/**
 * Copyright (c) Members of the EGEE Collaboration. 2004-2010. 
 * See http://www.eu-egee.org/partners/ for details on the copyright
 * holders.  
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); 
 * you may not use this file except in compliance with the License. 
 * You may obtain a copy of the License at 
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0 
 * 
 * Unless required by applicable law or agreed to in writing, software 
 * distributed under the License is distributed on an "AS IS" BASIS, 
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
 * See the License for the specific language governing permissions and 
 * limitations under the License.
 *
 *
 *  Authors:
 *  2009-
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     Mischa Sall\'e <msalle@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *     <grid-mw-security@nikhef.nl> 
 *
 *  2007-2009
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 *  2003-2007
 *     Martijn Steenbakkers <martijn@nikhef.nl>
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 */

#include <dlfcn.h>
#include <syslog.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <lcmaps/lcmaps.h>

#include "llgt_config.h"

#include "llgt_lcmaps.h"
#include "llgt_utils.h"

#ifndef LCMAPS_LIBDIR
    #define LCMAPS_LIBDIR ""
#endif
#ifndef LIBLCMAPS_SO
    #define LIBLCMAPS_SO "liblcmaps" LIBSUFF
#endif

static int set_liblcmaps_path(char ** liblcmaps_path);

static int set_liblcmaps_path(char ** liblcmaps_path)
{
    char * lcmaps_libdir       = NULL;
    char * lcmaps_moddir_sfx   = NULL;
    char * lcmaps_modules_dir  = NULL;
    struct stat st;
    int sn_retval;

    /* First get the $LLGT_LCMAPS_LIBDIR environment variable as a base setting */
    lcmaps_libdir = getenv("LLGT_LCMAPS_LIBDIR");
    if (lcmaps_libdir && ((lcmaps_libdir)[0] != '\0')) {
        /* If it's non-empty, check it is an absolute and valid directory */
        if ((lcmaps_libdir)[0] != '/' || stat(lcmaps_libdir, &st)!=0 || !S_ISDIR(st.st_mode))  {
            llgt_logmsg(LOG_WARNING, "%s: Ignoring $LLGT_LCMAPS_LIBDIR as \"%s\" is not an absolute path to a valid directory\n", __func__, lcmaps_libdir);
            lcmaps_libdir = LCMAPS_LIBDIR;
        }
        /* Get the potentially different $LCMAPS_MODULE_SFX, based on the customized libdir */
        lcmaps_moddir_sfx = getenv("LLGT_LCMAPS_MODULEDIR_SFX");
        if (!lcmaps_moddir_sfx) {
            lcmaps_moddir_sfx = LCMAPS_MODULEDIR_SFX; /* Take the input from the LLGT configure */
        }

        /* Concatting LCMAPS_LIBDIR and LCMAPS_MODULEDIR_SFX to be exported as $LCMAPS_MODULES_DIR */
        lcmaps_modules_dir = malloc(AP_MAXPATH);
        if (!lcmaps_modules_dir) {
            llgt_logmsg(LOG_ERR, "%s: Could not allocate memory: %s\n", __func__, strerror(errno));
            return 1;
        } else {
	    sn_retval=snprintf(lcmaps_modules_dir,AP_MAXPATH,"%s%s", lcmaps_libdir, lcmaps_moddir_sfx);
	    if (sn_retval<0)	{
		llgt_logmsg(LOG_WARNING, "snprintf failed when creating full modulespath, not setting LCMAPS_MODULES_DIR\n");
	    } else if ((size_t)sn_retval >= AP_MAXPATH) {
                llgt_logmsg(LOG_WARNING, "Full modulespath '%s%s' would be too long, not setting LCMAPS_MODULES_DIR\n", lcmaps_libdir,lcmaps_moddir_sfx);
            } else {
                llgt_logmsg(LOG_DEBUG, "Setting LCMAPS_MODULES_DIR to '%s'\n", lcmaps_modules_dir);
                setenv("LCMAPS_MODULES_DIR", lcmaps_modules_dir, 1);
            }
            free(lcmaps_modules_dir);
        }

        /* Setting the full path to the liblcmaps.so object */
        *liblcmaps_path = malloc(AP_MAXPATH);
	sn_retval=snprintf(*liblcmaps_path, AP_MAXPATH, "%s/%s", lcmaps_libdir, LIBLCMAPS_SO);
	if (sn_retval<0)    {
            llgt_logmsg(LOG_ERR, "snprintf failed when creating full path to lcmaps lib %s\n",LIBLCMAPS_SO);
            return 1;
	} else if ((size_t)sn_retval >= AP_MAXPATH) {
            llgt_logmsg(LOG_ERR, "Full path to %s \"%s/%s\" is too long\n", LIBLCMAPS_SO, lcmaps_libdir, LIBLCMAPS_SO);
            return 1;
        }
    } else {
        /* Just set the name of the shared object file, let the system figure it out */
	if ( (*liblcmaps_path = strdup(LCMAPS_LIBDIR LIBLCMAPS_SO)) == NULL)
	    return 1;
    }
    llgt_logmsg(LOG_DEBUG, "LCMAPS library path : \"%s\"\n", *liblcmaps_path);
    return 0;
}

static int policy_tokenize(char ***policies);

/*!
 * Splits LCMAPS_POLICY_NAME value into array of policy names.
 * \return number of policies or -1 on error
 */
static int policy_tokenize(char ***policies)	{
    char *input=getenv("LCMAPS_POLICY_NAME");
    char *inp_copy,*pos1,*pos2,**pol_array;
    size_t i;
    int j,npol=0;

    /* initialize */
    *policies=NULL;

    /* anything specified? */
    if (input==NULL || strlen(input)==0)
	return 0;

    /* Count the number of colons: upper limit for number of policies */
    npol=1;
    for (i=0; i<strlen(input); i++)	{
	if (input[i]==':')
	    npol++;
    }

    /* Make a local copy of the env variable, which we can modify */
    if ((inp_copy=strdup(input))==NULL)	{
	return -1;
    }
    /* Now malloc the memory needed for the policies: add one for a NULL
     * termination */
    if ( (pol_array=(char **)malloc(sizeof(char*)*((size_t)npol+1)))==NULL) {
	free(inp_copy);
	return -1;
    }

    /* Get all (non-empty) policies */
    pos1=inp_copy;
    /* restart from 0 since we want only non-empty policy names */
    npol=0;
    while (1)	{
	/* Check if there is a colon -> set it to null character */
	if ( (pos2=strchr(pos1,':')) != NULL )
	    pos2[0]='\0';
	/* Check if the name is non-empty */
	if (pos1[0]!='\0')  {
	    if ((pol_array[npol]=strdup(pos1))==NULL)	{
		/* Cleanup and return */
		free(inp_copy);
		for (j=0; j<npol; j++)
		    free(pol_array[j]);
		return -1;
	    }
	    npol++;
	}
	/* Was this the last entry? */
	if (pos2==NULL)
	    break;
	/* Update the starting pointer */
	pos1=pos2+1;
    }

    /* Check there is at least one policy */
    if (npol>0)	{
	pol_array[npol]=NULL;
	*policies=pol_array;
    } else  /* No policies: free pol_array */
	free(pol_array);
    /* Free temporary local storage */
    free(inp_copy);
    
    return npol;
}

void llgt_setup_lcmaps_environment(void)
{
    if (!getenv ("LCMAPS_DEBUG_LEVEL"))
       setenv ("LCMAPS_DEBUG_LEVEL", "3", 1);

/* Not required by default, because the setting of the LCMAPS_LOG_FILE will declare to log to file, or use Syslog instead */
#ifdef LCAS_LCMAPS_FORCE_LOG_TO_FILE
    if (!getenv ("LCMAPS_LOG_FILE"))
        setenv ("LCMAPS_LOG_FILE",  "/var/log/gt_lcas_lcmaps.log", 1);
#endif

    if (!getenv ("LCMAPS_DB_FILE"))
        setenv("LCMAPS_DB_FILE", SYSCONFDIR"/lcmaps/lcmaps.db", 1);

    if (!getenv ("LCMAPS_DIR"))
        setenv("LCMAPS_DIR", SYSCONFDIR, 1);

    /* if (!getenv ("LCMAPS_MOD_HOME")) */
        /* setenv("LCMAPS_MOD_HOME", LCMAPS_MODULEDIR, 1); */

    if (!getenv ("LCMAPS_POLICY_NAME"))
        setenv ("LCMAPS_POLICY_NAME", "", 1);
}


int llgt_run_lcmaps(gss_cred_id_t user_cred_handle, char * client_name, FILE * logfile, char ** username)
{
    /* USER MAPPING */
    /* Define lcmaps_handle static so that we can reuse it */
    static void *  lcmaps_handle   = NULL;
    int rc=1; /* Return code is error unless success */
    char *  llgt_dlclose_lcmaps=NULL;
    char *  error           = NULL;
    int     retval          = 0;
    char *  liblcmaps_path  = NULL;

    int i,npol=0;
    char **policies=NULL;

    int (*LcmapsInit)(FILE *);
    int (*LcmapsTerm)();
    void (*LcmapsEnableVomsAttributesVerification)();
    void (*LcmapsDisableVomsAttributesVerification)();
#if ALLOW_EMPTY_CREDENTIALS
    int (*LcmapsRunAndReturnUsername)(char*, gss_cred_id_t, char*, char**, int, char**);
#else
    int (*LcmapsRunAndReturnUsername)(gss_cred_id_t, char*, char**, int, char**);
#endif /* ALLOW_EMPTY_CREDENTIALS */

    /* Check we don't have a handle yet */
    if (lcmaps_handle==NULL)	{
	/* Get the path to the to be opened LCMAPS main library - Must be free'd
	 * */
	if (set_liblcmaps_path(&liblcmaps_path)) {
	    llgt_logmsg(LOG_ERR, "Couldn't set the path to \"%s\"\n", LIBLCMAPS_SO);
	    return 1;
	}

	/* I must have a path set */
	if (!liblcmaps_path) {
	    llgt_logmsg(LOG_ERR, "Failed set a name or path to find %s\n",
			LIBLCMAPS_SO);
	    return 1;
	}

	/* Now open the library */
        lcmaps_handle = dlopen(liblcmaps_path, RTLD_LAZY|RTLD_GLOBAL);
    
	if (!lcmaps_handle) {
	    llgt_logmsg(LOG_ERR,
		"Failed to dynamically load the library for LCMAPS \"%s\": %s\n",
		liblcmaps_path,dlerror());
	    free(liblcmaps_path);
	    return 1;
	}
	free(liblcmaps_path); liblcmaps_path=NULL;
    }

    /* Map the symbols */
    /* Init */
    LcmapsInit=dlsym(lcmaps_handle,"lcmaps_init");
    if ((error = dlerror()) != NULL) {
	llgt_logmsg(LOG_ERR, "LCMAPS module not compliant: Symbol \"%s\" not found: %s", "lcmaps_init", error);
	goto llgt_run_lcmaps_finalize;
    }
    /* Run */
    LcmapsRunAndReturnUsername=dlsym(lcmaps_handle,"lcmaps_run_and_return_username");
    if ((error = dlerror()) != NULL) {
	llgt_logmsg(LOG_ERR, "LCMAPS module not compliant: Symbol \"%s\" not found: %s", "lcmaps_run_and_return_username", error);
	goto llgt_run_lcmaps_finalize;
    }
    /* Term */
    LcmapsTerm=dlsym(lcmaps_handle,"lcmaps_term");
    if ((error = dlerror()) != NULL) {
	llgt_logmsg(LOG_ERR, "LCMAPS module not compliant: Symbol \"%s\" not found: %s", "lcmaps_term", error);
	goto llgt_run_lcmaps_finalize;
    }
    /* VOMS: Call to explicitly ENABLE the VOMS credential verification */
    LcmapsEnableVomsAttributesVerification=dlsym(lcmaps_handle,"lcmaps_enable_voms_attributes_verification");
    if ((error = dlerror()) != NULL) {
	llgt_logmsg(LOG_DEBUG, "LCMAPS module does not support explicit VOMS enable- or disablement. Symbol \"%s\" not found: %s", "lcmaps_enable_voms_attributes_verification", error);
    }
    /* VOMS: Call to explicitly DISABLE the VOMS credential verification */
    LcmapsDisableVomsAttributesVerification=dlsym(lcmaps_handle,"lcmaps_disable_voms_attributes_verification");
    if ((error = dlerror()) != NULL) {
	llgt_logmsg(LOG_DEBUG, "LCMAPS module does not support explicit VOMS enable- or disablement. Symbol \"%s\" not found: %s", "lcmaps_disable_voms_attributes_verification", error);
    }

    /* Initialize LCMAPS, do user mapping and terminate LCMAPS */
    retval=(*LcmapsInit)(logfile);
    if (retval)
    {
	llgt_logmsg(LOG_ERR, "LCMAPS initialization failure.");
	goto llgt_run_lcmaps_finalize;
    }

    /* VOMS: Check if the VOMS credential verification needs to be disabled
     * or explicitly enabled. Assuming there was a symbol in LCMAPS found
     * that is capable of performing the behavioural change. */
    if (LcmapsDisableVomsAttributesVerification && getenv("LLGT_VOMS_DISABLE_CREDENTIAL_CHECK")) {
	(*LcmapsDisableVomsAttributesVerification)();
    }
    if (LcmapsEnableVomsAttributesVerification && getenv("LLGT_VOMS_ENABLE_CREDENTIAL_CHECK")) {
	(*LcmapsEnableVomsAttributesVerification)();
    }

    /* Split the LCMAPS_POLICY_NAME contents into an array of policy 
     * names */
    if ( (npol=policy_tokenize(&policies)) < 0) {
	llgt_logmsg(LOG_ERR,
	    "Failed to parse value of env variable LCMAPS_POLICY_NAME");
	npol=0;
    }

#if ALLOW_EMPTY_CREDENTIALS
    retval=(*LcmapsRunAndReturnUsername)(client_name,
					 user_cred_handle,
					 NULL,
					 username,
					 npol,
					 policies
					);
#else
    retval=(*LcmapsRunAndReturnUsername)(user_cred_handle,
					 NULL,
					 username,
					 npol,
					 policies
					);
#endif

    /* Free memory malloced: note that the policies[0] points to the result
     * of the strdup */
    for (i=0; i<npol; i++)
	free(policies[i]);
    free(policies);

    if (retval) {
	llgt_logmsg(LOG_WARNING, "Warning: failed mapping. LCMAPS returned: %d\n", retval);
	if ((*LcmapsTerm)())
	    llgt_logmsg(LOG_ERR, "LCMAPS Termination failure!");

	goto llgt_run_lcmaps_finalize;
    }
    retval=(*LcmapsTerm)();
    if (retval) {
	llgt_logmsg(LOG_ERR, "LCMAPS Termination failure!");
	goto llgt_run_lcmaps_finalize;
    }

    /* END USER MAPPING */
    rc=0;

llgt_run_lcmaps_finalize:
    /* Do we want to skip dlclosing LCMAPS ? */
    if ((llgt_dlclose_lcmaps=getenv("LLGT_DLCLOSE_LCMAPS"))==NULL ||
	(strcasecmp(llgt_dlclose_lcmaps, "no")!=0 &&
	 strcasecmp(llgt_dlclose_lcmaps, "disabled")!=0 &&
	 strcasecmp(llgt_dlclose_lcmaps, "disable")!=0))
    {
	if (dlclose(lcmaps_handle))
	    llgt_logmsg(LOG_ERR, "Warning: dlclose() failed for lcmaps: %s\n",
		dlerror());
	lcmaps_handle=NULL;
    }

    return rc;
}
