/**
 * Copyright (c) Members of the EGEE Collaboration. 2004-2010.
 * See http://www.eu-egee.org/partners/ for details on the copyright
 * holders.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 *
 *  Authors:
 *  2009-
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     Mischa Sall\'e <msalle@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *     <grid-mw-security@nikhef.nl>
 *
 *  2007-2009
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 *  2003-2007
 *     Martijn Steenbakkers <martijn@nikhef.nl>
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 */


/*
 * Author of GT4 Mapping Call Out for LCMAPS:
 *           Oscar Koeroo <okoeroo@nikhef.nl>
 *
 * See LICENCE file for licence details
 */

/*
 * Globus Gridmap Callout
 * @author Markus Lorch, based on PRIMA module
 *         and Globus skeleton module by Sam Meder
 */

/*
 * Special thanks to:
 *           The UK National Grid Support (NGS) - http://www.grid-support.ac.uk/
 *           Robert Frank    - for supplying the code
 *           Stephen Pickles - for bringing me in contact with Robert Frank (and collegues)
 *
 * for the tips, hints and example code of how they solved this integration work of the Globus GT4 tools utilzing LCAS and LCMAPS in this interface.
 *
 */

#ifndef EXTERN_C_BEGIN
#    ifdef __cplusplus
#        define EXTERN_C_BEGIN extern "C" {
#        define EXTERN_C_END }
#    else
#        define EXTERN_C_BEGIN
#        define EXTERN_C_END
#    endif
#endif


/* General includes */
#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <strings.h>

#include <unistd.h>
#include <errno.h>

#include <gssapi.h>
#include <globus_gss_assist.h>
#include <globus_gridmap_callout_error.h>

#include "llgt_config.h"

/* LCAS and LCMAPS specific */
#ifndef DISABLE_LCAS
# include "llgt_lcas.h"
#endif
#include "llgt_lcmaps.h"

#include "llgt_utils.h"




EXTERN_C_BEGIN



/* prototypes */
globus_result_t lcmaps_callout( va_list ap);
static int llgt_run_lcas_authz_and_lcmaps_mapping (gss_ctx_id_t context, char ** local_identity);

/* -------- */

/***************************************************
Function:       run_lcas_authz_and_lcmaps_mapping
arguments:      gss_ctx_id_t context    - A context with all the credentials in it
                char ** local_identity  - A return value of the username to which the user was mapped by LCMAPS

Return:         1 : Error
                0 : All is well
****************************************************/
static int llgt_run_lcas_authz_and_lcmaps_mapping (gss_ctx_id_t context, char ** local_identity)
{
    int              retval             = 0;
    gss_cred_id_t    user_cred_handle   = GSS_C_NO_CREDENTIAL;
    char            *client_name        = NULL;
    FILE            *flog_lcmaps        = NULL;
#ifndef DISABLE_LCAS
    FILE            *flog_lcas          = NULL;
    char            *run_lcas      = NULL;
    int              do_run_lcas        = 1; /* Run LCAS by default */
#endif

    /* get user credential from context */
    user_cred_handle = llgt_get_user_cred_handle(context);
    client_name=llgt_get_client_name(context);

#ifndef DISABLE_LCAS
    /* Selectively run LCAS or not. Use "disabled" or "no" as value */
    if ((run_lcas = getenv("LLGT_RUN_LCAS"))) {
        if ((strcasecmp(run_lcas, "no") == 0) ||
            (strcasecmp(run_lcas, "disabled") == 0) ||
            (strcasecmp(run_lcas, "disable") == 0)) {

            /* Disable running LCAS */
            do_run_lcas = 0; /* Off */
        } else {
            /* Explicitly enable running LCAS */
            do_run_lcas = 1; /* On */
        }
    }

    /* When enabled, run LCAS */
    if (do_run_lcas) {
        /* Setup environment for lcas */
        llgt_setup_lcas_environment();
        if (getenv ("LCAS_LOG_FILE")) {
            flog_lcas = fopen (getenv ("LCAS_LOG_FILE"), "a");
            if (!flog_lcas)
                llgt_logmsg(LOG_ERR,
                        "Error: failed to open the LCAS log file at the $LCAS_LOG_FILE location: %s. Redirecting LCAS logs Syslog\n",
                        getenv ("LCAS_LOG_FILE"));
        }

        /* Execute LCAS with GLOBUS INTERFACE TO LCAS */
        if (llgt_run_lcas (user_cred_handle, client_name, flog_lcas)) {
            llgt_logmsg(LOG_ERR, "Execution of LCAS failed.\n");
            free(client_name);
            retval = 1;
            goto finalize;
        } else {
            llgt_logmsg(LOG_DEBUG, "Execution of LCAS successful.\n");
        }

	/* Close the log file, if used/opened */
	if (flog_lcas)	{
	    fclose(flog_lcas);
	    flog_lcas=NULL;
	}
    }
#endif

    /* Setup environment for lcmaps*/
    llgt_setup_lcmaps_environment();
    if (getenv ("LCMAPS_LOG_FILE")) {
        flog_lcmaps = fopen (getenv ("LCMAPS_LOG_FILE"), "a");
        if (!flog_lcmaps)
            llgt_logmsg(LOG_ERR,
                    "Error: failed to open the LCAS log file at the $LCAS_LOG_FILE location: %s. Redirecting LCMAPS logs Syslog\n",
                    getenv ("LCMAPS_LOG_FILE"));
    }
    /* Generate $JOB_REPOSITORY_ID and $GATEKEEPER_JM_ID for accounting purposes */
    llgt_create_jobid();

    /* Execute LCMAPS */
    if (llgt_run_lcmaps(user_cred_handle, client_name, flog_lcmaps, local_identity)) {
        llgt_logmsg(LOG_ERR, "Execution of LCMAPS failed.\n");
        free(client_name);
        retval = 1;
        goto finalize;
    }

    /* No longer need client_name */
    free(client_name);

    /* Implicit post-LCMAPS mapping to 'root' protection */
    if ((getuid() == 0) || geteuid() == 0 || getgid() == 0 || getegid() == 0) {
        /* Override the Implicit post-LCMAPS mapping to 'root' protection */
        if (getenv("LLGT_LIFT_PRIVILEGED_PROTECTION")) {
            llgt_logmsg(LOG_DEBUG, "Found $LLGT_NO_CHANGE_USER. Overriding post-LCMAPS 'root' protection.\n");
        } else if (getenv("LLGT_NO_CHANGE_USER")) {
            llgt_logmsg(LOG_DEBUG, "Found $LLGT_NO_CHANGE_USER. Overriding post-LCMAPS 'root' protection. Please set $LLGT_LIFT_PRIVILEGED_PROTECTION in stead\n");
        } else if (getenv("LLGT4_NO_CHANGE_USER")) {
            llgt_logmsg(LOG_DEBUG, "Found $LLGT4_NO_CHANGE_USER. Overriding post-LCMAPS 'root' protection. Please set $LLGT_LIFT_PRIVILEGED_PROTECTION in stead\n");
        } else {
            /* Meaning, I'm still root, which triggered this auto protect sequence.
             * Lift the implicit protection to run from the GSI-OpenSSHd by
             * exporting $LLGT4_NO_CHANGE_USER or $LLGT_NO_CHANGE_USER */
            free(*local_identity); *local_identity = NULL;
            llgt_logmsg(LOG_ERR,
		    "You are still root after the LCMAPS execution. "
		    "The implicit root-mapping safety is enabled. See documentation for details.\n");
            retval = 1;
            goto finalize;
        }
    }

    /* Great success */
    llgt_logmsg(LOG_DEBUG,
	    "Execution of LCMAPS Succeeded. "
	    "Registering the mapped local identity \"%s\" to the GSI-AuthZ interface\n",
	    *local_identity);
    retval = 0;

finalize:
    /* Close the log file, if used/opened */
    if (flog_lcmaps) {
        fclose(flog_lcmaps);
	flog_lcmaps=NULL;
    }
    return retval;
}


/* Main function */
globus_result_t lcmaps_callout( va_list ap)
{
    gss_ctx_id_t                        context                 = NULL;
    char *                              service                 = NULL;
    char *                              desired_identity        = NULL;
    char *                              identity_buffer         = NULL;
    unsigned int                        buffer_length           = 0;

    /* pointer to a buffer with local user name allocated by SAML subroutines */
    /* result of LCMAPS callout, cached to be able to reuse it: globus callout
     * typically does a local username lookup before checking the local username
     * is allowed (userok) */
    static char *                       local_identity          = NULL;
    globus_result_t                     result                  = GLOBUS_SUCCESS;
    gss_name_t                          peer                    = NULL;
    gss_buffer_desc                     peer_name_buffer; /* This is a struct */
    gss_name_t                          own                     = NULL;
    gss_buffer_desc                     own_name_buffer; /* This is a struct */
    OM_uint32                           major_status            = 0;
    OM_uint32                           minor_status            = 0;
    int                                 rc                      = 0;
    int                                 initiator               = 0;
    
    /* whether to reuse the result from a previous callout */
    char *                              llgt_cache_callout      = NULL;

    /* gss_priv_attribute_set              attributes;                */
    /* gss_id_set                          attribute_types_required; */

    /* ---------------------------------------------------- */

    context = va_arg(ap, gss_ctx_id_t);
    service = va_arg(ap, char *);
    desired_identity = va_arg(ap, char *);
    identity_buffer = va_arg(ap, char *);
    buffer_length = va_arg(ap, unsigned int);

    /* ---------------------------------------------------- */

    /* Setup logging for this plugin */
    llgt_open_log();

    /* load required modules */
    rc = globus_module_activate(GLOBUS_GRIDMAP_CALLOUT_ERROR_MODULE);
    if(rc!=0) {
        llgt_logmsg(LOG_ERR, "Error activating Globus GRIDMAP CALLOUT ERROR MODULE");
        result = GLOBUS_FAILURE;
        goto llgt_finalize;
    }

    /* If we have a previous local identity but we do not have a desired
     * identity or we do not want caching to be used (LLGT_CACHE_CALLOUT is no,
     * disable or disabled), we should remove the previous local identity */
    if (local_identity &&
	(desired_identity==NULL ||
	 ( (llgt_cache_callout=getenv("LLGT_CACHE_CALLOUT"))!=NULL &&
	      (strcasecmp(llgt_cache_callout, "no")==0 ||
	       strcasecmp(llgt_cache_callout, "disabled")==0 ||
	       strcasecmp(llgt_cache_callout, "disable")==0
	      )
	 )
	)
       )
    {
	/* Reset existing local identity */
	free(local_identity);
	local_identity=NULL;
    }

    /* If we are running in userok mode, i.e. we have a non-empty
     * desired_identity, and we have a cached result from LCMAPS, just check the
     * two match */
    if (desired_identity && local_identity) {
	/* We try a userok, and have a cached identity: can use that instead */
	if (strcmp(desired_identity, local_identity)==0)    {
	    /* We have a match, check it fits in the buffer */
	    if (strlen(local_identity) < buffer_length) {
		/* Copy local identity to Globus Arena and return  */
		strcpy(identity_buffer, local_identity);
		llgt_logmsg(LOG_INFO,
		    "Reusing cached \"LCMAPS\" result (service %s): \"%s\"",
		    service ? service : "unknown", identity_buffer);
		result=GLOBUS_SUCCESS;
		goto llgt_finalize;
	    }
	    /* Buffer too small: fatal error */
	    GLOBUS_GRIDMAP_CALLOUT_ERROR(
		    result,
		    GLOBUS_GRIDMAP_CALLOUT_BUFFER_TOO_SMALL,
		    ("Local identity length: %d, Buffer length: %d\n",
			strlen(local_identity), buffer_length));
	    /* Log error */
	    llgt_logmsg(LOG_ERR,
		    "Buffer length %d too small for identity %s",
		    buffer_length, local_identity);
	    /* Reset failed local identity */
	    free(local_identity);
	    local_identity=NULL;
	    result=GLOBUS_FAILURE;
	    goto llgt_finalize;
	}
	/* Cached result does not match desired identity */
	llgt_logmsg(LOG_WARNING,
		"Cached identity \"%s\" does not match desired identity \"%s\", "
		"rerunning mapping callout\n",
		local_identity, desired_identity);
	/* Reset local identity */
	free(local_identity);
	local_identity=NULL;
    }

    /* Need to do actual run: start necessary GSS modules */
    rc = globus_module_activate(GLOBUS_GSI_GSS_ASSIST_MODULE);
    if(rc!=0) {
        llgt_logmsg(LOG_ERR, "Error activating Globus GSS ASSIST MODULE");
        result = GLOBUS_FAILURE;
        goto llgt_finalize;
    }

    rc = globus_module_activate(GLOBUS_GSI_GSSAPI_MODULE);
    if(rc!=0) {
        llgt_logmsg(LOG_ERR, "Error activating Globus GSSAPI MODULE");
        result = GLOBUS_FAILURE;
        goto llgt_finalize;
    }

    /* find out who is the initiator of this connection (typically the client) */
    major_status = gss_inquire_context(&minor_status,
                                       context,
                                       GLOBUS_NULL,
                                       GLOBUS_NULL,
                                       GLOBUS_NULL,
                                       GLOBUS_NULL,
                                       GLOBUS_NULL,
                                       &initiator,
                                       GLOBUS_NULL);
    if(GSS_ERROR(major_status)) {
        GLOBUS_GRIDMAP_CALLOUT_GSS_ERROR(result, major_status, minor_status);
        llgt_logmsg(LOG_ERR, "Error inquiring context to find the initiator");
        goto llgt_finalize;
    }

    /* get our and the peer's name */
    major_status = gss_inquire_context(&minor_status,
                                       context,
                                       initiator ? &own : &peer,
                                       initiator ? &peer : &own,
                                       GLOBUS_NULL,
                                       GLOBUS_NULL,
                                       GLOBUS_NULL,
                                       GLOBUS_NULL,
                                       GLOBUS_NULL);
    if(GSS_ERROR(major_status)) {
        GLOBUS_GRIDMAP_CALLOUT_GSS_ERROR(result, major_status, minor_status);
        llgt_logmsg(LOG_ERR, "Error inquiring context to extract the identity of the peer");
        goto llgt_finalize;
    }

    major_status = gss_display_name(&minor_status,
                                    peer,
                                    &peer_name_buffer,
                                    GLOBUS_NULL);
    if(GSS_ERROR(major_status)) {
        GLOBUS_GRIDMAP_CALLOUT_GSS_ERROR(result, major_status, minor_status);
        gss_release_name(&minor_status, &peer);
        goto llgt_finalize;
    }

    gss_release_name(&minor_status, &peer);

    major_status = gss_display_name(&minor_status,
                                    own,
                                    &own_name_buffer,
                                    GLOBUS_NULL);
    if(GSS_ERROR(major_status)) {
        GLOBUS_GRIDMAP_CALLOUT_GSS_ERROR(result, major_status, minor_status);
        gss_release_name(&minor_status, &own);
        goto llgt_finalize;
    }

    gss_release_name(&minor_status, &own);


    /*************************
     * Start AuthZ with LCAS *
     *************************/
    if (llgt_run_lcas_authz_and_lcmaps_mapping (context, &local_identity)) {
        /* llgt_logmsg(LOG_ERR, "Failure in Authorization and Mapping"); */
        result = GLOBUS_FAILURE;
        goto llgt_finalize;
    }

    /* --------------------------------------------------------- */
    /* Copy local identity to Globus Arena and return */
    /* --------------------------------------------------------- */

    if (strlen(local_identity) + 1 > buffer_length) {
        GLOBUS_GRIDMAP_CALLOUT_ERROR(
		result,
		GLOBUS_GRIDMAP_CALLOUT_BUFFER_TOO_SMALL,
                ("Local identity length: %d Buffer length: %d\n",
		 strlen(local_identity),
		 buffer_length)
		);
        llgt_logmsg(LOG_ERR,
		"Failed map user. Buffer to small: local identity length: %d Buffer length: %d",
		strlen(local_identity),
		buffer_length);
    } else {
        strcpy(identity_buffer, local_identity);
        llgt_logmsg(LOG_INFO,
		"Callout to \"LCMAPS\" returned local user (service %s): \"%s\"",
		service ? service : "unknown", identity_buffer);
    }

    gss_release_buffer(&minor_status, &peer_name_buffer);
    gss_release_buffer(&minor_status, &own_name_buffer);

    result = GLOBUS_SUCCESS;

    /* syslog log closing */
    /* closelog(); */

llgt_finalize:
    llgt_close_log();

    /* free strings that have possibly been allocated */
    /* if they werent allocated they will be NULL and this does no harm */
/*    free(local_identity);*/ /* Do not free, we will reuse it in the userok
				 call */

    /* ------------------------------ */
    /* Deactivate the Globus modules */
    /* ------------------------------ */
    /* Seems unsafe to deactivate globus modules: some of them unload part of
     * OpenSSL: certainly GLOBUS_GSI_GSS_ASSIST_MODULE, perhaps others */
/*  globus_module_deactivate(GLOBUS_GSI_GSSAPI_MODULE);
    globus_module_deactivate(GLOBUS_GSI_GSS_ASSIST_MODULE);
    globus_module_deactivate(GLOBUS_GRIDMAP_CALLOUT_ERROR_MODULE);*/

    /* syslog log closing */
    /* closelog(); */

    return result;
}


EXTERN_C_END
