package latexDraw.util;

import static java.lang.Math.PI;

import java.awt.geom.Point2D;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.Serializable;

import latexDraw.figures.Line;


/** 
 * This class allows a LaTeXDrawPoint2D to be serializable (it's not the case for java 1.5)<br>
 * <br>
 * This file is part of LaTeXDraw<br>
 * Copyright (c) 2005-2008 Arnaud BLOUIN<br>
 *<br>
 *  LaTeXDraw is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.<br>
 *<br>
 *  LaTeXDraw is distributed without any warranty; without even the 
 *  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 *  PURPOSE. See the GNU General Public License for more details.<br>
 *<br>
 * 01/20/06<br>
 * @author Arnaud BLOUIN<br>
 * @version 2.0.0<br>
 */
public class LaTeXDrawPoint2D extends Point2D.Double implements Serializable
{
	private static final long serialVersionUID = 1L;

	
	/**
	 * Constructs a point from a java point2D
	 * @param pt The java point2D
	 */
	public LaTeXDrawPoint2D(Point2D pt)
	{
		if(pt==null)
			x = y = 0.;
		else
			setLocation(pt);
	}
	
	
	
	/**
	 * Constructs and initialises a Point2D with the specified coordinates.
	 * @param x The X-coordinate to which to set the newly constructed Point2D
	 * @param y The Y-coordinate to which to set the newly constructed Point2D
	 */
	public LaTeXDrawPoint2D(double x, double y)
	{
		super(x, y);
	}
	
	
	
	/**
	 * Constructs and initialises a Point2D with coordinates (0, 0).
	 */
	public LaTeXDrawPoint2D()
	{
		super();
	}
	
	
	
	/**
	 * Allows to deserialize the point2D.Double
	 * @param ois The input stream
	 */
	private void readObject(ObjectInputStream ois) throws IOException
	{
		x = ois.readDouble();
		y = ois.readDouble();
	}
	
	
	
	/**
	 * Allows to know if the point p is equal to "this" considering a gap.
	 * @param p The point to compare
	 * @param gap The approximation gap
	 * @return True if they are equals
	 * @exception IllegalArgumentException When <code>gap<0</code>
	 */
	public boolean equals(LaTeXDrawPoint2D p, double gap)
	{
		if(gap<0)
			throw new IllegalArgumentException();
		
		if(p==null) return false;
		//Fixed: < and > are replaced by <= and >=
		return p.x-gap<=x && p.x+gap>=x && p.y-gap<=y && p.y+gap>=y;
	}
	
	
	
	/**
	 * Allows to get middle point of these two points
	 * @param p The second point
	 * @return The middle point
	 */
	public LaTeXDrawPoint2D getMiddlePoint(LaTeXDrawPoint2D p)
	{
		return new LaTeXDrawPoint2D((x+p.x)/2., (y+p.y)/2.);
	}
	
	
	
	/**
	 * Allows to move a point with the difference between a the former position
	 * of a point p and its new.
	 * @param oldP The old position of the point of reference
	 * @param newP The new position of the point of reference
	 */
	public synchronized void move(LaTeXDrawPoint2D oldP, LaTeXDrawPoint2D newP)
	{
		x+=newP.x-oldP.x;
		y+=newP.y-oldP.y;
	}


	
	/**
	 * Returns vertically the point.
	 * @param origin The location of the vertical axe.
	 */
	public Point2D verticalSymmetry(LaTeXDrawPoint2D origin)
	{
		try
		{
			Line l = new Line(origin.y, (LaTeXDrawPoint2D)origin.clone(), false);
			LaTeXDrawPoint2D o2 = new LaTeXDrawPoint2D(x, origin.y);
			double distance = distance(o2);
			Line perp = l.getPerpendicularLine(o2, false);
			LaTeXDrawPoint2D[] pts = perp.findPoints(o2, distance);
			
			if(pts==null)
				return null;
			
			if(pts.length==1)
				return pts[0];
			
			return distance(pts[0])>distance(pts[1]) ? pts[0] : pts[1];
			
		}catch(Exception e) { return null; }
	}



	/**
	 * Returns horizontally the point.
	 * @param origin The location of the horizontal axe.
	 */
	public Point2D horizontalSymmetry(LaTeXDrawPoint2D origin)
	{
		try
		{
			Line l = new Line((LaTeXDrawPoint2D)origin.clone(), 
								new LaTeXDrawPoint2D(origin.x, origin.y-10), false);
			LaTeXDrawPoint2D o2 = new LaTeXDrawPoint2D(origin.x, y);
			double distance = distance(o2);
			Line perp = l.getPerpendicularLine(o2, false);
			LaTeXDrawPoint2D[] pts = perp.findPoints(o2, distance);
			
			if(pts==null)
				return null;
			
			if(pts.length==1)
				return pts[0];
			
			return distance(pts[0])>distance(pts[1]) ? pts[0] : pts[1];
			
		}catch(Exception e) { return null; }
	}
	
	
	
	
	/**
	 * Get a point by central symmetry.
	 * @param centre The centre of the symmetry.
	 * @return The resulting point.
	 * @since 1.9
	 */
	public LaTeXDrawPoint2D centralSymmetry(LaTeXDrawPoint2D centre)
	{
		return rotatePoint(centre, Math.PI);
	}
	
	
	
	/**
	 * Allows to rotate a point with as reference an other point.
	 * @param gravityC The point of reference.
	 * @param theta The angle of rotation (in rad).
	 * @return The rotated point.
	 * @since 1.9
	 */
	public LaTeXDrawPoint2D rotatePoint(LaTeXDrawPoint2D gravityC, double theta)
	{
		LaTeXDrawPoint2D pt = new LaTeXDrawPoint2D();
		double cosTheta;
		double sinTheta;

		if(theta<0.)
			theta = 2.*PI + theta;
		
		if((theta%(2.*PI))==0.)
			return (LaTeXDrawPoint2D)clone();
		
		if((theta%(2.*PI))==(PI/2.))
		{	
			cosTheta = 0.;
			sinTheta = 1.;
		}
		else
		{
			if((theta%(2.*PI))==PI)
			{
				cosTheta = -1.;
				sinTheta = 0.;
			}
			else
			{
				if((theta%(2.*PI))==(3.*PI/2.))
				{
					cosTheta = 0.;
					sinTheta = -1.;
				}
				else
				{
					cosTheta = Math.cos(theta);
					sinTheta = Math.sin(theta);
				}
			}
		}
		
		pt.x = cosTheta * (x - gravityC.x) - sinTheta * (y - gravityC.y) + gravityC.x;
		pt.y = sinTheta * (x - gravityC.x) + cosTheta * (y - gravityC.y) + gravityC.y;

		return pt;
	}
}
