package latexDraw.generators.svg;

import java.awt.geom.Point2D;
import java.util.Vector;

import latexDraw.figures.ArrowHead;
import latexDraw.figures.JoinedLines;
import latexDraw.figures.Line;
import latexDraw.figures.properties.Arrowable;
import latexDraw.parsers.svg.SVGAttributes;
import latexDraw.parsers.svg.SVGDocument;
import latexDraw.parsers.svg.elements.*;
import latexDraw.parsers.svg.elements.path.SVGPathSegLineto;
import latexDraw.parsers.svg.elements.path.SVGPathSegList;
import latexDraw.psTricks.PSTricksConstants;
import latexDraw.util.LaTeXDrawNamespace;
import latexDraw.util.LaTeXDrawPoint2D;

/**
 * Defines a SVG generator for some joined lines.<br>
 *<br>
 * This file is part of LaTeXDraw.<br>
 * Copyright (c) 2005-2008 Arnaud BLOUIN<br>
 *<br>
 *  LaTeXDraw is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.<br>
 *<br>
 *  LaTeXDraw is distributed without any warranty; without even the 
 *  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 *  PURPOSE. See the GNU General Public License for more details.<br>
 *<br>
 * 11/11/07<br>
 * @author Arnaud BLOUIN<br>
 * @version 0.1<br>
 */
public class LLinesSVGGenerator extends LPolygonSVGGenerator
{

	public LLinesSVGGenerator(JoinedLines f)
	{
		super(f);
	}

	
	
	/**
	 * Creates some lines using a SVG path.
	 * @param elt The SVG path.
	 */
	public LLinesSVGGenerator(SVGPathElement elt)
	{
		super(new JoinedLines(new LaTeXDrawPoint2D(), new LaTeXDrawPoint2D(1,1), true));
		
		if(elt==null || !elt.isLines())
			throw new IllegalArgumentException();
		
		SVGPathSegList segs = elt.getSegList();
		JoinedLines jl = (JoinedLines)getShape();
		int i, size;
		
		for(i=0, size=segs.size()-1; i<size; i++)
			jl.addPoint(new LaTeXDrawPoint2D(((SVGPathSegLineto)segs.elementAt(i)).getPoint()));
		
		jl.removePointAt(0);
		jl.removePointAt(0);
		jl.updateShape();
		jl.updateBorders();
		setSVGParameters(elt);
		applyTransformations(elt);
	}
	
	
	
	/**
	 * Creates some joined-liens from an SVG polyline element.
	 * @param elt The source element.
	 * @since 2.0.0
	 */
	public LLinesSVGGenerator(SVGPolyLineElement elt)
	{
		this(new JoinedLines(new LaTeXDrawPoint2D(), new LaTeXDrawPoint2D(1,1), true));
		
		JoinedLines jl = (JoinedLines)getShape();
		
		setSVGPolylineParameters(elt);
		jl.removePointAt(0);
		jl.removePointAt(0);
		jl.updateShape();
		jl.updateBorders();
		applyTransformations(elt);
	}
	
	
	
	
	public LLinesSVGGenerator(SVGGElement elt)
	{
		this(elt, true);
	}
	
	
	
	/**
	 * Creates some joined-lines from a latexdraw-SVG element.
	 * @param elt The source element.
	 * @since 2.0.0
	 */
	public LLinesSVGGenerator(SVGGElement elt, boolean withTransformation)
	{
		this(new JoinedLines(new LaTeXDrawPoint2D(), new LaTeXDrawPoint2D(1,1), true));

		setNumber(elt);
		SVGElement elt2 = getLaTeXDrawElement(elt, null);
		
		if(elt==null || elt2==null || !(elt2 instanceof SVGPolyLineElement))
			throw new IllegalArgumentException();
		
		JoinedLines l = (JoinedLines)getShape();
		ArrowHead arrowHead1 = l.getArrowHead1();
		ArrowHead arrowHead2 = l.getArrowHead2();
		SVGPolyLineElement main = (SVGPolyLineElement)elt2;
		setSVGLatexdrawParameters(elt);
		setSVGParameters(main);
		setSVGPolylineParameters(main);
		l.removePointAt(0);
		l.removePointAt(0);
		arrowHead1.setPosition(l.getPoint(0));
		arrowHead1.setLine(new Line(l.getPoint(0), l.getPoint(1), false));
		arrowHead1.setFigure(l);
		arrowHead2.setPosition(l.getPoint(-1));
		arrowHead2.setLine(new Line(l.getPoint(-1), l.getPoint(l.getNbPoints()-2), false));
		arrowHead2.setFigure(l);
		l.updateShape();
		l.updateBorders();
		
		setSVGArrow(arrowHead1, main.getAttribute(main.getUsablePrefix()+SVGAttributes.SVG_MARKER_START), main);
		setSVGArrow(arrowHead2, main.getAttribute(main.getUsablePrefix()+SVGAttributes.SVG_MARKER_END), main);
		homogeniseArrows(arrowHead1, arrowHead2);
		setSVGShadowParameters(getLaTeXDrawElement(elt, LaTeXDrawNamespace.XML_TYPE_SHADOW));
		setSVGDbleBordersParameters(getLaTeXDrawElement(elt, LaTeXDrawNamespace.XML_TYPE_DBLE_BORDERS));
		
		if(withTransformation)
			applyTransformations(elt);
	}
	
	
	
	private void setSVGPolylineParameters(SVGPolyLineElement e)
	{
		setSVGParameters(e);
		Vector<Point2D> ptsPol = e.getPoints2D();
		JoinedLines l = (JoinedLines)getShape();
		
		if(ptsPol==null)
			throw new IllegalArgumentException();
		
		for(Point2D pt : ptsPol)
			l.addPoint(new LaTeXDrawPoint2D(pt));
	}
	
	
	
	@Override
	public SVGElement toSVG(SVGDocument doc)
	{
		if(doc==null)
			return null;
		
		JoinedLines jl = (JoinedLines)shape;
        SVGElement root = new SVGGElement(doc), elt;
        int number = shape.getNumber();
		ArrowHead arrowHead1 = ((Arrowable)shape).getArrowHead1();
		ArrowHead arrowHead2 = ((Arrowable)shape).getArrowHead2();
		SVGDefsElement defs = doc.getFirstChild().getDefs();
		
		root.setAttribute(LaTeXDrawNamespace.LATEXDRAW_NAMESPACE + ':' + LaTeXDrawNamespace.XML_TYPE, LaTeXDrawNamespace.XML_TYPE_JOINED_LINES);
		root.setAttribute(SVGAttributes.SVG_ID, getSVGID());
		String points = "";//$NON-NLS-1$
		SVGElement arrow1 = null, arrow2 = null, arrow1Shad = null, arrow2Shad = null;
		String arrow1Name = "arrow1-" + number;//$NON-NLS-1$
		String arrow2Name = "arrow2-" + number;//$NON-NLS-1$
		String arrow1ShadName = "arrow1Shad-" + number;//$NON-NLS-1$
		String arrow2ShadName = "arrow2Shad-" + number;//$NON-NLS-1$

		if(!arrowHead1.isWithoutStyle())
		{
			arrow1     = new LArrowHeadSVGGenerator(arrowHead1).toSVG(doc, false);
			arrow1Shad = new LArrowHeadSVGGenerator(arrowHead1).toSVG(doc, true);

			arrow1.setAttribute(SVGAttributes.SVG_ID, arrow1Name);
			defs.appendChild(arrow1);

			if(shape.hasShadow())
			{
				arrow1Shad.setAttribute(SVGAttributes.SVG_ID, arrow1ShadName);
				defs.appendChild(arrow1Shad);
			}
		}

		if(!arrowHead2.isWithoutStyle())
		{
			arrow2     = new LArrowHeadSVGGenerator(arrowHead2).toSVG(doc, false);
			arrow2Shad = new LArrowHeadSVGGenerator(arrowHead2).toSVG(doc, true);

			arrow2.setAttribute(SVGAttributes.SVG_ID, arrow2Name);
			defs.appendChild(arrow2);

			if(shape.hasShadow())
			{
				arrow2Shad.setAttribute(SVGAttributes.SVG_ID, arrow2ShadName);
				defs.appendChild(arrow2Shad);
			}
		}

		for(LaTeXDrawPoint2D pt : jl.getPoints())
			points += pt.x + "," + pt.y + " ";//$NON-NLS-1$//$NON-NLS-2$

		if(shape.hasShadow())
		{
			SVGElement shad = new SVGPolyLineElement(doc);

			shad.setAttribute(SVGAttributes.SVG_POINTS, points);
			setSVGShadowAttributes(shad, false);
			root.appendChild(shad);

			if(arrow1Shad != null)
				shad.setAttribute(SVGAttributes.SVG_MARKER_START, "url(#" + arrow1ShadName + ")");//$NON-NLS-1$ //$NON-NLS-2$

			if(arrow2Shad != null)
				shad.setAttribute(SVGAttributes.SVG_MARKER_END, "url(#" + arrow2ShadName + ")");//$NON-NLS-1$ //$NON-NLS-2$
		}
		
        if(shape.hasShadow() && !shape.getLineStyle().equals(PSTricksConstants.LINE_NONE_STYLE) && shape.isFilled())
        {// The background of the borders must be filled is there is a shadow.
    		elt = new SVGPolyLineElement(doc);
    		elt.setAttribute(SVGAttributes.SVG_POINTS, points);
	        setSVGBorderBackground(elt, root);
        }

		elt = new SVGPolyLineElement(doc);
		elt.setAttribute(SVGAttributes.SVG_POINTS, points);
		root.appendChild(elt);

		if(shape.hasDoubleBoundary())
		{
			SVGElement dblBord = new SVGPolyLineElement(doc);

			dblBord.setAttribute(SVGAttributes.SVG_POINTS, points);
			setSVGDoubleBordersAttributes(dblBord);
			root.appendChild(dblBord);
		}

		setSVGAttributes(doc, elt, false);
		elt.setAttribute(LaTeXDrawNamespace.LATEXDRAW_NAMESPACE + ':' + LaTeXDrawNamespace.XML_ROTATION, String.valueOf(shape.getRotationAngle()));

		if(arrow1 != null)
			elt.setAttribute(SVGAttributes.SVG_MARKER_START, "url(#" + arrow1Name + ")");//$NON-NLS-1$ //$NON-NLS-2$

		if(arrow2 != null)
			elt.setAttribute(SVGAttributes.SVG_MARKER_END, "url(#" + arrow2Name + ")");//$NON-NLS-1$ //$NON-NLS-2$

		return root;
	}
}
