package latexDraw.figures;

import static java.lang.Math.toDegrees;

import java.awt.*;
import java.awt.geom.AffineTransform;
import java.awt.geom.Area;
import java.awt.geom.GeneralPath;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.util.Vector;

import latexDraw.figures.properties.BordersMovable;
import latexDraw.psTricks.DviPsColors;
import latexDraw.psTricks.PSTricksConstants;
import latexDraw.ui.LaTeXDrawFrame;
import latexDraw.ui.components.Delimitor;
import latexDraw.ui.components.MagneticGrid;
import latexDraw.util.LaTeXDrawPoint2D;


/** 
 * This class defines a rhombus.<br>
 *<br>
 * This file is part of LaTeXDraw.<br>
 * Copyright (c) 2005-2008 Arnaud BLOUIN<br>
 *<br>
 *  LaTeXDraw is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.<br>
 *<br>
 *  LaTeXDraw is distributed without any warranty; without even the 
 *  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 *  PURPOSE. See the GNU General Public License for more details.<br>
 *<br>
 * 01/20/06<br>
 * @author Arnaud BLOUIN<br>
 * @version 2.0.0<br>
 */
public class Rhombus extends LaTeXDrawPolygon implements BordersMovable
{
	private static final long serialVersionUID = 1L;

	/** The number of points of a rhombus */
	public static final short NB_POINTS_RHOMBUS = 4;
	
	

	/** 
	 * The constructor by default
	 */
	public Rhombus(boolean increaseMeter)
	{
		this(new LaTeXDrawPoint2D(), new LaTeXDrawPoint2D(), new LaTeXDrawPoint2D(), new LaTeXDrawPoint2D(), increaseMeter);
	}
	
	
	
	
	/**
	 * @param f The figure to copy.
	 * @param sameNumber True if the figure will have the same number of the copy.
	 * @throws IllegalArgumentException If f is null.
	 */
	public Rhombus(Figure f, boolean sameNumber)
	{
		super(f, sameNumber);
		
		borders = new LaTeXDrawRectangle(getPoint(0), getPoint(1), getPoint(2), getPoint(3), false);
		updateGravityCenter();
		shape = getInsideOutsideOrMiddleBorders();
		updateGravityCenter();
		isBordersMovable = true;
	}
	

	
	/**
	 * The constructor using four points
	 * @param pt1 The top left point of rectangle containing the rhombus
	 * @param pt2 The top right point of the rectangle containing the rhombus
	 * @param pt3 The bottom left point
	 * @param pt4 The bottom right point
	 */
	public Rhombus(LaTeXDrawPoint2D pt1, LaTeXDrawPoint2D pt2, 
			LaTeXDrawPoint2D pt3, LaTeXDrawPoint2D pt4, boolean increaseMeter)
	{
		super(pt1, pt2, pt3, pt4, increaseMeter);
		borders = new LaTeXDrawRectangle(pt1, pt2, pt3, pt4, false);
		shape = getInsideOutsideOrMiddleBorders();
		updateGravityCenter();
		isBordersMovable = true;
	}
	
	
	
	@Override
	public boolean addPoint(LaTeXDrawPoint2D pt)
	{
		if(pts==null) pts = new Vector<LaTeXDrawPoint2D>();
		
		if(pt!=null)
		{
			Delimitor d = new Delimitor(pt);
			d.setColorSet3();
			delimiters.add(d);
			pts.add(pt);
			
			return true;
		}
		
		return false;
	}

	
	
	
	
	@Override
	public void draw(Graphics2D g, Object antiAlias, Object rendering, Object alphaInter, Object colorRendering)
	{
		LaTeXDrawPoint2D NW = borders.getTheNWPoint(), SE = borders.getTheSEPoint();
		double cx = (NW.x+SE.x)/2., cy = (NW.y+SE.y)/2.;
		double c2x = Math.cos(rotationAngle)*cx - Math.sin(rotationAngle)*cy;
		double c2y = Math.sin(rotationAngle)*cx + Math.cos(rotationAngle)*cy;
		double c3x = Math.cos(-rotationAngle)*(cx-c2x) - Math.sin(-rotationAngle)*(cy-c2y);
		double c3y = Math.sin(-rotationAngle)*(cx-c2x) + Math.cos(-rotationAngle)*(cy-c2y);
		double dx=0, dy=0;
		boolean changeFillStyle = false;

		if(rotationAngle%(Math.PI*2)!=0)
		{		
			g.rotate(rotationAngle);
			g.translate(c3x,c3y);
		}
		
		if(hasShadow)
		{
			LaTeXDrawPoint2D cg = getGravityCenter();
			LaTeXDrawPoint2D shadowCg = (LaTeXDrawPoint2D)cg.clone();
			shadowCg.setLocation(cg.x+shadowSize, cg.y);
			shadowCg = Figure.rotatePoint(shadowCg, cg, shadowAngle);
			dx = shadowCg.x-cg.x;
			dy = cg.y-shadowCg.y;
		}
		
		Color formerCol = g.getColor();
		
		if(hasDoubleBoundary)
		{
			Shape s0 = shape;
			Shape s[] = getDbleBoundariesOutInOrMiddle(s0);
			Shape s1, s2, s3;

			if(bordersPosition.equals(PSTricksConstants.BORDERS_INSIDE))
			{
				s1 = s0;
				s2 = s[0];
				s3 = s[1];
			}
			else
				if(bordersPosition.equals(PSTricksConstants.BORDERS_MIDDLE))
				{
					s1 = s[0];
					s2 = s0;
					s3 = s[1];
				}
				else
				{
					s1 = s[0];
					s2 = s[1];
					s3 = s0;
				}

			if(lineStyle.equals(PSTricksConstants.LINE_NONE_STYLE))
			{
				if(hasShadow)
				{
					g.setStroke(new BasicStroke(thickness, BasicStroke.CAP_BUTT, BasicStroke.JOIN_MITER));
					g.translate(dx, dy);
					g.setColor(shadowColor);
					g.fill(s1);
					g.draw(s1);
					g.translate(-dx, -dy);
					
					if(!isFilled)
					{
						changeFillStyle = true;
						isFilled = true;
					}
				}
				
				g.setColor(doubleColor);
				g.setStroke(new BasicStroke((float)(doubleSep + thickness), BasicStroke.CAP_BUTT, BasicStroke.JOIN_MITER));
				g.draw(s2);
				fillFigure(g, antiAlias, rendering, alphaInter, colorRendering, s3);
				g.setColor(linesColor);
				g.setStroke(new BasicStroke(thickness, BasicStroke.CAP_BUTT, BasicStroke.JOIN_MITER));
				g.draw(s1);
				g.draw(s3);
			}
			else
			{
				if(hasShadow)
				{
					g.setStroke(new BasicStroke(thickness, BasicStroke.CAP_BUTT, BasicStroke.JOIN_MITER));
					g.translate(dx, dy);
					g.setColor(shadowColor);
					g.fill(s1);
					g.draw(s1);
					g.translate(-dx, -dy);
					
					g.setColor(interiorColor);
					g.setStroke(new BasicStroke((float)(thickness*2+doubleSep), BasicStroke.CAP_BUTT, BasicStroke.JOIN_MITER));
					g.draw(s2);
					
					if(!isFilled)
					{
						changeFillStyle = true;
						isFilled = true;
					}
				}
				
				if(lineStyle.equals(PSTricksConstants.LINE_DOTTED_STYLE))
					g.setStroke(new BasicStroke((float)(thickness*2+doubleSep), BasicStroke.CAP_ROUND,
							BasicStroke.JOIN_MITER, 1.f, new float[] { 0, (float)(thickness*2+doubleSep + dotSep) }, 0));
				else
					g.setStroke(new BasicStroke((float)(thickness*2+doubleSep),
							BasicStroke.CAP_BUTT, BasicStroke.JOIN_MITER, 1.f,
							new float[] { blackDashLength, whiteDashLength }, 0));
					
				fillFigure(g, antiAlias, rendering, alphaInter, colorRendering, s2);
				g.setColor(linesColor);
				g.draw(s2);
				g.setStroke(new BasicStroke((float)doubleSep, BasicStroke.CAP_BUTT, BasicStroke.JOIN_MITER));
				g.setColor(doubleColor);
				g.draw(s2);
			}				
		}
		else
		{
			if(hasShadow)
			{
				g.setStroke(new BasicStroke(thickness, BasicStroke.CAP_BUTT, BasicStroke.JOIN_MITER));
				g.translate(dx, dy);
				g.setColor(shadowColor);
				g.fill(shape);
				g.draw(shape);
				g.translate(-dx, -dy);
				
				if(!isFilled)
				{
					changeFillStyle = true;
					isFilled = true;
				}
				g.setColor(interiorColor);
				g.draw(shape);
			}
			
			if(lineStyle.equals(PSTricksConstants.LINE_NONE_STYLE))
				g.setStroke(new BasicStroke(thickness, BasicStroke.CAP_BUTT, BasicStroke.JOIN_MITER));
			else
				if(lineStyle.equals(PSTricksConstants.LINE_DOTTED_STYLE) )
				{
					g.setStroke(new BasicStroke(thickness, BasicStroke.CAP_ROUND,
							BasicStroke.JOIN_MITER, 1.f, new float[] { 0, thickness + dotSep }, 0));
				}
				else
					g.setStroke(new BasicStroke(thickness, BasicStroke.CAP_BUTT, BasicStroke.JOIN_MITER, 1.f,
							new float[] { blackDashLength, whiteDashLength }, 0));
			
			fillFigure(g, antiAlias, rendering, alphaInter, colorRendering,shape);
			g.setColor(linesColor);
			g.draw(shape);
		}

		g.setColor(formerCol);
		
		if(changeFillStyle) isFilled = false;
		
		if(rotationAngle%(Math.PI*2)!=0)
		{
			g.translate(-c3x, -c3y);
			g.rotate(-rotationAngle);
		}
		
		if(isSelected)
			borders.draw(g, false, antiAlias, rendering, alphaInter, colorRendering);
	}
	
	
	
	

	/**
	 * Allows to get the south-east point by taking into account
	 * the angle of rotation.
	 * @return The south-east point of the rotated rectangle
	 */
	@Override
	public LaTeXDrawPoint2D getTheSERotatedPoint()
	{
		if(borders!=null)
			return borders.getTheSERotatedPoint();
		
		return null;
	}
	
	
	
	
	
	/**
	 * Allows to get the north-west point by taking into account
	 * the angle of rotation.
	 * @return The north-west point of the rotated rectangle
	 */
	@Override
	public LaTeXDrawPoint2D getTheNWRotatedPoint()
	{
		if(borders!=null)
			return borders.getTheNWRotatedPoint();
		
		return null;
	}
	
	
	
	@Override
	public synchronized void setBordersPosition(String doubleLinePosition)
	{
		super.setBordersPosition(doubleLinePosition);
		shape = getInsideOutsideOrMiddleBorders();
	}
	
	
	
	
	@Override
	public synchronized void setLastPoint(LaTeXDrawPoint2D pt)
	{
		borders.setLastPoint(pt);
		updateGravityCenter();
		shape = getInsideOutsideOrMiddleBorders();
	}
	
	
	@Override
	public synchronized void setFirstPoint(LaTeXDrawPoint2D pt)
	{
		borders.setFirstPoint(pt);
		updateGravityCenter();
		shape = getInsideOutsideOrMiddleBorders();
	}
	
	
	
	@Override
	public synchronized void setRotationAngle(double theta)
	{
		if(!Double.isNaN(theta) && !Double.isInfinite(theta))
		{
			rotationAngle = theta%(Math.PI*2);
			
			if(borders!=null)
				borders.setRotationAngle(rotationAngle);
		}
	}
	
	
	

	@Override
	public void onDragged(Point formerPt, Point newPt) throws Exception
	{
		if(formerPt.equals(newPt)) return;
		
		borders.onDragged(formerPt, newPt);
		rotationAngle = borders.getRotationAngle();//update the angle(when rotation)
		updateGravityCenter();//update centre of gravity when reshaping
		shape = getInsideOutsideOrMiddleBorders();
	}
	
	
	
	
	@Override
	public synchronized String getCodePSTricks(DrawBorders drawBorders, float ppc)
	{
		LaTeXDrawPoint2D d = drawBorders.getOriginPoint();
		LaTeXDrawPoint2D pt1 = borders.getPoint(0), pt2 = borders.getPoint(1);
		LaTeXDrawPoint2D pt3 = borders.getPoint(2);
		double Xcenter = (pt1.x+pt2.x)/2. - d.x, Ycenter = d.y - (pt1.y+pt3.y)/2.;
		String add="", fillType=""; //$NON-NLS-1$ //$NON-NLS-2$
		boolean isFilledWasChanged = false;
		
		if(hasShadow)
		{
			fillType+=",shadow=true";//$NON-NLS-1$
			if((Math.toDegrees(shadowAngle)-Math.toDegrees(rotationAngle))!=PSTricksConstants.DEFAULT_SHADOW_ANGLE)
				fillType+=",shadowangle="+(float)(Math.toDegrees(shadowAngle)-Math.toDegrees(rotationAngle));//$NON-NLS-1$
			
			if(((float)shadowSize)!=((float)DEFAULT_SHADOW_SIZE))
				fillType+=",shadowsize="+(float)(shadowSize/PPC);//$NON-NLS-1$
			
			if(!shadowColor.equals(PSTricksConstants.DEFAULT_SHADOW_COLOR))
			{
				String name = DviPsColors.getColourName(shadowColor);
				if(name==null)
				{
					name = "color"+number+'e';//$NON-NLS-1$
					DviPsColors.addUserColour(shadowColor, name); 
				}
				add += ",shadowcolor=" + name; //$NON-NLS-1$
			}
			if(!isFilled)
			{
				isFilled = true;
				isFilledWasChanged = true;
			}
		}
		
		String str = getPSTricksCodeFilling(ppc);
		if(str.length()>0) fillType=fillType+','+str;
		
		str = getPSTricksCodeLine(ppc);
		if(str.length()>0) add=add+','+str;
		
		add+=",dimen="+bordersPosition;  //$NON-NLS-1$
		
		if(hasDoubleBoundary)
		{
			add+=",doubleline=true,doublesep="+(float)(doubleSep/ppc); //$NON-NLS-1$
			
			if(doubleColor!=PSTricksConstants.DEFAULT_DOUBLE_COLOR)
			{
				String name = DviPsColors.getColourName(doubleColor);
				if(name==null)
				{
					name = "color"+number+'d';//$NON-NLS-1$
					DviPsColors.addUserColour(doubleColor, name); 
				}
				add+= ",doublecolor="+name; //$NON-NLS-1$
			}
		}
		
		if(rotationAngle%(Math.PI*2)!=0)
			add+=",gangle="+(float)Math.toDegrees(-rotationAngle);//$NON-NLS-1$
		
		if(Math.abs(Xcenter) < 0.001) Xcenter = 0;
		if(Math.abs(Ycenter) < 0.001) Ycenter = 0;
		if(isFilledWasChanged) isFilled = false;
		
		return "\\psdiamond[linewidth=" + (thickness/ppc) +  //$NON-NLS-1$
		add + fillType + "](" + (float)(Xcenter/ppc) + ',' +  //$NON-NLS-1$
		(float)(Ycenter/ppc) + ")("  //$NON-NLS-1$
			+ (float)((Math.abs(pt1.x-pt2.x)/2.)/ppc) + ',' + (float)((Math.abs(pt1.y-pt3.y)/2.)/ppc) + ')';
	}
	
	

	
	
	@Override
	public boolean isIn(LaTeXDrawPoint2D p) 
	{
		LaTeXDrawPoint2D pt = rotateInvertPoint(p);
		
		if(isSelected && (borders.dNE.isIn(pt) || borders.dNW.isIn(pt) || borders.dSE.isIn(pt) || borders.dSW.isIn(pt) ||
				borders.dS.isIn(pt)  || borders.dN.isIn(pt) || borders.dE.isIn(pt)  || borders.dW.isIn(pt)))
			return true;
		
		GeneralPath gp;
		
		if(hasDoubleBoundary)
		{
			if(bordersPosition.equals(PSTricksConstants.BORDERS_INSIDE))
				 gp = getBorders(0, true);
			else if(bordersPosition.equals(PSTricksConstants.BORDERS_MIDDLE))
				 gp = getBorders(thickness*2+doubleSep, false);
			else gp = getBorders((thickness*2+doubleSep)*2, false);

			if(!gp.contains(pt))
				return false;

			if(isFilled || hasShadow || hasGradient())
				return true;
			
			if(bordersPosition.equals(PSTricksConstants.BORDERS_INSIDE))
				gp = getBorders((thickness*2+doubleSep)*2, true);
			else if(bordersPosition.equals(PSTricksConstants.BORDERS_MIDDLE))
				gp = getBorders(doubleSep+2*thickness, true);
			else gp = getBorders(0, true);
		}
		else
		{
			if(bordersPosition.equals(PSTricksConstants.BORDERS_INSIDE))
				 gp = getBorders(0, true);
			else if(bordersPosition.equals(PSTricksConstants.BORDERS_MIDDLE))
				 gp = getBorders(thickness, false);
			else gp = getBorders(thickness*2, false);
		
			if(!gp.contains(pt))
				return false;

			if(isFilled || hasShadow || hasGradient())
				return true;
			
			if(bordersPosition.equals(PSTricksConstants.BORDERS_INSIDE))
				 gp = getBorders(thickness*2, true);
			else if(bordersPosition.equals(PSTricksConstants.BORDERS_MIDDLE))
				 gp = getBorders(thickness, true);
			else gp = getBorders(0, true);
		}
		
		return !gp.contains(pt);
	}
	
	
	

	
	@Override
	public Object clone() throws CloneNotSupportedException
	{
		Rhombus r = (Rhombus) super.clone();
		
		r.borders = new LaTeXDrawRectangle(r.getPoint(0), r.getPoint(1), r.getPoint(2), r.getPoint(3), false);
		r.setThickness(thickness);
		r.updateShape();
	
		return r;
	}

	
	
	
	
	@Override
	public Shape createShape2D() 
	{
		LaTeXDrawPoint2D NW = getTheNWPoint(), SE = getTheSEPoint();
		Shape area = createNonRotatedShape2D();

		if(rotationAngle % (Math.PI*2) != 0.)
		{
			double cx = (NW.x + SE.x) / 2., cy = (NW.y + SE.y) / 2.;
			double c2x = Math.cos(rotationAngle) * cx - Math.sin(rotationAngle)* cy;
			double c2y = Math.sin(rotationAngle) * cx + Math.cos(rotationAngle)* cy;

			AffineTransform at = AffineTransform.getTranslateInstance(cx - c2x, cy - c2y);
			at.rotate(rotationAngle);
			area = at.createTransformedShape(area);
		}

		return area;
	}

	
	
	
	
	@Override
	public Shape createNonRotatedShape2D() 
	{
		Shape area;
		Shape s = getInsideOutsideOrMiddleBorders();

		if(hasDoubleBoundary)
		{
			Shape[] s2 = getDbleBoundariesOutInOrMiddle(s);
			Shape min;
			Shape max;

			if(bordersPosition.equals(PSTricksConstants.BORDERS_INSIDE))
			{
				max = s;
				min = s2[1];
			}
			else
				if(bordersPosition.equals(PSTricksConstants.BORDERS_MIDDLE))
				{
					max = s2[0];
					min = s2[1];
				}
				else
				{
					max = s2[0];
					min = s;
				}

			area = new Area(max);
			((Area)area).exclusiveOr(new Area(min));
		}
		else
			area = s;
		
		return area;
	}

	
	
	@SuppressWarnings("unchecked")
	private void readObject(ObjectInputStream ois) throws IOException, ClassNotFoundException
	{
		canHaveShadow = true;
		interiorColor = (Color) ois.readObject();
		lineStyle = (String) ois.readObject();
		rotationAngle = ois.readDouble();
		thickness = ois.readFloat();
		isFilled = ois.readBoolean();
		isSelected = ois.readBoolean();
		isOnRotation = ois.readBoolean();
		linesColor = (Color) ois.readObject();
		blackDashLength = ois.readFloat();
		dotSep = ois.readFloat();
		whiteDashLength = ois.readFloat();
		pts = (Vector) ois.readObject();
		borders = (LaTeXDrawRectangle) ois.readObject();
		
		delimiters = new Vector();
		for(int i=0, size = pts.size();i<size; i++)
			delimiters.add(new Delimitor(pts.elementAt(i)));
			
		if(LaTeXDrawFrame.getVersionOfFile().compareTo("1.5")>=0)//$NON-NLS-1$
		{
			hasDoubleBoundary = ois.readBoolean();
			doubleColor = (Color)ois.readObject();
			doubleSep = ois.readDouble();
			bordersPosition = (String)ois.readObject();
			if(!(LaTeXDrawFrame.getVersionOfFile().compareTo("1.6")>=0)) //$NON-NLS-1$
				ois.readBoolean();
			hatchingAngle = ois.readDouble();
			hatchingColor = (Color)ois.readObject();
			hatchingStyle = (String)ois.readObject();
			hatchingWidth = ois.readFloat();
			
			if(LaTeXDrawFrame.getVersionOfFile().compareTo("1.6") < 0)//$NON-NLS-1$
			{
				if(hatchingStyle.equals(DECREPETED_FILL_CROSS))
					hatchingStyle = PSTricksConstants.TOKEN_FILL_CROSSHATCH;
				else if(hatchingStyle.equals(DECREPETED_FILL_HORIZ))
					hatchingStyle = PSTricksConstants.TOKEN_FILL_HLINES;
				else if(hatchingStyle.equals(DECREPETED_FILL_VERT))
					hatchingStyle = PSTricksConstants.TOKEN_FILL_VLINES;
				else if(hatchingStyle.equals(DECREPETED_FILL_NO))
					hatchingStyle = PSTricksConstants.TOKEN_FILL_NONE;
			}
		}
		else
		{
			hasDoubleBoundary  = DEFAULT_HAS_DOUBLE_BOUNDARY;
			doubleColor = DEFAULT_DOUBLE_COLOR;
			doubleSep   = DEFAULT_DOUBLESEP;
			bordersPosition = DEFAULT_BORDERS_POSITION;
			hatchingAngle = DEFAULT_HATCH_ANGLE;
			hatchingColor = DEFAULT_HATCH_COL;
			hatchingStyle = DEFAULT_HATCH_STYLE;
			hatchingWidth = DEFAULT_HATCH_WIDTH;
		}
		
		if(LaTeXDrawFrame.getVersionOfFile().compareTo("1.7")>=0) //$NON-NLS-1$
		{
			hasShadow 	= ois.readBoolean();
			shadowAngle = ois.readDouble();
			shadowSize	= ois.readDouble();
			shadowColor	= (Color)ois.readObject();
			gradientEndColor = (Color)ois.readObject();
			gradientStartColor = (Color)ois.readObject();
			gradientAngle = ois.readDouble();
			gradientMidPoint = ois.readDouble();
		}
		else
		{
			hasShadow 	= DEFAULT_SHADOW_HAS;
			shadowAngle	= DEFAULT_SHADOW_ANGLE;
			shadowSize	= DEFAULT_SHADOW_SIZE;
			shadowColor	= DEFAULT_SHADOW_COLOR;
			gradientEndColor = PSTricksConstants.DEFAULT_GRADIENT_END_COLOR;
			gradientStartColor = PSTricksConstants.DEFAULT_GRADIENT_START_COLOR;
			gradientAngle = DEFAULT_GRADIENT_ANGLE;
			gradientMidPoint = DEFAULT_GRADIENT_MID_POINT;
		}
		
		if(LaTeXDrawFrame.getVersionOfFile().compareTo("1.8")>=0) //$NON-NLS-1$
			hatchingSep = ois.readDouble();
		else
			hatchingSep = DEFAULT_HATCH_SEP;
		
		shape = getInsideOutsideOrMiddleBorders();
	}




	@Override
	protected GeneralPath getBorders(double gap, boolean into)
	{
		if(gap==0.)
			return getMiddleBorders();
		
		LaTeXDrawPoint2D pt1 = getPoint(0), pt2 = getPoint(-1);
		GeneralPath gp = new GeneralPath();
		
		if(pt1.x<pt2.x+thickness && pt1.x>pt2.x-thickness)
		{
			gp.moveTo((float)(pt1.x+pt2.x)/2f, (float)pt1.y);
			gp.lineTo((float)(pt1.x+pt2.x)/2f, (float)pt2.y);
		}
		else
		{
			if(!into) 
				gap*=-1;
			
			LaTeXDrawPoint2D gc = getGravityCenter();
			LaTeXDrawPoint2D p1 = new LaTeXDrawPoint2D((pt1.x+pt2.x)/2., pt1.y);
			LaTeXDrawPoint2D p2 = new LaTeXDrawPoint2D(pt2.x, (pt1.y+pt2.y)/2.);
			LaTeXDrawPoint2D p3 = new LaTeXDrawPoint2D((pt1.x+pt2.x)/2., pt2.y);
			double cornerGap1 = getCornerGap(gc, p1, p2, gap)/2.;
			double cornerGap2 = getCornerGap(gc, p2, p3, gap)/2.;
			
		    if(p2.x<p3.x)
		    	cornerGap2*=-1;
		    
		    if(p1.y>p2.y)
		    	cornerGap1*=-1;
		    
		    gp.moveTo((float)p1.x, (float)(pt1.y + cornerGap1));
		    gp.lineTo((float)(pt2.x - cornerGap2), (float)p2.y);
		    gp.lineTo((float)p1.x, (float)(pt2.y - cornerGap1));
		    gp.lineTo((float)(pt1.x + cornerGap2), (float)p2.y);
		    gp.closePath();
		}
		
		return gp;
	}
	
	
	

	public GeneralPath getInsideBorders()
	{
		return getBorders(thickness, true);
	}


	
	
	
	public GeneralPath getOutsideBorders()
	{
		return getBorders(thickness, false);
	}

	
	

	
	public GeneralPath getMiddleBorders()
	{
		LaTeXDrawPoint2D pt1 = getPoint(0), pt2 = getPoint(-1);
		GeneralPath path = new GeneralPath();
		
		path.moveTo((float)(pt1.x+pt2.x)/2f, (float)pt2.y);
		path.lineTo((float)pt1.x, (float)(pt2.y+pt1.y)/2f);
		path.lineTo((float)(pt1.x+pt2.x)/2f, (float)pt1.y);
		path.lineTo((float)pt2.x, (float)(pt2.y+pt1.y)/2f);
		path.closePath();

		return path;
	}


	
	
	@Override
	public GeneralPath getInsideOutsideOrMiddleBorders()
	{
		GeneralPath path;

		if (bordersPosition.equals(PSTricksConstants.BORDERS_INSIDE))
			path = getInsideBorders();
		else
			if (bordersPosition.equals(PSTricksConstants.BORDERS_OUTSIDE))
				path = getOutsideBorders();
			else
				path = getMiddleBorders();

		return path;
	}
	
	
	
	
	
	
	@Override
	public GeneralPath[] getDbleBoundariesOutside(Shape classicBord)
	{
		LaTeXDrawPoint2D pt1 = getPoint(0), pt2 = getPoint(-1);
		GeneralPath[] sx = new GeneralPath[2];
		
		if(pt1.x<pt2.x+thickness && pt1.x>pt2.x-thickness)
		{
			sx[0] = new GeneralPath(classicBord);
			sx[1] = new GeneralPath(classicBord);
		}
		else
		{
			sx[0] = getBorders(3*thickness+2*doubleSep, false);
			sx[1] = getBorders(2*thickness+doubleSep, false);
		}
		
		return sx;
	}


	
	

	@Override
	public GeneralPath[] getDbleBoundariesInside(Shape classicBord)
	{
		LaTeXDrawPoint2D pt1 = getPoint(0), pt2 = getPoint(-1);
		GeneralPath[] sx = new GeneralPath[2];
		
		if(pt1.x<pt2.x+thickness && pt1.x>pt2.x-thickness)
		{
			sx[0] = new GeneralPath(classicBord);
			sx[1] = new GeneralPath(classicBord);
		}
		else
		{
			sx[1] = getBorders(3*thickness+2*doubleSep, true);
			sx[0] = getBorders(2*thickness+doubleSep, true);
		}
		
		return sx;
	}


	
	

	@Override
	public GeneralPath[] getDbleBoundariesMiddle(Shape classicBord)
	{
		LaTeXDrawPoint2D pt1 = getPoint(0), pt2 = getPoint(-1);
		GeneralPath[] sx = new GeneralPath[2];
		
		if(pt1.x<pt2.x+thickness && pt1.x>pt2.x-thickness)
		{
			sx[0] = new GeneralPath(classicBord);
			sx[1] = new GeneralPath(classicBord);
		}
		else
		{
			sx[0] = getBorders(thickness+doubleSep, false);
			sx[1] = getBorders(thickness+doubleSep, true);
		}
		
		return sx;
	}
	
	
	
	@Override
	public GeneralPath[] getDbleBoundariesOutInOrMiddle(Shape classicBord)
	{
		GeneralPath[] path;

		if(bordersPosition.equals(PSTricksConstants.BORDERS_INSIDE))
			path = getDbleBoundariesInside(classicBord);
		else
			if(bordersPosition.equals(PSTricksConstants.BORDERS_OUTSIDE))
				path = getDbleBoundariesOutside(classicBord);
			else
				path = getDbleBoundariesMiddle(classicBord);

		return path;
	}

	
	
	@Override
	public void shift(double shiftX,double shiftY) 
	{
		if(shiftX==0 && shiftY==0) return ;

		borders.shift(shiftX, shiftY);
		updateShape();
	}

	
	
	
	@Override
	public synchronized void setThickness(float val)
	{
		if(!Double.isInfinite(val) && !Double.isNaN(val) && val>0)
		{
			thickness = val;
			shape = getInsideOutsideOrMiddleBorders();
			borders.setThickness(thickness);
		}
	}
	
	
	
	@Override
	@Deprecated
	public void updateBorders()
	{
		/*
		 * Update the borders change the points of the rhombus. 
		 */
	}
	
	
	
	
	@Override
	public Shape createShadowShape()
	{
		if(!canHaveShadow || !hasShadow) return shape;
		
		Shape shadowS;
		
		if(hasDoubleBoundary)
		{
			if(bordersPosition.equals(PSTricksConstants.BORDERS_INSIDE))
				 shadowS = getBorders(0, true);
			else if(bordersPosition.equals(PSTricksConstants.BORDERS_MIDDLE))
				 shadowS = getBorders(thickness*2+doubleSep, false);
			else shadowS = getBorders((thickness*2+doubleSep)*2, false);
		}
		else
		{
			if(bordersPosition.equals(PSTricksConstants.BORDERS_INSIDE))
				 shadowS = getBorders(0, false);
			else if(bordersPosition.equals(PSTricksConstants.BORDERS_MIDDLE))
				 shadowS = getBorders(thickness, false);
			else shadowS = getBorders(thickness*2, false);
		}
		
		double dx=0, dy=0;
		LaTeXDrawPoint2D cg = getGravityCenter();
		LaTeXDrawPoint2D shadowCg = (LaTeXDrawPoint2D)cg.clone();
		AffineTransform at = new AffineTransform();
		
		shadowCg.setLocation(cg.x+shadowSize, cg.y);
		shadowCg = Figure.rotatePoint(shadowCg, cg, shadowAngle);
		dx = shadowCg.x-cg.x;
		dy = cg.y-shadowCg.y;
		at.translate(dx, dy);
		
		return at.createTransformedShape(shadowS);
	}
	
	
	
	
	@Override
	public String getPSTricksCodeFilling(float ppc)
	{
		String str="fillstyle=";//$NON-NLS-1$

		if(hatchingStyle.equals(PSTricksConstants.TOKEN_FILL_NONE))
		{
			if(isFilled)
				str += "solid"; //$NON-NLS-1$
			else str = "";//$NON-NLS-1$
		}
		else
			if(hatchingStyle.equals(PSTricksConstants.TOKEN_FILL_GRADIENT))
			{
				str+= "gradient,gradlines=2000";//$NON-NLS-1$
				
				if(!gradientStartColor.equals(PSTricksConstants.DEFAULT_GRADIENT_START_COLOR))
				{
					String name = DviPsColors.getColourName(gradientStartColor);
					if(name==null)
					{
						name = "color"+number+'g';//$NON-NLS-1$
						DviPsColors.addUserColour(gradientStartColor, name); 
					}
					str += ",gradbegin=" + name; //$NON-NLS-1$
				}
				
				if(!gradientEndColor.equals(PSTricksConstants.DEFAULT_GRADIENT_END_COLOR))
				{
					String name = DviPsColors.getColourName(gradientEndColor);
					if(name==null)
					{
						name = "color"+number+'f';//$NON-NLS-1$
						DviPsColors.addUserColour(gradientEndColor, name); 
					}
					str += ",gradend=" + name; //$NON-NLS-1$
				}
				
				if(gradientMidPoint!=PSTricksConstants.DEFAULT_GRADIENT_MID_POINT)
					str+=",gradmidpoint="+(float)gradientMidPoint;//$NON-NLS-1$
				
				if(((toDegrees(gradientAngle)-Math.toDegrees(rotationAngle))%360.)!=PSTricksConstants.DEFAULT_GRADIENT_ANGLE)
					str+=",gradangle="+(float)((toDegrees(gradientAngle)-Math.toDegrees(rotationAngle))%360.);//$NON-NLS-1$
			}
			else
			{
				if(hatchingStyle.equals(PSTricksConstants.TOKEN_FILL_CROSSHATCH))
					str += "crosshatch"; //$NON-NLS-1$
				else
					if (hatchingStyle.equals(PSTricksConstants.TOKEN_FILL_HLINES))
						str += "hlines"; //$NON-NLS-1$
					else
						str += "vlines"; //$NON-NLS-1$
	
				if(isFilled)
					str += "*"; //$NON-NLS-1$
	
				str += ",hatchwidth=" + (hatchingWidth / PPC) + ",hatchangle=" + //$NON-NLS-1$ //$NON-NLS-2$
						(float)((-Math.toDegrees(rotationAngle))%360.);
				//TODO hatching angle
				if(!hatchingColor.equals(PSTricksConstants.DEFAULT_HATCHING_COLOR))
				{
					String name = DviPsColors.getColourName(hatchingColor);
					if(name==null)
					{
						name = "color"+number+'c';//$NON-NLS-1$
						DviPsColors.addUserColour(hatchingColor, name); 
					}
					str += ",hatchcolor=" + name; //$NON-NLS-1$
				}
			}
		
		if(isFilled)
		{
			if (!interiorColor.equals(PSTricksConstants.DEFAULT_INTERIOR_COLOR))
			{
				String name = DviPsColors.getColourName(interiorColor);
				if(name==null)
				{
					name = "color"+number+'b';//$NON-NLS-1$
					DviPsColors.addUserColour(interiorColor, name); 
				}
				str += ",fillcolor=" + name; //$NON-NLS-1$
			}
		}
		
		return str;
	}




	@Override
	@Deprecated
	public void updateBorders(LaTeXDrawPoint2D pt)
	{
		// Nothing must be done.
	}
	
	
	
	@Override
	public void mirrorHorizontal(LaTeXDrawPoint2D origin)
	{
		borders.mirrorHorizontal(origin);
		updateShape();
	}



	@Override
	public void mirrorVertical(LaTeXDrawPoint2D origin)
	{
		borders.mirrorVertical(origin);
		updateShape();
	}
	
	
	
	@Override
	public synchronized LaTeXDrawPoint2D getLastPoint()
	{
		return borders.getLastPoint();
	}




	@Override
	public void updateToGrid(MagneticGrid grid)
	{
		borders.updateToGrid(grid);
		updateShape();
	}




	@Override
	public void rescaleX(double formerX, double newX, double percent, LaTeXDrawRectangle bound)
	{
		borders.rescaleX(formerX, newX, percent, bound);
		updateShape();
	}




	@Override
	public void rescaleY(double formerY, double newY, double percent, LaTeXDrawRectangle bound)
	{
		borders.rescaleY(formerY, newY, percent, bound);
		updateShape();
	}
	
	
	
	@Override
	public int hashCode()
	{
		return (int)(super.hashCode()/1.5);
	}
}
