package latexDraw.figures;

import java.awt.*;
import java.awt.geom.AffineTransform;
import java.awt.geom.GeneralPath;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.util.Vector;

import javax.swing.JLabel;

import latexDraw.psTricks.DviPsColors;
import latexDraw.psTricks.PSTricksConstants;
import latexDraw.ui.LaTeXDrawFrame;
import latexDraw.ui.components.Delimitor;
import latexDraw.ui.components.LaTeXDrawComboBox;
import latexDraw.ui.components.LabelListCellRenderer;
import latexDraw.ui.components.MagneticGrid;
import latexDraw.util.LaTeXDrawNumber;
import latexDraw.util.LaTeXDrawPoint2D;
import latexDraw.util.LaTeXDrawResources;


/** 
 * This class defines joined points.<br>
 *<br>
 * This file is part of LaTeXDraw<br>
 * Copyright (c) 2005-2008 Arnaud BLOUIN<br>
 *<br>
 *  LaTeXDraw is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.<br>
 *<br>
 *  LaTeXDraw is distributed without any warranty; without even the 
 *  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 *  PURPOSE. See the GNU General Public License for more details.<br>
 *<br>
 * 10/25/06<br>
 * @author Arnaud BLOUIN<br>
 * @version 2.0.0<br>
 */
public class AkinPoints extends LaTeXDrawPolygon
{
	private static final long serialVersionUID = 1L;

	/** The default value of interval. @since 1.9*/
	public static final int DEFAULT_INTERVAL = 5;
	
	/** The akin points are drawn as curves. @since 1.9*/
	public static final short TYPE_CURVES = 1;
	
	/** The akin points are drawn as lines. @since 1.9*/
	public static final short TYPE_LINES = 0;
	
	/** The type by default of the akin points. @since 1.9*/
	public static final short DEFAULT_TYPE = TYPE_CURVES;
	
	public static final String LABEL_TYPE_CHOICE = "typeChoice";//$NON-NLS-1$
	
	public static final boolean DEFAULT_OPEN = true;
	
	
	/** Define if the shape is closed or not. @since 1.9 */
	protected boolean open;
	
	/** Define the interval between the each point to use. @since 1.9*/
	protected int interval;
	
	/** Define the type of the akin points: lines or curves. @since 1.9*/
	protected int type;
	
	
	
	/**
	 * The constructor by default
	 */
	public AkinPoints(boolean increaseMeter)
	{
		this(new LaTeXDrawPoint2D(), increaseMeter);
	}
	
	
	
	
	/**
	 * The constructor using one point.
	 * @param pt The first point.
	 */
	public AkinPoints(LaTeXDrawPoint2D pt, boolean increaseMeter)
	{
		super(increaseMeter);
	
		open 		= DEFAULT_OPEN;
		type 		= DEFAULT_TYPE;
		interval 	= DEFAULT_INTERVAL;
		isBordersMovable 	 = false;
		isDashableOrDotable  = true;
		isDoubleBoundaryable = false;
		canBeFilled 		 = true;
		borders = new LaTeXDrawRectangle((LaTeXDrawPoint2D)pt.clone(), (LaTeXDrawPoint2D)pt.clone(), false);
		addPoint(pt);
		updateShape();
	}
	

	/**
	 * Create a list of choice in order to select the type of shape we want.
	 * @return The created list.
	 * @since 1.9
	 */
	public static LaTeXDrawComboBox createTypeChoice()
	{
		LaTeXDrawComboBox typeChoice = new LaTeXDrawComboBox();
		typeChoice.setRenderer(new LabelListCellRenderer());
		typeChoice.setName(LABEL_TYPE_CHOICE);
		typeChoice.setActionCommand(LABEL_TYPE_CHOICE);
		
		JLabel l = new JLabel(String.valueOf(TYPE_CURVES));
		l.setIcon(LaTeXDrawResources.curvesFreehandIcon);
		typeChoice.addItem(l);
		l = new JLabel(String.valueOf(TYPE_LINES));
		l.setIcon(LaTeXDrawResources.linesFreehandIcon);
		typeChoice.addItem(l);
		
		return typeChoice;
	}
	
	
	
	
	
	/**
	 * Allows to add a point to the polygon
	 * @param pt The point to be added
	 */
	@Override
	public boolean addPoint(LaTeXDrawPoint2D pt)
	{
		boolean add = true;
		
		if(!pts.isEmpty())
		{ // The problem with akin points, is that some akin points
			// can be equal and consequently, the code becomes heavier;
			// so we look the last point and of it's equal to the new point
			// we don't insert this new point.
			LaTeXDrawPoint2D  p = pts.lastElement();
			if(((int)p.x)==((int)pt.x) && ((int)p.y)==((int)pt.y))
				add = false;
		}
		
		if(add)
		{
			pts.add(pt);
			if(borders!=null)
				updateBorders();
			
			updateShape();
			
			return true;
		}
		return false;
	}

	
	
	@Override
	public boolean addPointAt(LaTeXDrawPoint2D pt, int id)
	{
		if(id == -1)
			id = pts.size()-1;
		
		if(id<0 || id>pts.size())
			throw new IllegalArgumentException(String.valueOf(id));
		
		if(pt!=null)
		{
			if(pts.isEmpty())
				pts.add(pt);
			else
				pts.add(id, pt);
			
			if(borders!=null)
				updateBorders(pt);
			
			updateShape();
			
			return true;
		}
		return false;
	}

	
	
	
	@Override
	public void draw(Graphics2D g, Object antiAlias, Object rendering, Object alphaInter, Object colorRendering)
	{
		if(pts==null || pts.isEmpty()) return ;
		
		if(lineStyle.equals(PSTricksConstants.LINE_NONE_STYLE))
			g.setStroke(new BasicStroke(thickness, BasicStroke.CAP_BUTT, BasicStroke.JOIN_MITER));
		else 
		if(lineStyle.equals(PSTricksConstants.LINE_DOTTED_STYLE))
			g.setStroke(new BasicStroke(thickness, BasicStroke.CAP_ROUND, BasicStroke.JOIN_MITER,
					1.f, new float[]{0,thickness+dotSep}, 0));
		else
		if(lineStyle.equals(PSTricksConstants.LINE_DASHED_STYLE))
			g.setStroke(new BasicStroke(thickness, BasicStroke.CAP_BUTT, BasicStroke.JOIN_MITER,
					1.f, new float[]{blackDashLength, whiteDashLength}, 0));
		
		Color formerCol = g.getColor();
		
		if(hasShadow)
		{
			Stroke formerS = g.getStroke();
			double dx=0, dy=0;
			LaTeXDrawPoint2D cg = getGravityCenter();
			LaTeXDrawPoint2D shadowCg = (LaTeXDrawPoint2D)cg.clone();
			
			shadowCg.setLocation(cg.x+shadowSize, cg.y);
			shadowCg = Figure.rotatePoint(shadowCg, cg, shadowAngle);
			dx = shadowCg.x-cg.x;
			dy = cg.y-shadowCg.y;
			g.setStroke(new BasicStroke(thickness, BasicStroke.CAP_BUTT, BasicStroke.JOIN_MITER));
			g.translate(dx, dy);
			g.setColor(shadowColor);
			g.draw(shape);
			
			if(isFilled() || hasGradient() || isHatched())
				g.fill(shape);
			
			g.translate(-dx, -dy);
			g.setColor(interiorColor);
			g.draw(shape);
			g.setStroke(formerS);
		}
		
		fillFigure(g, antiAlias, rendering, alphaInter, colorRendering, shape);
		
		g.setColor(linesColor);
        g.draw(shape);
		g.setColor(formerCol);
		
		if(isSelected && borders!=null)
			borders.draw(g, false, antiAlias, rendering, alphaInter, colorRendering);
	}

	
	
	
	
	@Override
	public boolean isIn(LaTeXDrawPoint2D pt)
	{
		if(isSelected && (borders.dNE.isIn(pt) || borders.dNW.isIn(pt) || 
			borders.dSE.isIn(pt) || borders.dSW.isIn(pt) || borders.dS.isIn(pt)  || 
			borders.dN.isIn(pt) || borders.dE.isIn(pt)  || borders.dW.isIn(pt)))
			return true;
		
		Stroke wideline = new BasicStroke(thickness, BasicStroke.CAP_BUTT, BasicStroke.JOIN_MITER);
		Shape s = wideline.createStrokedShape(shape);

		if(s.contains(pt))
			return true;

		if(isFilled() || hasGradient() || isHatched())
			if(shape.contains(pt))
				return true;
		
		return false;
	}

	
	
	
	@Override
	public void onClick(Point pt)
	{
		isSelected = true;
		if(borders!=null)
			borders.onClick(pt);
	}

	
	
	
	
	@Override
	public void setLastPoint(double x, double y) 
	{
		/* We cannot set the last point of akin points. */
	}



	
	@Override
	public String getCodePSTricks(DrawBorders drawBorders, float ppc)
	{
		if(getNbPoints()<2) return null;
		
		LaTeXDrawPoint2D d = drawBorders.getOriginPoint();
		String coord = "", add="", fillCode=""; //$NON-NLS-1$//$NON-NLS-2$//$NON-NLS-3$
		int i, size = getNbPoints();
		double threshold = 0.001;
		
		if(hasShadow)
		{
			add+=",shadow=true";//$NON-NLS-1$
			if(Math.toDegrees(shadowAngle)!=PSTricksConstants.DEFAULT_SHADOW_ANGLE)
				add+=",shadowangle="+(float)Math.toDegrees(shadowAngle);//$NON-NLS-1$
			
			if(((float)shadowSize) != ((float)DEFAULT_SHADOW_SIZE))
				add+=",shadowsize="+(float)(shadowSize/PPC);//$NON-NLS-1$
			
			if(!shadowColor.equals(PSTricksConstants.DEFAULT_SHADOW_COLOR))
			{
				String name = DviPsColors.getColourName(shadowColor);
				if(name==null)
				{
					name = "color"+number+'e';//$NON-NLS-1$
					DviPsColors.addUserColour(shadowColor, name); 
				}
				add += ",shadowcolor=" + name; //$NON-NLS-1$
			}
		}
			
		if(!linesColor.equals(PSTricksConstants.DEFAULT_LINE_COLOR))
		{
			String name = DviPsColors.getColourName(linesColor);
			if(name==null)
			{
				name = "color"+number;//$NON-NLS-1$
				DviPsColors.addUserColour(linesColor, "color"+number); //$NON-NLS-1$
			}
			add += ",linecolor="+name; //$NON-NLS-1$
		}
		
		String str = getPSTricksCodeFilling(ppc);
		if(str.length()>0) 
			fillCode=','+str;
		
		if(lineStyle.equals(PSTricksConstants.LINE_DOTTED_STYLE))
			add += ",linestyle="+lineStyle+",dotsep="+ //$NON-NLS-1$ //$NON-NLS-2$
			(dotSep/ppc)+ "cm";	//$NON-NLS-1$
		else
			if(lineStyle.equals(PSTricksConstants.LINE_DASHED_STYLE))
				add += ",linestyle="+lineStyle+",dash=" + //$NON-NLS-1$ //$NON-NLS-2$
				(blackDashLength/ppc) + "cm "+ //$NON-NLS-1$
				(whiteDashLength/ppc) + "cm";//$NON-NLS-1$
		
		if(type==TYPE_CURVES)
		{
			float prevx=0;
			float prevy=0;
			float curx = (float)pts.firstElement().x;
			float cury = (float)pts.firstElement().y;
	        float midx=0;
	        float midy=0;
			
	        coord = "\\moveto(" +LaTeXDrawNumber.getCutNumber((float)((curx-d.x)/ppc), threshold)+","+//$NON-NLS-1$//$NON-NLS-2$
	        		LaTeXDrawNumber.getCutNumber((float)((d.y-cury)/ppc), threshold)+")\n";//$NON-NLS-1$
	    	
	        if(pts.size()>interval)
	        {
	            prevx = curx;
	            prevy = cury;
	            curx = (float)pts.elementAt(interval).x; 
	            cury = (float)pts.elementAt(interval).y;
	            midx = (curx + prevx) / 2.0f;
	            midy = (cury + prevy) / 2.0f;
				
	            coord += "\\lineto(" +LaTeXDrawNumber.getCutNumber((float)((midx-d.x)/ppc), threshold)+","+//$NON-NLS-1$//$NON-NLS-2$
	            		LaTeXDrawNumber.getCutNumber((float)((d.y-midy)/ppc), threshold)+")\n";//$NON-NLS-1$
	        }
	        
	        for(i=interval*2; i<size; i+=interval) 
	        {
				float x1 = (midx + curx) / 2.0f;
				float y1 = (midy + cury) / 2.0f;
				prevx = curx;
				prevy = cury;
				curx = (float)pts.elementAt(i).x; 
				cury = (float)pts.elementAt(i).y;
				midx = (curx + prevx) / 2.0f;
				midy = (cury + prevy) / 2.0f;
				float x2 = (prevx + midx) / 2.0f;
				float y2 = (prevy + midy) / 2.0f;
				
	            coord += "\\curveto(" +LaTeXDrawNumber.getCutNumber((float)((x1-d.x)/ppc), threshold)+","+//$NON-NLS-1$//$NON-NLS-2$
	            		LaTeXDrawNumber.getCutNumber((float)((d.y-y1)/ppc), threshold)+")(" +  //$NON-NLS-1$
	            		LaTeXDrawNumber.getCutNumber((float)((x2-d.x)/ppc), threshold)+","+//$NON-NLS-1$
        				LaTeXDrawNumber.getCutNumber((float)((d.y-y2)/ppc), threshold)+")(" +  //$NON-NLS-1$
        				LaTeXDrawNumber.getCutNumber((float)((midx-d.x)/ppc), threshold)+","+//$NON-NLS-1$
						LaTeXDrawNumber.getCutNumber((float)((d.y-midy)/ppc), threshold)+")\n";//$NON-NLS-1$
	        }
	        
	        if((i-interval+1)<size)
	        {
	        	float x1 = (midx + curx) / 2.0f;
	        	float y1 = (midy + cury) / 2.0f;
	            prevx = curx;
	            prevy = cury;
	            curx = (float)pts.lastElement().x; 
	            cury = (float)pts.lastElement().y;
	            midx = (curx + prevx) / 2.0f;
	            midy = (cury + prevy) / 2.0f;
	            float x2 = (prevx + midx) / 2.0f;
	            float y2 = (prevy + midy) / 2.0f;
				
	            coord += "\\curveto(" + //$NON-NLS-1$
	            LaTeXDrawNumber.getCutNumber((float)((x1-d.x)/ppc), threshold)+","+//$NON-NLS-1$
	            LaTeXDrawNumber.getCutNumber((float)((d.y-y1)/ppc), threshold)+")(" +  //$NON-NLS-1$
				LaTeXDrawNumber.getCutNumber((float)((x2-d.x)/ppc), threshold)+","+//$NON-NLS-1$
				LaTeXDrawNumber.getCutNumber((float)((d.y-y2)/ppc), threshold)+")(" +  //$NON-NLS-1$
				LaTeXDrawNumber.getCutNumber((float)((pts.lastElement().x-d.x)/ppc), threshold)+","+//$NON-NLS-1$
				LaTeXDrawNumber.getCutNumber((float)((d.y-pts.lastElement().y)/ppc), threshold)+")\n";//$NON-NLS-1$
	        }
		}
		else
		{
			LaTeXDrawPoint2D p = pts.firstElement();
			
			coord = "\\moveto(" +LaTeXDrawNumber.getCutNumber((float)((p.x-d.x)/ppc), threshold)+","+//$NON-NLS-1$//$NON-NLS-2$
					LaTeXDrawNumber.getCutNumber((float)((d.y-p.y)/ppc), threshold)+")\n";//$NON-NLS-1$
			
			for(i=interval; i<size; i+=interval)
			{
				p = pts.elementAt(i);
				coord += "\\lineto(" +LaTeXDrawNumber.getCutNumber((float)((p.x-d.x)/ppc), threshold)+","+//$NON-NLS-1$//$NON-NLS-2$
						LaTeXDrawNumber.getCutNumber((float)((d.y-p.y)/ppc), threshold)+")\n";//$NON-NLS-1$
			}
			
			if((i-interval)<size)
				coord += "\\lineto(" +LaTeXDrawNumber.getCutNumber((float)((pts.lastElement().x-d.x)/ppc), threshold)+","+//$NON-NLS-1$//$NON-NLS-2$
				LaTeXDrawNumber.getCutNumber((float)((d.y-pts.lastElement().y)/ppc), threshold)+")\n";//$NON-NLS-1$
		}
		
		if(!isFilled && hasShadow() && isHatched())
		{
			String name = DviPsColors.getColourName(interiorColor);
			if(name==null)
			{
				name = "color"+number+'b';//$NON-NLS-1$
				DviPsColors.addUserColour(interiorColor, name); 
			}
			add += ",fillcolor=" + name; //$NON-NLS-1$
		}
		
		return "\\pscustom[linewidth=" + (thickness/ppc)+ fillCode + add +"]\n{\n\\newpath\n" + //$NON-NLS-1$//$NON-NLS-2$
				coord + (open ? "" : "\\closepath")+ (hasShadow ? "\\openshadow\n" : "") + '}';//$NON-NLS-1$//$NON-NLS-2$//$NON-NLS-3$//$NON-NLS-4$
	}

	

	
	@SuppressWarnings("unchecked")
	private void readObject(ObjectInputStream ois) throws IOException, ClassNotFoundException
	{
		canHaveShadow 			= true;
		isDoubleBoundaryable 	= false;
		isDashableOrDotable 	= true;
		canBeFilled				= true;
		interiorColor 	= (Color) ois.readObject();
		lineStyle 		= (String) ois.readObject();
		rotationAngle 	= ois.readDouble();
		thickness 		= ois.readFloat();
		isFilled 		= ois.readBoolean();
		isSelected 		= ois.readBoolean();
		isOnRotation 	= ois.readBoolean();
		linesColor 		= (Color) ois.readObject();
		blackDashLength = ois.readFloat();
		dotSep 			= ois.readFloat();
		whiteDashLength = ois.readFloat();
		pts 			= (Vector) ois.readObject();
		
		delimiters = new Vector<Delimitor>();
		for(int i=0, size = pts.size();i<size; i++)
			delimiters.add(new Delimitor(pts.elementAt(i)));
			
		if(LaTeXDrawFrame.getVersionOfFile().compareTo("1.5")>=0) //$NON-NLS-1$
		{
			hasDoubleBoundary 	= ois.readBoolean();
			doubleColor 		= (Color)ois.readObject();
			doubleSep 			= ois.readDouble();
		}
		else
		{
			hasDoubleBoundary  	= DEFAULT_HAS_DOUBLE_BOUNDARY;
			doubleColor 		= DEFAULT_DOUBLE_COLOR;
			doubleSep   		= DEFAULT_DOUBLESEP;
		}
		
		if(LaTeXDrawFrame.getVersionOfFile().compareTo("1.7")>=0) //$NON-NLS-1$
		{
			hasShadow 	= ois.readBoolean();
			shadowAngle = ois.readDouble();
			shadowSize	= ois.readDouble();
			shadowColor	= (Color)ois.readObject();
		}
		else
		{
			hasShadow 	= DEFAULT_SHADOW_HAS;
			shadowAngle	= DEFAULT_SHADOW_ANGLE;
			shadowSize	= DEFAULT_SHADOW_SIZE;
			shadowColor	= DEFAULT_SHADOW_COLOR;
		}
		
		if(LaTeXDrawFrame.getVersionOfFile().compareTo("1.9")>=0) //$NON-NLS-1$
		{
			interval 	= ois.readInt();
			type 		= ois.readInt();
			open 		= ois.readBoolean();
		}
		else
		{	
			interval	= DEFAULT_INTERVAL;
			type		= DEFAULT_TYPE;
			open		= DEFAULT_OPEN;
		}
		
		shape = createShape2D();
	}




	@Override
	public void updateToGrid(MagneticGrid grid)
	{
		super.updateToGrid(grid);
		
		int i=0;
		
		while(i<pts.size()-1)
			if(pts.elementAt(i).equals(pts.elementAt(i+1)))
				pts.remove(i);
			else
				i++;
		
		updateShape();
	}




	/**
	 * @return the interval.
	 * @since 1.9
	 */
	public int getInterval()
	{
		return interval;
	}




	/**
	 * @param interval the interval to set.
	 * @since 1.9
	 */
	public void setInterval(int interval)
	{
		if(interval>0)
		{
			this.interval = interval;
			updateShape();
			updateBorders();
		}
		else
			throw new IllegalArgumentException("The interval must be greater than 0.");//$NON-NLS-1$
	}




	/**
	 * @return the type.
	 * @since 1.9
	 */
	public int getType()
	{
		return type;
	}




	/**
	 * @param type the type to set.
	 * @since 1.9
	 */
	public void setType(int type)
	{
		if(type==TYPE_CURVES || type==TYPE_LINES)
		{
			this.type = type;
			updateShape();
			updateBorders();
		}
	}


	
	@Override
	public Shape createShape2D()
	{
		GeneralPath gp = new GeneralPath();
		int i, size = pts.size();
		
		if(getType()==TYPE_CURVES)
		{
			float prevx = (float)pts.lastElement().x;
			float prevy = (float)pts.lastElement().y;
			float curx = (float)pts.firstElement().x;
			float cury = (float)pts.firstElement().y;
	        float midx = (curx + prevx) / 2.0f;
	        float midy = (cury + prevy) / 2.0f;
			
        	gp.moveTo(curx, cury);
	    	
	        if(pts.size()>interval)
	        {
	            prevx = curx;
	            prevy = cury;
	            curx = (float)pts.elementAt(interval).x; 
	            cury = (float)pts.elementAt(interval).y;
	            midx = (curx + prevx) / 2.0f;
	            midy = (cury + prevy) / 2.0f;
	            
            	gp.lineTo(midx, midy);
	        }
	        
	        for(i=interval*2; i <pts.size(); i+=interval) 
	        {
	        	 float x1 = (midx + curx) / 2.0f;
	             float y1 = (midy + cury) / 2.0f;
	             prevx = curx;
	             prevy = cury;
	             curx = (float)pts.elementAt(i).x; 
	             cury = (float)pts.elementAt(i).y;
	             midx = (curx + prevx) / 2.0f;
	             midy = (cury + prevy) / 2.0f;
	             float x2 = (prevx + midx) / 2.0f;
	             float y2 = (prevy + midy) / 2.0f;
	             
            	 gp.curveTo(x1, y1, x2, y2, midx, midy);
	        }
	        
	        if((i-interval+1)<size)
	        {
	        	float x1 = (midx + curx) / 2.0f;
	        	float y1 = (midy + cury) / 2.0f;
	            prevx = curx;
	            prevy = cury;
	            curx = (float)pts.lastElement().x; 
	            cury = (float)pts.lastElement().y;
	            midx = (curx + prevx) / 2.0f;
	            midy = (cury + prevy) / 2.0f;
	            float x2 = (prevx + midx) / 2.0f;
	            float y2 = (prevy + midy) / 2.0f;
	            
            	gp.curveTo(x1, y1, x2, y2, (float)pts.lastElement().x, (float)pts.lastElement().y);
	        }
		}
		else
		{
			LaTeXDrawPoint2D p = pts.firstElement();
			
			gp.moveTo((float)p.x, (float)p.y);
			
			for(i=interval; i<size; i+=interval)
			{
				p = pts.elementAt(i);
				
				gp.lineTo((float)p.x, (float)p.y);
			}
			
			if((i-interval)<size)
				gp.lineTo((float)pts.lastElement().x, (float)pts.lastElement().y);
		}
		
		if(!isOpen())
			gp.closePath();
	
		return gp;
	}
	


	@Override
	public void updateShape()
	{
		updateGravityCenter();
		shape = createShape2D();
	}




	@Override
	public Object clone() throws CloneNotSupportedException
	{
		AkinPoints ak = (AkinPoints)super.clone();
		
		ak.interval = interval;
		ak.type		= type;
		ak.updateShape();
		
		return ak;
	}




	/**
	 * @return the open.
	 * @since 1.9
	 */
	public boolean isOpen()
	{
		return open;
	}




	/**
	 * @param open the open to set.
	 * @since 1.9
	 */
	public void setOpen(boolean open)
	{
		if(this.open!=open)
		{
			this.open = open;
			updateShape();
		}
	}




	@Override
	public Shape createShadowShape()
	{
		if(!hasShadow()) return shape;
		
		double dx, dy;
		LaTeXDrawPoint2D cg 		= getGravityCenter();
		LaTeXDrawPoint2D shadowCg 	= (LaTeXDrawPoint2D)cg.clone();
		AffineTransform at 			= new AffineTransform();
		
		shadowCg.setLocation(cg.x+shadowSize, cg.y);
		shadowCg = Figure.rotatePoint(shadowCg, cg, shadowAngle);
		dx = shadowCg.x-cg.x;
		dy = cg.y-shadowCg.y;
		at.translate(dx, dy);
		
		return at.createTransformedShape(shape);
	}

	
	
	@Override
	public void removePointAt(int id)
	{
		if(pts.isEmpty()) return ;
		
		if(id>=pts.size() || id<-1)
			throw new IllegalArgumentException();

		if(id==-1)
			id = pts.size()-1;
		
		pts.remove(id);
		
		if(borders!=null)
			updateBorders();
		
		updateShape();
	}
	
	
	
	@Override
	public int hashCode()
	{
		return super.hashCode()/4;
	}




	@Override
	public void updateBorders()
	{
		int i, size = pts.size();
		double NWx, NWy, SEy, SEx;
		
		if(getType()==TYPE_CURVES)
		{
			float prevx = (float)pts.lastElement().x;
			float prevy = (float)pts.lastElement().y;
			float curx = (float)pts.firstElement().x;
			float cury = (float)pts.firstElement().y;
	        float midx = (curx + prevx) / 2.0f;
	        float midy = (cury + prevy) / 2.0f;
			
	        NWx = SEx = curx;
	        NWy = SEy = cury;
	    	
	        if(pts.size()>interval)
	        {
	            prevx = curx;
	            prevy = cury;
	            curx = (float)pts.elementAt(interval).x; 
	            cury = (float)pts.elementAt(interval).y;
	            midx = (curx + prevx) / 2.0f;
	            midy = (cury + prevy) / 2.0f;
	        	
	        	if(midx<NWx) NWx = midx;
				else if(midx>SEx) SEx = midx;
				if(midy<NWy) NWy = midy;
				else if(midy>SEy) SEy = midy;
	        }
	        
	        for(i=interval*2; i <pts.size(); i+=interval) 
	        {
	             prevx = curx;
	             prevy = cury;
	             curx = (float)pts.elementAt(i).x; 
	             cury = (float)pts.elementAt(i).y;
	             midx = (curx + prevx) / 2.0f;
	             midy = (cury + prevy) / 2.0f;
	             
	        	if(midx<NWx) NWx = midx;
				else if(midx>SEx) SEx = midx;
				if(midy<NWy) NWy = midy;
				else if(midy>SEy) SEy = midy;
	        }
	        
	        if((i-interval+1)<size)
	        {
	            prevx = curx;
	            prevy = cury;
	            curx = (float)pts.lastElement().x; 
	            cury = (float)pts.lastElement().y;
	            midx = (curx + prevx) / 2.0f;
	            midy = (cury + prevy) / 2.0f;
	            
	        	if(pts.lastElement().x<NWx) NWx = pts.lastElement().x;
				else if(pts.lastElement().x>SEx) SEx = pts.lastElement().x;
				if(pts.lastElement().y<NWy) NWy = pts.lastElement().y;
				else if(pts.lastElement().y>SEy) SEy = pts.lastElement().y;
	        }
		}
		else
		{
			LaTeXDrawPoint2D p = pts.firstElement();
			NWx = SEx = p.x;		
			SEy = NWy = p.y;
			
			for(i=interval; i<size; i+=interval)
			{
				p = pts.elementAt(i);
				if(p.x<NWx) NWx = p.x;
				else if(p.x>SEx) SEx = p.x;
				if(p.y<NWy) NWy = p.y;
				else if(p.y>SEy) SEy = p.y;
			}
			
			if((i-interval)<size)
				if(pts.lastElement().x<NWx) NWx = pts.lastElement().x;
				else if(pts.lastElement().x>SEx) SEx = pts.lastElement().x;
				if(pts.lastElement().y<NWy) NWy = pts.lastElement().y;
				else if(pts.lastElement().y>SEy) SEy = pts.lastElement().y;
		}
		
		if(borders==null)
			borders = new LaTeXDrawRectangle(new LaTeXDrawPoint2D(NWx,NWy),
											 new LaTeXDrawPoint2D(SEx,SEy),false);
		else
		{
			borders.setLastPoint(SEx, SEy);
			borders.setFirstPoint(NWx, NWy);
		}
		updateGravityCenter();
	}
	
	
	@Override
	public boolean shadowFillsShape()
	{
		return false;
	}
	
	
	@Override
	public String toString()
	{
		String str = super.toString();
		
		return str + "{pts=" + pts + ", type=" + getType() + ", inter=" + getInterval() + ", open=" + isOpen() + "}"; //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$ //$NON-NLS-5$
	}
}
