/* This program 'tests' a tape by timing the time needed to read
 * each block on tape. %-<
 * Option -w to write an entire tape with a test pattern. (Destructive.)
 *	  -f to specify tape device.
 *	  -d to specify the treshold to be used in milliseconds.
 *	  -e to prevent erasing the tape afterwards (only with -w).
 * Written by Hennus Bergman. Copyright 1992 . All rights reserved.
 */


#include <sys/time.h>
#include <stdlib.h>
#include <unistd.h>
#include <stdio.h>
#include <errno.h>
#include <getopt.h>	/* assuming long opts */
#include <sys/mtio.h>
#include <sys/stat.h>

#define TPBUFSIZE	512

struct timezone tz = {0, 0};

int write_pat_flag 	= 0;
int no_erase_flag 	= 0;
long delay_val 		= 20000;   /* 1 block (512 bytes) is 8 msec @ 500kbps */
char *tapedev;

char *progname;

char buffer[TPBUFSIZE];


struct option longopts[] =
{
  {"file", required_argument, NULL, 'f'},
  {"write", no_argument, NULL, 'w'},
  {"no-erase", no_argument, NULL, 'e'},
  {"delay", required_argument, NULL, 'd'},
  {NULL, 0, NULL, 0}
};


void usage(void)
{
	printf("%s: Checks a tape for badly readable blocks by timing retries.\n"
	       "Usage:\t%s [-w [-e]] [-d msecs] [-f device]\n"
	       "\t[--write [--no-erase]] [--file device] [--delay msecs]\n\n"
	       "The -w option writes a test pattern.\n",
		progname, progname);
	exit(1);
} /* usage */


void check_type (char *dev)		/* adapted from gnu mt(1) */
{
	struct stat stats;

	if (!dev) {
		fputs("No tape device given.", stderr);
		exit(1);
	}
	if (stat (dev, &stats) == -1) {
		perror("cannot stat() tape device");
		exit(1);
	}
	if ((stats.st_mode & S_IFMT) != S_IFCHR) {
		puts("Argument of '-f' is not a character special file");
		exit(1);
	}
} /* check_type */


void rewind(FILE *f)
{
	struct mtop op;

	op.mt_op = MTREW;
	op.mt_count = 1;
	if (ioctl(fileno(f), MTIOCTOP, &op) != 0) {
		perror("could not rewind tape");
		exit(1);
	}
} /* rewind */


void erase_tape(FILE *f)
{
	struct mtop op;

	fputs("Erasing tape...", stderr);
	op.mt_op = MTERASE;
	op.mt_count = 1;
	if (ioctl(fileno(f), MTIOCTOP, &op) != 0) {
		perror("could not erase tape");
		exit(1);
	}
} /* erase_tape */


void zap_tape(void)
{
	FILE *f;
	int i;
	unsigned long nblocks = 0;

	if ((f = fopen(tapedev, "w")) == NULL) {
		perror("could not open tape device for writing");
		exit(1);
	}
	rewind(f);
	for (i=0; i<TPBUFSIZE; i++)
		buffer[i] = i;

	nblocks = 0;
	do {
		i = write(fileno(f), buffer, TPBUFSIZE);
		if (i==TPBUFSIZE) nblocks++;
	} while (i==TPBUFSIZE);

	printf("Wrote %lu blocks (%lu MB)\n", nblocks, nblocks * TPBUFSIZE / 1024 / 1024);
	fflush(stdout);

	if (i<0) {
		perror("write to tape failed");
#if 0
		exit(1);
#else
		printf("errno = %d\n", errno);
#endif
	} else if (i==0)
		puts("EOF on tape during writing.");
	else
		printf("Only wrote %d bytes\n", i);


	rewind(f);
	fclose(f);
} /* zap_tape */
	


void now(struct timeval *t)
{
	if (gettimeofday(t, &tz)!=0) {
		perror("Could not get time");
		exit(1);
	}
} /* now */	


void elapsed(struct timeval *d, struct timeval *te, struct timeval *ts)
{
	/* assume *te > *ts */
	if (te->tv_usec >= ts->tv_usec) {
		d->tv_usec = te->tv_usec - ts->tv_usec;
		d->tv_sec  = te->tv_sec - ts->tv_sec;
	} else {
		d->tv_usec = 1000 * 1000 - ts->tv_usec + te->tv_usec;
		d->tv_sec  = te->tv_sec - ts->tv_sec - 1;
	}
#if 0
	printf("elapsed: %lu, %lu\n", d->tv_sec, d->tv_usec);
	fflush(stdout);
#endif
} /* elapsed */



void run_test(void)
{
	FILE *f;
	int i;
	unsigned long nblocks = 0;
	struct timeval	ts, te, tr;
		
	if ((f = fopen(tapedev, "r")) == NULL) {
		perror("could not open tape device for reading");
		exit(1);
	}

	nblocks = 0;
	i = 0;
	do {
		now(&ts);
		i = read(fileno(f), buffer, TPBUFSIZE);
		now(&te);
		elapsed(&tr, &te, &ts);
		if (tr.tv_usec > delay_val) {
			printf("block %lu took %ld seconds and %ld microsec\n",
				nblocks, tr.tv_sec, tr.tv_usec);
			fflush(stdout);
#if 0
			printf("s: %lu, %lu\ne: %lu, %lu\n", ts.tv_sec, ts.tv_usec,
				te.tv_sec, te.tv_usec);
			fflush(stdout);
#endif
		}
		if (i==TPBUFSIZE) nblocks++;
	} while (i==TPBUFSIZE);

	printf("Read %lu blocks (%lu MB)\n", nblocks, nblocks * TPBUFSIZE / 1024 / 1024);
	fflush(stdout);

	if (i<0) {
		perror("read from tape failed");
		exit(1);
	}
	if (i==0)
		puts("Read EOF on tape.");
	else
		printf("Read only %d bytes in last block\n", i);

	rewind(f);
	if ((write_pat_flag != 0) && (no_erase_flag == 0))
		erase_tape(f);
	fclose(f);
} /* run_test */



int main(int argc, char *argv[])
{
	int i;

	progname = argv[0];

#ifdef DEFTAPE
	tapedev = DEFTAPE;
#else
	tapedev = NULL;
#endif


	while ((i = getopt_long (argc, argv, "ed:wf:", longopts, (int *) 0)) != EOF) {
		if (i == 'f')
			tapedev = optarg;
		else if (i == 'w')
			write_pat_flag = 1;
		else if (i == 'd')
			delay_val = atoi(optarg)*1000;
		else if (i == 'e')
			no_erase_flag = 1;
		else
			usage ();
	}
	if (optind != argc)
		usage ();

	check_type(tapedev);
	printf("%s: Running%s tape test on %s ...\n", progname, 
			(write_pat_flag)? " destructive" : "", tapedev);
	printf("delay treshold: %ld milliseconds\n", delay_val / 1000);
	fflush(stdout);
	if (write_pat_flag != 0)
		zap_tape();
	run_test();
	exit(0);
}

