/****************************************************************************
 *
 * Copyright (c) 1997-2002 Novell, Inc.
 * All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of version 2.1 of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, contact Novell, Inc.
 *
 * To contact Novell about this file by physical or electronic mail,
 * you may find current contact information at www.novell.com
 *
 ****************************************************************************/

/* Product defines */

#define PRODUCT_NAME                        "Hula API Library"
#define PRODUCT_VERSION                     "$Revision: 1.6 $"
#define PRODUCT_SHORT_NAME                  "msgapi.nlm"

#define CURRENT_PRODUCT_VERSION             3
#define CURRENT_PRODUCT_VERSION_MAJOR       5
#define CURRENT_PRODUCT_VERSION_MINOR       3

#include <config.h>
#include <xpl.h>
#include <xplresolve.h>
#include <connio.h>

#include <nmap.h>
#include <msgapi.h>
#include "msgapip.h"

#define MSGAPI_FLAG_EXITING                 (1 << 0)
#define MSGAPI_FLAG_STANDALONE              (1 << 1)
#define MSGAPI_FLAG_SUPPORT_CONNMGR         (1 << 2)
#define MSGAPI_FLAG_CLUSTERED               (1 << 3)
#define MSGAPI_FLAG_BOUND                   (1 << 4)
#define MSGAPI_FLAG_INTERNET_EMAIL_ADDRESS  (1 << 5)
#define MSGAPI_FLAG_MONITOR_RUNNING         (1 << 6)

#define LOOPBACK_ADDRESS                    0x0100007F
#define EMPTY_CHAR                          0x01

#define CONNMGR_STRUCT_SIZE                 (sizeof(ConnectionManagerCommand) + MAXEMAILNAMESIZE)

#define CONNMGR_REPLY_NOT_FOUND             0
#define CONNMGR_REPLY_FOUND                 1
#define CONNMGR_REPLY_FOUND_NAME            2

#define CONNMGR_REQ_STORE                   0
#define CONNMGR_REQ_CHECK                   1
#define CONNMGR_REQ_STORE_NAME              2
#define CONNMGR_REQ_CHECK_NAME              3

#if !defined(A_INTERNET_EMAIL_ADDRESS)
#define A_INTERNET_EMAIL_ADDRESS            "Internet EMail Address"
#endif

enum LibraryStates {
    LIBRARY_LOADED = 0, 
    LIBRARY_INITIALIZING, 
    LIBRARY_RUNNING, 
    LIBRARY_SHUTTING_DOWN, 
    LIBRARY_SHUTDOWN, 

    LIBRARY_MAX_STATES
} MSGAPIState = LIBRARY_LOADED;

struct {
    unsigned long flags;

    MDBHandle directoryHandle;

    XplThreadID groupID;

    struct {
        XplSemaphore version;
        XplSemaphore shutdown;
    } sem;

    struct {
        unsigned char dn[MDB_MAX_OBJECT_CHARS + 1];
        unsigned char tree[MDB_MAX_OBJECT_CHARS + 1];
    } server;

    struct {
        MDBValueStruct *storePaths;
        MDBValueStruct *storeDNs;

        struct {
            MDBValueStruct *names;
            MDBValueStruct *contexts;
            MDBValueStruct *addresses;
        } server;

#ifdef PARENTOBJECT_STORE
        struct {
            MDBValueStruct *objects;
            MDBValueStruct *stores;
        } parent;
#endif
    } vs;

    struct {
        unsigned long count;
        unsigned long allocated;

        unsigned char **list;
    } free;

    unsigned long connManager;

    XplRWLock configLock;

    struct {
        unsigned char work[XPL_MAX_PATH + 1];
        unsigned char nls[XPL_MAX_PATH + 1];
        unsigned char dbf[XPL_MAX_PATH + 1];
        unsigned char bin[XPL_MAX_PATH + 1];
        unsigned char lib[XPL_MAX_PATH + 1];
        unsigned char certificate[XPL_MAX_PATH + 1];
        unsigned char key[XPL_MAX_PATH + 1];
    } paths;

    struct {
        unsigned long local;

        unsigned char string[16];
    } address;

    struct {
        unsigned char moduleName[XPL_MAX_PATH + 1];
        unsigned char fileName[XPL_MAX_PATH + 1];

        XplPluginHandle handle;

        FindObjectCacheInitType init;
        FindObjectCacheShutdownType shutdown;
        FindObjectCacheType find;
        FindObjectCacheExType findEx;
        FindObjectStoreCacheType store;
    } cache;

    struct {
        unsigned char domain[MAXEMAILNAMESIZE + 1];
        unsigned long domainLength;
    } official;

    XplAtomic useCount;
} MsgGlobal;

typedef struct _ConnetionManagerCommand {
    unsigned long request;
    unsigned long address;
    unsigned long nameLen;
    unsigned long counter;
    unsigned char name[1];
} ConnectionManagerCommand;


EXPORT const unsigned char *
MsgGetServerDN(unsigned char *buffer)
{
    if (buffer) {
        return(strcpy(buffer, MsgGlobal.server.dn));
    }
    return(MsgGlobal.server.dn);
}

EXPORT BOOL
MsgSetServerState(const unsigned char *server, const unsigned char *setState)
{
    MDBValueStruct *state;

    state = MDBCreateValueStruct(MsgGlobal.directoryHandle, NULL);
    MDBAddValue(setState, state);

    if (server) {
        MDBWrite(server, MSGSRV_A_SERVER_STATUS, state);
    } else {
        MDBWrite(MsgGlobal.server.dn, MSGSRV_A_SERVER_STATUS, state);
    }

    MDBDestroyValueStruct(state);

    return(TRUE);
}

EXPORT BOOL
MsgFindObject(const unsigned char *user, unsigned char *dn, unsigned char *userType, struct in_addr *nmap, MDBValueStruct *v)
{
    BOOL retVal = FALSE;

    if (MsgGlobal.cache.find == NULL) {
        MDBValueStruct *userCtx;
        unsigned long i, j;
        unsigned char rdn[MDB_MAX_OBJECT_CHARS + 1];
        unsigned char *type;
	unsigned char localType[MDB_MAX_ATTRIBUTE_CHARS + 1];

        userCtx = MDBCreateValueStruct(MsgGlobal.directoryHandle, NULL);

        if (!userType) {
            type=localType;
        } else {
            type=userType;
        }

        XplRWReadLockAcquire(&MsgGlobal.configLock);
        for (i = 0; i < MsgGlobal.vs.server.contexts->Used; i++) {
            MDBSetValueStructContext(MsgGlobal.vs.server.contexts->Value[i], userCtx);
            if (MDBGetObjectDetails(user, type, rdn, dn, userCtx)) {

                if (MDBRead(user, MSGSRV_A_MESSAGING_DISABLED, userCtx) > 0) {
                    if (userCtx->Value[userCtx->Used-1][0] == '1') {
                        break;
                    } else {
                        MDBFreeValue(userCtx->Used - 1, userCtx);
                    }
                }
                if ((XplStrCaseCmp(type, C_USER) == 0) || (XplStrCaseCmp(type, MSGSRV_C_RESOURCE) == 0)) {
                    if (v) {
                        MDBAddValue(rdn, v);
                    }
                    if (nmap) {
                        nmap->s_addr=inet_addr(MsgGlobal.vs.server.addresses->Value[i]);
                    }
                    retVal = TRUE;
                } else if (XplStrCaseCmp(type, C_GROUP) == 0) {
                    if (v) {
                        MDBSetValueStructContext(MsgGlobal.vs.server.contexts->Value[i], v);
                        MDBRead(user, A_MEMBER, v);

                        /* This removes any context from a group member's name */
                        for (j = 0; j < v->Used; j++) {
                            unsigned char *ptr;
                            ptr = strrchr(v->Value[j], '\\');
                            if (ptr) {
                                memmove(v->Value[j], ptr + 1, strlen(ptr + 1) + 1);
                            }
                        }
                    }
                    retVal = TRUE;
                } else if (XplStrCaseCmp(type, C_ORGANIZATIONAL_ROLE) == 0) {
                    if (v) {
                        MDBSetValueStructContext(MsgGlobal.vs.server.contexts->Value[i], v);
                        MDBRead(user, A_ROLE_OCCUPANT, v);

                        /* This removes any context from a the roles name */
                        for (j = 0; j < v->Used; j++) {
                            unsigned char *ptr;
                            ptr = strrchr(v->Value[j], '\\');
                            if (ptr) {
                                memmove(v->Value[j], ptr + 1, strlen(ptr + 1) + 1);
                            }
                        }
                    }
                    retVal = TRUE;
                } else if (XplStrCaseCmp(type, "dynamicGroup") == 0) {
                    if (v) {
                        MDBSetValueStructContext(MsgGlobal.vs.server.contexts->Value[i], v);
                        MDBRead(user, A_MEMBER, v);

                        /* This removes any context from a group member's name */
                        for (j = 0; j < v->Used; j++) {
                            unsigned char *ptr;
                            ptr=strrchr(v->Value[j], '\\');
                            if (ptr) {
                                memmove(v->Value[j], ptr + 1, strlen(ptr + 1) + 1);
                            }
                        }
                    }
                    retVal = TRUE;
                }
                break;
            }
        }
        XplRWReadLockRelease(&MsgGlobal.configLock);

        MDBDestroyValueStruct(userCtx);
    } else {
        retVal = MsgGlobal.cache.find(user, dn, userType, nmap, v);
    }

    return(retVal);
}

EXPORT BOOL
MsgFindObjectEx(const unsigned char *user, unsigned char *dn, unsigned char *userType, struct in_addr *nmap, BOOL *disabled, MDBValueStruct *v)
{
    BOOL retVal = FALSE;

    if (MsgGlobal.cache.findEx == NULL) {
        MDBValueStruct *userCtx;
        unsigned long i, j;
        unsigned char rdn[MDB_MAX_OBJECT_CHARS + 1];
        unsigned char *type;
	unsigned char localType[MDB_MAX_ATTRIBUTE_CHARS + 1];

        userCtx = MDBCreateValueStruct(MsgGlobal.directoryHandle, NULL);

        if (!userType) {
            type = localType;
        } else {
            type = userType;
        }

        XplRWReadLockAcquire(&MsgGlobal.configLock);

        for (i = 0; i < MsgGlobal.vs.server.contexts->Used; i++) {
            MDBSetValueStructContext(MsgGlobal.vs.server.contexts->Value[i], userCtx);

            if (MDBGetObjectDetails(user, type, rdn, dn, userCtx)) {
                if (MDBRead(user, MSGSRV_A_MESSAGING_DISABLED, userCtx) > 0) {
                    if (userCtx->Value[userCtx->Used - 1][0] == '1') {
                        *disabled = TRUE;
                    } else {
                        *disabled = FALSE;
                    }
                    MDBFreeValue(userCtx->Used - 1, userCtx);
                } else {
                    *disabled = FALSE;
                }
                if ((XplStrCaseCmp(type, C_USER) == 0) || (XplStrCaseCmp(type, MSGSRV_C_RESOURCE) == 0)) {
                    if (v) {
                        MDBAddValue(rdn, v);
                    }
                    if (nmap) {
                        nmap->s_addr = inet_addr(MsgGlobal.vs.server.addresses->Value[i]);
                    }
                    retVal = TRUE;
                } else if (XplStrCaseCmp(type, C_GROUP) == 0) {
                    if (v) {
                        MDBSetValueStructContext(MsgGlobal.vs.server.contexts->Value[i], v);
                        MDBRead(user, A_MEMBER, v);

                        /* This removes any context from a group member's name */
                        for (j = 0; j < v->Used; j++) {
                            unsigned char *ptr;
                            ptr = strrchr(v->Value[j], '\\');
                            if (ptr) {
                                memmove(v->Value[j], ptr + 1, strlen(ptr + 1) + 1);
                            }
                        }
                    }
                    retVal = TRUE;
                } else if (XplStrCaseCmp(type, C_ORGANIZATIONAL_ROLE) == 0) {
                    if (v) {
                        MDBSetValueStructContext(MsgGlobal.vs.server.contexts->Value[i], v);
                        MDBRead(user, A_ROLE_OCCUPANT, v);

                        /* This removes any context from a the roles name */
                        for (j = 0; j < v->Used; j++) {
                            unsigned char *ptr;
                            ptr = strrchr(v->Value[j], '\\');
                            if (ptr) {
                                memmove(v->Value[j], ptr + 1, strlen(ptr + 1) + 1);
                            }
                        }
                    }
                    retVal = TRUE;
                } else if (XplStrCaseCmp(type, "dynamicGroup") == 0) {
                    if (v) {
                        MDBSetValueStructContext(MsgGlobal.vs.server.contexts->Value[i], v);
                        MDBRead(user, A_MEMBER, v);

                        /* This removes any context from a group member's name */
                        for (j = 0; j < v->Used; j++) {
                            unsigned char *ptr;
                            ptr = strrchr(v->Value[j], '\\');
                            if (ptr) {
                                memmove(v->Value[j], ptr + 1, strlen(ptr + 1) + 1);
                            }
                        }
                    }
                    retVal = TRUE;
                }
                break;
            }
        }
        XplRWReadLockRelease(&MsgGlobal.configLock);

        MDBDestroyValueStruct(userCtx);
    } else {
        retVal = MsgGlobal.cache.findEx(user, dn, userType, nmap, disabled, v);
    }

    return(retVal);
}

EXPORT const unsigned char *
MsgFindUserStore(const unsigned char *user, const unsigned char *defaultPath)
{
    if (MsgGlobal.cache.store == NULL) {
        MDBValueStruct *context;
        unsigned long i;
        
        XplRWReadLockAcquire(&MsgGlobal.configLock);
        context = MDBCreateValueStruct(MsgGlobal.directoryHandle, MsgGlobal.vs.server.contexts->Value[0]);

        i = 0;
        do {
            if (MDBIsObject(user, context)) {
                MDBDestroyValueStruct(context);
                if (MsgGlobal.vs.storePaths->Value[i][0] != EMPTY_CHAR) {
                    XplRWReadLockRelease(&MsgGlobal.configLock);
                    return(MsgGlobal.vs.storePaths->Value[i]);
                } else {
                    XplRWReadLockRelease(&MsgGlobal.configLock);
                    return(defaultPath);
                }
            } else {
                i++;
                if (i != MsgGlobal.vs.server.contexts->Used) {
                    MDBSetValueStructContext(MsgGlobal.vs.server.contexts->Value[i], context);
                }
            }
        } while (i != MsgGlobal.vs.server.contexts->Used);
        XplRWReadLockRelease(&MsgGlobal.configLock);

        MDBDestroyValueStruct(context);
        return(defaultPath);
    }

    return MsgGlobal.cache.store(user, defaultPath);
}

EXPORT BOOL
MsgFindServer(const unsigned char *user, unsigned char *dn)
{
    MDBValueStruct *userCtx;
    unsigned long i;
    BOOL retVal = FALSE;

    userCtx = MDBCreateValueStruct(MsgGlobal.directoryHandle, NULL);

    XplRWReadLockAcquire(&MsgGlobal.configLock);
    for (i = 0; i < MsgGlobal.vs.server.contexts->Used; i++) {
        MDBSetValueStructContext(MsgGlobal.vs.server.contexts->Value[i], userCtx);
        if (MDBGetObjectDetails(user, NULL, NULL, NULL, userCtx)) {
            if (MDBRead(user, MSGSRV_A_MESSAGING_DISABLED, userCtx)>0) {
                if (userCtx->Value[userCtx->Used-1][0]=='1') {
                    break;
                } else {
                    MDBFreeValues(userCtx);
                }
            }

            /* The user exists, and is enabled */
            if (dn) {
                strcpy(dn, MsgGlobal.vs.storeDNs->Value[i]);
            }

            retVal=TRUE;
        }
    }
    XplRWReadLockRelease(&MsgGlobal.configLock);

    MDBDestroyValueStruct(userCtx);
    return(retVal);
}

EXPORT BOOL
MsgReadIP(unsigned char *object, unsigned char *type, MDBValueStruct *v)
{
    int first,last, i;
    unsigned char *ptr;

    if (!v || !type) {
        return(FALSE);
    }

    first=v->Used;

    if (!object) {
        if (!MDBReadDN(MsgGlobal.server.dn, type, v)) {
            return(FALSE);
        }
    } else {
        if (!MDBReadDN(object, type, v)) {
            return(FALSE);
        }
    }

    last = v->Used;

    for (i = first; i < last; i++) {
        if ((ptr = strrchr(v->Value[i], '\\')) != NULL) {
            *ptr = '\0';
        }
        MDBRead(v->Value[i], MSGSRV_A_IP_ADDRESS, v);
    }

    /* The following loop removes the names read by the ReadDN above,
     * but keeps the IP addresses added afterwards :-) */
    for (i = first; i < last; i++) {
        MDBFreeValue(0, v);
    }

    if (v->Used > 0) {
        return(TRUE);
    } else {
        return(FALSE);
    }
}

EXPORT BOOL
MsgDomainExists(const unsigned char *domain, unsigned char *domainObjectDN)
{
    MDBValueStruct *domains;
    unsigned long i;
    unsigned long j;

    /* FIXME later; when we handle parent objects */
    return(FALSE);

    domains = MDBCreateValueStruct(MsgGlobal.directoryHandle, NULL);

    XplRWReadLockAcquire(&MsgGlobal.configLock);

    for (i = 0; i < MsgGlobal.vs.server.names->Used; i++) {
        MDBSetValueStructContext(MsgGlobal.vs.server.names->Value[i], domains);
        if(MDBRead(MSGSRV_AGENT_SMTP, MSGSRV_A_DOMAIN, domains) > 0) {
            for (j = 0; j < domains->Used; j++) {
                if (XplStrCaseCmp(domain, domains->Value[j])==0) {
                    XplRWReadLockRelease(&MsgGlobal.configLock);
                    MDBDestroyValueStruct(domains);

                    return(TRUE);
                }
            }
        }
        MDBFreeValues(domains);
    }
    MDBDestroyValueStruct(domains);
    XplRWReadLockRelease(&MsgGlobal.configLock);

    return(FALSE);
}

EXPORT int
MsgGetParentAttribute(const unsigned char *userDN, unsigned char *attribute, MDBValueStruct *v)
{
    unsigned long index;

    if (MDBRead(userDN, MSGSRV_A_PARENT_OBJECT, v) > 0) {
        index = v->Used - 1;
        if (MDBRead(v->Value[index], attribute, v)>0) {
            MDBFreeValue(index, v);
            return(v->Used);
        } else {
            MDBFreeValue(index, v);
            return(MDBRead(userDN, attribute, v));
        }
    } else {
        return(MDBRead(userDN, attribute, v));
    }
}

EXPORT unsigned char *
MsgGetUserEmailAddress(const unsigned char *userDNin, MDBValueStruct *userData, unsigned char *buffer, unsigned long bufLen)
{
    unsigned char *emailAddress = NULL;
    unsigned long originalValueCount = userData->Used;
    unsigned long lastValueCount = originalValueCount;
    unsigned char *user = NULL;
    unsigned char userDN[MDB_MAX_OBJECT_CHARS + 1];
    unsigned char *delim;

    /* Make a copy so we don't edit the const string */ 
    strcpy(userDN, userDNin);

    if (!(MsgGlobal.flags & MSGAPI_FLAG_INTERNET_EMAIL_ADDRESS)) {
        MDBRead(userDN, A_INTERNET_EMAIL_ADDRESS, userData);
        if (userData->Used > lastValueCount) {

            if (strchr(userData->Value[userData->Used - 1], '@')) {
		/* The user object has a value in the Internet Email Address Attribute */
                if (buffer) {
                    if (bufLen > strlen(userData->Value[userData->Used - 1])) {
                        emailAddress = buffer;
                        strcpy(emailAddress, userData->Value[userData->Used - 1]);
                    } 
                } else {
                    emailAddress = MemStrdup(userData->Value[userData->Used - 1]);
                }

                while (originalValueCount < userData->Used){
                    MDBFreeValue(userData->Used - 1, userData);
                }
                return(emailAddress);
            } else {
                user = userData->Value[userData->Used - 1];
                lastValueCount++;
            }
        }
    }

    /* The value in the Internet Email Address Attribute still needs a domain */

    delim = strrchr(userDN, '\\' );
    if (delim) {
        if (strchr(delim + 1, '@')) {
            /* The username is an address */
            if (buffer) {
                if (bufLen > strlen(delim + 1)) {
                    emailAddress = buffer;
                    strcpy(emailAddress, delim + 1);
                }                                          
            } else {
                emailAddress = MemStrdup(delim + 1);
            }
	    
            while (originalValueCount < userData->Used) {
		MDBFreeValue(userData->Used - 1, userData);
	    }
	
            return(emailAddress);
        }

        if (!user) {
            user = delim + 1;
        }            


        if (MsgGetParentAttribute(userDN, MSGSRV_A_DOMAIN, userData)) {
            /* The user or parent has a domain defined */
            if (buffer) {
                if (bufLen > (strlen(user) + strlen(userData->Value[userData->Used - 1]) + 1)) {
                    emailAddress = buffer;
                    sprintf(emailAddress, "%s@%s", user, userData->Value[userData->Used - 1]);
                }
            } else {
                emailAddress = MemMalloc(strlen(user) + strlen(userData->Value[userData->Used - 1]) + 2);
                if (emailAddress) {
                    sprintf(emailAddress, "%s@%s", user, userData->Value[userData->Used - 1]);
                }
            }

            while (originalValueCount < userData->Used){
                MDBFreeValue(userData->Used - 1, userData);
            }

            return(emailAddress);
        }

        *delim = '\0';
        MDBRead(userDN, MSGSRV_A_DOMAIN, userData);
        if (userData->Used > lastValueCount) {
            /* The container has a domain defined */
            if (buffer) {
                if (bufLen > (strlen(user) + strlen(userData->Value[userData->Used - 1]) + 1)) {
                    emailAddress = buffer;
                    sprintf(emailAddress, "%s@%s", user, userData->Value[userData->Used - 1]);                
                }
            } else {
                emailAddress = MemMalloc(strlen(user) + strlen(userData->Value[userData->Used - 1]) + 2);
                if (emailAddress) {
                    sprintf(emailAddress, "%s@%s", user, userData->Value[userData->Used - 1]);
                }
            }
        } else {
            /* Default  user@official */ 
            if (buffer) {
                if (bufLen > (strlen(user) + MsgGlobal.official.domainLength + 1)) {
                    emailAddress = buffer;
                    sprintf(emailAddress, "%s@%s", user, MsgGlobal.official.domain);
                }
            } else {
                emailAddress = MemMalloc(strlen(user) + MsgGlobal.official.domainLength + 2);
                if (emailAddress) {
                    sprintf(emailAddress, "%s@%s", user, MsgGlobal.official.domain);
                }
            }
        }

        *delim = '\\';

        for (;originalValueCount < userData->Used;){
            MDBFreeValue(userData->Used - 1, userData);
        }
        return(emailAddress);
    }

    /* The dn does not appear valid.  Treat it as rdn and do our best */
    if (buffer) {
        if (bufLen > (strlen(userDN) + MsgGlobal.official.domainLength + 1)) {
            emailAddress = buffer;
            sprintf(emailAddress, "%s@%s", userDN, MsgGlobal.official.domain);
        }
    } else {
        emailAddress = MemMalloc(strlen(userDN) + MsgGlobal.official.domainLength + 2);
        if (emailAddress) {
            sprintf(emailAddress, "%s@%s", userDN, MsgGlobal.official.domain);
        }
    }

    while (originalValueCount < userData->Used) {
	MDBFreeValue(userData->Used - 1, userData);
    }
    return(emailAddress);
}

EXPORT unsigned char *
MsgGetUserDisplayName(const unsigned char *userDN, MDBValueStruct *userData)
{
    unsigned char *displayName = NULL;
    unsigned long index = userData->Used;
    unsigned char *rdn;
    
    /* Read the Display Name */
    if ((MDBRead(userDN, A_FULL_NAME, userData) > 0) && (userData->Value[index][0] != '\0')) {
        displayName = MemStrdup(userData->Value[index]);
        while (index < userData->Used){
            MDBFreeValue(index, userData);
        }
        return(displayName);
    } 

    /* Create full name using First and Last */
    MDBRead(userDN, A_GIVEN_NAME, userData);
    MDBRead(userDN, A_SURNAME, userData);
    if (userData->Used == (index + 2)) {
        if ((userData->Value[index][0] != '\0') && (userData->Value[index + 1][0] != '\0')) {
	    displayName = MemMalloc (strlen (userData->Value[index]) + strlen (userData->Value[index + 1]) + 2);
            sprintf(displayName, "%s %s", userData->Value[index], userData->Value[index + 1]);
        } else if (userData->Value[index][0] != '\0') {
            displayName = MemStrdup(userData->Value[index]);
        } else if (userData->Value[index + 1][0] != '\0') {
            displayName = MemStrdup(userData->Value[index + 1]);
        } else {
            if ((rdn = strrchr(userDN, '\\'))) {
                displayName = MemStrdup(rdn + 1);
            } else {

                displayName = MemStrdup("");
            }
        }
    } else if (userData->Used == (index + 1)) {
        if (!(userData->Value[index][0] == ' ' && userData->Value[index][1] == '\0')) {
            displayName = MemStrdup(userData->Value[index]);
        } else {
	    /* FIXME: is this right? */
            if ((rdn = strrchr (userDN, '\\'))) {
                displayName = MemStrdup(rdn + 1);
            } else {
                displayName = MemStrdup("");
            }
        }
    } else {
        if ((rdn = strrchr(userDN, '\\'))) {
            displayName = MemStrdup(rdn + 1);
        } else {  
            displayName = MemStrdup("");
        }

    }

    while (index < userData->Used) {
        MDBFreeValue(index, userData);
    }


    return(displayName);
    
}

/*
 * Here's the logic:
 *
 * Step 1: We get the DN from the caller; we check if there's a parent DN, 
 * Step 2: If we have inheritance DN, so we read the inheritance configuration of the parent, if not, go to step 4c
 * Step 3a: Inheritance is Parent->User; read features attribute from parent
 * Step 3b: Inheritance is User->Parent; read features attribute from user
 * Step 4a: If feature disabled return disabled state
 * Step 4b: If feature from parent, read parent data attribute
 * Step 4c: If feature from user, read user data attribute
 * Step 5: If data attribute empty, try opposite DN attribute
 *
 */
EXPORT BOOL
MsgGetUserFeature(const unsigned char *userDN, unsigned char featureRow, unsigned long featureCol, unsigned char *attribute, MDBValueStruct *vOut)
{
    MDBValueStruct *v;
    unsigned char inheritance = '\0';
    unsigned char parentDN[MDB_MAX_OBJECT_CHARS+1];
    const unsigned char *objectDN;
    unsigned long i;
    BOOL looped;
    

    v = MDBCreateValueStruct(MsgGlobal.directoryHandle, NULL);

    /* Step 1 */
    if (MDBRead(userDN, MSGSRV_A_PARENT_OBJECT, v) > 0) {
	/* We have a parent */

        /* Step 2 */
        strcpy(parentDN, v->Value[0]);

        if (MDBRead(v->Value[0], MSGSRV_A_FEATURE_INHERITANCE, v) > 0) {
            /* We have a configured inheritance */
            if ((inheritance = v->Value[1][0]) == FEATURE_PARENT_FIRST) {            
                objectDN = parentDN;
            } else {
                objectDN = userDN;
            }
        } else {
	    /* No inheritance configured */
            objectDN = userDN;
        }
        MDBFreeValues(v);
    } else {
        parentDN[0] = '\0';
        objectDN = userDN;
    }

    /* objectDN now contains the DN we have determined we need to read first */

    looped=FALSE;

    /* Step 3 */
 ReadFeatureAttribute:
    if (MDBRead(objectDN, MSGSRV_A_FEATURE_SET, v) > 0) {
        /* We have features defined on the object */
        for (i = 0; i < v->Used; i++) {
	    /* Find our feature */
            if (v->Value[i][0] == featureRow) {
                switch (v->Value[i][featureCol]) {
		case FEATURE_NOT_AVAILABLE: {                                        
                    /* Feature disabled; return */
		    MDBDestroyValueStruct(v);

		    return(FALSE);
		}

		case FEATURE_AVAILABLE: {
                    /* Feature enabled */
		    i = v->Used;
		    break;
		}

		case FEATURE_USE_PARENT: 
                    /* Feature enabled via parent */
		case FEATURE_USE_USER: {
		    /* Feature enabled via user */
		    if (!looped && (parentDN[0] != '\0')) {
			if (objectDN == parentDN) {
			    objectDN = userDN;
			} else {
			    objectDN = parentDN;
			}

			MDBFreeValues(v);
			looped=TRUE;
			goto ReadFeatureAttribute;
		    } else { 
                        /* If loop exists, use user object */
			objectDN = userDN;
			i = v->Used;
		    }
		    break;
		}
                }
            }
        }

        /* We now have the DN of the object we need to read the data attribute from in ObjectDN */
        MDBFreeValues(v);
    } else {
        /* We didn't find a feature set configuration */
        if (inheritance == FEATURE_PARENT_FIRST) {
            /* We read the parent, didn't have feature config, let's read the user's feature set */
            inheritance = FEATURE_USER_FIRST;
            objectDN = userDN;
            goto ReadFeatureAttribute;
        } else {
            /* We read the user, don't have inheritance and don't have a feature set -> return enabled feature!    */
            goto ReadDataAttribute;
        }
    }

 ReadDataAttribute:
    /* Beyond this point, we can return success; we returned failure above */

    /* Clean up */
    MDBDestroyValueStruct(v);

    /* If Attribute NULL we just check if feature disabled */
    if (!attribute) {
        return(TRUE);
    }

    i = MDBRead(objectDN, attribute, vOut);
    if (i > 0) {
        return(i);
    }

    /* We didn't find what we were looking for, let's try the opposite DN (if possible) */
    if (parentDN[0] != '\0') {
        /* If we checked User, let's check parent, and vice versa */
        if (objectDN == userDN) {
            i = MDBRead(parentDN, attribute, vOut);
        } else {
            i = MDBRead(userDN, attribute, vOut);
        }

        return(i);
    } else {
        /* No parent DN, don't bother; we return success, but without data */
        return(0);
    }
}

EXPORT BOOL
MsgRegisterAddress(unsigned long address, unsigned char *name)
{
    struct sockaddr_in fromSin;
    struct sockaddr_in toSin;
    int sock;
    unsigned char commandBuffer[CONNMGR_STRUCT_SIZE];
    ConnectionManagerCommand *command=(ConnectionManagerCommand *)commandBuffer;

    XplRWReadLockAcquire(&MsgGlobal.configLock);
    if (!(MsgGlobal.flags & MSGAPI_FLAG_SUPPORT_CONNMGR)) {
        XplRWReadLockRelease(&MsgGlobal.configLock);
        return(FALSE);
    }
    XplRWReadLockRelease(&MsgGlobal.configLock);

    memset(&fromSin, 0, sizeof(struct sockaddr_in));
    memset(&toSin, 0, sizeof(struct sockaddr_in));
    sock = socket(PF_INET, SOCK_DGRAM, IPPROTO_UDP);
    if (sock >= 0) {
        toSin.sin_family = AF_INET;

        XplRWReadLockAcquire(&MsgGlobal.configLock);
        toSin.sin_addr.s_addr = MsgGlobal.connManager;
        XplRWReadLockRelease(&MsgGlobal.configLock);

        toSin.sin_port = htons(NMAP_PORT);

        fromSin.sin_family = AF_INET;
        fromSin.sin_addr.s_addr = 0;
        fromSin.sin_port = 0;

        if (bind(sock, (struct sockaddr *)&fromSin, sizeof(fromSin)) == 0) {
            command->address = address;

            /* The split may make it harder to read, but it gives a few cycles more performace, and in this code path it's important! */
            if (name) {
                command->nameLen = strlen(name);
                memcpy(command->name, name, command->nameLen);
                command->request = CONNMGR_REQ_STORE_NAME;
                sendto(sock, (unsigned char *)command, sizeof(ConnectionManagerCommand) + command->nameLen, 0, (struct sockaddr*)&toSin, sizeof(toSin));
            } else {
                command->request = CONNMGR_REQ_STORE;
                sendto(sock, (unsigned char *)command, sizeof(ConnectionManagerCommand), 0, (struct sockaddr*)&toSin, sizeof(toSin));
            }
        }
        IPclose(sock);
    }

    return(TRUE);
}

EXPORT BOOL
MsgIsKnownAddress(unsigned long address, unsigned char *name)
{
    struct sockaddr_in fromSin;
    struct sockaddr_in toSin;
    int sock;
    int size;
    unsigned char commandBuffer[CONNMGR_STRUCT_SIZE];
    ConnectionManagerCommand *command=(ConnectionManagerCommand *)commandBuffer;

    XplRWReadLockAcquire(&MsgGlobal.configLock);
    if (!(MsgGlobal.flags & MSGAPI_FLAG_SUPPORT_CONNMGR)) {
        XplRWReadLockRelease(&MsgGlobal.configLock);
        if (name) {
            name[0]='\0';
        }

        return(TRUE);
    }
    XplRWReadLockRelease(&MsgGlobal.configLock);

    memset(&fromSin, 0, sizeof(struct sockaddr_in));
    memset(&toSin, 0, sizeof(struct sockaddr_in));
    sock = socket(PF_INET, SOCK_DGRAM, IPPROTO_UDP);
    if (sock >= 0) {
        toSin.sin_family = AF_INET;
        XplRWReadLockAcquire(&MsgGlobal.configLock);
        toSin.sin_addr.s_addr = MsgGlobal.connManager;
        XplRWReadLockRelease(&MsgGlobal.configLock);
        toSin.sin_port = htons(NMAP_PORT);

        fromSin.sin_family = AF_INET;
        fromSin.sin_addr.s_addr = 0;
        fromSin.sin_port = 0;

        if (bind(sock, (struct sockaddr *)&fromSin, sizeof(fromSin)) == 0) {
            command->address = address;

            if (name) {
                command->request = CONNMGR_REQ_CHECK_NAME;
            } else {
                command->request = CONNMGR_REQ_CHECK;
            }
            sendto(sock, (unsigned char *)command, CONNMGR_STRUCT_SIZE, 0, (struct sockaddr*)&toSin, sizeof(toSin));

            size = XplIPRead((void *)sock, (unsigned char *)command, CONNMGR_STRUCT_SIZE, 19);

            IPclose(sock);

            if (size <= 0 || command->request == CONNMGR_REPLY_NOT_FOUND) {
                return(FALSE);
            } else {
                if (name) {
                    if (command->request == CONNMGR_REPLY_FOUND_NAME) {
                        memcpy(name, command->name, command->nameLen);
                        name[command->nameLen]='\0';
                    } else {
                        name[0]='\0';
                    }
                }

                return(TRUE);
            }
        }
        IPclose(sock);
    }

    return(FALSE);
}

EXPORT BOOL
MsgCleanPath(unsigned char *path)
{
    unsigned char *ptr;
    unsigned char *ptr2;

    /* This code strips off any server names that might be configured in the Path */
    ptr = path;
    while((ptr = strchr(ptr, '\\')) != NULL) {
        *ptr = '/';
    }

    ptr=strchr(path, ':');
    if (ptr) {
        *ptr = '\0';
        ptr2 = strchr(path, '/');
        if (ptr2) {
            *ptr=':';
            memmove(path, ptr2+1, strlen(ptr2+1)+1);
        } else {
            *ptr=':';
        }
    }

/* If you want all pathnames lowercase, uncomment the following */
#if 0
    ptr = path;
    while (*ptr) {
        if (isupper(*ptr)) {
            *ptr = tolower(*ptr);
        }
        ptr++;
    }
#endif

    /* strip off trailing /, to allow other code reliable appending of files by using %s/%s */
    ptr = path + strlen(path) - 1;
    if (*ptr == '/') {
        *ptr = '\0';
    }

    return(TRUE);
}

EXPORT void
MsgMakePath(unsigned char *path)
{
    unsigned char *ptr = path;
    unsigned char *ptr2;
    struct stat sb;

    ptr = strchr(path, '/');
    if (!ptr)
        ptr=strchr(path, '\\');

    while (ptr) {
        *ptr = '\0';
        if (stat(path, &sb) != 0) {
            XplMakeDir(path);
        }

        *ptr = '/';
        ptr2 = ptr;
        ptr = strchr(ptr2 + 1, '/');
        if (!ptr)
	    ptr = strchr(ptr2 + 1, '\\');
    }

    if (stat(path, &sb) != 0) {
        XplMakeDir(path);
    }
}

EXPORT const unsigned char *
MsgGetDBFDir(char *directory)
{
    if (directory) {
        strcpy(directory, MsgGlobal.paths.dbf);
    }
    return(MsgGlobal.paths.dbf);
}

EXPORT const unsigned char *
MsgGetWorkDir(char *directory)
{
    if (directory) {
        strcpy(directory, MsgGlobal.paths.work);
    }
    return(MsgGlobal.paths.work);
}

EXPORT const unsigned char *
MsgGetNLSDir(char *directory)
{
    if (directory) {
        strcpy(directory, MsgGlobal.paths.nls);
    }
    return(MsgGlobal.paths.nls);
}

EXPORT const unsigned char *
MsgGetBinDir(char *directory)
{
    if (directory) {
        strcpy(directory, MsgGlobal.paths.bin);
    }
    return(MsgGlobal.paths.bin);
}

EXPORT const unsigned char *
MsgGetLibDir(char *directory)
{
    if (directory) {
        strcpy(directory, MsgGlobal.paths.lib);
    }
    return(MsgGlobal.paths.bin);
}

EXPORT const unsigned char *
MsgGetTLSCertPath(char *path)
{
    if (path) {
        strcpy(path, MsgGlobal.paths.certificate);
    }
    return(MsgGlobal.paths.certificate);
}

EXPORT const unsigned char *
MsgGetTLSKeyPath(char *path)
{
    if (path) {
        strcpy(path, MsgGlobal.paths.key);
    }
    return(MsgGlobal.paths.key);
}

EXPORT unsigned long
MsgGetAgentBindIPAddress(void)
{
    if ((MsgGlobal.flags & MSGAPI_FLAG_BOUND)) {
        return(MsgGlobal.address.local);
    } else {
        return(0);
    }
}

EXPORT unsigned long
MsgGetHostIPAddress(void)
{
    return(MsgGlobal.address.local);
}

EXPORT const char *
MsgGetUnpriviledgedUser(void)
{
    if (HULA_USER[0] == '\0') {
	return NULL;
    } else {
	return HULA_USER;
    }
}

static void
MsgAddFreeList(unsigned char *ptr)
{
    MsgGlobal.free.list = MemRealloc(MsgGlobal.free.list, (MsgGlobal.free.count+1) * sizeof(unsigned char *));
    if (!MsgGlobal.free.list) {
        return;
    }
    MsgGlobal.free.list[MsgGlobal.free.count]=ptr;
    MsgGlobal.free.count++;
}

static void
MsgConfigMonitor(void)
{
    MDBValueStruct *config;
    unsigned long i;
    unsigned long j;
    unsigned long count;
    unsigned long realUsed;
    MDBValueStruct *servers, *serverContexts, *serverAddresses, *storePath, *serverDN;
    unsigned char nmapDn[MDB_MAX_OBJECT_CHARS + 1];
    unsigned char emptyString[] = " ";
    long prevConfigNumber = 0;
    struct tm *timeStruct;
    time_t utcTime;
    long tmp;

    MsgGlobal.flags |= MSGAPI_FLAG_MONITOR_RUNNING;

    XplRenameThread(XplGetThreadID(), "MsgAPI Config Monitor");

    config = MDBCreateValueStruct(MsgGlobal.directoryHandle, NULL);
    if (MDBRead(MSGSRV_ROOT, MSGSRV_A_CONFIG_CHANGED, config) > 0) {
        prevConfigNumber = atol(config->Value[0]);
    }

    while (!(MsgGlobal.flags & MSGAPI_FLAG_EXITING)) {
        for (i = 0; (i < 300) && !(MsgGlobal.flags & MSGAPI_FLAG_EXITING); i++) {
            XplDelay(1000);
        }

        /* Get Server Time Zone Offset */
        tzset();
        utcTime = time(NULL);
        timeStruct = localtime(&utcTime);
        if (timeStruct) {
            tmp = (((((((timeStruct->tm_year - 70) * 365) + timeStruct->tm_yday) * 24) + timeStruct->tm_hour) * 60) + timeStruct->tm_min) * 60;
            timeStruct = gmtime(&utcTime);
            tmp -= (((((((timeStruct->tm_year - 70) * 365) + timeStruct->tm_yday) * 24) + timeStruct->tm_hour) * 60) + timeStruct->tm_min) * 60;
            MsgDateSetUTCOffset(tmp); 
        }

        MDBFreeValues(config);

        if (!(MsgGlobal.flags & MSGAPI_FLAG_EXITING) && (MDBRead(MSGSRV_ROOT, MSGSRV_A_CONFIG_CHANGED, config)>0) && (atol(config->Value[0]) != prevConfigNumber)) {
            /* Clear what we just read */
            prevConfigNumber = atol(config->Value[0]);
            MDBFreeValues(config);
            
            /* Acquire Write Lock */            
            XplRWWriteLockAcquire(&MsgGlobal.configLock);

            if (MDBRead(MsgGlobal.server.dn, MSGSRV_A_CONNMGR_CONFIG, config)) {
                if (atoi(config->Value[0]) == 1) {
                    MsgGlobal.flags |= MSGAPI_FLAG_SUPPORT_CONNMGR;
                }
            }
            MDBFreeValues(config);

            if (MsgReadIP(MsgGlobal.server.dn, MSGSRV_A_CONNMGR, config)) {
                MsgGlobal.connManager = inet_addr(config->Value[0]);
            }
            MDBFreeValues(config);

            /* First, backup the old structures */
            servers = MsgGlobal.vs.server.names;
            serverContexts = MsgGlobal.vs.server.contexts;
            serverAddresses = MsgGlobal.vs.server.addresses;
            storePath = MsgGlobal.vs.storePaths;
            serverDN = MsgGlobal.vs.storeDNs;

            /* We've got clean globals, and a copy for later comparing */
            MsgGlobal.vs.server.names = MDBCreateValueStruct(MsgGlobal.directoryHandle, NULL);
            MsgGlobal.vs.server.contexts = MDBShareContext(MsgGlobal.vs.server.names);
            MsgGlobal.vs.server.addresses = MDBShareContext(MsgGlobal.vs.server.names);
            MsgGlobal.vs.storePaths = MDBShareContext(MsgGlobal.vs.server.names);
            MsgGlobal.vs.storeDNs = MDBShareContext(MsgGlobal.vs.server.names);

            /* This puts the local context always first */
            sprintf (nmapDn, "%s\\%s", MsgGlobal.server.dn, MSGSRV_AGENT_NMAP);
            if(MDBIsObject(nmapDn, MsgGlobal.vs.server.contexts)) {
                MDBReadDN(MsgGlobal.server.dn, MSGSRV_A_CONTEXT, MsgGlobal.vs.server.contexts);
                for (i = 0; i < MsgGlobal.vs.server.contexts->Used; i++) {
                    MDBRead(MsgGlobal.server.dn, MSGSRV_A_IP_ADDRESS, MsgGlobal.vs.server.addresses);
                    MDBAddValue(MsgGlobal.server.dn, MsgGlobal.vs.storeDNs);
                }
            }

            if (!(MsgGlobal.flags & MSGAPI_FLAG_STANDALONE)) {
                /* First time around we check all "real" classes */
                if (MDBEnumerateObjects(MSGSRV_ROOT, MSGSRV_C_SERVER, NULL, MsgGlobal.vs.server.names)) {
                    /* Put the local context(s) first */
                    for (i = 0; i < MsgGlobal.vs.server.names->Used; i++) {
                        /* MsgGlobal.server.dn is absolute, MsgGlobal.vs.server.names->Value isn't */
                        if (XplStrCaseCmp(MsgGlobal.server.dn+strlen(MsgGlobal.server.dn)-strlen(MsgGlobal.vs.server.names->Value[i]), MsgGlobal.vs.server.names->Value[i]) != 0) {
                            count = MsgGlobal.vs.server.contexts->Used;
                            if ((MDBRead(MsgGlobal.vs.server.names->Value[i], MSGSRV_A_IP_ADDRESS, MsgGlobal.vs.server.addresses)>0) &&
                                (MDBReadDN(MsgGlobal.vs.server.names->Value[i], MSGSRV_A_CONTEXT, MsgGlobal.vs.server.contexts)>0)) {
				MDBAddValue(MsgGlobal.vs.server.names->Value[i], MsgGlobal.vs.storeDNs);
				for (j = count+1; j<MsgGlobal.vs.server.contexts->Used; j++) {
				    MDBAddValue(MsgGlobal.vs.server.addresses->Value[count], MsgGlobal.vs.server.addresses);
				    MDBAddValue(MsgGlobal.vs.server.names->Value[i], MsgGlobal.vs.storeDNs);
				}
                            }
                        }
                    }
                }

                /* Now check any aliases that might be in the Internet Services container */
                realUsed=MsgGlobal.vs.server.names->Used;
                if (MDBEnumerateObjects(MSGSRV_ROOT, C_ALIAS, NULL, MsgGlobal.vs.server.names)) {
                    unsigned char realDN[MDB_MAX_OBJECT_CHARS+1];
                    unsigned char realType[MDB_MAX_ATTRIBUTE_CHARS+1];

                    /* Put the local context(s) first */
                    for (i = realUsed; i < MsgGlobal.vs.server.names->Used; i++) {
                        MDBGetObjectDetails(MsgGlobal.vs.server.names->Value[i], realType, NULL, realDN, MsgGlobal.vs.server.names);

                        /* MsgGlobal.server.dn is absolute, MsgGlobal.vs.server.names->Value isn't */
                        if ((XplStrCaseCmp(realType, MSGSRV_C_SERVER) == 0) && (XplStrCaseCmp(MsgGlobal.server.dn+strlen(MsgGlobal.server.dn) - strlen(realDN), realDN) != 0)) {
                            count = MsgGlobal.vs.server.contexts->Used;
                            if ((MDBRead(MsgGlobal.vs.server.names->Value[i], MSGSRV_A_IP_ADDRESS, MsgGlobal.vs.server.addresses) > 0) &&
                                (MDBReadDN(MsgGlobal.vs.server.names->Value[i], MSGSRV_A_CONTEXT, MsgGlobal.vs.server.contexts) > 0)) {
				MDBAddValue(MsgGlobal.vs.server.names->Value[i], MsgGlobal.vs.storeDNs);
				for (j = count + 1; j < MsgGlobal.vs.server.contexts->Used; j++) {
				    MDBAddValue(MsgGlobal.vs.server.addresses->Value[count], MsgGlobal.vs.server.addresses);
				    MDBAddValue(MsgGlobal.vs.server.names->Value[i], MsgGlobal.vs.storeDNs);
				}
                            }
                        }
                    }
                }
            }

            /* MDBAddValue doesn't add empty values, so fake it */
            /* with a ^A in the string */
            emptyString[0] = EMPTY_CHAR;
            /* Now read all the store path values */
            
            for (i = 0; i < MsgGlobal.vs.server.contexts->Used; i++) {
                if (MDBRead(MsgGlobal.vs.server.contexts->Value[i], MSGSRV_A_MESSAGE_STORE, MsgGlobal.vs.storePaths)==0) {
                    /* Add an empty string, no context store specified */
                    MDBAddValue(emptyString, MsgGlobal.vs.storePaths);
                } else {
                    MsgCleanPath(MsgGlobal.vs.storePaths->Value[i]);
                }
            }

#if 0
            for (i = 0; i < MsgGlobal.vs.server.contexts->Used; i++) {
                XplConsolePrintf("\rCtx:%-35s IP:%-15s Path:%s\n", MsgGlobal.vs.server.contexts->Value[i], MsgGlobal.vs.server.addresses->Value[i], MsgGlobal.vs.storePaths->Value[i]);
                XplConsolePrintf("\rDN :%-35s\n\n", MsgGlobal.vs.storeDNs->Value[i]);
            }
#endif

            /* Now figure out the differences and keep the old pointers if possible */
            for (i = 0; i < MsgGlobal.vs.server.contexts->Used; i++) {
                for (j = 0; j < serverContexts->Used; j++) {
                    if (storePath->Value[j]) {
                        if (strcmp(MsgGlobal.vs.storePaths->Value[i], storePath->Value[j]) == 0) {
                            MemFree(MsgGlobal.vs.storePaths->Value[i]);
                            MsgGlobal.vs.storePaths->Value[i] = storePath->Value[j];
                            storePath->Value[j] = NULL;
                            j = serverContexts->Used;
                        }
                    }
                }
            }

            for (i = 0; i < serverContexts->Used; i++) {
                if (storePath->Value[i]) {
                    MsgAddFreeList(storePath->Value[i]);
                }
            }
            storePath->Used = 0;

            MDBDestroyValueStruct(serverDN);
            MDBDestroyValueStruct(storePath);
            MDBDestroyValueStruct(serverAddresses);
            MDBDestroyValueStruct(serverContexts);
            MDBDestroyValueStruct(servers);

            XplRWWriteLockRelease(&MsgGlobal.configLock);
        }
    }
    MDBDestroyValueStruct(config);
    
    MsgGlobal.flags &= ~MSGAPI_FLAG_MONITOR_RUNNING;

    XplExitThread(TSR_THREAD, 0);
}

static BOOL
MsgLibraryStop(void)
{
    unsigned long i;

    while (MsgGlobal.flags & MSGAPI_FLAG_MONITOR_RUNNING) {
        XplDelay(1000);
    }

    MDBDestroyValueStruct(MsgGlobal.vs.storeDNs);
    MDBDestroyValueStruct(MsgGlobal.vs.storePaths);
    MDBDestroyValueStruct(MsgGlobal.vs.server.addresses);
    MDBDestroyValueStruct(MsgGlobal.vs.server.contexts);
    MDBDestroyValueStruct(MsgGlobal.vs.server.names);

#ifdef PARENTOBJECT_STORE
    MDBDestroyValueStruct(MsgGlobal.vs.parent.stores);
    MDBDestroyValueStruct(MsgGlobal.vs.parent.objects);
#endif

    for (i = 0; i < MsgGlobal.free.count; i++) {
        MemFree(MsgGlobal.free.list[i]);
    }
    MemFree(MsgGlobal.free.list);

    if (MsgGlobal.cache.handle) {
        MsgGlobal.cache.shutdown();

        XplReleaseDLLFunction(MsgGlobal.cache.fileName, "MsgGlobal.cache.init", MsgGlobal.cache.handle);
        XplReleaseDLLFunction(MsgGlobal.cache.fileName, "MsgGlobal.cache.shutdown", MsgGlobal.cache.handle);
        XplReleaseDLLFunction(MsgGlobal.cache.fileName, "FindObjectCache", MsgGlobal.cache.handle);
        XplReleaseDLLFunction(MsgGlobal.cache.fileName, "FindObjectStoreCache", MsgGlobal.cache.handle);

        XplUnloadDLL(MsgGlobal.cache.fileName, MsgGlobal.cache.handle);
    }

    XplRWLockDestroy(&MsgGlobal.configLock);

    return(TRUE);
}

static BOOL
MsgLibraryStart(void)
{
    unsigned long i;
    unsigned long j;
    unsigned long count;
    unsigned long realUsed;
    struct sockaddr_in server_sockaddr;
    unsigned char emptyString[]=" ";
    unsigned char nmapDn[MDB_MAX_OBJECT_CHARS + 1];
    unsigned char path[XPL_MAX_PATH + 1];
    MDBValueStruct *config;
    XplThreadID ID;

    /* Prepare later config updates */
    XplRWLockInit(&MsgGlobal.configLock);

    /* Read generic configuration info */
    config = MDBCreateValueStruct(MsgGlobal.directoryHandle, NULL);

    if (MDBRead(MsgGlobal.server.dn, MSGSRV_A_CONNMGR_CONFIG, config)) {
        if (atoi(config->Value[0]) == 1) {
            MsgGlobal.flags |= MSGAPI_FLAG_SUPPORT_CONNMGR;
        }
    }
    MDBFreeValues(config);

    if (MDBRead(MsgGlobal.server.dn, MSGSRV_A_CERTIFICATE_LOCATION, config)) {
        strcpy(MsgGlobal.paths.certificate, config->Value[0]);
        MsgCleanPath(MsgGlobal.paths.certificate);
    } else {
        if (MDBRead(MSGSRV_ROOT, MSGSRV_A_CERTIFICATE_LOCATION, config)) {
            strcpy(MsgGlobal.paths.certificate, config->Value[0]);
            MsgCleanPath(MsgGlobal.paths.certificate);
        } else {
            strcpy(MsgGlobal.paths.certificate, XPL_DEFAULT_CERT_PATH);
        }
    }
    MDBFreeValues(config);

    if (MDBRead(MsgGlobal.server.dn, MSGSRV_A_PRIVATE_KEY_LOCATION, config)) {
        strcpy(MsgGlobal.paths.key, config->Value[0]);
        MsgCleanPath(MsgGlobal.paths.key);
    } else {
        if (MDBRead(MSGSRV_ROOT, MSGSRV_A_PRIVATE_KEY_LOCATION, config)) {
            strcpy(MsgGlobal.paths.key, config->Value[0]);
            MsgCleanPath(MsgGlobal.paths.key);
        } else {
            strcpy(MsgGlobal.paths.key, XPL_DEFAULT_KEY_PATH);
        }
    }
    MDBFreeValues(config);

    if (MDBRead(MsgGlobal.server.dn, MSGSRV_A_CONFIGURATION, config) > 0) {
        for (i = 0; i < config->Used; i++) {
            if (XplStrCaseCmp(config->Value[i], "Clustered")==0) {
                MsgGlobal.flags |= MSGAPI_FLAG_CLUSTERED;
            }
            if (XplStrCaseCmp(config->Value[i], "Bind:Specified Address")==0) {
                MsgGlobal.flags |= MSGAPI_FLAG_BOUND;
            }
        }
    }
    if (!(MsgGlobal.flags & MSGAPI_FLAG_CLUSTERED)) {
        MsgGlobal.flags &= ~MSGAPI_FLAG_BOUND;
    }
    /* Config is free'd further down */
    MDBFreeValues(config);

    /* Read "context" related stuff */

    MsgGlobal.vs.server.names = MDBCreateValueStruct(MsgGlobal.directoryHandle, NULL);
    MsgGlobal.vs.server.contexts = MDBShareContext(MsgGlobal.vs.server.names);
    MsgGlobal.vs.server.addresses = MDBShareContext(MsgGlobal.vs.server.names);
    MsgGlobal.vs.storePaths = MDBShareContext(MsgGlobal.vs.server.names);
    MsgGlobal.vs.storeDNs = MDBShareContext(MsgGlobal.vs.server.names);

#ifdef PARENTOBJECT_STORE
    MsgGlobal.vs.parent.objects = MDBCreateValueStruct(MsgGlobal.directoryHandle, NULL);
    MsgGlobal.vs.parent.stores = MDBShareContext(MsgGlobal.vs.parent.objects);
#endif

    server_sockaddr.sin_addr.s_addr=XplGetHostIPAddress();
    sprintf(MsgGlobal.address.string,"%d.%d.%d.%d",
	    server_sockaddr.sin_addr.s_net,
	    server_sockaddr.sin_addr.s_host,
	    server_sockaddr.sin_addr.s_lh,
	    server_sockaddr.sin_addr.s_impno);

    MDBAddValue(MsgGlobal.address.string, MsgGlobal.vs.server.names);
    if (!(MsgGlobal.flags & MSGAPI_FLAG_CLUSTERED)) {
        MDBWrite(MsgGlobal.server.dn, MSGSRV_A_IP_ADDRESS, MsgGlobal.vs.server.names);
    } else {
        MDBFreeValues(MsgGlobal.vs.server.names);
        if (MDBRead(MsgGlobal.server.dn, MSGSRV_A_IP_ADDRESS, MsgGlobal.vs.server.names)>0) {
            strcpy(MsgGlobal.address.string, MsgGlobal.vs.server.names->Value[0]);
        }
    }

    MDBFreeValues(MsgGlobal.vs.server.names);
    MsgGlobal.address.local = inet_addr(MsgGlobal.address.string);

    /* This stuff is here to prevent a catch-22 with the IP address configuration above */
    if (MsgReadIP(MsgGlobal.server.dn, MSGSRV_A_CONNMGR, config)) {
        MsgGlobal.connManager = inet_addr(config->Value[0]);
    }

    MDBDestroyValueStruct(config);

    /* This puts the local context always first */
    sprintf(nmapDn, "%s\\%s", MsgGlobal.server.dn, MSGSRV_AGENT_NMAP);
    if(MDBIsObject(nmapDn, MsgGlobal.vs.server.contexts)) {
        MDBReadDN(MsgGlobal.server.dn, MSGSRV_A_CONTEXT, MsgGlobal.vs.server.contexts);
        for (i = 0; i < MsgGlobal.vs.server.contexts->Used; i++) {
            MDBRead(MsgGlobal.server.dn, MSGSRV_A_IP_ADDRESS, MsgGlobal.vs.server.addresses);
            MDBAddValue(MsgGlobal.server.dn, MsgGlobal.vs.storeDNs);
        }
    }

#ifdef PARENTOBJECT_STORE
    /* Read the parent object store configuration */
    if (MDBEnumerateObjects(MSGSRV_ROOT"\\"MSGSRV_PARENT_ROOT, MSGSRV_C_PARENTOBJECT, &MsgGlobal.vs.parent.objects)) {
        for (i=0; i<MsgGlobal.vs.parent.objects.Used; i++) {
            if (MDBRead(MsgGlobal.vs.parent.objects.Value[i], MSGSRV_A_MESSAGE_STORE, &MsgGlobal.vs.parent.stores)==0) {
                MDBAddValue(EmptyString, &MsgGlobal.vs.parent.stores);
            }
        }
    }
#endif

    /* FIXME: This code turns off distributed automatically if the server is not in internet services */
    /* We might not always want this? */
#if 0
    if (strstr(nmapDn, MSGSRV_ROOT)==NULL) {
        MsgGlobal.flags |= MSGAPI_FLAG_STANDALONE;
    }
#endif

    if (!(MsgGlobal.flags & MSGAPI_FLAG_STANDALONE)) {
        /* First time around we check all "real" classes */
        if (MDBEnumerateObjects(MSGSRV_ROOT, MSGSRV_C_SERVER, NULL, MsgGlobal.vs.server.names)) {
            /* Put the local context(s) first */
            for (i = 0; i < MsgGlobal.vs.server.names->Used; i++) {
                /* MsgGlobal.server.dn is absolute, MsgGlobal.vs.server.names->Value isn't */
                if (XplStrCaseCmp(MsgGlobal.server.dn + strlen(MsgGlobal.server.dn) - strlen(MsgGlobal.vs.server.names->Value[i]), MsgGlobal.vs.server.names->Value[i]) != 0) {
                    count = MsgGlobal.vs.server.contexts->Used;
                    if ((MDBRead(MsgGlobal.vs.server.names->Value[i], MSGSRV_A_IP_ADDRESS, MsgGlobal.vs.server.addresses) > 0) &&
                        (MDBReadDN(MsgGlobal.vs.server.names->Value[i], MSGSRV_A_CONTEXT, MsgGlobal.vs.server.contexts) > 0)) {
			MDBAddValue(MsgGlobal.vs.server.names->Value[i], MsgGlobal.vs.storeDNs);
			for (j = count + 1; j < MsgGlobal.vs.server.contexts->Used; j++) {
			    MDBAddValue(MsgGlobal.vs.server.addresses->Value[count], MsgGlobal.vs.server.addresses);
			    MDBAddValue(MsgGlobal.vs.server.names->Value[i], MsgGlobal.vs.storeDNs);
			}
                    }
                }
            }
        }

        /* Now check any aliases that might be in the Internet Services container */
        realUsed = MsgGlobal.vs.server.names->Used;
        if (MDBEnumerateObjects(MSGSRV_ROOT, C_ALIAS, NULL, MsgGlobal.vs.server.names)) {
            unsigned char realDn[MDB_MAX_OBJECT_CHARS + 1];
            unsigned char realType[MDB_MAX_ATTRIBUTE_CHARS + 1];

            /* Put the local context(s) first */
            for (i = realUsed; i < MsgGlobal.vs.server.names->Used; i++) {
                MDBGetObjectDetails(MsgGlobal.vs.server.names->Value[i], realType, NULL, realDn, MsgGlobal.vs.server.names);

                /* MsgGlobal.server.dn is absolute, MsgGlobal.vs.server.names->Value isn't */
                if ((XplStrCaseCmp(realType, MSGSRV_C_SERVER) == 0) && (XplStrCaseCmp(MsgGlobal.server.dn + strlen(MsgGlobal.server.dn) - strlen(realDn), realDn) != 0)) {
                    count = MsgGlobal.vs.server.contexts->Used;
                    if ((MDBRead(MsgGlobal.vs.server.names->Value[i], MSGSRV_A_IP_ADDRESS, MsgGlobal.vs.server.addresses) > 0) &&
                        (MDBReadDN(MsgGlobal.vs.server.names->Value[i], MSGSRV_A_CONTEXT, MsgGlobal.vs.server.contexts) > 0)) {
			MDBAddValue(MsgGlobal.vs.server.names->Value[i], MsgGlobal.vs.storeDNs);
			for (j = count + 1; j < MsgGlobal.vs.server.contexts->Used; j++) {
			    MDBAddValue(MsgGlobal.vs.server.addresses->Value[count], MsgGlobal.vs.server.addresses);
			    MDBAddValue(MsgGlobal.vs.server.names->Value[i], MsgGlobal.vs.storeDNs);
			}
                    }
                }
            }
        }
    }

    /* MDBAddValue doesn't add empty values, so fake it */
    /* with a ^A in the string */
    emptyString[0]=EMPTY_CHAR;
    /* Now read all the store path values */
    
    for (i = 0; i < MsgGlobal.vs.server.contexts->Used; i++) {
        if (MDBRead(MsgGlobal.vs.server.contexts->Value[i], MSGSRV_A_MESSAGE_STORE, MsgGlobal.vs.storePaths) == 0) {
            /* Add an empty string, no context store specified */
            MDBAddValue(emptyString, MsgGlobal.vs.storePaths);
        } else {
            MsgCleanPath(MsgGlobal.vs.storePaths->Value[i]);
        }
    }

    XplBeginThread(&ID, MsgConfigMonitor, 32767, NULL, i);

    /*    Attempt to load the plugable cache mechanism    */
    sprintf(MsgGlobal.cache.fileName, "%s%s", MsgGlobal.cache.moduleName, XPL_DLL_EXTENSION);
    sprintf(path, "%s/%s", XPL_DEFAULT_BIN_DIR, MsgGlobal.cache.fileName);

    MsgGlobal.cache.handle = XplLoadDLL(path);
    if (MsgGlobal.cache.handle != NULL) {
        MsgGlobal.cache.init = (FindObjectCacheInitType)XplGetDLLFunction(MsgGlobal.cache.fileName, "MsgGlobal.cache.init", MsgGlobal.cache.handle);
        MsgGlobal.cache.shutdown = (FindObjectCacheShutdownType)XplGetDLLFunction(MsgGlobal.cache.fileName, "MsgGlobal.cache.shutdown", MsgGlobal.cache.handle);
        MsgGlobal.cache.find = (FindObjectCacheType)XplGetDLLFunction(MsgGlobal.cache.fileName, "FindObjectCache", MsgGlobal.cache.handle);
        MsgGlobal.cache.findEx = (FindObjectCacheExType)XplGetDLLFunction(MsgGlobal.cache.fileName, "FindObjectExCache", MsgGlobal.cache.handle);
        MsgGlobal.cache.store = (FindObjectStoreCacheType)XplGetDLLFunction(MsgGlobal.cache.fileName, "FindObjectStoreCache", MsgGlobal.cache.handle);

        if (!MsgGlobal.cache.init || !MsgGlobal.cache.shutdown || !MsgGlobal.cache.find || !MsgGlobal.cache.findEx || !MsgGlobal.cache.store) {
            MsgGlobal.cache.init = NULL;
            MsgGlobal.cache.shutdown = NULL;
            MsgGlobal.cache.find = NULL;
            MsgGlobal.cache.findEx = NULL;
            MsgGlobal.cache.store = NULL;
        }

        if (MsgGlobal.cache.init) {
            MSGCacheInitStruct initData;

            initData.DirectoryHandle = MsgGlobal.directoryHandle;
            initData.ServerContexts = MsgGlobal.vs.server.contexts;
            initData.ServerAddresses = MsgGlobal.vs.server.addresses;
            initData.StorePath = MsgGlobal.vs.storePaths;
            initData.ConfigLock = &MsgGlobal.configLock;
            initData.DefaultFindObject = MsgFindObject;
            initData.DefaultFindObjectEx = MsgFindObjectEx;
            initData.DefaultFindObjectStore = MsgFindUserStore;
            initData.DefaultPathChar = EMPTY_CHAR;

	    if (!MsgGlobal.cache.init(&initData, path)) {
                MsgGlobal.cache.init = NULL;
                MsgGlobal.cache.shutdown = NULL;
                MsgGlobal.cache.find = NULL;
                MsgGlobal.cache.findEx = NULL;
                MsgGlobal.cache.store = NULL;
            }

        }
    }

    return(TRUE);
}

static BOOL
MsgReadConfiguration(void)
{
    MDBValueStruct *config;
    unsigned char serverDn[MDB_MAX_OBJECT_CHARS + 1];
    unsigned char credential[128];
    unsigned long i;
    struct tm *timeStruct;
    time_t utcTime;
    long tmp;
    FILE *eclients;

    /* Get Server Time Zone Offset */
    tzset();
    utcTime = time(NULL);
    timeStruct = localtime(&utcTime);
    if (timeStruct) {
        tmp = (((((((timeStruct->tm_year - 70) * 365) + timeStruct->tm_yday) * 24) + timeStruct->tm_hour) * 60) + timeStruct->tm_min) * 60;
        timeStruct = gmtime(&utcTime);
        tmp -=    (((((((timeStruct->tm_year - 70) * 365) + timeStruct->tm_yday) * 24) + timeStruct->tm_hour) * 60) + timeStruct->tm_min) * 60;

        MsgDateSetUTCOffset(tmp); 
    }

    memset(credential, 0, sizeof(credential));

    sprintf(serverDn, "%s/eclients.dat", XPL_DEFAULT_DBF_DIR);
    eclients = fopen(serverDn, "rb");
    if (eclients) {
        fread(credential, sizeof(unsigned char), sizeof(credential), eclients);

        fclose(eclients);
        eclients = NULL;
    } else {
        XplConsolePrintf("Insufficient privileges; shutting down.");
    }

    /* First, find out who we are. */
    if (!MDBGetServerInfo(serverDn, MsgGlobal.server.tree, NULL)) {
        XplConsolePrintf("Configuration failure; shutting down.");
        return(FALSE);
    }

    /* Use [Public] rights to read the host's messaging server. */
    MsgGlobal.directoryHandle = MDBAuthenticate("Hula", NULL, NULL);

    config = MDBCreateValueStruct(MsgGlobal.directoryHandle, NULL);
    if (!config) {
        XplConsolePrintf("Messaging server out of memory; shutting down.");
        MDBRelease(MsgGlobal.directoryHandle);
        MsgGlobal.directoryHandle = NULL;
        return(FALSE);
    }

    if (MDBReadDN(serverDn, MSGSRV_A_HULA_MESSAGING_SERVER, config)<1) {
        XplConsolePrintf("Messaging server not configured. Shutdown.");
        MDBRelease(MsgGlobal.directoryHandle);
        MsgGlobal.directoryHandle = NULL;
        return(FALSE);
    } else {
        strcpy(MsgGlobal.server.dn, config->Value[0]);
    }

    MDBDestroyValueStruct(config);

    MDBRelease(MsgGlobal.directoryHandle);

    MsgGlobal.directoryHandle = MDBAuthenticate("Hula", MsgGlobal.server.dn, credential);
    if (MsgGlobal.directoryHandle == NULL) {
        XplConsolePrintf("Messaging server credentials are invalid; shutting down.");
        MDBRelease(MsgGlobal.directoryHandle);
        MsgGlobal.directoryHandle = NULL;
        return(FALSE);
    }

    /* Read operating parameters, this is so complicated because in version 2.5
     * we changed the configuration of directories - before 2.5 we would always
     * append novonyx/mail to any given path, now we don't. The code tries to
     * automatically detect pre2.5 installs and change the DS attribute...
     */

    config = MDBCreateValueStruct(MsgGlobal.directoryHandle, NULL);

    if (!config) {
        XplConsolePrintf("Messaging server out of memory; shutting down.");
        MDBRelease(MsgGlobal.directoryHandle);
        MsgGlobal.directoryHandle = NULL;
        return(FALSE);
    }

    if (MDBRead(MsgGlobal.server.dn, MSGSRV_A_NLS_DIRECTORY, config)) {
        strcpy(MsgGlobal.paths.nls, config->Value[0]);
        MsgCleanPath(MsgGlobal.paths.nls);
        MsgMakePath(MsgGlobal.paths.nls);
        MDBFreeValues(config);
        if (MDBRead(MsgGlobal.server.dn, MSGSRV_A_DBF_DIRECTORY, config)) {
            strcpy(MsgGlobal.paths.dbf, config->Value[0]);
            MsgCleanPath(MsgGlobal.paths.dbf);
            MsgMakePath(MsgGlobal.paths.dbf);
            MDBFreeValues(config);
        } else {
            strcpy(MsgGlobal.paths.dbf, XPL_DEFAULT_DBF_DIR);
            MsgMakePath(MsgGlobal.paths.dbf);
        }

        if (MDBRead(MsgGlobal.server.dn, MSGSRV_A_BIN_DIRECTORY, config)) {
            strcpy(MsgGlobal.paths.bin, config->Value[0]);
            MsgCleanPath(MsgGlobal.paths.bin);
            MsgMakePath(MsgGlobal.paths.bin);
            MDBFreeValues(config);
        } else {
            strcpy(MsgGlobal.paths.bin, XPL_DEFAULT_BIN_DIR);
            MsgMakePath(MsgGlobal.paths.bin);
        }

        if (MDBRead(MsgGlobal.server.dn, MSGSRV_A_LIB_DIRECTORY, config)) {
            strcpy(MsgGlobal.paths.lib, config->Value[0]);
            MsgCleanPath(MsgGlobal.paths.lib);
            MsgMakePath(MsgGlobal.paths.lib);
            MDBFreeValues(config);
        } else {
            strcpy(MsgGlobal.paths.lib, XPL_DEFAULT_LIB_DIR);
            MsgMakePath(MsgGlobal.paths.lib);
        }

        if (MDBRead(MsgGlobal.server.dn, MSGSRV_A_WORK_DIRECTORY, config)) {
            strcpy(MsgGlobal.paths.work, config->Value[0]);
            MsgCleanPath(MsgGlobal.paths.work);
            MsgMakePath(MsgGlobal.paths.work);
            MDBFreeValues(config);
        } else {
            strcpy(MsgGlobal.paths.work, XPL_DEFAULT_WORK_DIR);
            MsgMakePath(MsgGlobal.paths.work);
        }
    } else {
        strcpy(MsgGlobal.paths.dbf, XPL_DEFAULT_DBF_DIR);
        MsgMakePath(MsgGlobal.paths.dbf);
        MDBAddValue(MsgGlobal.paths.dbf, config);
        MDBWrite(MsgGlobal.server.dn, MSGSRV_A_DBF_DIRECTORY, config);
        MDBFreeValues(config);

        strcpy(MsgGlobal.paths.nls, XPL_DEFAULT_NLS_DIR);
        MsgMakePath(MsgGlobal.paths.nls);
        MDBAddValue(MsgGlobal.paths.nls, config);
        MDBWrite(MsgGlobal.server.dn, MSGSRV_A_NLS_DIRECTORY, config);
        MDBFreeValues(config);

        strcpy(MsgGlobal.paths.bin, XPL_DEFAULT_BIN_DIR);
        MsgMakePath(MsgGlobal.paths.bin);
        MDBAddValue(MsgGlobal.paths.bin, config);
        MDBWrite(MsgGlobal.server.dn, MSGSRV_A_BIN_DIRECTORY, config);
        MDBFreeValues(config);

        if (MDBRead(MsgGlobal.server.dn, MSGSRV_A_WORK_DIRECTORY, config)) {
            strcpy(MsgGlobal.paths.work, config->Value[0]);
            MsgCleanPath(MsgGlobal.paths.work);
            strcat(MsgGlobal.paths.work, "/novonyx/mail");
            MsgMakePath(MsgGlobal.paths.work);
            MDBFreeValues(config);

            MDBAddValue(MsgGlobal.paths.work, config);
            MDBWrite(MsgGlobal.server.dn, MSGSRV_A_WORK_DIRECTORY, config);
            MDBFreeValues(config);
        } else {
            strcpy(MsgGlobal.paths.work, XPL_DEFAULT_WORK_DIR);
            MsgMakePath(MsgGlobal.paths.work);
            MDBAddValue(MsgGlobal.paths.work, config);
            MDBWrite(MsgGlobal.server.dn, MSGSRV_A_WORK_DIRECTORY, config);
            MDBFreeValues(config);
        }
    }

    /* Official Name */
    if (MDBRead(MsgGlobal.server.dn, MSGSRV_A_OFFICIAL_NAME, config)) {
        strcpy(MsgGlobal.official.domain, config->Value[0]);
        MsgGlobal.official.domainLength = strlen(MsgGlobal.official.domain);
    }
    MDBFreeValues(config);

    if (MDBRead(MsgGlobal.server.dn, MSGSRV_A_CONFIGURATION, config)) {
        for (i = 0; i < config->Used; i++) {
            if (XplStrNCaseCmp(config->Value[i], "FindObjectModule=", 17) == 0) {
                strcpy(MsgGlobal.cache.moduleName, config->Value[i]+17);
            } else if (XplStrNCaseCmp(config->Value[i], "IgnoreInternetEmailAddressAttribute", 35) == 0) {
                MsgGlobal.flags |= MSGAPI_FLAG_INTERNET_EMAIL_ADDRESS;
            }
        }
    }
    MDBFreeValues(config);

    if (MDBRead(MsgGlobal.server.dn, MSGSRV_A_SERVER_STANDALONE, config)) {
        if (config->Value[0][0]=='1') {
            MsgGlobal.flags |= MSGAPI_FLAG_STANDALONE;
        }
    }
    MDBDestroyValueStruct(config);

    return(TRUE);
}

static int
MsgLibraryInit(void)
{
    MsgGlobal.flags = 0;

    XplSafeWrite(MsgGlobal.useCount, 0);

    MsgGlobal.groupID = XplGetThreadGroupID();

    MsgGlobal.directoryHandle = NULL;

    MsgGlobal.connManager = 0x0100007F;

    MsgGlobal.free.count = 0;
    MsgGlobal.free.list = NULL;

    strcpy(MsgGlobal.cache.moduleName, "msgcache");
    MsgGlobal.cache.init = NULL;
    MsgGlobal.cache.shutdown = NULL;
    MsgGlobal.cache.find = NULL;
    MsgGlobal.cache.findEx = NULL;
    MsgGlobal.cache.store = NULL;

    MsgGlobal.official.domain[0] = '\0';
    MsgGlobal.official.domainLength = 0;

    XplOpenLocalSemaphore(MsgGlobal.sem.shutdown, 0);

    MemoryManagerOpen(NULL);

    MDBInit();

    if (!MsgReadConfiguration()) {
        XplConsolePrintf("Cannot read configuration. Shutting down");

        MemoryManagerClose(NULL);

        return(0);
    }

    MsgResolveStart();
    MsgLibraryStart();
    MsgDateStart();

    MSGAPIState = LIBRARY_RUNNING;

    return(0);
}

static void 
MsgLibraryShutdown(void)
{
    int oldGid;

    if (MSGAPIState == LIBRARY_RUNNING) {
        MSGAPIState = LIBRARY_SHUTTING_DOWN;

        MsgGlobal.flags |= MSGAPI_FLAG_EXITING;

        oldGid = XplSetThreadGroupID(MsgGlobal.groupID);

        MsgResolveStop();
        MsgLibraryStop();

        MemoryManagerClose(NULL);

        MSGAPIState = LIBRARY_SHUTDOWN;

        XplSetThreadGroupID(oldGid);
    }

    return;
}

MDBHandle 
MsgDirectoryHandle(void)
{
    return(MsgGlobal.directoryHandle);
}

BOOL
MsgResolveStart(void)
{
    MDBValueStruct *config;
    unsigned long i;

    if (!XplResolveStart()) {
	return FALSE;
    }
    
    config = MDBCreateValueStruct(MsgDirectoryHandle(), NULL);
    if (MDBRead(MsgGetServerDN(NULL), MSGSRV_A_RESOLVER, config) > 0) {
        for (i = 0; i < config->Used; i++) {
            XplDnsAddResolver(config->Value[i]);
/*            XplConsolePrintf("[%04d] MSGSRV_A_RESOLVER:%s", __LINE__, config->Value[i]);*/
        }
    } else {
        MDBSetValueStructContext(MsgGetServerDN(NULL), config);
        if (MDBRead(MSGSRV_AGENT_SMTP, MSGSRV_A_RESOLVER, config)>0) {
            for (i = 0; i < config->Used; i++) {
                XplDnsAddResolver(config->Value[i]);
/*		XplConsolePrintf("[%04d] MSGSRV_A_RESOLVER:%s", __LINE__, config->Value[i]);*/
            }
            MDBSetValueStructContext(NULL, config);
            MDBWrite(MsgGetServerDN(NULL), MSGSRV_A_RESOLVER, config);
        }
    }
    MDBDestroyValueStruct(config);

    return(TRUE);
}

BOOL
MsgResolveStop(void)
{
    return XplResolveStop();
}



BOOL 
MsgExiting(void)
{
    if (MsgGlobal.flags & MSGAPI_FLAG_EXITING) {
        return(TRUE);
    }

    return(FALSE);
}

EXPORT BOOL 
MsgShutdown(void)
{
    XplSafeDecrement(MsgGlobal.useCount);

    if (XplSafeRead(MsgGlobal.useCount) > 0) {
        return(FALSE);
    }

    MsgLibraryShutdown();

    return(TRUE);
}

EXPORT int 
MsgInit(void)
{
    if (MSGAPIState == LIBRARY_LOADED) {
        MSGAPIState = LIBRARY_INITIALIZING;

        MsgLibraryInit();
    }

    while (MSGAPIState < LIBRARY_RUNNING) {
        XplDelay(100);
    }

    if (MSGAPIState == LIBRARY_RUNNING) {
        XplSafeIncrement(MsgGlobal.useCount);
    }

    return((MSGAPIState == LIBRARY_RUNNING) ? (int)MsgGlobal.directoryHandle : 0);
}

/*  fixme - platform service implementation! */
#if 0
#if defined(NETWARE) || defined(LIBC)
XplThreadID                            TGid;

void MSGAPIShutdownSigHandler(int sigtype)
{
    static BOOL signaled = FALSE;
    XplThreadID oldTGID;

    if (!signaled) {
        oldTGID = XplSetThreadGroupID(TGid);    

        /*    Wake up the main thread to let him know we are shutting down.
	      Then perform a context (thread) switch to let him get out.
	      Finally, wait for him to signal the fact that he is gone.    */
        XplSignalLocalSemaphore(MsgGlobal.sem.shutdown);
        XplWaitOnLocalSemaphore(MsgGlobal.sem.shutdown);

        XplCloseLocalSemaphore(MsgGlobal.sem.shutdown);

        XplSetThreadGroupID(oldTGID);    
    }

    return;
}

int _NonAppCheckUnload(void)
{
    if (XplSafeRead(MsgGlobal.useCount) == 1) {
        MSGAPIShutdownSigHandler(SIGTERM);

        return(0);
    }

    XplConsolePrintf("MSGAPI: %d library users outstanding; cannot unload.\r\n", XplSafeRead(MsgGlobal.useCount));

    return(1);
}

int
main(int argc, char *argv[])
{
#if !defined(NETWARE)
    int            ccode;
    XplThreadID    id;
#endif

    MSGAPIState = LIBRARY_INITIALIZING;

    TGid = XplGetThreadGroupID();

    XplOpenLocalSemaphore(MsgGlobal.sem.version, 1);

#if !defined(NETWARE)
    XplSignalHandler(MSGAPIShutdownSigHandler);
#endif

    MsgLibraryInit();

    XplWaitOnLocalSemaphore(MsgGlobal.sem.shutdown);

    /*    Make sure the library users are gone before beginning to 
	  shutdown.    */
    while (XplSafeRead(MsgGlobal.useCount) > 1) {
        XplDelay(333);
    }

    MsgShutdown();

    XplCloseLocalSemaphore(MsgGlobal.sem.version);

    XplSignalLocalSemaphore(MsgGlobal.sem.shutdown);

    return(0);
}
#endif

#if defined(WIN32)
HANDLE    StatsMap;
HANDLE    SpamMap;
#define    MSGAPI_MEMORYMAP_STATS    "MSGAPIStats"
#define    MSGAPI_MEMORYMAP_SPAM    "MSGAPISpamStats"

BOOL WINAPI
DllMain(HINSTANCE hInst, DWORD Reason, LPVOID Reserved)
{
    if (Reason == DLL_PROCESS_ATTACH) {
        DisableThreadLibraryCalls(hInst);
    }

    return(TRUE);
}
#endif
#endif
