/****************************************************************************
 *
 * Copyright (c) 1997-2004 Novell, Inc.
 * All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of version 2.1 of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, contact Novell, Inc.
 *
 * To contact Novell about this file by physical or electronic mail,
 * you may find current contact information at www.novell.com
 *
 ****************************************************************************/

#include <config.h>
#include <xpl.h>
#include <connio.h>
#include <hulautil.h>
#include <memmgr.h>
#include <mdb.h>
#include <msgapi.h>
#include <management.h>

#define MAX_CONNECT_ATTEMPTS 120

struct {
    enum ManagementStates state;

    struct {
        unsigned char dn[MDB_MAX_OBJECT_CHARS + 1];
    } host;

    struct {
        unsigned short port;

        unsigned char dn[MDB_MAX_OBJECT_CHARS + 1];
        unsigned char credential[NMAP_HASH_SIZE + 1];
    } dmc;

    struct {
        Connection *conn;

        XplAtomic active;

        unsigned char dn[MDB_MAX_OBJECT_CHARS + 1];
        unsigned char signature[36];

        struct {
            unsigned long count;

            ManagementVariables *data;
        } variables;

        struct {
            unsigned long count;

            ManagementCommands *data;
        } commands;
    } server;

    MDBHandle directoryHandle;
} Management;

void 
MsgSendTrap(int TrapType, unsigned char *TrapMessage)
{
    int ccode;
    unsigned long length;
    unsigned char buffer[CONN_BUFSIZE + 1];
    Connection *conn = NULL;

    if (TrapMessage) {
        length = strlen(TrapMessage);

        conn = ConnAlloc(TRUE);
        if (conn) {
            ccode = ConnSocket(conn);

            if (ccode != -1) {
			    conn->socketAddress.sin_family = AF_INET;
			    conn->socketAddress.sin_addr.s_addr = MsgGetHostIPAddress();
			    conn->socketAddress.sin_port = Management.dmc.port;

                ccode = ConnConnect(conn, NULL, 0, NULL);
            }

            if (ccode != -1) {
                ccode = ConnRead(conn, buffer, sizeof(buffer));
            }

            if (ccode != -1) {
                ccode = ConnWriteF(conn, "TRAP %s %lu\r\n", Management.server.signature, length);
            }

            if (ccode != -1) {
                ccode = ConnFlush(conn);
            }

            if (ccode != -1) {
                ccode = ConnRead(conn, buffer, sizeof(buffer));
            }

            if (ccode != -1) {
                if (XplStrNCaseCmp(buffer, DMCMSG1003SEND_BYTES, 10) == 0) {
                    ccode = ConnWrite(conn, TrapMessage, length);
                } else {
                    ccode = -1;
                }
            }

            ConnWrite(conn, "QUIT\r\n", 6);
            ConnFlush(conn);
            ConnRead(conn, buffer, sizeof(buffer));
            ConnClose(conn, 0);
            ConnFree(conn);
            conn = NULL;
        }
    }

    return;
}

BOOL 
ManagementMemoryStats(unsigned char *Arguments, unsigned char **Response, BOOL *CloseConnection)
{
    int length;
    unsigned long x;
    unsigned long y;
    unsigned long total = 0;
    unsigned char *base;
    unsigned char *cur;
    BOOL result = TRUE;
    MemStatistics *stats = NULL;
    MemStatistics *next;

    if (CloseConnection) {
        *CloseConnection = FALSE;
    }

    if (Response) {
        if (!Arguments) {
            stats = MemAllocStatistics();
            if (stats) {
                base = cur = NULL;
                length = 0;
                next = stats;

                do {
                    cur = MemRealloc(base, length + 192);
                    if (cur) {
                        base = cur;
                        cur = base + length;

                        total += next->totalAlloc.size;

                        if (next->pitches) {
                            x = (next->hits * 100) / next->pitches;
                            y = (next->hits * 100) % next->pitches;
                        } else {
                            x = y = 0;
                        }

                        length += sprintf(cur, "%s: %lu [%lu@%lu] %lum %luM %lu.%02lu%%\r\n", next->name, next->totalAlloc.size, next->entry.allocated, next->entry.size, next->entry.minimum, next->entry.maximum, x, y);

                        next = next->next;

                        continue;
                    }

                    MemFree(base);

                    base = MemStrdup("Out of memory.\r\n");

                    result = FALSE;

                    break;
                } while (next);

                if (result) {
                    cur = MemRealloc(base, length + 32);
                    if (cur) {
                        base = cur;
                        cur = base + length;

                        length += sprintf(cur, "Total allocated = %lu\r\n", total);
                    }
                }

                *Response = base;

                MemFreeStatistics(stats);
            } else {
                *Response = MemStrdup("Out of memory.\r\n");
            }
        } else {
            *Response = MemStrdup("Arguments not allowed.\r\n");
        }

        if (*Response) {
            return(TRUE);
        }
    }

    return(FALSE);
}

static void 
HandleManagementClient(void *param)
{
    int ccode;
    size_t count;
    unsigned long index = 0;
    unsigned long length;
    size_t bufferSize = 0;
    size_t readSize;
    unsigned char *response = NULL;
    unsigned char *ptr;
    unsigned char *ptr2;
    unsigned char buffer[CONN_BUFSIZE + 1];
    BOOL close = FALSE;
    Connection *conn = param;

    if (conn) {
        /* Set TCP non blocking io */
        ccode = 1;
        setsockopt(conn->socket, IPPROTO_TCP, 1, (unsigned char *)&ccode, sizeof(ccode));
    } else {
        XplSafeDecrement(Management.server.active);

        XplExitThread(TSR_THREAD, 0);

        return;
    }

    if (XplIsLocalIPAddress(conn->socketAddress.sin_addr.s_addr) == FALSE) {
        ConnClose(conn, 0);
        ConnFree(conn);
        conn = NULL;

        XplSafeDecrement(Management.server.active);

        XplExitThread(TSR_THREAD, 0);
    }

    ConnWrite(conn, DMCMSG1000OK, DMCMSG1000OK_LEN);
    ConnFlush(conn);

    do {
        if (ConnRead(conn, buffer, sizeof(buffer)) == -1) {
            break;
        }

        /*    Process Client request    */
        switch(toupper(buffer[0])) {
            case 'C': {
                /* CMD <CommandIndex>[ Arguments] */
                if (buffer[3] == ' ') {
                    ptr = strchr(buffer + 4, ' ');
                    if (ptr) {
                        *ptr++ = '\0';
                    }

                    index = atol(buffer + 4);
                    if (index < Management.server.commands.count) {
                        ptr2 = NULL;
                        if (Management.server.commands.data[index].Execute(ptr, &ptr2, &close)) {
                            if (ptr2) {
                                count = strlen(ptr2);
                            } else {
                                count = 0;
                            }

                            ConnWriteF(conn, DMCMSG1001BYTES_COMING, count);
                            ConnWrite(conn, ptr2, count);
                            ConnWrite(conn, DMCMSG1000OK, DMCMSG1000OK_LEN);

                            if (ptr2) {
                                MemFree(ptr2);
                                ptr2 = NULL;
                            }

                            break;
                        }
                    }

                    ConnWrite(conn, DMCMSG2101BADPARAMS, DMCMSG2101BADPARAMS_LEN);
                    break;
                }

                ConnWrite(conn, DMCMSG2100BADCOMMAND, DMCMSG2100BADCOMMAND_LEN);
                break;
            }

            case 'E': {
                /* CORE */
                if (strcmp(buffer, "EROC") == 0) {
                    /*    Induce a segmentation fault.    */
                    ptr = NULL;
                    *ptr = 'C';
                }

                ConnWrite(conn, DMCMSG2100BADCOMMAND, DMCMSG2100BADCOMMAND_LEN);
                break;
            }

            case 'H': {
                /* HELP <VariableIndex> */
                if (buffer[4] == ' ') {
                    index = atol(buffer + 5);
                    if (index < Management.server.variables.count) {
                        count = strlen(Management.server.variables.data[index].Help);

                        ConnWriteF(conn, DMCMSG1001BYTES_COMING, count);
                        ConnWrite(conn, Management.server.variables.data[index].Help, count);
                        ConnWrite(conn, DMCMSG1000OK, DMCMSG1000OK_LEN);

                        break;
                    }

                    ConnWrite(conn, DMCMSG2101BADPARAMS, DMCMSG2101BADPARAMS_LEN);
                    break;
                }

                ConnWrite(conn, DMCMSG2100BADCOMMAND, DMCMSG2100BADCOMMAND_LEN);
                break;
            }

            case 'Q': {
                /* QUIT */
                close = TRUE;
                ConnWrite(conn, DMCMSG1000OK, DMCMSG1000OK_LEN);
                continue;
            }

            case 'R': {
                /* READ <VariableIndex> */
                if (buffer[4] == ' ') {
                    index = atol(buffer + 5);
                    if (index < Management.server.variables.count) {
                        do {
                            readSize = bufferSize;
                            if (Management.server.variables.data[index].Read(index, response, &readSize)) {
                                if (readSize < bufferSize) {
                                    ConnWriteF(conn, DMCMSG1001BYTES_COMING, readSize);
                                    ConnWrite(conn, response, readSize);
                                    ConnWrite(conn, DMCMSG1000OK, DMCMSG1000OK_LEN);
                                    break;
                                }

                                ptr = (unsigned char *)MemRealloc(response, readSize + 32);
                                if (ptr) {
                                    response = ptr;
                                    bufferSize = readSize + 32;
                                    continue;
                                }

                                ConnWrite(conn, DMCMSG2000NOMEMORY, DMCMSG2000NOMEMORY_LEN);
                                break;
                            }

                            ConnWrite(conn, DMCMSG2002READ_ERROR, DMCMSG2002READ_ERROR_LEN);
                            break;
                        } while (TRUE);

                        break;
                    }

                    ConnWrite(conn, DMCMSG2101BADPARAMS, DMCMSG2101BADPARAMS_LEN);
                    break;
                }

                ConnWrite(conn, DMCMSG2100BADCOMMAND, DMCMSG2100BADCOMMAND_LEN);
                break;
            }

            case 'W': {
                /* WRITE <VariableIndex> <Length> */
                if ((buffer[5] == ' ') && (isdigit(buffer[6]))) {
                    ptr = strchr(buffer + 6, ' ');
                    if (ptr && (isdigit(ptr[1]))) {
                        *ptr++ = '\0';
                        index = atol(buffer + 6);
                        readSize = atol(ptr);

                        if (index < Management.server.variables.count) {
                            if (Management.server.variables.data[index].Write) {
                                do {
                                    if (readSize < bufferSize) {
                                        ConnWrite(conn, DMCMSG1003SEND_VALUE, DMCMSG1003SEND_VALUE_LEN);
                                        ConnFlush(conn);

                                        ConnReadCount(conn, response, readSize);

                                        if (Management.server.variables.data[index].Write(index, response, readSize)) {
                                            ConnWrite(conn, DMCMSG1000OK, DMCMSG1000OK_LEN);
                                            break;
                                        }

                                        ConnWrite(conn, DMCMSG2002WRITE_ERROR, DMCMSG2002WRITE_ERROR_LEN);
                                        break;
                                    }

                                    ptr = (unsigned char *)MemRealloc(response, readSize + 32);
                                    if (ptr) {
                                        response = ptr;
                                        bufferSize = readSize + 32;
                                        continue;
                                    }

                                    ConnWrite(conn, DMCMSG2000NOMEMORY, DMCMSG2000NOMEMORY_LEN);
                                    break;
                                } while (TRUE);

                                break;
                            }

                            ConnWrite(conn, DMCMSG2104NOT_WRITEABLE, DMCMSG2104NOT_WRITEABLE_LEN);
                            break;
                        }
                    }

                    ConnWrite(conn, DMCMSG2101BADPARAMS, DMCMSG2101BADPARAMS_LEN);
                    break;
                }

                ConnWrite(conn, DMCMSG2100BADCOMMAND, DMCMSG2100BADCOMMAND_LEN);
                break;
            }

            default: {
                ConnWrite(conn, DMCMSG2100BADCOMMAND, DMCMSG2100BADCOMMAND_LEN);
                break;
            }
        }

        ConnFlush(conn);
    } while ((Management.state == MANAGEMENT_RUNNING) && !close);

    ConnClose(conn, 0);

    if (response != NULL) {
        MemFree(response);
        response = NULL;
    }

    XplSafeDecrement(Management.server.active);

    ConnFree(conn);
    conn = NULL;

    XplExitThread(TSR_THREAD, 0);
    return;
}

BOOL 
ManagementInit(const unsigned char *Identity, MDBHandle DirectoryHandle)
{
    unsigned long used;
    MDBValueStruct *config = NULL;
    BOOL result = FALSE;

    Management.state = MANAGEMENT_LOADING;

    XplSafeWrite(Management.server.active, 0);

    Management.host.dn[0] = '\0';
    Management.dmc.dn[0] = '\0';
    Management.server.dn[0] = '\0';

    memset(Management.dmc.credential, 0, sizeof(Management.dmc.credential));
    memset(Management.server.signature, 0, sizeof(Management.server.signature));

    Management.directoryHandle = NULL;

    if (Identity && Identity[0] && DirectoryHandle) {
        Management.state = MANAGEMENT_INITIALIZING;
        Management.directoryHandle = DirectoryHandle;

        result = TRUE;
    }

    if (result) {
        result = FALSE;

        if (MDBGetServerInfo(Management.host.dn, NULL, NULL)) {
            result = TRUE;
        }
    }

    if (result) {
        result = FALSE;

        config = MDBCreateValueStruct(Management.directoryHandle, NULL);

        if (config) {
            result = TRUE;
        }
    }

    if (result) {
        result = FALSE;

        if (MDBReadDN(Management.host.dn, MSGSRV_A_HULA_MESSAGING_SERVER, config)) {
            strcpy(Management.dmc.dn, config->Value[0]);

            MDBFreeValues(config);

            result = TRUE;
        }
    }

    if (result) {
        MDBSetValueStructContext(Management.dmc.dn, config);

        result = MDBIsObject(Identity, config);
        if (result) {
            sprintf(Management.server.dn, "%s\\%s", Management.host.dn, Identity);
        }

        MDBSetValueStructContext(NULL, config);
    }

    if (result) {
        result = FALSE;

        Management.dmc.port = htons(DMC_MANAGER_PORT);

        if (MDBRead(Management.dmc.dn, MSGSRV_A_CONFIGURATION, config)) {
            for (used = 0; used < config->Used; used++) {
                if ((XplStrNCaseCmp(config->Value[used], "DMC: ManagerPort=", 20) == 0) 
                        && (isdigit(config->Value[used][20]))) {
                    Management.dmc.port = htons(atol(config->Value[used] + 20));
                }
            }

            MDBFreeValues(config);
        }

        if (MDBRead(MSGSRV_ROOT, MSGSRV_A_ACL, config)) {
            result = HashCredential(Management.dmc.dn, config->Value[0], Management.dmc.credential);

            MDBFreeValues(config);
        }
    }

    if (config) {
        MDBDestroyValueStruct(config);
        config = NULL;
    }

    if (result) {
        Management.state = MANAGEMENT_READY;
    } else {
        Management.state = MANAGEMENT_STOPPED;
    }

    return(result);
}

void 
ManagementShutdown(void)
{
    if (Management.state < MANAGEMENT_STOPPING) {
        Management.state = MANAGEMENT_STOPPING;

        if ((Management.server.conn) && (Management.server.conn->socket != -1)) {
            IPclose(Management.server.conn->socket);
            Management.server.conn->socket = -1;
        }
    }

    return;
}

enum ManagementStates ManagementState(void)
{
    return(Management.state);
}

BOOL 
ManagementSetVariables(ManagementVariables *Variables, unsigned long Count)
{
    if (Management.state == MANAGEMENT_READY) {
        Management.server.variables.count = Count;
        Management.server.variables.data = Variables;
        return(TRUE);
    }

    return(FALSE);
}

BOOL 
ManagementSetCommands(ManagementCommands *Commands, unsigned long Count)
{
    if (Management.state == MANAGEMENT_READY) {
        Management.server.commands.count = Count;
        Management.server.commands.data = Commands;
        return(TRUE);
    }

    return(FALSE);
}

void 
ManagementServer(void *param)
{
    int i;
    int ccode;
    unsigned long used;
    unsigned long attempts = 0;
    unsigned char buffer[CONN_BUFSIZE + 1];
    unsigned char *name = NULL;
    MD5_CTX mdContext;
    XplThreadID id = 0;
    Connection *conn = NULL;

    if (Management.state == MANAGEMENT_READY) {
        name = strrchr(Management.server.dn, '\\');

        if (name) {
            name++;
        } else {
            name = Management.server.dn;
        }

        sprintf(buffer, "%s Management Server", name);
        XplRenameThread(XplGetThreadID(), buffer);

        Management.server.conn = ConnAlloc(FALSE);
        if (Management.server.conn) {
            Management.server.conn->socketAddress.sin_family = AF_INET;
            Management.server.conn->socketAddress.sin_addr.s_addr = MsgGetHostIPAddress();
        } else {
            Management.state = MANAGEMENT_STOPPING;
        }
    }

    if (Management.state == MANAGEMENT_READY) {
        if (ConnServerSocket(Management.server.conn, 2048) == -1) {
            Management.state = MANAGEMENT_STOPPING;
        }
    }

    if (Management.state == MANAGEMENT_READY) {
        conn = ConnAlloc(TRUE);
        if (conn) {
			conn->socketAddress.sin_family = AF_INET;
			conn->socketAddress.sin_addr.s_addr = MsgGetHostIPAddress();
			conn->socketAddress.sin_port = Management.dmc.port;
        } else {
            Management.state = MANAGEMENT_STOPPING;
        }
    }

    if (Management.state == MANAGEMENT_READY) {
        ccode = ConnSocket(conn);
        if (ccode == -1) {
            Management.state = MANAGEMENT_STOPPING;
        }
    }

    /* Register with the Hula Distributed Management Console */
    while (Management.state == MANAGEMENT_READY) {
        if (ConnConnect(conn, NULL, 0, NULL) != -1) {
            break;
        }

        if (attempts++ < MAX_CONNECT_ATTEMPTS) {
            XplDelay(5000);
            continue;
        }

        XplConsolePrintf("%s: Failed to register with DMC (Could not get connection)\n", name);
        Management.state = MANAGEMENT_STOPPING;
    }

    if (Management.state == MANAGEMENT_READY) {
        ccode = ConnRead(conn, buffer, sizeof(buffer));
        if (ccode != -1) {
            ccode = ConnWriteF(conn, "REGISTER \"%s\" %d -1\r\n", Management.server.dn, ntohs(Management.server.conn->socketAddress.sin_port));
        }

        if (ccode != -1) {
            ccode = ConnFlush(conn);
        }

        if (ccode != -1) {
            ccode = ConnReadAnswer(conn, buffer, sizeof(buffer));
        }

        if (ccode != -1) {
            if (strncmp(buffer, DMCMSG1002SALT_COMING, DMCMSG1002SALT_COMING_LEN - 2) == 0) {
                ccode = ConnReadAnswer(conn, buffer, sizeof(buffer));
            } else {
                ccode = -1;
            }
        }

        if (ccode != -1) {
            /* Generate and send our signature */
            MD5_Init(&mdContext);
            MD5_Update(&mdContext, buffer, strlen(buffer));
            MD5_Update(&mdContext, Management.dmc.credential, NMAP_HASH_SIZE);
            MD5_Update(&mdContext, Management.server.dn, strlen(Management.server.dn));
            MD5_Final(buffer, &mdContext);

            for (i = 0; i < 16; i++) {
                sprintf(Management.server.signature + (i * 2),"%02x", buffer[i]);
            }

            ccode = ConnWriteF(conn, "%s\r\n", Management.server.signature);
        }

        if (ccode != -1) {
            ccode = ConnFlush(conn);
        }

        if (ccode != -1) {
            ccode = ConnReadAnswer(conn, buffer, sizeof(buffer));
        }

        if (ccode != -1) {
            if (strncmp(buffer, DMCMSG1003SEND_VARIABLES, DMCMSG1003SEND_VARIABLES_LEN - 2) == 0) {
                for (used = 0; (used < Management.server.variables.count) && (ccode != -1); used++) {
                    ccode = ConnWriteF(conn, "%c \"%s\"\r\n", (Management.server.variables.data[used].Write == NULL)? 'R': 'W', Management.server.variables.data[used].Name);
                }

                if (ccode != -1) {
                    ccode = ConnWrite(conn, "END\r\n", 5);
                }
            } else {
                ccode = -1;
            }
        }

        if (ccode != -1) {
            ccode = ConnFlush(conn);
        }

        if (ccode != -1) {
            ccode = ConnReadAnswer(conn, buffer, sizeof(buffer));
        }

        if (ccode != -1) {
            if (strncmp(buffer, DMCMSG1003SEND_COMMANDS, DMCMSG1003SEND_COMMANDS_LEN - 2) == 0) {
                for (used = 0; (used < Management.server.commands.count) && (ccode != -1); used++) {
                    ccode = ConnWriteF(conn, "%s\r\n", Management.server.commands.data[used].Name);
                }

                if (ccode != -1) {
                    ccode = ConnWrite(conn, "END\r\n", 5);
                }
            } else {
                ccode = -1;
            }
        }

        if (ccode != -1) {
            ccode = ConnFlush(conn);
        }

        if (ccode != -1) {
            ccode = ConnReadAnswer(conn, buffer, sizeof(buffer));
        }

        if (ccode != -1) {
            if (strncmp(buffer, DMCMSG1000OK, DMCMSG1000OK_LEN - 2) != 0) {
                ccode = -1;
            }
        }

        if (ccode != -1) {
            Management.state = MANAGEMENT_RUNNING;
        } else {
            ConnWriteF(conn, "DEREGISTER %s\r\n", Management.server.signature);
            ConnFlush(conn);
            ConnRead(conn, buffer, sizeof(buffer));
        }

        ConnWrite(conn, "QUIT\r\n", 6);
        ConnFlush(conn);
        ConnRead(conn, buffer, sizeof(buffer));
        ConnClose(conn, 0);
        ConnFree(conn);

        conn = NULL;
    }

    /*    Listen for management requests    */
    while (Management.state == MANAGEMENT_RUNNING) {
       conn = ConnAlloc(TRUE);
        if (conn) {
            ccode = sizeof(conn->socketAddress);
            conn->socket = IPaccept(Management.server.conn->socket, (struct sockaddr *)&(conn->socketAddress), &ccode);
            if (conn->socket != -1) {
                if (Management.state == MANAGEMENT_RUNNING) {
                    XplBeginCountedThread(&id, HandleManagementClient, DMC_MANAGEMENT_STACKSIZE, conn, ccode, Management.server.active);

                    if (!ccode) {
                        continue;
                    }
                }

                ConnSend(conn, "QUIT\r\n", 6);

                ConnClose(conn, 0);

                ConnFree(conn);
                conn = NULL;

                continue;
            }

            ConnFree(conn);
            conn = NULL;

            switch (errno) {
                case ECONNABORTED:
#ifdef EPROTO
                case EPROTO:
#endif
                case EINTR: {
                    continue;
                }

                default: {
                    Management.state = MANAGEMENT_STOPPING;

                    break;
                }
            }
        }

        /* Shutdown! */
        break;
    }

    /* Shutting down */
    Management.state = MANAGEMENT_STOPPING;

    for (used = 0; XplSafeRead(Management.server.active) && (used < 60); i++) {
        XplDelay(1000);
    }

    /* DEREGISTER with the Distributed Management Console */
    conn = ConnAlloc(TRUE);
    if (conn) {
		conn->socketAddress.sin_family = AF_INET;
		conn->socketAddress.sin_addr.s_addr = MsgGetHostIPAddress();
		conn->socketAddress.sin_port = Management.dmc.port;
    }

    if (conn) {
        if (ConnSocket(conn) == -1) {
            ConnFree(conn);
            conn = NULL;
        }
    }

    while (conn) {
        if (ConnConnect(conn, NULL, 0, NULL) != -1) {
            break;
        }

        if (attempts++ < MAX_CONNECT_ATTEMPTS) {
            XplDelay(5000);
            continue;
        }

        ConnClose(conn, 0);
        ConnFree(conn);
        conn = NULL;
    }

    if (conn) {
        ConnRead(conn, buffer, sizeof(buffer));

        ConnWriteF(conn, "DEREGISTER %s\r\n", Management.server.signature);
        ConnFlush(conn);

        ConnRead(conn, buffer, sizeof(buffer));

        ConnClose(conn, 0);
        ConnFree(conn);
        conn = NULL;
    }

    Management.state = MANAGEMENT_STOPPED;

    XplExitThread(TSR_THREAD, 0);
}
