/****************************************************************************
 *
 * Copyright (c) 1997-2002 Novell, Inc.
 * All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of version 2.1 of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, contact Novell, Inc.
 *
 * To contact Novell about this file by physical or electronic mail,
 * you may find current contact information at www.novell.com
 *
 ****************************************************************************/

#include <config.h>
#include <xpl.h>
#include <mdb.h>
#include <msgapi.h>
#include <libical.h>
#include <logger.h>
#include <nmlib.h>

#include "nmapdp.h"

#define CALENDAR_TIMEOUT 45
#define CALENDAR_MAX_ITERATIONS 100

#pragma pack(push, 4)
typedef struct {
    unsigned long State; /* Item state */
    unsigned long UTCEnd; /* End of event; seconds since 1970 */
    unsigned long UTCStart; /* Start of event; seconds since 1970 */
    unsigned long UTCDue; /* Duedate of event; seconds since 1970 */
    unsigned char Type; /* Event type */
    unsigned long Sequence; /* Sequence number of event */
    unsigned long UID; /* Unique ID */
    unsigned char ICalUID[40]; /* First 40 chars of iCal UID */
    unsigned char Organizer[40]; /* First 40 chars of Organizer */
    unsigned char Summary[40]; /* First 40 chars of Summary */
    BOOL Recurring; /* Is event recurring */
    unsigned long RecurrenceID; /* Unique Recurrence Identifier */
    unsigned long UIDPos; /* Start of UID line */
    unsigned long OrganizerPos; /* Start of organizer line */
    unsigned long AttendeePos; /* Start of attendeelist */
    unsigned long AttendeeCount; /* Number of attendees */
    unsigned long AttendeeSize; /* Length of attendeelist */
    unsigned long Pos; /* Start of entry in .CAL file */
    unsigned long Size; /* Total size of entry */
    unsigned long CalSize; /* Start of public entry in .CAL file */
} CalendarInfoStructV3;
#pragma pack(pop)

CalendarInfoStruct *
AddCalendarInfo(NMAPClient *client, unsigned long increment)
{
    CalendarInfoStruct *tmp;

    tmp = MemRealloc(client->calendar.entries.info, ((client->calendar.entries.allocated + increment) * sizeof(CalendarInfoStruct)));
    if (tmp != NULL) {
        client->calendar.entries.info = tmp;
        client->calendar.entries.allocated += increment;
    } return(tmp);

    LoggerEvent(NMAP.handle.logging, LOGGER_SUBSYSTEM_GENERAL, LOGGER_EVENT_NMAP_OUT_OF_MEMORY, LOG_CRITICAL, 0, client->user, client->calendar.name, (client->calendar.entries.allocated + increment) * sizeof(CalendarInfoStruct), 0, NULL, 0);

    XplConsolePrintf("NMAPD: Out of memory processing user %s, calendar %s; request size: %lu bytes\r\n", client->user, client->calendar.name, (client->calendar.entries.allocated + increment) * sizeof(CalendarInfoStruct));

    return(tmp);
}

static int 
PrintCal(NMAPClient *client, MDBValueStruct *vs, unsigned char *path, unsigned char *pattern, int skip, unsigned char *newPath, unsigned long recurse, NMAPClient *sclient)
{
    int len;
    int len2;
    int ccode;
    unsigned long used;
    unsigned char *ptr;
    unsigned char *buffer;
    XplDir *dirEntry;
    XplDir *dirP;
    FILE *index;
    BOOL result;

    buffer = MemMalloc(strlen(path) + 7);
    if (buffer != NULL) {
        len = sprintf(buffer, "%s", path);
    } else {
        return(0);
    }

    XplSetThreadGroupID(NMAP.id.load);

    dirEntry = XplOpenDir(buffer);
    if (dirEntry != NULL) {
        len2 = strlen(pattern);
    } else {
        XplSetThreadGroupID(NMAP.id.group);

        MemFree(buffer);

        return(0);
    }

    ccode = 0;
    while ((ccode != -1) && ((dirP = XplReadDir(dirEntry)) != NULL)) {
        if ((dirP->d_attr & XPL_A_SUBDIR) && (dirP->d_name[0] != '.')) {
            sprintf(client->path, "%s/%s.cal", buffer, dirP->d_name);
            if (access(client->path, 0) != 0) {
                result = FALSE;

                sprintf(client->path, "%s/%s", buffer, dirP->d_name);
                len = XPLDosToLong(client->path, skip, newPath, XPL_MAX_PATH);

                /* Do we have a share with this name? */
                for (used = 0; used < vs->Used; used++) {
                    ptr = vs->Value[used];

                    if ((*(ptr++) != 'C') || (*ptr != 'A')) {
                        continue;
                    }

                    ptr = strrchr(++ptr, '\r');
                    if ((ptr == NULL) || (XplStrCaseCmp(++ptr, newPath) != 0)) {
                        continue;
                    }

                    result = TRUE;
                    break;
                }

                if (result == FALSE) {
                    if ((pattern[0] == '\0') || (XplStrNCaseCmp(newPath, pattern, len2) == 0)) {
                        newPath[len++] = '/';
                        newPath[len++] = '\r';
                        newPath[len++] = '\n';
                        newPath[len] = '\0';

                        if ((ccode = ConnWrite(client->conn, "2002-", 5)) != -1) {
                            ccode = ConnWrite(client->conn, newPath, len);
                        }
                    }
                }
            }

            if (recurse) {
                sprintf(client->path, "%s/%s", buffer, dirP->d_name);
                PrintCal(client, vs, client->path, pattern, skip, newPath, recurse - 1, sclient);

                XplSetThreadGroupID(NMAP.id.load);
            }

            continue;
        }
        
        if (QuickCmp(dirP->d_name + strlen(dirP->d_name) - 4, ".cal")) {
            if (sclient == NULL) {
                sprintf(client->path, "%s/%s", buffer, dirP->d_name);
                len = XPLDosToLong(client->path, skip, newPath, XPL_MAX_PATH) - 4;
                newPath[len] = '\0';
                if ((pattern[0] == '\0') || (XplStrNCaseCmp(newPath, pattern, len2) == 0)) { 
                    newPath[len++] = '\r';
                    newPath[len++] = '\n';
                    newPath[len] = '\0';

                    if ((ccode = ConnWrite(client->conn, "2002-", 5)) != -1) {
                        ccode = ConnWrite(client->conn, newPath, len);
                    }
                }

                continue;
            }

            sclient->calendar.entries.count = 0;
            sclient->calendar.entries.used = 0;

            sprintf(client->path, "%s/%s", buffer, dirP->d_name);
            len = XPLDosToLong(client->path, skip, sclient->calendar.name, sizeof(sclient->calendar.name)) - 4;
            sclient->calendar.name[len] = '\0';

            XplSetThreadGroupID(NMAP.id.group);

            sclient->calendar.flags = 0;

            sprintf(client->path, "%s/%s/%s.idc", client->store, client->user, sclient->calendar.name);
            used = ReadNLockAquire(&sclient->calendar.lock, &client->userHash, client->user, sclient->calendar.name, 0);
            if (used == NLOCK_AQUIRED) {
                index = fopen(client->path, "rb");
                if (index != NULL) {
                    fseek(index, 0, SEEK_SET);
                    if (fgets(client->line, CONN_BUFSIZE, index)) {
                        if ((strncmp(client->line, NMAP_CAL_IDX_VERSION, 8) == 0) && (client->line[8] == '\r')) {
                            fgets(client->line, CONN_BUFSIZE, index);
                            sclient->calendar.flags = atol(client->line);
                            fclose(index);
                        } else {
                            fclose(index);
                            if (ParseCalendar(sclient) == FALSE) {
                                sclient->calendar.flags = 0;
                            }
                        }
                    } else {
                        fclose(index);
                        if (ParseCalendar(sclient) == FALSE) {
                            sclient->calendar.flags = 0;
                        }
                    }
                }
                ReadNLockRelease(sclient->calendar.lock);
                sclient->calendar.lock = NULL;
            }

            XplSetThreadGroupID(NMAP.id.load);

            if (!(sclient->calendar.flags & RESOURCE_FLAG_UNSUBSCRIBED)) {
                sprintf(client->path, "%s/%s", buffer, dirP->d_name);
                len = XPLDosToLong(client->path, skip, newPath, XPL_MAX_PATH + 1) - 4;
                newPath[len] = '\0';
                if ((pattern[0] == '\0') || (XplStrNCaseCmp(newPath, pattern, len2) == 0)) { 
                    newPath[len++] = '\r';
                    newPath[len++] = '\n';
                    newPath[len] = '\0';

                    if ((ccode = ConnWrite(client->conn, "2002-", 5)) != -1) {
                        ccode = ConnWrite(client->conn, newPath, len);
                    }
                }

                continue;
            }

            ptr = strrchr(sclient->calendar.name, '/');
            if (ptr == NULL) {                    
                sprintf(client->path, "%s/%s", buffer, sclient->calendar.name);
            } else {
                sprintf(client->path, "%s%s", buffer, ptr);
            }

            if ((access(client->path, 0) == 0) 
                    && ((pattern[0] == '\0') || (XplStrNCaseCmp(sclient->calendar.name, pattern, len2) == 0))) {
                sclient->calendar.name[len++] = '/';
                sclient->calendar.name[len++] = '\r';
                sclient->calendar.name[len++] = '\n';
                sclient->calendar.name[len] = '\0';

                if ((ccode = ConnWrite(client->conn, "2002-", 5)) != -1) {
                    ccode = ConnWrite(client->conn, sclient->calendar.name, len);
                }
            }

            continue;
        }
    }

    XplCloseDir(dirEntry);

    XplSetThreadGroupID(NMAP.id.group);

    MemFree(buffer);

    return(ccode);
}

/* fixme - we should no longer be using file based locks. */
int 
WaitforAndLockCalendar(const unsigned char *store, unsigned char *user, unsigned char *calendar)
{
    int timeoutCount = CALENDAR_TIMEOUT;
    unsigned long offset;
    unsigned char path[XPL_MAX_PATH + 1];
    struct stat sb;
    FILE *handle;
    XplSemaphore semaphore;

    offset = ((user[0] << 5) || (user[1] & 0x1F)) & CSTORE_LOCK_ARRAY_SIZE;
    semaphore = NMAP.lock.calendar.sems[offset & 0x7F];

    sprintf(path, "%s/%s/%s.lcc", store, user, calendar);

    do {
        XplWaitOnLocalSemaphore(semaphore);
        if (NMAP.lock.calendar.array[offset] == 0) {
            NMAP.lock.calendar.array[offset] = 1;
            XplSignalLocalSemaphore(semaphore);

            if ((stat(path, &sb) == -1) || (sb.st_mtime < NMAP.time.startup)) {
                /* No existing valid lock was found, create one. */
                handle = fopen(path, "w+b");

                XplWaitOnLocalSemaphore(semaphore);
                NMAP.lock.calendar.array[offset] = 0;
                XplSignalLocalSemaphore(semaphore);

                if (handle != NULL) {
                    fclose(handle);
                    return(1);
                }

                return(0);
            }

            /* An existing lock was found, loop. */
            XplWaitOnLocalSemaphore(semaphore);
            NMAP.lock.calendar.array[offset] = 0;
        }

        XplSignalLocalSemaphore(semaphore);
        XplDelay(333);

        timeoutCount--;
    } while ((NMAP.state < NMAP_STOPPING) && (timeoutCount != 0));

    return(-1);
}

/* fixme - we should no longer be using file based locks. */
void
UnlockCalendar(const unsigned char *store, unsigned char *user, unsigned char *calendar)
{
    unsigned char path[XPL_MAX_PATH + 1];

    sprintf(path, "%s/%s/%s.lcc", store, user, calendar);
    unlink(path);

    return;
}

__inline static int
GetChangedCalFlags(unsigned char *idc, CalendarInfoStruct *entries, unsigned long used)
{
    int newPurges;
    unsigned long id;
    unsigned long state[2];
    unsigned char *base;
    FILE *index;

    /* Detect the delta between disk and memory; the only thing that could change are flags */
    newPurges = -1;
    base = (unsigned char *)entries;

    index = fopen(idc, "rb");
    if (index != NULL) {
        newPurges++;
        for (id = 0; id < used; id++, entries++) {
            if (fseek(index, NMAP_CAL_IDX_HEADERSIZE + ((unsigned char *)entries - base + offsetof(CalendarInfoStruct, State)), SEEK_SET) == 0) {
                /* This is tricky; we read the State and UTCend at once... */
                if (fread(&state, sizeof(unsigned long), 2, index) == 2) {
                    entries->UTCEnd = state[1];

                    if (state[0] & MSG_STATE_PURGED) {
                        newPurges++;;
                    }

                    if (entries->State == state[0]) {
                        continue;
                    }

                    entries->State = state[0];
                    continue;
                }
            }
            
            newPurges = -1;
            break;
        }

        fclose(index);
    }
    
    return(newPurges);
}

__inline static unsigned long 
CopyUnpurgedCalObjects(FILE *source, FILE *destination, CalendarInfoStruct *entries, unsigned long used, unsigned char *line, unsigned long lineLength)
{
    int ccode = 0;
    unsigned long delta;
    unsigned long newCount = 0;
    unsigned long count;
    unsigned long len;

    for (newCount = 0; (ccode != -1) && (used != 0); used--, entries++) {
        /* If the entry is deleted then just throw it away */
        if (!(entries->State & MSG_STATE_PURGED)) {
            newCount++;
            count = entries->Size;
            if (fseek(source, entries->Pos, SEEK_SET)) {
                if ((delta = entries->Pos - ftell(destination)) != 0) {
                    entries->Pos -= delta;
                    entries->UIDPos -= delta;
                    entries->OrganizerPos -= delta;
                    entries->AttendeePos -= delta;
                }

                while (count > 0) {
                    len = min(count, lineLength);
                    if (fread(line, sizeof(unsigned char), len, source) == len) {
                        if (fwrite(line, sizeof(unsigned char), len, destination) == len) {
                            count -= len;
                            continue;
                        }
                    }

                    return(0xFFFFFFFF);
                }

                continue;
            }

            return(0xFFFFFFFF);
        }
    }

    return(newCount);
}

BOOL 
StoreCalendar(NMAPClient *client)
{
    long result;
    unsigned long id;
    unsigned long newSize;
    unsigned long newCount;
    unsigned char calendar[XPL_MAX_PATH + 1];
    unsigned char temp[XPL_MAX_PATH + 1];
    unsigned char idc[XPL_MAX_PATH + 1];
    FILE *index;
    FILE *newStore;
    struct stat sb;
    CalendarInfoStruct *cInfo;

    if ((client->flags & NMAP_CSTORE_READONLY) == 0) {
        if ((client->user[0] != '\0') && (client->calendar.name[0] != '\0')) {
            result = PurgeNLockAquire(client->calendar.lock, 0);
        } else {
            return(FALSE);
        }
    } else {
        return(TRUE);
    }

    if (result == NLOCK_AQUIRED) {
        sprintf(idc, "%s/%s/%s.idc", client->store, client->user, client->calendar.name);
    } else {
        return(FALSE);
    }

    if (client->calendar.indexChanged) {
        result = GetChangedCalFlags(idc, client->calendar.entries.info, client->calendar.entries.used);
        if (result == 0) {
            stat(idc, &sb);
            client->calendar.indexTime = sb.st_mtime;
            client->calendar.indexChanged = FALSE;
        } else if (result > 0) {
            stat(idc, &sb);
            client->calendar.indexTime = sb.st_mtime;
            client->calendar.indexChanged = FALSE;
            client->calendar.changed = TRUE;
        } else {
            PurgeNLockRelease(client->calendar.lock);
            return(FALSE);
        }
    }

    while (client->calendar.changed) {
        sprintf(calendar, "%s/%s/%s.cal", client->store, client->user, client->calendar.name);
        client->calendar.fh = fopen(calendar, "rb");
        if (client->calendar.fh != NULL) {
            sprintf(temp, "%s/%s/%s.tmp", client->store, client->user, client->calendar.name);
        } else {
            break;
        }

        newStore = fopen(temp, "wb");
        if (newStore != NULL) {
            SetLongName(temp, client->line);
        } else {
            break;
        }

        newCount = CopyUnpurgedCalObjects(client->calendar.fh, newStore, client->calendar.entries.info, client->calendar.entries.used, client->line, CONN_BUFSIZE);
        if (newCount != 0xFFFFFFFF) {
            if (fseek(newStore, 0, SEEK_END) == 0) {
                newSize = ftell(newStore);

                if (!(NMAP.flushFlags & FLUSH_CAL_ON_PURGE)) {
                    fclose(newStore);
                } else {
                    XplFlushWrites(newStore);
                    fclose(newStore);
                }

                newStore = NULL;

                fclose(client->calendar.fh);
                client->calendar.fh = NULL;

                if (unlink(calendar) == 0) {
                    SetLongName(calendar, client->line);

                    index = fopen(idc, "wb");
                    if (index != NULL) {
                        SetLongName(idc, client->line);

                        fprintf(index, NMAP_CAL_IDX_VERSION"\r\n");
                        fprintf(index, "%010lu\r\n", client->calendar.flags);
                        fprintf(index, "%010lu\r\n", newSize);
                        fprintf(index, "%010lu\r\n", newCount);
                        fprintf(index, "%010lu\r\n", client->calendar.id);
                        fprintf(index, "%010lu\r\n", client->calendar.nextUID);

                        for (id = client->calendar.entries.used, cInfo = client->calendar.entries.info; id != 0; id--, cInfo++) {
                            if (!(cInfo->State & MSG_STATE_PURGED)) {
                                fwrite(cInfo, sizeof(CalendarInfoStruct), 1, index);
                            }
                        }

                        if (!(NMAP.flushFlags & FLUSH_CAL_ON_PURGE)) {
                            fclose(index);
                        } else {
                            XplFlushWrites(index);
                            fclose(index);
                        }

                        stat(idc, &sb);
                        client->calendar.indexTime = sb.st_mtime;

                        for (id = 5; id < 5; id++) {
                            if (rename(temp, calendar) == 0) {
                                PurgeNLockRelease(client->calendar.lock);
                                client->calendar.changed = FALSE;

                                return(TRUE);
                            }

                            XplDelay(111);
                        }
                    }
                }
            }
        }

        if (newStore) {
            fclose(newStore);
            newStore = NULL;
        }

        unlink(temp);

        if (client->calendar.fh) {
            fclose(client->calendar.fh);
            client->calendar.fh = NULL;
        }

        break;
    }
    
    PurgeNLockRelease(client->calendar.lock);

    LoggerEvent(NMAP.handle.logging, LOGGER_SUBSYSTEM_GENERAL, LOGGER_EVENT_WRITE_ERROR, LOG_WARNING, 0, client->calendar.name, client->user, 0, 0, NULL, 0);
    client->calendar.changed = FALSE;

    return(FALSE);
}

static BOOL
ParseV3Calendar(NMAPClient *client)
{
    unsigned long previous = 0xFFFFFFFF;
    unsigned long i;
    unsigned long version;
    unsigned long sequence;
    unsigned long utcStart;
    unsigned long utcEnd;
    unsigned long utcDue;
    unsigned long recurrenceID;
    unsigned long start;
    unsigned long size;
    unsigned long calSize;
    unsigned long len;
    unsigned long calState;
    unsigned long eventType;
    unsigned long OrganizerPos;
    unsigned long uidPos;
    unsigned long attendeePos;
    unsigned long attendeeCount;
    unsigned long attendeeSize;
    struct stat sb;
    unsigned char *ptr;
    unsigned char iCalUID[40];
    unsigned char organizer[40];
    unsigned char summary[40];
    BOOL recurring;
    FILE *fh;
    MDBValueStruct *uid;

    if (client->calendar.fh) {
        fclose(client->calendar.fh);
    }

    sprintf(client->path, "%s/%s/%s.cal", client->store, client->user, client->calendar.name);
    client->calendar.fh = fopen(client->path, "rb");
    if (!client->calendar.fh) {
        /* No store yet */
        return(FALSE);
    }

    /* Store new size; remember old size for index check */
    stat(client->path, &sb);
    size = client->calendar.size;
    client->calendar.size = sb.st_size;

    sprintf(client->path, "%s/%s/%s.idc", client->store, client->user, client->calendar.name);
    fh = fopen(client->path, "rb");
    if (fh) {
        stat(client->path, &sb);
        client->calendar.indexTime = sb.st_mtime;
        client->calendar.flags = 0;

        fgets(client->line, sizeof(client->line), fh);    /*    Old Version    */
        if (strncmp(client->line, "CNIMSv04", 8) == 0) {
            /*    Version 4 to Version 5 Update
            
                We are missing the mailbox flags
                We are missing the signature on each message information block

                This line in the buffer is the mailbox size.    */
            version = 4;
        } else if (strncmp(client->line, "CNIMSv03", 8) == 0) {
            /*    Version 3 to Version 5 Update
            
                We are missing the mailbox flags
                We are missing the signature on each message information block
                The value for .calendar.id is not trustworthy
                We are missing the next calendar uid

                This line in the buffer is the mailbox size.    */
            version = 3;
        } else {
            /*    Prior to version 3; toss it!    */
            fclose(fh);
            fh = NULL;

            uid = MDBCreateValueStruct(NMAP.handle.directory, NMAP.server.dn);

            client->calendar.flags = 0;
            client->calendar.entries.count = 0;
            client->calendar.entries.used = 0;
            client->calendar.entries.allocated = 0;
            client->calendar.entries.info = NULL;

            client->calendar.id = NMAP.universalCounter++;
            client->calendar.nextUID = NMAP.universalCounter++;
            sprintf(client->line, "%lu", NMAP.universalCounter);

            MDBAddValue(client->line, uid);
            MDBWrite(MSGSRV_AGENT_NMAP, MSGSRV_A_UID, uid);
            MDBDestroyValueStruct(uid);

            goto NoV3CalendarIndex;
        }

        fgets(client->line, sizeof(client->line),fh);
        calSize = atol(client->line);

        fgets(client->line, sizeof(client->line),fh);
        client->calendar.entries.count = atol(client->line);

        fgets(client->line, sizeof(client->line), fh);
        if (version == 4) {
            client->calendar.id = atol(client->line);

            fgets(client->line, sizeof(client->line), fh);
            client->calendar.nextUID = atol(client->line);
        } else {
            client->calendar.id = NMAP.universalCounter++;        /*    Set it to one; we didn't have one before so 
                                                                            one is always good on an upgrade */
            client->calendar.nextUID = NMAP.universalCounter++;            /* We didn't store UIDs on V3 (bug), let's come 
                                                                            up with an ID and start assigning them */

            uid = MDBCreateValueStruct(NMAP.handle.directory, NMAP.server.dn);
            sprintf(client->line, "%lu", NMAP.universalCounter);

            MDBAddValue(client->line, uid);
            MDBWrite(MSGSRV_AGENT_NMAP, MSGSRV_A_UID, uid);

            MDBDestroyValueStruct(uid);
        }

        /* We abuse UTCstart... */
        utcStart = ftell(fh);
        if (size > client->calendar.size) {
            MemFree(client->calendar.entries.info);
            client->calendar.entries.info = NULL;

            client->calendar.flags = 0;
            client->calendar.entries.count = 0;
            client->calendar.entries.allocated = 0;
            client->calendar.entries.used = 0;
        }

        if (client->calendar.entries.info == NULL) {
            client->calendar.entries.allocated = client->calendar.entries.count + 1;
            client->calendar.entries.info = MemMalloc(client->calendar.entries.allocated * sizeof(CalendarInfoStruct));
            if (client->calendar.entries.info == NULL) {
                ConnWrite(client->conn, MSG5001NOMEMORY, sizeof(MSG5001NOMEMORY) - 1);

                fclose(fh);

                FreeClientData(client);
                return(FALSE);
            }

            client->calendar.entries.used = client->calendar.entries.count;
            start = 0;
        } else {
            start = client->calendar.entries.used;
            if (client->calendar.entries.count > client->calendar.entries.allocated) {
                if (!AddCalendarInfo(client, (client->calendar.entries.count - client->calendar.entries.allocated) + 36)) {
                    ConnWrite(client->conn, MSG5001NOMEMORY, sizeof(MSG5001NOMEMORY) - 1);

                    fclose(fh);

                    FreeClientData(client);
                    return(FALSE);
                }
            }

            client->calendar.entries.used = client->calendar.entries.count;
        }

        fseek(fh, utcStart + (start * sizeof(CalendarInfoStructV3)), SEEK_SET);
        for (i = start; i < client->calendar.entries.count; i++) {
            client->calendar.entries.info[i].TailSig = NMAP_IDX_LINE_TAIL_SIGNATURE;

            fread(&(client->calendar.entries.info[i].State), sizeof(CalendarInfoStructV3), 1, fh);

            if (version == 3) {
                /* Assign a uid (we had a bug in v3 that didn't assign real UIDs) */
                client->calendar.entries.info[i].UID = client->calendar.nextUID++;
            }
        }

        fclose(fh);
        
        if (calSize == client->calendar.size) {        /* Calendar hasn't changed, just read the file and return */
            goto RewriteV5CalendarIndexFile;
        }

        if (client->calendar.entries.count) {
            fseek(client->calendar.fh, client->calendar.entries.info[client->calendar.entries.count - 1].Pos + client->calendar.entries.info[client->calendar.entries.count - 1].Size, SEEK_SET);
        } else {
            fseek(client->calendar.fh, 0, SEEK_SET);
        }
    } else {
NoV3CalendarIndex:
        uid = MDBCreateValueStruct(NMAP.handle.directory, NMAP.server.dn);

        client->calendar.flags = 0;
        client->calendar.entries.count = 0;
        client->calendar.entries.used = 0;
        client->calendar.entries.allocated = 0;
        client->calendar.entries.info = NULL;

        client->calendar.id = NMAP.universalCounter++;
        client->calendar.nextUID = NMAP.universalCounter++;
        sprintf(client->line, "%lu", NMAP.universalCounter);

        MDBAddValue(client->line, uid);
        MDBWrite(MSGSRV_AGENT_NMAP, MSGSRV_A_UID, uid);
        MDBDestroyValueStruct(uid);

        version = 4;
    }

    /* Some general preparations */
    organizer[39] = '\0';
    summary[39] = '\0';

    while (!feof(client->calendar.fh)) {
        if (fgets(client->line, CONN_BUFSIZE, client->calendar.fh)) {
            len = strlen(client->line);
            if (QuickCmp(client->line, "BEGIN:VCALENDAR\r\n")) {
                if (previous != 0xFFFFFFFF) {
                    client->calendar.entries.info[previous].TailSig = NMAP_IDX_LINE_TAIL_SIGNATURE;

                    client->calendar.entries.info[previous].Type = eventType;
                    client->calendar.entries.info[previous].Pos = start;
                    client->calendar.entries.info[previous].Size = ftell(client->calendar.fh) - 17 - start;
                    client->calendar.entries.info[previous].State = calState;
                    client->calendar.entries.info[previous].CalSize = calSize;
                    client->calendar.entries.info[previous].Sequence = sequence;
                    client->calendar.entries.info[previous].UTCStart = utcStart;
                    client->calendar.entries.info[previous].UTCEnd = utcEnd;
                    client->calendar.entries.info[previous].UTCDue = utcDue;
                    client->calendar.entries.info[previous].Recurring = recurring;
                    client->calendar.entries.info[previous].RecurrenceID = recurrenceID;
                    client->calendar.entries.info[previous].UIDPos = uidPos;
                    client->calendar.entries.info[previous].AttendeePos = attendeePos;
                    client->calendar.entries.info[previous].AttendeeCount = attendeeCount;
                    client->calendar.entries.info[previous].AttendeeSize = attendeeSize;
                    client->calendar.entries.info[previous].OrganizerPos = OrganizerPos;
                    client->calendar.entries.info[previous].UID = client->calendar.nextUID++;
                    strcpy(client->calendar.entries.info[previous].ICalUID, iCalUID);
                    strcpy(client->calendar.entries.info[previous].Organizer, organizer);
                    strcpy(client->calendar.entries.info[previous].Summary, summary);

                    client->calendar.entries.count++;
                }

                if ((previous = (client->calendar.entries.used)++) < client->calendar.entries.allocated) {
                    size = 0;
                } else if (AddCalendarInfo(client, 128)) {
                    size = 0;
                } else {
                    ConnWrite(client->conn, MSG5001NOMEMORY, sizeof(MSG5001NOMEMORY) - 1);

                    fclose(fh);

                    FreeClientData(client);
                    return(FALSE);
                }

                /* Prepare the next set */
                calSize = 0;
                utcStart = 0;
                utcEnd = 0;
                utcDue = 0;
                sequence = 0;
                recurring = FALSE;
                recurrenceID = 0;
                calState = 0;
                eventType = 0;
                OrganizerPos = 0;
                uidPos = 0;
                attendeePos = 0;
                attendeeCount = 0;
                attendeeSize = 0;
                iCalUID[0] = '\0';
                organizer[0] = '\0';
                summary[0] = '\0';
                start = ftell(client->calendar.fh) - 17;
            } else if (QuickCmp(client->line,   "END:VCALENDAR\r\n")) {
                calSize = ftell(client->calendar.fh) - start;
            } else if (QuickNCmp(client->line, "uid", 3)) {
                unsigned char    *ptr;

                 uidPos = ftell(client->calendar.fh)-len;
                if ((ptr = strchr(client->line + 3, ':')) == NULL) {
                    ptr = client->line+3;
                }
                
                if (len > 43) {
                    strncpy(iCalUID, ptr+1, 39);
                    iCalUID[39] = '\0';
                    CHOP_NEWLINE(iCalUID);
                } else {
                    memcpy(iCalUID, ptr + 1, len - 5);
                    CHOP_NEWLINE(iCalUID);
                }
            } else if (QuickNCmp(client->line, "NIMS-UTCSTART:", 14)) {
                utcStart = atol(client->line + 14);
            } else if (QuickNCmp(client->line, "NIMS-UTCEND:", 12)) {
                utcEnd = atol(client->line + 12);
            } else if (QuickNCmp(client->line, "NIMS-UTCDUE:", 12)) {
                utcDue = atol(client->line + 12);
            } else if (QuickNCmp(client->line, "NIMS-RECURRING", 14)) {
                recurring = TRUE;
            } else if (QuickNCmp(client->line, "NIMS-RID:", 9)) {
                recurrenceID = atol(client->line+9);
            } else if (QuickNCmp(client->line, "NIMS-TYPE:", 10)) {
                eventType = atol(client->line+10);
            } else if (QuickNCmp(client->line, "NIMS-STATE:", 11)) {
                calState = atol(client->line+11);
            } else if (QuickNCmp(client->line, "NIMS-ORGANIZER:", 15)) {
                OrganizerPos = ftell(client->calendar.fh) - len + 15;
                ptr = strchr(client->line, ' ');
                if (ptr[1] == '\r') {
                    ptr[0] = '\0';

                    strncpy(organizer, client->line + 15, 39);
                } else {
                    strncpy(organizer, ptr + 1, 39);
                    CHOP_NEWLINE(organizer);
                }
            } else if (QuickNCmp(client->line, "NIMS-ATTENDEE:", 14)) {
                if (attendeePos == 0) {
                    attendeePos = ftell(client->calendar.fh) - len;
                }

                attendeeCount++;
            } else if (QuickNCmp(client->line, "NIMS-ENDATTENDEE", 16)) {
                attendeeSize = ftell(client->calendar.fh) - len - attendeePos;
            } else if (QuickNCmp(client->line, "SEQUENCE", 8)) {
                unsigned char    *ptr;

                ptr = strchr(client->line+8, ':');
                if (ptr) {
                    sequence = atol(ptr+1);
                }
            } else if (QuickNCmp(client->line, "SUMMARY", 7)) {
                unsigned char    *ptr;

                ptr = strchr(client->line + 7, ':');
                if (ptr) {
                    strncpy(summary, ptr+1, 39);
                    CHOP_NEWLINE(summary);
                }
            }
        }
    }

    /* Handle the remaining item */
    if (previous != 0xFFFFFFFF) {
        client->calendar.entries.info[previous].TailSig = NMAP_IDX_LINE_TAIL_SIGNATURE;

        client->calendar.entries.info[previous].Type = eventType;
        client->calendar.entries.info[previous].Pos = start;
        client->calendar.entries.info[previous].Size = ftell(client->calendar.fh)-start;
        client->calendar.entries.info[previous].State = calState;
        client->calendar.entries.info[previous].CalSize = calSize;
        client->calendar.entries.info[previous].Sequence = sequence;
        client->calendar.entries.info[previous].UTCStart = utcStart;
        client->calendar.entries.info[previous].UTCEnd = utcEnd;
        client->calendar.entries.info[previous].UTCDue = utcDue;
        client->calendar.entries.info[previous].Recurring = recurring;
        client->calendar.entries.info[previous].RecurrenceID = recurrenceID;
        client->calendar.entries.info[previous].UIDPos = uidPos;
        client->calendar.entries.info[previous].AttendeePos = attendeePos;
        client->calendar.entries.info[previous].AttendeeCount = attendeeCount;
        client->calendar.entries.info[previous].AttendeeSize = attendeeSize;
        client->calendar.entries.info[previous].OrganizerPos = OrganizerPos;
        client->calendar.entries.info[previous].UID = client->calendar.nextUID++;
        strcpy(client->calendar.entries.info[previous].ICalUID, iCalUID);
        strcpy(client->calendar.entries.info[previous].Organizer, organizer);
        strcpy(client->calendar.entries.info[previous].Summary, summary);
        client->calendar.entries.count++;
    }

RewriteV5CalendarIndexFile:
    /* Write changed index file */
    if (! (client->flags & NMAP_READONLY)) {
        sprintf(client->path, "%s/%s/%s.idc",client->store, client->user, client->calendar.name);

        fh = fopen(client->path,"wb");
        fseek(client->calendar.fh, 0, SEEK_END);
        fprintf(fh, NMAP_CAL_IDX_VERSION"\r\n");
        fprintf(fh, "%010lu\r\n", client->calendar.flags);
        fprintf(fh, "%010lu\r\n", ftell(client->calendar.fh));
        fprintf(fh, "%010lu\r\n", client->calendar.entries.used);
        fprintf(fh, "%010lu\r\n", client->calendar.id);
        fprintf(fh, "%010lu\r\n", client->calendar.nextUID);

        for (i=0; i<client->calendar.entries.used; i++) {
            fwrite(&client->calendar.entries.info[i], sizeof(CalendarInfoStruct), 1, fh);
        }

        fclose(fh);
    }

    fclose(client->calendar.fh);
    client->calendar.fh = NULL;

    return(TRUE);
}

int 
ParseCalendar(NMAPClient *client)
{
    unsigned long previous;
    unsigned long i;
    unsigned long sequence;
    unsigned long utcStart;
    unsigned long utcEnd;
    unsigned long utcDue;
    unsigned long recurrenceID;
    unsigned long start;
    unsigned long size;
    unsigned long calSize;
    unsigned long len;
    unsigned long calState;
    unsigned long eventType;
    unsigned long organizerPos;
    unsigned long uidPos;
    unsigned long attendeePos;
    unsigned long attendeeCount;
    unsigned long attendeeSize;
    unsigned long estCount;
    unsigned char *ptr;
    unsigned char iCalUID[40];
    unsigned char organizer[40];
    unsigned char summary[40];
    CalendarInfoStruct *cInfo;
    struct stat sb;
    BOOL recurring;
    FILE *index;
    MDBValueStruct *uid;

    if ((client->user[0] != '\0') && (client->calendar.name[0] != '\0')) {
        if (WaitforAndLockCalendar(client->store, client->user, client->calendar.name)) {
            previous = 0xFFFFFFFF;
        } else {
            return(0);
        }
    } else {
        return(-1);
    }

V5StartOver:
    if (client->calendar.fh != NULL) {
        fclose(client->calendar.fh);
    }

    sprintf(client->path, "%s/%s/%s.cal", client->store, client->user, client->calendar.name);
    client->calendar.fh = fopen(client->path, "rb");
    if (client->calendar.fh != NULL) {
        /* Store new size; remember old size for index check */
        stat(client->path, &sb);
        size = client->calendar.size;
        client->calendar.size = sb.st_size;
    } else {
        /* No store yet */
        UnlockCalendar(client->store, client->user, client->calendar.name);
        return(-1);
    }

    sprintf(client->path, "%s/%s/%s.idc", client->store, client->user, client->calendar.name);
    index = fopen(client->path, "rb");
    if (index != NULL) {
        stat(client->path, &sb);
        client->calendar.indexTime = sb.st_mtime;

        client->line[7] = '\0';
        fgets(client->line, CONN_BUFSIZE, index);
        if ((client->line[0] == 'C') && (client->line[7] == '5')) {
            fgets(client->line, CONN_BUFSIZE, index);
            client->calendar.flags = atol(client->line);

            fgets(client->line, CONN_BUFSIZE, index);
            calSize = atol(client->line);

            fgets(client->line, CONN_BUFSIZE, index);
            client->calendar.entries.count = atol(client->line);

            fgets(client->line, CONN_BUFSIZE, index);
            client->calendar.id = atol(client->line);

            fgets(client->line, CONN_BUFSIZE, index);
            client->calendar.nextUID = atol(client->line);

            if (ftell(index) == NMAP_CAL_IDX_HEADERSIZE) {
                /* We abuse UTCstart... */
                utcStart = NMAP_CAL_IDX_HEADERSIZE;
            } else {
                fclose(index);
                index = NULL;

                unlink(client->path);

                LoggerEvent(NMAP.handle.logging, LOGGER_SUBSYSTEM_GENERAL, LOGGER_EVENT_INDEX_CORRUPTION, LOG_ERROR, 0, client->user, client->calendar.name, -1, 0, NULL, 0);

                goto V5StartOver;
            }
        } else {
            /* We didn't find our current IDX format, parse somewhere else and start over */
            if (feof(index)) {
                /* Probably 0 byte index file; remove it and start over */
                fclose(index);

                sprintf(client->path, "%s/%s/%s.idc", client->store, client->user, client->calendar.name);
                unlink(client->path);

                goto V5StartOver;
            }

            fclose(index);
            fclose(client->calendar.fh);
            client->calendar.fh = NULL;

            if (ParseV3Calendar(client)) {
                goto V5StartOver;
            }

            /*    We could not process, or recreate, the index file.  Help!    */
            UnlockCalendar(client->store, client->user, client->calendar.name);

            ConnWrite(client->conn, MSG5004INTERNALERR, sizeof(MSG5004INTERNALERR) - 1);

            XplConsolePrintf("NMAPD: Unable to use calendar index file %s for user %s; header corrupted.\r\n", client->calendar.name, client->user);

            LoggerEvent(NMAP.handle.logging, LOGGER_SUBSYSTEM_GENERAL, LOGGER_EVENT_INDEX_CORRUPTION, LOG_ERROR, 0, client->user, client->calendar.name, -1, 0, NULL, 0);

            FreeClientData(client);
            return(-1);
        }

        if (size <= client->calendar.size) {
            if (client->calendar.entries.info == NULL) {
                estCount = client->calendar.entries.count + ((client->calendar.size - calSize) >> 10) + 36;

                client->calendar.entries.info = MemMalloc(estCount * sizeof(CalendarInfoStruct));
                if (client->calendar.entries.info != NULL) {
                    client->calendar.entries.allocated = estCount;
                    client->calendar.entries.used = client->calendar.entries.count;
                    start = 0;
                } else {
                    UnlockCalendar(client->store, client->user, client->calendar.name);

                    ConnWrite(client->conn, MSG5001NOMEMORY, sizeof(MSG5001NOMEMORY) - 1);

                    fclose(index);

                    FreeClientData(client);
                    return(-1);
                }
            } else {
                start = client->calendar.entries.used;

                if (client->calendar.entries.count <= client->calendar.entries.allocated) {
                    estCount = ((client->calendar.size - calSize) >> 10) + 36;
                } else {
                    estCount = (client->calendar.entries.count - client->calendar.entries.allocated) + ((client->calendar.size - calSize) >> 9) + 36;
                    if (!AddCalendarInfo(client, estCount)) {
                        UnlockCalendar(client->store, client->user, client->calendar.name);

                        ConnWrite(client->conn, MSG5001NOMEMORY, sizeof(MSG5001NOMEMORY) - 1);

                        fclose(index);

                        FreeClientData(client);
                        return(-1);
                    }
                }

                client->calendar.entries.used = client->calendar.entries.count;
            }
        } else {
            estCount = (client->calendar.size >> 10) + 36;

            if (client->calendar.entries.info != NULL) {
                MemFree(client->calendar.entries.info);
            }

            client->calendar.entries.count = 0;
            client->calendar.entries.used  = 0;

            client->calendar.entries.info = MemMalloc(estCount * sizeof(CalendarInfoStruct));
            if (client->calendar.entries.info != NULL) {
                client->calendar.entries.allocated = estCount;
                start = 0;
            } else {
                UnlockCalendar(client->store, client->user, client->calendar.name);

                ConnWrite(client->conn, MSG5001NOMEMORY, sizeof(MSG5001NOMEMORY) - 1);

                fclose(index);

                FreeClientData(client);
                return(-1);
            }
        }

        fseek(index, utcStart + (start * sizeof(CalendarInfoStruct)), SEEK_SET);
        if (sb.st_mtime >= NMAP.time.startup) {
            for (i = client->calendar.entries.count, cInfo = &(client->calendar.entries.info[start]); i != start; i--, cInfo++) {
                fread(cInfo, sizeof(CalendarInfoStruct), 1, index);
            }
        } else {
            cInfo = &(client->calendar.entries.info[start]);
            i = client->calendar.entries.count;

            for (i = client->calendar.entries.count, cInfo = &(client->calendar.entries.info[start]); i != start; i--, cInfo++) {
                fread(cInfo, sizeof(CalendarInfoStruct), 1, index);

                if (cInfo->TailSig == NMAP_IDX_LINE_TAIL_SIGNATURE) {
                    continue;
                }

                /* This message information block is invalid; toss the index and restart. */
                MemFree(client->calendar.entries.info);
                client->calendar.entries.info = NULL;

                client->calendar.entries.count = 0;
                client->calendar.entries.allocated = 0;
                client->calendar.entries.used = 0;

                fclose(index);

                sprintf(client->path, "%s/%s/%s.idc", client->store, client->user, client->calendar.name);
                unlink(client->path);

                XplConsolePrintf("NMAPD: Unable to use index file %s for user %s; entry %lu corrupted.\r\n", client->calendar.name, client->user, i);
                LoggerEvent(NMAP.handle.logging, LOGGER_SUBSYSTEM_GENERAL, LOGGER_EVENT_INDEX_CORRUPTION, LOG_ERROR, 0, client->user, client->calendar.name, i, 0, NULL, 0);

                goto V5StartOver;
            }
        }

        fclose(index);
        
        if (calSize == client->calendar.size) {
            /* Calendar hasn't changed, just read the file and return */
            fclose(client->calendar.fh);
            client->calendar.fh = NULL;

            UnlockCalendar(client->store, client->user, client->calendar.name);
            return(TRUE);
        }

        if (client->calendar.entries.count) {
            fseek(client->calendar.fh, client->calendar.entries.info[client->calendar.entries.count-1].Pos+client->calendar.entries.info[client->calendar.entries.count-1].Size, SEEK_SET);
        } else {
            fseek(client->calendar.fh, 0, SEEK_SET);
        }
    } else {
        estCount = (client->calendar.size >> 9) + 36;

        client->calendar.flags = 0;
        client->calendar.entries.count = 0;
        client->calendar.entries.used = 0;
        client->calendar.entries.allocated = 0;
        client->calendar.entries.info = NULL;

        client->calendar.id = NMAP.universalCounter++;
        client->calendar.nextUID = NMAP.universalCounter++;
        sprintf(client->path, "%lu", NMAP.universalCounter);

        uid = MDBCreateValueStruct(NMAP.handle.directory, NMAP.server.dn);
        MDBAddValue(client->path, uid);
        MDBWrite(MSGSRV_AGENT_NMAP, MSGSRV_A_UID, uid);
        MDBDestroyValueStruct(uid);
    }

    /* Some general preparations */
    organizer[39]='\0';
    summary[39]='\0';

    while (!feof(client->calendar.fh)) {
        if (fgets(client->line, CONN_BUFSIZE, client->calendar.fh)) {
            len = strlen(client->line);
            if (QuickCmp(client->line, "BEGIN:VCALENDAR\r\n")) {
                if (previous != 0xFFFFFFFF) {
                    client->calendar.entries.info[previous].TailSig = NMAP_IDX_LINE_TAIL_SIGNATURE;

                    client->calendar.entries.info[previous].Type = eventType;
                    client->calendar.entries.info[previous].Pos = start;
                    client->calendar.entries.info[previous].Size = ftell(client->calendar.fh)-17-start;
                    client->calendar.entries.info[previous].State = calState;
                    client->calendar.entries.info[previous].CalSize = calSize;
                    client->calendar.entries.info[previous].Sequence = sequence;
                    client->calendar.entries.info[previous].UTCStart = utcStart;
                    client->calendar.entries.info[previous].UTCEnd = utcEnd;
                    client->calendar.entries.info[previous].UTCDue = utcDue;
                    client->calendar.entries.info[previous].Recurring = recurring;
                    client->calendar.entries.info[previous].RecurrenceID = recurrenceID;
                    client->calendar.entries.info[previous].UIDPos = uidPos;
                    client->calendar.entries.info[previous].AttendeePos = attendeePos;
                    client->calendar.entries.info[previous].AttendeeCount = attendeeCount;
                    client->calendar.entries.info[previous].AttendeeSize = attendeeSize;
                    client->calendar.entries.info[previous].OrganizerPos = organizerPos;
                    client->calendar.entries.info[previous].UID = client->calendar.nextUID++;

                    strcpy(client->calendar.entries.info[previous].ICalUID, iCalUID);
                    strcpy(client->calendar.entries.info[previous].Organizer, organizer);
                    strcpy(client->calendar.entries.info[previous].Summary, summary);

                    client->calendar.entries.count++;
                }

                if ((previous = (client->calendar.entries.used)++) < client->calendar.entries.allocated) {
                    size = 0;
                } else if (AddCalendarInfo(client, estCount)) {
                    estCount <<= 1;

                    size = 0;
                } else {
                    UnlockCalendar(client->store, client->user, client->calendar.name);

                    ConnWrite(client->conn, MSG5001NOMEMORY, sizeof(MSG5001NOMEMORY) - 1);

                    FreeClientData(client);
                    return(-1);
                }

                /* Prepare the next set */
                calSize = 0;
                utcStart = 0;
                utcEnd = 0;
                utcDue = 0;
                sequence = 0;
                recurring = FALSE;
                recurrenceID = 0;
                calState = 0;
                eventType = 0;
                organizerPos = 0;
                uidPos = 0;
                attendeePos = 0;
                attendeeCount = 0;
                attendeeSize = 0;
                iCalUID[0] = '\0';
                organizer[0] = '\0';
                summary[0] = '\0';

                start = ftell(client->calendar.fh) - 17;
            } else if (QuickCmp(client->line,   "END:VCALENDAR\r\n")) {
                calSize = ftell(client->calendar.fh) - start;
            } else if (QuickNCmp(client->line, "UID", 3)) {
                uidPos = ftell(client->calendar.fh) - len;
                if ((ptr = strchr(client->line + 3, ':')) == NULL) {
                    ptr = client->line + 3;
                }
                
                if (len > 43) {
                    strncpy(iCalUID, ptr + 1, 39);
                    iCalUID[39] = '\0';
                    CHOP_NEWLINE(iCalUID);
                } else {
                    memcpy(iCalUID, ptr + 1, len - 5);
                    iCalUID[len - 5] = '\0';
                    CHOP_NEWLINE(iCalUID);
                }
            } else if (QuickNCmp(client->line, "NIMS-UTCSTART:", 14)) {
                utcStart = atol(client->line + 14);
            } else if (QuickNCmp(client->line, "NIMS-UTCEND:", 12)) {
                utcEnd = atol(client->line + 12);
            } else if (QuickNCmp(client->line, "NIMS-UTCDUE:", 12)) {
                utcDue = atol(client->line + 12);
            } else if (QuickNCmp(client->line, "NIMS-RECURRING", 14)) {
                recurring = TRUE;
            } else if (QuickNCmp(client->line, "NIMS-RID:", 9)) {
                recurrenceID = atol(client->line + 9);
            } else if (QuickNCmp(client->line, "NIMS-TYPE:", 10)) {
                eventType = atol(client->line + 10);
            } else if (QuickNCmp(client->line, "NIMS-STATE:", 11)) {
                calState = atol(client->line + 11);
            } else if (QuickNCmp(client->line, "NIMS-ORGANIZER:", 15)) {
                organizerPos = ftell(client->calendar.fh)-len+15;
                ptr = strchr(client->line, ' ');
                if (ptr[1] == '\r') {
                    ptr[0] = '\0';
                    strncpy(organizer, client->line + 15, 39);
                } else {
                    strncpy(organizer, ptr + 1, 39);
                    CHOP_NEWLINE(organizer);
                }
            } else if (QuickNCmp(client->line, "NIMS-ATTENDEE:", 14)) {
                if (attendeePos == 0) {
                    attendeePos = ftell(client->calendar.fh) - len;
                }

                attendeeCount++;
            } else if (QuickNCmp(client->line, "NIMS-ENDATTENDEE", 16)) {
                attendeeSize = ftell(client->calendar.fh) - len - attendeePos;
            } else if (QuickNCmp(client->line, "SEQUENCE", 8)) {
                ptr = strchr(client->line+8, ':');
                if (ptr) {
                    sequence = atol(ptr+1);
                }
            } else if (QuickNCmp(client->line, "SUMMARY", 7)) {
                ptr = strchr(client->line+7, ':');
                if (ptr) {
                    strncpy(summary, ptr + 1, 39);
                    CHOP_NEWLINE(summary);
                }
            }
        }
    }

    /* Handle the remaining item */
    if (previous != 0xFFFFFFFF) {
        client->calendar.entries.info[previous].TailSig = NMAP_IDX_LINE_TAIL_SIGNATURE;

        client->calendar.entries.info[previous].Type = eventType;
        client->calendar.entries.info[previous].Pos = start;
        client->calendar.entries.info[previous].Size = ftell(client->calendar.fh)-start;
        client->calendar.entries.info[previous].State = calState;
        client->calendar.entries.info[previous].CalSize = calSize;
        client->calendar.entries.info[previous].Sequence = sequence;
        client->calendar.entries.info[previous].UTCStart = utcStart;
        client->calendar.entries.info[previous].UTCEnd = utcEnd;
        client->calendar.entries.info[previous].UTCDue = utcDue;
        client->calendar.entries.info[previous].Recurring = recurring;
        client->calendar.entries.info[previous].RecurrenceID = recurrenceID;
        client->calendar.entries.info[previous].UIDPos = uidPos;
        client->calendar.entries.info[previous].AttendeePos = attendeePos;
        client->calendar.entries.info[previous].AttendeeCount = attendeeCount;
        client->calendar.entries.info[previous].AttendeeSize = attendeeSize;
        client->calendar.entries.info[previous].OrganizerPos = organizerPos;
        client->calendar.entries.info[previous].UID = client->calendar.nextUID++;
        strcpy(client->calendar.entries.info[previous].ICalUID, iCalUID);
        strcpy(client->calendar.entries.info[previous].Organizer, organizer);
        strcpy(client->calendar.entries.info[previous].Summary, summary);

        client->calendar.entries.count++;
    }

    /* Write changed index file */
    if (!(client->flags & NMAP_CSTORE_READONLY)) {
        sprintf(client->path, "%s/%s/%s.idc",client->store, client->user, client->calendar.name);
        if (stat(client->path, &sb)!=0) {
            sb.st_size = 0;
        }

        if (sb.st_size != (int)(NMAP_CAL_IDX_HEADERSIZE + 1 + (client->calendar.entries.used * sizeof(CalendarInfoStruct)))) {
            index = fopen(client->path,"wb");
            fseek(client->calendar.fh, 0, SEEK_END);
            fprintf(index, NMAP_CAL_IDX_VERSION"\r\n");
            fprintf(index, "%010lu\r\n", client->calendar.flags);
            fprintf(index, "%010lu\r\n", ftell(client->calendar.fh));
            fprintf(index, "%010lu\r\n", client->calendar.entries.used);
            fprintf(index, "%010lu\r\n", client->calendar.id);
            fprintf(index, "%010lu\r\n", client->calendar.nextUID);

            for (i = 0, cInfo = client->calendar.entries.info; i < client->calendar.entries.used; i++, cInfo++) {
                fwrite(cInfo, sizeof(CalendarInfoStruct), 1, index);
            }

            if (!(NMAP.flushFlags & FLUSH_IDC_ON_PARSE)) {
                fclose(index);
            } else {
                XplFlushWrites(index);
                fclose(index);
            }
        }
    }

    fclose(client->calendar.fh);
    client->calendar.fh = NULL;

    UnlockCalendar(client->store, client->user, client->calendar.name);
    return(TRUE);
}

static BOOL
StoreCalendarAttendee(ICalObject *iCal, FILE *calendar)
{
    unsigned char rsvp;
    unsigned char role;
    unsigned char type;
    unsigned char state;
    unsigned char *address;
    unsigned char *delegator;
    unsigned char *delegated;
    unsigned char *name;
    unsigned long offset1;
    unsigned long offset2;
    unsigned long offset3;
    ICalVAttendee *attendee = iCal->Attendee;

    while (attendee) {
        if (attendee->RSVP) {
            rsvp = NMAP_CAL_RSVP_YES;
        } else {
            rsvp = NMAP_CAL_RSVP_NO;
        }

        switch (attendee->Type) {
            case ICAL_CUTYPE_INDIVIDUAL:            type = NMAP_CAL_TYPE_INDIVIDUAL;    break;
            case ICAL_CUTYPE_GROUP:                 type = NMAP_CAL_TYPE_GROUP;         break;
            case ICAL_CUTYPE_RESOURCE:              type = NMAP_CAL_TYPE_RESOURCE;      break;
            case ICAL_CUTYPE_ROOM:                  type = NMAP_CAL_TYPE_ROOM;          break;
            case ICAL_CUTYPE_UNKNOWN:               type = NMAP_CAL_TYPE_UNKNOWN;       break;
        }

        switch (attendee->State) {
            case ICAL_PARTSTAT_NEEDS_ACTION:        state = NMAP_CAL_STATE_NEED_ACT;    break;
            case ICAL_PARTSTAT_ACCEPTED:            state = NMAP_CAL_STATE_ACCEPTED;    break;
            case ICAL_PARTSTAT_DECLINED:            state = NMAP_CAL_STATE_DECLINED;    break;
            case ICAL_PARTSTAT_TENTATIVE:           state = NMAP_CAL_STATE_TENTATIVE;   break;
            case ICAL_PARTSTAT_DELEGATED:           state = NMAP_CAL_STATE_DELEGATED;   break;
            case ICAL_PARTSTAT_COMPLETED:           state = NMAP_CAL_STATE_COMPLETED;   break;
            case ICAL_PARTSTAT_IN_PROCESS:          state = NMAP_CAL_STATE_INPROCESS;   break;
        }

        switch(attendee->Role) {
            case ICAL_ROLE_CHAIR:                   role = NMAP_CAL_ROLE_CHAIR;         break;
            case ICAL_ROLE_REQUIRED_PARTICIPANT:    role = NMAP_CAL_ROLE_REQUIRED;      break;
            case ICAL_ROLE_OPTIONAL_PARTICIPANT:    role = NMAP_CAL_ROLE_OPTIONAL;      break;
            case ICAL_ROLE_NON_PARTICIPANT:         role = NMAP_CAL_ROLE_NOT;           break;
            
        }

        if (attendee->Address) {
            address = attendee->Address;
        } else {
            address = "";
        }

        if (attendee->Name) {
            name = attendee->Name;
        } else {
            name = "";
        }

        if (attendee->Delegated) {
            delegated = attendee->Delegated;
        } else {
            delegated = "";
        }

        if (attendee->DelegatedFrom) {
            delegator = attendee->DelegatedFrom;
        } else {
            delegator = "";
        }
        offset1 = NMAP_CAL_ADDRESS_POS + strlen(address) + 2;
        offset2 = offset1 + strlen(delegated) + 3;
        offset3 = offset2 + strlen(delegator) + 2;
        fprintf(calendar, "NIMS-ATTENDEE:%03d%03d%03d%c%c%c%c%s \"%s\" \"%s\" %s\r\n", 
                (int)offset1, (int)offset2, (int)offset3,
                state, role, rsvp, type,
                address, delegated, delegator, name);

        attendee = attendee->Next;
    }

    fprintf(calendar, "NIMS-ENDATTENDEE\r\n");

    return(TRUE);
}

int
StoreCalendarEvent(unsigned char *sender, unsigned char *authenticatedSender, unsigned char *recipient, unsigned char *calendarName, FILE *data, unsigned long eventSize, unsigned long flags, BOOL storeRaw)
{
    size_t count;
    unsigned long spaceUsed = 0;
    unsigned long enforcedQuota = 0;
    unsigned long pos;
    unsigned long type;
    unsigned char path[XPL_MAX_PATH + 1];
    unsigned char line[CONN_BUFSIZE + 1];
    unsigned char dn[MDB_MAX_OBJECT_CHARS + 1];
    const unsigned char *storePath;
    MDBValueStruct *user;
    FILE *calendar;
    ICalObject *iCal;

    storePath = MsgFindUserStore(recipient, NMAP.path.mail);
    sprintf(path, "%s/%s", storePath, recipient);

    if (NMAP.quota.useUser || NMAP.quota.useSystem) {
        spaceUsed = XplGetDiskspaceUsed(path);
        spaceUsed += (eventSize / 256 + 4) / 4;

        if (NMAP.quota.useUser) {
            user = MDBCreateValueStruct(NMAP.handle.directory, NULL);
            if (MsgFindObject(recipient, dn, NULL, NULL, user)) {
                MDBFreeValues(user);

                if ((MsgGetUserFeature(dn, FEATURE_NMAP_QUOTA, MSGSRV_A_USE_QUOTA, user) > 0) && (user->Value[0][0] == '1')) {
                    MDBFreeValues(user);
                    if (MsgGetUserFeature(dn, FEATURE_NMAP_QUOTA, MSGSRV_A_QUOTA_VALUE, user) > 0) {
                        enforcedQuota = atol(user->Value[0]);            
                    } else if (NMAP.quota.useSystem) {
                        enforcedQuota = NMAP.quota.defaultValue;
                    }
                } else if (NMAP.quota.useSystem) {
                    enforcedQuota = NMAP.quota.defaultValue;
                }
            }

            MDBDestroyValueStruct(user); 
        } else {
            enforcedQuota = NMAP.quota.defaultValue;    
        } 

        if (enforcedQuota > 0) {
            if ((spaceUsed > enforcedQuota) && ((recipient[0] != NMAP.postMaster[0]) || (XplStrCaseCmp(recipient, NMAP.postMaster) != 0))) {
                LoggerEvent(NMAP.handle.logging, LOGGER_SUBSYSTEM_GENERAL, LOGGER_EVENT_OVER_QUOTA, LOG_INFO, 0, recipient, sender, spaceUsed, enforcedQuota, NULL, 0);
                return(DELIVER_QUOTA_EXCEEDED);
            } 
        }
    }

    count = WaitforAndLockCalendar(storePath, recipient, calendarName);
    if (count == 1) {
        sprintf(path, "%s/%s/%s.cal", storePath, recipient, calendarName);
        calendar = fopen(path, "ab");
        if (calendar != NULL) {
            /* LoggerEvent(NMAP.handle.logging, LOGGER_SUBSYSTEM_CALENDAR, LOGGER_EVENT_ITEM_STORED, LOG_INFO, 0, recipient, authenticatedSender? authenticatedSender: sender, eventSize, 0, MIME_TEXT_PLAIN, strlen(calendarName) + 1, calendarName);*/
        } else {
            /* If calendar doesn't exist, we deliver into MAIN    */
            sprintf(path, "%s/%s/MAIN.cal", storePath, recipient);
            calendar = fopen(path, "ab");
            if (calendar != NULL) {
                /* LoggerEvent(NMAP.handle.logging, LOGGER_SUBSYSTEM_CALENDAR, LOGGER_EVENT_ITEM_STORED, LOG_INFO, 0, recipient, authenticatedSender? authenticatedSender: sender, eventSize, 0, MIME_TEXT_PLAIN, strlen(calendarName) + 1, calendarName); */
            } else {
                UnlockCalendar(storePath, recipient, calendarName);
                return(DELIVER_TRY_LATER);
            }
        }
    } else if (count == 0) {
        /* If recipient doesn't exist, we deliver into INBOX and create the directory entries. */
        sprintf(path, "%s/%s", storePath, recipient);
        XplMakeDir(path);

        sprintf(path, "%s/%s/MAIN.cal", storePath, recipient);
        calendar = fopen(path, "ab");
        if ((calendar != NULL) && (WaitforAndLockCalendar(storePath, recipient, "MAIN") == 1)) {
            /* LoggerEvent(NMAP.handle.logging, LOGGER_SUBSYSTEM_CALENDAR, LOGGER_EVENT_ITEM_STORED, LOG_INFO, 0, recipient, authenticatedSender? authenticatedSender: sender, eventSize, 0, MIME_TEXT_PLAIN, strlen(calendarName) + 1, "MAIN"); */
        } else {
            return(DELIVER_TRY_LATER);
        }
    } else {
        return(DELIVER_TRY_LATER);
    }

    /* Find the start of the ical object */
    do {
        pos = ftell(data);
        if (fgets(line, CONN_BUFSIZE, data) != NULL) {
            if (QuickCmp(line, "BEGIN:VCALENDAR\r\n")) {
                break;
            }
        }
    } while (!feof(data) && !ferror(data));

    if (feof(data)) {
        UnlockCalendar(storePath, recipient, calendarName);
        fclose(calendar);
        return(DELIVER_PROCESSING_ERROR);
    }

    fseek(data, pos, SEEK_SET);

    /* Here's where we parse the ical object */
    pos = ftell(data);
    iCal = ICalParseObject(data, NULL, eventSize);

    /* Fix the type */
    if (iCal->Type == ICAL_VTODO) {
        type = NMAP_CAL_TODO;
    } else if (iCal->Type == ICAL_VJOURNAL) {
        type = NMAP_CAL_JOURNAL;
    } else {
        type = NMAP_CAL_EVENT;
    }

    if (!ICalHasRule(iCal)) {
        /* Just copy the thing and write our special attributes */
        fseek(data, pos, SEEK_SET);
        while (!feof(data)) {
            count = fread(line, sizeof(unsigned char), CONN_BUFSIZE, data);
            if (count > 0) {
                if (fwrite(line, sizeof(unsigned char), count, calendar) < count) {
                    fclose(calendar);
                    UnlockCalendar(storePath, recipient, calendarName);
                    return(DELIVER_QUOTA_EXCEEDED);
                }
            }
        }

        /* Make sure there's a CR/LF at the end of the entry */
        if (count > 0) {
            if (line[count - 1] != LF) {
                fprintf(calendar, "\r\n");
            }
        }

        if (storeRaw == FALSE) {
            /* Now write our private data */
            if (type != NMAP_CAL_TODO) {
                fprintf(calendar, "NIMS-UTCEND:%010lu\r\nNIMS-UTCSTART:%010lu\r\nNIMS-TYPE:%d\r\n", iCal->End.UTC, iCal->Start.UTC, (int)type);
            } else {
                if (iCal->Completed.UTC == 0) {
                    fprintf(calendar, "NIMS-UTCEND:2147483647\r\nNIMS-UTCSTART:%010lu\r\nNIMS-UTCDUE:%010lu\r\nNIMS-TYPE:%d\r\n", iCal->Start.UTC, iCal->End.UTC, (int)type);
                } else {
                    fprintf(calendar, "NIMS-UTCEND:%010lu\r\nNIMS-UTCSTART:%010lu\r\nNIMS-UTCDUE:%010lu\r\nNIMS-TYPE:%d\r\n", iCal->Completed.UTC, iCal->Start.UTC, iCal->End.UTC, (int)type);
                }
            }

            if (iCal->Attendee) {
                StoreCalendarAttendee(iCal, calendar);
            }

            if (iCal->Organizer && iCal->Organizer->Address) {
                fprintf(calendar, "NIMS-ORGANIZER:%s %s\r\n\r\n", iCal->Organizer->Address, iCal->Organizer->Name? (char *)iCal->Organizer->Name: "");
            } else {
                fprintf(calendar, "\r\n");
            }
        }
    } else {
        int len1;
        int len2;
        unsigned long day;
        unsigned long month;
        unsigned long year;
        unsigned long h;
        unsigned long m;
        unsigned long s;
        unsigned long state = 0;
        unsigned long rid;
        unsigned long iterations = 0;
        unsigned char *originalRRule = NULL;
        unsigned char *originalDTStart = NULL;
        unsigned char *originalDTEnd = NULL;
        unsigned char *originalDTDue = NULL;
        unsigned char originalType = 0;
        BOOL more;
        BOOL skip = FALSE;
        MDBValueStruct *uid;
        ICalVRuleIterator iterator;

        /*
            We need to flatten the recurrence rule, writing the whole entry for every recurrence of the event
            While rewriting the data, we'll replace the DTSTART and DTEND with the data for the event; and
            drop any RRULE line...; we also need to add our own lines at the end.
        */
        rid = NMAP.universalCounter++;
        sprintf(line, "%lu", NMAP.universalCounter);

        uid = MDBCreateValueStruct(NMAP.handle.directory, NMAP.server.dn);
        MDBAddValue(line, uid);
        MDBWrite(MSGSRV_AGENT_NMAP, MSGSRV_A_UID, uid);
        MDBDestroyValueStruct(uid);

        more = ICalFirstRuleInstance(iCal, &iterator);
        while (more) {
            fseek(data, pos, SEEK_SET);
            while (!feof(data)) {
                if (fgets(line, CONN_BUFSIZE, data)!=NULL) {
                    if (skip) {
                        if (!isspace(line[0])) {
                            skip = FALSE;
                            originalType = 0;
                        } else {
                            switch(originalType) {
                                case 1: {
                                    CHOP_NEWLINE(originalRRule);

                                    len1 = strlen(originalRRule);
                                    len2 = strlen(line);

                                    originalRRule = MemRealloc(originalRRule, len1 + len2 + 1);
                                    if (!originalRRule) {
                                        continue;
                                    }

                                    memcpy(originalRRule + len1, line + 1, len2);
                                    break;
                                }

                                case 2: {
                                    CHOP_NEWLINE(originalDTStart);

                                    len1 = strlen(originalDTStart);
                                    len2 = strlen(line);

                                    originalDTStart = MemRealloc(originalDTStart, len1 + len2 + 1);
                                    if (!originalDTStart) {
                                        continue;
                                    }

                                    memcpy(originalDTStart + len1, line + 1, len2);
                                    break;
                                }

                                case 3: {
                                    CHOP_NEWLINE(originalDTEnd);

                                    len1 = strlen(originalDTEnd);
                                    len2 = strlen(line);

                                    originalDTEnd = MemRealloc(originalDTEnd, len1 + len2 + 1);
                                    if (!originalDTEnd) {
                                        continue;
                                    }

                                    memcpy(originalDTEnd + len1, line + 1, len2);
                                    break;
                                }

                                case 4: {
                                    CHOP_NEWLINE(originalDTDue);

                                    len1 = strlen(originalDTDue);
                                    len2 = strlen(line);

                                    originalDTEnd = MemRealloc(originalDTDue, len1 + len2 + 1);
                                    if (!originalDTDue) {
                                        continue;
                                    }

                                    memcpy(originalDTDue + len1, line + 1, len2);
                                    break;
                                }

                            }
                            continue;
                        }
                    }

                    if (QuickNCmp(line, "BEGIN:VTIMEZONE", 15)) {
                        state = 1;
                        fwrite(line, strlen(line), 1, calendar);
                    } else if (QuickNCmp(line, "END:VTIMEZONE", 13)) {
                        state = 0;
                        fwrite(line, strlen(line), 1, calendar);
                    } else if (state != 1) {
                        if (QuickNCmp(line, "DTSTART", 7)) {
                            if (!originalDTStart) {
                                originalDTStart = MemStrdup(line);
                                originalType = 2;
                            }

                            MsgGetDMY(iterator.UTC, &day, &month, &year, &h, &m, &s);
                            fprintf(calendar, "DTSTART:%04d%02d%02dT%02d%02d%02dZ\r\n", (int)year, (int)month, (int)day, (int)h, (int)m, (int)s);

                            skip = TRUE;
                        } else if (QuickNCmp(line, "DTEND", 5)) {
                            if (!originalDTEnd) {
                                originalDTEnd = MemStrdup(line);
                                originalType = 3;
                            }

                            MsgGetDMY(iterator.UTC + iterator.Duration, &day, &month, &year, &h, &m, &s);
                            fprintf(calendar, "DTEND:%04d%02d%02dT%02d%02d%02dZ\r\n", (int)year, (int)month, (int)day, (int)h, (int)m, (int)s);

                            skip = TRUE;
                        } else if (QuickNCmp(line, "DUE", 3)) {
                            if (!originalDTDue) {
                                originalDTDue = MemStrdup(line);
                                originalType = 4;
                            }

                            MsgGetDMY(iterator.UTC + iterator.Duration, &day, &month, &year, &h, &m, &s);
                            fprintf(calendar, "DUE:%04d%02d%02dT%02d%02d%02dZ\r\n", (int)year, (int)month, (int)day, (int)h, (int)m, (int)s);

                            skip = TRUE;
                        } else if (QuickNCmp(line, "RRULE", 5)) {
                            if (!originalRRule) {
                                originalRRule = MemStrdup(line);
                                originalType = 1;
                            }

                            skip = TRUE;
                        } else {
                            fwrite(line, strlen(line), 1, calendar);
                        }
                    } else {
                        fwrite(line, strlen(line), 1, calendar);
                    }
                }
            }

            if (storeRaw == FALSE) {
                if (type != NMAP_CAL_TODO) {
                    fprintf(calendar, "NIMS-UTCEND:%010lu\r\nNIMS-UTCSTART:%010lu\r\nNIMS-TYPE:%d\r\nNIMS-RECURRING\r\nNIMS-RID:%lu\r\n", 
                        iterator.UTC + iterator.Duration, iterator.UTC, (int)type, rid);
                } else {
                    fprintf(calendar, "NIMS-UTCEND:2147483647\r\nNIMS-UTCDUE:%010lu\r\nNIMS-UTCSTART:%010lu\r\nNIMS-TYPE:%d\r\nNIMS-RECURRING\r\nNIMS-RID:%lu\r\n", 
                        iterator.UTC + iterator.Duration, iterator.UTC, (int)type, rid);
                }

                if (iCal->Attendee) {
                    StoreCalendarAttendee(iCal, calendar);
                }

                if (originalRRule) {
                    fprintf(calendar, "NIMS-ORG-RRULE: %s", originalRRule);
                }

                if (originalDTStart) {
                    fprintf(calendar, "NIMS-ORG-DTSTART: %s", originalDTStart);
                }

                if (originalDTEnd) {
                    fprintf(calendar, "NIMS-ORG-DTEND: %s", originalDTEnd);
                }

                if (originalDTDue) {
                    fprintf(calendar, "NIMS-ORG-DTDUE: %s", originalDTDue);
                }

                if (iCal->Organizer && iCal->Organizer->Address) {
                    fprintf(calendar, "NIMS-ORGANIZER:%s %s\r\n\r\n", iCal->Organizer->Address, iCal->Organizer->Name? (char *)iCal->Organizer->Name: "");
                } else {
                    fprintf(calendar, "\r\n");
                }
            }

            spaceUsed += (eventSize / 256 + 4) / 4;

            more = ICalNextRuleInstance(iCal, &iterator);

            if ((enforcedQuota > 0) && (spaceUsed > enforcedQuota) && ((recipient[0] != NMAP.postMaster[0]) || (XplStrCaseCmp(recipient, NMAP.postMaster) != 0))) {
                LoggerEvent(NMAP.handle.logging, LOGGER_SUBSYSTEM_GENERAL, LOGGER_EVENT_OVER_QUOTA, LOG_INFO, 0, recipient, sender, spaceUsed, enforcedQuota, NULL, 0);
                return(DELIVER_QUOTA_EXCEEDED);
            }

            if (++iterations > CALENDAR_MAX_ITERATIONS) {
                more = FALSE;
            }
        }

        if (originalRRule) {
            MemFree(originalRRule);
        }

        if (originalDTStart) {
            MemFree(originalDTStart);
        }
        if (originalDTEnd) {
            MemFree(originalDTEnd);
        }
        if (originalDTDue) {
            MemFree(originalDTDue);
        }
    }

    ICalFreeObjects(iCal);

    XplSafeAdd(NMAP.stats.storedLocal.bytes, eventSize);
    XplSafeIncrement(NMAP.stats.storedLocal.count);

    fwrite("\r\n", sizeof(unsigned char), 2, calendar);

    if (!(NMAP.flushFlags & FLUSH_CAL_ON_STORE)) {
        fclose(calendar);
    } else {
        XplFlushWrites(calendar);
        fclose(calendar);
    }

    UnlockCalendar(storePath, recipient, calendarName);

    XplSafeIncrement(NMAP.stats.storedLocal.count);
    XplSafeIncrement(NMAP.stats.storedLocal.recipients);

    return(DELIVER_SUCCESS);
}

__inline static void 
CalendarOOBNotifications(NMAPClient *client, struct stat *sb)
{
    if (!(client->share.flags & NMAP_SHARE_CALENDAR)) {
        if (client->calendar.indexChanged == FALSE) {
            sprintf(client->path, "%s/%s/%s.idc", client->store, client->user, client->calendar.name);
            if ((stat(client->path, sb) == 0) && (client->calendar.indexTime != sb->st_mtime)) {
                client->calendar.indexChanged = TRUE;
                client->calendar.indexTime = sb->st_mtime;
            }
        }
    } else if (IsAvailableShare(client->user, client->calendar.name, NMAP_SHARE_CALENDAR, NULL, NULL)) {
        BOOL changed = FALSE;

        /* Synchronizing the shared resource may change the value for client->mailbox.indexTime. */
        if (SyncSharedCalendar(client, &changed, FALSE) && (changed)) {
            client->calendar.indexChanged = TRUE;
        }
    } else {
        /* This user no longer has permissions to the shared mailbox. */
        client->share.flags |= NMAP_SHARE_CALENDAR_DENIED;
    }

    return;
}


int 
NmapCommandCsopen(void *param)
{
    int ccode;
    unsigned char *ptr;
    unsigned char *resourceOwner;
    unsigned char *resourceName;
    struct stat sb;
    FILE *targetBox;
    BOOL result;
    NMAPClient *client = (NMAPClient *)param;
    
    if (client->states & NMAP_CLIENT_USER) {
        ptr = client->buffer + 6;
    } else {
        return(ConnWrite(client->conn, MSG3241NOUSER, sizeof(MSG3241NOUSER) - 1));
    }

    if ((*ptr++ == ' ') 
            && (*ptr) 
            && (!isspace(*ptr)) 
            && ((strchr(ptr, '.') == NULL) || ((strstr(ptr, "../") == NULL) && (strstr(ptr, "..\\") == NULL)))) {
        if (client->calendar.name[0] == '\0') {
            if (ReadNLockAquire(&client->calendar.lock, &client->userHash, client->user, ptr, 0) == NLOCK_AQUIRED) {
                sprintf(client->path, "%s/%s/%s.cal", client->store, client->user, ptr);
                targetBox = fopen(client->path, "rb");
            } else {
                return(ConnWrite(client->conn, MSG4120CALLOCKED, sizeof(MSG4120CALLOCKED) - 1));
            }
        } else {
            CalendarOOBNotifications(client, &sb);

            return(ConnWrite(client->conn, MSG4221CALDONE, sizeof(MSG4221CALDONE) - 1));
        }
    } else {
        return(ConnWrite(client->conn, MSG3010BADARGC, sizeof(MSG3010BADARGC) - 1));
    }

    if (targetBox != NULL) {
        fclose(targetBox);
        strcpy(client->calendar.name, ptr);

        if (ParseCalendar(client)) {
            if (!(client->flags & NMAP_CSTORE_READONLY)) {
                ccode = ConnWriteF(client->conn, "1000 %lu %s\r\n",client->calendar.id, MSG1000MBOX);
            } else {
                ccode = ConnWriteF(client->conn, "1020 %lu %s\r\n",client->calendar.id, MSG1020MBOX);
            }
        } else {
            client->calendar.name[0] = '\0';
            ccode = ConnWrite(client->conn, MSG4120CALLOCKED, sizeof(MSG4120CALLOCKED) - 1);
        }
    } else if (XplStrCaseCmp(ptr, "MAIN") == 0) {
        memcpy(ptr, "MAIN", 4);

        sprintf(client->path, "%s/%s/MAIN.cal", client->store, client->user);
        targetBox = fopen(client->path, "rb");
        if (targetBox != NULL) {
            fclose(targetBox);
            strcpy(client->calendar.name, "MAIN");
            if (ParseCalendar(client)) {
                if (!(client->flags & NMAP_CSTORE_READONLY)) {
                    ccode = ConnWriteF(client->conn, "1000 %lu %s\r\n", client->calendar.id, MSG1000MBOX);
                } else {
                    ccode = ConnWriteF(client->conn, "1020 %lu %s\r\n", client->calendar.id, MSG1020MBOX);
                }
            } else {
                client->calendar.name[0] = '\0';
                ccode = ConnWrite(client->conn, MSG4120CALLOCKED, sizeof(MSG4120CALLOCKED) - 1);
            }
        } else {
            client->calendar.name[0] = '\0';
            ccode = ConnWrite(client->conn, MSG4224NOCSTORE, sizeof(MSG4224NOCSTORE) - 1);
        }
    } else {
        if (IsAvailableShare(client->user, ptr, NMAP_SHARE_CALENDAR, &resourceOwner, &resourceName)) {
            ccode = ConnectToNMAPShare(client, resourceOwner, resourceName, NMAP_SHARE_CALENDAR, &(client->share.calendar.permissions));
            if ((ccode == 1000) && (client->share.calendar.permissions & NMAP_SHARE_READ)) {
                strcpy(client->calendar.name, ptr);

#if 0
                if (client->share.calendar.permissions & NMAP_SHARE_WRITE) {
                    client->flags &= ~NMAP_CSTORE_READONLY;
                } else {
                    client->flags |= NMAP_CSTORE_READONLY;
                }
#endif

                result = SelectNMAPShare(client, resourceOwner, resourceName, NMAP_SHARE_CALENDAR);
                if (result) {
                    client->share.flags |= NMAP_SHARE_CALENDAR;

                    if (!(client->flags & NMAP_CSTORE_READONLY)) {
                        ccode = ConnWriteF(client->conn, "1000 %lu %s\r\n", client->calendar.id, MSG1000MBOX);
                    } else {
                        ccode = ConnWriteF(client->conn, "1020 %lu %s\r\n", client->calendar.id, MSG1020MBOX);
                    }
                } else {
                    client->calendar.name[0] = '\0';

                    if (client->share.calendar.conn) {
                        NMAPQuit(client->share.calendar.conn);
                        ConnFree(client->share.calendar.conn);
                        client->share.calendar.conn = NULL;
                    }

                    if (client->calendar.entries.info != NULL) {
                        MemFree(client->calendar.entries.info);
                        client->calendar.entries.info = NULL;
                    }

                    client->calendar.entries.allocated = 0;
                    client->calendar.entries.count = 0;
                    client->calendar.entries.used = 0;

                    ccode = ConnWrite(client->conn, MSG4120CALLOCKED, sizeof(MSG4120CALLOCKED) - 1);
                }
            } else if (ccode == 1000) {
                NMAPQuit(client->share.calendar.conn);
                ConnFree(client->share.calendar.conn);

                client->calendar.name[0] = '\0';
                client->share.calendar.permissions = 0;
                client->share.flags &= ~(NMAP_SHARE_CALENDAR | NMAP_SHARE_CALENDAR_DENIED);

                ccode = ConnWrite(client->conn, MSG4240NOPERMISSION, sizeof(MSG4240NOPERMISSION) - 1);
            } else {
                client->calendar.name[0] = '\0';
                ConnWrite(client->conn, MSG4224CANTREADCAL, sizeof(MSG4224CANTREADCAL) - 1);
            }

            if (resourceOwner) {
                MemFree(resourceOwner);
            }

            if (resourceName) {
                MemFree(resourceName);
            }
        } else {
            client->calendar.name[0] = '\0';
            ccode = ConnWrite(client->conn, MSG4224NOCSTORE, sizeof(MSG4224NOCSTORE) - 1);
        }
    }

    if (client->calendar.name[0] == '\0') {
        ReadNLockRelease(client->calendar.lock);
        client->calendar.lock = NULL;
    }

    return(ccode);
}

int 
NmapCommandCal(void *param)
{
    int count;
    unsigned char *ptr;
    unsigned char *ptr2;
    NMAPClient *client = (NMAPClient *)param;

    ptr = client->buffer + 3;
    count = strlen(ptr);
    ptr += count;
    ptr2 = ptr + 3;

    do {
        *ptr2-- = *ptr--;
    } while (count-- != 0);

    memcpy(client->buffer, "CSOPEN", 6);

    return(NmapCommandCsopen(param));
}

int 
NmapCommandCsorg(void *param)
{
    int ccode;
    unsigned long id;
    unsigned char *ptr;
    struct stat sb;
    CalendarInfoStruct *cInfo;
    NMAPClient *client = (NMAPClient *)param;

    if (client->states & NMAP_CLIENT_USER) {
        if (client->calendar.name[0] != '\0') {
            CalendarOOBNotifications(client, &sb);

            ptr = client->buffer + 5;
        } else {
            return(ConnWrite(client->conn, MSG4225NOCALENDAR, sizeof(MSG4225NOCALENDAR) - 1));
        }
    } else {
        return(ConnWrite(client->conn, MSG3241NOUSER, sizeof(MSG3241NOUSER) - 1));
    }

    if ((*ptr++ == ' ') && (isdigit(*ptr))) {
        id = atol(ptr);
    } else {
        return(ConnWrite(client->conn, MSG3010BADARGC, sizeof(MSG3010BADARGC) - 1));
    }

    if (id && (id <= client->calendar.entries.used)) {
        id--;
    } else {
        return(ConnWrite(client->conn, MSG4220NOMSG, sizeof(MSG4220NOMSG) - 1));
    }

    if (!(client->share.flags & NMAP_SHARE_CALENDAR)) {
        cInfo = &(client->calendar.entries.info[id]);

        if (cInfo->OrganizerPos != 0) {
            sprintf(client->path, "%s/%s/%s.cal", client->store, client->user, client->calendar.name);
            client->calendar.fh = fopen(client->line, "rb");
            if (client->calendar.fh != NULL) {
                fseek(client->calendar.fh, cInfo->OrganizerPos, SEEK_SET);
            } else {
                return(ConnWrite(client->conn, MSG4224CANTREADCAL, sizeof(MSG4224CANTREADCAL) - 1));
            }

            strcpy(client->line, "2001 ");
            fgets(client->line + 5, CONN_BUFSIZE - 5, client->calendar.fh);
            fclose(client->calendar.fh);
            client->calendar.fh = NULL;

            ccode = ConnWrite(client->conn, client->line, strlen(client->line));
        } else {
            ccode = ConnWrite(client->conn, "2001 \r\n", 7);
        }
    } else if (!(client->share.flags & NMAP_SHARE_CALENDAR_DENIED)) {
        if (NMAPSendCommandF(client->share.calendar.conn, "CSORG %lu\r\n", id + 1) != -1) {
            ccode = ConnReadLine(client->share.calendar.conn, client->line, CONN_BUFSIZE);
            if (ccode != -1) {
                ccode = ConnWrite(client->conn, client->line, strlen(client->line));
            }
        } else {
            ccode = ConnWrite(client->conn, MSG4224CANTREADCAL, sizeof(MSG4224CANTREADCAL) - 1);
        }
    } else {
        ccode = ConnWrite(client->conn, MSG4240NOPERMISSION, sizeof(MSG4240NOPERMISSION) - 1);
    }

    return(ccode);
}

int 
NmapCommandCsinfo(void *param)
{
    int ccode;
    unsigned long id;
    unsigned long cUsed;
    unsigned char *ptr;
    CalendarInfoStruct *cInfo;
    struct stat sb;
    NMAPClient *client = (NMAPClient *)param;

    if (client->states & NMAP_CLIENT_USER) {
        if (client->calendar.name[0] != '\0') {
            CalendarOOBNotifications(client, &sb);

            ptr = client->buffer + 6;
        } else {
            return(ConnWrite(client->conn, MSG4225NOCALENDAR, sizeof(MSG4225NOCALENDAR) - 1));
        }
    } else {
        return(ConnWrite(client->conn, MSG3241NOUSER, sizeof(MSG3241NOUSER) - 1));
    }

    if (ptr[0] == '\0') {
        ccode = ConnWriteF(client->conn, "2002 %lu\r\n", client->calendar.entries.used);
        for (id = 0, cUsed = client->calendar.entries.used, cInfo = client->calendar.entries.info; (ccode != -1) && (id < cUsed); id++, cInfo++) {
            ccode = ConnWriteF(client->conn, "%lu %lu %lu %lu %lu %lu %lu %lu %lu %lu %lu %lu %s \"%s\" \"%s\"\r\n",
                id + 1,
                (long unsigned int)cInfo->Type,
                cInfo->Size,
                cInfo->CalSize,
                cInfo->State,
                cInfo->UTCStart,
                (cInfo->UTCEnd != 0x7fffffff)? cInfo->UTCEnd : cInfo->UTCDue,
                cInfo->Sequence,
                (long unsigned int)cInfo->Recurring,
                cInfo->RecurrenceID,
                cInfo->UID,
                client->calendar.id,
                cInfo->ICalUID,
                cInfo->Organizer,
                cInfo->Summary);
        }

        if (ccode != -1) {
            ccode = ConnWrite(client->conn, MSG1000OK, sizeof(MSG1000OK) - 1);
        }

        return(ccode);
    } else if ((ptr[0] != ' ') || (!isdigit(ptr[1]))) {
        return(ConnWrite(client->conn, MSG3010BADARGC, sizeof(MSG3010BADARGC) - 1));
    }

    id = atol(++ptr);
    if (id && (id <= client->calendar.entries.used)) {
        id--;
    } else {
        return(ConnWrite(client->conn, MSG4220NOMSG, sizeof(MSG4220NOMSG) - 1));
    }

    cInfo = &(client->calendar.entries.info[id]);
    ccode = ConnWriteF(client->conn, "2001 %lu %lu %lu %lu %lu %lu %lu %lu %lu %lu %lu %lu %s \"%s\" \"%s\"\r\n",
        id + 1,
        (long unsigned int)cInfo->Type,
        cInfo->Size,
        cInfo->CalSize,
        cInfo->State,
        cInfo->UTCStart,
        (cInfo->UTCEnd != 0x7fffffff)? cInfo->UTCEnd : cInfo->UTCDue,
        cInfo->Sequence,
        (long unsigned int)cInfo->Recurring,
        cInfo->RecurrenceID,
        cInfo->UID,
        client->calendar.id,
        cInfo->ICalUID,
        cInfo->Organizer,
        cInfo->Summary);

    return(ccode);
}

int 
NmapCommandCsfilt(void *param)
{
    int ccode = 0;
    unsigned long id;
    unsigned long utcStart;
    unsigned long utcEnd;
    unsigned long cUsed;
    unsigned char *ptr;
    unsigned char *ptr2;
    CalendarInfoStruct *cInfo;
    struct stat sb;
    NMAPClient *client = (NMAPClient *)param;

    if (client->states & NMAP_CLIENT_USER) {
        if (client->calendar.name[0] != '\0') {
            CalendarOOBNotifications(client, &sb);

            ptr = client->buffer + 6;
        } else {
            return(ConnWrite(client->conn, MSG4225NOCALENDAR, sizeof(MSG4225NOCALENDAR) - 1));
        }
    } else {
        return(ConnWrite(client->conn, MSG3241NOUSER, sizeof(MSG3241NOUSER) - 1));
    }

    if ((*ptr++ == ' ') && ((ptr2 = strchr(ptr, ' ')) != NULL)) {
        *ptr2++ = '\0';

        utcStart = atol(ptr);
        utcEnd = atol(ptr2);
    } else {
        return(ConnWrite(client->conn, MSG3010BADARGC, sizeof(MSG3010BADARGC) - 1));
    }

    for (cUsed = client->calendar.entries.used, id = 0, cInfo = client->calendar.entries.info; (ccode != -1)  && (id++ < cUsed); cInfo++) {
        if (!(cInfo->State & (MSG_STATE_PURGED | MSG_STATE_DELETED))) {
            if (((cInfo->UTCStart >= utcStart) && (cInfo->UTCStart <= utcEnd))
                    || ((cInfo->UTCEnd >= utcStart) && (cInfo->UTCEnd <= utcEnd))
                    || ((cInfo->UTCStart <= utcStart) && (cInfo->UTCEnd >= utcEnd))) {
                if (cInfo->UTCEnd != 0x7fffffff) {
                    ccode = ConnWriteF(client->conn, "2002-%lu %lu %lu %lu 0 %lu %lu\r\n",
                        id,
                        (long unsigned int)cInfo->Type,
                        cInfo->UTCStart,
                        cInfo->UTCEnd,
                        (long unsigned int)cInfo->Recurring,
                        cInfo->State);
                } else {
                    ccode = ConnWriteF(client->conn, "2002-%lu %lu %lu %lu %lu %lu %lu\r\n",
                        id,
                        (long unsigned int)cInfo->Type,
                        cInfo->UTCStart,
                        cInfo->UTCEnd,
                        cInfo->UTCDue,
                        (long unsigned int)cInfo->Recurring,
                        cInfo->State);
                }
            }
        }
    }

    if (ccode != -1) {
        ccode = ConnWrite(client->conn, MSG1000OK, sizeof(MSG1000OK) - 1);
    }

    return(ccode);
}

int 
NmapCommandCsfind(void *param)
{
    int ccode;
    unsigned long i;
    unsigned long id;
    unsigned long len;
    unsigned long count;
    unsigned long sequence;
    unsigned long utcStart;
    unsigned char *uid;
    unsigned char *ptr;
    unsigned char *ptr2;
    unsigned char buffer[CONN_BUFSIZE + 1];
    BOOL haveMatch = FALSE;
    struct stat sb;
    CalendarInfoStruct *cInfo;
    CalendarInfoStruct *cInfo2;
    NMAPClient *client = (NMAPClient *)param;

    
    if (client->states & NMAP_CLIENT_USER) {
        if (client->calendar.name[0] != '\0') {
            CalendarOOBNotifications(client, &sb);

            ptr = client->buffer + 6;
        } else {
            return(ConnWrite(client->conn, MSG4225NOCALENDAR, sizeof(MSG4225NOCALENDAR) - 1));
        }
    } else {
        return(ConnWrite(client->conn, MSG3241NOUSER, sizeof(MSG3241NOUSER) - 1));
    }

    if ((*ptr++ == ' ') && ((ptr2 = strchr(ptr, ' ')) != NULL)) {
        *ptr2++ = '\0';
        sequence = atol(ptr);
    } else {
        return(ConnWrite(client->conn, MSG3010BADARGC, sizeof(MSG3010BADARGC) - 1));
    }

    ptr = strchr(ptr2, ' ');
    if (ptr != NULL) {
        *ptr++ = '\0';

        utcStart = atol(ptr);
    } else {
        utcStart = 0;
    }

    uid = ptr2;

    if (!(client->share.flags & NMAP_SHARE_CALENDAR)) {
        len = strlen(uid);
        len = (len < (sizeof(cInfo->ICalUID) - 1))? len: sizeof(cInfo->ICalUID) - 1;

        /* fixme - improve handling of ICal uid's */
        for (id = 0, cInfo = client->calendar.entries.info; (id < client->calendar.entries.used) && (haveMatch == FALSE); id++, cInfo++) {
            if (!(cInfo->State & (MSG_STATE_PURGED | MSG_STATE_DELETED)) && (cInfo->Sequence <= sequence)) {
                if (QuickNCmp(cInfo->ICalUID, uid, len)) {
                    if (len < (sizeof(cInfo->ICalUID) - 1)) {
                        haveMatch = TRUE;
                        break;
                    } else {
                        /* We found an entry where the first n uid chars match, make sure the rest matches as well */
                        sprintf(client->path, "%s/%s/%s.cal", client->store, client->user, client->calendar.name);
                        client->calendar.fh = fopen(client->path, "rb");
                        if (client->calendar.fh != NULL) {
                            fseek(client->calendar.fh, cInfo->Pos, SEEK_SET);
                        } else {
                            return(ConnWrite(client->conn, MSG4224CANTREADCAL, sizeof(MSG4224CANTREADCAL) - 1));
                        }

                        len = 0;
                        count = 0;
                        while (count < cInfo->CalSize) {
                            fgets(client->line, CONN_BUFSIZE, client->calendar.fh);
                            len = strlen(client->line);
                            count += len;
                            if ((client->line[0] == 'U') 
                                    && (client->line[1] == 'I') 
                                    && (client->line[2] == 'D') 
                                    && ((client->line[3] == ':') || (client->line[3] == ';'))) {
                                /* keep reading more, the uid might be longer than one line */
                                while ((CONN_BUFSIZE - len) > 3) {
                                    fgets(buffer, CONN_BUFSIZE - len, client->calendar.fh);

                                    if (isspace(buffer[0])) {
                                        /* Append to the end of client->line; overwrite the CR/LF at the end of client->line */
                                        i = strlen(buffer + 1);
                                        memcpy(client->line + len - 2, buffer + 1, i + 1);
                                        len += i - 2;

                                        continue;
                                    }

                                    break;
                                }

                                CHOP_NEWLINE(client->line);

                                /* Need to do it this way; might have additional properties after uid */
                                ptr = strchr(client->line + 3, ':');
                                if (QuickCmp(ptr + 1, uid)) {
                                    haveMatch = TRUE;
                                    break;
                                }
                            }
                        }

                        fclose(client->calendar.fh);
                        client->calendar.fh = NULL;

                        if (haveMatch) {
                            break;
                        }
                    }
                }
            }
        }

        if (haveMatch) {
            if ((cInfo->Recurring == FALSE) || (/*    (cInfo->Recurring) &&*/ utcStart == 0)) {
                return(ConnWriteF(client->conn, "1000 %lu found\r\n", id + 1));
            }

            cInfo2 = &(client->calendar.entries.info[id]);

            haveMatch = FALSE;
            for (i = id; i < client->calendar.entries.used; i++, cInfo2++) {
                if ((cInfo2->RecurrenceID == cInfo->RecurrenceID) && (cInfo2->UTCStart == utcStart)) {
                    ccode = ConnWriteF(client->conn, "2002-%lu found\r\n", i + 1);
                    if (ccode != -1) {
                        ccode = ConnWrite(client->conn, MSG1000OK, sizeof(MSG1000OK) - 1);
                    }

                    return(ccode);
                }
            }
        }

        return(ConnWrite(client->conn, MSG4262NOTFOUND, sizeof(MSG4262NOTFOUND) - 1));
    } else if (!(client->share.flags & NMAP_SHARE_CALENDAR_DENIED)) {
        if ((ccode = NMAPSendCommandF(client->share.calendar.conn, "CSFIND %lu %s %lu\r\n", sequence, uid, utcStart)) != -1) {
            ccode = NMAPReadAnswerLine(client->share.calendar.conn, client->line, CONN_BUFSIZE, FALSE);
            while (ccode == 2002) {
                ccode = ConnWrite(client->conn, client->line, strlen(client->line));
                if (ccode != -1) {
                    ccode = NMAPReadAnswerLine(client->share.calendar.conn, client->line, CONN_BUFSIZE, FALSE);
                }
            }

            if (ccode != -1) {
                ccode = ConnWrite(client->conn, client->line, strlen(client->line));
            }

            return(ccode);
        }

        return(ConnWrite(client->conn, MSG4224CANTREADCAL, sizeof(MSG4224CANTREADCAL) - 1));
    }

    return(ConnWrite(client->conn, MSG4240NOPERMISSION, sizeof(MSG4240NOPERMISSION) - 1));
}

int 
NmapCommandCsgflg(void *param)
{
    unsigned long id;
    unsigned char *ptr;
    struct stat sb;
    NMAPClient *client = (NMAPClient *)param;

    if (client->states & NMAP_CLIENT_USER) {
        if (client->calendar.name[0] != '\0') {
            CalendarOOBNotifications(client, &sb);

            ptr = client->buffer + 6;
        } else {
            return(ConnWrite(client->conn, MSG4225NOCALENDAR, sizeof(MSG4225NOCALENDAR) - 1));
        }
    } else {
        return(ConnWrite(client->conn, MSG3241NOUSER, sizeof(MSG3241NOUSER) - 1));
    }

    if ((*ptr++ == ' ') && (isdigit(*ptr))) {
        id = atol(ptr);
    } else {
        return(ConnWrite(client->conn, MSG3010BADARGC, sizeof(MSG3010BADARGC) - 1));
    }

    if (id && (id <= client->calendar.entries.used)) {
        id--;
    } else {
        return(ConnWrite(client->conn, MSG4220NOMSG, sizeof(MSG4220NOMSG) - 1));
    }

    return(ConnWriteF(client->conn, "1000 %lu %s\r\n", client->calendar.entries.info[id].State, MSG1000FLAGINFO));
}

int 
NmapCommandCslist(void *param)
{
    int ccode;
    unsigned long id;
    unsigned long count;
    unsigned char *ptr;
    struct stat sb;
    CalendarInfoStruct *cInfo;
    NMAPClient *client = (NMAPClient *)param;

    if (client->states & NMAP_CLIENT_USER) {
        if (client->calendar.name[0] != '\0') {
            CalendarOOBNotifications(client, &sb);

            ptr = client->buffer + 6;
        } else {
            return(ConnWrite(client->conn, MSG4225NOCALENDAR, sizeof(MSG4225NOCALENDAR) - 1));
        }
    } else {
        return(ConnWrite(client->conn, MSG3241NOUSER, sizeof(MSG3241NOUSER) - 1));
    }

    if ((*ptr++ == ' ') && (isdigit(*ptr))) {
	ptr = strchr(ptr, ' ');
	id = atol(client->buffer + 7);
    } else {
        return(ConnWrite(client->conn, MSG3010BADARGC, sizeof(MSG3010BADARGC) - 1));
    }

    if (id && (id <= client->calendar.entries.used)) {
        id--;
    } else {
        return(ConnWrite(client->conn, MSG4220NOMSG, sizeof(MSG4220NOMSG) - 1));
    }

    if (!(client->share.flags & NMAP_SHARE_CALENDAR)) {
        if (WaitforAndLockCalendar(client->store, client->user, client->calendar.name) == 1) {
            cInfo = &(client->calendar.entries.info[id]);
        } else {
            return(ConnWrite(client->conn, MSG4120CALLOCKED, sizeof(MSG4120CALLOCKED) - 1));
        }

        sprintf(client->path, "%s/%s/%s.cal", client->store, client->user, client->calendar.name);
        client->calendar.fh = fopen(client->path,"rb");
        if (client->calendar.fh != NULL) {
            
	    if (!ptr) {
		/* send the complete entry */
		fseek(client->calendar.fh, cInfo->Pos, SEEK_SET);
		if (((ccode = ConnWriteF(client->conn, "2023 %lu\r\n", cInfo->Size)) != -1) 
                    && ((ccode = ConnWriteFromFile(client->conn, client->calendar.fh, cInfo->Size)) != -1)) {
		    ccode = ConnWrite(client->conn, MSG1000OK, sizeof(MSG1000OK) - 1);
		}
	    } else {
		ptr++;
		if (toupper(*ptr) == 'I') {
		    /* send just the ical object */
		    fseek(client->calendar.fh, cInfo->Pos, SEEK_SET);
		    if (((ccode = ConnWriteF(client->conn, "2023 %lu\r\n", cInfo->CalSize)) != -1) 
			&& ((ccode = ConnWriteFromFile(client->conn, client->calendar.fh, cInfo->CalSize)) != -1)) {
			ccode = ConnWrite(client->conn, MSG1000OK, sizeof(MSG1000OK) - 1);
		    }
		} else if (toupper(*ptr) == 'M') {
		    /* send just the meta data */
		    fseek(client->calendar.fh, cInfo->Pos + cInfo->CalSize, SEEK_SET);
		    if (((ccode = ConnWriteF(client->conn, "2023 %lu\r\n", cInfo->Size - cInfo->CalSize)) != -1) 
			&& ((ccode = ConnWriteFromFile(client->conn, client->calendar.fh, cInfo->Size - cInfo->CalSize)) != -1)) {
			ccode = ConnWrite(client->conn, MSG1000OK, sizeof(MSG1000OK) - 1);
		    }
		} else {
		    return(ConnWrite(client->conn, MSG3010BADARGC, sizeof(MSG3010BADARGC) - 1));
		}
	    }

            fclose(client->calendar.fh);
            client->calendar.fh = NULL;

            UnlockCalendar(client->store, client->user, client->calendar.name);

            return(ccode);
        }

        ccode = ConnWrite(client->conn, MSG4224CANTREADCAL, sizeof(MSG4224CANTREADCAL) - 1);

        UnlockCalendar(client->store, client->user, client->calendar.name);

        return(ccode);
    }

    if (!(client->share.flags & NMAP_SHARE_CALENDAR_DENIED)) {
        if (((ccode = NMAPSendCommandF(client->share.calendar.conn, "CSLIST %lu%s\r\n", id + 1, (ptr == NULL) ? "" : ptr)) != -1) 
                && ((ccode = NMAPReadAnswerLine(client->share.calendar.conn, client->line, CONN_BUFSIZE, FALSE)) != -1) 
                && (ccode == 2023)) {
            count = atol(client->line + 5);

            if (((ccode = ConnReadToConn(client->share.calendar.conn, client->conn, count)) != -1) 
                    && ((ccode = NMAPReadAnswerLine(client->share.calendar.conn, client->line, CONN_BUFSIZE, FALSE)) != -1)) {
                ccode = ConnWrite(client->conn, client->line, strlen(client->line));
            }
        } else if (ccode != -1) {
            ccode = ConnWrite(client->conn, client->line, strlen(client->line));
        } else {
            ccode = ConnWrite(client->conn, MSG4224CANTREADCAL, sizeof(MSG4224CANTREADCAL) - 1);
        }
    } else {
        ccode = ConnWrite(client->conn, MSG4240NOPERMISSION, sizeof(MSG4240NOPERMISSION) - 1);
    }

    return(ccode);
}

int 
NmapCommandCscrea(void *param)
{
    int ccode;
    unsigned char *ptr;
    unsigned char *ptr2;
    unsigned char *path;
    unsigned char *name;
    unsigned char tmpName[XPL_MAX_PATH + 1];
    struct stat sb;
    FILE *handle;
    NMAPClient *client = (NMAPClient *)param;
    MDBValueStruct *vs;

    if (client->states & NMAP_CLIENT_USER) {
        if (client->calendar.name[0] != '\0') {
            CalendarOOBNotifications(client, &sb);
        }

        ptr = client->buffer + 6;
    } else {
        return(ConnWrite(client->conn, MSG3241NOUSER, sizeof(MSG3241NOUSER) - 1));
    } 

    if ((*ptr == ' ') && !(ptr2 = strchr(++ptr, ' ')) 
            && ((strchr(ptr, '.') == NULL) || ((strstr(ptr, "../") == NULL) && (strstr(ptr, "..\\") == NULL)))) {
        path = ptr;
    } else {
        return(ConnWrite(client->conn, MSG3010BADARGC, sizeof(MSG3010BADARGC) - 1));
    }

    if ((strlen(client->store) + strlen(client->user) + strlen(path) + 4) < XPL_MAX_PATH) {
        /* Create directories, if required
           Name is blocked out automatically since the while loop only runs while there's a slash
           so the last element doesn't get created. And it shouldn't, since it's the name */
        name = strrchr(path, '/');
        if (name != NULL) {
            ptr = strchr(path, '/');

            while (ptr) {
                *ptr = '\0';

                sprintf(client->path, "%s/%s/%s", client->store, client->user, path);
                if (stat(client->path, &sb) != 0) {
                    XplMakeDir(client->path);
                    SetLongName(client->path, tmpName);
                }

                *ptr = '/';
                ptr2 = ptr;
                ptr = strchr(ptr2 + 1, '/');
            }

            *name = '/';
        }
    } else {
        return(ConnWrite(client->conn, MSG4230PATHTOOLONG, sizeof(MSG4230PATHTOOLONG) - 1));
    }

    sprintf(client->path, "%s/%s/%s.cal", client->store, client->user, path);
    handle = fopen(client->path, "r");
    if ((handle == NULL) 
            && (IsAvailableShare(client->user, path, NMAP_SHARE_CALENDAR, NULL, NULL) == FALSE)) {

        handle = fopen(client->path, "wb");
        if (handle != NULL) {
            fclose(handle);

            SetLongName(client->path, tmpName);

            sprintf(client->path, "%s/%s/%s.sdx", client->store, client->user, path);
            unlink(client->path);

            sprintf(client->path, "%s/%s/%s.sdc", client->store, client->user, path);
            unlink(client->path);

            sprintf(client->path, "%s/%s/%s.idc", client->store, client->user, path);
            handle = fopen(client->path, "wb");

            SetLongName(client->path, tmpName);

            fprintf(handle, "%s\r\n0000000000\r\n0000000000\r\n0000000000\r\n%010lu\r\n%010lu\r\n", NMAP_CAL_IDX_VERSION, NMAP.universalCounter++, NMAP.universalCounter++);

            sprintf(client->line, "%lu", NMAP.universalCounter);
            vs = MDBCreateValueStruct(NMAP.handle.directory, NMAP.server.dn);
            MDBAddValue(client->line, vs);
            MDBWrite(MSGSRV_AGENT_NMAP, MSGSRV_A_UID, vs);
            MDBDestroyValueStruct(vs);
            fclose(handle);

            ccode = ConnWrite(client->conn, MSG1000CALENDARMADE, sizeof(MSG1000CALENDARMADE) - 1);
        } else {
            ccode = ConnWriteF(client->conn, MSG3014REQNOTALLOWED, " Illegal calendar name.");
        }
    } else {
        if (handle != NULL) {
            fclose(handle);
        }

        ccode = ConnWrite(client->conn, MSG4226CALEXISTS, sizeof(MSG4226CALEXISTS) - 1);
    }

    return(ccode);
}

int 
NmapCommandCscheck(void *param)
{
    int ccode;
    unsigned long i;
    unsigned long totalCount = 0;
    unsigned long totalSize = 0;
    unsigned long eventCount = 0;
    unsigned long eventSize = 0;
    unsigned long todoCount = 0;
    unsigned long todoSize = 0;
    unsigned long journalCount = 0;
    unsigned long journalSize = 0;
    unsigned long deletedCount = 0;
    unsigned long deletedSize = 0;
    unsigned long purgedCount = 0;
    unsigned long purgedSize = 0;
    unsigned char *ptr;
    unsigned char *resourceOwner;
    unsigned char *resourceName;
    BOOL result;
    CalendarInfoStruct *cInfo;
    struct stat sb;
    NMAPClient *client = (NMAPClient *)param;
    NMAPClient *sclient;

    if (client->states & NMAP_CLIENT_USER) {
        if (client->calendar.name[0] != '\0') {
            CalendarOOBNotifications(client, &sb);
        }

        ptr = client->buffer + 7;
    } else {
        return(ConnWrite(client->conn, MSG3241NOUSER, sizeof(MSG3241NOUSER) - 1));
    }

    if ((*ptr++ == ' ') && (*ptr != ' ')) {
        sclient = NMAPClientAlloc();
    } else {
        return(ConnWrite(client->conn, MSG3010BADARGC, sizeof(MSG3010BADARGC) - 1));
    }

    if (sclient != NULL) {
        strcpy(sclient->user, client->user);
        strcpy(sclient->calendar.name, ptr);
        sclient->store = client->store;
        sclient->mailbox.newline = TRUE;

        sclient->userHash = client->userHash; 
    } else {
        return(ConnWrite(client->conn, MSG5001NOMEMORY, sizeof(MSG5001NOMEMORY) - 1));
    }

    sprintf(sclient->path, "%s/%s/%s.cal", sclient->store, sclient->user, sclient->calendar.name);
    if (access(sclient->path, 0) == 0) {
        if (ReadNLockAquire(&sclient->calendar.lock, &sclient->userHash, sclient->user, sclient->calendar.name, 0) == NLOCK_AQUIRED) {
            ParseCalendar(sclient);
            ReadNLockRelease(sclient->calendar.lock);
            sclient->calendar.lock = NULL;
        } else {
            NMAPClientFree(sclient);

            return(ConnWrite(client->conn, MSG4224CANTREADCAL, sizeof(MSG4224CANTREADCAL) - 1));
        }
    } else if (XplStrCaseCmp(sclient->calendar.name, "MAIN") == 0) {
        memcpy(sclient->calendar.name, "MAIN", 4);

        sprintf(sclient->path, "%s/%s/MAIN.cal", sclient->store, sclient->user);
        if (access(sclient->path, 0) == 0) {
            if (ReadNLockAquire(&sclient->calendar.lock, &sclient->userHash, sclient->user, sclient->calendar.name, 0) == NLOCK_AQUIRED) {
                ParseCalendar(sclient);
                ReadNLockRelease(sclient->calendar.lock);
                sclient->calendar.lock = NULL;
            }
        } else {
            NMAPClientFree(sclient);

            return(ConnWrite(client->conn, MSG4224NOCSTORE, sizeof(MSG4224NOCSTORE) - 1));
        }
    } else {
        if (IsAvailableShare(sclient->user, sclient->calendar.name, NMAP_SHARE_CALENDAR, &resourceOwner, &resourceName)) {
            if (((ccode = ConnectToNMAPShare(sclient, resourceOwner, resourceName, NMAP_SHARE_CALENDAR, &(sclient->share.calendar.permissions))) != -1) 
                    && (ccode == 1000) 
                    && (sclient->share.calendar.permissions & NMAP_SHARE_READ)){
                if (ReadNLockAquire(&sclient->calendar.lock, &sclient->userHash, sclient->user, sclient->calendar.name, 0) == NLOCK_AQUIRED) {
                    result = SelectNMAPShare(sclient, resourceOwner, resourceName, NMAP_SHARE_CALENDAR);
                    if (!result) {
                        ccode = ConnWrite(client->conn, MSG4224CANTREADCAL, sizeof(MSG4224CANTREADCAL) - 1);
                    }

                    ReadNLockRelease(sclient->calendar.lock);
                    sclient->calendar.lock = NULL;
                } else {
                    result = FALSE;
                    ccode = ConnWrite(client->conn, MSG4224CANTREADCAL, sizeof(MSG4224CANTREADCAL) - 1);
                }
            } else if (ccode == 1000) {
                result = FALSE;
                ccode = ConnWrite(client->conn, MSG4240NOPERMISSION, sizeof(MSG4240NOPERMISSION) - 1);
            } else {
                result = FALSE;
                ConnWrite(client->conn, MSG5004INTERNALERR, sizeof(MSG5004INTERNALERR) - 1);
            }


            MemFree(resourceOwner);
            MemFree(resourceName);

            NMAPQuit(sclient->share.calendar.conn);
            ConnFree(sclient->share.calendar.conn);
            sclient->share.calendar.conn = NULL;

            if (!result) {
                sclient->share.calendar.permissions = 0;
                sclient->share.flags &= ~(NMAP_SHARE_CALENDAR | NMAP_SHARE_CALENDAR_DENIED);

                if (sclient->calendar.entries.info != NULL) {
                    MemFree(sclient->calendar.entries.info);
                }

                NMAPClientFree(sclient);

                return(ccode);
            }
        } else {
            if (sclient->calendar.entries.info != NULL) {
                MemFree(sclient->calendar.entries.info);
            }

            NMAPClientFree(sclient);

            return(ConnWrite(client->conn, MSG4224NOCSTORE, sizeof(MSG4224NOCSTORE) - 1));
        }
    }

    for (i = 0, cInfo = sclient->calendar.entries.info; i < sclient->calendar.entries.used; i++, cInfo++) {
        totalSize += cInfo->Size;
        totalCount++;

        if (!(cInfo->State & (MSG_STATE_PURGED | MSG_STATE_DELETED))) {
            switch (cInfo->Type) {
                case NMAP_CAL_EVENT: {
                    eventSize += cInfo->Size;
                    eventCount++;
                    continue;
                }

                case NMAP_CAL_TODO: {
                    todoSize += cInfo->Size;
                    todoCount++;
                    continue;
                }

                case NMAP_CAL_JOURNAL: {
                    journalSize += cInfo->Size;
                    journalSize++;
                    continue;
                }
            }
        } else if (cInfo->State & MSG_STATE_DELETED) {
            deletedSize += cInfo->Size;
            deletedCount++;
            continue;
        } else {
            purgedSize += cInfo->Size;
            purgedCount++;
        }
    }

    if (sclient->calendar.entries.used > 0) {
        ccode = ConnWriteF(client->conn, "1000 %lu %lu %lu %lu %lu %lu %lu %lu %lu %lu %lu %lu %lu %lu\r\n",
            totalCount, totalSize, eventCount, eventSize, todoCount, todoSize, 
            journalCount, journalSize, deletedCount, deletedSize, purgedCount, purgedSize,
            sclient->calendar.entries.info[sclient->calendar.entries.used - 1].UID + 1, sclient->calendar.id);
    } else {
        ccode = ConnWriteF(client->conn, "1000 0 0 0 0 0 0 0 0 0 0 0 0 %lu %lu\r\n", sclient->calendar.nextUID, sclient->calendar.id);
    }

    if (sclient->calendar.entries.info) {
        MemFree(sclient->calendar.entries.info);
        sclient->calendar.entries.info = NULL;
    }

    NMAPClientFree(sclient);

    return(ccode);
}

int 
NmapCommandCscomp(void *param)
{
    int ccode;
    unsigned long id;
    unsigned long utcTime;
    unsigned char *ptr;
    unsigned char *ptr2;
    CalendarInfoStruct *cInfo;
    struct stat sb;
    FILE *index;
    NMAPClient *client = (NMAPClient *)param;

    if (client->states & NMAP_CLIENT_USER) {
        if (client->calendar.name[0] != '\0') {
            CalendarOOBNotifications(client, &sb);

            ptr = client->buffer + 6;
        } else {
            return(ConnWrite(client->conn, MSG4225NOCALENDAR, sizeof(MSG4225NOCALENDAR) - 1));
        }
    } else {
        return(ConnWrite(client->conn, MSG3241NOUSER, sizeof(MSG3241NOUSER) - 1));
    }

    if ((*ptr++ == ' ') && (*ptr != ' ') && ((ptr2 = strchr(ptr, ' ')) != NULL)) {
        *(ptr2++) = '\0';
    } else {
        return(ConnWrite(client->conn, MSG3010BADARGC, sizeof(MSG3010BADARGC) - 1));
    }

    id = atol(ptr);
    if (id && (id <= client->calendar.entries.used)) {
        id--;
    } else {
        return(ConnWrite(client->conn, MSG4220NOMSG, sizeof(MSG4220NOMSG) - 1));
    }

    utcTime = atol(ptr2);

    if (WaitforAndLockCalendar(client->store, client->user, client->calendar.name) == 1) {
        cInfo = &(client->calendar.entries.info[id]);
    } else {
        return(ConnWrite(client->conn, MSG4120CALLOCKED, sizeof(MSG4120CALLOCKED) - 1));
    }

    if (!(client->share.flags & NMAP_SHARE_CALENDAR)) {
        sprintf(client->path, "%s/%s/%s.cal", client->store, client->user, client->calendar.name);
        client->calendar.fh = fopen(client->path, "r+b");
        if (client->calendar.fh != NULL) {
            fseek(client->calendar.fh, cInfo->Pos + cInfo->CalSize, SEEK_SET);
        } else {
            UnlockCalendar(client->store, client->user, client->calendar.name);

            return(ConnWrite(client->conn, MSG1000OK, sizeof(MSG1000OK) - 1));
        }

        if (fgets(client->line, CONN_BUFSIZE, client->calendar.fh) != NULL) {
            if (QuickNCmp("NIMS-UTCEND:", client->line, 12)) {
                fseek(client->calendar.fh, cInfo->Pos + cInfo->CalSize + 12, SEEK_SET);
                fprintf(client->calendar.fh, "%010lu", utcTime);
            }
        }

        fclose(client->calendar.fh);
        client->calendar.fh = NULL;

        sprintf(client->path, "%s/%s/%s.idc",client->store, client->user,client->calendar.name);
    } else if (!(client->share.flags & NMAP_SHARE_CALENDAR_DENIED)) {
        if (((ccode = NMAPSendCommandF(client->share.calendar.conn, "CSCOMP %lu %lu\r\n", id + 1, utcTime)) != -1) 
                && ((ccode = NMAPReadAnswer(client->share.calendar.conn, client->line, CONN_BUFSIZE, FALSE)) != -1) 
                && (ccode == 1000)) {
            sprintf(client->path, "%s/%s/%s.sdc",client->store, client->user,client->calendar.name);
        } else if (ccode != -1) {
            UnlockCalendar(client->store, client->user, client->calendar.name);
            return(ConnWrite(client->conn, client->line, strlen(client->line)));
        } else {
            UnlockCalendar(client->store, client->user, client->calendar.name);
            return(ConnWrite(client->conn, MSG4224CANTREADCAL, sizeof(MSG4224CANTREADCAL) - 1));
        }
    } else {
        UnlockCalendar(client->store, client->user, client->calendar.name);

        return(ConnWrite(client->conn, MSG4240NOPERMISSION, sizeof(MSG4240NOPERMISSION) - 1));
    }

    cInfo->UTCEnd = utcTime;

    index = fopen(client->path, "r+b");
    fseek(index, NMAP_CAL_IDX_HEADERSIZE + (id * sizeof(CalendarInfoStruct)) + offsetof(CalendarInfoStruct, UTCEnd), SEEK_SET);
    fwrite(&utcTime, sizeof(unsigned long), 1, index);
    fclose(index);

    stat(client->path, &sb);
    client->calendar.indexTime = sb.st_mtime;

    UnlockCalendar(client->store, client->user, client->calendar.name);

    return(ConnWrite(client->conn, MSG1000OK, sizeof(MSG1000OK) - 1));
}

__inline static int 
CopyCalendarFromLocalToLocal(NMAPClient *client, unsigned long id, unsigned char *target)
{
    unsigned long len;
    unsigned long count;
    CalendarInfoStruct *cInfo;
    FILE *targetCalendar;

    WaitforAndLockCalendar(client->store, client->user, target);

    targetCalendar = fopen(client->path, "ab");
    if (targetCalendar != NULL) {
        sprintf(client->path, "%s/%s/%s.cal", client->store, client->user, client->calendar.name);
        client->calendar.fh = fopen(client->path, "rb");
        if (client->calendar.fh == NULL) {
            fclose(targetCalendar);

            UnlockCalendar(client->store, client->user, target);
            return(ConnWrite(client->conn, MSG4224CANTREADCAL, sizeof(MSG4224CANTREADCAL) - 1));
        }
    } else {
        UnlockCalendar(client->store, client->user, target);
        return(ConnWrite(client->conn, MSG4228CANTWRITECAL, sizeof(MSG4228CANTWRITECAL) - 1));
    }

    cInfo = &(client->calendar.entries.info[id]);
    count = cInfo->Size;
    fseek(client->calendar.fh, cInfo->Pos, SEEK_SET);

    while (count > 0) {
        if (count < CONN_BUFSIZE) {
            len = fread(client->line, sizeof(unsigned char), count, client->calendar.fh);
        } else {
            len = fread(client->line, sizeof(unsigned char), CONN_BUFSIZE, client->calendar.fh);
        }

        if (len > 0) {
            fwrite(client->line, sizeof(unsigned char), len, targetCalendar);
            count -= len;
        } else {
            /* fixme - correct error handling */
            break;
        }
    }

    fclose(client->calendar.fh);
    fclose(targetCalendar);
    client->calendar.fh = NULL;

    UnlockCalendar(client->store, client->user, target);

    return(ConnWrite(client->conn, MSG1000OK, sizeof(MSG1000OK) - 1));
}

__inline static int 
CopyCalendarFromLocalToShare(NMAPClient *client, unsigned long id, unsigned char *target)
{
    int ccode;
    unsigned long len;
    unsigned long count;
    unsigned char *resourceOwner;
    unsigned char *resourceName;
    CalendarInfoStruct *cInfo;
    NMAPClient *sclient = NULL;

    if (IsAvailableShare(client->user, target, NMAP_SHARE_CALENDAR, &resourceOwner, &resourceName)) {
        do {
            sprintf(client->path, "%s/%s/%s.cal", client->store, client->user, client->calendar.name);
            client->calendar.fh = fopen(client->path, "rb");
            if (client->calendar.fh) {
                sclient = NMAPClientAlloc();
            } else {
                ccode = ConnWrite(client->conn, MSG4224CANTREADCAL, sizeof(MSG4224CANTREADCAL) - 1);
                break;
            }

            if (sclient != NULL) {
                strcpy(sclient->user, client->user);
                sclient->userHash = client->userHash; 
                sclient->mailbox.newline = TRUE;
            } else {
                ccode = ConnWrite(client->conn, MSG5001NOMEMORY, sizeof(MSG5001NOMEMORY) - 1);
                break;
            }

            if (((ccode = ConnectToNMAPShare(sclient, resourceOwner, resourceName, NMAP_SHARE_CALENDAR, &(sclient->share.calendar.permissions))) != -1) 
                    && ((ccode = ConnectToNMAPShare(sclient, resourceOwner, resourceName, NMAP_SHARE_CALENDAR, &(sclient->share.calendar.permissions))) != -1) 
                    && (ccode == 1000) 
                    && (sclient->share.calendar.permissions & NMAP_SHARE_INSERT)) {
                ccode = ConnWriteF(sclient->share.calendar.conn, "USER %s\r\n", resourceOwner);
            } else if (ccode != 1000) {
                ccode = ConnWrite(client->conn, MSG4224CANTREADMBOX, sizeof(MSG4224CANTREADMBOX) - 1);
                break;
            } else {
                ccode = ConnWrite(client->conn, MSG4240NOPERMISSION, sizeof(MSG4240NOPERMISSION) - 1);
                break;
            }

            if ((ccode != -1) 
                    && ((ccode = NMAPReadAnswer(sclient->share.calendar.conn, sclient->line, CONN_BUFSIZE, FALSE)) != -1) 
                    && (ccode == 1000)) {
                cInfo = &(client->calendar.entries.info[id]);
                count = cInfo->Size;
                fseek(client->calendar.fh, cInfo->Pos, SEEK_SET);
            } else {
                ccode = ConnWrite(client->conn, MSG4224CANTREADMBOX, sizeof(MSG4224CANTREADMBOX) - 1);
                break;
            }

            /* fixme - concerned about reliable handling of transparent data. */
            if (((ccode = NMAPSendCommandF(sclient->share.calendar.conn, "CSSTRAW %s %lu FROM %s %s\r\n", resourceName, count, client->user, client->user)) != -1) 
                    && ((ccode = NMAPReadAnswer(sclient->share.calendar.conn, sclient->buffer, CONN_BUFSIZE, FALSE)) != -1) 
                    && (ccode == 2002)) {
                while ((count > 0) && (ccode != -1)) {
                    if (count < CONN_BUFSIZE) {
                        len = fread(client->line, sizeof(unsigned char), count, client->calendar.fh);
                    } else {
                        len = fread(client->line, sizeof(unsigned char), CONN_BUFSIZE, client->calendar.fh);
                    }

                    if (len > 0) {
                        if (client->line[0] != '.') {
                            ccode = ConnWrite(sclient->share.calendar.conn, client->line, len);
                        } else {
                            ccode = ConnWrite(sclient->share.calendar.conn, ".", 1);
                            if (ccode != -1) {
                                ccode = ConnWrite(sclient->share.calendar.conn, client->line, len);
                            }
                        }

                        count -= len;
                    } else {
                        /* fixme - correct error handling */
                        break;
                    }
                }
            } else {
                ccode = ConnWrite(client->conn, MSG4228CANTWRITECAL, sizeof(MSG4228CANTWRITECAL) - 1);
                break;
            }

            NMAPReadAnswerLine(sclient->share.calendar.conn, client->line, CONN_BUFSIZE, FALSE);
            ccode = ConnWrite(client->conn, client->line, strlen(client->line));
            break;
        } while (TRUE);

        MemFree(resourceOwner);
        MemFree(resourceName);

        if (client->calendar.fh) {
            fclose(client->calendar.fh);
            client->calendar.fh = NULL;
        }

        if (sclient) {
            if (sclient->calendar.entries.info != NULL) {
                MemFree(sclient->calendar.entries.info);
                sclient->calendar.entries.info = NULL;
            }

            NMAPQuit(sclient->share.calendar.conn);
            ConnFree(sclient->share.calendar.conn);

            NMAPClientFree(sclient);
        }
    } else {
        ccode = ConnWrite(client->conn, MSG4224NOMBOX, sizeof(MSG4224NOMBOX) - 1);
    }

    return(ccode);
}

__inline static int 
CopyCalendarFromShareToLocal(NMAPClient *client, unsigned long id, unsigned char *target)
{
    int ccode;
    unsigned long count;
    FILE *targetCalendar;

    WaitforAndLockCalendar(client->store, client->user, target);

    targetCalendar = fopen(client->path, "ab");
    if (targetCalendar != NULL) {
        if (((ccode = NMAPSendCommandF(client->share.calendar.conn, "CSLIST %lu\r\n", id + 1)) != -1) 
                && ((ccode = NMAPReadAnswer(client->share.calendar.conn, client->line, CONN_BUFSIZE, FALSE)) != -1) 
                && (ccode = 2023)) {
            count = atol(client->line + 5);
        } else {
            fclose(targetCalendar);
            UnlockCalendar(client->store, client->user, target);

            return(ConnWrite(client->conn, MSG4224CANTREADCAL, sizeof(MSG4224CANTREADCAL) - 1));
        }
    } else {
        UnlockCalendar(client->store, client->user, target);

        return(ConnWrite(client->conn, MSG4228CANTWRITECAL, sizeof(MSG4228CANTWRITECAL) - 1));
    }

    /* fixme - this is incorrect behaviour for transparent data */
    while ((ccode != -1) && (count > 0)) {
        ccode = ConnReadLine(client->share.calendar.conn, client->line, CONN_BUFSIZE);
        if (ccode != -1) {
            if (client->line[0] != '.') {
                fwrite(client->line, sizeof(unsigned char), ccode, targetCalendar);
            } else {
                fwrite(".", sizeof(unsigned char), 1, targetCalendar);
                fwrite(client->line, sizeof(unsigned char), ccode, targetCalendar);
            }

            count -= ccode;
        }
    }

    fclose(targetCalendar);

    UnlockCalendar(client->store, client->user, target);

    ccode = NMAPReadAnswerLine(client->share.calendar.conn, client->line, CONN_BUFSIZE, FALSE);
    if (ccode == 1000) {
        ccode = ConnWrite(client->conn, MSG1000OK, sizeof(MSG1000OK) - 1);
    } else {
        ConnWrite(client->conn, client->line, strlen(client->line));
    }

    return(ccode);
}

__inline static int 
CopyCalendarFromShareToShare(NMAPClient *client, unsigned long id, unsigned char *target)
{
    /* fixme - was broken; still broken. */
    return(-1);
}

int 
NmapCommandCscopy(void *param)
{
    int ccode;
    unsigned long id;
    unsigned char *ptr;
    unsigned char *ptr2;
    unsigned char *target;
    struct stat sb;
    NMAPClient *client = (NMAPClient *)param;

    if (client->states & NMAP_CLIENT_USER) {
        if (client->calendar.name[0] != '\0') {
            CalendarOOBNotifications(client, &sb);

            ptr = client->buffer + 6;
        } else {
            return(ConnWrite(client->conn, MSG4225NOCALENDAR, sizeof(MSG4225NOCALENDAR) - 1));
        }
    } else {
        return(ConnWrite(client->conn, MSG3241NOUSER, sizeof(MSG3241NOUSER) - 1));
    }

    if (NMAP.flags & NMAP_FLAG_DISK_SPACE_LOW) {
        ConnWrite(client->conn, MSG5221SPACELOW, sizeof(MSG5221SPACELOW) - 1);
        return(TRUE);
    }

    if ((*ptr++ == ' ') && (*ptr != ' ') && ((ptr2 = strchr(ptr, ' ')) != NULL)) {
        *ptr2++ = '\0';
    } else {
        return(ConnWrite(client->conn, MSG3010BADARGC, sizeof(MSG3010BADARGC) - 1));
    }

    target = ptr2;
    if (XplStrCaseCmp(target, client->calendar.name) != 0) {
        id = atoi(ptr);
    } else {
        return(ConnWrite(client->conn, MSG4227TGTMBOXSELECTED, sizeof(MSG4227TGTMBOXSELECTED) - 1));
    }

    if (id && (id <= client->calendar.entries.used)) {
        id--;
    } else {
        return(ConnWrite(client->conn, MSG4220NOMSG, sizeof(MSG4220NOMSG) - 1));
    }

    if (!(client->share.flags & NMAP_SHARE_CALENDAR)) {
        sprintf(client->path,"%s/%s/%s.cal", client->store, client->user, target);
        if (access(client->path, 0) == 0) {
            ccode = CopyCalendarFromLocalToLocal(client, id, target);
        } else {
            ccode = CopyCalendarFromLocalToShare(client, id, target);
        }
    } else if (!(client->share.flags & NMAP_SHARE_CALENDAR_DENIED)) {
        sprintf(client->path,"%s/%s/%s.cal", client->store, client->user, target);
        if (access(client->path, 0) == 0) {
            ccode = CopyCalendarFromShareToLocal(client, id, target);
        } else {
            ccode = CopyCalendarFromShareToShare(client, id, target);
        }
    } else {
        ccode = ConnWrite(client->conn, MSG4240NOPERMISSION, sizeof(MSG4240NOPERMISSION) - 1);
    }

    return(ccode);
}

int 
NmapCommandCsdele(void *param)
{
    unsigned long i;
    unsigned long id;
    unsigned long occurrences = 0;
    unsigned long recurrenceID;
    unsigned char *ptr;
    unsigned char *ptr2;
    CalendarInfoStruct *cInfo;
    struct stat sb;
    FILE *index;
    NMAPClient *client = (NMAPClient *)param;

    if (client->states & NMAP_CLIENT_USER) {
        if (client->calendar.name[0] != '\0') {
            CalendarOOBNotifications(client, &sb);

            ptr = client->buffer + 6;
        } else {
            return(ConnWrite(client->conn, MSG4225NOCALENDAR, sizeof(MSG4225NOCALENDAR) - 1));
        }
    } else {
        return(ConnWrite(client->conn, MSG3241NOUSER, sizeof(MSG3241NOUSER) - 1));
    }

    /* CSDELE <id>[ RECURRING] */
    if ((*ptr++ = ' ') && (isdigit(*ptr))) {
        ptr2 = strchr(ptr, ' ');
        if (!ptr2) {
            id = atol(ptr);
            if (id && (id <= client->calendar.entries.used)) {
                id--;
                cInfo = &(client->calendar.entries.info[id]);
            } else {
                return(ConnWrite(client->conn, MSG4220NOMSG, sizeof(MSG4220NOMSG) - 1));
            }
        } else {
            *ptr2++ = '\0';
            id = atol(ptr);
            if (id && (id <= client->calendar.entries.used)) {
                id--;
            } else {
                return(ConnWrite(client->conn, MSG4220NOMSG, sizeof(MSG4220NOMSG) - 1));
            }

            if (toupper(*ptr2) == 'R') {
                if (client->calendar.entries.info[id].Recurring) {
                    recurrenceID = client->calendar.entries.info[id].RecurrenceID;
                    cInfo = client->calendar.entries.info;
                } else {
                    cInfo = &(client->calendar.entries.info[id]);
                    ptr2 = NULL;
                }
            } else {
                return(ConnWrite(client->conn, MSG3010BADARGC, sizeof(MSG3010BADARGC) - 1));
            }
        }
    } else {
        return(ConnWrite(client->conn, MSG3010BADARGC, sizeof(MSG3010BADARGC) - 1));
    }

    if (!(client->share.flags & NMAP_SHARE_CALENDAR)) {
        sprintf(client->path, "%s/%s/%s.idc", client->store, client->user, client->calendar.name);
    } else if (!(client->share.flags & NMAP_SHARE_CALENDAR_DENIED) && (client->share.calendar.permissions & NMAP_SHARE_DELETE)) {
        sprintf(client->path, "%s/%s/%s.sdc", client->store, client->user, client->calendar.name);
    } else {
        return(ConnWrite(client->conn, MSG4240NOPERMISSION, sizeof(MSG4240NOPERMISSION) - 1));
    }

    index = NULL;
    if (ptr2 == NULL) {
        if (WaitforAndLockCalendar(client->store, client->user, client->calendar.name) == 1) {
            index = fopen(client->path,"r+b");
        } else {
            return(ConnWrite(client->conn, MSG4120CALLOCKED, sizeof(MSG4120CALLOCKED) - 1));
        }

        if (index != NULL) {
            fseek(index, NMAP_CAL_IDX_HEADERSIZE + ((unsigned char *)cInfo - (unsigned char *)client->calendar.entries.info) + offsetof(CalendarInfoStruct, State), SEEK_SET);

            cInfo->State |= MSG_STATE_DELETED;
            client->calendar.entries.count--;

            if (!(cInfo->State & MSG_STATE_RECENT)) {
                fwrite(&(cInfo->State), sizeof(unsigned long), 1, index);
            } else {
                cInfo->State &= ~MSG_STATE_RECENT;
                fwrite(&(cInfo->State), sizeof(unsigned long), 1, index);
                cInfo->State |= MSG_STATE_RECENT;
            }

            fclose(index);

            stat(client->path, &sb);
            client->calendar.indexTime = sb.st_mtime;

            UnlockCalendar(client->store, client->user, client->calendar.name);

            return(ConnWriteF(client->conn, "1000 %s\r\n", MSG1000DELETED));
        }

        UnlockCalendar(client->store, client->user, client->calendar.name);

        return(ConnWrite(client->conn, MSG4228CANTWRITECAL, sizeof(MSG4228CANTWRITECAL) - 1));
    }

    for (i = client->calendar.entries.used; i != 0; i--, cInfo++) {
        if (cInfo->RecurrenceID != recurrenceID) {
            continue;
        }

        if (index != NULL) {
            fseek(index, NMAP_CAL_IDX_HEADERSIZE + ((unsigned char *)cInfo - (unsigned char *)client->calendar.entries.info) + offsetof(CalendarInfoStruct, State), SEEK_SET);
        } else {
            if (WaitforAndLockCalendar(client->store, client->user, client->calendar.name) == 1) {
                index = fopen(client->path, "r+b");
            } else {
                return(ConnWrite(client->conn, MSG4120CALLOCKED, sizeof(MSG4120CALLOCKED) - 1));
            }

            if (index != NULL) {
                fseek(index, NMAP_CAL_IDX_HEADERSIZE + ((unsigned char *)cInfo - (unsigned char *)client->calendar.entries.info) + offsetof(CalendarInfoStruct, State), SEEK_SET);
            } else {
                UnlockCalendar(client->store, client->user, client->calendar.name);

                return(ConnWrite(client->conn, MSG4228CANTWRITECAL, sizeof(MSG4228CANTWRITECAL) - 1));
            }
        }

        cInfo->State |= MSG_STATE_DELETED;
        client->calendar.entries.count--;
        occurrences++;

        if (!(cInfo->State & MSG_STATE_RECENT)) {
            fwrite(&(cInfo->State), sizeof(unsigned long), 1, index);
        } else {
            cInfo->State &= ~MSG_STATE_RECENT;
            fwrite(&(cInfo->State), sizeof(unsigned long), 1, index);
            cInfo->State |= MSG_STATE_RECENT;
        }
    }

    if (index != NULL) {
        fclose(index);

        stat(client->path, &sb);
        client->calendar.indexTime = sb.st_mtime;

        UnlockCalendar(client->store, client->user, client->calendar.name);
    }

    return(ConnWriteF(client->conn, "1000 %lu Occurrences deleted\r\n", occurrences));
}

int 
NmapCommandCsdflg(void *param)
{
    int ccode;
    unsigned long id;
    unsigned long newFlags;
    unsigned long flags;
    unsigned char *ptr;
    unsigned char *ptr2;
    struct stat sb;
    FILE *index;
    NMAPClient *client = (NMAPClient *)param;

    if (client->states & NMAP_CLIENT_USER) {
        if (client->calendar.name[0] != '\0') {
            CalendarOOBNotifications(client, &sb);

            ptr = client->buffer + 6;
        } else {
            return(ConnWrite(client->conn, MSG4225NOCALENDAR, sizeof(MSG4225NOCALENDAR) - 1));
        }
    } else {
        return(ConnWrite(client->conn, MSG3241NOUSER, sizeof(MSG3241NOUSER) - 1));
    }

    if (!(client->flags & NMAP_CSTORE_READONLY)) {
        if ((*ptr++ == ' ') && ((ptr2 = strchr(ptr, ' ')) != NULL)) {
            *ptr2++ = '\0';
        } else {
            return(ConnWrite(client->conn, MSG3010BADARGC, sizeof(MSG3010BADARGC) - 1));
        }
    } else {
        return(ConnWrite(client->conn, MSG4243READONLY, sizeof(MSG4243READONLY) - 1));
    }

    id = atol(ptr);
    if (id && (id <= client->calendar.entries.used)) {
        id--;

        newFlags = atol(ptr2);
        flags = client->calendar.entries.info[id].State;
    } else {
        return(ConnWrite(client->conn, MSG4220NOMSG, sizeof(MSG4220NOMSG) - 1));
    }


    if ((flags & ~newFlags) == flags) {
        /* Flags are already set, ignore */
        return(ConnWriteF(client->conn, "1000 %lu %s\r\n", flags, MSG1000FLAGSET));
    }

    if (!(client->share.flags & NMAP_SHARE_CALENDAR)) {
        sprintf(client->path, "%s/%s/%s.idc", client->store, client->user, client->calendar.name);
    } else if (!(client->share.flags & NMAP_SHARE_CALENDAR_DENIED)) {
        if ((newFlags & MSG_STATE_READ) && !(client->share.calendar.permissions & NMAP_SHARE_SEEN) && (flags & MSG_STATE_READ)) {
            newFlags &= ~MSG_STATE_READ;                                
        }
  
        if ((newFlags & MSG_STATE_DELETED) && !(client->share.calendar.permissions & NMAP_SHARE_DELETE) && (flags & MSG_STATE_DELETED)) {
            newFlags &= ~MSG_STATE_DELETED;                                
        }
  
        sprintf(client->path, "%s/%s/%s.sdc", client->store, client->user, client->calendar.name);
    } else {
        return(ConnWrite(client->conn, MSG4240NOPERMISSION, sizeof(MSG4240NOPERMISSION) - 1));
    }

    if (WaitforAndLockCalendar(client->store, client->user, client->calendar.name) == 1) {
        index = fopen(client->path,"r+b");
    } else {
        return(ConnWrite(client->conn, MSG4120CALLOCKED, sizeof(MSG4120CALLOCKED) - 1));
    }

    if (index != NULL) {
        if (newFlags & MSG_STATE_PURGED) {
            newFlags &= ~MSG_STATE_PURGED;
        } else if ((newFlags & MSG_STATE_DELETED) && (flags & MSG_STATE_DELETED)) {
            client->calendar.entries.count++;
        }

        fseek(index, NMAP_CAL_IDX_HEADERSIZE + (id * sizeof(CalendarInfoStruct)) + offsetof(CalendarInfoStruct, State), SEEK_SET);

        flags &= ~newFlags;
        if (!(flags & MSG_STATE_RECENT)) {
            fwrite(&flags, sizeof(unsigned long), 1, index);
        } else {
            flags &= ~MSG_STATE_RECENT;
            fwrite(&flags, sizeof(unsigned long), 1, index);
            flags |= MSG_STATE_RECENT;
        }

        fclose(index);

        stat(client->path, &sb);
        client->calendar.indexTime = sb.st_mtime;

        client->calendar.entries.info[id].State = flags;
        UnlockCalendar(client->store, client->user, client->calendar.name);

        ccode = ConnWriteF(client->conn, "1000 %lu %s\r\n", flags, MSG1000FLAGSET);
    } else {
        UnlockCalendar(client->store, client->user, client->calendar.name);

        ccode = ConnWrite(client->conn, MSG4228CANTWRITECAL, sizeof(MSG4228CANTWRITECAL) - 1);
    }

    return(ccode);
}

int 
NmapCommandCsaflg(void *param)
{
    int ccode;
    unsigned long id;
    unsigned long newFlags;
    unsigned long flags;
    unsigned char *ptr;
    unsigned char *ptr2;
    struct stat sb;
    FILE *index;
    NMAPClient *client = (NMAPClient *)param;

    if (client->states & NMAP_CLIENT_USER) {
        if (client->calendar.name[0] != '\0') {
            CalendarOOBNotifications(client, &sb);

            ptr = client->buffer + 6;
        } else {
            return(ConnWrite(client->conn, MSG4225NOCALENDAR, sizeof(MSG4225NOCALENDAR) - 1));
        }
    } else {
        return(ConnWrite(client->conn, MSG3241NOUSER, sizeof(MSG3241NOUSER) - 1));
    }

    if (client->flags & NMAP_CSTORE_READONLY) {
        return(ConnWrite(client->conn, MSG4243READONLY, sizeof(MSG4243READONLY) - 1));
    }

    if ((*ptr++ = ' ') && (*ptr != ' ') && ((ptr2 = strchr(ptr, ' ')) != NULL)) {
        *ptr2++ = '\0';
    } else {
        return(ConnWrite(client->conn, MSG3010BADARGC, sizeof(MSG3010BADARGC) - 1));
    }

    id = atol(ptr);
    if (id && (id <= client->calendar.entries.used)) {
        id--;

        newFlags = atol(ptr2);
        flags = client->calendar.entries.info[id].State;
    } else {
        return(ConnWrite(client->conn, MSG4220NOMSG, sizeof(MSG4220NOMSG) - 1));
    }

    if ((flags & newFlags) == newFlags) {
        /* Flags are already set, ignore */
        return(ConnWriteF(client->conn, "1000 %lu %s\r\n", flags, MSG1000FLAGSET));
    }

    if (!(client->share.flags & NMAP_SHARE_CALENDAR)) {
        sprintf(client->path, "%s/%s/%s.idc", client->store, client->user, client->calendar.name);
    } else if (!(client->share.flags & NMAP_SHARE_CALENDAR_DENIED)) {
        if ((newFlags & MSG_STATE_READ) && !(client->share.calendar.permissions & NMAP_SHARE_SEEN)) {
            newFlags &= ~MSG_STATE_READ;
        }

        if ((newFlags & (MSG_STATE_DELETED | MSG_STATE_PURGED)) && !(client->share.calendar.permissions & NMAP_SHARE_DELETE)) {
            newFlags &= ~(MSG_STATE_DELETED | MSG_STATE_PURGED);
        }

        sprintf(client->path, "%s/%s/%s.sdc", client->store, client->user, client->calendar.name);
    } else {
        return(ConnWrite(client->conn, MSG4240NOPERMISSION, sizeof(MSG4240NOPERMISSION) - 1));
    }

    if (WaitforAndLockCalendar(client->store, client->user, client->calendar.name) == 1) {
        index = fopen(client->path,"r+b");
    } else {
        return(ConnWrite(client->conn, MSG4120CALLOCKED, sizeof(MSG4120CALLOCKED) - 1));
    }

    if (index != NULL) {
        if ((newFlags & MSG_STATE_DELETED) && !(flags & MSG_STATE_DELETED)) {
            client->calendar.entries.count--;
        }

        if (newFlags & MSG_STATE_PURGED) {
            client->calendar.changed = TRUE;
        }

        flags |= newFlags;

        fseek(index, NMAP_CAL_IDX_HEADERSIZE + (id * sizeof(CalendarInfoStruct)) + offsetof(CalendarInfoStruct, State), SEEK_SET);
        fwrite(&flags, sizeof(unsigned long), 1, index);
        fclose(index);

        stat(client->path, &sb);
        client->calendar.indexTime = sb.st_mtime;

        client->calendar.entries.info[id].State = flags;
        UnlockCalendar(client->store, client->user, client->calendar.name);

        ccode = ConnWriteF(client->conn, "1000 %lu %s\r\n", flags, MSG1000FLAGSET);
    } else {
        UnlockCalendar(client->store, client->user, client->calendar.name);

        ccode = ConnWrite(client->conn, MSG4228CANTWRITECAL, sizeof(MSG4228CANTWRITECAL) - 1);
    }

    return(ccode);
}

int 
NmapCommandCsatnd(void *param)
{
    int ccode;
    unsigned long id;
    unsigned long len;
    unsigned long count;
    unsigned long updatePos;
    unsigned char *ptr;
    unsigned char *state;
    unsigned char *address;
    CalendarInfoStruct *cInfo;
    struct stat sb;
    NMAPClient *client = (NMAPClient *)param;

    if (client->states & NMAP_CLIENT_USER) {
        if (client->calendar.name[0] != '\0') {
            CalendarOOBNotifications(client, &sb);

            ptr = client->buffer + 6;
        } else {
            return(ConnWrite(client->conn, MSG4225NOCALENDAR, sizeof(MSG4225NOCALENDAR) - 1));
        }
    } else {
        return(ConnWrite(client->conn, MSG3241NOUSER, sizeof(MSG3241NOUSER) - 1));
    }

    if ((*ptr++ = ' ') && (isdigit(*ptr))) {
        id = atol(ptr);
    } else {
        return(ConnWrite(client->conn, MSG3010BADARGC, sizeof(MSG3010BADARGC) - 1));
    }

    if (id && (id <= client->calendar.entries.used)) {
        id--;
    } else {
        return(ConnWrite(client->conn, MSG4220NOMSG, sizeof(MSG4220NOMSG) - 1));
    }

    if (!(client->share.flags & NMAP_SHARE_CALENDAR)) {
        cInfo = &(client->calendar.entries.info[id]);

        state = strchr(ptr, ' ');
        if (!state) {
            if (cInfo->AttendeeCount > 0) {
                sprintf(client->path, "%s/%s/%s.cal", client->store, client->user, client->calendar.name);
                client->calendar.fh = fopen(client->path,"rb");
                if (client->calendar.fh != NULL) {
                    fseek(client->calendar.fh, cInfo->AttendeePos, SEEK_SET);
                } else {
                    return(ConnWrite(client->conn, MSG4224CANTREADCAL, sizeof(MSG4224CANTREADCAL) - 1));
                }

                count = 0;
                ccode = ConnWriteF(client->conn, "2002-%lu %lu List coming up\r\n", cInfo->AttendeeCount, cInfo->AttendeeSize);

                while ((ccode != -1) && !feof(client->calendar.fh) && !ferror(client->calendar.fh) && (count < cInfo->AttendeeSize)) {
                    if (fgets(client->line, CONN_BUFSIZE, client->calendar.fh) != NULL) {
                        count += strlen(client->line);
                        CHOP_NEWLINE(client->line);

                        /* Parse the attendee line */
                        ccode = ConnWriteF(client->conn, "2002-%c %c %c %c %s\r\n", 
                            client->line[NMAP_CAL_STATE_POS], 
                            client->line[NMAP_CAL_ROLE_POS], 
                            client->line[NMAP_CAL_RSVP_POS], 
                            client->line[NMAP_CAL_TYPE_POS], 
                            client->line + NMAP_CAL_ADDRESS_POS); 
                    }
                }

                fclose(client->calendar.fh);
                client->calendar.fh = NULL;
            }

            return(ConnWrite(client->conn, MSG1000OK, sizeof(MSG1000OK) - 1));
        }

        /* We're setting attendee status */
        *(state++) = '\0';

        if (isspace(state[1])) {
            state[1] = '\0';
        } else {
            return(ConnWrite(client->conn, MSG3010BADARGC, sizeof(MSG3010BADARGC) - 1));
        }

        address = state + 2;
        len = strlen(address);

        if (WaitforAndLockCalendar(client->store, client->user, client->calendar.name) == 1) {
            sprintf(client->path, "%s/%s/%s.cal", client->store, client->user, client->calendar.name);
        } else {
            return(ConnWrite(client->conn, MSG4120CALLOCKED, sizeof(MSG4120CALLOCKED) - 1));
        }

        client->calendar.fh = fopen(client->path, "rb");
        if (client->calendar.fh != NULL) {
            fseek(client->calendar.fh, cInfo->AttendeePos, SEEK_SET);
        } else {
            return(ConnWrite(client->conn, MSG4224CANTREADCAL, sizeof(MSG4224CANTREADCAL) - 1));
        }

        updatePos = 0;
        count = 0;
        while (!feof(client->calendar.fh) && !ferror(client->calendar.fh) && (count < cInfo->AttendeeSize)) {
            if (fgets(client->line, CONN_BUFSIZE, client->calendar.fh) != NULL) {
                count += strlen(client->line);

                /* Check address with what we read */
                if ((QuickNCmp(client->line + NMAP_CAL_ADDRESS_POS, address, len)) && (client->line[NMAP_CAL_ADDRESS_POS + len] == ' ')) {
                    updatePos = cInfo->AttendeePos + count - strlen(client->line) + NMAP_CAL_STATE_POS;
                    break;
                }
            }
        }

        fclose(client->calendar.fh);

        if (updatePos != 0) {
            sprintf(client->path, "%s/%s/%s.cal", client->store, client->user, client->calendar.name);
            client->calendar.fh = fopen(client->path,"r+b");
            if (client->calendar.fh != NULL) {
                fseek(client->calendar.fh, updatePos, SEEK_SET);
                fwrite(state, sizeof(unsigned char), 1, client->calendar.fh);
                fclose(client->calendar.fh);
            } else {
                return(ConnWrite(client->conn, MSG4228CANTWRITECAL, sizeof(MSG4228CANTWRITECAL) - 1));
            }
        }

        client->calendar.fh = NULL;
        UnlockCalendar(client->store, client->user, client->calendar.name);

        return(ConnWrite(client->conn, MSG1000OK, sizeof(MSG1000OK) - 1));
    }
    
    if (!(client->share.flags & NMAP_SHARE_CALENDAR_DENIED)) {
        if (((ccode = NMAPSendCommandF(client->share.calendar.conn, "CSATND %s\r\n", ptr)) != -1) 
                && ((ccode = NMAPReadAnswerLine(client->share.calendar.conn, client->line, CONN_BUFSIZE, FALSE)) != -1)) {
            while (ccode == 2002) {
                ccode = ConnWrite(client->conn, client->line, strlen(client->line));
                if (ccode != -1) {
                    ccode = NMAPReadAnswerLine(client->share.calendar.conn, client->line, CONN_BUFSIZE, FALSE);
                }
            }

            if (ccode != -1) {
                ccode = ConnWrite(client->conn, client->line, strlen(client->line));
            }
        } else {
            ccode = ConnWrite(client->conn, MSG4228CANTWRITECAL, sizeof(MSG4228CANTWRITECAL) - 1);
        }
    } else {
        ccode = ConnWrite(client->conn, MSG4240NOPERMISSION, sizeof(MSG4240NOPERMISSION) - 1);
    }

    return(ccode);
}

int 
NmapCommandCsupda(void *param)
{
    int ccode;
    unsigned long id;
    unsigned long state[2];
    unsigned long cUsed;
    unsigned long totalSize;
    unsigned long totalCount;
    unsigned long purgedSize;
    unsigned long purgedCount;
    unsigned long deletedSize;
    unsigned long deletedCount;
    unsigned long newSize;
    unsigned long newCount;
    CalendarInfoStruct *cInfo;
    struct stat sb;
    FILE *index;
    NMAPClient *client = (NMAPClient *)param;

    /* fixme - verify command buffer; no arguments means no delimiter. */
    if (client->states & NMAP_CLIENT_USER) {
        if (client->calendar.name[0] != '\0') {
            CalendarOOBNotifications(client, &sb);

            if (!(client->share.flags & NMAP_SHARE_CALENDAR)) {
                sprintf(client->path, "%s/%s/%s.idc", client->store, client->user, client->calendar.name);
            } else if (!(client->share.flags & NMAP_SHARE_CALENDAR_DENIED)) {
                sprintf(client->path, "%s/%s/%s.sdc", client->store, client->user, client->calendar.name);
            } else {
                return(ConnWrite(client->conn, MSG4240NOPERMISSION, sizeof(MSG4240NOPERMISSION) - 1));
            }
        } else {
            return(ConnWrite(client->conn, MSG4225NOCALENDAR, sizeof(MSG4225NOCALENDAR) - 1));
        }
    } else {
        return(ConnWrite(client->conn, MSG3241NOUSER, sizeof(MSG3241NOUSER) - 1));
    }

    cUsed = client->calendar.entries.used;
    totalSize = 0;
    totalCount = 0;
    purgedSize = 0;
    purgedCount = 0;
    deletedSize = 0;
    deletedCount = 0;
    newSize = 0;
    newCount = 0;

    if (client->calendar.indexChanged) {
        /* Detect the delta between disk and memory; the only thing that could change are flags */
        if (WaitforAndLockCalendar(client->store, client->user, client->calendar.name) == 1) {
            index = fopen(client->path, "rb");
            if (index != NULL) {
                id = 0;
                cInfo = client->calendar.entries.info;
            } else {
                UnlockCalendar(client->store, client->user, client->calendar.name);

                return(ConnWrite(client->conn, MSG4224CANTREADCAL, sizeof(MSG4224CANTREADCAL) - 1));
            }
        } else {
            return(ConnWrite(client->conn, MSG4120CALLOCKED, sizeof(MSG4120CALLOCKED) - 1));
        }

        ccode = 0;
        for (; id < cUsed; (ccode != -1) && id++, cInfo++) {
            if (fseek(index, NMAP_CAL_IDX_HEADERSIZE + ((unsigned char *)cInfo - (unsigned char *)client->calendar.entries.info) + offsetof(CalendarInfoStruct, State), SEEK_SET) == 0) {
                /* This is tricky; we read the State and UTCend at once... */
                fread(state, sizeof(unsigned long), 2, index);

                cInfo->UTCEnd = state[1];
                if (state[0] & MSG_STATE_PURGED) {
                    client->calendar.changed = TRUE;
                }

                if (cInfo->State == state[0]) {
                    continue;
                }

                cInfo->State = state[0];

                ccode = ConnWriteF(client->conn, "6001-%lu %lu\r\n", id + 1, state[0]);

                continue;
            } else {
                fclose(index);

                UnlockCalendar(client->store, client->user, client->calendar.name);

                return(ConnWrite(client->conn, MSG4224CANTREADCAL, sizeof(MSG4224CANTREADCAL) - 1));
            }
        }

        fclose(index);

        client->calendar.indexChanged = FALSE;
        UnlockCalendar(client->store, client->user, client->calendar.name);
    }

    /* New calendar entries? */
    if (!(client->share.flags & NMAP_SHARE_CALENDAR)) {
        sprintf(client->path, "%s/%s/%s.cal", client->store, client->user, client->calendar.name);

        if ((stat(client->path, &sb) == 0) && (client->calendar.size != (unsigned long)sb.st_size)) {
            ParseCalendar(client);
        }
    } else if (!(client->share.flags & NMAP_SHARE_CALENDAR_DENIED) && (client->share.calendar.permissions & NMAP_SHARE_DELETE)) {
        if (SyncSharedCalendar(client, NULL, FALSE) == FALSE) {
            return(ConnWrite(client->conn, MSG4224CANTREADCAL, sizeof(MSG4224CANTREADCAL) - 1));
        }
    }

    cInfo = client->calendar.entries.info;
    for (id = 0; id < client->calendar.entries.used; id++, cInfo++) {
        totalSize += cInfo->Size;
        totalCount++;

        if (!(cInfo->State & (MSG_STATE_PURGED | MSG_STATE_DELETED))) {
            if (id < cUsed) {
                continue;
            }

            newSize += cInfo->Size;
            newCount++;
        } else if (cInfo->State & MSG_STATE_PURGED) {
            purgedSize += cInfo->Size;
            purgedCount++;
        } else {
            deletedSize += cInfo->Size;
            deletedCount++;
        }
    }

    return(ConnWriteF(client->conn, "1000 %s %lu %lu %lu %lu %lu %lu %lu %lu\r\n", client->user, totalCount, totalSize, newCount, newSize, purgedCount, purgedSize, deletedCount, deletedSize));
}

int 
NmapCommandCsunsubs(void *param)
{
    int ccode;
    unsigned long count;
    unsigned char *ptr;
    struct stat sb;
    FILE *index;
    NMAPClient *client = (NMAPClient *)param;
    NMAPClient *sclient;

    if (client->states & NMAP_CLIENT_USER) {
        if (client->calendar.name[0] != '\0') {
            CalendarOOBNotifications(client, &sb);
        }

        ptr = client->buffer + 8;
    } else {
        return(ConnWrite(client->conn, MSG3241NOUSER, sizeof(MSG3241NOUSER) - 1));
    }

    if ((*ptr++ == ' ') && *ptr) {
        if (XplStrCaseCmp(client->calendar.name, ptr) == 0) {
            return(ConnWrite(client->conn, MSG4227TGTCALSELECTED, sizeof(MSG4227TGTCALSELECTED) - 1));
        }

        if (strchr(ptr, '.')) {
            if ((strstr(ptr, "../") != NULL) || (strstr(ptr, "..\\") != NULL)) {
                return(ConnWrite(client->conn, MSG4224NOCSTORE, sizeof(MSG4224NOCSTORE) - 1));
            }
        }

        if (XplStrCaseCmp("MAIN", ptr) == 0) {
            return(ConnWriteF(client->conn, MSG3014REQNOTALLOWED, "  MAIN cannot be unsubscribed"));
        }
    } else {
        return(ConnWrite(client->conn, MSG3010BADARGC, sizeof(MSG3010BADARGC) - 1));
    }

    sclient = NMAPClientAlloc();
    if (sclient != NULL) {
        strcpy(sclient->user, client->user);
        sclient->userHash = client->userHash; 
        strcpy(sclient->calendar.name, ptr);
        sclient->store = client->store;
        sclient->mailbox.newline = TRUE;
    } else {
        return(ConnWrite(client->conn, MSG5001NOMEMORY, sizeof(MSG5001NOMEMORY) - 1));
    }

    sprintf(client->path, "%s/%s/%s.cal", sclient->store, sclient->user, sclient->calendar.name);
    if (access(client->path, 0) == 0) {
        if (ReadNLockAquire(&sclient->calendar.lock, &client->userHash, client->user, sclient->calendar.name, 0) == NLOCK_AQUIRED) {
            ParseCalendar(sclient);
            ReadNLockRelease(sclient->calendar.lock);
            sclient->calendar.lock = NULL;
        }

        if (!(sclient->calendar.flags & RESOURCE_FLAG_UNSUBSCRIBED)) {
            WaitforAndLockCalendar(client->store, client->user, sclient->calendar.name);

            sprintf(client->path, "%s/%s/%s.idc", client->store, client->user, sclient->calendar.name);
            index = fopen(client->path, "r+b");

            /* Calendars are subscribed by default, if no index file is found for the 
               calendar, it is considered subscribed. */
            if (index != NULL) {
                sclient->calendar.flags |= RESOURCE_FLAG_UNSUBSCRIBED;
                count = sprintf(client->line, "%010lu\r\n", sclient->calendar.flags);

                /* Skip    NMAP_CAL_IDX_VERSION "CNIMSv05/r/n" */
                fseek(index, 10, SEEK_SET);
                fwrite(client->path, sizeof(unsigned char), count, index);
                fclose(index);

                UnlockCalendar(client->store, client->user, sclient->calendar.name);

                ccode = ConnWrite(client->conn, MSG1000OK, sizeof(MSG1000OK) - 1);
            } else {
                UnlockCalendar(client->store, client->user, sclient->calendar.name);

                ccode = ConnWrite(client->conn, MSG4120CALLOCKED, sizeof(MSG4120CALLOCKED) - 1);
            }
        } else {
            ccode = ConnWrite(client->conn, MSG1000OK, sizeof(MSG1000OK) - 1);
        }
    } else if (IsAvailableShare(sclient->user, sclient->calendar.name, NMAP_SHARE_CALENDAR, NULL, NULL)) {
        /* Shared calendars are unsubscribed by default, since no index file was found for the 
           shared calendar, it is already considered unsubscribed.    */
        if (SetShareSubscription(sclient->store, sclient->user, sclient->userHash, sclient->calendar.name, NMAP_SHARE_CALENDAR, FALSE)) {
            ccode = ConnWrite(client->conn, MSG1000OK, sizeof(MSG1000OK) - 1);
        } else {
            ccode = ConnWrite(client->conn, MSG4120CALLOCKED, sizeof(MSG4120CALLOCKED) - 1);
        }
    } else {
        ccode = ConnWrite(client->conn, MSG4224NOCSTORE, sizeof(MSG4224NOCSTORE) - 1);
    }

    if (sclient->calendar.entries.info != NULL) {
        MemFree(sclient->calendar.entries.info);
    }

    NMAPClientFree(sclient);

    return(ccode);
}

int 
NmapCommandCsrmov(void *param)
{
    unsigned char *ptr;
    unsigned char scratch[(XPL_MAX_PATH * 2) + 1];
    struct stat sb;
    FILE *index;
    NMAPClient *client = (NMAPClient *)param;
    MDBValueStruct *vs;

    if (client->states & NMAP_CLIENT_USER) {
        if (client->calendar.name[0] != '\0') {
            CalendarOOBNotifications(client, &sb);
        }

        ptr = client->buffer + 6;
    } else {
        return(ConnWrite(client->conn, MSG3241NOUSER, sizeof(MSG3241NOUSER) - 1));
    } 

    if ((*ptr++ == ' ') && (*ptr != ' ') && *ptr) {
        if (strchr(ptr, '.')) {
            if ((strstr(ptr, "../") != NULL) || (strstr(ptr, "..\\") != NULL)) {
                return(ConnWrite(client->conn, MSG4224NOCSTORE, sizeof(MSG4224NOCSTORE) - 1));
            }
        }

        if (XplStrCaseCmp(client->calendar.name, ptr) == 0) {
            return(ConnWrite(client->conn, MSG4227TGTCALSELECTED, sizeof(MSG4227TGTCALSELECTED) - 1));
        }

        if ((strlen(client->store) + strlen(client->user) + strlen(ptr) + 4) > XPL_MAX_PATH) {
            return(ConnWrite(client->conn, MSG4230PATHTOOLONG, sizeof(MSG4230PATHTOOLONG) - 1));
        }
    } else {
        return(ConnWrite(client->conn, MSG3010BADARGC, sizeof(MSG3010BADARGC) - 1));
    }

    sprintf(client->path, "%s/%s/%s.cal", client->store, client->user, ptr);
    if (access(client->path, 0) == 0) {
        sprintf(scratch, "%s/%s/%s.idc", client->store, client->user, ptr);
        index = fopen(scratch, "rb");
        if (index != NULL) {
            fseek(index, 10, SEEK_SET);
            fgets(client->line, CONN_BUFSIZE, index);
            fclose(index);

            if (atol(client->line) & RESOURCE_FLAG_SHARED) {
                return(ConnWrite(client->conn, MSG4221CALSHARED, sizeof(MSG4221CALSHARED) - 1));
            }
        }
    } else if (IsAvailableShare(client->user, ptr, NMAP_SHARE_CALENDAR, NULL, NULL)) {
        if (RemoveAvailableShare(client, NMAP_SHARE_CALENDAR, ptr)) {
            sprintf(client->path, "%s/%s/%s.sdc", client->store, client->user, ptr);
            unlink(client->path);

            CollapseSubTree(client->store, client->user, ptr, scratch, sizeof(scratch) - 1);

            return(ConnWrite(client->conn, MSG1000OK, sizeof(MSG1000OK) - 1));
        }

        return(ConnWrite(client->conn, MSG4120CALLOCKED, sizeof(MSG4120CALLOCKED) - 1));
    } else {
        /* May be a dangling hierarchical element. */
        client->path[strlen(client->path) - 4] = '\0';
        if (access(client->path, 0) == 0) {
            if (rmdir(client->path) == 0) {
                CollapseSubTree(client->store, client->user, ptr, scratch, sizeof(scratch) - 1);

                return(ConnWrite(client->conn, MSG1000OK, sizeof(MSG1000OK) - 1));
            }

            return(ConnWrite(client->conn, MSG4120CALLOCKED, sizeof(MSG4120CALLOCKED) - 1));
        }

        return(ConnWrite(client->conn, MSG4224NOCSTORE, sizeof(MSG4224NOCSTORE) - 1));
    }

    if (unlink(client->path)) {
        return(ConnWrite(client->conn, MSG4224NOCSTORE, sizeof(MSG4224NOCSTORE) - 1));
    }

    sprintf(client->path, "%s/%s/%s.idc", client->store, client->user, ptr);
    unlink(client->path);

    CollapseSubTree(client->store, client->user, ptr, scratch, sizeof(scratch) - 1);

    if (XplStrCaseCmp(ptr, "MAIN")==0) {
        /* The guy removed the main calendar, should recreate it */
        sprintf(client->path, "%s/%s/MAIN.cal", client->store, client->user);
        fclose(fopen(client->path, "wb"));

        sprintf(client->path, "%s/%s/MAIL.idc", client->store, client->user);
        index = fopen(client->path, "wb");
        fprintf(index, "%s\r\n0000000000\r\n0000000000\r\n0000000000\r\n%010lu\r\n%010lu\r\n", NMAP_CAL_IDX_VERSION, NMAP.universalCounter++, NMAP.universalCounter++);
        sprintf(client->line, "%lu", NMAP.universalCounter);

        vs = MDBCreateValueStruct(NMAP.handle.directory, NMAP.server.dn);
        MDBAddValue(client->line, vs);
        MDBWrite(MSGSRV_AGENT_NMAP, MSGSRV_A_UID, vs);
        MDBDestroyValueStruct(vs);

        fclose(index);
    }

    return(ConnWrite(client->conn, MSG1000OK, sizeof(MSG1000OK) - 1));
}

int 
NmapCommandCsrnam(void *param)
{
    unsigned char *ptr;
    unsigned char *ptr2;
    unsigned char path[XPL_MAX_PATH + 1];
    struct stat sb;
    FILE *index;
    NMAPClient *client = (NMAPClient *)param;
    MDBValueStruct *vs;

    if (client->states & NMAP_CLIENT_USER) {
        if (client->calendar.name[0] != '\0') {
            CalendarOOBNotifications(client, &sb);
        }

        ptr = client->buffer + 6;
    } else {
        return(ConnWrite(client->conn, MSG3241NOUSER, sizeof(MSG3241NOUSER) - 1));
    } 

    if ((*ptr++ == ' ') && (!isspace(*ptr)) && *ptr && ((ptr2 = strchr(ptr, ' ')) != NULL)) {
        if (strchr(ptr, '.')) {
            if ((strstr(ptr, "../") != NULL) || (strstr(ptr, "..\\") != NULL)) {
                return(ConnWrite(client->conn, MSG4224NOCSTORE, sizeof(MSG4224NOCSTORE) - 1));
            }
        }

        *ptr2++ = '\0';

        if (XplStrCaseCmp(client->calendar.name, ptr) == 0) {
            return(ConnWrite(client->conn, MSG4227TGTCALSELECTED, sizeof(MSG4227TGTCALSELECTED) - 1));
        }

        if ((strlen(client->store) + strlen(client->user) + strlen(ptr) + 4) > XPL_MAX_PATH) {
            return(ConnWrite(client->conn, MSG4230PATHTOOLONG, sizeof(MSG4230PATHTOOLONG) - 1));
        }

        if ((strlen(client->store) + strlen(client->user) + strlen(ptr2) + 4) > XPL_MAX_PATH) {
            return(ConnWrite(client->conn, MSG4230PATHTOOLONG, sizeof(MSG4230PATHTOOLONG) - 1));
        }
    } else {
        return(ConnWrite(client->conn, MSG3010BADARGC, sizeof(MSG3010BADARGC) - 1));
    }

    sprintf(client->path, "%s/%s/%s.cal", client->store, client->user, ptr);
    if (access(client->path, 0) == 0) {
        sprintf(path, "%s/%s/%s.idc", client->store, client->user, ptr);
        index = fopen(path, "rb");
        if (index != NULL) {
            fseek(index, 9, SEEK_SET);
            fgets(client->line, CONN_BUFSIZE, index);
            fclose(index);

            if (atol(client->line) & RESOURCE_FLAG_SHARED) {
                return(ConnWrite(client->conn, MSG4221CALSHARED, sizeof(MSG4221CALSHARED) - 1));
            }
        }

        sprintf(path, "%s/%s/%s.cal", client->store, client->user, ptr2);
    } else if (IsAvailableShare(client->user, ptr, NMAP_SHARE_CALENDAR, NULL, NULL)) {
        /* fixme - enhancment (support renaming shared resources)?
        RenameAvailableShare(client, NMAP_SHARE_CALENDAR, ptr, ptr2); */

        return(ConnWrite(client->conn, MSG4240NOPERMISSION, sizeof(MSG4240NOPERMISSION) - 1));
    } else {
        return(ConnWrite(client->conn, MSG4224NOCSTORE, sizeof(MSG4224NOCSTORE) - 1));
    }

    if (access(path, 0) == 0) {
        return(ConnWrite(client->conn, MSG4226CALEXISTS, sizeof(MSG4226CALEXISTS) - 1));
    }

    if (rename(client->path, path)) {
        return(ConnWrite(client->conn, MSG4224NOCSTORE, sizeof(MSG4224NOCSTORE) - 1));
    }

    SetLongPathName(path);

    sprintf(client->path, "%s/%s/%s.idc",client->store, client->user, ptr);
    sprintf(path, "%s/%s/%s.idc",client->store,client->user, ptr2);
    rename(client->path, path);

    SetLongPathName(path);

    if (XplStrCaseCmp(ptr, "MAIN")==0) {
        sprintf(client->path, "%s/%s/MAIN.cal", client->store, client->user);
        fclose(fopen(client->path, "wb"));

        sprintf(client->path, "%s/%s/MAIN.idc",client->store, client->user);
        index=fopen(client->path, "wb");
        fprintf(index, "%s\r\n0000000000\r\n0000000000\r\n0000000000\r\n%010lu\r\n%010lu\r\n", NMAP_CAL_IDX_VERSION, NMAP.universalCounter++, NMAP.universalCounter++);
        sprintf(client->line, "%lu", NMAP.universalCounter);

        vs = MDBCreateValueStruct(NMAP.handle.directory, NMAP.server.dn);
        MDBAddValue(client->line, vs);
        MDBWrite(MSGSRV_AGENT_NMAP, MSGSRV_A_UID, vs);
        MDBDestroyValueStruct(vs);
        fclose(index);
    } else {
        sprintf(client->path, "%s/%s/%s", client->store, client->user, ptr);

        if (access(client->path, 0) == 0) {
            sprintf(path, "%s/%s/%s", client->store,client->user, ptr2);
            rename(client->path, path);
            SetLongPathName(path);
        }
    }

    return(ConnWrite(client->conn, MSG1000OK, sizeof(MSG1000OK) - 1));
}

int 
NmapCommandCsstraw(void *param)
{
    int ccode;
    long len;
    unsigned long count;
    unsigned long id;
    unsigned char *ptr;
    unsigned char *ptr2;
    unsigned char *sender;
    unsigned char *authSender;
    unsigned char *resourceOwner;
    unsigned char *resourceName;
    unsigned char *calendar;
    BOOL ready;
    struct stat sb;
    FILE *writeCal;
    NMAPClient *client = (NMAPClient *)param;

    if (client->states & NMAP_CLIENT_USER) {
        if (client->calendar.name[0] != '\0') {
            CalendarOOBNotifications(client, &sb);
        }

        ptr = client->buffer + 7;
    } else {
        return(ConnWrite(client->conn, MSG3241NOUSER, sizeof(MSG3241NOUSER) - 1));
    }

    /* CSSTRAW <calendar>[[ <count>][ FROM <sender> <authenticated sender>]]CRLF */
    if ((*ptr++ == ' ') && (!isspace(*ptr)) && *ptr) {
        sender = client->user;
        authSender = client->user;

        calendar = ptr;

        ptr2 = strchr(ptr, ' ');
        if (ptr2) {
            *ptr2++ = '\0';

            if (isdigit(*ptr2)) {
                count = atol(ptr2);

                ptr2 = strchr(ptr2, ' ');
                if (ptr2) {
                    *ptr2++ = '\0';
                }
            } else {
                count = 0;
            }

            if (ptr2 && (XplStrNCaseCmp(++ptr2, "FROM ", 5) == 0)) {
                sender = ptr2 + 5;
                ptr2 = strchr(sender, ' ');
                if (ptr2) {
                    *ptr2++ = '\0';

                    authSender = ptr2;
                } else {
                    return(ConnWrite(client->conn, MSG3010BADARGC, sizeof(MSG3010BADARGC) - 1));
                }
            } else if (ptr2) {
                return(ConnWrite(client->conn, MSG3010BADARGC, sizeof(MSG3010BADARGC) - 1));
            }
        } else {
            count = 0;
        }
    } else {
        return(ConnWrite(client->conn, MSG3010BADARGC, sizeof(MSG3010BADARGC) - 1));
    }

    if (!(NMAP.flags & NMAP_FLAG_DISK_SPACE_LOW)) {
        sprintf(client->path, "%s/%s/%s.cal", client->store, client->user, calendar);
    } else {
        return(ConnWrite(client->conn, MSG5221SPACELOW, sizeof(MSG5221SPACELOW) - 1));
    }

    if (access(client->path, 0) == 0) {
        XplWaitOnLocalSemaphore(NMAP.queue.semaphore);
        id = NMAP.queue.id++ & ((1 << 28) - 1);
        XplSignalLocalSemaphore(NMAP.queue.semaphore);

        sprintf(client->path, "%s/c%07lx.in", NMAP.path.spool, id);
        writeCal = fopen(client->path,"wb");
        if (writeCal) {
            ccode = ConnWrite(client->conn, "2002 Send calendar data\r\n", 25);
            if (ccode != -1) {
                ConnFlush(client->conn);
            }
        } else {
            return(ConnWrite(client->conn, MSG5221SPACELOW, sizeof(MSG5221SPACELOW) - 1));
        }
    } else if (IsAvailableShare(client->user, calendar, NMAP_SHARE_CALENDAR, &resourceOwner, &resourceName)) {
        /* fixme - not implemented
        HandleShareCSSTRAW(client, calendar, sender, authSender, count, resourceOwner, resourceName, line, CONN_BUFSIZE); */

        MemFree(resourceOwner);
        MemFree(resourceName);

        return(ConnWrite(client->conn, MSG1000NOTIMP, sizeof(MSG1000NOTIMP) - 1));
    } else {
        return(ConnWrite(client->conn, MSG4224NOMBOX, sizeof(MSG4224NOMBOX) - 1));
    }

    if (count > 0) {
        ccode = ConnReadToFile(client->conn, writeCal, count);
    } else {
        ccode = ConnReadToFileUntilEOS(client->conn, writeCal);
    }

    if (ccode != -1) {
        ready = TRUE;
    } else {
        ready = FALSE;
        count = errno;
    }

    fclose(writeCal);

    sprintf(client->path, "%s/c%07lx.in", NMAP.path.spool, id);

    if (ready) {
        stat(client->path, &sb);

        writeCal = fopen(client->path, "rb");
        if ((len = StoreCalendarEvent(sender, authSender, client->user, calendar, writeCal, sb.st_size, 0, TRUE)) == DELIVER_SUCCESS) {
            ccode = ConnWrite(client->conn, MSG1000OK, sizeof(MSG1000OK) - 1);
        } else {
            switch(len) {
                case DELIVER_QUOTA_EXCEEDED: {
                    ccode = ConnWrite(client->conn, MSG5220QUOTAERR, sizeof(MSG5220QUOTAERR) - 1);
                    break;
                }

                case DELIVER_PROCESSING_ERROR:
                default: {
                    ccode = ConnWrite(client->conn, MSG4228CANTWRITECAL, sizeof(MSG4228CANTWRITECAL) - 1);
                    break; 
                }
            }
        }

        if (!(NMAP.flushFlags & FLUSH_CAL_ON_STORE)) {
            fclose(writeCal);
        } else {
            XplFlushWrites(writeCal);
            fclose(writeCal);
        }
    } else if ((ccode == -1) && (count = ENOSPC)) {
        ccode = ConnWrite(client->conn, MSG5220QUOTAERR, sizeof(MSG5220QUOTAERR) - 1);
    } else {
        ccode = ConnWrite(client->conn, MSG4229CANTWRITEMBOX, sizeof(MSG4229CANTWRITEMBOX) - 1);
    }

    unlink(client->path);
        
    return(ccode);
}

int 
NmapCommandCsstor(void *param)
{
    int ccode;
    long len;
    unsigned long id;
    unsigned long count;
    unsigned char *ptr;
    unsigned char *ptr2;
    unsigned char *calendar;
    unsigned char *sender;
    unsigned char *authSender;
    unsigned char *owner;
    unsigned char *name;
    BOOL ready;
    struct stat sb;
    FILE *writeCal;
    NMAPClient *client = (NMAPClient *)param;

    if (client->states & NMAP_CLIENT_USER) {
        if (client->calendar.name[0] != '\0') {
            CalendarOOBNotifications(client, &sb);
        }

        ptr = client->buffer + 6;
    } else {
        return(ConnWrite(client->conn, MSG3241NOUSER, sizeof(MSG3241NOUSER) - 1));
    }

    /* CSSTOR <calendar>[[ <count>][ FROM <sender> <authenticated sender>]]CRLF */
    if ((*ptr++ == ' ') && (!isspace(*ptr)) && *ptr) {
        sender = client->user;
        authSender = client->user;

        calendar = ptr;

        ptr2 = strchr(ptr, ' ');
        if (ptr2) {
            *ptr2++ = '\0';

            if (isdigit(*ptr2)) {
                count = atol(ptr2);

                ptr2 = strchr(ptr2, ' ');
                if (ptr2) {
                    *ptr2++ = '\0';
                }
            } else {
                count = 0;
            }

            if (ptr2 && (XplStrNCaseCmp(ptr2, "FROM ", 5) == 0)) {
                sender = ptr2 + 5;
                ptr2 = strchr(sender, ' ');
                if (ptr2) {
                    *ptr2++ = '\0';

                    authSender = ptr2;
                } else {
                    return(ConnWrite(client->conn, MSG3010BADARGC, sizeof(MSG3010BADARGC) - 1));
                }
            } else if (ptr2) {
                return(ConnWrite(client->conn, MSG3010BADARGC, sizeof(MSG3010BADARGC) - 1));
            }
        } else {
            count = 0;
        }
    } else {
        return(ConnWrite(client->conn, MSG3010BADARGC, sizeof(MSG3010BADARGC) - 1));
    }

    if (!(NMAP.flags & NMAP_FLAG_DISK_SPACE_LOW)) {
        sprintf(client->path, "%s/%s/%s.cal", client->store, client->user, calendar);
    } else {
        return(ConnWrite(client->conn, MSG5221SPACELOW, sizeof(MSG5221SPACELOW) - 1));
    }

    if (access(client->path, 0) == 0) {
        XplWaitOnLocalSemaphore(NMAP.queue.semaphore);
        id = NMAP.queue.id++ & ((1 << 28) - 1);
        XplSignalLocalSemaphore(NMAP.queue.semaphore);

        sprintf(client->path, "%s/c%07lx.in", NMAP.path.spool, id);
        writeCal = fopen(client->path, "wb");
        if (writeCal) {
            ccode = ConnWrite(client->conn, "2002 Send calendar data\r\n", 25);
            if (ccode != -1) {
                ConnFlush(client->conn);
            }
        } else {
            return(ConnWrite(client->conn, MSG5221SPACELOW, sizeof(MSG5221SPACELOW) - 1));
        }
    } else if (IsAvailableShare(client->user, calendar, NMAP_SHARE_CALENDAR, &owner, &name)) {
        ccode = HandleShareCsstor(client, calendar, sender, authSender, count, owner, name);

        MemFree(owner);
        MemFree(name);

        return(ccode);
    } else {
        return(ConnWrite(client->conn, MSG4224NOMBOX, sizeof(MSG4224NOMBOX) - 1));
    }

    if (count > 0) {
        ccode = ConnReadToFile(client->conn, writeCal, count);
    } else {
        ccode = ConnReadToFileUntilEOS(client->conn, writeCal);
    }

    if (ccode != -1) {
        ready = TRUE;
    } else {
        ready = FALSE;
        count = errno;
    }

    fclose(writeCal);

    sprintf(client->path, "%s/c%07lx.in",NMAP.path.spool, id);

    if (ready) {
        stat(client->path, &sb);

        writeCal = fopen(client->path, "rb");
        if ((len = StoreCalendarEvent(client->user, client->user, client->user, calendar, writeCal, sb.st_size, 0, FALSE))==DELIVER_SUCCESS) {
            ccode = ConnWrite(client->conn, MSG1000OK, sizeof(MSG1000OK) - 1);
        } else {
            switch(len) {
                case DELIVER_QUOTA_EXCEEDED: {
                    ccode = ConnWrite(client->conn, MSG5220QUOTAERR, sizeof(MSG5220QUOTAERR) - 1);
                    break; 
                }

                case DELIVER_PROCESSING_ERROR:
                default: {
                    ccode = ConnWrite(client->conn, MSG4228CANTWRITECAL, sizeof(MSG4228CANTWRITECAL) - 1);
                    break; 
                }
            }
        }

        if (!(NMAP.flushFlags & FLUSH_CAL_ON_STORE)) {
            fclose(writeCal);
        } else {
            XplFlushWrites(writeCal);
            fclose(writeCal);
        }
    } else if ((ccode == -1) && (count = ENOSPC)) {
        ccode = ConnWrite(client->conn, MSG5220QUOTAERR, sizeof(MSG5220QUOTAERR) - 1);
    } else {
        ccode = ConnWrite(client->conn, MSG4229CANTWRITEMBOX, sizeof(MSG4229CANTWRITEMBOX) - 1);
    }

    unlink(client->path);
    
    return(ccode);
}

int 
NmapCommandCsstat(void *param)
{
    unsigned long i;
    unsigned long totalCount = 0;
    unsigned long totalSize = 0;
    unsigned long purgedCount = 0;
    unsigned long purgedSize = 0;
    unsigned long deletedCount = 0;
    unsigned long deletedSize = 0;
    CalendarInfoStruct *cInfo;
    struct stat sb;
    NMAPClient *client = (NMAPClient *)param;


    /* fixme - verify command buffer; no arguments means no delimiter. */
    if (client->states & NMAP_CLIENT_USER) {
        if (client->calendar.name[0] != '\0') {
            CalendarOOBNotifications(client, &sb);

            i = client->calendar.entries.used;
            cInfo = client->calendar.entries.info;
        } else {
            return(ConnWrite(client->conn, MSG4225NOCALENDAR, sizeof(MSG4225NOCALENDAR) - 1));
        }
    } else {
        return(ConnWrite(client->conn, MSG3241NOUSER, sizeof(MSG3241NOUSER) - 1));
    }

    while (i-- != 0) {
        totalSize += cInfo->Size;
        totalCount++;

        if (cInfo->State & MSG_STATE_DELETED) {
            deletedSize += cInfo->Size;
            deletedCount++;
        } else if (cInfo->State & MSG_STATE_PURGED) {
            purgedSize += cInfo->Size;
            purgedCount++;
        }

        cInfo++;
    }

    return(ConnWriteF(client->conn, "1000 %s %lu %lu 0 0 %lu %lu %lu %lu\r\n", client->user, totalCount, totalSize, purgedCount, purgedSize, deletedCount, deletedSize));
}

int 
NmapCommandCsshow(void *param)
{
    int ccode;
    unsigned long len;
    unsigned long count;
    unsigned char *ptr;
    unsigned char pattern[64];
    unsigned char scratch[(XPL_MAX_PATH * 2) + 1];
    struct stat sb;
    BOOL recurse = 256;
    NMAPClient *client = (NMAPClient *)param;
    MDBValueStruct *vs;

    if (client->states & NMAP_CLIENT_USER) {
        if (client->calendar.name[0] != '\0') {
            CalendarOOBNotifications(client, &sb);
        }

        ptr = client->buffer + 6;
    } else {
        return(ConnWrite(client->conn, MSG3241NOUSER, sizeof(MSG3241NOUSER) - 1));
    }

    if (*ptr == ' ') {
        strncpy(pattern, ptr + 1, sizeof(pattern) - 1);

        ptr = strchr(pattern, '*');
        if (ptr) {
            *ptr = '\0';
        }

        ptr = strchr(pattern, '%');
        if (ptr) {
            *ptr = '\0';

            ptr = pattern;
            recurse = 0;
            while (*ptr) {
                if (*ptr++ == '/') {
                    recurse++;
                }
            }
        }
    } else if (*ptr == '\0') {
        pattern[0] = '\0';
    } else {
        return(ConnWrite(client->conn, MSG3010BADARGC, sizeof(MSG3010BADARGC) - 1));
    }

    sprintf(scratch, "%s/%s", client->store, client->user);
    if (!XPLLongToDos(scratch, client->path, XPL_MAX_PATH)) {
        return(ConnWrite(client->conn, MSG4224NOSTORE, sizeof(MSG4224NOSTORE) - 1));
    }

    vs = MDBCreateValueStruct(NMAP.handle.directory, NULL);
    if (vs) {
        if (MsgFindObject(client->user, client->dn, NULL, NULL, NULL)) {
            ccode = ConnWrite(client->conn, "2002-Calendars coming up\r\n", 26);
        } else {
            MDBDestroyValueStruct(vs);

            return(ConnWriteF(client->conn, MSG5244USERLOOKUPFAILURE, client->user));
        }
    } else {
        return(ConnWrite(client->conn, MSG5001NOMEMORY, sizeof(MSG5001NOMEMORY) - 1));
    }

    MDBRead(client->dn, MSGSRV_A_AVAILABLE_SHARES, vs);
    MDBRead(MSGSRV_ROOT, MSGSRV_A_AVAILABLE_SHARES, vs);

    PrintCal(client, vs, client->path, pattern, strlen(client->path), scratch, recurse, NULL);

    len = strlen(pattern);
    for (count = 0; (ccode != -1) && (count < vs->Used); count++) {
        ptr = vs->Value[count];

        if ((*ptr++ != 'C') || (*ptr != 'A')) {
            continue;
        }

        ptr = strrchr(++ptr, '\r');
        if ((ptr++ == NULL) || ((len > 0) && (XplStrNCaseCmp(ptr, pattern, len) != 0))) {
            continue;
        }

        ccode = ConnWriteF(client->conn, "2002-%s\r\n", ptr);
    }

    if (ccode != -1) {
        ccode = ConnWrite(client->conn, MSG1000OK, sizeof(MSG1000OK) - 1);
    }

    MDBDestroyValueStruct(vs);

    return(ccode);
}

int 
NmapCommandCsshowsub(void *param)
{
    int ccode;
    unsigned long len;
    unsigned long count;
    unsigned char *ptr;
    unsigned char pattern[64];
    unsigned char scratch[(XPL_MAX_PATH * 2) + 1];
    BOOL recurse = 256;
    struct stat sb;
    FILE *index;
    NMAPClient *client = (NMAPClient *)param;
    NMAPClient *sclient;
    MDBValueStruct *vs;

    if (client->states & NMAP_CLIENT_USER) {
        if (client->calendar.name[0] != '\0') {
            CalendarOOBNotifications(client, &sb);
        }

        ptr = client->buffer + 9;
    } else {
        return(ConnWrite(client->conn, MSG3241NOUSER, sizeof(MSG3241NOUSER) - 1));
    }

    if (*ptr == ' ') {
        strncpy(pattern, ptr + 1, sizeof(pattern) - 1);

        ptr = strchr(pattern, '*');
        if (ptr) {
            *ptr = '\0';
        }

        ptr = strchr(pattern, '%');
        if (ptr) {
            *ptr = '\0';

            ptr = pattern;
            recurse = 0;
            while (*ptr) {
                if (*ptr++ == '/') {
                    recurse++;
                }
            }
        }
    } else if (*ptr == '\0') {
        pattern[0] = '\0';
    } else {
        return(ConnWrite(client->conn, MSG3010BADARGC, sizeof(MSG3010BADARGC) - 1));
    }

    sclient = NMAPClientAlloc();
    if (sclient != NULL) {
        strcpy(sclient->user, client->user);
        sclient->userHash = client->userHash; 
        sclient->store = client->store;
        sclient->mailbox.newline = TRUE;
    } else {
        return(ConnWrite(client->conn, MSG5001NOMEMORY, sizeof(MSG5001NOMEMORY) - 1));
    }

    sprintf(scratch, "%s/%s", client->store, client->user);
    if (!XPLLongToDos(scratch, client->path, XPL_MAX_PATH)) {
        return(ConnWrite(client->conn, MSG4224NOSTORE, sizeof(MSG4224NOSTORE) - 1));
    }

    vs = MDBCreateValueStruct(NMAP.handle.directory, NULL);
    if (vs) {
        if (MsgFindObject(client->user, client->dn, NULL, NULL, NULL)) {
            ccode = ConnWrite(client->conn, "2002-Calendars coming up\r\n", 26);
        } else {
            MDBDestroyValueStruct(vs);

            return(ConnWriteF(client->conn, MSG5244USERLOOKUPFAILURE, client->user));
        }
    } else {
        return(ConnWrite(client->conn, MSG5001NOMEMORY, sizeof(MSG5001NOMEMORY) - 1));
    }

    MDBRead(client->dn, MSGSRV_A_AVAILABLE_SHARES, vs);
    MDBRead(MSGSRV_ROOT, MSGSRV_A_AVAILABLE_SHARES, vs);

    PrintCal(client, vs, client->path, pattern, strlen(client->path), scratch, recurse, sclient);

    len = strlen(pattern);
    for (count = 0; (ccode != -1) && (count < vs->Used); count++) {
        ptr = vs->Value[count];

        if ((*(ptr++) != 'C') || (*ptr != 'A')) {
            continue;
        }

        ptr = strrchr(++ptr, '\r');
        if ((ptr++ == NULL) || ((len > 0) && (XplStrNCaseCmp(ptr, pattern, len) != 0))) {
            continue;
        }

        sprintf(sclient->path, "%s/%s/%s.sdc", client->store, client->user, ptr);
        if (ReadNLockAquire(&sclient->calendar.lock, &client->userHash, client->user, ptr, 0) == NLOCK_AQUIRED) {
            index = fopen(sclient->path, "rb");
            if (index != NULL) {
                fseek(index, 0, SEEK_SET);
                if (fgets(sclient->line, CONN_BUFSIZE, index)) {
                    if ((strncmp(sclient->line, NMAP_CAL_IDX_VERSION, 8) == 0) && (sclient->line[8] == '\r')) {
                        fgets(sclient->line, CONN_BUFSIZE, index);
                        sclient->calendar.flags = atol(sclient->line);
                    } else {
                        sclient->calendar.flags = RESOURCE_FLAG_UNSUBSCRIBED;
                    }
                } else {
                    sclient->calendar.flags = RESOURCE_FLAG_UNSUBSCRIBED;
                }

                fclose(index);
            } else {
                sclient->calendar.flags = RESOURCE_FLAG_UNSUBSCRIBED;
            }

            ReadNLockRelease(sclient->calendar.lock);
            sclient->calendar.lock = NULL;
        }

        if (!(sclient->calendar.flags & RESOURCE_FLAG_UNSUBSCRIBED)) {
            ccode = ConnWriteF(client->conn, "2002-%s\r\n", ptr);
        } else {
            sprintf(client->path, "%s/%s/%s", client->store, client->user, ptr);
            if (access(client->path, 0) == 0) {
                ccode = ConnWriteF(client->conn, "2002-%s/\r\n", ptr);
            }
        }
    }

    if (ccode != -1) {
        ccode = ConnWrite(client->conn, MSG1000OK, sizeof(MSG1000OK) - 1);
    }

    MDBDestroyValueStruct(vs);

    if (sclient->calendar.entries.info) {
        MemFree(sclient->calendar.entries.info);
        sclient->calendar.entries.info = NULL;
    }

    NMAPClientFree(sclient);

    return(ccode);
}

int 
NmapCommandCssubs(void *param)
{
    int ccode;
    unsigned long count;
    unsigned char *ptr;
    unsigned char *name;
    struct stat sb;
    FILE *index;
    NMAPClient *client = (NMAPClient *)param;
    NMAPClient *sclient;

    if (client->states & NMAP_CLIENT_USER) {
        if (client->calendar.name[0] != '\0') {
            CalendarOOBNotifications(client, &sb);
        }

        ptr = client->buffer + 6;
    } else {
        return(ConnWrite(client->conn, MSG3241NOUSER, sizeof(MSG3241NOUSER) - 1));
    }

    /* CSSUBS <calendar> */
    if ((*ptr++ == ' ') && (!isspace(*ptr)) && *ptr) {
        if (strchr(ptr, '.')) {
            if ((strstr(ptr, "../") != NULL) || (strstr(ptr, "..\\") != NULL)) {
                return(ConnWrite(client->conn, MSG4224NOCSTORE, sizeof(MSG4224NOCSTORE) - 1));
            }
        }
    } else {
        return(ConnWrite(client->conn, MSG3010BADARGC, sizeof(MSG3010BADARGC) - 1));
    }

    sclient = NMAPClientAlloc();
    if (sclient != NULL) {
        strcpy(sclient->user, client->user);
        sclient->userHash = client->userHash; 
        sclient->store = client->store;
        sclient->mailbox.newline = TRUE;
    } else {
        return(ConnWrite(client->conn, MSG5001NOMEMORY, sizeof(MSG5001NOMEMORY) - 1));
    }

    strcpy(sclient->calendar.name, ptr);

    sprintf(client->path, "%s/%s/%s.cal", client->store, client->user, sclient->calendar.name);
    if (access(client->path, 0) == 0) {
        if (ReadNLockAquire(&sclient->calendar.lock, &sclient->userHash, sclient->user, sclient->calendar.name, 0) == NLOCK_AQUIRED) {
            ParseCalendar(sclient);
            ReadNLockRelease(sclient->calendar.lock);
            sclient->calendar.lock = NULL;
        }

        if (sclient->calendar.flags & RESOURCE_FLAG_UNSUBSCRIBED) {
            WaitforAndLockCalendar(client->store, client->user, sclient->calendar.name);

            sprintf(client->path, "%s/%s/%s.idc", client->store, client->user, sclient->calendar.name);
            index = fopen(client->path, "r+b");
            if (index != NULL) {
                sclient->calendar.flags &= ~RESOURCE_FLAG_UNSUBSCRIBED;

                count = sprintf(client->line, "%010lu\r\n", sclient->calendar.flags);

                fseek(index, 10, SEEK_SET);        /*    Skip    NMAP_CAL_IDX_VERSION "CNIMSv05/r/n"    */
                fwrite(client->line, sizeof(unsigned char), 12, index);
                fclose(index);

                UnlockCalendar(client->store, client->user, sclient->calendar.name);

                ccode = ConnWrite(client->conn, MSG1000OK, sizeof(MSG1000OK) - 1);
            } else {
                UnlockCalendar(client->store, client->user, sclient->calendar.name);

                ccode = ConnWrite(client->conn, MSG4120CALLOCKED, sizeof(MSG4120CALLOCKED) - 1);
            }
        } else {
            ccode = ConnWrite(client->conn, MSG1000OK, sizeof(MSG1000OK) - 1);
        }
    } else if (IsAvailableShare(sclient->user, sclient->calendar.name, NMAP_SHARE_CALENDAR, NULL, NULL)) {
        name = strrchr(sclient->calendar.name, '/');
        if (name != NULL) {
            ptr = strchr(sclient->calendar.name, '/');

            while (ptr) {
                *ptr = '\0';

                sprintf(sclient->path, "%s/%s/%s", sclient->store, sclient->user, sclient->calendar.name);
                if (stat(sclient->path, &sb) != 0) {
                    XplMakeDir(sclient->path);
                    SetLongName(sclient->path, client->path);
                }

                *ptr = '/';
                ptr = strchr(ptr + 1, '/');
            }

            *name = '/';
        }

        if (SetShareSubscription(sclient->store, sclient->user, sclient->userHash, sclient->calendar.name, NMAP_SHARE_CALENDAR, TRUE)) {
            ccode = ConnWrite(client->conn, MSG1000OK, sizeof(MSG1000OK) - 1);
        } else {
            ccode = ConnWrite(client->conn, MSG4120CALLOCKED, sizeof(MSG4120CALLOCKED) - 1);
        }
    } else {
        ccode = ConnWrite(client->conn, MSG4224NOCSTORE, sizeof(MSG4224NOCSTORE) - 1);
    }

    if (sclient->calendar.entries.info) {
        MemFree(sclient->calendar.entries.info);
        sclient->calendar.entries.info = NULL;
    }

    NMAPClientFree(sclient);

    return(TRUE);
}

int 
NmapCommandCssalv(void *param)
{
    int ccode;
    unsigned long id;
    unsigned long delCount = 0;
    unsigned char *ptr;
    struct stat sb;
    FILE *index;
    BOOL all;
    CalendarInfoStruct *cInfo;
    NMAPClient *client = (NMAPClient *)param;

    if (client->states & NMAP_CLIENT_USER) {
        if (client->calendar.name[0] != '\0') {
            CalendarOOBNotifications(client, &sb);

            ptr = client->buffer + 6;
        } else {
            return(ConnWrite(client->conn, MSG4225NOCALENDAR, sizeof(MSG4225NOCALENDAR) - 1));
        }
    } else {
        return(ConnWrite(client->conn, MSG3241NOUSER, sizeof(MSG3241NOUSER) - 1));
    }

    if (*ptr == '\0') {
        all = TRUE;

        id = client->calendar.entries.used;
        cInfo = client->calendar.entries.info;
    } else if ((*ptr++ == ' ') && (*ptr != ' ') && *ptr) {
        all = FALSE;

        id = atol(ptr);
        if (id && (id <= client->calendar.entries.used)) {
            id--;
        } else {
            return(ConnWrite(client->conn, MSG4220NOMSG, sizeof(MSG4220NOMSG) - 1));
        }

        cInfo = &(client->calendar.entries.info[id]);
        id = 1;
    } else {
        return(ConnWrite(client->conn, MSG3010BADARGC, sizeof(MSG3010BADARGC) - 1));
    }

    if (!(client->share.flags & NMAP_SHARE_CALENDAR)) {
        sprintf(client->path, "%s/%s/%s.idc", client->store, client->user, client->calendar.name);
    } else if (!(client->share.flags & NMAP_SHARE_CALENDAR_DENIED) && (client->share.calendar.permissions & NMAP_SHARE_DELETE)) {
        sprintf(client->path, "%s/%s/%s.sdc", client->store, client->user, client->calendar.name);
    } else {
        return(ConnWrite(client->conn, MSG4240NOPERMISSION, sizeof(MSG4240NOPERMISSION) - 1));
    }

    index = NULL;
    for (; id != 0; id--, cInfo++) {
        if (!(cInfo->State & MSG_STATE_DELETED)) {
            continue;
        }

        cInfo->State &= ~MSG_STATE_DELETED;
        client->calendar.entries.count++;

        delCount++;

        /*  Be a bit fancy and save the lock/open until we actually need it; 
            in case someone does a SALV without deleted messages in the box */
        if (index != NULL) {
            fseek(index, NMAP_CAL_IDX_HEADERSIZE + ((unsigned char *)cInfo - (unsigned char *)client->calendar.entries.info) + offsetof(CalendarInfoStruct, State), SEEK_SET);
        } else {
            WaitforAndLockCalendar(client->store, client->user, client->calendar.name);

            index = fopen(client->path, "r+b");
            if (index != NULL) {
                fseek(index, NMAP_CAL_IDX_HEADERSIZE + ((unsigned char *)cInfo - (unsigned char *)client->calendar.entries.info) + offsetof(CalendarInfoStruct, State), SEEK_SET);
            } else {
                cInfo->State |= MSG_STATE_DELETED;

                UnlockCalendar(client->store, client->user, client->calendar.name);

                return(ConnWrite(client->conn, MSG4228CANTWRITECAL, sizeof(MSG4228CANTWRITECAL) - 1));
            }
        }

        if (!(cInfo->State & MSG_STATE_RECENT)) {
            fwrite(&(cInfo->State), sizeof(unsigned long), 1, index);
        } else {
            cInfo->State &= ~MSG_STATE_RECENT;
            fwrite(&(cInfo->State), sizeof(unsigned long), 1, index);
            cInfo->State |= MSG_STATE_RECENT;
        }
    }

    if (index != NULL) {
        fclose(index);

        stat(client->path, &sb);
        client->calendar.indexTime = sb.st_mtime;
        UnlockCalendar(client->store, client->user, client->calendar.name);
    }

    if (all) {
        ccode = ConnWriteF(client->conn, "1000 %lu %s\r\n", delCount, MSG1000SALVAGED);
    } else {
        ccode = ConnWriteF(client->conn, "1000 %s\r\n", MSG1000SALVAGED);
    }

    return(ccode);
}

int 
NmapCommandCssflg(void *param)
{
    int ccode;
    unsigned long id;
    unsigned long flags;
    unsigned long newFlags;
    unsigned char *ptr;
    unsigned char *ptr2;
    struct stat sb;
    FILE *index;
    NMAPClient *client = (NMAPClient *)param;

    if (client->states & NMAP_CLIENT_USER) {
        if (client->calendar.name[0] != '\0') {
            CalendarOOBNotifications(client, &sb);

            ptr = client->buffer + 6;
        } else {
            return(ConnWrite(client->conn, MSG4225NOCALENDAR, sizeof(MSG4225NOCALENDAR) - 1));
        }
    } else {
        return(ConnWrite(client->conn, MSG3241NOUSER, sizeof(MSG3241NOUSER) - 1));
    }

    if (client->flags & NMAP_CSTORE_READONLY) {
        return(ConnWrite(client->conn, MSG4243READONLY, sizeof(MSG4243READONLY) - 1));
    }

    if ((*ptr++ == ' ') && (*ptr != ' ') && ((ptr2 = strchr(ptr, ' ')) != NULL)) {
        *ptr2++ = '\0';
    } else {
        return(ConnWrite(client->conn, MSG3010BADARGC, sizeof(MSG3010BADARGC) - 1));
    }

    id = atol(ptr);
    if (id && (id <= client->calendar.entries.used)) {
        id--;
    } else {
        return(ConnWrite(client->conn, MSG4220NOMSG, sizeof(MSG4220NOMSG) - 1));
    }

    newFlags = atol(ptr2);
    flags = client->calendar.entries.info[id].State;

    if (flags == newFlags) {
        return(ConnWriteF(client->conn, "1000 %lu %s\r\n", flags, MSG1000FLAGSET));
    }

    if (!(client->share.flags & NMAP_SHARE_CALENDAR)) {
        sprintf(client->path, "%s/%s/%s.idc", client->store, client->user, client->calendar.name);
    } else if (!(client->share.flags & NMAP_SHARE_CALENDAR_DENIED)) {
        if ((newFlags & MSG_STATE_READ) && !(client->share.calendar.permissions & NMAP_SHARE_SEEN) && !(flags & MSG_STATE_READ)) {
            newFlags &= ~MSG_STATE_READ;
        }

        if ((newFlags & (MSG_STATE_DELETED | MSG_STATE_PURGED)) && !(client->share.calendar.permissions & NMAP_SHARE_DELETE)) {
            if ((newFlags & MSG_STATE_PURGED) && (!(flags & MSG_STATE_PURGED))) {
                newFlags &= ~MSG_STATE_PURGED;
            }

            if ((newFlags & MSG_STATE_DELETED) && (!(flags & MSG_STATE_DELETED))) {
                newFlags &= ~MSG_STATE_DELETED;
            }
        }

        sprintf(client->path, "%s/%s/%s.sdc", client->store, client->user, client->calendar.name);
    } else {
        return(ConnWrite(client->conn, MSG4240NOPERMISSION, sizeof(MSG4240NOPERMISSION) - 1));
    }

    if (WaitforAndLockCalendar(client->store, client->user, client->calendar.name) == 1) {
        index = fopen(client->path, "r+b");
    } else {
        return(ConnWrite(client->conn, MSG4120CALLOCKED, sizeof(MSG4120CALLOCKED) - 1));
    }

    if (index != NULL) {
        if ((newFlags & MSG_STATE_DELETED) && !(flags & MSG_STATE_DELETED)) {
            client->calendar.entries.count--;
        }

        if (!(newFlags & MSG_STATE_DELETED) && (flags & MSG_STATE_DELETED)) {
            client->calendar.entries.count++;
        }

        if (newFlags & MSG_STATE_PURGED) {
            client->calendar.changed = TRUE;
        }

        flags = newFlags;

        fseek(index, NMAP_CAL_IDX_HEADERSIZE + (id * sizeof(CalendarInfoStruct)) + offsetof(CalendarInfoStruct, State), SEEK_SET);
        fwrite(&flags, sizeof(unsigned long), 1, index);
        fclose(index);

        stat(client->path, &sb);
        client->calendar.indexTime = sb.st_mtime;

        client->calendar.entries.info[id].State = flags;
        UnlockCalendar(client->store, client->user, client->calendar.name);

        ccode = ConnWriteF(client->conn, "1000 %lu %s\r\n", flags, MSG1000FLAGSET);
    } else {
        UnlockCalendar(client->store, client->user, client->calendar.name);

        ccode = ConnWrite(client->conn, MSG4228CANTWRITECAL, sizeof(MSG4228CANTWRITECAL) - 1);
    }

    return(ccode);
}

int 
NmapCommandCssinfo(void *param)
{
    int ccode;
    unsigned long stopID;
    unsigned long startID;
    unsigned char *ptr;
    CalendarInfoStruct *cInfo;
    struct stat sb;
    NMAPClient *client = (NMAPClient *)param;

    if (client->states & NMAP_CLIENT_USER) {
        if (client->calendar.name[0] != '\0') {
            CalendarOOBNotifications(client, &sb);

            ptr = client->buffer + 7;
        } else {
            return(ConnWrite(client->conn, MSG4225NOCALENDAR, sizeof(MSG4225NOCALENDAR) - 1));
        }
    } else {
        return(ConnWrite(client->conn, MSG3241NOUSER, sizeof(MSG3241NOUSER) - 1));
    }

    if (*ptr == '\0') {
        startID = 0;
        stopID = client->calendar.entries.used;

        cInfo = client->calendar.entries.info;
    } else if ((*ptr++ == ' ') && (isdigit(*ptr))) {
        startID = atol(ptr);
        if (startID && (startID <= client->calendar.entries.used)) {
            startID--;
        } else {
            return(ConnWrite(client->conn, MSG4220NOMSG, sizeof(MSG4220NOMSG) - 1));
        }

        stopID = startID;

        cInfo = &(client->calendar.entries.info[startID]);
    } else {
        return(ConnWrite(client->conn, MSG3010BADARGC, sizeof(MSG3010BADARGC) - 1));
    }

    ccode = ConnWriteF(client->conn, "2002 %lu %lu\r\n", stopID - startID, client->calendar.id);

    for (; (ccode != -1) && (startID < stopID); startID++, cInfo++) {
        ccode = ConnWriteF(client->conn, "%lu %lu %lu %lu %lu %lu %lu %lu %lu %lu %lu %lu %lu %lu %lu %lu %lu %lu %lu %s %lu %s %lu %s\r\n", 
            startID + 1, 
            cInfo->State, 
            cInfo->UTCEnd, 
            cInfo->UTCStart, 
            cInfo->UTCDue, 
            (long unsigned int)cInfo->Type, 
            cInfo->Sequence, 
            cInfo->UID, 
            (long unsigned int)cInfo->Recurring, 
            cInfo->RecurrenceID, 
            cInfo->UIDPos, 
            cInfo->OrganizerPos, 
            cInfo->AttendeePos, 
            cInfo->AttendeeCount, 
            cInfo->AttendeeSize, 
            cInfo->Pos, 
            cInfo->Size, 
            cInfo->CalSize, 
            (long unsigned int)strlen(cInfo->ICalUID), 
            cInfo->ICalUID, 
            (long unsigned int)strlen(cInfo->Organizer), 
            cInfo->Organizer, 
            (long unsigned int)strlen(cInfo->Summary), 
            cInfo->Summary);
    }

    if (ccode != -1) {
        ccode = ConnWrite(client->conn, MSG1000OK, sizeof(MSG1000OK) - 1);
    }

    return(ccode);
}

int 
NmapCommandCspurg(void *param)
{
    int ccode;
    unsigned long delCount = 0;
    unsigned long startID;
    unsigned long stopID;
    unsigned long state;
    unsigned char *ptr;
    BOOL verbose = FALSE;
    BOOL markedOnly = FALSE;
    struct stat sb;
    FILE *index;
    CalendarInfoStruct *cInfo;
    NMAPClient *client = (NMAPClient *)param;

    if (client->states & NMAP_CLIENT_USER) {
        if (client->calendar.name[0] != '\0') {
            CalendarOOBNotifications(client, &sb);

            ptr = client->buffer + 6;
        } else {
            return(ConnWrite(client->conn, MSG4225NOCALENDAR, sizeof(MSG4225NOCALENDAR) - 1));
        }
    } else {
        return(ConnWrite(client->conn, MSG3241NOUSER, sizeof(MSG3241NOUSER) - 1));
    }

    if (client->flags & NMAP_CSTORE_READONLY) {
        return(ConnWrite(client->conn, MSG4243READONLY, sizeof(MSG4243READONLY) - 1));
    }

    if (*ptr == '\0') {
        startID = 0;
        stopID = client->calendar.entries.used;
    } else if  ((*ptr++ == ' ') && (isdigit(*ptr))) {
        startID = atol(ptr);
        if (startID && (startID <= client->calendar.entries.used)) {
            startID--;
        } else {
            return(ConnWrite(client->conn, MSG4220NOMSG, sizeof(MSG4220NOMSG) - 1));
        }

        stopID = startID;
    } else {
        return(ConnWrite(client->conn, MSG3010BADARGC, sizeof(MSG3010BADARGC) - 1));
    }

    if (toupper(client->buffer[5]) == 'V') {
        verbose = TRUE;
    } else if (toupper(client->buffer[5]) == 'M') {
        markedOnly = TRUE;
    }

    if (markedOnly == FALSE) {
        index = NULL;
    } else {
        startID = 0;
        stopID = client->calendar.entries.used;

        cInfo = client->calendar.entries.info;

        while (startID < stopID) {
            if (cInfo->State & MSG_STATE_PURGED) {
                delCount++;
            }

            startID++;
            cInfo++;
        }

        return(ConnWriteF(client->conn, "1000 %lu %s\r\n", delCount, MSG1000PURGED));
    }

    ccode = 0;
    cInfo = &(client->calendar.entries.info[startID]);
    while ((ccode != -1) && (startID < stopID)) {
        state = cInfo->State;
        if (state & (MSG_STATE_DELETED | MSG_STATE_PURGED)) {
            if (!(state & MSG_STATE_PURGED)) {
                state |= MSG_STATE_PURGED;
                state &= ~MSG_STATE_DELETED;

                if (index != NULL) {
                    fseek(index, NMAP_CAL_IDX_HEADERSIZE + (startID * sizeof(CalendarInfoStruct)) + offsetof(CalendarInfoStruct, State), SEEK_SET);
                } else {
                    WaitforAndLockCalendar(client->store, client->user, client->calendar.name);

                    if (!(client->share.flags & NMAP_SHARE_CALENDAR)) {
                        sprintf(client->path, "%s/%s/%s.idc", client->store, client->user, client->calendar.name);
                    } else if (!(client->share.flags & NMAP_SHARE_CALENDAR_DENIED) && (client->share.calendar.permissions & NMAP_SHARE_DELETE)) {
                        sprintf(client->path, "%s/%s/%s.sdc", client->store, client->user, client->calendar.name);
                    } else {
                        UnlockCalendar(client->store, client->user, client->calendar.name);
                        return(ConnWrite(client->conn, MSG4240NOPERMISSION, sizeof(MSG4240NOPERMISSION) - 1));
                    }

                    index = fopen(client->path, "r+b");
                    if (index != NULL) {
                        fseek(index, NMAP_CAL_IDX_HEADERSIZE + (startID * sizeof(CalendarInfoStruct)) + offsetof(CalendarInfoStruct, State), SEEK_SET);
                    } else {
                        UnlockCalendar(client->store, client->user, client->mailbox.name);
                        return(ConnWrite(client->conn, MSG4224CANTREADCAL, sizeof(MSG4224CANTREADCAL) - 1));
                    }
                }

                if (!(state & MSG_STATE_RECENT)) {
                    fwrite(&state, sizeof(unsigned long), 1, index);
                } else {
                    state &= ~MSG_STATE_RECENT;
                    fwrite(&state, sizeof(unsigned long), 1, index);
                    state |= MSG_STATE_RECENT;
                }

                cInfo->State = state;

                if (verbose) {
                    ccode = ConnWriteF(client->conn, "2002-%lu marked purged\r\n", startID + 1);
                }
            }

            delCount++;
        }

        startID++;
        cInfo++;
    }

    if (index != NULL) {
        fclose(index);

        stat(client->path, &sb);
        client->calendar.indexTime = sb.st_mtime;

        UnlockCalendar(client->store, client->user, client->calendar.name);
    }

    if (!(client->share.flags & NMAP_SHARE_CALENDAR)) {
        if (delCount > 0) {
            client->calendar.changed = TRUE;
        }

        StoreCalendar(client);

        client->calendar.entries.used = 0;
        client->calendar.entries.count = 0;
        client->calendar.size = 0;
        client->calendar.id = 0;
        client->calendar.nextUID = 0;

        ParseCalendar(client);
    }

    if (ccode != -1) {
        ccode = ConnWriteF(client->conn, "1000 %lu %s\r\n", delCount, MSG1000PURGED);
    }

    return(ccode);
}

int 
NmapCommandShareCal(void *param)
{
    int count;
    int ccode;
    unsigned int len;
    unsigned long used;
    unsigned long request = NMAP_SHARE_CALENDAR;
    unsigned char *ptr;
    unsigned char *ptr2;
    unsigned char buffer[CONN_BUFSIZE + 1];
    unsigned char resource[XPL_MAX_PATH + 1];
    BOOL result;
    struct stat sb;
    NMAPClient *client = (NMAPClient *)param;
    MDBValueStruct *vs = NULL;

    if (client->states & NMAP_CLIENT_USER) {
        if (client->calendar.name[0] != '\0') {
            CalendarOOBNotifications(client, &sb);
        }

        ptr = client->buffer + 9;
    } else {
        return(ConnWrite(client->conn, MSG3241NOUSER, sizeof(MSG3241NOUSER) - 1));
    }

    /* SHARE CAL[ <resource name>[ <grantee> <permissions>]] */
    if (*ptr == '\0') {
        result = MsgFindObject(client->user, client->dn, NULL, NULL, NULL);
    } else if ((*ptr++ == ' ') && (*ptr) && (!isspace(*ptr))) {
        request |= NMAP_SHARE_RESOURCE;

        ptr2 = strchr(ptr, ' ');
        if (ptr2 == NULL) {
            strcpy(resource, ptr);
        } else if ((ptr2[1]) && (!isspace(ptr2[1]))) {
            *ptr2++ = '\0';
            strcpy(resource, ptr);

            if (*ptr2 != '\0') {
                return(HandleShareCreate(client, request, resource, ptr2));
            }
        } else {
            return(ConnWrite(client->conn, MSG3010BADARGC, sizeof(MSG3010BADARGC) - 1));
        }

        result = MsgFindObject(client->user, client->dn, NULL, NULL, NULL);
    } else {
        return(ConnWrite(client->conn, MSG3010BADARGC, sizeof(MSG3010BADARGC) - 1));
    }

    if (result) {
        vs = MDBCreateValueStruct(NMAP.handle.directory, NULL);
    } else {
        return(ConnWrite(client->conn, MSG4224NOUSER, sizeof(MSG4224NOUSER) - 1));
    }

    if (vs) {
        strcpy(buffer, "CA");

        if (request & NMAP_SHARE_RESOURCE) {
            sprintf(buffer + strlen(buffer), "%s\r", resource);
        }

        count = strlen(buffer);

        ccode = ConnWrite(client->conn, "2002-Coming up\r\n", 16);

        MDBRead(client->dn, MSGSRV_A_OWNED_SHARES, vs);
        for (used = 0; (ccode != -1) && (used < vs->Used); used++) {
            if (XplStrNCaseCmp(buffer, vs->Value[used], count) != 0) {
                continue;
            }

            len = strlen(vs->Value[used]);
            vs->Value[used][len - 11] = '\0';

            ptr = strchr(vs->Value[used], '\r');
            if (ptr != NULL) {
                *(ptr++) = '\0';

                if (!(request & NMAP_SHARE_RESOURCE)) {
                    ccode = ConnWriteF(client->conn, "2002-%s %s %lu\r\n", vs->Value[used] + 2, ptr, atol(&(vs->Value[used][len - 10])));
                } else {
                    ccode = ConnWriteF(client->conn, "2002-%s %lu\r\n", ptr, atol(&(vs->Value[used][len - 10])));
                }
            }
        }

        if (ccode != -1) {
            ccode = ConnWrite(client->conn, MSG1000OK, sizeof(MSG1000OK) - 1);
        }

        MDBDestroyValueStruct(vs);
    } else {
        ccode = ConnWrite(client->conn, MSG5001NOMEMORY, sizeof(MSG5001NOMEMORY) - 1);
    }

    return(ccode);
}
