/****************************************************************************
 *
 * Copyright (c) 2001-2002 Novell, Inc.
 * All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of version 2.1 of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, contact Novell, Inc.
 *
 * To contact Novell about this file by physical or electronic mail,
 * you may find current contact information at www.novell.com
 *
 ****************************************************************************/

#include <config.h>
#include <xpl.h>
#include <openssl/md5.h>
#include <openssl/ssl.h>
#include <openssl/err.h>

#include <msgapi.h>
#include <nmap.h>

#include "modwebp.h"

unsigned char	*ModWebSVersion = "$Revision: 1.6 $";

#if !defined(A_INTERNET_EMAIL_ADDRESS)
#define	A_INTERNET_EMAIL_ADDRESS	"Internet EMail Address"
#endif

#if defined(MODWEB_DEBUG_BUILD)
int		SessionScreen;
#define	SessionDebugInit()	SessionScreen=XplCreateScreen("MW Session Debug Screen", 0)
#else
#define	SessionDebugInit()
#endif

unsigned long		DefaultConnectionTimeout	=	360;
unsigned long		DefaultLogoID					=	0;
unsigned long		DefaultLanguage				=	4;
unsigned long		DefaultTemplate				=	0;
unsigned char		DefaultDateFormatShort[128]= "%m/%d/%y";
unsigned char		DefaultDateFormatLong[128]	= "%A, %B %d, %Y";
unsigned char		DefaultTimeFormat[128]		= "%I:%M %p";
long					DefaultTimezoneOffset		=	0;
unsigned long		DefaultTimezoneID				=	0;
unsigned char		*DefaultMonthShort[12]		= { "Jan", "Feb", "Mar", "Apr", "May", "Jun", "Jul", "Aug", "Sep", "Oct", "Nov", "Dec"};
unsigned char		*DefaultMonthLong[12]		= { "January", "February", "March", "April", "May", "June", "July", "August", "September", "Octobert", "November", "December"};
unsigned char		*DefaultWDayShort[7]			= { "Sun", "Mon", "Tue", "Wed", "Thu", "Fri", "Sat"};
unsigned char		*DefaultWDayLong[7]			= { "Sunday", "Monday", "Tuesday", "Wednesday", "Thursday", "Friday", "Saturday"};
unsigned char		*DefaultAMPM[2]				= { "AM", "PM"};
unsigned long		SessionMonitorInterval		=	5*60;
time_t				ModwebTimeStamp				= 0;

SessionStruct		*SessionDB[MAX_SESSION_ENTRIES];
XplSemaphore		SessionMutex[MAX_SESSION_ENTRIES];
unsigned long		SessionStack[MAX_SESSION_ENTRIES];
long					SessionStackPtr=MAX_SESSION_ENTRIES-1;

TSessionStruct		*TSessionDB[MAX_TSESSION_ENTRIES];
XplSemaphore		TSessionMutex[MAX_TSESSION_ENTRIES];
unsigned long		TSessionStack[MAX_TSESSION_ENTRIES];
long					TSessionStackPtr=MAX_TSESSION_ENTRIES-1;

ModWebStatistics	ModWebStats;

BOOL
MWConnectUserToNMAPServer(SessionStruct *Session)
{
	struct sockaddr_in	soc_address;
	unsigned char			Answer[BUFSIZE+1];
	int						ReplyInt;

	SessionDebug("Connection session %x to NMAP\n", (unsigned int)Session);

	/* Connect to NMAP server */
	soc_address.sin_family=AF_INET;
	soc_address.sin_port=htons(NMAP_PORT);
	soc_address.sin_addr=Session->NMAPAddress;
	Session->NMAPs=IPsocket(PF_INET, SOCK_STREAM, IPPROTO_TCP);
	

	ReplyInt=IPconnect(Session->NMAPs, (struct sockaddr *)&soc_address, sizeof(soc_address));
	if (ReplyInt) {
		IPclose(Session->NMAPs);
		Session->NMAPs=-1;
		LoggerEvent(LogHandle, LOGGER_SUBSYSTEM_GENERAL, LOGGER_EVENT_NMAP_UNAVAILABLE, LOG_ERROR, 0, NULL, NULL, XplHostToLittle(soc_address.sin_addr.s_addr), ReplyInt, NULL, 0);
		MWDebug("a\n");
		
		return(FALSE);
	}

	/* Set TCP non blocking io */
	ReplyInt=1;
	setsockopt(Session->NMAPs, IPPROTO_TCP, 1, (unsigned char *)&ReplyInt, sizeof(ReplyInt));

	ReplyInt=MWGetNMAPAnswer(Session, Answer, sizeof(Answer),TRUE);

	switch (ReplyInt) {
		case NMAP_READY: {
			break;
		}

		case 4242: {
			unsigned char	*ptr, *salt;
			MD5_CTX			mdContext;
			unsigned char	digest[16];
			unsigned char	HexDigest[33];
			int				i;

			ptr=strchr(Answer, '<');
			if (ptr) {
				ptr++;
				salt=ptr;
				if ((ptr=strchr(ptr, '>'))!=NULL) {
					*ptr='\0';
				}

				MD5_Init(&mdContext);
				MD5_Update(&mdContext, salt, strlen(salt));
				MD5_Update(&mdContext, NMAPHash, NMAP_HASH_SIZE);
				MD5_Final(digest, &mdContext);
				for (i=0; i<16; i++) {
					snprintf(HexDigest+(i*2), sizeof(HexDigest)-(i*2), "%02x",digest[i]);
				}
				ReplyInt=snprintf(Answer, sizeof(Answer), "AUTH %s\r\n", HexDigest);

				MWSendNMAPServer(Session, Answer, ReplyInt);
				if (MWGetNMAPAnswer(Session, Answer, sizeof(Answer), TRUE)==1000) {
					break;
				}
			}
			/* Fall-through */
		}

		default:{
			IPclose(Session->NMAPs);
			Session->NMAPs=-1;
			LoggerEvent(LogHandle, LOGGER_SUBSYSTEM_GENERAL, LOGGER_EVENT_NMAP_UNAVAILABLE, LOG_ERROR, 0, NULL, NULL, XplHostToLittle(soc_address.sin_addr.s_addr), ReplyInt, NULL, 0);
			MWDebug("b\n");

			return(FALSE);
		}
	}

	if (Session->User) {
		ReplyInt=snprintf(Answer, sizeof(Answer), "USER %s\r\n", Session->User);
		MWSendNMAPServer(Session, Answer, ReplyInt);
		ReplyInt=MWGetNMAPAnswer(Session, Answer, sizeof(Answer), TRUE);
		if (ReplyInt!=1000) {
			IPclose(Session->NMAPs);
			Session->NMAPs=-1;
			LoggerEvent(LogHandle, LOGGER_SUBSYSTEM_GENERAL, LOGGER_EVENT_NMAP_UNAVAILABLE, LOG_ERROR, 0, NULL, NULL, XplHostToLittle(soc_address.sin_addr.s_addr), ReplyInt, NULL, 0);
			MWDebug("c\n");

			return(FALSE);
		}
		ReplyInt=snprintf(Answer, sizeof(Answer), "FLAG %lu\r\n", (unsigned long)(NMAP_SHOWDELETED | NMAP_OOBMESSAGES | NMAP_KEEPUSER));
		MWSendNMAPServer(Session, Answer, ReplyInt);
		ReplyInt=MWGetNMAPAnswer(Session, Answer, sizeof(Answer), TRUE);
		if (ReplyInt!=1000) {
			IPclose(Session->NMAPs);
			Session->NMAPs=-1;
			LoggerEvent(LogHandle, LOGGER_SUBSYSTEM_GENERAL, LOGGER_EVENT_NMAP_UNAVAILABLE, LOG_ERROR, 0, NULL, NULL, XplHostToLittle(soc_address.sin_addr.s_addr), ReplyInt, NULL, 0);
			MWDebug("d\n");

			return(FALSE);
		}
	}

	MWDebug("e\n");
	
	return(TRUE);
}


void SessionMonitor(void)
{
	int 				i;
	unsigned long	counter;
	SessionStruct	*session;
	TSessionStruct	*tSession;

	XplRenameThread(XplGetThreadID(), "ModWeb SessionExpire Monitor");

	SessionDebug("SessionMonitorInterval:%d\n", (unsigned int)SessionMonitorInterval);

	while (!Exiting) {
		SessionDebug("SessionMonitor: Entering lurk mode\n");

		/*
			Every five minutes we check for and update settings, and set the time.
			That way we can just read ModwebTimeStamp anywhere we would normally
			call time(NULL);
		*/

		/*
			The session monitor needs to run every 5 minutes (SessionMonitorInterval).  In order to
			make sure we bail when Exiting is set we will sleep for no more than 15 seconds at a
			time.

			We also need to update ModwebTimeStamp every minute.  ModwebTimeStamp is used to reduce
			our calls to time(NULL); to a minumum.
		*/

		ModwebTimeStamp = time(NULL);
		counter = 0;

		do {
			i = 0;

			do {
				XplDelay(15 * 1000);
				i++;
			} while (i < 4 && !Exiting);

			ModwebTimeStamp += 60;
			counter += 60;
		} while (counter < SessionMonitorInterval && !Exiting);

		SessionDebug("SessionMonitor(): Now checking...\n");

		if (!Exiting) {
			for (counter = 0; counter < MAX_SESSION_ENTRIES; counter++) {
				if (SessionDB[counter] == NULL) {
					continue;
				}

				XplExamineLocalSemaphore(SessionMutex[counter], i);
				/* Do not try to use a connection that's in use */

				if (i > 0) {
					XplWaitOnLocalSemaphore(SessionMutex[counter]);

					if ((SessionDB[counter]->Timestamp + SessionDB[counter]->ConnectionTimeout) < ModwebTimeStamp) {
						SessionDebug("SessionMonitor: session %x expired (Time:%x, age:%x)\n", (unsigned int)SessionDB[counter], (unsigned int)ModwebTimeStamp, (unsigned int)SessionDB[counter]->Timestamp+(unsigned int)SessionDB[counter]->ConnectionTimeout);

						session = SessionDB[counter];
						SessionDB[counter] = NULL;

						/* Note: We still hold the mutex lock; MWDestroySession will release it for us */
						MWDestroySession(session);
					} else {
						XplSignalLocalSemaphore(SessionMutex[counter]);
						SessionDebug("SessionMonitor: session %x lingering but not too old\n", (unsigned int)SessionDB[counter]);
					}
				}
			}

			SessionDebug("TSessionMonitor(): Now checking timeout storage sessions...\n");

			for (counter = 0; counter < MAX_TSESSION_ENTRIES; counter++) {
				if (TSessionDB[counter] == NULL) {
					continue;
				}

				XplExamineLocalSemaphore(TSessionMutex[counter], i);
				/* Do not try to use a connection that's in use */

				if (i > 0) {
					XplWaitOnLocalSemaphore(TSessionMutex[counter]);
				
					if ((TSessionDB[counter]->Timestamp + 300) < ModwebTimeStamp) {
						SessionDebug("TSessionMonitor: tSession %x expired (Time:%x, age:%x)\n", (unsigned int)TSessionDB[counter], (unsigned int)ModwebTimeStamp, (unsigned int)TSessionDB[counter]->Timestamp + 300);

						tSession = TSessionDB[counter];
						TSessionDB[counter] = NULL;

						/* Note: We still hold the mutex lock; DestroyTSession will release it for us */
						DestroyTSession(tSession);
					} else {
						XplSignalLocalSemaphore(TSessionMutex[counter]);
						SessionDebug("SessionMonitor: tSession %x lingering but not too old\n", (unsigned int)TSessionDB[counter]);
					}
				}
			}
		}
	}

	SessionDebug("session monitor exiting\n");

	XplConsolePrintf("\rMODWEBD: Session expiration monitor done.\r\n");

	XplSafeDecrement(ModWebServerThreads);

	return;
}

BOOL
MWDestroySession(SessionStruct *Session)
{
	unsigned long	i;

	SessionDebug("MWDestroySession: %x\n", (unsigned int)Session);

	if (!Session) {
		return(FALSE);
	}

	if (Session->User) {
		MemFree(Session->User);
	}

	if (Session->RealUser) {
		MemFree(Session->RealUser);
	}

	if (Session->UserDN) {
		MemFree(Session->UserDN);
	}

	if (Session->Charset) {
		MemFree(Session->Charset);
	}

	if (Session->OfficialDomain) {
		MemFree(Session->OfficialDomain);
	}

	if (Session->EMailAddress) {
		MemFree(Session->EMailAddress);
	}

	if (Session->DisplayName) {
		MemFree(Session->DisplayName);
	}

	if (Session->DateFormatShort) {
		MemFree(Session->DateFormatShort);
	}

	if (Session->DateFormatLong) {
		MemFree(Session->DateFormatLong);
	}

	if (Session->TimeFormat) {
		MemFree(Session->TimeFormat);
	}

	if (Session->Title) {
		MemFree(Session->Title);
	}

	if (Session->FolderList) {
		MDBDestroyValueStruct(Session->FolderList);
	}

	if (Session->FolderUnreadCount) {
		MemFree(Session->FolderUnreadCount);
	}

	if (Session->FolderDisplayNames) {
		MDBDestroyValueStruct(Session->FolderDisplayNames);
	}

	if (Session->SentFolder) {
		MemFree(Session->SentFolder);
	}

	if (Session->HTMLCodec) {
		StreamStruct	*EncodingCodec;
		StreamStruct	*NetworkOutCodec;

		EncodingCodec=Session->HTMLCodec->Next;
		NetworkOutCodec=EncodingCodec->Next;

		MWReleaseStream(EncodingCodec);
		MWReleaseStream(NetworkOutCodec);
		MWReleaseStream(Session->HTMLCodec);
	}

	if (Session->NMAPs!=-1) {
		/* Send a CSPURG to purge out any deleted calender items */
		unsigned char buffer[BUFSIZE];
		MWSendNMAPServer(Session, "CSPURG\r\n",9);
		MWGetNMAPAnswer(Session, buffer, BUFSIZE, FALSE);
		MWSendNMAPServer(Session, "QUIT\r\n", 6);
		IPclose(Session->NMAPs);
		  Session->NMAPs=-1;
	}

	if (Session->ModuleData) {
		for (i=0; i<TModuleCount; i++) {
			TModules[i].DestroySession(Session, Session->ModuleData[i]);
		}

		MemFree(Session->ModuleData);
	}

	if (Session->UserValues) {
	    UserValue *node;
	    UserValue *nextNode;

	    node = Session->UserValues;
   
	    do {
		nextNode = node->next;
		MemFree(node->value);
		MemFree(node);
		node = nextNode;
	    } while (node);
	}

	XplWaitOnLocalSemaphore(SessionDBSemaphore);

	SessionDB[Session->SessionID]=NULL;
	SessionStack[++SessionStackPtr]=Session->SessionID;

	SessionDebug("Releasing session %x mutex\n", (unsigned int)Session);
	XplSignalLocalSemaphore(SessionMutex[Session->SessionID]);
	XplSignalLocalSemaphore(SessionDBSemaphore);

	MemFree(Session);

	XplSafeIncrement(ModWebStats.Clients.Serviced);
	XplSafeDecrement(ModWebStats.Clients.Incoming);
	return(TRUE);
}


BOOL
DestroyAllSessions(void)
{
	unsigned long	i;

	SessionDebug("DestroyAllSessions called\n");

	for (i=0; i<MAX_SESSION_ENTRIES; i++) {
		if (SessionDB[i]!=NULL) {
			SessionDebug("Accquiring session %x mutex\n", (unsigned int)SessionDB[i]);
			XplWaitOnLocalSemaphore(SessionMutex[i]);
			MWDestroySession(SessionDB[i]);
		}
	}

	for (i=0; i<MAX_TSESSION_ENTRIES; i++) {
		if (TSessionDB[i]!=NULL) {
			SessionDebug("Accquiring tsession %x mutex\n", (unsigned int)TSessionDB[i]);
			XplWaitOnLocalSemaphore(TSessionMutex[i]);
			DestroyTSession(TSessionDB[i]);
		}
	}

	return(TRUE);
}

BOOL
CreateSession(ConnectionStruct *Client, unsigned char *Username, unsigned char *Password, BOOL *Disabled)
{
	SessionStruct			*Session;
	unsigned char			UserDN[MAXEMAILNAMESIZE*2+1];
	unsigned char			*ptr;
	MDBValueStruct			*Config;
	int						i;
	StreamStruct			*NetworkOutCodec;
	StreamStruct			*EncodingCodec;

	Session=MemMalloc(sizeof(SessionStruct));

	if (!Session) {
		return(FALSE);
	}

	SessionDebug("CreateSession: User:%s Password:%s\n", (char *)Username, (char *)Password ? (char *)Password : (char *)"Proxy Access");

	memset(Session, 0, sizeof(SessionStruct));
	Config = MDBCreateValueStruct(ModWebDirectoryHandle, NULL);

	/* note: Needs to change if NMAP ever starts using SSL */
	Session->Read = SessionFuncTbl->read;
	Session->Write = SessionFuncTbl->write;

	Client->Session=Session;
	Session->NMAPs=-1;
	Session->CookieID = Client->CookieID;

	/* Get all codecs required for our HTML translator */
	Session->HTMLCodec=MWGetStream(NULL, NULL, FALSE);
	EncodingCodec=MWGetStream(Session->HTMLCodec, "text/plain", FALSE);
	NetworkOutCodec=MWGetStream(EncodingCodec, NULL, FALSE);

	if (Session->HTMLCodec && EncodingCodec && NetworkOutCodec) {
		Session->HTMLCodec->Codec=MWStreamFromMemory;
		NetworkOutCodec->StreamData=(void *)Client;
		NetworkOutCodec->Codec=MWStreamToMWClient;
	} else {
		/*
			Didn't get enough memory for all codecs, clean up and reset; the MWHTMLSendClient function
			knows how to handle a NULL Session->HTMLCodec
		*/
		if (Session->HTMLCodec) {
			MWReleaseStream(Session->HTMLCodec);
			Session->HTMLCodec=NULL;
		}
		if (EncodingCodec) {
			MWReleaseStream(EncodingCodec);
		}
		if (NetworkOutCodec) {
			MWReleaseStream(NetworkOutCodec);
		}
	}



	snprintf(UserDN, sizeof(UserDN), "MW:%s\n", Username);
	XplRenameThread(XplGetThreadID(), UserDN);

	if (!MsgFindObject(Username, UserDN, NULL, &Session->NMAPAddress, Config)) {
		LoggerEvent(LogHandle, LOGGER_SUBSYSTEM_AUTH, LOGGER_EVENT_UNKNOWN_USER, LOG_NOTICE, 0, Username, (Password) ? (char *)Password : "Proxy Access", XplHostToLittle(Client->cs.sin_addr.s_addr), 0, NULL, 0);

		SessionDebug("CreateSession: MsgFindObject(%s) failed\n", Username);
		i=sizeof(Client->cs);
		IPgetpeername(Client->s, (struct sockaddr *)&(Client->cs), &i);
		MDBDestroyValueStruct(Config);
			if (Session->HTMLCodec) {
				MWReleaseStream(Session->HTMLCodec);
			}
			if (EncodingCodec) {
				MWReleaseStream(EncodingCodec);
			}
			if (NetworkOutCodec) {
				MWReleaseStream(NetworkOutCodec);
			}

		MemFree(Session);
		Client->Session=NULL;
		return(FALSE);
	}

	/*
		If Password is NULL we ignore security checks.
	*/
	if (Password) {
		if (!MDBVerifyPassword(UserDN, Password, Config)) {
			LoggerEvent(LogHandle, LOGGER_SUBSYSTEM_AUTH, LOGGER_EVENT_WRONG_PASSWORD, LOG_NOTICE, 0, Username, Password, XplHostToLittle(Client->cs.sin_addr.s_addr), 0, NULL, 0);

			SessionDebug("CreateSession: MDBVerifyPassword(%s, %s) failed\n", UserDN, Password);
			i=sizeof(Client->cs);
			IPgetpeername(Client->s, (struct sockaddr *)&(Client->cs), &i);
			XplSafeIncrement(ModWebStats.WrongPassword);
			MDBDestroyValueStruct(Config);
			if (Session->HTMLCodec) {
				MWReleaseStream(Session->HTMLCodec);
			}
			if (EncodingCodec) {
				MWReleaseStream(EncodingCodec);
			}
			if (NetworkOutCodec) {
				MWReleaseStream(NetworkOutCodec);
			}
			MemFree(Session);
			Client->Session=NULL;
			return(FALSE);
		}

		if (!MsgGetUserFeature(UserDN, FEATURE_MODWEB, NULL, NULL)) {
			LoggerEvent(LogHandle, LOGGER_SUBSYSTEM_AUTH, LOGGER_EVENT_DISABLED_FEATURE, LOG_NOTICE, 0, Username, NULL, XplHostToLittle(Client->cs.sin_addr.s_addr), 0, NULL, 0);

			SessionDebug("CreateSession: FEATURE_MODWEB for %s disabled\n", Username);
			if (Disabled) {
				*Disabled=TRUE;
			}
			MDBDestroyValueStruct(Config);
			if (Session->HTMLCodec) {
				MWReleaseStream(Session->HTMLCodec);
			}
			if (EncodingCodec) {
				MWReleaseStream(EncodingCodec);
			}
			if (NetworkOutCodec) {
				MWReleaseStream(NetworkOutCodec);
			}
			MemFree(Session);
			Client->Session=NULL;
			return(FALSE);
		}
	}

	Session->User=MemStrdup(Config->Value[0]);
	Session->UserDN=MemStrdup(UserDN);

	/* We now have the user's DN, real username and NMAP store address; let's load the other defaults */
	Session->SessionUID=time(NULL);

	/* Connection Timeout */
	MDBFreeValues(Config);
	if (MsgGetUserFeature(UserDN, FEATURE_MODWEB, MSGSRV_A_TIMEOUT, Config) > 0) {
		Session->ConnectionTimeout=atol(Config->Value[0])*60;
	}
	if (Session->ConnectionTimeout<1) {
		Session->ConnectionTimeout=DefaultConnectionTimeout;
	} else if (Session->ConnectionTimeout>40*60) {
		Session->ConnectionTimeout=40*60;
		MDBFreeValues(Config);
		MDBAddValue("40", Config);
		MDBWrite(UserDN, MSGSRV_A_TIMEOUT, Config);
	}
	SessionDebug("CreateSession: %s ConnectionTimeout:%d\n", Username, (int)Session->ConnectionTimeout);

	/* Language */
	MDBFreeValues(Config);
	if (MsgGetUserFeature(UserDN, FEATURE_MODWEB, MSGSRV_A_LANGUAGE, Config) > 0) {
		Session->Language=atol(Config->Value[0]);
	} else {
		Session->Language=DefaultLanguage;

		/* Determine language; check Client->Language */
		FindLanguage(Client->Language, Session->Language);
	}
	SessionDebug("CreateSession: %s Language:%d\n", Username, (unsigned int)Session->Language);

	/* Template */
	MDBFreeValues(Config);
	i=-1;
	if (MDBReadDN(UserDN, MSGSRV_A_TEMPLATE, Config) > 0) {
		ptr=strrchr(Config->Value[0], '\\');
		if (ptr) {
			ptr++;
		} else {
			ptr=Config->Value[0];
		}
		i=MWFindTemplate(ptr);
		if (i!=-1) {
			MWSetSessionTemplate(i, Session->Language, Session);
		}
	}
	MDBFreeValues(Config);
	if (i==-1) {
		if (MsgGetUserFeature(UserDN, FEATURE_MODWEB, MSGSRV_A_DEFAULT_TEMPLATE, Config) >0 ) {
			ptr=strrchr(Config->Value[0], '\\');
			if (ptr) {
				ptr++;
			} else {
				ptr=Config->Value[0];
			}
			i=MWFindTemplate(ptr);
			if (i!=-1) {
				MWSetSessionTemplate(i, Session->Language, Session);
			} else {
				MWSetSessionTemplate(DefaultTemplate, Session->Language, Session);
			}
		} else {
			MWSetSessionTemplate(DefaultTemplate, Session->Language, Session);
		}
	}

	/* Timezone */
	MDBFreeValues(Config);
	if (MDBRead(UserDN, MSGSRV_A_TIMEZONE, Config) > 0) {
		Session->TimezoneID=atol(Config->Value[0]);
	} else {
		if (MsgGetParentAttribute(UserDN, MSGSRV_A_TIMEZONE, Config) > 0) {
			Session->TimezoneID=atol(Config->Value[0]);
		} else {
			Session->TimezoneID=DefaultTimezoneID;
		}
	}
	Session->TimezoneOffset=MsgGetUTCOffsetByDate(Session->TimezoneID, 0, 0, 0, 0);
	SessionDebug("CreateSession: %s TimezoneOffset:%d\n", Username, (unsigned int)Session->TimezoneOffset);
	
	/* Time & Date formats */
	MDBFreeValues(Config);
	if (MsgGetUserFeature(UserDN, FEATURE_MODWEB, MSGSRV_A_LOCALE, Config) > 0) {
		for (i=0; i<Config->Used; i++) {
			if (MWQuickNCmp(Config->Value[i], "DS:", 3)) {
				Session->DateFormatShort=MemStrdup(Config->Value[i]+3);
			} else if (MWQuickNCmp(Config->Value[i], "DL:", 3)) {
				Session->DateFormatLong=MemStrdup(Config->Value[i]+3);
			} else if (MWQuickNCmp(Config->Value[i], "T:", 2)) {
				Session->TimeFormat=MemStrdup(Config->Value[i]+2);
			} else if (MWQuickNCmp(Config->Value[i], "WDS:", 4)) {
				Session->StartWeekday=atol(Config->Value[i]+4);
			}
		}
	}
	if (!Session->DateFormatShort) {
		Session->DateFormatShort=MemStrdup(DefaultDateFormatShort);
	}
	if (!Session->DateFormatLong) {
		Session->DateFormatLong=MemStrdup(DefaultDateFormatLong);
	}
	if (!Session->TimeFormat) {
		Session->TimeFormat=MemStrdup(DefaultTimeFormat);
	}
	Session->DateFormat.monthShort=DefaultMonthShort;
	Session->DateFormat.monthLong=DefaultMonthLong;
	Session->DateFormat.wDayShort=DefaultWDayShort;
	Session->DateFormat.wDayLong=DefaultWDayLong;
	Session->DateFormat.AmPm=DefaultAMPM;
	Session->DateFormat.timezoneOffset=Session->TimezoneOffset;
	Session->DateFormat.wDayStart=Session->StartWeekday;

	SessionDebug("CreateSession: %s TimezoneOffset:%d\n", Username, (unsigned int)Session->TimezoneOffset);
	

	/* Title */
	MDBFreeValues(Config);
	if (MsgGetUserFeature(UserDN, FEATURE_MODWEB, MSGSRV_A_TITLE, Config)) {
		Session->Title=MemStrdup(Config->Value[0]);
	}
														 
	/* Charset */
	MDBFreeValues(Config);
	if (MsgGetUserFeature(UserDN, FEATURE_MODWEB, MSGSRV_A_DEFAULT_CHARSET, Config)) {
		Session->Charset=MemStrdup(Config->Value[0]);
	} else {
		Session->Charset=MemStrdup("UTF-8");
	}
														 
	/* Official Domain */
	MDBFreeValues(Config);
	if (MsgGetUserFeature(Session->UserDN, FEATURE_MODWEB, MSGSRV_A_OFFICIAL_NAME, Config)) {
		Session->OfficialDomain=MemStrdup(Config->Value[0]);
	} else {
		Session->OfficialDomain=MemStrdup(OfficialDomain);
	}

	/* Read the email address */
	MDBFreeValues(Config);
	Session->EMailAddress = MsgGetUserEmailAddress(Session->UserDN, Config, NULL, 0);

	/* Read the Display Name */
	MDBFreeValues(Config);
	Session->DisplayName = MsgGetUserDisplayName(Session->UserDN, Config);

	/* UI Preferences */
	MDBFreeValues(Config);
	if (MsgGetUserFeature(UserDN, FEATURE_MODWEB, MSGSRV_A_PREFERENCES, Config) > 0) {
		for (i=0; i<Config->Used; i++) {
			if (MWQuickNCmp(Config->Value[i], "ModWeb:MsgPerPage=", 18)) {
				Session->MessagesPerPage=atol(Config->Value[i]+18);
			} else if (MWQuickNCmp(Config->Value[i], "ModWeb:Logo=", 12)) {
				Session->LogoID=atol(Config->Value[i]+12);
			}
		}
	}

	if (Session->LogoID==0) {
		Session->LogoID=DefaultLogoID;
	}

	if (Session->MessagesPerPage==0) {
		Session->MessagesPerPage=10;
	}

	MDBFreeValues(Config);
	if (MsgGetUserFeature(UserDN, FEATURE_MODWEB, MSGSRV_A_COLOR, Config) > 0) {
		for(i = 0; i < Config->Used; i++) {
			if (MWQuickNCmp(Config->Value[i], "Page: ", 6)) {
				sscanf(Config->Value[i], "Page: %s %s", Session->Colors[COLOR_PAGE_FG], Session->Colors[COLOR_PAGE_BG]);
			} else if (MWQuickNCmp(Config->Value[i], "Border: ", 8)) {
				sscanf(Config->Value[i], "Border: %s %s", Session->Colors[COLOR_BORDER_FG], Session->Colors[COLOR_BORDER_BG]);
			} else if (MWQuickNCmp(Config->Value[i], "Section: ", 9)) {
				sscanf(Config->Value[i], "Section: %s %s", Session->Colors[COLOR_SECTION_FG], Session->Colors[COLOR_SECTION_BG]);
			} else if (MWQuickNCmp(Config->Value[i], "Fieldname: ", 11)) {
				sscanf(Config->Value[i], "Fieldname: %s %s", Session->Colors[COLOR_FIELDNAME_FG], Session->Colors[COLOR_FIELDNAME_BG]);
			} else if (MWQuickNCmp(Config->Value[i], "Fieldbody: ", 11)) {
				sscanf(Config->Value[i], "Fieldbody: %s %s", Session->Colors[COLOR_FIELDBODY_FG], Session->Colors[COLOR_FIELDBODY_BG]);
			}
		}
	}

	if (Session->Colors[COLOR_PAGE_FG][0]=='\0') {
		/* Use the colors from the template, since the user' colors are not set */
		snprintf(Session->Colors[COLOR_PAGE_FG], sizeof(Session->Colors[COLOR_PAGE_FG]), "%06x", Templates[Session->TemplateID]->Colors[0][0]);
		snprintf(Session->Colors[COLOR_PAGE_BG], sizeof(Session->Colors[COLOR_PAGE_BG]), "%06x", Templates[Session->TemplateID]->Colors[0][1]);
		snprintf(Session->Colors[COLOR_BORDER_FG], sizeof(Session->Colors[COLOR_BORDER_FG]), "%06x", Templates[Session->TemplateID]->Colors[1][0]);
		snprintf(Session->Colors[COLOR_BORDER_BG], sizeof(Session->Colors[COLOR_BORDER_BG]), "%06x", Templates[Session->TemplateID]->Colors[1][1]);
		snprintf(Session->Colors[COLOR_SECTION_FG], sizeof(Session->Colors[COLOR_SECTION_FG]), "%06x", Templates[Session->TemplateID]->Colors[2][0]);
		snprintf(Session->Colors[COLOR_SECTION_BG], sizeof(Session->Colors[COLOR_SECTION_BG]), "%06x", Templates[Session->TemplateID]->Colors[2][1]);
		snprintf(Session->Colors[COLOR_FIELDNAME_FG], sizeof(Session->Colors[COLOR_FIELDNAME_FG]), "%06x", Templates[Session->TemplateID]->Colors[3][0]);
		snprintf(Session->Colors[COLOR_FIELDNAME_BG], sizeof(Session->Colors[COLOR_FIELDNAME_BG]), "%06x", Templates[Session->TemplateID]->Colors[3][1]);
		snprintf(Session->Colors[COLOR_FIELDBODY_FG], sizeof(Session->Colors[COLOR_FIELDBODY_FG]), "%06x", Templates[Session->TemplateID]->Colors[4][0]);
		snprintf(Session->Colors[COLOR_FIELDBODY_BG], sizeof(Session->Colors[COLOR_FIELDBODY_BG]), "%06x", Templates[Session->TemplateID]->Colors[4][1]);

	}
	MDBFreeValues(Config);

	if (MsgGetUserFeature(UserDN, FEATURE_MODWEB, MSGSRV_A_PREFERENCES, Config)) {
		for (i = 0; i < Config->Used; i++) {
			if (strlen(Config->Value[i]) > 23 && MWQuickNCmp(Config->Value[i], "Webmail:SentFolderName:", 23)) {
				Session->SentFolder = MemStrdup(Config->Value[i] + 23);
				break;
			}
		}
	}
	MDBDestroyValueStruct(Config);

	/* Folder Structure */
	Session->FolderList = MDBCreateValueStruct(ModWebDirectoryHandle, NULL);
	Session->FolderDisplayNames = MDBCreateValueStruct(ModWebDirectoryHandle, NULL);

	Session->Timestamp = ModwebTimeStamp;

	/*
		Keep track of the elapsed time since the last time we talked to NMAP.
	*/
	Session->NMAPTimestamp = ModwebTimeStamp;
	
	Session->ModuleData=MemMalloc(sizeof(void *)*TModuleCount);
	if (Session->ModuleData==NULL && TModuleCount>0) {
		MWDestroySession(Session);
		Client->Session=NULL;

		LoggerEvent(LogHandle, LOGGER_SUBSYSTEM_GENERAL, LOGGER_EVENT_OUT_OF_MEMORY, LOG_CRITICAL, 0, __FILE__, NULL, (sizeof(void *) * TModuleCount), __LINE__, NULL, 0);
		return(FALSE);
	} else {
		memset(Session->ModuleData, 0, sizeof(void *)*TModuleCount);
	}
	XplSafeIncrement(ModWebStats.Clients.Incoming);

	/* We need to insert the session into our array */
	SessionDebug("CreateSession: Inserting session into global table\n");
	XplWaitOnLocalSemaphore(SessionDBSemaphore);
	if (SessionStackPtr>=0) {
		Session->SessionID=SessionStack[SessionStackPtr--];
		SessionDB[Session->SessionID]=Session;
		SessionDebug("Accquiring session %x mutex\n", (unsigned int)Session);
		XplWaitOnLocalSemaphore(SessionMutex[Session->SessionID]);
		XplSignalLocalSemaphore(SessionDBSemaphore);
	} else {
		/* Session Database is full, should clean up */
		XplSignalLocalSemaphore(SessionDBSemaphore);

		// FIXME - need error handling
		MWDestroySession(Session);
		Client->Session=NULL;
		LoggerEvent(LogHandle, LOGGER_SUBSYSTEM_AUTH, LOGGER_EVENT_SESSION_LIMIT_REACHED, LOG_ERROR, 0, NULL, NULL, MAX_SESSION_ENTRIES, 0, NULL, 0);
		return(FALSE);
	}

	if (!MWConnectUserToNMAPServer(Session)) {
		MWDestroySession(Session);
		Client->Session=NULL;
		return(FALSE);
	}

	for (i=0; i<TModuleCount; i++) {
		TModules[i].InitSession(Session, &(Session->ModuleData[i]));
	}

	/* We leave with the session mutex accquired */

	LoggerEvent(LogHandle, LOGGER_SUBSYSTEM_AUTH, LOGGER_EVENT_LOGIN, LOG_INFO, 0, Username, NULL, XplHostToLittle(Client->cs.sin_addr.s_addr), 0, NULL, 0);

	SessionDebug("CreateSession: Session:%x: ID:%d UID:%d, Template:%d\n", (unsigned int)Session, (int)Session->SessionID, (int)Session->SessionUID, (int)Session->TemplateID);
	return(TRUE);
}

SessionStruct
*RetrieveSession(unsigned long SessionID, unsigned long SessionUID)
{
	SessionStruct	*Session;

	SessionDebug("RetrieveSession(%d, %d)\n", (int)SessionID, (int)SessionUID);

	if (SessionID>=MAX_SESSION_ENTRIES) {
		SessionDebug("Error: RetrieveSession: SessionID>=MAX_SESSION_ENTRIES\n");
		return(NULL);
	}

	SessionDebug("Accquiring session %x mutex\n", (unsigned int)SessionDB[SessionID]);
	XplWaitOnLocalSemaphore(SessionMutex[SessionID]);
	Session=SessionDB[SessionID];
	
	if (!Session || Session->SessionUID!=SessionUID) {
		if (!Session) {
			SessionDebug("Error: RetrieveSession: Session NULL\n");
		} else {
			SessionDebug("Error: RetrieveSession: Session UIDs don't match\n");
		}
		SessionDebug("Releasing session %x mutex\n", (unsigned int)SessionDB[SessionID]);
		XplSignalLocalSemaphore(SessionMutex[SessionID]);
		return(NULL);
	}

	XplRenameThread(XplGetThreadID(), Session->User);

	if (Session->NMAPs==-1) {
		if (!MWConnectUserToNMAPServer(Session)) {
			SessionDebug("Releasing session %x mutex\n", (unsigned int)Session);
			XplSignalLocalSemaphore(SessionMutex[Session->SessionID]);
			SessionDebug("Error: RetrieveSession: MWConnectUserToNMAPServer failed\n");
			return(NULL);
		}
	}

	Session->Timestamp = ModwebTimeStamp;

	/*
		Now we need to check to see if the time we have left before NMAP times
		out is less than the time we have left before the session times out.
		If that is the case we need to send a NOOP to NMAP.
	*/

	if ((signed long)(NMAP_CONNECTION_TIMEOUT - (ModwebTimeStamp - Session->NMAPTimestamp)) < (signed long)(Session->ConnectionTimeout)) {
		unsigned char			buffer[BUFSIZE + 1];
#if 0
		XplConsolePrintf("\nMODWEBD: RetrieveSession() found NMAP time remaining less than modweb time remaining.\r\n");
		XplConsolePrintf("         User:                       %s\r\n", Session->User);
		XplConsolePrintf("         ModwebTimeStamp:            %lu\r\n", ModwebTimeStamp);
		XplConsolePrintf("         Session->NMAPTimestamp:     %lu\r\n", Session->NMAPTimestamp);
		XplConsolePrintf("         Session->ConnectionTimeout: %lu\r\n", Session->ConnectionTimeout);
#endif

		MWSendNMAPServer(Session, "NOOP\r\n", 6);
		MWGetNMAPAnswer(Session, buffer, BUFSIZE, TRUE);
	}

	/* We leave with the session mutex accquired */

	SessionDebug("RetrieveSession: Connected to NMAP, timestamp:%x\n", (unsigned int)Session->Timestamp);
	return(Session);
}

BOOL
DestroyTSession(TSessionStruct *TSession)
{
	unsigned char	Path[XPL_MAX_PATH+1];

	SessionDebug("DestroyTSession: %x\n", (unsigned int)TSession);

	if (!TSession) {
		return(FALSE);
	}

	snprintf(Path, sizeof(Path), "%s/T%lu", WorkDir, TSession->SessionID);
	unlink(Path);

	XplWaitOnLocalSemaphore(TSessionDBSemaphore);

	TSessionDB[TSession->SessionID]=NULL;
	TSessionStack[++TSessionStackPtr]=TSession->SessionID;
	SessionDebug("Releasing session %x mutex\n", (unsigned int)TSession);
	XplSignalLocalSemaphore(TSessionMutex[TSession->SessionID]);
	XplSignalLocalSemaphore(TSessionDBSemaphore);

	MemFree(TSession);

	return(TRUE);
}

TSessionStruct
*RetrieveTSession(unsigned long SessionID, unsigned long SessionUID)
{
	TSessionStruct	*TSession;

	SessionDebug("RetrieveTSession(%d, %d)\n", (int)SessionID, (int)SessionUID);

	if (SessionID>=MAX_TSESSION_ENTRIES) {
		SessionDebug("Error: RetrieveSession: SessionID>=MAX_TSESSION_ENTRIES\n");
		return(NULL);
	}

	SessionDebug("Accquiring session %x mutex\n", (unsigned int)TSessionDB[SessionID]);
	XplWaitOnLocalSemaphore(TSessionMutex[SessionID]);
	TSession=TSessionDB[SessionID];
	
	if (!TSession || TSession->SessionUID!=SessionUID) {
		if (!TSession) {
			SessionDebug("Error: RetrieveTSession: Session NULL\n");
		} else {
			SessionDebug("Error: RetrieveTSession: Session UIDs don't match\n");
		}
		SessionDebug("Releasing tsession %x mutex\n", (unsigned int)TSessionDB[SessionID]);
		XplSignalLocalSemaphore(TSessionMutex[SessionID]);
		return(NULL);
	}

	return(TSession);
}


TSessionStruct
*CreateTSession(void)
{
	TSessionStruct			*TSession;

	TSession=MemMalloc(sizeof(TSessionStruct));

	if (!TSession) {
		return(NULL);
	}

	SessionDebug("CreateTSession\n");

	memset(TSession, 0, sizeof(TSessionStruct));
	TSession->SessionUID = time(NULL);
	TSession->Timestamp = ModwebTimeStamp;

	/* We need to insert the session into our array */
	SessionDebug("CreateTSession: Inserting session into global table\n");

	XplWaitOnLocalSemaphore(TSessionDBSemaphore);
	if (TSessionStackPtr>=0) {
		TSession->SessionID=TSessionStack[TSessionStackPtr--];
		TSessionDB[TSession->SessionID]=TSession;
		SessionDebug("Accquiring tsession %x mutex\n", (unsigned int)TSession);
		XplWaitOnLocalSemaphore(TSessionMutex[TSession->SessionID]);
		XplSignalLocalSemaphore(TSessionDBSemaphore);
	} else {
		/* Session Database is full, should clean up */
		XplSignalLocalSemaphore(TSessionDBSemaphore);

		MemFree(TSession);
		return(NULL);
	}

	return(TSession);
}


BOOL
SessionDBInit(void)
{
	unsigned long	i;

	SessionDebugInit();
	memset(&SessionDB, 0, sizeof(SessionStruct *)*MAX_SESSION_ENTRIES);
	memset(&TSessionDB, 0, sizeof(TSessionStruct *)*MAX_TSESSION_ENTRIES);

	SessionDebug("Creating session database & semaphores\n");
	for (i=0; i<MAX_SESSION_ENTRIES; i++) {
		SessionStack[i]=i;
		XplOpenLocalSemaphore(SessionMutex[i], 1);
	}

	for (i=0; i<MAX_TSESSION_ENTRIES; i++) {
		TSessionStack[i]=i;
		XplOpenLocalSemaphore(TSessionMutex[i], 1);
	}
	SessionDebug("Session globals initialized\n");
	return(TRUE);
}

BOOL
SessionDBShutdown(void)
{
	unsigned long	i;

	SessionDebug("Destroying session semaphores\n");
	for (i=0; i<MAX_SESSION_ENTRIES; i++) {
		XplCloseLocalSemaphore(SessionMutex[i]);
	}

	for (i=0; i<MAX_TSESSION_ENTRIES; i++) {
		XplCloseLocalSemaphore(TSessionMutex[i]);
	}

	SessionDebug("Session globals destroyed\n");
	return(TRUE);
}
