/****************************************************************************
 *
 * Copyright (c) 2001-2002 Novell, Inc.
 * All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of version 2.1 of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, contact Novell, Inc.
 *
 * To contact Novell about this file by physical or electronic mail,
 * you may find current contact information at www.novell.com
 *
 ****************************************************************************/

#include <config.h>
#include <xpl.h>

#include <msgapi.h>
#include <nmap.h>
#include <hulautil.h>
#include <libical.h>
#include "modwebp.h"
#include "modweb.ary"

unsigned char	*ModWebDVersion = "$Revision: 1.5 $";

/* Modified Base64 encoding/decoding tables; used in mUTF-7  */
#if 0 
static unsigned char MBASE64[] = {"ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+,"};
#endif
static unsigned char MBASE64_R[256] = {
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x3E, 0x3F, 0xFF, 0xFF, 0xFF,
	0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3A, 0x3B, 0x3C, 0x3D, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E,
	0x0F, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0x1A, 0x1B, 0x1C, 0x1D, 0x1E, 0x1F, 0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 0x28,
	0x29, 0x2A, 0x2B, 0x2C, 0x2D, 0x2E, 0x2F, 0x30, 0x31, 0x32, 0x33, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
};


__inline
static void Base64EncodeUnit(unsigned char *dest, unsigned char b64_buf[3], unsigned char table[]) {
	dest[0] = table[((b64_buf[0] & 0xFC) >> 2)];
	dest[1] = table[((b64_buf[0] & 0x03) << 4) | ((b64_buf[1] & 0xF0) >> 4)];
	dest[2] = table[((b64_buf[1] & 0x0F) << 2) | ((b64_buf[2] & 0xC0) >> 6)];
	dest[3] = table[((b64_buf[2] & 0x3F) << 0)];
}

static int
UTF8EncodeChar(unicode unichar, unsigned char *dest)
{
	if (unichar < 0x0080) {					/* 0000-007F */
		dest[0] = (char)(unichar);
		return(1);
	}
	if (unichar < 0x0800) {					/* 0080-07FF */
		dest[0] = (char)(0xC0 | ((unichar & 0x07C0) >> 6));
		dest[1] = (char)(0x80 | (unichar & 0x003F));
		return(2);
	}
														/* 0800-FFFF */
	dest[0] = (char)(0xE0 | ((unichar & 0xF000) >> 12));
	dest[1] = (char)(0x80 | ((unichar & 0x0FC0) >> 6));
	dest[2] = (char)(0x80 | (unichar & 0x003F));
	return(3);	
}

static int
FolderUTF7toUTF8(const unsigned char *UTF7, unsigned char *UTF8)
{
	unsigned long	Src;
	unsigned long	Dst;
	BOOL				Done=FALSE;
	unsigned long	i;
	unsigned long	UDst;
	unsigned char	Unicode[6];
	unsigned char	Base64[4];

	Src=0;
	Dst=0;

	ModDebug("FolderUTF7toUTF8(): Decoding %s\n", UTF7);

	while (!Done) {
		switch(UTF7[Src]) {
			case 0x7f: {
				UTF8[Dst]=' ';
				Dst++;
				Src++;
				break;
			}

			case '&': {
				Src++;
				if (UTF7[Src]!='-') {

					UDst=0;

					do {
						/* We get four encoded chars, translate them */
						for (i=0; i<4; i++) {
							if ((UTF7[Src]!='\0') && (UTF7[Src]!='-')) {
								Base64[i]=UTF7[Src++];
							} else {
								Base64[i]='A';
							}
						}

						/* We've got the encoded char in Base64, padded if neccessary, now translate to unicode */
						i=0;
						Unicode[UDst++] = ((MBASE64_R[Base64[0]] << 2) & 0xFC) | ((MBASE64_R[Base64[1]] >> 4) & 0x03);
						Unicode[UDst++] = ((MBASE64_R[Base64[1]] << 4) & 0xF0) | ((MBASE64_R[Base64[2]] >> 2) & 0x0F);
						Unicode[UDst++] = ((MBASE64_R[Base64[2]] << 6) & 0xC0) | ((MBASE64_R[Base64[3]] >> 0) & 0x3F);

						/* Convert to UTF-8 every 6 bytes (= 3 Unicode characters, = 2 loop iterations) */
						if (UDst==6) {
							for (i = 0; i < 6; i += 2) {
								if (((Unicode[i] << 8) | Unicode[i + 1]) != 0) {	/* May have padding */
									Dst += UTF8EncodeChar((unicode )(Unicode[i] << 8) | Unicode[i + 1], UTF8 + Dst);
								}
							}
							UDst=0;
						}						
					} while (UTF7[Src]!='\0' && UTF7[Src]!='-');

					if (UTF7[Src]=='-') {
						/* Whack the terminator */
						Src++;
					}

					if (UDst==3) {
						Dst += UTF8EncodeChar((unicode )(Unicode[0] << 8) | Unicode[1], UTF8 + Dst);
					}

				} else {
					UTF8[Dst]='&';
					Dst++;
					Src++;
				}

				break;
			}

			case '\0': {
				UTF8[Dst]='\0';
				Done=TRUE;
				break;
			}

			default: {
				UTF8[Dst++]=UTF7[Src++];
				break;
			}
		}
	}

	return(Dst);
}

static int
SortFolder(const void *Folder1, const void *Folder2)
{
	BOOL			Folder1Inbox = FALSE;
	BOOL			Folder2Inbox = FALSE;
	BOOL			Folder1Main = FALSE;
	BOOL			Folder2Main = FALSE;

	/* Ignore the first three characters.  It is being used to specify the type */

	/* Special case INBOX -- it always goes first */
	if (MWQuickNCmp(*(unsigned char **)Folder1 + 3, "INBOX", 5)) {
		Folder1Inbox = TRUE;
	}

	if (MWQuickNCmp(*(unsigned char **)Folder2 + 3, "INBOX", 5)) {
		Folder2Inbox = TRUE;
	}

	/* Special case MAIN -- it always goes last */
	if (MWQuickNCmp(*(unsigned char **)Folder1 + 3, "MAIN", 4)) {
		Folder1Main = TRUE;
	}

	if (MWQuickNCmp(*(unsigned char **)Folder2 + 3, "MAIN", 4)) {
		Folder2Main = TRUE;
	}

	if (Folder1Inbox) {
		if (Folder2Inbox) {
			/* Both folders are INBOX* */

			return(XplStrCaseCmp(*(unsigned char **)Folder1 + 3, *(unsigned char **)Folder2 + 3));
		} else {
			return(-1);
		}
	} else if (Folder2Inbox) {
		/* Folder 2 is INBOX* and folder 1 isn't */

		return(1);
	} else if (Folder1Main) {
		if (Folder2Main) {
			/* Both folders are MAIN* */

			return(XplStrCaseCmp(*(unsigned char **)Folder1 + 3, *(unsigned char **)Folder2 + 3));
		} else {
			return(1);
		}
	} else if (Folder2Main) {
		/* Folder 2 is MAIN* and folder 1 isn't */

		return(-1);
	}
		
	/* Neither are in the INBOX hierarchy, just compare them normally */
	return(XplStrCaseCmp(*(unsigned char **)Folder1 + 3, *(unsigned char **)Folder2 + 3));
}

static BOOL
UpdateFolderList(SessionStruct *Session)
{
	int				ReplyInt;
	unsigned char	Answer[BUFSIZE + 1];
	unsigned long	i;
	unsigned char	SelectedFolder[BUFSIZE + 1];
	unsigned char	SelectedCalendar[BUFSIZE + 1];
	BOOL				CalendarEnabled;

	if (MsgGetUserFeature(Session->UserDN, FEATURE_CALENDAR, NULL, NULL)) {
		CalendarEnabled = TRUE;
	} else {
		CalendarEnabled = FALSE;
	}

	/* Store the currently selected folder and calendar name so we can set it again later */
	if (Session->CurrentFolder < Session->FolderList->Used) {
		unsigned char		*folder = Session->FolderList->Value[Session->CurrentFolder];
		HulaStrNCpy(SelectedFolder, folder, BUFSIZE);
	} else {
		SelectedFolder[0] = '\0';
	}

	Session->CurrentFolder = 0;

	if (Session->CurrentCalendar < Session->FolderList->Used) {
		unsigned char		*folder = Session->FolderList->Value[Session->CurrentCalendar];

		HulaStrNCpy(SelectedCalendar, folder, BUFSIZE);
	} else {
		SelectedCalendar[0] = '\0';
	}

	Session->CurrentCalendar = 0;

	MDBFreeValues(Session->FolderList);
	MDBFreeValues(Session->FolderDisplayNames);
	if (Session->FolderUnreadCount) {
		MemFree(Session->FolderUnreadCount);
		Session->FolderUnreadCount = NULL;
		Session->FolderTotalCount = NULL;
	}

	/*
		Now, read the new list of mailboxs and folder elements. 
	*/
	MWSendNMAPServer(Session, "RESOURCE\r\n", 10);
	ReplyInt = MWGetNMAPAnswer(Session, Answer, sizeof(Answer), TRUE);
	if (ReplyInt != 2002) {
#if 0
		EnterDebugger();
#endif
		ModDebug("UpdateFolderList(): Bad NMAP response:%s\n", Answer);
		return(FALSE);
	}

	while ((ReplyInt = MWGetNMAPAnswer(Session, Answer, BUFSIZE, TRUE)) != 1000) {
		/* Strip off any trailing '/'s.  We know from the first char if is a real folder or not */
		i = strlen(Answer + 1);
		if (Answer[i] == '/') {
			Answer[i] = '\0';
		}

		if (strlen(Answer) > 3) {
			if (!CalendarEnabled && *Answer == 'C') {
				/*
					If calendaring is not enabled switch calendars to Hierarchical Element
				*/

				Answer[0] = 'H';
				Answer[1] = '0';
				Answer[2] = '0';
			}

			MDBAddValue(Answer, Session->FolderList);
		}
	}

	qsort(Session->FolderList->Value, Session->FolderList->Used, sizeof(unsigned char *), SortFolder);

	for (i = 0; i < Session->FolderList->Used; i++) {
		unsigned char		*folder = strrchr(Session->FolderList->Value[i] + 3, '/');

		if (!folder) {
			folder = Session->FolderList->Value[i] + 3;
		} else {
			folder++;
		}

		FolderUTF7toUTF8(folder, Answer);
		MDBAddValue(Answer, Session->FolderDisplayNames);
	}

	Session->FolderUnreadCount = MemMalloc((Session->FolderList->Used + 1) * sizeof(unsigned long) * 2);
	if (Session->FolderUnreadCount) {
		Session->FolderTotalCount = Session->FolderUnreadCount + (Session->FolderList->Used + 1);

		for (i = 0; i < Session->FolderList->Used; i++) {
			unsigned char		*folder = Session->FolderList->Value[i];

			Session->FolderUnreadCount[i] = 0;
			Session->FolderTotalCount[i] = 0;

			if (*folder == 'M') {
				ReplyInt = snprintf(Answer, sizeof(Answer), "CHECK %s\r\n", folder + 3);
				MWSendNMAPServer(Session, Answer, ReplyInt);

				ReplyInt = MWGetNMAPAnswer(Session, Answer, sizeof(Answer), TRUE);
				if (ReplyInt == 1000) {
					unsigned long		read;
					unsigned long		total;
					unsigned long		purged;

					if (sscanf(Answer, "%lu %*u %*u %*u %lu %*u %lu", &total, &purged, &read) == 3) {
						Session->FolderTotalCount[i] = total - purged;
						Session->FolderUnreadCount[i] = total - read - purged;
					}
				}
			} 

			/* note: calendars inherently don't have unread items */
			/* they are either implicitly or explicitly accepted. */
		}
	} 
else {
#if 0
		EnterDebugger();
#endif
	}

	/* Now reselect the folder and calendar */
	for (i = 0; i < Session->FolderList->Used; i++) {
		unsigned char		*folder = Session->FolderList->Value[i];

		if (MWQuickCmp(SelectedFolder, folder)) {
			Session->CurrentFolder = i;
		} else if (MWQuickCmp(SelectedCalendar, folder)) {
			Session->CurrentCalendar = i;
		}
	}

	/* Check to see if folder list changed since we obtained a CurrentDisplayName */
	if (Session->CurrentDisplayFolder > Session->FolderDisplayNames->Used) {
		/* Reset to default folder */
		Session->CurrentDisplayFolder = 0;
	}
		

	return(TRUE);
}

static BOOL
SetUserVariable(ConnectionStruct *client, unsigned long id, unsigned char *value, unsigned long len)
{
    UserValue *node;

    node = client->Session->UserValues;
   
    while (node) {
	if (node->id == id) {
	    if (node->value) {
		MemFree(node->value);
	    }
	    
	    node->value = MemMalloc(len + 1);
	    if (node->value) {
		
		memcpy(node->value, value, len);
		node->value[len] = '\0';
		return(TRUE);
	    }
	    return(FALSE);
	}

	node = node->next;
    }

    /* it doesn't exist yet */
    node = MemCalloc(1, sizeof(UserValue)); 
    if (node) {
	node->id = id;

	node->value = MemMalloc(len + 1);
	if (node->value) {
	    memcpy(node->value, value, len);
	    node->value[len] = '\0';
	    node->next = client->Session->UserValues;
	    client->Session->UserValues = node;
	    return(TRUE);
	}
	MemFree(node);
    }
    return(FALSE);
}

static unsigned char *
GetUserVariable(ConnectionStruct *client, unsigned long id)
{
    UserValue *node;

    node = client->Session->UserValues;
   
    while (node) {
	if (node->id == id) {
	    return(node->value);
	}

	node = node->next;
    }

    return("");
}

BOOL
SendHTTPHeaderDynamic(ConnectionStruct *Client, unsigned char *contentType, unsigned long contentTypeLen)
{
	if (Client->SentHeader==FALSE) {
		MWSendClient(Client, "HTTP/1.1 200 Ok\r\nPragma: no-cache\r\nConnection: ", 47);

		if (Client->KeepAlive) {
			MWSendClient(Client, "keep-alive\r\n", 12);
		} else {
			MWSendClient(Client, "close\r\n", 7);
		}

		if (Client->Cookie) {
			MWSendClient(Client, Client->Temp, snprintf(Client->Temp, sizeof (Client->Temp), "Set-Cookie: SID%lu=%lu; path=/\r\n", Client->Session->CookieID, Client->Cookie));
		} else if (Client->ClearCookie) {
			MWSendClient(Client, Client->Temp, snprintf(Client->Temp, sizeof(Client->Temp), "Set-Cookie: SID%lu=0; path=/\r\n", Client->Session->CookieID));
		}

		switch (Client->DeviceType) {
			case DEVICE_HTML: 
			default: {
				MWSendClient(Client, "Content-type: ", strlen("Content-type: "));
				MWSendClient(Client, contentType, contentTypeLen);
				MWSendClient(Client, "\r\n\r\n", strlen("\r\n\r\n"));
				break;
			}

			case DEVICE_WML: {
			    if ((contentTypeLen != strlen("text/html")) || !MWQuickCmp(contentType, "text/html")) {
				MWSendClient(Client, "Content-type: ", strlen("Content-type: "));
				MWSendClient(Client, contentType, contentTypeLen);
				MWSendClient(Client, "\r\n\r\n", strlen("\r\n\r\n"));
				break;
			    } else {
				MWSendClient(Client, "Content-type: text/vnd.wap.wml\r\n\r\n", 34);
				break;
			    }
			}
		}

		Client->SentHeader=TRUE;
	}
	return(TRUE);
}

void
MWReleaseStreams(StreamStruct *FirstStream)
{
	if (FirstStream) {
		StreamStruct *CurrentStream;
		StreamStruct *PreviousStream;

		do {
			CurrentStream = FirstStream;
			PreviousStream = NULL;
			while (CurrentStream->Next != NULL) {
				PreviousStream = CurrentStream;
				CurrentStream = CurrentStream->Next;
			}
			MWReleaseStream(CurrentStream);
			if (PreviousStream) {
				PreviousStream->Next = NULL;
			}
		} while(PreviousStream);
	}
}

BOOL
MWDisplayMIMEParam(ConnectionStruct *Client, unsigned char *Value, StreamStruct **MIMEParamStream)
{
	StreamStruct	*memoryInCodec;

	if (*MIMEParamStream) {
		memoryInCodec = *MIMEParamStream;
		memoryInCodec->StreamData = Value;
		memoryInCodec->StreamLength = strlen(Value);	
		memoryInCodec->Codec(memoryInCodec, memoryInCodec->Next);
		return(TRUE);
	} else {
		StreamStruct	*rfc2231ValueDecodeCodec;
		StreamStruct	*htmlEncodeCodec;
		StreamStruct	*networkOutCodec;

		memoryInCodec = MWGetStream(NULL, NULL, FALSE);
		if (memoryInCodec) {
			memoryInCodec->Codec = MWStreamFromMemory;

			rfc2231ValueDecodeCodec = MWGetStream(memoryInCodec, "RFC2231_NMAP_Value", FALSE);
			if (rfc2231ValueDecodeCodec) {
				rfc2231ValueDecodeCodec->Charset = Client->Session->Charset;
				htmlEncodeCodec = MWGetStream(rfc2231ValueDecodeCodec, "text/plain", FALSE);
				if (htmlEncodeCodec) {
					networkOutCodec = MWGetStream(htmlEncodeCodec, NULL, FALSE);
					if (networkOutCodec) {
						networkOutCodec->Codec = MWStreamToMWClient;
						networkOutCodec->StreamData = Client;

						/* send parameter */
						memoryInCodec->StreamData = Value;
						memoryInCodec->StreamLength = strlen(Value);	
						memoryInCodec->Codec(memoryInCodec, rfc2231ValueDecodeCodec);

						/* save stream */
						*MIMEParamStream = memoryInCodec;
						return(TRUE);
					}
					MWReleaseStream(htmlEncodeCodec);
				}
				MWReleaseStream(rfc2231ValueDecodeCodec);
			}
			MWReleaseStream(memoryInCodec);
		}

		MWSendClient(Client, Value, strlen(Value));
		return(FALSE);
	}
}

static BOOL
MWHTTPSendMIMEParam(ConnectionStruct *Client, unsigned char *Name, unsigned char *Value, unsigned char *UserCharset)
{
	StreamStruct	*memoryInCodec;

	if (MIMEParamEncodingforHTTP == ANE_RESTRICT_TO_7BIT) {	 
		/* Mem -> RFC2231_Value_Decode -> quoted_string_encode -> Browser */
		memoryInCodec = MWGetStream(NULL, NULL, FALSE);
		if (memoryInCodec) {
			StreamStruct	*rfc2231ValueDecodeCodec;

			memset(memoryInCodec, 0, sizeof(StreamStruct));
			memoryInCodec->Codec = MWStreamFromMemory;

			rfc2231ValueDecodeCodec = MWGetStream(memoryInCodec, "RFC2231_NMAP_Value", FALSE);
			if (rfc2231ValueDecodeCodec) {
				StreamStruct	*quotedStringCodec;

				rfc2231ValueDecodeCodec->Charset = UserCharset;
				quotedStringCodec = MWGetStream(rfc2231ValueDecodeCodec, "quoted-string", TRUE);
				if (quotedStringCodec) {
					StreamStruct	*netOutCodec;
					
					netOutCodec = MWGetStream(quotedStringCodec, NULL, FALSE);
					if (netOutCodec) {
						netOutCodec->Codec = MWStreamToMWClient;
						netOutCodec->StreamData = Client;

						memoryInCodec->StreamData = Value;
						memoryInCodec->StreamLength = strlen(Value);	

						/* send parameter */
						MWSendClient(Client, Name, strlen(Name));
						MWSendClient(Client, "=", 1);
						memoryInCodec->Codec(memoryInCodec, rfc2231ValueDecodeCodec);

						MWReleaseStream(netOutCodec);
						MWReleaseStream(quotedStringCodec);
						MWReleaseStream(rfc2231ValueDecodeCodec);
						MWReleaseStream(memoryInCodec);
						return(TRUE);
					}
					MWReleaseStream(quotedStringCodec);
				}
				MWReleaseStream(rfc2231ValueDecodeCodec);
			}
			MWReleaseStream(memoryInCodec);
		}
	} else if (MIMEParamEncodingforHTTP == ANE_RFC2231) {
		/* Mem -> RFC2231_Value_Decode -> UserCharset -> RFC2231_Value_Encode -> Browser */
		memoryInCodec = MWGetStream(NULL, NULL, FALSE);
		if (memoryInCodec) {
			StreamStruct	*rfc2231ValueDecodeCodec;

			memset(memoryInCodec, 0, sizeof(StreamStruct));
			memoryInCodec->Codec = MWStreamFromMemory;
			memoryInCodec->StreamData = Value;
			memoryInCodec->StreamLength = strlen(Value);	

			rfc2231ValueDecodeCodec = MWGetStream(memoryInCodec, "RFC2231_NMAP_Value", FALSE);
			if (rfc2231ValueDecodeCodec) {
				StreamStruct	*charsetCodec;
				rfc2231ValueDecodeCodec->Charset = UserCharset;
				charsetCodec = MWGetStream(rfc2231ValueDecodeCodec, UserCharset, TRUE);
				if (charsetCodec) {
					StreamStruct	*rfc2231ValueEncodeCodec;
					rfc2231ValueEncodeCodec = MWGetStream(charsetCodec, "RFC2231_Param_Value", TRUE);
					if (rfc2231ValueEncodeCodec) {
						StreamStruct	*netOutCodec;

						rfc2231ValueEncodeCodec->Charset = UserCharset;
						rfc2231ValueEncodeCodec->StreamData = (void *)Name;

						netOutCodec = MWGetStream(rfc2231ValueEncodeCodec, NULL, FALSE);
						if (netOutCodec) {
							netOutCodec->Codec = MWStreamToMWClient;
							netOutCodec->StreamData = Client;

							/* send parameter */
							memoryInCodec->Codec(memoryInCodec, rfc2231ValueDecodeCodec);

							MWReleaseStream(netOutCodec);
							MWReleaseStream(rfc2231ValueEncodeCodec);
							MWReleaseStream(charsetCodec);
							MWReleaseStream(rfc2231ValueDecodeCodec);
							MWReleaseStream(memoryInCodec);
							return(TRUE);
						}
						MWReleaseStream(rfc2231ValueEncodeCodec);
					}
					MWReleaseStream(charsetCodec);
				}
				MWReleaseStream(rfc2231ValueDecodeCodec);
			}
			MWReleaseStream(memoryInCodec);
		}
	} else if (MIMEParamEncodingforHTTP == ANE_RFC2047) {
		/* Mem -> RFC2231_Value_Decode -> RFC2047_Encode -> Browser */
		memoryInCodec = MWGetStream(NULL, NULL, FALSE);
		if (memoryInCodec) {
			StreamStruct	*rfc2231ValueDecodeCodec;

			memset(memoryInCodec, 0, sizeof(StreamStruct));
			memoryInCodec->Codec = MWStreamFromMemory;
			memoryInCodec->StreamData = Value;
			memoryInCodec->StreamLength = strlen(Value);	

			rfc2231ValueDecodeCodec = MWGetStream(memoryInCodec, "RFC2231_NMAP_Value", FALSE);
			if (rfc2231ValueDecodeCodec) {
				StreamStruct	*rfc2047EncodeCodec;
				rfc2231ValueDecodeCodec->Charset = UserCharset;
				rfc2047EncodeCodec = MWGetStream(rfc2231ValueDecodeCodec, "RFC2047_Name", TRUE);
				if (rfc2047EncodeCodec) {
					StreamStruct	*netOutCodec;

					rfc2047EncodeCodec->Charset = UserCharset;
					
					netOutCodec = MWGetStream(rfc2047EncodeCodec, NULL, FALSE);
					if (netOutCodec) {
						netOutCodec->Codec = MWStreamToMWClient;
						netOutCodec->StreamData = Client;

						/* send parameter */
						MWSendClient(Client, Name, strlen(Name));
						MWSendClient(Client, "=", 1);
						memoryInCodec->Codec(memoryInCodec, rfc2231ValueDecodeCodec);

						MWReleaseStream(netOutCodec);
						MWReleaseStream(rfc2047EncodeCodec);
						MWReleaseStream(rfc2231ValueDecodeCodec);
						MWReleaseStream(memoryInCodec);
						return(TRUE);
					}
					MWReleaseStream(rfc2047EncodeCodec);
				}
				MWReleaseStream(rfc2231ValueDecodeCodec);
			}
			MWReleaseStream(memoryInCodec);
		}
	} else {
		/* Mem -> RFC2231_Value_Decode -> UserCharset -> File */
		memoryInCodec = MWGetStream(NULL, NULL, FALSE);
		if (memoryInCodec) {
			StreamStruct	*rfc2231ValueDecodeCodec;

			memset(memoryInCodec, 0, sizeof(StreamStruct));
			memoryInCodec->Codec = MWStreamFromMemory;
			memoryInCodec->StreamData = Value;
			memoryInCodec->StreamLength = strlen(Value);	

			rfc2231ValueDecodeCodec = MWGetStream(memoryInCodec, "RFC2231_NMAP_Value", FALSE);
			if (rfc2231ValueDecodeCodec) {
				StreamStruct	*charsetCodec;
				rfc2231ValueDecodeCodec->Charset = UserCharset;
				charsetCodec = MWGetStream(rfc2231ValueDecodeCodec, UserCharset, TRUE);
				if (charsetCodec) {
					StreamStruct	*netOutCodec;

					netOutCodec = MWGetStream(charsetCodec, NULL, FALSE);
					if (netOutCodec) {
						netOutCodec->Codec = MWStreamToMWClient;
						netOutCodec->StreamData = Client;

						/* send parameter */
						MWSendClient(Client, Name, strlen(Name));
						MWSendClient(Client, "=\"", 2);
						memoryInCodec->Codec(memoryInCodec, rfc2231ValueDecodeCodec);
						MWSendClient(Client, "\"", 1);

						MWReleaseStream(netOutCodec);
						MWReleaseStream(charsetCodec);
						MWReleaseStream(rfc2231ValueDecodeCodec);
						MWReleaseStream(memoryInCodec);
						return(TRUE);
					}
					MWReleaseStream(charsetCodec);
				}
				MWReleaseStream(rfc2231ValueDecodeCodec);
			}
			MWReleaseStream(memoryInCodec);
		}
	}

	/* send parameter */
	MWSendClient(Client, Name, strlen(Name));
	MWSendClient(Client, "=\"", 2);
	MWSendClient(Client, Value, strlen(Value));
	MWSendClient(Client, "\"\r\n", 3);
	return(FALSE);
}

BOOL
MWSendHTTPHeader(ConnectionStruct *Client, unsigned char *Type, unsigned long Size, unsigned char *Filename, BOOL Cache)
{
	MWSendClient(Client, "HTTP/1.1 200 Ok\r\n", 17);
//	if (!Cache) {
//		MWSendClient(Client, "Pragma: no-cache\r\n", 18);
//	}
	if (Client->KeepAlive) {
		MWSendClient(Client, "Connection: keep-alive\r\n", 24);
	} else {
		MWSendClient(Client, "Connection: close\r\n", 19);
	}
	if (Client->Cookie) {
		MWSendClient(Client, Client->Temp, snprintf(Client->Temp, sizeof(Client->Temp), "Set-Cookie: SID%lu=%lu\r\n", Client->Session->CookieID, Client->Cookie));
	}

	if (Filename) {
		MWSendClient(Client, "Content-Disposition: attachment; ", 33);
		MWHTTPSendMIMEParam(Client, "filename", Filename, Client->Session->Charset);
		MWSendClient(Client, "\r\n", 2);
		MWSendClient(Client, Client->Temp, snprintf(Client->Temp, sizeof(Client->Temp), "Content-Type: application/octect-stream\r\n\r\n"));
		return(TRUE);
	}

	if (Size>0) {
		MWSendClient(Client, Client->Temp, snprintf(Client->Temp, sizeof(Client->Temp), "Content-Type: %s\r\nContent-length: %lu\r\n\r\n", Type, Size));
	} else {
		MWSendClient(Client, Client->Temp, snprintf(Client->Temp, sizeof(Client->Temp), "Content-Type: %s\r\n\r\n", Type));
	}
	return(TRUE);
}

static BOOL
HandleImage(ConnectionStruct *Client, unsigned long Template, unsigned long Language, unsigned long ImageID)
{
	unsigned char	*Image;
	unsigned long	Length;

	Client->KeepAlive=TRUE;

	if (Template>=TemplateCount || ImageID>=Templates[Template]->ImageCount || Language>=Templates[Template]->LanguageCount) {
		Client->KeepAlive=FALSE;
		return(FALSE);
	}

	MWDebug("HandleImage(ImageID:%d Language:%x) called\n", (int)ImageID, (unsigned int)Language);

	Image=TemplateImages[Template*TemplateMaxLanguages+Language][ImageID];

	Length=*(unsigned long *)Image;
	Image+=sizeof(unsigned long);

	if (Image[0]=='G') {
		MWSendHTTPHeader(Client, "image/gif", Length, NULL, TRUE);
	} else {
		MWSendHTTPHeader(Client, "image/jpeg", Length, NULL, TRUE);
	}
	MWSendClient(Client, Image, Length);

	return(TRUE);
}

static BOOL
TemplateHandleStaticObject(ConnectionStruct *Client, TemplateObjectOverlay *StaticObject)
{
    Client->KeepAlive=FALSE;

    if (!CheckDiskForTemplateUpdates) {
	MWSendHTTPHeader(Client, StaticObject->ContentType, StaticObject->ObjectLen, NULL, TRUE);
	MWSendClient(Client, StaticObject->Object, StaticObject->ObjectLen);

	return(TRUE);
    } else {
	unsigned char *path;
	int pathSize;
	struct stat sb;
	unsigned char *object;
	FILE *objectFile;
	long ret;

	pathSize = strlen(Templates[Client->Session->TemplateID]->SourcePath) + StaticObject->FileNameLen;
	path = MemMalloc(pathSize);
	if (path) {
	    snprintf(path, pathSize, "%s%s", Templates[Client->Session->TemplateID]->SourcePath, StaticObject->FileName);
	    ret = stat(path, &sb);
	    if ((ret != 0) || (sb.st_mtime == StaticObject->FileModTime)) {
		MWSendHTTPHeader(Client, StaticObject->ContentType, StaticObject->ObjectLen, NULL, TRUE);
		MWSendClient(Client, StaticObject->Object, StaticObject->ObjectLen);
		MemFree(path);
		return(TRUE);
	    } else {
		object = MemMalloc(sb.st_size);
		if (object) {
		    objectFile = fopen(path, "r");
		    if (objectFile) {
			fread(object, sizeof(unsigned char), sb.st_size, objectFile);
			MWSendHTTPHeader(Client, StaticObject->ContentType, sb.st_size, NULL, TRUE);
			MWSendClient(Client, object, sb.st_size);
			MemFree(object);
			MemFree(path);
			fclose(objectFile);
			return(TRUE);
		    }
		    MemFree(object);
		}
		
	    }
	    MemFree(path);
	}

	MWSendHTTPHeader(Client, StaticObject->ContentType, 26, NULL, TRUE);
	MWSendClient(Client, "HULAMODWEB: Out of Memmory!", 26);

	return(FALSE);
    }
}

BOOL
HandleLogo(ConnectionStruct *Client, unsigned long LogoID)
{
	long	Length;
	unsigned long	Read;
	struct stat		sb;
	unsigned char	Path[XPL_MAX_PATH+1];
	FILE				*Image;
	BOOL				JPEG=FALSE;

	snprintf(Path, sizeof(Path), "%s/%d.gif", LogoDir, (int)LogoID);
	if (stat(Path, &sb)!=0) {
		snprintf(Path, sizeof(Path), "%s/%d.jpg", LogoDir, (int)LogoID);
		if (stat(Path, &sb)!=0) {
			Client->KeepAlive=FALSE;
			return(FALSE);
		}
		JPEG=TRUE;
	}

	Image=fopen(Path, "rb");
	if (!Image) {
		Client->KeepAlive=FALSE;
		return(FALSE);
	}
	Client->KeepAlive=TRUE;

	MWDebug("HandleLogo(Path:%s) called\n", Path);

	if (!JPEG) {
		MWSendHTTPHeader(Client, "image/gif", sb.st_size, NULL, TRUE);
	} else {
		MWSendHTTPHeader(Client, "image/jpeg", sb.st_size, NULL, TRUE);
	}

	Length=0;
	while ((Length<sb.st_size) && !feof(Image) && !ferror(Image)) {
		if ((Read=fread(Client->Temp, sizeof(unsigned char), BUFSIZE, Image))>0) {
			MWSendClient(Client, Client->Temp, Read);
			Length+=Read;
		}
	}
	fclose(Image);
	return(TRUE);
}

BOOL
MWHandleNamedTemplate(void *ClientIn, unsigned char *TemplateName, void *ObjectData)
{
	ConnectionStruct			*Client=(ConnectionStruct *)ClientIn;
	SessionStruct				*Session=Client->Session;
	unsigned long				TemplateID;
	unsigned char				URL[256];
	unsigned char				Answer[BUFSIZE+1];
	unsigned long				ReplyInt;
	unsigned char				*Template;
	TokenOverlayStruct		*Token;
	register unsigned char	*ptr;
	MWHandleTemplateFunc		TemplateHandler = NULL;
	StreamDescStruct        *StreamListPtr=MWStreamList;
	unsigned long           i=0;

	if (!Client || !Session || !ObjectData || !TemplateName || (TemplateID=MWFindTemplatePage(TemplateName, Session->TemplateID))==(unsigned long)-1) {
		return(FALSE);
	}

	/* Find the handler for this template */
	while (StreamListPtr[i].Charset!=NULL) {
		if (MWQuickCmp(TemplateName, StreamListPtr[i].Charset)) {
			TemplateHandler=(MWHandleTemplateFunc)StreamListPtr[i].TemplateHandler;
			break;
		}
		i++;
	}

	MWDebug("MWHandleNamedTemplate(Sesson:%x TemplateID:%d) called\n", (unsigned int)Session, (int)TemplateID);

	SendHTTPHeaderDynamic(Client, Session->Pages[TemplateID]->ContentType, Session->Pages[TemplateID]->ContentTypeLen);

	Template = Session->Pages[TemplateID]->Object;
	Token=(TokenOverlayStruct *)Template;

NextToken:
	while (Token->TokenID!=0) {
		switch(Token->TokenID) {
			case T_TEXT: {
				MWSendClient(Client, Token->Data+Token->ArgumentOffsetOrID[0], Token->ArgumentSize[0]);
				break;
			}

			case T_QSTRING:
			case T_STRING: {
				ptr=Session->Strings[Token->ArgumentOffsetOrID[0]];
				MWSendClient(Client, ptr, strlen(ptr));
				break;
			}

			case T_IMAGE: {
				MWEncodeURL(Session, URL, URL_TYPE_IMAGE, DISPLAY_IMAGE, Session->TemplateID, Session->Language, Token->ArgumentOffsetOrID[0], 0);
				MWSendClient(Client, URL, strlen(URL));
				break;
			}

			case T_STATIC_OBJECT: {
				MWEncodeURL(Session, URL, URL_TYPE_STATIC, DISPLAY_STATIC_OBJECT, Session->TemplateID, Token->ArgumentOffsetOrID[0], 0, 0);
				MWSendClient(Client, URL, strlen(URL));
				break;
			}

			case T_LOGOUT: {
				MWEncodeURL(Session, URL, URL_TYPE_LINK, REQUEST_LOGOUT, Token->ArgumentOffsetOrID[0], 0, 0, 0);
				MWSendClient(Client, URL, strlen(URL));
				break;
			}

			case T_PAGEID:
			case T_PAGE: {
				MWEncodeURL(Session, URL, URL_TYPE_LINK, DISPLAY_TEMPLATE, Token->ArgumentOffsetOrID[0], 0, 0, 0);
				MWSendClient(Client, URL, strlen(URL));
				break;
			}

			case T_LOGO: {
				if (Session->LogoID==0) {
					MWEncodeURL(Session, URL, URL_TYPE_IMAGE, DISPLAY_IMAGE, Session->TemplateID, Session->Language, Token->ArgumentOffsetOrID[0], 0);
				} else {
					MWEncodeURL(Session, URL, URL_TYPE_IMAGE, DISPLAY_LOGO, Session->LogoID, 0, 0, 0);
				}
				MWSendClient(Client, URL, strlen(URL));
				break;
			}

			case T_APP_TITLE: {
				if (Session->Title) {
					MWSendClient(Client, Session->Title, strlen(Session->Title));
				} else if (DefaultTitle) {
					MWSendClient(Client, DefaultTitle, strlen(DefaultTitle));
				} else {
					MWSendClient(Client, Templates[Session->TemplateID]->Name, strlen(Templates[Session->TemplateID]->Name));
				}
				break;
			}

			case T_COLOR: {
				switch (Token->ArgumentOffsetOrID[0]) {
					case AA_PAGEFG: {
						MWSendClient(Client, Session->Colors[COLOR_PAGE_FG], 6);
						break;
					}

					case AA_PAGEBG: {
						MWSendClient(Client, Session->Colors[COLOR_PAGE_BG], 6);
						break;
					}

					case AA_BORDERFG: {
						MWSendClient(Client, Session->Colors[COLOR_BORDER_FG], 6);
						break;
					}

					case AA_BORDERBG: {
						MWSendClient(Client, Session->Colors[COLOR_BORDER_BG], 6);
						break;
					}

					case AA_SECTIONFG: {
						MWSendClient(Client, Session->Colors[COLOR_SECTION_FG], 6);
						break;
					}

					case AA_SECTIONBG: {
						MWSendClient(Client, Session->Colors[COLOR_SECTION_BG], 6);
						break;
					}

					case AA_FIELDNAMEFG: {
						MWSendClient(Client, Session->Colors[COLOR_FIELDNAME_FG], 6);
						break;
					}

					case AA_FIELDNAMEBG: {
						MWSendClient(Client, Session->Colors[COLOR_FIELDNAME_BG], 6);
						break;
					}

					case AA_FIELDBODYFG: {
						MWSendClient(Client, Session->Colors[COLOR_FIELDBODY_FG], 6);
						break;
					}

					case AA_FIELDBODYBG: {
						MWSendClient(Client, Session->Colors[COLOR_FIELDBODY_BG], 6);
						break;
					}
				}
				break;
			}

			case T_USERNAME: {
				MWSendClient(Client, Session->User, strlen(Session->User));
				break;
			}

			case T_FULLNAME: {
				MWSendClient(Client, Session->DisplayName, strlen(Session->DisplayName));
				break;
			}

			case T_UID: {
				MWSendClient(Client, Client->Temp, snprintf(Client->Temp, sizeof(Client->Temp), "%lu", Session->SessionUID));
				break;
			}

			case T_LANGUAGELIST: {
				unsigned long	*LanguageList=(unsigned long *)Templates[Session->TemplateID] +sizeof(TemplateStruct);
				unsigned long	i;

				for (i=0; i<Templates[Session->TemplateID]->LanguageCount; i++) {
					if (XplReturnLanguageName(LanguageList[i], URL)==0) {
						ReplyInt=snprintf(Answer, sizeof(Answer), "<OPTION VALUE=\"%d\"%s>%s</OPTION>\r\n", (int)LanguageList[i], (Session->Language==i) ? " SELECTED" : "", URL);
						MWSendClient(Client, Answer, ReplyInt);
					}
				}
				break;
			}

			case T_DATETIME_SETUP: {
				Session->DateFormat.monthShort=&Session->Strings[Token->ArgumentOffsetOrID[0]];
				Session->DateFormat.monthLong=&Session->Strings[Token->ArgumentOffsetOrID[1]];
				Session->DateFormat.wDayShort=&Session->Strings[Token->ArgumentOffsetOrID[2]];
				Session->DateFormat.wDayLong=&Session->Strings[Token->ArgumentOffsetOrID[3]];
				Session->DateFormat.AmPm=&Session->Strings[Token->ArgumentOffsetOrID[4]];
				break;
			}

			case T_SESS_OVERRIDE: {
				switch (Token->ArgumentOffsetOrID[0]) {
					case AA_LANGUAGE: {
						Session->Language = Token->ArgumentOffsetOrID[1];
						break;
					}	
					
					case AA_STARTWEEKDAY: {
						Session->StartWeekday = Token->ArgumentOffsetOrID[1];
						break;
					}	

					case AA_MESSAGESPERPAGE: {
						Session->MessagesPerPage = Token->ArgumentOffsetOrID[1];
						break;
					}	

					case AA_LOGOID: {
						Session->LogoID = Token->ArgumentOffsetOrID[1];
						break;
					}	

					case AA_CONNECTIONTIMEOUT: {
						Session->ConnectionTimeout = Token->ArgumentOffsetOrID[1];
						break;
					}	

					default: {
						break;
					}	
				}
				break;
			}

			default: {
				unsigned long	GotoToken;

				if (TemplateHandler) {
					ReplyInt=TemplateHandler(Client, Session, TemplateID, Token, &GotoToken, ObjectData);

					if (ReplyInt == TOKEN_DONE) {
					    if (Token->TokenID != 0) {
						Template+=Token->Length;
						Token=(TokenOverlayStruct *)Template;
					    }
					    goto NextToken;
					} else if (ReplyInt == TOKEN_MOVE_FORWARD) {
					    do {
						if (Token->TokenID != 0) {
						    Template += Token->Length;
						    Token = (TokenOverlayStruct *)Template;
						    continue;
						}
						break;
					    } while (Token->TokenID != GotoToken);
					    goto NextToken;
					} else if (ReplyInt == TOKEN_MOVE_BACKWARD) {
					    do {
						if (Token != (TokenOverlayStruct *)(Session->Pages[TemplateID]->Object)) {
						    Template -= Token->PrevSize;
						    Token = (TokenOverlayStruct *)Template;
						    continue;
						}
						break;
					    } while (Token->TokenID != GotoToken);
					}
				}

				MWDebug("Have unknown token %x\n", (unsigned int)Token->TokenID);
				break;
			}
		}
		Template+=Token->Length;
		Token=(TokenOverlayStruct *)Template;
	}
	return(TRUE);
}

static BOOL
ShouldDisplayFolder(SessionStruct *Session, unsigned long ID)
{
    if (Session->ShowUnsubscribedFolders) {
	/* we show everything */
	return(TRUE);
    }

    if (Session->FolderList->Value[ID][2] != '0') {
	/* this one is subscribed anyway */
	return(TRUE);
    }

    if ((ID + 1) < Session->FolderList->Used) {
	/* we still might need to display this folder if there are subfolders that are subscribed */
	unsigned char *originalFolder = Session->FolderList->Value[ID];
	unsigned long originalFolderLen = strlen(Session->FolderList->Value[ID]);
	unsigned char *currentFolder;
	unsigned long currentFolderLen;

	do {
	    if (++ID < Session->FolderList->Used) {
		currentFolder = Session->FolderList->Value[ID];
		currentFolderLen = strlen(currentFolder);
		
		if ((originalFolderLen < currentFolderLen) && 
		    MWQuickNCmp(originalFolder + 3, currentFolder + 3, originalFolderLen - 3) &&
		    (currentFolder[originalFolderLen] == '/')) {
		    if (currentFolder[2] != '0') {
			return(TRUE);
		    }

		    continue;
		}
	    }

	   break;
	} while (TRUE);
    }

    return(FALSE);
}

static BOOL
FeatureIsLicenced(ConnectionStruct *Client, SessionStruct *Session, int Feature)
{
	switch (Feature) {
		case AA_Complex_Rules:
		case AA_User_Proxy:
		case AA_Message_Priority:
		case AA_Shared_Folders:
		case AA_Busy_Search:
		case AA_Shared_Calendar:
		case AA_Shares:
		case AA_DelegateOption:
			return(TRUE);

		default:
			return(FALSE);
	}

	return(FALSE);

}


BOOL
MWHandleTemplate(ConnectionStruct *Client, SessionStruct *Session, unsigned long TemplateID)
{
	unsigned char				URL[256];
	unsigned char				Answer[BUFSIZE+1];
	unsigned long				ReplyInt;
	unsigned char				*Template;
	TokenOverlayStruct		*Token;
	register unsigned char	*ptr;

	if (!Session) {
		MWSendClient(Client, "HTTP/1.1 404 Document does not exist\r\nPragma: no-cache\r\nContent-type: text/html\r\nConnection: close\r\n", 100);
		Client->KeepAlive=FALSE;
		MWSendClient(Client, "<H1>Document not found</H1>", 27);

		return(TRUE);
	}

	MWDebug("HandleTemplate(Sesson:%x TemplateID:%d) called\n", (unsigned int)Session, (int)TemplateID);

	SendHTTPHeaderDynamic(Client, Session->Pages[TemplateID]->ContentType, Session->Pages[TemplateID]->ContentTypeLen);

	Template = Session->Pages[TemplateID]->Object;
	Token = (TokenOverlayStruct *)Template;

	if (Session->FolderListIsDirty || (Session->FolderList && Session->FolderList->Used == 0)) {
		if (UpdateFolderList(Session) == TRUE) {
			;/* common case */
		} else {
			/* This is a bad error because we lost NMAP connection */
			/* Destory Session ensures that bad NMAP session is not used */
			MWDestroySession(Session);
			Client->Session=NULL;

			/* EndClientConnection is easier to than sending a reasonable response */
			return(EndClientConnection(Client));
		}

		Session->FolderListIsDirty = FALSE;
	}

NextToken:
	while (Token->TokenID!=0) {
		switch(Token->TokenID) {
			case T_TEXT: {
				MWSendClient(Client, Token->Data+Token->ArgumentOffsetOrID[0], Token->ArgumentSize[0]);
				break;
			}

			case T_QSTRING:
			case T_STRING: {
				ptr=Session->Strings[Token->ArgumentOffsetOrID[0]];
				MWSendClient(Client, ptr, strlen(ptr));
				break;
			}

			case T_IMAGE: {
				MWEncodeURL(Session, URL, URL_TYPE_IMAGE, DISPLAY_IMAGE, Session->TemplateID, Session->Language, Token->ArgumentOffsetOrID[0], 0);
				MWSendClient(Client, URL, strlen(URL));
				break;
			}

			case T_LOGOUT: {
				MWEncodeURL(Session, URL, URL_TYPE_LINK, REQUEST_LOGOUT, Token->ArgumentOffsetOrID[0], 0, 0, 0);
				MWSendClient(Client, URL, strlen(URL));
				break;
			}

		    case T_VARIABLE: {
			if (Token->ArgumentOffsetOrID[0] == AA_GET) {
			    unsigned char *value;

			    value = GetUserVariable(Client, Token->ArgumentOffsetOrID[1]);
			    MWSendClient(Client, value, strlen(value));
			    break;
			}

			if (Token->ArgumentOffsetOrID[0] == AA_SET) {
			    SetUserVariable(
				Client, 
				Token->ArgumentOffsetOrID[1], 
				(unsigned char *)(Token->Data + Token->ArgumentOffsetOrID[2]), 
				Token->ArgumentSize[2]);
			    break;
			}
			break;
		    }

			case T_CONDITION: {
			    BOOL condition;
			    unsigned long jumpID;
			    
			    jumpID = Token->ArgumentOffsetOrID[0];

			    if (Token->ArgumentOffsetOrID[1] == AA_ENDIF) {
				break;
			    }

			    switch (Token->ArgumentOffsetOrID[2]) {		 
				case AA_VARIABLE: {
				    condition = MWQuickNCmp(GetUserVariable(Client, Token->ArgumentOffsetOrID[3]), Token->Data + Token->ArgumentOffsetOrID[4], Token->ArgumentSize[4]);
				    break;
				}

				case AA_HASUNREADMESSAGES: {
				    if (Session->CurrentDisplayFolder >= Session->FolderList->Used) {
					do {
					    if (Token->TokenID != 0) {
						Template += Token->Length;
						Token = (TokenOverlayStruct *)Template;
						continue;
					    }								
					    break;
					} while ((Token->TokenID != T_CONDITION) || (Token->ArgumentOffsetOrID[0] != jumpID));

					goto NextToken;
				    }

				    condition = Session->FolderUnreadCount[Session->CurrentDisplayFolder];
				    break;
				}

				case AA_ISSELECTEDFOLDER: {
				    if (Session->CurrentDisplayFolder < Session->FolderList->Used) {
					if (Session->CurrentFolder ==  Session->CurrentDisplayFolder) {
					    unsigned char *folder = Session->FolderList->Value[Session->CurrentDisplayFolder];
				
					    if (folder[0] == 'M') {

						condition = TRUE;
						break;
					    }			    
					}
				    }

				    condition = FALSE;
				    break;
				}

				case AA_ISSELECTEDCALENDAR: {
				    if (Session->CurrentDisplayFolder < Session->FolderList->Used) {
					if (Session->CurrentCalendar ==  Session->CurrentDisplayFolder) {
					    unsigned char *calendar = Session->FolderList->Value[Session->CurrentDisplayFolder];
				
					    if (calendar[0] == 'C') 
					    {
						condition = TRUE;
						break;
					    }			    
					}

				    }

				    condition = FALSE;
				    break;
				}

				default: {
				    do {
					if (Token->TokenID != 0) {
					    Template += Token->Length;
					    Token = (TokenOverlayStruct *)Template;
					    continue;
					}								
					break;
				    } while ((Token->TokenID != T_CONDITION) || (Token->ArgumentOffsetOrID[0] != jumpID));

				    goto NextToken;
				}
			    }

			    if (((Token->ArgumentOffsetOrID[1] == AA_IF) && !condition) || ((Token->ArgumentOffsetOrID[1] == AA_ELSE) && condition)) {
				do {
				    if (Token->TokenID != 0) {
					Template += Token->Length;
					Token = (TokenOverlayStruct *)Template;
					continue;
				    }								
				    break;
				} while ((Token->TokenID != T_CONDITION) || (Token->ArgumentOffsetOrID[0] != jumpID));

				goto NextToken;
			    }

			   break;			
			}

			case T_PAGEID:
			case T_PAGE: {
				MWEncodeURL(Session, URL, URL_TYPE_LINK, DISPLAY_TEMPLATE, Token->ArgumentOffsetOrID[0], 0, 0, 0);
				MWSendClient(Client, URL, strlen(URL));
				break;
			}

			case T_STATIC_OBJECT: {
				MWEncodeURL(Session, URL, URL_TYPE_STATIC, DISPLAY_STATIC_OBJECT, Session->TemplateID, Token->ArgumentOffsetOrID[0], 0, 0);
				MWSendClient(Client, URL, strlen(URL));
				break;
			}

			case T_LOGO: {
				if (Session->LogoID==0) {
					MWEncodeURL(Session, URL, URL_TYPE_IMAGE, DISPLAY_IMAGE, Session->TemplateID, Session->Language, Token->ArgumentOffsetOrID[0], 0);
				} else {
					MWEncodeURL(Session, URL, URL_TYPE_IMAGE, DISPLAY_LOGO, Session->LogoID, 0, 0, 0);
				}
				MWSendClient(Client, URL, strlen(URL));
				break;
			}

			case T_RELOGIN: {
				int				count;

				/* First, accquire a TSession */
				Client->TSession=CreateTSession();

				/* Now store all data in the session */
				snprintf(Answer, sizeof(Answer), "%s/T%lu", WorkDir, Client->TSession->SessionID);
				Client->PostData=fopen(Answer, "wb");
				fprintf(Client->PostData, "%lu\r\n", Client->ContentLength+Client->BufferPtr);
				fprintf(Client->PostData, "%c%s\r\n", Client->EncodingType==FORM_DATA_ENCODED ? 'f' : 'u', Client->FormSeparator ? (char *)Client->FormSeparator : "");
				fprintf(Client->PostData, "%s\r\n", Client->URL);

				if (Client->BufferPtr>0) {
					fwrite(Client->Buffer, 1, Client->BufferPtr, Client->PostData);
					Client->BufferPtr=0;
				}
				while (Client->ContentLength>0) {
					count=DoClientRead(Client, Client->Buffer+Client->BufferPtr, BUFSIZE-Client->BufferPtr, CONNECTION_TIMEOUT);
					if ((count<1) || (Exiting)) {
						return(EndClientConnection(Client));
					}
					Client->BufferPtr+=count;
					if ((unsigned long)Client->BufferPtr>Client->ContentLength) {
						fwrite(Client->Buffer, 1, Client->ContentLength, Client->PostData);
						memmove(Client->Buffer, Client->Buffer+Client->ContentLength, Client->BufferPtr-Client->ContentLength);
						Client->BufferPtr-=Client->ContentLength;
						Client->ContentLength=0;
					} else {
						fwrite(Client->Buffer, 1, Client->BufferPtr, Client->PostData);
						Client->ContentLength-=Client->BufferPtr;
						Client->BufferPtr=0;
					}
				}
				fclose(Client->PostData);
				Client->PostData=NULL;
				Client->TSession->Timestamp = ModwebTimeStamp;

				//MWEncodeURL(Session, URL, 'f', REQUEST_RELOGIN, (unsigned long)Client->TSession, 0, 0, 0);
				MWSendClient(Client, "f", 1);
				break;
			}

			case T_STOREDDATA: {
				if (Client->TSession) {
					unsigned long		i;

					MWEncodeURL(Session, URL, 'x', 0, Client->TSession->SessionID, Client->TSession->SessionUID, 0, 0);
					ReplyInt=snprintf(Answer, sizeof(Answer), "<INPUT TYPE=\"Hidden\" NAME=\"TSession\" VALUE=\"%s\">", URL);
					ReleaseTSession((unsigned int)Client->TSession);
					for (i = 0; i < 10; i++) {
						Client->SessionUID[i] = 0;
					}
					Client->TSession=NULL;
					MWSendClient(Client, Answer, ReplyInt);
				}
				break;
			}

			case T_APP_TITLE: {
				if (Session->Title) {
					MWSendClient(Client, Session->Title, strlen(Session->Title));
				} else if (DefaultTitle) {
					MWSendClient(Client, DefaultTitle, strlen(DefaultTitle));
				} else {
					MWSendClient(Client, Templates[Session->TemplateID]->Name, strlen(Templates[Session->TemplateID]->Name));
				}
				break;
			}


			case T_COLOR: {
				switch (Token->ArgumentOffsetOrID[0]) {
					case AA_PAGEFG: {
						MWSendClient(Client, Session->Colors[COLOR_PAGE_FG], 6);
						break;
					}

					case AA_PAGEBG: {
						MWSendClient(Client, Session->Colors[COLOR_PAGE_BG], 6);
						break;
					}

					case AA_BORDERFG: {
						MWSendClient(Client, Session->Colors[COLOR_BORDER_FG], 6);
						break;
					}

					case AA_BORDERBG: {
						MWSendClient(Client, Session->Colors[COLOR_BORDER_BG], 6);
						break;
					}

					case AA_SECTIONFG: {
						MWSendClient(Client, Session->Colors[COLOR_SECTION_FG], 6);
						break;
					}

					case AA_SECTIONBG: {
						MWSendClient(Client, Session->Colors[COLOR_SECTION_BG], 6);
						break;
					}

					case AA_FIELDNAMEFG: {
						MWSendClient(Client, Session->Colors[COLOR_FIELDNAME_FG], 6);
						break;
					}

					case AA_FIELDNAMEBG: {
						MWSendClient(Client, Session->Colors[COLOR_FIELDNAME_BG], 6);
						break;
					}

					case AA_FIELDBODYFG: {
						MWSendClient(Client, Session->Colors[COLOR_FIELDBODY_FG], 6);
						break;
					}

					case AA_FIELDBODYBG: {
						MWSendClient(Client, Session->Colors[COLOR_FIELDBODY_BG], 6);
						break;
					}
				}
				break;
			}

			case T_USERNAME: {
				MWSendClient(Client, Session->User, strlen(Session->User));
				break;
			}

			case T_FULLNAME: {
				MWSendClient(Client, Session->DisplayName, strlen(Session->DisplayName));
				break;
			}

			case T_UID: {
				MWSendClient(Client, Client->Temp, snprintf(Client->Temp, sizeof(Client->Temp), "%lu", Session->SessionUID));
				break;
			}

			case T_LANGUAGELIST: {
				unsigned long	*LanguageList=(unsigned long *)((unsigned char *)Templates[Session->TemplateID] +sizeof(TemplateStruct));
				unsigned long	i;

				for (i=0; i<Templates[Session->TemplateID]->LanguageCount; i++) {
					if (XplReturnLanguageName(LanguageList[i], URL)==0) {
						ReplyInt=snprintf(Answer, sizeof(Answer), "<OPTION VALUE=\"%d\"%s>%s</OPTION>\r\n", (int)LanguageList[i], (Session->Language==i) ? " SELECTED" : "", URL);
						MWSendClient(Client, Answer, ReplyInt);
					}
				}
				break;
			}

			case T_DATETIME_SETUP: {
				Session->DateFormat.monthShort=&Session->Strings[Token->ArgumentOffsetOrID[0]];
				Session->DateFormat.monthLong=&Session->Strings[Token->ArgumentOffsetOrID[1]];
				Session->DateFormat.wDayShort=&Session->Strings[Token->ArgumentOffsetOrID[2]];
				Session->DateFormat.wDayLong=&Session->Strings[Token->ArgumentOffsetOrID[3]];
				Session->DateFormat.AmPm=&Session->Strings[Token->ArgumentOffsetOrID[4]];
				break;
			}

			case T_SESS_OVERRIDE: {
				switch (Token->ArgumentOffsetOrID[0]) {
					case AA_LANGUAGE: {
						Session->Language = Token->ArgumentOffsetOrID[1];
						break;
					}	
					
					case AA_STARTWEEKDAY: {
						Session->StartWeekday = Token->ArgumentOffsetOrID[1];
						break;
					}	

					case AA_MESSAGESPERPAGE: {
						Session->MessagesPerPage = Token->ArgumentOffsetOrID[1];
						break;
					}	

					case AA_LOGOID: {
						Session->LogoID = Token->ArgumentOffsetOrID[1];
						break;
					}	

					case AA_CONNECTIONTIMEOUT: {
						Session->ConnectionTimeout = Token->ArgumentOffsetOrID[1];
						break;
					}	

					default: {
						break;
					}	
				}
				break;
			}

			case T_ASSIGN_FOLDER_ICON: {
				switch (Token->ArgumentOffsetOrID[0]) {
					case AA_MAILBOX:					Session->MailBoxIcon = Token->ArgumentOffsetOrID[1];
					case AA_SHAREDMAILBOX:			Session->MailBoxSharedIcon = Token->ArgumentOffsetOrID[1];
					case AA_SHARINGMAILBOX:			Session->MailBoxSharingIcon = Token->ArgumentOffsetOrID[1];
					case AA_CALENDAR:					Session->CalendarIcon = Token->ArgumentOffsetOrID[1];
					case AA_SHAREDCALENDAR:			Session->CalendarSharedIcon = Token->ArgumentOffsetOrID[1];
					case AA_SHARINGCALENDAR:		Session->CalendarSharingIcon = Token->ArgumentOffsetOrID[1];
					case AA_HIERARCHICALELEMENT:	Session->HierarchicalElementIcon = Token->ArgumentOffsetOrID[1];
					default:								break;
				}
				break;
			}

			case T_FOLDERLIST_REFRESH: {
				break;
			}

			case T_FOLDERLIST_BEGIN: {
				if (Token->ArgumentOffsetOrID[0] == AA_TRUE) {
					Session->ShowUnsubscribedFolders = TRUE;
				} else {
					Session->ShowUnsubscribedFolders = FALSE;
				}

				if (Session->FolderList->Used == 0 || Session->CurrentDisplayFolder > Session->FolderList->Used) {
					/* We don't have any folders, or we walked past the end so we just leave */

					while (Token->TokenID != T_FOLDERLIST_END && Token->TokenID != 0) {
						Template += Token->Length;
						Token = (TokenOverlayStruct *)Template;
					}
					goto NextToken;
				}
				break;
			}

			case T_FOLDERLIST_END: {
			    do {
				Session->CurrentDisplayFolder++;
				if (Session->CurrentDisplayFolder < Session->FolderList->Used) {
				    if (ShouldDisplayFolder(Session, Session->CurrentDisplayFolder)) {
					while (Token->TokenID != T_FOLDERLIST_BEGIN && Token->TokenID != 0) {
					    Template -= Token->PrevSize;
					    Token = (TokenOverlayStruct *)Template;
					}
					goto NextToken;
				    }

				    continue;
				}

				break;

			    } while(TRUE);

			    Session->CurrentDisplayFolder = 0;

			    break;
			}

			case T_FOLDERLIST_SELECT: {
				MWEncodeURL(Session, URL, URL_TYPE_LINK, REQUEST_SELECT_FOLDER, Session->CurrentDisplayFolder, Token->ArgumentOffsetOrID[0], Token->ArgumentOffsetOrID[1], Token->ArgumentOffsetOrID[2]);
				MWSendClient(Client, URL, strlen(URL));

				break;
			}

			case T_FOLDERLIST_VALUE: {
				if (Session->CurrentDisplayFolder < Session->FolderList->Used) {
					unsigned char		*Folder = Session->FolderList->Value[Session->CurrentDisplayFolder];

					switch (Token->ArgumentOffsetOrID[0]) {
						case AA_FOLDERDISPLAYNAME: {

							Folder = Session->FolderDisplayNames->Value[Session->CurrentDisplayFolder];
							MWHTMLSendClient(Client, Folder, strlen(Folder));
							break;
						}
						case AA_FOLDERCURRENTID: {
							ReplyInt = snprintf(Client->Temp, sizeof(Client->Temp), "%lu", Session->CurrentDisplayFolder + 1);
							MWSendClient(Client, Client->Temp, ReplyInt);
							break;
						}
						case AA_FOLDERTYPE: {
							/* FolderType (Arg2 = Mailbox string, Arg3 = Calendar string, Arg4 = Hierarchical Element string) */
							switch (toupper(*Folder)) {
								case 'M': {		/* Mailbox */
									MWSendClient(Client, Token->Data + Token->ArgumentOffsetOrID[1], Token->ArgumentSize[1]);
									break;
								}
								case 'C': {		/* Calendar */
									MWSendClient(Client, Token->Data + Token->ArgumentOffsetOrID[2], Token->ArgumentSize[2]);
									break;
								}
								default: {		/* Hierarchical Element */
									MWSendClient(Client, Token->Data + Token->ArgumentOffsetOrID[3], Token->ArgumentSize[3]);
									break;
								}
							}
							
							break;
						}
						case AA_IMAGE: {
							/* Image (Images are set with AssignFolderImage) */

							switch (toupper(*Folder)) {
								case 'M': {																																										/* Mailbox					*/
									if (Folder[2] == '1' || Session->ShowUnsubscribedFolders) {																									/*		Subscribed			*/
										if (Folder[1] == '0') {																																				/*		Not shared			*/
											MWEncodeURL(Session, URL, URL_TYPE_IMAGE, DISPLAY_IMAGE, Session->TemplateID, Session->Language, Session->MailBoxIcon, 0);
										} else if (Folder[1] == '1') {																																	/*		I'm sharing			*/
											MWEncodeURL(Session, URL, URL_TYPE_IMAGE, DISPLAY_IMAGE, Session->TemplateID, Session->Language, Session->MailBoxSharingIcon, 0);
										} else {																																									/*		Shared with me		*/
											MWEncodeURL(Session, URL, URL_TYPE_IMAGE, DISPLAY_IMAGE, Session->TemplateID, Session->Language, Session->MailBoxSharedIcon, 0);
										}
									} else {																																										/*		Unsubscribed		*/
										MWEncodeURL(Session, URL, URL_TYPE_IMAGE, DISPLAY_IMAGE, Session->TemplateID, Session->Language, Session->HierarchicalElementIcon, 0);
									}
									break;
								}
								case 'C': {																																										/* Calendar					*/
									if (Folder[2] == '1' || Session->ShowUnsubscribedFolders) {																									/*		Subscribed			*/
										if (Folder[1] == '0') {																																				/*		Not shared			*/
											MWEncodeURL(Session, URL, URL_TYPE_IMAGE, DISPLAY_IMAGE, Session->TemplateID, Session->Language, Session->CalendarIcon, 0);
										} else if (Folder[1] == '1') {																																	/*		I'm sharing			*/
											MWEncodeURL(Session, URL, URL_TYPE_IMAGE, DISPLAY_IMAGE, Session->TemplateID, Session->Language, Session->CalendarSharingIcon, 0);
										} else {																																									/*		Shared with me		*/
											MWEncodeURL(Session, URL, URL_TYPE_IMAGE, DISPLAY_IMAGE, Session->TemplateID, Session->Language, Session->CalendarSharedIcon, 0);
										}
									} else {																																										/*		Unsubscribed		*/
										MWEncodeURL(Session, URL, URL_TYPE_IMAGE, DISPLAY_IMAGE, Session->TemplateID, Session->Language, Session->HierarchicalElementIcon, 0);
									}
									break;
								}
								default: {																																										/* Hierarchical Element */
									MWEncodeURL(Session, URL, URL_TYPE_IMAGE, DISPLAY_IMAGE, Session->TemplateID, Session->Language, Session->HierarchicalElementIcon, 0);
									break;
								}
							}
							MWSendClient(Client, URL, strlen(URL));

							break;
						}

					    case AA_PREFIX: {
						unsigned long prevDepth;
						unsigned long curDepth;
						long prevIndex;
						
						if (Session->CurrentDisplayFolder > 0) {
						    prevIndex = Session->CurrentDisplayFolder - 1;
						    do {
							if (Session->FolderList->Value[prevIndex][2] != 0) {
							    unsigned char *prevFolder = Session->FolderList->Value[prevIndex];
						    
							    prevDepth = 1;

							    do {
								prevFolder = strchr(++prevFolder, '/');
								if (prevFolder) {
								    prevDepth++;
								    continue;
								}
								break;
							    } while (TRUE);

							    break;
							}
							
							prevIndex--;
							if (prevIndex > -1) {
							    continue;
							}
							prevDepth = 0;
							break;
						    } while (TRUE);
						} else {
						    prevDepth = 0;
						}

						curDepth = 1;
						do {
						    Folder = strchr(++Folder, '/');
						    if (Folder) {
							curDepth++;
							continue;
						    }
						    break;
						} while (TRUE);
						
						if (prevDepth < curDepth) {
						    MWSendClient(Client, Token->Data + Token->ArgumentOffsetOrID[1], Token->ArgumentSize[1]);
						}

						MWSendClient(Client, Token->Data + Token->ArgumentOffsetOrID[2], Token->ArgumentSize[2]);
						break;
					    }

					    case AA_POSTFIX: {
						unsigned long nextDepth;
						unsigned long curDepth;
						long nextIndex;
						
						nextIndex = Session->CurrentDisplayFolder + 1;
						if (nextIndex < Session->FolderList->Used) {
						    do {
							if (ShouldDisplayFolder(Session, nextIndex)) {
							    unsigned char *nextFolder = Session->FolderList->Value[nextIndex];
						    
							    nextDepth = 1;

							    do {
								nextFolder = strchr(++nextFolder, '/');
								if (nextFolder) {
								    nextDepth++;
								    continue;
								}
								break;
							    } while (TRUE);

							    break;
							}
							
							nextIndex++;
							if (nextIndex < Session->FolderList->Used) {
							    continue;
							}

							nextDepth = 0;
							break;

						    } while (TRUE);

						} else {
						    nextDepth = 0;
						}

						curDepth = 1;
						do {
						    Folder = strchr(++Folder, '/');
						    if (Folder) {
							curDepth++;
							continue;
						    }
						    break;
						} while (TRUE);
						
						if (nextDepth > curDepth) {
						    break;
						}

						MWSendClient(Client, Token->Data + Token->ArgumentOffsetOrID[1], Token->ArgumentSize[1]);

						while (nextDepth < curDepth) {
						    MWSendClient(Client, Token->Data + Token->ArgumentOffsetOrID[2], Token->ArgumentSize[2]);

						    if (curDepth > 1) {
							MWSendClient(Client, Token->Data + Token->ArgumentOffsetOrID[1], Token->ArgumentSize[1]);
						    }
						    curDepth--;
						}

						break;
					    }

						case AA_INDENT: {
							/* Indent Arg2 = Indention string, sent once for each level of indention */

							do {
								Folder = strchr(++Folder, '/');
								if (Folder) {
									MWSendClient(Client, Token->Data + Token->ArgumentOffsetOrID[1], Token->ArgumentSize[1]);
								}
							} while (Folder);

							break;
						}

						case AA_SUBSCRIBE: {
							/* Subscribed, Arg2 = Subscribed String, Arg3 = Unsubscribed String */

							if (Folder[2] == '1') {
								MWSendClient(Client, Token->Data + Token->ArgumentOffsetOrID[1], Token->ArgumentSize[1]);
							} else {
								MWSendClient(Client, Token->Data + Token->ArgumentOffsetOrID[2], Token->ArgumentSize[2]);
							}
							break;
						}
						case AA_SHARE: {
							/* Share, Arg2 = Not Shared String, Arg3 = Shared String, Arg4 = Sharing String */

							switch(Folder[1]) {
								case '0': {
									MWSendClient(Client, Token->Data + Token->ArgumentOffsetOrID[1], Token->ArgumentSize[1]);
									break;
								}
								case '1':
                                case '3': {
									MWSendClient(Client, Token->Data + Token->ArgumentOffsetOrID[2], Token->ArgumentSize[2]);
									break;
								}
								case '2': {
									MWSendClient(Client, Token->Data + Token->ArgumentOffsetOrID[3], Token->ArgumentSize[3]);
									break;
                                }
							}
							break;
						}
						case AA_FOLDERFULLNAME: {
							MWSendClient(Client, Folder + 3, strlen(Folder + 3));
							break;
						}
						case AA_HASUNREAD: {
							if (Session->FolderUnreadCount[Session->CurrentDisplayFolder] > 0) {
								MWSendClient(Client, Token->Data + Token->ArgumentOffsetOrID[1], Token->ArgumentSize[1]);
							} else {
								MWSendClient(Client, Token->Data + Token->ArgumentOffsetOrID[2], Token->ArgumentSize[2]);
							}
							break;
						}
						case AA_UNREADCOUNT: {
							ReplyInt = snprintf(Answer, sizeof(Answer), "%lu", Session->FolderUnreadCount[Session->CurrentDisplayFolder]);
							MWSendClient(Client, Answer, ReplyInt);
							break;
						}
						case AA_TOTALCOUNT: {
							ReplyInt = snprintf(Answer, sizeof(Answer), "%lu", Session->FolderTotalCount[Session->CurrentDisplayFolder]);
							MWSendClient(Client, Answer, ReplyInt);
							break;
						}
					}
				}
				break;
			}
			case T_LANGUAGE_BEGIN: {
				unsigned long	*LanguageList=(unsigned long *)((unsigned char *)Templates[Session->TemplateID] +sizeof(TemplateStruct));

				if (Token->ArgumentOffsetOrID[0] != LanguageList[Session->Language]) {
					unsigned long	GotoToken;

					GotoToken = T_LANGUAGE_END;

					while (Token->TokenID != GotoToken && Token->TokenID != 0) {
						Template += Token->Length;
						Token = (TokenOverlayStruct *)Template;
					}

					goto NextToken;
				}

				break;
			}

			case T_LANGUAGE_END: {
					break;
			}


			case T_SHOWFEATURE_BEGIN: {
				/* Based on licenced feature either ignore or move to else token */
				if (FeatureIsLicenced(Client, Session, Token->ArgumentOffsetOrID[0]) == TRUE) {
					/* FeatureIsLicenced for this user we are done */
					break;
				} else {
					/* This feature is not licenced */
					/* Move to the end token */
					unsigned long	GotoToken;

					GotoToken = T_SHOWFEATURE_ELSE;

					while (Token->TokenID != GotoToken && Token->TokenID != 0) {
						Template += Token->Length;
						Token = (TokenOverlayStruct *)Template;
					}

					goto NextToken;

				}
					
					
				break;
			}

			case T_SHOWFEATURE_ELSE: {
				/* Based on licenced feature either ignore or move to end token */
				if (!FeatureIsLicenced(Client, Session, Token->ArgumentOffsetOrID[0]) == TRUE) {
					/* FeatureIsLicenced for this user we are done */
					break;
				} else {
					/* This feature is not licenced */
					/* Move to the end token */
					unsigned long	GotoToken;

					GotoToken = T_SHOWFEATURE_END;

					while (Token->TokenID != GotoToken && Token->TokenID != 0) {
						Template += Token->Length;
						Token = (TokenOverlayStruct *)Template;
					}

					goto NextToken;

				}
					
					
				break;
			}



			case T_SHOWFEATURE_END: {
				break;
			}


			default: {
				unsigned long	i;
				unsigned long	GotoToken;
				if (Session->ModuleData) {
					for (i = 0; i < TModuleCount; i++) {
						if (TModules[i].TokenRangeStart<=Token->TokenID && Token->TokenID<=TModules[i].TokenRangeEnd) {
							ReplyInt = TModules[i].HandleTemplate(Client, Session, TemplateID, Token, &GotoToken, Session->ModuleData[i]);

							if (ReplyInt == TOKEN_DONE) {
							    if (Token->TokenID != 0) {
								Template += Token->Length;
								Token = (TokenOverlayStruct *)Template;
							    }
							    goto NextToken;
							} else if (ReplyInt == TOKEN_MOVE_FORWARD) {
							    do {
								if (Token->TokenID != 0) {
								    Template += Token->Length;
								    Token = (TokenOverlayStruct *)Template;
								    continue;
								}								
								break;
							    } while (Token->TokenID != GotoToken);

							    goto NextToken;

							} else if (ReplyInt == TOKEN_MOVE_BACKWARD) {
							    do {
								if (Token != (TokenOverlayStruct *)(Session->Pages[TemplateID]->Object)) {
								    Template -= Token->PrevSize;
								    Token = (TokenOverlayStruct *)Template;
								    continue;
								}								
								break;
							    } while (Token->TokenID != GotoToken);

							    goto NextToken;
							}
						}
					}
				}

				MWDebug("Have unknown token %x\n", (unsigned int)Token->TokenID);
				break;
			}
		}

		Template += Token->Length;
		Token = (TokenOverlayStruct *)Template;
	}

	return(TRUE);
}


#if 0
static BOOL
ProcessFolderList(ConnectionStruct *Client, SessionStruct *Session)
{
	unsigned long	ValueSize;
	unsigned char	Type[128]="", Filename[128]="";

	while (MWGetFormNameEx(Client, Client->Temp, Type, Filename, BUFSIZE)) {
		TemplateDebug("MWGetFormName name >%s<, filename:>%s<, type:>%s<\n", Client->Temp, Filename, Type);
		ValueSize=BUFSIZE;
		while (MWGetFormValue(Client, Client->Temp, &ValueSize)) {
			TemplateDebug("Got field value:%d\n", (int)ValueSize);
			ValueSize=BUFSIZE;
		}
	}
	
	return(TRUE);
}
#endif

BOOL
HandleURL(ConnectionStruct *Client, SessionStruct *Session, URLStruct *URL)
{
 	MWDebug("HandleURL(Sesson:%x URL->Request:%d) called\n", (unsigned int)Session, (int)URL->Request);

	switch(URL->Request) {
		case DISPLAY_TEMPLATE: {
			Client->KeepAlive=FALSE;
			MWHandleTemplate(Client, Session, URL->Argument[0]);
			return(TRUE);
		}

		case DISPLAY_IMAGE: {
			HandleImage(Client, URL->Argument[0], URL->Argument[1], URL->Argument[2]);
			return(TRUE);
		}

	    case DISPLAY_STATIC_OBJECT: {
        if ((URL->Argument[0] < TemplateCount) 
                && (URL->Argument[1] < Templates[URL->Argument[0]]->CommonStaticObjectCount)) {
            if (Session) {
    			TemplateHandleStaticObject(Client, Session->StaticObjects[URL->Argument[1]]);
            } else {
                TemplateObjectOverlay **StaticObjects;

                StaticObjects = (TemplateObjectOverlay **)((unsigned char *)Templates[URL->Argument[0]] + 
				sizeof(TemplateStruct) + 
				sizeof(unsigned long)*Templates[URL->Argument[0]]->LanguageCount+ 
				(
					sizeof(unsigned char *) *
					(
						Templates[URL->Argument[0]]->StringCount+
						Templates[URL->Argument[0]]->ImageCount
					)*
					Templates[URL->Argument[0]]->LanguageCount
				) +
				sizeof(unsigned char *) * Templates[URL->Argument[0]]->NameCount +
				sizeof(unsigned char *) * Templates[URL->Argument[0]]->TemplateCount );

    			TemplateHandleStaticObject(Client, StaticObjects[URL->Argument[1]]);

    			return(TRUE);
		    }
		}

		Client->KeepAlive=FALSE;
		return(FALSE);
	    }

		case DISPLAY_LOGO: {
			HandleLogo(Client, URL->Argument[0]);
			return(TRUE);
		}

		case REQUEST_LOGOUT: {
			MWDebug("Have logout request\n");
			Client->ClearCookie = TRUE;
			Client->KeepAlive = FALSE;
			if (!UseFormLogoutRedirectURL) {
				MWHandleTemplate(Client, Session, URL->Argument[0]);
			} else {
				MWRedirectClient(Client, FormLogoutRedirectURL);
			}
			MWDestroySession(Session);
			Client->Session=NULL;
			return(TRUE);
		}

		case REQUEST_LOGOUT_REDIR: {
			MWDebug("Have logout redirect request\n");
			MWRedirectClient(Client, Session->Strings[URL->Argument[0]]);
			MWDestroySession(Session);
			Client->Session=NULL;
			return(TRUE);
		}

		case REQUEST_RELOGIN: {
			MWDebug("Have relogin request\n");
			return(TRUE);
		}

		case REQUEST_PROXY_SWITCH: {
			unsigned char		Answer[BUFSIZE+1];
			BOOL					Disabled				= TRUE;
			BOOL					IsResource			= FALSE;
			SessionStruct		*OriginalSession	= Session;
			unsigned long		CookieID				= Client->CookieID;
			unsigned long		i;

			MWURLExtraDecode(Client, Client->Temp, sizeof(Client->Temp));

			/*
				Proxy

				We now need to create a new session, and set a cookie for it.  The cookie has to have a
				different name than the original, so that we can be logged in to both at the same time.

				In order to really keep everything seperate we also need to add the user name to all
				href targets.  This should be enough to allow both to work at once.
			*/

			MWSendNMAPServer(Session, "SHOWPROXY\r\n", 11);
			if (MWGetNMAPAnswer(Session, Answer, BUFSIZE, TRUE) == 2002) {
				while ((i = MWGetNMAPAnswer(Session, Answer, BUFSIZE, TRUE)) != 1000) {
					unsigned char		*ptr;

					ptr = strrchr(Answer, ' ');
					if (ptr) {
						*ptr = '\0';
					}
					if (MWQuickCmp(Answer, Client->Temp)) {
						Disabled = FALSE;
					}
				}
			}

			/* Is it a resource? */
			if (Disabled == TRUE) {
				unsigned char		Class[MDB_MAX_ATTRIBUTE_CHARS + 1];
				MDBValueStruct		*config;

				config = MDBCreateValueStruct(ModWebDirectoryHandle, NULL);

				if (MsgFindObject(Client->Temp, Answer, Class, NULL, config)) {

					if (MDBReadDN(Session->UserDN, MSGSRV_A_MANAGED_RESOURCES, config) > 0) {
						for (i = 0; i < config->Used; i++) {
							if (MWQuickCmp(Answer, config->Value[i])) {
								MDBFreeValues(config);

								/*
									Alright it looks like it matches, Lets check to see if we
									are on the resource as well.
								*/
								if (MDBReadDN(Answer, MSGSRV_A_RESOURCE_MANAGER, config)) {
									for (i = 0; i < config->Used; i++) {
										if (MWQuickCmp(Session->UserDN, config->Value[i])) {
											Disabled = FALSE;
											IsResource = TRUE;
										}
									}
								}
							}
						}
					}
				}
				MDBDestroyValueStruct(config);
			}

			/* This is NEEDED with all the switching going on */
			Client->KeepAlive = FALSE;
			if (!Disabled && CreateSession(Client, Client->Temp, NULL, &Disabled)) {
				unsigned char			buffer[BUFSIZE+1];

				/*
					Put some of our own settings into the new session.
				*/
				Client->Session->Language = OriginalSession->Language;
				Client->Session->TemplateID = OriginalSession->TemplateID;
				Client->Session->Strings = OriginalSession->Strings;
				Client->Session->Images = OriginalSession->Images;
				Client->Session->Pages = OriginalSession->Pages;
				Client->Session->StaticObjects = OriginalSession->StaticObjects;
				Client->Session->RealUser = MemStrdup(OriginalSession->User);

				memcpy(Client->Session->Colors, OriginalSession->Colors, sizeof(Client->Session->Colors));

				/*
					Display the default template for the new session.  We need to release
					the semaphore from the original session first.
				*/
				XplSignalLocalSemaphore(SessionMutex[OriginalSession->SessionID]);

				Client->Cookie = Client->Session->SessionUID;
				Client->Session->CookieID = CookieID;

				/*
					We need to figure out what our own rights are.  Since we have the new user's session now
					we have a connection to NMAP, so we can send a PROXY command and find ourselfs in the
					list.  That will give us the rights.
				*/

				if (!IsResource) {
					Client->Session->ReadOnly = TRUE;

					MWSendNMAPServer(Client->Session, "PROXY\r\n", 7);
					if (MWGetNMAPAnswer(Client->Session, buffer, sizeof(buffer), TRUE) == 2002) {
						while ((MWGetNMAPAnswer(Client->Session, buffer, BUFSIZE, TRUE)) != 1000) {
							unsigned long			len = strlen(Session->User);

							if (MWQuickNCmp(buffer, Session->User, len) && buffer[len] == ' ') {
								/* We found ourselfs */

								if (buffer[len + 1] == '2') {
									Client->Session->ReadOnly = FALSE;
								}
							}
						}
					}
				} else {
					/* Resources are always read-write */

					Client->Session->ReadOnly = FALSE;
				}

				MWHandleTemplate(Client, Client->Session, Templates[Client->Session->TemplateID]->InitialTemplate);
			} else {
				/*
					Failed to proxy to the other user for whatever reason, so we'll just display
					the failure page.
				*/
				Session = OriginalSession;
				Client->Session = OriginalSession;

				MWHandleTemplate(Client, Session, URL->Argument[0]);
			}

			return(TRUE);
		}

		case REQUEST_SELECT_FOLDER: {
			unsigned long		folderID = URL->Argument[0];
			unsigned long		Page = URL->Argument[1];

			if (folderID < Session->FolderList->Used) {
				unsigned char		*folder = Session->FolderList->Value[folderID];

				switch (*folder) {
					case 'M': {
						Session->CurrentFolder = folderID;

						/* Is this the sent folder? */
						if (Session->SentFolder && MWQuickCmp(folder + 3, Session->SentFolder)) {
							Page = URL->Argument[2];
						}
						break;
					}

					case 'C': {
						Session->CurrentCalendar = folderID;

						Page = URL->Argument[3];
						break;
					}
				}
			}
			MWHandleTemplate(Client, Session, Page);
		}

		default: {
			unsigned long	i;

			Client->KeepAlive=FALSE;
			for (i=0; i<TModuleCount; i++) {
				if (TModules[i].TokenRangeStart<=URL->Request && URL->Request<=TModules[i].TokenRangeEnd) {
					if (TModules[i].HandleURL(Client, Session, URL, Session->ModuleData[i])) {
						break;
					}
				}
			}

			if (i==TModuleCount) { MWDebug("HandleURL():Unknown URL request type:%d\n", (int)URL->Request); }
			break;
		}
	}
	return(TRUE);
}

extern unsigned int NamedURLIDs[];

BOOL
MWHandleNamedURL(ConnectionStruct *Client, SessionStruct *Session, URLStruct *URL)
{
	unsigned long	i;


 	MWDebug("MWHandleURL(Sesson:%x URL->Request:%d) called\n", (unsigned int)Session, (int)URL->Request);

	/* validate NamedURL falls within Range */
	if ( (URL->Request < NM_MAX_NAMED_URLIDS) && NamedURLIDs[URL->Request]) {
		/* translate Named ID to URL - ID  see MWRegisterURLs */
		URL->Request = NamedURLIDs[URL->Request];
	} else { 
		return(FALSE);
	}


	/* Find correct module */
	Client->KeepAlive=FALSE;

	for (i=0; i<TModuleCount; i++) {
		if (TModules[i].TokenRangeStart<=URL->Request && URL->Request<=TModules[i].TokenRangeEnd) {
			if (TModules[i].HandleURL(Client, Session, URL, Session->ModuleData[i])) {
				return(TRUE);
			}			
		}
	}

	MWDebug("MWHandleNamedURL():Unknown URL request type:%d\n", (int)URL->Request);
	return(FALSE);

}

#define SEQUENCE_ALLOC_BLOCK_SIZE 50
#define ICS_BEGIN_STRING "BEGIN:VCALENDAR\r\nCALSCALE:GREGORIAN\r\nPRODID:-//Hula//NONSGML Hula Calendar\r\nMETHOD:PUBLISH\r\nVERSION:2.0\r\n"
#define ICS_END_STRING "END:VCALENDAR\r\n"

#define COMP_START 0
#define COMP_GOTBEGIN 1
#define COMP_GOTEND 2
BOOL
MWSendIcs(ConnectionStruct *Client, SessionStruct *Session)
{
    unsigned char buffer[BUFSIZE + 1];
    unsigned char *compbufptr;
    unsigned long *sequenceList = NULL;
    unsigned long sequenceListAllocSize;
    unsigned long sequenceListSize;
    unsigned long i;
    unsigned long objectSize = 0;
    unsigned long bufFullSize = 0;
    unsigned long lenOfPartialLine = 0;
    unsigned int progress = COMP_START;

    sequenceList = (unsigned long *)MemMalloc(sizeof(unsigned long) * SEQUENCE_ALLOC_BLOCK_SIZE);
    if (sequenceList) {
	sequenceListAllocSize = SEQUENCE_ALLOC_BLOCK_SIZE;
	sequenceListSize = 0;

	MWSendNMAPServer(Session, buffer, snprintf(buffer, sizeof(buffer), "CSFILT %lu %lu\r\n", time(NULL) + Session->TimezoneOffset, (unsigned long)0xffffffff));
	while (MWGetNMAPAnswer(Session, buffer, sizeof(buffer), TRUE) == 2002) {
	    unsigned long seqNumber;
	    unsigned long Type;

	    sscanf(buffer, "%lu %lu %*u %*u %*u %*u %*u", &seqNumber, &Type);

	    /* Type 1 == Appointment.  We don't care about tasks and notes */
	    if (Type == 1) {
		unsigned long *tmp;

		if (sequenceListSize < sequenceListAllocSize) {
		    sequenceList[sequenceListSize] = seqNumber;
		    sequenceListSize++;
		    continue;
		}

		tmp = MemRealloc(sequenceList, (sequenceListAllocSize + SEQUENCE_ALLOC_BLOCK_SIZE) * sizeof(unsigned long));
		if (tmp) {
		    sequenceList = tmp;
		    sequenceList[sequenceListSize] = seqNumber;
		    sequenceListSize++;
		    sequenceListAllocSize += SEQUENCE_ALLOC_BLOCK_SIZE;
		    continue;
		}
	    }
	}

	snprintf(buffer, BUFSIZE, "%s.ics", Session->User);
	MWSendHTTPHeader(Client, "text/calendar", 0, buffer, FALSE); 

	MWSendClient(Client, ICS_BEGIN_STRING, strlen(ICS_BEGIN_STRING));	

	for (i = 0; i < sequenceListSize; i++) {
	    MWSendNMAPServer(Session, buffer, snprintf(buffer, sizeof(buffer), "CSLIST %lu ICAL\r\n", sequenceList[i]));
	    if (MWGetNMAPAnswer(Session, buffer, sizeof(buffer), TRUE) == 2023) {
		objectSize = atol(buffer);
		progress = COMP_START;
		lenOfPartialLine = 0;

		if (Session->NBufferPtr > 0) {
		    if ((unsigned int)Session->NBufferPtr >= objectSize) {
			/* We already have it in our buffer, so just send it */
			memcpy (buffer, Session->NBuffer, objectSize);
			buffer[objectSize] = '\0';
			if ((compbufptr = strstr (buffer, "BEGIN:VEVENT")) != NULL) {
			    char *endptr;

			    if ((endptr = strstr (compbufptr, "END:VEVENT")) != NULL) {
				*(endptr + 12) = '\0';
			    }
			    MWSendClient (Client, compbufptr, strlen (compbufptr));
			}

			/* Preserve data in the buffer */
			Session->NBufferPtr = strlen(Session->NBuffer + objectSize);
			memmove(Session->NBuffer, Session->NBuffer + objectSize, Session->NBufferPtr + objectSize);

			if (MWGetNMAPAnswer(Session, buffer, sizeof(buffer), TRUE) == 1000) {
			    continue;
			}

			break;
		    }

		    /* We have part of it in our buffer */
		    memcpy (buffer, Session->NBuffer, Session->NBufferPtr);
		    buffer[Session->NBufferPtr] = '\0';
		    if ((compbufptr = strstr (buffer, "BEGIN:VEVENT")) != NULL) {
			char *endptr;
			char *newline = NULL;

			progress = COMP_GOTBEGIN;
			if ((endptr = strstr (compbufptr, "END:VEVENT")) != NULL) {
			    progress = COMP_GOTEND;
			    *(endptr + 12) = '\0';
			} else {
			    newline = strrchr (compbufptr, '\n');
			    newline++;
			    lenOfPartialLine = strlen (newline);
			}
			MWSendClient (Client, compbufptr, strlen (compbufptr) - lenOfPartialLine);
			if (lenOfPartialLine != 0) {
			    memmove (buffer, newline, strlen (newline) + 1);
			}
		    } else {
			bufFullSize = Session->NBufferPtr;
		    }
		    objectSize -= Session->NBufferPtr;
		    Session->NBufferPtr = 0;
		    Session->NBuffer[0] = '\0';
		}

		do {
		    unsigned long len;
		    char *newline = NULL;

		    if (objectSize < sizeof(buffer) - (bufFullSize + 1) - lenOfPartialLine) {
			len = MWDirectNMAPRead(Session, buffer + bufFullSize + lenOfPartialLine, objectSize);
		    } else {
			len = MWDirectNMAPRead(Session, buffer + bufFullSize + lenOfPartialLine, sizeof(buffer) - (bufFullSize + 1) - lenOfPartialLine);
		    }

		    if (len > 0) {
			if (COMP_GOTEND != progress) {
			    char *endptr;

			    buffer[bufFullSize + len + lenOfPartialLine] = '\0';
			    lenOfPartialLine = 0;
			    compbufptr = buffer;
			    if (COMP_GOTBEGIN != progress) {
				compbufptr = strstr (buffer, "BEGIN:VEVENT");
				if (NULL == compbufptr) {
				    bufFullSize += len;
				    objectSize -= len;
				    continue;
				}
				progress = COMP_GOTBEGIN;
			    }
			    if ((endptr = strstr (compbufptr, "END:VEVENT")) != NULL) {
				progress = COMP_GOTEND;
				*(endptr + 12) = '\0';
			    } else {
				newline = strrchr (compbufptr, '\n');
				newline++;
				lenOfPartialLine = strlen (newline);
			    }
			    MWSendClient (Client, compbufptr, strlen (compbufptr) - lenOfPartialLine);
			    if (lenOfPartialLine != 0) {
				memmove (buffer, newline, strlen (newline) + 1);
			    }
			    bufFullSize = 0;
			}
			    
			objectSize -= len;
			continue;
		    }
		    
		    break;
		} while (objectSize > 0);

		if (MWGetNMAPAnswer(Session, buffer, sizeof(buffer), TRUE) == 1000) {
		    continue;
		}

		break;
	    }
	    
	    break;
	}

	MWSendClient(Client, ICS_END_STRING, strlen(ICS_END_STRING));
	MemFree(sequenceList);
    }

    return(TRUE);
}


