/****************************************************************************
 *
 * Copyright (c) 2001-2002 Novell, Inc.
 * All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of version 2.1 of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, contact Novell, Inc.
 *
 * To contact Novell about this file by physical or electronic mail,
 * you may find current contact information at www.novell.com
 *
 ****************************************************************************/

#include <config.h>
#include <limits.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <time.h>
#include <xpl.h>
#include <connio.h>
#include <msgapi.h>
#include <hulautil.h>
#include "imapd.h"

/*------------------------------------------------------------------------------ */
/*							Internal Definitions														*/
/*------------------------------------------------------------------------------ */
#define NUL              0
#define CARRIAGE_RETURN 13
#define LINE_FEED       10
#define SPACE           32
#define QUOTE           34
#define ALL_BITS_SET    0xff
#define ALL_BITS_CLEAR  0x00
#define BY_POSITION     1
#define BY_UID          2 


#define ELEMENTSIZE (1/*4*/) /*sizeof( unsigned long)) */
#define BITCOUNT (8 * ELEMENTSIZE)
#define BVGET( bitvector, message) (bitvector[message / BITCOUNT] &  (0x80/*000000*/ /*(1 << BITCOUNT)*/ >> (message % BITCOUNT)))
#define BVSET( bitvector, message) (bitvector[message / BITCOUNT] |= (0x80 >> (message % BITCOUNT)))
const int DATE_MONTH_STR_LEN = 3;

unsigned char	*IMSearchVersion = "$Revision: 1.1.1.1 $";

/*----------------------------------------------------------------------------- */
/*							Internal Types and Structures */
/*----------------------------------------------------------------------------- */
typedef enum {
	string, ident, keyword, symbol
} EMIT;

enum {
	DATE_DAY = 0, DATE_MONTH, DATE_YEAR
};

/*typedef unsigned long BITVECTOR; */
typedef unsigned char BITVECTOR;


/*----------------------------------------------------------------------------- */
/*							Internal Function Prototypes */
/*----------------------------------------------------------------------------- */
BITVECTOR *command( IMAPSEARCH *Search);
BITVECTOR *uid( IMAPSEARCH *Search);
BITVECTOR *search( IMAPSEARCH *Search);
BITVECTOR *searchAND( IMAPSEARCH *Search);
BITVECTOR *search_key( IMAPSEARCH *Search);
BITVECTOR *set( IMAPSEARCH *Search, BITVECTOR *bv);
BITVECTOR *sequence_num( IMAPSEARCH *Search, BITVECTOR *bv);
void       astring( IMAPSEARCH *Search);
void       header_fld_name( IMAPSEARCH *Search);
void       number( IMAPSEARCH *Search);
void       date( IMAPSEARCH *Search);
void       InSymbol( IMAPSEARCH *Search);
void       TestTerminal( IMAPSEARCH *Search, SYMBOL sy);
void       NextCh( IMAPSEARCH *Search);
void       Emit( IMAPSEARCH *Search, EMIT emit, char *text);
BITVECTOR *BVEmpty( BITVECTOR *bv, unsigned long msgCount, int initTo);
BITVECTOR *BVIntersect( BITVECTOR *bv1, BITVECTOR *bv2, unsigned long byteSize);
BITVECTOR *BVInvert( BITVECTOR *bv, unsigned long byteSize);
BITVECTOR *BVUnion( BITVECTOR *bv1, BITVECTOR *bv2, unsigned long byteSize);
void       ImapSearchInit( void);
BITVECTOR *ImapSearchQuery( IMAPSEARCH *Search);
/*BITVECTOR *ImapSearchQuery2( IMAPSEARCH *Search); */

/*----------------------------------------------------------------------------- */
/*							Internal Globals */
/*----------------------------------------------------------------------------- */
static char *g_key[] = /* IMAP keywords */
{	"", 
	"UID",        "SEARCH",        "CHARSET",     "BODY",
	"TEXT",       "TO",            "FROM",        "SUBJECT",
	"CC",         "BCC",           "HEADER",      "NEW",
	"OLD",        "RECENT",        "ANSWERED",    "UNANSWERED",
	"DELETED",    "UNDELETED",     "DRAFT",       "UNDRAFT",
	"FLAGGED",    "UNFLAGGED",     "SEEN",        "UNSEEN",
	"KEYWORD",    "UNKEYWORD",     "LARGER",      "SMALLER",
	"NOT",        "OR",            "BEFORE",      "ON",
	"SINCE",      "SENTBEFORE",    "SENTON",      "SENTSINCE",
	"ALL"
};

static SYMBOL g_keysy[] = /* symbols corresponding to keywords */
{	-1,
	UIDsy,        SEARCHsy,        CHARSETsy,     BODYsy,
	TEXTsy,       TOsy,            FROMsy,        SUBJECTsy,
	CCsy,         BCCsy,           HEADERsy,      NEWsy,
	OLDsy,        RECENTsy,        ANSWEREDsy,    UNANSWEREDsy,
	DELETEDsy,    UNDELETEDsy,     DRAFTsy,       UNDRAFTsy,
	FLAGGEDsy,    UNFLAGGEDsy,     SEENsy,        UNSEENsy,
	KEYWORDsy,    UNKEYWORDsy,     LARGERsy,      SMALLERsy,
	NOTsy,        ORsy,            BEFOREsy,      ONsy,
	SINCEsy,      SENTBEFOREsy,    SENTONsy,      SENTSINCEsy,
	ALLsy
};

static unsigned int g_keylen[sizeof( g_key)/sizeof(g_key[0])];

static unsigned int g_nkw = sizeof( g_key)/sizeof(g_key[0]);

static BOOL g_ImapSearchInit = FALSE;

#if 0
typedef enum /* NMAP symbols not present in IMAP symbols */
{	HEXsy = 1, BINsy,
	COUNTsy,
	FLAGsy,
	SIZEsy,
	RECEIVEDsy, SENTsy,
	HDRsy,
} other;

static char *g_otherkey[] = /* NMAP keywords that are not present in IMAP keywords */
{	"",
	"HEX", "BIN",
	"COUNT",
	"FLAG",
	"SIZE",
	"RECEIVED", "SENT",
	"HDR",
};

static other g_othersy[] = /* NMAP symbols corresponding to keywords */
{	-1,
	HEXsy, BINsy,
	COUNTsy,
	FLAGsy,
	SIZEsy,
	RECEIVEDsy, SENTsy,
	HDRsy,
};
#endif


/*----------------------------------------------------------------------------- */
/*							Exported Globals */
/*----------------------------------------------------------------------------- */
char *g_month[] = /* all months */
{
	"Jan", "Feb", "Mar", "Apr", "May", "Jun", "Jul", "Aug", "Sep", "Oct", "Nov", "Dec"
};



/*----------------------------------------------------------------------------- */
/*							Functions - Internal and External */
/*							All kept alphabetically ordered. */
/*----------------------------------------------------------------------------- */


/*----------------------------------------------------------------------------
;xxx
Title:		
Globals:	
Return:		
Descrip:	

------------------------------------------------------------------------------*/
/*xxx (	)						// Formal parameters */
/*{ */
/*} // xxx */



/*----------------------------------------------------------------------------
;ImapSearch
Title:		
Globals:	
Return:		
Descrip:	search main

------------------------------------------------------------------------------*/
long 
IMAPSearch(ImapClient* Client, BOOL ByUID)
{
	char        Answer[1024];
	/*int         msg, len; */
	BITVECTOR  *bv = 0;

	if (ByUID) {
	    /* Fixme - Shouldn't the response be different for UID Search? */
	}

	/*---------------------------- */
	/* is a charset parameter of other than us-ascii being used */
	if (XplStrNCaseCmp(Client->Command + 6, " charset ", 9) == 0) {
		if (XplStrNCaseCmp(Client->Command + 6, " charset us-ascii ", 18) == 0) {
			unsigned long len;

			len = strlen(Client->Command + 6 + 18);
			memmove(Client->Command + 7, Client->Command + 6 + 18, len);
			*(Client->Command + 7 + len) = '\0';
		} else if (XplStrNCaseCmp(Client->Command + 6, " charset \"us-ascii\" ", 20) == 0) {
			unsigned long len;

			len = strlen(Client->Command + 6 + 20);
			memmove(Client->Command + 7, Client->Command + 6 + 20, len);
			*(Client->Command + 7 +len) = '\0';
		} else {
			snprintf(Answer, sizeof(Answer), "%s NO Search error: charset not supported\r\n", Client->Ident);
			SendClient(Client, Answer, strlen( Answer));
			return(CLIENT_CONTINUE);
		}
	}
	 	
	/*------------------------ */
	/* one time initialization */
	if(!g_ImapSearchInit) {
		ImapSearchInit();
	}

	Client->Search = (IMAPSEARCH*)MemMalloc(sizeof(IMAPSEARCH));
	if(!Client->Search) {
		snprintf(Answer, sizeof(Answer), "%s NO Search error: could not allocate memory\r\n", Client->Ident);
		SendClient(Client, Answer, strlen( Answer));
		return(CLIENT_CONTINUE);
	}
	memset(Client->Search, 0, sizeof(IMAPSEARCH)); 

	/*------------------- */
	/* fill out shortcuts */
	Client->Search->Command = Client->Command;
	Client->Search->Client = (void*)Client;

	/*------------------------ */
	/* set default return type */
	Client->Search->resultByPosition = TRUE;

	/*---------------- */
	/* prime the pump! */
	Client->Search->ch = Client->Search->Command[0];

	InSymbol(Client->Search);

	/*------------------------- */
	/* examine the search query */
	if(Client->Search->sy != endofbufsy)
	{
		/*command( Client->Search, &bv); */
		bv = command(Client->Search);
	}

	/*------------------------------------ */
	/* send the required untagged response */
	snprintf( Answer, sizeof(Answer), "* %s", g_key[SEARCHsy]);
	SendClient( Client, Answer, strlen( Answer));

	/*-------------- */
	/* send the hits */
	if( bv)
	{
		BOOL byPos = Client->Search->resultByPosition;
		unsigned long msg, len, count = Client->Search->msgCount;
		
		memset( Answer, 0, sizeof( Answer));

		for( msg = 0, len = 0; msg < count; msg++)
		{
			/*---------- */
			/* hit check */
			if( BVGET( bv, msg))
			{
				len += sprintf( Answer + len, " %lu", (byPos) ?msg+1 :Client->UIDList[msg]);
			}

			/*---------------------------------------------- */
			/* ugly protection code to avoid buffer overflow */
			/* TODO make a more efficient check for overflow */
			if( len > (sizeof(Answer)-20))
			{
				SendClient( Client, Answer, strlen( Answer));
				len = 0;
			}
		}

		/*------------------- */
		/* send the leftovers */
		SendClient( Client, Answer, strlen( Answer));
		MemFree( bv);
	}


	/*------------------------------------------------ */
	/* send the required tagged response OK, NO or BAD */
	snprintf(
		Answer, 
		sizeof(Answer), 
		"\r\n%s OK %s%s%s completed\r\n",
		Client->Ident, 
		!Client->Search->resultByPosition ? g_key[UIDsy] :"", 
		!Client->Search->resultByPosition ? " " :"",
		g_key[SEARCHsy]);

	SendClient( Client, Answer, strlen( Answer));

	/*--------------------------- */
	/* cleanup memory allocations */
	MemFree( Client->Search);
	Client->Search = 0;
	return(CLIENT_CONTINUE);
}

/*----------------------------------------------------------------------------
;command
Title:		
Globals:	
Return:		
Descrip:		
	command         ::= tag SPACE (command_any / command_auth / command_nonauth / command_select) CRLF
	Modal based on state
	only command_select is to be implemented

	ignore tag, done previously

	command_select  ::= "CHECK" / "CLOSE" / "EXPUNGE" / copy / fetch / store / uid / search
	Valid only when in Selected state
	only uid and search are to be implemented, 
	therefore command_select is folded into command

------------------------------------------------------------------------------*/
BITVECTOR *command( IMAPSEARCH *Search)
{
	switch( Search->sy)
	{
		case UIDsy:
			return uid( Search);
		break;

		case SEARCHsy:
			return search( Search);
		break;
	default:
	  return(0);
	}
	/* TestTerminal( Search, crlfsy); */
	return 0;
}

/*----------------------------------------------------------------------------
;uid
Title:		
Globals:	
Return:		
Descrip:	
	uid             ::= "UID" SPACE search   
	UID used instead of message sequence numbers

------------------------------------------------------------------------------*/
BITVECTOR *uid( IMAPSEARCH *Search)
{
	/* set default return type */
	Search->resultByPosition = FALSE;
	TestTerminal( Search, UIDsy);
	TestTerminal( Search, spacesy);
	return search( Search);
}


/*----------------------------------------------------------------------------
;search
Title:		
Globals:	
Return:		
Descrip:	
	search          ::= "SEARCH" SPACE ["CHARSET" SPACE astring SPACE] 1#search_key 
	[CHARSET] MUST be registered with IANA

------------------------------------------------------------------------------*/
BITVECTOR *search( IMAPSEARCH *Search)
{
	Emit( Search, keyword, 0);
	/* TODO - see if chosing between HEX and BIN can be configurable */
	Emit( Search, string, "HEX");
/*	Emit( Search, string, "BIN"); */
	TestTerminal( Search, SEARCHsy);
	TestTerminal( Search, spacesy);

	/*-------- */
	/* CHARSET */
	if( CHARSETsy == Search->sy) {
		InSymbol( Search);
		TestTerminal( Search, spacesy);
		astring( Search);
		TestTerminal( Search, spacesy);
	}
	
	/*------------------ */
	/* get the msg count */
	Search->msgCount = ((ImapClient*)Search->Client)->Messages;

	Search->bvByteSize = (Search->msgCount / BITCOUNT) * ELEMENTSIZE;
	if( Search->msgCount % BITCOUNT)	{
		Search->bvByteSize += ELEMENTSIZE;
	}

	return searchAND( Search);
}

BITVECTOR *searchAND( IMAPSEARCH *Search)
{
	BITVECTOR *bv1     = 0;
	BITVECTOR *bvFinal = 0;

	/*---------------------------------- */
	/* do the first key, return if error  */
	if((bvFinal = search_key( Search)) == NULL)
	{
		return NULL;
	}

	/*------------------------------------------ */
	/* if there is a space we expect another key */
	if( spacesy == Search->sy)
	{
		InSymbol( Search);
	}
	else
	{
		return bvFinal;
	}
	
	/*------------------------------ */
	/* for the remaining search keys */
	while( 1)
	{
		/*----------------------------- */
		/* do this key, return if error */
		if((bv1 = search_key( Search)) == NULL)
		{
			MemFree( bvFinal);
			return NULL;
		}

		/*------------------------------------------- */
		/* this is AND'ing all the keys at this level */
		BVIntersect( bvFinal, bv1, Search->bvByteSize);

		/*-------------------------------------------- */
		/* free this one, we only need to keep bvFinal */
		MemFree( bv1);

		if( spacesy == Search->sy)
		{
			InSymbol( Search);
		}
		else
		{
			break;
		}
	}
	return bvFinal;
}




/*----------------------------------------------------------------------------
;search_key
Title:		
Globals:	
Return:		
Descrip:		
	search_key      ::=  see below for production rules
	change IMAP syntax to NMAP syntax

------------------------------------------------------------------------------*/
BITVECTOR *search_key( IMAPSEARCH *Search)
{
	BITVECTOR *bv1 = 0;
	BITVECTOR *bv2 = 0;

	switch( Search->sy)
	{
		case NEWsy:        /* "NEW" ie RECENT UNSEEN */
			/* get recent */
			Emit( Search, string, "FLAG");
			Search->sy = RECENTsy;
			Emit( Search, symbol, 0);
			bv1 = ImapSearchQuery( Search);
			if( !bv1)
			{
				return 0;
			}

			/* get seen */
			Emit( Search, string, "FLAG");
			Search->sy = SEENsy;
			Emit( Search, symbol, 0);
			bv2 = ImapSearchQuery( Search);
			if( !bv2)
			{
				MemFree( bv1);
				return 0;
			}

			/* unseen */
			BVInvert( bv2, Search->bvByteSize);
			/* AND */
			BVIntersect( bv1, bv2, Search->bvByteSize);
			MemFree( bv2);
			InSymbol( Search);
		return bv1;

		case ANSWEREDsy:   /* "ANSWERED"  */
		case DELETEDsy:    /* "DELETED" */
		case DRAFTsy:      /* "DRAFT" */
		case FLAGGEDsy:    /* "FLAGGED" */
		case SEENsy:       /* "SEEN" */
		case RECENTsy:     /* "RECENT" */
			Emit( Search, string, "FLAG");
			Emit( Search, keyword, 0);
			InSymbol( Search);
		return ImapSearchQuery( Search);

		case UNANSWEREDsy: /* "UNANSWERED" */
		case UNDELETEDsy:  /* "UNDELETED" */
		case UNDRAFTsy:    /* "UNDRAFT" */
		case UNFLAGGEDsy:  /* "UNFLAGGED" */
		case UNSEENsy:     /* "UNSEEN" */
		case OLDsy:        /* "OLD" ie NOT RECENT */
			/* change to presence search */
			switch( Search->sy)
			{
				case UNANSWEREDsy: Search->sy = ANSWEREDsy; break;
				case UNDELETEDsy:  Search->sy = DELETEDsy;  break;
				case UNDRAFTsy:    Search->sy = DRAFTsy;    break;
				case UNFLAGGEDsy:  Search->sy = FLAGGEDsy;  break;
				case UNSEENsy:     Search->sy = SEENsy;     break;
				case OLDsy:        Search->sy = RECENTsy;   break;
			default: // should not happen
			  break;
			}
			Emit( Search, string, "FLAG");
			Emit( Search, symbol, 0);
			InSymbol( Search);
			bv1 = ImapSearchQuery( Search);
			if( !bv1)
			{
				return 0;
			}
			/* un* */
			BVInvert( bv1, Search->bvByteSize);
		return bv1;

		
		case KEYWORDsy:   /* "KEYWORD"   SPACE flag_keyword */
		case UNKEYWORDsy: /* "UNKEYWORD" SPACE flag_keyword */
			/* KEYWORD AND UNKEYWORD facilitate non-permanent flags, we do not */
			/* support ANY non-permanent flags, so here we're just */
			/* covering the possibility that permanent flag may also be passed */
			Emit( Search, string, "FLAG");
			InSymbol( Search);
			TestTerminal( Search, spacesy);
			Emit( Search, ident, 0);
			TestTerminal( Search, atomsy);
		return ImapSearchQuery( Search);

		case LARGERsy:  /* "LARGER"  SPACE number */
		case SMALLERsy: /* "SMALLER" SPACE number */
			Emit( Search, string, "SIZE");
			Emit( Search, keyword, 0);
			InSymbol( Search);
			if( spacesy == Search->sy)
			{
				number( Search);
			}
		return ImapSearchQuery( Search);

		case BEFOREsy:     /* "BEFORE"     SPACE date  */
		case ONsy:         /* "ON"         SPACE date */
		case SINCEsy:      /* "SINCE"      SPACE date */
			Emit( Search, string, "RECEIVED");
			goto CommonDate;
		case SENTBEFOREsy: /* "SENTBEFORE" SPACE date */
		case SENTONsy:     /* "SENTON"     SPACE date */
		case SENTSINCEsy:  /* "SENTSINCE"  SPACE date */
			Emit( Search, string, "SENT");
			CommonDate:
			switch( Search->sy)
			{
				case BEFOREsy:
				case SENTBEFOREsy:
					Emit( Search, string, "BEFORE");
				break;

				case ONsy:
				case SENTONsy:
					Emit( Search, string, "ON");
				break;
				
				case SINCEsy:
				case SENTSINCEsy:
					Emit( Search, string, "SINCE");
				break;
			default: // should not happen
			  break;
			}
			InSymbol( Search);
			if( spacesy == Search->sy)
			{
				date( Search);
			}
		return ImapSearchQuery( Search);

		case NOTsy: /* "NOT" SPACE search_key */
			/* do not emit NOT keyword because it is not handled in NMAP */
			InSymbol( Search);
			TestTerminal( Search, spacesy);
			if( !(bv1 = search_key( Search)))
			{
				return 0;
			}
			BVInvert( bv1, Search->bvByteSize);
		return bv1;
	
		case ORsy: /* "OR" SPACE search_key SPACE search_key */
			/* do not emit OR keyword because it is not handled in NMAP */
			InSymbol( Search);
			TestTerminal( Search, spacesy);
			if( !(bv1 = search_key( Search)))
			{
				return 0;
			}
			TestTerminal( Search, spacesy);
			if( !(bv2 = search_key( Search)))
			{
				MemFree( bv1);
				return 0;
			}
			BVUnion( bv1, bv2, Search->bvByteSize);
			MemFree( bv2);
		return bv1;

		case ALLsy: /* "ALL" */
			/*Emit( Search, keyword, 0); */
			InSymbol( Search);
		return BVEmpty( bv1, Search->msgCount, ALL_BITS_SET);

		case UIDsy: /* "UID" SPACE set */
			InSymbol( Search);
			/* remember we want UIDs */
			Search->messageSet = BY_UID;
			TestTerminal( Search, spacesy);
			bv1 = BVEmpty( bv1, Search->msgCount, ALL_BITS_CLEAR);
			if( !bv1)
			{
				return 0;
			}
		return set( Search, bv1);

		case largestsy:  /* set */
		case nz_numbersy:
			/* remember we want POSITIONs */
			Search->messageSet = BY_POSITION;
			bv1 = BVEmpty( bv1, Search->msgCount, ALL_BITS_CLEAR);
			if( !bv1)
			{
				return 0;
			}
		return set( Search, bv1);

		case lparensy: /* "(" 1#search_key ")" */
			InSymbol( Search);
			bv1 = searchAND( Search);
			TestTerminal( Search, rparensy);
		return bv1;

		case TOsy:      /* "TO"      SPACE astring */
		case FROMsy:    /* "FROM"    SPACE astring */
		case SUBJECTsy: /* "SUBJECT" SPACE astring */
		case CCsy:      /* "CC"      SPACE astring */
		case BCCsy:     /* "BCC"     SPACE astring */
			Emit( Search, string, "HDR");
			Emit( Search, ident, 0);
			goto CommonText;
		case BODYsy:    /* "BODY"    SPACE astring */
			Emit( Search, string, "BODY");
			goto CommonText;
		case TEXTsy:    /* "TEXT"    SPACE astring */
			Emit( Search, string, "TEXT");
			CommonText:
			InSymbol( Search);
			TestTerminal( Search, spacesy);
			astring( Search);
		return ImapSearchQuery( Search);

		case HEADERsy: /* "HEADER" SPACE header_fld_name SPACE astring */
			Emit( Search, string, "HDR");
			InSymbol( Search);
			TestTerminal( Search, spacesy);
			header_fld_name( Search);
			TestTerminal( Search, spacesy);
			astring( Search);
		return ImapSearchQuery( Search);
	default: // should not happen
	  break;;
	}
	return 0;
}



/*----------------------------------------------------------------------------
;set
Title:		
Globals:	
Return:		
Descrip:	
	set             ::= sequence_num / (sequence_num ":" sequence_num) / (set "," set)
	Identifies a set of messages.  For message
    sequence numbers, these are consecutive
    numbers from 1 to the number of messages in
    the mailbox
    Comma delimits individual numbers, colon
    delimits between two numbers inclusive.
    Example: 2,4:7,9,12:* is 2,4,5,6,7,9,12,13,14,15 
	for a mailbox with 15 messages.

------------------------------------------------------------------------------*/
BITVECTOR *set( IMAPSEARCH *Search, BITVECTOR *bv)
{
	unsigned long first, last, msg;

	do {
		if(Search->messageSet == BY_UID) {
			if (Search->sy == nz_numbersy) {
				first = Search->identNum;
			} else {
				first = ULONG_MAX;
			}
			last=first;

			InSymbol(Search);

			if(Search->sy == colonsy) {
				InSymbol(Search);
				if(Search->sy == nz_numbersy) {
					last = Search->identNum;
				} else {
					last = ULONG_MAX;
				}
				if (!UIDtoNUMRange(Search->Client, first, last, (long *)&first, (long *)&last)) {
					return(bv);
				}
				InSymbol(Search);
			} else {
			    long retValue;

			    retValue = UIDtoNUM(Search->Client, first);
			    if (retValue == -1) {
				return(bv);
			    }
			    first = retValue;
			    last = first;
			}
		} else {
			if (Search->sy == nz_numbersy) {
				first = Search->identNum - 1;
			} else {
				first = Search->msgCount - 1;
			}
			last=first;

			InSymbol(Search);

			if(Search->sy == colonsy) {
				InSymbol(Search);
				if(Search->sy == nz_numbersy) {
					last = Search->identNum - 1;
				} else {
					last = Search->msgCount - 1;
				}
				InSymbol(Search);
			}
			if ((first >= Search->msgCount) || (last >= Search->msgCount)) {
				return(bv);
			}
		}

		for (msg = first; msg <= last; msg++) {
			BVSET(bv, msg);
		}

		if(Search->sy != commasy) {
			return bv;
		}
		InSymbol( Search);		/* Process the next set */
	} while(TRUE);
}



/*----------------------------------------------------------------------------
;sequence_num
Title:		
Globals:	
Return:		
Descrip:	this is a specialization of TestTerminal
	sequence_num    ::= nz_number / "*"
	* is the largest number in use.
	For message sequence numbers, it is the number of messages in the mailbox.  
	For unique identifiers, it is the unique identifier of the last message in the mailbox.

------------------------------------------------------------------------------*/
BITVECTOR *sequence_num( IMAPSEARCH *Search, BITVECTOR *bv)
{
	unsigned long msg;

	switch( Search->sy)
	{
		case largestsy:
			switch( Search->messageSet)
			{
				case BY_POSITION:
				case BY_UID:
					BVSET( bv, (Search->msgCount - 1)); /* remember we are 0 based...avoid side effects with parens */
				break;
			}
			InSymbol( Search);
		break;

		case nz_numbersy:
			/*Emit( Search, keyword, 0); */
			switch( Search->messageSet)
			{
				case BY_POSITION:
					if( Search->identNum <= Search->msgCount)
					{
						BVSET( bv, (Search->identNum - 1)); /* remember we are 0 based...avoid side effects with parens */
					}
				break;
				case BY_UID:
					/* the number returned is zero based */
					msg = UIDtoNUM( Search->Client, Search->identNum);
					if( msg <= Search->msgCount)
					{
						BVSET( bv, msg);
					}
				break;
			}
			InSymbol( Search);
		break;
	default: // should not happen
	  break;
	}
	return bv;
}



/*----------------------------------------------------------------------------
;astring
Title:		
Globals:	
Return:		
Descrip:	this is a specialization of TestTerminal
	astring         ::= atom / string

------------------------------------------------------------------------------*/
void astring( IMAPSEARCH *Search)
{
	switch( Search->sy)
	{
		case atomsy:
		case stringsy:
			Emit( Search, ident, 0);
			InSymbol( Search);
		break;
	default: // should not happen
	  break;

	}
}


/*----------------------------------------------------------------------------
;header_fld_name
Title:		
Globals:	
Return:		
Descrip:	this is a specialization of TestTerminal
	header_fld_name ::= astring
	similar to astring except that here all headers are valid

------------------------------------------------------------------------------*/
void header_fld_name( IMAPSEARCH *Search)
{
	switch( Search->sy)
	{
		case atomsy:
		case stringsy:
		case TOsy:
		case FROMsy:
		case SUBJECTsy:
		case CCsy:
		case BCCsy:
			Emit( Search, ident, 0);
			InSymbol( Search);
		break;
	default: // should not happen
	  break;

	}
}


/*----------------------------------------------------------------------------
;date
Title:		
Globals:	
Return:		
Descrip:	this is a specialization of InSymbol
	date            ::= date_text / <"> date_text <">
	date_text       ::= date_day "-" date_month "-" date_year
	date_day        ::= 1*2digit 
	date_month      ::= "Jan" / "Feb" / "Mar" / "Apr" / "May" / "Jun" / "Jul" / "Aug" / "Sep" / "Oct" / "Nov" / "Dec"
	date_year       ::= 4digit

------------------------------------------------------------------------------*/
void date( IMAPSEARCH *Search)
{
	BOOL      quoted = FALSE;
	time_t    dateInSec;
	struct tm date;
	int       i, datepart, month;

	Search->sy = datesy;

	memset( &date, 0, sizeof( struct tm));
	date.tm_isdst = -1;

	if( '"' == Search->ch)
	{
		/* remember quoted */
		quoted = TRUE;
		NextCh( Search);
	}
	
	/*-------------------------------- */
	/* for each of the parts of a date */
	for( datepart = 0; datepart < 3; datepart++)
	{
		i =0;
		while( (Search->ch!= ' ') &&  (Search->ch != '"') && (Search->ch != 0) && (Search->ch != '-'))
		{
			if( i < IDENTSIZE)
			{
				Search->ident[i] = Search->ch;
				i++;
			}
			NextCh( Search);
		}

		if( i < IDENTSIZE)
		{
			Search->ident[i] = 0;
			Search->identLen = i;
		}
		else
		{
			Search->ident[IDENTSIZE+1] = 0;
			Search->identLen = IDENTSIZE;
		}

		switch( datepart)
		{
			case DATE_DAY: 
				date.tm_mday = atol( Search->ident); 
				/* remove the dash */
				NextCh( Search);
			break;
			
			case DATE_MONTH:
				for( month = 0; month < 12; month++ ) 
				{
					if( 0 == XplStrNCaseCmp( Search->ident, g_month[ month], DATE_MONTH_STR_LEN))
					{
						date.tm_mon = month;
						break;
					}
				}
				/* remove the dash */
				NextCh( Search);
			break;

			case DATE_YEAR: 
				date.tm_year = atol( Search->ident);
				date.tm_year -= 1900; /* adjust for mktime */
			break;
		}
	}

	if( quoted && ('"' == Search->ch))
	{
		/* remove the last quote */
		NextCh( Search);
	}

	dateInSec = mktime( &date);
	sprintf( Search->ident, "%lu", dateInSec);
	Search->identLen = strlen( Search->ident);
	Emit( Search, ident, 0);
	InSymbol( Search);
}


/*----------------------------------------------------------------------------
;number
Title:		
Globals:	
Return:		
Descrip:	this is a specialization of InSymbol
	nz_number       ::= digit_nz *digit // Non-zero unsigned 32-bit integer (0 < n < 4,294,967,296)
	digit           ::= "0" / digit_nz
	digit_nz        ::= "1" / "2" / "3" / "4" / "5" / "6" / "7" / "8" / "9"

------------------------------------------------------------------------------*/
void number( IMAPSEARCH *Search)
{
	int i = 0;
	unsigned long num;

	switch( Search->ch)
	{
		case '0':
		case '1':
		case '2':
		case '3':
		case '4':
		case '5':
		case '6':
		case '7':
		case '8':
		case '9':
			i =0;
			while( isdigit( Search->ch))
			{
				if( i < IDENTSIZE)
				{
					Search->ident[i] = Search->ch;
					i++;
				}

				NextCh( Search);
			}

			if( i < IDENTSIZE)
			{
				Search->ident[i] = 0;
				Search->identLen = i;
			}
			else
			{
				Search->ident[IDENTSIZE+1] = 0;
				Search->identLen = IDENTSIZE;
			}
			num = atol( Search->ident);
			Search->sy = (0 == num) ?numbersy :nz_numbersy;
		break;
	}

	Emit( Search, ident, 0);
	InSymbol( Search);
}


/*----------------------------------------------------------------------------
;ImapSearchInit
Title:		
Globals:	
Return:		
Descrip:	One time initialization of search data

------------------------------------------------------------------------------*/
void ImapSearchInit( void)
{
	int i, len;

	/* search has been initialized */
	g_ImapSearchInit = TRUE;

	/* count the length of the keywords */
	len = sizeof( g_key) / sizeof( g_key[0]);
	for( i = 0; i < len; i++) {
		g_keylen[i] = strlen( g_key[i]);
	}
}


/*----------------------------------------------------------------------------
;TestTerminal
Title:		
Globals:	
Return:		
Descrip:	Test the current token against a given token

------------------------------------------------------------------------------*/
void TestTerminal( IMAPSEARCH *Search, SYMBOL sy)
{
	if( Search->sy == sy)
	{
		InSymbol( Search);
	}
	else
	{
		Search->sy = errorsy;
	}
}


/*----------------------------------------------------------------------------
;NexCh
Title:		
Globals:	
Return:		
Descrip:	Get the next character in the buffer

------------------------------------------------------------------------------*/
void NextCh( IMAPSEARCH *Search) 
{ 
	Search->offsetIn++; 
	Search->ch = Search->Command[ Search->offsetIn];
}


/*----------------------------------------------------------------------------
;Emit
Title:		
Globals:	
Return:		
Descrip:	Emit strings, keywords or current identifier to buffer

------------------------------------------------------------------------------*/
void Emit( IMAPSEARCH *Search, EMIT emit, char *text)
{
	int len = 0;

	/*--------------------------------------------------- */
	/* first let's check if this will overflow our buffer */
	switch( emit)
	{
		case string: /* we add a space, count it */
			if( text)
			{
				len = strlen( text);
				if( BUFSIZE <= Search->offsetOut + len)
				{
				    return; /* Fixme - allow for bigger searches (NMAP's command buffersize is limited) */
				}
			}
			else
			{
				return;
			}
		break;

		case keyword: /*!! we add a space, count it */
			/* ?? */
		break;

		case ident: /*!! we add 2 quotes and a space, need to count them */
			if( BUFSIZE <= Search->offsetOut + Search->identLen)
			{
			    return; /* Fixme - allow for bigger searches (NMAP's command buffersize is limited) */
			}
		break;

		case symbol:
			/*?? */
		break;
	}

	/*------------------------------------------- */
	/* we're done thinking, now add to the buffer */
	switch( emit)
	{
		case string: /* the passed in string */
			strcpy( Search->bufferOut + Search->offsetOut, text);
			Search->offsetOut += len;
			Search->bufferOut[ Search->offsetOut] = ' ';
			Search->offsetOut++;
			Search->bufferOut[ Search->offsetOut] = 0;
		break;
		case keyword: /* the curent identifier not quoted */
			strcpy( Search->bufferOut + Search->offsetOut, Search->ident);
			Search->offsetOut += Search->identLen;
			Search->bufferOut[ Search->offsetOut] = ' ';
			Search->offsetOut++;
			Search->bufferOut[ Search->offsetOut] = 0;
		break;
		case symbol: /* the current symbol */
			strcpy( Search->bufferOut + Search->offsetOut, g_key[ Search->sy]);
			Search->offsetOut += g_keylen[ Search->sy];
			Search->bufferOut[ Search->offsetOut] = ' ';
			Search->offsetOut++;
			Search->bufferOut[ Search->offsetOut] = 0;
		break;
		case ident: /* the current identifier quoted */
			Search->bufferOut[ Search->offsetOut] = '"';
			Search->offsetOut++;
			strcpy( Search->bufferOut + Search->offsetOut, Search->ident);
			Search->offsetOut += Search->identLen;
			Search->bufferOut[ Search->offsetOut] = '"';
			Search->offsetOut++;
			Search->bufferOut[ Search->offsetOut] = ' ';
			Search->offsetOut++;
			Search->bufferOut[ Search->offsetOut] = 0;
		break;
	}
}


/*----------------------------------------------------------------------------
;InSymbol
Title:		
Globals:	
Return:		
Descrip:	Tokenize a stream of characters
	quoted          ::= <"> *QUOTED_CHAR <">
	QUOTED_CHAR     ::= <any TEXT_CHAR except quoted_specials> "\" quoted_specials
	TEXT_CHAR       ::= <any CHAR except CR and LF>
	CHAR            ::= <any 7-bit US-ASCII character except NUL, 0x01 - 0x7f>
	CR              ::= <ASCII CR, carriage return, 0x0D>
	LF              ::= <ASCII LF, line feed, 0x0A>
	quoted_specials ::= <"> / "\"

	literal         ::= "{" number "}" CRLF *CHAR8 // Number represents the number of CHAR8 octets
	CHAR8           ::= <any 8-bit octet except NUL, 0x01 - 0xff>
	CRLF            ::= CR LF

	ATOM or reserved word
	must be at least length 1 or greater
	atom            ::= 1*ATOM_CHAR
	ATOM_CHAR       ::= <any CHAR except atom_specials>
	CHAR            ::= <any 7-bit US-ASCII character except NUL, 0x01 - 0x7f>
	atom_specials   ::= "(" / ")" / "{" / SPACE / CTL / list_wildcards / quoted_specials
	SPACE           ::= <ASCII SP, space, 0x20>
	CTL             ::= <any ASCII control character and DEL, 0x00 - 0x1f, 0x7f>
	list_wildcards  ::= "%" / "*"
	quoted_specials ::= <"> / "\"

------------------------------------------------------------------------------*/
void InSymbol( IMAPSEARCH *Search)
{
	unsigned long i; /*, len; */
	unsigned char lch; /*, cch; */

	lch = Search->ch;
	NextCh( Search);

	switch( lch)
	{
		case '0':
		case '1':
		case '2':
		case '3':
		case '4':
		case '5':
		case '6':
		case '7':
		case '8':
		case '9':
			i =0;
			Search->ident[i] = lch;
			i++;
			while( isdigit( Search->ch))
			{
				if( i < IDENTSIZE)
				{
					Search->ident[i] = Search->ch;
					i++;
				}

				NextCh( Search);
			}

			if( i < IDENTSIZE)
			{
				Search->ident[i] = 0;
				Search->identLen = i;
			}
			else
			{
				Search->ident[IDENTSIZE+1] = 0;
				Search->identLen = IDENTSIZE;
			}
			Search->identNum = atol( Search->ident);
			Search->sy = (0 == Search->identNum) ?numbersy :nz_numbersy;
		break;

		case NUL: /* null */
			Search->sy = endofbufsy;
		break;

		case SPACE:
			Search->sy = spacesy;
		break;

		case QUOTE: /* " start of string */
			/* quoted          ::= <"> *QUOTED_CHAR <"> */
			/* QUOTED_CHAR     ::= <any TEXT_CHAR except quoted_specials> "\" quoted_specials */
			/* TEXT_CHAR       ::= <any CHAR except CR and LF> */
			/* CHAR            ::= <any 7-bit US-ASCII character except NUL, 0x01 - 0x7f> */
			/* CR              ::= <ASCII CR, carriage return, 0x0D> */
			/* LF              ::= <ASCII LF, line feed, 0x0A> */
			/* quoted_specials ::= <"> / "\" */

			for( i = 0;;i++ )
			{
				/* the other quote */
				if( '"' == Search->ch)
				{
					/* eat the last quote */
					 NextCh( Search);
					break;
				}

				/* not allowed */
				if( (NUL == Search->ch) 
					||  (CARRIAGE_RETURN == Search->ch) 
					||  (LINE_FEED == Search->ch))
				{
					/* error */
					Search->sy = errorsy;
					return;
				}

				/* quoted special */
				if( '\\' == Search->ch)
				{
					/* next char has to be a quoted special */
					NextCh( Search);
					if( ('"' != Search->ch) && ('\\' != Search->ch))
					{
						/* error */
						Search->sy = errorsy;
						return;
					}
				}

				if( i < IDENTSIZE)
				{
					Search->ident[i] = Search->ch;
				}

				NextCh( Search);
			}

			if( i < IDENTSIZE)
			{
				Search->ident[i] = 0;
				Search->identLen = i;
			}
			else
			{
				Search->ident[IDENTSIZE+1] = 0;
				Search->identLen = IDENTSIZE;
			}

			Search->sy = stringsy;
		break;
#if 0
		case CARRIAGE_RETURN:
			if( LINE_FEED == Search->ch)
			{
				/*NextCh( Search);  */
				Search->sy = crlfsy;
			}
			else
			{
				Search->sy = crsy;
			}
		break;
#endif
		case LINE_FEED:  Search->sy = lfsy; break;
		case '}': Search->sy = rbracesy; break;
		case ',': Search->sy = commasy; break;
		case ':': Search->sy = colonsy; break;
		case '*': Search->sy = largestsy; break;
		case '(': Search->sy = lparensy; break;
		case ')': Search->sy = rparensy; break;
		case '[': Search->sy = lbracksy; break;
		case ']': Search->sy = rbracksy; break;
		case '{': 
			Search->sy = lbracesy;
			#if 0 /* simple literal */
			/* call GrabOctet? */
			/* literal         ::= "{" number "}" CRLF *CHAR8 // Number represents the number of CHAR8 octets */
			/* CHAR8           ::= <any 8-bit octet except NUL, 0x01 - 0xff> */
			/* CRLF            ::= CR LF */
			TestTerminal( Search, lbracesy);
			TestTerminal( Search, numbersy);
			TestTerminal( Search, rbracesy);
			TestTerminal( Search, crlfsy);
			/* get some data */
			/* test data integrity */
			#endif /* end simple literal */
		break;

		default:
			/* ATOM or reserved word */
			/* must be at least length 1 or greater */
			/* atom            ::= 1*ATOM_CHAR */
			/* ATOM_CHAR       ::= <any CHAR except atom_specials> */
			/* CHAR            ::= <any 7-bit US-ASCII character except NUL, 0x01 - 0x7f> */
			/* atom_specials   ::= "(" / ")" / "{" / SPACE / CTL / list_wildcards / quoted_specials */
			/* SPACE           ::= <ASCII SP, space, 0x20> */
			/* CTL             ::= <any ASCII control character and DEL, 0x00 - 0x1f, 0x7f> */
			/* list_wildcards  ::= "%" / "*" */
			/* quoted_specials ::= <"> / "\" */

			/* what an ugly way to catch lch */
			i = 0;
			if( isprint( lch)
				&& (lch != ' ') 
				&& (lch != NUL)
				&& (lch != '(') 
				&& (lch != ')') 
				&& (lch != '{') 
				&& (lch != '%') 
				&& (lch != '*') 
				&& (lch != '\\') 
				&& (lch != '"'))
			{
				Search->ident[i] = lch;
				i++;
			}
			else
			{
				Search->sy = errorsy;
				break;
			}
			
			/* while it is a valid atom char */
			while( isprint( Search->ch)
				&& (Search->ch != ' ') 
				&& (Search->ch != NUL)
				&& (Search->ch != '(') 
				&& (Search->ch != ')') 
				&& (Search->ch != '{') 
				&& (Search->ch != '%') 
				&& (Search->ch != '*') 
				&& (Search->ch != '\\') 
				&& (Search->ch != '"'))
			{
				if( i < IDENTSIZE)
				{
					Search->ident[i] = Search->ch;
					i++;
				}

				NextCh( Search);
			}

			if( i < IDENTSIZE)
			{
				Search->ident[i] = 0;
				Search->identLen = i;
			}
			else
			{
				Search->ident[IDENTSIZE+1] = 0;
				Search->identLen = IDENTSIZE;
			}

			/* mark it as an atom for now */
			Search->sy = atomsy;
			
			/* is it a keyword? */
			for( i = 1; i < g_nkw; i++ ) 
			{
				/* only compare if same length */
				if( g_keylen[i] == Search->identLen 
					&& 0 == XplStrNCaseCmp( Search->ident, g_key[i], g_keylen[i]))
				{
					/* record the match */
					Search->sy = g_keysy[i];
					break;
				}
			}
		break;
	}
}


/*----------------------------------------------------------------------------
;BVIntersect
Title:		
Globals:	
Return:		
Descrip:	Bitwise intersection (AND) of bitvectors

------------------------------------------------------------------------------*/
BITVECTOR *BVIntersect( BITVECTOR *bv1, BITVECTOR *bv2, unsigned long byteSize)
{
	unsigned long i;

	if( !bv1 || !bv2)
	{
		return 0;
	}

	for( i = 0; i < byteSize; i++)
	{
		bv1[i] = bv1[i] & bv2[i];
	}
	return bv1;
}


/*----------------------------------------------------------------------------
;BVInvert
Title:		
Globals:	
Return:		
Descrip:	Bitwise inversion (NOT) of bitvector

------------------------------------------------------------------------------*/
BITVECTOR *BVInvert( BITVECTOR *bv, unsigned long byteSize)
{
	unsigned long i;

	if( !bv)
	{
		return 0;
	}

	for( i = 0; i < byteSize; i++)
	{
		bv[i] = ~bv[i];
	}
	return bv;
}


/*----------------------------------------------------------------------------
;BVUnion
Title:		
Globals:	
Return:		
Descrip:	Bitwise union (OR) of bitvectors

------------------------------------------------------------------------------*/
BITVECTOR *BVUnion( BITVECTOR *bv1, BITVECTOR *bv2, unsigned long byteSize)
{
	unsigned long i;

	if( !bv1 || ! bv2)
	{
		return 0;
	}

	for( i = 0; i < byteSize; i++)
	{
		bv1[i] = bv1[i] | bv2[i];
	}
	return bv1;
}


/*----------------------------------------------------------------------------
;BVEmpty
Title:		
Globals:	
Return:		
Descrip:	Zero initialize bitvector

------------------------------------------------------------------------------*/
BITVECTOR *BVEmpty( BITVECTOR *bv, unsigned long msgCount, int initTo)
{
	unsigned long bvSize;
	size_t        memorySize;

	/*-------------------------- */
	/* don't cause a memory leak || don't malloc(0) */
	if ((bv) || (msgCount == 0))
	{
		return 0;
	}

	/*------------------ */
	/* size of bitvector */
	bvSize = msgCount / BITCOUNT;
	if( msgCount % BITCOUNT)
	{
		bvSize++;
	}
	memorySize = bvSize * ELEMENTSIZE;

	/*------------------------------ */
	/* allocate memory for bitvector */
	bv = (BITVECTOR*)MemMalloc( memorySize);
	if( !bv)
	{
		/* TODO - choose correct error msg */
		/*SendClient( Client, MSG5001NOMEMORY, MSG5001NOMEMORY_LEN); */
		return 0;
	}
	memset( bv, initTo, memorySize);

	return bv;
}


/*----------------------------------------------------------------------------
;ImapSearchQuery
Title:		
Globals:	
Return:		
Descrip:	Send query to and receive answer from NMAP using the selected format

------------------------------------------------------------------------------*/
BITVECTOR *ImapSearchQuery( IMAPSEARCH *Search)
{
	char         Reply[3072];
	char        *temp1; /* = (char*)bv; */
	unsigned int i, j;
	BITVECTOR   *bv = 0;

/*	return ImapSearchQuery2( Search); */

	memset( Reply, 0, sizeof( Reply));
	sprintf( Search->bufferOut+strlen( Search->bufferOut), "\r\n");
	SendNMAPServer( Search->Client, Search->bufferOut, strlen( Search->bufferOut));
	GetNMAPAnswer( Search->Client, Reply, sizeof( Reply), FALSE);
	/* reset bufferOut for each new search key since we are sending 1 query */
	Search->offsetOut = 0;
	Emit( Search, string, g_key[ SEARCHsy]);
	Emit( Search, string, "HEX");
	Search->bufferOut[Search->offsetOut] = 0;

	/*------------------------------ */
	/* allocate memory for bitvector */
	bv = (BITVECTOR*)MemMalloc( Search->bvByteSize);
	if(!bv) {
		/* TODO - choose correct error msg */
		/*SendClient( Search->Client, MSG5001NOMEMORY, MSG5001NOMEMORY_LEN); */
		return 0;
	}
	memset( bv, 0, Search->bvByteSize);

	temp1 = (char*)bv;

	for( i = 0, j = 0; i < Search->bvByteSize; i++)	{
		switch( Reply[j]) {						/* high nibble */
			case '0': temp1[i] = 0x00; break;
			case '1': temp1[i] = 0x10; break;
			case '2': temp1[i] = 0x20; break;
			case '3': temp1[i] = 0x30; break;
			case '4': temp1[i] = 0x40; break;
			case '5': temp1[i] = 0x50; break;
			case '6': temp1[i] = 0x60; break;
			case '7': temp1[i] = 0x70; break;
			case '8': temp1[i] = 0x80; break;
			case '9': temp1[i] = 0x90; break;
			case 'A': temp1[i] = 0xA0; break;
			case 'B': temp1[i] = 0xB0; break;
			case 'C': temp1[i] = 0xC0; break;
			case 'D': temp1[i] = 0xD0; break;
			case 'E': temp1[i] = 0xE0; break;
			case 'F': temp1[i] = 0xF0; break;
		}

		j++;

		switch( Reply[j]) { /* low nibble */
			case '0': temp1[i] |= 0x0; break;
			case '1': temp1[i] |= 0x1; break;
			case '2': temp1[i] |= 0x2; break;
			case '3': temp1[i] |= 0x3; break;
			case '4': temp1[i] |= 0x4; break;
			case '5': temp1[i] |= 0x5; break;
			case '6': temp1[i] |= 0x6; break;
			case '7': temp1[i] |= 0x7; break;
			case '8': temp1[i] |= 0x8; break;
			case '9': temp1[i] |= 0x9; break;
			case 'A': temp1[i] |= 0xA; break;
			case 'B': temp1[i] |= 0xB; break;
			case 'C': temp1[i] |= 0xC; break;
			case 'D': temp1[i] |= 0xD; break;
			case 'E': temp1[i] |= 0xE; break;
			case 'F': temp1[i] |= 0xF; break;
		}

		j++;
	}
	return bv;
}

void
InitSearchModule(void)
{
	return;
}
