// Copyright 2017 The Hugo Authors. All rights reserved.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
// http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package helpers

import (
	"io"
	"strconv"
	"sync/atomic"

	"github.com/olekukonko/tablewriter"
)

type ProcessingStats struct {
	Name string

	Pages           uint64
	PaginatorPages  uint64
	Static          uint64
	ProcessedImages uint64
	Files           uint64
	Aliases         uint64
	Sitemaps        uint64
	Cleaned         uint64
}

type processingStatsTitleVal struct {
	name string
	val  uint64
}

func (s *ProcessingStats) toVals() []processingStatsTitleVal {
	return []processingStatsTitleVal{
		processingStatsTitleVal{"Pages", s.Pages},
		processingStatsTitleVal{"Paginator pages", s.PaginatorPages},
		processingStatsTitleVal{"Non-page files", s.Files},
		processingStatsTitleVal{"Static files", s.Static},
		processingStatsTitleVal{"Processed images", s.ProcessedImages},
		processingStatsTitleVal{"Aliases", s.Aliases},
		processingStatsTitleVal{"Sitemaps", s.Sitemaps},
		processingStatsTitleVal{"Cleaned", s.Cleaned},
	}
}

func NewProcessingStats(name string) *ProcessingStats {
	return &ProcessingStats{Name: name}
}

func (s *ProcessingStats) Incr(counter *uint64) {
	atomic.AddUint64(counter, 1)
}

func (s *ProcessingStats) Add(counter *uint64, amount int) {
	atomic.AddUint64(counter, uint64(amount))
}

func (s *ProcessingStats) Table(w io.Writer) {
	titleVals := s.toVals()
	data := make([][]string, len(titleVals))
	for i, tv := range titleVals {
		data[i] = []string{tv.name, strconv.Itoa(int(tv.val))}
	}

	table := tablewriter.NewWriter(w)

	table.AppendBulk(data)
	table.SetHeader([]string{"", s.Name})
	table.SetBorder(false)
	table.Render()

}

func ProcessingStatsTable(w io.Writer, stats ...*ProcessingStats) {
	names := make([]string, len(stats)+1)

	var data [][]string

	for i := 0; i < len(stats); i++ {
		stat := stats[i]
		names[i+1] = stat.Name

		titleVals := stat.toVals()

		if i == 0 {
			data = make([][]string, len(titleVals))
		}

		for j, tv := range titleVals {
			if i == 0 {
				data[j] = []string{tv.name, strconv.Itoa(int(tv.val))}
			} else {
				data[j] = append(data[j], strconv.Itoa(int(tv.val)))
			}

		}

	}

	table := tablewriter.NewWriter(w)

	table.AppendBulk(data)
	table.SetHeader(names)
	table.SetBorder(false)
	table.Render()

}
