#!/usr/bin/env python
# Copyright (C) 2005-2006 Sivan Greenberg. <sivan@ubuntu.com>
# 
# This software is distributed under the terms and conditions of the GNU General
# Public License. See http://www.gnu.org/copyleft/gpl.html for details.

import sys
import dbus
import fsMisc

class RawDeviceDictionary(dict):
    """
    When instantiated, DeviceDictionary instance will have all of system's devices information
    fetched from HAL, accessible by the "info.udi" property as a key.

    Usage:
        from DeviceInfo import DeviceDictionary
        a = DeviceDictionary()
        CdromInformation = a["/org/freedesktop/Hal/devices/storage_model_HL_DT_STDVD_ROM_GDR8163B"]
        # a Now contains all the properties for the DVD ROM drive
        # Let's check if this is really a dvd capable drive:
        print CdromInformation["storage.cdrom.dvd"]
        True
    """
    def __init__(self,category=['All']):
        checkHalIsRunning()
        self.bus = dbus.SystemBus()
        self.hal_manager_obj = self.bus.get_object("org.freedesktop.Hal","/org/freedesktop/Hal/Manager")
        self.hal_manager = dbus.Interface(self.hal_manager_obj,"org.freedesktop.Hal.Manager")
        device_names = self.hal_manager.GetAllDevices()
        device_names.sort()
        for name in device_names:
            device_dbus_obj = self.bus.get_object("org.freedesktop.Hal" ,name)
            properties = device_dbus_obj.GetAllProperties(dbus_interface="org.freedesktop.Hal.Device")
            if category[0] != "All":
                if properties.has_key("info.category"): 
                    if properties["info.category"] in category:
                            self[properties["info.udi"]] = properties
            else:
                self[properties["info.udi"]] = properties


class StorageDeviceDict(dict):
    """
    Instantiates into an dictionary holding more readable form
    of available media devices on a user's machine. This includes
    information about any child volumes the device might have.
    """
    def __init__(self):
        SVDevices = RawDeviceDictionary(category=['storage','volume'])
        for DeviceUDI in SVDevices:
            Name = Type = Product = Vendor = ReadWrite = ""
            DevNode =  SVDevices[DeviceUDI]["block.device"]
            DeviceProperties = SVDevices[DeviceUDI]
            if DeviceProperties["info.category"] == "storage":
                try:
                    Product =  DeviceProperties["info.product"]
                except KeyError:
                    Product = ''
                try:
                    Vendor  =  DeviceProperties["storage.vendor"]
                except KeyError:
                    Vendor = ''
                Name = 1 and Product or Vendor
                Type    =  DeviceProperties["storage.drive_type"]
                ReadWrite = 0
                if Type=="cdrom":
                    ReadWrite = 1 or \
                                DeviceProperties["storage.cdrom.cdr"] or \
                                DeviceProperties["storage.cdrom.cdrw"] or \
                                DeviceProperties["storage.cdrom.dvdram"] or \
                                DeviceProperties["storage.cdrom.dvdrw"]
                    if not Name:
                        Name = 'Unknown CDROM'
                else:
                    ReadWrite = 1 and Type == "disk" or Type == "floppy"
                    if not Name:
                        Name = 'Unknown drive'
                Removable = 1 and DeviceProperties["storage.removable"]
                self[DevNode] = { 'udi' : DeviceProperties["info.udi"],
                                    'Name' : Name,
                                    'Type' : Type,
                                    'ReadWrite' : ReadWrite,
                                    'Removable' : Removable,
                                    'Empty' : True,
                                    'Volumes' : {} }

        for DeviceUDI in SVDevices:
            DevNode = SVDevices[DeviceUDI]["block.device"]
            DeviceProperties = SVDevices[DeviceUDI]
            Parent = DeviceProperties["info.parent"]
            if DeviceProperties["info.category"] == "volume":
                for dev in self:
                    if SVDevices[self[dev]['udi']]['info.category'] != "volume" and Parent == self[dev]['udi']:
                        self[dev]['Empty'] = False
                        self[dev]['Volumes'].setdefault(DevNode,{})["VolumeIsMounted"] = DeviceProperties["volume.is_mounted"]
                        self[dev]['Volumes'].setdefault(DevNode,{})["VolumeMountPoint"] = DeviceProperties["volume.mount_point"]
                        self[dev]['Volumes'].setdefault(DevNode,{})["VolumeLabel"] = DeviceProperties["volume.label"]
                        self[dev]['Volumes'].setdefault(DevNode,{})["VolumeSize"] = DeviceProperties["volume.size"]
                        self[dev]['Volumes'].setdefault(DevNode,{})["FreeSpaceBytes"] = fsMisc.free_space(DeviceProperties["volume.mount_point"])
                        if DeviceProperties["volume.is_disc"]:
                            self[dev]['Volumes'].setdefault(DevNode,{})["Blank"] = DeviceProperties["volume.disc.is_blank"]
                            self[dev]['Volumes'].setdefault(DevNode,{})["Data"] = DeviceProperties["volume.disc.has_data"]
                            self[dev]['Volumes'].setdefault(DevNode,{})["Rewritable"] = DeviceProperties["volume.disc.is_rewritable"]
                            self[dev]['Volumes'].setdefault(DevNode,{})["Appendable"] = DeviceProperties["volume.disc.is_appendable"]
                            try:
                                self[dev]['Volumes'].setdefault(DevNode,{})["Capacity"] = DeviceProperties["volume.disc.capacity"]
                            except KeyError:
                                self[dev]['Volumes'].setdefault(DevNode,{})["Capacity"] = "N/A"


class HUBDeviceDict(StorageDeviceDict):
    """
    specify 'rwOnly=true' to include in the list only the devices
    that can be written to. This will exclude devices that support
    read only access.

    By default lists all storage devices.
    """
    def __init__(self,rwOnly=False,rmvOnly=False):
        StorageDeviceDict.__init__(self)
        tmp = [dev for dev in self \
                if (rwOnly and not self[dev]['ReadWrite']) or \
                    (rmvOnly and not self[dev]['Removable']) ]
        for dev in tmp:
            self.pop(dev)



class NoHaldError(Exception):
    pass


def checkHalIsRunning(guiControlled=False):
    bus = dbus.SystemBus()
    proxy_obj = bus.get_object('org.freedesktop.DBus', '/org/freedesktop/DBus')
    dbus_iface = dbus.Interface(proxy_obj, 'org.freedesktop.DBus')
    if 'org.freedesktop.Hal' not in dbus_iface.ListNames():
#        if guiControlled:
#            return False
#        else:
            raise NoHaldError
                                        



def print_devs(RawDeviceDictionary): # used to get print outs of RawDeviceDictionary
    if RawDeviceDict != None:
       for key in RawDeviceDict:
            print "\n * Device UDI %s \n" % key
            for thisKey in RawDeviceDict[key]:
                print "\t %s => %s " % (thisKey,RawDeviceDict[key][thisKey])
            print "\n"
    else:
        print "hald is not running!"


def print_SDD(sdict): # generic print outs of key => value hashes
    for dev in sdict:
        print " %s ==> %s " % (dev, sdict[dev])


if __name__ == "__main__":
    try:
        a = HUBDeviceDict(True,True)
    except NoHaldError:
        print "Cannot get device list. Check that you'r /usr/sbin/hald is running?"
        sys.exit(1)
    print_SDD(a) 
