#!/usr/bin/env python
# Copyright (C) 2005-2006 Sivan Greenberg. <sivan@ubuntu.com>
# 
# This software is distributed under the terms and conditions of the GNU General
# Public License. See http://www.gnu.org/copyleft/gpl.html for details.

import sys
import os
import re
from DeviceInfo import *
import subprocess
from pexpect import *

DEBUG_MODE = True


class NotBurnDeviceError(Exception):
    pass

class NonExistantDevice(Exception):
    pass

class CannotUnMountError(Exception):
    pass


class CDBurner:
    def __init__(self,device_node,isofilepath, multisession=False):
        self.curPrecent = self.prevPrecent = 0        
        if multisession:
            if DEBUG_MODE: print "* Preparing a multi session CD"
            self.BURN_CMDLINE = " -vv -eject dev=%s  -multi -pad -data %s" # multisession burn
        else:
            self.BURN_CMDLINE = " -vv -eject dev=%s -data %s"
            
        self.device_node = device_node
        self.isofilepath = isofilepath
        
        self.curOutputLine = ''
        self.oper_log = []
        
        devlist = HUBDeviceDict(True,True)
        
        try:
            self.properties = devlist[device_node]
        except KeyError:
            raise NonExistantDevice
        
        if self.properties["Type"] != "cdrom":
            raise NotBurnDeviceError

        # check if this device is already mounted and thus inaccessible and
        # automatically unmount it.
        try:
            aVolumeIsMounted = self.properties['Volumes'][device_node]['VolumeIsMounted']
        except KeyError:
            # no volumes on this target so it cannot be mounted, do nothing
            aVolumeIsMounted = False

        if aVolumeIsMounted:
            if DEBUG_MODE: print "* Attempting to unmount %s" % device_node
            retcode = subprocess.call(["pumount",device_node])
            if retcode!=0:
                raise CannotUnMountError
        

        if DEBUG_MODE: print "self.properties: ", self.properties ,"\n"
        self.isofile = isofilepath

        self.BURN_CMDLINE = self.BURN_CMDLINE % (self.device_node, self.isofilepath)
        if DEBUG_MODE: print "self.BURN_CMDLINE: ", self.BURN_CMDLINE ,"\n"

    def pre_build_kick(self): # returns the spawned object reference for further manipulation by GUI code.
        if DEBUG_MODE: print "Executing cdrecord to burn image file to cd."
        self.proc = spawn("cdrecord",self.BURN_CMDLINE.split())
        # COMMENT TO ALLOW TESTING AROUND THE BURNING PROCESS, AFTER IT'S BEEN TESTED FOR A WHILE.
        # DUMMY BURN COMMAND = ls , PLACEHOLDER = ****
        #self.proc = spawn("ls",["-la"])
        return self.proc
      
    def build(self):
        outputLine = ''
        proc_isatty = True
        while proc_isatty:
            try:
                outputLine = self.proc.read_nonblocking(80,1)
                outputLine.strip()
                print "outputLine = [%s]" % outputLine
            except TIMEOUT:
                yield True
            if outputLine!='':
                self.curOutputLine = outputLine
                self.oper_log.append(self.curOutputLine)
            proc_isatty = self.proc.isalive()
            yield True
        yield False


    def reportLine(self):
        return self.curOutputLine

    def reportPrecent(self):
        regexp = re.compile(r'.*Track\ \d*\:\ *(\d*)\ *of\ *(\d*)\ *MB',re.IGNORECASE)
        curLine = self.curOutputLine        
        match = regexp.match(curLine)
        if match:
            try:
                portion = float(match.group(1))
                total = float(match.group(2)) 
                print "portion = %s" % portion
                print "total = %s" % total
                if total==0:
                    total = portion = 1
                self.curPrecent = int((portion / total) * 100)
            except ValueError:
                self.curPrecent = self.prevPrecent
        else:
            self.curPrecent = self.prevPrecent
        self.prevPrecent = self.curPrecent
        return self.curPrecent

    def reportStatus(self):
        regexpFixating = re.compile(r'TBD',re.IGNORECASE)
        curLine = self.curOutputLine
        matchFixating = regexpFixating.match(curOutputLine)
        # to be continued...
        

if __name__ == "__main__":
    try:
        device_node = sys.argv[1]
        isofilepath = sys.argv[2]
    except IndexError:
        print "Please provide burner file system device node and an iso file to burn."
        sys.exit(1)
    try:
        Burner = CDBurner(device_node,isofilepath,False)
    except NotBurnDeviceError, NonExistantDevice:
        print " %s does not seem to be a valid cdrom device. try another?" % device_node
        sys.exit(1)
    procObj = Burner.pre_build_kick()
    g = Burner.build()
    while g.next():
        print "output from subprocess: %s" % Burner.reportLine()
    if DEBUG_MODE:
        print "log:"
        print Burner.oper_log
    


    
