/***************************************************************************
 * C++ Implementation:                                                     *
 * Copyright (C) 2012-2014 by Eduard Kalinowski                            *
 * Germany, Lower Saxony, Hanover                                          *
 * eduard_kalinowski@yahoo.de                                              *
 *                                                                         *
 * HTTraQt is free software; you can redistribute it and/or                *
 * modify it under the terms of the GNU Lesser General Public              *
 * License as published by the Free Software Foundation; either            *
 * version 3 of the License, or (at your option) any later version.        *
 *                                                                         *
 * HTTraQt is distributed in the hope that it will be useful, but          *
 * WITHOUT ANY WARRANTY; without even the implied warranty of              *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.                    *
 *                                                                         *
 * See the GNU Lesser General Public License for more details.             *
 *                                                                         *
 * You should have received a copy of the GNU Lesser General Public        *
 * License along with HTTraQt. If not, see  http://www.gnu.org/licenses    *
 ***************************************************************************/

#include <QtGui>
#include <QSystemTrayIcon>
#include <QApplication>
#include <QMenu>
#include <QDir>
#include <QIcon>
#include <QTreeView>
#include <QSound>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <exception>
#include <assert.h>

// #include "./includes/htinterface.h"
// #include "./includes/htsstrings.h"

#include "htinterface.h"
#include "htsstrings.h"

#include "../version.h"

#include "includes/httraqt.h"

#include "ui_mainForm.h"

#include "includes/buttonPanel.h"
#include "../options/includes/OptionsDialog.h"
#include "includes/AboutDialog.h"
#include "includes/NewProjTab.h"
#include "includes/OptionsTab.h"
#include "includes/StartTab.h"
#include "includes/ConfirmTab.h"
#include "includes/ProgressTab.h"
#include "includes/FinalTab.h"


class OptionsDialog;

class ProgressTab;


bool writeWIniFile(QIODevice &device, const QSettings::SettingsMap &map)
{
    QDataStream out(&device);

    if ( device.isOpen () && ! map.empty() ) {
        QMap<QString, QVariant>::const_iterator it;

        for ( it = map.begin(); it != map.end(); ++it ) {
            QString buffer;
            QString val = it.value().toString();

            if (val.length() > 0) {
                if (val.indexOf("\"") == 0) {
                    val.remove(0, 1);
                }

                if (val.lastIndexOf("\"") == val.length() - 1) {
                    val.remove(val.length() - 1, 1);
                }

                val.remove("%0d%0a");
            }

            buffer = QString ( "%1=%2\n" ).arg ( it.key(), val );

            if ( ! buffer.isEmpty() ) {
                device.write ( buffer.toAscii() );
            }

            buffer.clear();
        }

        return true;
    }

    return false;
}


bool readWIniFile(QIODevice &device, QSettings::SettingsMap &map)
{
    QDataStream in(&device);

    if ( device.isOpen () ) {
        if ( ! map.empty() ) {
            map.clear();
        }

        while ( ! device.atEnd() ) {
            QByteArray line = device.readLine().trimmed();

            if ( ! line.startsWith ( "#" ) && line.contains ( "=" ) ) {
                int pos = line.indexOf("=");
                QString key = line.left(pos).trimmed ();// tmp.at ( 0 ).trimmed() );
                QString val = line.mid(pos + 1).trimmed(); // tmp.at ( 1 ).trimmed() );

                if (val.length() > 0) {
                    if (val.indexOf("\"") == 0) {
                        val.remove(0, 1);
                    }

                    if (val.lastIndexOf("\"") == val.length() - 1) {
                        val.remove(val.length() - 1, 1);
                    }

                    val.remove("%0d%0a");
                }

                if ( ! key.isEmpty() /*&& ! val.isEmpty()*/ ) {
                    if ( val.contains ( QRegExp ( "^[0-9]+$" ) ) ) {
                        map.insert ( key, QVariant ( val ).toInt() );
                    } else {
                        map.insert ( key, QVariant ( val ).toString() );
                    }
                }
            }
        }

        return true;
    }

    return false;
}


static const QSettings::Format WIniFormat = QSettings:: registerFormat ("ini", readWIniFile, writeWIniFile);

HTTraQt* mainWidget;

using namespace std;



QString translateString ( int id )
{
    QString m;

    if (id < 0) {
        return "";
    }

    if (id >= 0 && id < (sizeof(engText) / sizeof(char*))) {
        m = QString(engText[id]);

        if (id == 383 || id == 401) {
            m.replace("%HOME%", QString(QDir::homePath() + "/"));
        }
    }

    if (mainWidget->translateTable.size() == 0) {
        return m;
    }

    for (QStringList::Iterator iter = mainWidget->translateTable.begin(); iter != mainWidget->translateTable.end(); ++iter) {
        if (m == (*iter)) {
            iter++;

            if (m.simplified().length() >= 1) { // if nothing
                m = (*iter);
            }

            break;
        }
    }

    //     m.replace("&amp;", "&");
    //     m.replace( "&gt;", ">");
    //     m.replace( "&lt;", "<");

    return m;
}



int messageBox(const QString &title, const QString &text, int ticon)
{
    int ret;
    QMessageBox* msgBox = NULL;

    if (ticon == QMessageBox::Question) {
        msgBox = new QMessageBox(QMessageBox::Question, title, text, QMessageBox::Yes | QMessageBox::No);

        msgBox->setButtonText(QMessageBox::Yes, translateString(_YES));
        msgBox->setButtonText(QMessageBox::No, translateString(_NO));
    }

    if (ticon == QMessageBox::Critical || ticon == QMessageBox::Warning || QMessageBox::Information) {
        if (ticon == QMessageBox::Critical) {
            msgBox = new QMessageBox(QMessageBox::Critical, title, text, QMessageBox::Ok);
        }

        if (ticon == QMessageBox::Warning) {
            msgBox = new QMessageBox(QMessageBox::Warning, title, text, QMessageBox::Ok);
        }

        if (ticon == QMessageBox::Information) {
            msgBox = new QMessageBox(QMessageBox::Information, title, text, QMessageBox::Ok);
        }

        msgBox->setButtonText(QMessageBox::Ok, translateString(_OK));
    }

    if (msgBox == NULL) {
        //         qDebug() << "not found dialog" << ticon;
        return 0;
    }

    if (mainWidget->programStyleSheet.length() > 0) {
        msgBox->setStyleSheet(mainWidget->programStyleSheet);
    }

    ret = msgBox->exec();

    delete msgBox;
    //     qDebug() << "mydialog" << ret;
    return ret;
}


void MyThread::run()
{
    httraq_main();
}


#if 0
void TimerBox::showEvent ( QShowEvent * event )
{
    currentTime = 0;
    QString s;
    QTime t(0, 0, 0);
    QTime n;
    n = t.addSecs(timeout - currentTime);

    s = translateString(_WAITTIME) + n.toString(" hh:mm:ss");
    setText(s);

    if (autoClose) {
        this->startTimer(1000);
    }
}


void TimerBox::setTimeout ( int t )
{
    if (t <= 0) {
        timeout = 0;
    } else {
        timeout = t;
    }
}


void TimerBox::setAutoClose ( bool a )
{
    autoClose = a;
}


void TimerBox::timerEvent(QTimerEvent *event)
{
    currentTime++;

    if (currentTime >= timeout) {
        this->done(0);
    }

    QString s;
    QTime t(0, 0, 0);
    QTime n;
    n = t.addSecs(timeout - currentTime);

    s = translateString(_WAITTIME) + n.toString(" hh:mm:ss");
    setText(s);
}
#endif


QVariant DirModel::headerData ( int section, Qt::Orientation orientation, int role ) const
{
    if (orientation == Qt::Horizontal) {
        if (role != Qt::DisplayRole) {
            return QVariant();
        }

        switch (section) {
            case 0:
                return mainWidget->currentWorkDir;

                //             case 1: return tr("My Size Header");
                //             case 2: return tr("My Type Header");
                //             case 3: return tr("My Modified Header");
            default:
                return QVariant();
        }
    }
}


void HTTraQt::launch()
{
    cmdArgumentsOptions();

    writeSettings(false);

    initSInfo();

    // wait time, if entered
    QString hh, mm, ss;
    bool okHH, okMM, okSS;
    int  intHH, intMM, intSS;
    intHH = intMM = intSS = 0;

    hh = (*(ConfirmTab*)widgets[3]).labelHH->text(); // HH
    mm = (*(ConfirmTab*)widgets[3]).labelMM->text(); // MM
    ss = (*(ConfirmTab*)widgets[3]).labelSS->text(); // SS

    if (hh.length() > 0) {
        intHH = hh.toInt(&okHH);
    }

    if (intHH < 0 || intHH > 23) {
        okHH = false;
        intHH = 0;
    }

    if (mm.length() > 0) {
        intMM = mm.toInt(&okMM);
    }

    if (intMM < 0 || intMM > 59) {
        okMM = false;
        intMM = 0;
    }

    if (ss.length() > 0) {
        intSS = ss.toInt(&okSS);
    }

    if (intSS < 0 || intSS > 59) {
        okSS = false;
        intSS = 0;
    }

    if (intHH > 0 || intMM > 0 || intSS > 0) {
        int mSec = 1000 * (intHH * 3600 + intMM * 60 + intSS);
        QTimer::singleShot(mSec, this, SLOT(onStopAll()));
        //         TimerBox msgBox;
        //         msgBox.setIcon(QMessageBox::Information);
        //         msgBox.setStandardButtons(QMessageBox::Ok);
        //         msgBox.setAutoClose(true);
        //         msgBox.setTimeout(intHH * 3600 + intMM * 60 + intSS); //Closes after three seconds
        //         msgBox.exec();
    }

    int result = 0;
    {
        QString projectDir;
        projectDir =  currentWorkDir + "/" + currentProject + "/";

        // on efface le doit.log, pour annuler les parametres anciens et en redéfinir de nouveaux
        // c'est ici une logique qui diffère de la version en ligne de commande
        if (QFile::exists(projectDir + "hts-cache/new.zip") || QFile::exists(projectDir + "hts-cache/new.ndx")) {    // un cache est présent
            QFile fl(projectDir + "hts-cache/doit.log");

            if (fl.exists()) {
                fl.remove();
            }

            if (fl.open(QFile::WriteOnly) == true) {
                fl.close();
            }
        }
    }


    //     if (global_opt != NULL) {
    //         hts_free_opt(global_opt);
    //         global_opt = NULL;
    //     }

    termine = 0;
    termine_requested = 0;

    //     global_opt = hts_create_opt();

    timerProgressUpdate->start(1000); //progress tab ones per second to update

    timerDirRefresh->start(10000); // ones per 10 seconds


    mth = new MyThread;

    // options correcture after starting
    //     QTimer::singleShot(1000, this, SLOT(setOptions()));

    connect(mth, SIGNAL(finished ()), this, SLOT(threadFinished()));
    connect(mth, SIGNAL(terminated ()), this, SLOT(threadFinished()));

    mth->start();
}


void HTTraQt::refreshDirModel()
{
    dModel.refresh();
    statusBar()->showMessage("Update dir list", 1000);
}


HTTraQt::HTTraQt(QWidget* parent, Qt::WindowFlags flags)
    : QMainWindow(parent, flags)
{
    setupUi(this);

    mainWidget = this;

    currentAppDir = qApp->applicationDirPath();

    if (currentAppDir.lastIndexOf("/build") > 0) { // build dir detection
        currentAppDir.remove("/build");
        //         currentAppDir = QDir::currentPath();
    }

    currentLang = "English";

    _singular = new QSharedMemory("HTTraQt", this);


    QFont sysFont = qApp->font();
    sysFont = sysFont;
    // now it's depended from system settings ;)
    //     sysFont.setPointSize(10);

    fontSize = sysFont.pointSize();

    mth = NULL;

    readGUISettings();

    //     createFontSizeMenu();

    programStyleSheet = QString().sprintf("font-size: %dpt", fontSize);

    if ( fontSize == -1) {
        fontSize = sysFont.pixelSize();
        programStyleSheet = QString().sprintf("font-size: %dpx", fontSize);
    }

    if (programStyleSheet.length() > 0) {
        setStyleSheet(programStyleSheet);
    }


    if (readLangDir() == false) { // init from langFiles variable in format "filename:language"
        messageBox(translateString(_ERR),
                   "Directory with other languages not found\nDefault GUI language is english", QMessageBox::Critical);
    }

    currentTab = 0;

    groupMain->setFont(sysFont);
    groupButtons->setFont(sysFont);
    dirView->setFont(sysFont);

    // this function i need only for converting! normally is disabled
    // old help file format -> new
    //     convertTranslateFile();

    buttonsWidget = new buttonPanel(this);
    gridLayoutB->addWidget(buttonsWidget);

    widgets[0] = new StartTab(this);
    gridLayoutM->addWidget(widgets[0]);

    widgets[1] = new NewProjTab(this);
    gridLayoutM->addWidget(widgets[1]);

    widgets[2] = new OptionsTab(this);
    gridLayoutM->addWidget(widgets[2]);

    widgets[3] = new ConfirmTab(this);
    gridLayoutM->addWidget(widgets[3]);

    widgets[4] = new ProgressTab(this);
    gridLayoutM->addWidget(widgets[4]);

    widgets[5] = new FinalTab(this);
    gridLayoutM->addWidget(widgets[5]);

    setFontForWidgets();

    timerProgressUpdate = new QTimer(this);
    timerDirRefresh =  new QTimer(this);

    connect(timerProgressUpdate, SIGNAL(timeout()), widgets[4], SLOT(update()));
    connect(timerDirRefresh, SIGNAL(timeout()), this, SLOT(refreshDirModel()));


    //     connect (groupMain, SIGNAL( resizeEvent(QResizeEvent*) ), this, SLOT(resizeTabs(QResizeEvent*)));

    // flag de termination

    termine = 0;
    process = NULL;
    termine_requested = 0;
    shell_terminated = 0;
    soft_term_requested = 0;

    //     QVBoxLayout* layout = new QVBoxLayout(splitter_2);
    //     layout->addWidget(splitter);

    setWindowIcon(QIcon(main_xpm));

    dModel.setReadOnly(true);

    dirView->setModel(&dModel);

    setCentralWidget(centralwidget);

    dirView->setColumnHidden(1, true);
    dirView->setColumnHidden(2, true);
    dirView->setColumnHidden(3, true);
    dirView->setAnimated(false);
    dirView->setIndentation(20);
    dirView->setSortingEnabled(true);

    connect(dirView, SIGNAL(expanded(const QModelIndex &)), this, SLOT(treeItemClicked(const QModelIndex &)));
    connect(dirView, SIGNAL(collapsed(const QModelIndex &)), this, SLOT(treeItemClicked(const QModelIndex &)));

    resetDefaultOptions();

    readSettings(true); // global settings

    statusBar()->showMessage("Current directory: " + currentWorkDir);

    createActions();
    createToolBars();
    createStatusBar();

    createTrayIcon();

    if (getLangTable() == false) {
        messageBox(translateString(_ERR),
                   "Can't open language file!\nDefault GUI language is english", QMessageBox::Critical);

        currentLang = "English";
    }

    for (QVector<QAction*>::iterator itL = actLangSelect.begin(); itL != actLangSelect.end(); ++itL) {
        if ((*itL)->text() == currentLang) {
            (*itL)->setChecked(true);
            break;
        }
    }

#if 0

    if(checkInstanceRuns() == true) {
        int res = QMessageBox::question(0, "Warning", "Application already running\nDo you want to continue?", QMessageBox::Yes | QMessageBox::No);

        if (res == QMessageBox::No) {
            exit(-1);
        }
    }

#endif
    setLangGUI();

    activatePage(0);
}


void HTTraQt::resizeTabs(QResizeEvent* se)
{
    qDebug() << "resize";
}

void HTTraQt::setFontForWidgets()
{
    buttonsWidget->setStyleSheet(programStyleSheet);

    for (int i = 0; i < 6; i++) {
        widgets[i]->setStyleSheet(programStyleSheet);
    }
}


void HTTraQt::clearStatsBuffer(void)
{
    // SInfo.refresh
    for (int i = 0; i < NStatsBuffer; i++) {
        StatsBuffer[i].state = ""; // etat
        StatsBuffer[i].name = "";  // nom
        StatsBuffer[i].file = "";  // fichier
        StatsBuffer[i].url_sav = "";
        StatsBuffer[i].back = NULL;
        StatsBuffer[i].size = 0;
        StatsBuffer[i].sizetot = 0;
    }
}


void HTTraQt::threadFinished(void)
{
    if (global_opt != NULL) {
        hts_free_opt(global_opt);
        global_opt = NULL;
    }

    buttonsWidget->onNext(); // to log page
    onEndMirror();

    timerProgressUpdate->stop();
    timerDirRefresh->stop();

    /* Aborted mirror or finished? */
    {
        QString projectDir;
        projectDir =  currentWorkDir + "/" + currentProject + "/";

        if (soft_term_requested || termine_requested) {
            QFile fl(projectDir + "hts-cache/interrupted.lock");

            if (fl.open(QFile::WriteOnly) == true) {
                fl.close();
            }
        } else {
            QFile::remove(projectDir + "hts-cache/interrupted.lock");
        }
    }

    delete mth;
    mth = NULL;

    //     correctIndexHtml();
}


//
bool HTTraQt::treeItemClicked(const QModelIndex &m)
{
    //     dirView->resizeColumnToContents(0);
}


// true if new project
bool HTTraQt::rebuildWorkDirView()
{
    bool newProj = false;

    if (currentWorkDir.length() == 0) {
        //         currentWorkDir = QDir::homePath() + "/My Web Sites";
        readSettings(true);
    }

    QDir wDir(currentWorkDir);

    if (wDir.exists() == false) {
        wDir.mkpath(currentWorkDir);
        newProj = true;
    }

    dirView->reset();
    dirView->setRootIndex(dModel.index(currentWorkDir));
    dModel.refresh();

    // something in hts-cache?
    if (checkContinue(false) == false) {
        newProj = true;
    }

    return newProj;
}


void HTTraQt::initSInfo()
{
    //     SInfo.ask_refresh = 0;
    SInfo.refresh = 0;
    SInfo.stat_bytes = 0;
    SInfo.stat_time = 0;
    SInfo.lien_n = 0;
    SInfo.lien_tot = 0;
    SInfo.stat_nsocket = 0;
    SInfo.rate = 0;
    SInfo.irate = 0;
    SInfo.ft = 0;
    SInfo.stat_written = 0;
    SInfo.stat_updated = 0;
    SInfo.stat_errors = 0;
    SInfo.stat_warnings = 0;
    SInfo.stat_infos = 0;
    SInfo.stat_timestart = mtime_local();
    SInfo.stat_back = 0;
}


// reprise possible?
bool HTTraQt::checkContinue(bool msg)
{
    QString projectDir;
    projectDir =  currentWorkDir + "/" + currentProject + "/";

    if (QFile::exists(projectDir + "hts-cache/new.zip")  ||
            ((QFile::exists(projectDir + "hts-cache/new.dat")) && (QFile::exists(projectDir + "hts-cache/new.ndx")))) {  // il existe déja un cache précédent.. renommer
        return true;
    }

    if (QFile::exists(projectDir + "hts-cache/old.zip") ||
            ((QFile::exists(projectDir + "hts-cache/old.dat")) && (QFile::exists(projectDir + "hts-cache/old.ndx")))) {  // il existe déja un ancien cache précédent.. renommer
        return true;
    }

    if ( msg == true) {
        messageBox(translateString(_ERR), translateString(_CANTFIND), QMessageBox::Critical);
    }

    return false;
}


void HTTraQt::afterChangepathlog()
{
    QString st = "";
    char tempo[8192];
    bool modify;
    QString projectDir;
    projectDir =  currentWorkDir + "/" + currentProject + "/";
#if 0

    if (fexist(fconcat(catbuff, tempo, "winprofile.ini"))) {  // un cache est présent
        if (fsize(fconcat(catbuff, tempo, "winprofile.ini")) > 0) { // taille log contrôle>0
            int i;

            for(i = 0; i < (int) strlen(tempo); i++) {
                if (tempo[i] == '/') {
                    tempo[i] = '\\';
                }
            }

            Read_profile(fconcat(catbuff, tempo, "winprofile.ini"), 0);

            // peut on modifier?
            int pos = m_ctl_todo.GetCurSel();

            if ((pos == LAST_ACTION) || (pos == LAST_ACTION - 1) || (pos == 0)) {
                modify = true;
            } else {
                modify = false;
            }

            // existe: update
            if (fexist(fconcat(catbuff, tempo, "hts-cache/new.zip")) ||
                    (fexist(fconcat(catbuff, tempo, "hts-cache/new.dat"))) && (fexist(fconcat(catbuff, tempo, "hts-cache/new.ndx")))
               ) {  // il existe déja un cache précédent.. renommer
                if (modify) {
                    if ((!fexist(fconcat(catbuff, tempo, "hts-in_progress.lock"))) &&
                            (!fexist(fconcat(catbuff, tempo, "hts-cache/interrupted.lock")))
                       ) {
                        m_ctl_todo.SetCurSel(LAST_ACTION);
                    } else {
                        m_ctl_todo.SetCurSel(LAST_ACTION - 1);
                    }
                }

                log_flip = 1;
            } else if (log_flip) {
                if (modify) {
                    m_ctl_todo.SetCurSel(0);
                }

                log_flip = 0;
            }

            OnSelchangetodo();
        }
    } else if (log_flip) {
        m_ctl_todo.SetCurSel(0);
        log_flip = 0;
    }

#endif
}



void HTTraQt::renameOldToNew()
{
    QString projectDir;
    projectDir =  currentWorkDir + "/" + currentProject + "/";

    if (QFile::exists(projectDir + "hts-cache/old.dat") && QFile::exists(projectDir + "hts-cache/old.ndx")) {
        if (QFile::remove(projectDir + "hts-cache/new.dat") == false) {
            messageBox(translateString(_ERR), translateString(_ERRDELETING), QMessageBox::Warning);
        }

        if ( QFile::remove(projectDir + "hts-cache/new.ndx") == false) {
            messageBox(translateString(_ERR), translateString(_ERRDELETING), QMessageBox::Warning);
        }
    }

    if (QFile::remove(projectDir + "hts-cache/new.lst") == false ) {
        messageBox(translateString(_ERR), translateString(_ERRDELETING), QMessageBox::Warning);
    }

    if (QFile::exists(projectDir + "hts-cache/old.zip")) {
        if (QFile::remove(projectDir + "hts-cache/new.zip") == false ) {
            messageBox(translateString(_ERR), translateString(_ERRDELETING), QMessageBox::Warning);
        }
    }

    QFile::remove(projectDir + "hts-cache/new.txt");
    QFile::rename(projectDir + "hts-cache/old.zip", projectDir + "hts-cache/new.zip");
    QFile::rename(projectDir + "hts-cache/old.dat", projectDir + "hts-cache/new.dat");
    QFile::rename(projectDir + "hts-cache/old.ndx", projectDir + "hts-cache/new.ndx");
    QFile::rename(projectDir + "hts-cache/old.lst", projectDir + "hts-cache/new.lst");
    QFile::rename(projectDir + "hts-cache/old.txt", projectDir + "hts-cache/new.txt");
}


QString HTTraQt::sizeToText(long long int s)
{
    QString t;

    if (s < 0) {
        return QString(" size is wrong!");
    }

    if (s > 1024 && s < 1024 * 1024) {
        t = QString::number(s / 1024) + " kB";
    } else if (s > (1024 * 1024)) {
        t = QString::number(s / (1024 * 1024)) + " MB";
    } else {
        t = QString::number(s / 1024) + " B";
    }

    return t;
}


void HTTraQt::onEndMirror()
{
    QStringList ext;
    QString oldName, oldShort;
    QString newName, newShort;
    long int oldSize = -1;
    long int newSize = -1;

    //     endInProgress();

    ext << "zip" << "dat" << "ndx";

    QString projectDir;
    projectDir =  currentWorkDir + "/" + currentProject + "/";

    for (QStringList::iterator iExt = ext.begin(); iExt != ext.end(); ++iExt) {
        QFile oldF;
        QFile newF;
        oldName = projectDir + "hts-cache/old." + (*iExt);
        oldShort = "old."  + (*iExt);
        newName = projectDir + "hts-cache/new." + (*iExt);
        newShort = "new."  + (*iExt);

        oldF.setFileName (oldName);
        newF.setFileName (newName);

        if (oldF.exists() == true && newF.exists() == true ) {
            oldSize = oldF.size();
            newSize = newF.size();
        }

        if (oldSize > 0 && newSize > 0) {
            break;
        }
    }

    if (oldSize == -1) { // nothing to overwrite, project was new
        return;
    }

    if (newSize == -1) {
        messageBox(translateString(_ERR), translateString(_MIRROR_ERR), QMessageBox::Warning);

        return; // nothing was found
    }

#if 0

    if (termine == 1 || termine_requested == 1) {
        if (QFile::exists(projectDir + "hts-cache/old.zip") ||
                ((QFile::exists(projectDir + "hts-cache/old.dat")) && (QFile::exists(projectDir + "hts-cache/old.ndx")))) {
            QString info;
            QString oldSizeTxt = sizeToText(oldSize);
            QString newSizeTxt = sizeToText(newSize);

            info = "\n\n" + translateString("File") + ": " + oldShort +  " " + translateString("size") + " = " + oldSizeTxt;
            info += "\n" + translateString("File") + ": " + newShort +  " " + translateString("size") + " = " + newSizeTxt;

            if (messageBox("Aborted", translateString(_MIRROR_ABORTED) + info, QMessageBox::Yes | QMessageBox::No) == QMessageBox::Yes) {
                renameOldToNew();
            }
        }
    }

#endif
}

void HTTraQt::contextMenuEvent(QContextMenuEvent *)
{
#if 0
    QMenu* contextMenu = new QMenu("boo", this);
    Q_CHECK_PTR(contextMenu);
    QString caption = QString("<font color=darkblue><u><b>"
                              "Context Menu</b></u></font>");
    //caption->setAlignment( Qt::AlignCenter );
    // contextMenu->setTitle( "boo" );
    contextMenu->addMenu(QString("&New"));            //,  SLOT(news()), QKeySequence(Qt::CTRL+Qt::Key_N) );
    contextMenu->addMenu(QString("&Open..."));            //, this, SLOT(open()), CTRL+Key_O );
    contextMenu->addMenu(QString("&Save"));            //, this, SLOT(save()), CTRL+Key_S );

    QMenu *submenu = new QMenu(this);
    Q_CHECK_PTR(submenu);
    submenu->addMenu(QString("&Print to printer"));            //, this, SLOT(printer()) );
    submenu->addMenu(QString("Print to &file"));            //, this, SLOT(file()) );
    submenu->addMenu(QString("Print to fa&x"));            //, this, SLOT(fax()) );
    contextMenu->addMenu(QString("&Print"));            //, submenu );
    contextMenu->exec(QCursor::pos());
    delete contextMenu;
#endif
}


void HTTraQt::displayProgressMsg()
{
    QByteArray msg;
    QString result;
    //   QProcess* actualP = NULL;

    process->setReadChannel(QProcess::StandardError);
    msg = process->readAllStandardError();
    //     if ( msg.length() > 1 ) {
    result = msg.data();
    //       actualP = process;
    //     qDebug() << "displayProgressMsg:" << result ;
    //     }

    //   if ( result.length() < 1 )
    //     return;
}

void HTTraQt::displayOutputMsg()
{
    QString line;

    process->setReadChannel(QProcess::StandardOutput);

    while (process->canReadLine()) {
        line = process->readLine().trimmed();
    }

    //     qDebug() << "displayOutputMsg:" << line ;
}


void HTTraQt::processFinished(int exitCode, QProcess::ExitStatus exitStatus)
{
}


void HTTraQt::createActions()
{
    //     newProjAct->setShortcut ( tr ( "Ctrl+N" ) );
    newProjAct->setStatusTip(translateString(_OPENGUI));
    connect(newProjAct, SIGNAL(triggered()), this, SLOT(newProject()));

    //     openProjAct->setShortcut ( tr ( "Ctrl+O" ) );
    //     openProjAct->setStatusTip(translateString("Open project"));
    //     connect(openProjAct, SIGNAL(triggered()), this, SLOT(openProject()));

    //     saveProjAct->setShortcut ( tr ( "Ctrl+S" ) );
    //     saveProjAct->setStatusTip(translateString("Save project"));
    //     connect(saveProjAct, SIGNAL(triggered()), this, SLOT(saveProject()));

    //     delProjAct->setShortcut ( tr ( "Ctrl+D" ) );
    //     delProjAct->setStatusTip(translateString("Delete a project"));
    //     connect(delProjAct, SIGNAL(triggered()), this, SLOT(deleteProject()));


    browseAct->setShortcut ( tr ( "Ctrl+B" ) );
    browseAct->setStatusTip(translateString(_BROWSEEXISTS));
    connect(browseAct, SIGNAL(triggered()), this, SLOT(browseSites()));

    //     loadDefOptAct->setShortcut ( tr ( "Ctrl+L" ) );
    loadDefOptAct->setStatusTip(translateString(_LOADDEF));
    connect(loadDefOptAct, SIGNAL(triggered()), this, SLOT(loadDefaultOptions()));

    saveOptAct->setStatusTip(translateString(_SAVEDEF));
    connect(saveOptAct, SIGNAL(triggered()), this, SLOT(saveDefaultOptions()));

    //     resetOptToDefAct->setShortcut ( tr ( "Ctrl+R" ) );
    resetOptToDefAct->setStatusTip(translateString(_RESETTODEF));
    connect(resetOptToDefAct, SIGNAL(triggered()), this, SLOT(resetToDefault()));

    loadOptAct->setStatusTip(translateString(_LOADOPT));
    connect(loadOptAct, SIGNAL(triggered()), this, SLOT(loadOptionsFromFile()));

    //     recentAct->setShortcut ( tr ( "Ctrl+R" ) );
    //     recentAct->setStatusTip(translateString("Recent file"));
    //     connect(recentAct, SIGNAL(triggered()), this, SLOT(recent()));

    //     exitAct->setShortcut ( tr ( "Ctrl+X" ) );
    exitAct->setStatusTip(translateString(_EXIT));
    connect(exitAct, SIGNAL(triggered()), this, SLOT(quit()));

    saveOptAsAct->setStatusTip(translateString(_SAVEOPTAS));
    connect(saveOptAsAct, SIGNAL(triggered()), this, SLOT(saveOptionsAs()));

    //     selectFontAct->setStatusTip(translateString(_SEL_FONT));
    //     connect(selectFontAct, SIGNAL(triggered()), this, SLOT(selectFontSize()));

    // fontSizePrefAct->setStatusTip(translateString(_SEL_FONT));
    langPrefAct->setStatusTip(translateString(_LANGPREF));
    //     fontPrefAct->setStatusTip(translateString(_SEL_FONT));
    //     connect(langPrefAct, SIGNAL(triggered()), this, SLOT(langPreference()));

    //     modifyOptAct->setShortcut ( tr ( "Ctrl+M" ) );
    modifyOptAct->setStatusTip(translateString(_MODIFYOPT));
    connect(modifyOptAct, SIGNAL(triggered()), this, SLOT(modifyOptions()));

    //     pauseOptAct->setShortcut ( tr ( "Ctrl+P" ) );
    pauseOptAct->setStatusTip(translateString(_PAUSETRANSF));
    connect(pauseOptAct, SIGNAL(triggered()), this, SLOT(pauseTransfer()));

    viewLogAct->setStatusTip(translateString(_VIEW_LOG));
    connect(viewLogAct, SIGNAL(triggered()), this, SLOT(viewLog()));

    viewErrLogAct->setStatusTip(translateString(_VIEWERRLOG));
    connect(viewErrLogAct, SIGNAL(triggered()), this, SLOT(viewErrorLog()));

    //     viewTransferAct->setStatusTip(translateString(_VIEWFILETRANSFERS));
    //     connect(viewTransferAct, SIGNAL(triggered()), this, SLOT(viewTransfers()));

    //     hideAct->setShortcut ( tr ( "Ctrl+H" ) );
    hideAct->setStatusTip(translateString(_HIDE));
    connect(hideAct, SIGNAL(triggered()), this, SLOT(hideWindow()));

    restAct = new QAction(translateString(_OPEN), this);
    //     restAct->setStatusTip(translateString(_RESTORE));
    connect(restAct, SIGNAL(triggered()), this, SLOT(restoreWindow()));

    //     winStatusAct->setStatusTip(translateString("Status Bar"));
    //     connect(winStatusAct, SIGNAL(triggered()), this, SLOT(windowStatusBar()));

    //     checkAct->setShortcut ( tr ( "Ctrl+U" ) );
    checkAct->setStatusTip(translateString(_TOPROJECTPAGE));
    connect(checkAct, SIGNAL(triggered()), this, SLOT(checkUpdates()));

    contentAct->setStatusTip(translateString(_CONTENTS));
    connect(contentAct, SIGNAL(triggered()), this, SLOT(contens()));

    stepByStepAct->setStatusTip(translateString(_STEPBYSTEP));
    connect(stepByStepAct, SIGNAL(triggered()), this, SLOT(stepByStep()));

    //     aboutAct->setShortcut ( tr ( "Ctrl+A" ) );
    aboutAct->setStatusTip(translateString(_ABOUTPROG));
    connect(aboutAct, SIGNAL(triggered()), this, SLOT(about()));

    aboutQtAct->setStatusTip(translateString(_ABOUTQT));
    aboutQtAct->setFont(sysFont);
    connect(aboutQtAct, SIGNAL(triggered()), this, SLOT(aboutQt()));
}


void HTTraQt::createTrayIcon()
{
    if (!QSystemTrayIcon::isSystemTrayAvailable()) {
        hideAct->setEnabled(false);
        return;
    }

    QApplication::setQuitOnLastWindowClosed(false);

    trayIconMenu = new QMenu(this);

    trayIconMenu->addAction(restAct);
    trayIconMenu->addSeparator();

    trayIconMenu->addAction(modifyOptAct);
    trayIconMenu->addAction(viewLogAct);
    trayIconMenu->addAction(viewErrLogAct); // view err log
    trayIconMenu->addSeparator();

    trayIconMenu->addAction(pauseOptAct); // pause
    trayIconMenu->addAction(aboutAct); // about
    trayIconMenu->addAction(exitAct);

    trayIcon = new QSystemTrayIcon(this);

    connect(trayIcon, SIGNAL(activated(QSystemTrayIcon::ActivationReason)),
            this, SLOT(iconActivated(QSystemTrayIcon::ActivationReason)));

    trayIcon->setContextMenu(trayIconMenu);

    trayIcon->setIcon(QIcon(httraqt_icon));
}


void HTTraQt::iconActivated(QSystemTrayIcon::ActivationReason reason)
{
    switch (reason) {
        case QSystemTrayIcon::Trigger:
            //             qDebug() << "trigger";
            break;

        case QSystemTrayIcon::DoubleClick:
            //             qDebug() << "double";
            emit restoreWindow();
            //          iconComboBox->setCurrentIndex((iconComboBox->currentIndex() + 1)
            //                                        % iconComboBox->count());
            break;

        case QSystemTrayIcon::MiddleClick:
            //             qDebug() << "middle";
            //          showMessage();
            break;

        default:
            ;
    }
}


void HTTraQt::hideWindow()
{
    // to system tray
    trayIcon->show();
    hide();
}


void HTTraQt::restoreWindow()
{
    // to normal
    trayIcon->hide();
    showNormal();
}


void HTTraQt::newProject()
{
    QString app;
    app = qApp->applicationDirPath() + "/httraqt";

    //     qDebug() << app;

    QProcess *myProcess = new QProcess();
    myProcess->start(app);
}


void HTTraQt::browseSites()
{
    QString name = currentWorkDir + "/" + currentProject + "/index.html";

    if (QFile::exists(name) == true ) {
        QDesktopServices::openUrl(QUrl::fromLocalFile(qPrintable( name )));
    }
}


void HTTraQt::loadDefaultOptions()
{
    int ans = messageBox(translateString(_QUEST), translateString(_LOADDEFAULT), QMessageBox::Question);

    if (ans == QMessageBox::Yes) {
        readSettings(true);
    }
}


void HTTraQt::saveDefaultOptions()
{
    int ans = messageBox(translateString(_QUEST), translateString(_SAVEDEFAULT), QMessageBox::Question);

    if (ans == QMessageBox::Yes) {
        writeSettings(true); // global options
    }
}


void HTTraQt::resetToDefault()
{
    int ans = messageBox(translateString(_QUEST), translateString(_RESETDEFAULT), QMessageBox::Question);

    if (ans == QMessageBox::Yes) {
        resetDefaultOptions();
    }
}


void HTTraQt::saveOptions(QSettings *s, bool gl)
{
    QVector<options>::iterator iopt;

    //     s->registerFormat ("ini", readIniFile, writeIniFile);

    QString os, st;
    GetProfile("BrowserID", st);
    GetProfile("osID", os);

    if (st.length() > 0) {
        if (st.contains("%s")) {
            st.replace("%s", os);
        }

        SetProfile("UserID", st);
    }

    for (iopt = prOptions.begin(); iopt != prOptions.end(); ++iopt) {
        if (gl == true) { // do not save as global
            if ((*iopt).name == "CurrentUrl" || (*iopt).name == "CurrentURLList") { // only in project settings
                continue;
            }
        }

        if ((*iopt).type == TEXT) {
            QString v = (*iopt).value.toString();
            profile_code(v);
            //             qDebug() << "save" << v;
            s->setValue((*iopt).name, v);
        }

        if ((*iopt).type == FLOAT) {
            if ((*iopt).value.toFloat() >= 0) {
                float fTmp = (*iopt).value.toFloat();
                QString fTxt;
                fTxt = QString().sprintf("%.4f", fTmp);
                //                 qDebug() << "save" << (*iopt).name << fTxt;
                s->setValue((*iopt).name, (qint64)( fTxt.toFloat() * 1024.0 * 1024.0));
            } else {
                s->setValue((*iopt).name, "");
            }
        }

        if ((*iopt).type == INTEGER) {
            if ((*iopt).value.toInt() != -1) {
                s->setValue((*iopt).name, (*iopt).value.toInt());
            } else {
                s->setValue((*iopt).name, "");
            }
        }
    }

    s->sync();
}


void HTTraQt::loadOptions(QSettings *s)
{
    resetDefaultOptions();

    //     s->registerFormat ("ini", readIniFile, writeIniFile);

    // now reading of settings
    QVector<options>::iterator iopt;

    for (iopt = prOptions.begin(); iopt != prOptions.end(); ++iopt) {
        int tInt;
        QString tStr;
        float tFloat;

        if ((*iopt).type == TEXT) {
            tStr = s->value((*iopt).name, (*iopt).value).toString();
            profile_decode(tStr);
            (*iopt).value = tStr;

            continue;
        }

        if ((*iopt).type == FLOAT) {
            bool iOk;
            tFloat = s->value((*iopt).name, (*iopt).value).toFloat(&iOk);

            if (iOk == true) {
                if (tFloat >= 0) {
                    QString tTxt;
                    tTxt = QString().sprintf("%.4f", tFloat / (1024.0 * 1024.0));
                    tFloat = tTxt.toFloat();
                }

                //                 qDebug() << "load" << (*iopt).name << tFloat << (qint64)tFloat;
                (*iopt).value = tFloat; // overwrite the default value if right
            } else {
                (*iopt).value = -1.0;
            }

            continue;
        }

        if ((*iopt).type == INTEGER) {
            bool iOk;
            tInt = s->value((*iopt).name, (*iopt).value).toInt(&iOk);

            if (iOk == true) {
                (*iopt).value = tInt; // overwrite the default value if right
            }

            continue;
        }
    }

    // to implement the checking of options file!
}


void HTTraQt::loadOptionsFromFile()
{
    QSettings* s;

    QString fileName = QFileDialog::getOpenFileName(NULL, translateString(_OPENFILE),
                       currentWorkDir, "HTTrack Settings (linprofile.ini winprofile.ini)");

    if (fileName.length() == 0) {
        return;
    }

    s = new QSettings(fileName, WIniFormat);

    loadOptions(s);
}


bool HTTraQt::checkInstanceRuns()
{
    if(_singular->attach(QSharedMemory::ReadOnly)) {
        //         _singular->detach();
        return false;
    }

    if(_singular->create(1)) {
        return true;
    }

    return false;
}



void HTTraQt::saveOptionsAs()
{
    QSettings* s;

    QString fileName = QFileDialog::getSaveFileName(NULL, translateString(_SAVEFILE),
                       currentWorkDir, "winprofile.ini");

    if (fileName.length() == 0) {
        return;
    }

    s = new QSettings(fileName, WIniFormat);

    saveOptions(s, false);
}


void HTTraQt::pauseTransfer()
{
    if (pauseOptAct->isChecked() == true) {
        hts_setpause(global_opt, 1);
    } else {
        hts_setpause(global_opt, 0);
    }
}


void HTTraQt::onStopAll()
{
    pauseOptAct->setChecked(false);

    // qDebug() << "onStopAll";
    if (soft_term_requested) {
        termine_requested = 1;
    } else {
        soft_term_requested = 1;
        hts_request_stop(mainWidget->global_opt, 0);
    }
}


void HTTraQt::viewTransfers()
{
}



// this is only possible, if programm installed in home directory, like firefox
// in other case only display information about update,
// normal updates only from repository
void HTTraQt::checkUpdates()
{
    QDesktopServices::openUrl(QUrl("http://httraqt.sourceforge.net/", QUrl::TolerantMode));
}


// zeige hilfe
void HTTraQt::contens()
{
    QDesktopServices::openUrl(QUrl::fromLocalFile(qPrintable( helpDir + "/index.html")));
}


void HTTraQt::stepByStep()
{
    QDesktopServices::openUrl(QUrl::fromLocalFile(qPrintable( helpDir + "/step.html")));
}


void HTTraQt::createToolBars()
{
    //     fileToolBar()->addToolBar(translateString("File"));
    //     fileToolBar->addAction(newLetterAct);
    //     fileToolBar->addAction(saveAct);
    //     fileToolBar->addAction(printAct);
    //
    //     editToolBar = addToolBar(translateString("Edit"));
    //     editToolBar->addAction(undoAct);
}


void HTTraQt::createStatusBar()
{
    statusBar()->showMessage(translateString(_READY), 2000);
}


void HTTraQt::getOptions()
{
    currentOptionsFile = "";

    if (currentProject.length() > 0) {
        QString linname, winname;
        linname = currentWorkDir + "/" + currentProject + "/linprofile.ini";
        winname = currentWorkDir + "/" + currentProject + "/hts-cache/winprofile.ini";

        if (QFile::exists(linname) == true) {
            currentOptionsFile = linname;
        }

        if (QFile::exists(winname) == true) {
            currentOptionsFile = winname;
        }
    }

    if (currentOptionsFile.length() > 0) {
        readSettings(false); // project
    } else {
        readSettings(true); // global
    }

    SetProfile("MaxTime", -1);
    OptionsDialog *oDia = new OptionsDialog(this);

    delete oDia;
}


void HTTraQt::resetDefaultOptions()
{
    prOptions.clear();

    prOptions << (options) {
        "Near", INTEGER, 1
    };
    prOptions << (options) {
        "ParseAll", INTEGER, 1
    };
    prOptions << (options) {
        "Test", INTEGER, 0
    };
    prOptions << (options) {
        "HTMLFirst", INTEGER, 0
    };
    prOptions << (options) {
        "Dos", INTEGER, 0
    };
    prOptions << (options) {
        "ISO9660", INTEGER, 0
    };
    prOptions << (options) {
        "NoErrorPages", INTEGER, 0
    };
    prOptions << (options) {
        "NoExternalPages", INTEGER, 1
    };
    prOptions << (options) {
        "NoPwdInPages", INTEGER, 0
    };
    prOptions << (options) {
        "NoQueryStrings", INTEGER, 1
    };
    prOptions << (options) {
        "NoPurgeOldFiles", INTEGER, 0
    };
    prOptions << (options) {
        "Build", INTEGER, 0
    };
    prOptions << (options) {
        "WildCardFilters", TEXT, "+*.png +*.gif +*.jpg +*.css +*.js -ad.doubleclick.net/*"
    };
    prOptions << (options) {
        "Depth", INTEGER, -1
    };
    prOptions << (options) {
        "ExtDepth", INTEGER, 0
    };
    prOptions << (options) {
        "CurrentAction", INTEGER, 0
    };
    prOptions << (options) {
        "MaxHtml", FLOAT, -1
    };
    prOptions << (options) {
        "MaxOther", FLOAT, -1
    };
    prOptions << (options) {
        "MaxAll", FLOAT, -1
    };
    prOptions << (options) {
        "MaxWait", INTEGER, 0
    };
    prOptions << (options) {
        "MaxTime", INTEGER, -1
    };
    prOptions << (options) {
        "MaxRate", FLOAT, -1
    };
    prOptions << (options) {
        "MaxConn", INTEGER, -1
    };
    prOptions << (options) {
        "MaxLinks", INTEGER, -1
    };
    prOptions << (options) {
        "Proxy", TEXT, ""
    };
    prOptions << (options) {
        "CurrentUrl", TEXT, ""
    };
    prOptions << (options) {
        "CurrentURLList", TEXT, ""
    };
    prOptions << (options) {
        "Category", TEXT, ""
    };
    prOptions << (options) {
        "BuildString", TEXT, "%h%p/%n%q.%t"
    };
    prOptions << (options) {
        "UseHTTPProxyForFTP", INTEGER, 0
    };
    prOptions << (options) {
        "PrimaryScan", INTEGER, 4
    };
    prOptions << (options) {
        "Debug", INTEGER, 0
    };
    prOptions << (options) {
        "UserID", TEXT, "Mozilla/5.0 (X11; U; Linux i686; I; rv:17.0.1) Gecko/20100101 Firefox/17.0.1"
    };
    prOptions << (options) {
        "BrowserID", TEXT, "Mozilla/5.0 (%s; rv:17.0.1) Gecko/20100101 Firefox/17.0.1"
    };
    prOptions << (options) {
        "osID", TEXT, "X11; U; Linux i686; I"
    };
    prOptions << (options) {
        "Travel", INTEGER, 1
    };
    prOptions << (options) {
        "GlobalTravel", INTEGER, 0
    };
    prOptions << (options) {
        "RewriteLinks", INTEGER, 0
    };
    prOptions << (options) {
        "Footer", TEXT, "<!-- Mirrored from %s%s by HTTraQt Website Copier/1.x [Karbofos 2010-2014] %s -->"
    };
    prOptions << (options) {
        "Cache", INTEGER, 1
    };
    prOptions << (options) {
        "TimeOut", INTEGER, -1
    };
    prOptions << (options) {
        "Sockets", INTEGER, 8
    };
    prOptions << (options) {
        "Retry", INTEGER, -1
    };
    prOptions << (options) {
        "KeepAlive", INTEGER, 1
    };
    prOptions << (options) {
        "RemoveTimeout", INTEGER, 0
    };
    prOptions << (options) {
        "RemoveRateout", INTEGER, 0
    };
    prOptions << (options) {
        "RateOut", INTEGER, 0
    };
    prOptions << (options) {
        "StoreAllInCache", INTEGER, 0
    };
    prOptions << (options) {
        "MIMEDefsMime1", TEXT, ""
    };
    prOptions << (options) {
        "MIMEDefsMime2", TEXT, ""
    };
    prOptions << (options) {
        "MIMEDefsMime3", TEXT, ""
    };
    prOptions << (options) {
        "MIMEDefsMime4", TEXT, ""
    };
    prOptions << (options) {
        "MIMEDefsMime5", TEXT, ""
    };
    prOptions << (options) {
        "MIMEDefsMime6", TEXT, ""
    };
    prOptions << (options) {
        "MIMEDefsMime7", TEXT, ""
    };
    prOptions << (options) {
        "MIMEDefsMime8", TEXT, ""
    };
    prOptions << (options) {
        "MIMEDefsExt1", TEXT, ""
    };
    prOptions << (options) {
        "MIMEDefsExt2", TEXT, ""
    };
    prOptions << (options) {
        "MIMEDefsExt3", TEXT, ""
    };
    prOptions << (options) {
        "MIMEDefsExt4", TEXT, ""
    };
    prOptions << (options) {
        "MIMEDefsExt5", TEXT, ""
    };
    prOptions << (options) {
        "MIMEDefsExt6", TEXT, ""
    };
    prOptions << (options) {
        "MIMEDefsExt7", TEXT, ""
    };
    prOptions << (options) {
        "MIMEDefsExt8", TEXT, ""
    };
    prOptions << (options) {
        "NoRecatch", INTEGER, 0
    };
    prOptions << (options) {
        "Log", INTEGER, 0
    };
    prOptions << (options) {
        "Index", INTEGER, 1
    };
    prOptions << (options) {
        "BuildTopIndex", INTEGER, 1
    };
    prOptions << (options) {
        "WordIndex", INTEGER, 1
    };
    prOptions << (options) {
        "LogType", INTEGER, 0
    };
    prOptions << (options) {
        "URLHack", INTEGER, 1
    };
    prOptions << (options) {
        "Cookies", INTEGER, 1
    };
    prOptions << (options) {
        "UpdateHack", INTEGER, 1
    };
    prOptions << (options) {
        "ParseJava", INTEGER, 1
    };
    prOptions << (options) {
        "TolerantRequests", INTEGER, 0
    };
    prOptions << (options) {
        "HTTP10", INTEGER, 0
    };
    prOptions << (options) {
        "CheckType", INTEGER, 0
    };
    prOptions << (options) {
        "FollowRobotsTxt", INTEGER, 2
    };
    prOptions << (options) {
        "ProxyPort", INTEGER, 21
    };
}


void HTTraQt::writeGUISettings(void)
{
    QSettings* s;
    s = new QSettings(QSettings::UserScope, "KarboSoft", "HTTraQt");
    s->setValue("pos", pos());
    s->setValue("size", size());
    s->setValue("WorkDir", currentWorkDir);
    s->setValue("LANGUAGE", currentLang);
    s->setValue("LASTPROJ", currentProject);
    //     s->setValue("FontSize", fontSize);
    //     s->setValue("GUIFont", sysFont);
    // qDebug() << "writeGUISettings";
    int i = 0;

    for (QStringList::Iterator iDir = lastDirs.begin(); iDir != lastDirs.end(); iDir++, i++) {
        if (i > 8) { // max last dirs
            break;
        }

        s->setValue("LASDIR" + QString::number(i), (*iDir));
    }

    s->sync();
}


void HTTraQt::writeSettings(bool global)
{
    QSettings* s;
    QString fname;

    if (global == true) {
        s = new QSettings(QSettings::UserScope, "KarboSoft", "HTTraQt");
        s->setValue("WorkDir", currentWorkDir);
        s->setValue("LANGUAGE", currentLang);
        //                 qDebug() << "writesettings global";
    } else {
        if (currentProject.length() == 0) {
            return;
        }

        fname = currentWorkDir + "/" + currentProject/* + "/hts-cache"*/;
        QDir d(fname);

        if (d.exists() == false) {
            d.mkpath((const QString) fname);
        }

        if (d.mkpath(fname + "/hts-cache") == false) {
            qDebug() << "can not create";
        }

        fname += "/hts-cache/winprofile.ini";
        //         qDebug() << "writesettings local" << fname;
        s = new QSettings(fname, WIniFormat);
    }

    saveOptions(s, global);
}

// to get locale and convert to internal string
QString HTTraQt::getLocaleString()
{
    QString res;
    QLocale lSys = QLocale::system();

    switch (lSys.language()) {
        case QLocale::Bulgarian:
            res = "Bulgarian";
            break;

        case QLocale::Spanish:
            res = "Castellano";
            break;

        case QLocale::Czech:
            res = "Česky";
            break;

        case QLocale::Chinese:
            res = "Chinese-Simplified";
            break;

        case QLocale::Danish:
            res = "Dansk";
            break;

        case QLocale::German:
            res = "Deutsch";
            break;

        case QLocale::Estonian:
            res = "Eesti";
            break;

        case QLocale::C:
            res = "English";
            break;

        case QLocale::Finnish:
            res = "Finnish";
            break;

        case QLocale::French:
            res = "Français";
            break;

        case QLocale::Greek:
            res = "Greek";
            break;

        case QLocale::Italian:
            res = "Italiano";
            break;

        case QLocale::Japanese:
            res = "Japanese";
            break;

        case QLocale::Hungarian:
            res = "Magyar";
            break;

        case QLocale::Netherlands:
            res = "Nederlands";
            break;

        case QLocale::NorwegianNynorsk:
            res = "Norsk";
            break;

        case QLocale::Polish:
            res = "Polski";
            break;

        case QLocale::Brazil:
            res = "Português-Brasil";
            break;

        case QLocale::Portuguese:
            res = "Português";
            break;

        case QLocale::Romanian:
            res = "Romanian";
            break;

        case QLocale::Russian:
            res = "Russian";
            break;

        case QLocale::Slovak:
            res = "Slovak";
            break;

        case QLocale::Slovenian:
            res = "Slovenian";
            break;

        case QLocale::Swedish:
            res = "Svenska";
            break;

        case QLocale::Turkish:
            res = "Turkish";
            break;

        case QLocale::Ukrainian:
            res = "Ukrainian";
            break;

        default:
            res = "English";
            break;
    }

    return res;
}


void HTTraQt::readGUISettings()
{
    QSettings* s;
    s = new QSettings(QSettings::UserScope, "KarboSoft", "HTTraQt");
    // qDebug() << "readGUISettings";
    QPoint pos = s->value("pos", QPoint(200, 200)).toPoint();
    QSize size = s->value("size", QSize(840, 640)).toSize();
    resize(size);
    move(pos);

    QString l;
    l = getLocaleString();

    currentLang = s->value("LANGUAGE", l).toString();
    currentProject = s->value("LASTPROJ").toString();
    sysFont = sysFont.toString();//QFont(s->value("GUIFont", sysFont.toString()).toString());
    //     qDebug() << sysFont;
    int sz = sysFont.pointSize();

    if ( sz == -1) {
        sz = sysFont.pixelSize();
    }

    fontSize = sz;

    //     setFontForWidgets();

    for (int i = 0; i < 8; i++) {
        QString d = s->value("LASDIR" + QString::number(i)).toString();
        QDir dr;

        if (d.length() == 0) {
            break;
        }

        if (dr.exists(d) == true) {
            lastDirs << d;
        }
    }

    QDir dir;
    QStringList dirsLang;
    dirsLang << "/usr/share/httraqt/" << "/usr/local/share/httraqt/" << currentAppDir;

    foreach(QString entry, dirsLang) {
        helpDir = entry + "/help/";

        dir = QDir(helpDir);

        if (dir.exists() == true) {
            break;
        } else {
            helpDir = "";
        }
    }

    foreach(QString entry, dirsLang) {
        langDir = entry + "/lang/";

        dir = QDir(langDir);

        if (dir.exists() == true) {
            break;
        } else {
            langDir = "";
        }
    }
}


void HTTraQt::readSettings(bool global)
{
    QSettings* s;
    QString fname;

    int num;
    QString text;

    if (global == true) {
        s = new QSettings(QSettings::UserScope, "KarboSoft", "HTTraQt");
        currentWorkDir = s->value("WorkDir", QDir::homePath() + "/My Web Sites").toString();
    } else {
        fname = currentWorkDir + "/" + currentProject + "/linprofile.ini";

        if (QFile::exists(fname) == false) { // exists
            fname = currentWorkDir + "/" + currentProject + "/hts-cache/winprofile.ini";

            if (QFile::exists(fname) == false) { // exists
                fname = "";
            }
        }
    }

    if (fname.length() == 0) { // file not found, global settings
        s = new QSettings(QSettings::UserScope, "KarboSoft", "HTTraQt");
    } else {
        s = new QSettings(fname, WIniFormat);
    }

    currentOptionsFile = fname;

    loadOptions(s);
}


void HTTraQt::onCancelAll()
{
}


void HTTraQt::onQuit()
{
    quit();
}


void HTTraQt::activatePage(int pageNum)
{
    switch (pageNum) {
        case 0: { // start tab
            currentTab = 0;

            // actions
            browseAct->setEnabled(false);

            viewLogAct->setEnabled(false);
            viewErrLogAct->setEnabled(false);
            pauseOptAct->setEnabled(false);
            modifyOptAct->setEnabled(false);

            //             menuMirror->setEnabled(false);
            //             menuLog->setEnabled(false);
            break;
        }

        case 1: { // new project tab or select project
            //             qDebug() << "seite 1: " << currentWorkDir;
            (*(NewProjTab*)widgets[1]).editBasePath->setText(currentWorkDir);
            (*(NewProjTab*)widgets[1]).selectProj(currentProject);

            currentTab = 1;

            //actions
            browseAct->setEnabled(false);

            viewLogAct->setEnabled(false);
            viewErrLogAct->setEnabled(false);
            pauseOptAct->setEnabled(false);
            modifyOptAct->setEnabled(false);

            //             menuMirror->setEnabled(false);
            //             menuLog->setEnabled(false);
            break;
        }

        case 2: { // options tab
            if (currentTab == 1) {   // prüfe, ob die parameter i.o. sind: aus NewProjTab
                if ((*(NewProjTab*)widgets[1]).testIt() == false) {   // projektname muss vorhanden sein
                    return;
                }
            }

            bool nProj = rebuildWorkDirView();

            QString url;
            GetProfile("CurrentUrl", url);
            (*(OptionsTab*)widgets[2]).textEdit->setText(url);

            QString urlFile;
            GetProfile("CurrentURLList", urlFile);
            (*(OptionsTab*)widgets[2]).label1286->setText(urlFile);

            if (nProj == false) {
                (*(OptionsTab*)widgets[2]).updateLastAction();
                getOptions();
            } else {
                m_todo = 0;
                (*(OptionsTab*)widgets[2]).comboAction->setCurrentIndex(m_todo);
                readSettings(true); // global
                m_todo = 0;
            }

            // wenn projekt existierte, einlesen von diversen settings und url liste.

            if ((m_todo == LAST_ACTION) || (m_todo == LAST_ACTION - 1)) {
                if (checkContinue(true) == false) {
                    return;
                }
            }

            currentTab = 2;

            //actions
            browseAct->setEnabled(false);

            viewLogAct->setEnabled(false);
            viewErrLogAct->setEnabled(false);
            pauseOptAct->setEnabled(false);
            modifyOptAct->setEnabled(false);

            //             menuMirror->setEnabled(false);
            //             menuLog->setEnabled(false);
            break;
        }

        case 3: { // confirm tab with delay time
            if (currentTab == 2) {   // prüfe, ob die parameter i.o. sind: aus OptionsTab
                if ((*(OptionsTab*)widgets[2]).testIt() == false) {    // die url liste muss vorhanden sein
                    messageBox(translateString(_NOURL), translateString(_URLNOTCOMP), QMessageBox::Critical);

                    return;
                }
            }

            (*(OptionsTab*)widgets[2]).updateLastAction();

            QString url = (*(OptionsTab*)widgets[2]).textEdit->toPlainText();
            SetProfile("CurrentUrl", url);

            QString urlFile = (*(OptionsTab*)widgets[2]).label1286->text();
            SetProfile("CurrentURLList", urlFile);

            (*(ConfirmTab*)widgets[3]).labelHH->clear(); // HH
            (*(ConfirmTab*)widgets[3]).labelMM->clear(); // MM
            (*(ConfirmTab*)widgets[3]).labelSS->clear(); // SS

            currentTab = 3;

            getMainOptionsFromGUI();
            writeSettings(false); // write project settings

            // check current action, if
            QString prDir;
            prDir =  currentWorkDir + "/" + currentProject + "/";

            //actions
            browseAct->setEnabled(false);

            viewLogAct->setEnabled(false);
            viewErrLogAct->setEnabled(false);
            pauseOptAct->setEnabled(false);
            modifyOptAct->setEnabled(false);

            menuPreferences->setEnabled(true);
            break;
        }

        case 4: { // progress tab

            currentTab = 4;

            //actions
            browseAct->setEnabled(true);

            viewLogAct->setEnabled(true);
            viewErrLogAct->setEnabled(true);
            pauseOptAct->setEnabled(true);
            modifyOptAct->setEnabled(true);

            menuPreferences->setEnabled(false);

            launch();

            break;
        }

        case 5: { // log tab
            currentTab = 5;

            // actions
            browseAct->setEnabled(true);

            viewLogAct->setEnabled(false);
            viewErrLogAct->setEnabled(false);
            pauseOptAct->setEnabled(false);
            modifyOptAct->setEnabled(false);

            menuPreferences->setEnabled(true);
            break;
        }

        default:
            break;
    }

    for (int i = 0; i <= 5; i++) {
        if (i == pageNum) {
            if (widgets[i]->isHidden() == true) {
                widgets[i]->show();
            }
        } else {
            if (widgets[i]->isHidden() == false) {
                widgets[i]->hide();
            }
        }
    }

    buttonsWidget->onButtons(pageNum);
}


HTTraQt::~HTTraQt()
{
    if(_singular->isAttached()) {
        _singular->detach();
    }

    quit();
}


int HTTraQt::removeFolder(QDir & dir)
{
    int res = 0;
    //list of dirs
    QStringList lstDirs  = dir.entryList(QDir::Dirs | QDir::AllDirs | QDir::NoDotAndDotDot);
    //file list
    QStringList lstFiles = dir.entryList(QDir::Files);
    //rm files
    foreach(QString entry, lstFiles) {
        QString entryAbsPath = dir.absolutePath() + "/" + entry;
        QFile::remove(entryAbsPath);
    }

    //for directories recursive
    foreach(QString entry, lstDirs) {
        QString entryAbsPath = dir.absolutePath() + "/" + entry;
        QDir dr;
        dr = QDir(entryAbsPath);
        removeFolder(dr);
    }

    //deleting of directory
    if (!QDir().rmdir(dir.absolutePath())) {
        res = 1;
    }

    return res;
}


void HTTraQt::viewLog()
{
    QString name = currentWorkDir + "/" + currentProject + "/"  "hts-log.txt";

    // fichier log existe ou on est télécommandé par un !
    if (QFile::exists(name) == true ) {
        QDesktopServices::openUrl(QUrl::fromLocalFile(qPrintable( name )));
        return;
    }

    //     name = currentWorkDir + "/" + currentProject + "/"  "hts-err.txt";
    //
    //     if (QFile::exists(name) == true ) {
    //         QDesktopServices::openUrl(QUrl::fromLocalFile(qPrintable( name )));
    //         return;
    //     }

    QString l = translateString(_NOLOGFILES);
    l.replace("%s", currentProject);

    messageBox(translateString(_ERROR), l, QMessageBox::Warning);
}


void HTTraQt::viewErrorLog()
{
    QString name = currentWorkDir + "/" + currentProject + "/"  "hts-err.txt";

    if (QFile::exists(name) == true ) {
        QDesktopServices::openUrl(QUrl::fromLocalFile(qPrintable( name )));
        return;
    }

    QString l = translateString(_NOLOGFILES);
    l.replace("%s", currentProject);
    messageBox(translateString(_ERROR), l, QMessageBox::Warning);
}

#if 0
void HTTraQt::createFontSizeMenu()
{
    fontSizeMenu = menuPreferences->addMenu(translateString(_SEL_FONT));
    fsizeGroup = new QActionGroup(this);
    QStringList szList;
    szList << "9" << "10" << "12" << "14" << "16" << "18" << "20";

    foreach(QString entry, szList) {
        QAction *tmpAction = new QAction(entry, fontSizePrefAct);
        tmpAction->setCheckable(true);

        fontSizeMenu->addAction(tmpAction);
        fsizeGroup->addAction(tmpAction);

        if (fontSize == entry.toInt()) {
            tmpAction->setChecked(true);
        }

        actFSizeSelect.push_back(tmpAction);
    }

    connect(fsizeGroup, SIGNAL(triggered(QAction*)), this, SLOT(selectFontSize(QAction*)));
}
#endif



bool HTTraQt::readLangDir()
{
    //     QString sSheet2 = QString("QMenuBar { %1; }").arg( programStyleSheet );
    bool found = false;
    QString lngDirName;
    QStringList dirsLang;
    QDir dir;
    dirsLang << "/usr/share/httraqt/" << "/usr/local/share/httraqt/" << currentAppDir;

    dictFormat = 0;

    foreach(QString entry, dirsLang) {
        lngDirName = entry + "/lang/";

        dir = QDir(lngDirName);

        if (dir.exists() == true) {
            found = true;
            break;
        }

        lngDirName = entry + "/language/";
        dir = QDir(lngDirName);

        if (dir.exists() == true) {
            found = true;
            break;
        }
    }

    if(found == false) {
        return false;
    }

    langFiles.clear();

    QStringList fList = dir.entryList(QStringList("*.utf"));

    found = false;

    langMenu = menuPreferences->addMenu(translateString(_LANGUAGE));
    //     langMenu->setStyleSheet(sSheet2);


    langGroup = new QActionGroup(this);
    //     langGroup->setFont(sysFont);

    for (QStringList::Iterator iL = fList.begin(); iL != fList.end(); iL++) {
        QFile fLang(lngDirName + *iL);

        if (fLang.exists() == false) {
            continue;
        }

        QString iconName;

        if (fLang.open(QIODevice::ReadOnly)) {      //wird eingelesen
            QTextStream stream(&fLang);
            stream.setCodec("UTF-8");
            QString line, nm;

            int lines = 0;

            while (!stream.atEnd()) {
                line = stream.readLine(); // line of text excluding '\n'
                lines++;

                if (line == "LANGUAGE_NAME") {
                    line = stream.readLine();
                    lines++;
                    nm = line;
                    continue;
                }

                if (line == "LANGUAGE_ISO") {
                    line = stream.readLine();
                    selectedLang = line;
                    lines++;

                    iconName = lngDirName + "flags/" + line + ".png";

                    found = true;

                    langFiles += (*iL) + ":" + nm;
                    QAction *tmpAction = new QAction(nm, langPrefAct);
                    tmpAction->setCheckable(true);
                    //                     tmpAction->setFont(sysFont);

                    if (QFile::exists(iconName) == true) {
                        QFile flIcon(iconName);

                        if (flIcon.size() < 1024 ) { // checking of filesize
                            tmpAction->setIcon(QIcon(iconName));
                        }
                    }

                    langGroup->addAction(tmpAction);
                    langMenu->addAction(tmpAction);

                    if (currentLang == nm) {
                        tmpAction->setChecked(true);
                    }

                    actLangSelect.push_back(tmpAction);
                    break;
                }

                if (lines > 8) {
                    break;
                }
            }

            fLang.close();

        } else {
            continue;
        }
    }

    connect(langGroup, SIGNAL(triggered(QAction*)), this, SLOT(setLang(QAction*)));

    return (found);
}


void HTTraQt::selectFontSize(QAction* mnu)
{
    QString lngStr;
    //     mnu = fsizeGroup->checkedAction();

    lngStr = mnu->text();
    fontSize = lngStr.toInt();

    mnu->setChecked(true);

    int sz = sysFont.pointSize();

    // for lang menu and for fontsize menu
    //     QString menuSheet, menuSheet2;

    if ( sz == -1) {
        //         sz = sysFont.pixelSize();
        programStyleSheet = QString().sprintf("font-size: %dpx", fontSize);
        //         menuSheet = QString().sprintf("QMenuBar { font-size: %dpx; }", fontSize);
        //         menuSheet = QString().sprintf("QMenuBar { font-family %s; font-size: %dpx}", sysFont.family().data(), fontSize);
    } else {
        programStyleSheet = QString().sprintf("font-size: %dpt", fontSize);
        //         menuSheet = QString().sprintf("QMenuBar { font-size: %dpt; }", fontSize);
        //           menuSheet = QString().sprintf("QMenuBar { font-family %s; font-size: %dpt}", sysFont.family().data(), fontSize);
        //         menuSheet2 = QString().sprintf("QMenuBar { font-weight: normal; font-style: normal; font-size: %dpx }", fontSize-1);
    }

    QString sSheet2 = QString("QMenu { %1; }").arg( programStyleSheet );
    //     qDebug() << sysFont.family();
    setStyleSheet(programStyleSheet);
    //     qDebug() << langMenu->styleSheet();
    langMenu->setStyleSheet(sSheet2);
    //     langGroup->setStyleSheet(menuSheet);
    //     fontSizeMenu->setStyleSheet(sSheet2);

    setFontForWidgets();
}


void HTTraQt::setLang(QAction* mnu)
{
    QString lngStr;
    mnu = langGroup->checkedAction();

    lngStr = mnu->text();
    currentLang = lngStr;

    if (getLangTable() == false) {
        qDebug() << "setLang" << false;
    }

    QVector<QAction*>::iterator itL;

    disconnect(langGroup, SIGNAL(triggered(QAction*)), this, SLOT(setLang(QAction*)));

    mnu->setChecked(true);

    connect(langGroup, SIGNAL(triggered(QAction*)), this, SLOT(setLang(QAction*)));

    setLangGUI();
}


void HTTraQt::convertTranslateFile()
{
    QVector<hInfo> hList;

    QFile hdrFile(currentAppDir + "/sources/main/includes/lang.h.orig");
    //         qDebug() << hdrFile.fileName();

    if (hdrFile.open(QIODevice::ReadOnly)) {
        QTextStream stream(&hdrFile);
        stream.setCodec("UTF-8");
        int num = 1;

        while (!stream.atEnd()) {
            QString ll = stream.readLine();

            if (ll.indexOf("#define") != 0 || ll.indexOf("\"") <= 0) {
                continue;
            }

            int pos1 = ll.indexOf(QRegExp("[_A-Z]"));
            int pos2 = ll.indexOf(" ", pos1);
            QString name = ll.mid(pos1, pos2 - pos1);

            pos1 = ll.indexOf("\"", pos2);
            QString txt = ll.mid(pos1);
            bool found = false;

            for (QVector<hInfo>::iterator il = hList.begin(); il != hList.end(); ++il) {
                if ((*il).tInfo == txt) {
                    found = true;
                    break;
                }
            }

            if (found == false) {
                hInfo tmp;
                tmp.name = name;
                tmp.num = num;
                txt.replace("\t", "\\t");
                txt.replace("\n", "\\n");
                tmp.tInfo = txt.mid(1, txt.length() - 2);

                hList << tmp;
                num++;
            }
        }

        hdrFile.close();

        QFile hdrOutFile(currentAppDir + "/sources/main/includes/lang.h.new");
        //     qDebug() << langTrFile.fileName();

        if (hdrOutFile.open(QIODevice::WriteOnly)) {
            QTextStream stream(&hdrOutFile);
            stream.setCodec("UTF-8");
            stream << "#ifndef LANGUAGE_HEADER\n";
            stream << "#define LANGUAGE_HEADER\n\n\n";

            stream << "struct engText { \n";

            for (QVector<hInfo>::iterator il = hList.begin(); il != hList.end(); ++il) {
                stream << "\"" << (*il).tInfo << "\", // " << QString::number((*il).num) << "\n";
            }

            stream << "};\n\n";

            for (QVector<hInfo>::iterator il = hList.begin(); il != hList.end(); ++il) {
                stream << "#define " << (*il).name << " "  << QString::number((*il).num) << "\n";
            }

            stream << "\n\n#endif\n";
            hdrOutFile.close();
        } else {
            qDebug() << "cannot open" << hdrOutFile.fileName();
            return;
        }
    } else {
        qDebug() << "cannot open" << hdrFile.fileName();
        return;
    }


    bool found = false;
    QString lngDirName = currentAppDir + "/lang/";
    QDir dir(lngDirName);

    QStringList fList = dir.entryList(QStringList("*.utf"));

    for (QStringList::Iterator iL = fList.begin(); iL != fList.end(); iL++) {
        QStringList dictionary;
        dictionary.clear();

        QFile fLang(lngDirName + *iL);

        if (fLang.open(QIODevice::ReadOnly)) {      //wird eingelesen
            QTextStream stream(&fLang);
            stream.setCodec("UTF-8");
            QString line;

            while (!stream.atEnd()) {
                line = stream.readLine(); // line of text excluding '\n'
                dictionary << line;
            }

            fLang.close();
        } else {
            qDebug() << "not found" << *iL;
        }

        qDebug() << (*iL) << "lines: " << dictionary.size();
        // #ifdef TRANSLATE_FILE
        QFile langTrFile(lngDirName + (*iL) + ".new");
        //     qDebug() << langTrFile.fileName();
        //         int n = 1;
        bool trBeg = false;

        if (langTrFile.open(QIODevice::WriteOnly)) {
            QTextStream stream(&langTrFile);
            stream.setCodec("UTF-8");

            for (QStringList::iterator it = dictionary.begin(); it != dictionary.end(); ++it) {
                if (trBeg == false) {
                    if ((*it).length() == 0) {
                        stream << (*it) << "\r\n";
                    }

                    if ((*it).indexOf("OK") == 0) { // beg of translations detected
                        trBeg = true;
                    }
                }

                if (trBeg == false) {
                    stream << (*it) << "\r\n";
                } else {
                    QString origText;
                    origText = (*it);
                    it++; // jede zweite

                    if (it == dictionary.end()) {
                        break;
                    }

                    QString ll = *it;

                    //                 ll.replace("\\r\\n", "\n");
                    if (ll.length() > 0) {
                        ll.replace("winhttrack", "httraqt");
                        ll.replace(" (B/s)", "");
                        ll.replace("C:", "%HOME%");
                        ll.replace("c:", "%HOME%");
                        //                         ll.replace("\\n", "\n");
                        //                         ll.replace("\\t", "\t");
                        ll.replace("\\r\\n", "\\n");
                        ll.replace( "\\\\", "/");
                        //                         ll.replace("\\t", "\t");
                        ll.replace( "&gt;", ">");
                        ll.replace( "&lt;", "<");
                    } else {
                        continue;
                    }

                    if (origText.length() > 0) {
                        origText.replace("winhttrack", "httraqt");
                        origText.replace(" (B/s)", "");
                        //                         origText.replace("\\n", "\n");
                        origText.replace("C:", "%HOME%");
                        origText.replace("c:", "%HOME%");
                        //                          origText.replace("\\t", "\t");
                        origText.replace("\\r\\n", "\\n");
                        origText.replace( "\\\\", "/");
                        origText.replace( "&amp;", "&");
                        origText.replace( "&gt;", ">");
                        origText.replace( "&lt;", "<");
                    } else {
                        continue;
                    }

                    if (ll.length() > 0) {
                        ll.replace( "\\\\", "/");
                        ll.replace( "&amp;", "&");
                    } else {
                        continue;
                    }

                    //                     if (origText.length() > 0) {
                    //                         origText.replace( "/", "\\\\");
                    //                     }
                    //                     else{
                    //                         continue;
                    //                     }

                    if (ll.length() > 0) {
                        bool found = false;

                        for (QVector<hInfo>::iterator itr = hList.begin(); itr != hList.end(); ++itr) {
                            //                             origText = "\"" + origText + "\"";
                            if ( origText == (*itr).tInfo) {
                                stream << QString().sprintf("t%03d=", (*itr).num) << ll << "\r\n";
                                found = true;
                                break;
                            }

                            //                         stream << (*itr).tInfo << ", // " << QString::number((*itr).num) << "\n";
                        }

                        if (found == false) {
                            qDebug() << "file" << *iL << "line not found" << origText;
                        }
                    }

                    //                     n++;
                }
            }

            langTrFile.close();
        } else {
            qDebug() << "cannot open" <<  langTrFile.fileName();
        }
    }
}


bool HTTraQt::getLangTable()
{
    QString ll;
    QString lang = currentLang;
    QString fileLang = "";

    for (QStringList::Iterator iLang = langFiles.begin(); iLang != langFiles.end(); iLang++) {
        if ((*iLang).contains(":" + lang) > 0) {
            fileLang = *iLang;
            fileLang.remove(":" + lang);
            break;
        }
    }

    if (fileLang == "") {
        return (false);
    }

    QFile langFile(langDir + fileLang);

    if (langFile.exists() == false) {
        messageBox(translateString(_ERROR), "Language file not exists!\n\n"
                   + langDir + "\n\n" + fileLang, QMessageBox::Warning);

        return (false);
    }

    translateTable.clear();

    if (langFile.open(QIODevice::ReadOnly)) {
        QTextStream stream(&langFile);
        stream.setCodec("UTF-8");

        ll = stream.readLine();

        if (ll.length() > 0) {
            ll.remove("\n");
        }

        if (ll == "LANGUAGE_NAME") {
            translateTable << ll;
        } else {
            messageBox(translateString(_FILEERR), "File is not a language file!", QMessageBox::Warning);

            langFile.close();
            return (false);
        }

        while (!stream.atEnd()) {
            ll = stream.readLine();
            //             qDebug() << ll;

            if (ll.length() > 0) {
                ll.remove("\n");
            }

            if (ll.length() == 0) { // english original text not found
                ll = stream.readLine();

                if (stream.atEnd() == true) {
                    break;
                } else {
                    continue;
                }
            }

            if (ll.length() > 0) {
                ll.replace( "\\r\\n", "\n");
                ll.replace( "\\n", "\n");
                ll.replace( "\\\\", "/");
                ll.replace( "&amp;", "&");
                ll.replace( "&gt;", ">");
                ll.replace( "&lt;", "<");
            }

            if (ll.indexOf(QRegExp("t[0-9]{3}=")) == 0) {
                qDebug() << "neues format";
                dictFormat = 2; // new format
            }

            if (ll.length() > 0) {
                if (ll.indexOf("C:/") >= 0) {
                    ll.replace("C:/", QString(QDir::homePath() + "/"));
                }

                if (ll.indexOf("c:/") >= 0) {
                    ll.replace("c:/", QString(QDir::homePath() + "/"));
                }

                if (ll.indexOf("%HOME%") >= 0) {
                    ll.replace("%HOME%", QString(QDir::homePath() + "/"));
                }
            }

            translateTable << ll;
        }

        langFile.close();

        return (true);
    }


    return (false);
}


void HTTraQt::setLangGUI()
{
    (*(StartTab*)widgets[0]).translate();
    (*(NewProjTab*)widgets[1]).translate();
    (*(OptionsTab*)widgets[2]).translate();
    (*(ConfirmTab*)widgets[3]).translate();
    (*(ProgressTab*)widgets[4]).translate();
    (*(FinalTab*)widgets[5]).translate();

    // actions translate
    translateActions();

    //     dModel.setHeaderData(0, Qt::Horizontal, "Dir");
    // // setHeaderData (0,Qt::Horizontal, tr("Display Date"));
    //     dModel.setHeaderData (1,Qt::Horizontal, tr("Media File"));
    //     dModel.setHeaderData (2,Qt::Horizontal, tr("Display time"));
    //
    //     dirView->setModel(&dModel);

    rebuildWorkDirView();

    //      setFontForWidgets();
}


void HTTraQt::translateActions(QAction* act, int id)
{
    QString tmp;
    QString str = translateString(id);
    int posTab;

    if (str.length() == 0) {
        return;
    }

    posTab = str.indexOf("\\t");
    tmp = translateString(id);

    if (posTab > 0) {
        QString shrtcut = str.mid(posTab + 2);
        QStringList tmpC = tmp.split("\\t");
        act->setText(tmpC.at(0));
        act->setFont(sysFont);
        act->setShortcut(shrtcut);
    } else {
        act->setText(tmp);
        act->setFont(sysFont);
    }
}


void HTTraQt::translateActions()
{
    //     menubar;
    //     QString sSheet;
    //     int sz = sysFont.pointSize();
    //
    //     if ( sz == -1) {
    //         sz = sysFont.pixelSize();
    //         sSheet = QString("QMenuBar { font-size: %1px; }").arg( fontSize );
    //     } else {
    //         sSheet = QString("QMenuBar { font-size: %1pt; }").arg( fontSize );
    //     }

    buttonsWidget->translate();

    //     QString sSheet2 = QString("QMenuBar { font-size: %1px; }").arg( fontSize + 2 );
    //     menubar->setStyleSheet(sSheet);
    menuFile->setTitle(translateString(_FILE));
    //      menuFile->setStyleSheet(sSheet);
    //     menuFile->setFont(sysFont);
    translateActions(newProjAct, _P18);
    //     newProjAct->setStatusTip ( tr ( "Create a new project" ) );

    //     translateActions(openProjAct, "&amp;Open...\\tCtrl+O");
    //     openProjAct->setStatusTip ( tr ( "Open project" ) );

    //     translateActions(saveProjAct, "&amp;Save\\tCtrl+S");
    //     saveProjAct->setStatusTip ( tr ( "Save project" ) );

    //     translateActions(delProjAct, "&amp;Delete...");
    //     delProjAct->setStatusTip ( tr ( "Delete a project" ) );

    translateActions(browseAct, _P23);
    browsProjAct->setStatusTip ( translateString ( _BROWSEEXISTS ) );

    translateActions(exitAct, _EXIT);
    //     exitAct->setStatusTip ( tr ( "Exit" ) );

    menuPreferences->setTitle(translateString(_PREF));
    //     menuPreferences->setStyleSheet(sSheet);
    translateActions(loadDefOptAct, _LOADDEF);
    //     loadDefOptAct->setStatusTip ( tr ( "Load default options" ) );

    translateActions(saveOptAct, _SAVEDEF);
    //     saveOptAct->setStatusTip ( tr ( "Save default options" ) );

    translateActions(resetOptToDefAct, _RESETTODEF);
    //     resetOptToDefAct->setStatusTip ( tr ( "Reset to default options" ) );

    translateActions(loadOptAct,  _LOADOPT);
    //     loadOptAct->setStatusTip ( tr ( "Load options..." ) );

    //     translateActions(recentAct, "Recent file");
    //     recentAct->setStatusTip ( tr ( "Recent file" ) );

    translateActions(saveOptAsAct, _SAVEOPTAS);
    //     translateActions(selectFontAct, _SEL_FONT);
    //     saveOptAsAct->setStatusTip ( tr ( "Save options as..." ) );
    langMenu->setTitle(translateString(_LANGPREF));
    //     fontSizeMenu->setTitle(translateString(_SEL_FONT));

    //     langMenu->setFont(sysFont);
    //     langMenu->setStyleSheet(sSheet2);
    // langGroup->setProperty()
    //     langPrefAct->setStatusTip ( tr ( "Language preference..." ));

    menuMirror->setTitle(translateString(_MIRROR));
    // menuMirror->setStyleSheet(sSheet);
    translateActions(modifyOptAct, _MODIFYOPT);
    //     modifyOptAct->setStatusTip ( tr ( "Modify options" ) );

    pauseOptAct->setCheckable(true);
    translateActions(pauseOptAct, _PAUSETRANSF);
    //     pauseOptAct->setStatusTip ( tr ( "Pause transfer" ) );

    menuLog->setTitle(translateString(_LOG));
    //     menuLog->setStyleSheet(sSheet);
    translateActions(viewLogAct, _VIEW_LOG);
    //     viewLogAct->setStatusTip ( translateString("View log" ) );

    translateActions(viewErrLogAct, _VIEWERRLOG);
    //     viewErrLogAct->setStatusTip ( tr ( "View error log" ) );

    //     translateActions(viewTransferAct,  _VIEWFILETRANSFERS);
    //     viewTransferAct->setStatusTip ( tr ( "View file transfers" ) );

    menuWindow->setTitle(translateString(_WINDOW));
    //  menuWindow->setStyleSheet(sSheet);
    translateActions(hideAct, _HIDE);
    translateActions(restAct, _OPEN);
    //     hideAct->setStatusTip ( tr ( "Hide" ) );

    //     winStatusAct->setStatusTip ( tr ( "Status Bar" ) );

    menuHelp->setTitle(translateString(_HELP));
    //     menuHelp->setStyleSheet(sSheet);
    translateActions(checkAct, _TOPROJECTPAGE);
    translateActions(stepByStepAct, _STEPBYSTEP);
    translateActions(contentAct, _CONTENTS);
    translateActions(aboutAct, _ABOUTPROG);
}



void HTTraQt::quit()
{
    int ans;
    ans = messageBox(translateString(_QUIT), translateString(_REALLYQUIT), QMessageBox::Question);

    if (ans == QMessageBox::No) {
        return;
    }

    writeGUISettings();

    qApp->quit();
}


void HTTraQt::closeEvent(QCloseEvent* ce)
{
    int ans = messageBox(translateString(_QUIT), translateString(_REALLYQUIT), QMessageBox::Question);

    if (ans == QMessageBox::No) {
        ce->ignore();
        return;
    }

    writeGUISettings();
    //     writeSettings(false);

    ce->accept();
    return;
}


void HTTraQt::aboutQt()
{
    QMessageBox::aboutQt(this, PROGRAM_NAME);
}


void HTTraQt::about()
{
    AboutDialog *ad;
    ad = new AboutDialog(this);

    ad->exec();
    delete ad;
}


// modif RX 10/10/98 pour gestion des , et des tabs
void HTTraQt::StripControls(QString &st)
{
    //     st.replace(0x09, " ");
    //     st.replace(0x0a, " ");
    //     st.replace(0x0d, " ");
    st.simplified();
    st.replace("  ", " ");
}


// modifs RX 10/10/98: gestion des ,
QString HTTraQt::change(char* chaine, char c)
{
    int comma = 1;
    int first = 1;
    QString chaine1;

    for (int i = 0; i < (int) strlen(chaine); i++) {
        switch (chaine[i]) {
            case 10:
            case 13:
            case 9:
            case ' ':
            case ',':
                comma = 1;
                break;

            default:

                if (comma) {
                    if (!first) {
                        chaine1 += ' ';
                    } else {
                        first = 0;
                    }

                    chaine1  += c;

                    comma = 0;
                }

                chaine1 += chaine[i];

                break;
        }
    }

    return chaine1;
}


// Ecriture profiles
void HTTraQt::profile_code(QString& from)
{
    from.replace("%%", "%");    // delete double entries
    from.replace("\n\n", "\n");    // delete double entries
    from.replace("%", "%%");
    from.replace("\r", "%0a");
    from.replace("\n", "%0d%0a");
    from.replace("\t", "%09");
}


void HTTraQt::profile_decode(QString& from)
{
    from.replace("%0d%0a", "\n");
    from.replace("%0d", "\n");
    from.replace("%0a", "\r");
    from.replace("%09", "\t");
    from.replace("%%", "%");
}


void HTTraQt::SetProfile(QString variName, float val)
{
    QVector<options>::iterator iopt;

    if (variName.length() < 1) {
        return;
    }

    for (iopt = prOptions.begin(); iopt != prOptions.end(); ++iopt) {
        if ((*iopt).name == variName) {
            if ((*iopt).type != FLOAT) {
                qDebug() << "Wrong type!" << (*iopt).name;
                break;
            }

            (*iopt).value = val;
            break;
        }
    }
}


void HTTraQt::SetProfile(QString variName, int val)
{
    QVector<options>::iterator iopt;

    if (variName.length() < 1) {
        return;
    }

    for (iopt = prOptions.begin(); iopt != prOptions.end(); ++iopt) {
        if ((*iopt).name == variName) {
            if ((*iopt).type != INTEGER) {
                qDebug() << "Wrong type!" << (*iopt).name;
                break;
            }

            (*iopt).value = val;
            break;
        }
    }
}


void  HTTraQt::SetProfile(QString variName, const QString &val)
{
    QVector<options>::iterator iopt;

    if (variName.length() < 1) {
        return;
    }

    for (iopt = prOptions.begin(); iopt != prOptions.end(); ++iopt) {
        if ((*iopt).name == variName) {
            if ((*iopt).type != TEXT) {
                qDebug() << "Wrong type!" << (*iopt).name;
                break;
            }

            (*iopt).value = val;
            break;
        }
    }
}


void HTTraQt::GetProfile(QString variName, float &val)
{
    QVector<options>::iterator iopt;

    if (variName.length() < 1) {
        return;
    }

    for (iopt = prOptions.begin(); iopt != prOptions.end(); ++iopt) {
        if ((*iopt).name == variName) {
            if ((*iopt).type != FLOAT) {
                qDebug() << "Wrong type!" << (*iopt).name;
                break;
            }

            val = (*iopt).value.toFloat();
            break;
        }
    }
}


void HTTraQt::GetProfile(QString variName, int &val)
{
    QVector<options>::iterator iopt;

    if (variName.length() < 1) {
        return;
    }

    for (iopt = prOptions.begin(); iopt != prOptions.end(); ++iopt) {
        if ((*iopt).name == variName) {
            if ((*iopt).type != INTEGER) {
                qDebug() << "Wrong type!" << (*iopt).name;
                break;
            }

            val = (*iopt).value.toInt();
            break;
        }
    }
}


void HTTraQt::GetProfile(QString variName, QString &val)
{
    QVector<options>::iterator iopt;

    if (variName.length() < 1) {
        return;
    }

    for (iopt = prOptions.begin(); iopt != prOptions.end(); ++iopt) {
        if ((*iopt).name == variName) {
            if ((*iopt).type != TEXT) {
                qDebug() << "Wrong type!" << (*iopt).name;
                break;
            }

            val = (*iopt).value.toString();
            break;
        }
    }
}


void HTTraQt::getMainOptionsFromGUI()
{
    QString str;
    QStringList sList;
    //     int val;

    // str = firstWidget->label1027->currentText();
    // SetProfileString(mainOptions, "Category", str);
    str = (*(OptionsTab*)widgets[2]).textEdit->toPlainText();
    SetProfile("CurrentUrl", str);

    //     str.replace("\n", " ");
    //   qDebug() << "set current" << str;
    // interprate the combobox
    //     val = (*(OptionsTab*)widgets[2]).comboAction->currentIndex();
    //     str = translateString((*(OptionsTab*)widgets[2]).comb01);
    //     sList = str.split("\n");
    //     str = sList.at(val);
    //     SetProfile("CurrentAction", val);

    //end of combobox
    str = (*(OptionsTab*)widgets[2]).label1286->text();
    SetProfile("CurrentURLList", str);
}


void HTTraQt::gOptions(bool dialog)
{
    if (mth == NULL ) {
        return;
    }

    OptionsDialog *oDia;
    httrackp *opt = hts_create_opt();

    opt->log = opt->errlog = NULL;

    // dévalider champs (non modifiés)
    opt->maxsite = -1;
    opt->maxfile_nonhtml = -1;
    opt->maxfile_html = -1;
    opt->maxsoc = -1;
    opt->nearlink = -1;
    opt->timeout = -1;
    opt->rateout = -1;
    opt->maxtime = -1;
    //     opt->check_type = 0;
    opt->mms_maxtime = -1;
    opt->maxrate = -1;
    StringClear(opt->user_agent);
    opt->retry = -1;
    opt->hostcontrol = -1;
    opt->errpage = -1;
    opt->travel = -1;
    opt->external = -1;
    opt->delete_old = -1;
    opt->parseall = -1;
    opt->delete_old = -1;
    opt->travel = 0;       // NOTE: NE SERA PRIS EN COMPTE QUE LE BIT 8

    if (dialog == true) {
        oDia = new OptionsDialog(this);
        //         oDia->setFont(sysFont);
        oDia->exec();
    }

    int n;
    float nf;
    GetProfile("Test", n);

    if (n) {
        opt->travel |= 256;
    }

    //     GetProfile("CheckType", n);
    //     opt->check_type = n;

    GetProfile("ParseAll", n);

    if (n) {
        opt->parseall = 1;
    } else {
        opt->parseall = 0;
    }

    // near link,err page
    GetProfile("Near", n);

    if (n) {
        opt->nearlink = 1;
    } else {
        opt->nearlink = 0;
    }

    GetProfile("NoErrorPages", n);

    if (n) {
        opt->errpage = 1;
    } else {
        opt->errpage = 0;
    }

    GetProfile("NoExternalPages", n);

    if (n) {
        opt->external = 1;
    } else {
        opt->external = 0;
    }

    GetProfile("NoPurgeOldFiles", n);

    if (n) {
        opt->delete_old = 1;
    } else {
        opt->delete_old = 0;
    }


    // host control
    {
        int a = 0;
        GetProfile("RemoveTimeout", n);

        if (n > 0) {
            a += 1;
        }

        GetProfile("RemoveRateout", n);

        if (n > 0) {
            a += 2;
        }

        opt->hostcontrol = a;
    }

    // sockets
    GetProfile("Sockets", n);

    if (n >= 0) {
        opt->maxsoc = n;
        maxProgressRows = n;
    } else {
        maxProgressRows = 8;
        opt->maxsoc = 8;
    }

    // maxfile_nonhtml
    GetProfile("MaxOther", nf);

    if (nf > 0) {
        opt->maxfile_nonhtml = (qint64)(nf * 1024.0 * 1024.0);
    } else {
        opt->maxfile_nonhtml = -1;
    }

    // maxfile_html
    GetProfile("MaxHtml", nf);

    if (nf > 0) {
        opt->maxfile_html = (qint64)nf * 1024.0 * 1024.0;
    } else {
        opt->maxfile_html = -1;
    }

    // maxsite
    GetProfile("MaxAll", nf);

    if (nf > 0) {
        opt->maxsite = (qint64)(nf * 1024.0 * 1024.0);
    } else {
        opt->maxsite = -1;
    }

    // fragment
    GetProfile("MaxWait", n);

    if (n > 0) {
        opt->fragment = (qint64)n;
    } else {
        opt->fragment = -1;
    }

    // timeout
    GetProfile("TimeOut", n);

    if (n >= 0) {
        opt->timeout = n;
    } else {
        opt->timeout = -1;
    }

    // rateout
    GetProfile("RateOut", n);

    if (n != 0) {
        opt->rateout = n;
    } else {
        opt->rateout = -1;
    }

    // maxtime
    GetProfile("MaxTime", n);

    if (n > 0) {
        opt->maxtime = n;
    } else {
        opt->maxtime = -1;
    }

    // maxrate
    GetProfile("MaxRate", nf);

    if (nf > 0) {
        opt->maxrate = (int)(nf * 1024.0 * 1024.0);
    } else {
        opt->maxrate = -1;
    }

    // max. connect
    GetProfile("MaxConn", n);

    if (n > 0) {
        opt->maxconn = n;
    } else {
        opt->maxconn = -1;
    }

    // retry
    GetProfile("Retry", n);

    if (n > 0) {
        opt->retry = n;
    } else {
        opt->retry = -1;
    }

    // user_agent
    QString os, st;
    GetProfile("BrowserID", st);
    GetProfile("osID", os);

    if (st.length() > 0) {
        if (st.contains("%s")) {
            st.replace("%s", os);
        }

        SetProfile("UserID", st);
        StringCopy(opt->user_agent, st.toLatin1().data());
    }

    if (global_opt != NULL) {
        copy_htsopt(opt, global_opt);
    }

    hts_free_opt(opt);

    if (dialog == true) {
        delete oDia;
        writeSettings(false);
    }
}


void HTTraQt::setOptions()
{
    gOptions(false); // without dialog
}


void HTTraQt::modifyOptions()
{
    gOptions(true);
}



void HTTraQt::cmdArgumentsOptions()
{
    QString st;
    int n;
    float nf;
    int n1;
    char choixdeb;
    int action;

    cmdOpt = "";
    action = (*(OptionsTab*)widgets[2]).comboAction->currentIndex(); //GetProfile("CurrentAction", action);

    m_todo = action;

    QString array;
    array = "wWgY!/i";

    choixdeb = array.at(action).toLatin1();

    if (choixdeb != 'W') {
        cmdOpt += "-q ";
    }

    GetProfile("BuildTopIndex", n);

    if (n == 0) {
        cmdOpt += "-%i0 ";
    } else {
        cmdOpt += "-%i ";
    }

    if (choixdeb == '/') {
        cmdOpt += "-i ";
    } else if (choixdeb != '!') {
        cmdOpt += ("-" + QString(choixdeb) + " ");
    }

    QString url, urllist;
    GetProfile("CurrentUrl", url);
    url.replace("\n", " ");
    url.simplified();

    GetProfile("CurrentURLList", urllist);

    if (url.length() == 0 && urllist.length() == 0) {
        // nothong to download
        return;
    }

    if (url.length() > 0) { // ersetzen!
        cmdOpt += " " + url + " ";//+"\"";
    }

    if (urllist.length() > 0) {
        cmdOpt += (" -%L \"" + urllist + "\" ");
    }

    GetProfile("Depth", n);

    if (n >= 0) {
        cmdOpt += ("-r" + QString::number(n) + " ");
    }

    GetProfile("ExtDepth", n);

    if (n >= 0) {
        cmdOpt += ("-%e" + QString::number(n) + " ");
    }

    if (choixdeb == '/') {
        cmdOpt += "-C1 ";
    } else {
        GetProfile("Cache", n);

        if (n == 0) {
            cmdOpt += "-C0 ";
        } else {
            cmdOpt += "-C2 ";
        }
    }

    GetProfile("NoRecatch", n);

    if (n != 0) {
        cmdOpt += "-%n ";
    }

    GetProfile("Test", n);

    if (n == 1) {
        cmdOpt += "-t ";
    }

    GetProfile("ParseAll", n);

    if (n == 1) {
        cmdOpt += "-%P ";
    } else {
        cmdOpt += "-%P0 ";
    }

    GetProfile("Near", n);

    if (n == 1) {
        cmdOpt += "-n ";
    }

    GetProfile("NoExternalPages", n);

    if (n != 0) {
        cmdOpt += "-x ";
    }

    GetProfile("NoPurgeOldFiles", n);

    if (n != 0) {
        cmdOpt += "-X0 ";
    }

    GetProfile("NoPwdInPages", n);

    if (n != 0) {
        cmdOpt += "-%x ";
    }

    GetProfile("NoQueryStrings", n);

    if (n != 1) { // include-query-string
        cmdOpt += "-%q ";
    }

    GetProfile("FollowRobotsTxt", n);

    if (n >= 0) {
        cmdOpt += ("-s" + QString::number(n) + " ");
    }

    GetProfile("Cookies", n);

    if (n == 0) {
        cmdOpt += "-b0 ";
    }

    GetProfile("CheckType", n);

    if (n > 0) {
        cmdOpt += ("-u" + QString::number(n) + " ");
    }

    GetProfile("ParseJava", n);

    if (n == 0) {
        cmdOpt += "-j0 ";
    }

    GetProfile("StoreAllInCache", n);

    if (n != 0) {
        cmdOpt += "-k ";
    }

    GetProfile("LogType", n);

    if (n == 1) {
        cmdOpt += "-z ";
        GetProfile("Debug", n);

        if (n == 1) {
            cmdOpt += "-%H ";
        }
    } else if (n == 2) {
        cmdOpt += "-Z ";
        GetProfile("Debug", n);

        if (n == 1) {
            cmdOpt += "-%H ";
        }
    }

    GetProfile("HTTP10", n);

    if (n != 0) {
        cmdOpt += "-%h ";
    }

    GetProfile("TolerantRequests", n);

    if (n > 0) {
        cmdOpt += "-%B ";
    }

    GetProfile("UpdateHack", n);

    if (n > 0) {
        cmdOpt += "-%s ";
    }

    GetProfile("URLHack", n);

    if (n > 0) {
        cmdOpt += "-%u ";
    } else {
        cmdOpt += "-%u0 ";
    }

    if (choixdeb != 'g') {
        QString bld;

        GetProfile("Build", n);

        if (n == 14) {   // i.O.
            GetProfile("BuildString", st);
            bld = ("-N \"" + st + "\"");
            //             cmdOpt += bld;
        } else {
            if (n >= 0 && n <= 5) {
                bld = "-N" + QString::number(n);
            } else if (n >= 6 && n <= 11) {
                bld = "-N10" + QString::number(n - 6);
            } else if (n == 12) {
                bld = "-N99";
            } else if (n == 13) {
                bld = "-N199";
            }
        }

        if (bld.length() > 0) {
            cmdOpt += (bld + " ");
        }
    }

    GetProfile("Dos", n);
    GetProfile("ISO9660", n1);

    if (n > 0) {
        cmdOpt += "-L0 ";
    } else if (n1 > 0) {
        cmdOpt += "-L2 ";
    }

    GetProfile("Index", n);

    if (n == 0) {
        cmdOpt += "-I0 ";
    }

    GetProfile("WordIndex", n);

    if (n == 0) {
        cmdOpt += "-%I0 ";
    } /*else {

        cmdOpt += "-%I ";
    }*/
    /*
        GetProfile("HTMLFirst", n);

        if (n == 1) {
            cmdOpt += "-p7 ";
        }
    */
    QString filter;
    GetProfile("PrimaryScan", n);      // filter

    if (n >= 0 && n <= 2) {
        filter = "-p" + QString::number(n) + " ";
    } else if (n == 3) {   /* default */
        GetProfile("HTMLFirst", n1);

        if (!n1) {
            filter = "-p3 ";
        }
    } else {
        if (n == 4) {
            filter = "-p7 ";
        }
    }


    GetProfile("Travel", n);

    if (n == 0) {
        filter += "-S ";
    } else if (n == 1) {
        filter += "-D ";
    } else if (n == 2) {
        filter += "-U ";
    } else if (n == 3) {
        filter += "-B ";
    }

    //
    GetProfile("GlobalTravel", n);

    if (n == 0) {
        filter += "-a ";
    } else if (n == 1) {
        filter += "-d ";
    } else if (n == 2) {
        filter += "-l ";
    } else if (n == 3) {
        filter += "-e ";
    }

    //
    GetProfile("RewriteLinks", n);

    if (n == 0) {
        filter += "-K0 ";
    } else if (n == 1) {
        filter += "-K ";
    } else if (n == 2) {
        filter += "-K3 ";
    } else if (n == 3) {
        filter += "-K4 ";
    }

    cmdOpt += filter;

    //sizemax
    GetProfile("MaxAll", nf);

    if (n > 0) {
        cmdOpt += ("-M" + QString::number((qint64)(nf * 1024.0 * 1024.0)) + " ");
    }

    //pausebytes
    GetProfile("MaxWait", n);

    if (n > 0) {
        cmdOpt += ("-G" + QString::number(n) + " ");
    }

    float nf1;
    GetProfile("MaxHtml", nf);
    GetProfile("MaxOther", nf1);

    if (nf >= 0 ||  nf1 >= 0) {
        cmdOpt += "-m";

        if (nf1 > 0) {
            cmdOpt += QString::number((qint64)(nf1 * 1024.0 * 1024.0));
        } else {
            cmdOpt += "0";
        }

        if (nf > 0) {
            cmdOpt += ("," + QString::number((qint64)(nf * 1024.0 * 1024.0)));
        } else {
            cmdOpt += ",0";
        }

        cmdOpt += " ";
    }

    GetProfile("Sockets", n);

    if (n >= 0) {
        maxProgressRows = n;
    } else {
        maxProgressRows = 8;
    }

    cmdOpt += ("-c" + QString::number(n) + " ");

    GetProfile("TimeOut", n);

    if (n > 0) {
        cmdOpt += ("-T" + QString::number(n) + " ");
    }

    GetProfile("RateOut", n);

    if (n > 0) {
        cmdOpt += ("-J" + QString::number(n) + " ");
    }

    GetProfile("Retry", n);

    if (n > 0) {
        cmdOpt += ("-R" + QString::number(n) + " ");
    }

    int a = 0;
    GetProfile("RemoveTimeout", n);

    if (n > 0) {
        a += 1;
    }

    GetProfile("RemoveRateout", n);

    if (n > 0) {
        a += 2;
    }

    if (a > 0 ) {
        cmdOpt += ("-H" + QString::number(a) + " ");
    }

    GetProfile("KeepAlive", n);

    if (n > 0) {
        cmdOpt += "-%k ";
    } else {
        cmdOpt += "-%k0 ";
    }

    GetProfile("Log", n);

    if (n != 0) {
        cmdOpt += "-f2 ";
    } else {
        cmdOpt += "-Q ";
    }

    GetProfile("NoErrorPages", n);

    if (n > 0) {
        cmdOpt += "-o0 ";
    }

    //
    GetProfile("MaxTime", n);

    if (n > 0) {
        cmdOpt += ("-E" + QString::number(n) + " ");         // max timeout
    }

    GetProfile("MaxRate", nf);

    if (nf > 0.0) {
        cmdOpt += ("-A" + QString::number((int)(nf * 1024.0 * 1024.0)) + " ");     // max rate
    }

    //     maxConnections = 0;

    GetProfile("MaxConn", n);

    if (n > 0) {
        cmdOpt += ("-%c" + QString::number(n) + " ");      // max connections
        //         maxConnections = n;
    }

    GetProfile("MaxLinks", n);

    if (n > 0) {
        cmdOpt += ("-#L" + QString::number(n) + " ");         // max links
    }

    GetProfile("UseHTTPProxyForFTP", n);

    if (n > 0) {
        cmdOpt += "-%f ";
    } /*else {

        cmdOpt += "-%f0 ";
    }*/

    //     cmdOpt += "#f";  // flush

    QString os;
    GetProfile("BrowserID", st);
    GetProfile("osID", os);

    if (st.length() > 0) {
        if (st.contains("%s")) {
            st.replace("%s", os);
        }

        // qDebug() << "cmd" << st;
        cmdOpt += (" -F \"" + st + "\" ");
    }

    GetProfile("Footer", st);

    if (st.length() > 0) {
        QString stTmp = st;
        cmdOpt += (" -%F \"" + stTmp + "\" ");
    }

    GetProfile("Proxy", st);

    if (st.length() > 0) {
        GetProfile("ProxyPort", n);
        cmdOpt += (" -P " + st + ":" + QString::number(n) + " ");
    }

    // lang iso
    QString lng_iso = selectedLang;//translateString("LANGUAGE_ISO");

    if (lng_iso.length() > 0) {
        cmdOpt +=  " -%l \"" + lng_iso;

        if (lng_iso != "en") {
            cmdOpt += ",en";
        }

        cmdOpt += "\" ";
    }

    // end of lang
    if (choixdeb == '!') {
        cmdOpt += " --testlinks ";
    } else if (choixdeb == 'Y') {
        cmdOpt += " --mirrorlinks ";
    }

    if (currentProject.length() > 0) {
        cmdOpt += " -O \"" + currentWorkDir + "/" + currentProject + "/\" ";
    }

    GetProfile("WildCardFilters", st);

    if (st.length() > 0) {
        cmdOpt += (" " + st);
    }

    for (a = '1'; a < '9'; a++) {
        QString st1, st2;
        GetProfile(("MIMEDefsMime" + a), st1);
        GetProfile(("MIMEDefsExt" + a), st2);

        if (st1.length() > 0 && st2.length() > 0) {
            cmdOpt += (" --assume " + st2 + "=" + st1 + " ");
        }
    }

    // clean
    StripControls(cmdOpt);
}


void HTTraQt::setMainOptionsToGUI()
{
    QString str;
    QStringList sList;
    //     int val;

    readSettings(true);
    // GetProfileString(mainOptions, "Category", str);
    //   str = firstWidget->label1027->Text();
    GetProfile("CurrentUrl", str);
    str.replace(" ", "\n");

    (*(OptionsTab*)widgets[2]).textEdit->setText(str);

    str = (*(OptionsTab*)widgets[2]).comb01;
    sList = str.split("\n");
    //     GetProfile("CurrentAction", val);
    //   val =0;
    //   for (val = 0; val < sList.size(); val++){
    //     if (sList.at(val) == str)
    //       break;
    //   }
    //     (*(OptionsTab*)widgets[2]).comboAction->setCurrentIndex(val);

    GetProfile("CurrentURLList", str);
    (*(OptionsTab*)widgets[2]).label1286->setText(str);
}


bool HTTraQt::RemoveEmptyDir(QString path)
{
    int ans = messageBox(translateString(_REMOVE), translateString(_SURE), QMessageBox::Question);

    if (ans == QMessageBox::No) {
        return false;
    }

    QDirIterator it(path, QDirIterator::Subdirectories);

    while (it.hasNext()) {
        QString n = it.next();

        if (n == "." || n == "..") {
            continue;
        }

        QString fullPath = path + "/" + n;

        QDir subd(fullPath);

        if (subd.rmdir(fullPath) == false) {
            messageBox(translateString(_ERR), translateString(_ANYDIRNOTEMPTY), QMessageBox::Warning);

            return false;
        }
    }

    QStringList names;

    QString maske = path + "/*.*";
    QDir d(path);
    names = d.entryList(QStringList("*.*"));

    for (QStringList::Iterator ist = names.begin(); ist != names.end(); ++ist) {
        if (*ist == "." || *ist == "..") {   // is dir
            continue;
        } else {
            d.remove(path + "/" + (*ist));
        }
    }

    if (d.remove(maske) == true) {
        d.rmdir(path);
    }

#ifdef WIN32
    QDir::setCurrent("C:\\");

#else
    QString home = QDir::homePath();

    QDir::setCurrent(home);

#endif

    return true;
}

