/* ptal-pml -- PTAL command-line PML client */

/* Copyright (C) 2000-2002 Hewlett-Packard Company
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * is provided AS IS, WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, and
 * NON-INFRINGEMENT.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston,
 * MA 02111-1307, USA.
 *
 * In addition, as a special exception, Hewlett-Packard Company
 * gives permission to link the code of this program with any
 * version of the OpenSSL library which is distributed under a
 * license identical to that listed in the included LICENSE.OpenSSL
 * file, and distribute linked combinations including the two.
 * You must obey the GNU General Public License in all respects
 * for all of the code used other than OpenSSL.  If you modify
 * this file, you may extend this exception to your version of the
 * file, but you are not obligated to do so.  If you do not wish to
 * do so, delete this exception statement from your version.
 */

/* Original author: David Paschal */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "ptal.h"


/* TODO: Finish implementing CMD_GETNEXT and CMD_TRAP. */
#define CMD_NONE	0
#define CMD_DEV		1
#define CMD_GET		2
/* #define CMD_GETNEXT	3 */
#define CMD_SET_OID	4
#define CMD_SET_TYPE	5
#define CMD_SET_VALUE	6
/* #define CMD_TRAP	7 */

#define PTAL_PML_TYPE_BINARY_STRING (PTAL_PML_TYPE_BINARY+0x1000)

int strLooksLike(char *arg,char *str) {
	return strstr(str,arg)==str;
}

int parseType(char *s) {
	if (strLooksLike(s,"enumeration")) {
		return PTAL_PML_TYPE_ENUMERATION;
	}
	if (strLooksLike(s,"integer")) {
		return PTAL_PML_TYPE_SIGNED_INTEGER;
	}
	if (strLooksLike(s,"string")) {
		return PTAL_PML_TYPE_STRING;
	}
	if (strLooksLike(s,"binary")) {
		return PTAL_PML_TYPE_BINARY;
	}
	if (strLooksLike(s,"bstring")) {
		return PTAL_PML_TYPE_BINARY_STRING;
	}
	if (strLooksLike(s,"null")) {
		return PTAL_PML_TYPE_NULL_VALUE;
	}
	if (strLooksLike(s,"collection")) {
		return PTAL_PML_TYPE_COLLECTION;
	}
	return PTAL_ERROR;
}

int parseGetCommand(char *arg,char *str,int *pType) {
	int len=strlen(str);
	if (strstr(arg,str)!=arg) return 0;
	if (pType) {
		*pType=PTAL_ERROR;
		if (arg[len]=='-') {
			*pType=parseType(arg+len+1);
			if (*pType==PTAL_ERROR) return 0;
		}
	}
	return 1;
}

char *printType(int type) {
	switch (type) {
	   case PTAL_PML_TYPE_ENUMERATION:
		return "enumeration";
	   case PTAL_PML_TYPE_SIGNED_INTEGER:
		return "integer";
	   case PTAL_PML_TYPE_STRING:
		return "string";
	   case PTAL_PML_TYPE_BINARY_STRING:
		return "bstring";
	   case PTAL_PML_TYPE_BINARY:
		return "binary";
	   case PTAL_PML_TYPE_NULL_VALUE:
		return "null";
	   case PTAL_PML_TYPE_COLLECTION:
		return "collection";
	   default:
		return "?";
	}
}

int printObject(ptalPmlObject_t obj,int status,int type) {
	int symbolSet,i,len,ivalue;
	char oid[PTAL_PML_MAX_OID_LEN];
	char svalue[PTAL_PML_MAX_VALUE_LEN+1];

	if (ptalPmlGetID(obj,oid,PTAL_PML_MAX_OID_LEN)==PTAL_ERROR) {

		return PTAL_ERROR;
	}
	for (i=0;oid[i];i++) {
		printf("%s%d",i?".":"",((unsigned)oid[i])&0xFF);
	}
	printf("  ");

	if (type==PTAL_ERROR) {
		type=ptalPmlGetType(obj);
	}
	if (status==PTAL_ERROR || type==PTAL_ERROR) {
		printf("failed (status=0x%2.2X)!\n",ptalPmlGetStatus(obj));

	} else if (type==PTAL_PML_TYPE_ENUMERATION ||
	    type==PTAL_PML_TYPE_SIGNED_INTEGER ||
	    type==PTAL_PML_TYPE_NULL_VALUE ||
	    type==PTAL_PML_TYPE_COLLECTION) {
		if (ptalPmlGetIntegerValue(obj,0,&ivalue)==PTAL_ERROR) {
			printf("failed (ptalPmlGetIntegerValue)!\n");
		} else {
			printf("%s  %d (0x%8.8X)\n",
				printType(type),ivalue,ivalue);
		}

	} else if (type==PTAL_PML_TYPE_STRING) {
		len=ptalPmlGetStringValue(obj,&symbolSet,svalue,
			PTAL_PML_MAX_VALUE_LEN);
		if (len==PTAL_ERROR) {
			printf("failed (ptalPmlGetStringValue)!\n");
		} else {
			printf("%s(symbolSet=0x%4.4X,len=%d)  \"%s\"\n",
				printType(type),symbolSet,len,svalue);
		}

	} else if (type==PTAL_PML_TYPE_BINARY_STRING) {
		len=ptalPmlGetValue(obj,0,svalue,PTAL_PML_MAX_VALUE_LEN);
		if (len==PTAL_ERROR) {
			printf("failed (ptalPmlGetValue(bstring))!\n");
		} else {
			printf("%s(len=%d)  \"%s\"\n",
				printType(type),len,svalue);
		}

	} else {
		len=ptalPmlGetValue(obj,0,svalue,PTAL_PML_MAX_VALUE_LEN);
		if (len==PTAL_ERROR) {
			printf("failed (ptalPmlGetValue(type=%d))!\n",type);
		} else {
			if (type==PTAL_PML_TYPE_BINARY) {
				printf("%s(len=%d) ",
					printType(type),len);
			} else {
				printf("%s(type=0x%2.2X,len=%d) ",
					printType(type),type,len);
			}
			for (i=0;i<len;i++) {
				printf(" %2.2X",
					((unsigned char)svalue[i])&0xFF);
			}
			printf("\n");
		}
	}

	return PTAL_OK;
}

int main(int argc,char **argv) {
	char *argv0=*argv;
	int cmd=CMD_DEV,type=PTAL_ERROR,r;
#ifdef CMD_TRAP
	int trapRequested=0
#endif
	ptalDevice_t dev=0;
	ptalPmlObject_t obj=0,next;

	ptalInit();

	while (42) {
		argc--; argv++; if (argc<=0) break;

		if (cmd==CMD_DEV) {
			if (**argv=='-') goto syntaxError;
			if (!(dev=ptalDeviceOpen(*argv))) {
				PTAL_LOG_ERROR("%s: invalid device \"%s\"!\n",
					argv0,*argv);
				return 3;
			}
			if (ptalPmlOpen(dev)==PTAL_ERROR) {
				PTAL_LOG_ERROR("%s: error opening PML "
					"for device \"%s\"!\n",
					argv0,*argv);
				return 4;
			}
			cmd=CMD_NONE;

		} else if (cmd==CMD_SET_TYPE) {
			type=parseType(*argv);
			if (type==PTAL_ERROR) {
				goto syntaxError;
			}
			cmd=CMD_SET_VALUE;

		} else if (cmd==CMD_SET_VALUE) {
			if (type==PTAL_PML_TYPE_STRING) {
				ptalPmlSetStringValue(obj,0,*argv,
					strlen(*argv));

			} else if (type==PTAL_PML_TYPE_BINARY_STRING) {
				type=PTAL_PML_TYPE_BINARY;
				ptalPmlSetValue(obj,type,*argv,strlen(*argv));

			} else if (type==PTAL_PML_TYPE_BINARY) {
				char *ptr=*argv,*endptr;
				char value[PTAL_PML_MAX_VALUE_LEN];
				int lenValue;

				for (lenValue=0;
				     lenValue<PTAL_PML_MAX_VALUE_LEN;
				     lenValue++) {
					value[lenValue]=strtol(ptr,&endptr,16);
					if (endptr==ptr) break;
					ptr=endptr;
				}
				ptalPmlSetValue(obj,PTAL_PML_TYPE_BINARY,
					value,lenValue);

			} else if (type==PTAL_PML_TYPE_ENUMERATION ||
			    type==PTAL_PML_TYPE_SIGNED_INTEGER ||
			    type==PTAL_PML_TYPE_NULL_VALUE ||
			    type==PTAL_PML_TYPE_COLLECTION) {
				ptalPmlSetIntegerValue(obj,type,
					strtol(*argv,0,0));

			} else {
				goto syntaxError;
			}

			r=ptalPmlRequestSet(obj);
			/* ptalPmlRequestGet(obj); */
			printObject(obj,r,PTAL_ERROR);

			cmd=CMD_SET_OID;

		} else if (!strcmp(*argv,"dev")) {
			cmd=CMD_DEV;

#ifdef CMD_GETNEXT
		} else if (parseGetCommand(*argv,"getnext",&type)) {
			cmd=CMD_GETNEXT;
#endif

		} else if (parseGetCommand(*argv,"get",&type)) {
			cmd=CMD_GET;

		} else if (!strcmp(*argv,"set")) {
			cmd=CMD_SET_OID;

#ifdef CMD_TRAP
		} else if (!strcmp(*argv,"trap")) {
			cmd=CMD_TRAP;
#endif

		} else {
			if (cmd==CMD_NONE) {
				goto syntaxError;
			}

			if (!(obj=ptalPmlAllocate(dev))) {
				PTAL_LOG_ERROR("%s: error allocating "
					"PML object for \"%s\"!\n",
					argv0,*argv);
				return 5;
			}
			if (ptalPmlSetAsciiID(obj,*argv)==PTAL_ERROR) {
				PTAL_LOG_ERROR("%s: error parsing PML "
					"object ID \"%s\"!\n\n",
					argv0,*argv);
				goto syntaxError;
			}

#ifdef CMD_GETNEXT
			if (cmd==CMD_GETNEXT) {
				next=obj;
				goto doGet;
			}
#endif
			if (cmd==CMD_GET) {
				next=0;
#ifdef CMD_GETNEXT
doGet:
#endif
				printObject(obj,ptalPmlRequestGet(obj,next),
					type);

#ifdef CMD_TRAP
			} else if (cmd==CMD_TRAP) {
				trapRequested=1;
				if (ptalPmlRequestSetTrap(obj,1)==PTAL_ERROR) {

				}
#endif

			} else if (cmd==CMD_SET_OID) {
				cmd=CMD_SET_TYPE;
			}
		}
	}

	if (cmd==CMD_DEV || cmd==CMD_SET_TYPE || cmd==CMD_SET_VALUE) {
syntaxError:
		PTAL_LOG_ERROR(
"Syntax: %s <devname> [<cmd> [<cmd>]...]\n"
"Where <devname> is one of:\n"
			,argv0);
		ptalDeviceEnumerate(0,
			ptalDeviceEnumeratePrintCallback,0);
		PTAL_LOG_ERROR(
"<cmd> is one of:\n"
	"\tdev <devname>\n"
	"\tget[-<type>] [<oid>]...\n"
	/* "\tgetnext [<oid>]...\n" */ /* CMD_GETNEXT */
	"\tset [<oid> <type> <value>]...\n"
	/* "\ttrap [<oid>]...\n" */ /* CMD_TRAP */
"<oid> is a dotted-decimal PML object ID "
"(for example, \"1.1.3.2\")\n"
"<type> is one of:\n"
	"\tenumeration\n"
	"\tinteger\n"
	"\tstring\n"
	"\tbinary\n"
	"\tbstring  (same as binary, but parsed/printed as a string)\n"
	"\tcollection\n"
	"\tnull\n"
"For \"enumeration\", \"integer\", \"collection\", and \"null\" types,\n"
	"\t<value> is one of:\n"
	"\t<decimal>\n"
	"\t0<octal>\n"
	"\t0x<hex>\n"
"For \"string\" and \"bstring\" types, <value> is a string enclosed in\n"
	"\tdouble quotes (\").\n"
"For \"binary\" type, <value> is a sequence of hex numbers separated by\n"
	"\tspaces and enclosed in double quotes (\").\n"
			);
		return 1;
	}

#ifdef CMD_TRAP
	if (trapRequested) {



	}
#endif

	return 0;
}
