# Perl module to send reports to the Xymon system monitor
# (formerly known as Hobbit)
#
# Copyright (C) 2008-2011  Christoph Berg <myon@debian.org>
# Copyright (C) 2011       Axel Beckert <abe@debian.org>
#
# Permission is hereby granted, free of charge, to any person
# obtaining a copy of this software and associated documentation
# files (the "Software"), to deal in the Software without
# restriction, including without limitation the rights to use,
# copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the
# Software is furnished to do so, subject to the following
# conditions:
#
# The above copyright notice and this permission notice shall be
# included in all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
# OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
# HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
# WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
# FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
# OTHER DEALINGS IN THE SOFTWARE.

package Hobbit;

use strict;
use warnings;

my %color = (
	clear => 0,
	green => 1,
	purple => 2,
	yellow => 3,
	red => 4,
);

my $gself = {
	hostname => 'hobbit.pm',
	test => 'uninitialized',
};
bless $gself;

sub max_color ($$)
{
	my ($a, $b) = @_;
	die "color $a unknown" unless exists $color{$a};
	die "color $b unknown" unless exists $color{$b};
	return $color{$b} > $color{$a} ? $b : $a;
}

sub new ($)
{
	my $class = shift;
	my $arg = shift;
	unless (ref $arg) {
		$arg = {
			test => $arg,
		};
	}
	unless ($arg->{test}) {
		print STDERR "$0: test name undefined\n";
		exit 1;
	}
	my $self = {
		type => ($arg->{type} || 'status'),
		color => 'clear',
		text => $arg->{text} || '',
		hostname => ($arg->{hostname} || $ENV{CLIENTHOSTNAME} ||
			$ENV{MACHINEDOTS} || $ENV{MACHINE} || "unknown"),
		test => $arg->{test},
		title => $arg->{title},
		ttl => $arg->{ttl},
	};
	$gself = $self;
	bless $self;
}

sub add_color ($)
{
	my ($self, $color) = @_;
	$self->{color} = max_color ($self->{color}, $color);
}

sub print ($)
{
	my ($self, $text) = @_;
	$self->{text} .= $text;
}

sub color_print ($$)
{
	my ($self, $color, $text) = @_;
	$self->add_color ($color);
	$self->print ($text);
}

sub color_line ($$)
{
	my ($self, $color, $text) = @_;
	$self->color_print ($color, "&$color $text");
}

sub send ()
{
	my $self = shift;
	if ($self->{ttl} and
	    $self->{ttl} =~ /^\d+/ and # there might be an h/d/w suffix
	    $self->{type} eq 'status') {
	    $self->{type} = "$self->{type}+$self->{ttl}";
	}
	my $report = "$self->{type} $self->{hostname}.$self->{test}";
	if ($self->{type} =~ m/^status/) {
		my $date = scalar localtime;
		my $title = '';
		if ($self->{color} eq 'green') {
			$title = "$self->{test} OK";
		} elsif ($self->{color} eq 'yellow' or $self->{color} eq 'red') {
			$title = "$self->{test} NOT ok";
		}
		$title = ' - ' . ($self->{title} ? $self->{title} : $title)
			if ($self->{title} or $title);
		$report .= " $self->{color} $date$title";
	}
	$report .= "\n$self->{text}";
	$report .= "\n" unless ($report =~ /\n\n$/);
	if ($ENV{BB} and $ENV{BBDISP}) {
		open F, "| $ENV{BB} $ENV{BBDISP} @";
		print F $report;
		close F;
	} else {
		print $report;
	}
}

sub moan ($)
{
	my $msg = shift;
	my $date = scalar localtime;
	print STDERR "$date $0 $gself->{hostname}.$gself->{test}: $msg";
	$gself->color_line ('yellow', "Warning: $msg");
}

sub croak ($)
{
	my $msg = shift;
	my $date = scalar localtime;
	print STDERR "$date $0 $gself->{hostname}.$gself->{test}: $msg";
	$gself->color_line ('red', "Error: $msg");
	$gself->send();
	exit 1;
}

$SIG{__WARN__} = \&moan;
$SIG{__DIE__} = \&croak;

1;
