/* Copyright (c) 2000, 2016, Oracle and/or its affiliates. All rights reserved.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; version 2 of the License.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301  USA */

#include "vio_priv.h"

#ifdef HAVE_OPENSSL

#if OPENSSL_VERSION_NUMBER >= 0x10100000L
#define CRYPTO_num_locks() (1)
#include <openssl/dh.h>
#include <openssl/bn.h>
#define ERR_free_strings()
#define EVP_cleanup()
#define SSL_library_init()
#define OpenSSL_add_all_algorithms()
#define SSL_load_error_strings()
#define CRYPTO_cleanup_all_ex_data()
// Function removed after OpenSSL 1.1.0
#define ERR_remove_state(x)
#endif

#include <openssl/bio.h>

/*
  Diffie-Hellman key.
  Generated using: >openssl dhparam -5 -C 2048

  -----BEGIN DH PARAMETERS-----
  MIIBCAKCAQEAil36wGZ2TmH6ysA3V1xtP4MKofXx5n88xq/aiybmGnReZMviCPEJ
  46+7VCktl/RZ5iaDH1XNG1dVQmznt9pu2G3usU+k1/VB4bQL4ZgW4u0Wzxh9PyXD
  glm99I9Xyj4Z5PVE4MyAsxCRGA1kWQpD9/zKAegUBPLNqSo886Uqg9hmn8ksyU9E
  BV5eAEciCuawh6V0O+Sj/C3cSfLhgA0GcXp3OqlmcDu6jS5gWjn3LdP1U0duVxMB
  h/neTSCSvtce4CAMYMjKNVh9P1nu+2d9ZH2Od2xhRIqMTfAS1KTqF3VmSWzPFCjG
  mjxx/bg6bOOjpgZapvB6ABWlWmRmAAWFtwIBBQ==
  -----END DH PARAMETERS-----
 */
static unsigned char dh2048_p[]=
{
  0x8A, 0x5D, 0xFA, 0xC0, 0x66, 0x76, 0x4E, 0x61, 0xFA, 0xCA, 0xC0, 0x37,
  0x57, 0x5C, 0x6D, 0x3F, 0x83, 0x0A, 0xA1, 0xF5, 0xF1, 0xE6, 0x7F, 0x3C,
  0xC6, 0xAF, 0xDA, 0x8B, 0x26, 0xE6, 0x1A, 0x74, 0x5E, 0x64, 0xCB, 0xE2,
  0x08, 0xF1, 0x09, 0xE3, 0xAF, 0xBB, 0x54, 0x29, 0x2D, 0x97, 0xF4, 0x59,
  0xE6, 0x26, 0x83, 0x1F, 0x55, 0xCD, 0x1B, 0x57, 0x55, 0x42, 0x6C, 0xE7,
  0xB7, 0xDA, 0x6E, 0xD8, 0x6D, 0xEE, 0xB1, 0x4F, 0xA4, 0xD7, 0xF5, 0x41,
  0xE1, 0xB4, 0x0B, 0xE1, 0x98, 0x16, 0xE2, 0xED, 0x16, 0xCF, 0x18, 0x7D,
  0x3F, 0x25, 0xC3, 0x82, 0x59, 0xBD, 0xF4, 0x8F, 0x57, 0xCA, 0x3E, 0x19,
  0xE4, 0xF5, 0x44, 0xE0, 0xCC, 0x80, 0xB3, 0x10, 0x91, 0x18, 0x0D, 0x64,
  0x59, 0x0A, 0x43, 0xF7, 0xFC, 0xCA, 0x01, 0xE8, 0x14, 0x04, 0xF2, 0xCD,
  0xA9, 0x2A, 0x3C, 0xF3, 0xA5, 0x2A, 0x83, 0xD8, 0x66, 0x9F, 0xC9, 0x2C,
  0xC9, 0x4F, 0x44, 0x05, 0x5E, 0x5E, 0x00, 0x47, 0x22, 0x0A, 0xE6, 0xB0,
  0x87, 0xA5, 0x74, 0x3B, 0xE4, 0xA3, 0xFC, 0x2D, 0xDC, 0x49, 0xF2, 0xE1,
  0x80, 0x0D, 0x06, 0x71, 0x7A, 0x77, 0x3A, 0xA9, 0x66, 0x70, 0x3B, 0xBA,
  0x8D, 0x2E, 0x60, 0x5A, 0x39, 0xF7, 0x2D, 0xD3, 0xF5, 0x53, 0x47, 0x6E,
  0x57, 0x13, 0x01, 0x87, 0xF9, 0xDE, 0x4D, 0x20, 0x92, 0xBE, 0xD7, 0x1E,
  0xE0, 0x20, 0x0C, 0x60, 0xC8, 0xCA, 0x35, 0x58, 0x7D, 0x3F, 0x59, 0xEE,
  0xFB, 0x67, 0x7D, 0x64, 0x7D, 0x8E, 0x77, 0x6C, 0x61, 0x44, 0x8A, 0x8C,
  0x4D, 0xF0, 0x12, 0xD4, 0xA4, 0xEA, 0x17, 0x75, 0x66, 0x49, 0x6C, 0xCF,
  0x14, 0x28, 0xC6, 0x9A, 0x3C, 0x71, 0xFD, 0xB8, 0x3A, 0x6C, 0xE3, 0xA3,
  0xA6, 0x06, 0x5A, 0xA6, 0xF0, 0x7A, 0x00, 0x15, 0xA5, 0x5A, 0x64, 0x66,
  0x00, 0x05, 0x85, 0xB7,
};

static unsigned char dh2048_g[]={
  0x05,
};

static DH *get_dh2048(void)
{
  DH *dh;
  if ((dh=DH_new()))
  {
    BIGNUM *p = BN_bin2bn(dh2048_p,sizeof(dh2048_p),NULL);
    BIGNUM *g = BN_bin2bn(dh2048_g,sizeof(dh2048_g),NULL);
#if OPENSSL_VERSION_NUMBER < 0x10100000L
    dh->p=p;
    dh->g=g;
    if (! dh->p || ! dh->g)
#else
    if (!DH_set0_pqg(dh, p, NULL, g))
#endif
    {
      DH_free(dh);
      dh=0;
    }
  }
  return(dh);
}


static void
report_errors()
{
  unsigned long	l;
  const char*	file;
  const char*	data;
  int		line,flags;

  DBUG_ENTER("report_errors");

  while ((l=ERR_get_error_line_data(&file,&line,&data,&flags)) != 0)
  {
#ifndef DBUG_OFF				/* Avoid warning */
    char buf[200];
    DBUG_PRINT("error", ("OpenSSL: %s:%s:%d:%s\n", ERR_error_string(l,buf),
			 file,line,(flags & ERR_TXT_STRING) ? data : "")) ;
#endif
  }
  DBUG_VOID_RETURN;
}

static const char*
ssl_error_string[] = 
{
  "No error",
  "Unable to get certificate",
  "Unable to get private key",
  "Private key does not match the certificate public key",
  "SSL_CTX_set_default_verify_paths failed",
  "Failed to set ciphers to use",
  "SSL_CTX_new failed",
  "SSL_CTX_set_tmp_dh failed"
};

const char*
sslGetErrString(enum enum_ssl_init_error e)
{
  DBUG_ASSERT(SSL_INITERR_NOERROR < e && e < SSL_INITERR_LASTERR);
  return ssl_error_string[e];
}

static my_bool check_same_file(const char *cert_file, const char *key_file)
{
  MY_STAT cert_stat;
  MY_STAT key_stat;

  if (!my_stat(cert_file, &cert_stat, MYF(0)) ||
      !my_stat(key_file, &key_stat, MYF(0)))
  {
    DBUG_PRINT("error", ("Unable to stat the files."));
    return FALSE;
  }

  return cert_stat.st_ino == key_stat.st_ino &&
          cert_stat.st_dev == key_stat.st_dev;
}

static int vio_load_cert(SSL_CTX* ctx, const char *cert_file, BIO* bio)
{
  X509* cert = NULL;

  if (BIO_read_filename(bio, cert_file) != 1 ||
      !PEM_read_bio_X509(bio, &cert, NULL, NULL))
  {
    return 1;
  }

  int ret = SSL_CTX_use_certificate(ctx, cert);
  X509_free(cert);

  return ret <= 0;
}

static int vio_load_key(SSL_CTX* ctx, const char *key_file,
                        my_bool same_file, BIO* bio)
{
  // read key file if not in same file as cert
  if (!same_file)
  {
    if (BIO_read_filename(bio, key_file) != 1)
    {
      DBUG_PRINT("error", ("BIO_WRITE for key %s failed", key_file));
      return 1;
    }
  }
  else
  {
    if (BIO_reset(bio) < 0) {
      DBUG_PRINT("error", ("BIO_reset failed ret"));
      return 1;
    }
  }

  EVP_PKEY* key = NULL;
  if (!PEM_read_bio_PrivateKey(bio, &key, NULL, NULL))
    return 1;

  int ret = SSL_CTX_use_PrivateKey(ctx, key);
  EVP_PKEY_free(key);

  return ret <= 0;
}

static int
vio_set_cert_stuff(SSL_CTX *ctx, const char *cert_file, const char *key_file,
                   enum enum_ssl_init_error* error)
{
  DBUG_ENTER("vio_set_cert_stuff");
  DBUG_PRINT("enter", ("ctx: 0x%lx  cert_file: %s  key_file: %s",
		       (long) ctx, cert_file, key_file));

  *error = SSL_INITERR_NOERROR;

  my_bool same_file = FALSE;
  if (cert_file == NULL) {
    if (key_file == NULL)
      DBUG_RETURN(0);

    cert_file = key_file;
    same_file = TRUE;
  }
  else if (key_file == NULL) {
    key_file = cert_file;
    same_file = TRUE;
  }
  else
    same_file = check_same_file(key_file, cert_file);


  const char* error_file = NULL;
  BIO* bio = BIO_new(BIO_s_file());
  if (bio == NULL)
    *error = SSL_INITERR_MEMFAIL;
  else
  {
    // load the cert
    if (vio_load_cert(ctx, cert_file, bio))
    {
      *error = SSL_INITERR_CERT;
      error_file = cert_file;
    }
    // load the key
    else if (vio_load_key(ctx, key_file, same_file, bio))
    {
      *error = SSL_INITERR_KEY;
      error_file = key_file;
    }
    /*
      If we are using DSA, we can copy the parameters from the private key
      Now we know that a key and cert have been set against the SSL context
    */
    else if (!SSL_CTX_check_private_key(ctx))
    {
      *error= SSL_INITERR_NOMATCH;
    }

    BIO_free_all(bio);
  }

  if (*error != SSL_INITERR_NOERROR)
  {
    char err_string[MYSQL_ERRMSG_SIZE];
    if (error_file)
      snprintf(err_string, sizeof(err_string), "%s from '%s'",
              sslGetErrString(*error), error_file);
    else
      snprintf(err_string, sizeof(err_string), "%s", sslGetErrString(*error));
    DBUG_PRINT("error", ("%s", err_string));
    DBUG_EXECUTE("error", ERR_print_errors_fp(DBUG_FILE););
    //NO_LINT_DEBUG
    fprintf(stderr, "SSL error: %s\n", err_string);
    fflush(stderr);
    DBUG_RETURN(1);
  }

  DBUG_RETURN(0);
}

static my_bool ssl_initialized = FALSE;

#ifndef HAVE_YASSL
/* OpenSSL specific */

#ifdef HAVE_PSI_INTERFACE
static PSI_rwlock_key key_rwlock_openssl;

static PSI_rwlock_info openssl_rwlocks[]=
{
  { &key_rwlock_openssl, "CRYPTO_dynlock_value::lock", 0}
};
#endif


typedef struct CRYPTO_dynlock_value
{
  mysql_rwlock_t lock;
} openssl_lock_t;


/* Array of locks used by openssl internally for thread synchronization.
   The number of locks is equal to CRYPTO_num_locks.
*/
static openssl_lock_t *openssl_stdlocks;

/*OpenSSL callback functions for multithreading. We implement all the functions
  as we are using our own locking mechanism.
*/
#if OPENSSL_VERSION_NUMBER < 0x10100000L
static void openssl_lock(int mode, openssl_lock_t *lock,
                         const char *file MY_ATTRIBUTE((unused)),
                         int line MY_ATTRIBUTE((unused)))
{
  int err = 0;
  char const *what = 0;

  switch (mode) {
    case CRYPTO_LOCK|CRYPTO_READ:
      what = "read lock";
      err= mysql_rwlock_rdlock(&lock->lock);
      break;
    case CRYPTO_LOCK|CRYPTO_WRITE:
      what = "write lock";
      err= mysql_rwlock_wrlock(&lock->lock);
      break;
    case CRYPTO_UNLOCK|CRYPTO_READ:
    case CRYPTO_UNLOCK|CRYPTO_WRITE:
      what = "unlock";
      err= mysql_rwlock_unlock(&lock->lock);
      break;
    default:
      /* Unknown locking mode. */
      DBUG_PRINT("error",
        ("Fatal OpenSSL: %s:%d: interface problem (mode=0x%x)\n",
          file, line, mode));

      //NO_LINT_DEBUG
      fprintf(stderr, "Fatal: OpenSSL interface problem (mode=0x%x)", mode);
      fflush(stderr);
      abort();
  }
  if (err && what) {
    DBUG_PRINT("error",
      ("Fatal OpenSSL: %s:%d: can't %s OpenSSL lock\n",
      file, line, what));

    //NO_LINT_DEBUG
    fprintf(stderr, "Fatal: can't %s OpenSSL lock", what);
    fflush(stderr);
    abort();
  }
}

static void openssl_lock_function(int mode, int n, const char *file, int line)
{
  if (n < 0 || n > CRYPTO_num_locks())
  {
    /* Lock number out of bounds. */
    DBUG_PRINT("error",
      ("Fatal OpenSSL: %s:%d: interface problem (n = %d)", file, line, n));

    //NO_LINT_DEBUG
    fprintf(stderr, "Fatal: OpenSSL interface problem (n = %d)", n);
    fflush(stderr);
    abort();
  }
  openssl_lock(mode, &openssl_stdlocks[n], file, line);
}

static openssl_lock_t *openssl_dynlock_create(
                         const char *file MY_ATTRIBUTE((unused)),
                         int line MY_ATTRIBUTE((unused)))
{
  DBUG_PRINT("info", ("openssl_dynlock_create: %s:%d", file, line));

  openssl_lock_t *lock= (openssl_lock_t*)
                          my_malloc(sizeof(openssl_lock_t),MYF(0));
  mysql_rwlock_init(key_rwlock_openssl, &lock->lock);

  return lock;
}


static void openssl_dynlock_destroy(openssl_lock_t *lock,
                         const char *file MY_ATTRIBUTE((unused)),
                         int line MY_ATTRIBUTE((unused)))
{
  DBUG_PRINT("info", ("openssl_dynlock_destroy: %s:%d", file, line));

  mysql_rwlock_destroy(&lock->lock);
  my_free(lock);
}

static unsigned long openssl_id_function()
{
  return (unsigned long) pthread_self();
}
#endif

//End of mutlithreading callback functions

static void init_ssl_locks()
{
  int i= 0;
#ifdef HAVE_PSI_INTERFACE
  const char* category= "sql";
  int count= array_elements(openssl_rwlocks);
  mysql_rwlock_register(category, openssl_rwlocks, count);
#endif

  openssl_stdlocks= (openssl_lock_t*) OPENSSL_malloc(CRYPTO_num_locks() *
    sizeof(openssl_lock_t));
  for (i= 0; i < CRYPTO_num_locks(); ++i)
#ifdef HAVE_PSI_INTERFACE
    mysql_rwlock_init(key_rwlock_openssl, &openssl_stdlocks[i].lock);
#else
    mysql_rwlock_init(0, &openssl_stdlocks[i].lock);
#endif
}

static void set_lock_callback_functions(my_bool init)
{
#if OPENSSL_VERSION_NUMBER < 0x10100000L
  CRYPTO_set_locking_callback(init ? openssl_lock_function : NULL);
  CRYPTO_set_id_callback(init ? openssl_id_function : NULL);
  CRYPTO_set_dynlock_create_callback(init ? openssl_dynlock_create : NULL);
  CRYPTO_set_dynlock_destroy_callback(init ? openssl_dynlock_destroy : NULL);
  CRYPTO_set_dynlock_lock_callback(init ? openssl_lock : NULL);
#else
  (void)init;
#endif
}

static void init_lock_callback_functions()
{
  set_lock_callback_functions(TRUE);
}

static void deinit_lock_callback_functions()
{
  set_lock_callback_functions(FALSE);
}

void vio_ssl_end()
{
  int i= 0;

  if (ssl_initialized) {
    ERR_remove_state(0);
    ERR_free_strings();
    EVP_cleanup();

    CRYPTO_cleanup_all_ex_data();

    deinit_lock_callback_functions();

    for (; i < CRYPTO_num_locks(); ++i)
      mysql_rwlock_destroy(&openssl_stdlocks[i].lock);
    OPENSSL_free(openssl_stdlocks);

    ssl_initialized= FALSE;
  }
}

#endif //OpenSSL specific

void ssl_start()
{
  if (!ssl_initialized)
  {
#ifndef HAVE_YASSL
#if !defined(OPENSSL_IS_BORINGSSL) && OPENSSL_VERSION_NUMBER < 0x10100000L
    // if locks already initialized, don't override
    if (CRYPTO_get_locking_callback()) {
      DBUG_PRINT("info", ("Openssl crypto library already initialized."));
      return;
    }
#endif
#endif

    ssl_initialized= TRUE;

    SSL_library_init();
    OpenSSL_add_all_algorithms();
    SSL_load_error_strings();

#ifndef HAVE_YASSL
    init_ssl_locks();
    init_lock_callback_functions();
#endif
  }
}

/************************ VioSSLFd **********************************/
static struct st_VioSSLFd *
new_VioSSLFd(const char *key_file, const char *cert_file,
             const char *ca_file, const char *ca_path,
             const char *cipher, my_bool is_client,
             enum enum_ssl_init_error *error,
             const char *crl_file, const char *crl_path)
{
  DH *dh;
  struct st_VioSSLFd *ssl_fd;
  long ssl_ctx_options= SSL_OP_NO_SSLv2 | SSL_OP_NO_SSLv3;
  DBUG_ENTER("new_VioSSLFd");
  DBUG_PRINT("enter",
             ("key_file: '%s'  cert_file: '%s'  ca_file: '%s'  ca_path: '%s'  "
              "cipher: '%s' crl_file: '%s' crl_path: '%s' ",
              key_file ? key_file : "NULL",
              cert_file ? cert_file : "NULL",
              ca_file ? ca_file : "NULL",
              ca_path ? ca_path : "NULL",
              cipher ? cipher : "NULL",
              crl_file ? crl_file : "NULL",
              crl_path ? crl_path : "NULL"));


  if (!(ssl_fd= ((struct st_VioSSLFd*)
                 my_malloc(sizeof(struct st_VioSSLFd),MYF(0)))))
    DBUG_RETURN(0);
  ssl_fd->owned = TRUE;

  if (!(ssl_fd->ssl_context= SSL_CTX_new(is_client ?
                                         SSLv23_client_method() :
                                         SSLv23_server_method())))
  {
    *error= SSL_INITERR_MEMFAIL;
    DBUG_PRINT("error", ("%s", sslGetErrString(*error)));
    report_errors();
    my_free(ssl_fd);
    DBUG_RETURN(0);
  }

  SSL_CTX_set_options(ssl_fd->ssl_context, ssl_ctx_options);
  if (!is_client)
  {
    SSL_CTX_set_session_cache_mode(ssl_fd->ssl_context, SSL_SESS_CACHE_OFF);
  }

  /*
    Set the ciphers that can be used
    NOTE: SSL_CTX_set_cipher_list will return 0 if
    none of the provided ciphers could be selected
  */
  if (cipher &&
      SSL_CTX_set_cipher_list(ssl_fd->ssl_context, cipher) == 0)
  {
    *error= SSL_INITERR_CIPHERS;
    DBUG_PRINT("error", ("%s", sslGetErrString(*error)));
    report_errors();
    free_vio_ssl_fd(ssl_fd);
    DBUG_RETURN(0);
  }

  /* There should be no pending errors at this point */
  DBUG_ASSERT(ERR_peek_error() == 0);

  /* Load certs from the trusted ca */
  if (SSL_CTX_load_verify_locations(ssl_fd->ssl_context, ca_file, ca_path) <= 0)
  {
    DBUG_PRINT("warning", ("SSL_CTX_load_verify_locations failed"));
    if (ca_file || ca_path)
    {
      /* fail only if ca file or ca path were supplied and looking into 
         them fails. */
      *error= SSL_INITERR_BAD_PATHS;
      DBUG_PRINT("error", ("SSL_CTX_load_verify_locations failed : %s", 
                 sslGetErrString(*error)));
      report_errors();
      free_vio_ssl_fd(ssl_fd);
      DBUG_RETURN(0);
    }

    /* otherwise go use the defaults */
    if (SSL_CTX_set_default_verify_paths(ssl_fd->ssl_context) == 0)
    {
      *error= SSL_INITERR_BAD_PATHS;
      DBUG_PRINT("error", ("%s", sslGetErrString(*error)));
      report_errors();
      free_vio_ssl_fd(ssl_fd);
      DBUG_RETURN(0);
    }
  }

  /* SSL_CTX_load_verify_locations sometimes leaves the error stack
     not clear.  Let's clear it. */
  ERR_clear_error();

  if (crl_file || crl_path)
  {
#ifdef HAVE_YASSL
    DBUG_PRINT("warning", ("yaSSL doesn't support CRL"));
    DBUG_ASSERT(0);
#else
    X509_STORE *store= SSL_CTX_get_cert_store(ssl_fd->ssl_context);
    /* Load crls from the trusted ca */
    if (X509_STORE_load_locations(store, crl_file, crl_path) == 0 ||
        X509_STORE_set_flags(store,
                             X509_V_FLAG_CRL_CHECK | 
                             X509_V_FLAG_CRL_CHECK_ALL) == 0)
    {
      DBUG_PRINT("warning", ("X509_STORE_load_locations for CRL failed"));
      *error= SSL_INITERR_BAD_PATHS;
      DBUG_PRINT("error", ("%s", sslGetErrString(*error)));
      report_errors();
      free_vio_ssl_fd(ssl_fd);
      DBUG_RETURN(0);
    }
#endif
  }

  if (vio_set_cert_stuff(ssl_fd->ssl_context, cert_file, key_file, error))
  {
    DBUG_PRINT("error", ("vio_set_cert_stuff failed"));
    report_errors();
    free_vio_ssl_fd(ssl_fd);
    DBUG_RETURN(0);
  }

  /* DH stuff */
  dh= get_dh2048();
  if (SSL_CTX_set_tmp_dh(ssl_fd->ssl_context, dh) == 0)
  {
    *error= SSL_INITERR_DHFAIL;
    DBUG_PRINT("error", ("%s", sslGetErrString(*error)));
    report_errors();
    DH_free(dh);
    SSL_CTX_free(ssl_fd->ssl_context);
    my_free(ssl_fd);
    DBUG_RETURN(0);
  }
  DH_free(dh);

#ifndef OPENSSL_NO_ECDH
  {
    /* Elliptic Curve */
    EC_KEY *ecdh = EC_KEY_new_by_curve_name(NID_X9_62_prime256v1);
    if (ecdh) {
      SSL_CTX_set_tmp_ecdh(ssl_fd->ssl_context, ecdh);
      EC_KEY_free(ecdh);
    }
  }
#endif

  DBUG_PRINT("exit", ("OK 1"));

  DBUG_RETURN(ssl_fd);
}


/************************ VioSSLConnectorFd **********************************/
struct st_VioSSLFd *
new_VioSSLConnectorFd(const char *key_file, const char *cert_file,
                      const char *ca_file, const char *ca_path,
                      const char *cipher, enum enum_ssl_init_error* error,
                      const char *crl_file, const char *crl_path)
{
  struct st_VioSSLFd *ssl_fd;
  int verify= SSL_VERIFY_PEER;

  /*
    Turn off verification of servers certificate if both
    ca_file and ca_path is set to NULL
  */
  if (ca_file == 0 && ca_path == 0)
    verify= SSL_VERIFY_NONE;

  if (!(ssl_fd= new_VioSSLFd(key_file, cert_file, ca_file,
                             ca_path, cipher, TRUE, error,
                             crl_file, crl_path)))
  {
    return 0;
  }

  /* Init the VioSSLFd as a "connector" ie. the client side */

  SSL_CTX_set_verify(ssl_fd->ssl_context, verify, NULL);

  return ssl_fd;
}


/************************ VioSSLConnectorFdFromContext ************************/
struct st_VioSSLFd *
new_VioSSLConnectorFdFromContext(SSL_CTX* context,
                                 enum enum_ssl_init_error *error)
{
  DBUG_ENTER("new_VioSSLConnectorFdFromContext");
  struct st_VioSSLFd *ssl_fd;
  if (!(ssl_fd= ((struct st_VioSSLFd*)
                 my_malloc(sizeof(struct st_VioSSLFd),MYF(0))))) {
    *error = SSL_INITERR_MEMFAIL;
     DBUG_RETURN(NULL);
  }
  ssl_fd->ssl_context = context;
  ssl_fd->owned = FALSE;

  DBUG_RETURN(ssl_fd);
}


/************************ VioSSLAcceptorFd **********************************/
struct st_VioSSLFd *
new_VioSSLAcceptorFd(const char *key_file, const char *cert_file,
		     const char *ca_file, const char *ca_path,
		     const char *cipher, enum enum_ssl_init_error* error,
                     const char *crl_file, const char *crl_path)
{
  struct st_VioSSLFd *ssl_fd;
  int verify= SSL_VERIFY_PEER | SSL_VERIFY_CLIENT_ONCE;
  if (!(ssl_fd= new_VioSSLFd(key_file, cert_file, ca_file,
                             ca_path, cipher, FALSE, error,
                             crl_file, crl_path)))
  {
    return 0;
  }
  /* Init the the VioSSLFd as a "acceptor" ie. the server side */

  /* Set max number of cached sessions, returns the previous size */
  SSL_CTX_sess_set_cache_size(ssl_fd->ssl_context, 128);

  SSL_CTX_set_verify(ssl_fd->ssl_context, verify, NULL);

  /*
    Set session_id - an identifier for this server session
    Use the ssl_fd pointer
   */
  SSL_CTX_set_session_id_context(ssl_fd->ssl_context,
				 (const unsigned char *)ssl_fd,
				 sizeof(ssl_fd));

  return ssl_fd;
}

void free_vio_ssl_fd(struct st_VioSSLFd *fd)
{
  if (fd->owned) {
    SSL_CTX_free(fd->ssl_context);
  }
  my_free(fd);
}
#endif /* HAVE_OPENSSL */
