(**
 * Copyright (c) 2014, Facebook, Inc.
 * All rights reserved.
 *
 * This source code is licensed under the BSD-style license found in the
 * LICENSE file in the "hack" directory of this source tree. An additional grant
 * of patent rights can be found in the PATENTS file in the same directory.
 *
 *)

open Utils
open Typing_defs

module Env = Typing_env


(* Module checking if a type is generic, I like to use an exception for this sort
 * of things, the code is more readable (subjective :-), and the exception never
 * escapes anyway.
*)
module IsGeneric: sig

  (* Give back the name and position of a generic if found *)
  val ty: ty -> string option
end = struct

  exception Found of string

  let rec ty (_, x) = ty_ x
  and ty_ = function
    | Tgeneric ("this", ty) -> ty_opt ty
    | Tgeneric (x, _) -> raise (Found x)
    | Tanon _
    | Tany | Tmixed | Tprim _ -> ()
    | Tarray (_, ty1, ty2) ->
        ty_opt ty1; ty_opt ty2
    | Tvar _ -> () (* Expansion got rid of Tvars ... *)
    | Toption x -> ty x
    | Tfun fty ->
        List.iter ty (List.map snd fty.ft_params);
        ty fty.ft_ret
    | Tabstract (_, tyl, x) ->
        List.iter ty tyl; ty_opt x
    | Tapply (_, tyl)
    | Ttuple tyl -> List.iter ty tyl
    | Tunresolved tyl -> List.iter ty tyl
    | Tobject -> ()
    | Tshape fdm ->
        SMap.iter (fun _ v -> ty v) fdm

  and ty_opt = function None -> () | Some x -> ty x

  let ty x = try ty x; None with Found x -> Some x

end

let rename env old_name new_name ty_to_rename = 
  let rec ty env (r, t) = (match t with
    | Tgeneric (x, ty) -> 
        let name = if x = old_name then new_name else x in
        let env, ty = ty_opt env ty in
        env, (r, Tgeneric (name, ty))
    | Tanon _
    | Tany | Tmixed | Tprim _-> env, (r, t)
    | Tarray (local, ty1, ty2) ->
        let env, ty1 = ty_opt env ty1 in
        let env, ty2 = ty_opt env ty2 in
        env, (r, Tarray (local, ty1, ty2))
    | Tvar n -> 
        let env, t = Env.get_type env n in
        let n' = Env.fresh() in
        let env = Env.rename env n n' in
        let env, t = ty env t in
        let env = Env.add env n' t in
        env, (r, Tvar n')
    | Toption x ->
        let env, x = ty env x in
        env, (r, Toption x)
    | Tfun fty ->
        let env, params = List.fold_right (fun (s_opt, t) (env, params) ->
          let env, t= ty env t in
          env, (s_opt, t)::params
        ) fty.ft_params (env, []) in
        let env, ret = ty env fty.ft_ret in
        env, (r, Tfun { fty with
          ft_params = params;
          ft_ret = ret;
        })
    | Tabstract (id, l, x) -> 
        let env, tyl = tyl env l in
        let env, x = ty_opt env x in
        env, (r, Tabstract (id, l, x))
    | Tapply (id, l) -> 
        let env, l = tyl env l in
        env, (r, Tapply(id, l))
    | Ttuple l ->
        let env, l = tyl env l in
        env, (r, Ttuple l)
    | Tunresolved l ->
        let env, l = tyl env l in
        env, (r, Tunresolved l)
    | Tobject -> env, (r, Tobject)
    | Tshape fdm -> 
        let env, fdm = SMap.fold (fun k v (env, fdm) ->
          let env, v = ty env v in
          env, SMap.add k v fdm
        ) fdm (env, SMap.empty) in
        env, (r, Tshape fdm ))

  and ty_opt env = function 
    | None -> env, None 
    | Some x -> 
        let env, x = ty env x in
        env, Some x

  and tyl env l = List.fold_right (fun t (env, l) ->
    let env, t = ty env t in
    env, t::l
  ) l (env, [])

  in ty env ty_to_rename

(* Function making sure that a type can be generalized, in our case it just
 * means the type should be monomorphic
*)
let no_generic p local_var_id env =
  let env, ty = Env.get_local env local_var_id in
  let ty = Typing_expand.fully_expand env ty in
  match IsGeneric.ty ty with
  | None -> env
  | Some x -> 
      Errors.generic_static p x;
      env

