(**
 * Copyright (c) 2014, Facebook, Inc.
 * All rights reserved.
 *
 * This source code is licensed under the BSD-style license found in the
 * LICENSE file in the "hack" directory of this source tree. An additional grant
 * of patent rights can be found in the PATENTS file in the same directory.
 *
 *)


open Utils
open Typing_defs
open Nast

module Dep = Typing_deps.Dep

(* The following classes are used to make sure we make no typing
 * mistake when interacting with the database. The database knows
 * how to associate a string to a string. We need to deserialize
 * the string and make sure the type is correct. By using these
 * modules, the places where there could be a typing mistake are
 * very well isolated.
*)

(* Module used to represent serialized classes *)
module Class = struct
  type t = class_type
  let prefix = Prefix.make()
end

(* a function type *)
module Fun = struct
  type t = fun_type
  let prefix = Prefix.make()
end

module Typedef = struct

  type visibility =
    | Public
    | Private

  type tdef = visibility * Typing_defs.tparam list * ty option * ty * Pos.t

  type tdef_or_error =
    | Error
    | Ok of tdef

  type t = tdef_or_error
  let prefix = Prefix.make()
end

module GConst = struct
  type t = ty
  let prefix = Prefix.make()
end

module Funs = SharedMem.WithCache(Fun)
module Classes = SharedMem.WithCache(Class)
module Typedefs = SharedMem.WithCache(Typedef)
module GConsts = SharedMem.WithCache(GConst)

type funs    = Funs.t
type classes = Classes.t

type fake_members = {
    last_call : Pos.t option;
    invalid   : SSet.t;
    valid     : SSet.t;
  }

type local = ty list * ty
type local_env = fake_members * local IMap.t

type env = {
    pos     : Pos.t      ;
    tenv    : ty  IMap.t ;
    subst   : int IMap.t ;
    lenv    : local_env ;
    genv    : genv       ;
    todo    : tfun list  ;
    pclasses : SSet.t;
    pfuns    : SSet.t;
  }

and genv = {
    mode    : Ast.mode;
    return  : ty         ;
    parent  : ty         ;
    self_id : string     ;
    self    : ty         ;
    static  : bool       ;
    has_yield : bool     ;
    allow_null_as_void : bool;
    f_type  : Ast.fun_type;
    anons   : anon IMap.t;
    droot   : Typing_deps.Dep.variant option  ;
    file    : string;
  }

(* An anonymous function
 * the environment + the fun parameters + the captured identifiers
*)
and anon = env -> fun_params -> env * ty
and tfun = env -> env

let fresh () =
  Ident.tmp()

let fresh_type () =
  Reason.none, Tvar (Ident.tmp())

let add_subst env x x' =
  if x <> x'
  then { env with subst = IMap.add x x' env.subst }
  else env

let rec get_var env x =
  let x' = IMap.get x env.subst in
  (match x' with
  | None -> env, x
  | Some x' ->
      let env, x' = get_var env x' in
      let env = add_subst env x x' in
      env, x'
  )

let rename env x x' =
  let env, x = get_var env x in
  let env, x' = get_var env x' in
  let env = add_subst env x x' in
  env

let add env x ty =
  let env, x = get_var env x in
  { env with tenv = IMap.add x ty env.tenv }

let get_type env x =
  let env, x = get_var env x in
  let ty = IMap.get x env.tenv in
  match ty with
  | None -> env, (Reason.none, Tany)
  | Some ty -> env, ty


let get_type_unsafe env x =
  let ty = IMap.get x env.tenv in
  match ty with
  | None ->
      env, (Reason.none, Tany)
  | Some ty -> env, ty

let expand_type env x =
  match x with
  | _, Tvar x -> get_type env x
  | x -> env, x

let has_type env x =
  let env, x = get_var env x in
  IMap.mem x env.tenv

let make_ft env p params ret_ty =
  let arity = List.length params in
  {
   ft_pos = p;
   ft_unsafe    = false;
   ft_abstract  = false;
   ft_arity_min = arity;
   ft_arity_max = arity;
   ft_tparams   = [];
   ft_params    = params;
   ft_ret       = ret_ty;
  }

let rec debug stack env (r, ty) =
  let o = print_string in
  (match r with Reason.Rlost_info _ -> o "~lost" | _ -> ());
  match ty with
  | Tunresolved tyl -> o "intersect("; debugl stack env tyl; o ")"
  | Ttuple _ -> o "tuple"
  | Tarray (_, None, None) -> o "array"
  | Tarray (_, Some x, None) -> o "array<"; debug stack env x; o ">"
  | Tarray (_, Some x, Some y) -> o "array<"; debug stack env x; o ", ";
      debug stack env y; o ">"
  | Tarray _ -> assert false
  | Tmixed -> o "mixed"
  | Tabstract (x, argl, _)
  | Tapply (x, argl) ->
      Printf.printf "App %s" (snd x);
      o "<"; List.iter (fun x -> debug stack env x; o ", ") argl;
      o ">"
  | Tany -> o "X"
  | Tanon _ -> o "anonymous"
  | Tfun ft ->
      o "fun ";
      List.iter (fun (_, x) -> debug stack env x; o ", ") ft.ft_params;
      o " -> ";
      debug stack env ft.ft_ret
  | Toption ty -> o "option("; debug stack env ty; o ")"
  | Tprim p ->
      (match p with
      | Tvoid -> o "Tvoid"
      | Tint -> o "Tint"
      | Tbool -> o "Tbool"
      | Tfloat -> o "Tfloat"
      | Tstring -> o "Tstring"
      | Tnum -> o "Tnum"
      | Tresource -> o "Tresource"
      )
  | Tgeneric (s, x) ->
      o s;
      (match x with
      | None -> ()
      | Some x -> o "<"; debug stack env x; o ">"
      )
  | Tvar x ->
      let env, x = get_var env x in
      if ISet.mem x stack
      then o (Ident.debug x)
      else
        let stack = ISet.add x stack in
        let _, y = get_var env x in
        o "["; o (string_of_int y); o "]";
        (match get_type env x with
        | _, (_, Tany) -> o (Ident.debug x)
        | _, ty -> debug stack env ty)
  | Tobject -> o "object"
  | Tshape fdm -> o "shape(";
      SMap.iter begin fun k v ->
        o k; o " => "; debug stack env v
      end fdm;
      o ")"

and debugl stack env x =
  let o = print_string in
  match x with
  | [] -> ()
  | [x] -> debug stack env x
  | x :: rl -> debug stack env x; o ", "; debugl stack env rl

let debug env ty = debug ISet.empty env ty; print_newline()

let empty_fake_members = {
    last_call = None;
    invalid   = SSet.empty;
    valid     = SSet.empty;
}

let empty_local = empty_fake_members, IMap.empty

let empty file = {
  pos     = Pos.none;
  tenv    = IMap.empty;
  subst   = IMap.empty;
  lenv    = empty_local;
  todo    = [];
  pclasses = SSet.empty;
  pfuns = SSet.empty;
  genv    = {
    mode    = Ast.Mstrict;
    return  = fresh_type();
    self_id = "";
    self    = Reason.none, Tany;
    static  = false;
    parent  = Reason.none, Tany;
    has_yield = false;
    allow_null_as_void = false;
    f_type  = Ast.FSync;
    anons   = IMap.empty;
    droot   = None;
    file    = file;
  }
}

let add_class x y =
  Classes.add x y

let add_typedef x y =
  Typedefs.add x (Typedef.Ok y)

let is_typedef env x =
  match Typedefs.get x with
  | None -> false
  | Some _ -> true

let add_typedef_error x =
  Typedefs.add x Typedef.Error

(* Adds a new function (global) *)
let add_fun x ft =
  Funs.add x ft

(* Manipulating the environment *)
let add_wclass env x =
  let dep = Dep.Class x in
  Typing_deps.add_idep env.genv.droot dep;
  let env = { env with pclasses = SSet.add x env.pclasses } in
  env

(* When we want to type something with a fresh typing environment *)
let fresh_tenv env f =
  let genv = env.genv in
  let genv = { genv with has_yield = false; allow_null_as_void = false } in
  f { env with todo = []; tenv = IMap.empty; genv = genv }

let get_class env x =
  let env = add_wclass env x in
  env, Classes.get x

let get_typedef env x =
  let env = add_wclass env x in
  env, Typedefs.get x

let class_exists x =
  Classes.mem x

let add_extends_dependency env x =
  let dep = Dep.Class x in
  Typing_deps.add_idep env.genv.droot dep;
  Typing_deps.add_idep env.genv.droot (Dep.Extends x);
  ()

let get_class_dep env x =
  let env = add_wclass env x in
  add_extends_dependency env x;
  let class_ = Classes.get x in
  match class_ with
  | None -> env, None
  | Some class_ ->
      env, Some class_

(* Used to access class constants. *)
let get_const env class_ mid =
  let env = add_wclass env class_.tc_name in
  let dep = Dep.Const (class_.tc_name, mid) in
  Typing_deps.add_idep env.genv.droot dep;
  env, SMap.get mid class_.tc_consts

(* Used to access "global constants". That is constants that were
 * introduced with "const X = ...;" at topelevel, or "define('X', ...);"
 *)
let get_gconst env cst_name =
  Typing_deps.add_idep env.genv.droot (Dep.GConst cst_name);
  GConsts.get cst_name

let get_static_member is_method env class_ mid =
  let env = add_wclass env class_.tc_name in
  let dep = if is_method then Dep.SMethod (class_.tc_name, mid)
  else Dep.SCVar (class_.tc_name, mid) in
  Typing_deps.add_idep env.genv.droot dep;
  env, if is_method then SMap.get mid class_.tc_smethods
  else SMap.get mid class_.tc_scvars

let suggest_member members mid =
  let members = SMap.fold begin fun x ce acc ->
    let pos = Reason.to_pos (fst ce.ce_type) in
    SMap.add (String.lowercase x) (pos, x) acc
  end members SMap.empty
  in
  SMap.get mid members

let suggest_static_member is_method class_ mid =
  let mid = String.lowercase mid in
  let members = if is_method then class_.tc_smethods else class_.tc_scvars in
  suggest_member members mid

let method_exists class_ mid =
  SMap.mem mid class_.tc_methods

let get_member is_method env class_ mid =
  let env = add_wclass env class_.tc_name in
  let dep = if is_method then Dep.Method (class_.tc_name, mid)
  else Dep.CVar (class_.tc_name, mid) in
  Typing_deps.add_idep env.genv.droot dep;
  env, if is_method then (SMap.get mid class_.tc_methods)
  else SMap.get mid class_.tc_cvars

let suggest_member is_method class_ mid =
  let mid = String.lowercase mid in
  let members = if is_method then class_.tc_methods else class_.tc_cvars in
  suggest_member members mid

let get_construct env class_ =
  let env = add_wclass env class_.tc_name in
  let dep = Dep.Cstr (class_.tc_name) in
  Typing_deps.add_idep env.genv.droot dep;
  env, class_.tc_construct

let get_todo env =
  env.todo

let get_return env =
  env.genv.return

let set_return env x =
  let genv = env.genv in
  let genv = { genv with return = x } in
  { env with genv = genv }

let with_return env f =
  let ret = get_return env in
  let env = f env in
  set_return env ret

let has_yield env = env.genv.has_yield
let allow_null_as_void env = env.genv.allow_null_as_void
let is_static env = env.genv.static
let get_self env = env.genv.self
let get_self_id env = env.genv.self_id
let get_parent env = env.genv.parent

let get_fn_type env = env.genv.f_type

let get_fun env x =
  let dep = Dep.Fun x in
  Typing_deps.add_idep env.genv.droot dep;
  let env = { env with pfuns = SSet.add x env.pfuns } in
  env, Funs.get x

let set_has_yield ?(value=true) env =
  {env with genv = {env.genv with has_yield = value }}

let set_allow_null_as_void ?(allow=true) env =
  let genv = env.genv in
  let genv = { genv with allow_null_as_void = allow } in
  { env with genv = genv }

let set_fn_type env fn_type =
  let genv = env.genv in
  let genv = { genv with f_type = fn_type } in
  { env with genv = genv }

let add_todo env x =
  { env with todo = x :: env.todo }

let add_anonymous env x =
  let genv = env.genv in
  let anon_id = Ident.tmp() in
  let genv = { genv with anons = IMap.add anon_id x genv.anons } in
  { env with genv = genv }, anon_id

let get_anonymous env x =
  IMap.get x env.genv.anons

let set_self_id env x =
  let genv = env.genv in
  let genv = { genv with self_id = x } in
  { env with genv = genv }

let set_self env x =
  let genv = env.genv in
  let genv = { genv with self = x } in
  { env with genv = genv }

let set_parent env x =
  let genv = env.genv in
  let genv = { genv with parent = x } in
  { env with genv = genv }

let set_static env =
  let genv = env.genv in
  let genv = { genv with static = true } in
  { env with genv = genv }

let set_mode env mode =
  let genv = env.genv in
  let genv = { genv with mode = mode } in
  { env with genv = genv }

let set_root env root =
  let genv = env.genv in
  let genv = { genv with droot = Some root } in
  { env with genv = genv }

let get_mode env = env.genv.mode

let is_strict env = get_mode env = Ast.Mstrict
let is_decl env = get_mode env = Ast.Mdecl

(*
let debug_env env =
  Classes.iter begin fun cid class_ ->
    Printf.printf "Type of class %s:" cid;
    Printf.printf "{ ";
    SMap.iter begin fun m _ ->
      Printf.printf "%s " m;
    end class_.tc_methods;
    Printf.printf "}\n"
  end env.genv.classes
*)
(*****************************************************************************)
(* This is used when we want member variables to be treated like locals
 * We want to handle the following:
 * if($this->x) {
 *   ... $this->x ...
 * }
 * The trick consists in relplacing $this->x with a "fake" local. So that
 * all the logic that normally applies to locals is applied in cases like
 * this. Hence the name: FakeMembers.
 * All the fake members are thrown away at the first call.
 * We keep the invalidated fake members for better error messages.
 *)
(*****************************************************************************)

let get_last_call env =
  match (fst env.lenv).last_call with
  | None -> assert false
  | Some pos -> pos

let rec lost_info fake_name stack env ty =
  let info r = Reason.Rlost_info (fake_name, r, get_last_call env) in
  match ty with
  | _, Tvar v when ISet.mem v stack -> env, ty
  | _, Tvar v ->
      let stack = ISet.add v stack in
      let env, v' = get_var env v in
      (match IMap.get v' env.tenv with
      | None ->
          env, ty
      | Some ty ->
          let env, ty = lost_info fake_name stack env ty in
          let env = add env v ty in
          env, ty
      )
  | r, Tunresolved tyl ->
      let env, tyl = lfold (lost_info fake_name stack) env tyl in
      env, (info r, Tunresolved tyl)
  | r, ty ->
      env, (info r, ty)

let forget_members env call_pos =
  let fake_members, locals = env.lenv in
  let old_invalid = fake_members.invalid in
  let new_invalid = fake_members.valid in
  let new_invalid = SSet.union new_invalid old_invalid in
  let fake_members = {
    last_call = Some call_pos;
    invalid = new_invalid;
    valid = SSet.empty;
  } in
  { env with lenv = fake_members, locals }

module FakeMembers = struct

  let make_id obj_name member_name =
    let obj_name =
      match obj_name with
      | _, This -> this
      | _, Lvar (_, x) -> x
      | _ -> assert false
    in
    string_of_int obj_name^"->"^member_name

  let make_static_id cid member_name =
    let class_name =
      match cid with
      | CIparent -> "parent"
      | CIself -> "self"
      | CIstatic -> "static"
      | CI (_, x) -> x
    in
    class_name^"::"^member_name

  let get env obj member_name =
    match obj with
    | _, This
    | _, Lvar _ ->
        let id = make_id obj member_name in
        if SSet.mem id (fst env.lenv).valid
        then Some (Hashtbl.hash id)
        else None
    | _ -> None

  let is_invalid env obj member_name =
    match obj with
    | _, This
    | _, Lvar _ ->
        SSet.mem (make_id obj member_name) (fst env.lenv).invalid
    | _ -> false

  let get_static env cid member_name =
    let name = make_static_id cid member_name in
    if SSet.mem name (fst env.lenv).valid
    then Some (Hashtbl.hash name)
    else None

  let is_static_invalid env cid member_name =
    SSet.mem (make_static_id cid member_name) (fst env.lenv).invalid

  let add_member env fake_id =
    let fake_members, locals = env.lenv in
    let valid = SSet.add fake_id fake_members.valid in
    let fake_members = { fake_members with valid = valid } in
    { env with lenv = fake_members, locals }

  let make p env obj_name member_name =
    let my_fake_local_id = make_id obj_name member_name in
    let env = add_member env my_fake_local_id in
    env, Hashtbl.hash my_fake_local_id

 let make_static p env class_name member_name =
   let my_fake_local_id = make_static_id class_name member_name in
   let env = add_member env my_fake_local_id in
    env, Hashtbl.hash my_fake_local_id

end


(*****************************************************************************)
(* Locals *)
(*****************************************************************************)

(* We want to "take a picture" of the current type
 * that is, the current type shouldn't be affected by a
 * future unification.
 *)

let rec unbind seen env ty =
  let env, ty = expand_type env ty in
  if List.mem ty seen
  then env, ty
  else
    let seen = ty :: seen in
    match ty with
    | r, Tunresolved tyl ->
        let env, tyl = lfold (unbind seen) env tyl in
        env, (r, Tunresolved tyl)
    | ty -> env, ty

let unbind = unbind []

(* We maintain 2 states for a local, all the types that the
 * local ever had (cf integrate in typing.ml), and the type
 * that the local currently has.
 *)
let set_local env x new_type =
  let fake_members, locals = env.lenv in
  let env, new_type = unbind env new_type in
  let all_types =
    match IMap.get x locals with
    | None -> []
    | Some (x, _) -> x
  in
  let all_types =
    if List.exists (fun x -> x = new_type) all_types
    then all_types
    else new_type :: all_types
  in
  let local = all_types, new_type in
  let locals = IMap.add x local locals in
  let env = { env with lenv = fake_members, locals } in
  env

let get_local env x =
  let lcl = IMap.get x (snd env.lenv) in
  match lcl with
  | None -> env, (Reason.Rnone, Tany)
  | Some (_, x) -> env, x

(*****************************************************************************)
(* This function is called when we are about to type-check a block that will
 * later be fully_integrated (cf Typing.fully_integrage).
 * Integration is about keeping track of all the types that a local had in
 * its lifetime. It's necessary to correctly type-check catch blocks.
 * After we type-check a block, we want to take all the types that the local
 * had in this block, and add it to the list of possible types.
 *
 * However, we are not interested in the types that the local had *before*
 * we started typing the block.
 *
 * A concrete example:
 *
 * $x = null;
 *
 * $x = 'hello'; // the type of $x is string
 *
 * while (...) {
 *   $x = 0;
 * }
 *
 * The type of $x is string or int, NOT string or int or ?_.
 * We don't really care about the fact that $x could be null before the
 * block.
 *
 * This is what freeze_local does, just before we start type-checking the
 * while loop, we "freeze" the type of locals to the current environment.
 *)
(*****************************************************************************)

let freeze_local_env env =
  let (members, locals) = env.lenv in
  let locals = IMap.map (fun (_, type_) -> [type_], type_) locals in
  let lenv = members, locals in
  { env with lenv = lenv }

(*****************************************************************************)
(* Sets up/cleans up the environment when typing an anonymous function. *)
(*****************************************************************************)

let anon anon_lenv env f =
  (* Setting up the environment. *)
  let old_lenv = env.lenv in
  let old_return = get_return env in
  let old_has_yield = env.genv.has_yield in
  let outer_f_type = get_fn_type env in
  let env = set_has_yield ~value:false env in
  let env = { env with lenv = anon_lenv } in
  (* Typing *)
  let env, result = f env in
  (* Cleaning up the environment. *)
  let env = { env with lenv = old_lenv } in
  let env = set_has_yield ~value:old_has_yield env in
  let env = set_return env old_return in
  let env = set_fn_type env outer_f_type in
  env, result
