/* 
 * Copyright (C) 2001-2012 Michael Fuchs
 *
 * This file is part of herold.
 * 
 * herold is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * herold is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with herold.  If not, see <http://www.gnu.org/licenses/>.  
 */
package org.dbdoclet.trafo.html.docbook.editor;

import java.util.HashMap;

import org.dbdoclet.option.OptionException;
import org.dbdoclet.service.StringServices;
import org.dbdoclet.trafo.html.docbook.DocBookTransformer;
import org.dbdoclet.trafo.tag.docbook.Article;
import org.dbdoclet.trafo.tag.docbook.Book;
import org.dbdoclet.trafo.tag.docbook.BridgeHead;
import org.dbdoclet.trafo.tag.docbook.Chapter;
import org.dbdoclet.trafo.tag.docbook.DocBookDocument;
import org.dbdoclet.trafo.tag.docbook.DocBookElement;
import org.dbdoclet.trafo.tag.docbook.DocBookFragment;
import org.dbdoclet.trafo.tag.docbook.DocBookTagFactory;
import org.dbdoclet.trafo.tag.docbook.Para;
import org.dbdoclet.trafo.tag.docbook.Part;
import org.dbdoclet.trafo.tag.docbook.PartIntro;
import org.dbdoclet.trafo.tag.docbook.RefEntry;
import org.dbdoclet.trafo.tag.docbook.RefName;
import org.dbdoclet.trafo.tag.docbook.RefNameDiv;
import org.dbdoclet.trafo.tag.docbook.RefPurpose;
import org.dbdoclet.trafo.tag.docbook.RefSect1;
import org.dbdoclet.trafo.tag.docbook.RefSect2;
import org.dbdoclet.trafo.tag.docbook.RefSect3;
import org.dbdoclet.trafo.tag.docbook.RefSect4;
import org.dbdoclet.trafo.tag.docbook.RefSect5;
import org.dbdoclet.trafo.tag.docbook.Reference;
import org.dbdoclet.trafo.tag.docbook.Sect1;
import org.dbdoclet.trafo.tag.docbook.Sect2;
import org.dbdoclet.trafo.tag.docbook.Sect3;
import org.dbdoclet.trafo.tag.docbook.Sect4;
import org.dbdoclet.trafo.tag.docbook.Sect5;
import org.dbdoclet.trafo.tag.docbook.Section;
import org.dbdoclet.trafo.tag.docbook.Title;
import org.dbdoclet.trafo.tag.html.HeaderElement;
import org.dbdoclet.trafo.tag.html.HtmlDocument;
import org.dbdoclet.trafo.tag.html.HtmlElement;
import org.dbdoclet.xiphias.dom.NodeImpl;
import org.w3c.dom.Element;
import org.w3c.dom.Node;

public class HeaderBaseEditor extends Editor {

	private static HashMap<String, HashMap<String, String>> validHtmlParentPathMap;

	// private static Class<?>[] bookMap = { Book.class, Chapter.class,
	// Sect1.class, Sect2.class, Sect3.class,
	// Sect4.class };

	// private static Class<?>[] articleMap = { Article.class, Sect1.class,
	// Sect2.class, Sect3.class,
	// Sect4.class, Sect5.class };

	private static Class<?>[] bookMap = { Book.class, Chapter.class,
			Section.class, Section.class, Section.class, Section.class };

	private static Class<?>[] articleMap = { Article.class, Section.class,
			Section.class, Section.class, Section.class, Section.class };

	private static Class<?>[] partIntroMap = { PartIntro.class, Section.class,
			Section.class, Section.class, Section.class, Section.class };

	private static Class<?>[] chapterMap = { Chapter.class, Section.class,
			Section.class, Section.class, Section.class, Section.class };

	private static Class<?>[] sect1Map = { Sect1.class, Sect2.class,
			Sect3.class, Sect4.class, Sect5.class, Sect5.class };

	private static Class<?>[] sect2Map = { Sect2.class, Sect3.class,
			Sect4.class, Sect5.class, Sect5.class, Sect5.class };

	private static Class<?>[] sect3Map = { Sect3.class, Sect4.class,
			Sect5.class, Sect5.class, Sect5.class, Sect5.class };

	private static Class<?>[] referenceMap = { Reference.class, RefEntry.class,
			RefSect1.class, RefSect2.class, RefSect3.class, RefSect4.class };

	private static Class<?>[] refSect1Map = { RefSect1.class, RefSect2.class,
			RefSect3.class, RefSect4.class, RefSect5.class, RefSect5.class };

	private static Class<?>[] refSect2Map = { RefSect2.class, RefSect3.class,
			RefSect4.class, RefSect5.class, RefSect5.class, RefSect5.class };

	private static Class<?>[] refSect3Map = { RefSect3.class, RefSect4.class,
			RefSect5.class, RefSect5.class, RefSect5.class, RefSect5.class };

	private static Class<?>[] saveMap = { Para.class, Para.class, Para.class,
			Para.class, Para.class, Para.class };

	static {

		validHtmlParentPathMap = new HashMap<String, HashMap<String, String>>();
		validHtmlParentPathMap.put(new HtmlDocument().getNodeName(),
				HtmlElement.getAttributeMap());
		validHtmlParentPathMap.put(org.dbdoclet.trafo.tag.html.Html.getTag(),
				HtmlElement.getAttributeMap());
		validHtmlParentPathMap.put(org.dbdoclet.trafo.tag.html.Body.getTag(),
				HtmlElement.getAttributeMap());
		validHtmlParentPathMap.put(org.dbdoclet.trafo.tag.html.Center.getTag(),
				HtmlElement.getAttributeMap());
		validHtmlParentPathMap.put(org.dbdoclet.trafo.tag.html.Div.getTag(),
				HtmlElement.getAttributeMap());
		validHtmlParentPathMap.put(org.dbdoclet.trafo.tag.html.A.getTag(),
				HtmlElement.getAttributeMap());
	}

	private Class<?>[] map;

	// private HeaderElement header;

	public DocBookElement createSectionChild(HeaderElement header,
			Element parent) throws OptionException {

		DocBookTagFactory dbfactory = getTagFactory();
		DocBookElement section = null;

		if (parent == null || parent instanceof Para
				|| isValidHeader(header) == false) {
			section = dbfactory.createBridgeHead();
		}

		if (parent instanceof Book || parent instanceof Part) {
			section = dbfactory.createChapter();
		}

		if (parent instanceof Chapter || parent instanceof PartIntro) {
			section = dbfactory.createSection();
		}

		if (parent instanceof Article) {
			section = dbfactory.createSection();
		}

		if (parent instanceof Section) {
			section = dbfactory.createSection();
		}

		if (parent instanceof Sect1) {
			section = dbfactory.createSect2();
		}

		if (parent instanceof Sect2) {
			section = dbfactory.createSect3();
		}

		if (parent instanceof Sect3) {
			section = dbfactory.createSect4();
		}

		if (parent instanceof Sect4) {
			section = dbfactory.createSect5();
		}

		if (parent instanceof Sect5) {
			section = dbfactory.createSimpleSect();
		}

		if (parent instanceof Reference) {
			section = dbfactory.createRefEntry();
		}

		if (parent instanceof RefEntry) {
			section = dbfactory.createRefSect1();
		}

		if (parent instanceof RefSect1) {
			section = dbfactory.createRefSect2();
		}

		if (parent instanceof RefSect2) {
			section = dbfactory.createRefSect3();
		}

		if (parent instanceof RefSect3) {
			section = dbfactory.createRefSect4();
		}

		if (parent instanceof RefSect4) {
			section = dbfactory.createRefSect5();
		}

		if (parent instanceof RefSect5) {
			section = dbfactory.createSimpleSect();
		}

		if (section != null) {
			copyCommonAttributes(header, section);
		}

		return section;
	}

	public DocBookElement createSectionPara(Element parent)
			throws OptionException {

		DocBookElement para;

		DocBookTagFactory dbfactory = getTagFactory();

		if ((parent != null) && parent instanceof Para) {

			para = (DocBookElement) parent;

		} else {

			para = dbfactory.createPara();

			if (((Para) para).isValidParent(getParent())) {
				getParent().appendChild(para);
			} else {
				para = getParent();
			}
		}

		BridgeHead bridgeHead = dbfactory.createBridgeHead();
		bridgeHead.appendChild(getHtmlElement().getTextContent());
		para.appendChild(bridgeHead);

		// Emphasis emph = dbfactory.createEmphasis();
		// emph.setRole(getOptions().getEmphasisBoldRole());
		// emph.setParentNode(para);
		// emph.setRemap(header.getNodeName());
		// emph.appendChild(getChild().getText());
		// para.appendChild(emph);

		return para;
	}

	@Override
	public EditorInstruction edit(EditorInstruction values)
			throws EditorException {

		setValues(super.edit(values));
		DocBookTagFactory dbfactory = values.getTagFactory();

		try {

			Element levelParent;
			DocBookElement sect;

			DocBookTransformer transformer = getTransformer();

			HtmlElement child = getHtmlElement();

			DocBookElement parent = getParent();

			if (parent != null) {
				setCurrent(parent);
			}

			traverse(true);

			map = null;

			Element root = getDocumentElement();

			if (isRootBook() || isRoot(Part.getTag())) {
				map = bookMap;
			}

			if (isRootArticle()) {
				map = articleMap;
			}

			if (isRoot(Chapter.getTag())) {
				map = chapterMap;
			}

			if (isRoot(PartIntro.getTag())) {
				map = partIntroMap;
			}

			if (isRoot(Sect1.getTag())) {
				map = sect1Map;
			}

			if (isRoot(Sect2.getTag())) {
				map = sect2Map;
			}

			if (isRoot(Sect3.getTag())) {
				map = sect3Map;
			}

			if (isRoot(Reference.getTag())) {
				map = referenceMap;
			}

			if (isRoot(RefSect1.getTag())) {
				map = refSect1Map;
			}

			if (isRoot(RefSect2.getTag())) {
				map = refSect2Map;
			}

			if (isRoot(RefSect3.getTag())) {
				map = refSect3Map;
			}

			if (map == null) {
				map = saveMap;
			}

			HeaderElement header = (HeaderElement) child;

			int level = header.getLevel();
			levelParent = findParentForLevel(header, level);

			if ((levelParent == null)
					&& ((getCodeContext() == DocBookTransformer.DocumentElementType.OVERVIEW) || (getCodeContext() == DocBookTransformer.DocumentElementType.BOOK))) {

				sect = dbfactory.createChapter();
				sect.setRemap(header.getNodeName());

				levelParent = root;

			} else {

				sect = createSectionChild(header, levelParent);
			}

			if (sect != null) {

				if (sect instanceof Para) {

					traverse(false);

					// Try to force a new paragraph after the header.
					if (parent instanceof Para
							&& (parent.getParentNode() != null)) {
						setParent((DocBookElement) parent.getParentNode());
					}

					setCurrent(getParent());

				} else {

					if (levelParent != null) {

						levelParent.appendChild(sect);

						if (sect instanceof RefEntry) {

							RefNameDiv refNameDiv = dbfactory
									.createRefNameDiv();
							RefName refName = dbfactory.createRefName("");
							RefPurpose refPurpose = dbfactory
									.createRefPurpose();

							refNameDiv.appendChild(refName);
							refNameDiv.appendChild(refPurpose);

							RefSect1 refSect1 = dbfactory.createRefSect1();

							sect.appendChild(refNameDiv);
							sect.appendChild(refSect1);
							sect = refSect1;
						}

						if (sect instanceof BridgeHead) {

							// setParent(sect);
							setCurrent(sect);

						} else {

							Title title = dbfactory.createTitle();
							sect.appendChild(title);
							setParent(sect);
							setCurrent(title);
						}
					}
				}

			} else {

				sect = getCurrent();
			}

			String id = child.getId();

			if ((id != null) && (id.length() > 0)) {

				sect.setId(transformer.createUniqueId(id));
			}

			if (root != null) {
				root.getClass().getName();
			}

			return finalizeValues();

		} catch (OptionException oops) {

			throw new EditorException(oops);
		} // end of try-catch
	}

	public DocBookElement findParentForLevel(HeaderElement header, int level) {

		DocBookElement parent = null;
		Class<?> parentClass = null;

		if ((level < 0) || (level > 6)) {
			level = 6;
		}

		if (level > map.length) {
			level = map.length - 1;
		}

		for (int i = (level - 1); i >= 0; i--) {

			parentClass = map[i];

			parent = getCurrent();

			if (parent != null) {

				String tag = header.getNodeName();
				String remap = parent.getRemap();

				int tagLevel = getLevelByTagName(tag);
				int parentLevel = getLevelByTagName(remap);

				while (tagLevel != -1 && parentLevel != -1
						&& parentLevel >= tagLevel) {

					DocBookElement ancestor = (DocBookElement) NodeImpl
							.findParent(parent, parentClass);

					if (ancestor == null) {
						break;
					}

					parent = ancestor;
					parentLevel = getLevelByTagName(parent.getRemap());
				}

				break;
			}
		}

		return parent;
	}

	private int getLevelByTagName(String tagName) {

		if (tagName == null) {
			return -1;
		}

		int index = tagName.indexOf(':');

		if (index != -1) {
			tagName = tagName.substring(0, index);
		}

		if (tagName.length() != 2) {
			return -1;
		}

		tagName = tagName.trim().toLowerCase();

		if (tagName.startsWith("h") == false) {
			return -1;
		}

		tagName = StringServices.cutPrefix(tagName, "h");

		int level = -1;

		try {
			level = Integer.parseInt(tagName);
		} catch (NumberFormatException oops) {
			return -1;
		}

		return level;
	}

	private Element getDocumentElement() {

		NodeImpl rootNode = getCurrent().getRoot();

		Element root = null;

		if (rootNode instanceof DocBookDocument) {

			DocBookDocument doc = (DocBookDocument) rootNode;
			root = doc.getDocumentElement();

		} else if (rootNode instanceof Element) {

			root = (Element) rootNode;

		} else {
			throw new IllegalStateException(
					"Root node must be of type DocBookDocument or Element."
							+ "Found root node of type "
							+ rootNode.getClass().getName() + "!");
		}

		return root;
	}

	public boolean isRoot(String tagName) {

		Node root = getDocumentElement();

		if (root instanceof DocBookFragment) {

			Node firstChild = ((DocBookFragment) root).getFirstElement();

			if (firstChild != null) {
				root = firstChild;
			}
		}

		if (root == null) {
			throw new IllegalStateException("The field root must not be null!");
		}

		if (tagName.equalsIgnoreCase(root.getNodeName())) {
			return true;
		} else {
			return false;
		}
	}

	public boolean isRootArticle() {

		Node root = getDocumentElement();

		if (root instanceof DocBookFragment) {
			root = ((DocBookFragment) root).getFirstElement();
		}

		if (root instanceof Article) {
			return true;
		} else {
			return false;
		}
	}

	public boolean isRootBook() {

		Element root = getDocumentElement();

		if (root instanceof DocBookFragment) {
			root = ((DocBookFragment) root).getFirstElement();
		}

		if (root instanceof Book
				|| (getCodeContext() == DocBookTransformer.DocumentElementType.BOOK)
				|| (getCodeContext() == DocBookTransformer.DocumentElementType.OVERVIEW)) {
			return true;
		} else {
			return false;
		}
	}

	private boolean isValidHeader(HeaderElement header) {

		boolean rc = header.validateParentPath(validHtmlParentPathMap);
		return rc;
	}
}
