/* 
 * Copyright (C) 2001-2012 Michael Fuchs
 *
 * This file is part of herold.
 * 
 * herold is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * herold is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with herold.  If not, see <http://www.gnu.org/licenses/>.  
 */
package org.dbdoclet.trafo;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.PrintWriter;
import java.io.StringWriter;

import javax.xml.transform.ErrorListener;
import javax.xml.transform.TransformerException;

import org.dbdoclet.service.StringServices;

/**
 * Die Klasse <code>TrafoResult</code> speichert das Protokoll einer
 * Transformation.
 * 
 * @author michael
 * 
 */
public class TrafoResult implements ErrorListener {

	private File file;
	private StringBuffer buffer;
	private byte[] data;
	private Throwable throwable;

	public TrafoResult() {
		super();
		buffer = new StringBuffer();
	}

	public TrafoResult(File file) {

		if (file == null) {

			throw new IllegalArgumentException(
					"The argument file must not be null!");
		}

		this.file = file;
		buffer = new StringBuffer();
	}

	public String getBuffer() {
		return buffer.toString();
	}

	public byte[] getData() {
		return data;
	}

	public File getFile() {
		return file;
	}

	public void append(String line) {
		buffer.append(line);
	}

	public void setData(byte[] data) {
		this.data = data;
	}

	public void setThrowable(Throwable throwable) {
		this.throwable = throwable;
	}

	public Throwable getThrowable() {
		return throwable;
	}

	/**
	 * Generiert eine Zusammenfassung des Protokolls
	 */
	@Override
	public String toString() {

		StringBuilder builder = new StringBuilder();

		builder.append('\n');
		builder.append(StringServices.createBox("TRANSFORMATION PROTOCOL"));

		if (buffer != null) {
			builder.append("=== STDOUT/STDERR ===\n");
			builder.append(buffer);
		}

		if (throwable != null) {

			if (throwable instanceof FileNotFoundException) {
				
				builder.append("[ERROR] File not found: " + throwable.getMessage());
				
			} else {
			
				builder.append("\n=== EXCEPTION ===\n");
				StringWriter strBuffer = new StringWriter();
				throwable.printStackTrace(new PrintWriter(strBuffer));
				builder.append(strBuffer.toString());
			}
		} 

		builder.append(">>TRANSFORMATION FINISHED.\n\n");
		
		return builder.toString();
	}

	public void warning(TransformerException exception)
			throws TransformerException {
		buffer.append(exception.getMessage());
		buffer.append('\n');
	}

	public void error(TransformerException exception)
			throws TransformerException {
		buffer.append("[error] " + exception.getMessage());
		buffer.append('\n');
	}

	public void fatalError(TransformerException exception)
			throws TransformerException {
		buffer.append("[fatal] " + exception.getMessage());
		buffer.append('\n');
	}
}
