package org.dbdoclet.trafo.script;

import java.io.File;
import java.util.ArrayList;
import java.util.TreeMap;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.dbdoclet.service.FileServices;
import org.dbdoclet.trafo.param.BooleanParam;
import org.dbdoclet.trafo.param.NumberParam;
import org.dbdoclet.trafo.param.Param;
import org.dbdoclet.trafo.param.TextParam;

public class Script {

	private File inputFile;
	private String inputFormat;
	private String outputFormat;
	private File outputFile;
	private TreeMap<String, TreeMap<String, Param<?>>> sectionMap;
	private TreeMap<String, Param<?>> currentParamMap;
	private File inputDirectory;
	private File outputDirectory;

	public Script() {
		sectionMap = new TreeMap<String, TreeMap<String, Param<?>>>();
	}

	public void addBoolParam(String name,
			boolean flag) {
		
		if (currentParamMap == null) {
			throw new IllegalStateException("Field currentParamMap must not be null!");
		}
		
		BooleanParam param = (BooleanParam) currentParamMap.get(name); 

		if (param == null) { 
			param = new BooleanParam(name, flag);	
			currentParamMap.put(name, param);
		} else {
			param.addValue(flag);
		}
	}

	public void addBoolParam(String name, String bool) {

		addBoolParam(name, Boolean.valueOf(bool));	
	}

	public void addNumberParam(String name, int number) {

		if (currentParamMap == null) {
			throw new IllegalStateException("Field currentParamMap must not be null!");
		}
		
		NumberParam param = (NumberParam) currentParamMap.get(name); 

		if (param == null) { 
			param = new NumberParam(name, number);	
			currentParamMap.put(name, param);
		} else {
			param.addValue(number);
		}
	}

	public void addNumberParam(String name, String number) {
		addNumberParam(name, Integer.valueOf(number));
	}

	public void addTextParam(String name, String text) {

		if (currentParamMap == null) {
			throw new IllegalStateException("Field currentParamMap must not be null!");
		}
		
		TextParam param = (TextParam) currentParamMap.get(name); 

		if (param == null) { 
			param = new TextParam(name, new String(text));	
			currentParamMap.put(name, param);
		} else {
			param.addValue(new String(text));
		}
	}

	public String dump() {

		StringBuilder buffer = new StringBuilder();
		
		for (String section : sectionMap.keySet()) {
		
			System.out.println("%%% Section: " + section);
			
			TreeMap<String, Param<?>> paramMap = sectionMap.get(section);
			
			for (String name : paramMap.keySet()) {
				
				Param<?> param = paramMap.get(name);
				System.out.println(param.toString());
			}
		}
		
		return buffer.toString();
	}

	public ArrayList<File[]> getInOutFilePair() {

		ArrayList<File[]> pairList = new ArrayList<File[]>();

		File inf = inputFile;
		File outf = outputFile;

		if (inputDirectory != null) {
			inf = new File(FileServices.appendFileName(inputDirectory,
					inf.getAbsolutePath()));
		}

		if (outputDirectory != null) {
			outf = new File(FileServices.appendFileName(outputDirectory,
					outf.getAbsolutePath()));
		}

		if (inf.exists() && outf.exists()) {

			File[] filePair = new File[2];
			filePair[0] = inf;
			filePair[1] = outf;
			pairList.add(filePair);
			return pairList;
		}

		if (inputDirectory != null && inputFile != null
				&& outputDirectory != null && outputFile != null) {

			Pattern pattern = Pattern.compile(inputFile.getName());

			for (File file : inputDirectory.listFiles()) {

				Matcher matcher = pattern.matcher(file.getName());

				if (matcher.matches()) {
					
					String outFileName = FileServices.appendFileName(outputDirectory, matcher.replaceFirst(outputFile
							.getName()));
					
					File[] filePair = new File[2];
					filePair[0] = file;
					filePair[1] = new File(outFileName);
					pairList.add(filePair);
				}
			}
		}

		return pairList;
	}

	public File getInputDirectory() {
		return inputDirectory;
	}

	public File getInputFile() {
		return inputFile;
	}

	public String getInputFormat() {
		return inputFormat;
	}

	public int getIntParameter(String section, String name, int def) {

		Param<?> param = getParameter(section, name);

		if (param == null) {
			return def;
		}

		return (Integer) param.getValue();
	}

	public File getOutputDirectory() {
		return outputDirectory;
	}

	public File getOutputFile() {
		return outputFile;
	}

	public String getOutputFormat() {
		return outputFormat;
	}

	public Param<?> getParameter(String section, String name) {

		TreeMap<String, Param<?>> paramMap = sectionMap.get(section);

		if (paramMap == null) {
			return null;
		}

		return paramMap.get(name);
	}

	public String getTextParameter(String section, String name, String def) {

		Param<?> param = getParameter(section, name);

		if (param == null) {
			return def;
		}

		return param.getString();
	}

	public ArrayList<String> getTextParameterList(String section,
			String name, ArrayList<String> def) {

		Param<?> param = getParameter(section, name);

		if (param == null) {
			return def;
		}

		ArrayList<?> values = param.getValues();
		
		ArrayList<String> list = new ArrayList<String>(values.size());
		
		for (Object value : values) {
			list.add(value.toString());
		}
		
		return list;
	}

	public boolean isParameterEnabled(String section, String name, boolean def) {

		Param<?> param = getParameter(section, name);

		if (param == null) {
			return def;
		}

		return param.isEnabled();
	}

	public void selectSection(String section) {

		TreeMap<String, Param<?>> paramMap = sectionMap.get(section);
		
		if (paramMap == null) {
			paramMap = new TreeMap<String, Param<?>>();
			sectionMap.put(section, paramMap);
		}
		
		currentParamMap = paramMap;
	}

	public void setInputDirectory(String dirName) {
		inputDirectory = new File(dirName);
	}

	public void setInputFile(File inputFile) {
		this.inputFile = inputFile;
	}

	public void setInputFile(String fileName) {
		inputFile = new File(fileName);
	}

	public void setInputFormat(String inputFormat) {
		this.inputFormat = inputFormat;
	}

	public void setOutputDirectory(String dirName) {
		outputDirectory = new File(dirName);
	}

	public void setOutputFile(File outputFile) {
		this.outputFile = outputFile;
	}

	public void setOutputFile(String fileName) {
		outputFile = new File(fileName);
	}

	public void setOutputFormat(String outputFormat) {
		this.outputFormat = outputFormat;
	}

	public void setTextParam(String name, String text) {

		TextParam param = (TextParam) currentParamMap.get(name); 
		param = new TextParam(name, new String(text));	
		currentParamMap.put(name, param);
	}
}
