/* 
 * Copyright (C) 2001-2012 Michael Fuchs
 *
 * This file is part of herold.
 * 
 * herold is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * herold is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with herold.  If not, see <http://www.gnu.org/licenses/>.  
 */
package org.dbdoclet.trafo.html.docbook.editor;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;

import org.dbdoclet.service.FileServices;
import org.dbdoclet.trafo.html.docbook.DBTConstants;
import org.dbdoclet.trafo.script.Script;
import org.dbdoclet.trafo.tag.docbook.DocBookElement;
import org.dbdoclet.trafo.tag.docbook.DocBookTagFactory;
import org.dbdoclet.trafo.tag.docbook.Emphasis;
import org.dbdoclet.trafo.tag.docbook.Entry;
import org.dbdoclet.trafo.tag.docbook.ImageData;
import org.dbdoclet.trafo.tag.docbook.ImageObject;
import org.dbdoclet.trafo.tag.docbook.Link;
import org.dbdoclet.trafo.tag.docbook.Para;
import org.dbdoclet.trafo.tag.docbook.Screen;
import org.dbdoclet.trafo.tag.docbook.Term;
import org.dbdoclet.trafo.tag.docbook.ULink;
import org.dbdoclet.trafo.tag.html.Img;
import org.dbdoclet.xiphias.ImageServices;

public class ImgEditor extends Editor {

	private static HashMap<String, String> validFormatMap;

	static {

		validFormatMap = new HashMap<String, String>();

		validFormatMap.put("BASE64", "BASE64");
		validFormatMap.put("BMP", "BMP");
		validFormatMap.put("CGM-BINARY", "CGM-BINARY");
		validFormatMap.put("CGM-CHAR", "CGM-CHAR");
		validFormatMap.put("CGM-CLEAR", "CGM-CLEAR");
		validFormatMap.put("DITROFF", "DITROFF");
		validFormatMap.put("DVI", "DVI");
		validFormatMap.put("EPS", "EPS");
		validFormatMap.put("EQN", "EQN");
		validFormatMap.put("FAX", "FAX");
		validFormatMap.put("GIF", "GIF");
		validFormatMap.put("GIF87a", "GIF87a");
		validFormatMap.put("GIF89a", "GIF89a");
		validFormatMap.put("IGES", "IGES");
		validFormatMap.put("JPEG", "JPEG");
		validFormatMap.put("JPG", "JPG");
		validFormatMap.put("linespecific", "linespecific");
		validFormatMap.put("PCX", "PCX");
		validFormatMap.put("PIC", "PIC");
		validFormatMap.put("PNG", "PNG");
		validFormatMap.put("PS", "PS");
		validFormatMap.put("SGML", "SGML");
		validFormatMap.put("SVG", "SVG");
		validFormatMap.put("TBL", "TBL");
		validFormatMap.put("TEX", "TEX");
		validFormatMap.put("TIFF", "TIFF");
		validFormatMap.put("WMF", "WMF");
		validFormatMap.put("WPG", "WPG");
	}

	@Override
	public EditorInstruction edit(EditorInstruction values)
			throws EditorException {

		setValues(super.edit(values));

		DocBookTagFactory dbfactory = values.getTagFactory();
		Script script = getScript();

		Img img = (Img) getHtmlElement();

		Boolean useAbsoluteImagePaths = script.isParameterEnabled(
				DBTConstants.SECTION_DOCBOOK,
				DBTConstants.PARAM_DOCBOOK_USE_ABSOLUTE_IMAGE_PATH, false);

		ArrayList<String> imageDataFormats = createImageDataFormatList(
				script.getTextParameterList(DBTConstants.SECTION_DOCBOOK,
						DBTConstants.PARAM_DOCBOOK_IMAGEDATA_FORMATS,
						new ArrayList<String>()), img.getSrc());

		String imagePath = script.getTextParameter(
				DBTConstants.SECTION_DOCBOOK,
				DBTConstants.PARAM_DOCBOOK_IMAGE_PATH, "");

		String src = img.getSrc();

		if ((src == null) || (src.length() == 0)) {
			return finalizeValues();
		}

		if (imagePath != null && imagePath.length() > 0
				&& FileServices.isAbsolutePath(src) == false) {
			src = FileServices.appendFileName(imagePath, src);
		}

		validateSrc(src);

		DocBookElement media;
		DocBookElement parent = getParent();

		if (parent instanceof Emphasis || parent instanceof Entry
				|| parent instanceof Para || parent instanceof Screen
				|| parent instanceof Term || parent instanceof ULink
				|| parent instanceof Link) {

			media = dbfactory.createInlineMediaObject();
			parent.appendChild(media);
			media.setParentNode(getCurrent());

			setAnything(media.getParentNode());

		} else {

			DocBookElement figure;

			String title = img.getTitle();

			if ((title != null) && (title.length() > 0)) {
				figure = dbfactory.createFigure(title);
			} else {
				figure = dbfactory.createInformalFigure();
			}

			figure.setParentNode(parent);

			if (figure.validate()) {

				parent.appendChild(figure);

				media = dbfactory.createMediaObject();
				figure.appendChild(media);

				setAnything(figure);

			} else {

				traverse(true);
				return finalizeValues();
			}
		}

		traverse(false);
		setCurrent(parent);

		File file = new File(src);

		if (useAbsoluteImagePaths) {
			file = file.getAbsoluteFile();
		}

		try {

			createHtmlImageData(media, dbfactory, imageDataFormats, img, file);
			createFoImageData(media, dbfactory, imageDataFormats, img, file);

		} catch (IOException oops) {
			throw new EditorException(oops);
		}

		return finalizeValues();
	}

	private ArrayList<String> createImageDataFormatList(
			ArrayList<String> paramList, String src) {

		ArrayList<String> formatList = new ArrayList<String>();

		for (String param : paramList) {

			if (param == null) {
				continue;
			}

			if (formatList.contains(param.toUpperCase()) == false) {
				formatList.add(param.toUpperCase());
			}
		}

		if (src != null) {

			String value = FileServices.getExtension(src);

			if (value != null && value.trim().length() > 0
					&& formatList.contains(value.toUpperCase()) == false) {
				formatList.add(0, value.toUpperCase());
			}
		}

		return formatList;
	}

	public static void createHtmlImageData(DocBookElement parent, DocBookTagFactory dbfactory,
			List<String> imageDataFormats, Img img,
			File file) throws IOException {

		String fileRef = FileServices.normalizePath(file.getPath());

		if (FileServices.isAbsolutePath(fileRef)) {

			fileRef = FileServices.normalizePath(fileRef);

			if (fileRef.startsWith("/")) {
				fileRef = "file://" + fileRef;
			} else {
				fileRef = "file:///" + fileRef;
			}
		}

		fileRef = FileServices.getFileBase(fileRef);
		int index = 0;
		
		for (String format : imageDataFormats) {

			String width = img.getWidth();
			String height = img.getHeight();
			String align = img.getAlign();
			
			ImageObject image = dbfactory.createImageObject();
		
			if (index == 0) {
				image.setRole("html");
			} else {
				image.setRole("html-" + format.toLowerCase());
			}
			
			ImageData data = dbfactory.createImageData();
			data.setScaleFit(true);

			if (width != null && width.length() > 0) {
				data.setContentWidth(width);
			}

			if (height != null && height.length() > 0) {
				data.setContentDepth(height);
			}

			if (file.exists() && format.equalsIgnoreCase("BASE64")) {

				String fileName = FileServices.getFileBase(file) + ".base64";
				FileServices.writeFromString(new File(fileName),
						ImageServices.toXml(file));
			}

			data.setFormat(format);

			if (align != null && align.length() > 0) {
				data.setAlign(align);
			}

			data.setFileRef(fileRef + "." + format.toLowerCase());
			image.appendChild(data);
			parent.appendChild(image);			
			index++;
		}
	}

	public static void createFoImageData(DocBookElement parent, DocBookTagFactory dbfactory, List<String> imageDataFormats,
			 Img img, File file) throws IOException {

		String fileRef = FileServices.normalizePath(file.getPath());
		fileRef = FileServices.getFileBase(fileRef);

		if (FileServices.isAbsolutePath(fileRef)) {

			fileRef = FileServices.normalizePath(fileRef);

			if (fileRef.startsWith("/")) {
				fileRef = "file://" + fileRef;
			} else {
				fileRef = "file:///" + fileRef;
			}
		}

		int index = 0;
		
		for (String format : imageDataFormats) {

			ImageObject image = dbfactory.createImageObject();
			
			if (index == 0) {
				image.setRole("fo");
			} else {
				image.setRole("fo-" + format.toLowerCase());
			}

			String align = img.getAlign();
			
			ImageData data = dbfactory.createImageData();

			data.setScaleFit(true);
			data.setWidth("100%");
			data.setContentDepth("100%");
			data.setFormat(format);
			
			if (align != null && align.length() > 0) {
				data.setAlign(align);
			}

			data.setFileRef(fileRef + "." + format.toLowerCase());

			image.appendChild(data);
			parent.appendChild(image);
			index++;
		}
	}

	public static String validateFormat(String format) {

		if (format == null) {

			throw new IllegalArgumentException("Parameter format is null!");
		}

		format = format.toUpperCase();

		if (validFormatMap.get(format) != null) {
			return format;
		} else {
			return "JPG";
		}
	}

	private String validateSrc(String src) {

		if (src == null) {
			throw new IllegalArgumentException("Parameter src is null!");
		}

		if (src.startsWith("http:") && (src.indexOf("?") > 0)) {
			src = src.substring(0, src.indexOf("?"));
		}

		return src;
	}
}
