/* 
 * Copyright (C) 2001-2012 Michael Fuchs
 *
 * This file is part of herold.
 * 
 * herold is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * herold is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with herold.  If not, see <http://www.gnu.org/licenses/>.  
 */
package org.dbdoclet.trafo.html.docbook.editor;

import org.dbdoclet.trafo.tag.docbook.Anchor;
import org.dbdoclet.trafo.tag.docbook.Chapter;
import org.dbdoclet.trafo.tag.docbook.DocBookElement;
import org.dbdoclet.trafo.tag.docbook.DocBookTagFactory;
import org.dbdoclet.trafo.tag.docbook.Para;
import org.dbdoclet.trafo.tag.html.Div;

public class DivEditor extends Editor {

	private static final int TIP = 1;
	private static final int NOTE = 2;
	private static final int WARNING = 3;
	private static final int CAUTION = 4;
	private static final int IMPORTANT = 5;

	private void addAdmonition(DocBookTagFactory dbfactory, int type,
			String title) {

		DocBookElement admon;
		DocBookElement parent = getParent();

		switch (type) {
		case CAUTION:
			admon = dbfactory.createCaution();
			break;
		case IMPORTANT:
			admon = dbfactory.createImportant();
			break;
		case NOTE:
			admon = dbfactory.createNote();
			break;
		case TIP:
			admon = dbfactory.createTip();
			break;
		case WARNING:
			admon = dbfactory.createWarning();
			break;
		default:
			admon = null;
			break;
		}

		if (admon == null) {
			return;
		}

		if (title != null && title.length() > 0) {
			admon.appendChild(dbfactory.createTitle(title));
		}

		Para para = dbfactory.createPara();
		admon.appendChild(para);
		parent.appendChild(admon);
		setCurrent(para);
	}

	@Override
	public EditorInstruction edit(EditorInstruction values)
			throws EditorException {

		setValues(super.edit(values));

		DocBookTagFactory dbfactory = values.getTagFactory();

		String type = null;

		Div div = (Div) getHtmlElement();
		String id = div.getId();

		String clazz = div.getCssClass();
		String title = div.getTitle();

		DocBookElement parent = getParent();

		traverse(true);
		ignore(false);

		if ((id != null) && (id.length() > 0)) {
			type = id.toLowerCase();
		}

		if ((type == null) && (clazz != null) && (clazz.length() > 0)) {
			type = clazz.toLowerCase();
		}

		if ((type != null) && (type.length() > 0)) {

			if (type.startsWith("example")) {

				if ((title != null) && (title.length() > 0)) {
					setCurrent(dbfactory.createExample(title));
				} else {
					setCurrent(dbfactory.createInformalExample());
				}

				parent.appendChild(getCurrent());
				getCurrent().setParentNode(parent);
				return finalizeValues();
			}

			if (type.startsWith("formalpara")) {

				if ((title != null) && (title.length() > 0)) {
					setCurrent(dbfactory.createFormalPara(title));
				} else {
					setCurrent(dbfactory.createFormalPara());
				}

				parent.appendChild(getCurrent());
				return finalizeValues();
			}

			if (type.startsWith("caution")) {

				addAdmonition(dbfactory, CAUTION, title);
				return finalizeValues();
			}

			if (type.startsWith("important")) {

				addAdmonition(dbfactory, IMPORTANT, title);
				return finalizeValues();
			}

			if (type.startsWith("note")) {

				addAdmonition(dbfactory, NOTE, title);
				return finalizeValues();
			}

			if (type.startsWith("tip") || type.startsWith("hint")) {

				addAdmonition(dbfactory, TIP, title);
				return finalizeValues();
			}

			if (type.startsWith("warning")) {

				addAdmonition(dbfactory, WARNING, title);
				return finalizeValues();
			}

		}

		if (parent.isBook()) {
			
			Chapter chapter = dbfactory.createChapter("@TODO");
			parent.appendChild(chapter);
			
			Para para = dbfactory.createPara();
			chapter.appendChild(para);
			
			parent = para;
			setCurrent(para);
		}
		
		if (id != null) {
			Anchor anchor = dbfactory.createAnchor();
			anchor.validate();
			copyCommonAttributes(getHtmlElement(), anchor);
			parent.appendChild(anchor);
		}

		traverse(true);
		ignore(true);

		return finalizeValues();
	}
}
