/* 
 * Copyright (C) 2001-2012 Michael Fuchs
 *
 * This file is part of herold.
 * 
 * herold is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * herold is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with herold.  If not, see <http://www.gnu.org/licenses/>.  
 */
package org.dbdoclet.herold;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStreamReader;
import java.net.URL;
import java.util.ResourceBundle;

import org.dbdoclet.log.Logger;
import org.dbdoclet.option.BooleanOption;
import org.dbdoclet.option.FileOption;
import org.dbdoclet.option.Option;
import org.dbdoclet.option.OptionException;
import org.dbdoclet.option.OptionList;
import org.dbdoclet.option.SelectOption;
import org.dbdoclet.option.StringOption;
import org.dbdoclet.service.FileServices;
import org.dbdoclet.service.MessageServices;
import org.dbdoclet.service.ResourceServices;
import org.dbdoclet.service.StringServices;
import org.dbdoclet.trafo.TrafoException;
import org.dbdoclet.trafo.TrafoScriptManager;
import org.dbdoclet.trafo.html.docbook.DBTConstants;
import org.dbdoclet.trafo.html.docbook.DocBookTransformer;
import org.dbdoclet.trafo.script.Script;

/**
 * The class <code>Herold</code> implements a console converter to transform
 * HTML code to DocBook XML.
 * 
 * @author <a href ="mailto:mfuchs@unico-consulting.com">Michael Fuchs</a>
 * @version 1.0
 */
public class Herold {

	private static Logger logger = Logger.getLogger(Herold.class);
	private static FileOption optIn;
	private static FileOption optOut;
	private static ResourceBundle res = ResourceBundle
			.getBundle("org/dbdoclet/herold/Resources");
	private boolean verbose = false;
	private static boolean systemExitEnabled = true;

	public static void main(String[] args) {

		OptionList options = null;
		int rc = 0;

		try {

			Option<?> option;
			options = new OptionList(args);

			// help
			option = new BooleanOption("help", "h");
			option.setMediumName("?");
			options.add(option);

			// license
			option = new BooleanOption("license", "L");
			options.add(option);

			// version
			option = new BooleanOption("version", "V");
			options.add(option);

			options.validate(true);

			if (options.getFlag("help", false)) {
				printUsage();
				return;
			}

			if (options.getFlag("version", false)) {
				printVersion();
				return;
			}

			if (options.getFlag("license", false)) {
				printLicense();
				return;
			}

			options = createOptionList(args);

			if (options.validate() == false) {
				throw new OptionException(options.getError());
			}

			Logger.setRootLogLevel(Logger.ERROR);
			Logger.setLoggerLevel(DocBookTransformer.class,
					getLoggingLevel(options.getTextOption("logging-level")
							.getValue()));
			Logger.setRootOutputFormat(Logger.SHORT);

			Herold converter = new Herold();

			FileOption optProfile = (FileOption) options.getOption("profile");

			Script script;

			if (optProfile.isUnset() == false) {
				script = TrafoScriptManager.parseScript(optProfile.getValue());
			} else {
				script = new Script();
			}

			File outFile = converter.processCommandLineOptions(options, script);
			script.setOutputFile(outFile);
			script.setInputFile(optIn.getValue());

			// System.out.println(script.dump());
			converter.convert(script);

		} catch (OptionException oops) {

			if ((options != null) && (options.getFlag("help", false) == false)) {

				printUsage();

				String msg = oops.getMessage();

				if (msg != null) {
					System.err.println(msg);
				}

			} else {
				oops.printStackTrace();
				printUsage();
			}

			rc = 1;

		} catch (FileNotFoundException oops) {

			oops.printStackTrace();
			rc = 5;

		} catch (IOException oops) {

			oops.printStackTrace();
			rc = 6;

		} catch (Exception oops) {

			oops.printStackTrace();
			rc = 7;
		}

		if (systemExitEnabled == true) {
			System.exit(rc);
		}
	}

	private static OptionList createOptionList(String[] args) {

		Option<?> option;
		SelectOption selopt;
		BooleanOption bopt;
		StringOption sopt;

		OptionList options = new OptionList(args);

		// add-index
		bopt = new BooleanOption(DBTConstants.SECTION_DOCBOOK.toLowerCase()
				+ "-" + DBTConstants.PARAM_DOCBOOK_ADD_INDEX, "x");
		bopt.setValue(false);
		options.add(bopt);

		// destination-encoding
		sopt = new StringOption(DBTConstants.SECTION_DOCBOOK.toLowerCase()
				+ "-" + DBTConstants.PARAM_DOCBOOK_DESTINATION_ENCODING, "d");
		sopt.setDefault("UTF-8");
		options.add(sopt);

		// decompose-tables
		bopt = new BooleanOption(DBTConstants.SECTION_DOCBOOK.toLowerCase()
				+ "-" + DBTConstants.PARAM_DOCBOOK_DECOMPOSE_TABLES,
				"T");
		bopt.setDefault(false);
		options.add(bopt);

		// document-element
		selopt = new SelectOption(DBTConstants.SECTION_DOCBOOK.toLowerCase()
				+ "-" + DBTConstants.PARAM_DOCBOOK_DOCUMENT_ELEMENT,
				"r");
		
		String[] optv2 = { "article", "book", "reference" };
		selopt.setList(optv2);
		selopt.setDefault("article");
		options.add(selopt);

		// title
		sopt = new StringOption(DBTConstants.SECTION_DOCBOOK.toLowerCase()
				+ "-" + DBTConstants.PARAM_DOCBOOK_TITLE, "t");
		sopt.setDefault("http://www.dbdoclet.org/herold");
		options.add(sopt);

		// source-encoding
		sopt = new StringOption(DBTConstants.SECTION_HTML.toLowerCase()
				+ "-" + DBTConstants.PARAM_HTML_SOURCE_ENCODING, "s");
		sopt.setDefault("UTF-8");
		options.add(sopt);

		// in
		optIn = new FileOption("in", "i");
		optIn.isRequired(true);
		optIn.isExisting(true);
		options.add(optIn);

		// loglevel
		SelectOption optLogLevel = new SelectOption("logging-level", "l");
		optLogLevel.setList(new String[] { "debug5", "debug4", "debug3",
				"debug2", "debug", "info", "warn", "error", "fatal" });
		optLogLevel.setDefault("error");
		options.add(optLogLevel);

		// out
		optOut = new FileOption("out", "o");
		options.add(optOut);

		FileOption optProfile = new FileOption("profile", "p");
		optProfile.isExisting(true);
		options.add(optProfile);

		// verbose
		option = new BooleanOption("verbose", "v");
		options.add(option);

		return options;
	}

	private static int getLoggingLevel(String value) {

		if (value == null) {
			return Logger.WARN;
		}

		value = value.trim().toLowerCase();

		if (value.equals("fatal")) {
			return Logger.FATAL;
		}

		if (value.equals("error")) {
			return Logger.ERROR;
		}

		if (value.equals("warn")) {
			return Logger.WARN;
		}

		if (value.equals("info")) {
			return Logger.INFO;
		}

		if (value.equals("debug")) {
			return Logger.DEBUG;
		}

		if (value.equals("debug2")) {
			return Logger.DEBUG2;
		}

		if (value.equals("debug3")) {
			return Logger.DEBUG3;
		}

		if (value.equals("debug4")) {
			return Logger.DEBUG4;
		}

		if (value.equals("debug5")) {
			return Logger.DEBUG5;
		}

		return Logger.WARN;
	}

	private static String getVersion() {

		Package p = Herold.class.getPackage();
		return p.getImplementationVersion();
	}

	private static void printLicense() throws IOException {

		URL url = ResourceServices
				.getResourceAsUrl("/org/dbdoclet/herold/COPYING");

		BufferedReader reader = new BufferedReader(new InputStreamReader(
				url.openStream()));
		String line = reader.readLine();

		while (line != null) {
			println(line);
			line = reader.readLine();
		}

		reader.close();
	}

	private static void println(String str) {
		System.out.println(str);
	}

	private static void printUsage() {

		try {

			String resname = ResourceServices.getString(res, "C_USAGE");
			String buffer = ResourceServices.getResourceAsString(resname);

			println(MessageServices.format(buffer, getVersion()));

		} catch (IOException oops) {

			logger.fatal("Printing usage message failed!", oops);
		}
	}

	private static void printVersion() {

		println("herold version \"" + getVersion() + "\"");
	}

	public void convert(Script script) throws TrafoException {

		DocBookTransformer trafo = new DocBookTransformer(
				script.getOutputFile());

		if (verbose == true) {
			trafo.addProgressListener(new ConsoleProgressListener(false));
		}

		trafo.setScript(script);
		trafo.convert(script.getInputFile(), script.getOutputFile());

		if (verbose == true) {
			System.out.println();
		}
	}

	public boolean isCanceled() {
		return false;
	}

	public File processCommandLineOptions(OptionList options, Script script) {

		for (Option<?> option : options) {

			String name = option.getLongName();

			if (name.startsWith(DBTConstants.SECTION_DOCBOOK.toLowerCase())) {
				script.selectSection(DBTConstants.SECTION_DOCBOOK);
				name = StringServices.cutPrefix(name,
						DBTConstants.SECTION_DOCBOOK.toLowerCase());
			} else if (name.startsWith(DBTConstants.SECTION_HTML.toLowerCase())) {
				script.selectSection(DBTConstants.SECTION_HTML);
				name = StringServices.cutPrefix(name,
						DBTConstants.SECTION_HTML.toLowerCase());
			} else if (name.startsWith(DBTConstants.SECTION_GROOVY
					.toLowerCase())) {
				script.selectSection(DBTConstants.SECTION_GROOVY);
				name = StringServices.cutPrefix(name,
						DBTConstants.SECTION_GROOVY.toLowerCase());
			}

			name = StringServices.cutPrefix(name, "-");

			if (option.isUnset() == false) {
				switch (option.getType()) {

				case BOOLEAN:
					script.addBoolParam(name, (Boolean) option.getValue());
					break;

				case TEXT:
					script.addTextParam(name, option.getValue().toString());
					break;
				}
			}
		}

		setVerbose(options.getFlag("verbose", false));

		File outFile;

		if (optOut.isUnset()) {

			String outFileName = FileServices.getFileBase(optIn.getValue())
					+ ".xml";
			// outFile = FileServices.createUniqueFile(new File(outFileName));
			outFile = new File(outFileName);

		} else {
			outFile = optOut.getValue();
		}

		return outFile;
	}

	public void setVerbose(boolean verbose) {
		this.verbose = verbose;
	}

	public void convert(File htmlFile, File xmlFile) throws TrafoException {

		Script script = new Script();
		script.setInputFile(htmlFile);
		script.setOutputFile(xmlFile);
		convert(script);
	}

	public static void setSystemExitEnabled(boolean systemExitEnabled) {
		Herold.systemExitEnabled = systemExitEnabled;
	}
}
