{-# LANGUAGE UndecidableInstances #-}
-- Search for UndecidableInstances to see why this is needed
{- |
Module      :  Control.Monad.Reader.Class
Copyright   :  (c) Andy Gill 2001,
               (c) Oregon Graduate Institute of Science and Technology 2001,
               (c) Jeff Newbern 2003-2007,
               (c) Andriy Palamarchuk 2007
License     :  BSD-style (see the file LICENSE)

Maintainer  :  libraries@haskell.org
Stability   :  experimental
Portability :  non-portable (multi-param classes, functional dependencies)

[Computation type:] Computations which read values from a shared environment.

[Binding strategy:] Monad values are functions from the environment to a value.
The bound function is applied to the bound value, and both have access
to the shared environment.

[Useful for:] Maintaining variable bindings, or other shared environment.

[Zero and plus:] None.

[Example type:] @'Reader' [(String,Value)] a@

The 'Reader' monad (also called the Environment monad).
Represents a computation, which can read values from
a shared environment, pass values from function to function,
and execute sub-computations in a modified environment.
Using 'Reader' monad for such computations is often clearer and easier
than using the 'Control.Monad.State.State' monad.

  Inspired by the paper
  /Functional Programming with Overloading and Higher-Order Polymorphism/,
    Mark P Jones (<http://web.cecs.pdx.edu/~mpj/>)
    Advanced School of Functional Programming, 1995.
-}

module Control.Monad.Reader.Class (
    MonadReader(..),
    asks,
    ) where

import Control.Monad.Trans.Cont as Cont
import Control.Monad.Trans.Error
import Control.Monad.Trans.Identity
import Control.Monad.Trans.List
import Control.Monad.Trans.Maybe
import Control.Monad.Trans.Reader (ReaderT)
import qualified Control.Monad.Trans.Reader as ReaderT (ask, local)
import qualified Control.Monad.Trans.RWS.Lazy as LazyRWS (RWST, ask, local)
import qualified Control.Monad.Trans.RWS.Strict as StrictRWS (RWST, ask, local)
import Control.Monad.Trans.State.Lazy as Lazy
import Control.Monad.Trans.State.Strict as Strict
import Control.Monad.Trans.Writer.Lazy as Lazy
import Control.Monad.Trans.Writer.Strict as Strict

import Control.Monad.Trans.Class (lift)
import Control.Monad
import Data.Monoid

-- ----------------------------------------------------------------------------
-- class MonadReader
--  asks for the internal (non-mutable) state.

-- | See examples in "Control.Monad.Reader".
-- Note, the partially applied function type @(->) r@ is a simple reader monad.
-- See the @instance@ declaration below.
class (Monad m) => MonadReader r m | m -> r where
    -- | Retrieves the monad environment.
    ask   :: m r

    -- | Executes a computation in a modified environment.
    local :: (r -> r)	-- ^ The function to modify the environment.
          -> m a	-- ^ @Reader@ to run in the modified environment.
          -> m a

-- | Retrieves a function of the current environment.
asks :: (MonadReader r m)
    => (r -> a)		-- ^ The selector function to apply to the environment.
    -> m a
asks f = do
    r <- ask
    return (f r)

-- ----------------------------------------------------------------------------
-- The partially applied function type is a simple reader monad

instance MonadReader r ((->) r) where
    ask       = id
    local f m = m . f

instance (Monad m) => MonadReader r (ReaderT r m) where
    ask = ReaderT.ask
    local = ReaderT.local

instance (Monad m, Monoid w) => MonadReader r (LazyRWS.RWST r w s m) where
    ask = LazyRWS.ask
    local = LazyRWS.local

instance (Monad m, Monoid w) => MonadReader r (StrictRWS.RWST r w s m) where
    ask = StrictRWS.ask
    local = StrictRWS.local

-- ---------------------------------------------------------------------------
-- Instances for other mtl transformers
--
-- All of these instances need UndecidableInstances,
-- because they do not satisfy the coverage condition.

instance (MonadReader r' m) => MonadReader r' (ContT r m) where
    ask   = lift ask
    local = Cont.liftLocal ask local

instance (Error e, MonadReader r m) => MonadReader r (ErrorT e m) where
    ask   = lift ask
    local = mapErrorT . local

instance (MonadReader r m) => MonadReader r (IdentityT m) where
    ask   = lift ask
    local = mapIdentityT . local

instance (MonadReader r m) => MonadReader r (ListT m) where
    ask   = lift ask
    local = mapListT . local

instance (MonadReader r m) => MonadReader r (MaybeT m) where
    ask   = lift ask
    local = mapMaybeT . local

instance (MonadReader r m) => MonadReader r (Lazy.StateT s m) where
    ask   = lift ask
    local = Lazy.mapStateT . local

instance (MonadReader r m) => MonadReader r (Strict.StateT s m) where
    ask   = lift ask
    local = Strict.mapStateT . local

instance (Monoid w, MonadReader r m) => MonadReader r (Lazy.WriterT w m) where
    ask   = lift ask
    local = Lazy.mapWriterT . local

instance (Monoid w, MonadReader r m) => MonadReader r (Strict.WriterT w m) where
    ask   = lift ask
    local = Strict.mapWriterT . local
