-- KeyInfo.hs: OpenPGP (RFC4880) fingerprinting methods
-- Copyright © 2012-2013  Clint Adams
-- This software is released under the terms of the Expat license.
-- (See the LICENSE file).

module Codec.Encryption.OpenPGP.KeyInfo (
   keySize
 , pkalgoAbbrev
) where

import qualified Crypto.PubKey.RSA as RSA
import qualified Crypto.PubKey.DSA as DSA
import qualified Crypto.Hash.MD5 as MD5
import qualified Crypto.Hash.SHA1 as SHA1
import qualified Data.ByteString as B
import Data.Bits (shiftR)
import Data.List (unfoldr)

import Codec.Encryption.OpenPGP.Types

keySize (RSAPubKey x) = RSA.public_size x * 8
keySize (DSAPubKey x) = bitcount . DSA.params_p . DSA.public_params $ x
keySize (ElGamalPubKey x) = bitcount $ x !! 0

bitcount = (*8) . length . unfoldr (\x -> if x == 0 then Nothing else Just (True, x `shiftR` 8))

pkalgoAbbrev RSA = "R"
pkalgoAbbrev DSA = "D"
pkalgoAbbrev ElgamalEncryptOnly = "g"
pkalgoAbbrev DeprecatedRSAEncryptOnly = "-"
pkalgoAbbrev DeprecatedRSASignOnly = "_"
pkalgoAbbrev EC = "e"
pkalgoAbbrev ECDSA = "E"
pkalgoAbbrev ForbiddenElgamal = "f"
pkalgoAbbrev DH = "d"
pkalgoAbbrev (OtherPKA _) = "."
