-- | Haskell Interface to GLFW (<http://www.glfw.org>).
--   Supports GLFW API version 2.7.2.
--
--   GLFW thread functions are not supported by this module; use
--   Haskell thread instead.

{-# LANGUAGE CPP, ExistentialQuantification, ForeignFunctionInterface, TypeFamilies #-}

module Graphics.UI.GLFW
  ( -- * Data types
    Version
  , DisplayBits(..)
  , WindowMode(..)
  , VideoMode(..)
  , KeyButtonState(..)
  , Key(..)
  , SpecialKey(..)
  , MouseButton(..)
  , Joystick (..)
  , JoystickParam (..)
  , SpecialFeature(..)
  , TextureFlag(..)
  , BitmapFont(..)
  , WindowCloseCallback
  , WindowSizeCallback
  , WindowRefreshCallback
  , KeyCallback
  , CharCallback
  , MouseButtonCallback
  , MousePosCallback
  , MouseWheelCallback
    -- * Initialization and Termination
  , initialize
  , terminate
  , version
    -- * Window Handling
  , openWindow
  , closeWindow
  , windowTitle
  , windowSize
  , windowPos
  , iconifyWindow
  , restoreWindow
  , swapBuffers
  , swapInterval
    -- * Video Modes
  , videoModes
  , desktopMode
    -- * Input Handling
  , pollEvents
  , waitEvents
  , getKey
  , getMouseButton
  , mousePos
  , mouseWheel
  , joystickParam
  , joystickPos
  , joystickPos'
  , joystickButtons
    -- * Callbacks
    -- ** Window callbacks
  , windowSizeCallback
  , windowCloseCallback
  , windowRefreshCallback
    -- ** Input callbacks
  , keyCallback
  , charCallback
  , mouseButtonCallback
  , mousePosCallback
  , mouseWheelCallback
    -- * Timing
  , time
  , sleep
    -- * OpenGL Extension Support
  , extensionSupported
  , glVersion
    -- * Texture loading
  , loadTexture2D
  , loadMemoryTexture2D
    -- * Text rendering
  , renderString
    -- * Miscellaneous
  , enableSpecial
  , disableSpecial
    -- * Window parameters and hints
  , ParamVal
  , Param(..)
  , Hint(..)
  , Opened(..)
  , Active(..)
  , Iconified(..)
  , Accelerated(..)
  , RedBits(..)
  , GreenBits(..)
  , BlueBits(..)
  , AlphaBits(..)
  , DepthBits(..)
  , StencilBits(..)
  , RefreshRate(..)
  , AccumRedBits(..)
  , AccumGreenBits(..)
  , AccumBlueBits(..)
  , AccumAlphaBits(..)
  , AuxBuffers(..)
  , Stereo(..)
  , NoResize(..)
  , FSAASamples(..)
  , OpenGLVersionMajor(..)
  , OpenGLVersionMinor(..)
  , OpenGLForwardCompat(..)
  , OpenGLDebugContext(..)
  , OpenGLProfile(..)
  , Profile(..)
) where

import Control.Monad (liftM, liftM2)
import Data.IORef    (IORef, atomicModifyIORef, newIORef, readIORef, writeIORef)
import Foreign
import Foreign.C
import Graphics.Rendering.OpenGL (GLfloat, ($=))
import qualified Graphics.Rendering.OpenGL as GL

-- | Version is represented by (major, minor, revision), used
--   in gettable variable 'version'.
type Version = (Int, Int, Int)

-- | Bit depth of GL display buffers, used in 'openWindow'.
data DisplayBits
  = DisplayRGBBits !Int !Int !Int
  | DisplayAlphaBits !Int
  | DisplayDepthBits !Int
  | DisplayStencilBits !Int
  deriving (Eq, Show)

-- | Window or Fullscreen mode, used in 'openWindow'.
data WindowMode
  = Window
  | FullScreen
  deriving (Eq, Show)

instance Enum WindowMode where
  fromEnum Window     = 0x00010001
  fromEnum FullScreen = 0x00010002
  toEnum 0x00010001 = Window
  toEnum 0x00010002 = FullScreen
  toEnum _          = error "GLFW: WindowMode toEnum out of bounds"
  
-- | We use type families to organize Window params that
--   can be retrieved using 'getParam' of the 'Param' class.
--   The value of a param 'a' is of type 'ParamVal' 'a', 
--   where 'ParamVal' is a type family defined as follows: 
--  
-- > ParamVal Opened      = Bool
-- > ParamVal Active      = Bool
-- > ParamVal Iconified   = Bool
-- > ParamVal Accelerated = Bool
-- > ParamVal RedBits     = Int
-- > ParamVal GreenBits   = Int
-- > ParamVal BlueBits    = Int
-- > ParamVal AlphaBits   = Int
-- > ParamVal DepthBits   = Int
-- > ParamVal StencilBits = Int
-- 
--   The following are both params and hints that can be set using 
--   'openWindowHint' of the 'Hint' class.
-- 
-- > ParamVal RefreshRate         = Int
-- > ParamVal AccumRedBits        = Int
-- > ParamVal AccumGreenBits      = Int
-- > ParamVal AccumBlueBits       = Int
-- > ParamVal AccumAlphaBits      = Int
-- > ParamVal AuxBuffers          = Int
-- > ParamVal Stereo              = Bool
-- > ParamVal NoResize            = Bool
-- > ParamVal FSAASamples         = Int
-- > ParamVal OpenGLVersionMajor  = Int
-- > ParamVal OpenGLVersionMinor  = Int
-- > ParamVal OpenGLForwardCompat = Bool
-- > ParamVal OpenGLDebugContext  = Bool
-- > ParamVal OpenGLProfile       = Profile
--

type family ParamVal a

class Param a where
    getParam :: a -> IO (ParamVal a)
    
class Param a => Hint a where
    openWindowHint :: a -> ParamVal a -> IO ()
    
data Opened = Opened
data Active = Active
data Iconified = Iconified
data Accelerated = Accelerated
data RedBits = RedBits
data GreenBits = GreenBits
data BlueBits = BlueBits
data AlphaBits = AlphaBits
data DepthBits = DepthBits
data StencilBits = StencilBits
data RefreshRate = RefreshRate
data AccumRedBits = AccumRedBits
data AccumGreenBits = AccumGreenBits
data AccumBlueBits = AccumBlueBits
data AccumAlphaBits = AccumAlphaBits
data AuxBuffers = AuxBuffers
data Stereo = Stereo
data NoResize = NoResize
data FSAASamples = FSAASamples
data OpenGLVersionMajor = OpenGLVersionMajor
data OpenGLVersionMinor = OpenGLVersionMinor
data OpenGLForwardCompat = OpenGLForwardCompat
data OpenGLDebugContext = OpenGLDebugContext
data OpenGLProfile = OpenGLProfile

type instance ParamVal Opened = Bool
type instance ParamVal Active = Bool
type instance ParamVal Iconified = Bool
type instance ParamVal Accelerated = Bool
type instance ParamVal RedBits = Int
type instance ParamVal GreenBits = Int
type instance ParamVal BlueBits = Int
type instance ParamVal AlphaBits = Int
type instance ParamVal DepthBits = Int
type instance ParamVal StencilBits = Int
type instance ParamVal RefreshRate = Int
type instance ParamVal AccumRedBits = Int
type instance ParamVal AccumGreenBits = Int
type instance ParamVal AccumBlueBits = Int
type instance ParamVal AccumAlphaBits = Int
type instance ParamVal AuxBuffers = Int
type instance ParamVal Stereo = Bool
type instance ParamVal NoResize = Bool
type instance ParamVal FSAASamples = Int
type instance ParamVal OpenGLVersionMajor = Int
type instance ParamVal OpenGLVersionMinor = Int
type instance ParamVal OpenGLForwardCompat = Bool
type instance ParamVal OpenGLDebugContext = Bool
type instance ParamVal OpenGLProfile = Profile

-- | OpenGL profiles, used in 'openWindowHint' with 'OpenGLProfile'.
data Profile
  = DefaultProfile
  | OpenGLCoreProfile
  | OpenGLCompatProfile
  deriving (Eq, Show)
  
instance Enum Profile where
  fromEnum DefaultProfile      = 0
  fromEnum OpenGLCoreProfile   = 0x00050001
  fromEnum OpenGLCompatProfile = 0x00050002
  toEnum 0          = DefaultProfile
  toEnum 0x00050001 = OpenGLCoreProfile
  toEnum 0x00050002 = OpenGLCompatProfile
  toEnum _          = error "GLFW: Profile toEnum out of bounds"

-- | Video modes used in gettable variables 'videoModes' and 'desktopMode'.
data VideoMode = VideoMode
  { videoWidth     :: !Int
  , videoHeight    :: !Int
  , videoRedBits   :: !Int
  , videoBlueBits  :: !Int
  , videoGreenBits :: !Int
  } deriving (Eq, Show)

-- | Key or button state.
data KeyButtonState
  = Release
  | Press
  deriving (Eq, Show, Enum)

-- | Key is represented by either a character key or a special key.
data Key
  = CharKey Char
  | SpecialKey SpecialKey
  deriving (Eq, Ord, Show)

instance Enum Key where
  fromEnum (CharKey c)    = fromEnum c
  fromEnum (SpecialKey c) = fromEnum c
  toEnum c =
      if c < 256
        then CharKey (toEnum c)
        else SpecialKey (toEnum c)

-- | Special key is a key not represented in the 32 - 127 printable ASCII range.
data SpecialKey
  = UNKNOWN
  | ESC
  | F1
  | F2
  | F3
  | F4
  | F5
  | F6
  | F7
  | F8
  | F9
  | F10
  | F11
  | F12
  | F13
  | F14
  | F15
  | F16
  | F17
  | F18
  | F19
  | F20
  | F21
  | F22
  | F23
  | F24
  | F25
  | UP
  | DOWN
  | LEFT
  | RIGHT
  | LSHIFT
  | RSHIFT
  | LCTRL
  | RCTRL
  | LALT
  | RALT
  | TAB
  | ENTER
  | BACKSPACE
  | INSERT
  | DEL
  | PAGEUP
  | PAGEDOWN
  | HOME
  | END
  | KP_0
  | KP_1
  | KP_2
  | KP_3
  | KP_4
  | KP_5
  | KP_6
  | KP_7
  | KP_8
  | KP_9
  | KP_DIVIDE
  | KP_MULTIPLY
  | KP_SUBTRACT
  | KP_ADD
  | KP_DECIMAL
  | KP_EQUAL
  | KP_ENTER
  deriving (Eq, Ord, Show)

instance Enum SpecialKey where
  fromEnum UNKNOWN     = -1
  fromEnum ESC         = 257
  fromEnum F1          = 258
  fromEnum F2          = 259
  fromEnum F3          = 260
  fromEnum F4          = 261
  fromEnum F5          = 262
  fromEnum F6          = 263
  fromEnum F7          = 264
  fromEnum F8          = 265
  fromEnum F9          = 266
  fromEnum F10         = 267
  fromEnum F11         = 268
  fromEnum F12         = 269
  fromEnum F13         = 270
  fromEnum F14         = 271
  fromEnum F15         = 272
  fromEnum F16         = 273
  fromEnum F17         = 274
  fromEnum F18         = 275
  fromEnum F19         = 276
  fromEnum F20         = 277
  fromEnum F21         = 278
  fromEnum F22         = 279
  fromEnum F23         = 280
  fromEnum F24         = 281
  fromEnum F25         = 282
  fromEnum UP          = 283
  fromEnum DOWN        = 284
  fromEnum LEFT        = 285
  fromEnum RIGHT       = 286
  fromEnum LSHIFT      = 287
  fromEnum RSHIFT      = 288
  fromEnum LCTRL       = 289
  fromEnum RCTRL       = 290
  fromEnum LALT        = 291
  fromEnum RALT        = 292
  fromEnum TAB         = 293
  fromEnum ENTER       = 294
  fromEnum BACKSPACE   = 295
  fromEnum INSERT      = 296
  fromEnum DEL         = 297
  fromEnum PAGEUP      = 298
  fromEnum PAGEDOWN    = 299
  fromEnum HOME        = 300
  fromEnum END         = 301
  fromEnum KP_0        = 302
  fromEnum KP_1        = 303
  fromEnum KP_2        = 304
  fromEnum KP_3        = 305
  fromEnum KP_4        = 306
  fromEnum KP_5        = 307
  fromEnum KP_6        = 308
  fromEnum KP_7        = 309
  fromEnum KP_8        = 310
  fromEnum KP_9        = 311
  fromEnum KP_DIVIDE   = 312
  fromEnum KP_MULTIPLY = 313
  fromEnum KP_SUBTRACT = 314
  fromEnum KP_ADD      = 315
  fromEnum KP_DECIMAL  = 316
  fromEnum KP_EQUAL    = 317
  fromEnum KP_ENTER    = 318
  toEnum 257 = ESC
  toEnum 258 = F1
  toEnum 259 = F2
  toEnum 260 = F3
  toEnum 261 = F4
  toEnum 262 = F5
  toEnum 263 = F6
  toEnum 264 = F7
  toEnum 265 = F8
  toEnum 266 = F9
  toEnum 267 = F10
  toEnum 268 = F11
  toEnum 269 = F12
  toEnum 270 = F13
  toEnum 271 = F14
  toEnum 272 = F15
  toEnum 273 = F16
  toEnum 274 = F17
  toEnum 275 = F18
  toEnum 276 = F19
  toEnum 277 = F20
  toEnum 278 = F21
  toEnum 279 = F22
  toEnum 280 = F23
  toEnum 281 = F24
  toEnum 282 = F25
  toEnum 283 = UP
  toEnum 284 = DOWN
  toEnum 285 = LEFT
  toEnum 286 = RIGHT
  toEnum 287 = LSHIFT
  toEnum 288 = RSHIFT
  toEnum 289 = LCTRL
  toEnum 290 = RCTRL
  toEnum 291 = LALT
  toEnum 292 = RALT
  toEnum 293 = TAB
  toEnum 294 = ENTER
  toEnum 295 = BACKSPACE
  toEnum 296 = INSERT
  toEnum 297 = DEL
  toEnum 298 = PAGEUP
  toEnum 299 = PAGEDOWN
  toEnum 300 = HOME
  toEnum 301 = END
  toEnum 302 = KP_0
  toEnum 303 = KP_1
  toEnum 304 = KP_2
  toEnum 305 = KP_3
  toEnum 306 = KP_4
  toEnum 307 = KP_5
  toEnum 308 = KP_6
  toEnum 309 = KP_7
  toEnum 310 = KP_8
  toEnum 311 = KP_9
  toEnum 312 = KP_DIVIDE
  toEnum 313 = KP_MULTIPLY
  toEnum 314 = KP_SUBTRACT
  toEnum 315 = KP_ADD
  toEnum 316 = KP_DECIMAL
  toEnum 317 = KP_EQUAL
  toEnum 318 = KP_ENTER
  toEnum _   = UNKNOWN

-- | Mouse button is represented by left, right, middle or a number from 0 to
--   7. /Differs from the C API./
data MouseButton
  = ButtonLeft
  | ButtonRight
  | ButtonMiddle
  | ButtonNo Int
  deriving (Eq, Show)

instance Enum MouseButton where
  fromEnum ButtonLeft   = 0
  fromEnum ButtonRight  = 1
  fromEnum ButtonMiddle = 2
  fromEnum (ButtonNo i)
    | i < 0                       = 0
    | i < _GLFW_MOUSE_BUTTON_LAST = i
    | otherwise                   = _GLFW_MOUSE_BUTTON_LAST
  toEnum 0 = ButtonLeft
  toEnum 1 = ButtonRight
  toEnum 2 = ButtonMiddle
  toEnum i = ButtonNo i

-- | Joystick is represent a number from 0 to 15. /Differs from the C API./
newtype Joystick
  = Joystick Int
  deriving (Eq, Show)

instance Enum Joystick where
  fromEnum (Joystick i)
    | i < 0                   = 0
    | i < _GLFW_JOYSTICK_LAST = i
    | otherwise               = _GLFW_JOYSTICK_LAST
  toEnum i | i < _GLFW_JOYSTICK_LAST = Joystick i
  toEnum _                           = Joystick _GLFW_JOYSTICK_LAST

-- | Joystick parameters
data JoystickParam
  -- | Indicates whether the joystick is present.
  = Present
  -- | Number of axes supported by the joystick.
  | Axes
  -- | Number of buttons supported by the joystick.
  | Buttons
  deriving (Eq, Show)

instance Enum JoystickParam where
  fromEnum Present = 0x00050001
  fromEnum Axes    = 0x00050002
  fromEnum Buttons = 0x00050003
  toEnum 0x00050001 = Present
  toEnum 0x00050002 = Axes
  toEnum 0x00050003 = Buttons
  toEnum _          = error "GLFW: JoystickParam toEnum out of bounds"

-- | Special features used in 'enableSpecial' and 'disableSpecial'.
data SpecialFeature
  -- | When enabled, the mouse cursor is visible and mouse coordinates are relative to the
  --   upper left corner of the client area of the GLFW window. The coordinates are limited
  --   to the client area of the window.
  = MouseCursor
  -- | When enabled, keys which are pressed will not be released until they are physically
  --   released and checked with 'getKey'.
  | StickyKey
  -- | When enabled, mouse buttons which are pressed will not be released until they are
  --   physically released and checked with 'getMouseButton'.
  | StickyMouseButton
  -- | When enabled, pressing standard system key combinations, such as ALT+TAB under Windows,
  --   will give the normal behavior.
  | SystemKey
  -- | When enabled, the key and character callback functions are called repeatedly when a
  --   key is held down long enough (according to the system key repeat configuration).
  | KeyRepeat
  -- | When enabled, 'pollEvents' is automatically called each time 'swapBuffers' is called,
  --   immediately after the buffer swap itself. Enabled by default.
  | AutoPollEvent
  deriving (Eq, Show)

instance Enum SpecialFeature where
  fromEnum MouseCursor       = 0x00030001
  fromEnum StickyKey         = 0x00030002
  fromEnum StickyMouseButton = 0x00030003
  fromEnum SystemKey         = 0x00030004
  fromEnum KeyRepeat         = 0x00030005
  fromEnum AutoPollEvent     = 0x00030006
  toEnum 0x00030001 = MouseCursor
  toEnum 0x00030002 = StickyKey
  toEnum 0x00030003 = StickyMouseButton
  toEnum 0x00030004 = SystemKey
  toEnum 0x00030005 = KeyRepeat
  toEnum 0x00030006 = AutoPollEvent
  toEnum _          = error "GLFW: SpecialFeature toEnum out of bounds"

-- | Texture flag used in 'loadTexture2D' and 'loadMemoryTexture2D'.
data TextureFlag
  -- | Do not rescale to the closest 2^m x 2^n resolution.
  = NoRescale
  -- | Specifies that the origin of the loaded image should be in the upper left corner
  --   (default is the lower left corner).
  | OriginUL
  -- | Automatically build and upload all mipmap levels.
  | BuildMipMaps
  -- | Treat single component images as alpha maps rather than luminance maps.
  | AlphaMap
  deriving (Eq, Show)

instance Enum TextureFlag where
  fromEnum NoRescale    = 0x00000001
  fromEnum OriginUL     = 0x00000002
  fromEnum BuildMipMaps = 0x00000004
  fromEnum AlphaMap     = 0x00000008
  toEnum 0x00000001 = NoRescale
  toEnum 0x00000002 = OriginUL
  toEnum 0x00000004 = BuildMipMaps
  toEnum 0x00000008 = AlphaMap
  toEnum _          = error "GLFW: TextureFlag toEnum out of bounds"

-- | Built-in bitmap font used in 'renderString'.
data BitmapFont
  = Fixed8x16 -- ^ 8x16 fixed width font.
  deriving (Eq, Enum, Show)

-- Mouse button definitions
_GLFW_MOUSE_BUTTON_LAST :: Int
_GLFW_MOUSE_BUTTON_LAST = 7

-- Joystick identifiers
_GLFW_JOYSTICK_LAST :: Int
_GLFW_JOYSTICK_LAST = 15

-- Time spans longer than this (seconds) are considered to be infinity
_GLFW_INFINITY :: Double
_GLFW_INFINITY = 100000

-- Callback function type
type GLFWwindowsizefun    = Int -> Int -> IO ()
type GLFWwindowclosefun   = IO Bool
type GLFWwindowrefreshfun = IO ()
type GLFWmousebuttonfun   = Int -> Int -> IO ()
type GLFWmouseposfun      = Int -> Int -> IO ()
type GLFWmousewheelfun    = Int -> IO ()
type GLFWkeyfun           = Int -> Int -> IO ()
type GLFWcharfun          = Int -> Int -> IO ()

-- | Initialize GLFW library. Returns 'True' if successful, 'False' otherwise. Must
--   be called before any other GLFW functions.
initialize :: IO Bool
initialize = liftM toEnum glfwInit

foreign import ccall unsafe glfwInit :: IO Int

-- | Terminate GLFW library after use. Before a program terminates, GLFW has to
--   be terminated in order to free up resources, etc.
terminate :: IO ()
terminate = glfwTerminate >> glfwCleanup

foreign import ccall unsafe glfwTerminate :: IO ()

-- | Returns the GLFW C library version numbers.
version :: GL.GettableStateVar Version
version = GL.makeGettableStateVar $
  with 0 $ \x ->
  with 0 $ \y ->
  with 0 $ \z -> do
    glfwGetVersion x y z
    peek3 (x, y, z)

foreign import ccall unsafe glfwGetVersion :: Ptr Int -> Ptr Int -> Ptr Int -> IO ()

-- | Open a window. Returns 'True' if successful, 'False' otherwise. GLFW
--   applications can only open one window.
--
--   If width is zero and height is not, width will be calculated as width = 4/3 height.
--
--   If height is zero and width is not, height will be calculated as height = 3/4 width.
--
--   If both width and height are zero, width is set to 640 and height to 480.
--
--   Display bits default to 0 if no value is specified, meaning that particular buffer is not created.
--
--   In fullscreen mode a resolution that best matches the given window dimensions will be chosen.
--
--   In fullscreen mode the mouse cursor is hidden by default. To change the visibility of the mouse
--   cursor, see, 'enableSpecial' and 'disableSpecial'.
openWindow :: GL.Size -> [DisplayBits] -> WindowMode -> IO Bool
openWindow (GL.Size w h) bits mode = do
  writeIORef fontTextures []
  liftM toEnum $ glfwOpenWindow (fromIntegral w) (fromIntegral h) r' g' b' a' d' s' $ fromEnum mode
  where
    (r', g', b', a', d', s') = gather bits (0, 0, 0, 0, 0, 0)

    gather :: [DisplayBits] -> (Int, Int, Int, Int, Int, Int) -> (Int, Int, Int, Int, Int, Int)
    gather (x:xs) (r, g, b, a, d, s) = gather xs $ case x of
      DisplayRGBBits     r_ g_ b_ -> (r_, g_, b_, a , d , s )
      DisplayAlphaBits   a_       -> (r , g , b , a_, d , s )
      DisplayDepthBits   d_       -> (r , g , b , a , d_, s )
      DisplayStencilBits s_       -> (r , g , b , a , d , s_)
    gather [] vs = vs

foreign import ccall unsafe glfwOpenWindow :: Int -> Int -> Int -> Int -> Int -> Int -> Int -> Int -> Int -> IO Int

-- | Close the open window and destroy the associated OpenGL context.
foreign import ccall unsafe "glfwCloseWindow" closeWindow :: IO ()

-- | Vertical monitor refresh rate in Hz (only used for fullscreen windows).
--   Zero means system default. Use with caution: specifying a refresh rate can override the system's settings,
--   in which case the display may be suboptimal, fail or even damage the monitor.
instance Hint RefreshRate where
    openWindowHint RefreshRate = glfwOpenWindowHint 0x0002000B 
    
-- | Specify the number of bits for the red channel of the accumulation buffer.
instance Hint AccumRedBits where
    openWindowHint AccumRedBits = glfwOpenWindowHint 0x0002000C 
    
-- | Specify the number of bits for the green channel of the accumulation buffer.
instance Hint AccumGreenBits where
    openWindowHint AccumGreenBits = glfwOpenWindowHint 0x0002000D 
    
-- | Specify the number of bits for the blue channel of the accumulation buffer.
instance Hint AccumBlueBits where
    openWindowHint AccumBlueBits = glfwOpenWindowHint 0x0002000E 
    
-- | Specify the number of bits for the alpha channel of the accumulation buffer.
instance Hint AccumAlphaBits where
    openWindowHint AccumAlphaBits = glfwOpenWindowHint 0x0002000F 

-- | Specify the number of auxiliary buffers.
instance Hint AuxBuffers where
    openWindowHint AuxBuffers = glfwOpenWindowHint 0x00020010 
    
-- | Specify if stereo rendering should be supported.
--   If Stereo is requested on a call to 'openWindowHint', but no stereo rendering pixel formats / framebuffer
--   configs are available, 'openWindow' will fail.
instance Hint Stereo where
    openWindowHint Stereo = glfwOpenWindowHint 0x00020011 . fromEnum 

-- | Specify whether the window can be resized by the user.
instance Hint NoResize where
    openWindowHint NoResize = glfwOpenWindowHint 0x00020012 . fromEnum 
    
-- | Specify the number of samples to use for the multisampling buffer.
instance Hint FSAASamples where
    openWindowHint FSAASamples = glfwOpenWindowHint 0x00020013
    
-- | Specify the major number of the desired minimum OpenGL version.
instance Hint OpenGLVersionMajor where
    openWindowHint OpenGLVersionMajor = glfwOpenWindowHint 0x00020014 
    
-- | Specify the minor number of the desired minimum OpenGL version.
instance Hint OpenGLVersionMinor where
    openWindowHint OpenGLVersionMinor = glfwOpenWindowHint 0x00020015 

-- | Specify whether the OpenGL context should be forward-compatible (i.e. disallow legacy functionality).
--   This should only be used when requesting OpenGL version 3.0 or above.
instance Hint OpenGLForwardCompat where
    openWindowHint OpenGLForwardCompat = glfwOpenWindowHint 0x00020016 . fromEnum 
   
-- | Specify whether a debug context should be created.
instance Hint OpenGLDebugContext where
    openWindowHint OpenGLDebugContext = glfwOpenWindowHint 0x00020017 . fromEnum 
    
-- | Specify the OpenGL profile the context should implement. For available profiles see 'Profile'.
instance Hint OpenGLProfile where
    openWindowHint OpenGLProfile = glfwOpenWindowHint 0x00020018 . fromEnum 

foreign import ccall unsafe glfwOpenWindowHint :: Int -> Int -> IO ()

-- | Set the title of the opened window.
windowTitle :: GL.SettableStateVar String
windowTitle = GL.makeSettableStateVar setter
  where setter = flip withCString glfwSetWindowTitle

foreign import ccall unsafe glfwSetWindowTitle :: CString -> IO ()

-- | Get or set the size of the opened window.
--
--   The dimensions denote the size of the client area of the window (i.e. excluding any
--   window borders and decorations).
--
--   If the window is in fullscreen mode when setting new dimensions, the video mode will
--   be changed to a resolution that closest matches the given dimensions.
--
--   The setter has no effect if the window is iconified.
--
--   The OpenGL context is guaranteed to be preserved when the window is resized.
windowSize :: GL.StateVar GL.Size
windowSize = GL.makeStateVar getter setter
  where getter =
          with 0 $ \w ->
          with 0 $ \h -> do
            glfwGetWindowSize w h
            liftM2 GL.Size (fmap fromIntegral $ peek w) (fmap fromIntegral $ peek h)
        setter (GL.Size w h) = glfwSetWindowSize (fromIntegral w) (fromIntegral h)

foreign import ccall unsafe glfwGetWindowSize :: Ptr Int -> Ptr Int -> IO ()
foreign import ccall unsafe glfwSetWindowSize :: Int -> Int -> IO ()

-- | Set the position of the opened window.
--
--   The setter has no effect in fullscreen mode or if the window is iconified.
--
--   On multi-monitor systems, the behavior of the setter is ill-defined.
windowPos :: GL.SettableStateVar GL.Position
windowPos = GL.makeSettableStateVar $ \(GL.Position x y) -> glfwSetWindowPos x y

foreign import ccall unsafe glfwSetWindowPos :: GL.GLint -> GL.GLint -> IO ()

-- | Iconify the window.
foreign import ccall unsafe "glfwIconifyWindow" iconifyWindow :: IO ()

-- | Restore the window after iconification.
foreign import ccall unsafe "glfwRestoreWindow" restoreWindow :: IO ()

-- | Swap the back and front color buffers of the window. If 'AutoPollEvent' is
--   enabled by 'enableSpecial' (which is the default), it also polls for new
--   events before the swapping.
foreign import ccall safe "glfwSwapBuffers" swapBuffers :: IO ()

-- | Set the minimum number of monitor retraces between each each buffer swap
--   performed by 'swapBuffers'. If set to zero, buffer swaps will not be
--   synchronized to the vertical refresh of the monitor.
swapInterval :: GL.SettableStateVar Int
swapInterval = GL.makeSettableStateVar glfwSwapInterval

foreign import ccall unsafe glfwSwapInterval :: Int -> IO ()

-- | Query the window opened status.
instance Param Opened where
    getParam Opened = fmap toEnum $ glfwGetWindowParam 0x00020001

-- | Query the window active status.
instance Param Active where
    getParam Active = fmap toEnum $ glfwGetWindowParam 0x00020002
    
-- | Query the window iconified status.
instance Param Iconified where
    getParam Iconified = fmap toEnum $ glfwGetWindowParam 0x00020003
    
-- | Query the window hardware accelerated status.
instance Param Accelerated where
    getParam Accelerated = fmap toEnum $ glfwGetWindowParam 0x00020004
    
-- | Query the number of bits for the red color component.
instance Param RedBits where
    getParam RedBits = glfwGetWindowParam 0x00020005
    
-- | Query the number of bits for the green color component.
instance Param GreenBits where
    getParam GreenBits = glfwGetWindowParam 0x00020006
    
-- | Query the number of bits for the blue color component.
instance Param BlueBits where
    getParam BlueBits = glfwGetWindowParam 0x00020007
    
-- | Query the number of bits for the alpha buffer.
instance Param AlphaBits where
    getParam AlphaBits = glfwGetWindowParam 0x00020008
    
-- | Query the number of bits for the depth buffer.
instance Param DepthBits where
    getParam DepthBits = glfwGetWindowParam 0x00020009
    
-- | Query the number of bits for the stencil buffer.
instance Param StencilBits where
    getParam StencilBits = glfwGetWindowParam 0x0002000A
    
-- | Query the vertical monitor refresh rate in Hz (only used for fullscreen windows).
instance Param RefreshRate where
    getParam RefreshRate = glfwGetWindowParam 0x0002000B
    
-- | Query the number of bits for the red channel of the accumulation buffer.
instance Param AccumRedBits where
    getParam AccumRedBits = glfwGetWindowParam 0x0002000C
    
-- | Query the number of bits for the green channel of the accumulation buffer.
instance Param AccumGreenBits where
    getParam AccumGreenBits = glfwGetWindowParam 0x0002000D
    
-- | Query the number of bits for the blue channel of the accumulation buffer.
instance Param AccumBlueBits where
    getParam AccumBlueBits = glfwGetWindowParam 0x0002000E
    
-- | Query the number of bits for the alpha channel of the accumulation buffer.
instance Param AccumAlphaBits where
    getParam AccumAlphaBits = glfwGetWindowParam 0x0002000F
    
-- | Query the number of auxiliary buffers.
instance Param AuxBuffers where
    getParam AuxBuffers = glfwGetWindowParam 0x00020010
    
-- | Query whether the window supports stereo rendering.
instance Param Stereo where
    getParam Stereo = fmap toEnum $ glfwGetWindowParam 0x00020011
    
-- | Query whether the window can be resized by the user.
instance Param NoResize where
    getParam NoResize = fmap toEnum $ glfwGetWindowParam 0x00020012
    
-- | Query the number used for the multisampling buffer.
instance Param FSAASamples where
    getParam FSAASamples = glfwGetWindowParam 0x00020013
    
-- | Query the OpenGL major version.
instance Param OpenGLVersionMajor where
    getParam OpenGLVersionMajor = glfwGetWindowParam 0x00020014
    
-- | Query the OpenGL minor version.
instance Param OpenGLVersionMinor where
    getParam OpenGLVersionMinor = glfwGetWindowParam 0x00020015
    
-- | Query whether the current OpenGL context is forward-compatible.
instance Param OpenGLForwardCompat where
    getParam OpenGLForwardCompat = fmap toEnum $ glfwGetWindowParam 0x000200016
    
-- | Query whether the current OpenGL context is a debug context.
instance Param OpenGLDebugContext where
    getParam OpenGLDebugContext = fmap toEnum $ glfwGetWindowParam 0x00020017
    
-- | Query the OpenGL 'Profile' implemented by the current context.
instance Param OpenGLProfile where
    getParam OpenGLProfile = fmap toEnum $ glfwGetWindowParam 0x00020018

foreign import ccall unsafe glfwGetWindowParam :: Int -> IO Int

-- | Callback type for 'windowSizeCallback'.
type WindowSizeCallback = GL.Size -> IO ()

-- | Set the function that will be called every time the window size changes.
windowSizeCallback :: GL.SettableStateVar WindowSizeCallback
windowSizeCallback = GL.makeSettableStateVar (\f -> do
  let g w h = f $ GL.Size (fromIntegral w) (fromIntegral h)
  ptr <- glfwWrapFun2 g
  glfwSetCallbackIORef glfwWindowsizefun ptr
  glfwSetWindowSizeCallback ptr)

-- glfwSetWindowSizeCallback executes the callback (see window.c), ccall cannot be marked unsafe
foreign import ccall safe glfwSetWindowSizeCallback :: FunPtr GLFWwindowsizefun -> IO ()

-- | Callback type for 'windowCloseCallback'. The callback should return 'True' to close the window, and 'False' otherwise.
type WindowCloseCallback = IO Bool

-- | Set the function that will be called when the window is closed.
windowCloseCallback :: GL.SettableStateVar WindowCloseCallback
windowCloseCallback = GL.makeSettableStateVar (\f -> do
  ptr <- glfwWrapFunB f
  glfwSetCallbackIORef glfwWindowclosefun ptr
  glfwSetWindowCloseCallback ptr)

foreign import ccall safe glfwSetWindowCloseCallback :: FunPtr GLFWwindowclosefun -> IO ()

-- | Callback type for 'windowRefreshCallback'.
type WindowRefreshCallback = IO ()

-- | Set the function that will be called when the window client area needs to
--   be refreshed, which occurs when any part of the window client area is
--   damaged and needs to be repainted (for instance, if a part of the window
--   that was previously occluded by another window has become visible).
windowRefreshCallback :: GL.SettableStateVar WindowRefreshCallback
windowRefreshCallback = GL.makeSettableStateVar setter
  where
    setter f = do
      ptr <- glfwWrapFun0 f
      glfwSetCallbackIORef glfwWindowrefreshfun ptr
      glfwSetWindowRefreshCallback ptr

foreign import ccall unsafe glfwSetWindowRefreshCallback :: FunPtr GLFWwindowrefreshfun -> IO ()

-- | Get a list of detected 'VideoMode's, the max number of which is limited to
--   256 for now.
videoModes :: GL.GettableStateVar [VideoMode]
videoModes = GL.makeGettableStateVar getter
  where
    getter = withArray (replicate (sizeOfVideoMode * maxCount) 0) $ \arr ->
      glfwGetVideoModes arr maxCount >>= filterMode arr
    filterMode _ 0 = return []
    filterMode a c = do
      [w, h, r, b, g] <- peekArray 5 a
      let a' = advancePtr a sizeOfVideoMode
      rest <- filterMode a' (pred c)
      return $ VideoMode w h r b g : rest

foreign import ccall unsafe glfwGetVideoModes :: Ptr Int -> Int -> IO Int

sizeOfVideoMode :: Int
sizeOfVideoMode = 5

maxCount :: Int
maxCount = 256

-- | Get the 'VideoMode' of current desktop.
desktopMode :: GL.GettableStateVar VideoMode
desktopMode = GL.makeGettableStateVar getter
  where
    getter = withArray (replicate sizeOfVideoMode 0) $ \arr -> do
      glfwGetDesktopMode arr
      [w, h, r, b, g] <- peekArray 5 arr
      return $ VideoMode w h r b g

foreign import ccall unsafe glfwGetDesktopMode :: Ptr Int -> IO ()

-- | Poll events, such as user input and window events. Upon calling this
--   function, all window states, keyboard states and mouse states are updated.
--   If any related callback functions are registered, these are called during
--   the call of 'pollEvents'.
--
--   'pollEvents' is called implicitly from 'swapBuffers' if 'AutoPollEvent' is
--   enabled (as it is by default).
foreign import ccall safe "glfwPollEvents" pollEvents :: IO ()

-- | Wait for events, such as user input and window events. The calling thread
--   will be put to sleep until any event appears in the event queue.
--   When events are ready, the events will be processed just as they are
--   processed by 'pollEvents'.
foreign import ccall safe "glfwWaitEvents" waitEvents :: IO ()

-- | Return a 'KeyButtonState', either 'Release' or 'Press', of the indicated
--   key.
--
--   A window must be opened for the function to have any effect, and 'pollEvents',
--   'waitEvents' or 'swapBuffers' (with 'AutoPollEvent' enabled) must be called
--   before any keyboard events are recorded and reported by 'getKey'.
getKey :: Enum a => a -> IO KeyButtonState
getKey = fmap toEnum . glfwGetKey . fromEnum

foreign import ccall unsafe glfwGetKey :: Int -> IO Int

-- | Return a 'KeyButtonState', either 'Release' or 'Press', of the indicated
--   mouse button.
--
--   A window must be opened for the function to have any effect, and 'pollEvents',
--   'waitEvents' or 'swapBuffers' (with 'AutoPollEvent' enabled) must be called
--   before any mouse events are recorded and reported by 'getMouseButton'.
getMouseButton :: MouseButton -> IO KeyButtonState
getMouseButton = fmap toEnum . glfwGetMouseButton . fromEnum

foreign import ccall unsafe glfwGetMouseButton :: Int -> IO Int

-- | Set or get the mouse position.
--
--   A window must be opened for the getter to have any effect, and 'pollEvents',
--   'waitEvents' or 'swapBuffers' (with 'AutoPollEvent' enabled) must be called
--   before any mouse movements are recorded and reported by 'mousePos'.
--
--   When setting the mouse position, if the cursor is visible (not disabled), the
--   cursor will be moved to the specified position, relative to the upper left
--   corner of the window client area and with the Y-axis down. If the cursor is
--   hidden (disabled), only the mouse position that is reported by the getter is
--   changed.
mousePos :: GL.StateVar GL.Position
mousePos = GL.makeStateVar getter setter
  where
    getter = with 0 (\x -> with 0 (\y -> do
      glfwGetMousePos x y
      mx <- peek x
      my <- peek y
      return $ GL.Position (fromIntegral mx) (fromIntegral my)))
    setter (GL.Position x y) = glfwSetMousePos (fromIntegral x) (fromIntegral y)

foreign import ccall unsafe glfwGetMousePos :: Ptr Int -> Ptr Int -> IO ()
foreign import ccall unsafe glfwSetMousePos :: Int -> Int -> IO ()

-- | Set or get the mouse wheel position.
--
--   A window must be opened for the getter to have any effect, and 'pollEvents',
--   'waitEvents' or 'swapBuffers' (with 'AutoPollEvent' enabled) must be called
--   before any wheel movements are recorded and reported by 'mouseWheel'.
mouseWheel :: GL.StateVar Int
mouseWheel = GL.makeStateVar glfwGetMouseWheel glfwSetMouseWheel

foreign import ccall unsafe glfwGetMouseWheel :: IO Int
foreign import ccall unsafe glfwSetMouseWheel :: Int -> IO ()

-- | Get joystick parameters.
--
--   The joystick information is updated every time the getter is queried.
--
--   No window has to be opened for joystick information to be available.
joystickParam :: Joystick -> JoystickParam -> GL.GettableStateVar Int
joystickParam j param = GL.makeGettableStateVar (glfwGetJoystickParam (fromEnum j) (fromEnum param))

foreign import ccall unsafe glfwGetJoystickParam :: Int -> Int -> IO Int

-- | Get a certain number of axis positions for the given joystick. If the
--   number of positions requested is is greater than the number available, the
--   unavailable positions will be 0.
--
--   The joystick state is updated every time the getter is queried.
--
--   No window has to be opened for joystick input to be available.
joystickPos :: Joystick -> Int -> GL.GettableStateVar [Float]
joystickPos j n = GL.makeGettableStateVar $
  withArray (replicate n 0) $ \a -> do
    _ <- glfwGetJoystickPos (fromEnum j) a n
    peekArray n a

-- | Get joystick positions. The returned list contains the positions
--   for all available axes for the given joystick.
--
--   The joystick state is updated every time the getter is queried.
--
--   No window has to be opened for joystick input to be available.
joystickPos' :: Joystick -> GL.GettableStateVar [Float]
joystickPos' j = GL.makeGettableStateVar $ do
  n <- glfwGetJoystickParam (fromEnum j) (fromEnum Axes)
  allocaArray n $ \a -> do
    n' <- glfwGetJoystickPos (fromEnum j) a n
    peekArray n' a

foreign import ccall unsafe glfwGetJoystickPos :: Int -> Ptr Float -> Int -> IO Int

-- | Get joystick button states. The returned list contains the states
--   for all available buttons for the given joystick.
--
--   The joystick state is updated every time the getter is queried.
--
--   No window has to be opened for joystick input to be available.
joystickButtons :: Joystick -> GL.GettableStateVar [KeyButtonState]
joystickButtons j = GL.makeGettableStateVar $ do
  n <- glfwGetJoystickParam (fromEnum j) (fromEnum Buttons)
  allocaArray n $ \a -> do
    n' <- glfwGetJoystickButtons (fromEnum j) a n
    l  <- peekArray n' a
    return $ map (toEnum . fromEnum) l

foreign import ccall unsafe glfwGetJoystickButtons :: Int -> Ptr Int8 -> Int -> IO Int

-- | Callback type for 'keyCallback'.
type KeyCallback = Key -> KeyButtonState -> IO ()

-- | Set the function that will be called when there is a key event, i.e.,
--   every time the state of a single key is changed. The reported keys are
--   unaffected by any modifiers (such as SHIFT or ALT).
keyCallback :: GL.SettableStateVar KeyCallback
keyCallback = GL.makeSettableStateVar setter
  where
    setter f = do
      let g k s = f (toEnum k) (toEnum s)
      ptr <- glfwWrapFun2 g
      glfwSetCallbackIORef glfwKeyfun ptr
      glfwSetKeyCallback ptr

foreign import ccall unsafe glfwSetKeyCallback :: FunPtr GLFWkeyfun -> IO ()

-- | Callback type for 'charCallback'.
type CharCallback = Char -> KeyButtonState -> IO ()

-- | Set the function that will be called when there is a character event,
--   i.e., every time a key that results in a printable Unicode character is
--   pressed or released. Characters are affected by modifiers (such as SHIFT
--   or ALT).
charCallback :: GL.SettableStateVar CharCallback
charCallback = GL.makeSettableStateVar setter
  where
    setter f = do
      let g k s = f (toEnum k) (toEnum s)
      ptr <- glfwWrapFun2 g
      glfwSetCallbackIORef glfwCharfun ptr
      glfwSetCharCallback ptr

foreign import ccall unsafe glfwSetCharCallback :: FunPtr GLFWcharfun -> IO ()

-- | Callback type for 'mouseButtonCallback'.
type MouseButtonCallback = MouseButton -> KeyButtonState -> IO ()

-- | Set the function that will be called when there is a mouse button event,
--   i.e., every time a mouse button is pressed or released.
mouseButtonCallback :: GL.SettableStateVar MouseButtonCallback
mouseButtonCallback = GL.makeSettableStateVar setter
  where
    setter f = do
      let g b s = f (toEnum b) (toEnum s)
      ptr <- glfwWrapFun2 g
      glfwSetCallbackIORef glfwMousebuttonfun ptr
      glfwSetMouseButtonCallback ptr

foreign import ccall unsafe glfwSetMouseButtonCallback :: FunPtr GLFWmousebuttonfun -> IO ()

-- | Callback type for 'mousePosCallback'.
type MousePosCallback = GL.Position -> IO ()

-- | Set the function that will be called when there is a mouse motion event,
--   i.e., every time they mouse is moved.
mousePosCallback :: GL.SettableStateVar MousePosCallback
mousePosCallback = GL.makeSettableStateVar setter
  where
    setter f = do
      let g x y = f =<< do
#if _GLASGOW_HASKELL__ >= 610
            return $ GL.Position (fromIntegral x) (fromIntegral y)
#else
      -- Work around bug in GHC FFI
      -- See http://hackage.haskell.org/trac/ghc/ticket/2594
            with 0 $ \ptr -> do
              poke ptr x
              x32 <- peek (castPtr ptr) :: IO Int32
              poke ptr y
              y32 <- peek (castPtr ptr) :: IO Int32
              return $ GL.Position (fromIntegral x32) (fromIntegral y32)
#endif
      ptr <- glfwWrapFun2 g
      glfwSetCallbackIORef glfwMouseposfun ptr
      glfwSetMousePosCallback ptr

-- glfwSetMousePosCallback executes the callback (see input.c), ccall cannot be marked unsafe
foreign import ccall safe glfwSetMousePosCallback :: FunPtr GLFWmouseposfun -> IO ()

-- | Callback type for 'mouseWheelCallback'.
type MouseWheelCallback = Int -> IO ()

-- | Set the function that will be called when there is a mouse wheel event,
--   i.e., every time the mouse wheel is turned.
mouseWheelCallback :: GL.SettableStateVar MouseWheelCallback
mouseWheelCallback  = GL.makeSettableStateVar setter
  where
    setter f = do
#if __GLASGOW_HASKELL__ >= 610
      ptr <- glfwWrapFun1 f
#else
      -- Work around bug in GHC FFI
      -- See http://hackage.haskell.org/trac/ghc/ticket/2594
      let g x = with 0 \ptr -> do
                    poke ptr x
                    x32 <- peek (castPtr ptr) :: IO Int32
                    f (fromIntegral x32)
      ptr <- glfwWrapFun1 g
#endif
      glfwSetCallbackIORef glfwMousewheelfun ptr
      glfwSetMouseWheelCallback ptr

-- glfwSetMouseWheelCallback executes the callback (see input.c), ccall cannot be marked unsafe
foreign import ccall safe glfwSetMouseWheelCallback :: FunPtr GLFWmousewheelfun -> IO ()

-- | Get or set the value of the high precision timer. The time is measured in
--   seconds as a double precision floating point number.
--
--   Unless the timer has been set by the programmer, the time is measured as
--   the number of seconds that have passed since 'initialize' was called.
--
--   The resolution of the timer depends on which system the program is running on.
time :: GL.StateVar Double
time = GL.makeStateVar glfwGetTime glfwSetTime

foreign import ccall unsafe glfwGetTime :: IO Double
foreign import ccall unsafe glfwSetTime :: Double -> IO ()

-- | Put the calling thread to sleep for the requested period of time in
--   seconds.
--
--   The minimum amount of time it is possible to sleep is generally in the
--   range 1ms to 20ms.
foreign import ccall unsafe "glfwSleep" sleep :: Double -> IO ()

-- | Return 'True' if the extension is supported, 'False' otherwise.
extensionSupported :: String -> IO Bool
extensionSupported = liftM toEnum . flip withCString glfwExtensionSupported

foreign import ccall unsafe glfwExtensionSupported :: CString -> IO Int

-- | Returns the version numbers for the currently used OpenGL implementation.
glVersion :: GL.GettableStateVar Version
glVersion = GL.makeGettableStateVar $
  with 0 $ \x ->
  with 0 $ \y ->
  with 0 $ \z -> do
    glfwGetGLVersion x y z
    peek3 (x, y, z)
    
foreign import ccall unsafe glfwGetGLVersion :: Ptr Int -> Ptr Int -> Ptr Int -> IO ()

-- TODO:
-- foreign import ccall unsafe glfwGetProcAddress :: Ptr CChar -> FunPtr ?
-- foreign import ccall unsafe glfwGetGLVersion :: Ptr Int -> Ptr Int -> Ptr Int -> IO ()

-- | Enable a 'SpecialFeature'.
enableSpecial :: SpecialFeature -> IO ()
enableSpecial = glfwEnable . fromEnum

foreign import ccall unsafe glfwEnable :: Int -> IO ()

-- | Disable a 'SpecialFeature'.
disableSpecial :: SpecialFeature -> IO ()
disableSpecial = glfwDisable . fromEnum

foreign import ccall unsafe glfwDisable :: Int -> IO ()

--foreign import ccall unsafe glfwReadImage :: Ptr CChar -> Ptr GLFWimage -> Int -> IO ()
--foreign import ccall unsafe glfwFreeImage :: Ptr GLFWimage -> IO ()

-- | Read an image from a file specified by the given string and upload the
--   image to OpenGL texture memory.
--
--   If BuildMipMaps flag is given, all mipmap levels for the loaded texture
--   are generated and uploaded to texture memory.
--
--   Unless the OriginUL flag is given, the origin of the texture is the lower
--   left corner of the loaded image. If OriginUL is given, however, the first
--   pixel is the upper left corner.
--
--   For single component images (i.e. gray scale), the texture is uploaded as
--   an alpha mask if the flag AlphaMap is set, otherwise it's uploaded as a
--   luminance texture.
--
--   It only supports the Truevision Targa verson 1 file (.tga). Supported
--   pixel formats are: 8-bit gray scale, 8-bit paletted (24/32-bit color),
--   24-bit true color and 32-bit true color + alpha.
--
--   Paletted images are translated into true color or true color + alpha pixel
--   formats.
--
--   The read texture is always rescaled to the nearest larger 2^m x 2^n
--   resolution using bilinear interpolation if necessary.
loadTexture2D :: String -> [TextureFlag] -> IO Bool
loadTexture2D fname flag = do
  r <- withCString fname (`glfwLoadTexture2D` readFlag flag) 
  return $ toEnum r

foreign import ccall unsafe glfwLoadTexture2D :: CString -> Int -> IO Int

readFlag :: forall t. (Enum t) => [t] -> Int
readFlag = foldr ((.|.) . fromEnum) 0

-- | Read an image from the memory buffer (the given byte string) and
--   upload the image to OpenGL texture memory. The rest is similar
--   to 'loadTexture2D'.
loadMemoryTexture2D :: String -> [TextureFlag] -> IO Bool
loadMemoryTexture2D arr flag = withCAStringLen arr (\(ptr, len) -> do
  r <- glfwLoadMemoryTexture2D ptr len (readFlag flag)
  return $ toEnum r)

foreign import ccall unsafe glfwLoadMemoryTexture2D :: Ptr CChar -> Int -> Int -> IO Int

glfwWindowsizefun    :: IORef (Maybe (FunPtr GLFWwindowsizefun))
glfwWindowclosefun   :: IORef (Maybe (FunPtr GLFWwindowclosefun))
glfwWindowrefreshfun :: IORef (Maybe (FunPtr GLFWwindowrefreshfun))
glfwMousebuttonfun   :: IORef (Maybe (FunPtr GLFWmousebuttonfun))
glfwMouseposfun      :: IORef (Maybe (FunPtr GLFWmouseposfun))
glfwMousewheelfun    :: IORef (Maybe (FunPtr GLFWmousewheelfun))
glfwKeyfun           :: IORef (Maybe (FunPtr GLFWkeyfun))
glfwCharfun          :: IORef (Maybe (FunPtr GLFWcharfun))
glfwWindowsizefun    = unsafePerformIO (newIORef Nothing)
glfwWindowclosefun   = unsafePerformIO (newIORef Nothing)
glfwWindowrefreshfun = unsafePerformIO (newIORef Nothing)
glfwMousebuttonfun   = unsafePerformIO (newIORef Nothing)
glfwMouseposfun      = unsafePerformIO (newIORef Nothing)
glfwMousewheelfun    = unsafePerformIO (newIORef Nothing)
glfwKeyfun           = unsafePerformIO (newIORef Nothing)
glfwCharfun          = unsafePerformIO (newIORef Nothing)

foreign import ccall unsafe "wrapper" glfwWrapFun2 :: (Int -> Int -> IO ()) -> IO (FunPtr (Int -> Int -> IO ()))
foreign import ccall unsafe "wrapper" glfwWrapFun1 :: (Int -> IO ()) -> IO (FunPtr (Int -> IO ()))
foreign import ccall unsafe "wrapper" glfwWrapFunB :: IO Bool -> IO (FunPtr (IO Bool))
foreign import ccall unsafe "wrapper" glfwWrapFun0 :: IO () -> IO (FunPtr (IO ()))

glfwSetCallbackIORef :: forall a. IORef (Maybe (FunPtr a)) -> FunPtr a -> IO ()
glfwSetCallbackIORef ref f =
  atomicModifyIORef ref (\g -> (Just f, g)) >>=
    maybe (return ()) freeHaskellFunPtr

glfwCleanup :: IO ()
glfwCleanup = do
  cleanup glfwWindowsizefun
  cleanup glfwWindowclosefun
  cleanup glfwWindowrefreshfun
  cleanup glfwMousebuttonfun
  cleanup glfwMouseposfun
  cleanup glfwMousewheelfun
  cleanup glfwKeyfun
  cleanup glfwCharfun
  where
    cleanup :: IORef (Maybe (FunPtr a)) -> IO ()
    cleanup ref =
      atomicModifyIORef ref (\g -> (Nothing, g)) >>=
        maybe (return ()) freeHaskellFunPtr

-- text rendering

fontTextures :: IORef [(BitmapFont, GL.TextureObject)]
fontTextures = unsafePerformIO (newIORef [])

loadFont :: BitmapFont -> IO GL.TextureObject
loadFont name = do
  l <- readIORef fontTextures
  case lookup name l of
    Just font -> do
      (r, _) <- GL.areTexturesResident [font]
      if length r == 1
        then return font
        else loadIt $ filter ((/=name) . fst) l
    Nothing -> loadIt l
  where
    bitmap = case name of
                 Fixed8x16 -> bitmap8x16
    loadIt l = do
      objs <- GL.genObjectNames 1
      case objs of
        [font] -> do
          GL.textureBinding GL.Texture2D $= Just font
          -- this next line is important, otherwise it won't render the texture!
          GL.textureFilter GL.Texture2D $= ((GL.Linear', Nothing), GL.Linear')
          _ <- loadMemoryTexture2D bitmap [OriginUL, NoRescale]
          writeIORef fontTextures ((name, font):l)
          return font
        _ -> error "GLFW: loadFont Unable to load texture"

renderChar :: (Enum a) => t -> a -> IO ()
renderChar _ c = do
  let y = fromIntegral (fromEnum c `rem` 16 * 16) / 256
      x = fromIntegral (fromEnum c `quot` 16 * 8) / 128
      dx = 8 / 128
      dy = 16 / 256
      h = 16
      w = 8
  GL.preservingMatrix $ GL.renderPrimitive GL.Quads (do
    GL.texCoord (texCoord2 x y)
    GL.vertex (vertex3 0 h 0)
    GL.texCoord (texCoord2 x (y + dy))
    GL.vertex (vertex3 0 0 0)
    GL.texCoord (texCoord2 (x + dx) (y + dy))
    GL.vertex (vertex3 w 0 0)
    GL.texCoord (texCoord2 (x + dx) y)
    GL.vertex (vertex3 w h 0))
  GL.translate (vector3 w 0 0)

-- | Render a text string using the given 'BitmapFont'. Text is rendered
--   through texture, and is only possible with alpha enabled.
renderString :: BitmapFont -> String -> IO ()
renderString name s = do
  font <- loadFont name
  GL.texture GL.Texture2D $= GL.Enabled
  GL.textureBinding GL.Texture2D $= Just font
  GL.preservingMatrix $ mapM_ (renderChar font) s
  GL.texture GL.Texture2D $= GL.Disabled

peek3 :: (Storable t1, Storable t2, Storable t) => (Ptr t, Ptr t1, Ptr t2) -> IO (t, t1, t2)
peek3 (x, y, z) = do
  x' <- peek x
  y' <- peek y
  z' <- peek z
  return (x', y', z')

vector3 :: GLfloat -> GLfloat -> GLfloat -> GL.Vector3 GLfloat
vector3 = GL.Vector3

vertex3 :: GLfloat -> GLfloat -> GLfloat -> GL.Vertex3 GLfloat
vertex3 = GL.Vertex3

texCoord2 :: GLfloat -> GLfloat -> GL.TexCoord2 GLfloat
texCoord2 = GL.TexCoord2

bitmap8x16 :: String
bitmap8x16 = "\x00\x00\x0a\x00\x00\x00\x00\x00\x00\x00\x00\x00\x80\x00\x00\x01\x20\x08\xff\xff\xff\xff\x00\xa7\xff\xff\xff\x00\x86\xff\xff\xff\xf3\x00\xff\xff\xff\x34\xcf\xff\xff\xff\x00\x01\xff\xff\xff\x25\xff\xff\xff\xf6\x82\xff\xff\xff\xef\x04\xff\xff\xff\xf0\xff\xff\xff\xef\xff\xff\xff\x00\xff\xff\xff\x25\xff\xff\xff\xf6\x82\xff\xff\xff\xef\x01\xff\xff\xff\xf0\xff\xff\xff\xef\xa8\xff\xff\xff\x00\x01\xff\xff\xff\x25\xff\xff\xff\xf6\x82\xff\xff\xff\xef\x04\xff\xff\xff\xf0\xff\xff\xff\xef\xff\xff\xff\x00\xff\xff\xff\x25\xff\xff\xff\xf6\x82\xff\xff\xff\xef\x04\xff\xff\xff\xf0\xff\xff\xff\xef\xff\xff\xff\x00\xff\xff\xff\x25\xff\xff\xff\xf6\x82\xff\xff\xff\xef\x01\xff\xff\xff\xf0\xff\xff\xff\xef\xb0\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x05\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x2c\xff\xff\xff\xe8\xff\xff\xff\x07\xa4\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\xb0\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x01\xff\xff\xff\xb6\xff\xff\xff\x68\x86\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x84\xff\xff\xff\x00\x04\xff\xff\xff\x0d\xff\xff\xff\xa8\xff\xff\xff\xf1\xff\xff\xff\xea\xff\xff\xff\x88\x8a\xff\xff\xff\x00\x04\xff\xff\xff\x28\xff\xff\xff\xcc\xff\xff\xff\xf3\xff\xff\xff\xb8\xff\xff\xff\x12\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x89\xff\xff\xff\x00\x04\xff\xff\xff\x5f\xff\xff\xff\xe6\xff\xff\xff\xf3\xff\xff\xff\xc5\xff\xff\xff\x32\x82\xff\xff\xff\x00\x02\xff\xff\xff\xf3\xff\xff\xff\xf4\xff\xff\xff\xff\x81\xff\xff\xff\xf3\x81\xff\xff\xff\x00\x05\xff\xff\xff\x27\xff\xff\xff\xd7\xff\xff\xff\x01\xff\xff\xff\xe0\xff\xff\xff\xf0\xff\xff\xff\x5e\x81\xff\xff\xff\x00\x03\xff\xff\xff\x47\xff\xff\xff\xf3\xff\xff\xff\xf4\xff\xff\xff\xff\x81\xff\xff\xff\xf3\x00\xff\xff\xff\x1d\x88\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x02\xff\xff\xff\x40\xff\xff\xff\xdb\xff\xff\xff\x01\x85\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x84\xff\xff\xff\x00\x05\xff\xff\xff\x99\xff\xff\xff\xaa\xff\xff\xff\x08\xff\xff\xff\x12\xff\xff\xff\xcb\xff\xff\xff\x65\x89\xff\xff\xff\x00\x04\xff\xff\xff\xc7\xff\xff\xff\x75\xff\xff\xff\x02\xff\xff\xff\x9f\xff\xff\xff\x99\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x88\xff\xff\xff\x00\x05\xff\xff\xff\x0e\xff\xff\xff\xf9\xff\xff\xff\x49\xff\xff\xff\x00\xff\xff\xff\x3f\xff\xff\xff\x4a\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x83\xff\xff\xff\x00\x05\xff\xff\xff\x27\xff\xff\xff\xd7\xff\xff\xff\x00\xff\xff\xff\x17\xff\xff\xff\x51\xff\xff\xff\xdf\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x8b\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x01\xff\xff\xff\xc9\xff\xff\xff\x54\x85\xff\xff\xff\x00\x01\xff\xff\xff\x01\xff\xff\xff\x0e\x84\xff\xff\xff\x00\x01\xff\xff\xff\xf0\xff\xff\xff\x2e\x81\xff\xff\xff\x00\x01\xff\xff\xff\x5b\xff\xff\xff\xbf\x88\xff\xff\xff\x00\x05\xff\xff\xff\x11\xff\xff\xff\xff\xff\xff\xff\x10\xff\xff\xff\x00\xff\xff\xff\x3d\xff\xff\xff\xe1\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x88\xff\xff\xff\x00\x02\xff\xff\xff\x1a\xff\xff\xff\xff\xff\xff\xff\x0d\x86\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xd7\x81\xff\xff\xff\x00\x01\xff\xff\xff\x2f\xff\xff\xff\xf6\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x8b\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x01\xff\xff\xff\x54\xff\xff\xff\xca\x85\xff\xff\xff\x00\x01\xff\xff\xff\x26\xff\xff\xff\xfb\x83\xff\xff\xff\x00\x02\xff\xff\xff\x19\xff\xff\xff\xff\xff\xff\xff\x06\x81\xff\xff\xff\x00\x01\xff\xff\xff\x33\xff\xff\xff\xe7\x88\xff\xff\xff\x00\x01\xff\xff\xff\x24\xff\xff\xff\xfd\x81\xff\xff\xff\x00\x01\xff\xff\xff\x2a\xff\xff\xff\xf3\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x89\xff\xff\xff\x00\x02\xff\xff\xff\xac\xff\xff\xff\xa8\xff\xff\xff\x01\x85\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xd7\x81\xff\xff\xff\x00\x01\xff\xff\xff\x7f\xff\xff\xff\xdd\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x8b\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x02\xff\xff\xff\x01\xff\xff\xff\xdb\xff\xff\xff\x40\x84\xff\xff\xff\x00\x02\xff\xff\xff\x0c\xff\xff\xff\xf3\xff\xff\xff\x32\x82\xff\xff\xff\x00\x01\xff\xff\xff\x24\xff\xff\xff\xfc\x82\xff\xff\xff\x00\x01\xff\xff\xff\x29\xff\xff\xff\xf3\x88\xff\xff\xff\x00\x05\xff\xff\xff\x12\xff\xff\xff\xff\xff\xff\xff\x10\xff\xff\xff\x00\xff\xff\xff\x3d\xff\xff\xff\xe1\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x89\xff\xff\xff\x00\x02\xff\xff\xff\x0b\xff\xff\xff\xca\xff\xff\xff\x81\x85\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x83\xff\xff\xff\x00\x05\xff\xff\xff\x27\xff\xff\xff\xd7\xff\xff\xff\x00\xff\xff\xff\x4d\xff\xff\xff\xd1\xff\xff\xff\x19\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x8b\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x83\xff\xff\xff\x00\x01\xff\xff\xff\x68\xff\xff\xff\xb6\x85\xff\xff\xff\x00\x02\xff\xff\xff\x47\xff\xff\xff\xe0\xff\xff\xff\x2e\x81\xff\xff\xff\x00\x02\xff\xff\xff\x1a\xff\xff\xff\xff\xff\xff\xff\x06\x81\xff\xff\xff\x00\x01\xff\xff\xff\x33\xff\xff\xff\xe9\x89\xff\xff\xff\x00\x04\xff\xff\xff\xcc\xff\xff\xff\x76\xff\xff\xff\x02\xff\xff\xff\xa0\xff\xff\xff\x99\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x8a\xff\xff\xff\x00\x01\xff\xff\xff\x36\xff\xff\xff\xe8\x85\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x83\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xd7\xff\xff\xff\x00\xff\xff\xff\x93\xff\xff\xff\x44\x84\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x8b\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x83\xff\xff\xff\x00\x02\xff\xff\xff\x07\xff\xff\xff\xe8\xff\xff\xff\x2c\x85\xff\xff\xff\x00\x01\xff\xff\xff\x58\xff\xff\xff\xd3\x81\xff\xff\xff\x00\x02\xff\xff\xff\x01\xff\xff\xff\xf3\xff\xff\xff\x2f\x81\xff\xff\xff\x00\x01\xff\xff\xff\x5c\xff\xff\xff\xc5\x89\xff\xff\xff\x00\x04\xff\xff\xff\x2c\xff\xff\xff\xcf\xff\xff\xff\xf3\xff\xff\xff\xb9\xff\xff\xff\x12\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x8a\xff\xff\xff\x00\x01\xff\xff\xff\x1f\xff\xff\xff\xf6\x85\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x83\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xd7\xff\xff\xff\x00\xff\xff\xff\x63\xff\xff\xff\x81\x83\xff\xff\xff\x00\x02\xff\xff\xff\xcc\xff\xff\xff\xf4\xff\xff\xff\xfb\x8b\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x84\xff\xff\xff\x00\x01\xff\xff\xff\x7c\xff\xff\xff\xa2\x82\xff\xff\xff\x00\x04\xff\xff\xff\x4f\xff\xff\xff\x25\xff\xff\xff\x00\xff\xff\xff\x69\xff\xff\xff\xe3\x82\xff\xff\xff\x00\x05\xff\xff\xff\xa2\xff\xff\xff\xac\xff\xff\xff\x09\xff\xff\xff\x14\xff\xff\xff\xcd\xff\xff\xff\x72\x90\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x8b\xff\xff\xff\x00\x00\xff\xff\xff\x0a\x84\xff\xff\xff\x00\x02\xff\xff\xff\x07\xff\xff\xff\x2d\xff\xff\xff\xf7\x81\xff\xff\xff\x07\x81\xff\xff\xff\x00\x01\xff\xff\xff\x23\xff\xff\xff\xdd\x81\xff\xff\xff\x00\x01\xff\xff\xff\x86\xff\xff\xff\xcf\x91\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xff\x82\xff\xff\xff\xf3\x04\xff\xff\xff\xf4\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xff\x82\xff\xff\xff\xf3\x01\xff\xff\xff\xf4\xff\xff\xff\xf7\x84\xff\xff\xff\x00\x02\xff\xff\xff\x11\xff\xff\xff\xee\xff\xff\xff\x1d\x81\xff\xff\xff\x00\x04\xff\xff\xff\x4d\xff\xff\xff\xd0\xff\xff\xff\xf5\xff\xff\xff\xdc\xff\xff\xff\x49\x82\xff\xff\xff\x00\x05\xff\xff\xff\x13\xff\xff\xff\xaf\xff\xff\xff\xf1\xff\xff\xff\xec\xff\xff\xff\x96\xff\xff\xff\x03\x90\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xff\x82\xff\xff\xff\xf3\x04\xff\xff\xff\xf4\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xff\x82\xff\xff\xff\xf3\x04\xff\xff\xff\xf4\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xff\x82\xff\xff\xff\xf3\x01\xff\xff\xff\xf4\xff\xff\xff\xf7\x8a\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x84\xff\xff\xff\x00\x84\xff\xff\xff\xff\x81\xff\xff\xff\x00\x05\xff\xff\xff\x06\xff\xff\xff\xef\xff\xff\xff\x39\xff\xff\xff\x00\xff\xff\xff\x7b\xff\xff\xff\xc1\x91\xff\xff\xff\x00\xd0\xff\xff\xff\x00\x00\xff\xff\xff\x21\x82\xff\xff\xff\xf3\x00\xff\xff\xff\x56\x92\xff\xff\xff\x00\x04\xff\xff\xff\x51\xff\xff\xff\xdc\xff\xff\xff\xf5\xff\xff\xff\xce\xff\xff\xff\x2b\x82\xff\xff\xff\x00\x04\xff\xff\xff\x4f\xff\xff\xff\xcc\xff\xff\xff\x02\xff\xff\xff\xf3\xff\xff\xff\x30\x89\xff\xff\xff\x00\xe0\xff\xff\xff\x00\x04\xff\xff\xff\x4f\xff\xff\xff\xd7\xff\xff\xff\x07\xff\xff\xff\xff\xff\xff\xff\x3b\x99\xff\xff\xff\x00\xff\xff\xff\xff\x00\xff\xff\xff\xff\x00\xf7\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x85\xff\xff\xff\x00\x01\xff\xff\xff\x25\xff\xff\xff\xf6\x82\xff\xff\xff\xef\x04\xff\xff\xff\xf0\xff\xff\xff\xef\xff\xff\xff\x00\xff\xff\xff\x25\xff\xff\xff\xf6\x82\xff\xff\xff\xef\x01\xff\xff\xff\xf0\xff\xff\xff\xef\xb0\xff\xff\xff\x00\x01\xff\xff\xff\x25\xff\xff\xff\xf6\x82\xff\xff\xff\xef\x04\xff\xff\xff\xf0\xff\xff\xff\xef\xff\xff\xff\x00\xff\xff\xff\x25\xff\xff\xff\xf6\x82\xff\xff\xff\xef\x01\xff\xff\xff\xf0\xff\xff\xff\xef\x8d\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x98\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x85\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\xb0\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x8a\xff\xff\xff\x00\x05\xff\xff\xff\x25\xff\xff\xff\xf7\xff\xff\xff\xef\xff\xff\xff\xf1\xff\xff\xff\xff\xff\xff\xff\x9c\x97\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x85\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x84\xff\xff\xff\x00\x00\xff\xff\xff\x04\x85\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x81\xff\xff\xff\x00\x01\xff\xff\xff\xa3\xff\xff\xff\xf7\x89\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x89\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x04\xff\xff\xff\x13\xff\xff\xff\xa8\xff\xff\xff\xe9\xff\xff\xff\xa4\xff\xff\xff\x1b\x83\xff\xff\xff\x00\x04\xff\xff\xff\x03\xff\xff\xff\xae\xff\xff\xff\x6a\xff\xff\xff\x27\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x84\xff\xff\xff\xff\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x85\xff\xff\xff\x00\x03\xff\xff\xff\x41\xff\xff\xff\xdf\xff\xff\xff\xe4\xff\xff\xff\xff\x81\xff\xff\xff\xdf\x07\xff\xff\xff\x1b\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\xff\xff\xff\x8e\xff\xff\xff\xf1\xff\xff\xff\xc4\xff\xff\xff\x16\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x01\xff\xff\xff\x21\xff\xff\xff\xd7\x84\xff\xff\xff\x00\x02\xff\xff\xff\xe6\xff\xff\xff\x80\xff\xff\xff\x18\x83\xff\xff\xff\x00\x05\xff\xff\xff\x27\xff\xff\xff\xfb\xff\xff\xff\x00\xff\xff\xff\x08\xff\xff\xff\xeb\xff\xff\xff\xf7\x89\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x89\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x09\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x01\xff\xff\xff\x9a\xff\xff\xff\x39\xff\xff\xff\x04\xff\xff\xff\x67\xff\xff\xff\xe0\xff\xff\xff\x19\x83\xff\xff\xff\x00\x03\xff\xff\xff\x05\xff\xff\xff\xb1\xff\xff\xff\x62\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x02\xff\xff\xff\x07\xff\xff\xff\x2d\xff\xff\xff\xf7\x81\xff\xff\xff\x07\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x87\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x83\xff\xff\xff\x00\x05\xff\xff\xff\x27\xff\xff\xff\xff\xff\xff\xff\x76\xff\xff\xff\x01\xff\xff\xff\x9e\xff\xff\xff\x9b\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x89\xff\xff\xff\x00\x04\xff\xff\xff\x2d\xff\xff\xff\x93\xff\xff\xff\xe9\xff\xff\xff\xa6\xff\xff\xff\x3a\x81\xff\xff\xff\x00\x05\xff\xff\xff\x27\xff\xff\xff\xfb\xff\xff\xff\x00\xff\xff\xff\x54\xff\xff\xff\xaf\xff\xff\xff\xf7\x89\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x89\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x11\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x76\xff\xff\xff\x67\xff\xff\xff\xd8\xff\xff\xff\x00\xff\xff\xff\xb0\xff\xff\xff\x91\xff\xff\xff\x9c\xff\xff\xff\x00\xff\xff\xff\x2a\xff\xff\xff\x2d\xff\xff\xff\x06\xff\xff\xff\x00\xff\xff\xff\x08\xff\xff\xff\xb4\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x83\xff\xff\xff\x00\x05\xff\xff\xff\x27\xff\xff\xff\xff\xff\xff\xff\xef\xff\xff\xff\xf1\xff\xff\xff\xca\xff\xff\xff\x35\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x83\xff\xff\xff\x00\x05\xff\xff\xff\x27\xff\xff\xff\xff\xff\xff\xff\x10\xff\xff\xff\x00\xff\xff\xff\x3d\xff\xff\xff\xe1\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x8b\xff\xff\xff\x00\x0a\xff\xff\xff\x09\xff\xff\xff\x5f\xff\xff\xff\xca\xff\xff\xff\xcd\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\xff\xff\xff\x00\xff\xff\xff\xac\xff\xff\xff\x57\xff\xff\xff\xf7\x89\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x08\xff\xff\xff\x7a\xff\xff\xff\x04\xff\xff\xff\x33\xff\xff\xff\xb6\xff\xff\xff\xeb\xff\xff\xff\x7a\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x12\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x0d\xff\xff\xff\xf6\xff\xff\xff\x32\xff\xff\xff\xf7\xff\xff\xff\x48\xff\xff\xff\x5a\xff\xff\xff\x35\xff\xff\xff\xe5\xff\xff\xff\x00\xff\xff\xff\x2a\xff\xff\xff\x6b\xff\xff\xff\x98\xff\xff\xff\xa7\xff\xff\xff\xa9\xff\xff\xff\x86\xff\xff\xff\x5d\xff\xff\xff\x2a\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x83\xff\xff\xff\x00\x05\xff\xff\xff\x27\xff\xff\xff\xfb\xff\xff\xff\x00\xff\xff\xff\x04\xff\xff\xff\x87\xff\xff\xff\xce\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfd\x81\xff\xff\xff\x00\x01\xff\xff\xff\x2a\xff\xff\xff\xf3\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x8b\xff\xff\xff\x00\x0a\xff\xff\xff\x26\xff\xff\xff\x8e\xff\xff\xff\xe5\xff\xff\xff\xa1\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\xff\xff\xff\x0c\xff\xff\xff\xce\xff\xff\xff\x27\xff\xff\xff\xf7\x89\xff\xff\xff\x00\x05\xff\xff\xff\x27\xff\xff\xff\xff\xff\xff\xff\x0e\xff\xff\xff\x00\xff\xff\xff\x2e\xff\xff\xff\xf3\x82\xff\xff\xff\x00\x08\xff\xff\xff\x81\xff\xff\xff\xe9\xff\xff\xff\xcb\xff\xff\xff\x50\xff\xff\xff\x13\xff\xff\xff\x83\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x12\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x1f\xff\xff\xff\xfd\xff\xff\xff\x27\xff\xff\xff\xff\xff\xff\xff\xed\xff\xff\xff\x63\xff\xff\xff\x33\xff\xff\xff\xeb\xff\xff\xff\x00\xff\xff\xff\x26\xff\xff\xff\xd6\xff\xff\xff\xf0\xff\xff\xff\xd1\xff\xff\xff\x4d\xff\xff\xff\x2e\xff\xff\xff\x5f\xff\xff\xff\x5d\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x81\xff\xff\xff\x00\x01\xff\xff\xff\x31\xff\xff\xff\xf2\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x83\xff\xff\xff\x00\x05\xff\xff\xff\x27\xff\xff\xff\xff\xff\xff\xff\x10\xff\xff\xff\x00\xff\xff\xff\x3d\xff\xff\xff\xe0\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x89\xff\xff\xff\x00\x04\xff\xff\xff\x57\xff\xff\xff\xbf\xff\xff\xff\xe3\xff\xff\xff\x7b\xff\xff\xff\x15\x81\xff\xff\xff\x00\x05\xff\xff\xff\x27\xff\xff\xff\xfb\xff\xff\xff\x5c\xff\xff\xff\x7f\xff\xff\xff\x27\xff\xff\xff\xf7\x89\xff\xff\xff\x00\x05\xff\xff\xff\x27\xff\xff\xff\xff\xff\xff\xff\x78\xff\xff\xff\x01\xff\xff\xff\x7c\xff\xff\xff\xce\x89\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x0f\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x01\xff\xff\xff\xe9\xff\xff\xff\x3a\xff\xff\xff\xf7\xff\xff\xff\x38\xff\xff\xff\xe9\xff\xff\xff\x79\xff\xff\xff\xa8\xff\xff\xff\x00\xff\xff\xff\x10\xff\xff\xff\x17\xff\xff\xff\x00\xff\xff\xff\x6a\xff\xff\xff\xe6\x84\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x83\xff\xff\xff\x00\x05\xff\xff\xff\x27\xff\xff\xff\xfb\xff\xff\xff\x00\xff\xff\xff\x04\xff\xff\xff\x89\xff\xff\xff\xce\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x83\xff\xff\xff\x00\x05\xff\xff\xff\x27\xff\xff\xff\xff\xff\xff\xff\x78\xff\xff\xff\x02\xff\xff\xff\x9f\xff\xff\xff\x98\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x89\xff\xff\xff\x00\x02\xff\xff\xff\xc4\xff\xff\xff\x56\xff\xff\xff\x04\x83\xff\xff\xff\x00\x02\xff\xff\xff\x27\xff\xff\xff\xfb\xff\xff\xff\xb3\x81\xff\xff\xff\x27\x00\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x0d\xff\xff\xff\x2e\xff\xff\xff\xcd\xff\xff\xff\x1b\xff\xff\xff\x00\xff\xff\xff\x07\xff\xff\xff\xb4\xff\xff\xff\x5a\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\xff\xff\xff\x6e\xff\xff\xff\xe8\xff\xff\xff\xe7\xff\xff\xff\x47\x89\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x09\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x65\xff\xff\xff\xa0\xff\xff\xff\xff\xff\xff\xff\xe6\xff\xff\xff\xa2\xff\xff\xff\xda\xff\xff\xff\x1f\x82\xff\xff\xff\x00\x02\xff\xff\xff\x01\xff\xff\xff\x6c\xff\xff\xff\xd4\x84\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x83\xff\xff\xff\x00\x05\xff\xff\xff\x27\xff\xff\xff\xff\xff\xff\xff\xef\xff\xff\xff\xf0\xff\xff\xff\xc9\xff\xff\xff\x34\x82\xff\xff\xff\x00\x02\xff\xff\xff\xbc\xff\xff\xff\xe4\xff\xff\xff\xfb\x83\xff\xff\xff\x00\x05\xff\xff\xff\x27\xff\xff\xff\xfb\xff\xff\xff\x8d\xff\xff\xff\xf1\xff\xff\xff\xc3\xff\xff\xff\x14\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x90\xff\xff\xff\x00\x05\xff\xff\xff\x27\xff\xff\xff\xfd\xff\xff\xff\xc9\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x04\xff\xff\xff\x42\xff\xff\xff\xd5\xff\xff\xff\x25\xff\xff\xff\xb8\xff\xff\xff\x75\x91\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x04\xff\xff\xff\x61\xff\xff\xff\xd3\xff\xff\xff\xe7\xff\xff\xff\x99\xff\xff\xff\x15\x82\xff\xff\xff\x00\x03\xff\xff\xff\x42\xff\xff\xff\xf7\xff\xff\xff\xe5\xff\xff\xff\x37\x83\xff\xff\xff\x00\x02\xff\xff\xff\x03\xff\xff\xff\x2a\xff\xff\xff\xf7\x81\xff\xff\xff\x03\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x8d\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x85\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xff\x82\xff\xff\xff\xf3\x04\xff\xff\xff\xf4\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xff\x82\xff\xff\xff\xf3\x01\xff\xff\xff\xf4\xff\xff\xff\xf7\x90\xff\xff\xff\x00\x05\xff\xff\xff\x27\xff\xff\xff\xff\xff\xff\xff\x77\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xf7\x83\xff\xff\xff\x00\x02\xff\xff\xff\x5a\xff\xff\xff\xf9\xff\xff\xff\x90\x92\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xff\x82\xff\xff\xff\xf3\x04\xff\xff\xff\xf4\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xff\x82\xff\xff\xff\xf3\x01\xff\xff\xff\xf4\xff\xff\xff\xf7\x88\xff\xff\xff\x00\x04\xff\xff\xff\x01\xff\xff\xff\x11\xff\xff\xff\x00\xff\xff\xff\x60\xff\xff\xff\xe4\x83\xff\xff\xff\x00\x84\xff\xff\xff\xff\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x86\xff\xff\xff\x00\x04\xff\xff\xff\x40\xff\xff\xff\x9e\xff\xff\xff\x1e\xff\xff\xff\xb7\xff\xff\xff\x06\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x85\xff\xff\xff\x00\xd7\xff\xff\xff\x00\x04\xff\xff\xff\x09\xff\xff\xff\xce\xff\xff\xff\xed\xff\xff\xff\xde\xff\xff\xff\x72\x83\xff\xff\xff\x00\x04\xff\xff\xff\x03\xff\xff\xff\x04\xff\xff\xff\x00\xff\xff\xff\x06\xff\xff\xff\x02\x8b\xff\xff\xff\x00\x02\xff\xff\xff\x84\xff\xff\xff\xe0\xff\xff\xff\x26\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x85\xff\xff\xff\x00\xe0\xff\xff\xff\x00\x04\xff\xff\xff\x16\xff\xff\xff\xbf\xff\xff\xff\x1c\xff\xff\xff\xc9\xff\xff\xff\x0c\x99\xff\xff\xff\x00\xe1\xff\xff\xff\x00\x02\xff\xff\xff\x69\xff\xff\xff\xea\xff\xff\xff\x44\x9a\xff\xff\xff\x00\xff\xff\xff\xff\x00\xf8\xff\xff\xff\x00\x02\xff\xff\xff\xdd\xff\xff\xff\xe3\xff\xff\xff\x39\x83\xff\xff\xff\x00\x01\xff\xff\xff\x25\xff\xff\xff\xf6\x82\xff\xff\xff\xef\x04\xff\xff\xff\xf0\xff\xff\xff\xef\xff\xff\xff\x00\xff\xff\xff\x25\xff\xff\xff\xf6\x82\xff\xff\xff\xef\x01\xff\xff\xff\xf0\xff\xff\xff\xef\xa9\xff\xff\xff\x00\x02\xff\xff\xff\xd4\xff\xff\xff\xde\xff\xff\xff\x5f\x83\xff\xff\xff\x00\x01\xff\xff\xff\x25\xff\xff\xff\xf6\x82\xff\xff\xff\xef\x04\xff\xff\xff\xf0\xff\xff\xff\xef\xff\xff\xff\x00\xff\xff\xff\x25\xff\xff\xff\xf6\x82\xff\xff\xff\xef\x01\xff\xff\xff\xf0\xff\xff\xff\xef\x8b\xff\xff\xff\x00\x01\xff\xff\xff\x95\xff\xff\xff\xf4\x81\xff\xff\xff\xf3\x9a\xff\xff\xff\x00\x01\xff\xff\xff\x69\xff\xff\xff\xc9\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x9a\xff\xff\xff\x00\x02\xff\xff\xff\x86\xff\xff\xff\xf0\xff\xff\xff\xf3\x8c\xff\xff\xff\x00\x01\xff\xff\xff\x5d\xff\xff\xff\xe1\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x8b\xff\xff\xff\x00\x02\xff\xff\xff\x1e\xff\xff\xff\xa3\xff\xff\xff\x24\x9b\xff\xff\xff\x00\x02\xff\xff\xff\x06\xff\xff\xff\xf2\xff\xff\xff\x26\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x90\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x05\xff\xff\xff\x27\xff\xff\xff\xfb\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x27\xff\xff\xff\xf7\x83\xff\xff\xff\x00\x01\xff\xff\xff\x28\xff\xff\xff\xf9\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x8c\xff\xff\xff\x00\x02\xff\xff\xff\x09\xff\xff\xff\xa2\xff\xff\xff\x3b\x81\xff\xff\xff\x00\x02\xff\xff\xff\xf3\xff\xff\xff\xf4\xff\xff\xff\xff\x81\xff\xff\xff\xf3\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x83\xff\xff\xff\x00\x03\xff\xff\xff\x41\xff\xff\xff\xdf\xff\xff\xff\xe4\xff\xff\xff\xff\x81\xff\xff\xff\xdf\x00\xff\xff\xff\x1b\x82\xff\xff\xff\x00\x02\xff\xff\xff\x1a\xff\xff\xff\xfc\xff\xff\xff\x7d\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x90\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x05\xff\xff\xff\x27\xff\xff\xff\xfb\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x27\xff\xff\xff\xf7\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x8b\xff\xff\xff\x00\x03\xff\xff\xff\x1c\xff\xff\xff\x03\xff\xff\xff\x4b\xff\xff\xff\xe2\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x85\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x85\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x85\xff\xff\xff\x00\x02\xff\xff\xff\x72\xff\xff\xff\xd4\xff\xff\xff\xd5\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x89\xff\xff\xff\x00\x85\xff\xff\xff\xf3\x02\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x05\xff\xff\xff\x27\xff\xff\xff\xfb\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x27\xff\xff\xff\xf7\x83\xff\xff\xff\x00\x02\xff\xff\xff\x24\xff\xff\xff\xfd\xff\xff\xff\x01\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x88\xff\xff\xff\x00\x06\xff\xff\xff\x6d\xff\xff\xff\x4c\xff\xff\xff\x05\xff\xff\xff\x5b\xff\xff\xff\xe1\xff\xff\xff\xe6\xff\xff\xff\x79\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x85\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x85\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x85\xff\xff\xff\x00\x03\xff\xff\xff\xcd\xff\xff\xff\x4a\xff\xff\xff\xe2\xff\xff\xff\x2e\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x82\xff\xff\xff\xf3\x8a\xff\xff\xff\x00\x06\xff\xff\xff\x27\xff\xff\xff\xfb\xff\xff\xff\x00\xff\xff\xff\x7f\xff\xff\xff\x5c\xff\xff\xff\x27\xff\xff\xff\xf7\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x05\xff\xff\xff\x27\xff\xff\xff\xfb\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x27\xff\xff\xff\xf6\x83\xff\xff\xff\x00\x02\xff\xff\xff\x08\xff\xff\xff\xf2\xff\xff\xff\x4c\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x82\xff\xff\xff\xf3\x82\xff\xff\xff\x00\x05\xff\xff\xff\x4a\xff\xff\xff\xa9\xff\xff\xff\xe2\xff\xff\xff\x9b\xff\xff\xff\x43\xff\xff\xff\x02\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x85\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x85\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x84\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xec\xff\xff\xff\x04\xff\xff\xff\x8f\xff\xff\xff\x87\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x90\xff\xff\xff\x00\x06\xff\xff\xff\x27\xff\xff\xff\xfb\xff\xff\xff\x01\xff\xff\xff\xd0\xff\xff\xff\xbb\xff\xff\xff\x27\xff\xff\xff\xf7\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x05\xff\xff\xff\x27\xff\xff\xff\xfc\xff\xff\xff\x29\xff\xff\xff\xf8\xff\xff\xff\x29\xff\xff\xff\xf2\x84\xff\xff\xff\x00\x02\xff\xff\xff\x70\xff\xff\xff\xff\xff\xff\xff\xc5\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x8a\xff\xff\xff\x00\x05\xff\xff\xff\x09\xff\xff\xff\x52\xff\xff\xff\xa6\xff\xff\xff\xdb\xff\xff\xff\x93\xff\xff\xff\x08\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x85\xff\xff\xff\x00\x02\xff\xff\xff\x6a\xff\xff\xff\xff\xff\xff\xff\x3f\x84\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x84\xff\xff\xff\x00\x04\xff\xff\xff\x81\xff\xff\xff\x9a\xff\xff\xff\x00\xff\xff\xff\x39\xff\xff\xff\xdf\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x89\xff\xff\xff\x00\x85\xff\xff\xff\xef\x07\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\xff\xff\xff\x3c\xff\xff\xff\x8f\xff\xff\xff\xbb\xff\xff\xff\x3b\xff\xff\xff\xf7\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x05\xff\xff\xff\x27\xff\xff\xff\xff\xff\xff\xff\x50\xff\xff\xff\xff\xff\xff\xff\x4d\xff\xff\xff\xe1\x83\xff\xff\xff\x00\x02\xff\xff\xff\x0e\xff\xff\xff\xf9\xff\xff\xff\x3e\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x88\xff\xff\xff\x00\x07\xff\xff\xff\x3b\xff\xff\xff\xad\xff\xff\xff\xd2\xff\xff\xff\xa7\xff\xff\xff\x1e\xff\xff\xff\x07\xff\xff\xff\x4e\xff\xff\xff\x01\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x84\xff\xff\xff\x00\x04\xff\xff\xff\x0f\xff\xff\xff\xe8\xff\xff\xff\x82\xff\xff\xff\xcd\xff\xff\xff\x01\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x84\xff\xff\xff\x00\x01\xff\xff\xff\xdb\xff\xff\xff\x42\x81\xff\xff\xff\x00\x02\xff\xff\xff\xe2\xff\xff\xff\x39\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x90\xff\xff\xff\x00\x06\xff\xff\xff\x27\xff\xff\xff\xfb\xff\xff\xff\x8d\xff\xff\xff\x3d\xff\xff\xff\x6d\xff\xff\xff\x88\xff\xff\xff\xf7\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x05\xff\xff\xff\x27\xff\xff\xff\xfc\xff\xff\xff\xcf\xff\xff\xff\xa1\xff\xff\xff\xdc\xff\xff\xff\x8f\x83\xff\xff\xff\x00\x01\xff\xff\xff\x26\xff\xff\xff\xfc\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x89\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x86\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x84\xff\xff\xff\x00\x04\xff\xff\xff\x8b\xff\xff\xff\xa3\xff\xff\xff\x00\xff\xff\xff\xcc\xff\xff\xff\x60\x82\xff\xff\xff\x00\x02\xff\xff\xff\xbc\xff\xff\xff\xe4\xff\xff\xff\xfb\x83\xff\xff\xff\x00\x02\xff\xff\xff\x36\xff\xff\xff\xe7\xff\xff\xff\x02\x81\xff\xff\xff\x00\x02\xff\xff\xff\x8e\xff\xff\xff\x92\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x90\xff\xff\xff\x00\x06\xff\xff\xff\x27\xff\xff\xff\xfb\xff\xff\xff\xc8\xff\xff\xff\x01\xff\xff\xff\x1b\xff\xff\xff\xd8\xff\xff\xff\xf7\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x8c\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfa\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x89\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x85\xff\xff\xff\x00\x02\xff\xff\xff\x07\xff\xff\xff\x2d\xff\xff\xff\xf7\x81\xff\xff\xff\x07\x81\xff\xff\xff\x00\x06\xff\xff\xff\x21\xff\xff\xff\xf2\xff\xff\xff\x20\xff\xff\xff\x00\xff\xff\xff\x46\xff\xff\xff\xe5\xff\xff\xff\x0b\x8a\xff\xff\xff\x00\x02\xff\xff\xff\x0a\xff\xff\xff\xd1\xff\xff\xff\x13\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xff\x82\xff\xff\xff\xf3\x04\xff\xff\xff\xf4\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xff\x82\xff\xff\xff\xf3\x01\xff\xff\xff\xf4\xff\xff\xff\xf7\x90\xff\xff\xff\x00\x02\xff\xff\xff\x27\xff\xff\xff\xff\xff\xff\xff\x9b\x81\xff\xff\xff\x00\x01\xff\xff\xff\xc8\xff\xff\xff\xf7\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x8c\xff\xff\xff\x00\x01\xff\xff\xff\x51\xff\xff\xff\xec\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xff\x82\xff\xff\xff\xf3\x04\xff\xff\xff\xf4\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xff\x82\xff\xff\xff\xf3\x01\xff\xff\xff\xf4\xff\xff\xff\xf7\x89\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x85\xff\xff\xff\x00\x84\xff\xff\xff\xff\x81\xff\xff\xff\x00\x01\xff\xff\xff\xab\xff\xff\xff\x93\x82\xff\xff\xff\x00\x01\xff\xff\xff\xbe\xff\xff\xff\x82\x82\xff\xff\xff\x00\x02\xff\xff\xff\x2d\xff\xff\xff\xbe\xff\xff\xff\x0e\x85\xff\xff\xff\x00\x01\xff\xff\xff\x63\xff\xff\xff\xa2\x82\xff\xff\xff\x00\xa9\xff\xff\xff\x00\x02\xff\xff\xff\x88\xff\xff\xff\xf4\xff\xff\xff\xfb\x8b\xff\xff\xff\x00\x02\xff\xff\xff\xd8\xff\xff\xff\xe9\xff\xff\xff\x72\x9b\xff\xff\xff\x00\x02\xff\xff\xff\x50\xff\xff\xff\xb5\xff\xff\xff\xfb\x86\xff\xff\xff\x00\x01\xff\xff\xff\x03\xff\xff\xff\x05\x8e\xff\xff\xff\x00\x02\xff\xff\xff\x48\xff\xff\xff\xc2\xff\xff\xff\x0b\x84\xff\xff\xff\x00\x02\xff\xff\xff\x02\xff\xff\xff\xd2\xff\xff\xff\x45\x81\xff\xff\xff\x00\xd9\xff\xff\xff\x00\x00\xff\xff\xff\x06\x86\xff\xff\xff\x00\x02\xff\xff\xff\x1e\xff\xff\xff\xcc\xff\xff\xff\x06\x84\xff\xff\xff\x00\x02\xff\xff\xff\x20\xff\xff\xff\xb8\xff\xff\xff\x01\x92\xff\xff\xff\x00\xe2\xff\xff\xff\x00\x01\xff\xff\xff\x8c\xff\xff\xff\x8b\x85\xff\xff\xff\x00\x01\xff\xff\xff\x8d\xff\xff\xff\x77\x92\xff\xff\xff\x00\xff\xff\xff\xff\x00\xb9\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\xc3\xff\xff\xff\x00\x01\xff\xff\xff\x25\xff\xff\xff\xf6\x82\xff\xff\xff\xef\x04\xff\xff\xff\xf0\xff\xff\xff\xef\xff\xff\xff\x00\xff\xff\xff\x25\xff\xff\xff\xf6\x82\xff\xff\xff\xef\x01\xff\xff\xff\xf0\xff\xff\xff\xef\x82\xff\xff\xff\x00\x01\xff\xff\xff\x9e\xff\xff\xff\x42\xa5\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x83\xff\xff\xff\x00\x01\xff\xff\xff\x25\xff\xff\xff\xf6\x82\xff\xff\xff\xef\x04\xff\xff\xff\xf0\xff\xff\xff\xef\xff\xff\xff\x00\xff\xff\xff\x25\xff\xff\xff\xf6\x82\xff\xff\xff\xef\x01\xff\xff\xff\xf0\xff\xff\xff\xef\x8c\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\xa1\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x01\xff\xff\xff\x65\xff\xff\xff\x9a\x97\xff\xff\xff\x00\x02\xff\xff\xff\x11\xff\xff\xff\xee\xff\xff\xff\x1c\x8a\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x8a\xff\xff\xff\x00\x04\xff\xff\xff\x02\xff\xff\xff\xf3\xff\xff\xff\xf1\xff\xff\xff\xff\xff\xff\xff\x9c\xa0\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x01\xff\xff\xff\x31\xff\xff\xff\xeb\x89\xff\xff\xff\x00\x02\xff\xff\xff\x04\xff\xff\xff\x00\xff\xff\xff\x27\x84\xff\xff\xff\xff\x00\xff\xff\xff\x33\x84\xff\xff\xff\x00\x01\xff\xff\xff\x7c\xff\xff\xff\xa2\x84\xff\xff\xff\x00\x03\xff\xff\xff\x68\xff\xff\xff\xef\xff\xff\xff\xf3\xff\xff\xff\x47\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x8b\xff\xff\xff\x00\x02\xff\xff\xff\x8d\xff\xff\xff\x29\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x02\xff\xff\xff\xf3\xff\xff\xff\xf4\xff\xff\xff\xff\x81\xff\xff\xff\xf3\x82\xff\xff\xff\x00\x04\xff\xff\xff\x46\xff\xff\xff\xdd\xff\xff\xff\xf4\xff\xff\xff\xc6\xff\xff\xff\x25\x81\xff\xff\xff\x00\x03\xff\xff\xff\x47\xff\xff\xff\xf3\xff\xff\xff\xf4\xff\xff\xff\xff\x81\xff\xff\xff\xf3\x00\xff\xff\xff\x1d\x81\xff\xff\xff\x00\x04\xff\xff\xff\x6d\xff\xff\xff\xf1\xff\xff\xff\xde\xff\xff\xff\x74\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x87\xff\xff\xff\x00\x05\xff\xff\xff\x18\xff\xff\xff\x80\xff\xff\xff\xe6\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x83\xff\xff\xff\x07\x84\xff\xff\xff\x00\x02\xff\xff\xff\x07\xff\xff\xff\xe8\xff\xff\xff\x2c\x83\xff\xff\xff\x00\x02\xff\xff\xff\x08\xff\xff\xff\xf6\xff\xff\xff\x3c\x84\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x8b\xff\xff\xff\x00\x02\xff\xff\xff\x29\xff\xff\xff\x86\xff\xff\xff\xf7\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x83\xff\xff\xff\x00\x05\xff\xff\xff\x02\xff\xff\xff\xe9\xff\xff\xff\x47\xff\xff\xff\x00\xff\xff\xff\x76\xff\xff\xff\xbe\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x83\xff\xff\xff\x00\x05\xff\xff\xff\x07\xff\xff\xff\xf6\xff\xff\xff\x50\xff\xff\xff\x04\xff\xff\xff\xa0\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x8a\xff\xff\xff\x00\x07\xff\xff\xff\x3a\xff\xff\xff\xa7\xff\xff\xff\xea\xff\xff\xff\x93\xff\xff\xff\x2d\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x88\xff\xff\xff\x00\x01\xff\xff\xff\x68\xff\xff\xff\xb6\x84\xff\xff\xff\x00\x01\xff\xff\xff\x24\xff\xff\xff\xfc\x85\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x85\xff\xff\xff\x00\x04\xff\xff\xff\x25\xff\xff\xff\xf3\xff\xff\xff\x00\xff\xff\xff\x2a\xff\xff\xff\x06\x81\xff\xff\xff\x00\x01\xff\xff\xff\x88\xff\xff\xff\xf7\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x83\xff\xff\xff\x00\x01\xff\xff\xff\x1d\xff\xff\xff\xfd\x81\xff\xff\xff\x00\x01\xff\xff\xff\x2a\xff\xff\xff\xed\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x83\xff\xff\xff\x00\x05\xff\xff\xff\x23\xff\xff\xff\xfd\xff\xff\xff\x03\xff\xff\xff\x00\xff\xff\xff\x3a\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x89\xff\xff\xff\x00\x03\xff\xff\xff\xcd\xff\xff\xff\xca\xff\xff\xff\x5f\xff\xff\xff\x09\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x87\xff\xff\xff\x00\x02\xff\xff\xff\x01\xff\xff\xff\xdb\xff\xff\xff\x40\x84\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x85\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x85\xff\xff\xff\x00\x07\xff\xff\xff\x27\xff\xff\xff\xfb\xff\xff\xff\x00\xff\xff\xff\x67\xff\xff\xff\xc1\xff\xff\xff\xa8\xff\xff\xff\x57\xff\xff\xff\x0d\x84\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x83\xff\xff\xff\x00\x01\xff\xff\xff\x26\xff\xff\xff\xfb\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf6\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x89\xff\xff\xff\x00\x03\xff\xff\xff\xa1\xff\xff\xff\xe5\xff\xff\xff\x8e\xff\xff\xff\x26\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x87\xff\xff\xff\x00\x01\xff\xff\xff\x54\xff\xff\xff\xc9\x85\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x85\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x03\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x56\x83\xff\xff\xff\xf3\x01\xff\xff\xff\xf4\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x04\xff\xff\xff\x22\xff\xff\xff\x74\xff\xff\xff\xc0\xff\xff\xff\xb5\xff\xff\xff\x53\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x8a\xff\xff\xff\x00\x07\xff\xff\xff\x15\xff\xff\xff\x7c\xff\xff\xff\xe3\xff\xff\xff\xbf\xff\xff\xff\x57\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x87\xff\xff\xff\x00\x01\xff\xff\xff\xca\xff\xff\xff\x54\x85\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x85\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x88\xff\xff\xff\x00\x06\xff\xff\xff\x3b\xff\xff\xff\xad\xff\xff\xff\xd2\xff\xff\xff\xa1\xff\xff\xff\x00\xff\xff\xff\x1b\xff\xff\xff\x3f\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x8c\xff\xff\xff\x00\x05\xff\xff\xff\x04\xff\xff\xff\x56\xff\xff\xff\xc4\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x86\xff\xff\xff\x00\x02\xff\xff\xff\x40\xff\xff\xff\xdb\xff\xff\xff\x01\x85\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x85\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x89\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x86\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x02\xff\xff\xff\xbc\xff\xff\xff\xe4\xff\xff\xff\xfb\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x90\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x86\xff\xff\xff\x00\x01\xff\xff\xff\xb6\xff\xff\xff\x68\x86\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x85\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x89\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x85\xff\xff\xff\x00\x02\xff\xff\xff\x07\xff\xff\xff\x2d\xff\xff\xff\xf7\x81\xff\xff\xff\x07\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x91\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xff\x82\xff\xff\xff\xf3\x04\xff\xff\xff\xf4\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xff\x82\xff\xff\xff\xf3\x01\xff\xff\xff\xf4\xff\xff\xff\xf7\x90\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x85\xff\xff\xff\x00\x02\xff\xff\xff\x2c\xff\xff\xff\xe8\xff\xff\xff\x07\x86\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x85\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xff\x82\xff\xff\xff\xf3\x04\xff\xff\xff\xf4\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xff\x82\xff\xff\xff\xf3\x01\xff\xff\xff\xf4\xff\xff\xff\xf7\x89\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x85\xff\xff\xff\x00\x84\xff\xff\xff\xff\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x83\xff\xff\xff\x00\x02\xff\xff\xff\x48\xff\xff\xff\xaf\xff\xff\xff\x02\x8a\xff\xff\xff\x00\xb0\xff\xff\xff\x00\x02\xff\xff\xff\xd8\xff\xff\xff\xf4\xff\xff\xff\xfb\x85\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x9b\xff\xff\xff\x00\x02\xff\xff\xff\x4f\xff\xff\xff\xb4\xff\xff\xff\xfb\x87\xff\xff\xff\x00\x01\xff\xff\xff\x07\xff\xff\xff\x01\x8b\xff\xff\xff\x00\x02\xff\xff\xff\x41\xff\xff\xff\xc9\xff\xff\xff\x0c\x84\xff\xff\xff\x00\x04\xff\xff\xff\x5e\xff\xff\xff\xcc\xff\xff\xff\x06\xff\xff\xff\xf3\xff\xff\xff\x1d\x81\xff\xff\xff\x00\xe1\xff\xff\xff\x00\x02\xff\xff\xff\x14\xff\xff\xff\xd0\xff\xff\xff\x08\x83\xff\xff\xff\x00\x04\xff\xff\xff\x2c\xff\xff\xff\xc9\xff\xff\xff\x06\xff\xff\xff\xef\xff\xff\xff\x02\x91\xff\xff\xff\x00\xe1\xff\xff\xff\x00\x01\xff\xff\xff\xa5\xff\xff\xff\x61\x9b\xff\xff\xff\x00\xff\xff\xff\xff\x00\xff\xff\xff\xff\x00\x01\xff\xff\xff\x25\xff\xff\xff\xf6\x82\xff\xff\xff\xef\x04\xff\xff\xff\xf0\xff\xff\xff\xef\xff\xff\xff\x00\xff\xff\xff\x25\xff\xff\xff\xf6\x82\xff\xff\xff\xef\x01\xff\xff\xff\xf0\xff\xff\xff\xef\xab\xff\xff\xff\x00\x02\xff\xff\xff\x81\xff\xff\xff\xe5\xff\xff\xff\xab\x81\xff\xff\xff\x00\x01\xff\xff\xff\x25\xff\xff\xff\xf6\x82\xff\xff\xff\xef\x04\xff\xff\xff\xf0\xff\xff\xff\xef\xff\xff\xff\x00\xff\xff\xff\x25\xff\xff\xff\xf6\x82\xff\xff\xff\xef\x01\xff\xff\xff\xf0\xff\xff\xff\xef\xb0\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x8a\xff\xff\xff\x00\x01\xff\xff\xff\x99\xff\xff\xff\x4b\x8d\xff\xff\xff\x00\x03\xff\xff\xff\x25\xff\xff\xff\xf6\xff\xff\xff\xef\xff\xff\xff\x5c\x8b\xff\xff\xff\x00\x02\xff\xff\xff\x0f\xff\xff\xff\xfd\xff\xff\xff\x2c\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\xb0\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x8a\xff\xff\xff\x00\x01\xff\xff\xff\x56\xff\xff\xff\xb3\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x02\xff\xff\xff\x94\xff\xff\xff\xcd\xff\xff\xff\x03\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x84\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x81\xff\xff\xff\x00\x02\xff\xff\xff\x28\xff\xff\xff\xf2\xff\xff\xff\x33\x81\xff\xff\xff\x00\x01\xff\xff\xff\x25\xff\xff\xff\xfc\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x83\xff\xff\xff\x00\x02\xff\xff\xff\x02\xff\xff\xff\x00\xff\xff\xff\x01\x82\xff\xff\xff\x00\x00\xff\xff\xff\x01\x85\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xff\x83\xff\xff\xff\xf3\x00\xff\xff\xff\x25\x81\xff\xff\xff\x00\x04\xff\xff\xff\x41\xff\xff\xff\xd9\xff\xff\xff\xf9\xff\xff\xff\xc3\xff\xff\xff\x21\x82\xff\xff\xff\x00\x05\xff\xff\xff\x0b\xff\xff\xff\x96\xff\xff\xff\xe8\xff\xff\xff\xf3\xff\xff\xff\xd9\xff\xff\xff\x6e\x81\xff\xff\xff\x00\x04\xff\xff\xff\x6d\xff\xff\xff\xe7\xff\xff\xff\xc0\xff\xff\xff\x84\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x84\xff\xff\xff\x00\x01\xff\xff\xff\x28\xff\xff\xff\xf9\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x81\xff\xff\xff\x00\x02\xff\xff\xff\x39\xff\xff\xff\xf8\xff\xff\xff\x2e\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x84\xff\xff\xff\x00\x05\xff\xff\xff\x27\xff\xff\xff\xfb\xff\xff\xff\x00\xff\xff\xff\x02\xff\xff\xff\xc9\xff\xff\xff\x7f\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x03\xff\xff\xff\x0b\xff\xff\xff\xb3\xff\xff\xff\x00\xff\xff\xff\x80\x82\xff\xff\xff\x00\x03\xff\xff\xff\xb2\xff\xff\xff\x0b\xff\xff\xff\x00\xff\xff\xff\x7f\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x85\xff\xff\xff\x00\x05\xff\xff\xff\x02\xff\xff\xff\xe8\xff\xff\xff\x56\xff\xff\xff\x06\xff\xff\xff\x82\xff\xff\xff\xba\x82\xff\xff\xff\x00\x0c\xff\xff\xff\xa5\xff\xff\xff\x9b\xff\xff\xff\x0e\xff\xff\xff\x00\xff\xff\xff\x1f\xff\xff\xff\x4b\xff\xff\xff\x00\xff\xff\xff\x07\xff\xff\xff\xf6\xff\xff\xff\x38\xff\xff\xff\x00\xff\xff\xff\x8a\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x84\xff\xff\xff\x00\x01\xff\xff\xff\x0e\xff\xff\xff\x5d\x83\xff\xff\xff\x00\x05\xff\xff\xff\x27\xff\xff\xff\xfb\xff\xff\xff\x00\xff\xff\xff\x04\xff\xff\xff\xd3\xff\xff\xff\x82\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x84\xff\xff\xff\x00\x05\xff\xff\xff\x27\xff\xff\xff\xfb\xff\xff\xff\x02\xff\xff\xff\x79\xff\xff\xff\xcb\xff\xff\xff\x03\x82\xff\xff\xff\x00\x01\xff\xff\xff\x2b\xff\xff\xff\xf7\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x04\xff\xff\xff\x1c\xff\xff\xff\xcd\xff\xff\xff\x72\xff\xff\xff\x4b\xff\xff\xff\x9a\x82\xff\xff\xff\x00\x04\xff\xff\xff\x70\xff\xff\xff\xcc\xff\xff\xff\x1c\xff\xff\xff\x9e\xff\xff\xff\x4a\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x85\xff\xff\xff\x00\x01\xff\xff\xff\x1d\xff\xff\xff\xfd\x81\xff\xff\xff\x00\x01\xff\xff\xff\x2a\xff\xff\xff\xec\x81\xff\xff\xff\x00\x02\xff\xff\xff\x0c\xff\xff\xff\xfb\xff\xff\xff\x0c\x84\xff\xff\xff\x00\x05\xff\xff\xff\x23\xff\xff\xff\xfd\xff\xff\xff\x01\xff\xff\xff\x00\xff\xff\xff\x35\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x8a\xff\xff\xff\x00\x05\xff\xff\xff\x27\xff\xff\xff\xfc\xff\xff\xff\x17\xff\xff\xff\x7b\xff\xff\xff\xd3\xff\xff\xff\x05\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x84\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xff\xff\xff\xff\xb6\xff\xff\xff\xef\xff\xff\xff\x29\x83\xff\xff\xff\x00\x01\xff\xff\xff\x77\xff\xff\xff\xcd\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x04\xff\xff\xff\xe4\xff\xff\xff\x65\xff\xff\xff\x00\xff\xff\xff\xcb\xff\xff\xff\x05\x83\xff\xff\xff\x00\x03\xff\xff\xff\x58\xff\xff\xff\xe4\xff\xff\xff\x08\xff\xff\xff\xd3\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x85\xff\xff\xff\x00\x01\xff\xff\xff\x26\xff\xff\xff\xfb\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf6\x81\xff\xff\xff\x00\x01\xff\xff\xff\x22\xff\xff\xff\xff\x82\xff\xff\xff\xef\x04\xff\xff\xff\xf0\xff\xff\xff\xed\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x81\xff\xff\xff\xef\x01\xff\xff\xff\xf1\xff\xff\xff\xff\x81\xff\xff\xff\xef\x88\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xff\xff\xff\xff\xd3\xff\xff\xff\xf3\xff\xff\xff\x35\x84\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x84\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xfb\xff\xff\xff\x9c\xff\xff\xff\xc5\xff\xff\xff\x02\x82\xff\xff\xff\x00\x02\xff\xff\xff\xed\xff\xff\xff\xf9\xff\xff\xff\x4c\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x04\xff\xff\xff\x55\xff\xff\xff\xd9\xff\xff\xff\x2d\xff\xff\xff\x88\xff\xff\xff\x59\x82\xff\xff\xff\x00\x04\xff\xff\xff\x2b\xff\xff\xff\xd4\xff\xff\xff\x55\xff\xff\xff\x60\xff\xff\xff\x87\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xff\x82\xff\xff\xff\xef\x00\xff\xff\xff\xd1\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x02\xff\xff\xff\x0b\xff\xff\xff\xfb\xff\xff\xff\x25\x81\xff\xff\xff\x00\x04\xff\xff\xff\x3a\xff\xff\xff\xde\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x84\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x83\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xfb\xff\xff\xff\x8c\xff\xff\xff\xda\xff\xff\xff\x09\x84\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x84\xff\xff\xff\x00\x05\xff\xff\xff\x27\xff\xff\xff\xfb\xff\xff\xff\x01\xff\xff\xff\xa2\xff\xff\xff\xa5\xff\xff\xff\x01\x82\xff\xff\xff\x00\x01\xff\xff\xff\x69\xff\xff\xff\xde\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x03\xff\xff\xff\x37\xff\xff\xff\xdb\xff\xff\xff\x02\xff\xff\xff\xb3\x82\xff\xff\xff\x00\x04\xff\xff\xff\xdb\xff\xff\xff\x37\xff\xff\xff\x00\xff\xff\xff\xb3\xff\xff\xff\x02\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x85\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x08\xff\xff\xff\xa2\xff\xff\xff\xb4\xff\xff\xff\x0e\xff\xff\xff\x09\xff\xff\xff\xab\xff\xff\xff\x8b\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x84\xff\xff\xff\x00\x01\xff\xff\xff\x20\xff\xff\xff\xd3\x83\xff\xff\xff\x00\x05\xff\xff\xff\x27\xff\xff\xff\xfb\xff\xff\xff\x00\xff\xff\xff\x9f\xff\xff\xff\xb7\xff\xff\xff\x05\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x84\xff\xff\xff\x00\x06\xff\xff\xff\x27\xff\xff\xff\xfb\xff\xff\xff\x00\xff\xff\xff\x02\xff\xff\xff\xa8\xff\xff\xff\xa0\xff\xff\xff\x01\x81\xff\xff\xff\x00\x01\xff\xff\xff\x28\xff\xff\xff\xfa\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x83\xff\xff\xff\x00\x02\xff\xff\xff\x20\xff\xff\xff\x00\xff\xff\xff\x15\x82\xff\xff\xff\x00\x00\xff\xff\xff\x1f\x81\xff\xff\xff\x00\x00\xff\xff\xff\x14\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x85\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x08\xff\xff\xff\x0a\xff\xff\xff\x98\xff\xff\xff\xea\xff\xff\xff\xef\xff\xff\xff\x9f\xff\xff\xff\x08\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x90\xff\xff\xff\x00\x06\xff\xff\xff\x27\xff\xff\xff\xfb\xff\xff\xff\x00\xff\xff\xff\x03\xff\xff\xff\xb0\xff\xff\xff\xa8\xff\xff\xff\x01\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x84\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x86\xff\xff\xff\x00\x01\xff\xff\xff\x26\xff\xff\xff\xfb\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x90\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x83\xff\xff\xff\x07\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x8a\xff\xff\xff\x00\x04\xff\xff\xff\x05\xff\xff\xff\x02\xff\xff\xff\x00\xff\xff\xff\x05\xff\xff\xff\x01\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xff\x82\xff\xff\xff\xf3\x04\xff\xff\xff\xf4\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xff\x82\xff\xff\xff\xf3\x01\xff\xff\xff\xf4\xff\xff\xff\xf7\x90\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x81\xff\xff\xff\x00\x02\xff\xff\xff\x07\xff\xff\xff\xc0\xff\xff\xff\x97\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x84\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x86\xff\xff\xff\x00\x02\xff\xff\xff\x18\xff\xff\xff\xff\xff\xff\xff\x22\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xff\x82\xff\xff\xff\xf3\x04\xff\xff\xff\xf4\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xff\x82\xff\xff\xff\xf3\x01\xff\xff\xff\xf4\xff\xff\xff\xf7\x90\xff\xff\xff\x00\x00\xff\xff\xff\x27\x84\xff\xff\xff\xff\x03\xff\xff\xff\x0b\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x8a\xff\xff\xff\x00\x04\xff\xff\xff\x2b\xff\xff\xff\xb2\xff\xff\xff\x1c\xff\xff\xff\xbb\xff\xff\xff\x08\x81\xff\xff\xff\x00\xa9\xff\xff\xff\x00\x03\xff\xff\xff\x27\xff\xff\xff\xff\xff\xff\xff\xf3\xff\xff\xff\x5e\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x87\xff\xff\xff\x00\x02\xff\xff\xff\x97\xff\xff\xff\xee\xff\xff\xff\xae\xaa\xff\xff\xff\x00\x03\xff\xff\xff\x01\xff\xff\xff\x05\xff\xff\xff\x00\xff\xff\xff\x07\x84\xff\xff\xff\x00\x03\xff\xff\xff\xc1\xff\xff\xff\x06\xff\xff\xff\xf3\xff\xff\xff\xcc\x83\xff\xff\xff\x00\x02\xff\xff\xff\x68\xff\xff\xff\xe8\xff\xff\xff\x2f\x82\xff\xff\xff\x00\xe0\xff\xff\xff\x00\x04\xff\xff\xff\x5b\xff\xff\xff\xd7\xff\xff\xff\x07\xff\xff\xff\xff\xff\xff\xff\x27\x82\xff\xff\xff\x00\x03\xff\xff\xff\x09\xff\xff\xff\xbc\xff\xff\xff\x23\xff\xff\xff\xb2\x92\xff\xff\xff\x00\xe9\xff\xff\xff\x00\x02\xff\xff\xff\x5e\xff\xff\xff\xeb\xff\xff\xff\x33\x92\xff\xff\xff\x00\xff\xff\xff\xff\x00\xb0\xff\xff\xff\x00\x03\xff\xff\xff\x8c\xff\xff\xff\xf4\xff\xff\xff\xe6\xff\xff\xff\x46\xca\xff\xff\xff\x00\x01\xff\xff\xff\x25\xff\xff\xff\xf6\x82\xff\xff\xff\xef\x04\xff\xff\xff\xf0\xff\xff\xff\xef\xff\xff\xff\x00\xff\xff\xff\x25\xff\xff\xff\xf6\x82\xff\xff\xff\xef\x01\xff\xff\xff\xf0\xff\xff\xff\xef\xa3\xff\xff\xff\x00\x01\xff\xff\xff\x6a\xff\xff\xff\xd2\x8a\xff\xff\xff\x00\x01\xff\xff\xff\x25\xff\xff\xff\xf6\x82\xff\xff\xff\xef\x04\xff\xff\xff\xf0\xff\xff\xff\xef\xff\xff\xff\x00\xff\xff\xff\x25\xff\xff\xff\xf6\x82\xff\xff\xff\xef\x01\xff\xff\xff\xf0\xff\xff\xff\xef\xb0\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\xa3\xff\xff\xff\x00\x01\xff\xff\xff\x2b\xff\xff\xff\xf8\x8a\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\xb0\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x8a\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x83\xff\xff\xff\x00\x05\xff\xff\xff\x13\xff\xff\xff\xb6\xff\xff\xff\xef\xff\xff\xff\xf3\xff\xff\xff\xc9\xff\xff\xff\x2d\x82\xff\xff\xff\x00\x01\xff\xff\xff\xfd\xff\xff\xff\xf6\x83\xff\xff\xff\xf3\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x83\xff\xff\xff\x00\x01\xff\xff\xff\xff\xff\xff\xff\xf5\x82\xff\xff\xff\xf3\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x90\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xff\x83\xff\xff\xff\xf3\x00\xff\xff\xff\x15\x81\xff\xff\xff\x00\x04\xff\xff\xff\x41\xff\xff\xff\xd9\xff\xff\xff\xf9\xff\xff\xff\xc3\xff\xff\xff\x21\x82\xff\xff\xff\x00\x05\xff\xff\xff\x0b\xff\xff\xff\x96\xff\xff\xff\xe8\xff\xff\xff\xf3\xff\xff\xff\xd9\xff\xff\xff\x6e\x81\xff\xff\xff\x00\x04\xff\xff\xff\x6d\xff\xff\xff\xe7\xff\xff\xff\xc0\xff\xff\xff\x84\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x8a\xff\xff\xff\x00\x01\xff\xff\xff\x21\xff\xff\xff\xd7\x83\xff\xff\xff\x00\x05\xff\xff\xff\x21\xff\xff\xff\x77\xff\xff\xff\x0e\xff\xff\xff\x00\xff\xff\xff\x84\xff\xff\xff\xc0\x82\xff\xff\xff\x00\x01\xff\xff\xff\x96\xff\xff\xff\xa2\x87\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x83\xff\xff\xff\x00\x01\xff\xff\xff\xab\xff\xff\xff\x93\x84\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x90\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x85\xff\xff\xff\x00\x05\xff\xff\xff\x02\xff\xff\xff\xe8\xff\xff\xff\x56\xff\xff\xff\x06\xff\xff\xff\x82\xff\xff\xff\xba\x82\xff\xff\xff\x00\x0c\xff\xff\xff\xa5\xff\xff\xff\x9b\xff\xff\xff\x0e\xff\xff\xff\x00\xff\xff\xff\x1f\xff\xff\xff\x4b\xff\xff\xff\x00\xff\xff\xff\x07\xff\xff\xff\xf6\xff\xff\xff\x38\xff\xff\xff\x00\xff\xff\xff\x8a\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x94\xff\xff\xff\x00\x01\xff\xff\xff\x30\xff\xff\xff\xf1\x82\xff\xff\xff\x00\x02\xff\xff\xff\x0d\xff\xff\xff\xe3\xff\xff\xff\x4a\x86\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x83\xff\xff\xff\x00\x02\xff\xff\xff\x0c\xff\xff\xff\xd7\xff\xff\xff\x59\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x00\xff\xff\xff\xb9\x82\xff\xff\xff\xbf\x83\xff\xff\xff\x00\x00\xff\xff\xff\xdf\x81\xff\xff\xff\xf3\x00\xff\xff\xff\xaa\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x85\xff\xff\xff\x00\x01\xff\xff\xff\x1d\xff\xff\xff\xfd\x81\xff\xff\xff\x00\x01\xff\xff\xff\x2a\xff\xff\xff\xec\x81\xff\xff\xff\x00\x02\xff\xff\xff\x0c\xff\xff\xff\xfb\xff\xff\xff\x0c\x84\xff\xff\xff\x00\x05\xff\xff\xff\x23\xff\xff\xff\xfd\xff\xff\xff\x01\xff\xff\xff\x00\xff\xff\xff\x35\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x8f\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfa\x83\xff\xff\xff\x00\x02\xff\xff\xff\x4f\xff\xff\xff\xe3\xff\xff\xff\x0e\x85\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x84\xff\xff\xff\x00\x02\xff\xff\xff\x2a\xff\xff\xff\xe8\xff\xff\xff\x2a\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x03\xff\xff\xff\x1b\xff\xff\xff\x52\xff\xff\xff\x20\xff\xff\xff\x58\x83\xff\xff\xff\x00\x03\xff\xff\xff\x06\xff\xff\xff\x08\xff\xff\xff\x07\xff\xff\xff\x04\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x85\xff\xff\xff\x00\x01\xff\xff\xff\x26\xff\xff\xff\xfb\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf6\x81\xff\xff\xff\x00\x01\xff\xff\xff\x22\xff\xff\xff\xff\x82\xff\xff\xff\xef\x04\xff\xff\xff\xf0\xff\xff\xff\xed\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x09\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x3d\xff\xff\xff\x76\xff\xff\xff\x2a\xff\xff\xff\xfb\xff\xff\xff\x0b\xff\xff\xff\x90\xff\xff\xff\x18\x8c\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x84\xff\xff\xff\x00\x01\xff\xff\xff\xac\xff\xff\xff\x9b\x85\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x85\xff\xff\xff\x00\x02\xff\xff\xff\x59\xff\xff\xff\xd7\xff\xff\xff\x0c\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x04\xff\xff\xff\x08\xff\xff\xff\xea\xff\xff\xff\x64\xff\xff\xff\xa4\xff\xff\xff\xf7\x83\xff\xff\xff\x00\x03\xff\xff\xff\x52\xff\xff\xff\xc0\xff\xff\xff\xc2\xff\xff\xff\x31\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xff\x82\xff\xff\xff\xef\x00\xff\xff\xff\xc2\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x02\xff\xff\xff\x0b\xff\xff\xff\xfb\xff\xff\xff\x28\x81\xff\xff\xff\x00\x04\xff\xff\xff\x3a\xff\xff\xff\xde\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x04\xff\xff\xff\x51\xff\xff\xff\xbd\xff\xff\xff\xfd\xff\xff\xff\xab\xff\xff\xff\x37\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x87\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x84\xff\xff\xff\x00\x02\xff\xff\xff\x18\xff\xff\xff\xed\xff\xff\xff\x43\x84\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x86\xff\xff\xff\x00\x01\xff\xff\xff\x94\xff\xff\xff\xaa\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x04\xff\xff\xff\x1c\xff\xff\xff\xff\xff\xff\xff\x24\xff\xff\xff\x34\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x04\xff\xff\xff\x07\xff\xff\xff\xf1\xff\xff\xff\x18\xff\xff\xff\x48\xff\xff\xff\xc9\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x85\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x08\xff\xff\xff\xa2\xff\xff\xff\xb6\xff\xff\xff\x0f\xff\xff\xff\x08\xff\xff\xff\xab\xff\xff\xff\x8b\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x04\xff\xff\xff\x52\xff\xff\xff\xbd\xff\xff\xff\xfd\xff\xff\xff\xab\xff\xff\xff\x37\x83\xff\xff\xff\x00\x01\xff\xff\xff\x20\xff\xff\xff\xd3\x87\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x85\xff\xff\xff\x00\x02\xff\xff\xff\x64\xff\xff\xff\xde\xff\xff\xff\x0b\x81\xff\xff\xff\x00\x03\xff\xff\xff\x02\xff\xff\xff\xdf\xff\xff\xff\xe4\xff\xff\xff\xfb\x83\xff\xff\xff\x00\x00\xff\xff\xff\xd0\x81\xff\xff\xff\xf3\x01\xff\xff\xff\xf5\xff\xff\xff\xff\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x03\xff\xff\xff\x8b\xff\xff\xff\xea\xff\xff\xff\xf2\xff\xff\xff\xf5\x82\xff\xff\xff\x00\x04\xff\xff\xff\x23\xff\xff\xff\xff\xff\xff\xff\x01\xff\xff\xff\x2d\xff\xff\xff\xf2\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x85\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x08\xff\xff\xff\x0a\xff\xff\xff\x98\xff\xff\xff\xea\xff\xff\xff\xef\xff\xff\xff\x9f\xff\xff\xff\x08\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x09\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x3d\xff\xff\xff\x75\xff\xff\xff\x2a\xff\xff\xff\xfb\xff\xff\xff\x0a\xff\xff\xff\x90\xff\xff\xff\x18\x8c\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x86\xff\xff\xff\x00\x01\xff\xff\xff\xbf\xff\xff\xff\x93\x90\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x03\xff\xff\xff\x20\xff\xff\xff\x00\xff\xff\xff\x59\xff\xff\xff\xdc\x82\xff\xff\xff\x00\x04\xff\xff\xff\x08\xff\xff\xff\xf4\xff\xff\xff\x3c\xff\xff\xff\x6a\xff\xff\xff\xcd\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x07\x00\xff\xff\xff\x06\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x91\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xff\x82\xff\xff\xff\xf3\x04\xff\xff\xff\xf4\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xff\x82\xff\xff\xff\xf3\x01\xff\xff\xff\xf4\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x8d\xff\xff\xff\x00\x03\xff\xff\xff\x7c\xff\xff\xff\xf3\xff\xff\xff\xf4\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x83\xff\xff\xff\xf3\x01\xff\xff\xff\xf8\xff\xff\xff\xfd\x83\xff\xff\xff\x00\x01\xff\xff\xff\x17\xff\xff\xff\x98\x8a\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xff\x82\xff\xff\xff\xf3\x04\xff\xff\xff\xf4\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xff\x82\xff\xff\xff\xf3\x01\xff\xff\xff\xf4\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x03\xff\xff\xff\x76\xff\xff\xff\xec\xff\xff\xff\xe8\xff\xff\xff\x54\x83\xff\xff\xff\x00\x03\xff\xff\xff\x5f\xff\xff\xff\xec\xff\xff\xff\xe1\xff\xff\xff\x3b\x81\xff\xff\xff\x00\x00\xff\xff\xff\x27\x83\xff\xff\xff\xff\x00\xff\xff\xff\xfb\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x83\xff\xff\xff\x00\x03\xff\xff\xff\xa2\xff\xff\xff\x49\xff\xff\xff\x54\xff\xff\xff\x95\x83\xff\xff\xff\x00\x02\xff\xff\xff\x1e\xff\xff\xff\xc1\xff\xff\xff\x15\x82\xff\xff\xff\x00\xb2\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\xbc\xff\xff\xff\x00\x03\xff\xff\xff\x0a\xff\xff\xff\xc7\xff\xff\xff\xc2\xff\xff\xff\x06\x84\xff\xff\xff\x00\x02\xff\xff\xff\x3b\xff\xff\xff\xca\xff\xff\xff\x11\x81\xff\xff\xff\x00\xe0\xff\xff\xff\x00\x04\xff\xff\xff\x21\xff\xff\xff\xb4\xff\xff\xff\x1c\xff\xff\xff\xbe\xff\xff\xff\x04\x83\xff\xff\xff\x00\x01\xff\xff\xff\x1b\xff\xff\xff\xbe\x93\xff\xff\xff\x00\xe1\xff\xff\xff\x00\x02\xff\xff\xff\x7d\xff\xff\xff\xe7\xff\xff\xff\x38\x85\xff\xff\xff\x00\x01\xff\xff\xff\x8b\xff\xff\xff\x65\x92\xff\xff\xff\x00\xff\xff\xff\xff\x00\xb8\xff\xff\xff\x00\x02\xff\xff\xff\xdd\xff\xff\xff\xe3\xff\xff\xff\x39\xc3\xff\xff\xff\x00\x01\xff\xff\xff\x25\xff\xff\xff\xf6\x82\xff\xff\xff\xef\x04\xff\xff\xff\xf0\xff\xff\xff\xef\xff\xff\xff\x00\xff\xff\xff\x25\xff\xff\xff\xf6\x82\xff\xff\xff\xef\x01\xff\xff\xff\xf0\xff\xff\xff\xef\xaa\xff\xff\xff\x00\x01\xff\xff\xff\x69\xff\xff\xff\xc9\x83\xff\xff\xff\x00\x01\xff\xff\xff\x25\xff\xff\xff\xf6\x82\xff\xff\xff\xef\x04\xff\xff\xff\xf0\xff\xff\xff\xef\xff\xff\xff\x00\xff\xff\xff\x25\xff\xff\xff\xf6\x82\xff\xff\xff\xef\x01\xff\xff\xff\xf0\xff\xff\xff\xef\xb0\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x01\xff\xff\xff\x75\xff\xff\xff\x80\xa5\xff\xff\xff\x00\x02\xff\xff\xff\x06\xff\xff\xff\xf2\xff\xff\xff\x26\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\xb0\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x02\xff\xff\xff\x0b\xff\xff\xff\xe7\xff\xff\xff\x10\x83\xff\xff\xff\x00\x03\xff\xff\xff\x67\xff\xff\xff\xef\xff\xff\xff\xec\xff\xff\xff\x84\x83\xff\xff\xff\x00\x02\xff\xff\xff\xf3\xff\xff\xff\xf4\xff\xff\xff\xff\x81\xff\xff\xff\xf3\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x83\xff\xff\xff\x00\x03\xff\xff\xff\x47\xff\xff\xff\xf3\xff\xff\xff\xf4\xff\xff\xff\xff\x81\xff\xff\xff\xf3\x00\xff\xff\xff\x1d\x82\xff\xff\xff\x00\x02\xff\xff\xff\x1a\xff\xff\xff\xfc\xff\xff\xff\x7d\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x04\xff\xff\xff\x46\xff\xff\xff\xd1\xff\xff\xff\xe9\xff\xff\xff\xa7\xff\xff\xff\x1e\x89\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xff\x83\xff\xff\xff\xf3\x00\xff\xff\xff\x15\x81\xff\xff\xff\x00\x04\xff\xff\xff\x41\xff\xff\xff\xd9\xff\xff\xff\xf9\xff\xff\xff\xc3\xff\xff\xff\x21\x82\xff\xff\xff\x00\x05\xff\xff\xff\x0b\xff\xff\xff\x96\xff\xff\xff\xe8\xff\xff\xff\xf3\xff\xff\xff\xd9\xff\xff\xff\x6e\x81\xff\xff\xff\x00\x04\xff\xff\xff\x6d\xff\xff\xff\xe7\xff\xff\xff\xc0\xff\xff\xff\x84\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x83\xff\xff\xff\x00\x01\xff\xff\xff\xa4\xff\xff\xff\x6d\x83\xff\xff\xff\x00\x04\xff\xff\xff\x21\xff\xff\xff\x05\xff\xff\xff\x18\xff\xff\xff\xd5\xff\xff\xff\x5e\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x85\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x85\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x85\xff\xff\xff\x00\x02\xff\xff\xff\x72\xff\xff\xff\xd4\xff\xff\xff\xd5\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x09\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x08\xff\xff\xff\x9a\xff\xff\xff\x19\xff\xff\xff\x05\xff\xff\xff\x67\xff\xff\xff\xe2\xff\xff\xff\x1b\x88\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x85\xff\xff\xff\x00\x05\xff\xff\xff\x02\xff\xff\xff\xe8\xff\xff\xff\x56\xff\xff\xff\x06\xff\xff\xff\x82\xff\xff\xff\xba\x82\xff\xff\xff\x00\x0c\xff\xff\xff\xa5\xff\xff\xff\x9e\xff\xff\xff\x0f\xff\xff\xff\x00\xff\xff\xff\x20\xff\xff\xff\x4d\xff\xff\xff\x00\xff\xff\xff\x07\xff\xff\xff\xf6\xff\xff\xff\x38\xff\xff\xff\x00\xff\xff\xff\x8a\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x83\xff\xff\xff\x00\x01\xff\xff\xff\x63\xff\xff\xff\xb6\x86\xff\xff\xff\x00\x01\xff\xff\xff\x5f\xff\xff\xff\xbc\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x85\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x85\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x85\xff\xff\xff\x00\x03\xff\xff\xff\xcd\xff\xff\xff\x4a\xff\xff\xff\xe2\xff\xff\xff\x2e\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x09\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x61\xff\xff\xff\x43\xff\xff\xff\xda\xff\xff\xff\xe7\xff\xff\xff\x4d\xff\xff\xff\x89\xff\xff\xff\x9d\x88\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x85\xff\xff\xff\x00\x01\xff\xff\xff\x1d\xff\xff\xff\xfd\x81\xff\xff\xff\x00\x01\xff\xff\xff\x2a\xff\xff\xff\xec\x81\xff\xff\xff\x00\x02\xff\xff\xff\x0c\xff\xff\xff\xfb\xff\xff\xff\x0e\x84\xff\xff\xff\x00\x05\xff\xff\xff\x23\xff\xff\xff\xfd\xff\xff\xff\x01\xff\xff\xff\x00\xff\xff\xff\x35\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x83\xff\xff\xff\x00\x01\xff\xff\xff\x3a\xff\xff\xff\xe5\x83\xff\xff\xff\x00\x04\xff\xff\xff\x40\xff\xff\xff\xd9\xff\xff\xff\xe3\xff\xff\xff\x84\xff\xff\xff\xe7\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x85\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x85\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x84\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xec\xff\xff\xff\x04\xff\xff\xff\x8f\xff\xff\xff\x87\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x09\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x06\xff\xff\xff\xe6\xff\xff\xff\xd9\xff\xff\xff\x52\xff\xff\xff\x01\xff\xff\xff\x10\xff\xff\xff\x35\xff\xff\xff\xe5\x88\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x85\xff\xff\xff\x00\x01\xff\xff\xff\x26\xff\xff\xff\xfb\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf6\x81\xff\xff\xff\x00\x01\xff\xff\xff\x22\xff\xff\xff\xff\x82\xff\xff\xff\xd7\x04\xff\xff\xff\xda\xff\xff\xff\xe2\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x83\xff\xff\xff\x00\x01\xff\xff\xff\x2c\xff\xff\xff\xf5\x83\xff\xff\xff\x00\x04\xff\xff\xff\xe3\xff\xff\xff\x65\xff\xff\xff\x02\xff\xff\xff\x98\xff\xff\xff\xf3\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x85\xff\xff\xff\x00\x02\xff\xff\xff\x6a\xff\xff\xff\xff\xff\xff\xff\x3f\x84\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x84\xff\xff\xff\x00\x04\xff\xff\xff\x81\xff\xff\xff\x9a\xff\xff\xff\x00\xff\xff\xff\x39\xff\xff\xff\xdf\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x05\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x21\xff\xff\xff\xfd\xff\xff\xff\xfc\xff\xff\xff\x01\x81\xff\xff\xff\x00\x01\xff\xff\xff\x35\xff\xff\xff\xea\x81\xff\xff\xff\x00\x04\xff\xff\xff\x3b\xff\xff\xff\xad\xff\xff\xff\xd2\xff\xff\xff\xa7\xff\xff\xff\x1b\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xff\x82\xff\xff\xff\xef\x00\xff\xff\xff\xc2\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x02\xff\xff\xff\x0b\xff\xff\xff\xfb\xff\xff\xff\x2b\x81\xff\xff\xff\x00\x04\xff\xff\xff\x37\xff\xff\xff\xe0\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x83\xff\xff\xff\x00\x01\xff\xff\xff\x39\xff\xff\xff\xe5\x82\xff\xff\xff\x00\x05\xff\xff\xff\x1e\xff\xff\xff\xff\xff\xff\xff\x05\xff\xff\xff\x00\xff\xff\xff\x37\xff\xff\xff\xea\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x84\xff\xff\xff\x00\x04\xff\xff\xff\x0f\xff\xff\xff\xe8\xff\xff\xff\x82\xff\xff\xff\xcd\xff\xff\xff\x01\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x84\xff\xff\xff\x00\x01\xff\xff\xff\xdb\xff\xff\xff\x42\x81\xff\xff\xff\x00\x04\xff\xff\xff\xe2\xff\xff\xff\x39\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x09\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x04\xff\xff\xff\xef\xff\xff\xff\xec\xff\xff\xff\x4f\xff\xff\xff\x01\xff\xff\xff\x12\xff\xff\xff\x86\xff\xff\xff\xa1\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x85\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x08\xff\xff\xff\xa2\xff\xff\xff\xbb\xff\xff\xff\x15\xff\xff\xff\x0d\xff\xff\xff\xad\xff\xff\xff\x8e\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x83\xff\xff\xff\x00\x01\xff\xff\xff\x62\xff\xff\xff\xb6\x82\xff\xff\xff\x00\x05\xff\xff\xff\x1d\xff\xff\xff\xff\xff\xff\xff\x05\xff\xff\xff\x00\xff\xff\xff\x36\xff\xff\xff\xca\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x84\xff\xff\xff\x00\x04\xff\xff\xff\x8b\xff\xff\xff\xa3\xff\xff\xff\x00\xff\xff\xff\xcc\xff\xff\xff\x60\x82\xff\xff\xff\x00\x02\xff\xff\xff\xbc\xff\xff\xff\xe4\xff\xff\xff\xfb\x83\xff\xff\xff\x00\x02\xff\xff\xff\x36\xff\xff\xff\xe7\xff\xff\xff\x02\x81\xff\xff\xff\x00\x04\xff\xff\xff\x8e\xff\xff\xff\x92\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x09\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x77\xff\xff\xff\xa8\xff\xff\xff\xe0\xff\xff\xff\xf3\xff\xff\xff\xb5\xff\xff\xff\xd5\xff\xff\xff\x17\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x85\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x08\xff\xff\xff\x0a\xff\xff\xff\x98\xff\xff\xff\xec\xff\xff\xff\xf2\xff\xff\xff\xa1\xff\xff\xff\x09\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x83\xff\xff\xff\x00\x01\xff\xff\xff\xa2\xff\xff\xff\x6b\x83\xff\xff\xff\x00\x04\xff\xff\xff\xe2\xff\xff\xff\x63\xff\xff\xff\x01\xff\xff\xff\x9b\xff\xff\xff\x7f\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x83\xff\xff\xff\x00\x06\xff\xff\xff\x21\xff\xff\xff\xf2\xff\xff\xff\x20\xff\xff\xff\x00\xff\xff\xff\x46\xff\xff\xff\xe5\xff\xff\xff\x0b\x90\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x04\xff\xff\xff\x71\xff\xff\xff\xd8\xff\xff\xff\xe3\xff\xff\xff\x7b\xff\xff\xff\x0a\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x07\x00\xff\xff\xff\x06\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x84\xff\xff\xff\x00\x00\xff\xff\xff\x09\x8b\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xff\x82\xff\xff\xff\xf3\x04\xff\xff\xff\xf4\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xff\x82\xff\xff\xff\xf3\x01\xff\xff\xff\xf4\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x02\xff\xff\xff\x0b\xff\xff\xff\xe6\xff\xff\xff\x10\x83\xff\xff\xff\x00\x04\xff\xff\xff\x3f\xff\xff\xff\xda\xff\xff\xff\xf2\xff\xff\xff\xad\xff\xff\xff\x09\x82\xff\xff\xff\x00\x02\xff\xff\xff\xf3\xff\xff\xff\xf4\xff\xff\xff\xff\x81\xff\xff\xff\xf3\x81\xff\xff\xff\x00\x01\xff\xff\xff\xab\xff\xff\xff\x93\x82\xff\xff\xff\x00\x01\xff\xff\xff\xbe\xff\xff\xff\x82\x82\xff\xff\xff\x00\x01\xff\xff\xff\x17\xff\xff\xff\x98\x8b\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xff\x82\xff\xff\xff\xf3\x04\xff\xff\xff\xf4\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xff\x82\xff\xff\xff\xf3\x01\xff\xff\xff\xf4\xff\xff\xff\xf7\x89\xff\xff\xff\x00\x02\xff\xff\xff\x4f\xff\xff\xff\xb4\xff\xff\xff\xfb\x83\xff\xff\xff\x00\x00\xff\xff\xff\x27\x83\xff\xff\xff\xff\x00\xff\xff\xff\xfb\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x84\xff\xff\xff\x00\x01\xff\xff\xff\x71\xff\xff\xff\x98\x84\xff\xff\xff\x00\x02\xff\xff\xff\x39\xff\xff\xff\xb6\xff\xff\xff\x05\x82\xff\xff\xff\x00\x91\xff\xff\xff\x00\x01\xff\xff\xff\x74\xff\xff\xff\x80\x9d\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\xbf\xff\xff\xff\x00\x01\xff\xff\xff\x92\xff\xff\xff\x96\x82\xff\xff\xff\x00\x02\xff\xff\xff\x34\xff\xff\xff\xd2\xff\xff\xff\x15\x83\xff\xff\xff\x00\xe1\xff\xff\xff\x00\x02\xff\xff\xff\x2f\xff\xff\xff\xc0\xff\xff\xff\x01\x84\xff\xff\xff\x00\x02\xff\xff\xff\x0d\xff\xff\xff\xc6\xff\xff\xff\x06\x92\xff\xff\xff\x00\xe2\xff\xff\xff\x00\x01\xff\xff\xff\x99\xff\xff\xff\x71\x84\xff\xff\xff\x00\x01\xff\xff\xff\x91\xff\xff\xff\x5f\x93\xff\xff\xff\x00\xff\xff\xff\xff\x00\xd9\xff\xff\xff\x00\x02\xff\xff\xff\x4e\xff\xff\xff\xc0\xff\xff\xff\xc1\xa2\xff\xff\xff\x00\x01\xff\xff\xff\x25\xff\xff\xff\xf6\x82\xff\xff\xff\xef\x04\xff\xff\xff\xf0\xff\xff\xff\xef\xff\xff\xff\x00\xff\xff\xff\x25\xff\xff\xff\xf6\x82\xff\xff\xff\xef\x01\xff\xff\xff\xf0\xff\xff\xff\xef\xb0\xff\xff\xff\x00\x01\xff\xff\xff\x25\xff\xff\xff\xf6\x82\xff\xff\xff\xef\x04\xff\xff\xff\xf0\xff\xff\xff\xef\xff\xff\xff\x00\xff\xff\xff\x25\xff\xff\xff\xf6\x82\xff\xff\xff\xef\x01\xff\xff\xff\xf0\xff\xff\xff\xef\x8b\xff\xff\xff\x00\x01\xff\xff\xff\x0e\xff\xff\xff\xac\xa2\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x84\xff\xff\xff\x00\x01\xff\xff\xff\xae\xff\xff\xff\x47\xa9\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x98\xff\xff\xff\x00\x00\xff\xff\xff\x0f\x8e\xff\xff\xff\x00\x01\xff\xff\xff\x24\xff\xff\xff\x05\x85\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x83\xff\xff\xff\x00\x01\xff\xff\xff\x34\xff\xff\xff\xce\x83\xff\xff\xff\x00\x04\xff\xff\xff\x52\xff\xff\xff\xdd\xff\xff\xff\xf7\xff\xff\xff\xcd\xff\xff\xff\x31\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x01\xff\xff\xff\x95\xff\xff\xff\xaf\x82\xff\xff\xff\x00\x04\xff\xff\xff\x6e\xff\xff\xff\xcc\xff\xff\xff\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x09\xff\xff\xff\x35\xff\xff\xff\xe8\xff\xff\xff\x18\xff\xff\xff\x00\xff\xff\xff\x03\xff\xff\xff\xca\xff\xff\xff\x6c\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x90\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xff\x83\xff\xff\xff\xf3\x07\xff\xff\xff\x15\xff\xff\xff\x00\xff\xff\xff\x88\xff\xff\xff\x69\xff\xff\xff\xbd\xff\xff\xff\xf4\xff\xff\xff\xe7\xff\xff\xff\x82\x82\xff\xff\xff\x00\x0c\xff\xff\xff\x0b\xff\xff\xff\x96\xff\xff\xff\xe8\xff\xff\xff\xf3\xff\xff\xff\xd9\xff\xff\xff\x6e\xff\xff\xff\x00\xff\xff\xff\x4e\xff\xff\xff\xa5\xff\xff\xff\xc2\xff\xff\xff\xf4\xff\xff\xff\xc1\xff\xff\xff\x18\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x83\xff\xff\xff\x00\x01\xff\xff\xff\x9d\xff\xff\xff\x72\x82\xff\xff\xff\x00\x05\xff\xff\xff\x0a\xff\xff\xff\xf4\xff\xff\xff\x57\xff\xff\xff\x01\xff\xff\xff\x7f\xff\xff\xff\xce\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x09\xff\xff\xff\x0f\xff\xff\xff\xe7\xff\xff\xff\x47\xff\xff\xff\x00\xff\xff\xff\x0e\xff\xff\xff\xea\xff\xff\xff\x39\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x04\xff\xff\xff\x7c\xff\xff\xff\xb4\xff\xff\xff\x00\xff\xff\xff\x7d\xff\xff\xff\xb6\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x90\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x85\xff\xff\xff\x00\x06\xff\xff\xff\x08\xff\xff\xff\xe7\xff\xff\xff\x9b\xff\xff\xff\x08\xff\xff\xff\x12\xff\xff\xff\xcb\xff\xff\xff\x62\x81\xff\xff\xff\x00\x05\xff\xff\xff\xa5\xff\xff\xff\x9e\xff\xff\xff\x0f\xff\xff\xff\x00\xff\xff\xff\x20\xff\xff\xff\x4d\x81\xff\xff\xff\x00\x02\xff\xff\xff\xd0\xff\xff\xff\x9e\xff\xff\xff\x02\x81\xff\xff\xff\x9f\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x83\xff\xff\xff\x00\x01\xff\xff\xff\xe5\xff\xff\xff\x33\x82\xff\xff\xff\x00\x05\xff\xff\xff\x1f\xff\xff\xff\xff\xff\xff\xff\x04\xff\xff\xff\x00\xff\xff\xff\x30\xff\xff\xff\xee\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x04\xff\xff\xff\x5b\xff\xff\xff\xd9\xff\xff\xff\x05\xff\xff\xff\x89\xff\xff\xff\xa3\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x04\xff\xff\xff\x02\xff\xff\xff\xc4\xff\xff\xff\x90\xff\xff\xff\xe5\xff\xff\xff\x17\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x90\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x85\xff\xff\xff\x00\x0a\xff\xff\xff\x05\xff\xff\xff\xf9\xff\xff\xff\xb7\xff\xff\xff\x02\xff\xff\xff\x00\xff\xff\xff\x5b\xff\xff\xff\xbe\xff\xff\xff\x00\xff\xff\xff\x0c\xff\xff\xff\xfb\xff\xff\xff\x0e\x84\xff\xff\xff\x00\x05\xff\xff\xff\x0f\xff\xff\xff\xff\xff\xff\xff\xae\xff\xff\xff\x25\xff\xff\xff\x3c\xff\xff\xff\xe2\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x02\xff\xff\xff\x13\xff\xff\xff\xff\xff\xff\xff\x0c\x82\xff\xff\xff\x00\x05\xff\xff\xff\x02\xff\xff\xff\xd7\xff\xff\xff\x58\xff\xff\xff\x03\xff\xff\xff\x84\xff\xff\xff\xaa\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x83\xff\xff\xff\x00\x03\xff\xff\xff\xbd\xff\xff\xff\x96\xff\xff\xff\xea\xff\xff\xff\x19\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x83\xff\xff\xff\x00\x02\xff\xff\xff\x29\xff\xff\xff\xff\xff\xff\xff\x64\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x90\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x85\xff\xff\xff\x00\x09\xff\xff\xff\x1d\xff\xff\xff\xfb\xff\xff\xff\x7b\xff\xff\xff\x8c\xff\xff\xff\x00\xff\xff\xff\x33\xff\xff\xff\xe8\xff\xff\xff\x00\xff\xff\xff\x22\xff\xff\xff\xff\x82\xff\xff\xff\xd7\x08\xff\xff\xff\xda\xff\xff\xff\xe2\xff\xff\xff\x00\xff\xff\xff\x24\xff\xff\xff\xfd\xff\xff\xff\x12\xff\xff\xff\xaf\xff\xff\xff\x2d\xff\xff\xff\xf3\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x01\xff\xff\xff\x22\xff\xff\xff\xff\x84\xff\xff\xff\x00\x04\xff\xff\xff\x23\xff\xff\xff\xe2\xff\xff\xff\xfc\xff\xff\xff\xcd\xff\xff\xff\x0e\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xff\x81\xff\xff\xff\xef\x01\xff\xff\xff\xf1\xff\xff\xff\xf7\x83\xff\xff\xff\x00\x02\xff\xff\xff\x29\xff\xff\xff\xff\xff\xff\xff\x7f\x82\xff\xff\xff\x00\x05\xff\xff\xff\x27\xff\xff\xff\xff\xff\xff\xff\x0e\xff\xff\xff\x00\xff\xff\xff\x2e\xff\xff\xff\xf3\x83\xff\xff\xff\x00\x03\xff\xff\xff\x90\xff\xff\xff\xcd\xff\xff\xff\xc6\xff\xff\xff\x02\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x90\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xff\x82\xff\xff\xff\xef\x00\xff\xff\xff\xc2\x81\xff\xff\xff\x00\x0a\xff\xff\xff\x25\xff\xff\xff\xfd\xff\xff\xff\x00\xff\xff\xff\xa7\xff\xff\xff\x5d\xff\xff\xff\x28\xff\xff\xff\xf4\xff\xff\xff\x00\xff\xff\xff\x0b\xff\xff\xff\xfb\xff\xff\xff\x2b\x81\xff\xff\xff\x00\x08\xff\xff\xff\x37\xff\xff\xff\xe0\xff\xff\xff\x00\xff\xff\xff\x13\xff\xff\xff\xff\xff\xff\xff\x0f\xff\xff\xff\x46\xff\xff\xff\xb8\xff\xff\xff\xe0\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x02\xff\xff\xff\x12\xff\xff\xff\xff\xff\xff\xff\x0c\x83\xff\xff\xff\x00\x04\xff\xff\xff\xc6\xff\xff\xff\x5e\xff\xff\xff\x03\xff\xff\xff\x87\xff\xff\xff\x9e\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x83\xff\xff\xff\x00\x02\xff\xff\xff\x67\xff\xff\xff\xe8\xff\xff\xff\xc1\x82\xff\xff\xff\x00\x05\xff\xff\xff\x27\xff\xff\xff\xff\xff\xff\xff\x78\xff\xff\xff\x01\xff\xff\xff\x7c\xff\xff\xff\xce\x82\xff\xff\xff\x00\x04\xff\xff\xff\x45\xff\xff\xff\xd9\xff\xff\xff\x0a\xff\xff\xff\xae\xff\xff\xff\x7d\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x90\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x85\xff\xff\xff\x00\x06\xff\xff\xff\x18\xff\xff\xff\xff\xff\xff\xff\x0a\xff\xff\xff\x09\xff\xff\xff\xc5\xff\xff\xff\x60\xff\xff\xff\xeb\x81\xff\xff\xff\x00\x05\xff\xff\xff\xa2\xff\xff\xff\xbb\xff\xff\xff\x15\xff\xff\xff\x0d\xff\xff\xff\xad\xff\xff\xff\x8e\x81\xff\xff\xff\x00\x04\xff\xff\xff\xce\xff\xff\xff\x78\xff\xff\xff\x02\xff\xff\xff\xc6\xff\xff\xff\x9e\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x83\xff\xff\xff\x00\x01\xff\xff\xff\xe5\xff\xff\xff\x33\x82\xff\xff\xff\x00\x05\xff\xff\xff\x1d\xff\xff\xff\xff\xff\xff\xff\x04\xff\xff\xff\x00\xff\xff\xff\x30\xff\xff\xff\xed\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x04\xff\xff\xff\x12\xff\xff\xff\xeb\xff\xff\xff\x34\xff\xff\xff\xc6\xff\xff\xff\x60\x81\xff\xff\xff\x00\x05\xff\xff\xff\x27\xff\xff\xff\xfb\xff\xff\xff\x6e\xff\xff\xff\xe8\xff\xff\xff\xe7\xff\xff\xff\x47\x81\xff\xff\xff\x00\x09\xff\xff\xff\x12\xff\xff\xff\xe2\xff\xff\xff\x39\xff\xff\xff\x00\xff\xff\xff\x15\xff\xff\xff\xe4\xff\xff\xff\x35\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x90\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x86\xff\xff\xff\x00\x05\xff\xff\xff\xed\xff\xff\xff\x31\xff\xff\xff\x00\xff\xff\xff\x1c\xff\xff\xff\xe2\xff\xff\xff\xc9\x81\xff\xff\xff\x00\x05\xff\xff\xff\x0a\xff\xff\xff\x98\xff\xff\xff\xec\xff\xff\xff\xf2\xff\xff\xff\xa1\xff\xff\xff\x09\x81\xff\xff\xff\x00\x06\xff\xff\xff\x2f\xff\xff\xff\xd2\xff\xff\xff\xf3\xff\xff\xff\xaf\xff\xff\xff\xbb\xff\xff\xff\x22\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x83\xff\xff\xff\x00\x01\xff\xff\xff\x9c\xff\xff\xff\x72\x82\xff\xff\xff\x00\x05\xff\xff\xff\x07\xff\xff\xff\xee\xff\xff\xff\x55\xff\xff\xff\x01\xff\xff\xff\x81\xff\xff\xff\xc8\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x08\xff\xff\xff\x99\xff\xff\xff\xa0\xff\xff\xff\x00\xff\xff\xff\x35\xff\xff\xff\xe9\xff\xff\xff\x12\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x8d\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x90\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x07\x00\xff\xff\xff\x06\x82\xff\xff\xff\x00\x05\xff\xff\xff\x91\xff\xff\xff\xaa\xff\xff\xff\x09\xff\xff\xff\x12\xff\xff\xff\xcc\xff\xff\xff\xac\x84\xff\xff\xff\x00\x00\xff\xff\xff\x09\x87\xff\xff\xff\x00\x02\xff\xff\xff\x1a\xff\xff\xff\x0b\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xff\x82\xff\xff\xff\xf3\x04\xff\xff\xff\xf4\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xff\x82\xff\xff\xff\xf3\x01\xff\xff\xff\xf4\xff\xff\xff\xf7\x83\xff\xff\xff\x00\x01\xff\xff\xff\x34\xff\xff\xff\xce\x83\xff\xff\xff\x00\x04\xff\xff\xff\x40\xff\xff\xff\xd4\xff\xff\xff\xf5\xff\xff\xff\xc2\xff\xff\xff\x24\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x02\xff\xff\xff\x34\xff\xff\xff\xf2\xff\xff\xff\x1a\x81\xff\xff\xff\x00\x04\xff\xff\xff\xa1\xff\xff\xff\x9c\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x8d\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xff\x82\xff\xff\xff\xf3\x04\xff\xff\xff\xf4\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xff\x82\xff\xff\xff\xf3\x01\xff\xff\xff\xf4\xff\xff\xff\xf7\x90\xff\xff\xff\x00\x00\xff\xff\xff\x27\x83\xff\xff\xff\xff\x00\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x06\xff\xff\xff\x08\xff\xff\xff\x9d\xff\xff\xff\xec\xff\xff\xff\xef\xff\xff\xff\x9a\xff\xff\xff\x95\xff\xff\xff\x37\x81\xff\xff\xff\x00\x02\xff\xff\xff\x01\xff\xff\xff\xa3\xff\xff\xff\x64\x8a\xff\xff\xff\x00\x93\xff\xff\xff\x00\x01\xff\xff\xff\xae\xff\xff\xff\x47\x99\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x9e\xff\xff\xff\x00\x04\xff\xff\xff\x52\xff\xff\xff\xcc\xff\xff\xff\x02\xff\xff\xff\xf3\xff\xff\xff\x25\x97\xff\xff\xff\x00\x05\xff\xff\xff\x08\xff\xff\xff\x02\xff\xff\xff\x00\xff\xff\xff\x01\xff\xff\xff\xa1\xff\xff\xff\x86\x8b\xff\xff\xff\x00\xe1\xff\xff\xff\x00\x02\xff\xff\xff\x1a\xff\xff\xff\xcd\xff\xff\xff\x06\x9a\xff\xff\xff\x00\xe1\xff\xff\xff\x00\x01\xff\xff\xff\xb3\xff\xff\xff\x51\x9b\xff\xff\xff\x00\xff\xff\xff\xff\x00\xb0\xff\xff\xff\x00\x04\xff\xff\xff\x4c\xff\xff\xff\xe5\xff\xff\xff\xf2\xff\xff\xff\xb4\xff\xff\xff\x11\xab\xff\xff\xff\x00\x03\xff\xff\xff\x02\xff\xff\xff\xea\xff\xff\xff\xf1\xff\xff\xff\x76\x8b\xff\xff\xff\x00\x02\xff\xff\xff\x5e\xff\xff\xff\xf3\xff\xff\xff\x87\x8a\xff\xff\xff\x00\x01\xff\xff\xff\x25\xff\xff\xff\xf6\x82\xff\xff\xff\xef\x04\xff\xff\xff\xf0\xff\xff\xff\xef\xff\xff\xff\x00\xff\xff\xff\x25\xff\xff\xff\xf6\x82\xff\xff\xff\xef\x01\xff\xff\xff\xf0\xff\xff\xff\xef\xa1\xff\xff\xff\x00\x04\xff\xff\xff\x32\xff\xff\xff\x23\xff\xff\xff\x02\xff\xff\xff\x96\xff\xff\xff\x9a\x89\xff\xff\xff\x00\x01\xff\xff\xff\x25\xff\xff\xff\xf6\x82\xff\xff\xff\xef\x04\xff\xff\xff\xf0\xff\xff\xff\xef\xff\xff\xff\x00\xff\xff\xff\x25\xff\xff\xff\xf6\x82\xff\xff\xff\xef\x01\xff\xff\xff\xf0\xff\xff\xff\xef\x81\xff\xff\xff\x00\x03\xff\xff\xff\xbc\xff\xff\xff\xed\xff\xff\xff\xdd\xff\xff\xff\x4d\x8d\xff\xff\xff\x00\x02\xff\xff\xff\x08\xff\xff\xff\x47\xff\xff\xff\xcb\x8b\xff\xff\xff\x00\x02\xff\xff\xff\x07\xff\xff\xff\x05\xff\xff\xff\xbb\x8a\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\xa4\xff\xff\xff\x00\x01\xff\xff\xff\x36\xff\xff\xff\xe2\x89\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x03\xff\xff\xff\x3b\xff\xff\xff\x01\xff\xff\xff\x5e\xff\xff\xff\xe1\x8e\xff\xff\xff\x00\x01\xff\xff\xff\x79\xff\xff\xff\x6d\x8c\xff\xff\xff\x00\x01\xff\xff\xff\x25\xff\xff\xff\x83\x8a\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x8a\xff\xff\xff\x00\x01\xff\xff\xff\x8a\xff\xff\xff\xca\x85\xff\xff\xff\x00\x04\xff\xff\xff\x68\xff\xff\xff\xdb\xff\xff\xff\xf2\xff\xff\xff\xdc\xff\xff\xff\x6c\x81\xff\xff\xff\x00\x05\xff\xff\xff\xc2\xff\xff\xff\x85\xff\xff\xff\x00\xff\xff\xff\x4e\xff\xff\xff\xf9\xff\xff\xff\x02\x81\xff\xff\xff\x00\x04\xff\xff\xff\x2c\xff\xff\xff\xd8\xff\xff\xff\xeb\xff\xff\xff\x86\xff\xff\xff\xf5\x82\xff\xff\xff\x00\x04\xff\xff\xff\x93\xff\xff\xff\xb2\xff\xff\xff\x00\xff\xff\xff\x7a\xff\xff\xff\xcb\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x83\xff\xff\xff\x00\x01\xff\xff\xff\x63\xff\xff\xff\xe8\x8b\xff\xff\xff\x00\x04\xff\xff\xff\x02\xff\xff\xff\x8b\xff\xff\xff\xd9\xff\xff\xff\xdc\xff\xff\xff\xad\x8a\xff\xff\xff\x00\x04\xff\xff\xff\x16\xff\xff\xff\xac\xff\xff\xff\xd1\xff\xff\xff\xcd\xff\xff\xff\xa9\x89\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x8a\xff\xff\xff\x00\x02\xff\xff\xff\x23\xff\xff\xff\xfd\xff\xff\xff\x30\x83\xff\xff\xff\x00\x05\xff\xff\xff\x63\xff\xff\xff\xd2\xff\xff\xff\x25\xff\xff\xff\x00\xff\xff\xff\x46\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x05\xff\xff\xff\xe6\xff\xff\xff\xb9\xff\xff\xff\x00\xff\xff\xff\x83\xff\xff\xff\xfd\xff\xff\xff\x21\x81\xff\xff\xff\x00\x04\xff\xff\xff\xc7\xff\xff\xff\x7b\xff\xff\xff\x02\xff\xff\xff\x9f\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x08\xff\xff\xff\xcb\xff\xff\xff\xd9\xff\xff\xff\x04\xff\xff\xff\xb4\xff\xff\xff\xec\xff\xff\xff\x08\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x04\xff\xff\xff\x01\xff\xff\xff\x7f\xff\xff\xff\xf8\xff\xff\xff\xd1\xff\xff\xff\x0f\x8a\xff\xff\xff\x00\x04\xff\xff\xff\x7d\xff\xff\xff\xaf\xff\xff\xff\x04\xff\xff\xff\x00\xff\xff\xff\x37\x81\xff\xff\xff\x00\x06\xff\xff\xff\x03\xff\xff\xff\xaa\xff\xff\xff\x1c\xff\xff\xff\x00\xff\xff\xff\x06\xff\xff\xff\xa5\xff\xff\xff\x1a\x81\xff\xff\xff\x00\x01\xff\xff\xff\xba\xff\xff\xff\x8b\x81\xff\xff\xff\x00\x00\xff\xff\xff\x2a\x84\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x8b\xff\xff\xff\x00\x01\xff\xff\xff\xb8\xff\xff\xff\x96\x83\xff\xff\xff\x00\x01\xff\xff\xff\xdb\xff\xff\xff\x45\x81\xff\xff\xff\x00\x10\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x0c\xff\xff\xff\xe9\xff\xff\xff\xcd\xff\xff\xff\x00\xff\xff\xff\xb8\xff\xff\xff\xc5\xff\xff\xff\x46\xff\xff\xff\x00\xff\xff\xff\x10\xff\xff\xff\xff\xff\xff\xff\x12\xff\xff\xff\x00\xff\xff\xff\x3e\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x09\xff\xff\xff\x08\xff\xff\xff\xee\xff\xff\xff\x81\xff\xff\xff\x4b\xff\xff\xff\xaf\xff\xff\xff\xc0\xff\xff\xff\x3b\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x04\xff\xff\xff\x9f\xff\xff\xff\xd4\xff\xff\xff\x1f\xff\xff\xff\x8a\xff\xff\xff\xbc\x8a\xff\xff\xff\x00\x01\xff\xff\xff\xe5\xff\xff\xff\x34\x85\xff\xff\xff\x00\x04\xff\xff\xff\x6d\xff\xff\xff\xd8\xff\xff\xff\x1b\xff\xff\xff\xb2\xff\xff\xff\xa0\x81\xff\xff\xff\x00\x02\xff\xff\xff\x10\xff\xff\xff\xff\xff\xff\xff\x15\x87\xff\xff\xff\x00\x01\xff\xff\xff\x1c\xff\xff\xff\xb5\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x8b\xff\xff\xff\x00\x02\xff\xff\xff\x4e\xff\xff\xff\xf1\xff\xff\xff\x0a\x81\xff\xff\xff\x00\x02\xff\xff\xff\x13\xff\xff\xff\xff\xff\xff\xff\x0b\x81\xff\xff\xff\x00\x0c\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x32\xff\xff\xff\xcc\xff\xff\xff\xad\xff\xff\xff\x25\xff\xff\xff\xd2\xff\xff\xff\x90\xff\xff\xff\x6b\xff\xff\xff\x00\xff\xff\xff\x24\xff\xff\xff\xfd\x81\xff\xff\xff\x00\x01\xff\xff\xff\x2a\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x09\xff\xff\xff\x3b\xff\xff\xff\xc3\xff\xff\xff\x3c\xff\xff\xff\xcf\xff\xff\xff\x71\xff\xff\xff\x8c\xff\xff\xff\x73\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x08\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x17\xff\xff\xff\xff\xff\xff\xff\x24\xff\xff\xff\x00\xff\xff\xff\x52\xff\xff\xff\xe6\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x83\xff\xff\xff\x00\x02\xff\xff\xff\x16\xff\xff\xff\xff\xff\xff\xff\x07\x86\xff\xff\xff\x00\x03\xff\xff\xff\x76\xff\xff\xff\xf4\xff\xff\xff\xac\xff\xff\xff\x02\x81\xff\xff\xff\x00\x01\xff\xff\xff\x23\xff\xff\xff\xfd\x86\xff\xff\xff\x00\x85\xff\xff\xff\x1b\x00\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x8b\xff\xff\xff\x00\x02\xff\xff\xff\x02\xff\xff\xff\xe1\xff\xff\xff\x63\x81\xff\xff\xff\x00\x01\xff\xff\xff\x23\xff\xff\xff\xfd\x81\xff\xff\xff\x00\x05\xff\xff\xff\x86\xff\xff\xff\xf0\xff\xff\xff\xef\xff\xff\xff\x00\xff\xff\xff\x57\xff\xff\xff\xae\x81\xff\xff\xff\x7d\x09\xff\xff\xff\xb3\xff\xff\xff\x72\xff\xff\xff\x90\xff\xff\xff\x00\xff\xff\xff\x10\xff\xff\xff\xff\xff\xff\xff\x12\xff\xff\xff\x00\xff\xff\xff\x3d\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x09\xff\xff\xff\x73\xff\xff\xff\x8f\xff\xff\xff\x04\xff\xff\xff\xe9\xff\xff\xff\x2c\xff\xff\xff\x58\xff\xff\xff\xab\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x08\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x0d\xff\xff\xff\xf3\xff\xff\xff\x07\xff\xff\xff\x77\xff\xff\xff\xe4\xff\xff\xff\x49\x83\xff\xff\xff\x00\x01\xff\xff\xff\x22\xff\xff\xff\xdb\x83\xff\xff\xff\x00\x01\xff\xff\xff\x24\xff\xff\xff\xfd\x87\xff\xff\xff\x00\x03\xff\xff\xff\xa0\xff\xff\xff\xdc\xff\xff\xff\xcc\xff\xff\xff\x0e\x81\xff\xff\xff\x00\x02\xff\xff\xff\x0b\xff\xff\xff\xfb\xff\xff\xff\x20\x85\xff\xff\xff\x00\x85\xff\xff\xff\xeb\x00\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x8c\xff\xff\xff\x00\x01\xff\xff\xff\x7c\xff\xff\xff\xca\x81\xff\xff\xff\x00\x02\xff\xff\xff\x13\xff\xff\xff\xff\xff\xff\xff\x0d\x84\xff\xff\xff\x00\x06\xff\xff\xff\x7c\xff\xff\xff\x90\xff\xff\xff\x4c\xff\xff\xff\xe2\xff\xff\xff\x81\xff\xff\xff\x54\xff\xff\xff\xb5\x81\xff\xff\xff\x00\x04\xff\xff\xff\xc7\xff\xff\xff\x7c\xff\xff\xff\x02\xff\xff\xff\xa0\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x01\xff\xff\xff\xab\xff\xff\xff\x5c\x82\xff\xff\xff\x00\x04\xff\xff\xff\x25\xff\xff\xff\xe3\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x03\xff\xff\xff\x62\xff\xff\xff\xe7\xff\xff\xff\x8d\xff\xff\xff\x13\x8a\xff\xff\xff\x00\x02\xff\xff\xff\x16\xff\xff\xff\xff\xff\xff\xff\x0b\x85\xff\xff\xff\x00\x05\xff\xff\xff\x97\xff\xff\xff\xbb\xff\xff\xff\x07\xff\xff\xff\x8b\xff\xff\xff\xc4\xff\xff\xff\x0a\x81\xff\xff\xff\x00\x04\xff\xff\xff\xa8\xff\xff\xff\xab\xff\xff\xff\x0d\xff\xff\xff\x06\xff\xff\xff\x51\x84\xff\xff\xff\x00\x01\xff\xff\xff\x09\xff\xff\xff\x3d\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x87\xff\xff\xff\x00\x02\xff\xff\xff\x17\xff\xff\xff\xf9\xff\xff\xff\x30\x81\xff\xff\xff\x00\x01\xff\xff\xff\xda\xff\xff\xff\x4b\x84\xff\xff\xff\x00\x06\xff\xff\xff\xa2\xff\xff\xff\x72\xff\xff\xff\x1b\xff\xff\xff\xff\xff\xff\xff\x4f\xff\xff\xff\x36\xff\xff\xff\xda\x81\xff\xff\xff\x00\x04\xff\xff\xff\x2c\xff\xff\xff\xd7\xff\xff\xff\xec\xff\xff\xff\x8c\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x01\xff\xff\xff\xe3\xff\xff\xff\x28\x83\xff\xff\xff\x00\x03\xff\xff\xff\xef\xff\xff\xff\x1b\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x05\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x07\xff\xff\xff\xd9\xff\xff\xff\x48\x8d\xff\xff\xff\x00\x01\xff\xff\xff\xe3\xff\xff\xff\x3e\x84\xff\xff\xff\x00\x02\xff\xff\xff\x02\xff\xff\xff\x7e\xff\xff\xff\x0a\x81\xff\xff\xff\x00\x01\xff\xff\xff\x78\xff\xff\xff\x13\x81\xff\xff\xff\x00\x04\xff\xff\xff\x0e\xff\xff\xff\xa1\xff\xff\xff\xed\xff\xff\xff\xee\xff\xff\xff\x91\x84\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x88\xff\xff\xff\x00\x01\xff\xff\xff\xa9\xff\xff\xff\x96\x81\xff\xff\xff\x00\x08\xff\xff\xff\x60\xff\xff\xff\xd8\xff\xff\xff\x29\xff\xff\xff\x00\xff\xff\xff\x21\xff\xff\xff\x57\xff\xff\xff\x00\xff\xff\xff\xc7\xff\xff\xff\x54\x82\xff\xff\xff\x00\x02\xff\xff\xff\x18\xff\xff\xff\xfa\xff\xff\xff\x03\x8f\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x07\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x18\xff\xff\xff\xfd\xff\xff\xff\x2f\xff\xff\xff\x00\xff\xff\xff\x30\x8b\xff\xff\xff\x00\x04\xff\xff\xff\x77\xff\xff\xff\xc4\xff\xff\xff\x13\xff\xff\xff\x07\xff\xff\xff\x52\x99\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xff\x82\xff\xff\xff\xf3\x04\xff\xff\xff\xf4\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xff\x82\xff\xff\xff\xf3\x01\xff\xff\xff\xf4\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x84\xff\xff\xff\x00\x83\xff\xff\xff\xf3\x01\xff\xff\xff\xf8\xff\xff\xff\xf1\x82\xff\xff\xff\x00\x07\xff\xff\xff\x64\xff\xff\xff\xd8\xff\xff\xff\xf3\xff\xff\xff\xdc\xff\xff\xff\x5e\xff\xff\xff\x00\xff\xff\xff\xec\xff\xff\xff\x36\x83\xff\xff\xff\x00\x01\xff\xff\xff\xf8\xff\xff\xff\x24\x8f\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xff\x82\xff\xff\xff\xf3\x04\xff\xff\xff\xf4\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xff\x82\xff\xff\xff\xf3\x01\xff\xff\xff\xf4\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x03\xff\xff\xff\x76\xff\xff\xff\xec\xff\xff\xff\xf0\xff\xff\xff\xb3\x8b\xff\xff\xff\x00\x04\xff\xff\xff\x01\xff\xff\xff\x87\xff\xff\xff\xe9\xff\xff\xff\xf0\xff\xff\xff\x98\x99\xff\xff\xff\x00\xff\xff\xff\xff\x00\xff\xff\xff\xff\x00\xff\xff\xff\xff\x00\xff\xff\xff\xff\x00\xd1\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\xab\xff\xff\xff\x00\x01\xff\xff\xff\x25\xff\xff\xff\xf6\x82\xff\xff\xff\xef\x04\xff\xff\xff\xf0\xff\xff\xff\xef\xff\xff\xff\x00\xff\xff\xff\x25\xff\xff\xff\xf6\x82\xff\xff\xff\xef\x01\xff\xff\xff\xf0\xff\xff\xff\xef\xb0\xff\xff\xff\x00\x01\xff\xff\xff\x25\xff\xff\xff\xf6\x82\xff\xff\xff\xef\x04\xff\xff\xff\xf0\xff\xff\xff\xef\xff\xff\xff\x00\xff\xff\xff\x25\xff\xff\xff\xf6\x82\xff\xff\xff\xef\x01\xff\xff\xff\xf0\xff\xff\xff\xef\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x86\xff\xff\xff\x00\x03\xff\xff\xff\x25\xff\xff\xff\xef\xff\xff\xff\x25\xff\xff\xff\xeb\xa0\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\xb0\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x86\xff\xff\xff\x00\x03\xff\xff\xff\x27\xff\xff\xff\xfb\xff\xff\xff\x27\xff\xff\xff\xf7\xa0\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x0d\xff\xff\xff\x18\xff\xff\xff\xaf\xff\xff\xff\xf2\xff\xff\xff\xe9\xff\xff\xff\x97\xff\xff\xff\xa9\xff\xff\xff\xb9\xff\xff\xff\x00\xff\xff\xff\x0d\xff\xff\xff\xa4\xff\xff\xff\xed\xff\xff\xff\xf0\xff\xff\xff\xab\xff\xff\xff\x11\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x86\xff\xff\xff\x00\x02\xff\xff\xff\x3f\xff\xff\xff\xff\xff\xff\xff\x77\x84\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x85\xff\xff\xff\x00\x02\xff\xff\xff\x3e\xff\xff\xff\xff\xff\xff\xff\x79\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x86\xff\xff\xff\x00\x0b\xff\xff\xff\x27\xff\xff\xff\xfb\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\xc1\xff\xff\xff\x54\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xff\xff\xff\xff\xf3\xff\xff\xff\xeb\x81\xff\xff\xff\x00\x04\xff\xff\xff\x0d\xff\xff\xff\xa8\xff\xff\xff\xf1\xff\xff\xff\xea\xff\xff\xff\x88\x82\xff\xff\xff\x00\x0c\xff\xff\xff\x8f\xff\xff\xff\xf3\xff\xff\xff\xd6\xff\xff\xff\x5d\xff\xff\xff\xdf\xff\xff\xff\xf1\xff\xff\xff\x93\xff\xff\xff\x00\xff\xff\xff\x28\xff\xff\xff\xcd\xff\xff\xff\xf3\xff\xff\xff\xb8\xff\xff\xff\x12\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x0d\xff\xff\xff\xc7\xff\xff\xff\x96\xff\xff\xff\x0b\xff\xff\xff\x09\xff\xff\xff\xa6\xff\xff\xff\xfd\xff\xff\xff\x20\xff\xff\xff\x00\xff\xff\xff\x9a\xff\xff\xff\x8c\xff\xff\xff\x05\xff\xff\xff\x0a\xff\xff\xff\xa8\xff\xff\xff\xa5\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x86\xff\xff\xff\x00\x02\xff\xff\xff\x88\xff\xff\xff\xc1\xff\xff\xff\xbf\x84\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x85\xff\xff\xff\x00\x02\xff\xff\xff\x93\xff\xff\xff\xbf\xff\xff\xff\xce\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x86\xff\xff\xff\x00\x09\xff\xff\xff\x27\xff\xff\xff\xfb\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x7d\xff\xff\xff\x9e\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x83\xff\xff\xff\x00\x14\xff\xff\xff\x99\xff\xff\xff\xaa\xff\xff\xff\x08\xff\xff\xff\x12\xff\xff\xff\xcb\xff\xff\xff\x65\xff\xff\xff\x00\xff\xff\xff\x19\xff\xff\xff\xff\xff\xff\xff\x1f\xff\xff\xff\x56\xff\xff\xff\xff\xff\xff\xff\x3f\xff\xff\xff\x01\xff\xff\xff\x32\xff\xff\xff\x00\xff\xff\xff\xc7\xff\xff\xff\x6e\xff\xff\xff\x01\xff\xff\xff\x98\xff\xff\xff\x99\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x0d\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x1b\xff\xff\xff\xef\xff\xff\xff\x04\xff\xff\xff\x00\xff\xff\xff\x1a\xff\xff\xff\xe9\xff\xff\xff\xbe\xff\xff\xff\x7d\xff\xff\xff\x00\xff\xff\xff\xf0\xff\xff\xff\x0b\x81\xff\xff\xff\x00\x01\xff\xff\xff\x35\xff\xff\xff\xeb\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x86\xff\xff\xff\x00\x03\xff\xff\xff\xd1\xff\xff\xff\x43\xff\xff\xff\xf3\xff\xff\xff\x0d\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x84\xff\xff\xff\x00\x04\xff\xff\xff\x01\xff\xff\xff\xe6\xff\xff\xff\x2d\xff\xff\xff\xec\xff\xff\xff\x23\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x8b\xff\xff\xff\x00\x09\xff\xff\xff\x27\xff\xff\xff\xfb\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x38\xff\xff\xff\xfd\xff\xff\xff\xef\xff\xff\xff\xf1\xff\xff\xff\xfb\x83\xff\xff\xff\x00\x01\xff\xff\xff\xf0\xff\xff\xff\x2e\x81\xff\xff\xff\x00\x07\xff\xff\xff\x5a\xff\xff\xff\xbf\xff\xff\xff\x00\xff\xff\xff\x14\xff\xff\xff\xfd\xff\xff\xff\x31\xff\xff\xff\x29\xff\xff\xff\xfa\x82\xff\xff\xff\x00\x05\xff\xff\xff\x11\xff\xff\xff\xff\xff\xff\xff\x0f\xff\xff\xff\x00\xff\xff\xff\x3b\xff\xff\xff\xe1\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x0d\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x11\xff\xff\xff\xe2\xff\xff\xff\x00\xff\xff\xff\x01\xff\xff\xff\xbe\xff\xff\xff\x81\xff\xff\xff\x37\xff\xff\xff\xd2\xff\xff\xff\x19\xff\xff\xff\xff\xff\xff\xff\x0b\x81\xff\xff\xff\x00\x01\xff\xff\xff\x39\xff\xff\xff\xea\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x85\xff\xff\xff\x00\x04\xff\xff\xff\x1b\xff\xff\xff\xf2\xff\xff\xff\x04\xff\xff\xff\xc0\xff\xff\xff\x51\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x84\xff\xff\xff\x00\x04\xff\xff\xff\x3d\xff\xff\xff\xd5\xff\xff\xff\x00\xff\xff\xff\x9e\xff\xff\xff\x78\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x01\xff\xff\xff\x0e\xff\xff\xff\x5d\x85\xff\xff\xff\x00\x0a\xff\xff\xff\x43\xff\xff\xff\x8c\xff\xff\xff\xfb\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x02\xff\xff\xff\xf0\xff\xff\xff\x26\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x02\xff\xff\xff\x19\xff\xff\xff\xff\xff\xff\xff\x06\x81\xff\xff\xff\x00\x01\xff\xff\xff\x33\xff\xff\xff\xe7\x81\xff\xff\xff\x00\x08\xff\xff\xff\x72\xff\xff\xff\xe4\xff\xff\xff\xf2\xff\xff\xff\xff\xff\xff\xff\xef\xff\xff\xff\xf0\xff\xff\xff\xee\xff\xff\xff\x24\xff\xff\xff\xfd\x81\xff\xff\xff\x00\x01\xff\xff\xff\x2a\xff\xff\xff\xf3\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x0d\xff\xff\xff\xad\xff\xff\xff\x4b\xff\xff\xff\x7a\xff\xff\xff\xba\xff\xff\xff\x01\xff\xff\xff\x27\xff\xff\xff\xf1\xff\xff\xff\x24\xff\xff\xff\xff\xff\xff\xff\x8d\xff\xff\xff\x08\xff\xff\xff\x0d\xff\xff\xff\xae\xff\xff\xff\xa2\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xff\x82\xff\xff\xff\xef\x00\xff\xff\xff\x86\x81\xff\xff\xff\x00\x04\xff\xff\xff\x63\xff\xff\xff\xb6\xff\xff\xff\x00\xff\xff\xff\x7e\xff\xff\xff\x9a\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x84\xff\xff\xff\x00\x04\xff\xff\xff\x92\xff\xff\xff\x82\xff\xff\xff\x00\xff\xff\xff\x4b\xff\xff\xff\xcd\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x84\xff\xff\xff\x00\x00\xff\xff\xff\x90\x81\xff\xff\xff\xff\x02\xff\xff\xff\xfb\xff\xff\xff\x27\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x08\xff\xff\xff\xaf\xff\xff\xff\x62\xff\xff\xff\x27\xff\xff\xff\xff\xff\xff\xff\xef\xff\xff\xff\xa0\xff\xff\xff\x00\xff\xff\xff\x24\xff\xff\xff\xfc\x82\xff\xff\xff\x00\x01\xff\xff\xff\x2a\xff\xff\xff\xf3\x83\xff\xff\xff\x00\x0a\xff\xff\xff\x2a\xff\xff\xff\xf9\xff\xff\xff\x00\xff\xff\xff\x2a\xff\xff\xff\xec\xff\xff\xff\x12\xff\xff\xff\xff\xff\xff\xff\x10\xff\xff\xff\x00\xff\xff\xff\x3d\xff\xff\xff\xe1\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x03\xff\xff\xff\x10\xff\xff\xff\xc8\xff\xff\xff\xde\xff\xff\xff\x14\x82\xff\xff\xff\x00\x06\xff\xff\xff\x15\xff\xff\xff\xff\xff\xff\xff\x66\xff\xff\xff\xdd\xff\xff\xff\xeb\xff\xff\xff\xa6\xff\xff\xff\x10\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x85\xff\xff\xff\x00\x04\xff\xff\xff\xac\xff\xff\xff\x75\xff\xff\xff\x00\xff\xff\xff\x3b\xff\xff\xff\xe3\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x83\xff\xff\xff\x00\x09\xff\xff\xff\x01\xff\xff\xff\xe5\xff\xff\xff\x2f\xff\xff\xff\x00\xff\xff\xff\x06\xff\xff\xff\xf0\xff\xff\xff\x22\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x83\xff\xff\xff\x00\x01\xff\xff\xff\x10\xff\xff\xff\xfc\x81\xff\xff\xff\xff\x02\xff\xff\xff\xfb\xff\xff\xff\x27\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x03\xff\xff\xff\x6b\xff\xff\xff\x9e\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x02\xff\xff\xff\x1b\xff\xff\xff\xff\xff\xff\xff\x02\x81\xff\xff\xff\x00\x01\xff\xff\xff\x3a\xff\xff\xff\xe8\x81\xff\xff\xff\x00\x0c\xff\xff\xff\x2e\xff\xff\xff\x03\xff\xff\xff\x6d\xff\xff\xff\xff\xff\xff\xff\x2d\xff\xff\xff\x60\xff\xff\xff\xc6\xff\xff\xff\x00\xff\xff\xff\xcc\xff\xff\xff\x76\xff\xff\xff\x02\xff\xff\xff\xa0\xff\xff\xff\x99\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x02\xff\xff\xff\x08\xff\xff\xff\xec\xff\xff\xff\x38\x84\xff\xff\xff\x00\x01\xff\xff\xff\xe1\xff\xff\xff\x42\x85\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x84\xff\xff\xff\x00\x06\xff\xff\xff\x04\xff\xff\xff\xf0\xff\xff\xff\x33\xff\xff\xff\x00\xff\xff\xff\x04\xff\xff\xff\xf3\xff\xff\xff\x2c\x81\xff\xff\xff\x00\x02\xff\xff\xff\xdc\xff\xff\xff\xf4\xff\xff\xff\xff\x81\xff\xff\xff\xf3\x03\xff\xff\xff\x3c\xff\xff\xff\x00\xff\xff\xff\x3c\xff\xff\xff\xdc\x82\xff\xff\xff\x00\x04\xff\xff\xff\xa5\xff\xff\xff\x78\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x83\xff\xff\xff\x00\x00\xff\xff\xff\x1e\x82\xff\xff\xff\xff\x02\xff\xff\xff\xfb\xff\xff\xff\x27\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x03\xff\xff\xff\x26\xff\xff\xff\xda\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x02\xff\xff\xff\x02\xff\xff\xff\xf5\xff\xff\xff\x1e\x81\xff\xff\xff\x00\x01\xff\xff\xff\x76\xff\xff\xff\xc1\x81\xff\xff\xff\x00\x0c\xff\xff\xff\x80\xff\xff\xff\xee\xff\xff\xff\xdf\xff\xff\xff\x7e\xff\xff\xff\xe8\xff\xff\xff\xe9\xff\xff\xff\x49\xff\xff\xff\x00\xff\xff\xff\x2c\xff\xff\xff\xcf\xff\xff\xff\xf3\xff\xff\xff\xb9\xff\xff\xff\x12\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x04\xff\xff\xff\x1b\xff\xff\xff\xfd\xff\xff\xff\x26\xff\xff\xff\x00\xff\xff\xff\x21\x82\xff\xff\xff\x00\x05\xff\xff\xff\x6f\xff\xff\xff\xd4\xff\xff\xff\x2a\xff\xff\xff\x00\xff\xff\xff\x12\xff\xff\xff\x1a\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x84\xff\xff\xff\x00\x02\xff\xff\xff\x3e\xff\xff\xff\xee\xff\xff\xff\x01\x81\xff\xff\xff\x00\x01\xff\xff\xff\xb7\xff\xff\xff\x75\x82\xff\xff\xff\x00\x01\xff\xff\xff\x25\xff\xff\xff\xfb\x8b\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x84\xff\xff\xff\x00\x00\xff\xff\xff\xd3\x81\xff\xff\xff\xff\x02\xff\xff\xff\xfb\xff\xff\xff\x27\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x02\xff\xff\xff\xe1\xff\xff\xff\x3e\xff\xff\xff\xfb\x83\xff\xff\xff\x00\x05\xff\xff\xff\xac\xff\xff\xff\x81\xff\xff\xff\x06\xff\xff\xff\x48\xff\xff\xff\xee\xff\xff\xff\x69\x90\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xff\x82\xff\xff\xff\xf3\x04\xff\xff\xff\xf4\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xff\x82\xff\xff\xff\xf3\x01\xff\xff\xff\xf4\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x03\xff\xff\xff\x7d\xff\xff\xff\xe8\xff\xff\xff\xf0\xff\xff\xff\xa9\x83\xff\xff\xff\x00\x04\xff\xff\xff\x73\xff\xff\xff\xdd\xff\xff\xff\xf3\xff\xff\xff\xe0\xff\xff\xff\x40\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xff\x82\xff\xff\xff\xf3\x03\xff\xff\xff\xe3\xff\xff\xff\x00\xff\xff\xff\x87\xff\xff\xff\xaf\x82\xff\xff\xff\x00\x01\xff\xff\xff\x74\xff\xff\xff\xbe\x82\xff\xff\xff\x00\x02\xff\xff\xff\x0d\xff\xff\xff\xfb\xff\xff\xff\x26\x8a\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xff\x82\xff\xff\xff\xf3\x04\xff\xff\xff\xf4\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xff\x82\xff\xff\xff\xf3\x01\xff\xff\xff\xf4\xff\xff\xff\xf7\x89\xff\xff\xff\x00\x05\xff\xff\xff\x1f\xff\xff\xff\xb2\xff\xff\xff\xf3\xff\xff\xff\xff\xff\xff\xff\xf4\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x04\xff\xff\xff\x9d\xff\xff\xff\xf7\xff\xff\xff\xff\xff\xff\xff\xf3\xff\xff\xff\xcc\x81\xff\xff\xff\x00\x04\xff\xff\xff\x1c\xff\xff\xff\xc6\xff\xff\xff\xf8\xff\xff\xff\xe0\xff\xff\xff\x81\x91\xff\xff\xff\x00\xb2\xff\xff\xff\x00\x03\xff\xff\xff\x79\xff\xff\xff\xf0\xff\xff\xff\xf3\xff\xff\xff\x3c\xc1\xff\xff\xff\x00\x03\xff\xff\xff\x15\xff\xff\xff\xcc\xff\xff\xff\x06\xff\xff\xff\xdc\x82\xff\xff\xff\x00\xe8\xff\xff\xff\x00\x04\xff\xff\xff\x0f\xff\xff\xff\xd7\xff\xff\xff\x07\xff\xff\xff\xff\xff\xff\xff\xe3\x91\xff\xff\xff\x00\xe8\xff\xff\xff\x00\x04\xff\xff\xff\x0f\xff\xff\xff\xd7\xff\xff\xff\x07\xff\xff\xff\xff\xff\xff\xff\xe3\x91\xff\xff\xff\x00\xff\xff\xff\xff\x00\xd7\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\xa5\xff\xff\xff\x00\x01\xff\xff\xff\x25\xff\xff\xff\xf6\x82\xff\xff\xff\xef\x04\xff\xff\xff\xf0\xff\xff\xff\xef\xff\xff\xff\x00\xff\xff\xff\x25\xff\xff\xff\xf6\x82\xff\xff\xff\xef\x01\xff\xff\xff\xf0\xff\xff\xff\xef\xb0\xff\xff\xff\x00\x01\xff\xff\xff\x25\xff\xff\xff\xf6\x82\xff\xff\xff\xef\x04\xff\xff\xff\xf0\xff\xff\xff\xef\xff\xff\xff\x00\xff\xff\xff\x25\xff\xff\xff\xf6\x82\xff\xff\xff\xef\x01\xff\xff\xff\xf0\xff\xff\xff\xef\x88\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\xa5\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\xb0\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x88\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\xa5\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x84\xff\xff\xff\x00\x09\xff\xff\xff\x7a\xff\xff\xff\xf1\xff\xff\xff\xd6\xff\xff\xff\x35\xff\xff\xff\x19\xff\xff\xff\xd6\xff\xff\xff\xf4\xff\xff\xff\xf0\xff\xff\xff\xa7\xff\xff\xff\x0f\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xff\x83\xff\xff\xff\xf3\x00\xff\xff\xff\x15\x81\xff\xff\xff\x00\x04\xff\xff\xff\x41\xff\xff\xff\xd9\xff\xff\xff\xf4\xff\xff\xff\xc4\xff\xff\xff\x25\x82\xff\xff\xff\x00\x05\xff\xff\xff\x0b\xff\xff\xff\x96\xff\xff\xff\xe8\xff\xff\xff\xf3\xff\xff\xff\xd9\xff\xff\xff\x6e\x81\xff\xff\xff\x00\x04\xff\xff\xff\x6d\xff\xff\xff\xe7\xff\xff\xff\xc0\xff\xff\xff\x84\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x83\xff\xff\xff\x00\x09\xff\xff\xff\x27\xff\xff\xff\xfb\xff\xff\xff\xa3\xff\xff\xff\xf3\xff\xff\xff\x9b\xff\xff\xff\xc8\xff\xff\xff\xa5\xff\xff\xff\x00\xff\xff\xff\xc3\xff\xff\xff\x71\x82\xff\xff\xff\x00\x01\xff\xff\xff\xa2\xff\xff\xff\x97\x81\xff\xff\xff\x00\x04\xff\xff\xff\x0d\xff\xff\xff\xa8\xff\xff\xff\xf1\xff\xff\xff\xea\xff\xff\xff\x88\x82\xff\xff\xff\x00\x05\xff\xff\xff\x7f\xff\xff\xff\xf1\xff\xff\xff\xe5\xff\xff\xff\x6d\xff\xff\xff\xff\xff\xff\xff\x1b\x81\xff\xff\xff\x00\x04\xff\xff\xff\x28\xff\xff\xff\xcc\xff\xff\xff\xf3\xff\xff\xff\xb8\xff\xff\xff\x12\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x83\xff\xff\xff\x00\x0a\xff\xff\xff\x15\xff\xff\xff\xfc\xff\xff\xff\x25\xff\xff\xff\x57\xff\xff\xff\xd9\xff\xff\xff\x13\xff\xff\xff\x3c\xff\xff\xff\x01\xff\xff\xff\x0c\xff\xff\xff\xad\xff\xff\xff\xa1\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x85\xff\xff\xff\x00\x05\xff\xff\xff\x02\xff\xff\xff\xe8\xff\xff\xff\x4c\xff\xff\xff\x00\xff\xff\xff\x79\xff\xff\xff\xbe\x82\xff\xff\xff\x00\x0c\xff\xff\xff\xa5\xff\xff\xff\x9b\xff\xff\xff\x0e\xff\xff\xff\x00\xff\xff\xff\x1f\xff\xff\xff\x4b\xff\xff\xff\x00\xff\xff\xff\x07\xff\xff\xff\xf6\xff\xff\xff\x38\xff\xff\xff\x00\xff\xff\xff\x8a\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x83\xff\xff\xff\x00\x09\xff\xff\xff\x27\xff\xff\xff\xff\xff\xff\xff\x5b\xff\xff\xff\x02\xff\xff\xff\x98\xff\xff\xff\xfa\xff\xff\xff\x09\xff\xff\xff\x00\xff\xff\xff\x6b\xff\xff\xff\xc5\x81\xff\xff\xff\x00\x02\xff\xff\xff\x07\xff\xff\xff\xf1\xff\xff\xff\x3f\x81\xff\xff\xff\x00\x0d\xff\xff\xff\x99\xff\xff\xff\xaa\xff\xff\xff\x08\xff\xff\xff\x12\xff\xff\xff\xcb\xff\xff\xff\x65\xff\xff\xff\x00\xff\xff\xff\x17\xff\xff\xff\xef\xff\xff\xff\x1b\xff\xff\xff\x04\xff\xff\xff\x8b\xff\xff\xff\xff\xff\xff\xff\x1b\x81\xff\xff\xff\x00\x04\xff\xff\xff\xc7\xff\xff\xff\x75\xff\xff\xff\x02\xff\xff\xff\x9f\xff\xff\xff\x99\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x06\xff\xff\xff\x0a\xff\xff\xff\x00\xff\xff\xff\x13\xff\xff\xff\xfb\xff\xff\xff\x2a\xff\xff\xff\x5b\xff\xff\xff\xd1\x83\xff\xff\xff\x00\x01\xff\xff\xff\x36\xff\xff\xff\xea\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x85\xff\xff\xff\x00\x01\xff\xff\xff\x1d\xff\xff\xff\xfd\x81\xff\xff\xff\x00\x01\xff\xff\xff\x2a\xff\xff\xff\xed\x81\xff\xff\xff\x00\x02\xff\xff\xff\x0c\xff\xff\xff\xfb\xff\xff\xff\x0c\x84\xff\xff\xff\x00\x05\xff\xff\xff\x23\xff\xff\xff\xfd\xff\xff\xff\x01\xff\xff\xff\x00\xff\xff\xff\x35\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x83\xff\xff\xff\x00\x05\xff\xff\xff\x27\xff\xff\xff\xfd\xff\xff\xff\x04\xff\xff\xff\x00\xff\xff\xff\x35\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x06\xff\xff\xff\x16\xff\xff\xff\xfb\xff\xff\xff\xf2\xff\xff\xff\xef\xff\xff\xff\xf5\xff\xff\xff\xe4\xff\xff\xff\x01\x81\xff\xff\xff\x00\x01\xff\xff\xff\xf0\xff\xff\xff\x2e\x81\xff\xff\xff\x00\x10\xff\xff\xff\x5a\xff\xff\xff\xbf\xff\xff\xff\x00\xff\xff\xff\x11\xff\xff\xff\xef\xff\xff\xff\x30\xff\xff\xff\x00\xff\xff\xff\x15\xff\xff\xff\xff\xff\xff\xff\x1b\xff\xff\xff\x00\xff\xff\xff\x11\xff\xff\xff\xff\xff\xff\xff\x10\xff\xff\xff\x00\xff\xff\xff\x3d\xff\xff\xff\xe1\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x06\xff\xff\xff\x5d\xff\xff\xff\xb1\xff\xff\xff\x4c\xff\xff\xff\x71\xff\xff\xff\xea\xff\xff\xff\xbf\xff\xff\xff\x1f\x83\xff\xff\xff\x00\x01\xff\xff\xff\x39\xff\xff\xff\xe9\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x85\xff\xff\xff\x00\x01\xff\xff\xff\x26\xff\xff\xff\xfb\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf6\x81\xff\xff\xff\x00\x01\xff\xff\xff\x22\xff\xff\xff\xff\x82\xff\xff\xff\xef\x04\xff\xff\xff\xf0\xff\xff\xff\xed\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x06\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x77\xff\xff\xff\xef\xff\xff\xff\xf1\xff\xff\xff\xff\x81\xff\xff\xff\xef\x03\xff\xff\xff\x4d\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x04\xff\xff\xff\xbb\xff\xff\xff\x82\xff\xff\xff\x00\xff\xff\xff\xb3\xff\xff\xff\x8f\x81\xff\xff\xff\x00\x02\xff\xff\xff\x19\xff\xff\xff\xff\xff\xff\xff\x06\x81\xff\xff\xff\x00\x01\xff\xff\xff\x33\xff\xff\xff\xe7\x81\xff\xff\xff\x00\x08\xff\xff\xff\x5c\xff\xff\xff\xd5\xff\xff\xff\xf0\xff\xff\xff\xef\xff\xff\xff\xff\xff\xff\xff\x1a\xff\xff\xff\x00\xff\xff\xff\x24\xff\xff\xff\xfd\x81\xff\xff\xff\x00\x01\xff\xff\xff\x2a\xff\xff\xff\xf3\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x0b\xff\xff\xff\x0b\xff\xff\xff\x71\xff\xff\xff\xb7\xff\xff\xff\x5f\xff\xff\xff\x04\xff\xff\xff\x00\xff\xff\xff\x0e\xff\xff\xff\x28\xff\xff\xff\x00\xff\xff\xff\x19\xff\xff\xff\xbc\xff\xff\xff\x9b\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xff\x82\xff\xff\xff\xef\x00\xff\xff\xff\xc2\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x02\xff\xff\xff\x0b\xff\xff\xff\xfb\xff\xff\xff\x28\x81\xff\xff\xff\x00\x04\xff\xff\xff\x3a\xff\xff\xff\xde\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x02\xff\xff\xff\x54\xff\xff\xff\xff\xff\xff\xff\x26\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x04\xff\xff\xff\x63\xff\xff\xff\xb5\xff\xff\xff\x00\xff\xff\xff\xe6\xff\xff\xff\x37\x81\xff\xff\xff\x00\x01\xff\xff\xff\x24\xff\xff\xff\xfc\x82\xff\xff\xff\x00\x01\xff\xff\xff\x29\xff\xff\xff\xf3\x84\xff\xff\xff\x00\x09\xff\xff\xff\x0e\xff\xff\xff\xff\xff\xff\xff\x15\xff\xff\xff\x00\xff\xff\xff\x12\xff\xff\xff\xff\xff\xff\xff\x10\xff\xff\xff\x00\xff\xff\xff\x3d\xff\xff\xff\xe1\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x0c\xff\xff\xff\x69\xff\xff\xff\xe7\xff\xff\xff\xdd\xff\xff\xff\x4c\xff\xff\xff\x5f\xff\xff\xff\xb8\xff\xff\xff\x54\xff\xff\xff\x27\xff\xff\xff\xff\xff\xff\xff\xe4\xff\xff\xff\xe9\xff\xff\xff\x9b\xff\xff\xff\x0b\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x85\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x08\xff\xff\xff\xa2\xff\xff\xff\xb6\xff\xff\xff\x0f\xff\xff\xff\x08\xff\xff\xff\xab\xff\xff\xff\x8b\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x0c\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x53\xff\xff\xff\xac\xff\xff\xff\xef\xff\xff\xff\xae\xff\xff\xff\xe2\xff\xff\xff\xa7\xff\xff\xff\x36\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x03\xff\xff\xff\x10\xff\xff\xff\xe3\xff\xff\xff\x1a\xff\xff\xff\xde\x82\xff\xff\xff\x00\x02\xff\xff\xff\x1a\xff\xff\xff\xff\xff\xff\xff\x07\x81\xff\xff\xff\x00\x01\xff\xff\xff\x34\xff\xff\xff\xe9\x81\xff\xff\xff\x00\x05\xff\xff\xff\x6a\xff\xff\xff\x3a\xff\xff\xff\x1b\xff\xff\xff\x82\xff\xff\xff\xe9\xff\xff\xff\x01\x81\xff\xff\xff\x00\x04\xff\xff\xff\xcc\xff\xff\xff\x76\xff\xff\xff\x02\xff\xff\xff\xa0\xff\xff\xff\x99\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x07\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x13\xff\xff\xff\xfb\xff\xff\xff\x2b\xff\xff\xff\x57\xff\xff\xff\xe2\x81\xff\xff\xff\x00\x02\xff\xff\xff\x13\xff\xff\xff\x27\xff\xff\xff\xfb\x85\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x85\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x08\xff\xff\xff\x0a\xff\xff\xff\x98\xff\xff\xff\xea\xff\xff\xff\xef\xff\xff\xff\x9f\xff\xff\xff\x08\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x04\xff\xff\xff\x79\xff\xff\xff\xb7\xff\xff\xff\x04\xff\xff\xff\xda\xff\xff\xff\x43\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x83\xff\xff\xff\x00\x02\xff\xff\xff\xb3\xff\xff\xff\x6c\xff\xff\xff\x87\x82\xff\xff\xff\x00\x02\xff\xff\xff\x01\xff\xff\xff\xf3\xff\xff\xff\x31\x81\xff\xff\xff\x00\x01\xff\xff\xff\x5e\xff\xff\xff\xc5\x81\xff\xff\xff\x00\x04\xff\xff\xff\x6b\xff\xff\xff\xdd\xff\xff\xff\xf9\xff\xff\xff\xd3\xff\xff\xff\x3f\x82\xff\xff\xff\x00\x04\xff\xff\xff\x2c\xff\xff\xff\xcf\xff\xff\xff\xf3\xff\xff\xff\xb9\xff\xff\xff\x12\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x07\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x14\xff\xff\xff\xfc\xff\xff\xff\x2b\xff\xff\xff\x57\xff\xff\xff\xb8\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x85\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x85\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x91\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x09\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x1b\xff\xff\xff\xf0\xff\xff\xff\x2a\xff\xff\xff\x00\xff\xff\xff\x51\xff\xff\xff\xd9\xff\xff\xff\x06\x8a\xff\xff\xff\x00\x02\xff\xff\xff\x5b\xff\xff\xff\xd3\xff\xff\xff\x2f\x83\xff\xff\xff\x00\x05\xff\xff\xff\xa2\xff\xff\xff\xb2\xff\xff\xff\x11\xff\xff\xff\x1e\xff\xff\xff\xd1\xff\xff\xff\x72\x83\xff\xff\xff\x00\x00\xff\xff\xff\x01\x8b\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xff\x82\xff\xff\xff\xf3\x04\xff\xff\xff\xf4\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xff\x82\xff\xff\xff\xf3\x01\xff\xff\xff\xf4\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x03\xff\xff\xff\x74\xff\xff\xff\xeb\xff\xff\xff\xaf\xff\xff\xff\x0c\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xff\x82\xff\xff\xff\xf3\x00\xff\xff\xff\x5a\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xff\x82\xff\xff\xff\xf3\x00\xff\xff\xff\xef\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x91\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xff\x82\xff\xff\xff\xf3\x04\xff\xff\xff\xf4\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xff\x82\xff\xff\xff\xf3\x04\xff\xff\xff\xf4\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\xa7\xff\xff\xff\x97\x82\xff\xff\xff\x00\x01\xff\xff\xff\xc1\xff\xff\xff\x7c\x8a\xff\xff\xff\x00\x02\xff\xff\xff\x41\xff\xff\xff\xf4\xff\xff\xff\x1f\x83\xff\xff\xff\x00\x05\xff\xff\xff\x13\xff\xff\xff\xaf\xff\xff\xff\xf4\xff\xff\xff\xee\xff\xff\xff\x96\xff\xff\xff\x03\x81\xff\xff\xff\x00\x03\xff\xff\xff\x02\xff\xff\xff\xa5\xff\xff\xff\xb2\xff\xff\xff\x6d\x83\xff\xff\xff\x00\x03\xff\xff\xff\x3b\xff\xff\xff\x72\xff\xff\xff\xa3\xff\xff\xff\xbc\x82\xff\xff\xff\x00\xe0\xff\xff\xff\x00\x03\xff\xff\xff\x0d\xff\xff\xff\xf2\xff\xff\xff\x56\xff\xff\xff\xd2\x8b\xff\xff\xff\x00\x03\xff\xff\xff\x17\xff\xff\xff\xfa\xff\xff\xff\x43\xff\xff\xff\xeb\x83\xff\xff\xff\x00\x04\xff\xff\xff\x0a\xff\xff\xff\xe1\xff\xff\xff\x6a\xff\xff\xff\xa0\xff\xff\xff\x0c\x81\xff\xff\xff\x00\xe0\xff\xff\xff\x00\x03\xff\xff\xff\x14\xff\xff\xff\xfb\xff\xff\xff\x50\xff\xff\xff\xdc\x83\xff\xff\xff\x00\x04\xff\xff\xff\x26\xff\xff\xff\x81\xff\xff\xff\x36\xff\xff\xff\xba\xff\xff\xff\x72\x83\xff\xff\xff\x00\x02\xff\xff\xff\x6a\xff\xff\xff\xe2\xff\xff\xff\x5c\x8a\xff\xff\xff\x00\xe1\xff\xff\xff\x00\x02\xff\xff\xff\x66\xff\xff\xff\xde\xff\xff\xff\x3d\x83\xff\xff\xff\x00\x05\xff\xff\xff\x03\xff\xff\xff\xc3\xff\xff\xff\xd3\xff\xff\xff\x41\xff\xff\xff\xc2\xff\xff\xff\x02\x90\xff\xff\xff\x00\xff\xff\xff\xff\x00\xff\xff\xff\xff\x00\x01\xff\xff\xff\x25\xff\xff\xff\xf6\x82\xff\xff\xff\xef\x04\xff\xff\xff\xf0\xff\xff\xff\xef\xff\xff\xff\x00\xff\xff\xff\x25\xff\xff\xff\xf6\x82\xff\xff\xff\xef\x01\xff\xff\xff\xf0\xff\xff\xff\xef\xb0\xff\xff\xff\x00\x01\xff\xff\xff\x25\xff\xff\xff\xf6\x82\xff\xff\xff\xef\x04\xff\xff\xff\xf0\xff\xff\xff\xef\xff\xff\xff\x00\xff\xff\xff\x25\xff\xff\xff\xf6\x82\xff\xff\xff\xef\x01\xff\xff\xff\xf0\xff\xff\xff\xef\xb0\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\xab\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\xb0\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x08\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x10\xff\xff\xff\xaa\xff\xff\xff\xf1\xff\xff\xff\xff\xff\xff\xff\xd9\xff\xff\xff\x2b\x85\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x81\xff\xff\xff\x00\x05\xff\xff\xff\x27\xff\xff\xff\xff\xff\xff\xff\xf3\xff\xff\xff\xf0\xff\xff\xff\xbe\xff\xff\xff\x43\x84\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x83\xff\xff\xff\x00\x04\xff\xff\xff\x2e\xff\xff\xff\xd7\xff\xff\xff\xec\xff\xff\xff\x8e\xff\xff\xff\xf7\x84\xff\xff\xff\x00\x06\xff\xff\xff\x88\xff\xff\xff\xec\xff\xff\xff\xe3\xff\xff\xff\x62\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x90\xff\xff\xff\x00\x01\xff\xff\xff\xd9\xff\xff\xff\x5f\x82\xff\xff\xff\x00\x01\xff\xff\xff\x8f\xff\xff\xff\xac\x81\xff\xff\xff\x00\x04\xff\xff\xff\x0d\xff\xff\xff\xa8\xff\xff\xff\xf1\xff\xff\xff\xea\xff\xff\xff\x88\x82\xff\xff\xff\x00\x04\xff\xff\xff\x83\xff\xff\xff\xf3\xff\xff\xff\xe0\xff\xff\xff\x78\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x04\xff\xff\xff\x28\xff\xff\xff\xcc\xff\xff\xff\xf3\xff\xff\xff\xb8\xff\xff\xff\x12\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x08\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x1d\xff\xff\xff\x69\xff\xff\xff\x2f\xff\xff\xff\xf7\xff\xff\xff\x60\xff\xff\xff\xc4\x85\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x81\xff\xff\xff\x00\x06\xff\xff\xff\x27\xff\xff\xff\xfb\xff\xff\xff\x00\xff\xff\xff\x06\xff\xff\xff\x4d\xff\xff\xff\xe9\xff\xff\xff\x3c\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x83\xff\xff\xff\x00\x04\xff\xff\xff\xc9\xff\xff\xff\x75\xff\xff\xff\x02\xff\xff\xff\xa2\xff\xff\xff\xf7\x83\xff\xff\xff\x00\x02\xff\xff\xff\x13\xff\xff\xff\xff\xff\xff\xff\x29\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x01\xff\xff\xff\x4f\xff\xff\xff\x1a\x81\xff\xff\xff\x00\x01\xff\xff\xff\x32\xff\xff\xff\x36\x88\xff\xff\xff\x00\x01\xff\xff\xff\x8d\xff\xff\xff\xaf\x82\xff\xff\xff\x00\x01\xff\xff\xff\xdf\xff\xff\xff\x60\x81\xff\xff\xff\x00\x0c\xff\xff\xff\x99\xff\xff\xff\xaa\xff\xff\xff\x08\xff\xff\xff\x12\xff\xff\xff\xcb\xff\xff\xff\x65\xff\xff\xff\x00\xff\xff\xff\x18\xff\xff\xff\xff\xff\xff\xff\x2c\xff\xff\xff\x05\xff\xff\xff\xa0\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x04\xff\xff\xff\xc7\xff\xff\xff\x75\xff\xff\xff\x02\xff\xff\xff\x9f\xff\xff\xff\x99\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x03\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x2d\xff\xff\xff\xf1\x81\xff\xff\xff\x00\x00\xff\xff\xff\x27\x85\xff\xff\xff\xff\x81\xff\xff\xff\x27\x00\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x71\xff\xff\xff\xb0\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x05\xff\xff\xff\x11\xff\xff\xff\xff\xff\xff\xff\x10\xff\xff\xff\x00\xff\xff\xff\x3e\xff\xff\xff\xf7\x83\xff\xff\xff\x00\x01\xff\xff\xff\x26\xff\xff\xff\xfb\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x05\xff\xff\xff\x40\xff\xff\xff\xdc\xff\xff\xff\xcd\xff\xff\xff\xbe\xff\xff\xff\xda\xff\xff\xff\x26\x88\xff\xff\xff\x00\x06\xff\xff\xff\x41\xff\xff\xff\xfd\xff\xff\xff\xf0\xff\xff\xff\xef\xff\xff\xff\xf2\xff\xff\xff\xfc\xff\xff\xff\x15\x81\xff\xff\xff\x00\x01\xff\xff\xff\xf0\xff\xff\xff\x2e\x81\xff\xff\xff\x00\x08\xff\xff\xff\x5a\xff\xff\xff\xbf\xff\xff\xff\x00\xff\xff\xff\x11\xff\xff\xff\xfb\xff\xff\xff\x41\xff\xff\xff\x00\xff\xff\xff\x34\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x05\xff\xff\xff\x11\xff\xff\xff\xff\xff\xff\xff\x10\xff\xff\xff\x00\xff\xff\xff\x3d\xff\xff\xff\xe1\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x03\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x83\xff\xff\xff\xc9\x81\xff\xff\xff\x00\x05\xff\xff\xff\x0a\xff\xff\xff\xde\xff\xff\xff\x30\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x38\xff\xff\xff\xe4\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x01\xff\xff\xff\x24\xff\xff\xff\xfd\x81\xff\xff\xff\x00\x01\xff\xff\xff\x2a\xff\xff\xff\xf7\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x04\xff\xff\xff\x0b\xff\xff\xff\xe3\xff\xff\xff\x2f\xff\xff\xff\x5b\xff\xff\xff\xbe\x89\xff\xff\xff\x00\x05\xff\xff\xff\x04\xff\xff\xff\xef\xff\xff\xff\x46\xff\xff\xff\x00\xff\xff\xff\x77\xff\xff\xff\xc7\x81\xff\xff\xff\x00\x02\xff\xff\xff\x19\xff\xff\xff\xff\xff\xff\xff\x06\x81\xff\xff\xff\x00\x01\xff\xff\xff\x33\xff\xff\xff\xe7\x81\xff\xff\xff\x00\x04\xff\xff\xff\x5f\xff\xff\xff\xd8\xff\xff\xff\xf0\xff\xff\xff\xf1\xff\xff\xff\xf6\x81\xff\xff\xff\x00\x01\xff\xff\xff\x24\xff\xff\xff\xfd\x81\xff\xff\xff\x00\x01\xff\xff\xff\x2a\xff\xff\xff\xf3\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x04\xff\xff\xff\x42\xff\xff\xff\xd6\xff\xff\xff\xff\xff\xff\xff\xdc\xff\xff\xff\x30\x82\xff\xff\xff\x00\x04\xff\xff\xff\x49\xff\xff\xff\xc7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x2a\xff\xff\xff\xf3\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x05\xff\xff\xff\x12\xff\xff\xff\xff\xff\xff\xff\x10\xff\xff\xff\x00\xff\xff\xff\x3e\xff\xff\xff\xf7\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x04\xff\xff\xff\x0c\xff\xff\xff\xe3\xff\xff\xff\x2b\xff\xff\xff\x56\xff\xff\xff\xc2\x8a\xff\xff\xff\x00\x04\xff\xff\xff\xa9\xff\xff\xff\x88\xff\xff\xff\x00\xff\xff\xff\xb9\xff\xff\xff\x7a\x81\xff\xff\xff\x00\x01\xff\xff\xff\x24\xff\xff\xff\xfc\x82\xff\xff\xff\x00\x01\xff\xff\xff\x29\xff\xff\xff\xf3\x84\xff\xff\xff\x00\x01\xff\xff\xff\x2d\xff\xff\xff\xf1\x81\xff\xff\xff\x00\x05\xff\xff\xff\x12\xff\xff\xff\xff\xff\xff\xff\x10\xff\xff\xff\x00\xff\xff\xff\x3d\xff\xff\xff\xe1\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x06\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x05\xff\xff\xff\xed\xff\xff\xff\x94\xff\xff\xff\xf7\x85\xff\xff\xff\x00\x03\xff\xff\xff\xa3\xff\xff\xff\x65\xff\xff\xff\x27\xff\xff\xff\xfb\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x39\xff\xff\xff\xe5\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x83\xff\xff\xff\x00\x04\xff\xff\xff\xcc\xff\xff\xff\x76\xff\xff\xff\x02\xff\xff\xff\xa4\xff\xff\xff\xf7\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x05\xff\xff\xff\x41\xff\xff\xff\xd9\xff\xff\xff\xcc\xff\xff\xff\xbf\xff\xff\xff\xd9\xff\xff\xff\x24\x89\xff\xff\xff\x00\x04\xff\xff\xff\x5d\xff\xff\xff\xca\xff\xff\xff\x05\xff\xff\xff\xf4\xff\xff\xff\x2e\x81\xff\xff\xff\x00\x02\xff\xff\xff\x1a\xff\xff\xff\xff\xff\xff\xff\x07\x81\xff\xff\xff\x00\x01\xff\xff\xff\x34\xff\xff\xff\xe9\x81\xff\xff\xff\x00\x04\xff\xff\xff\x54\xff\xff\xff\x1e\xff\xff\xff\x01\xff\xff\xff\x86\xff\xff\xff\xc9\x82\xff\xff\xff\x00\x04\xff\xff\xff\xcc\xff\xff\xff\x76\xff\xff\xff\x02\xff\xff\xff\xa0\xff\xff\xff\x99\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x06\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x22\xff\xff\xff\xff\xff\xff\xff\x2a\xff\xff\xff\xf7\x85\xff\xff\xff\x00\x03\xff\xff\xff\x12\xff\xff\xff\xd6\xff\xff\xff\x3c\xff\xff\xff\xfb\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x72\xff\xff\xff\xb5\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x83\xff\xff\xff\x00\x04\xff\xff\xff\x2f\xff\xff\xff\xd7\xff\xff\xff\xec\xff\xff\xff\x8e\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x02\xff\xff\xff\xdc\xff\xff\xff\xf4\xff\xff\xff\xff\x81\xff\xff\xff\xf3\x03\xff\xff\xff\x69\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x01\xff\xff\xff\x4d\xff\xff\xff\x1a\x81\xff\xff\xff\x00\x01\xff\xff\xff\x30\xff\xff\xff\x39\x89\xff\xff\xff\x00\x03\xff\xff\xff\x13\xff\xff\xff\xf9\xff\xff\xff\x4e\xff\xff\xff\xe0\x82\xff\xff\xff\x00\x02\xff\xff\xff\x01\xff\xff\xff\xf3\xff\xff\xff\x31\x81\xff\xff\xff\x00\x01\xff\xff\xff\x5e\xff\xff\xff\xc5\x81\xff\xff\xff\x00\x04\xff\xff\xff\x68\xff\xff\xff\xe0\xff\xff\xff\xf5\xff\xff\xff\xcb\xff\xff\xff\x2e\x82\xff\xff\xff\x00\x04\xff\xff\xff\x2c\xff\xff\xff\xcf\xff\xff\xff\xf3\xff\xff\xff\xb9\xff\xff\xff\x12\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x08\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x0b\xff\xff\xff\xf7\xff\xff\xff\x5a\xff\xff\xff\xf7\xff\xff\xff\x21\xff\xff\xff\x30\x84\xff\xff\xff\x00\x02\xff\xff\xff\x55\xff\xff\xff\xc4\xff\xff\xff\xfb\x81\xff\xff\xff\x00\x06\xff\xff\xff\x27\xff\xff\xff\xfb\xff\xff\xff\x00\xff\xff\xff\x06\xff\xff\xff\x4d\xff\xff\xff\xeb\xff\xff\xff\x47\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x86\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x92\xff\xff\xff\x00\x02\xff\xff\xff\xc5\xff\xff\xff\xce\xff\xff\xff\x95\x83\xff\xff\xff\x00\x05\xff\xff\xff\xa2\xff\xff\xff\xb2\xff\xff\xff\x11\xff\xff\xff\x1e\xff\xff\xff\xd1\xff\xff\xff\x72\x90\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xff\x82\xff\xff\xff\xf3\x04\xff\xff\xff\xf4\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xff\x82\xff\xff\xff\xf3\x01\xff\xff\xff\xf4\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x04\xff\xff\xff\x68\xff\xff\xff\xea\xff\xff\xff\xff\xff\xff\xff\xdc\xff\xff\xff\x40\x85\xff\xff\xff\x00\x01\xff\xff\xff\xae\xff\xff\xff\xfb\x81\xff\xff\xff\x00\x05\xff\xff\xff\x27\xff\xff\xff\xff\xff\xff\xff\xf3\xff\xff\xff\xf1\xff\xff\xff\xc5\xff\xff\xff\x50\x82\xff\xff\xff\x00\x81\xff\xff\xff\xf3\x01\xff\xff\xff\xf4\xff\xff\xff\xff\x82\xff\xff\xff\xf3\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xff\x82\xff\xff\xff\xf3\x04\xff\xff\xff\xf4\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xff\x82\xff\xff\xff\xf3\x01\xff\xff\xff\xf4\xff\xff\xff\xf7\x8a\xff\xff\xff\x00\x02\xff\xff\xff\x06\xff\xff\xff\xb8\xff\xff\xff\x36\x84\xff\xff\xff\x00\x02\xff\xff\xff\x79\xff\xff\xff\xff\xff\xff\xff\x49\x83\xff\xff\xff\x00\x05\xff\xff\xff\x13\xff\xff\xff\xaf\xff\xff\xff\xf4\xff\xff\xff\xee\xff\xff\xff\x96\xff\xff\xff\x03\x89\xff\xff\xff\x00\x03\xff\xff\xff\x08\xff\xff\xff\xae\xff\xff\xff\x20\xff\xff\xff\xa5\x82\xff\xff\xff\x00\x91\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x9f\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\xa4\xff\xff\xff\x00\x02\xff\xff\xff\x16\xff\xff\xff\xd0\xff\xff\xff\x2e\x93\xff\xff\xff\x00\x03\xff\xff\xff\xcc\xff\xff\xff\x06\xff\xff\xff\xf3\xff\xff\xff\x12\x83\xff\xff\xff\x00\x02\xff\xff\xff\x44\xff\xff\xff\xe4\xff\xff\xff\x21\x82\xff\xff\xff\x00\xe0\xff\xff\xff\x00\x04\xff\xff\xff\x51\xff\xff\xff\xc9\xff\xff\xff\x02\xff\xff\xff\xef\xff\xff\xff\x24\x82\xff\xff\xff\x00\x04\xff\xff\xff\x01\xff\xff\xff\xba\xff\xff\xff\x3c\xff\xff\xff\x65\xff\xff\xff\x95\x91\xff\xff\xff\x00\xe9\xff\xff\xff\x00\x03\xff\xff\xff\x1e\xff\xff\xff\xdd\xff\xff\xff\xd3\xff\xff\xff\x09\x91\xff\xff\xff\x00\xff\xff\xff\xff\x00\xff\xff\xff\xff\x00\x01\xff\xff\xff\x25\xff\xff\xff\xf6\x82\xff\xff\xff\xef\x04\xff\xff\xff\xf0\xff\xff\xff\xef\xff\xff\xff\x00\xff\xff\xff\x25\xff\xff\xff\xf6\x82\xff\xff\xff\xef\x01\xff\xff\xff\xf0\xff\xff\xff\xef\xb0\xff\xff\xff\x00\x01\xff\xff\xff\x25\xff\xff\xff\xf6\x82\xff\xff\xff\xef\x04\xff\xff\xff\xf0\xff\xff\xff\xef\xff\xff\xff\x00\xff\xff\xff\x25\xff\xff\xff\xf6\x82\xff\xff\xff\xef\x01\xff\xff\xff\xf0\xff\xff\xff\xef\xb0\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\xb0\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\xb0\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x07\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x02\xff\xff\xff\xe6\xff\xff\xff\x00\xff\xff\xff\x9d\xff\xff\xff\x50\x82\xff\xff\xff\x00\x05\xff\xff\xff\x72\xff\xff\xff\xdc\xff\xff\xff\xf3\xff\xff\xff\xee\xff\xff\xff\xa2\xff\xff\xff\x0f\x82\xff\xff\xff\x00\x04\xff\xff\xff\x02\xff\xff\xff\x8b\xff\xff\xff\xea\xff\xff\xff\xf1\xff\xff\xff\x9b\x81\xff\xff\xff\x00\x05\xff\xff\xff\x04\xff\xff\xff\xb0\xff\xff\xff\xed\xff\xff\xff\xf5\xff\xff\xff\xc6\xff\xff\xff\x2b\x82\xff\xff\xff\x00\x01\xff\xff\xff\x0f\xff\xff\xff\xa4\x81\xff\xff\xff\xee\x00\xff\xff\xff\x92\x81\xff\xff\xff\x00\x05\xff\xff\xff\x18\xff\xff\xff\xc2\xff\xff\xff\xf1\xff\xff\xff\xf4\xff\xff\xff\xc9\xff\xff\xff\x37\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x02\xff\xff\xff\xf3\xff\xff\xff\xf4\xff\xff\xff\xff\x82\xff\xff\xff\xf3\x88\xff\xff\xff\x00\x01\xff\xff\xff\xa1\xff\xff\xff\x97\x82\xff\xff\xff\x00\x01\xff\xff\xff\x5f\xff\xff\xff\xdc\x81\xff\xff\xff\x00\x04\xff\xff\xff\x0d\xff\xff\xff\xa8\xff\xff\xff\xf1\xff\xff\xff\xea\xff\xff\xff\x88\x82\xff\xff\xff\x00\x04\xff\xff\xff\x83\xff\xff\xff\xf3\xff\xff\xff\xe0\xff\xff\xff\x78\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x04\xff\xff\xff\x28\xff\xff\xff\xcc\xff\xff\xff\xf3\xff\xff\xff\xb8\xff\xff\xff\x12\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x03\xff\xff\xff\xc6\xff\xff\xff\x25\xff\xff\xff\x68\xff\xff\xff\x83\x82\xff\xff\xff\x00\x05\xff\xff\xff\x58\xff\xff\xff\x28\xff\xff\xff\x00\xff\xff\xff\x12\xff\xff\xff\xb5\xff\xff\xff\xa5\x82\xff\xff\xff\x00\x04\xff\xff\xff\x7d\xff\xff\xff\xc0\xff\xff\xff\x12\xff\xff\xff\x05\xff\xff\xff\x4f\x81\xff\xff\xff\x00\x05\xff\xff\xff\x08\xff\xff\xff\x6e\xff\xff\xff\x0b\xff\xff\xff\x04\xff\xff\xff\x8a\xff\xff\xff\xc7\x82\xff\xff\xff\x00\x04\xff\xff\xff\xac\xff\xff\xff\xa6\xff\xff\xff\x0c\xff\xff\xff\x05\xff\xff\xff\x4d\x81\xff\xff\xff\x00\x05\xff\xff\xff\x16\xff\xff\xff\x4d\xff\xff\xff\x09\xff\xff\xff\x01\xff\xff\xff\x72\xff\xff\xff\xd9\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x8b\xff\xff\xff\x00\x01\xff\xff\xff\x55\xff\xff\xff\xe5\x82\xff\xff\xff\x00\x01\xff\xff\xff\xb0\xff\xff\xff\x90\x81\xff\xff\xff\x00\x0c\xff\xff\xff\x99\xff\xff\xff\xaa\xff\xff\xff\x08\xff\xff\xff\x12\xff\xff\xff\xcb\xff\xff\xff\x65\xff\xff\xff\x00\xff\xff\xff\x18\xff\xff\xff\xff\xff\xff\xff\x2c\xff\xff\xff\x05\xff\xff\xff\xa0\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x04\xff\xff\xff\xc7\xff\xff\xff\x75\xff\xff\xff\x02\xff\xff\xff\x9f\xff\xff\xff\x99\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x03\xff\xff\xff\x93\xff\xff\xff\x5a\xff\xff\xff\x33\xff\xff\xff\xb7\x86\xff\xff\xff\x00\x01\xff\xff\xff\x38\xff\xff\xff\xef\x82\xff\xff\xff\x00\x01\xff\xff\xff\xe5\xff\xff\xff\x3c\x88\xff\xff\xff\x00\x01\xff\xff\xff\x30\xff\xff\xff\xf1\x81\xff\xff\xff\x00\x02\xff\xff\xff\x0d\xff\xff\xff\xfc\xff\xff\xff\x1c\x87\xff\xff\xff\x00\x02\xff\xff\xff\x03\xff\xff\xff\x75\xff\xff\xff\xe2\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x8b\xff\xff\xff\x00\x06\xff\xff\xff\x0e\xff\xff\xff\xf9\xff\xff\xff\xf2\xff\xff\xff\xef\xff\xff\xff\xf0\xff\xff\xff\xfd\xff\xff\xff\x43\x81\xff\xff\xff\x00\x01\xff\xff\xff\xf0\xff\xff\xff\x2e\x81\xff\xff\xff\x00\x08\xff\xff\xff\x5b\xff\xff\xff\xbf\xff\xff\xff\x00\xff\xff\xff\x11\xff\xff\xff\xfb\xff\xff\xff\x41\xff\xff\xff\x00\xff\xff\xff\x34\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x05\xff\xff\xff\x10\xff\xff\xff\xff\xff\xff\xff\x10\xff\xff\xff\x00\xff\xff\xff\x3d\xff\xff\xff\xe1\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x09\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\xeb\xff\xff\xff\xf4\xff\xff\xff\xf8\xff\xff\xff\xef\xff\xff\xff\xfc\xff\xff\xff\xef\xff\xff\xff\x7e\x84\xff\xff\xff\x00\x01\xff\xff\xff\x39\xff\xff\xff\xe8\x81\xff\xff\xff\x00\x02\xff\xff\xff\x16\xff\xff\xff\xff\xff\xff\xff\x09\x87\xff\xff\xff\x00\x02\xff\xff\xff\x0b\xff\xff\xff\x95\xff\xff\xff\xc9\x81\xff\xff\xff\x00\x01\xff\xff\xff\x23\xff\xff\xff\xfd\x86\xff\xff\xff\x00\x04\xff\xff\xff\x4c\xff\xff\xff\xba\xff\xff\xff\xf1\xff\xff\xff\xd4\xff\xff\xff\x45\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x85\xff\xff\xff\x00\x03\xff\xff\xff\x96\xff\xff\xff\xee\xff\xff\xff\xdb\xff\xff\xff\x50\x82\xff\xff\xff\x00\x09\xff\xff\xff\xbd\xff\xff\xff\x7d\xff\xff\xff\x00\xff\xff\xff\x46\xff\xff\xff\xf1\xff\xff\xff\x04\xff\xff\xff\x00\xff\xff\xff\x19\xff\xff\xff\xff\xff\xff\xff\x06\x81\xff\xff\xff\x00\x01\xff\xff\xff\x33\xff\xff\xff\xe7\x81\xff\xff\xff\x00\x04\xff\xff\xff\x5f\xff\xff\xff\xd8\xff\xff\xff\xf0\xff\xff\xff\xf1\xff\xff\xff\xf6\x81\xff\xff\xff\x00\x01\xff\xff\xff\x23\xff\xff\xff\xfd\x81\xff\xff\xff\x00\x01\xff\xff\xff\x29\xff\xff\xff\xf3\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x04\xff\xff\xff\x0e\xff\xff\xff\xd7\xff\xff\xff\x07\xff\xff\xff\x9f\xff\xff\xff\x4e\x84\xff\xff\xff\x00\x02\xff\xff\xff\x16\xff\xff\xff\xbc\xff\xff\xff\x88\x81\xff\xff\xff\x00\x01\xff\xff\xff\x24\xff\xff\xff\xfd\x86\xff\xff\xff\x00\x04\xff\xff\xff\x30\xff\xff\xff\xb5\xff\xff\xff\xf7\xff\xff\xff\xcc\xff\xff\xff\x28\x81\xff\xff\xff\x00\x02\xff\xff\xff\x0d\xff\xff\xff\xfc\xff\xff\xff\x1c\x84\xff\xff\xff\x00\x03\xff\xff\xff\x10\xff\xff\xff\xf9\xff\xff\xff\x5c\xff\xff\xff\x0d\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x02\xff\xff\xff\xa7\xff\xff\xff\xf1\xff\xff\xff\xff\x81\xff\xff\xff\xef\x00\xff\xff\xff\x20\x82\xff\xff\xff\x00\x03\xff\xff\xff\x24\xff\xff\xff\x01\xff\xff\xff\x63\xff\xff\xff\xe8\x82\xff\xff\xff\x00\x04\xff\xff\xff\x71\xff\xff\xff\xbe\xff\xff\xff\x00\xff\xff\xff\x87\xff\xff\xff\xaa\x81\xff\xff\xff\x00\x01\xff\xff\xff\x24\xff\xff\xff\xfc\x82\xff\xff\xff\x00\x01\xff\xff\xff\x29\xff\xff\xff\xf3\x84\xff\xff\xff\x00\x01\xff\xff\xff\x2d\xff\xff\xff\xf1\x81\xff\xff\xff\x00\x05\xff\xff\xff\x13\xff\xff\xff\xff\xff\xff\xff\x0f\xff\xff\xff\x00\xff\xff\xff\x3c\xff\xff\xff\xe1\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x05\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x4a\xff\xff\xff\xef\xff\xff\xff\xf9\x81\xff\xff\xff\xf3\x08\xff\xff\xff\xf9\xff\xff\xff\xef\xff\xff\xff\x2f\xff\xff\xff\x00\xff\xff\xff\x24\xff\xff\xff\xef\xff\xff\xff\xfd\xff\xff\xff\xc4\xff\xff\xff\x0f\x81\xff\xff\xff\x00\x02\xff\xff\xff\x16\xff\xff\xff\xff\xff\xff\xff\x0a\x84\xff\xff\xff\x00\x03\xff\xff\xff\x07\xff\xff\xff\xeb\xff\xff\xff\x9a\xff\xff\xff\x2b\x84\xff\xff\xff\x00\x04\xff\xff\xff\xac\xff\xff\xff\xa7\xff\xff\xff\x0c\xff\xff\xff\x06\xff\xff\xff\x4e\x81\xff\xff\xff\x00\x05\xff\xff\xff\x14\xff\xff\xff\xfc\xff\xff\xff\x35\xff\xff\xff\x00\xff\xff\xff\x13\xff\xff\xff\x36\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x87\xff\xff\xff\x00\x01\xff\xff\xff\x5c\xff\xff\xff\xd5\x82\xff\xff\xff\x00\x04\xff\xff\xff\x25\xff\xff\xff\xf7\xff\xff\xff\x07\xff\xff\xff\xc9\xff\xff\xff\x5e\x81\xff\xff\xff\x00\x02\xff\xff\xff\x1a\xff\xff\xff\xff\xff\xff\xff\x06\x81\xff\xff\xff\x00\x01\xff\xff\xff\x33\xff\xff\xff\xe9\x81\xff\xff\xff\x00\x04\xff\xff\xff\x54\xff\xff\xff\x1e\xff\xff\xff\x01\xff\xff\xff\x86\xff\xff\xff\xc9\x82\xff\xff\xff\x00\x04\xff\xff\xff\xce\xff\xff\xff\x71\xff\xff\xff\x00\xff\xff\xff\x9c\xff\xff\xff\x99\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x03\xff\xff\xff\x6c\xff\xff\xff\x82\xff\xff\xff\x10\xff\xff\xff\xdd\x84\xff\xff\xff\x00\x02\xff\xff\xff\x07\xff\xff\xff\x74\xff\xff\xff\xc8\x82\xff\xff\xff\x00\x01\xff\xff\xff\xe3\xff\xff\xff\x3c\x84\xff\xff\xff\x00\x02\xff\xff\xff\x22\xff\xff\xff\xff\xff\xff\xff\x05\x85\xff\xff\xff\x00\x01\xff\xff\xff\x0f\xff\xff\xff\xa3\x81\xff\xff\xff\xee\x00\xff\xff\xff\x91\x82\xff\xff\xff\x00\x04\xff\xff\xff\x67\xff\xff\xff\xe0\xff\xff\xff\xf4\xff\xff\xff\xe4\xff\xff\xff\x6f\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x02\xff\xff\xff\x22\xff\xff\xff\xfb\xff\xff\xff\x01\x85\xff\xff\xff\x00\x02\xff\xff\xff\xcc\xff\xff\xff\xf0\xff\xff\xff\x40\x83\xff\xff\xff\x00\x03\xff\xff\xff\xd9\xff\xff\xff\x50\xff\xff\xff\xf9\xff\xff\xff\x14\x81\xff\xff\xff\x00\x02\xff\xff\xff\x01\xff\xff\xff\xf3\xff\xff\xff\x2f\x81\xff\xff\xff\x00\x01\xff\xff\xff\x5c\xff\xff\xff\xc5\x81\xff\xff\xff\x00\x04\xff\xff\xff\x68\xff\xff\xff\xe0\xff\xff\xff\xf5\xff\xff\xff\xcb\xff\xff\xff\x2e\x82\xff\xff\xff\x00\x04\xff\xff\xff\x2e\xff\xff\xff\xd0\xff\xff\xff\xed\xff\xff\xff\xb9\xff\xff\xff\x12\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x0b\xff\xff\xff\x38\xff\xff\xff\xb5\xff\xff\xff\x00\xff\xff\xff\xda\xff\xff\xff\x13\xff\xff\xff\x00\xff\xff\xff\x1a\xff\xff\xff\x1c\xff\xff\xff\x00\xff\xff\xff\x02\xff\xff\xff\x72\xff\xff\xff\xe2\x82\xff\xff\xff\x00\x04\xff\xff\xff\x77\xff\xff\xff\xc2\xff\xff\xff\x13\xff\xff\xff\x07\xff\xff\xff\x50\x81\xff\xff\xff\x00\x05\xff\xff\xff\x05\xff\xff\xff\xeb\xff\xff\xff\x5a\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\x3b\x91\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x04\xff\xff\xff\x04\xff\xff\xff\xef\xff\xff\xff\x5d\xff\xff\xff\x00\xff\xff\xff\x2c\x82\xff\xff\xff\x00\x03\xff\xff\xff\x0d\xff\xff\xff\x00\xff\xff\xff\x59\xff\xff\xff\xe6\x83\xff\xff\xff\x00\x02\xff\xff\xff\x8d\xff\xff\xff\xce\xff\xff\xff\xc5\x83\xff\xff\xff\x00\x05\xff\xff\xff\xa2\xff\xff\xff\xac\xff\xff\xff\x09\xff\xff\xff\x14\xff\xff\xff\xcd\xff\xff\xff\x72\x8a\xff\xff\xff\x00\x01\xff\xff\xff\x02\xff\xff\xff\x05\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xff\x82\xff\xff\xff\xf3\x04\xff\xff\xff\xf4\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xff\x82\xff\xff\xff\xf3\x01\xff\xff\xff\xf4\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x0b\xff\xff\xff\x08\xff\xff\xff\xe5\xff\xff\xff\x00\xff\xff\xff\xa5\xff\xff\xff\x48\xff\xff\xff\x00\xff\xff\xff\x45\xff\xff\xff\xd1\xff\xff\xff\xf1\xff\xff\xff\xf2\xff\xff\xff\xc2\xff\xff\xff\x36\x82\xff\xff\xff\x00\x04\xff\xff\xff\x01\xff\xff\xff\x87\xff\xff\xff\xe9\xff\xff\xff\xf0\xff\xff\xff\x98\x82\xff\xff\xff\x00\x04\xff\xff\xff\x3f\xff\xff\xff\xd4\xff\xff\xff\xf5\xff\xff\xff\xda\xff\xff\xff\x4a\x91\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xff\x82\xff\xff\xff\xf3\x04\xff\xff\xff\xf4\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xff\x82\xff\xff\xff\xf3\x01\xff\xff\xff\xf4\xff\xff\xff\xf7\x83\xff\xff\xff\x00\x03\xff\xff\xff\x4f\xff\xff\xff\xde\xff\xff\xff\xf4\xff\xff\xff\xcd\x82\xff\xff\xff\x00\x03\xff\xff\xff\x78\xff\xff\xff\xed\xff\xff\xff\xec\xff\xff\xff\x82\x83\xff\xff\xff\x00\x02\xff\xff\xff\x41\xff\xff\xff\xff\xff\xff\xff\x79\x83\xff\xff\xff\x00\x05\xff\xff\xff\x13\xff\xff\xff\xaf\xff\xff\xff\xf1\xff\xff\xff\xec\xff\xff\xff\x96\xff\xff\xff\x03\x82\xff\xff\xff\x00\x03\xff\xff\xff\x80\xff\xff\xff\x82\xff\xff\xff\xd6\xff\xff\xff\x09\x83\xff\xff\xff\x00\x02\xff\xff\xff\x0c\xff\xff\xff\xb9\xff\xff\xff\x10\x82\xff\xff\xff\x00\xea\xff\xff\xff\x00\x00\xff\xff\xff\x0b\x85\xff\xff\xff\x00\x03\xff\xff\xff\xb5\xff\xff\xff\x87\xff\xff\xff\x7c\xff\xff\xff\x3b\x84\xff\xff\xff\x00\x02\xff\xff\xff\x3d\xff\xff\xff\xb8\xff\xff\xff\x02\x81\xff\xff\xff\x00\xe0\xff\xff\xff\x00\x04\xff\xff\xff\x31\xff\xff\xff\x80\xff\xff\xff\x7b\xff\xff\xff\xb1\xff\xff\xff\x2d\x84\xff\xff\xff\x00\x01\xff\xff\xff\xad\xff\xff\xff\x67\x92\xff\xff\xff\x00\xe0\xff\xff\xff\x00\x04\xff\xff\xff\x01\xff\xff\xff\x6e\xff\xff\xff\x52\xff\xff\xff\x1e\xff\xff\xff\x43\x84\xff\xff\xff\x00\x02\xff\xff\xff\x19\xff\xff\xff\xe7\xff\xff\xff\x2f\x91\xff\xff\xff\x00\xff\xff\xff\xff\x00\xff\xff\xff\xff\x00\x01\xff\xff\xff\x25\xff\xff\xff\xf6\x82\xff\xff\xff\xef\x04\xff\xff\xff\xf0\xff\xff\xff\xef\xff\xff\xff\x00\xff\xff\xff\x25\xff\xff\xff\xf6\x82\xff\xff\xff\xef\x01\xff\xff\xff\xf0\xff\xff\xff\xef\xb0\xff\xff\xff\x00\x01\xff\xff\xff\x25\xff\xff\xff\xf6\x82\xff\xff\xff\xef\x04\xff\xff\xff\xf0\xff\xff\xff\xef\xff\xff\xff\x00\xff\xff\xff\x25\xff\xff\xff\xf6\x82\xff\xff\xff\xef\x01\xff\xff\xff\xf0\xff\xff\xff\xef\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\xaa\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\xb0\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\xaa\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x89\xff\xff\xff\x00\x01\xff\xff\xff\xff\xff\xff\xff\xf6\x81\xff\xff\xff\xf3\x00\xff\xff\xff\xeb\x81\xff\xff\xff\x00\x09\xff\xff\xff\x27\xff\xff\xff\xff\xff\xff\xff\xf3\xff\xff\xff\xf2\xff\xff\xff\xed\xff\xff\xff\xb5\xff\xff\xff\x26\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x08\xff\xff\xff\x80\xff\xff\xff\xb5\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\xff\xff\xff\x8b\xff\xff\xff\xf1\xff\xff\xff\xc4\xff\xff\xff\x16\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x84\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x05\xff\xff\xff\x09\xff\xff\xff\x95\xff\xff\xff\xed\xff\xff\xff\xff\xff\xff\xff\xe7\xff\xff\xff\x56\x88\xff\xff\xff\x00\x01\xff\xff\xff\xa1\xff\xff\xff\x97\x82\xff\xff\xff\x00\x01\xff\xff\xff\x60\xff\xff\xff\xdc\x81\xff\xff\xff\x00\x04\xff\xff\xff\x0d\xff\xff\xff\xa8\xff\xff\xff\xf1\xff\xff\xff\xea\xff\xff\xff\x88\x82\xff\xff\xff\x00\x04\xff\xff\xff\x83\xff\xff\xff\xf3\xff\xff\xff\xe0\xff\xff\xff\x78\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x04\xff\xff\xff\x28\xff\xff\xff\xcc\xff\xff\xff\xf3\xff\xff\xff\xb8\xff\xff\xff\x12\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x89\xff\xff\xff\x00\x02\xff\xff\xff\x86\xff\xff\xff\xb2\xff\xff\xff\x02\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x81\xff\xff\xff\x00\x05\xff\xff\xff\x08\xff\xff\xff\x88\xff\xff\xff\xcb\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x81\xff\xff\xff\x00\x09\xff\xff\xff\x04\xff\xff\xff\xe5\xff\xff\xff\x42\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xff\xff\xff\xff\x76\xff\xff\xff\x02\xff\xff\xff\x9f\xff\xff\xff\x9b\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x84\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x05\xff\xff\xff\xa2\xff\xff\xff\xb3\xff\xff\xff\x38\xff\xff\xff\xf7\xff\xff\xff\x0e\xff\xff\xff\x29\x88\xff\xff\xff\x00\x02\xff\xff\xff\x55\xff\xff\xff\xe6\xff\xff\xff\x01\x81\xff\xff\xff\x00\x01\xff\xff\xff\xb1\xff\xff\xff\x90\x81\xff\xff\xff\x00\x0c\xff\xff\xff\x99\xff\xff\xff\xaa\xff\xff\xff\x08\xff\xff\xff\x12\xff\xff\xff\xcb\xff\xff\xff\x65\xff\xff\xff\x00\xff\xff\xff\x18\xff\xff\xff\xff\xff\xff\xff\x2c\xff\xff\xff\x05\xff\xff\xff\xa0\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x04\xff\xff\xff\xc7\xff\xff\xff\x75\xff\xff\xff\x02\xff\xff\xff\x9f\xff\xff\xff\x99\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x8a\xff\xff\xff\x00\x01\xff\xff\xff\xa6\xff\xff\xff\x95\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x2e\xff\xff\xff\xef\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x81\xff\xff\xff\x00\x01\xff\xff\xff\x56\xff\xff\xff\xcf\x81\xff\xff\xff\x00\x05\xff\xff\xff\x27\xff\xff\xff\xff\xff\xff\xff\x10\xff\xff\xff\x00\xff\xff\xff\x3d\xff\xff\xff\xe1\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x84\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x07\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x0b\xff\xff\xff\xfb\xff\xff\xff\x22\xff\xff\xff\x27\xff\xff\xff\xf7\x8a\xff\xff\xff\x00\x02\xff\xff\xff\x0e\xff\xff\xff\xf9\xff\xff\xff\xf8\x81\xff\xff\xff\xf7\x01\xff\xff\xff\xff\xff\xff\xff\x43\x81\xff\xff\xff\x00\x01\xff\xff\xff\xf0\xff\xff\xff\x2e\x81\xff\xff\xff\x00\x08\xff\xff\xff\x5b\xff\xff\xff\xbf\xff\xff\xff\x00\xff\xff\xff\x11\xff\xff\xff\xfb\xff\xff\xff\x41\xff\xff\xff\x00\xff\xff\xff\x34\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x05\xff\xff\xff\x10\xff\xff\xff\xff\xff\xff\xff\x10\xff\xff\xff\x00\xff\xff\xff\x3d\xff\xff\xff\xe1\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x8a\xff\xff\xff\x00\x02\xff\xff\xff\x06\xff\xff\xff\xc4\xff\xff\xff\x6e\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x81\xff\xff\xff\x00\x09\xff\xff\xff\x0f\xff\xff\xff\x9a\xff\xff\xff\x99\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\xff\xff\xff\x00\xff\xff\xff\x16\xff\xff\xff\xd7\xff\xff\xff\x59\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfd\x81\xff\xff\xff\x00\x01\xff\xff\xff\x2a\xff\xff\xff\xf3\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfc\x84\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x07\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x23\xff\xff\xff\xfd\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xf7\x83\xff\xff\xff\x00\x01\xff\xff\xff\x25\xff\xff\xff\xf4\x81\xff\xff\xff\xef\x00\xff\xff\xff\x02\x82\xff\xff\xff\x00\x09\xff\xff\xff\xbd\xff\xff\xff\x7c\xff\xff\xff\x00\xff\xff\xff\x45\xff\xff\xff\xf1\xff\xff\xff\x04\xff\xff\xff\x00\xff\xff\xff\x19\xff\xff\xff\xff\xff\xff\xff\x06\x81\xff\xff\xff\x00\x01\xff\xff\xff\x33\xff\xff\xff\xe7\x81\xff\xff\xff\x00\x04\xff\xff\xff\x5f\xff\xff\xff\xd8\xff\xff\xff\xf0\xff\xff\xff\xf1\xff\xff\xff\xf6\x81\xff\xff\xff\x00\x01\xff\xff\xff\x23\xff\xff\xff\xfd\x81\xff\xff\xff\x00\x01\xff\xff\xff\x29\xff\xff\xff\xf3\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x8b\xff\xff\xff\x00\x02\xff\xff\xff\x16\xff\xff\xff\xe3\xff\xff\xff\x38\x81\xff\xff\xff\x00\x0d\xff\xff\xff\x27\xff\xff\xff\xff\xff\xff\xff\xef\xff\xff\xff\xf4\xff\xff\xff\xef\xff\xff\xff\x62\xff\xff\xff\x01\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xff\xff\xff\xff\xef\xff\xff\xff\xfb\xff\xff\xff\xcc\xff\xff\xff\x0e\x81\xff\xff\xff\x00\x05\xff\xff\xff\x27\xff\xff\xff\xff\xff\xff\xff\x10\xff\xff\xff\x00\xff\xff\xff\x3d\xff\xff\xff\xe2\x82\xff\xff\xff\x00\x02\xff\xff\xff\x27\xff\xff\xff\xff\xff\xff\xff\x1e\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x07\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x0b\xff\xff\xff\xfb\xff\xff\xff\x21\xff\xff\xff\x27\xff\xff\xff\xf7\x84\xff\xff\xff\x00\x01\xff\xff\xff\xa5\xff\xff\xff\x27\x84\xff\xff\xff\x00\x04\xff\xff\xff\x71\xff\xff\xff\xbe\xff\xff\xff\x00\xff\xff\xff\x87\xff\xff\xff\xaa\x81\xff\xff\xff\x00\x01\xff\xff\xff\x24\xff\xff\xff\xfc\x82\xff\xff\xff\x00\x01\xff\xff\xff\x29\xff\xff\xff\xf3\x84\xff\xff\xff\x00\x01\xff\xff\xff\x2d\xff\xff\xff\xf1\x81\xff\xff\xff\x00\x05\xff\xff\xff\x13\xff\xff\xff\xff\xff\xff\xff\x0f\xff\xff\xff\x00\xff\xff\xff\x3c\xff\xff\xff\xe1\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x8c\xff\xff\xff\x00\x01\xff\xff\xff\x63\xff\xff\xff\xbf\x81\xff\xff\xff\x00\x05\xff\xff\xff\x27\xff\xff\xff\xfb\xff\xff\xff\x00\xff\xff\xff\x02\xff\xff\xff\x7e\xff\xff\xff\xbb\x81\xff\xff\xff\x00\x05\xff\xff\xff\x27\xff\xff\xff\xfb\xff\xff\xff\x00\xff\xff\xff\x03\xff\xff\xff\x86\xff\xff\xff\xb5\x81\xff\xff\xff\x00\x05\xff\xff\xff\x27\xff\xff\xff\xff\xff\xff\xff\x79\xff\xff\xff\x02\xff\xff\xff\x9f\xff\xff\xff\x9d\x82\xff\xff\xff\x00\x08\xff\xff\xff\x27\xff\xff\xff\xfd\xff\xff\xff\x9e\xff\xff\xff\x0c\xff\xff\xff\x08\xff\xff\xff\x2b\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x05\xff\xff\xff\xa1\xff\xff\xff\xb1\xff\xff\xff\x36\xff\xff\xff\xf7\xff\xff\xff\x0d\xff\xff\xff\x28\x82\xff\xff\xff\x00\x02\xff\xff\xff\x12\xff\xff\xff\xc2\xff\xff\xff\x14\x83\xff\xff\xff\x00\x04\xff\xff\xff\x25\xff\xff\xff\xf7\xff\xff\xff\x07\xff\xff\xff\xc8\xff\xff\xff\x5e\x81\xff\xff\xff\x00\x02\xff\xff\xff\x1a\xff\xff\xff\xff\xff\xff\xff\x06\x81\xff\xff\xff\x00\x01\xff\xff\xff\x33\xff\xff\xff\xe9\x81\xff\xff\xff\x00\x04\xff\xff\xff\x54\xff\xff\xff\x1e\xff\xff\xff\x01\xff\xff\xff\x86\xff\xff\xff\xc9\x82\xff\xff\xff\x00\x04\xff\xff\xff\xce\xff\xff\xff\x71\xff\xff\xff\x00\xff\xff\xff\x9c\xff\xff\xff\x99\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x03\xff\xff\xff\x27\xff\xff\xff\xfb\xff\xff\xff\x27\xff\xff\xff\xf7\x86\xff\xff\xff\x00\x01\xff\xff\xff\x2f\xff\xff\xff\xf0\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x81\xff\xff\xff\x00\x01\xff\xff\xff\x2d\xff\xff\xff\xf0\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x81\xff\xff\xff\x00\x01\xff\xff\xff\x31\xff\xff\xff\xef\x81\xff\xff\xff\x00\x05\xff\xff\xff\x27\xff\xff\xff\xfb\xff\xff\xff\x8d\xff\xff\xff\xf1\xff\xff\xff\xc5\xff\xff\xff\x17\x82\xff\xff\xff\x00\x08\xff\xff\xff\x27\xff\xff\xff\xfb\xff\xff\xff\x56\xff\xff\xff\xe0\xff\xff\xff\xf2\xff\xff\xff\x71\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x05\xff\xff\xff\x09\xff\xff\xff\x99\xff\xff\xff\xf0\xff\xff\xff\xff\xff\xff\xff\xe8\xff\xff\xff\x57\x83\xff\xff\xff\x00\x01\xff\xff\xff\x58\xff\xff\xff\xb1\x84\xff\xff\xff\x00\x03\xff\xff\xff\xd9\xff\xff\xff\x4f\xff\xff\xff\xf8\xff\xff\xff\x14\x81\xff\xff\xff\x00\x02\xff\xff\xff\x01\xff\xff\xff\xf3\xff\xff\xff\x2f\x81\xff\xff\xff\x00\x01\xff\xff\xff\x5c\xff\xff\xff\xc5\x81\xff\xff\xff\x00\x04\xff\xff\xff\x68\xff\xff\xff\xe0\xff\xff\xff\xf5\xff\xff\xff\xcb\xff\xff\xff\x2e\x82\xff\xff\xff\x00\x04\xff\xff\xff\x2e\xff\xff\xff\xd0\xff\xff\xff\xed\xff\xff\xff\xb9\xff\xff\xff\x12\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x03\xff\xff\xff\x27\xff\xff\xff\xfb\xff\xff\xff\x27\xff\xff\xff\xf7\x83\xff\xff\xff\x00\x04\xff\xff\xff\x73\xff\xff\xff\x16\xff\xff\xff\x02\xff\xff\xff\x8b\xff\xff\xff\xc0\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x81\xff\xff\xff\x00\x01\xff\xff\xff\x75\xff\xff\xff\xca\x81\xff\xff\xff\x00\x05\xff\xff\xff\x27\xff\xff\xff\xfb\xff\xff\xff\x00\xff\xff\xff\x03\xff\xff\xff\x89\xff\xff\xff\xc7\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x8d\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x83\xff\xff\xff\x00\x03\xff\xff\xff\x0b\xff\xff\xff\x18\xff\xff\xff\x4a\xff\xff\xff\xef\x84\xff\xff\xff\x00\x02\xff\xff\xff\x8d\xff\xff\xff\xce\xff\xff\xff\xc5\x83\xff\xff\xff\x00\x05\xff\xff\xff\xa2\xff\xff\xff\xac\xff\xff\xff\x09\xff\xff\xff\x14\xff\xff\xff\xcd\xff\xff\xff\x72\x8b\xff\xff\xff\x00\x00\xff\xff\xff\x07\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xff\x82\xff\xff\xff\xf3\x04\xff\xff\xff\xf4\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xff\x82\xff\xff\xff\xf3\x01\xff\xff\xff\xf4\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x03\xff\xff\xff\x27\xff\xff\xff\xfb\xff\xff\xff\x27\xff\xff\xff\xf7\x83\xff\xff\xff\x00\x04\xff\xff\xff\x97\xff\xff\xff\xe8\xff\xff\xff\xf4\xff\xff\xff\xbd\xff\xff\xff\x20\x81\xff\xff\xff\x00\x05\xff\xff\xff\x27\xff\xff\xff\xff\xff\xff\xff\xf3\xff\xff\xff\xf4\xff\xff\xff\xc9\xff\xff\xff\x2e\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xff\x81\xff\xff\xff\xf3\x01\xff\xff\xff\xc1\xff\xff\xff\x28\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x8d\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xff\x82\xff\xff\xff\xf3\x04\xff\xff\xff\xf4\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xff\x82\xff\xff\xff\xf3\x01\xff\xff\xff\xf4\xff\xff\xff\xf7\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x83\xff\xff\xff\x00\x03\xff\xff\xff\x10\xff\xff\xff\xd0\xff\xff\xff\xf1\xff\xff\xff\x72\x84\xff\xff\xff\x00\x02\xff\xff\xff\x41\xff\xff\xff\xff\xff\xff\xff\x79\x83\xff\xff\xff\x00\x05\xff\xff\xff\x13\xff\xff\xff\xaf\xff\xff\xff\xf1\xff\xff\xff\xec\xff\xff\xff\x96\xff\xff\xff\x03\x81\xff\xff\xff\x00\x04\xff\xff\xff\x02\xff\xff\xff\xaf\xff\xff\xff\x1c\xff\xff\xff\xb5\xff\xff\xff\x08\x83\xff\xff\xff\x00\x01\xff\xff\xff\x28\xff\xff\xff\xad\x83\xff\xff\xff\x00\xaf\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\xb8\xff\xff\xff\x00\x01\xff\xff\xff\x01\xff\xff\xff\x09\x84\xff\xff\xff\x00\x02\xff\xff\xff\x30\xff\xff\xff\xe9\xff\xff\xff\x3c\x83\xff\xff\xff\x00\x02\xff\xff\xff\x0f\xff\xff\xff\xca\xff\xff\xff\x1d\x83\xff\xff\xff\x00\xe0\xff\xff\xff\x00\x04\xff\xff\xff\x05\xff\xff\xff\xbe\xff\xff\xff\x19\xff\xff\xff\xba\xff\xff\xff\x21\x84\xff\xff\xff\x00\x01\xff\xff\xff\x93\xff\xff\xff\x81\x92\xff\xff\xff\x00\xe1\xff\xff\xff\x00\x02\xff\xff\xff\x3e\xff\xff\xff\xe8\xff\xff\xff\x7a\x84\xff\xff\xff\x00\x02\xff\xff\xff\x51\xff\xff\xff\xd5\xff\xff\xff\x05\x92\xff\xff\xff\x00\xff\xff\xff\xff\x00\xbb\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\xc1\xff\xff\xff\x00\x01\xff\xff\xff\x25\xff\xff\xff\xf6\x82\xff\xff\xff\xef\x04\xff\xff\xff\xf0\xff\xff\xff\xef\xff\xff\xff\x00\xff\xff\xff\x25\xff\xff\xff\xf6\x82\xff\xff\xff\xef\x01\xff\xff\xff\xf0\xff\xff\xff\xef\x9d\xff\xff\xff\x00\x01\xff\xff\xff\x36\xff\xff\xff\x0b\x8c\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x01\xff\xff\xff\x25\xff\xff\xff\xf6\x82\xff\xff\xff\xef\x04\xff\xff\xff\xf0\xff\xff\xff\xef\xff\xff\xff\x00\xff\xff\xff\x25\xff\xff\xff\xf6\x82\xff\xff\xff\xef\x01\xff\xff\xff\xf0\xff\xff\xff\xef\xb0\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x9c\xff\xff\xff\x00\x02\xff\xff\xff\x51\xff\xff\xff\xec\xff\xff\xff\x37\x8c\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\xb0\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x85\xff\xff\xff\x00\x84\xff\xff\xff\xff\x02\xff\xff\xff\x00\xff\xff\xff\xa1\xff\xff\xff\x97\x82\xff\xff\xff\x00\x01\xff\xff\xff\x60\xff\xff\xff\xdc\x81\xff\xff\xff\x00\x05\xff\xff\xff\x0b\xff\xff\xff\xa5\xff\xff\xff\xf0\xff\xff\xff\xfd\xff\xff\xff\xbd\xff\xff\xff\x05\x81\xff\xff\xff\x00\x04\xff\xff\xff\x83\xff\xff\xff\xf3\xff\xff\xff\xe0\xff\xff\xff\x78\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x04\xff\xff\xff\x2f\xff\xff\xff\xd8\xff\xff\xff\xec\xff\xff\xff\x8d\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x84\xff\xff\xff\x00\x85\xff\xff\xff\xf3\x02\xff\xff\xff\x00\xff\xff\xff\xa1\xff\xff\xff\x97\x82\xff\xff\xff\x00\x04\xff\xff\xff\x60\xff\xff\xff\xdc\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x81\xff\xff\xff\x00\x01\xff\xff\xff\xa3\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x04\xff\xff\xff\x83\xff\xff\xff\xf3\xff\xff\xff\xe0\xff\xff\xff\x77\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x85\xff\xff\xff\x00\x02\xff\xff\xff\x07\xff\xff\xff\x2d\xff\xff\xff\xf7\x81\xff\xff\xff\x07\x03\xff\xff\xff\x00\xff\xff\xff\x55\xff\xff\xff\xe6\xff\xff\xff\x01\x81\xff\xff\xff\x00\x01\xff\xff\xff\xb1\xff\xff\xff\x90\x81\xff\xff\xff\x00\x0c\xff\xff\xff\x95\xff\xff\xff\xaa\xff\xff\xff\x08\xff\xff\xff\x12\xff\xff\xff\xcb\xff\xff\xff\x73\xff\xff\xff\x00\xff\xff\xff\x18\xff\xff\xff\xff\xff\xff\xff\x2c\xff\xff\xff\x05\xff\xff\xff\xa0\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x04\xff\xff\xff\xcc\xff\xff\xff\x75\xff\xff\xff\x02\xff\xff\xff\x9f\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x8b\xff\xff\xff\x00\x02\xff\xff\xff\x55\xff\xff\xff\xe6\xff\xff\xff\x01\x81\xff\xff\xff\x00\x08\xff\xff\xff\xb1\xff\xff\xff\x90\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\xff\xff\xff\x00\xff\xff\xff\x08\xff\xff\xff\xeb\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x05\xff\xff\xff\x18\xff\xff\xff\xff\xff\xff\xff\x2d\xff\xff\xff\x05\xff\xff\xff\xa0\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x01\xff\xff\xff\x01\xff\xff\xff\x0e\x86\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x02\xff\xff\xff\x0e\xff\xff\xff\xf9\xff\xff\xff\xf8\x81\xff\xff\xff\xf7\x01\xff\xff\xff\xff\xff\xff\xff\x43\x81\xff\xff\xff\x00\x01\xff\xff\xff\xee\xff\xff\xff\x2e\x81\xff\xff\xff\x00\x08\xff\xff\xff\x5b\xff\xff\xff\xc6\xff\xff\xff\x00\xff\xff\xff\x11\xff\xff\xff\xfb\xff\xff\xff\x41\xff\xff\xff\x00\xff\xff\xff\x34\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x05\xff\xff\xff\x12\xff\xff\xff\xff\xff\xff\xff\x10\xff\xff\xff\x00\xff\xff\xff\x3d\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x86\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x02\xff\xff\xff\x0e\xff\xff\xff\xf9\xff\xff\xff\xf8\x81\xff\xff\xff\xf7\x08\xff\xff\xff\xff\xff\xff\xff\x43\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\xff\xff\xff\x00\xff\xff\xff\x54\xff\xff\xff\xaf\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x05\xff\xff\xff\x10\xff\xff\xff\xfa\xff\xff\xff\x3c\xff\xff\xff\x00\xff\xff\xff\x34\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x86\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x83\xff\xff\xff\x00\x09\xff\xff\xff\xbd\xff\xff\xff\x7c\xff\xff\xff\x00\xff\xff\xff\x45\xff\xff\xff\xf1\xff\xff\xff\x04\xff\xff\xff\x00\xff\xff\xff\x19\xff\xff\xff\xff\xff\xff\xff\x06\x81\xff\xff\xff\x00\x01\xff\xff\xff\x33\xff\xff\xff\xea\x81\xff\xff\xff\x00\x04\xff\xff\xff\x5f\xff\xff\xff\xd8\xff\xff\xff\xf0\xff\xff\xff\xf1\xff\xff\xff\xf6\x81\xff\xff\xff\x00\x01\xff\xff\xff\x24\xff\xff\xff\xfd\x81\xff\xff\xff\x00\x01\xff\xff\xff\x2a\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x86\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x83\xff\xff\xff\x00\x0c\xff\xff\xff\xbd\xff\xff\xff\x7c\xff\xff\xff\x00\xff\xff\xff\x45\xff\xff\xff\xf1\xff\xff\xff\x04\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\xff\xff\xff\x00\xff\xff\xff\xac\xff\xff\xff\x57\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x04\xff\xff\xff\x58\xff\xff\xff\xd0\xff\xff\xff\xe2\xff\xff\xff\xe4\xff\xff\xff\xf6\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x86\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x83\xff\xff\xff\x00\x04\xff\xff\xff\x71\xff\xff\xff\xbe\xff\xff\xff\x00\xff\xff\xff\x87\xff\xff\xff\xaa\x81\xff\xff\xff\x00\x01\xff\xff\xff\x24\xff\xff\xff\xfc\x82\xff\xff\xff\x00\x01\xff\xff\xff\x29\xff\xff\xff\xf4\x84\xff\xff\xff\x00\x01\xff\xff\xff\x2d\xff\xff\xff\xf1\x81\xff\xff\xff\x00\x05\xff\xff\xff\x11\xff\xff\xff\xff\xff\xff\xff\x10\xff\xff\xff\x00\xff\xff\xff\x3d\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x84\xff\xff\xff\x00\x81\xff\xff\xff\xef\x01\xff\xff\xff\xf1\xff\xff\xff\xff\x81\xff\xff\xff\xef\x81\xff\xff\xff\x00\x04\xff\xff\xff\x71\xff\xff\xff\xbe\xff\xff\xff\x00\xff\xff\xff\x87\xff\xff\xff\xaa\x81\xff\xff\xff\x00\x05\xff\xff\xff\x27\xff\xff\xff\xfb\xff\xff\xff\x0c\xff\xff\xff\xce\xff\xff\xff\x27\xff\xff\xff\xf7\x85\xff\xff\xff\x00\x01\xff\xff\xff\x2f\xff\xff\xff\xf1\x81\xff\xff\xff\x00\x05\xff\xff\xff\x27\xff\xff\xff\xff\xff\xff\xff\x06\xff\xff\xff\x00\xff\xff\xff\x2a\xff\xff\xff\xf3\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x86\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x83\xff\xff\xff\x00\x04\xff\xff\xff\x25\xff\xff\xff\xf7\xff\xff\xff\x07\xff\xff\xff\xc8\xff\xff\xff\x5e\x81\xff\xff\xff\x00\x02\xff\xff\xff\x1a\xff\xff\xff\xff\xff\xff\xff\x06\x81\xff\xff\xff\x00\x01\xff\xff\xff\x33\xff\xff\xff\xe9\x81\xff\xff\xff\x00\x04\xff\xff\xff\x54\xff\xff\xff\x1e\xff\xff\xff\x01\xff\xff\xff\x86\xff\xff\xff\xc9\x82\xff\xff\xff\x00\x04\xff\xff\xff\xcb\xff\xff\xff\x77\xff\xff\xff\x02\xff\xff\xff\xa1\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x86\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x83\xff\xff\xff\x00\x04\xff\xff\xff\x25\xff\xff\xff\xf7\xff\xff\xff\x07\xff\xff\xff\xc8\xff\xff\xff\x5e\x81\xff\xff\xff\x00\x05\xff\xff\xff\x27\xff\xff\xff\xfb\xff\xff\xff\x5c\xff\xff\xff\x7f\xff\xff\xff\x27\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x04\xff\xff\xff\x65\xff\xff\xff\x3f\xff\xff\xff\x20\xff\xff\xff\x9c\xff\xff\xff\xc9\x81\xff\xff\xff\x00\x05\xff\xff\xff\x27\xff\xff\xff\xff\xff\xff\xff\x5d\xff\xff\xff\x00\xff\xff\xff\x63\xff\xff\xff\xce\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x86\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x84\xff\xff\xff\x00\x03\xff\xff\xff\xd9\xff\xff\xff\x4f\xff\xff\xff\xf8\xff\xff\xff\x14\x81\xff\xff\xff\x00\x02\xff\xff\xff\x01\xff\xff\xff\xf3\xff\xff\xff\x2f\x81\xff\xff\xff\x00\x01\xff\xff\xff\x5c\xff\xff\xff\xc4\x81\xff\xff\xff\x00\x04\xff\xff\xff\x68\xff\xff\xff\xe0\xff\xff\xff\xf5\xff\xff\xff\xcb\xff\xff\xff\x2e\x82\xff\xff\xff\x00\x04\xff\xff\xff\x30\xff\xff\xff\xd7\xff\xff\xff\xec\xff\xff\xff\x8f\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x01\xff\xff\xff\x01\xff\xff\xff\x0e\x86\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x84\xff\xff\xff\x00\x03\xff\xff\xff\xd9\xff\xff\xff\x4f\xff\xff\xff\xf8\xff\xff\xff\x14\x81\xff\xff\xff\x00\x02\xff\xff\xff\x27\xff\xff\xff\xfb\xff\xff\xff\xb3\x81\xff\xff\xff\x27\x00\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x04\xff\xff\xff\x60\xff\xff\xff\xdc\xff\xff\xff\xf9\xff\xff\xff\xcb\xff\xff\xff\x2e\x81\xff\xff\xff\x00\x05\xff\xff\xff\x27\xff\xff\xff\xfb\xff\xff\xff\x7b\xff\xff\xff\xc9\xff\xff\xff\xe3\xff\xff\xff\x47\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x84\xff\xff\xff\x00\x03\xff\xff\xff\x12\xff\xff\xff\x66\xff\xff\xff\x50\xff\xff\xff\xf7\x84\xff\xff\xff\x00\x02\xff\xff\xff\x8d\xff\xff\xff\xce\xff\xff\xff\xc5\x83\xff\xff\xff\x00\x05\xff\xff\xff\xa2\xff\xff\xff\xac\xff\xff\xff\x09\xff\xff\xff\x14\xff\xff\xff\xcd\xff\xff\xff\x72\x90\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x8d\xff\xff\xff\x00\x02\xff\xff\xff\x8d\xff\xff\xff\xce\xff\xff\xff\xc5\x82\xff\xff\xff\x00\x05\xff\xff\xff\x27\xff\xff\xff\xfd\xff\xff\xff\xc9\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xf7\x83\xff\xff\xff\x00\x01\xff\xff\xff\x01\xff\xff\xff\x06\x84\xff\xff\xff\x00\x03\xff\xff\xff\x05\xff\xff\xff\x01\xff\xff\xff\x00\xff\xff\xff\x02\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xff\x82\xff\xff\xff\xf3\x04\xff\xff\xff\xf4\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xff\x82\xff\xff\xff\xf3\x01\xff\xff\xff\xf4\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x84\xff\xff\xff\x00\x03\xff\xff\xff\x13\xff\xff\xff\xb0\xff\xff\xff\xea\xff\xff\xff\xf7\x84\xff\xff\xff\x00\x02\xff\xff\xff\x41\xff\xff\xff\xff\xff\xff\xff\x79\x83\xff\xff\xff\x00\x05\xff\xff\xff\x13\xff\xff\xff\xaf\xff\xff\xff\xf1\xff\xff\xff\xec\xff\xff\xff\x96\xff\xff\xff\x03\x90\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xff\x82\xff\xff\xff\xf3\x04\xff\xff\xff\xf4\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xff\x82\xff\xff\xff\xf3\x01\xff\xff\xff\xf4\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x8d\xff\xff\xff\x00\x02\xff\xff\xff\x41\xff\xff\xff\xff\xff\xff\xff\x79\x82\xff\xff\xff\x00\x05\xff\xff\xff\x27\xff\xff\xff\xff\xff\xff\xff\x77\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xf7\x83\xff\xff\xff\x00\x02\xff\xff\xff\x04\xff\xff\xff\xb7\xff\xff\xff\x22\x83\xff\xff\xff\x00\x04\xff\xff\xff\x86\xff\xff\xff\x49\xff\xff\xff\xb6\xff\xff\xff\xe6\xff\xff\xff\x0c\x81\xff\xff\xff\x00\xe1\xff\xff\xff\x00\x01\xff\xff\xff\x01\xff\xff\xff\x07\x84\xff\xff\xff\x00\x03\xff\xff\xff\x02\xff\xff\xff\x03\xff\xff\xff\x00\xff\xff\xff\x02\x85\xff\xff\xff\x00\x02\xff\xff\xff\x26\xff\xff\xff\xcc\xff\xff\xff\x0f\x82\xff\xff\xff\x00\x04\xff\xff\xff\x37\xff\xff\xff\xf0\xff\xff\xff\x64\xff\xff\xff\x7f\xff\xff\xff\x43\x81\xff\xff\xff\x00\xe1\xff\xff\xff\x00\x02\xff\xff\xff\x07\xff\xff\xff\xd0\xff\xff\xff\x19\x83\xff\xff\xff\x00\x03\xff\xff\xff\x3e\xff\xff\xff\x71\xff\xff\xff\xa4\xff\xff\xff\xc6\x92\xff\xff\xff\x00\xe2\xff\xff\xff\x00\x01\xff\xff\xff\x57\xff\xff\xff\xb2\x83\xff\xff\xff\x00\x04\xff\xff\xff\x09\xff\xff\xff\xda\xff\xff\xff\x8c\xff\xff\xff\x9d\xff\xff\xff\x0e\x91\xff\xff\xff\x00\xff\xff\xff\xff\x00\xb7\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\xc5\xff\xff\xff\x00\x01\xff\xff\xff\x25\xff\xff\xff\xf6\x82\xff\xff\xff\xef\x04\xff\xff\xff\xf0\xff\xff\xff\xef\xff\xff\xff\x00\xff\xff\xff\x25\xff\xff\xff\xf6\x82\xff\xff\xff\xef\x01\xff\xff\xff\xf0\xff\xff\xff\xef\x92\xff\xff\xff\x00\x04\xff\xff\xff\x0e\xff\xff\xff\x8a\xff\xff\xff\xdc\xff\xff\xff\xf0\xff\xff\xff\xa8\x90\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x8d\xff\xff\xff\x00\x01\xff\xff\xff\x25\xff\xff\xff\xf6\x82\xff\xff\xff\xef\x01\xff\xff\xff\xf0\xff\xff\xff\xef\xb0\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x91\xff\xff\xff\x00\x05\xff\xff\xff\x0a\xff\xff\xff\xd0\xff\xff\xff\xa4\xff\xff\xff\x1f\xff\xff\xff\x01\xff\xff\xff\x18\x90\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x85\xff\xff\xff\x00\x02\xff\xff\xff\x1f\xff\xff\xff\x4c\xff\xff\xff\x10\x81\xff\xff\xff\x00\x04\xff\xff\xff\x19\xff\xff\xff\x46\xff\xff\xff\x0e\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\xb0\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x89\xff\xff\xff\x00\x04\xff\xff\xff\x15\xff\xff\xff\xc1\xff\xff\xff\xf4\xff\xff\xff\x9f\xff\xff\xff\x01\x82\xff\xff\xff\x00\x01\xff\xff\xff\x85\xff\xff\xff\xbb\x84\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x8d\xff\xff\xff\x00\x05\xff\xff\xff\x27\xff\xff\xff\xfb\xff\xff\xff\x8e\xff\xff\xff\xf1\xff\xff\xff\xc4\xff\xff\xff\x16\x81\xff\xff\xff\x00\x09\xff\xff\xff\x57\xff\xff\xff\xee\xff\xff\xff\x52\xff\xff\xff\x00\xff\xff\xff\x07\xff\xff\xff\x9a\xff\xff\xff\xee\xff\xff\xff\x44\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x90\xff\xff\xff\x00\x01\xff\xff\xff\xa1\xff\xff\xff\x97\x82\xff\xff\xff\x00\x08\xff\xff\xff\x60\xff\xff\xff\xdc\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xff\xff\xff\xff\xf3\xff\xff\xff\xef\xff\xff\xff\xbd\xff\xff\xff\x42\x82\xff\xff\xff\x00\x04\xff\xff\xff\x83\xff\xff\xff\xdd\xff\xff\xff\xbc\xff\xff\xff\x8a\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x04\xff\xff\xff\x25\xff\xff\xff\xcb\xff\xff\xff\xf3\xff\xff\xff\xb4\xff\xff\xff\x0f\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x89\xff\xff\xff\x00\x04\xff\xff\xff\x9d\xff\xff\xff\xa9\xff\xff\xff\x0d\xff\xff\xff\xcf\xff\xff\xff\x56\x82\xff\xff\xff\x00\x08\xff\xff\xff\xe3\xff\xff\xff\x47\xff\xff\xff\x3e\xff\xff\xff\xd6\xff\xff\xff\xea\xff\xff\xff\xa1\xff\xff\xff\xef\xff\xff\xff\x27\xff\xff\xff\xfb\x8d\xff\xff\xff\x00\x05\xff\xff\xff\x27\xff\xff\xff\xff\xff\xff\xff\x76\xff\xff\xff\x01\xff\xff\xff\x9e\xff\xff\xff\x9b\x81\xff\xff\xff\x00\x09\xff\xff\xff\x10\xff\xff\xff\xc8\xff\xff\xff\xae\xff\xff\xff\x0f\xff\xff\xff\x42\xff\xff\xff\xe8\xff\xff\xff\x7f\xff\xff\xff\x10\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x90\xff\xff\xff\x00\x02\xff\xff\xff\x55\xff\xff\xff\xe6\xff\xff\xff\x01\x81\xff\xff\xff\x00\x10\xff\xff\xff\xb1\xff\xff\xff\x90\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\xff\xff\xff\x00\xff\xff\xff\x05\xff\xff\xff\x45\xff\xff\xff\xe3\xff\xff\xff\x3c\xff\xff\xff\x00\xff\xff\xff\x18\xff\xff\xff\xff\xff\xff\xff\x16\xff\xff\xff\x00\xff\xff\xff\x88\xff\xff\xff\xf7\x82\xff\xff\xff\x00\x04\xff\xff\xff\xc4\xff\xff\xff\x7b\xff\xff\xff\x02\xff\xff\xff\xa5\xff\xff\xff\x93\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x89\xff\xff\xff\x00\x04\xff\xff\xff\xed\xff\xff\xff\x24\xff\xff\xff\x00\xff\xff\xff\x55\xff\xff\xff\xa1\x81\xff\xff\xff\x00\x09\xff\xff\xff\x14\xff\xff\xff\xff\xff\xff\xff\x14\xff\xff\xff\xe9\xff\xff\xff\x5a\xff\xff\xff\x03\xff\xff\xff\x88\xff\xff\xff\xf7\xff\xff\xff\x27\xff\xff\xff\xfb\x8d\xff\xff\xff\x00\x05\xff\xff\xff\x27\xff\xff\xff\xff\xff\xff\xff\x10\xff\xff\xff\x00\xff\xff\xff\x3d\xff\xff\xff\xe1\x81\xff\xff\xff\x00\x09\xff\xff\xff\x03\xff\xff\xff\x6b\xff\xff\xff\xe8\xff\xff\xff\x90\xff\xff\xff\xd4\xff\xff\xff\xc2\xff\xff\xff\x0e\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x90\xff\xff\xff\x00\x02\xff\xff\xff\x0e\xff\xff\xff\xf9\xff\xff\xff\xf8\x81\xff\xff\xff\xf7\x04\xff\xff\xff\xff\xff\xff\xff\x43\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x08\xff\xff\xff\x6a\xff\xff\xff\xb0\xff\xff\xff\x00\xff\xff\xff\x11\xff\xff\xff\xfb\xff\xff\xff\x49\xff\xff\xff\x00\xff\xff\xff\x32\xff\xff\xff\xf7\x81\xff\xff\xff\x00\x05\xff\xff\xff\x10\xff\xff\xff\xff\xff\xff\xff\x12\xff\xff\xff\x00\xff\xff\xff\x3e\xff\xff\xff\xdf\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x88\xff\xff\xff\x00\x01\xff\xff\xff\x15\xff\xff\xff\xed\x81\xff\xff\xff\x00\x01\xff\xff\xff\x1d\xff\xff\xff\xc5\x81\xff\xff\xff\x00\x09\xff\xff\xff\x23\xff\xff\xff\xfd\xff\xff\xff\x22\xff\xff\xff\xfd\xff\xff\xff\x02\xff\xff\xff\x00\xff\xff\xff\x31\xff\xff\xff\xf7\xff\xff\xff\x27\xff\xff\xff\xfb\x8d\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfd\x81\xff\xff\xff\x00\x01\xff\xff\xff\x2a\xff\xff\xff\xf3\x82\xff\xff\xff\x00\x01\xff\xff\xff\x0a\xff\xff\xff\x78\x81\xff\xff\xff\xf0\x00\xff\xff\xff\x3c\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x91\xff\xff\xff\x00\x08\xff\xff\xff\xbd\xff\xff\xff\x7c\xff\xff\xff\x00\xff\xff\xff\x45\xff\xff\xff\xf1\xff\xff\xff\x04\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x36\xff\xff\xff\xe4\x81\xff\xff\xff\x00\x04\xff\xff\xff\x5f\xff\xff\xff\xd8\xff\xff\xff\xf0\xff\xff\xff\xf1\xff\xff\xff\xf6\x81\xff\xff\xff\x00\x01\xff\xff\xff\x24\xff\xff\xff\xfd\x81\xff\xff\xff\x00\x01\xff\xff\xff\x2a\xff\xff\xff\xf3\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x88\xff\xff\xff\x00\x05\xff\xff\xff\x23\xff\xff\xff\xda\xff\xff\xff\x00\xff\xff\xff\xae\xff\xff\xff\x0b\xff\xff\xff\xd3\x81\xff\xff\xff\x00\x09\xff\xff\xff\x16\xff\xff\xff\xff\xff\xff\xff\x11\xff\xff\xff\xe8\xff\xff\xff\x5c\xff\xff\xff\x04\xff\xff\xff\x8a\xff\xff\xff\xf7\xff\xff\xff\x27\xff\xff\xff\xff\x81\xff\xff\xff\xef\x01\xff\xff\xff\xc2\xff\xff\xff\x2b\x89\xff\xff\xff\x00\x05\xff\xff\xff\x27\xff\xff\xff\xff\xff\xff\xff\x10\xff\xff\xff\x00\xff\xff\xff\x3d\xff\xff\xff\xe0\x83\xff\xff\xff\x00\x03\xff\xff\xff\x08\xff\xff\xff\xbe\xff\xff\xff\xc6\xff\xff\xff\x0b\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x91\xff\xff\xff\x00\x04\xff\xff\xff\x71\xff\xff\xff\xbe\xff\xff\xff\x00\xff\xff\xff\x87\xff\xff\xff\xaa\x81\xff\xff\xff\x00\x06\xff\xff\xff\xd7\xff\xff\xff\xff\xff\xff\xff\xef\xff\xff\xff\xc5\xff\xff\xff\x00\xff\xff\xff\x2a\xff\xff\xff\xf2\x84\xff\xff\xff\x00\x01\xff\xff\xff\x2e\xff\xff\xff\xf1\x81\xff\xff\xff\x00\x05\xff\xff\xff\x11\xff\xff\xff\xff\xff\xff\xff\x12\xff\xff\xff\x00\xff\xff\xff\x41\xff\xff\xff\xdd\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x88\xff\xff\xff\x00\x05\xff\xff\xff\x20\xff\xff\xff\xdc\xff\xff\xff\x00\xff\xff\xff\x88\xff\xff\xff\x0d\xff\xff\xff\xcf\x82\xff\xff\xff\x00\x00\xff\xff\xff\xe9\x81\xff\xff\xff\x3b\x09\xff\xff\xff\xd3\xff\xff\xff\xe9\xff\xff\xff\x9b\xff\xff\xff\xf6\xff\xff\xff\x27\xff\xff\xff\xfb\xff\xff\xff\x00\xff\xff\xff\x05\xff\xff\xff\x8e\xff\xff\xff\xc8\x89\xff\xff\xff\x00\x05\xff\xff\xff\x27\xff\xff\xff\xff\xff\xff\xff\x78\xff\xff\xff\x02\xff\xff\xff\x9f\xff\xff\xff\x98\x83\xff\xff\xff\x00\x03\xff\xff\xff\x11\xff\xff\xff\xc9\xff\xff\xff\x79\xff\xff\xff\x02\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x8a\xff\xff\xff\x00\x03\xff\xff\xff\x4c\xff\xff\xff\xdb\xff\xff\xff\xe3\xff\xff\xff\x4e\x82\xff\xff\xff\x00\x04\xff\xff\xff\x25\xff\xff\xff\xf7\xff\xff\xff\x07\xff\xff\xff\xc8\xff\xff\xff\x5e\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x3a\xff\xff\xff\xe3\x81\xff\xff\xff\x00\x04\xff\xff\xff\x62\xff\xff\xff\x3b\xff\xff\xff\x1c\xff\xff\xff\x99\xff\xff\xff\xc9\x82\xff\xff\xff\x00\x04\xff\xff\xff\xc7\xff\xff\xff\x7e\xff\xff\xff\x00\xff\xff\xff\x8d\xff\xff\xff\x98\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x88\xff\xff\xff\x00\x05\xff\xff\xff\x08\xff\xff\xff\xf2\xff\xff\xff\x02\xff\xff\xff\x00\xff\xff\xff\x29\xff\xff\xff\xb3\x82\xff\xff\xff\x00\x01\xff\xff\xff\x92\xff\xff\xff\xa1\x82\xff\xff\xff\x00\x03\xff\xff\xff\x48\xff\xff\xff\xe6\xff\xff\xff\x27\xff\xff\xff\xfb\x81\xff\xff\xff\x00\x01\xff\xff\xff\x31\xff\xff\xff\xf1\x89\xff\xff\xff\x00\x05\xff\xff\xff\x27\xff\xff\xff\xfb\xff\xff\xff\x8d\xff\xff\xff\xf1\xff\xff\xff\xc3\xff\xff\xff\x14\x82\xff\xff\xff\x00\x03\xff\xff\xff\x0a\xff\xff\xff\x74\xff\xff\xff\xe8\xff\xff\xff\x54\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x89\xff\xff\xff\x00\x04\xff\xff\xff\x0e\xff\xff\xff\xf5\xff\xff\xff\x2b\xff\xff\xff\x5a\xff\xff\xff\xe3\x83\xff\xff\xff\x00\x03\xff\xff\xff\xd9\xff\xff\xff\x4f\xff\xff\xff\xf8\xff\xff\xff\x14\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x74\xff\xff\xff\xaf\x81\xff\xff\xff\x00\x04\xff\xff\xff\x60\xff\xff\xff\xdc\xff\xff\xff\xf9\xff\xff\xff\xcb\xff\xff\xff\x2e\x82\xff\xff\xff\x00\x04\xff\xff\xff\x29\xff\xff\xff\xcf\xff\xff\xff\xf7\xff\xff\xff\xfb\xff\xff\xff\x27\x81\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x04\xff\xff\xff\x27\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x89\xff\xff\xff\x00\x04\xff\xff\xff\xbd\xff\xff\xff\x66\xff\xff\xff\x01\xff\xff\xff\x94\xff\xff\xff\x6a\x82\xff\xff\xff\x00\x0c\xff\xff\xff\x13\xff\xff\xff\xdf\xff\xff\xff\x7c\xff\xff\xff\x0b\xff\xff\xff\x13\xff\xff\xff\xc7\xff\xff\xff\x96\xff\xff\xff\x27\xff\xff\xff\xfb\xff\xff\xff\x00\xff\xff\xff\x03\xff\xff\xff\x8b\xff\xff\xff\xca\x92\xff\xff\xff\x00\x03\xff\xff\xff\x27\xff\xff\xff\xd5\xff\xff\xff\xba\xff\xff\xff\x19\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xfb\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xf7\x89\xff\xff\xff\x00\x04\xff\xff\xff\x16\xff\xff\xff\xfd\xff\xff\xff\x2a\xff\xff\xff\x58\xff\xff\xff\xcd\x83\xff\xff\xff\x00\x02\xff\xff\xff\x8d\xff\xff\xff\xce\xff\xff\xff\xc5\x82\xff\xff\xff\x00\x06\xff\xff\xff\x27\xff\xff\xff\xfb\xff\xff\xff\x00\xff\xff\xff\x06\xff\xff\xff\x4e\xff\xff\xff\xeb\xff\xff\xff\x3d\x83\xff\xff\xff\x00\x01\xff\xff\xff\x05\xff\xff\xff\x01\x83\xff\xff\xff\x00\x03\xff\xff\xff\x29\xff\xff\xff\x1c\xff\xff\xff\x80\xff\xff\xff\x94\x82\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xff\x82\xff\xff\xff\xf3\x04\xff\xff\xff\xf4\xff\xff\xff\xf7\xff\xff\xff\x00\xff\xff\xff\x27\xff\xff\xff\xff\x82\xff\xff\xff\xf3\x01\xff\xff\xff\xf4\xff\xff\xff\xf7\x89\xff\xff\xff\x00\x04\xff\xff\xff\x28\xff\xff\xff\xd0\xff\xff\xff\xf2\xff\xff\xff\xa5\xff\xff\xff\x04\x83\xff\xff\xff\x00\x0b\xff\xff\xff\x19\xff\xff\xff\x9e\xff\xff\xff\xe4\xff\xff\xff\xea\xff\xff\xff\x9d\xff\xff\xff\x0a\xff\xff\xff\x27\xff\xff\xff\xff\xff\xff\xff\xf3\xff\xff\xff\xf4\xff\xff\xff\xc7\xff\xff\xff\x2e\x83\xff\xff\xff\x00\x02\xff\xff\xff\x15\xff\xff\xff\xc0\xff\xff\xff\x1c\x8b\xff\xff\xff\x00\x02\xff\xff\xff\x07\xff\xff\xff\x2a\xff\xff\xff\x12\x83\xff\xff\xff\x00\x01\xff\xff\xff\x27\xff\xff\xff\xff\x82\xff\xff\xff\xf3\x01\xff\xff\xff\xf4\xff\xff\xff\xf7\x8a\xff\xff\xff\x00\x03\xff\xff\xff\x7b\xff\xff\xff\xf2\xff\xff\xff\xbf\xff\xff\xff\x1a\x83\xff\xff\xff\x00\x02\xff\xff\xff\x41\xff\xff\xff\xff\xff\xff\xff\x79\x82\xff\xff\xff\x00\x05\xff\xff\xff\x27\xff\xff\xff\xff\xff\xff\xff\xf3\xff\xff\xff\xf1\xff\xff\xff\xc2\xff\xff\xff\x48\x83\xff\xff\xff\x00\x02\xff\xff\xff\x19\xff\xff\xff\xb9\xff\xff\xff\x07\x83\xff\xff\xff\x00\x03\xff\xff\xff\x24\xff\xff\xff\xaf\xff\xff\xff\xf8\xff\xff\xff\x4d\x82\xff\xff\xff\x00\xb0\xff\xff\xff\x00\x02\xff\xff\xff\x10\xff\xff\xff\xc9\xff\xff\xff\x3a\xae\xff\xff\xff\x00\x81\xff\xff\xff\x04\x8b\xff\xff\xff\x00\x02\xff\xff\xff\x08\xff\xff\xff\xc4\xff\xff\xff\x2e\x84\xff\xff\xff\x00\x04\xff\xff\xff\x06\xff\xff\xff\xce\xff\xff\xff\x54\xff\xff\xff\x72\xff\xff\xff\x14\x81\xff\xff\xff\x00\xe1\xff\xff\xff\x00\x02\xff\xff\xff\x02\xff\xff\xff\xc5\xff\xff\xff\x24\x9a\xff\xff\xff\x00\xe1\xff\xff\xff\x00\x01\xff\xff\xff\x77\xff\xff\xff\x8f\x9b\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x54\x52\x55\x45\x56\x49\x53\x49\x4f\x4e\x2d\x58\x46\x49\x4c\x45\x2e\x00"
