/*
 * happydigger - program for cataloging archaeological finds
 * Copyright (C) 2004, 2005, 2007 Joop Stakenborg <pg4i@amsat.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA
 */

#include <gtk/gtk.h>
#include <stdio.h>
#include <string.h>

#include "interface.h"
#include "utils.h"
#include "callbacks.h"
#include "preferences.h"
#include "db.h"

extern GtkWidget *window;
extern GtkWidget *mapchooserdialog;
extern statestype state;
extern sqlite3 *db;
extern preferencestype preferences;

GtkWidget *progressdialog;
gint i, records, pages;
FILE *f;

void
on_progresscancelbutton_clicked        (GtkButton       *button,
                                        gpointer         user_data)
{
	i = records;
}


gboolean
on_mapchooserdialog_delete_event       (GtkWidget       *widget,
                                        GdkEvent        *event,
                                        gpointer         user_data)
{
	gtk_widget_set_sensitive (window, 1);
	return FALSE;
}


void
on_folder_cancel_button_clicked        (GtkButton       *button,
                                        gpointer         user_data)
{
	gtk_widget_set_sensitive (window, 1);
	gtk_widget_destroy (gtk_widget_get_toplevel (GTK_WIDGET(button)));
}

static gboolean create_thumbnail (gchar *filename, GdkPixbuf *pb, gint w, gint h)
{
	FILE *fd;

	if (!filename || !pb)
		return FALSE;
	fd = fopen(filename, "wb");
	if (fd)
	{
		GdkPixbuf *tmp_im;
	 	tmp_im = gdk_pixbuf_scale_simple (pb, w, h, GDK_INTERP_BILINEAR);
		gdk_pixbuf_save (tmp_im, filename, "jpeg", NULL, "quality", "50", NULL);
		g_object_unref (tmp_im);
		return TRUE;
	}
	return FALSE;
}

/* binary file copy */
static void copyfile (gchar *from, gchar *to)
{
	FILE *fin, *fout;
	gchar buf[1025];
	gint numread, numwrite;

	fin = fopen (from, "r");
	if (fin)
	{
		fout = fopen (to, "w");
		if (fout)
		{
			while (!feof(fin))
			{
				numread = fread (buf, 1, sizeof(buf), fin);
				numwrite = fwrite (buf, 1, numread, fout);
			}
			fclose (fout);
		}
		fclose (fin);
	}
}

/* close file and add footer */
static FILE *closefile (FILE *fp)
{
	fprintf (fp, "</table>\n");

	fprintf (fp, "<p>\n");
	fprintf (fp, "<table width=\"100%%\">\n");
	if (strncasecmp (preferences.webpagefootnote, "^", 1))
	{
		fprintf (fp, "  <tr><td align=\"center\">\n");
		fprintf (fp, "%s\n", preferences.webpagefootnote);
		fprintf (fp, "  </td></tr>\n");
	}

	fprintf (fp, "  <tr><td align=\"center\">\n");
	fprintf (fp,
"    <i>\n    Created with the help of\n    <a href=\"http://happydigger.nongnu.org\">happydigger</a> %s\n    </i>\n", VERSION);
	fprintf (fp, "  </td></tr>\n");
	fprintf (fp, "</table>\n");
	fprintf (fp, "</p>\n");
	fprintf (fp, "</body>\n");
	fprintf (fp, "</html>");
	fflush (f);
	fclose (fp);
	fp = NULL;
	return fp;
}

/* header and title */
static void openfile (FILE *fp)
{
	fprintf (fp, "<!DOCTYPE html PUBLIC \"-//W3C//DTD HTML 4.01 Transitional//EN\"\n");
	fprintf (fp, "   \"http://www.w3.org/TR/html4/loose.dtd\">\n");
	fprintf (fp, "<html>\n");
	fprintf (fp, "<head>\n");
	fprintf (fp, "<META HTTP-EQUIV=\"Content-Type\" CONTENT=\"text/html; charset=UTF-8\">\n");
	fprintf (fp, "<META NAME=\"keywords\" CONTENT=\"happydigger, metal detecting, treasure hunting, archaeology, finds\">\n");
	if (strncasecmp (preferences.webpagetitle, "^", 1))
	{
		fprintf (fp, "<title>\n");
		fprintf (fp, "%s\n", preferences.webpagetitle);
		fprintf (fp, "</title>\n");
	}
	fprintf (fp, "<link rel=\"stylesheet\" href=\"style.css\" type=\"text/css\">\n");
	fprintf (fp, "</head>\n");
	if (strncasecmp (preferences.webpagetitle, "^", 1))
	{
		fprintf (fp, "<center>\n");
		fprintf (fp, "<h2>\n");
		fprintf (fp, "%s\n", preferences.webpagetitle);
		fprintf (fp, "</h2>\n");
		fprintf (fp, "</center>\n\n");
	}

	/* page number here */
	fprintf (fp, "<!-- page number -->\n");
	fprintf (fp, "<table width=\"100%%\">\n");
	fprintf (f, "  <tr><td align=\"right\">\n");
	if (pages == 1)
		fprintf (f, "    page 1 / 1\n");
	else
	{
		if (i / preferences.recordsperpage + 1 == 2)
			fprintf (f, "    <a href=\"index.html\"><<</a>\n    &nbsp;\n");
		else if (i / preferences.recordsperpage + 1 > 2)
			fprintf (f, "    <a href=\"page%03d.html\"><<</a>\n    &nbsp;\n",
				i / preferences.recordsperpage - 1);
		fprintf (f, "    page %d / %d\n", i / preferences.recordsperpage + 1, pages);
		if (i / preferences.recordsperpage + 1 < pages)
			fprintf (f, "    &nbsp;\n    <a href=\"page%03d.html\">>></a>\n",
				i / preferences.recordsperpage + 1);
	}
	fprintf (f, "  </td></tr>\n");
	fprintf (fp, "</table>\n\n");

	fprintf (fp, "<table width=\"100%%\">\n");
	fflush (f);
}

/* creation of the webpage and copying of images */
static int
webpage_callback (void *folder, int argc, char **argv, char **columnNames)
{
	gchar *file, *imagename, *thumbname, *imagepath;
	GtkWidget *bar;
	GdkPixbuf *pix;
	gdouble w, h, ratio;

	bar = lookup_widget (progressdialog, "progressbar");
	gtk_progress_bar_set_fraction (GTK_PROGRESS_BAR(bar), (float)i / records);
	while (gtk_events_pending ()) gtk_main_iteration (); 
	if ((pages == 1) || (i - preferences.recordsperpage < 0))
	{
		if (!f)
		{
			file = g_strdup_printf
				("%s%sindex.html", (gchar *)folder, G_DIR_SEPARATOR_S);
			f = fopen (file, "w");
			g_free (file);
			openfile (f);
		}
	}
	else if (!(i % preferences.recordsperpage))
	{
		if (f) f = closefile (f);
		file = g_strdup_printf
			("%s%spage%03d.html", (gchar *)folder, G_DIR_SEPARATOR_S,
				i / preferences.recordsperpage);
		f = fopen (file, "w");
		g_free (file);
		openfile (f);
	}

	/* 2 columns */
	if (!(i % 2)) fprintf (f, "  <tr><td>\n    <table width=\"50%%\" align=\"left\">\n");
	else fprintf (f, "    <table width=\"50%%\" align=\"right\">\n");

	fprintf (f, "      <tr><td align=\"center\">\n");
	if (g_utf8_strlen (argv[14], -1) > 0)
	{
		imagename = g_strdup_printf ("img%03da.jpeg", records - i -1);
		thumbname = g_strdup_printf ("img%03da_thumb.jpeg", records - i -1);
		imagepath = g_build_filename (folder, imagename, NULL);
		copyfile (argv[14], imagepath);
		pix = gdk_pixbuf_new_from_file (argv[14], NULL);
		imagepath = g_build_filename (folder, thumbname, NULL);
		if (pix)
		{
			w = gdk_pixbuf_get_width (pix);
			h = gdk_pixbuf_get_height (pix);
			if (w >= h )
			{
				ratio = w/h;
				create_thumbnail (imagepath, pix, 100.0 * ratio, 100);
			}
			else
			{
				ratio = h/w;
				create_thumbnail (imagepath, pix, 100, 100.0 * ratio);
			}
			fprintf (f,
				"      <div class=\"thumb\"><a href=\"%s\"><img src=\"%s\" alt=\"%s\"></a></div>\n",
				imagename, thumbname, imagename);
			g_object_unref (pix);
		}
		g_free (imagepath);
		g_free (imagename);
		g_free (thumbname);
	}
	/* second image */
	if (g_utf8_strlen (argv[15], -1) > 0)
	{
		imagename = g_strdup_printf ("img%03db.jpeg", records - i - 1);
		thumbname = g_strdup_printf ("img%03db_thumb.jpeg", records - i -1);
		imagepath = g_build_filename (folder, imagename, NULL);
		copyfile (argv[15], imagepath);
		pix = gdk_pixbuf_new_from_file (argv[15], NULL);
		imagepath = g_build_filename (folder, thumbname, NULL);
		if (pix)
		{
			w = gdk_pixbuf_get_width (pix);
			h = gdk_pixbuf_get_height (pix);
			if (w >= h )
			{
				ratio = w/h;
				create_thumbnail (imagepath, pix, 100.0 * ratio, 100);
			}
			else
			{
				ratio = h/w;
				create_thumbnail (imagepath, pix, 100, 100.0 * ratio);
			}
			fprintf (f,
				"      <div class=\"thumb\"><a href=\"%s\"><img src=\"%s\" alt=\"%s\"></a></div>\n",
				imagename, thumbname, imagename);
			g_object_unref (pix);
			g_free (imagepath);
			g_free (imagename);
		}
	}
	fprintf (f, "      </td></tr>\n");
	fprintf (f, "      <tr><td>\n");
	fprintf (f, "<div class=\"title\">%s</div>\n", argv[6]);
	fprintf (f, "      </td></tr>\n");

	fprintf (f, "    </table>\n");
	if (i % 2) fprintf (f, "  </td></tr>\n");
	fflush (f);

	if (i == records - 1) f = closefile (f);

	i++;
	return 0;
}

/* select a folder and query the database */
void
on_folder_open_button_clicked          (GtkButton       *button,
                                        gpointer         user_data)
{
	GtkWidget *dialog, *entry, *footentry;
	gchar *folder, *query, *err, *webpagetitle, *webpagefootnote,
		*cssdestpath, *cssorigpath;
	gint ret, remainder;

	i = 0;
	f = NULL;

	dialog = gtk_widget_get_toplevel (GTK_WIDGET(button));
	folder = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (dialog));
	state.webmapdir = g_strconcat (folder, G_DIR_SEPARATOR_S, NULL);

	records = max_db ();
	pages = records / preferences.recordsperpage;
	remainder = records % preferences.recordsperpage;
	if (remainder > 0) pages++;

	if (preferences.reverseorder == 1)
		query = g_strdup_printf ("SELECT * from %s ORDER BY %s DESC;",
			state.table, state.sortby);
	else
		query = g_strdup_printf ("SELECT * from %s ORDER BY %s ASC;",
			state.table, state.sortby);

	entry = lookup_widget (dialog, "webpageentry");
	webpagetitle = gtk_editable_get_chars (GTK_EDITABLE(entry), 0, -1);
	if (g_utf8_strlen (webpagetitle, -1) > 0)
		preferences.webpagetitle = g_strdup (webpagetitle);
	else
		preferences.webpagetitle = g_strdup ("^");
	g_free (webpagetitle);

	footentry = lookup_widget (dialog, "webpagefootentry");
	webpagefootnote = gtk_editable_get_chars (GTK_EDITABLE(footentry), 0, -1);
	if (g_utf8_strlen (webpagefootnote, -1) > 0)
		preferences.webpagefootnote = g_strdup (webpagefootnote);
	else
		preferences.webpagefootnote = g_strdup ("^");
	g_free (webpagefootnote);

	progressdialog = (GtkWidget *)create_progressdialog();
	gtk_widget_destroy (dialog);
	gtk_widget_show (progressdialog);

	cssdestpath = g_build_filename (folder, "style.css", NULL);
	cssorigpath = g_build_filename (PACKAGE_DATA_DIR, "style.css", NULL);
	copyfile (cssorigpath, cssdestpath);
	g_free (cssdestpath);
	g_free (cssorigpath);

	ret = sqlite3_exec (db, query, webpage_callback, folder, &err);
	if (ret != SQLITE_OK)
		g_print ("Error on SELECT: %s\n", err);
	g_free (query);
	g_free (folder);

	gtk_widget_destroy (progressdialog);
	gtk_widget_set_sensitive (window, 1);
}

