{-# OPTIONS -fglasgow-exts #-}
module HAppS.MACID.Checkpoint(CheckpointEvent(..), loadState, periodicCheckpoints, checkpointAndExit) where


import HAppS.MACID.Saver
import HAppS.MACID.Serialize
import HAppS.MACID.SideEffect
import HAppS.MACID.Transaction
import HAppS.MACID.Types

import Control.Monad (when)
import Control.Concurrent
import Control.Exception as E
import qualified Data.ByteString.Char8 as P
import qualified Data.Map as M
import qualified Data.Set as S
import System.IO
import Text.Printf
import Data.Typeable

import System.Log.Logger

logMC = logM "HAppS.MACID.Checkpoint"

{- State on disk:

* ${TXD}/events-*       event files in ascending order
* ${TXD}/side-effects-* completed side-effects
* ${TXD}/checkpoint-*   checkpoint files in ascending order
* ${TXD}/current        pointer to last checkpoint

-}

formatFilePath :: TxConfig -> String -> FilePath
formatFilePath conf str = printf "%s-%010d" str $ txcFileNum_ conf

-- | Load state
loadState :: Serialize st => 
             TxRun st -> TxConfig -> EvLoaders -> st 
          -> (st -> TxConfig -> IO ())-> IO ()
loadState txrun@(TxRun _ _ tisrc _) conf0 evl st0 runner = do
  logMC NOTICE "> Starting to load state"
  nullSaver <- createSaverImpl NullSaver
  let conf = conf0 { txcSaverImpl_ = nullSaver }
  saver <- createSaverImpl $ txcSaver conf0
  let rf def fp = handle (\_ -> return def) (fmap (concatMap P.unpack) (saverGet saver fp) >>= readIO)
  cs <- rf [("version","0")] "current"
  let ml s = maybe (fail ("Field not defined in current file: "++show s)) readIO $ lookup s cs
  ver <- ml "version" :: IO Int
  logMC NOTICE ("> State version "++show ver)
  case ver of
    -- pristine
    0 -> do putMVar tisrc 1
            let conf' = conf { txcFileNum_ = 0 }
            eh      <- saverOpen saver (formatFilePath conf' "events")
            saverClose saver
            runner st0 conf' { txcSaverImpl_ = eh }
            -- Here we can safely checkpoint as there won't be any pending side-effects to block things.
            addCheckpoint txrun
    1 -> fail "Version 1 state dir without side-effects, upgrade not supported"
    -- version 2, loading
    2 -> do putMVar tisrc =<< ml "txId"
            txcF  <- ml "txcFileNum"
            let conf' = conf { txcFileNum_ = txcF }
            raw     <- saverGet saver (formatFilePath conf' "checkpoint")
            (st,"") <- decodeStringM $ concatMap P.unpack raw
            runner st conf'
            nextFileNum <- loadEvents conf' tisrc saver evl
            -- switch to a new saver in a synchronous fashion, not blocking on side effects.
            logMC NOTICE ("> switching to real side-effect saver")
            syncNextSaver txrun saver (nextFileNum - 1)
            -- and do a checkpoint but don't wait for it to be ready.
            forkIO $ addCheckpoint txrun
            return ()
    x -> fail ("Tried to load state with version "++show x++" which is not supported.")

-- | Run this after state has been loaded. Return last filenum.
loadEvents :: TxConfig -> TxIdSource -> SaverImpl -> EvLoaders -> IO Int
loadEvents conf tisrc realSaver evl = either (\_ -> return (txcFileNum_ conf)) return =<< E.try (do
  logMC NOTICE ("> Starting to load side-effects "++show (txcFileNum_ conf))
  raw <- fmap (either (const []) id) $ E.try $ saverGet realSaver (formatFilePath conf "side-effects")
  let ses = S.fromList $ flip concatMap (lines $ concat $ map P.unpack raw) $ \line ->
              case readsPrec 0 line of
                [(num,"")] -> [num]
                _          -> []
  logMC NOTICE ("> Starting to load events "++show (txcFileNum_ conf))
  let work k "" = return k
      work k xs = do (Wrap s,r) <- decodeStringM xs
                     case M.lookup s evl of
                         Just ehr -> do (t,r) <- ehr r ses
                                        work (max k t) r
                         Nothing  -> do logMC WARNING ("No loader for events of: "++show s++" ignoring.")
                                        return k
  raw <- saverGet realSaver (formatFilePath conf "events")
  km <- work 0 $ concatMap P.unpack raw
  modifyMVar_ tisrc $ \old -> return (max old (km+1))
  loadEvents (conf { txcFileNum_ = 1 + txcFileNum_ conf }) tisrc realSaver evl)

data CheckpointEvent = CheckpointEvent 
                       deriving (Typeable,Read,Show) -- for defaultSerialize
instance Serialize CheckpointEvent where
    typeString _ = "CheckpointEvent"
    encodeStringM CheckpointEvent = return ""
    decodeStringM s = return (CheckpointEvent,s)

-- | Run a checkpoint every N seconds or N events which ever happens sooner.
periodicCheckpoints :: Serialize a => TxRun a -> Seconds -> Int -> IO ()
periodicCheckpoints txrun@(TxRun _ _ tisrc _) secs n = forkIO (loop secs 0) >> return ()
    where loop ptime ptid = do let ctime = min 60 ptime
                               threadDelay (1000000 * ctime)
                               tid <- readMVar tisrc
                               if tid-ptid > fromIntegral n || ptime <= ctime
                                  then do addCheckpoint txrun
                                          loop secs tid
                                  else do loop (ptime - ctime) ptid
                         

-- | Run a checkpoint *now* in a synchronous fashion.
addCheckpoint :: (Serialize a) => TxRun a -> IO ()
addCheckpoint txrun@(TxRun queue _ tisrc _) = do
  logMC NOTICE "CHECKPOINT start"
  stv  <- newEmptyMVar
  tev <- addTxContext tisrc CheckpointEvent
  putMVar queue $! IHR (typeString $ proxy CheckpointEvent) tev $ checkpointHandler txrun stv
  (conf,st) <- takeMVar stv
  doSaveCheckpoint txrun conf st

-- | Switch to the next saver *now* in a synchronous fashion.
syncNextSaver :: Serialize a => TxRun a -> SaverImpl -> Int -> IO ()
syncNextSaver txrun@(TxRun queue _ tisrc _) saver fn = do
  logMC NOTICE "syncNextSaver start"
  stv  <- newEmptyMVar
  tev <- addTxContext tisrc CheckpointEvent
  putMVar queue $! IHR (typeString $ proxy CheckpointEvent) tev $ \_ _ conf -> do
    let conf' = conf { txcSaverImpl_ = saver, txcFileNum_ = fn}
    conf'' <- saverNext txrun conf' $ \s c -> SwitchNow s (doCleanup c >> putMVar stv ())
    return (Nothing, Just conf'', return ())
  takeMVar stv
  logMC NOTICE "syncNextSaver done"

checkpointAndExit :: Serialize a => TxRun a -> IO ()
checkpointAndExit txrun@(TxRun queue _ tisrc _) = do
  logMC NOTICE "CHECKPOINT start"
  stv  <- newEmptyMVar
  wmv  <- newEmptyMVar
  tev <- addTxContext tisrc CheckpointEvent
  putMVar queue $! IHR (typeString $ proxy CheckpointEvent) tev $ \x st conf -> do
    x <- checkpointHandler txrun stv x st conf
    (conf,st) <- takeMVar stv
    doSaveCheckpoint txrun conf st
    putMVar wmv ()
    takeMVar stv -- this blocks forever
    return x
  takeMVar wmv

checkpointHandler :: Serialize a => 
                     TxRun a -> MVar (TxConfig,a) -> TxContext CheckpointEvent -> a -> TxConfig -> IO (Maybe a, Maybe TxConfig, IO ())
checkpointHandler txrun stv _ st conf = do
  logMC NOTICE "CHECKPOINT> state grabber> start"
  conf'' <- saverNext txrun conf $ \s c -> Sequence s (putMVar stv (c,st))
  logMC NOTICE "CHECKPOINT> state grabber> ok"
  return (Nothing,Just conf'',return ())

saverNext :: TxRun a -> TxConfig -> (SaverImpl -> TxConfig -> SideEffect) -> IO TxConfig
saverNext (TxRun _ sideQueue _ _) conf comp = do
  saverClose (txcSaverImpl_ conf)
  let conf'  = conf  { txcFileNum_ = 1 + txcFileNum_ conf }
  nsaver <- saverOpen (txcSaverImpl_ conf') (formatFilePath conf' "events")
  sides  <- saverOpen (txcSaverImpl_ conf') (formatFilePath conf' "side-effects")
  let conf'' = conf' { txcSaverImpl_ = nsaver }
  writeChan sideQueue $ comp sides conf''
  return conf''

doSaveCheckpoint :: Serialize a => TxRun a -> TxConfig -> a -> IO ()
doSaveCheckpoint (TxRun _ _ tisrc _) conf st = do
  logMC NOTICE "CHECKPOINT side-effects ok"
  csaver <- saverOpen (txcSaverImpl_ conf) (formatFilePath conf "checkpoint")
  encodedSt <- encodeFPS st
  saverAdd csaver encodedSt (return ())
  saverClose csaver
  logMC NOTICE "CHECKPOINT writing checkpoint file ok"
  tx <- newTxId tisrc
  logMC NOTICE ("CHECKPOINT txid "++show tx)
  let cs = [("version","2"),("txId",show tx),("txcFileNum",show $ txcFileNum_ conf)]
  saverAtomicReplace (txcSaverImpl_ conf) "current" $ show cs
  doCleanup conf
  logMC NOTICE "CHECKPOINT: end"

doCleanup conf =
  when (txcFileNum_ conf > 1) $ fmap (const()) $ do
    let conf' = conf { txcFileNum_ = txcFileNum_ conf - 2 }
    saverArchive (txcSaverImpl_ conf) (formatFilePath conf' "checkpoint")
    saverArchive (txcSaverImpl_ conf) (formatFilePath conf' "side-effects")
    saverArchive (txcSaverImpl_ conf) (formatFilePath conf' "events")
