/* 
 * File: gmlplaylist.c
 *
 * Medialib search and playlist editor
 * Copyright (c) 2005 Johannes Heimansberg
 *
 * requires GTK+ 2.6 or better and XMMS2 DR2 or better
 *
 * Released under the GNU General Public License v2
 */

#include <stdlib.h>
#include <string.h>
#include <gtk/gtk.h>
#include <gtk/gtksignal.h>
#include <gtk/gtkdialog.h>
#include <gdk/gdkkeysyms.h>
#include "gmedialib.h"
#include "gtrackinfo/trackinfo.h"
#include "gtrackinfo/gtrackinfo.h"
#include "addrnd.xpm"
#include "shuffle.xpm"
#include "rank0.xpm"
#include "rank1.xpm"
#include "rank2.xpm"
#include "rank3.xpm"
#include "rank4.xpm"

extern xmmsc_connection_t *connection; /* available from gxmms2.c */

static void update_pl_entry(xmmsc_result_t *res, void *gmlptr, GtkTreeIter *iter)
{
	gint32              id;
	GMedialib           *gml = (GMedialib *)gmlptr;
	gchar               *url, *filename = "", *artist_tmp = "", *title_tmp = "";
	gchar               *album, *artist = NULL, *title = NULL, *channel = NULL;
	gchar               *prefix = "";
	gchar               album_with_tracknr[256];
	gint                tracknr = -1, rank = -1;
	gchar               *rankstr = NULL;

	xmmsc_result_get_dict_entry_str  (res,   "artist",  &artist_tmp);
	xmmsc_result_get_dict_entry_str  (res,   "title",   &title_tmp);
	xmmsc_result_get_dict_entry_str  (res,   "album",   &album);
	xmmsc_result_get_dict_entry_int32(res,   "id",      &id);
	xmmsc_result_get_dict_entry_str  (res,   "url",     &url);
	xmmsc_result_get_dict_entry_int32(res,   "tracknr", &tracknr);
	xmmsc_result_get_dict_entry_str  (res,   "rank",    &rankstr);
	if (xmmsc_result_get_dict_entry_str(res, "channel", &channel)) {
		prefix = "[Stream] ";
		album = channel;
	} else if (album == NULL)
		album = "[Unknown Album]";

	if (rankstr != NULL) rank = atoi(rankstr);

	if (tracknr > 0)
		snprintf(album_with_tracknr, 256, "%s [%03d]", album, tracknr);
	else
		snprintf(album_with_tracknr, 256, "%s", album);

	if (url)
		filename = g_path_get_basename(url);
	else
		filename = "[Unknown]";

	if (artist_tmp == NULL && channel == NULL) {
		artist = g_malloc(sizeof(gchar) * strlen(filename) + 1);
		g_strlcpy(artist, filename, sizeof(gchar) * strlen(filename) + 1);
	} else if (artist_tmp != NULL) {
		artist = g_malloc(sizeof(gchar) * (strlen(artist_tmp) + strlen(prefix) + 1));
		g_snprintf(artist, strlen(artist_tmp) + strlen(prefix) + 1, 
		           "%s%s", prefix, artist_tmp);
	} else if (channel != NULL) {
		artist = g_malloc(sizeof(gchar) * (strlen(prefix) + 1));
		g_snprintf(artist, strlen(prefix) + 1, 
		           "%s", prefix);
	}
	if (title_tmp == NULL) {			
		title = g_malloc(sizeof(gchar) * strlen(filename) + 1);
		g_strlcpy(title, filename, sizeof(gchar) * strlen(filename) + 1);
	} else {
		title = g_malloc(sizeof(gchar) * strlen(title_tmp) + 1);
		g_strlcpy(title, title_tmp, sizeof(gchar) * strlen(title_tmp) + 1);		
	}

	if (gml->playlist.ls != NULL) {
		gtk_list_store_set(gml->playlist.ls, iter, 
		                   1, id,
		                   2, artist, 3, title,
		                   4, album_with_tracknr, -1);
		if (rank >= 0 && rank <= 4)
			gtk_list_store_set(gml->playlist.ls, iter, 
			                   5, gml->playlist.icon_rank[rank], -1);
	}
	g_free(artist);
	g_free(title);

	if (url)
		g_free(filename);

	xmmsc_result_unref(res);
}

void gml_playlist_n_add_track_with_info(xmmsc_result_t *res2, void *gmlptr)
{
	guint               pos;
	gint32              id;
	static GtkTreeIter  iter;
	GMedialib           *gml = (GMedialib *)gmlptr;

	pos = gtk_tree_model_iter_n_children(GTK_TREE_MODEL(gml->playlist.ls), NULL);

	xmmsc_result_get_dict_entry_int32(res2,   "id",      &id);

	if (gml->playlist.ls != NULL) {
		gtk_list_store_append(gml->playlist.ls, &iter);
		gtk_list_store_set(gml->playlist.ls, &iter, 
		                   0, pos,    1, id, -1);
	}

	update_pl_entry(res2, gml, &iter);
}

static void n_pl_playlist_list(xmmsc_result_t *res, void *userdata)
{
	GMedialib                  *gml = (GMedialib *)userdata;
	static char                *pref[] = { "client/generic", "server", NULL };

	for (; xmmsc_result_list_valid(res); xmmsc_result_list_next(res)) {
		xmmsc_result_t *res2;
		guint           i;

		if (!xmmsc_result_get_uint(res, &i))
			printf("Broken result\n");

		res2 = xmmsc_medialib_get_info(connection, i);
		xmmsc_result_source_preference_set(res2, pref);
		xmmsc_result_notifier_set(res2, gml_playlist_n_add_track_with_info, gml);
		xmmsc_result_unref(res2);
	}
	xmmsc_result_unref(res);
	gml_set_statusbar_text(gml, "Ready.");
	gtk_widget_set_sensitive(gml->playlist.button_up,   FALSE);
	gtk_widget_set_sensitive(gml->playlist.button_down, FALSE);
}

void gml_pl_refresh_playlist(GMedialib *gml)
{
	xmmsc_result_t *res;

	gml_set_statusbar_text(gml, "Loading playlist...");

	if (gml->playlist.ls != NULL) {
		gtk_list_store_clear(gml->playlist.ls);
	}

	res = xmmsc_playlist_list(connection);
	xmmsc_result_notifier_set(res, n_pl_playlist_list, gml);
	xmmsc_result_unref(res);
}

static void n_pl_update_current_pos(xmmsc_result_t *res, void *arg)
{
	guint pos = 0;
	GMedialib *gml = (GMedialib *)arg;
	xmmsc_result_get_uint(res, &pos);
	gml->playlist.current_pos = pos;
	gml_playlist_update_entry(gml, gml->playlist.current_pos);
	gtk_widget_queue_draw(gml->playlist.list);
	xmmsc_result_unref(res);
}

static void bc_handle_current_id(xmmsc_result_t *res, void *userdata)
{
	xmmsc_result_t *res2;
	res2 = xmmsc_playlist_current_pos(connection);
	xmmsc_result_notifier_set(res2, n_pl_update_current_pos, userdata);
	xmmsc_result_unref(res2);
}

static void bc_handle_medialib_entry_changed(xmmsc_result_t *res, void *gmlptr)
{
	GMedialib *gml = GMEDIALIB(gmlptr);

	gml_playlist_update_entry(gml, gml->playlist.current_pos);
}

static void update_current_pl_pos(GMedialib *gml)
{
	xmmsc_result_t *res;

	res = xmmsc_playlist_current_pos(connection);
	xmmsc_result_notifier_set(res, n_pl_update_current_pos, gml);
	xmmsc_result_unref(res);
}

static void pl_cell_data_function(GtkTreeViewColumn *col,
                                  GtkCellRenderer   *renderer,
                                  GtkTreeModel      *model,
                                  GtkTreeIter       *iter,
                                  gpointer           user_data)
{
	GMedialib *gml = GMEDIALIB(user_data);
	guint      pos = 0;
	guint      cur_pos = (guint)gml->playlist.current_pos;

	gtk_tree_model_get(model, iter, 0, &pos, -1);

	if (pos == cur_pos)
		g_object_set(renderer, "weight", 700, "weight-set", TRUE, NULL);
	else
		g_object_set(renderer, "weight-set", FALSE, NULL);
}

static void n_playback_start(xmmsc_result_t *res, void *userdata)
{
	GMedialib *gml = (GMedialib *)userdata;
	if (xmmsc_result_iserror(res)) {
		gchar buffer[128];
		snprintf(buffer, 128, "ERROR: Couldn't jump to that song: %s", 
		         xmmsc_result_get_error(res));
		gml_set_statusbar_text(gml, buffer);
	} else {
		xmmsc_result_unref(res);
		gml_set_statusbar_text(gml, "Ready.");
	}
}

static void n_tickle(xmmsc_result_t *res, void *userdata)
{
	GMedialib *gml = (GMedialib *)userdata;
	if (xmmsc_result_iserror(res)) {
		gchar buffer[128];
		snprintf(buffer, 128, "ERROR: Couldn't jump to that song: %s", 
		         xmmsc_result_get_error(res));
		gml_set_statusbar_text(gml, buffer);
	} else {
		xmmsc_result_t *res2;
		res2 = xmmsc_playback_start(connection);
		xmmsc_result_notifier_set(res2, n_playback_start, userdata);
		xmmsc_result_unref(res2);
		xmmsc_result_unref(res);
	}
}

static void n_jump_to_track(xmmsc_result_t *res, void *userdata)
{
	GMedialib *gml = (GMedialib *)userdata;
	if (xmmsc_result_iserror(res)) {
		gchar buffer[128];
		snprintf(buffer, 128, "ERROR: Couldn't jump to that song: %s", 
		         xmmsc_result_get_error(res));
		gml_set_statusbar_text(gml, buffer);
	} else {
		xmmsc_result_t *res2;
		res2 = xmmsc_playback_tickle(connection);
		xmmsc_result_notifier_set(res2, n_tickle, userdata);
		xmmsc_result_unref(res2);
		xmmsc_result_unref(res);
	}
}

static void cb_pl_on_tree_view_row_activated(GtkTreeView *view, 
                                             GtkTreePath *path,
                                             GtkTreeViewColumn *col,
                                             gpointer userdata)
{
	GtkTreeIter     iter;
	GtkTreeModel   *model;
	xmmsc_result_t *res;

	model = gtk_tree_view_get_model(view);

	if (gtk_tree_model_get_iter(model, &iter, path)) {
		guint pos;

		gtk_tree_model_get(model, &iter, 0, &pos, -1);

		/* jump to track */
		res = xmmsc_playlist_set_next(connection, pos);
		xmmsc_result_notifier_set(res, n_jump_to_track, userdata);
		xmmsc_result_unref(res);
	}
}

static void cb_pl_on_tree_selection_changed(GtkTreeSelection *sel, 
                                            gpointer userdata)
{
	GMedialib   *gml = (GMedialib *)userdata;
	guint       cnt = 0;
	
	cnt = gtk_tree_selection_count_selected_rows(sel);

	if (cnt == 1) {
		gtk_widget_set_sensitive(gml->playlist.button_up,   TRUE);
		gtk_widget_set_sensitive(gml->playlist.button_down, TRUE);
	} else {
		//gtk_widget_set_sensitive(gml->playlist.button_up,   FALSE);
		//gtk_widget_set_sensitive(gml->playlist.button_down, FALSE);
	}
}

enum { UP = -1, DOWN = 1 };

struct params_2
{
	GMedialib   *gml;
	guint        direction;
	GtkTreeIter  selected_row;
};

static void n_playlist_move(xmmsc_result_t *res, void *arguments)
{
	struct params_2  *p = (struct params_2 *)arguments;
	GtkTreeSelection *sel = 
		gtk_tree_view_get_selection(GTK_TREE_VIEW(p->gml->playlist.list));
	GtkTreeModel     *model = GTK_TREE_MODEL(p->gml->playlist.ls);
	GList            *list = gtk_tree_selection_get_selected_rows(sel, &model);
	GtkTreePath      *path = list->data;

	GtkTreeIter     selected_row, other_row;

	guint           pos, id, other_pos, other_id;
	gchar           *artist = NULL,       *track = NULL, *album = NULL;
	gchar           *other_artist = NULL, *other_track = NULL;
	gchar           *other_album = NULL;
	GdkPixbuf       *icon_rank, *icon_other_rank;

	/*gtk_tree_model_get_iter(GTK_TREE_MODEL(p->gml->playlist.ls),
	                        &selected_row, path);*/
	gtk_tree_model_get(model, &(p->selected_row), 0, &pos,    1, &id, 
	                                              2, &artist, 3, &track, 
	                                              4, &album,  5, &icon_rank, -1);

	/*if (xmmsc_result_iserror(res)) {
		gchar buf[128];
		snprintf(buf, 128, "Unable to move playlist entry: %s",
		        xmmsc_result_get_error(res));
		gml_set_statusbar_text(p->gml, buf);
	} else */ {
		if (p->direction == DOWN)
			gtk_tree_path_next(path);
		else
			gtk_tree_path_prev(path);
		gtk_tree_model_get_iter(GTK_TREE_MODEL(p->gml->playlist.ls),
		                        &other_row, path);
		if (p->direction == DOWN)
			gtk_tree_path_prev(path);
		else
			gtk_tree_path_next(path);
		gtk_tree_model_get_iter(GTK_TREE_MODEL(p->gml->playlist.ls),
		                        &(p->selected_row), path);
		gtk_tree_model_get(model, &other_row, 
		                          0, &other_pos, 
		                          1, &other_id, 
		                          2, &other_artist, 
		                          3, &other_track, 
		                          4, &other_album, 
		                          5, &icon_other_rank,
		                          -1);

printf("Item to move: pos=%d id=%d artist=%s track=%s\n", pos, id, artist, track);
printf("Swap for: pos=%d id=%d artist=%s track=%s\n", other_pos, other_id, other_artist, other_track);

		gtk_list_store_set(p->gml->playlist.ls,
		                   &(p->selected_row),
		                   0, pos,
	                       1, other_id,
	                       2, other_artist,
	                       3, other_track,
	                       4, other_album,
	                       5, icon_other_rank,
	                       -1);

   		gtk_list_store_set(p->gml->playlist.ls,
		                   &other_row,
		                   0, other_pos,
		                   1, id,
		                   2, artist,
		                   3, track,
		                   4, album,
		                   5, icon_rank,
		                   -1);
		                   
		/* if (p->direction == DOWN)
			gtk_tree_path_next(path);
		else
			gtk_tree_path_prev(path); */
		/*gtk_tree_view_set_cursor(GTK_TREE_VIEW(p->gml->playlist.list),
		                         path,
		                         NULL,
		                         FALSE);*/
		gtk_widget_grab_focus(GTK_WIDGET(p->gml->playlist.list));
		gtk_tree_view_scroll_to_cell(GTK_TREE_VIEW(p->gml->playlist.list),
		                             path,
		                             NULL,
		                             TRUE,
		                             0.5,
		                             0.5);
		g_free(artist);
		g_free(track);
		gml_set_statusbar_text(p->gml, "Ready.");
	}

	p->gml->playlist.move_finished = TRUE;

	g_list_foreach(list, (GFunc)gtk_tree_path_free, NULL);
	g_list_free(list);
	update_current_pl_pos(p->gml);
	g_free(p);
	//xmmsc_result_unref(res);
}

static void pl_move_selected_item(GtkTreeModel *model,
                                  GtkTreePath  *path,
                                  GtkTreeIter  *iter,
                                  gpointer     userdata)
{
	xmmsc_result_t  *res;
	guint            pos;
	struct params_2 *p = malloc(sizeof(struct params_2));

	p->gml = ((struct params_2 *)userdata)->gml;
	p->direction = ((struct params_2 *)userdata)->direction;
	p->selected_row = *iter;

	gtk_tree_model_get(model, iter, 0, &pos, -1);

	printf("Move Pos %d %d.\n", pos, p->direction);
	
	res = xmmsc_playlist_move(connection, pos, pos + p->direction);
	//xmmsc_result_notifier_set(res, n_playlist_move, p);
	xmmsc_result_unref(res);
	n_playlist_move(NULL, p);
	//g_free((struct params_2 *)userdata);
}

static void pl_move_item(GMedialib *gml, guint direction)
{
	xmmsc_result_t  *res;
	struct params_2 *p = g_malloc(sizeof(struct params_2));
	GtkTreeSelection *sel = 
		gtk_tree_view_get_selection(GTK_TREE_VIEW(gml->playlist.list));
	GtkTreeModel    *model = GTK_TREE_MODEL(gml->playlist.ls);
	GtkTreeIter     selected_row;
	GList           *list = gtk_tree_selection_get_selected_rows(sel, &model);
	GtkTreePath     *path = list->data;
	guint           cnt = gtk_tree_selection_count_selected_rows(sel);
	guint           pos;


	/* if (cnt == 1 && gml->playlist.move_finished) { /* exactly one row selected */
	/*	gml->playlist.move_finished = FALSE;
		p->gml = gml;
		p->direction = direction;

		gtk_tree_model_get_iter(GTK_TREE_MODEL(p->gml->playlist.ls),
		                        &selected_row, path);
		gtk_tree_model_get(model, &selected_row, 0, &pos, -1);

		g_list_foreach(list, (GFunc)gtk_tree_path_free, NULL);
		g_list_free(list);

		res = xmmsc_playlist_move(connection, pos, pos + direction);
		xmmsc_result_notifier_set(res, n_playlist_move, p);
		xmmsc_result_unref(res);
	} else */
	if (gml->playlist.move_finished) {
		printf("Move multiple items!\n");
		GtkTreeSelection	*sel;
		//GList               *element;

		p->gml = gml;
		p->direction = direction;

		sel = gtk_tree_view_get_selection(GTK_TREE_VIEW(gml->playlist.list));
		gtk_tree_selection_selected_foreach(sel, pl_move_selected_item, p);

		/*selection_list = g_list_sort(selection_list, comp_func);
		element = sel;
		while (element != NULL) {
			xmmsc_result_t *res;
			guint           pos = *(guint *)(element->data);

			res = xmmsc_playlist_remove(connection, pos);
			xmmsc_result_notifier_set(res, n_playlist_remove, NULL);
			xmmsc_result_unref(res);

			g_free(element->data);
			element = element->next;
		}
		g_list_free(sel);
		sel = NULL;*/
	}
}

static void cb_pl_button_down_pressed(GtkWidget *widget, gpointer userdata)
{
	pl_move_item((GMedialib *)userdata, DOWN);
}

static void cb_pl_button_up_pressed(GtkWidget *widget, gpointer userdata)
{
	pl_move_item((GMedialib *)userdata, UP);
}

static void n_shuffle_or_sort_ready(xmmsc_result_t *res, void *userdata)
{
	GMedialib *gml = (GMedialib *)userdata;
	update_current_pl_pos(gml);
	gml_pl_refresh_playlist(gml);
	xmmsc_result_unref(res);
}

static void cb_pl_shuffle_button_pressed(GtkWidget *widget, gpointer userdata)
{
	xmmsc_result_t *res;
	
	res = xmmsc_playlist_shuffle(connection);
	xmmsc_result_notifier_set(res, n_shuffle_or_sort_ready, userdata);
	xmmsc_result_unref(res);
}

static void cb_pl_sort_button_pressed(GtkWidget *widget, gpointer userdata)
{
	xmmsc_result_t *res;
	
	res = xmmsc_playlist_sort(connection, "artist");
	xmmsc_result_notifier_set(res, n_shuffle_or_sort_ready, userdata);
	xmmsc_result_unref(res);
}

static void n_query_finished(xmmsc_result_t *res, void *arguments)
{
	GMedialib      *gml = (GMedialib *)arguments;
	GtkTreeIter    iter;
	xmmsc_result_t *res2;
	guint          pl_length = 0;

	if (xmmsc_result_iserror(res)) {
		printf("%s\n", xmmsc_result_get_error(res));
		return;
	}

	for (; xmmsc_result_list_valid(res); xmmsc_result_list_next(res)) {
		gint32 id;
		xmmsc_result_get_dict_entry_int32(res, "id",      &id);

		/* get playlist length */
		pl_length = 
			gtk_tree_model_iter_n_children(GTK_TREE_MODEL(gml->playlist.ls),
			                               NULL);

		/* add id to playlist */
		res2 = xmmsc_playlist_add_id(connection, id);
		xmmsc_result_unref(res2);
		
		/* add item to gtk list */
		if (gml->playlist.ls != NULL) {
			gtk_list_store_append(gml->playlist.ls, &iter);
			gtk_list_store_set(gml->playlist.ls, &iter, 
			                   0, pl_length, 1, id, -1);
		}
		gml_playlist_update_entry(gml, pl_length);
	}
	xmmsc_result_unref(res);
	gml_set_statusbar_text(gml, "Ready.");
}

struct params_entry_update {
	GMedialib   *gml;
	GtkTreeIter  iter;
};

static void n_update_playlist_entry(xmmsc_result_t *res, void *arg)
{
	struct params_entry_update *p = (struct params_entry_update *)arg;
	static GtkTreeIter          iter;

	iter = p->iter;

	if (xmmsc_result_iserror(res))
		printf("%s\n", xmmsc_result_get_error(res));
	else
		update_pl_entry(res, p->gml, &iter);

	g_free(p);
}

void gml_playlist_update_entry(GMedialib *gml, guint pos)
{
	struct params_entry_update *p = g_malloc(sizeof(struct params_entry_update));
	GtkTreeModel *model = GTK_TREE_MODEL(gml->playlist.ls);
	GtkTreeIter   iter;
	guint         poscnt = 0;
	static char  *pref[] = { "client/generic", "server", NULL };

	p->gml = gml;

	if (gtk_tree_model_get_iter_first(model, &iter)) {
		do {
			gint            id;
			xmmsc_result_t *res;

		    gtk_tree_model_get(model, &iter, 1, &id, -1);
			if (pos == poscnt) {
				p->iter = iter;
				res = xmmsc_medialib_get_info(connection, id);
				xmmsc_result_source_preference_set(res, pref);
				xmmsc_result_notifier_set(res, n_update_playlist_entry, p);
				xmmsc_result_unref(res);
			}
			poscnt++;
		} while (gtk_tree_model_iter_next(model, &iter));
	}
}

static void cb_pl_add_rnd_button_pressed(GtkWidget *widget, gpointer userdata)
{
	xmmsc_result_t *res;
	gchar          *query = "SELECT id FROM Media WHERE key='url' " \
	                        "ORDER BY RANDOM() LIMIT 10";
	GMedialib      *gml = (GMedialib *)userdata;
	static char    *pref[] = { "client/generic", "server", NULL };

	gml_set_statusbar_text(gml, "Adding random songs...");

	res = xmmsc_medialib_select(connection, query);
	xmmsc_result_source_preference_set(res, pref);
	xmmsc_result_notifier_set(res, n_query_finished, userdata);
	xmmsc_result_unref(res);
}

static void cb_pl_reload_button_pressed(GtkWidget *widget, gpointer pt);

static GList *selection_list = NULL;

static void pl_delete_selection(GtkTreeModel *model,
                                GtkTreePath  *path,
                                GtkTreeIter  *iter,
                                gpointer     userdata)
{
	guint pos, *value;

	gtk_tree_model_get(model, iter, 0, &pos, -1);

	value = g_malloc(sizeof(guint));
	*value = pos;
	selection_list = g_list_append(selection_list, (gpointer)value);
	/* gtk_list_store_remove(GTK_LIST_STORE(model), &selected_row); */
}

static gint comp_func(gconstpointer a, gconstpointer b)
{
	return (*(guint *)a > *(guint *)b ? -1 : 1);
}

static void n_playlist_remove(xmmsc_result_t *res, void *arguments)
{
	if (xmmsc_result_iserror(res)) {
		fprintf(stderr, "Couldn't remove item (%s)\n", 
		        xmmsc_result_get_error(res));
	}
	xmmsc_result_unref(res);
}

static void cb_pl_button_delete_pressed(GtkWidget *widget, gpointer userdata)
{
	GMedialib			*gml = (GMedialib *)userdata;
	GtkTreeSelection	*sel;
	GList               *element;

	sel = gtk_tree_view_get_selection(GTK_TREE_VIEW(gml->playlist.list));
	gtk_tree_selection_selected_foreach(sel, pl_delete_selection, NULL);

	selection_list = g_list_sort(selection_list, comp_func);
	element = selection_list;
	while (element != NULL)
	{
		xmmsc_result_t *res;
		guint           pos = *(guint *)(element->data);

		res = xmmsc_playlist_remove(connection, pos);
		xmmsc_result_notifier_set(res, n_playlist_remove, NULL);
		xmmsc_result_unref(res);

		g_free(element->data);
		element = element->next;
	}
	g_list_free(selection_list);
	selection_list = NULL;

	update_current_pl_pos(gml);
	gml_pl_refresh_playlist(gml);
}

static gboolean cb_pl_selection_key_press(GtkWidget   *treeview,
                                          GdkEventKey *event,
                                          gpointer    userdata)
{
	gboolean result = FALSE;

	if (event->type == GDK_KEY_PRESS && 
	    (event->keyval == GDK_Delete || event->keyval == GDK_KP_Delete)) {
		cb_pl_button_delete_pressed(treeview, userdata);
		result = TRUE;
	}
	if (event->type == GDK_KEY_PRESS && event->state == GDK_MOD1_MASK) {
	    switch (event->keyval) {
	    	case GDK_Up:
	    		pl_move_item((GMedialib *)userdata, UP);
	    		result = TRUE;
	    		break;
	    	case GDK_Down:
	    		pl_move_item((GMedialib *)userdata, DOWN);
	    		result = TRUE;
	    		break;
	    }
	}
	return result;
}

static void pl_play_selection(GtkTreeModel  *model,
                              GtkTreePath   *path,
                              GtkTreeIter   *iter,
                              gpointer       userdata)
{
	guint          pos;
	xmmsc_result_t *res;

	gtk_tree_model_get(model, iter, 0, &pos, -1);

	/* jump to track */
	res = xmmsc_playlist_set_next(connection, pos);
	xmmsc_result_notifier_set(res, n_jump_to_track, userdata);
	xmmsc_result_unref(res);
}

static void n_media_lib_get_info(xmmsc_result_t *res, void *arg)
{
	trackinfo       track;
	GtkWidget       *dialog;

	trackinfo_update(res, &track);
	dialog = gtrackinfo_new();
	gtrackinfo_set_info(GTRACKINFO(dialog), &track);
	gtk_widget_show(dialog);
	xmmsc_result_unref(res);
}

static void pl_info_about_selection(GtkTreeModel  *model,
                                    GtkTreePath   *path,
                                    GtkTreeIter   *iter,
                                    gpointer       userdata)
{
	guint           id;
	xmmsc_result_t *res;

	gtk_tree_model_get(model, iter, 1, &id, -1);
	res = xmmsc_medialib_get_info(connection, id);
    xmmsc_result_notifier_set(res, n_media_lib_get_info, NULL);
    xmmsc_result_unref(res);
}

struct params
{
	GMedialib   *gml;
	GtkTreeIter iter;
};

static void n_playlist_current_pos(xmmsc_result_t *res, void *arguments)
{
	struct params *pm = (struct params *)arguments;
	guint         pos = 0, p = 0;
	GtkTreePath   *path;

	xmmsc_result_get_uint(res, &pos);
	xmmsc_result_unref(res);
	pm->gml->playlist.current_pos = pos;

	gtk_tree_model_get(GTK_TREE_MODEL(pm->gml->playlist.ls), 
	                   &(pm->iter), 0, &p, -1);
	while (p != pos && 
	       gtk_tree_model_iter_next(GTK_TREE_MODEL(pm->gml->playlist.ls), 
	                                &(pm->iter))) {
		gtk_tree_model_get(GTK_TREE_MODEL(pm->gml->playlist.ls), 
		                   &(pm->iter), 0, &p, -1);
	}
	path = gtk_tree_model_get_path(GTK_TREE_MODEL(pm->gml->playlist.ls), 
	                               &(pm->iter));
	gtk_tree_view_set_cursor(GTK_TREE_VIEW(pm->gml->playlist.list),
	                         path,
	                         NULL,
	                         FALSE);
	gtk_widget_grab_focus(GTK_WIDGET(pm->gml->playlist.list));
	gtk_tree_view_scroll_to_cell(GTK_TREE_VIEW(pm->gml->playlist.list),
	                             path,
	                             NULL,
	                             TRUE,
	                             0.5,
	                             0.5);
	gtk_tree_path_free(path);
	g_free(pm);
}

static void cb_pl_button_track_pressed(GtkWidget *widget, gpointer userdata)
{
	xmmsc_result_t   *res;
	struct params    *p;

	p = g_malloc(sizeof(struct params));
	p->gml = (GMedialib *)userdata;
	
	if (gtk_tree_model_get_iter_first(GTK_TREE_MODEL(p->gml->playlist.ls),
	                                  &(p->iter))) {
		res = xmmsc_playlist_current_pos(connection);
		xmmsc_result_notifier_set(res, n_playlist_current_pos, p);
		xmmsc_result_unref(res);
	}
}

static void cb_pl_save_pl_button_pressed(GtkWidget *widget, gpointer userdata);

static void n_playlist_clear(xmmsc_result_t *res, void *userdata)
{
	GMedialib *gml = (GMedialib *)userdata;
	if (gml->playlist.ls != NULL)
		gtk_list_store_clear(gml->playlist.ls);
	gml_set_statusbar_text(gml, "Ready.");
}

static void cb_pl_clear_button_pressed(GtkWidget *widget, gpointer userdata)
{
	xmmsc_result_t *res;
	GMedialib      *gml = (GMedialib *)userdata;
	gint           result;
	GtkWidget      *dialog = gtk_message_dialog_new (GTK_WINDOW(gml),
                                  GTK_DIALOG_DESTROY_WITH_PARENT,
                                  GTK_MESSAGE_QUESTION,
                                  GTK_BUTTONS_NONE,
                                  "Do you want to save the current playlist?");
	gtk_dialog_add_buttons(GTK_DIALOG(dialog), 
	                       GTK_STOCK_YES,    GTK_RESPONSE_YES,
	                       GTK_STOCK_NO,     GTK_RESPONSE_NO,
	                       GTK_STOCK_CANCEL, GTK_RESPONSE_REJECT,
	                       NULL);

	result = gtk_dialog_run(GTK_DIALOG(dialog));
	if (result == GTK_RESPONSE_YES) {
		/* save the playlist */
		gtk_widget_destroy(dialog);
		cb_pl_save_pl_button_pressed(widget, gml);
	} else {
		gtk_widget_destroy(dialog);
	}
	if (result != GTK_RESPONSE_REJECT) {
		res = xmmsc_playlist_clear(connection);
		xmmsc_result_notifier_set(res, n_playlist_clear, userdata);
		xmmsc_result_unref(res);
	}
}

static void n_playlist_save(xmmsc_result_t *res, void *userdata)
{
	GMedialib *gml = (GMedialib *)userdata;
	if (xmmsc_result_iserror(res)) {
		gml_set_statusbar_text(gml, (gchar *)xmmsc_result_get_error(res));
	} else {
		gml_playlists_update_playlists(gml);
		gml_set_statusbar_text(gml, "Playlist saved successfully.");
	}
	xmmsc_result_unref(res);
}

static void cb_pl_save_pl_button_pressed(GtkWidget *widget, gpointer userdata)
{
	xmmsc_result_t *res;
	gint           result;
	GtkWidget      *dialog = gtk_dialog_new_with_buttons("Save playlist...",
	                                                     GTK_WINDOW(userdata),
	                                                     GTK_DIALOG_MODAL | 
	                                                     GTK_DIALOG_DESTROY_WITH_PARENT,
	                                                     GTK_STOCK_OK,
	                                                     GTK_RESPONSE_ACCEPT,
	                                                     GTK_STOCK_CANCEL,
	                                                     GTK_RESPONSE_REJECT,
	                                                     NULL);
	GtkWidget      *label = gtk_label_new("Save as:");
	GtkWidget      *entry = gtk_entry_new();
	gtk_widget_set_size_request(GTK_WIDGET(dialog), 300, 100);
	gtk_window_set_resizable(GTK_WINDOW(dialog), FALSE);
	gtk_container_add(GTK_CONTAINER(GTK_DIALOG(dialog)->vbox), label);
	gtk_container_add(GTK_CONTAINER(GTK_DIALOG(dialog)->vbox), entry);
	gtk_widget_show_all(dialog);

	result = gtk_dialog_run(GTK_DIALOG(dialog));
	if (result == GTK_RESPONSE_ACCEPT) {
		/* save the playlist */
		res = xmmsc_medialib_playlist_save_current(connection, 
		                                           gtk_entry_get_text(
		                                           GTK_ENTRY(entry)));
		xmmsc_result_notifier_set(res, n_playlist_save, userdata);
		xmmsc_result_unref(res);
	}
	gtk_widget_destroy(dialog);
}

static void cb_pl_reload_button_pressed(GtkWidget *widget, gpointer pt)
{
	GMedialib *gml = (GMedialib *)pt;
	gml_pl_refresh_playlist(gml);
}

static void search_for_artist_foreach_func(GtkTreeModel  *model,
                                           GtkTreePath   *path,
                                           GtkTreeIter   *selected_row,
                                           gpointer       pt)
{
	GMedialib       *gml = (GMedialib *)pt;
	gchar *artist;

	gtk_tree_model_get(model, selected_row, 2, &artist, -1);

	gml_search_do_search(gml, SEARCH_ARTIST, artist);
	gtk_notebook_set_current_page(GTK_NOTEBOOK(gml->notebook), 0);
	g_free(artist);
}

static void cb_pl_popup_search_artist_pressed(GtkWidget *menuitem,
                                              gpointer  userdata)
{
	GMedialib        *gml = (GMedialib *)userdata;
	GtkTreeSelection *sel;

	sel = gtk_tree_view_get_selection(GTK_TREE_VIEW(gml->playlist.list));
    gtk_tree_selection_selected_foreach(sel, search_for_artist_foreach_func,
                                        (gpointer)gml);
}

static void search_for_album_foreach_func(GtkTreeModel  *model,
                                          GtkTreePath   *path,
                                          GtkTreeIter   *selected_row,
                                          gpointer       pt)
{
	GMedialib *gml = (GMedialib *)pt;
	gchar     *album;

	gtk_tree_model_get(model, selected_row, 4, &album, -1);

	/* remove tracknr [xxx] from album name: */
	if (album[strlen(album)-1] == ']' &&
	    album[strlen(album)-5] == '[')
		album[strlen(album)-6] = '\0';

	gml_search_do_search(gml, SEARCH_ALBUM, album);
	gtk_notebook_set_current_page(GTK_NOTEBOOK(gml->notebook), 0);
	g_free(album);
}

static void cb_pl_popup_search_album_pressed(GtkWidget *menuitem,
                                              gpointer  userdata)
{
	GMedialib        *gml = (GMedialib *)userdata;
	GtkTreeSelection *sel;

	sel = gtk_tree_view_get_selection(GTK_TREE_VIEW(gml->playlist.list));
    gtk_tree_selection_selected_foreach(sel, search_for_album_foreach_func,
                                        (gpointer)gml);
}

static void cb_pl_popup_play_pressed(GtkWidget *menuitem,
                                     gpointer  userdata)
{
	GMedialib        *gml = (GMedialib *)userdata;
	GtkTreeSelection *sel;
	sel = gtk_tree_view_get_selection(GTK_TREE_VIEW(gml->playlist.list));
    gtk_tree_selection_selected_foreach(sel, pl_play_selection,
                                        (gpointer)gml);
}

static void cb_pl_popup_info_pressed(GtkWidget *menuitem,
                                     gpointer  userdata)
{
	GMedialib        *gml = (GMedialib *)userdata;
	GtkTreeSelection *sel;
	sel = gtk_tree_view_get_selection(GTK_TREE_VIEW(gml->playlist.list));
	gtk_tree_selection_selected_foreach(sel, pl_info_about_selection, NULL);
}

struct params_3
{
	GMedialib *gml;
	gint       rank;
};

static void pl_rate_selected_tracks(GtkTreeModel  *model,
                                    GtkTreePath   *path,
                                    GtkTreeIter   *iter,
                                    gpointer       userdata)
{
	guint			id;
	xmmsc_result_t *res;
	GMedialib      *gml = ((struct params_3 *)userdata)->gml;
	gint            rank = ((struct params_3 *)userdata)->rank;
	gchar           rankstr[8] = "";

	gtk_tree_model_get(model, iter, 1, &id, -1);

	if (rank >= 0 && rank <= 4) {
		gtk_list_store_set(gml->playlist.ls, iter, 
		                   5, gml->playlist.icon_rank[rank], -1);
		snprintf(rankstr, 7, "%d", rank);
		res = xmmsc_medialib_entry_property_set_with_source(connection, id, 
		                                                    "client/generic",
		                                                    "rank", rankstr);
		xmmsc_result_unref(res);
	}
}

static void cb_pl_popup_rank_track_4(GtkWidget *menuitem,
                                     gpointer  userdata)
{
	GMedialib        *gml = (GMedialib *)userdata;
	GtkTreeSelection *sel;
	struct params_3   params;

	params.gml = gml;
	params.rank = 4;

	sel = gtk_tree_view_get_selection(GTK_TREE_VIEW(gml->playlist.list));
	gtk_tree_selection_selected_foreach(sel, pl_rate_selected_tracks, &params);
}

static void cb_pl_popup_rank_track_3(GtkWidget *menuitem,
                                     gpointer  userdata)
{
	GMedialib        *gml = (GMedialib *)userdata;
	GtkTreeSelection *sel;
	struct params_3   params;

	params.gml = gml;
	params.rank = 3;

	sel = gtk_tree_view_get_selection(GTK_TREE_VIEW(gml->playlist.list));
	gtk_tree_selection_selected_foreach(sel, pl_rate_selected_tracks, &params);
}

static void cb_pl_popup_rank_track_2(GtkWidget *menuitem,
                                     gpointer  userdata)
{
	GMedialib        *gml = (GMedialib *)userdata;
	GtkTreeSelection *sel;
	struct params_3   params;

	params.gml = gml;
	params.rank = 2;

	sel = gtk_tree_view_get_selection(GTK_TREE_VIEW(gml->playlist.list));
	gtk_tree_selection_selected_foreach(sel, pl_rate_selected_tracks, &params);
}

static void cb_pl_popup_rank_track_1(GtkWidget *menuitem,
                                     gpointer  userdata)
{
	GMedialib        *gml = (GMedialib *)userdata;
	GtkTreeSelection *sel;
	struct params_3   params;

	params.gml = gml;
	params.rank = 1;

	sel = gtk_tree_view_get_selection(GTK_TREE_VIEW(gml->playlist.list));
	gtk_tree_selection_selected_foreach(sel, pl_rate_selected_tracks, &params);
}

static void cb_pl_popup_rank_track_0(GtkWidget *menuitem,
                                     gpointer  userdata)
{
	GMedialib        *gml = (GMedialib *)userdata;
	GtkTreeSelection *sel;
	struct params_3   params;

	params.gml = gml;
	params.rank = 0;

	sel = gtk_tree_view_get_selection(GTK_TREE_VIEW(gml->playlist.list));
	gtk_tree_selection_selected_foreach(sel, pl_rate_selected_tracks, &params);
}

static void cb_column_pos_visible(GtkWidget *menuitem,
                                  gpointer   userdata)
{
	GMedialib *gml = (GMedialib *)userdata;
	gml_pl_set_column_pos_visible(gml, !gml_pl_get_column_pos_visible(gml));
}

static void cb_column_id_visible(GtkWidget *menuitem,
                                 gpointer   userdata)
{
	GMedialib *gml = (GMedialib *)userdata;
	gml_pl_set_column_id_visible(gml, !gml_pl_get_column_id_visible(gml));
}

static void cb_column_artist_visible(GtkWidget *menuitem,
                                     gpointer   userdata)
{
	GMedialib *gml = (GMedialib *)userdata;
	gml_pl_set_column_artist_visible(gml, !gml_pl_get_column_artist_visible(gml));
}

static void cb_column_track_visible(GtkWidget *menuitem,
                                    gpointer   userdata)
{
	GMedialib *gml = (GMedialib *)userdata;
	gml_pl_set_column_track_visible(gml, !gml_pl_get_column_track_visible(gml));
}

static void cb_column_album_visible(GtkWidget *menuitem,
                                    gpointer   userdata)
{
	GMedialib *gml = (GMedialib *)userdata;
	gml_pl_set_column_album_visible(gml, !gml_pl_get_column_album_visible(gml));
}

static void cb_column_ranking_visible(GtkWidget *menuitem,
                                      gpointer   userdata)
{
	GMedialib *gml = (GMedialib *)userdata;
	gml_pl_set_column_ranking_visible(gml, !gml_pl_get_column_ranking_visible(gml));
}

static void pl_view_popup_menu(GtkWidget      *treeview,
                               GdkEventButton *event,
                               gpointer       userdata)
{
	GtkWidget        *menu, *menu_rank, *menu_columns, *menuitem;
	GtkTreeSelection *selection;
	GMedialib        *gml = (GMedialib *)userdata;
	gboolean          multiple = FALSE, none = FALSE;
	int               i;

	struct submenu
	{
		gchar    *name;
		gboolean (*get_fn)();
		void     (*callback_fn)();
	};

	struct submenu sm_columns[] = 
		{ {"#", gml_pl_get_column_pos_visible, cb_column_pos_visible},
		  {"ID", gml_pl_get_column_id_visible, cb_column_id_visible},
		  {"Artist", gml_pl_get_column_artist_visible, cb_column_artist_visible},
		  {"Track", gml_pl_get_column_track_visible, cb_column_track_visible},
		  {"Album", gml_pl_get_column_album_visible, cb_column_album_visible},
		  {"Ranking", gml_pl_get_column_ranking_visible, cb_column_ranking_visible},
		  { NULL, NULL, NULL } };

	struct submenu sm_rank[] =
		{ {"4 (Best)",  NULL, cb_pl_popup_rank_track_4},
		  {"3",         NULL, cb_pl_popup_rank_track_3},
		  {"2",         NULL, cb_pl_popup_rank_track_2},
		  {"1",         NULL, cb_pl_popup_rank_track_1},
		  {"0 (Worst)", NULL, cb_pl_popup_rank_track_0},
		  { NULL, NULL, NULL } };

	selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(gml->playlist.list));
	if (gtk_tree_selection_count_selected_rows(selection) > 1)
		multiple = TRUE;
	if (gtk_tree_selection_count_selected_rows(selection) == 0)
		none = TRUE;

	/* Submenu: Track ranking */
	menu_rank = gtk_menu_new();

	i = 0;
	while (sm_rank[i].name != NULL) {
		menuitem = gtk_menu_item_new_with_mnemonic(sm_rank[i].name);
		if (none) gtk_widget_set_sensitive(menuitem, FALSE);
		gtk_menu_shell_append(GTK_MENU_SHELL(menu_rank), menuitem);
		g_signal_connect(menuitem, "activate",
		                 G_CALLBACK(sm_rank[i].callback_fn), gml);
		i++;
	}

	/* Submenu: Visible columns */
	menu_columns = gtk_menu_new();

	i = 0;
	while (sm_columns[i].name != NULL) {
		menuitem = gtk_check_menu_item_new_with_mnemonic(sm_columns[i].name);
		gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(menuitem),
		                               sm_columns[i].get_fn(gml));
		gtk_menu_shell_append(GTK_MENU_SHELL(menu_columns), menuitem);
		g_signal_connect(menuitem, "activate",
		                 G_CALLBACK(sm_columns[i].callback_fn), gml);
		i++;
	}

	/* Context menu */
	menu = gtk_menu_new();

	menuitem = gtk_image_menu_item_new_from_stock(GTK_STOCK_MEDIA_PLAY, NULL);
	if (multiple || none) gtk_widget_set_sensitive(menuitem, FALSE);

	g_signal_connect(menuitem, "activate",
	                 G_CALLBACK(cb_pl_popup_play_pressed), userdata);

	gtk_menu_shell_append(GTK_MENU_SHELL(menu), menuitem);

	menuitem = gtk_image_menu_item_new_from_stock(GTK_STOCK_DIALOG_INFO, NULL);
	if (multiple || none) gtk_widget_set_sensitive(menuitem, FALSE);

	g_signal_connect(menuitem, "activate",
	                 G_CALLBACK(cb_pl_popup_info_pressed), userdata);

	gtk_menu_shell_append(GTK_MENU_SHELL(menu), menuitem);

	menuitem = gtk_image_menu_item_new_from_stock(GTK_STOCK_REMOVE, NULL);
	if (none) gtk_widget_set_sensitive(menuitem, FALSE);

	g_signal_connect(menuitem, "activate",
	                 G_CALLBACK(cb_pl_button_delete_pressed), userdata);

	gtk_menu_shell_append(GTK_MENU_SHELL(menu), menuitem);

	menuitem = gtk_image_menu_item_new_with_mnemonic("Search for _Artist");
	gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(menuitem), 
	                              gtk_image_new_from_stock(GTK_STOCK_FIND, 
	                                                       GTK_ICON_SIZE_SMALL_TOOLBAR));
	if (multiple || none) gtk_widget_set_sensitive(menuitem, FALSE);

	g_signal_connect(menuitem, "activate",
	                 G_CALLBACK(cb_pl_popup_search_artist_pressed), userdata);

	gtk_menu_shell_append(GTK_MENU_SHELL(menu), menuitem);

	menuitem = gtk_image_menu_item_new_with_mnemonic("Search for Al_bum");
	gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(menuitem), 
	                              gtk_image_new_from_stock(GTK_STOCK_FIND, 
	                                                       GTK_ICON_SIZE_SMALL_TOOLBAR));
	if (multiple || none) gtk_widget_set_sensitive(menuitem, FALSE);

	g_signal_connect(menuitem, "activate",
	                 G_CALLBACK(cb_pl_popup_search_album_pressed), userdata);

	gtk_menu_shell_append(GTK_MENU_SHELL(menu), menuitem);

	menuitem = gtk_menu_item_new_with_mnemonic("_Rate track(s)");
	if (none) gtk_widget_set_sensitive(menuitem, FALSE);
	gtk_menu_item_set_submenu(GTK_MENU_ITEM(menuitem), menu_rank);
	gtk_menu_shell_append(GTK_MENU_SHELL(menu), menuitem);

	menuitem = gtk_separator_menu_item_new();
	gtk_menu_shell_append(GTK_MENU_SHELL(menu), menuitem);

	menuitem = gtk_menu_item_new_with_mnemonic("_Columns");
	gtk_menu_item_set_submenu(GTK_MENU_ITEM(menuitem), menu_columns);
	gtk_menu_shell_append(GTK_MENU_SHELL(menu), menuitem);

	gtk_widget_show_all(menu);

	gtk_menu_popup(GTK_MENU(menu), NULL, NULL, NULL, NULL,
	               (event != NULL) ? event->button : 0,
	               gdk_event_get_time((GdkEvent*)event));
}

static gboolean cb_pl_selection_button_press(GtkWidget      *treeview,
                                             GdkEventButton *event,
                                             gpointer       userdata)
{
	gboolean result = FALSE;
	if (event->type == GDK_BUTTON_PRESS  &&  event->button == 3) {
		pl_view_popup_menu(treeview, event, userdata);
		result = TRUE;
	}
	return result;
}

static gboolean cb_pl_selection_popup_menu(GtkWidget *treeview, 
                                           gpointer  userdata)
{
	pl_view_popup_menu(treeview, NULL, userdata);
	return TRUE;
}


gboolean gml_pl_get_column_pos_visible(GMedialib *gml)
{
	return gml->playlist.column_pos_visible;
}

gboolean gml_pl_get_column_id_visible(GMedialib *gml)
{
	return gml->playlist.column_id_visible;
}

gboolean gml_pl_get_column_artist_visible(GMedialib *gml)
{
	return gml->playlist.column_artist_visible;
}

gboolean gml_pl_get_column_track_visible(GMedialib *gml)
{
	return gml->playlist.column_track_visible;
}

gboolean gml_pl_get_column_album_visible(GMedialib *gml)
{
	return gml->playlist.column_album_visible;
}

gboolean gml_pl_get_column_ranking_visible(GMedialib *gml)
{
	return gml->playlist.column_ranking_visible;
}

void gml_pl_set_column_pos_visible(GMedialib *gml, gboolean visible)
{
	gml->playlist.column_pos_visible = visible;
	gtk_tree_view_column_set_visible(gml->playlist.column_pos, visible);
}

void gml_pl_set_column_id_visible(GMedialib *gml, gboolean visible)
{
	gml->playlist.column_id_visible = visible;
	gtk_tree_view_column_set_visible(gml->playlist.column_id, visible);
}

void gml_pl_set_column_artist_visible(GMedialib *gml, gboolean visible)
{
	gml->playlist.column_artist_visible = visible;
	gtk_tree_view_column_set_visible(gml->playlist.column_artist, visible);
}

void gml_pl_set_column_track_visible(GMedialib *gml, gboolean visible)
{
	gml->playlist.column_track_visible = visible;
	gtk_tree_view_column_set_visible(gml->playlist.column_track, visible);
}

void gml_pl_set_column_album_visible(GMedialib *gml, gboolean visible)
{
	gml->playlist.column_album_visible = visible;
	gtk_tree_view_column_set_visible(gml->playlist.column_album, visible);
}

void gml_pl_set_column_ranking_visible(GMedialib *gml, gboolean visible)
{
	gml->playlist.column_ranking_visible = visible;
	gtk_tree_view_column_set_visible(gml->playlist.column_ranking, visible);
}

void gml_pl_setup_xmms_callbacks(GMedialib *gml)
{
	XMMS_CALLBACK_SET(connection, xmmsc_broadcast_playback_current_id,
 	                  bc_handle_current_id, gml);
	XMMS_CALLBACK_SET(connection, xmmsc_broadcast_medialib_entry_changed,
		              bc_handle_medialib_entry_changed, gml);
}

void gml_create_playlist(GMedialib *gml)
{
	GtkCellRenderer   *renderer_pixbuf;
	GtkTreeSelection  *selection;
	GtkTooltips       *tooltips;
	GdkPixbuf         *icon;

	gml->playlist.frame = gtk_frame_new("Playlist");
	gtk_container_set_border_width(GTK_CONTAINER(gml->playlist.frame), 2);

	/* table playlist */
	gml->playlist.table = gtk_table_new(2, 9, FALSE);
	gtk_container_set_border_width(GTK_CONTAINER(gml->playlist.table), 3);
	gtk_table_set_row_spacings(GTK_TABLE(gml->playlist.table), 4);
	gtk_table_set_col_spacings(GTK_TABLE(gml->playlist.table), 4);
	gtk_container_add(GTK_CONTAINER(gml->playlist.frame), gml->playlist.table);

	/* clear playlist button */
	gml->playlist.button_new_pl = gtk_button_new();
	gml->playlist.image_new = gtk_image_new_from_stock(GTK_STOCK_NEW,
	                                     GTK_ICON_SIZE_LARGE_TOOLBAR);
	gtk_button_set_relief(GTK_BUTTON(gml->playlist.button_new_pl), 
	                      GTK_RELIEF_NONE);
	gtk_container_add(GTK_CONTAINER(gml->playlist.button_new_pl), 
	                  gml->playlist.image_new);
	gtk_table_attach(GTK_TABLE(gml->playlist.table),
		             gml->playlist.button_new_pl,
		             0, 1, 0, 1, GTK_SHRINK, GTK_SHRINK, 0, 0);

	/* save playlist button */
	gml->playlist.button_save_pl = gtk_button_new();
	gml->playlist.image_save = gtk_image_new_from_stock(GTK_STOCK_SAVE,
	                                         GTK_ICON_SIZE_LARGE_TOOLBAR);
	gtk_button_set_relief(GTK_BUTTON(gml->playlist.button_save_pl), 
	                      GTK_RELIEF_NONE);
	gtk_container_add(GTK_CONTAINER(gml->playlist.button_save_pl), 
	                  gml->playlist.image_save);
	gtk_table_attach(GTK_TABLE(gml->playlist.table),
		             gml->playlist.button_save_pl,
		             1, 2, 0, 1, GTK_SHRINK, GTK_SHRINK, 0, 0);


	/* add 10 random songs button */
	gml->playlist.button_add_rnd = gtk_button_new();
	icon = gdk_pixbuf_new_from_xpm_data((const char **)addrnd_xpm);
	gml->playlist.image_add_rnd = gtk_image_new_from_pixbuf(icon);
	gtk_button_set_relief(GTK_BUTTON(gml->playlist.button_add_rnd), 
	                      GTK_RELIEF_NONE);
	gtk_container_add(GTK_CONTAINER(gml->playlist.button_add_rnd), 
	                  gml->playlist.image_add_rnd);
	gtk_table_attach(GTK_TABLE(gml->playlist.table),
		             gml->playlist.button_add_rnd,
		             2, 3, 0, 1, GTK_SHRINK, GTK_SHRINK, 0, 0);

	/* shuffle playlist button */
	gml->playlist.button_shuffle_pl = gtk_button_new();
	icon = gdk_pixbuf_new_from_xpm_data((const char **)shuffle_xpm);
	gml->playlist.image_shuffle_pl = gtk_image_new_from_pixbuf(icon);
	gtk_button_set_relief(GTK_BUTTON(gml->playlist.button_shuffle_pl),
	                      GTK_RELIEF_NONE);
	gtk_container_add(GTK_CONTAINER(gml->playlist.button_shuffle_pl), 
	                  gml->playlist.image_shuffle_pl);
	gtk_table_attach(GTK_TABLE(gml->playlist.table),
		             gml->playlist.button_shuffle_pl,
		             3, 4, 0, 1, GTK_SHRINK, GTK_SHRINK, 0, 0);


	/* sort playlist button */
	gml->playlist.button_sort_pl = gtk_button_new();
	gml->playlist.image_sort_pl = gtk_image_new_from_stock(
	                              	GTK_STOCK_SORT_ASCENDING, 
	                              	GTK_ICON_SIZE_LARGE_TOOLBAR);
	gtk_button_set_relief(GTK_BUTTON(gml->playlist.button_sort_pl),
	                      GTK_RELIEF_NONE);
	gtk_container_add(GTK_CONTAINER(gml->playlist.button_sort_pl), 
	                  gml->playlist.image_sort_pl);
	gtk_table_attach(GTK_TABLE(gml->playlist.table),
		             gml->playlist.button_sort_pl,
		             4, 5, 0, 1, GTK_SHRINK, GTK_SHRINK, 0, 0);

	/* reload playlist button */
	gml->playlist.button_reload_pl = gtk_button_new();
	gml->playlist.image_reload = gtk_image_new_from_stock(GTK_STOCK_REFRESH, 
	                                            GTK_ICON_SIZE_LARGE_TOOLBAR);
	gtk_button_set_relief(GTK_BUTTON(gml->playlist.button_reload_pl),
	                                 GTK_RELIEF_NONE);
	gtk_container_add(GTK_CONTAINER(gml->playlist.button_reload_pl),
	                  gml->playlist.image_reload);
	gtk_table_attach(GTK_TABLE(gml->playlist.table),
		             gml->playlist.button_reload_pl,
		             5, 6, 0, 1, GTK_SHRINK, GTK_SHRINK, 0, 0);

	/* show-current-track button */
	gml->playlist.button_track = gtk_button_new();
	gml->playlist.image_track = gtk_image_new_from_stock(GTK_STOCK_JUMP_TO,
	                                           GTK_ICON_SIZE_LARGE_TOOLBAR);
	gtk_button_set_relief(GTK_BUTTON(gml->playlist.button_track),
	                      GTK_RELIEF_NONE);
	gtk_container_add(GTK_CONTAINER(gml->playlist.button_track),
	                  gml->playlist.image_track);
	gtk_table_attach(GTK_TABLE(gml->playlist.table),
		             gml->playlist.button_track,
		             6, 7, 0, 1, GTK_SHRINK, GTK_SHRINK, 0, 0);
	gtk_widget_set_sensitive(gml->playlist.button_track, TRUE);

	/* up button */
	gml->playlist.button_up = gtk_button_new();
	gml->playlist.image_up = gtk_image_new_from_stock(GTK_STOCK_GO_UP, 
	                                      GTK_ICON_SIZE_LARGE_TOOLBAR);
	gtk_button_set_relief(GTK_BUTTON(gml->playlist.button_up), GTK_RELIEF_NONE);
	gtk_container_add(GTK_CONTAINER(gml->playlist.button_up),
	                  gml->playlist.image_up);
	gtk_table_attach(GTK_TABLE(gml->playlist.table),
		             gml->playlist.button_up,
		             7, 8, 0, 1, GTK_SHRINK, GTK_SHRINK, 0, 0);
	gtk_widget_set_sensitive(gml->playlist.button_up, FALSE);

	/* down button */
	gml->playlist.button_down = gtk_button_new();
	gml->playlist.image_down = gtk_image_new_from_stock(GTK_STOCK_GO_DOWN, 
	                                          GTK_ICON_SIZE_LARGE_TOOLBAR);
	gtk_button_set_relief(GTK_BUTTON(gml->playlist.button_down),
	                      GTK_RELIEF_NONE);
	gtk_container_add(GTK_CONTAINER(gml->playlist.button_down),
	                  gml->playlist.image_down);
	gtk_table_attach(GTK_TABLE(gml->playlist.table),
		             gml->playlist.button_down,
		             8, 9, 0, 1, GTK_SHRINK, GTK_SHRINK, 0, 0);
	gtk_widget_set_sensitive(gml->playlist.button_down, FALSE);

	/* list playlist */
	gml->playlist.ls = gtk_list_store_new(6, G_TYPE_INT, G_TYPE_INT, 
	                                      G_TYPE_STRING, G_TYPE_STRING,
	                                      G_TYPE_STRING, GDK_TYPE_PIXBUF);

	gml->playlist.list = 
		gtk_tree_view_new_with_model(GTK_TREE_MODEL(gml->playlist.ls));
	gtk_tree_view_set_enable_search(GTK_TREE_VIEW(gml->playlist.list), TRUE);

	selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(gml->playlist.list));
    gtk_tree_selection_set_mode(selection, GTK_SELECTION_MULTIPLE);

	gml->playlist.scroll_widget = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(
		GTK_SCROLLED_WINDOW(gml->playlist.scroll_widget),
		GTK_POLICY_AUTOMATIC, GTK_POLICY_ALWAYS);

	gtk_container_add(GTK_CONTAINER(gml->playlist.scroll_widget),
	                  gml->playlist.list);

	gtk_table_attach(GTK_TABLE(gml->playlist.table),
		             gml->playlist.scroll_widget,
		             0, 9, 1, 2,
		             GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, 0, 0);

	gml->playlist.column_pos = gtk_tree_view_column_new_with_attributes("#", 
	                                                         gml->renderer_text,
	                                                         "text", 0, NULL);
	gtk_tree_view_append_column(GTK_TREE_VIEW(gml->playlist.list),
	                            gml->playlist.column_pos);
	gtk_tree_view_column_set_resizable(gml->playlist.column_pos, TRUE);
	gtk_tree_view_column_set_cell_data_func(gml->playlist.column_pos,
	                                        gml->renderer_text,
	                                        pl_cell_data_function, gml, NULL);

	gml->playlist.column_id = gtk_tree_view_column_new_with_attributes("ID",
	                                                        gml->renderer_text,
	                                                        "text", 1, NULL);
	gtk_tree_view_append_column(GTK_TREE_VIEW(gml->playlist.list),
	                            gml->playlist.column_id);
	gtk_tree_view_column_set_resizable(gml->playlist.column_id, TRUE);
	gtk_tree_view_column_set_cell_data_func(gml->playlist.column_id,
	                                        gml->renderer_text,
	                                        pl_cell_data_function, gml, NULL);

	gml->playlist.column_artist = 
		gtk_tree_view_column_new_with_attributes("Artist", gml->renderer_text,
		                                         "text", 2, NULL);
	gtk_tree_view_append_column(GTK_TREE_VIEW(gml->playlist.list),
	                            gml->playlist.column_artist);
	gtk_tree_view_column_set_resizable(gml->playlist.column_artist, TRUE);
	gtk_tree_view_column_set_cell_data_func(gml->playlist.column_artist,
	                                        gml->renderer_text,
	                                        pl_cell_data_function, gml, NULL);

	gml->playlist.column_track = 
		gtk_tree_view_column_new_with_attributes("Track", gml->renderer_text,
		                                         "text", 3, NULL);
	gtk_tree_view_append_column(GTK_TREE_VIEW(gml->playlist.list),
	                            gml->playlist.column_track);
	gtk_tree_view_column_set_resizable(gml->playlist.column_track, TRUE);
	gtk_tree_view_column_set_cell_data_func(gml->playlist.column_track,
	                                        gml->renderer_text,
	                                        pl_cell_data_function, gml, NULL);

	gml->playlist.column_album = 
		gtk_tree_view_column_new_with_attributes("Album [#]", gml->renderer_text,
		                                         "text", 4, NULL);
	gtk_tree_view_append_column(GTK_TREE_VIEW(gml->playlist.list),
	                            gml->playlist.column_album);
	gtk_tree_view_column_set_resizable(gml->playlist.column_album, TRUE);
	gtk_tree_view_column_set_cell_data_func(gml->playlist.column_album,
	                                        gml->renderer_text,
	                                        pl_cell_data_function, gml, NULL);


	renderer_pixbuf = gtk_cell_renderer_pixbuf_new();
	gml->playlist.column_ranking = 
		gtk_tree_view_column_new_with_attributes("Ranking", renderer_pixbuf,
		                                         "pixbuf", 5, NULL);
	gtk_tree_view_append_column(GTK_TREE_VIEW(gml->playlist.list),
	                            gml->playlist.column_ranking);
	gtk_tree_view_column_set_resizable(gml->playlist.column_ranking, FALSE);


	/* ranking icons */
	gml->playlist.icon_rank[0] = 
		gdk_pixbuf_new_from_xpm_data((const char **)rank0_xpm);
	gml->playlist.icon_rank[1] = 
		gdk_pixbuf_new_from_xpm_data((const char **)rank1_xpm);
	gml->playlist.icon_rank[2] = 
		gdk_pixbuf_new_from_xpm_data((const char **)rank2_xpm);
	gml->playlist.icon_rank[3] = 
		gdk_pixbuf_new_from_xpm_data((const char **)rank3_xpm);
	gml->playlist.icon_rank[4] = 
		gdk_pixbuf_new_from_xpm_data((const char **)rank4_xpm);

	/* tooltips */
	tooltips = gtk_tooltips_new();
	gtk_tooltips_set_tip(tooltips, gml->playlist.button_new_pl, 
	                     "New playlist", NULL);
	gtk_tooltips_set_tip(tooltips, gml->playlist.button_save_pl, 
	                     "Save playlist", NULL);
	gtk_tooltips_set_tip(tooltips, gml->playlist.button_shuffle_pl, 
	                     "Shuffle the playlist", NULL);
	gtk_tooltips_set_tip(tooltips, gml->playlist.button_sort_pl, 
	                     "Sort the playlist by artist", NULL);
	gtk_tooltips_set_tip(tooltips, gml->playlist.button_add_rnd, 
	                     "Add 10 random songs to the playlist", NULL);
	gtk_tooltips_set_tip(tooltips, gml->playlist.button_reload_pl, 
	                     "Refresh the playlist\n(Useful if another XMMS2 " \
	                     "client has altered the playlist)", NULL);
	gtk_tooltips_set_tip(tooltips, gml->playlist.button_up, 
	                     "Move selected track one position upwards", NULL);
	gtk_tooltips_set_tip(tooltips, gml->playlist.button_down, 
	                     "Move selected track one position downwards", NULL);
	gtk_tooltips_set_tip(tooltips, gml->playlist.button_track, 
	                     "Show currently played track", NULL);

	gml_pl_set_column_pos_visible    (gml, TRUE);
	gml_pl_set_column_id_visible     (gml, TRUE);
	gml_pl_set_column_artist_visible (gml, TRUE);
	gml_pl_set_column_track_visible  (gml, TRUE);
	gml_pl_set_column_album_visible  (gml, TRUE);
	gml_pl_set_column_ranking_visible(gml, TRUE);

    g_signal_connect(G_OBJECT(gml->playlist.button_reload_pl),  "clicked",
	                 G_CALLBACK(cb_pl_reload_button_pressed),
	                 (gpointer)gml);
    g_signal_connect(G_OBJECT(gml->playlist.button_shuffle_pl), "clicked",
	                 G_CALLBACK(cb_pl_shuffle_button_pressed),
	                 (gpointer)gml);
    g_signal_connect(G_OBJECT(gml->playlist.button_sort_pl),    "clicked",
	                 G_CALLBACK(cb_pl_sort_button_pressed),
	                 (gpointer)gml);
    g_signal_connect(G_OBJECT(gml->playlist.button_add_rnd),    "clicked",
	                 G_CALLBACK(cb_pl_add_rnd_button_pressed),
	                 (gpointer)gml);
    g_signal_connect(G_OBJECT(gml->playlist.button_new_pl),     "clicked",
	                 G_CALLBACK(cb_pl_clear_button_pressed),
	                 (gpointer)gml);
    g_signal_connect(G_OBJECT(gml->playlist.button_save_pl),    "clicked",
	                 G_CALLBACK(cb_pl_save_pl_button_pressed),
	                 (gpointer)gml);
    g_signal_connect(G_OBJECT(gml->playlist.button_track),      "clicked",
	                 G_CALLBACK(cb_pl_button_track_pressed),
	                 (gpointer)gml);
    g_signal_connect(G_OBJECT(gml->playlist.button_up),         "clicked",
	                 G_CALLBACK(cb_pl_button_up_pressed),
	                 (gpointer)gml);
    g_signal_connect(G_OBJECT(gml->playlist.button_down),       "clicked",
	                 G_CALLBACK(cb_pl_button_down_pressed),
	                 (gpointer)gml);
    g_signal_connect(G_OBJECT(gml->playlist.list),        "row-activated",
	                 G_CALLBACK(cb_pl_on_tree_view_row_activated),
	                 (gpointer)gml);
	gml->playlist.sel = 
		gtk_tree_view_get_selection(GTK_TREE_VIEW(gml->playlist.list));
    g_signal_connect(gml->playlist.sel,                         "changed",
	                 G_CALLBACK(cb_pl_on_tree_selection_changed),
	                 (gpointer)gml);
    g_signal_connect(G_OBJECT(gml->playlist.list),   "button-press-event",
                     G_CALLBACK(cb_pl_selection_button_press),
                     (gpointer)gml);
    g_signal_connect(G_OBJECT(gml->playlist.list),           "popup-menu",
                     G_CALLBACK(cb_pl_selection_popup_menu),
                     (gpointer)gml);
	g_signal_connect(G_OBJECT(gml->playlist.list),      "key-press-event",
	                 G_CALLBACK(cb_pl_selection_key_press),
	                 (gpointer)gml);
	gml_pl_setup_xmms_callbacks(gml);

	update_current_pl_pos(gml);
	gml->playlist.move_finished = TRUE;
}
