/**
 *
 * @file     interface.c
 * @author   Aleix Conchillo Flaque <aleix@member.fsf.org>
 * @date     Tue Aug 05, 2003 21:59
 * @brief    XMMS GNOME applet GUI
 *
 * $Id: interface.c,v 1.9 2005/01/05 00:22:54 aleix Exp $
 *
 * @if copyright
 *
 * Copyright (C) 2003, 2004, 2005 Aleix Conchillo Flaque
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * @endif
 */

#include "interface.h"

#include "callbacks.h"
#include "playerctrl.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <string.h>

/* Static functions definition (declaration at end of file) */
static void init_menu(XMMSData *xmms);
static void init_stock_icons(XMMSData *xmms);
static void register_stock_icons(XMMSData *xmms, GtkIconFactory *factory);
static void reload_images(XMMSButton *b, GtkStyle *style,
                          GtkIconSize icon_size);
static void create_track_control(XMMSData *xmms);
static void create_play_control(XMMSData *xmms);
static void create_button(XMMSButton *b, const gchar *tooltip,
                          const gchar *stock_id, GCallback press_func,
                          XMMSData *xmms);
static void ref_and_remove(GtkWidget *w);
static void remove_button(XMMSButton *b);
static void remove_main_box(XMMSData *xmms);
static void remove_track_control(XMMSData *xmms);
static void remove_play_control(XMMSData *xmms);
static void destroy_widget(GtkWidget **w);
static void destroy_button(XMMSButton *b);
static void destroy_main_box(XMMSData *xmms);
static void destroy_track_control(XMMSData *xmms);
static void destroy_play_control(XMMSData *xmms);


/* Global stock icon identifiers */
static const gchar *GXMMS_PREV       = "gxmms_prev";
static const gchar *GXMMS_PLAY_PAUSE = "gxmms_play_pause";
static const gchar *GXMMS_STOP       = "gxmms_stop";
static const gchar *GXMMS_NEXT       = "gxmms_next";
static const gchar *GXMMS_EJECT      = "gxmms_eject";

/* Image files extension */
static const gchar *GXMMS_IMAGE_EXT = ".png";


void
gui_create(XMMSData *xmms)
{
    GtkWidget *applet = xmms->applet;

    init_stock_icons(xmms);

    /* Create main frame */
    xmms->frame = gtk_hbox_new(FALSE, 0);
    gtk_widget_show(xmms->frame);

    g_signal_connect_after(G_OBJECT(xmms->frame), "realize",
                           G_CALLBACK(cb_realize), xmms);

    xmms->icon_size = gtk_icon_size_register("gxmms_icon_size", 5, 5);

    xmms->tooltips = gtk_tooltips_new();

    create_track_control(xmms);
    create_play_control(xmms);

    /* Add main frame to applet */
    gtk_container_add(GTK_CONTAINER(applet), xmms->frame);

    /* Get initial size and orientation */
    xmms->size = panel_applet_get_size(PANEL_APPLET(applet));
    xmms->orient = panel_applet_get_orient(PANEL_APPLET(applet));

    /* Setup menu */
    init_menu(xmms);

    /* Somwhere we'd probably have access to the applet but not to our
       XMMSData */
    g_object_set_data(G_OBJECT(applet), "xmms", xmms);

    g_signal_connect(applet, "destroy", G_CALLBACK (cb_destroy), xmms);
    g_signal_connect(applet, "change_size", G_CALLBACK(cb_change_size), xmms);
    g_signal_connect(applet, "change_orient", G_CALLBACK(cb_change_orient),
                     xmms);

    gtk_widget_show(applet);
}

void
gui_destroy(XMMSData *xmms)
{
    timer_off(xmms);

    g_object_unref(xmms->tooltips);

    destroy_main_box(xmms);

    destroy_widget(&xmms->frame);

    g_free(xmms);
}

void
gui_setup(XMMSData *xmms)
{
    gint spacing;
    GtkWidget *track_box;
    GtkWidget *play_box;
    GtkWidget *main_box;
    GtkWidget *progress;

    if (xmms->main_box)
    {
        remove_main_box(xmms);
    }

    track_box = xmms->track_control.box;
    progress = xmms->track_control.progress;

    if ((xmms->orient == PANEL_APPLET_ORIENT_DOWN)
        || (xmms->orient == PANEL_APPLET_ORIENT_UP))
    {
        main_box = xmms->main_box = gtk_vbox_new(FALSE, 1);
        gtk_widget_set_size_request(xmms->main_box, 80, -1);

        play_box = xmms->play_control.box = gtk_hbox_new(TRUE, 0);

        gtk_widget_set_size_request(track_box, -1, 8);
        gtk_widget_set_size_request(play_box, -1, 15);

        gtk_progress_bar_set_orientation(GTK_PROGRESS_BAR(progress),
                                         GTK_PROGRESS_LEFT_TO_RIGHT);

        spacing = 1;
    }
    else
    {
        main_box = xmms->main_box = gtk_hbox_new(FALSE, 1);
        gtk_widget_set_size_request(xmms->main_box, -1, 80);

        play_box = xmms->play_control.box = gtk_vbox_new(TRUE, 0);

        gtk_widget_set_size_request(track_box, 8, -1);
        gtk_widget_set_size_request(play_box, 15, -1);

        gtk_progress_bar_set_orientation(GTK_PROGRESS_BAR(progress),
                                         GTK_PROGRESS_TOP_TO_BOTTOM);

        spacing = 0;
    }

    /* Add scroll wheel mask and scroll handler */
    gtk_widget_add_events(xmms->applet,GDK_SCROLL_MASK);
    g_signal_connect(xmms->applet, "scroll_event", G_CALLBACK(cb_scrollwheel), xmms);

    gtk_box_pack_start(GTK_BOX(play_box), xmms->play_control.prev.button,
                       TRUE, TRUE, 0);
    gtk_box_pack_start(GTK_BOX(play_box), xmms->play_control.play_pause.button,
                       TRUE, TRUE, 0);
    gtk_box_pack_start(GTK_BOX(play_box), xmms->play_control.stop.button,
                       TRUE, TRUE, 0);
    gtk_box_pack_start(GTK_BOX(play_box), xmms->play_control.next.button,
                       TRUE, TRUE, 0);
    gtk_box_pack_start(GTK_BOX(play_box), xmms->play_control.eject.button,
                       TRUE, TRUE, 0);

    if ((xmms->orient == PANEL_APPLET_ORIENT_DOWN)
        || (xmms->orient == PANEL_APPLET_ORIENT_UP)
        || (xmms->orient == PANEL_APPLET_ORIENT_LEFT))
    {
        gtk_box_pack_start(GTK_BOX(xmms->main_box), track_box, TRUE, TRUE, 0);
        gtk_box_pack_start(GTK_BOX(xmms->main_box), play_box, TRUE, TRUE, 0);
    }
    else
    {
        gtk_box_pack_start(GTK_BOX(xmms->main_box), play_box, TRUE, TRUE, 0);
        gtk_box_pack_start(GTK_BOX(xmms->main_box), track_box, TRUE, TRUE, 0);
    }

    gtk_box_pack_start(GTK_BOX(xmms->frame), main_box, TRUE, TRUE, spacing);

    gtk_widget_show_all(xmms->main_box);

    gui_update(xmms);
}

void
gui_update(XMMSData *xmms)
{
    gint pos;
    gchar *track;
    gint minutes;
    gint seconds;
    gint aux_time;
    gint cur_time;
    gint total_time;
    gdouble width;
    GtkWidget *progress;
    BonoboUIComponent *popup;
    gchar info[256] = _("No track");

    progress = xmms->track_control.progress;
    if (!player_is_running(xmms))
    {
        gtk_tooltips_set_tip(xmms->tooltips, xmms->track_control.box, info,
                             NULL);
        gtk_progress_bar_set_fraction(GTK_PROGRESS_BAR(progress), 0);
        return;
    }

    pos = player_get_playlist_pos(xmms);
    track = player_get_playlist_title(xmms, pos);
    total_time = player_get_playlist_time(xmms, pos);
    cur_time = player_get_output_time(xmms);

    width = total_time ? ((gdouble) cur_time / (gdouble) total_time) : 0;

    gtk_progress_bar_set_fraction(GTK_PROGRESS_BAR(progress), width);

    aux_time = cur_time / 1000;
    seconds = aux_time % 60;
    minutes = aux_time / 60;
    if (track != NULL)
    {
        sprintf(info, "%d. %s (%02d:%02d)", pos + 1, track, minutes, seconds);
    }
    gtk_tooltips_set_tip(xmms->tooltips, xmms->track_control.box, info, NULL);

    /* Update popup checkboxes */
    popup = panel_applet_get_popup_component(PANEL_APPLET(xmms->applet));

    bonobo_ui_component_set_prop(popup, "/commands/Repeat", "state",
                                 player_is_repeat(xmms) ? "1" : "0", NULL);
    bonobo_ui_component_set_prop(popup, "/commands/Shuffle", "state",
                                 player_is_shuffle(xmms) ? "1" : "0", NULL);
}

void
gui_theme_load(XMMSData *xmms, GtkStyle *style)
{
    reload_images(&xmms->play_control.prev, style, xmms->icon_size);
    reload_images(&xmms->play_control.play_pause, style, xmms->icon_size);
    reload_images(&xmms->play_control.stop, style, xmms->icon_size);
    reload_images(&xmms->play_control.next, style, xmms->icon_size);
    reload_images(&xmms->play_control.eject, style, xmms->icon_size);
}


/* Static functions declaration */

void
gui_show_error(XMMSData *xmms, const gchar *msg)
{
    GtkWidget *dialog;

    dialog = gtk_message_dialog_new(NULL, GTK_DIALOG_DESTROY_WITH_PARENT,
                                    GTK_MESSAGE_ERROR, GTK_BUTTONS_OK, msg);

    g_signal_connect(dialog, "response", G_CALLBACK(gtk_widget_destroy), NULL);
    g_signal_connect(G_OBJECT(dialog), "destroy",
                     G_CALLBACK(gtk_widget_destroyed), &dialog);
    gtk_window_set_resizable(GTK_WINDOW(dialog), FALSE);
    gtk_window_set_screen(GTK_WINDOW(dialog),
                          gtk_widget_get_screen(xmms->applet));
    gtk_widget_show(dialog);
}

/* Static functions declaration */

static void
init_menu(XMMSData *xmms)
{
#ifdef HAVE_XMMS
    static const char* menu_xml_file = "GNOME_gxmmsApplet.xml";
#else
    static const char* menu_xml_file = "GNOME_gxmms_bmpApplet.xml";
#endif
    static const BonoboUIVerb gxmms_menu_verbs[] = {
        BONOBO_UI_UNSAFE_VERB("Repeat",   cb_repeat_menu),
        BONOBO_UI_UNSAFE_VERB("Shuffle",  cb_shuffle_menu),
        BONOBO_UI_UNSAFE_VERB("MainWin",  cb_main_win_menu),
        BONOBO_UI_UNSAFE_VERB("PlWin",    cb_pl_win_menu),
        BONOBO_UI_UNSAFE_VERB("EQWin",    cb_eq_win_menu),
        BONOBO_UI_UNSAFE_VERB("ShowHide", cb_show_hide_menu),
        BONOBO_UI_UNSAFE_VERB("Prefer",   cb_player_pref_menu),
        BONOBO_UI_UNSAFE_VERB("Quit",     cb_quit_menu),
        BONOBO_UI_UNSAFE_VERB("About",    cb_about_menu),
        BONOBO_UI_VERB_END
    };
    BonoboUIComponent *popup;
    GtkWidget *applet = xmms->applet;

    panel_applet_setup_menu_from_file(PANEL_APPLET(applet),
                                      NULL,
                                      menu_xml_file,
                                      NULL,
                                      gxmms_menu_verbs,
                                      xmms);

    popup = panel_applet_get_popup_component(PANEL_APPLET(applet));

    bonobo_ui_component_add_listener(popup, "Repeat",
                                     (BonoboUIListenerFn) cb_toggle_repeat,
                                     xmms);
    bonobo_ui_component_add_listener(popup, "Shuffle",
                                     (BonoboUIListenerFn) cb_toggle_shuffle,
                                     xmms);

    bonobo_ui_component_set_prop(popup, "/commands/Repeat", "state",
                                 player_is_repeat(xmms) ? "1" : "0", NULL);
    bonobo_ui_component_set_prop(popup, "/commands/Shuffle", "state",
                                 player_is_shuffle(xmms) ? "1" : "0", NULL);
}

static void
init_stock_icons(XMMSData *xmms)
{
    GtkIconFactory *factory;

    factory = gtk_icon_factory_new();
    gtk_icon_factory_add_default(factory);

    register_stock_icons(xmms, factory);

    g_object_unref(factory);
}

static void
register_stock_icons(XMMSData *xmms, GtkIconFactory *factory)
{
    gint i;
    static gchar file_name[1024];
    const gchar *items[] = { GXMMS_PREV, GXMMS_PLAY_PAUSE, GXMMS_STOP,
                             GXMMS_NEXT, GXMMS_EJECT };

    for (i = 0; i < sizeof(items) / sizeof(items[0]); ++i)
    {
	GdkPixbuf *pixbuf;
	GtkIconSet *icon_set;
        GError *error = NULL;

#ifdef HAVE_XMMS
        sprintf(file_name, "%s/%s%s", GXMMS_DATADIR, items[i], GXMMS_IMAGE_EXT);
#else
        sprintf(file_name, "%s/%s%s", GXMMS_BMP_DATADIR, items[i], GXMMS_IMAGE_EXT);
#endif
	pixbuf = gdk_pixbuf_new_from_file(file_name, &error);
        if (pixbuf == NULL)
        {
            gui_show_error(xmms, error->message);
        }

	icon_set = gtk_icon_set_new_from_pixbuf(pixbuf);
	gtk_icon_factory_add(factory, items[i], icon_set);

	gtk_icon_set_unref(icon_set);
	g_object_unref(G_OBJECT(pixbuf));
    }
}

static void
reload_images(XMMSButton *b, GtkStyle *style, GtkIconSize icon_size)
{
    GtkIconSet *icon_set;

    icon_set = gtk_style_lookup_icon_set(style, b->stock_id);
    gtk_image_set_from_icon_set(GTK_IMAGE(b->image), icon_set, icon_size);
}

static void
create_track_control(XMMSData *xmms)
{
    GtkObject *adj;
    GtkWidget *progress;
    GtkWidget *track_box;

    /* To catch progress bar events (real-time tooltips) */
    track_box = xmms->track_control.box = gtk_event_box_new();

    adj = gtk_adjustment_new(0.0, 0.0, 1.0, 0.1, 0.1, 0.0);

    progress = xmms->track_control.progress = gtk_progress_bar_new();
    gtk_progress_set_adjustment(GTK_PROGRESS(progress), GTK_ADJUSTMENT(adj));

    gtk_container_add(GTK_CONTAINER(track_box), progress);

    g_signal_connect(track_box, "button_press_event",
                     G_CALLBACK(cb_seek), xmms->applet);

    gtk_widget_show(progress);
}

static void
create_play_control(XMMSData *xmms)
{
    create_button(&xmms->play_control.prev, _("Previous track"),
                  GXMMS_PREV, G_CALLBACK(cb_prev_clicked), xmms);

    create_button(&xmms->play_control.play_pause, _("Play / Pause"),
                  GXMMS_PLAY_PAUSE, G_CALLBACK(cb_play_pause_clicked), xmms);

    create_button(&xmms->play_control.stop, _("Stop"),
                  GXMMS_STOP, G_CALLBACK(cb_stop_clicked), xmms);

    create_button(&xmms->play_control.next, _("Next track"),
                  GXMMS_NEXT, G_CALLBACK(cb_next_clicked), xmms);

    create_button(&xmms->play_control.eject, _("Eject"),
                  GXMMS_EJECT, G_CALLBACK(cb_eject_clicked), xmms);
}

static void
create_button(XMMSButton *b, const gchar *tooltip,
              const gchar *stock_id, GCallback press_func, XMMSData *xmms)
{
    GtkWidget *button;
    GtkWidget *image;

    b->stock_id = stock_id;

    button = b->button = gtk_button_new();
    gtk_button_set_relief(GTK_BUTTON(button), GTK_RELIEF_HALF);

    image = b->image = gtk_image_new_from_stock(stock_id, xmms->icon_size);
    gtk_widget_show(image);

    gtk_container_add(GTK_CONTAINER(button), image);

    gtk_tooltips_set_tip(xmms->tooltips, button, tooltip, NULL);

    g_signal_connect(button, "button_press_event",
                     G_CALLBACK(cb_button_press), xmms->applet);
    g_signal_connect(button, "clicked", G_CALLBACK(press_func), xmms);

    gtk_widget_show(button);
}

static void
ref_and_remove(GtkWidget *w)
{
    GtkWidget *parent = w->parent;

    if(parent)
    {
        g_object_ref(G_OBJECT(w));
        gtk_container_remove(GTK_CONTAINER(parent), w);
    }
}

static void
remove_button(XMMSButton *b)
{
    ref_and_remove(b->button);
}

static void
remove_main_box(XMMSData *xmms)
{
    g_return_if_fail(GTK_IS_CONTAINER(xmms->main_box));

    remove_track_control(xmms);
    remove_play_control(xmms);

    destroy_widget(&xmms->main_box);
}

static void
remove_track_control(XMMSData *xmms)
{
    g_return_if_fail(GTK_IS_CONTAINER(xmms->track_control.box));

    /* Do not need to remove progress bar from its container */
    ref_and_remove(xmms->track_control.box);
}

static void
remove_play_control(XMMSData *xmms)
{
    g_return_if_fail(GTK_IS_CONTAINER(xmms->play_control.box));

    remove_button(&xmms->play_control.prev);
    remove_button(&xmms->play_control.play_pause);
    remove_button(&xmms->play_control.stop);
    remove_button(&xmms->play_control.next);
    remove_button(&xmms->play_control.eject);

    destroy_widget(&xmms->play_control.box);
}

static void
destroy_widget(GtkWidget **w)
{
    if ((w == NULL) || (*w == NULL))
    {
        return;
    }

    gtk_widget_destroy(*w);
    *w = NULL;
}

static void
destroy_button(XMMSButton *b)
{
    if (b == NULL)
    {
        return;
    }

    destroy_widget(&b->image);
    destroy_widget(&b->button);
}

static void
destroy_main_box(XMMSData *xmms)
{
    g_return_if_fail(GTK_IS_CONTAINER(xmms->main_box));

    destroy_track_control(xmms);
    destroy_play_control(xmms);

    destroy_widget(&xmms->main_box);
}

static void
destroy_track_control(XMMSData *xmms)
{
    g_return_if_fail(GTK_IS_CONTAINER(xmms->track_control.box));

    timer_off(xmms);

    destroy_widget(&xmms->track_control.progress);

    destroy_widget(&xmms->track_control.box);
}

static void
destroy_play_control(XMMSData *xmms)
{
    g_return_if_fail(GTK_IS_CONTAINER(xmms->play_control.box));

    destroy_button(&xmms->play_control.prev);
    destroy_button(&xmms->play_control.play_pause);
    destroy_button(&xmms->play_control.stop);
    destroy_button(&xmms->play_control.next);
    destroy_button(&xmms->play_control.eject);

    destroy_widget(&xmms->play_control.box);
}
