/*
 *  $Id: formats3d.c 22829 2020-05-19 08:53:44Z yeti-dn $
 *  Copyright (C) 2009-2020 David Necas (Yeti).
 *  E-mail: yeti@gwyddion.net.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
 */

/**
 * [FILE-MAGIC-USERGUIDE]
 * VTK structured grid file
 * .vtk
 * Export
 **/

/**
 * [FILE-MAGIC-USERGUIDE]
 * PLY 3D Polygon File Format
 * .ply
 * Export
 **/

/**
 * [FILE-MAGIC-USERGUIDE]
 * Wavefront OBJ 3D geometry
 * .obj
 * Read Export
 **/

/**
 * [FILE-MAGIC-USERGUIDE]
 * Object File Format 3D geometry
 * .off
 * Export
 **/

/**
 * [FILE-MAGIC-USERGUIDE]
 * Stereolitography STL 3D geometry (binary)
 * .stl
 * Read Export
 **/

/**
 * [FILE-MAGIC-USERGUIDE]
 * XYZ data
 * .xyz, .dat
 * Read Export
 **/

/**
 * [FILE-MAGIC-MISSING]
 * Avoding clash with a standard file format.
 **/

#include "config.h"
#include <math.h>
#include <string.h>
#include <glib/gstdio.h>
#include <libgwyddion/gwymacros.h>
#include <libgwymodule/gwymodule-file.h>
#include <libprocess/stats.h>
#include <libgwydgets/gwyradiobuttons.h>
#include <libgwydgets/gwydgetutils.h>
#include <app/gwyapp.h>
#include <app/gwymoduleutils-file.h>

#include "err.h"

typedef enum {
    TRIANGULATION_NONE = 0,    /* Only vertices, if the format supports it. */
    TRIANGULATION_PLAIN,
    TRIANGULATION_MIDPOINT,
    TRIANGULATION_RANDOM,
    TRIANGULATION_FOLLOW,
    NTRIANGULATIONS
} TriangulationType;

typedef enum {
    ZSCALE_USER = 0,
    ZSCALE_PHYSICAL,
    ZSCALE_AUTO,
    NZSCALES
} ZScaleType;

typedef struct {
    gdouble zscale;
    ZScaleType zscale_type;
    TriangulationType triang_type;
    gboolean swap_xy;
    gboolean flip_z;
} Export3DArgs;

typedef struct {
    gchar *xy_units;
    gchar *z_units;
    gboolean do_rasterise;
} Import3DArgs;

typedef gboolean (*Export3DFunc)(FILE *fh,
                                 GwyDataField *dfield,
                                 const gchar *title,
                                 GArray *vertices,
                                 GArray *triangles,
                                 const Export3DArgs *args);
typedef GwySurface* (*Import3DFunc)(guchar *buffer,
                                    gsize size,
                                    GError **error);

typedef struct {
    gdouble dx;             /* Pixel x-length; 1 for square pixels. */
    gdouble dy;             /* Pixel y-length; 1 for square pixels. */
    gdouble zscale_1_1;     /* Z scale we consider 1:1. */
    gdouble zscale_auto;    /* Scale to get reasonable z vs. x and y. */
} ScalingInfo;

typedef struct {
    guint a;
    guint b;
    guint c;
} TriangleIndices;

typedef struct {
    Export3DArgs *args;
    const ScalingInfo *scinfo;
    GtkWidget *dialogue;
    GSList *scales;
    GtkWidget *zscale;
    GSList *triangs;
    GtkWidget *swap_xy;
    GtkWidget *flip_z;
} Export3DControls;

typedef struct {
    Import3DArgs *args;
    GwySurface *surface;
    GtkWidget *dialogue;
    GtkWidget *xmin;
    GtkWidget *xmax;
    GtkWidget *xunit;
    GtkWidget *ymin;
    GtkWidget *ymax;
    GtkWidget *yunit;
    GtkWidget *zmin;
    GtkWidget *zmax;
    GtkWidget *zunit;
    GtkWidget *xy_units;
    GtkWidget *z_units;
    GtkWidget *do_rasterise;
} Import3DControls;

typedef struct {
    const gchar *name;
    const gchar *title;
    const gchar *description;
    GwyFileDetectFunc detect;
    Export3DFunc export_;
    Import3DFunc import;
    gboolean has_triang_none;
} Format3D;

static gboolean      module_register     (void);
static gint          detect3d_vtk        (const GwyFileDetectInfo *fileinfo,
                                          gboolean only_name,
                                          const gchar *name);
static gboolean      export3d_vtk        (FILE *fh,
                                          GwyDataField *dfield,
                                          const gchar *title,
                                          GArray *vertices,
                                          GArray *triangles,
                                          const Export3DArgs *args);
static gint          detect3d_ply        (const GwyFileDetectInfo *fileinfo,
                                          gboolean only_name,
                                          const gchar *name);
static gboolean      export3d_ply        (FILE *fh,
                                          GwyDataField *dfield,
                                          const gchar *title,
                                          GArray *vertices,
                                          GArray *triangles,
                                          const Export3DArgs *args);
static gint          detect3d_obj        (const GwyFileDetectInfo *fileinfo,
                                          gboolean only_name,
                                          const gchar *name);
static gboolean      export3d_obj        (FILE *fh,
                                          GwyDataField *dfield,
                                          const gchar *title,
                                          GArray *vertices,
                                          GArray *triangles,
                                          const Export3DArgs *args);
static GwySurface*   import3d_obj        (guchar *buffer,
                                          gsize size,
                                          GError **error);
static gint          detect3d_off        (const GwyFileDetectInfo *fileinfo,
                                          gboolean only_name,
                                          const gchar *name);
static gboolean      export3d_off        (FILE *fh,
                                          GwyDataField *dfield,
                                          const gchar *title,
                                          GArray *vertices,
                                          GArray *triangles,
                                          const Export3DArgs *args);
static gint          detect3d_stl        (const GwyFileDetectInfo *fileinfo,
                                          gboolean only_name,
                                          const gchar *name);
static gboolean      export3d_stl        (FILE *fh,
                                          GwyDataField *dfield,
                                          const gchar *title,
                                          GArray *vertices,
                                          GArray *triangles,
                                          const Export3DArgs *args);
static GwySurface*   import3d_stl        (guchar *buffer,
                                          gsize size,
                                          GError **error);
static gint          detect3d_xyz        (const GwyFileDetectInfo *fileinfo,
                                          gboolean only_name,
                                          const gchar *name);
static GwySurface*   import3d_xyz        (guchar *buffer,
                                          gsize size,
                                          GError **error);
static gboolean      formats3d_export    (GwyContainer *data,
                                          const gchar *filename,
                                          GwyRunType mode,
                                          GError **error,
                                          const gchar *name);
static gboolean      export3d_dialogue   (const Format3D *fmt,
                                          Export3DArgs *args,
                                          const ScalingInfo *scinfo);
static void          zscale_changed      (GtkEntry *entry,
                                          Export3DControls *controls);
static void          zscale_type_changed (GtkRadioButton *radio,
                                          Export3DControls *controls);
static void          update_zscale_entry (Export3DControls *controls);
static void          triang_type_changed (GtkRadioButton *radio,
                                          Export3DControls *controls);
static void          swap_xy_changed     (GtkToggleButton *toggle,
                                          Export3DControls *controls);
static void          flip_z_changed      (GtkToggleButton *toggle,
                                          Export3DControls *controls);
static void          update_sensitivity  (Export3DControls *controls);
static void          export3d_load_args  (GwyContainer *settings,
                                          Export3DArgs *args);
static void          export3d_save_args  (GwyContainer *settings,
                                          const Export3DArgs *args);
static GwyContainer* formats3d_import    (const gchar *filename,
                                          GwyRunType mode,
                                          GError **error,
                                          const gchar *name);
static GwyDataField* check_regular_grid  (GwySurface *surface);
static gboolean      import3d_dialogue   (const Format3D *fmt,
                                          Import3DArgs *args,
                                          GwySurface *surface,
                                          GwyDataField *image);
static void          construct_range     (GtkTable *table,
                                          const gchar *name,
                                          gint row,
                                          GtkWidget **from,
                                          GtkWidget **to,
                                          GtkWidget **unit);
static gint          construct_units     (Import3DControls *controls,
                                          GtkTable *table,
                                          gint row);
static void          update_range_lables (GtkWidget *from,
                                          GtkWidget *to,
                                          GtkWidget *unit,
                                          gdouble min,
                                          gdouble max,
                                          const gchar *unitstring);
static void          xyunits_changed     (Import3DControls *controls,
                                          GtkEntry *entry);
static void          zunits_changed      (Import3DControls *controls,
                                          GtkEntry *entry);
static void          do_rasterise_changed(Import3DControls *controls,
                                          GtkToggleButton *toggle);
static void          import3d_load_args  (GwyContainer *settings,
                                          Import3DArgs *args);
static void          import3d_save_args  (GwyContainer *settings,
                                          const Import3DArgs *args);

static const Format3D formats3d[] = {
    {
        "vtk3d", "VTK", N_("VTK structured grid (.vtk)"),
        &detect3d_vtk, &export3d_vtk, NULL,
        TRUE
    },
    {
        "ply3d", "PLY", N_("Polygon file format (.ply)"),
        &detect3d_ply, &export3d_ply, NULL,
        FALSE,
    },
    {
        "obj3d", "OBJ", N_("Wavefront geometry definition (.obj)"),
        &detect3d_obj, &export3d_obj, &import3d_obj,
        FALSE,
    },
    {
        "off3d", "OFF", N_("Object File Format (.off)"),
        &detect3d_off, &export3d_off, NULL,
        FALSE,
    },
    {
        "stl3d", "STL", N_("Stereolitography STL (.stl)"),
        &detect3d_stl, &export3d_stl, &import3d_stl,
        FALSE,
    },
    /* The function used to be called rawxyz when it has its own module;
     * keep the name.  It is not a real 3D format anyway. */
    {
        "rawxyz", "XYZ", N_("XYZ data files (.xyz)"),
        &detect3d_xyz, NULL, &import3d_xyz,
        FALSE,
    }
};

static const Export3DArgs export3d_defaults = {
    1.0, ZSCALE_AUTO, TRIANGULATION_PLAIN,
    TRUE, FALSE,
};

static const Import3DArgs import3d_defaults = {
    NULL, NULL, TRUE,
};

static GwyModuleInfo module_info = {
    GWY_MODULE_ABI_VERSION,
    &module_register,
    N_("Exports images as miscellaneous 3D data formats "
       "and imports XYZ points from 3D formats."),
    "Yeti <yeti@gwyddion.net>",
    "1.0",
    "David Nečas (Yeti)",
    "2020",
};

GWY_MODULE_QUERY2(module_info, formats3d)

static gboolean
module_register(void)
{
    guint i;

    for (i = 0; i < G_N_ELEMENTS(formats3d); i++) {
        const Format3D *fmt = formats3d + i;
        gwy_file_func_register(fmt->name, _(fmt->description),
                               fmt->detect,
                               fmt->import ? formats3d_import : NULL,
                               NULL,
                               fmt->export_ ? formats3d_export : NULL);
    }

    return TRUE;
}

static gint
detect3d_vtk(const GwyFileDetectInfo *fileinfo,
             gboolean only_name,
             G_GNUC_UNUSED const gchar *name)
{
    g_return_val_if_fail(only_name, 0);
    return g_str_has_suffix(fileinfo->name_lowercase, ".vtk") ? 30 : 0;
}

static gint
detect3d_ply(const GwyFileDetectInfo *fileinfo,
             gboolean only_name,
             G_GNUC_UNUSED const gchar *name)
{
    g_return_val_if_fail(only_name, 0);
    return g_str_has_suffix(fileinfo->name_lowercase, ".ply") ? 30 : 0;
}

static inline gint
try_to_match_keyword(const gchar *s, const gchar *k, guint maxlen)
{
    gint i;

    for (i = 1; i < maxlen; i++) {
        /* End of keyword means we know matched the keyword, or not. */
        if (!k[i])
            return g_ascii_isspace(s[i]) ? i : -i;
        /* Hitting a different character means we know we did not match but
         * we want to move to the keyword end. */
        if (k[i] != s[i]) {
            while (k[i])
                i++;
            return -i;
        }
    }

    /* We exhausted the buffer without deciding. */
    return 0;
}

static gint
detect3d_obj(const GwyFileDetectInfo *fileinfo,
             gboolean only_name,
             G_GNUC_UNUSED const gchar *name)
{
    enum { kw_min = 98, kw_max = 118 };
    static const gchar keywords[] =
        "bevel\0bmat\0bsp\0bzp\0\0"
        "c_interp\0call\0cdc\0cdp\0con\0csh\0cstype\0ctech\0curv\0curv2\0\0"
        "d_interp\0deg\0\0"
        "end\0\0"
        "f\0\0"
        "g\0\0"
        "hole\0\0"
        "l\0lod\0\0"
        "maplib\0mg\0mtllib\0\0"
        "o\0\0"
        "p\0parm\0\0"
        "res\0\0"
        "s\0scrv\0shadow\0shadow_obj\0sp\0stech\0step\0surf\0\0"
        "trace\0trace_obj\0trim\0\0"
        "usemap\0usemtl\0\0"
        "v\0vn\0vp\0vt\0";
    static const gint offsets[] = {
        0, 20, 75, 89, 94, 97, 100, -1, -1, -1, 106, 113, -1, 131, 134, -1,
        142, 147, 192, 214, 229
    };

    const gchar *h;
    guint i, hlen, ngood = 0, nbad = 0;
    gboolean line_ended_with_backslash = FALSE;
    gint j, k;
    guchar c;

    if (only_name)
        return g_str_has_suffix(fileinfo->name_lowercase, ".obj") ? 15 : 0;

    if (fileinfo->buffer_len < 60)
        return 0;

    hlen = fileinfo->buffer_len-1;
    h = fileinfo->head;
    i = 0;
    while (TRUE) {
        while (i < hlen && g_ascii_isspace(h[i]))
            i++;
        if (i == hlen)
            goto decide;

        /* Try to weed out binary files quickly. */
        c = h[i];
        if (!g_ascii_isprint(c)) {
            gwy_debug("non-ASCII");
            return 0;
        }

        if (c >= kw_min && c <= kw_max && offsets[c - kw_min] != -1) {
            gwy_debug("promising character %c", c);
            /* This could be the start of a valid keyword. */
            j = offsets[c - kw_min];
            while (TRUE) {
                k = try_to_match_keyword(h + i, keywords + j, hlen - i);
                if (!k)
                    goto decide;
                if (k > 0) {
                    gwy_debug("matched keyword %s", keywords + j);
                    i += k;
                    ngood++;
                    break;
                }
                j -= k;
                j++;
                if (!keywords[j]) {
                    gwy_debug("failed to match any keyword");
                    i++;
                    nbad++;
                    break;
                }
                gwy_debug("failed to match keyword %s, but trying another",
                          keywords-1 + j + k);
            }
        }
        else if (c == '#') {
            gwy_debug("comment");
            i++;
        }
        else if (!line_ended_with_backslash) {
            gwy_debug("bad line %.*s", 12, h + i);
            i++;
            nbad++;
        }
        else {
            gwy_debug("previous line must have ended with backlash");
        }

        while (i < hlen && h[i] != '\r' && h[i] != '\n') {
            if (!g_ascii_isprint(h[i])) {
                gwy_debug("non-ASCII");
                return 0;
            }
            line_ended_with_backslash = (h[i] == '\\');
            i++;
        }
        if (i == hlen)
            goto decide;

        if (nbad >= 3) {
            gwy_debug("too many bad lines");
            return 0;
        }
        if (ngood >= 12*(nbad + 1)) {
            gwy_debug("lots of keywords found");
            return 50;
        }
        if (line_ended_with_backslash) {
            gwy_debug("line ended with backslash");
        }
    }
decide:
    gwy_debug("exhausted entire buffer, ngood=%u, nbad=%u -> score %u",
              ngood, nbad, 50*ngood/(12*(nbad + 1)));
    return 50*ngood/(12*(nbad + 1));
}

static gint
detect3d_off(const GwyFileDetectInfo *fileinfo,
             gboolean only_name,
             G_GNUC_UNUSED const gchar *name)
{
    g_return_val_if_fail(only_name, 0);
    return g_str_has_suffix(fileinfo->name_lowercase, ".off") ? 30 : 0;
}

static gint
detect3d_stl(const GwyFileDetectInfo *fileinfo,
             gboolean only_name,
             G_GNUC_UNUSED const gchar *name)
{
    guint i, j, ntri, ngood = 0;
    const guchar *p;
    gdouble block[12];

    if (only_name)
        g_str_has_suffix(fileinfo->name_lowercase, ".stl") ? 30 : 0;

    /* 80 bytes + 50 bytes * n-of-triangles */
    if (fileinfo->file_size < 134
        || fileinfo->file_size % 50 != 34
        || fileinfo->buffer_len < 134)
        return 0;

    p = fileinfo->head + 80;
    ntri = gwy_get_guint32_le(&p);
    if (ntri != (fileinfo->file_size - 84)/50)
        return 0;

    ntri = (fileinfo->buffer_len - 84)/50;
    ntri = MIN(ntri, 12);
    /* Check if attribute counts seem to be 0. */
    for (i = 0; i < ntri; i++) {
        if (fileinfo->head[132 + 50*i] != 0 || fileinfo->head[133 + 50*i] != 0)
            return 0;
    }

    /* Check the numbers.  Random stuff tends to produce infs, nans and
     * numbers of weird magnitudes.  Normals can be rubbish or zeros, so check
     * only coordinates.  */
    for (i = 0; i < ntri; i++) {
        gwy_convert_raw_data(fileinfo->head + 84 + i*50, 12, 1,
                             GWY_RAW_DATA_FLOAT, GWY_BYTE_ORDER_LITTLE_ENDIAN,
                             block, 1.0, 0.0);
        for (j = 0; j < 12; j++) {
            if (gwy_isnan(block[i]) || gwy_isinf(block[i]))
                return 0;
            if (block[i] && (fabs(block[i]) > 1e30 || fabs(block[i]) < 1e-30))
                return 0;
            if (j > 4 && (block[i] || (fabs(block[i]) < 1e12
                                       && fabs(block[i]) > 1e-12)))
                ngood++;
        }
    }

    return 50*ngood/(12*ntri);
}

static gint
detect3d_xyz(const GwyFileDetectInfo *fileinfo,
             gboolean only_name,
             G_GNUC_UNUSED const gchar *name)
{
    const gchar *s;
    gchar *end;
    guint i;

    if (only_name)
        return g_str_has_suffix(fileinfo->name_lowercase, ".xyz") ? 20 : 0;

    s = fileinfo->head;
    for (i = 0; i < 6; i++) {
        g_ascii_strtod(s, &end);
        if (end == s) {
            /* If we encounter garbage at the first line, give it a one more
             * chance. */
            if (i || !(s = strchr(s, '\n')))
                return 0;
            goto next_line;
        }
        s = end;
        while (g_ascii_isspace(*s) || *s == ';' || *s == ',')
             s++;
        g_ascii_strtod(s, &end);
        if (end == s)
            return 0;
        s = end;
        while (g_ascii_isspace(*s) || *s == ';' || *s == ',')
             s++;
        g_ascii_strtod(s, &end);
        if (end == s)
            return 0;

        s = end;
        while (*s == ' ' || *s == '\t')
            s++;
        if (*s != '\n' && *s != '\r')
            return 0;

next_line:
        do {
            s++;
        } while (g_ascii_isspace(*s));
    }

    return 50;
}

static const Format3D*
find_format(const gchar *name)
{
    guint i;

    for (i = 0; i < G_N_ELEMENTS(formats3d); i++) {
        const Format3D *fmt = formats3d + i;
        if (gwy_strequal(fmt->name, name))
            return fmt;
    }

    return NULL;
}

/*************************************************************************
 *
 * Export
 *
 *************************************************************************/

static void
make_scaling_info(GwyDataField *dfield, ScalingInfo *scinfo)
{
    gdouble min, max, dx, dy, a;
    gint xres, yres;

    xres = gwy_data_field_get_xres(dfield);
    yres = gwy_data_field_get_yres(dfield);
    dx = gwy_data_field_get_dx(dfield);
    dy = gwy_data_field_get_dy(dfield);
    gwy_data_field_get_min_max(dfield, &min, &max);

    scinfo->zscale_1_1 = 1.0/sqrt(dx*dy);
    scinfo->dx = dx*scinfo->zscale_1_1;
    scinfo->dy = dy*scinfo->zscale_1_1;
    a = dx*xres*scinfo->dx * dy*yres*scinfo->dy;
    scinfo->zscale_auto = (max <= min) ? 0.0 : 0.2*sqrt(a)/(max - min);
}

static inline void
make_triangle_split(GArray *triangles, guint k, guint xres,
                    gboolean main_diagonal)
{
    TriangleIndices abc;

    if (main_diagonal) {
        abc.a = k;
        abc.b = k+xres;
        abc.c = k+xres+1;
        g_array_append_val(triangles, abc);
        abc.b = k+xres+1;
        abc.c = k+1;
        g_array_append_val(triangles, abc);
    }
    else {
        abc.a = k+1;
        abc.b = k;
        abc.c = k+xres;
        g_array_append_val(triangles, abc);
        abc.b = k+xres;
        abc.c = k+xres+1;
        g_array_append_val(triangles, abc);
    }
}

static void
make_triangulation(GwyDataField *dfield,
                   const ScalingInfo *scinfo,
                   const Export3DArgs *args,
                   GArray *vertices,
                   GArray *triangles)
{
    gint xres, yres, i, j, k, stride;
    gdouble z, min, dx = scinfo->dx, dy = scinfo->dy, qz = scinfo->zscale_1_1;
    const gdouble *d;
    GwyXYZ xyz;

    xres = gwy_data_field_get_xres(dfield);
    yres = gwy_data_field_get_yres(dfield);
    min = gwy_data_field_get_min(dfield);

    if (args->zscale_type == ZSCALE_USER)
        qz *= args->zscale;
    else if (args->zscale_type == ZSCALE_AUTO)
        qz *= scinfo->zscale_auto;

    g_array_set_size(vertices, 0);
    g_array_set_size(triangles, 0);
    d = gwy_data_field_get_data_const(dfield);

    if (args->triang_type == TRIANGULATION_MIDPOINT) {
        stride = 2*xres - 1;
        for (i = 0; i < yres-1; i++) {
            xyz.y = i*dy;
            for (j = 0; j < xres; j++) {
                k = i*xres + j;
                z = d[k];
                xyz.x = j*dx;
                xyz.z = qz*(z - min);
                g_array_append_val(vertices, xyz);
            }

            xyz.y = (i + 0.5)*dy;
            for (j = 0; j < xres-1; j++) {
                TriangleIndices abc;

                k = i*xres + j;
                z = 0.25*(d[k] + d[k+1] + d[k+xres] + d[k+xres+1]);
                xyz.x = (j + 0.5)*dx;
                xyz.z = qz*(z - min);
                g_array_append_val(vertices, xyz);

                k = i*stride + j;
                abc.a = k;
                abc.b = k + xres;
                abc.c = k+1;
                g_array_append_val(triangles, abc);
                abc.a = abc.c;
                abc.c = k + stride+1;
                g_array_append_val(triangles, abc);
                abc.a = abc.c;
                abc.c = k + stride;
                g_array_append_val(triangles, abc);
                abc.a = abc.c;
                abc.c = k;
                g_array_append_val(triangles, abc);
            }
        }

        xyz.y = (yres - 1)*dy;
        for (j = 0; j < xres; j++) {
            k = i*xres + j;
            z = d[k];
            xyz.x = j*dx;
            xyz.z = qz*(z - min);
            g_array_append_val(vertices, xyz);
        }
        return;
    }

    for (i = 0; i < yres; i++) {
        xyz.y = i*dy;
        for (j = 0; j < xres; j++, d++) {
            xyz.x = j*dx;
            xyz.z = qz*(*d - min);
            g_array_append_val(vertices, xyz);
        }
    }
    if (args->triang_type == TRIANGULATION_NONE)
        return;

    if (args->triang_type == TRIANGULATION_RANDOM) {
        GRand *rng = g_rand_new();
        guint ranval = 0, havebits = 0;

        for (i = 0; i < yres-1; i++) {
            for (j = 0; j < xres-1; j++) {
                if (!havebits) {
                    ranval = g_rand_int(rng);
                    havebits = 31;
                }
                make_triangle_split(triangles, i*xres + j, xres, (ranval & 1));
                ranval >>= 1;
                havebits--;
            }
        }

        g_rand_free(rng);
    }
    else if (args->triang_type == TRIANGULATION_FOLLOW
             && xres > 4
             && yres > 4) {
        gdouble zmaj, zmin;
        gboolean ismaj;

        for (i = 1; i < yres-2; i++) {
            for (j = 1; j < xres-2; j++) {
                k = i*xres + j;
                zmaj = fabs(d[k] + d[k+xres+1] - d[k-xres-1] - d[k + 2*xres+2]);
                zmin = fabs(d[k+1] + d[k+xres] - d[k-xres+2] - d[k + 2*xres-1]);
                ismaj = (zmaj <= zmin);
                make_triangle_split(triangles, i*xres + j, xres, ismaj);
                if (j == 1)
                    make_triangle_split(triangles, k-1, xres, ismaj);
                if (i == 1 && j == 1)
                    make_triangle_split(triangles, 0, xres, ismaj);
                if (i == 1)
                    make_triangle_split(triangles, k-xres, xres, ismaj);
                if (i == 1 && j == xres-3)
                    make_triangle_split(triangles, k+1-xres, xres, ismaj);
                if (j == xres-3)
                    make_triangle_split(triangles, k+1, xres, ismaj);
                if (i == yres-3 && j == 1)
                    make_triangle_split(triangles, k+xres-1, xres, ismaj);
                if (i == yres-3)
                    make_triangle_split(triangles, k+xres, xres, ismaj);
                if (i == yres-3 && j == xres-3)
                    make_triangle_split(triangles, k+xres+1, xres, ismaj);
            }
        }
    }
    else {
        for (i = 0; i < yres-1; i++) {
            for (j = 0; j < xres-1; j++)
                make_triangle_split(triangles, i*xres + j, xres, TRUE);
        }
    }
}

static void
fix_triangulation(GArray *vertices, GArray *triangles,
                  gboolean flip_z, gboolean swap_xy)
{
    guint k, nvert = vertices->len, ntri = triangles->len;

    if (flip_z) {
        for (k = 0; k < nvert; k++) {
            GwyXYZ *xyz = &g_array_index(vertices, GwyXYZ, k);
            xyz->z *= 1.0;
        }
    }

    if (swap_xy) {
        for (k = 0; k < nvert; k++) {
            GwyXYZ *xyz = &g_array_index(vertices, GwyXYZ, k);
            GWY_SWAP(gdouble, xyz->x, xyz->y);
        }
    }
    else {
        for (k = 0; k < ntri; k++) {
            TriangleIndices *abc = &g_array_index(triangles, TriangleIndices,
                                                  k);
            GWY_SWAP(guint, abc->b, abc->c);
        }
    }
}

static gboolean
formats3d_export(G_GNUC_UNUSED GwyContainer *data,
                 const gchar *filename,
                 GwyRunType mode,
                 GError **error,
                 const gchar *name)
{
    const Format3D *fmt;
    Export3DArgs args;
    ScalingInfo scinfo;
    const guchar *title = NULL;
    GwyDataField *dfield;
    GArray *vertices, *triangles;
    gint id;
    gboolean ok;
    FILE *fh;

    fmt = find_format(name);
    g_return_val_if_fail(fmt, FALSE);
    g_return_val_if_fail(fmt->export_, FALSE);

    gwy_app_data_browser_get_current(GWY_APP_DATA_FIELD, &dfield,
                                     GWY_APP_DATA_FIELD_ID, &id,
                                     0);

    if (!dfield) {
        err_NO_CHANNEL_EXPORT(error);
        return FALSE;
    }

    export3d_load_args(gwy_app_settings_get(), &args);
    make_scaling_info(dfield, &scinfo);

    gwy_container_gis_string(data, gwy_app_get_data_title_key_for_id(id),
                             &title);
    if (!title)
        title = _("Untitled");

    if (!fmt->has_triang_none && args.triang_type == TRIANGULATION_NONE)
        args.triang_type = TRIANGULATION_PLAIN;

    if (mode == GWY_RUN_INTERACTIVE) {
        ok = export3d_dialogue(fmt, &args, &scinfo);
        export3d_save_args(gwy_app_settings_get(), &args);
        if (!ok) {
            err_CANCELLED(error);
            return FALSE;
        }
    }

    if (!(fh = gwy_fopen(filename, "w"))) {
        err_OPEN_WRITE(error);
        return FALSE;
    }

    vertices = g_array_new(FALSE, FALSE, sizeof(GwyXYZ));
    triangles = g_array_new(FALSE, FALSE, sizeof(TriangleIndices));
    make_triangulation(dfield, &scinfo, &args, vertices, triangles);
    fix_triangulation(vertices, triangles, args.swap_xy, args.flip_z);

    ok = fmt->export_(fh, dfield, title, vertices, triangles, &args);
    if (!ok)
        err_WRITE(error);
    fclose(fh);

    g_array_free(vertices, TRUE);
    g_array_free(triangles, TRUE);

    if (ok)
        return TRUE;

    g_unlink(filename);
    return FALSE;
}

static gboolean
write_vertex_lines(FILE *fh,
                   GArray *vertices,
                   gchar sep,
                   const gchar *prefix)
{
    guint k, nvert = vertices->len;
    const GwyXYZ *xyz;
    gchar bufx[24], bufy[24], bufz[24];

    for (k = 0; k < nvert; k++) {
        xyz = &g_array_index(vertices, GwyXYZ, k);
        g_ascii_formatd(bufx, sizeof(bufx), "%.9g", xyz->x);
        g_ascii_formatd(bufy, sizeof(bufy), "%.9g", xyz->y);
        g_ascii_formatd(bufz, sizeof(bufz), "%.9g", xyz->z);
        gwy_fprintf(fh, "%s%s%c%s%c%s",
                    prefix, bufx, sep, bufy, sep, bufz);
        if (fputc('\n', fh) == EOF)
            return FALSE;
    }
    return TRUE;
}

static gboolean
write_triangle_lines(FILE *fh,
                     GArray *triangles,
                     gchar sep,
                     const gchar *prefix)
{
    guint k, ntri = triangles->len;
    const TriangleIndices *abc;

    for (k = 0; k < ntri; k++) {
        abc = &g_array_index(triangles, TriangleIndices, k);
        gwy_fprintf(fh, "%s%u%c%u%c%u",
                    prefix, abc->a, sep, abc->b, sep, abc->c);
        if (fputc('\n', fh) == EOF)
            return FALSE;
    }
    return TRUE;
}

static gboolean
export3d_vtk(FILE *fh,
             GwyDataField *dfield,
             const gchar *title,
             GArray *vertices,
             GArray *triangles,
             const Export3DArgs *args)
{
    guint xres, yres, nvert, ntri;

    xres = gwy_data_field_get_xres(dfield);
    yres = gwy_data_field_get_yres(dfield);
    nvert = vertices->len;
    ntri = triangles->len;

    /* Do not bother checking errors here.  If some write fails we will get
     * more errors below. */
    fputs("# vtk DataFile Version 2.0\n", fh);
    gwy_fprintf(fh, "%s\n", title);
    fputs("ASCII\n", fh);
    if (args->triang_type == TRIANGULATION_NONE) {
        g_assert(nvert == xres*yres);
        fputs("DATASET STRUCTURED_GRID\n", fh);
        gwy_fprintf(fh, "DIMENSIONS %u %u 1\n", xres, yres);
        gwy_fprintf(fh, "POINTS %u float\n", nvert);
        return write_vertex_lines(fh, vertices, '\n', "");
    }

    fputs("DATASET POLYDATA\n", fh);
    gwy_fprintf(fh, "POINTS %u float\n", nvert);
    if (!write_vertex_lines(fh, vertices, '\n', ""))
        return FALSE;

    gwy_fprintf(fh, "POLYGONS %u %u\n", ntri, 4*ntri);
    if (!write_triangle_lines(fh, triangles, ' ', "3 "))
        return FALSE;

    return TRUE;
}

static gboolean
export3d_ply(FILE *fh,
             G_GNUC_UNUSED GwyDataField *dfield,
             const gchar *title,
             GArray *vertices,
             GArray *triangles,
             const Export3DArgs *args)
{
    guint nvert, ntri;

    g_return_val_if_fail(args->triang_type != TRIANGULATION_NONE, FALSE);
    nvert = vertices->len;
    ntri = triangles->len;

    /* Do not bother checking errors here.  If some write fails we will get
     * more errors below. */
    fputs("ply\n", fh);
    fputs("format ascii 1.0\n", fh);
    fputs("comment exported from Gwyddion\n", fh);
    gwy_fprintf(fh, "comment title %s\n", title);
    gwy_fprintf(fh, "element vertex %u\n", nvert);
    fputs("property float x\n", fh);
    fputs("property float y\n", fh);
    fputs("property float z\n", fh);
    gwy_fprintf(fh, "element face %u\n", ntri);
    fputs("property list uchar int vertex_index\n", fh);
    fputs("end_header\n", fh);

    if (!write_vertex_lines(fh, vertices, ' ', ""))
        return FALSE;
    if (!write_triangle_lines(fh, triangles, ' ', "3 "))
        return FALSE;

    return TRUE;
}

static gboolean
export3d_obj(FILE *fh,
             G_GNUC_UNUSED GwyDataField *dfield,
             const gchar *title,
             GArray *vertices,
             GArray *triangles,
             const Export3DArgs *args)
{
    g_return_val_if_fail(args->triang_type != TRIANGULATION_NONE, FALSE);

    /* Do not bother checking errors here.  If some write fails we will get
     * more errors below. */
    fputs("# exported from Gwyddion\n", fh);
    gwy_fprintf(fh, "# title %s\n", title);
    fputc('\n', fh);
    fputs("g surface\n", fh);
    fputc('\n', fh);

    if (!write_vertex_lines(fh, vertices, ' ', "v "))
        return FALSE;
    fputc('\n', fh);

    if (!write_triangle_lines(fh, triangles, ' ', "f "))
        return FALSE;

    return TRUE;
}

static gboolean
export3d_off(FILE *fh,
             G_GNUC_UNUSED GwyDataField *dfield,
             const gchar *title,
             GArray *vertices,
             GArray *triangles,
             const Export3DArgs *args)
{
    guint nvert, ntri;

    g_return_val_if_fail(args->triang_type != TRIANGULATION_NONE, FALSE);
    nvert = vertices->len;
    ntri = triangles->len;

    /* Do not bother checking errors here.  If some write fails we will get
     * more errors below. */
    gwy_fprintf(fh, "OFF %u %u 0\n", nvert, ntri);
    fputs("# exported from Gwyddion\n", fh);
    gwy_fprintf(fh, "# title %s\n", title);
    fputc('\n', fh);

    if (!write_vertex_lines(fh, vertices, ' ', ""))
        return FALSE;
    fputc('\n', fh);

    if (!write_triangle_lines(fh, triangles, ' ', "3 "))
        return FALSE;

    return TRUE;
}

static gboolean
export3d_stl(FILE *fh,
             G_GNUC_UNUSED GwyDataField *dfield,
             G_GNUC_UNUSED const gchar *title,
             GArray *vertices,
             GArray *triangles,
             const Export3DArgs *args)
{
    guchar buf[80];
    const TriangleIndices *abc;
    const GwyXYZ *a, *b, *c;
    guint ntri, k;
    guint swapme = (G_BYTE_ORDER == G_BIG_ENDIAN ? 3 : 0);
    gfloat values[4*3];
    guint32 t;

    g_return_val_if_fail(args->triang_type != TRIANGULATION_NONE, FALSE);
    ntri = triangles->len;

    /* 80 bytes long text header/comment/what-have-you. */
    gwy_clear(buf, sizeof(buf));
    g_snprintf(buf, sizeof(buf), "STL binary data exported from Gwyddion");
    if (fwrite(buf, 1, sizeof(buf), fh) != sizeof(buf))
        return FALSE;

    /* Number of triangles. */
    t = GUINT32_TO_LE(ntri);
    if (fwrite(&t, sizeof(guint32), 1, fh) != 1)
        return FALSE;

    /* Triangle loop, each block 50 bytes long. */
    buf[48] = buf[49] = 0;   /* Attribute byte count, i.e. zero. */
    for (k = 0; k < ntri; k++) {
        abc = &g_array_index(triangles, TriangleIndices, k);
        a = &g_array_index(vertices, GwyXYZ, abc->a);
        b = &g_array_index(vertices, GwyXYZ, abc->b);
        c = &g_array_index(vertices, GwyXYZ, abc->c);

        values[0] = (a->y*b->z - a->z*b->y
                     + b->y*c->z - b->z*c->y
                     + c->y*a->z - c->z*a->y);
        values[1] = (a->z*b->x - a->x*b->z
                     + b->z*c->x - b->x*c->z
                     + c->z*a->x - c->x*a->z);
        values[2] = (a->x*b->y - a->y*b->x
                     + b->x*c->y - b->y*c->x
                     + c->x*a->y - c->y*a->x);
        values[3] = a->x;
        values[4] = a->y;
        values[5] = a->z;
        values[6] = b->x;
        values[7] = b->y;
        values[8] = b->z;
        values[9] = c->x;
        values[10] = c->y;
        values[11] = c->z;
        gwy_memcpy_byte_swap((guint8*)values, buf, sizeof(gfloat), 4*3, swapme);
        if (fwrite(buf, 1, 50, fh) != 50)
            return FALSE;
    }

    return TRUE;
}

static gboolean
export3d_dialogue(const Format3D *fmt,
                  Export3DArgs *args,
                  const ScalingInfo *scinfo)
{
    static const GwyEnum triang_types[] = {
        { N_("None (only points)"),         TRIANGULATION_NONE,     },
        { N_("Plain along main diagonals"), TRIANGULATION_PLAIN,    },
        { N_("With pixel midpoints"),       TRIANGULATION_MIDPOINT, },
        { N_("Random orientation"),         TRIANGULATION_RANDOM,   },
        { N_("Following features"),         TRIANGULATION_FOLLOW,   },
    };

    Export3DControls controls;
    GtkWidget *dialogue, *table, *label;
    gint response, row = 0, omit_none;
    gchar *title = NULL;

    controls.args = args;
    controls.scinfo = scinfo;

    omit_none = !fmt->has_triang_none;
    title = g_strdup_printf(_("Export %s"), fmt->title);
    dialogue = gtk_dialog_new_with_buttons(title, NULL, 0,
                                           GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
                                           GTK_STOCK_OK, GTK_RESPONSE_OK,
                                           NULL);
    g_free(title);
    controls.dialogue = dialogue;
    gtk_dialog_set_default_response(GTK_DIALOG(dialogue), GTK_RESPONSE_OK);
    gwy_help_add_to_file_dialog(GTK_DIALOG(dialogue), GWY_HELP_DEFAULT);

    table = gtk_table_new(5 + NZSCALES + NTRIANGULATIONS, 2, FALSE);
    gtk_table_set_row_spacings(GTK_TABLE(table), 2);
    gtk_table_set_col_spacings(GTK_TABLE(table), 6);
    gtk_container_set_border_width(GTK_CONTAINER(table), 12);
    gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialogue)->vbox), table,
                       TRUE, TRUE, 0);
    row = 0;

    label = gtk_label_new_with_mnemonic(_("Z scale:"));
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
    gtk_table_attach(GTK_TABLE(table), label,
                     0, 1, row, row+1, GTK_FILL, 0, 0, 0);
    row++;

    controls.scales = gwy_radio_buttons_createl(G_CALLBACK(zscale_type_changed),
                                                &controls,
                                                args->zscale_type,
                                                _("_Automatic Z-scale"),
                                                ZSCALE_AUTO,
                                                _("_Physical 1:1"),
                                                ZSCALE_PHYSICAL,
                                                _("Other _scale:"),
                                                ZSCALE_USER,
                                                NULL);
    row = gwy_radio_buttons_attach_to_table(controls.scales, GTK_TABLE(table),
                                            2, row);

    controls.zscale = gtk_entry_new();
    gtk_entry_set_width_chars(GTK_ENTRY(controls.zscale), 9);
    gtk_table_attach(GTK_TABLE(table), controls.zscale,
                     1, 2, row, row+1, GTK_EXPAND | GTK_FILL, 0, 0, 0);
    update_zscale_entry(&controls);
    g_signal_connect(controls.zscale, "changed",
                     G_CALLBACK(zscale_changed), &controls);
    row++;

    gtk_table_set_row_spacing(GTK_TABLE(table), row-1, 8);
    label = gtk_label_new_with_mnemonic(_("Triangulation type:"));
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
    gtk_table_attach(GTK_TABLE(table), label,
                     0, 1, row, row+1, GTK_FILL, 0, 0, 0);
    row++;

    controls.triangs = gwy_radio_buttons_create(triang_types + omit_none,
                                                NTRIANGULATIONS - omit_none,
                                                G_CALLBACK(triang_type_changed),
                                                &controls,
                                                args->triang_type);
    row = gwy_radio_buttons_attach_to_table(controls.triangs, GTK_TABLE(table),
                                            2, row);

    gtk_table_set_row_spacing(GTK_TABLE(table), row-1, 8);
    controls.swap_xy
        = gtk_check_button_new_with_mnemonic(_("Swap X and Y axes"));
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(controls.swap_xy),
                                 args->swap_xy);
    g_signal_connect(controls.swap_xy, "toggled",
                     G_CALLBACK(swap_xy_changed), &controls);
    gtk_table_attach(GTK_TABLE(table), controls.swap_xy,
                     0, 2, row, row+1, GTK_EXPAND | GTK_FILL, 0, 0, 0);
    row++;

    controls.flip_z = gtk_check_button_new_with_mnemonic(_("Flip Z axis"));
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(controls.flip_z),
                                 args->flip_z);
    g_signal_connect(controls.flip_z, "toggled",
                     G_CALLBACK(flip_z_changed), &controls);
    gtk_table_attach(GTK_TABLE(table), controls.flip_z,
                     0, 2, row, row+1, GTK_EXPAND | GTK_FILL, 0, 0, 0);
    row++;

    gtk_widget_show_all(dialogue);
    update_sensitivity(&controls);

    response = gtk_dialog_run(GTK_DIALOG(dialogue));
    if (response != GTK_RESPONSE_NONE)
        gtk_widget_destroy(dialogue);

    return response == GTK_RESPONSE_OK;
}

static void
zscale_changed(GtkEntry *entry, Export3DControls *controls)
{
    Export3DArgs *args = controls->args;

    if (args->zscale_type == ZSCALE_USER)
        args->zscale = g_strtod(gtk_entry_get_text(entry), NULL);
}

static void
zscale_type_changed(G_GNUC_UNUSED GtkRadioButton *radio,
                    Export3DControls *controls)
{
    Export3DArgs *args = controls->args;

    args->zscale_type = gwy_radio_buttons_get_current(controls->scales);
    if (args->zscale_type == ZSCALE_AUTO) {
        args->zscale = controls->scinfo->zscale_auto;
        update_zscale_entry(controls);
    }
    else if (args->zscale_type == ZSCALE_PHYSICAL) {
        args->zscale = controls->scinfo->zscale_1_1;
        update_zscale_entry(controls);
    }
    update_sensitivity(controls);
}

static void
triang_type_changed(G_GNUC_UNUSED GtkRadioButton *radio,
                    Export3DControls *controls)
{
    Export3DArgs *args = controls->args;

    args->triang_type = gwy_radio_buttons_get_current(controls->triangs);
}

static void
swap_xy_changed(GtkToggleButton *toggle,
                Export3DControls *controls)
{
    Export3DArgs *args = controls->args;

    args->swap_xy = gtk_toggle_button_get_active(toggle);
}

static void
flip_z_changed(GtkToggleButton *toggle,
               Export3DControls *controls)
{
    Export3DArgs *args = controls->args;

    args->flip_z = gtk_toggle_button_get_active(toggle);
}

static void
update_zscale_entry(Export3DControls *controls)
{
    gchar *s = g_strdup_printf("%.9g", controls->args->zscale);

    gtk_entry_set_text(GTK_ENTRY(controls->zscale), s);
    g_free(s);
}

static void
update_sensitivity(Export3DControls *controls)
{
    gboolean sens = (controls->args->zscale_type == ZSCALE_USER);

    gtk_widget_set_sensitive(controls->zscale, sens);
}

static const gchar flip_z_key[]      = "/module/export3d/flip_z";
static const gchar swap_xy_key[]     = "/module/export3d/swap_xy";
static const gchar triang_type_key[] = "/module/export3d/triang_type";
static const gchar zscale_key[]      = "/module/export3d/zscale";
static const gchar zscale_type_key[] = "/module/export3d/zscale_type";

static void
export3d_load_args(GwyContainer *settings, Export3DArgs *args)
{
    *args = export3d_defaults;

    gwy_container_gis_enum_by_name(settings, zscale_type_key,
                                   &args->zscale_type);
    gwy_container_gis_double_by_name(settings, zscale_key, &args->zscale);
    gwy_container_gis_enum_by_name(settings, triang_type_key,
                                   &args->triang_type);
    gwy_container_gis_boolean_by_name(settings, flip_z_key, &args->flip_z);
    gwy_container_gis_boolean_by_name(settings, swap_xy_key, &args->swap_xy);

    args->zscale_type = CLAMP(args->zscale_type, 0, NZSCALES-1);
    args->triang_type = CLAMP(args->triang_type, 0, NTRIANGULATIONS-1);
}

static void
export3d_save_args(GwyContainer *settings, const Export3DArgs *args)
{
    gwy_container_set_enum_by_name(settings, zscale_type_key,
                                   args->zscale_type);
    gwy_container_set_double_by_name(settings, zscale_key, args->zscale);
    gwy_container_set_enum_by_name(settings, triang_type_key,
                                   args->triang_type);
    gwy_container_set_boolean_by_name(settings, flip_z_key, args->flip_z);
    gwy_container_set_boolean_by_name(settings, swap_xy_key, args->swap_xy);
}

/*************************************************************************
 *
 * Import
 *
 *************************************************************************/

static GwyContainer*
formats3d_import(const gchar *filename,
                 GwyRunType mode,
                 GError **error,
                 const gchar *name)
{
    const Format3D *fmt;
    GwyContainer *settings, *container = NULL;
    GwySurface *surface = NULL;
    GwyDataField *image = NULL;
    Import3DArgs args;
    GwySIUnit *xyunit = NULL, *zunit = NULL;
    gint power10xy, power10z;
    gdouble q;
    gchar *buffer = NULL;
    gsize size;
    GError *err = NULL;
    gboolean ok;
    guint k;

    fmt = find_format(name);
    g_return_val_if_fail(fmt, NULL);
    g_return_val_if_fail(fmt->import, NULL);

    if (!g_file_get_contents(filename, &buffer, &size, &err)) {
        err_GET_FILE_CONTENTS(error, &err);
        goto fail;
    }

    surface = fmt->import(buffer, size, error);
    g_free(buffer);

    if (!surface)
        return FALSE;

    if (!surface->n) {
        err_NO_DATA(error);
        g_object_unref(surface);
        goto fail;
    }

    image = check_regular_grid(surface);

    gwy_clear(&args, 1);
    settings = gwy_app_settings_get();
    import3d_load_args(settings, &args);
    if (mode == GWY_RUN_INTERACTIVE) {
        ok = import3d_dialogue(fmt, &args, surface, image);
        import3d_save_args(settings, &args);
        if (!ok) {
            err_CANCELLED(error);
            goto fail;
        }
    }

    container = gwy_container_new();
    xyunit = gwy_si_unit_new_parse(args.xy_units, &power10xy);
    zunit = gwy_si_unit_new_parse(args.z_units, &power10z);

    if (image && args.do_rasterise) {
        gwy_si_unit_assign(gwy_data_field_get_si_unit_xy(image), xyunit);
        if (power10xy) {
            q = pow10(power10xy);
            gwy_data_field_set_xreal(image, q*gwy_data_field_get_xreal(image));
            gwy_data_field_set_yreal(image, q*gwy_data_field_get_yreal(image));
            gwy_data_field_set_xoffset(image,
                                       q*gwy_data_field_get_xoffset(image));
            gwy_data_field_set_yoffset(image,
                                       q*gwy_data_field_get_yoffset(image));
        }

        gwy_si_unit_assign(gwy_data_field_get_si_unit_z(image), zunit);
        if (power10z)
            gwy_data_field_multiply(image, pow10(power10z));

        gwy_container_set_object(container, gwy_app_get_data_key_for_id(0),
                                 image);
        gwy_app_channel_title_fall_back(container, 0);
        gwy_file_channel_import_log_add(container, 0, NULL, filename);
    }
    else {
        gwy_si_unit_assign(gwy_surface_get_si_unit_xy(surface), xyunit);
        if (power10xy) {
            q = pow10(power10xy);
            for (k = 0; k < surface->n; k++) {
                surface->data[k].x *= q;
                surface->data[k].y *= q;
            }
            gwy_surface_invalidate(surface);
        }

        gwy_si_unit_assign(gwy_surface_get_si_unit_z(surface), zunit);
        if (power10z) {
            q = pow10(power10z);
            for (k = 0; k < surface->n; k++)
                surface->data[k].z *= q;
            gwy_surface_invalidate(surface);
        }

        gwy_container_set_object(container, gwy_app_get_surface_key_for_id(0),
                                 surface);
        gwy_app_xyz_title_fall_back(container, 0);
        gwy_file_xyz_import_log_add(container, 0, NULL, filename);
    }

fail:
    g_free(args.xy_units);
    g_free(args.z_units);
    GWY_OBJECT_UNREF(surface);
    GWY_OBJECT_UNREF(image);
    GWY_OBJECT_UNREF(xyunit);
    GWY_OBJECT_UNREF(zunit);

    return container;
}

/* Create a data field directly if the XY positions form a complete regular
 * grid.  */
static GwyDataField*
check_regular_grid(GwySurface *surface)
{
    GwyXY xymin, dxy;
    guint n, xres, yres, k;
    GwyDataField *dfield;
    gdouble *data;
    guint *map;

    n = surface->n;
    if (!(map = gwy_check_regular_2d_grid((const gdouble*)surface->data, 3, n,
                                          -1.0, &xres, &yres, &xymin, &dxy)))
        return NULL;

    dfield = gwy_data_field_new(xres, yres, xres*dxy.x, yres*dxy.y, FALSE);
    data = gwy_data_field_get_data(dfield);
    for (k = 0; k < n; k++)
        data[k] = surface->data[map[k]].z;
    g_free(map);

    gwy_data_field_set_xoffset(dfield, xymin.x);
    gwy_data_field_set_yoffset(dfield, xymin.y);
    gwy_surface_copy_units_to_data_field(surface, dfield);

    return dfield;
}

static gchar
figure_out_comma_fix_char(const gchar *line)
{
    gchar *comma, *end;

    /* Not a number, try again. */
    if (!g_ascii_strtod(line, &end) && end == line)
        return 0;

    /* There are decimal dots => POSIX. */
    if (strchr(line, '.'))
        return ' ';

    /* There are no commas => POSIX. */
    comma = strchr(line, ',');
    if (!comma)
        return ' ';

    /* There are spaces after commas => POSIX. */
    if (g_regex_match_simple(",[ \t]", line, G_REGEX_NO_AUTO_CAPTURE, 0))
        return ' ';

    /* There is a contiguous block of digits and commas => POSIX. */
    if (g_regex_match_simple("[0-9],[0-9]+,[0-9]", line,
                             G_REGEX_NO_AUTO_CAPTURE, 0))
        return ' ';

    /* There are commas and may actually be inside numbers.  Assume the decimal
     * separator is coma. */
    return '.';
}

static gboolean
read_one_point(const gchar *s, GwyXYZ *pt)
{
    gchar *end;

    if (!(pt->x = g_ascii_strtod(s, &end)) && end == s)
        return FALSE;

    s = end;
    while (g_ascii_isspace(*s))
        s++;

    if (!(pt->y = g_ascii_strtod(s, &end)) && end == s)
        return FALSE;

    s = end;
    while (g_ascii_isspace(*s))
        s++;

    if (!(pt->z = g_ascii_strtod(s, &end)) && end == s)
        return FALSE;

    return TRUE;
}

static GwySurface*
import3d_xyz(guchar *buffer, G_GNUC_UNUSED gsize size,
             G_GNUC_UNUSED GError **error)
{
    GwySurface *surface;
    GArray *points;
    gchar *line, *end, *p = buffer;
    char comma_fix_char = 0;
    GwyXYZ pt;

    points = g_array_new(FALSE, FALSE, sizeof(GwyXYZ));
    for (line = gwy_str_next_line(&p); line; line = gwy_str_next_line(&p)) {
        if (!line[0] || line[0] == '#')
            continue;

        if (!comma_fix_char) {
            comma_fix_char = figure_out_comma_fix_char(line);
            if (!comma_fix_char)
                continue;
        }

        for (end = line; *end; end++) {
            if (*end == ';')
                *end = ' ';
            else if (*end == ',')
                *end = comma_fix_char;
        }

        if (read_one_point(line, &pt))
            g_array_append_val(points, pt);
    }

    surface = gwy_surface_new_from_data((GwyXYZ*)points->data, points->len);
    g_array_free(points, TRUE);

    return surface;
}

static GwySurface*
import3d_obj(guchar *buffer, G_GNUC_UNUSED gsize size,
             G_GNUC_UNUSED GError **error)
{
    GwySurface *surface;
    GArray *points;
    gchar *p = buffer, *line;
    GwyXYZ pt;

    points = g_array_new(FALSE, FALSE, sizeof(GwyXYZ));
    while ((line = gwy_str_next_line(&p))) {
        while (*line == ' ' || *line == '\t')
            line++;

        /* We have a fairly simplistic view of vertex lines and we do not
         * care about anything else. */
        if (line[0] != 'v' || (line[1] != ' ' && line[1] != '\t'))
            continue;

        if (read_one_point(line + 2, &pt))
            g_array_append_val(points, pt);
    }

    surface = gwy_surface_new_from_data((GwyXYZ*)points->data, points->len);
    g_array_free(points, TRUE);

    return surface;
}

static guint
point_hash(gconstpointer key)
{
    const guint32 *data = (const guint32*)key;
    guint32 h;

    h = data[0];
    h ^= (data[1] >> 10) | ((data[1] & 0x3ff) << 22);
    h ^= (data[2] >> 22) | ((data[2] & 0x3fffff) << 10);
    return h;
}

static gboolean
point_equal(gconstpointer keya, gconstpointer keyb)
{
    const guint32 *a = (const guint32*)keya;
    const guint32 *b = (const guint32*)keyb;

    return a[0] == b[0] && a[1] == b[1] && a[2] == b[2];
}

static GwySurface*
import3d_stl(guchar *buffer, gsize size, GError **error)
{
    GHashTable *hash;
    GwySurface *surface;
    GwyXYZ *points;
    guint ntri, i, j, npt = 0;
    gdouble ptblock[9];
    const guchar *p;
    GwyXYZ pt;

    if (size < 134 || size % 50 != 34) {
        g_set_error(error, GWY_MODULE_FILE_ERROR, GWY_MODULE_FILE_ERROR_DATA,
                    _("File is truncated."));
        return NULL;
    }

    p = buffer + 80;
    ntri = gwy_get_guint32_le(&p);
    if (err_SIZE_MISMATCH(error, 84 + ntri*50, size, TRUE))
        return NULL;

    /* This is an upper bound. */
    points = g_new(GwyXYZ, 3*ntri);
    hash = g_hash_table_new(point_hash, point_equal);
    for (i = 0; i < ntri; i++) {
        gwy_convert_raw_data(p + 50*i + 3*4, 9, 1,
                             GWY_RAW_DATA_FLOAT, GWY_BYTE_ORDER_LITTLE_ENDIAN,
                             ptblock, 1.0, 0.0);
        for (j = 0; j < 9; j++) {
            if (gwy_isinf(ptblock[j]) || gwy_isnan(ptblock[j])) {
                g_set_error(error, GWY_MODULE_FILE_ERROR,
                            GWY_MODULE_FILE_ERROR_DATA,
                            _("File contains NaNs or infinities."));
                g_hash_table_destroy(hash);
                g_free(points);
                return NULL;
            }
        }
        for (j = 0; j < 3; j++) {
            pt.x = ptblock[0];
            pt.y = ptblock[1];
            pt.z = ptblock[2];
            if (!g_hash_table_lookup(hash, &pt))
                points[npt++] = pt;
        }
    }
    g_hash_table_destroy(hash);

    surface = gwy_surface_new_from_data(points, npt);
    g_free(points);

    return surface;
}

static gboolean
import3d_dialogue(const Format3D *fmt,
                  Import3DArgs *args,
                  GwySurface *surface,
                  GwyDataField *image)
{
    GtkWidget *dialogue, *label;
    GtkTable *table;
    Import3DControls controls;
    gint row, response;
    gchar *title, *s;

    controls.args = args;
    controls.surface = surface;

    title = g_strdup_printf(_("Import %s"), fmt->title);
    dialogue = gtk_dialog_new_with_buttons(title, NULL, 0,
                                           GTK_STOCK_CANCEL,
                                           GTK_RESPONSE_CANCEL,
                                           GTK_STOCK_OK,
                                           GTK_RESPONSE_OK,
                                           NULL);
    g_free(title);
    gtk_dialog_set_default_response(GTK_DIALOG(dialogue), GTK_RESPONSE_OK);
    gwy_help_add_to_file_dialog(GTK_DIALOG(dialogue), GWY_HELP_DEFAULT);
    controls.dialogue = dialogue;

    table = GTK_TABLE(gtk_table_new(6 + !!image, 5, FALSE));
    gtk_table_set_row_spacings(table, 2);
    gtk_table_set_col_spacings(table, 6);
    gtk_container_set_border_width(GTK_CONTAINER(table), 4);
    gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialogue)->vbox), GTK_WIDGET(table),
                       TRUE, TRUE, 0);
    row = 0;

    label = gtk_label_new(_("Number of points:"));
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
    gtk_table_attach(table, label, 0, 1, row, row+1, GTK_FILL, 0, 0, 0);

    s = g_strdup_printf("%u", surface->n);
    label = gtk_label_new(s);
    g_free(s);
    gtk_misc_set_alignment(GTK_MISC(label), 1.0, 0.5);
    gtk_table_attach(table, label, 1, 2, row, row+1, GTK_FILL, 0, 0, 0);
    row++;

    construct_range(table, _("X-range:"), row++,
                    &controls.xmin, &controls.xmax, &controls.xunit);
    construct_range(table, _("Y-range:"), row++,
                    &controls.ymin, &controls.ymax, &controls.yunit);
    construct_range(table, _("Z-range:"), row++,
                    &controls.zmin, &controls.zmax, &controls.zunit);
    gtk_table_set_row_spacing(table, row-1, 8);

    row = construct_units(&controls, table, row);
    g_signal_connect_swapped(controls.xy_units, "changed",
                             G_CALLBACK(xyunits_changed), &controls);
    g_signal_connect_swapped(controls.z_units, "changed",
                             G_CALLBACK(zunits_changed), &controls);
    gtk_entry_set_text(GTK_ENTRY(controls.xy_units), args->xy_units);
    gtk_entry_set_text(GTK_ENTRY(controls.z_units), args->z_units);
    xyunits_changed(&controls, GTK_ENTRY(controls.xy_units));
    zunits_changed(&controls, GTK_ENTRY(controls.z_units));

    if (image) {
        controls.do_rasterise
            = gtk_check_button_new_with_mnemonic(_("Create image _directly "
                                                   "from regular points"));
        gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(controls.do_rasterise),
                                     args->do_rasterise);
        gtk_table_attach(table, controls.do_rasterise,
                         0, 5, row, row+1, GTK_FILL, 0, 0, 0);
        g_signal_connect_swapped(controls.do_rasterise, "toggled",
                                 G_CALLBACK(do_rasterise_changed), &controls);
    }

    gtk_widget_show_all(dialogue);

    do {
        response = gtk_dialog_run(GTK_DIALOG(dialogue));
        switch (response) {
            case GTK_RESPONSE_CANCEL:
            case GTK_RESPONSE_DELETE_EVENT:
            gtk_widget_destroy(dialogue);
            case GTK_RESPONSE_NONE:
            return FALSE;
            break;

            case GTK_RESPONSE_OK:
            break;

            default:
            g_assert_not_reached();
            break;
        }
    } while (response != GTK_RESPONSE_OK);

    gtk_widget_destroy(dialogue);

    return TRUE;
}

static void
construct_range(GtkTable *table, const gchar *name, gint row,
                GtkWidget **from, GtkWidget **to, GtkWidget **unit)
{
    GtkWidget *label;

    label = gtk_label_new(name);
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
    gtk_table_attach(table, label, 0, 1, row, row+1, GTK_FILL, 0, 0, 0);

    *from = label = gtk_label_new(NULL);
    gtk_misc_set_alignment(GTK_MISC(label), 1.0, 0.5);
    gtk_table_attach(table, label, 1, 2, row, row+1, GTK_FILL, 0, 0, 0);

    label = gtk_label_new("–");
    gtk_misc_set_alignment(GTK_MISC(label), 0.5, 0.5);
    gtk_table_attach(table, label, 2, 3, row, row+1, GTK_FILL, 0, 0, 0);

    *to = label = gtk_label_new(NULL);
    gtk_misc_set_alignment(GTK_MISC(label), 1.0, 0.5);
    gtk_table_attach(table, label, 3, 4, row, row+1, GTK_FILL, 0, 0, 0);

    *unit = label = gtk_label_new(NULL);
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
    gtk_table_attach(table, label, 4, 5, row, row+1,
                     GTK_EXPAND | GTK_FILL, 0, 0, 0);
}

static gint
construct_units(Import3DControls *controls,
                GtkTable *table,
                gint row)
{
    Import3DArgs *args = controls->args;
    GtkWidget *label;

    label = gtk_label_new_with_mnemonic(_("_Lateral units:"));
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
    gtk_table_attach(table, label, 0, 1, row, row+1,
                     GTK_EXPAND | GTK_FILL, 0, 0, 0);
    controls->xy_units = gtk_entry_new();
    gtk_label_set_mnemonic_widget(GTK_LABEL(label), controls->xy_units);
    gtk_entry_set_text(GTK_ENTRY(controls->xy_units), args->xy_units);
    gtk_entry_set_width_chars(GTK_ENTRY(controls->xy_units), 6);
    gtk_table_attach(table, controls->xy_units, 1, 4, row, row+1,
                     GTK_EXPAND | GTK_FILL, 0, 0, 0);
    row++;

    label = gtk_label_new_with_mnemonic(_("_Value units:"));
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
    gtk_table_attach(table, label, 0, 1, row, row+1,
                     GTK_EXPAND | GTK_FILL, 0, 0, 0);
    controls->z_units = gtk_entry_new();
    gtk_label_set_mnemonic_widget(GTK_LABEL(label), controls->z_units);
    gtk_entry_set_text(GTK_ENTRY(controls->z_units), args->z_units);
    gtk_entry_set_width_chars(GTK_ENTRY(controls->z_units), 6);
    gtk_table_attach(table, controls->z_units, 1, 4, row, row+1,
                     GTK_EXPAND | GTK_FILL, 0, 0, 0);
    row++;

    return row;
}

static void
update_range_lables(GtkWidget *from, GtkWidget *to, GtkWidget *unit,
                    gdouble min, gdouble max, const gchar *unitstring)
{
    GwySIValueFormat *vf;
    GwySIUnit *siunit;
    gint power10;
    gchar *s;

    siunit = gwy_si_unit_new_parse(unitstring, &power10);
    min *= pow10(power10);
    max *= pow10(power10);
    vf = gwy_si_unit_get_format_with_digits(siunit, GWY_SI_UNIT_FORMAT_VFMARKUP,
                                            MAX(fabs(min), fabs(max)), 3,
                                            NULL);
    s = g_strdup_printf("%.*f", vf->precision, min/vf->magnitude);
    gtk_label_set_markup(GTK_LABEL(from), s);
    g_free(s);
    s = g_strdup_printf("%.*f", vf->precision, max/vf->magnitude);
    gtk_label_set_markup(GTK_LABEL(to), s);
    g_free(s);
    gtk_label_set_markup(GTK_LABEL(unit), vf->units);
    gwy_si_unit_value_format_free(vf);
    g_object_unref(siunit);
}

static void
xyunits_changed(Import3DControls *controls,
                GtkEntry *entry)
{
    Import3DArgs *args = controls->args;
    gdouble xmin, xmax, ymin, ymax;
    gchar *s;

    s = args->xy_units;
    args->xy_units = gtk_editable_get_chars(GTK_EDITABLE(entry), 0, G_MAXINT);
    g_free(s);
    gwy_surface_get_xrange(controls->surface, &xmin, &xmax);
    update_range_lables(controls->xmin, controls->xmax, controls->xunit,
                        xmin, xmax, args->xy_units);
    gwy_surface_get_yrange(controls->surface, &ymin, &ymax);
    update_range_lables(controls->ymin, controls->ymax, controls->yunit,
                        ymin, ymax, args->xy_units);
}

static void
zunits_changed(Import3DControls *controls,
               GtkEntry *entry)
{
    Import3DArgs *args = controls->args;
    gdouble zmin, zmax;
    gchar *s;

    s = args->z_units;
    args->z_units = gtk_editable_get_chars(GTK_EDITABLE(entry), 0, G_MAXINT);
    g_free(s);
    gwy_surface_get_min_max(controls->surface, &zmin, &zmax);
    update_range_lables(controls->zmin, controls->zmax, controls->zunit,
                        zmin, zmax, args->z_units);
}

static void
do_rasterise_changed(Import3DControls *controls,
                     GtkToggleButton *toggle)
{
    controls->args->do_rasterise = gtk_toggle_button_get_active(toggle);
}

static const gchar do_rasterise_key[] = "/module/import3d/do-rasterise";
static const gchar xy_units_key[]     = "/module/import3d/xy-units";
static const gchar z_units_key[]      = "/module/import3d/z-units";

static void
import3d_load_args(GwyContainer *settings, Import3DArgs *args)
{
    *args = import3d_defaults;

    gwy_container_gis_string_by_name(settings, xy_units_key,
                                     (const guchar**)&args->xy_units);
    gwy_container_gis_string_by_name(settings, z_units_key,
                                     (const guchar**)&args->z_units);
    gwy_container_gis_boolean_by_name(settings, do_rasterise_key,
                                      &args->do_rasterise);

    args->xy_units = g_strdup(args->xy_units ? args->xy_units : "");
    args->z_units = g_strdup(args->z_units ? args->z_units : "");
}

static void
import3d_save_args(GwyContainer *settings, const Import3DArgs *args)
{
    gwy_container_set_const_string_by_name(settings, xy_units_key,
                                           args->xy_units);
    gwy_container_set_const_string_by_name(settings, z_units_key,
                                           args->z_units);
    gwy_container_set_boolean_by_name(settings, do_rasterise_key,
                                      args->do_rasterise);
}

/* vim: set cin et ts=4 sw=4 cino=>1s,e0,n0,f0,{0,}0,^0,\:1s,=0,g1s,h0,t0,+1s,c3,(0,u0 : */
