/*
 *  @(#) $Id: calibrate.c 16089 2014-03-30 14:25:07Z yeti-dn $
 *  Copyright (C) 2003-2013 David Necas (Yeti), Petr Klapetek.
 *  E-mail: yeti@gwyddion.net, klapetek@gwyddion.net.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
 */

#include "config.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <libgwyddion/gwymacros.h>
#include <libgwyddion/gwymath.h>
#include <gtk/gtk.h>
#include <libprocess/stats.h>
#include <libgwydgets/gwydgetutils.h>
#include <libgwydgets/gwycombobox.h>
#include <libgwydgets/gwystock.h>
#include <libgwymodule/gwymodule-process.h>
#include <app/gwyapp.h>

#define CALIBRATE_RUN_MODES (GWY_RUN_IMMEDIATE | GWY_RUN_INTERACTIVE)

/* for compatibility checks */
#define EPSILON 1e-6

typedef struct {
    GwyContainer *data;
    gint id;
} GwyDataObjectId;

typedef struct {
    gdouble xratio;
    gdouble yratio;
    gdouble zratio;
    gint xyexponent;
    gint zexponent;
    gboolean square;
    gboolean new_channel;
    gdouble xreal;
    gdouble yreal;
    gdouble zreal;
    gdouble x0;
    gdouble y0;
    gdouble zshift;
    gdouble xorig;
    gdouble yorig;
    gdouble zorig;
    gdouble x0orig;
    gdouble y0orig;
    gint xyorigexp;
    gint zorigexp;
    gint xres;
    gint yres;
    gchar *xyunit;
    gchar *xyunitorig;
    gchar *zunit;
    gchar *zunitorig;
    GwyDataObjectId sizeid;
    GwyDataObjectId targetid;
} CalibrateArgs;

typedef struct {
    CalibrateArgs *args;
    GtkWidget *dialog;
    GtkObject *xratio;
    GtkObject *yratio;
    GtkObject *zratio;
    GtkWidget *xratio_spin;
    GtkWidget *xratio_label;
    GtkWidget *yratio_spin;
    GtkWidget *yratio_label;
    GtkWidget *match_size;
    GtkWidget *size_chooser;
    GtkWidget *xyexponent;
    GtkWidget *zexponent;
    GtkWidget *xpower10;
    GtkWidget *ypower10;
    GtkWidget *zpower10;
    GtkWidget *square;
    GtkWidget *new_channel;
    GtkObject *xreal;
    GtkObject *yreal;
    GtkObject *zreal;
    GtkWidget *xreal_spin;
    GtkWidget *xreal_label;
    GtkWidget *yreal_spin;
    GtkWidget *yreal_label;
    GtkObject *x0;
    GtkObject *y0;
    GtkObject *zshift;
    gboolean in_update;
    GtkWidget *xyunits;
    GtkWidget *zunits;
    GtkWidget *ok;
} CalibrateControls;

static gboolean module_register        (void);
static void     calibrate              (GwyContainer *data,
                                        GwyRunType run);
static gboolean calibrate_dialog       (CalibrateArgs *args,
                                        GwyDataField *dfield);
static void     dialog_reset           (CalibrateControls *controls,
                                        CalibrateArgs *args);
static void     xratio_changed         (GtkAdjustment *adj,
                                        CalibrateControls *controls);
static void     yratio_changed         (GtkAdjustment *adj,
                                        CalibrateControls *controls);
static void     zratio_changed         (GtkAdjustment *adj,
                                        CalibrateControls *controls);
static void     xreal_changed          (GtkAdjustment *adj,
                                        CalibrateControls *controls);
static void     yreal_changed          (GtkAdjustment *adj,
                                        CalibrateControls *controls);
static void     x0_changed             (GtkAdjustment *adj,
                                        CalibrateControls *controls);
static void     y0_changed             (GtkAdjustment *adj,
                                        CalibrateControls *controls);
static void     zshift_changed         (GtkAdjustment *adj,
                                        CalibrateControls *controls);
static void     zreal_changed          (GtkAdjustment *adj,
                                        CalibrateControls *controls);
static void     square_changed         (GtkWidget *check,
                                        CalibrateControls *controls);
static void     new_channel_changed    (GtkWidget *check,
                                        CalibrateControls *controls);
static void     xyexponent_changed     (GtkWidget *combo,
                                        CalibrateControls *controls);
static void     zexponent_changed      (GtkWidget *combo,
                                        CalibrateControls *controls);
static void     calibrate_dialog_update(CalibrateControls *controls,
                                        CalibrateArgs *args);
static void     calibrate_sanitize_args(CalibrateArgs *args);
static void     calibrate_load_args    (GwyContainer *container,
                                        CalibrateArgs *args);
static void     calibrate_save_args    (GwyContainer *container,
                                        CalibrateArgs *args);
static void     units_change           (GtkWidget *button,
                                        CalibrateControls *controls);
static void     match_size_changed     (GtkToggleButton *toggle,
                                        CalibrateControls *controls);
static void     size_channel_changed   (GwyDataChooser *toggle,
                                        CalibrateControls *controls);
static gboolean mould_filter(GwyContainer *data,
             gint id,
             gpointer user_data);
static void     set_combo_from_unit    (GtkWidget *combo,
                                        const gchar *str,
                                        gint basepower);

static const CalibrateArgs calibrate_defaults = {
    1.0,
    1.0,
    1.0,
    -6,
    -6,
    TRUE,
    TRUE,
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0, 0, 0, 0,
    "m", "m", "m", "m",
    { NULL, 0 },
    { NULL, 0 },
};

static GwyModuleInfo module_info = {
    GWY_MODULE_ABI_VERSION,
    &module_register,
    N_("Recalibrates scan lateral dimensions or value range."),
    "Petr Klapetek <klapetek@gwyddion.net>",
    "2.12",
    "David Nečas (Yeti) & Petr Klapetek",
    "2003",
};

GWY_MODULE_QUERY(module_info)

static gboolean
module_register(void)
{
    gwy_process_func_register("calibrate",
                              (GwyProcessFunc)&calibrate,
                              N_("/_Basic Operations/_Dimensions and Units..."),
                              GWY_STOCK_DATA_MEASURE,
                              CALIBRATE_RUN_MODES,
                              GWY_MENU_FLAG_DATA,
                              N_("Change physical dimensions, units "
                                 "or value scale"));

    return TRUE;
}

static void
calibrate(GwyContainer *data, GwyRunType run)
{
    GwyDataField *dfields[3];
    GQuark quarks[3];
    GQuark quark;
    gint oldid, newid;
    CalibrateArgs args;
    gboolean ok;
    GwySIUnit *siunitxy, *siunitz;

    g_return_if_fail(run & CALIBRATE_RUN_MODES);
    gwy_app_data_browser_get_current(GWY_APP_DATA_FIELD, dfields + 0,
                                     GWY_APP_MASK_FIELD, dfields + 1,
                                     GWY_APP_SHOW_FIELD, dfields + 2,
                                     GWY_APP_DATA_FIELD_KEY, quarks + 0,
                                     GWY_APP_MASK_FIELD_KEY, quarks + 1,
                                     GWY_APP_SHOW_FIELD_KEY, quarks + 2,
                                     GWY_APP_DATA_FIELD_ID, &oldid,
                                     0);
    g_return_if_fail(dfields[0]);

    calibrate_load_args(gwy_app_settings_get(), &args);
    args.targetid.data = data;
    args.targetid.id = oldid;
    args.xorig = gwy_data_field_get_xreal(dfields[0]);
    args.yorig = gwy_data_field_get_yreal(dfields[0]);
    args.zorig = gwy_data_field_get_max(dfields[0])
                 - gwy_data_field_get_min(dfields[0]);
    args.xres = gwy_data_field_get_xres(dfields[0]);
    args.yres = gwy_data_field_get_yres(dfields[0]);
    args.x0orig = gwy_data_field_get_xoffset(dfields[0]);
    args.y0orig = gwy_data_field_get_yoffset(dfields[0]);
    args.xyorigexp = 3*floor(log10(args.xorig*args.yorig)/6);
    args.zorigexp = 3*floor(log10(args.zorig)/3);
    args.xreal = args.xratio * args.xorig;
    args.yreal = args.yratio * args.yorig;
    args.zreal = args.zratio * args.zorig;
    args.xyexponent = 3*floor(log10(args.xreal*args.yreal)/6);
    args.zexponent = 3*floor(log10(args.zreal)/3);
    args.x0 = args.x0orig;
    args.y0 = args.y0orig;
    siunitxy = gwy_data_field_get_si_unit_xy(dfields[0]);
    args.xyunitorig = gwy_si_unit_get_string(siunitxy,
                                             GWY_SI_UNIT_FORMAT_VFMARKUP);
    args.xyunit = g_strdup(args.xyunitorig);
    siunitz = gwy_data_field_get_si_unit_z(dfields[0]);
    args.zunitorig = gwy_si_unit_get_string(siunitz,
                                            GWY_SI_UNIT_FORMAT_VFMARKUP);
    args.zunit = g_strdup(args.zunitorig);

    if (run == GWY_RUN_INTERACTIVE) {
        ok = calibrate_dialog(&args, dfields[0]);
        calibrate_save_args(gwy_app_settings_get(), &args);
        if (!ok) {
            g_free(args.xyunit);
            g_free(args.xyunitorig);
            g_free(args.zunit);
            g_free(args.zunitorig);
            return;
        }
    }

    if (!args.new_channel) {
        guint n = 1;
        if (dfields[1]) {
            n++;
            if (dfields[2])
                n++;
        }
        else if (dfields[2]) {
            quarks[1] = quarks[2];
            quarks[2] = 0;
            n++;
        }
        gwy_app_undo_qcheckpointv(data, n, quarks);
    }

    if (args.new_channel)
        dfields[0] = gwy_data_field_duplicate(dfields[0]);

    gwy_data_field_set_xreal(dfields[0], args.xreal);
    gwy_data_field_set_yreal(dfields[0], args.yreal);
    if (args.zratio != 1.0)
        gwy_data_field_multiply(dfields[0], args.zratio);
    if (args.zshift != 0.0)
        gwy_data_field_add(dfields[0], args.zshift);
    gwy_data_field_set_xoffset(dfields[0], args.x0);
    gwy_data_field_set_yoffset(dfields[0], args.y0);
    if (args.xyunit != args.xyunitorig) {
        siunitxy = gwy_data_field_get_si_unit_xy(dfields[0]);
        gwy_si_unit_set_from_string(siunitxy, args.xyunit);
    }
    if (args.zunit != args.zunitorig) {
        siunitz = gwy_data_field_get_si_unit_z(dfields[0]);
        gwy_si_unit_set_from_string(siunitz, args.zunit);
    }

    if (dfields[1]) {
        if (args.new_channel)
            dfields[1] = gwy_data_field_duplicate(dfields[1]);

        gwy_data_field_set_xreal(dfields[1], args.xreal);
        gwy_data_field_set_yreal(dfields[1], args.yreal);
        gwy_data_field_set_xoffset(dfields[1], args.x0);
        gwy_data_field_set_xoffset(dfields[1], args.y0);
        if (args.xyunit != args.xyunitorig) {
            siunitxy = gwy_data_field_get_si_unit_xy(dfields[1]);
            gwy_si_unit_set_from_string(siunitxy, args.xyunit);
        }
    }

    if (dfields[2]) {
        if (args.new_channel)
            dfields[2] = gwy_data_field_duplicate(dfields[2]);

        gwy_data_field_set_xreal(dfields[2], args.xreal);
        gwy_data_field_set_yreal(dfields[2], args.yreal);
        gwy_data_field_set_xoffset(dfields[2], args.x0);
        gwy_data_field_set_xoffset(dfields[2], args.y0);
        if (args.xyunit != args.xyunitorig) {
            siunitxy = gwy_data_field_get_si_unit_xy(dfields[2]);
            gwy_si_unit_set_from_string(siunitxy, args.xyunit);
        }
    }

    if (args.new_channel) {
        newid = gwy_app_data_browser_add_data_field(dfields[0], data, TRUE);
        g_object_unref(dfields[0]);
        gwy_app_sync_data_items(data, data, oldid, newid, FALSE,
                                GWY_DATA_ITEM_GRADIENT,
                                GWY_DATA_ITEM_RANGE,
                                GWY_DATA_ITEM_MASK_COLOR,
                                0);
        if (dfields[1]) {
            quark = gwy_app_get_mask_key_for_id(newid);
            gwy_container_set_object(data, quark, dfields[1]);
            g_object_unref(dfields[1]);
        }
        if (dfields[2]) {
            quark = gwy_app_get_show_key_for_id(newid);
            gwy_container_set_object(data, quark, dfields[2]);
            g_object_unref(dfields[2]);
        }

        gwy_app_set_data_field_title(data, newid, _("Recalibrated Data"));
        gwy_app_channel_log_add(data, oldid, newid, "proc::calibrate", NULL);
    }
    else {
        guint i;

        for (i = 0; i < 3; i++) {
            if (dfields[i])
                gwy_data_field_data_changed(dfields[i]);
        }

        if (args.xratio != 1.0 || args.yratio != 1.0)
            gwy_app_data_clear_selections(data, oldid);

        gwy_app_channel_log_add(data, oldid, oldid, "proc::calibrate", NULL);
    }

    g_free(args.xyunit);
    g_free(args.xyunitorig);
    g_free(args.zunit);
    g_free(args.zunitorig);
}

static gboolean
calibrate_dialog(CalibrateArgs *args,
                 GwyDataField *dfield)
{
    enum { RESPONSE_RESET = 1};
    GtkWidget *dialog, *spin, *table, *label;
    GwySIUnit *unit;
    CalibrateControls controls;
    gint row, response;

    dialog = gtk_dialog_new_with_buttons(_("Dimensions and Units"), NULL, 0,
                                         _("_Reset"), RESPONSE_RESET,
                                         GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
                                         NULL);
    controls.dialog = dialog;

    controls.ok = gtk_dialog_add_button(GTK_DIALOG(dialog), GTK_STOCK_OK, GTK_RESPONSE_OK);
    gtk_dialog_set_default_response(GTK_DIALOG(dialog), GTK_RESPONSE_OK);

    controls.args = args;
    controls.in_update = TRUE;

    table = gtk_table_new(17, 4, FALSE);
    gtk_table_set_row_spacings(GTK_TABLE(table), 2);
    gtk_table_set_col_spacings(GTK_TABLE(table), 6);
    gtk_container_set_border_width(GTK_CONTAINER(table), 4);
    gtk_container_add(GTK_CONTAINER(GTK_DIALOG(dialog)->vbox), table);
    row = 0;

    /***** New Real Dimensions *****/
    label = gwy_label_new_header(_("New Real Dimensions"));
    gtk_table_attach(GTK_TABLE(table), label,
                     0, 3, row, row+1, GTK_EXPAND | GTK_FILL, 0, 0, 0);
    row++;

    controls.match_size
        = gtk_check_button_new_with_mnemonic(_("_Match pixel size:"));
    gtk_table_attach(GTK_TABLE(table), controls.match_size,
                     0, 1, row, row+1, GTK_EXPAND | GTK_FILL, 0, 0, 0);
    g_signal_connect(controls.match_size, "toggled",
                     G_CALLBACK(match_size_changed), &controls);

    controls.size_chooser = gwy_data_chooser_new_channels();
    gtk_widget_set_sensitive(controls.size_chooser, FALSE);
    gwy_data_chooser_set_filter(GWY_DATA_CHOOSER(controls.size_chooser),
                                mould_filter, &args->targetid, NULL);
    gtk_table_attach_defaults(GTK_TABLE(table), controls.size_chooser,
                              1, 3, row, row+1);
    if (gwy_data_chooser_get_active(GWY_DATA_CHOOSER(controls.size_chooser),
                                    NULL)) {
        g_signal_connect(controls.size_chooser, "changed",
                         G_CALLBACK(size_channel_changed), &controls);
    }
    else {
        gtk_widget_set_sensitive(controls.match_size, FALSE);
        gtk_widget_set_no_show_all(controls.match_size, TRUE);
        gtk_widget_set_no_show_all(controls.size_chooser, TRUE);
    }
    row++;

    label = gtk_label_new_with_mnemonic(_("_X range:"));
    controls.xreal_label = label;
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
    gtk_table_attach(GTK_TABLE(table), label,
                     0, 1, row, row+1, GTK_EXPAND | GTK_FILL, 0, 0, 0);

    controls.xreal = gtk_adjustment_new(args->xreal/pow10(args->xyexponent),
                                        0.01, 10000, 1, 10, 0);
    spin = gtk_spin_button_new(GTK_ADJUSTMENT(controls.xreal), 1, 2);
    controls.xreal_spin = spin;
    gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(spin), TRUE);
    gtk_table_attach(GTK_TABLE(table), spin,
                     1, 2, row, row+1, GTK_EXPAND | GTK_FILL, 0, 0, 0);

    unit = gwy_data_field_get_si_unit_xy(dfield);
    controls.xyexponent
        = gwy_combo_box_metric_unit_new(G_CALLBACK(xyexponent_changed),
                                        &controls, -15, 6, unit,
                                        args->xyexponent);
    gtk_table_attach(GTK_TABLE(table), controls.xyexponent, 2, 3, row, row+2,
                     GTK_EXPAND | GTK_FILL | GTK_SHRINK, 0, 0, 0);

    controls.xyunits = gtk_button_new_with_label(gwy_sgettext("verb|Change"));
    g_object_set_data(G_OBJECT(controls.xyunits), "id", (gpointer)"xy");
    gtk_table_attach(GTK_TABLE(table), controls.xyunits,
                     3, 4, row, row+2,
                     GTK_EXPAND | GTK_FILL | GTK_SHRINK, 0, 0, 0);
    row++;

    label = gtk_label_new_with_mnemonic(_("_Y range:"));
    controls.yreal_label = label;
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
    gtk_table_attach(GTK_TABLE(table), label,
                     0, 1, row, row+1, GTK_EXPAND | GTK_FILL, 0, 0, 0);

    controls.yreal = gtk_adjustment_new(args->yreal/pow10(args->xyexponent),
                                        0.01, 10000, 1, 10, 0);
    spin = gtk_spin_button_new(GTK_ADJUSTMENT(controls.yreal), 1, 2);
    controls.yreal_spin = spin;
    gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(spin), TRUE);
    gtk_table_attach(GTK_TABLE(table), spin,
                     1, 2, row, row+1, GTK_EXPAND | GTK_FILL, 0, 0, 0);
    row++;

    controls.square
        = gtk_check_button_new_with_mnemonic(_("_Square samples"));
    gtk_table_attach(GTK_TABLE(table), controls.square,
                     0, 3, row, row+1, GTK_EXPAND | GTK_FILL, 0, 0, 0);
    row++;

    /***** Lateral Offsets *****/

    label = gtk_label_new_with_mnemonic(_("_X offset:"));
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
    gtk_table_attach(GTK_TABLE(table), label,
                     0, 1, row, row+1, GTK_EXPAND | GTK_FILL, 0, 0, 0);

    controls.x0 = gtk_adjustment_new(args->x0/pow10(args->xyexponent),
                                     -10000, 10000, 1, 10, 0);
    spin = gtk_spin_button_new(GTK_ADJUSTMENT(controls.x0), 1, 2);
    gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(spin), TRUE);
    gtk_table_attach(GTK_TABLE(table), spin,
                     1, 2, row, row+1, GTK_EXPAND | GTK_FILL, 0, 0, 0);
    row++;

    label = gtk_label_new_with_mnemonic(_("_Y offset:"));
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
    gtk_table_attach(GTK_TABLE(table), label,
                     0, 1, row, row+1, GTK_EXPAND | GTK_FILL, 0, 0, 0);

    controls.y0 = gtk_adjustment_new(args->y0/pow10(args->xyexponent),
                                     -10000, 10000, 1, 10, 0);
    spin = gtk_spin_button_new(GTK_ADJUSTMENT(controls.y0), 1, 2);
    gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(spin), TRUE);
    gtk_table_attach(GTK_TABLE(table), spin,
                     1, 2, row, row+1, GTK_EXPAND | GTK_FILL, 0, 0, 0);
    gtk_table_set_row_spacing(GTK_TABLE(table), row, 8);
    row++;

    /***** Value Range *****/
    label = gwy_label_new_header(_("Value Range"));
    gtk_table_attach(GTK_TABLE(table), label,
                     0, 3, row, row+1, GTK_EXPAND | GTK_FILL, 0, 0, 0);
    row++;

    label = gtk_label_new_with_mnemonic(_("_Z range:"));
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
    gtk_table_attach(GTK_TABLE(table), label,
                     0, 1, row, row+1, GTK_EXPAND | GTK_FILL, 0, 0, 0);

    controls.zreal = gtk_adjustment_new(args->zreal/pow10(args->zexponent),
                                        -10000, 10000, 1, 10, 0);
    spin = gtk_spin_button_new(GTK_ADJUSTMENT(controls.zreal), 1, 2);
    gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(spin), TRUE);
    gtk_table_attach(GTK_TABLE(table), spin,
                     1, 2, row, row+1, GTK_EXPAND | GTK_FILL, 0, 0, 0);

    unit = gwy_data_field_get_si_unit_z(dfield);
    controls.zexponent
        = gwy_combo_box_metric_unit_new(G_CALLBACK(zexponent_changed),
                                        &controls, -15, 6, unit,
                                        args->zexponent);
    gtk_table_attach(GTK_TABLE(table), controls.zexponent, 2, 3, row, row+1,
                     GTK_EXPAND | GTK_FILL | GTK_SHRINK, 0, 0, 0);

    controls.zunits = gtk_button_new_with_label(gwy_sgettext("verb|Change"));
    g_object_set_data(G_OBJECT(controls.zunits), "id", (gpointer)"z");
    gtk_table_attach(GTK_TABLE(table), controls.zunits,
                     3, 4, row, row+1,
                     GTK_EXPAND | GTK_FILL | GTK_SHRINK, 0, 0, 0);
    row++;

    /***** Value Shift *****/

    label = gtk_label_new_with_mnemonic(_("Z shi_ft:"));
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
    gtk_table_attach(GTK_TABLE(table), label,
                     0, 1, row, row+1, GTK_EXPAND | GTK_FILL, 0, 0, 0);

    controls.zshift = gtk_adjustment_new(args->zshift/pow10(args->zexponent),
                                         -10000, 10000, 1, 10, 0);
    spin = gtk_spin_button_new(GTK_ADJUSTMENT(controls.zshift), 1, 2);
    gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(spin), TRUE);
    gtk_table_attach(GTK_TABLE(table), spin,
                     1, 2, row, row+1, GTK_EXPAND | GTK_FILL, 0, 0, 0);
    gtk_table_set_row_spacing(GTK_TABLE(table), row, 8);
    row++;

    /***** Calibration Coefficients *****/
    label = gwy_label_new_header(_("Calibration Coefficients"));
    gtk_table_attach(GTK_TABLE(table), label,
                     0, 3, row, row+1, GTK_EXPAND | GTK_FILL, 0, 0, 0);
    row++;

    controls.xratio = gtk_adjustment_new(args->xratio, 0.001, 1000, 0.1, 1, 0);
    spin = gwy_table_attach_spinbutton(table, row,
                                       _("_X calibration factor:"), "",
                                       controls.xratio);
    controls.xratio_spin = spin;
    controls.xratio_label = gwy_table_get_child_widget(table, row, 0);
    controls.xpower10 = gwy_table_get_child_widget(table, row, 2);
    gtk_spin_button_set_digits(GTK_SPIN_BUTTON(spin), 3);
    row++;

    controls.yratio = gtk_adjustment_new(args->yratio, 0.001, 1000, 0.1, 1, 0);
    spin = gwy_table_attach_spinbutton(table, row,
                                       _("_Y calibration factor:"), "",
                                       controls.yratio);
    controls.yratio_spin = spin;
    controls.yratio_label = gwy_table_get_child_widget(table, row, 0);
    controls.ypower10 = gwy_table_get_child_widget(table, row, 2);
    gtk_spin_button_set_digits(GTK_SPIN_BUTTON(spin), 3);
    row++;

    controls.zratio = gtk_adjustment_new(args->zratio, -1000, 1000, 0.1, 1, 0);
    spin = gwy_table_attach_spinbutton(table, row,
                                       _("_Z calibration factor:"), "",
                                       controls.zratio);
    controls.zpower10 = gwy_table_get_child_widget(table, row, 2);
    gtk_spin_button_set_digits(GTK_SPIN_BUTTON(spin), 3);
    row++;

    /***** Options *****/
    label = gwy_label_new_header(_("Options"));
    gtk_table_attach(GTK_TABLE(table), label,
                     0, 3, row, row+1, GTK_EXPAND | GTK_FILL, 0, 0, 0);
    row++;

    controls.new_channel
        = gtk_check_button_new_with_mnemonic(_("Create new channel"));
    gtk_table_attach(GTK_TABLE(table), controls.new_channel,
                     0, 3, row, row+1, GTK_EXPAND | GTK_FILL, 0, 0, 0);
    row++;

    g_signal_connect(controls.xreal, "value-changed",
                     G_CALLBACK(xreal_changed), &controls);
    g_signal_connect(controls.yreal, "value-changed",
                     G_CALLBACK(yreal_changed), &controls);
    g_signal_connect(controls.xyunits, "clicked",
                     G_CALLBACK(units_change), &controls);
    g_signal_connect(controls.zunits, "clicked",
                     G_CALLBACK(units_change), &controls);
    g_signal_connect(controls.x0, "value-changed",
                     G_CALLBACK(x0_changed), &controls);
    g_signal_connect(controls.y0, "value-changed",
                     G_CALLBACK(y0_changed), &controls);
    g_signal_connect(controls.zshift, "value-changed",
                     G_CALLBACK(zshift_changed), &controls);
    g_signal_connect(controls.zreal, "value-changed",
                     G_CALLBACK(zreal_changed), &controls);
    g_signal_connect(controls.xratio, "value-changed",
                     G_CALLBACK(xratio_changed), &controls);
    g_signal_connect(controls.yratio, "value-changed",
                     G_CALLBACK(yratio_changed), &controls);
    g_signal_connect(controls.zratio, "value-changed",
                     G_CALLBACK(zratio_changed), &controls);
    g_signal_connect(controls.square, "toggled",
                     G_CALLBACK(square_changed), &controls);
    g_signal_connect(controls.new_channel, "toggled",
                     G_CALLBACK(new_channel_changed), &controls);

    controls.in_update = FALSE;
    /* sync all fields */
    calibrate_dialog_update(&controls, args);

    gtk_widget_show_all(dialog);
    do {
        response = gtk_dialog_run(GTK_DIALOG(dialog));
        switch (response) {
            case GTK_RESPONSE_CANCEL:
            case GTK_RESPONSE_DELETE_EVENT:
            gtk_widget_destroy(dialog);
            case GTK_RESPONSE_NONE:
            return FALSE;
            break;

            case GTK_RESPONSE_OK:
            break;

            case RESPONSE_RESET:
            dialog_reset(&controls, args);
            break;

            default:
            g_assert_not_reached();
            break;
        }
    } while (response != GTK_RESPONSE_OK);

    gtk_widget_destroy(dialog);

    return TRUE;
}

static void
dialog_reset(CalibrateControls *controls,
             CalibrateArgs *args)
{
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(controls->square),
                                 calibrate_defaults.square);
    /* Don't reset this one
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(controls->new_channel),
                                 calibrate_defaults.new_channel);
                                 */

    set_combo_from_unit(controls->xyexponent,
                        args->xyunitorig, args->xyorigexp);
    gwy_enum_combo_box_set_active(GTK_COMBO_BOX(controls->xyexponent),
                                  args->xyorigexp);
    gtk_adjustment_set_value(GTK_ADJUSTMENT(controls->xreal),
                             args->xorig/pow10(args->xyorigexp));
    gtk_adjustment_set_value(GTK_ADJUSTMENT(controls->yreal),
                             args->yorig/pow10(args->xyorigexp));
    gtk_adjustment_set_value(GTK_ADJUSTMENT(controls->yratio),
                             calibrate_defaults.yratio);
    gtk_adjustment_set_value(GTK_ADJUSTMENT(controls->xratio),
                             calibrate_defaults.xratio);

    set_combo_from_unit(controls->zexponent,
                        args->zunitorig, args->zorigexp);
    gwy_enum_combo_box_set_active(GTK_COMBO_BOX(controls->zexponent),
                                  args->zorigexp);
    gtk_adjustment_set_value(GTK_ADJUSTMENT(controls->zreal),
                             args->zorig/pow10(args->zorigexp));
    gtk_adjustment_set_value(GTK_ADJUSTMENT(controls->zratio),
                             calibrate_defaults.zratio);

    gtk_adjustment_set_value(GTK_ADJUSTMENT(controls->x0),
                             args->x0orig/pow10(args->xyexponent));
    gtk_adjustment_set_value(GTK_ADJUSTMENT(controls->y0),
                             args->y0orig/pow10(args->xyexponent));
    gtk_adjustment_set_value(GTK_ADJUSTMENT(controls->zshift),
                             calibrate_defaults.zshift);

    calibrate_dialog_update(controls, args);
}

static void
xratio_changed(GtkAdjustment *adj,
               CalibrateControls *controls)
{
    CalibrateArgs *args = controls->args;

    if (controls->in_update)
        return;

    controls->in_update = TRUE;
    args->xratio = gtk_adjustment_get_value(adj)
                   * pow10(args->xyexponent - args->xyorigexp);
    args->xreal = args->xratio * args->xorig;
    if (args->square) {
        args->yreal = args->xreal/args->xres * args->yres;
        args->yratio = args->yreal/args->yorig;
    }
    calibrate_dialog_update(controls, args);
    controls->in_update = FALSE;
}

static void
yratio_changed(GtkAdjustment *adj,
               CalibrateControls *controls)
{
    CalibrateArgs *args = controls->args;

    if (controls->in_update)
        return;

    controls->in_update = TRUE;
    args->yratio = gtk_adjustment_get_value(adj)
                   * pow10(args->xyexponent - args->xyorigexp);
    args->yreal = args->yratio * args->yorig;
    if (args->square) {
        args->xreal = args->yreal/args->yres * args->xres;
        args->xratio = args->xreal/args->xorig;
    }
    calibrate_dialog_update(controls, args);
    controls->in_update = FALSE;
}

static void
zratio_changed(GtkAdjustment *adj,
               CalibrateControls *controls)
{
    CalibrateArgs *args = controls->args;

    if (controls->in_update)
        return;

    controls->in_update = TRUE;
    args->zratio = gtk_adjustment_get_value(adj)
                   * pow10(args->zexponent - args->zorigexp);

    args->zreal = args->zratio * args->zorig;

    calibrate_dialog_update(controls, args);
    controls->in_update = FALSE;
}

static void
xreal_changed(GtkAdjustment *adj,
              CalibrateControls *controls)
{
    CalibrateArgs *args = controls->args;

    if (controls->in_update)
        return;

    controls->in_update = TRUE;
    args->xreal = gtk_adjustment_get_value(adj) * pow10(args->xyexponent);
    args->xratio = args->xreal/args->xorig;
    if (args->square) {
        args->yreal = args->xreal/args->xres * args->yres;
        args->yratio = args->yreal/args->yorig;
    }
    calibrate_dialog_update(controls, args);
    controls->in_update = FALSE;

}

static void
yreal_changed(GtkAdjustment *adj,
              CalibrateControls *controls)
{
    CalibrateArgs *args = controls->args;

    if (controls->in_update)
        return;

    controls->in_update = TRUE;

    args->yreal = gtk_adjustment_get_value(adj) * pow10(args->xyexponent);
    args->yratio = args->yreal/args->yorig;
    if (args->square) {
        args->xreal = args->yreal/args->yres * args->xres;
        args->xratio = args->xreal/args->xorig;
    }
    calibrate_dialog_update(controls, args);
    controls->in_update = FALSE;
}

static void
x0_changed(GtkAdjustment *adj,
           CalibrateControls *controls)
{
    CalibrateArgs *args = controls->args;

    if (controls->in_update)
        return;

    controls->in_update = TRUE;
    args->x0 = gtk_adjustment_get_value(adj) * pow10(args->xyexponent);
    calibrate_dialog_update(controls, args);
    controls->in_update = FALSE;

}

static void
y0_changed(GtkAdjustment *adj,
           CalibrateControls *controls)
{
    CalibrateArgs *args = controls->args;

    if (controls->in_update)
        return;

    controls->in_update = TRUE;

    args->y0 = gtk_adjustment_get_value(adj) * pow10(args->xyexponent);
    calibrate_dialog_update(controls, args);
    controls->in_update = FALSE;
}

static void
zshift_changed(GtkAdjustment *adj,
               CalibrateControls *controls)
{
    CalibrateArgs *args = controls->args;

    if (controls->in_update)
        return;

    controls->in_update = TRUE;

    args->zshift = gtk_adjustment_get_value(adj) * pow10(args->zexponent);
    calibrate_dialog_update(controls, args);
    controls->in_update = FALSE;
}

static void
zreal_changed(GtkAdjustment *adj,
              CalibrateControls *controls)
{
    CalibrateArgs *args = controls->args;

    if (controls->in_update)
        return;

    controls->in_update = TRUE;

    args->zreal = gtk_adjustment_get_value(adj) * pow10(args->zexponent);
    args->zratio = args->zreal/args->zorig;
    calibrate_dialog_update(controls, args);
    controls->in_update = FALSE;
}

static void
square_changed(GtkWidget *check,
               CalibrateControls *controls)
{
    CalibrateArgs *args = controls->args;

    if (controls->in_update)
        return;

    controls->in_update = TRUE;

    args->square
        = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(check));
    if (args->square) {
        args->yreal = args->xreal/args->xres * args->yres;
        args->yratio = args->yreal/args->yorig;
    }
    calibrate_dialog_update(controls, args);
    controls->in_update = FALSE;
}

static void
new_channel_changed(GtkWidget *check,
                    CalibrateControls *controls)
{
    CalibrateArgs *args = controls->args;
    args->new_channel = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(check));
}

static void
xyexponent_changed(GtkWidget *combo,
                   CalibrateControls *controls)
{
    CalibrateArgs *args = controls->args;

    if (controls->in_update)
        return;

    controls->in_update = TRUE;
    args->xyexponent = gwy_enum_combo_box_get_active(GTK_COMBO_BOX(combo));
    args->xreal = gtk_adjustment_get_value(GTK_ADJUSTMENT(controls->xreal))
                  * pow10(args->xyexponent);
    args->x0 = gtk_adjustment_get_value(GTK_ADJUSTMENT(controls->x0))
               * pow10(args->xyexponent);
    args->xratio = args->xreal/args->xorig;
    args->yreal = gtk_adjustment_get_value(GTK_ADJUSTMENT(controls->yreal))
                  * pow10(args->xyexponent);
    args->y0 = gtk_adjustment_get_value(GTK_ADJUSTMENT(controls->y0))
               * pow10(args->xyexponent);
    args->yratio = args->yreal/args->yorig;
    calibrate_dialog_update(controls, args);
    controls->in_update = FALSE;
}

static void
zexponent_changed(GtkWidget *combo,
                  CalibrateControls *controls)
{
    CalibrateArgs *args = controls->args;

    if (controls->in_update)
        return;

    controls->in_update = TRUE;

    args->zexponent = gwy_enum_combo_box_get_active(GTK_COMBO_BOX(combo));
    args->zreal = gtk_adjustment_get_value(GTK_ADJUSTMENT(controls->zreal))
                  * pow10(args->zexponent);
    args->zratio = args->zreal/args->zorig;
    calibrate_dialog_update(controls, args);
    controls->in_update = FALSE;
}

static void
units_change(GtkWidget *button,
             CalibrateControls *controls)
{
    GtkWidget *dialog, *hbox, *label, *entry;
    const gchar *id, *unit;
    gint response;
    CalibrateArgs *args = controls->args;

    if (controls->in_update)
        return;

    controls->in_update = TRUE;

    id = g_object_get_data(G_OBJECT(button), "id");
    dialog = gtk_dialog_new_with_buttons(_("Change Units"),
                                         NULL,
                                         GTK_DIALOG_MODAL
                                         | GTK_DIALOG_NO_SEPARATOR,
                                         GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
                                         GTK_STOCK_OK, GTK_RESPONSE_OK,
                                         NULL);
    gtk_dialog_set_default_response(GTK_DIALOG(dialog), GTK_RESPONSE_OK);

    hbox = gtk_hbox_new(FALSE, 6);
    gtk_container_set_border_width(GTK_CONTAINER(hbox), 4);
    gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), hbox,
                       FALSE, FALSE, 0);

    label = gtk_label_new_with_mnemonic(_("New _units:"));
    gtk_box_pack_start(GTK_BOX(hbox), label, TRUE, TRUE, 0);

    entry = gtk_entry_new();
    gtk_entry_set_activates_default(GTK_ENTRY(entry), TRUE);
    gtk_box_pack_start(GTK_BOX(hbox), entry, TRUE, TRUE, 0);

    gtk_widget_show_all(dialog);
    response = gtk_dialog_run(GTK_DIALOG(dialog));
    if (response != GTK_RESPONSE_OK) {
        gtk_widget_destroy(dialog);
        return;
    }

    unit = gtk_entry_get_text(GTK_ENTRY(entry));
    if (gwy_strequal(id, "xy")) {
        set_combo_from_unit(controls->xyexponent, unit, 0);
        g_free(controls->args->xyunit);
        controls->args->xyunit = g_strdup(unit);
    }
    else if (gwy_strequal(id, "z")) {
        set_combo_from_unit(controls->zexponent, unit, 0);
        g_free(controls->args->zunit);
        controls->args->zunit = g_strdup(unit);
    }

    gtk_widget_destroy(dialog);

    calibrate_dialog_update(controls, args);
    controls->in_update = FALSE;
}

static void
match_size_changed(GtkToggleButton *toggle,
                   CalibrateControls *controls)
{
    gboolean matching = gtk_toggle_button_get_active(toggle);

    gtk_widget_set_sensitive(controls->size_chooser, matching);
    gtk_widget_set_sensitive(controls->xreal_spin, !matching);
    gtk_widget_set_sensitive(controls->xreal_label, !matching);
    gtk_widget_set_sensitive(controls->yreal_spin, !matching);
    gtk_widget_set_sensitive(controls->yreal_label, !matching);
    gtk_widget_set_sensitive(controls->xratio_spin, !matching);
    gtk_widget_set_sensitive(controls->xratio_label, !matching);
    gtk_widget_set_sensitive(controls->yratio_spin, !matching);
    gtk_widget_set_sensitive(controls->yratio_label, !matching);
    gtk_widget_set_sensitive(controls->xyunits, !matching);
    gtk_widget_set_sensitive(controls->xyexponent, !matching);
    gtk_widget_set_sensitive(controls->square, !matching);
    /* This is fixed to sensitive again in size_channel_changed() any channel
     * is selected. */
    gtk_dialog_set_response_sensitive(GTK_DIALOG(controls->dialog),
                                      GTK_RESPONSE_OK, !matching);

    if (matching)
        size_channel_changed(GWY_DATA_CHOOSER(controls->size_chooser),
                             controls);
}

static void
size_channel_changed(GwyDataChooser *chooser,
                     CalibrateControls *controls)
{
    CalibrateArgs *args = controls->args;
    GwyDataField *mould;
    GQuark quark;
    gdouble dx, dy;
    GwySIUnit *unit;

    args->sizeid.data = gwy_data_chooser_get_active(chooser, &args->sizeid.id);
    if (!args->sizeid.data) {
        gtk_dialog_set_response_sensitive(GTK_DIALOG(controls->dialog),
                                          GTK_RESPONSE_OK, FALSE);
        return;
    }

    quark = gwy_app_get_data_key_for_id(args->sizeid.id);
    mould = GWY_DATA_FIELD(gwy_container_get_object(args->sizeid.data, quark));
    dx = gwy_data_field_get_xmeasure(mould);
    dy = gwy_data_field_get_ymeasure(mould);
    unit = gwy_data_field_get_si_unit_xy(mould);

    controls->in_update = TRUE;
    args->xreal = dx * args->xres;
    args->yreal = dy * args->yres;
    args->xratio = args->xreal/args->xorig;
    args->yratio = args->yreal/args->yorig;
    args->xyexponent = 3*floor(log10(args->xreal*args->yreal)/6);
    g_free(args->xyunit);
    args->xyunit = gwy_si_unit_get_string(unit, GWY_SI_UNIT_FORMAT_PLAIN);
    args->square = (fabs(log(dx/dy)) <= EPSILON);

    set_combo_from_unit(controls->xyexponent, args->xyunit, args->xyexponent);
    gwy_enum_combo_box_set_active(GTK_COMBO_BOX(controls->xyexponent),
                                  args->xyexponent);
    calibrate_dialog_update(controls, args);
    controls->in_update = FALSE;
}

static gboolean
mould_filter(GwyContainer *data,
             gint id,
             gpointer user_data)
{
    GwyDataObjectId *object = (GwyDataObjectId*)user_data;
    return data != object->data || id != object->id;
}

static void
set_combo_from_unit(GtkWidget *combo,
                    const gchar *str,
                    gint basepower)
{
    GwySIUnit *unit;
    gint power10;

    unit = gwy_si_unit_new_parse(str, &power10);
    power10 += basepower;
    gwy_combo_box_metric_unit_set_unit(GTK_COMBO_BOX(combo),
                                       power10 - 6, power10 + 6, unit);
    g_object_unref(unit);
}

static void
calibrate_dialog_update(CalibrateControls *controls,
                        CalibrateArgs *args)
{
    gchar buffer[32];
    gint e;

    gtk_adjustment_set_value(GTK_ADJUSTMENT(controls->xreal),
                             args->xreal/pow10(args->xyexponent));
    gtk_adjustment_set_value(GTK_ADJUSTMENT(controls->yreal),
                             args->yreal/pow10(args->xyexponent));
    gtk_adjustment_set_value(GTK_ADJUSTMENT(controls->x0),
                             args->x0/pow10(args->xyexponent));
    gtk_adjustment_set_value(GTK_ADJUSTMENT(controls->y0),
                             args->y0/pow10(args->xyexponent));

    gtk_adjustment_set_value(GTK_ADJUSTMENT(controls->zreal),
                             args->zreal/pow10(args->zexponent));
    gtk_adjustment_set_value(GTK_ADJUSTMENT(controls->zshift),
                             args->zshift/pow10(args->zexponent));

    gtk_adjustment_set_value(GTK_ADJUSTMENT(controls->xratio),
                             args->xratio/pow10(args->xyexponent
                                                - args->xyorigexp));
    gtk_adjustment_set_value(GTK_ADJUSTMENT(controls->yratio),
                             args->yratio/pow10(args->xyexponent
                                                - args->xyorigexp));
    gtk_adjustment_set_value(GTK_ADJUSTMENT(controls->zratio),
                             args->zratio/pow10(args->zexponent
                                                - args->zorigexp));

    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(controls->square),
                                 args->square);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(controls->new_channel),
                                 args->new_channel);

    e = args->xyexponent - args->xyorigexp;
    if (!e)
        buffer[0] = '\0';
    else
        g_snprintf(buffer, sizeof(buffer), "× 10<sup>%d</sup>", e);
    gtk_label_set_markup(GTK_LABEL(controls->xpower10), buffer);
    gtk_label_set_markup(GTK_LABEL(controls->ypower10), buffer);

    e = args->zexponent - args->zorigexp;
    if (!e)
        buffer[0] = '\0';
    else
        g_snprintf(buffer, sizeof(buffer), "× 10<sup>%d</sup>", e);
    gtk_label_set_markup(GTK_LABEL(controls->zpower10), buffer);

    if (args->zratio == 0 || args->zreal == 0)
        gtk_widget_set_sensitive(controls->ok, FALSE);
    else
        gtk_widget_set_sensitive(controls->ok, TRUE);
}

static const gchar xratio_key[]      = "/module/calibrate/xratio";
static const gchar yratio_key[]      = "/module/calibrate/yratio";
static const gchar zratio_key[]      = "/module/calibrate/zratio";
static const gchar zshift_key[]      = "/module/calibrate/zshift";
static const gchar square_key[]      = "/module/calibrate/square";
static const gchar new_channel_key[] = "/module/calibrate/new_channel";

static void
calibrate_sanitize_args(CalibrateArgs *args)
{
    args->xratio = CLAMP(args->xratio, 1e-15, 1e15);
    args->yratio = CLAMP(args->yratio, 1e-15, 1e15);
    args->zratio = CLAMP(args->zratio, 1e-15, 1e15);
    args->zshift = CLAMP(args->zshift, -1e-9, 1e9);
    args->square = !!args->square;
    args->new_channel = !!args->new_channel;
}

static void
calibrate_load_args(GwyContainer *container,
                    CalibrateArgs *args)
{
    *args = calibrate_defaults;

    gwy_container_gis_double_by_name(container, xratio_key, &args->xratio);
    gwy_container_gis_double_by_name(container, yratio_key, &args->yratio);
    gwy_container_gis_double_by_name(container, zratio_key, &args->zratio);
    gwy_container_gis_double_by_name(container, zshift_key, &args->zshift);
    gwy_container_gis_boolean_by_name(container, square_key, &args->square);
    gwy_container_gis_boolean_by_name(container, new_channel_key,
                                      &args->new_channel);
    calibrate_sanitize_args(args);
}

static void
calibrate_save_args(GwyContainer *container,
                    CalibrateArgs *args)
{
    gwy_container_set_double_by_name(container, xratio_key, args->xratio);
    gwy_container_set_double_by_name(container, yratio_key, args->yratio);
    gwy_container_set_double_by_name(container, zratio_key, args->zratio);
    gwy_container_set_double_by_name(container, zshift_key, args->zshift);
    gwy_container_set_boolean_by_name(container, square_key, args->square);
    gwy_container_set_boolean_by_name(container, new_channel_key,
                                      args->new_channel);
}

/* vim: set cin et ts=4 sw=4 cino=>1s,e0,n0,f0,{0,}0,^0,\:1s,=0,g1s,h0,t0,+1s,c3,(0,u0 : */
