/*
 * Copyright 2012 Canonical Ltd.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; version 3.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

import QtQuick 2.0
import QtGraphicalEffects 1.0
import Ubuntu.Components 0.1
import Ubuntu.Components.ListItems 0.1 as ListItem
import Ubuntu.Components.Popups 0.1
import Friends 0.1

Item {
    id: tile
    width: parent.width
    anchors {
        topMargin: units.gu(1)
        leftMargin: units.gu(1)
        bottomMargin: units.gu(1)
        rightMargin: units.gu(1)
    }
    property real detailsOpacity : 0
    property string service: column_0
    property string accountId: column_1
    property string messageId: column_2
    property string stream: column_3
    property string sender: column_4
    property string sender_nick: column_6
    property bool from_me: column_7
    property string timeString: friendsUtils.createTimeString(column_8)
    property string message: column_9
    property string avatar: column_10
    property string url: column_11
    property int likes: column_12 > 0 ? column_12 : liked ? 1 : 0
    property bool liked: column_13
    property string link_picture: column_14
    property string link_name: column_15
    property string link_url: column_16
    property string link_desc: column_17
    property string link_caption: column_18
    property string link_icon: column_19
    property string location: column_20
    property string latitude: column_21
    property string longitude: column_22
    property string likesString: likes > 1 ? i18n.tr("people liked this") : i18n.tr("person liked this")

    Rectangle {
        id: tileRect
        height: status_update_content.height + detailsWrapper.height
        width: parent.width - units.gu(1)
        color: "white"
        //color: detailsOpacity > 0 ? "#f8f8f8" : "white"
        anchors.right: parent.right

        FriendsUtils {
            id: friendsUtils
        }

        FriendsDispatcher {
            id: friends
            onLikeComplete: {
                favoriteSpinner.visible = false;
                if (success) {
                    console.log ("Like succeeded");
                } else {
                    console.log ("Like failed: " + errorMessage);
                }
            }
            onUnlikeComplete: {
                favoriteSpinner.visible = false;
                 if (success) {
                     console.log ("Like succeeded");
                 } else {
                     console.log ("UnLike failed: " + errorMessage);
                 }
             }
            onRetweetComplete: {
                retweetSpinner.visible = false;
                if (success) {
                     console.log ("Retweet completed successfully");
                } else {
                     console.log ("Retweet failed: " + errorMessage);
                }
             }
        }

        UbuntuShape {
            id: avatarImage
            height: units.dp(48)
            width: units.dp(48)
            anchors {
                left: parent.left
                top: parent.top
                leftMargin: units.gu(1)
                topMargin: units.gu(1)
            }
            image: Image {
                source: Qt.resolvedUrl(avatar)
                asynchronous: true
                fillMode: Image.PreserveAspectFit
            }

            MouseArea {
                anchors.fill: avatarImage
                onClicked: {
                    Qt.openUrlExternally(url);
                }
            }
        }

        Image {
            id: privateIcon
            anchors {
                right: avatarImage.right
                bottom: avatarImage.bottom
                bottomMargin: units.dp(3)
                rightMargin: units.dp(3)
            }
            source: "images/private.png"
            asynchronous: true
            visible: stream == "private"
        }

        Image {
            id: replyIcon
            anchors {
                right: avatarImage.right
                bottom: avatarImage.bottom
                bottomMargin: units.dp(3)
                rightMargin: units.dp(3)
            }
            source: "images/replies.png"
            asynchronous: true
            visible: stream == "mentions"
        }

        Image {
            id: serviceIcon
            anchors {
                left: parent.left
                bottom: status_update_content.bottom
                topMargin: units.gu(1)
                leftMargin: units.gu(1)
                bottomMargin: units.gu(1)
            }
            source: "images/"+service+".png"
            width: 16
            height: 16
            asynchronous: true
        }

        Item {
            id: status_update_content
            anchors {
                left: avatarImage.right
                right: parent.right
                top: parent.top
                topMargin: units.gu(1)
                leftMargin: units.gu(1)
                bottomMargin: units.gu(1)
                rightMargin: units.gu(1)
            }
            height: childrenRect.height

            MouseArea {
                anchors.fill: status_update_content
                onClicked: {
                    if (tile.state != 'Details') {
                        tile.state = 'Details';
                    } else {
                        tile.state = 'List';
                    }
                }
            }

            Column {
                spacing: units.gu(1)
                anchors {
                    left: parent.left;
                    right: parent.right
                    bottomMargin: units.gu(2)
                }
                Label {
                    text: sender
                    fontSize: "medium"
                    font.bold: true
                    width: parent.width
                }
                Label {
                    id: messageText
                    text: message
                    visible: message.length > 0
                    wrapMode: Text.WrapAtWordBoundaryOrAnywhere
                    width: parent.width - units.gu(3)
                    linkColor: "gray"
                    onLinkActivated: Qt.openUrlExternally(link)
                }
                Label {
                    id: locationName
                    text: location
                    visible: location.length > 0
                    wrapMode: Text.WrapAtWordBoundaryOrAnywhere
                    width: parent.width - units.gu(3)
                    MouseArea {
                        anchors.fill: locationName
                        onClicked: {
                            if (url.length > 0)
                            {
                                Qt.openUrlExternally(url);
                            }
                        }
                    }
                }
                Label {
                    id: linkName
                    text: link_name
                    visible: link_name.length > 0
                    wrapMode: Text.WrapAtWordBoundaryOrAnywhere
                    width: parent.width - units.gu(3)
                    MouseArea {
                        anchors.fill: linkName
                        onClicked: {
                            if (link_url.length > 0)
                                Qt.openUrlExternally(link_url);

                        }
                    }
                }
                Label {
                    id: linkDesc
                    text: link_desc
                    visible: link_desc.length > 0
                    wrapMode: Text.WrapAtWordBoundaryOrAnywhere
                    width: parent.width - units.gu(3)
                    onLinkActivated: Qt.openUrlExternally(link)
                }

                Item {
                    id: favorite
                    width: parent.width - avatarImage.width
                    height: favoriteIcon.height + units.gu(1)
                    anchors {
                        left: parent.left
                        bottomMargin: units.gu(1)
                    }
                    Image {
                        id: favoriteIcon
                        width: 16
                        height: 16
                        opacity: liked ? 1.0 : 0.1
                        anchors {
                            left: parent.left
                            right: likesLabel.left
                        }
                        source: "images/favorite.png"
                        asynchronous: true
                        MouseArea {
                            anchors.fill: favoriteIcon
                            onClicked: {
                                favoriteSpinner.visible = true;
                                if (liked) {
                                    friends.unlikeAsync(accountId, messageId);
                                }
                                else {
                                    friends.likeAsync(accountId, messageId);
                                }
                            }
                        }
                        ActivityIndicator {
                            id: favoriteSpinner
                            anchors.centerIn: parent
                            width: parent.width
                            visible: false
                            running: visible
                        }
                    }
                    Label {
                        id: likesLabel
                        text: (likes > 0) ? likes + " " + likesString : ""
                        anchors {
                            left: favoriteIcon.right
                        }
                        fontSize: "small"
                        color: "gray"
                    }
                }

                Item {
                    id: repeatIcon
                    anchors {
                        right: parent.right
                        bottom: parent.bottom
                        rightMargin: units.gu(2)
                        bottomMargin: units.gu(1)
                    }

                    opacity: detailsOpacity
                    height: from_me ? 0 : childrenRect.height
                    // FIXME: Use features API instead
                    visible: (service == "twitter" || service == "identica") && !from_me && stream != "private"
                    Image {
                        id: repeatIconImage
                        width: 16
                        height: 16
                        source: "images/share.png"
                        asynchronous: true
                        visible: retweetSpinner.visble ? false : true
                        MouseArea {
                            anchors.fill: parent
                            onClicked: {
                                console.log ("Retweeting " + messageId + " from " + accountId);
                                retweetSpinner.visible = true
                                friends.retweetAsync(accountId, messageId);
                            }
                        }
                        ActivityIndicator {
                            id: retweetSpinner
                            anchors.centerIn: parent
                            width: parent.width
                            visible: false
                            running: visible
                        }
                    }
                }
            }

            Row {
                anchors.right: parent.right
                anchors.rightMargin: units.gu(1)
                anchors.top: parent.top
                spacing: units.gu(1)

                Label {
                    id: time
                    text: timeString
                    fontSize: "small"
                    color: "gray"
                }
            }
        }

        Item {
            id: detailsWrapper
            height: detailsLoader.height + units.gu(1)
            width: parent.width
            opacity: detailsOpacity
            visible:  detailsOpacity > 0 ? true : false

            anchors {
                top: status_update_content.bottom
                left: parent.left
                right: parent.right
            }

            Loader {
                id: detailsLoader
                width: parent.width
                visible: true
            }

            onVisibleChanged: {
                if (visible && detailsLoader.state != Loader.Ready) {
                    detailsLoader.source = "StatusUpdateTileDetails.qml";
                }
            }
        }
    }

    DropShadow {
        id: tileShadow
        anchors.fill: tileRect
        horizontalOffset: 3
        verticalOffset: 3
        radius: 8.0
        samples: 16
        cached: true
        fast: true
        color: "#f0f0f0"
        source: tileRect
    }

    states: [State {
                name: "Details"
                PropertyChanges { target: tile; detailsOpacity: 1; x: 0 }
                StateChangeScript {
                    script: {
                        listView.positionViewAtIndex(listView.indexAt(tile.x, tile.y), ListView.Contain);
                    }
                }

            },
            State {
                name: "List"
                PropertyChanges { target: tile; detailsOpacity: 0; x: 0 }
            }]


    transitions: Transition {
        NumberAnimation {
            duration: 300; properties: "detailsOpacity,x,height,width"
        }
    }
}
