/*
 * Copyright (C) 2013 Canonical, Ltd.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 3.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

import QtQuick 2.0
import Ubuntu.Components 0.1
import Ubuntu.Components.ListItems 0.1 as ListItem
import Ubuntu.OnlineAccounts 0.1
import Friends 0.1

Rectangle {
    id: post
    anchors {
        fill: parent
    }
    color: "#f8f8f8"

    FriendsDispatcher {
        id: friends
        onSendComplete: {
             if (success) {
                 activitySpinner.visible = false;
                 entryArea.text = "";
                 console.log ("Send completed successfully");
                 pageStack.pop();
             } else {
                 activitySpinner.visible = false;
                 console.log ("Send failed: " + errorMessage);
             }
         }
    }

    Item {
        id: entry
        property var inputEntry: entryArea
        property var lastCursorPosition
        anchors {
            top: parent.top
            left: parent.left
            right: parent.right
            topMargin: units.gu(1)
        }
        height: childrenRect.height

        TextArea {
            id: entryArea
            anchors {
                left: parent.left
                leftMargin: units.gu(1)
                right: sendButton.left
                rightMargin: units.gu(1)
                topMargin: units.gu(2)
                bottomMargin: units.gu(1.5)
            }
            focus: true
            font.pixelSize: FontUtils.sizeToPixels("medium")
            font.family: "Ubuntu"
            // send message if return was pressed
            Keys.onReturnPressed: sendButton.clicked(null)
            Keys.onEscapePressed: text = ""
            height: units.gu(8)
            placeholderText: i18n.tr("Compose")
            autoSize: true
            maximumLineCount: 0
            color: text.length < 140 ? "gray" : "red"
            textFormat: TextEdit.PlainText
            contentWidth: width - units.gu(5)
            onHeightChanged: {
                if (height < units.gu(8)) {
                    height = units.gu(8);
                }
            }

            onTextChanged: {
                var enteredText = text.substring(entry.lastCursorPosition, cursorPosition);
                if (enteredText.substring(0,4) == "http") {
                    var shortUrl = friends.urlShorten(enteredText);
                    if (shortUrl.length > 4) {
                        text = text.replace (enteredText, shortUrl);
                        cursorPosition = text.length;
                    }
                }
                entry.lastCursorPosition = cursorPosition;
            }

            Item {
                anchors {
                    right: entryArea.right
                    top: entryArea.top
                    rightMargin: units.gu(1)
                }
                height: units.gu(4)
                width: childrenRect.width
                Text {
                    text: 140 - entryArea.text.length
                    color: "gray"
                    font.family: "Ubuntu"
                    font.bold: true
                    font.pixelSize: FontUtils.sizeToPixels("large")
                    opacity: 0.3
                }
            }
            Image {
                id: clearIcon
                anchors {
                    right: entryArea.right
                    bottom: entryArea.bottom
                    rightMargin: units.gu(1)
                }
                source: "images/clear-search.png"
                opacity: 0.3
                MouseArea {
                    anchors.fill: clearIcon
                    onClicked: {
                        entryArea.text = "";
                    }
                }
            }
        }

        Button {
            id: sendButton
            anchors {
                right: parent.right
                rightMargin: units.gu(1)
                bottom: entryArea.bottom
            }
            width: units.gu(9)
            height: units.gu(4)
            text: i18n.tr("Send")
            enabled: entryArea.text.length < 140
            onClicked: {
                console.log ("Post: " + entryArea.text);
                activitySpinner.visible = true
                for (var i=0; i<accountsModel.count; i++) {
                    if (accountsModel.get(i).sendEnabled) {
                        friends.sendForAccountAsync(accountsModel.get(i).id, entryArea.text);
                    }
                }
            }
        }

        ActivityIndicator {
            id: activitySpinner
            anchors.centerIn: entryArea
            visible: false
            running: visible
        }
    }

    AccountServiceModel {
        id: accounts
        serviceType: "microblogging"
        Component.onCompleted: {
            for (var i=0; i<accounts.count; i++) {
                var displayName = accounts.get(i, "displayName");
                var accountId = accounts.get(i, "accountId");
                var serviceName = accounts.get(i, "serviceName");
                var features = friends.featuresForProtocol(serviceName.toLowerCase().replace(".",""));
                if (features.indexOf("send") > -1) {
                    console.log (serviceName + " Supports send");
                    /* FIXME: we should get the iconName and serviceName from the accountService
                       but I am not sure we can access that from JS */
                    accountsModel.append({
                        "displayName": displayName,
                        "id": accountId,
                        "provider": serviceName,
                        "iconName": serviceName.toLowerCase().replace(".",""),
                        "sendEnabled": true
                    });
                }
            }
        }
    }

    ListModel {
        id: accountsModel
    }

    Item {
        id: accountsArea
        width: parent.width
        anchors {
            top: entry.bottom
            topMargin: units.gu(5)
        }

        Column {
            id: accountsView
            width: parent.width
            spacing: units.gu(1)
            Repeater {
                width: parent.width
                model: accountsModel
                delegate: Item {
                    width: parent.width
                    height: childrenRect.height

                    ListItem.Standard {
                        text: displayName
                        opacity: sw.checked ? 1.0 : 0.3

                        height: visible ? icon.height : 0
                        icon: UbuntuShape {
                            id: icon
                            anchors.rightMargin: units.gu(1)
                            height: 48
                            width: 48
                            image: Image {
                                fillMode: Image.PreserveAspectFit
                                source: "image://gicon/"+iconName
                            }
                        }
                        control {
                            Switch {
                                id: sw
                                checked: sendEnabled
                                onCheckedChanged: {
                                    accountsModel.setProperty(index, "sendEnabled", sw.checked);
                                }
                            }
                        }
                        showDivider: false
                    }
                }
            }
        }
    }
}

