import threading, operator, json, types, sys, mx.DateTime
import gobject, dbus.glib, dbus, dbus.service
from . import microblog, config, resources, urlshorter, gintegration
from traceback import print_exc

try:
  import indicate
except:
  indicate = None

gobject.threads_init()

DBUS_TYPE_BLACKLIST = [
  types.InstanceType,
  types.MethodType,
  types.ClassType]

def dsanitize(data):
  data = data.copy()
  for k, v in data.items():
    if not v and isinstance(v, list):
      data[k] = dbus.Array([], "s")
    elif v == None:
      data[k] = ""
    elif type(v) in DBUS_TYPE_BLACKLIST:
      del data[k]
    elif type(v).__name__ == "DateTime":
      data[k] = int(v.gmtime())
    elif type(v).__name__ == "SRE_Match":
      data[k] = True
    elif isinstance(v, list):
      data[k] = [dbus.Dictionary(dsanitize(item), "sv", variant_level=5)
        if isinstance(item, dict) else item for item in v]
  return data

class Protocol(dbus.service.Object):
  __dbus_object_path__  = None

  def __init__(self, id):
    self.__dbus_object_path__  = "/com/gwibber/Protocol/%s" % id
    
    self.protocol = microblog.PROTOCOLS[id]
    self.info = self.protocol.PROTOCOL_INFO
    self.pid = id 

    self.bus = dbus.SessionBus()
    bus_name = dbus.service.BusName("com.Gwibber", bus=self.bus)
    dbus.service.Object.__init__(self, bus_name, self.__dbus_object_path__)

  @dbus.service.method("com.Gwibber")
  def features(self): return self.info["features"]

  @dbus.service.method("com.Gwibber")
  def name(self): return self.info["name"]

  @dbus.service.method("com.Gwibber")
  def id(self): return self.pid

  @dbus.service.method("com.Gwibber")
  def config(self): return self.info["config"]

class Protocols(dbus.service.Object):
  __dbus_object_path__ = "/com/gwibber/Protocol"

  def __init__(self):
    self.bus = dbus.SessionBus()
    bus_name = dbus.service.BusName("com.Gwibber", bus=self.bus)
    dbus.service.Object.__init__(self, bus_name, self.__dbus_object_path__)
    self.protocols = [Protocol(n) for n in microblog.PROTOCOLS.keys()]

  @dbus.service.method("com.Gwibber")
  def support(self, feature):
    return [p.id() for p in self.protocols if feature in p.features()]

  @dbus.service.method("com.Gwibber", out_signature="a{sa{sv}}")
  def data(self):
    return dict([(p.id(), {
        "name": p.name(),
        "config": p.config(),
        "features": p.features(),
      }) for p in self.protocols])

class Feature(dbus.service.Object):
  __dbus_object_path__ = None

  def __init__(self, opname):
    self.__dbus_object_path__ = "/com/gwibber/Feature/%s" % opname
    self.operation = getattr(microblog.op, opname)
    self.opname = opname
    
    self.props = [
      "enabled", "function", "return_value", "first_only", "stream",
      "search", "icon", "transient", "dynamic", "account_tree", "path"]

    self.bus = dbus.SessionBus()
    bus_name = dbus.service.BusName("com.Gwibber", bus=self.bus)
    dbus.service.Object.__init__(self, bus_name, self.__dbus_object_path__)

  @dbus.service.method("com.Gwibber")
  def id(self): return self.opname

  @dbus.service.method("com.Gwibber", out_signature="a{sv}")
  def properties(self):
    return dict(dict([(p, getattr(self.operation, p) or "")
      for p in self.props]), **{"name": self.opname})

class Features(dbus.service.Object):
  __dbus_object_path__ = "/com/gwibber/Feature"

  def __init__(self):
    self.bus = dbus.SessionBus()
    bus_name = dbus.service.BusName("com.Gwibber", bus=self.bus)
    dbus.service.Object.__init__(self, bus_name, self.__dbus_object_path__)
    self.features = [Feature(n) for n in microblog.FEATURES]

  @dbus.service.method("com.Gwibber", out_signature="a{sa{sv}}")
  def data(self):
    return dict([(o.id(), o.properties()) for o in self.features])

class Messages(dbus.service.Object, microblog.OperationResultHandler):
  __dbus_object_path__ = "/com/gwibber/Messages"
  def __init__(self):
    microblog.OperationResultHandler.__init__(self)
    self.bus = dbus.SessionBus()
    bus_name = dbus.service.BusName("com.Gwibber", bus=self.bus)
    dbus.service.Object.__init__(self, bus_name, "/com/gwibber/Messages")
    self.indicator_items = {}
    self.errors = []
    self.preferences = config.Preferences()

    if indicate:
      self.indicate = indicate.indicate_server_ref_default()
      self.indicate.set_type("message.gwibber")
      self.indicate.set_desktop_file(resources.get_desktop_file())
      self.indicate.connect("server-display", self.on_indicator_activate)
      self.indicate.show()

  def op2dbus(self, o):
    return dsanitize({
        "accountid": o.account["id"],
        "protocol": o.protocol_name,
        "opname": o.opname,
        "source": o.source,
        "path": o.path,
        #"args": dbus.Array(o.args, "v", 5),
        "opid": o.opid,
      })

  @dbus.service.method("com.Gwibber")
  def mark_all_read(self):
    # FIXME: we need to uncomment these lines and mark messages ready when 
    # we can open the replies tab
    for i in list(self.indicator_items.values()): i.hide()
    self.indicator_items = {}

  @dbus.service.method("com.Gwibber")
  def get_messages(self, filters, count):
    data = microblog.OperationResultHandler.get_messages(self, filters, count)
    return dbus.Dictionary({
      "count": dbus.Int32(data["count"]),
      "total": dbus.Int32(data["total"]),
      "messages": dbus.Array([dsanitize(m) for m in data["messages"]], "a{sv}", 5)},
      "sv")

  @dbus.service.method("com.Gwibber", out_signature="aa{sv}")
  def get_errors(self):
    return [dsanitize(e) for e in self.errors]

  @dbus.service.method("com.Gwibber")
  def get_usernames(self):
    return microblog.OperationResultHandler.get_usernames(self)

  @dbus.service.signal("com.Gwibber", signature="a{sv}")
  def finish_loading(self, opdata): pass

  @dbus.service.signal("com.Gwibber")
  def loading_complete(self):
    microblog.OperationResultHandler.loading_complete(self)

  @dbus.service.signal("com.Gwibber")
  def operations_started(self):
    microblog.OperationResultHandler.operations_started(self)

  @dbus.service.signal("com.Gwibber")
  def error(self, opdata, error): pass

  def new_message(self, message):
    microblog.OperationResultHandler.new_message(self, message)
    # only flag a message as new for the indicator or notification of it is 
    # newer than the last focus of the client or if the client has never 
    # store a last_focus_time (to prevent spamming with notifications)
    self.last_focus_time = None
    self.last_focus_time = self.preferences["last_focus_time"]
    if self.last_focus_time:
      if (message["time"] > self.last_focus_time) or \
        (hasattr(message, "is_unread") and message["is_unread"]):
        for m in self.messages.keys():
          if message["gwibber_id"] in m:
            return
        if "is_reply" in message and message["is_reply"]:
          if indicate:
            if message["id"] not in self.indicator_items:
              try:
                indicator = indicate.Indicator()
              except:
                indicator = indicate.IndicatorMessage()
              indicator.connect("user-display", self.on_indicator_reply_activate)
              indicator.set_property("subtype", "im")
              indicator.set_property("sender", message["sender_nick"])
              indicator.set_property("body", message["text"])
              indicator.set_property_time("time", 
                mx.DateTime.DateTimeFromTicks(message["time"]).localtime().ticks())
              self.indicator_items[message["id"]] = indicator
              indicator.show()
        self.show_notification_bubble(message)

  def on_indicator_activate(self, indicator):
    dbus.mainloop.glib.DBusGMainLoop(set_as_default=True)
    client_bus = dbus.SessionBus()
    try:
      client_obj = client_bus.get_object("com.GwibberClient",
        "/com/GwibberClient", follow_name_owner_changes = True,
        introspect = False)
      gw = dbus.Interface(client_obj, "com.GwibberClient")
      gw.focus_client(reply_handler=self.handle_focus_reply, 
                      error_handler=self.handle_focus_error)
    except dbus.DBusException:
      print_exc()


  def on_indicator_reply_activate(self, indicator):
    client_bus = dbus.SessionBus()
    try:
      client_obj = client_bus.get_object("com.GwibberClient", "/com/GwibberClient")
      gw = dbus.Interface(client_obj, "com.GwibberClient")
      gw.show_replies(reply_handler=self.handle_focus_reply,  
                      error_handler=self.handle_focus_error)
      indicator.hide()
    except dbus.DBusException:
      print_exc()

  def show_notification_bubble(self, message):
    if self.preferences["show_notifications"] and \
      gintegration.can_notify and \
      message["username"] != message["sender_nick"]:
        body = microblog.support.xml_escape(message["text"])
        #until image caching is working again, we will post the gwibber icon
        #image = hasattr(message, "image_path") and message["image_path"] or ''
        image = resources.icon("gwibber")
        expire_timeout = 5000
        n = gintegration.notify(message["sender"], body, image, expire_timeout)

  def handle_focus_reply(self):
    pass

  def handle_focus_error(self):
    pass

  def operation_failed(self, o, error):
    microblog.OperationResultHandler.operation_failed(self, o, error)
    err, op = dbus.Dictionary(dsanitize(error), "sv"), self.op2dbus(o)
    self.error(op, err)
    error["op"] = op 
    self.errors.append(error)
  
  def operation_complete(self, o, messages):
    microblog.OperationResultHandler.operation_complete(self, o, messages)
    self.finish_loading(dbus.Dictionary(self.op2dbus(o), "sv", 5))

class Microblog(dbus.service.Object):
  __dbus_object_path__ = "/com/gwibber/Microblog"
  
  def __init__(self, accounts):
    self.bus = dbus.SessionBus()
    bus_name = dbus.service.BusName("com.Gwibber", bus=self.bus)
    dbus.service.Object.__init__(self, bus_name, "/com/gwibber/Microblog")
    
    self.messages = Messages()
    self.opmanager = microblog.OperationManager(accounts, self.messages)
    self.protocols = Protocols()
    self.features = Features()
    self.shortener = URLShorten()
    self.accounts = accounts

  def dbus2op(self, data):
    args = []

    if "args" in data:
      for a in data["args"]:
        if a.startswith("message://"):
          filter = ["/" + a.split("/", 1)[-1].lstrip("/")]
          args.append(self.messages.get_messages(filter, 1)["messages"][0])
        else: args.append(a)
    
    return microblog.Operation(
      getattr(microblog.op, data["opname"]),
      self.accounts.get_account(data["accountid"]),
      data["source"], data["id"], args)

  @dbus.service.method("com.Gwibber")
  def perform_scheduled_operations(self):
    self.opmanager.perform_scheduled_operations()

  @dbus.service.method("com.Gwibber", in_signature="ia{sv}")
  def schedule_operation(self, interval, opdata):
    o = self.dbus2op(opdata)
    self.opmanager.schedule_operation(interval, o)
    if "immediate" in opdata and opdata["immediate"]:
      self.opmanager.operation(o)

  @dbus.service.method("com.Gwibber")
  def unschedule_operation(self, source, id):
    self.opmanager.unschedule_operation(source, id)

  @dbus.service.method("com.Gwibber", out_signature="a{sa{sv}}")
  def protocols(self): return self.protocols.data()

  @dbus.service.method("com.Gwibber", out_signature="a{sa{sv}}")
  def features(self): return self.features.data()

  @dbus.service.method("com.Gwibber", in_signature="a{sv}")
  def operation(self, opdata):
    o = self.dbus2op(opdata)
    self.opmanager.operation(o)

class Controller(dbus.service.Object):
  __dbus_object_path__ = "/com/gwibber/Daemon"

  def __init__(self):
    self.bus = dbus.SessionBus()
    bus_name = dbus.service.BusName("com.Gwibber", bus=self.bus)
    dbus.service.Object.__init__(self, bus_name, "/com/gwibber/Daemon")

    self.server = Microblog(config.Accounts())
    self.mainloop = gobject.MainLoop()
    self.mainloop.run()

  @dbus.service.method("com.Gwibber")
  def shutdown(self):
    self.mainloop.quit()
    sys.exit()

class URLShorten(dbus.service.Object):
  __dbus_object_path__ = "/com/gwibber/URLShorten"

  def __init__(self):
    self.bus = dbus.SessionBus()
    bus_name = dbus.service.BusName("com.Gwibber", bus=self.bus)
    dbus.service.Object.__init__(self, bus_name, self.__dbus_object_path__)

  @dbus.service.method("com.Gwibber")
  def shorten(self, url, service):
    if self.is_short(url): return url
    try:
      s = urlshorter.PROTOCOLS[service].URLShorter()
      return s.short(url)
    except: return url

  @dbus.service.method("com.Gwibber")
  def is_short(self, url):
    for us in urlshorter.PROTOCOLS.values():
      if url.startswith(us.PROTOCOL_INFO["fqdn"]):
        return True
    return False


