from . import op, digg, flickr, brightkite, friendfeed
from . import twitter, jaiku, identica, laconica, qaiku
from . import opencollaboration, facebook

import glib, threading, Queue, time, logging, urllib
import sys, operator, traceback, types, mx.DateTime
from gettext import lgettext as _

PROTOCOLS = {
  #"jaiku": jaiku,
  "digg": digg,
  "twitter": twitter,
  "facebook": facebook,
  "friendfeed": friendfeed,
  "flickr": flickr,
  "identica": identica,
  "laconica": laconica,
  "brightkite": brightkite,
  "qaiku": qaiku,
  "opencollaboration": opencollaboration
}

FEATURES = {
  "send": op.send,
  "send_thread": op.send_thread,
  "reply": op.reply,
  "thread": op.thread,
  "responses": op.responses,
  "receive": op.receive,
  "search": op.search,
  "search_url": op.search_url,
  "tag": op.tag,
  "user_messages": op.user_messages,
  "group": op.group,
  "public": op.public,
  "retweet": op.retweet,
  "delete": op.delete,
  "like": op.like,
  "favorites": op.favorites,
  "private": op.private,
}

INTERVAL_DURATION = 60
THREADPOOL_SIZE = 3
DRY_RUN = False

def is_iterable(i):
  return isinstance(i, list) or isinstance(i, types.GeneratorType)

def features(account):
  return PROTOCOLS[account["protocol"]].PROTOCOL_INFO["features"]

op.features = features

logger = logging.getLogger("gwibber")
logger.setLevel(logging.DEBUG)

class Operation:
  def __init__(self, operation, account, source, id, args=[]):
    self.operation = operation
    self.opname = operation.__name__
    self.path = operation.path
    self.account = account
    self.stream_name = operation.stream or self.opname
    self.protocol_name = account["protocol"]
    self.protocol = PROTOCOLS[self.protocol_name]
    self.opid = "/%s/%s" % (source, id)
    self.client = self.protocol.Client(account)
    self.function = getattr(self.client, operation.function or self.opname, None)
    self.query = args[0] if args else None
    self.source = source
    self.args = args
    self.id = id

  def perform_operation(self, *args):
    return self.function(*(args or self.args))

  def __str__(self):
    return "<%s:%s>" % (self.opname, self.protocol_name)

class OperationThread(threading.Thread):
  def __init__(self, operation_queue, output_result_handler):
    threading.Thread.__init__(self)
    self.operation_queue = operation_queue
    self.output_result_handler = output_result_handler

  def sanitize(self, message):
    m = message.__dict__
    m["account"] = m["account"]["id"]
    m["client"] = ""
    return m

  def perform_operation(self, o):
    if DRY_RUN: return []
    if not o.operation.return_value: o.perform_operation()
    else:
      message_store = {}
      path_format = o.path + "%(timestamp)s_%(messageid)s"

      output = o.perform_operation()
      if not is_iterable(output): output = [output]

      if isinstance(o.query, dict) and "gwibber_path" in o.query:
        query = urllib.quote_plus("message:/" + o.query["gwibber_path"])
      else: query = o.query

      for message in output:
        if not hasattr(message, "time"):
          print "Dropping message with bad time attribute:", message.__dict__
        else:
          id = path_format % {
            "protocol": o.protocol_name,
            "accountid": o.account["id"],
            "stream": o.stream_name,
            "timestamp": int(message.time),
            "messageid": message.id,
            "query": query}
          
          message.gwibber_path = id
          message.gwibber_id = id.split("/")[-1]
          message_store[id] = self.sanitize(message)

      return message_store

  def run(self):
    while True:
      opdata = self.operation_queue.get()
      logger.debug("Performing Op: %s" % opdata)
      try:
        messages = self.perform_operation(opdata)
        glib.idle_add (self.output_result_handler.process_opdata, opdata, messages)
      except Exception as e:
        glib.idle_add (self.output_result_handler.process_opdata,
         opdata, {
          "traceback": traceback.format_exc(),
          "type": type(e).__name__,
          "error": str(e),
          "time": mx.DateTime.utc().ticks()
        })
      self.operation_queue.task_done()
      if self.operation_queue.empty():
        self.operation_queue.join()
        glib.idle_add (self.output_result_handler.process_opdata, "finished", messages)

class OperationResultHandler():
  def __init__(self):
    self.already_finished = False
    self.messages = {}

  def filter_messages(self, filters):
    for path, message in self.messages.items():
      for f in filters:
        if f in path: yield message

  def get_messages(self, filters, count):
    data = list(self.filter_messages(filters))
    data.sort(key=operator.itemgetter("time"), reverse=True)
    if count == 0: return {"count": len(data), "total": len(data), "messages": data}
    else: return {"count": count, "total": len(data), "messages": data[:count]}

  def get_usernames(self, filters=None):
    return dict([
      ((m["sender_nick"], m["account"]),
        dict((i, m[i]) for i in [
          "sender_nick",
          "protocol",
          "account",
          "image",
          "sender"]))
        for m in list(self.messages.values())
          if ((m.account in filters) if filters else True)]).values()

  def loading_complete(self):
    logger.debug("Loading Complete")

  def operations_started(self):
    logger.debug("Starting scheduled operations")
  
  def operation_complete(self, o, messages):
    logger.debug("Finished Op: %s" % o)

  def operation_failed(self, o, error):
    logger.debug("Failed Op: %s - %s" % (o, error["traceback"]))

  def new_message(self, message):
    pass

  def process_opdata (self, opdata, messages):
      if opdata == "finished":
        if not self.already_finished:
          self.already_finished = True
          self.loading_complete()
      elif isinstance(messages, dict) and "error" in messages:
        self.already_finished = False
        self.operation_failed(opdata, messages)
      else:
        self.operations_started()
        self.already_finished = False
        if messages and isinstance(messages, dict):
          for k,v in messages.items():
            if not k in self.messages: self.new_message(v)
          self.messages.update(messages)
        self.operation_complete(opdata, messages)
      return False


class OperationManager():
  def __init__(self, accounts, operation_result_handler):
    self.accounts = accounts
    self.current_interval = 0
    self.scheduled_operations = {}
    self.operation_queue = Queue.Queue()
    self.operation_result_handler = operation_result_handler
    
    for i in range(THREADPOOL_SIZE):
      t = OperationThread(self.operation_queue, self.operation_result_handler)
      t.name = str(i)
      t.daemon = True
      t.start()

  def schedule_operation(self, interval, o):
    logger.debug("Scheduled Op: %s" % o)
    o.interval = interval
    o.glib_source_id = glib.timeout_add_seconds (INTERVAL_DURATION * interval, self.perform_scheduled_operation, o)
    self.scheduled_operations[o.opid] = o

  def unschedule_operation(self, source, id):
    opid = "/%s/%s" % (source, id)
    if opid in self.scheduled_operations:
      glib.source_remove (self.scheduled_operations[opid].glib_source_id)
      del self.scheduled_operations[opid]
    else:
      for i in self.scheduled_operations.keys():
        if i.startswith(opid):
          glib.source_remove (self.scheduled_operations[i].glib_source_id)
          del self.scheduled_operations[i]

  def perform_scheduled_operations(self):
    operations = list(self.scheduled_operations.values())
    if len(operations) == 0:
      logger.debug("No operations scheduled")
    for o in operations:
      self.perform_scheduled_operation(o)

  def perform_scheduled_operation(self, o):
    if o.operation.check(o.account):
      self.operation(o)
    return True

  def operation(self, o):
    logger.debug("Queueing Op: %s" % o)
    self.operation_queue.put(o)

