import gtk, microblog, gintegration, resources

class MessageAction:
  icon = None
  label = None

  @classmethod
  def get_icon_path(self, size=16, use_theme=True):
    return resources.icon(self.icon, size, use_theme)
    
  @classmethod
  def include(self, client, msg):
    return client.can(self.__name__, msg.protocol)

  @classmethod
  def action(self, w, client, msg):
    pass

class reply(MessageAction):
  icon = "mail-reply-sender"
  label = "_Reply"

  @classmethod
  def action(self, w, client, msg):
    client.reply_target = msg
    client.target_bar.set_account(msg.account)
    client.target_bar.show()
    
    if hasattr(msg, "is_private") and msg.is_private:
      client.input.set_text("d %s " % msg.sender_nick)
    else:
      if client.preferences["reply_append_colon"]:
        client.input.set_text("@%s: " % msg.sender_nick)
      else:
        client.input.set_text("@%s " % msg.sender_nick)

    client.input.textview.grab_focus()
    tb = client.input.textview.get_buffer()
    tb.place_cursor(tb.get_end_iter())

class thread(MessageAction):
  icon = "mail-reply-all"
  label = "View reply t_hread"

  @classmethod
  def action(self, w, client, msg):
    tab_label = msg.original_title if hasattr(msg, "original_title") else msg.text
    client.add_transient_stream(msg.account, "thread",
        "message:/" + msg.gwibber_path, "Thread")

class retweet(MessageAction):
  icon = "mail-forward"
  label = "R_etweet"

  @classmethod
  def action(self, w, client, msg):
    if not client.preferences["global_retweet"]:
      client.reply_target = msg
      client.target_bar.set_account(msg.account)
      client.target_bar.show()

    if client.preferences["retweet_style_via"]:
      client.input.set_text("%s (via @%s)" % (msg.text, msg.sender_nick))
    else:
      client.input.set_text(u"\u267a @%s: %s" % (msg.sender_nick, msg.text))

    client.input.textview.grab_focus()
    tb = client.input.textview.get_buffer()
    tb.place_cursor(tb.get_end_iter())

class like(MessageAction):
  icon = "bookmark_add"
  label = "_Like this message"

  @classmethod
  def action(self, w, client, msg):
    msg.account.get_client().like(msg)
    d = gtk.MessageDialog(buttons=gtk.BUTTONS_OK)
    d.set_markup("You have marked this message as liked.")
    if d.run(): d.destroy()

class delete(MessageAction):
  icon = "gtk-delete"
  label = "_Delete this message"

  @classmethod
  def action(self, w, client, msg):
    msg.account.get_client().delete(msg)
    d = gtk.MessageDialog(buttons=gtk.BUTTONS_OK)
    d.set_markup("The message has been deleted.")
    if d.run(): d.destroy()

  @classmethod
  def include(self, client, msg):
    return MessageAction.include(client, msg) and \
      msg.sender_nick == msg.username

class search(MessageAction):
  icon = "gtk-find"
  label = "_Search for a query"

  @classmethod
  def action(self, w, client, query=None):
    pass

class read(MessageAction):
  icon = "mail-read"
  label = "View _Message"

  @classmethod
  def action(self, w, client, msg):
    if hasattr(msg, "url") and msg.url:
      gintegration.load_url(msg.url)

  @classmethod
  def include(self, client, msg):
    return True

class user(MessageAction):
  icon = "face-monkey"
  label = "View user _Profile"
  
  @classmethod
  def action(self, w, client, acct=None, name=None):
    client.add_transient_stream(acct, "user_messages", name)

class menu(MessageAction):
  @classmethod
  def action(self, w, client, msg):
    client.on_message_action_menu(msg)

class tag(MessageAction):
  @classmethod
  def action(self, w, client, query):
    print "Searching for tag", query

class tomboy(MessageAction):
  icon = "tomboy"
  label = "Save to _Tomboy"

  @classmethod
  def action(self, w, client, msg):
    gintegration.create_tomboy_note(
      "%s message from %s at %s\n\n%s\n\nSource: %s" % (
        msg.protocol.capitalize(),
        msg.sender, msg.time, msg.text, msg.url))

  @classmethod
  def include(self, client, msg):
    return gintegration.service_is_running("org.gnome.Tomboy")

class more(MessageAction):
  @classmethod
  def action(self, w, client):
    client.extended_view = True
    client.update_view()

class account(MessageAction):
  @classmethod
  def action(self, w, client, id, action=None):
    if action == "create":
      client.accounts.on_account_create(None, id)
    elif action == "delete":
      acct = client.accounts.get_account(id)
      client.accounts.on_account_delete(acct)
    else:
      acct = client.accounts.get_account(id)
      client.accounts.show_properties_dialog(acct, action)

MENU_ITEMS = [reply, thread, retweet, read, user, like, delete, tomboy]
