/*  GWhere
 *  Copyright (C) 2000  Sbastien LECACHEUR
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#include "gwsupport.h"
#include "gwcatalogmanager.h"

#if defined ( OS_CYGWIN) || defined ( OS_WIN32)
	/* A port for Cygwin OS, the GLib should be define this macro, but it doesn't work on my Cygwin OS */
	#define NAME_MAX 255
#elif defined ( OS_FREEBSD) || defined ( OS_OPENBSD) || defined ( OS_NETBSD)
	/* For NAME_MAX and PATH_MAX */
	#include <sys/syslimits.h>
#endif

#include <gtk/gtk.h>
#include <sys/stat.h>
#include <sys/types.h> /* opendir */
#include <dirent.h> /* opendir */

/* Notes that exist win32 implementation of these librairies. */
#if defined ( OS_LINUX) || defined ( OS_CYGWIN) || defined ( OS_FREEBSD) || defined ( OS_OPENBSD) || defined ( OS_NETBSD)
    #include <pwd.h>
    #include <grp.h>
#elif defined ( OS_WIN32)
/*  Should be used later with an external library (YAPL)
    #include <pwd.h>
*/
    #include <dirent.h>
#endif

#include <time.h>
#include <stdlib.h> /* strtol strtoul strtoull */
#include <stdio.h> /* perror */
#include <string.h> /* strncmp */
#include <errno.h>

/* To open and close files */
#include <fcntl.h>
#include <unistd.h> /* chdir readlink get_current_working_dir_name getcwd */

/* To catch any SIGSEGV restore memory */
#include <signal.h> /* signal */
#include <setjmp.h> /* setjmp longjmp */

#include "data/gwdata.h"
#include "tools/gwtools.h"
#include "res/gwres.h"
#include "gui/gwgui.h"

#include "gwapplicationmanager.h"
#include "gwpluginsmanager.h"
#include "gwguimanager.h"
#include "gwsupermount.h"


#if defined ( OS_WIN32)
	#define FORMAT1_DISK_SHORT "//\n%s:%d:%s:%s:%"##G_GUINT64_FORMAT##":%"##G_GUINT64_FORMAT##":%ld:%s:"
	#define FORMAT1_DISK_LONG "//\n%s:%d:%s:%s:%"##G_GUINT64_FORMAT##":%"##G_GUINT64_FORMAT##":%ld:%s:%s\n"
	#define FORMAT1_FILE_SHORT  "%s:%s:%s:%s:%ld:%"##G_GUINT64_FORMAT##":%ld:%ld:%ld:"
	#define FORMAT1_FILE_LONG "%s:%s:%s:%s:%ld:%"##G_GUINT64_FORMAT##":%ld:%ld:%ld:%s\n"
	#define FORMAT2_DISK_SHORT "//\n%s:%d:%s:%s:%s:%s:%"##G_GUINT64_FORMAT##":%"##G_GUINT64_FORMAT##":%ld:%s:%d:"
	#define FORMAT2_DISK_LONG "//\n%s:%d:%s:%s:%s:%s:%"##G_GUINT64_FORMAT##":%"##G_GUINT64_FORMAT##":%ld:%s:%d:%s\n"
	#define FORMAT2_FILE_SHORT "%s:%s:%s:%s:%ld:%"##G_GUINT64_FORMAT##":%ld:%ld:%ld:%d:"
	#define FORMAT2_FILE_LONG "%s:%s:%s:%s:%ld:%"##G_GUINT64_FORMAT##":%ld:%ld:%ld:%d:%s\n"
#else
	#ifdef __STDC__
//		#define CONCAT3(x,y,z) x##y##z
//		#define CONCAT5(v,w,x,y,z) v##w##x##y##z
		#define CONCAT3(x,y,z) x/**/y/**/z/**/
		#define CONCAT5(v,w,x,y,z) v/**/w/**/x/**/y/**/z/**/
	#else
		#define CONCAT3(x,y,z) x/**/y/**/z/**/
		#define CONCAT5(v,w,x,y,z) v/**/w/**/x/**/y/**/z/**/
	#endif

	#define FORMAT1_DISK_SHORT  CONCAT5("//\n%s:%d:%s:%s:%",G_GUINT64_FORMAT,":%",G_GUINT64_FORMAT,":%ld:%s:")
	#define FORMAT1_DISK_LONG  CONCAT5("//\n%s:%d:%s:%s:%",G_GUINT64_FORMAT,":%",G_GUINT64_FORMAT,":%ld:%s:%s\n")
	#define FORMAT1_FILE_SHORT  CONCAT3("%s:%s:%s:%s:%ld:%",G_GUINT64_FORMAT,":%ld:%ld:%ld:")
	#define FORMAT1_FILE_LONG  CONCAT3("%s:%s:%s:%s:%ld:%",G_GUINT64_FORMAT,":%ld:%ld:%ld:%s\n")
	#define FORMAT2_DISK_SHORT  CONCAT5("//\n%s:%d:%s:%s:%s:%s:%",G_GUINT64_FORMAT,":%",G_GUINT64_FORMAT,":%ld:%s:%d:")
	#define FORMAT2_DISK_LONG  CONCAT5("//\n%s:%d:%s:%s:%s:%s:%",G_GUINT64_FORMAT,":%",G_GUINT64_FORMAT,":%ld:%s:%d:%s\n")
	#define FORMAT2_FILE_SHORT  CONCAT3("%s:%s:%s:%s:%ld:%",G_GUINT64_FORMAT,":%ld:%ld:%ld:%d:")
	#define FORMAT2_FILE_LONG  CONCAT3("%s:%s:%s:%s:%ld:%",G_GUINT64_FORMAT,":%ld:%ld:%ld:%d:%s\n")
#endif


jmp_buf env;


void gw_catalog_manager_rollback_plugin_call ( int signum) {
#ifdef GW_DEBUG_MODE
	gw_application_manager_log_msg ( 0, __FILE__, __LINE__, __PRETTY_FUNCTION__, "catch signum %d",signum);
#endif


	longjmp ( env, 1);
}


gint catalog_load_file ( struct catalog_file_info *catalog_info, GtkCTree *ctree, GtkCTreeNode *parent)
{
	gchar *buf = NULL;
	gchar **tmp;
	gchar *file;
	gzFile zfic;
	GtkCList *clist;
	GtkWindow *window = gw_gui_manager_main_interface_get_main_window ( );
	GtkCList *clist_info = gw_notebook_catalog_get_clist_explorer ( window);
	GtkCTreeNode *root = NULL;
	gchar *text_utf8 = NULL;
	gchar *tmp_text = NULL;
	GWStringBuffer *sb = NULL;
#if GW_DEBUG_TIME
	time_t begin;
	time ( &begin);
#endif


#ifdef GW_DEBUG_MODE
	gw_application_manager_log_msg ( 0, __FILE__, __LINE__, __PRETTY_FUNCTION__, NULL);
#endif

	file = catalog_file_info_get_full_name ( catalog_info);

#if GW_DEBUG_MODE
	gw_application_manager_log_msg ( 0, __FILE__, __LINE__, __PRETTY_FUNCTION__, "file name = %s", file);
#endif

	if ( (zfic = gzopen ( file, "rb")) == NULL )
	{
#if GW_DEBUG_MODE
		gw_application_manager_log_msg ( 0, __FILE__, __LINE__, __PRETTY_FUNCTION__, "cannot open this file (z)");
#endif

		gw_msg_box_create ( window, _( "Load file"), _( "Error : cannot open this file."));

		return -1;
	}

        /* Reads the file version */
	if ( (buf = gw_zfile_readline_sb ( zfic, &sb)) == NULL )
	{
#if GW_DEBUG_MODE
		gw_application_manager_log_msg ( 0, __FILE__, __LINE__, __PRETTY_FUNCTION__, "cannot read the file version");
#endif

		gw_msg_box_create ( window, _( "Load file"), _( "Error : cannot read the file version."));

		if ( gzclose ( zfic) != Z_OK )
		{
			gw_msg_box_create ( window, _( "Load catalog"), _( "Error : cannot close file."));
		}

		return -1;
	}

	/* Checks the file version */
	tmp = g_strsplit ( buf, ":", 0);

	if ( strcmp ( tmp[0], PROJECT_NAME) != 0 )
	{
#if GW_DEBUG_MODE
		gw_application_manager_log_msg ( 0, __FILE__, __LINE__, __PRETTY_FUNCTION__, "it's not a GWhere file");
#endif

		gw_msg_box_create ( window, _( "Load file"), _( "Error : it's not a GWhere file."));
		g_strfreev ( tmp);

		if ( gzclose ( zfic) != Z_OK )
		{
			gw_msg_box_create ( window, _( "Load catalog"), _( "Error : cannot close file."));
		}

		return -1;
	}

	/* Sets the new window title */
	tmp_text = g_strconcat ( PROJECT_NAME, " - ", catalog_file_info_get_full_name ( catalog_info), NULL);
	g_strdup_to_gtk_text ( tmp_text, text_utf8);
	g_free ( tmp_text);
	gtk_window_set_title ( window, text_utf8);
	g_free ( text_utf8);

	/* Freezes the exploration tree */
	clist = GTK_CLIST ( ctree);
	gtk_clist_freeze ( clist);
	gtk_clist_freeze ( clist_info);

	/* Sets file version */
	catalog_file_info_set_version ( catalog_info, g_strdup ( tmp[2]));

	/* Sets the programme maker of the file */
	if ( tmp[3] != NULL )
	{
		catalog_file_info_set_program_maker ( catalog_info, g_strdup ( tmp[3]));
	}
	else
	{
		catalog_file_info_set_program_maker ( catalog_info, g_strconcat ( PROJECT_NAME, " ", tmp[1], NULL));
	}

	/* Checks the file version */
	switch ( strtol ( tmp[2], NULL, 10))
	{
		case 1:	/* Loads the 1st version of GWhere file */
			catalog_load_file_version_1 ( catalog_info, zfic, ctree, parent);

			break;

		case 2:	/* Loads the 2nd version of GWhere file */
			catalog_load_file_version_2 ( catalog_info, zfic, ctree, parent, sb);

			break;

		default:/* This version is unknown */
#if GW_DEBUG_MODE
			gw_application_manager_log_msg ( 0, __FILE__, __LINE__, __PRETTY_FUNCTION__, "this GWhere file version is unknown");
#endif

			gw_msg_box_create ( window, _( "Load file"), _( "Error : this GWhere version file is unknown."));

			break;
	}

	if ( sb != NULL) gw_string_buffer_free ( sb);

	if ( gzclose ( zfic) == -1 )
	{
#if GW_DEBUG_MODE
		gw_application_manager_log_msg ( 0, __FILE__, __LINE__, __PRETTY_FUNCTION__, "cannot close this file after loading");
#endif

		g_strfreev ( tmp);
		gw_msg_box_create ( window, _( "Load file"), _( "Error : cannot close this file after loading."));

		return -1;
	}

	/* Refreshes the exploration tree */
	root = gw_gui_manager_main_interface_get_tree_root ( );
	gtk_ctree_sort_recursive ( ctree, root);
	gtk_ctree_select ( ctree, root);
	gtk_clist_thaw ( clist);
	gtk_clist_thaw ( clist_info);

#if GW_DEBUG_MODE
	gw_application_manager_log_msg ( 0, __FILE__, __LINE__, __PRETTY_FUNCTION__, "select tree's node root");
#endif

	g_strfreev ( tmp);

#if GW_DEBUG_TIME
	gw_application_manager_log_msg ( 0, __FILE__, __LINE__, __PRETTY_FUNCTION__, "Elapsed time to load the catalog: %d secs", time ( NULL) - begin);
#endif

	return 0;
}


gint catalog_load_file_version_1 ( struct catalog_file_info *catalog_info, gzFile fic, GtkCTree *ctree, GtkCTreeNode *parent)
{
	gchar *buf = NULL, *text[1];
	GtkCTreeNode *node = NULL;
	struct disk_data *disk_data = NULL;
	struct disk_info *disk_info = NULL;
	GdkPixmap *pix;
	GdkBitmap *msk;
	GtkCTreeNode *root = gw_gui_manager_main_interface_get_tree_root ( );
	GtkWindow *window = gw_gui_manager_main_interface_get_main_window ( );


#ifdef GW_DEBUG_MODE
	gw_application_manager_log_msg ( 0, __FILE__, __LINE__, __PRETTY_FUNCTION__, NULL);
#endif

	/* Creates catalog root node */
	if ( catalog_file_info_get_title ( catalog_info)!=NULL && strlen(catalog_file_info_get_title ( catalog_info))>0 ) {
		g_strdup_to_gtk_text ( catalog_file_info_get_title ( catalog_info), text[0]);
	} else {
		g_strdup_to_gtk_text ( catalog_file_info_get_name ( catalog_info), text[0]);
	}
	root = gtk_ctree_insert_node ( ctree, NULL, NULL, text, 5, PixCatalog, MskCatalog, PixCatalog, MskCatalog, FALSE, TRUE);
	g_free ( text[0]);
	text[0] = NULL;
	gw_gui_manager_main_interface_set_tree_root ( root);
	gtk_ctree_node_set_row_data_full ( ctree, root, catalog_info, (GtkDestroyNotify)catalog_file_info_free);
	parent = root;

	while ( (buf = gw_zfile_readline ( fic, &buf, -1)) != NULL)
	{
		if ( strcmp ( buf, "//") == 0 )
		{
			g_free ( buf);
			buf = NULL;
			buf = gw_zfile_readline ( fic, &buf, -1);

			/* Creates disk node */
			disk_info = catalog_disk_info_new_from_str ( buf, 1, NULL);
			disk_data = disk_data_new ( );
			disk_data_set_info ( disk_data, disk_info);
			pix = disk_info_get_pixmap ( disk_info);
			msk = disk_info_get_bitmap ( disk_info);
			g_strdup_to_gtk_text ( disk_info_get_name ( disk_info), text[0]);
			node = gtk_ctree_insert_node ( ctree, parent, NULL, text, 5, pix, msk, pix, msk, FALSE, FALSE);
			g_free ( text[0]);
			text[0] = NULL;
			gtk_ctree_node_set_row_data_full ( ctree, node, disk_data, (GtkDestroyNotify)disk_data_free);

			/* Loads disk */
			if ( catalog_load_disk_1 ( fic, ctree, node) == -1 )
			{
				gw_msg_box_create ( window, _( "Load file"), _( "Error : during the loading of disk."));

				if ( buf != NULL )
				{
					g_free ( buf);
					buf = NULL;
				}

				return -1;
			}
		}
		else
		{
			/* Writes descriptions */
		}

		if ( buf != NULL )
		{
			g_free ( buf);
			buf = NULL;
		}
	}

	if ( buf != NULL )
	{
		g_free ( buf);
		buf = NULL;
	}

	return 0;
}


gint catalog_load_file_version_2 ( struct catalog_file_info *catalog_info, gzFile fic, GtkCTree *ctree, GtkCTreeNode *parent, GWStringBuffer *sb)
{
	gchar *text[1];
	gchar *buf = NULL;
	gchar **tmp = NULL;
	GtkCTreeNode *node = NULL;
	struct disk_data *disk_data = NULL;
	struct disk_info *disk_info = NULL;
	struct category *category;
	GdkPixmap *pix;
	GdkBitmap *msk;
	GtkCTreeNode *root = gw_gui_manager_main_interface_get_tree_root ( );
	GtkWindow *window = gw_gui_manager_main_interface_get_main_window ( );
	gchar *tmp_buf = NULL;


#ifdef GW_DEBUG_MODE
	gw_application_manager_log_msg ( 0, __FILE__, __LINE__, __PRETTY_FUNCTION__, NULL);
#endif


	/* Loads catalog informations */
	if ( (buf = gw_zfile_readline_sb ( fic, &sb)) != NULL )
	{
		tmp = g_strsplit ( buf, ":", 0);

		if ( tmp != NULL )
		{
			/* Reads catalog name */
			if ( tmp[0] != NULL)
			{
				if ( strlen ( tmp[0]) > 0 )
				{
					/* Case : the catalog gets a name */
					tmp_buf = gw_file_to_str ( tmp[0]);
					catalog_file_info_set_title ( catalog_info, tmp_buf);
					g_strdup_to_gtk_text ( catalog_file_info_get_title ( catalog_info), text[0]);
				}
				else
				{
					/* Case : default catalog name is file name */
					g_strdup_to_gtk_text ( catalog_file_info_get_name ( catalog_info), text[0]);
				}
			}
			else
			{
				/* Case : default catalog name is file name */
				g_strdup_to_gtk_text ( catalog_file_info_get_name ( catalog_info), text[0]);
			}

			/* Reads catalog description */
			if ( tmp[1] != NULL )
			{
				if ( strlen ( tmp[1]) > 0 )
				{
					tmp_buf = gw_file_to_str ( tmp[1]);
					catalog_file_info_set_description ( catalog_info, tmp_buf);
				}
			}

			g_strfreev ( tmp);
		}
	}

	/* Creates catalog root node */
	root = gtk_ctree_insert_node ( ctree, NULL, NULL, text, 5, PixCatalog, MskCatalog, PixCatalog, MskCatalog, FALSE, TRUE);
	g_free ( text[0]);
	gw_gui_manager_main_interface_set_tree_root ( root);
	gtk_ctree_node_set_row_data_full ( ctree, root, catalog_info, (GtkDestroyNotify)catalog_file_info_free);
	parent = root;

	while ( (buf = gw_zfile_readline_sb ( fic, &sb)) != NULL )
	{
#if GW_DEBUG_MODE
		gw_application_manager_log_msg ( 0, __FILE__, __LINE__, __PRETTY_FUNCTION__, "current line=%s", buf);
#endif
		if ( strcmp ( buf, "//") == 0 )
		{
#if GW_DEBUG_MODE
			gw_application_manager_log_msg ( 0, __FILE__, __LINE__, __PRETTY_FUNCTION__, "loading new disk...");
#endif

			gw_zfile_readline_sb ( fic, &sb);

#if GW_DEBUG_MODE
			gw_application_manager_log_msg ( 0, __FILE__, __LINE__, __PRETTY_FUNCTION__, "current line=%s", buf);
#endif

			/* Creates a disk node */
			disk_info = catalog_disk_info_new_from_str ( buf, 2, catalog_info);
			disk_data = disk_data_new ( );
			disk_data_set_info ( disk_data, disk_info);
			pix = disk_info_get_pixmap ( disk_info);
			msk = disk_info_get_bitmap ( disk_info);
			g_strdup_to_gtk_text ( disk_info_get_name ( disk_info), text[0]);
			node = gtk_ctree_insert_node ( ctree, parent, NULL, text, 5, pix, msk, pix, msk, FALSE, FALSE);
			g_free ( text[0]);
			gtk_ctree_node_set_row_data_full ( ctree, node, disk_data, (GtkDestroyNotify)disk_data_free);

			/* Loads disk */
			if ( catalog_load_disk_2 ( fic, ctree, node, sb) == -1 )
			{
				gw_msg_box_create ( window, _( "Load file"), _( "Error : during the loading of disk."));

				return -1;
			}
		}
		else
		{
			/* Loads categories */
			tmp = g_strsplit ( buf, ":", 2);

			if ( tmp[0] != NULL )
			{
				category = category_new ( );
				tmp_buf = gw_file_to_str ( tmp[0]);
				category_set_name ( category, tmp_buf);

				if ( tmp[1] != NULL )
				{
					if ( strcmp ( tmp[1], "" ) != 0 )
					{
						tmp_buf = gw_file_to_str ( tmp[1]);
						category_set_description ( category, tmp_buf);
					}
				}

				catalog_file_info_append_category ( catalog_info, category);
			}

			g_strfreev ( tmp);
		}
	}

	return 0;
}


gint catalog_load_disk_1 ( gzFile fic, GtkCTree *ctree, GtkCTreeNode *parent)
{
	guint ret, compt = 0;
	gchar *buf = NULL, *text [1];
	struct disk_data *disk_data = NULL;
	struct dir_data *dir_data = NULL;
	struct file_info *file_info = NULL;
	GtkCTreeNode *node = NULL;


#ifdef GW_DEBUG_MODE
	gw_application_manager_log_msg ( 0, __FILE__, __LINE__, __PRETTY_FUNCTION__, NULL);
#endif

	/* Gets disk informations */
	disk_data = gtk_ctree_node_get_row_data ( ctree, parent);

	while ( strcmp ( (buf = gw_zfile_readline ( fic, &buf, -1)), "\\") != 0 )
	{
		if ( strcmp ( buf, "/") == 0 )
		{
			/* Case : is a folder */
			compt++;
			g_free ( buf);
			buf = NULL;
			buf = gw_zfile_readline ( fic, &buf, -1);

			/* Allocates memory for datas structures */
			file_info = catalog_file_info_new_from_str ( buf/*, TRUE*/, 1, NULL);

			/* Creates folder node */
			dir_data = dir_data_new ( );
			dir_data_set_info ( dir_data, file_info);
			g_strdup_to_gtk_text ( file_info_get_name ( file_info), text[0]);
			node = gtk_ctree_insert_node ( ctree, parent, NULL, text, 5, PixFolder, MskFolder, PixOpenFolder, MskOpenFolder, FALSE, FALSE);
			gtk_ctree_node_set_row_data_full ( ctree, node, dir_data, (GtkDestroyNotify)dir_data_free);

			/* Loads folder */
			if ( (ret = catalog_load_dir_1 ( fic, ctree, node)) == 0)
			{
				if ( dir_data_get_num_file(dir_data)>0 ) {
					gtk_ctree_set_node_info ( ctree, node, text[0], 5, PixFolder, MskFolder, PixOpenFolder, MskOpenFolder, TRUE, FALSE);
				} else {
					gtk_ctree_set_node_info ( ctree, node, text[0], 5, PixLockedFolder, MskLockedFolder, PixLockedFolder, MskLockedFolder, TRUE, FALSE);
				}
			} else {
				/* It is not locked */
			}

			g_free ( text[0]);
			text[0] = NULL;
		}
		else
		{
			/* Case : is a file */
			/* Allocates memory for datas structures */
			file_info = catalog_file_info_new_from_str ( buf/*, FALSE*/, 1, NULL);

			/* Adds to files list the file informations */
			disk_data_append_child ( disk_data, file_info);
		}

		if ( buf != NULL )
		{
			g_free ( buf);
			buf = NULL;
		}
	}

	if ( buf != NULL )
	{
		g_free ( buf);
		buf = NULL;
	}

	return compt;
}


gint catalog_load_disk_2 (gzFile fic, GtkCTree *ctree, GtkCTreeNode *parent, GWStringBuffer *sb)
{
	guint ret, compt = 0;
	gchar *text[1];
	gchar *buf = NULL;
	struct catalog_file_info *catalog_info = NULL;
	struct disk_data *disk_data = NULL;
	struct dir_data *dir_data = NULL;
	struct file_info *file_info = NULL;
	GtkCTreeNode *node = NULL;
	GtkCTreeNode *root = gw_gui_manager_main_interface_get_tree_root ( );


#ifdef GW_DEBUG_MODE
	gw_application_manager_log_msg ( 0, __FILE__, __LINE__, __PRETTY_FUNCTION__, NULL);
#endif

	/* Gets catalog informations */
	catalog_info = gtk_ctree_node_get_row_data ( ctree, root);

	/* Gets disk information */
	disk_data = gtk_ctree_node_get_row_data ( ctree, parent);

	while ( strcmp ( (buf = gw_zfile_readline_sb ( fic, &sb)), "\\") != 0 )
	{
		if ( strcmp ( buf, "/") == 0 )
		{
			/* Case : is a folder */
			compt++;
			buf = gw_zfile_readline_sb ( fic, &sb);

			/* Allocates datas structure */
			file_info = catalog_file_info_new_from_str ( buf/*, TRUE*/, 2, catalog_info);

			/* Creates folder node */
			dir_data = dir_data_new ( );
			dir_data_set_info ( dir_data, file_info);
			g_strdup_to_gtk_text ( file_info_get_name ( file_info), text[0]);
			node = gtk_ctree_insert_node ( ctree, parent, NULL, text, 5, PixFolder, MskFolder, PixOpenFolder, MskOpenFolder, FALSE, FALSE);
			gtk_ctree_node_set_row_data_full ( ctree, node, dir_data, (GtkDestroyNotify)dir_data_free);

			/* Loads folder */
			if ( (ret = catalog_load_dir_2 ( fic, ctree, node, sb)) == 0)
			{
				if ( dir_data_get_num_file(dir_data)>0 ) {
					gtk_ctree_set_node_info ( ctree, node, text[0], 5, PixFolder, MskFolder, PixOpenFolder, MskOpenFolder, TRUE, FALSE);
				} else {
					gtk_ctree_set_node_info ( ctree, node, text[0], 5, PixLockedFolder, MskLockedFolder, PixLockedFolder, MskLockedFolder, TRUE, FALSE);
				}
			} else {
				/* It not locked */
			}

			g_free ( text[0]);
		}
		else
		{
			/* Case : is a file */
			/* Allocates datas structure */
			file_info = catalog_file_info_new_from_str ( buf/*, FALSE*/, 2, catalog_info);

			/* Adds to files list the file informations */
			disk_data_append_child ( disk_data, file_info);
		}
	}

	return compt;
}


gint catalog_load_dir_1 ( gzFile fic, GtkCTree *ctree, GtkCTreeNode *parent)
{
	guint ret, compt = 0;
	gchar *buf = NULL, *text [1];
	struct dir_data *dir_data = NULL;
	struct file_info *file_info = NULL;
	GtkCTreeNode *node = NULL;


#ifdef GW_DEBUG_MODE
	gw_application_manager_log_msg ( 0, __FILE__, __LINE__, __PRETTY_FUNCTION__, NULL);
#endif

	/* Gets folder informations. */
	dir_data = gtk_ctree_node_get_row_data ( ctree, parent);

	while ( strcmp ( (buf = gw_zfile_readline ( fic, &buf, -1)), "\\") != 0 )
	{
		if ( strcmp ( buf, "/") == 0 )
		{
			/* Case : is a folder */
			compt++;
			g_free ( buf);
			buf = NULL;
			buf = gw_zfile_readline ( fic, &buf, -1);

			/* Allocates datas structure */
			file_info = catalog_file_info_new_from_str ( buf/*, TRUE*/, 1, NULL);

			/* Creates folder node */
			dir_data = dir_data_new ( );
			dir_data_set_info ( dir_data, file_info);
			g_strdup_to_gtk_text ( file_info_get_name ( file_info), text[0]);
			node = gtk_ctree_insert_node ( ctree, parent, NULL, text, 5, PixFolder, MskFolder, PixOpenFolder, MskOpenFolder, FALSE, FALSE);
			gtk_ctree_node_set_row_data_full ( ctree, node, dir_data, (GtkDestroyNotify)dir_data_free);

			/* Loads folder with a recursive call */
			if ( (ret = catalog_load_dir_1 ( fic, ctree, node)) == 0)
			{
				if ( dir_data_get_num_file(dir_data)>0 ) {
					gtk_ctree_set_node_info ( ctree, node, text[0], 5, PixFolder, MskFolder, PixOpenFolder, MskOpenFolder, TRUE, FALSE);
				} else {
					gtk_ctree_set_node_info ( ctree, node, text[0], 5, PixLockedFolder, MskLockedFolder, PixLockedFolder, MskLockedFolder, TRUE, FALSE);
				}
			} else {
				/* It is not locked */
			}

			g_free ( text[0]);
			text[0] = NULL;
		}
		else
		{
			/* Case : is a file */
			/* Allocates datas structure */
			file_info = catalog_file_info_new_from_str ( buf/*, TRUE*/, 1, NULL);

			/* Adds to files list the file informations */
			dir_data_append_child ( dir_data, file_info);
		}

		if ( buf != NULL )
		{
			g_free ( buf);
			buf = NULL;
		}
	}

	if ( buf != NULL )
	{
		g_free ( buf);
		buf = NULL;
	}

	return compt;
}


gint catalog_load_dir_2 ( gzFile fic, GtkCTree *ctree, GtkCTreeNode *parent, GWStringBuffer *sb)
{
	guint ret, compt = 0;
	gchar *text [1];
	gchar *buf = NULL;
	struct catalog_file_info *catalog_info = NULL;
	struct dir_data *dir_data = NULL;
	struct file_info *file_info = NULL;
	GtkCTreeNode *node = NULL;
	GtkCTreeNode *root = gw_gui_manager_main_interface_get_tree_root ( );


#ifdef GW_DEBUG_MODE
	gw_application_manager_log_msg ( 0, __FILE__, __LINE__, __PRETTY_FUNCTION__, NULL);
#endif

	/* Gets catalog informations */
	catalog_info = gtk_ctree_node_get_row_data ( ctree, root);

	/* Gets folder informations */
	dir_data = gtk_ctree_node_get_row_data ( ctree, parent);

	while ( strcmp ( (buf = gw_zfile_readline_sb ( fic, &sb)), "\\") != 0 )
	{
		if ( strcmp ( buf, "/") == 0 )
		{
			/* Case : is a folder */
			compt++;
			buf = gw_zfile_readline_sb ( fic, &sb);

			/* Allocates datas structures */
			file_info = catalog_file_info_new_from_str ( buf/*, TRUE*/, 2, catalog_info);

			/* Create folder node */
			dir_data = dir_data_new ( );
			dir_data_set_info ( dir_data, file_info);
			g_strdup_to_gtk_text ( file_info_get_name ( file_info), text[0]);
			node = gtk_ctree_insert_node ( ctree, parent, NULL, text, 5, PixFolder, MskFolder, PixOpenFolder, MskOpenFolder, FALSE, FALSE);
			gtk_ctree_node_set_row_data_full ( ctree, node, dir_data, (GtkDestroyNotify)dir_data_free);

			/* Loads folder with a recursive call */
			if ( (ret = catalog_load_dir_2 ( fic, ctree, node, sb)) == 0)
			{
				if ( dir_data_get_num_file(dir_data)>0 ) {
					gtk_ctree_set_node_info ( ctree, node, text[0], 5, PixFolder, MskFolder, PixOpenFolder, MskOpenFolder, TRUE, FALSE);
				} else {
					gtk_ctree_set_node_info ( ctree, node, text[0], 5, PixLockedFolder, MskLockedFolder, PixLockedFolder, MskLockedFolder, TRUE, FALSE);
				}
			} else {
				/* It is not locked */
			}

			g_free ( text[0]);
		}
		else
		{
			/* Case : is a file */
			/* Allocates datas structure */
			file_info = catalog_file_info_new_from_str ( buf/*, FALSE*/, 2, catalog_info);

			/* Adds to files list the file informations */
			dir_data_append_child ( dir_data, file_info);
		}
	}

	return compt;
}


gint catalog_add_disk ( struct scan_disk_info *scan_disk, GtkCTree *ctree, GtkCTreeNode *parent)
{
	GtkCTreeNode *node = NULL;
	gchar *text [1];
	gchar past_dir[512];
	struct disk_info * disk_info = NULL;
	struct disk_data * disk_data = NULL;
	struct dir_info *dir_info = NULL;
	struct catalog_file_info *file_info = NULL;
	gulong date;
	GdkPixmap *pix;
	GdkBitmap *msk;
	gchar *error_msg = NULL;
	struct vfs_stats *vfs = NULL;
	struct category *category = NULL;
	guint64 disk_size = G_GINT64_CONSTANT ( 0);
	GList *forbidden = NULL, *self = NULL, *current = NULL;
	gint i = 0;


#if GW_DEBUG_TIME
	time_t begin;
	time ( &begin);
#endif

#ifdef GW_DEBUG_MODE
	gw_application_manager_log_msg ( 0, __FILE__, __LINE__, __PRETTY_FUNCTION__, NULL);
#endif
gtk_clist_freeze ( GTK_CLIST ( ctree));

	g_strdup_to_gtk_text ( scan_disk_info_get_disk_name ( scan_disk), text[0]);
	vfs = scan_disk_info_get_vfs ( scan_disk);
	category = scan_disk_info_get_disk_category ( scan_disk);

#if GW_DEBUG_MODE
	gw_application_manager_log_msg ( 0, __FILE__, __LINE__, __PRETTY_FUNCTION__, NULL);
	vfs_stats_display ( vfs);
#endif

	/* Creates disk node */
	disk_info = disk_info_new ( );
	disk_info_set_name ( disk_info, g_strdup ( scan_disk_info_get_disk_name ( scan_disk)));
	disk_info_set_fsname ( disk_info, g_strdup ( vfs_stats_get_name ( vfs)));
	disk_info_set_dir ( disk_info, g_strdup ( vfs_stats_get_dir ( vfs)));
	disk_info_set_num ( disk_info, scan_disk_info_get_disk_num_archive ( scan_disk));
	if ( vfs_stats_get_type ( vfs) != NULL )
	{
		disk_info_set_type ( disk_info, g_strdup ( vfs_stats_get_type ( vfs)));
	}
	disk_info_set_volume ( disk_info, g_strdup ( vfs_stats_get_label ( vfs)));
	disk_size = vfs_stats_get_total_used_space ( vfs);
	disk_info_set_full ( disk_info, disk_size);
	disk_size = vfs_stats_get_total_free_space ( vfs);
	disk_info_set_free ( disk_info, disk_size);
	disk_info_set_date ( disk_info, time ( &date));
	disk_info_set_serial ( disk_info, g_strdup ( vfs_stats_get_serial_number ( vfs)));
	disk_info_set_category ( disk_info, scan_disk_info_get_disk_category ( scan_disk));
	disk_info_set_description ( disk_info, NULL);

	disk_data = disk_data_new ( );
	disk_data_set_info ( disk_data, disk_info);
	pix = disk_info_get_pixmap ( disk_info);
	msk = disk_info_get_bitmap ( disk_info);
	node = gtk_ctree_insert_node ( ctree, parent, NULL, text, 5, pix, msk, pix, msk, FALSE, FALSE);
	gtk_ctree_node_set_row_data_full ( ctree, node, disk_data, (GtkDestroyNotify)disk_data_free);
	g_free ( text[0]);

	if ( scan_disk_info_get_explore_tree_structure ( scan_disk) == TRUE )
	{
		/* Gets current folder */
		if ( getcwd ( past_dir, 512) == NULL )
		{
			error_msg = g_strconcat ( _( "Error : cannot get current folder."), "\n", strerror ( errno), NULL);
			scan_disk_info_set_error_msg ( scan_disk, error_msg);

			/* Remove the new disk */
			gtk_ctree_remove_node ( ctree, node);
			gtk_clist_thaw ( GTK_CLIST ( ctree));

			return -1;
		}

		/* Freezes the display of exploration tree */
		/*gtk_clist_freeze ( GTK_CLIST ( ctree));*/
		dir_info = dir_info_new ( );

		/* Builds the list of all the folders which must not be scanned.
		   These forbidden folders are included mount point of the current disk to scan. */
		forbidden = gw_get_drives_list ( forbidden);

		/* Removes the disk itself from the forbidden list. */
		if ( (self = g_list_find_custom ( forbidden, disk_info_get_dir ( disk_info), (GCompareFunc)strcmp)) != NULL ) {
			g_free ( self->data);
			forbidden = g_list_remove_link ( forbidden, self);
		}

		/* Removes all the mount points which are not included in the current disk to scan. */
		for ( i = 0; i < g_list_length ( forbidden); i++) {
			current = g_list_nth ( forbidden, i);
			/* Removes all mount points which are smaller or which start by the current disk to scan. */
			if ( strlen ( current->data) < strlen ( disk_info_get_dir ( disk_info)) || strncmp ( disk_info_get_dir ( disk_info), current->data, strlen ( disk_info_get_dir ( disk_info)))!=0 ) {
				i--;
				g_free ( current->data);
				forbidden = g_list_remove_link ( forbidden, current);
				current = NULL;
			}
		}

		/* Interrupt all signals */
		for ( i = 0; i < NSIG; i++) {
			if ( signal ( i, gw_catalog_manager_rollback_plugin_call) == SIG_ERR) {
#if GW_DEBUG_MODE
				gw_application_manager_log_msg ( 0, __FILE__, __LINE__, __PRETTY_FUNCTION__, "signum %d interruption is not ready", i);
#endif
			} else {
#if GW_DEBUG_MODE
				gw_application_manager_log_msg ( 0, __FILE__, __LINE__, __PRETTY_FUNCTION__, "signum %d interruption is ready", i);
#endif
			}
		}

		/* Adds the new disk */
		if ( catalog_add_dir ( scan_disk, vfs_stats_get_dir ( vfs), dir_info, ctree, node, &forbidden) == -1 )
		{
			/* Remove the new disk */
			gtk_ctree_remove_node ( ctree, node);
			gtk_clist_thaw ( GTK_CLIST ( ctree));

			/* Goes to the current application directory */
			chdir ( past_dir);

			/* Frees the forbidden list. */
			g_list_foreach ( forbidden, (GFunc)g_free, NULL);
			g_list_free ( forbidden);

			return -1;
		}

		/* Stop the signals interruption */
		for ( i = 0; i < NSIG; i++) {
			if ( signal ( i, SIG_DFL) == SIG_ERR) {
#if GW_DEBUG_MODE
				gw_application_manager_log_msg ( 0, __FILE__, __LINE__, __PRETTY_FUNCTION__, "signum %d interruption is not stopped", i);
#endif
			} else {
#if GW_DEBUG_MODE
				gw_application_manager_log_msg ( 0, __FILE__, __LINE__, __PRETTY_FUNCTION__, "signum %d interruption is stopped", i);
#endif
			}
		}


		/* Frees the forbidden list. */
		g_list_foreach ( forbidden, (GFunc)g_free, NULL);
		g_list_free ( forbidden);

		/* Bug to fix : why we did it?
		size = dir_info_get_size ( dir_info);
		disk_info_set_full ( disk_info, size);
		*/
		dir_info_free ( dir_info);
		/*gtk_ctree_sort_recursive ( ctree, parent);*/
		/*gtk_clist_thaw ( GTK_CLIST ( ctree));*/

		/* Goes to the begining folder */
		if ( chdir ( past_dir) == -1 )
		{
			error_msg = g_strconcat ( _( "The disk has been added"), "\n", _( "Error : cannot go to working folder."), "\n", strerror ( errno), NULL);
			scan_disk_info_set_error_msg ( scan_disk, error_msg);
		}
	}

	file_info = gtk_ctree_node_get_row_data ( ctree, parent);
	catalog_file_info_set_ismodified ( file_info, TRUE);

	/* BUG : fix me please */
	/* The total used space different from all files size. */
	if ( !scan_disk_info_get_stop ( scan_disk) )
	{
		scan_disk_info_get_scan_disk_set_current_statment ( scan_disk)( scan_disk_info_get_statment_controler ( scan_disk), (long double)vfs_stats_get_total_used_space ( vfs));
	}

	gtk_ctree_sort_recursive ( ctree, parent);
	gtk_clist_thaw ( GTK_CLIST ( ctree));

	/* Refreshes the exploration tree */
	gtk_ctree_collapse_recursive ( ctree, parent);
	gtk_ctree_expand ( ctree, parent);
	gtk_ctree_select ( ctree, parent);

#if GW_DEBUG_TIME
	gw_application_manager_log_msg ( 0, __FILE__, __LINE__, __PRETTY_FUNCTION__, "Elapsed time to load the catalog: %u secs\n", time ( NULL) - begin);
#endif

	return 0;
}


gint catalog_add_dir ( struct scan_disk_info *scan_disk, gchar *directory, struct dir_info *d, GtkCTree *ctree, GtkCTreeNode *parent, GList **forbidden)
{
	DIR *dir;
	struct dirent *file;
	struct stat f_info;
	struct passwd *f_proprio;
	struct group *f_group;
	guint ret;
	struct dir_data *dir_data = NULL, *tmp_dir_data = NULL;
	struct dir_info *dir_info = NULL;
	struct file_info *file_data = NULL, *parent_info = NULL;
	gchar *info_description = NULL;
	gchar buf[1024];
	gchar *text[1];
	gchar *name = NULL, *error_msg = NULL;
	GtkCTreeNode *node = NULL, *root = NULL;
	struct category *category = NULL;
	char *ext = NULL;
	func_get_file_descr_t func_get_file_descr = NULL;
	func_get_files_descr_t func_get_files_descr = NULL;
	func_get_parent_descr_t func_get_parent_descr = NULL;
	gchar*** files_descr_list = NULL;
	GList *child = NULL;
	int i = 0;
#ifdef GW_DEBUG_MODE
	char *dir_name = NULL;
#endif
	gchar *tmp = NULL;
	GList *self = NULL;


#ifdef GW_DEBUG_MODE
	gw_application_manager_log_msg ( 0, __FILE__, __LINE__, __PRETTY_FUNCTION__, "scan the folder : %s", directory);
#endif

	if ( scan_disk_info_get_stop ( scan_disk) )
	{
#if GW_DEBUG_MODE
		gw_application_manager_log_msg ( 0, __FILE__, __LINE__, __PRETTY_FUNCTION__, "stop scan disk.");
#endif

		error_msg = g_strdup ( _( "The disk hasn't been added\nAdding canceled by user"));
		scan_disk_info_set_error_msg ( scan_disk, error_msg);

		return -1;
	}

	category = scan_disk_info_get_disk_category ( scan_disk);

	if ( chdir ( directory) == -1 )
	{
		switch ( errno)
		{
			case EACCES :	/* Folder is locked */
					gtk_ctree_node_get_text ( ctree, parent, 0, &name);
					gtk_ctree_set_node_info ( ctree, parent, name, 5, PixLockedFolder, MskLockedFolder, PixOpenFolder, MskOpenFolder, TRUE, FALSE);

					return 0;

			case EFAULT:	error_msg = g_strconcat ( _( "The disk hasn't been added\n"), g_strdup_printf ( _( "%s outside the process's allocated address space."), directory), NULL);
					scan_disk_info_set_error_msg ( scan_disk, error_msg);

					return -1;

			case ENAMETOOLONG:error_msg = g_strconcat ( _( "The disk hasn't been added\n"), g_strdup_printf ( _( "A component of the following pathname exceeded %d characters, or an entire path name exceeded %d characters : %s."), NAME_MAX, PATH_MAX, directory), NULL);
					scan_disk_info_set_error_msg ( scan_disk, error_msg);

					return -1;

			case ENOENT:	error_msg = g_strconcat ( _( "The disk hasn't been added\n"), g_strdup_printf ( _( "The following directory does not exist : %s."), directory), NULL);
					scan_disk_info_set_error_msg ( scan_disk, error_msg);

					return -1;

			case ENOMEM:	error_msg = g_strconcat ( _( "The disk hasn't been added\n"), _( "Insufficient memory to complete the operation."), NULL);
					scan_disk_info_set_error_msg ( scan_disk, error_msg);

					return -1;

			case ENOTDIR:	error_msg = g_strconcat ( _( "The disk hasn't been added\n"), g_strdup_printf ( _( "A component of the path prefix is not a directory : %s."), directory), NULL);
					scan_disk_info_set_error_msg ( scan_disk, error_msg);

					return -1;

#if defined ( OS_LINUX) || defined ( OS_CYGWIN) || defined ( OS_FREEBSD) || defined ( OS_OPENBSD) || defined ( OS_NETBSD)
			case ELOOP:	error_msg = g_strconcat ( _( "The disk hasn't been added\n"), g_strdup_printf ( _( "Too many symbolic links were encountered in resolving following path : %s."), directory), NULL);
					scan_disk_info_set_error_msg ( scan_disk, error_msg);

					return -1;
#endif
			case EIO:	error_msg = g_strconcat ( _( "The disk hasn't been added\n"), g_strdup_printf ( _( "An I/O error occurred when change directory to : %s."), directory), NULL);
					scan_disk_info_set_error_msg ( scan_disk, error_msg);

					return -1;

			default:	error_msg = g_strconcat ( _( "The disk hasn't been added\n"), _( "Unknown error"), NULL);
					scan_disk_info_set_error_msg ( scan_disk, error_msg);

					return -1;
		}
	}

#ifdef GW_DEBUG_MODE
	dir_name = getcwd ( NULL, -1);
#endif

	/* Check if the current folder is a forbidden folder. In this case
	   must not scan this one. */
	tmp = (gchar*)g_malloc ( 1025 * sizeof ( gchar));
	tmp = getcwd ( tmp, 1024);
	if ( (self = g_list_find_custom ( *forbidden, tmp, (GCompareFunc)strcmp)) != NULL ) {
		/* If the current folder is forbidden, removesit from the forbidden list
		   because it will not be encountered at time. */
		g_free ( self->data);
		*forbidden = g_list_remove_link ( *forbidden, self);
		if (tmp!=NULL) g_free ( tmp);
		tmp = NULL;

		if ( chdir ( "..") == -1 )
		{
			error_msg = g_strdup ( _( "The disk hasn't been added\nError : cannot go to parent folder."));
			scan_disk_info_set_error_msg ( scan_disk, error_msg);

			return -1;
		}

		return 0;
	}

	if (tmp!=NULL) g_free ( tmp);
	tmp = NULL;

	if ( ( dir = opendir ( ".")) == NULL )
	{
		switch ( errno )
		{
			case EACCES : /* Folder is locked */
					gtk_ctree_node_get_text ( ctree, parent, 0, &name);
					gtk_ctree_set_node_info ( ctree, parent, name, 5, PixLockedFolder, MskLockedFolder, PixOpenFolder, MskOpenFolder, TRUE, FALSE);

					return 0;

			case EMFILE:	error_msg = g_strconcat ( _( "The disk hasn't been added\n"), _( "Too many file descriptors in use by process."), NULL);
					scan_disk_info_set_error_msg ( scan_disk, error_msg);

					return -1;

			case ENFILE:	error_msg = g_strconcat ( _( "The disk hasn't been added\n"), _( "Too many files are currently open in the system."), NULL);
					scan_disk_info_set_error_msg ( scan_disk, error_msg);

					return -1;

			case ENOENT:	error_msg = g_strconcat ( _( "The disk hasn't been added\n"), (tmp = g_strdup_printf ( _( "The following directory does not exist, or name is an empty string : %s."), directory)), NULL);
					g_free ( tmp);
					scan_disk_info_set_error_msg ( scan_disk, error_msg);

					return -1;

			case ENOMEM:	error_msg = g_strconcat ( _( "The disk hasn't been added\n"), _( "Insufficient memory to complete the operation."), NULL);
					scan_disk_info_set_error_msg ( scan_disk, error_msg);

					return -1;

			case ENOTDIR:	error_msg = g_strconcat ( _( "The disk hasn't been added\n"), (tmp = g_strdup_printf ( _( "The following directory name is not a directory : %s."), directory)), NULL);
					g_free ( tmp);
					scan_disk_info_set_error_msg ( scan_disk, error_msg);

					return -1;

			default:	error_msg = g_strconcat ( _( "The disk hasn't been added\n"), _( "Unknown error"), NULL);
					scan_disk_info_set_error_msg ( scan_disk, error_msg);

					return -1;
		}
	}


	dir_data = gtk_ctree_node_get_row_data ( ctree, parent);
	root = gw_gui_manager_main_interface_get_tree_root ( );

	while ( (file = readdir ( dir)) != NULL )
	{
#if defined ( OS_WIN32) || defined ( OS_CYGWIN)
		/* Skips this system folder. It's an unscanable folder. */
		if ( (GTK_CTREE_ROW ( parent)->parent == root) && ((strcmp ( file->d_name, "System Volume Information") == 0) || (strcmp ( file->d_name, "RECYCLER") == 0)) )
		{
#ifdef GW_DEBUG_MODE
			gw_application_manager_log_msg ( 0, __FILE__, __LINE__, __PRETTY_FUNCTION__, "warning encounter %s%s%s", dir_name, G_DIR_SEPARATOR_S, file->d_name);
#endif
			continue;
		}
#endif

#if defined ( OS_WIN32)
		if ( (ret = stat ( file->d_name, &f_info)) == -1 )
#else
		if ( (ret = lstat ( file->d_name, &f_info)) == -1 )
#endif
		{
			continue;
		}
		else
		{
#ifdef GW_DEBUG_MODE
			gw_application_manager_log_msg ( 0, __FILE__, __LINE__, __PRETTY_FUNCTION__, "add %s%s%s", dir_name, G_DIR_SEPARATOR_S, file->d_name);
#endif

			file_data = file_info_new ( );

			/* File name */
			file_info_set_name ( file_data, g_strdup ( file->d_name));
			g_strdup_to_gtk_text ( file_info_get_name ( file_data), text[0]);

			/* File rights */
			file_info_set_rights ( file_data, f_info.st_mode);

			switch ( S_IFMT & f_info.st_mode )
			{
#if defined ( OS_LINUX) || defined ( OS_CYGWIN) || defined ( OS_FREEBSD) || defined ( OS_OPENBSD) || defined ( OS_NETBSD)
				case S_IFLNK:	memset ( buf, '\0', 1024);
#ifdef GW_DEBUG_MODE
						gw_application_manager_log_msg ( 0, __FILE__, __LINE__, __PRETTY_FUNCTION__, "current link is %s/%s", directory, file->d_name);
#endif
						/* With not use G_DIR_SEPARATOR?!? */
						if ( readlink ( (tmp = g_strconcat ( directory, "/", file->d_name, NULL)), buf, 1023) != -1 )
						{
							if ( tmp != NULL )
							{
								g_free ( tmp);
								tmp = NULL;
							}

							file_info_set_name ( file_data, g_strconcat ( file->d_name, " -> ", buf, NULL));
						}

						break;

#endif
				case S_IFDIR:	/* If is a directory (different to "." and "..") add a subfolder to the parent folder */
						if ( file_info_get_is_real_directory ( file_data) )
						{
							dir_info_add_sub_folder ( d);
						}

						break;
			}

#if defined ( OS_LINUX) || defined ( OS_CYGWIN) || defined ( OS_FREEBSD) || defined ( OS_OPENBSD) || defined ( OS_NETBSD)
			/* Owner file */
			f_proprio = getpwuid ( f_info.st_uid);
			if ( f_proprio != NULL )
			{
				file_info_set_owner ( file_data, g_strdup ( f_proprio->pw_name));
			}
#endif

#if defined ( OS_LINUX) || defined ( OS_CYGWIN) || defined ( OS_FREEBSD) || defined ( OS_OPENBSD) || defined ( OS_NETBSD)
			/* Owner group file */
			f_group = getgrgid ( f_info.st_gid);
			if ( f_group != NULL )
			{
				file_info_set_group ( file_data, g_strdup ( f_group->gr_name));
			}
#endif

			/* File size */
			file_info_set_size ( file_data, f_info.st_size);

			if ( !scan_disk_info_get_stop ( scan_disk) )
			{
				/* What it is the real size, f_info.st_size or f_info.st_blksize * f_info.st_blocks? */
				scan_disk_info_get_scan_disk_set_current_statment ( scan_disk)( scan_disk_info_get_statment_controler ( scan_disk), f_info.st_size);
			}

			/* File inode */
			file_info_set_inode ( file_data, f_info.st_ino);

			/* Creation file date */
			file_info_set_cdate ( file_data, f_info.st_ctime);

			/* Last file access date */
			file_info_set_adate ( file_data, f_info.st_atime);

			/* Last file modification date */
			file_info_set_mdate ( file_data, f_info.st_mtime);

			/* Category file */
			file_info_set_category ( file_data, category);

			info_description = NULL;
			if ( scan_disk_info_get_make_description ( scan_disk))
			{
				if ( setjmp ( env) == 0 ) {
					ext = strrchr ( file->d_name, '.');

					if ( ext != NULL )
					{
						ext++;

#ifdef GW_DEBUG_MODE
						gw_application_manager_log_msg ( 0, __FILE__, __LINE__, __PRETTY_FUNCTION__, "extension is %s", ext);
#endif

						gw_plugins_manager_get_file_descr_func ( ext, &func_get_file_descr);

						if ( func_get_file_descr != NULL)
						{
#ifdef GW_DEBUG_MODE
							gw_application_manager_log_msg ( 0, __FILE__, __LINE__, __PRETTY_FUNCTION__, "start plugin for file description");
#endif

							info_description = func_get_file_descr ( file->d_name, NULL);
						}
						else
						{
#ifdef GW_DEBUG_MODE
							gw_application_manager_log_msg ( 0, __FILE__, __LINE__, __PRETTY_FUNCTION__, "no plugin for file description");
#endif
						}
					}

					gw_plugins_manager_get_parent_descr_func ( file->d_name, &func_get_parent_descr);

					if ( func_get_parent_descr != NULL)
					{
#ifdef GW_DEBUG_MODE
						gw_application_manager_log_msg ( 0, __FILE__, __LINE__, __PRETTY_FUNCTION__, "start plugin for parent description");
#endif

						if ( dir_data != NULL )
						{
							parent_info = dir_data_get_info ( dir_data);

							if ( (parent_info != NULL) && ((file_info_get_description ( parent_info) == NULL) || (strlen ( file_info_get_description ( parent_info)) == 0)) )
							{
								file_info_set_description ( parent_info, func_get_parent_descr ( file->d_name, NULL));
							}
						}
						else
						{
#ifdef GW_DEBUG_MODE
							gw_application_manager_log_msg ( 0, __FILE__, __LINE__, __PRETTY_FUNCTION__, "warning!! There is no parent data");
#endif
						}
					}
					else
					{
#ifdef GW_DEBUG_MODE
						gw_application_manager_log_msg ( 0, __FILE__, __LINE__, __PRETTY_FUNCTION__, "no plugin for parent description");
#endif
					}

					gw_plugins_manager_get_files_descr_func ( file->d_name, &func_get_files_descr);

					if ( func_get_files_descr != NULL)
					{
#ifdef GW_DEBUG_MODE
						gw_application_manager_log_msg ( 0, __FILE__, __LINE__, __PRETTY_FUNCTION__, "start plugin for files description");
#endif

						if ( files_descr_list == NULL )
						{
							files_descr_list = func_get_files_descr ( file->d_name, NULL);
						}
						else
						{
#ifdef GW_DEBUG_MODE
							gw_application_manager_log_msg ( 0, __FILE__, __LINE__, __PRETTY_FUNCTION__, "warning!! There is already files description list");
#endif
						}
					}
					else
					{
#ifdef GW_DEBUG_MODE
						gw_application_manager_log_msg ( 0, __FILE__, __LINE__, __PRETTY_FUNCTION__, "no plugin for files description");
#endif
					}
				} else {
					/* An critical error occured, should exit GWhere proprely */
					tmp = (gchar*)g_malloc ( 1025 * sizeof ( gchar));
					memset ( tmp, '\0', 1025);
					tmp = getcwd ( tmp, 1024);
					if ( tmp == NULL) {
						tmp = g_strdup_printf ( "...%s%s", G_DIR_SEPARATOR_S, directory);
					}

					error_msg = g_strdup_printf ( _( "A critical error occured when extracting file information of %s%s%s.\nSave your catalog if it is possible, exit GWhere and report this trouble to the GWhere project team please."), tmp, G_DIR_SEPARATOR_S, file->d_name);
					if ( tmp != NULL) {
						g_free ( tmp);
						tmp = NULL;
					}

					scan_disk_info_set_error_msg ( scan_disk, error_msg);

					return -1;
				}
			}

			/* Description file */
			file_info_set_description ( file_data, info_description);

			if ( file_info_get_isdirectory ( file_data) == TRUE )
			{
				/* Case : is a folder */
				ret = 0;

				if ( file_info_get_is_real_directory ( file_data) )
				{
					/* Adds the folder node to tree */
					node = gtk_ctree_insert_node ( ctree, parent, NULL, text, 5, PixFolder, MskFolder, PixOpenFolder, MskOpenFolder, FALSE, FALSE);

					tmp_dir_data = dir_data_new ( );
					dir_data_set_info ( tmp_dir_data, file_data);
					gtk_ctree_node_set_row_data_full ( ctree, node, tmp_dir_data, (GtkDestroyNotify)dir_data_free);

					/* Recursive call to catalog_add_dir */
					dir_info = dir_info_new ( );

					if ( catalog_add_dir ( scan_disk, file->d_name, dir_info, ctree, node, forbidden) == -1 )
					{
						if ( text[0] != NULL )
						{
							g_free ( text[0]);
						}

						return -1;
					}

					dir_info_add_size ( d, dir_info_get_size ( dir_info));
					file_info_set_size ( file_data, dir_info_get_size ( dir_info));

					if ( dir_info_get_sub_folder ( dir_info) == 0)
					{
						if ( dir_data_get_num_file(tmp_dir_data)>0 ) {
							gtk_ctree_set_node_info ( ctree, node, text[0], 5, PixFolder, MskFolder, PixOpenFolder, MskOpenFolder, TRUE, FALSE);
						} else {
							gtk_ctree_set_node_info ( ctree, node, text[0], 5, PixLockedFolder, MskLockedFolder, PixLockedFolder, MskLockedFolder, TRUE, FALSE);
						}
					}

					dir_info_free ( dir_info);
				}
				else
				{
					dir_data_append_child ( dir_data, file_data);
				}
			}
			else
			{
			  	/* Case : is a file */
			  	/* Adds the file to file list */
			  	dir_info_add_size ( d, file_info_get_size ( file_data));
				dir_data_append_child ( dir_data, file_data);
			}

			g_free ( text[0]);
		}
	}

	if ( files_descr_list != NULL )
	{
#ifdef GW_DEBUG_MODE
		gw_application_manager_log_msg ( 0, __FILE__, __LINE__, __PRETTY_FUNCTION__, "set all files description...");
#endif

		for ( i = 0; files_descr_list[i] != NULL; i++)
		{
#ifdef GW_DEBUG_MODE
			gw_application_manager_log_msg ( 0, __FILE__, __LINE__, __PRETTY_FUNCTION__, "set file description number %d", i);
			gw_application_manager_log_msg ( 0, __FILE__, __LINE__, __PRETTY_FUNCTION__, "chearching file named %s", files_descr_list[i][0]);
#endif

			if ( (child = dir_data_get_child_named ( dir_data, files_descr_list[i][0])) != NULL )
			{
				if ( (file_data = (child->data)) != NULL )
				{
					if ( (file_info_get_description ( file_data) == NULL) || (strlen ( file_info_get_description ( file_data)) == 0) )
					{
						file_info_set_description ( file_data, files_descr_list[i][1]);
					}
					else
					{
						if ( files_descr_list[i][1] != NULL )
						{
							g_free ( files_descr_list[i][1]);
						}
					}
				}
				else
				{
					if ( files_descr_list[i][1] != NULL )
					{
						g_free ( files_descr_list[i][1]);
					}
				}
			}
			else
			{
#ifdef GW_DEBUG_MODE
				gw_application_manager_log_msg ( 0, __FILE__, __LINE__, __PRETTY_FUNCTION__, "warning!! child not founded");
#endif
			}

			if ( files_descr_list[i][0] != NULL )
			{
				/* TODO : to remove or not? */
				g_free ( files_descr_list[i][0]);
			}

			/* TODO : to remove or not? */
			g_free ( files_descr_list[i]);
		}

		g_free ( files_descr_list);
		files_descr_list = NULL;

#ifdef GW_DEBUG_MODE
		gw_application_manager_log_msg ( 0, __FILE__, __LINE__, __PRETTY_FUNCTION__, "all files description are setted");
#endif
	}

#ifdef GW_DEBUG_MODE
	free ( dir_name);
#endif

	if ( closedir ( dir) == -1 )
	{
		error_msg = g_strdup ( _( "The disk hasn't been added\nError : cannot close current folder."));
		scan_disk_info_set_error_msg ( scan_disk, error_msg);

		return -1;
	}

	if ( chdir ( "..") == -1 )
	{
		error_msg = g_strdup ( _( "The disk hasn't been added\nError : cannot go to parent folder."));
		scan_disk_info_set_error_msg ( scan_disk, error_msg);

		return -1;
	}

	return 0;
}


gint catalog_save_file ( gchar *file, GtkCTree *ctree, GtkCTreeNode *parent)
{
#ifdef GW_DEBUG_MODE
	gw_application_manager_log_msg ( 0, __FILE__, __LINE__, __PRETTY_FUNCTION__, NULL);
#endif

	return catalog_save_file_version_2 ( file, ctree, parent);
}


gint catalog_save_file_version_1 ( gchar *file, GtkCTree *ctree, GtkCTreeNode *parent)
{
	gchar *buf = NULL;
	gint fic;
	GtkCTreeNode *node = NULL;
	struct catalog_file_info *file_info = NULL;
	struct disk_data *disk_data = NULL;
	struct disk_info *disk_info = NULL;
	GtkWindow *window = gw_gui_manager_main_interface_get_main_window ( );


#ifdef GW_DEBUG_MODE
	gw_application_manager_log_msg ( 0, __FILE__, __LINE__, __PRETTY_FUNCTION__, NULL);
#endif

	/* Creates file */
	if ( (fic = open ( file, O_WRONLY|O_TRUNC|O_CREAT, 0666)) == -1 )
	{
		gw_msg_box_create ( window, _( "Save catalog"), _( "Error : cannot create file."));
		return -1;
	}

	/* Writes file version */
	buf = g_strconcat ( PROJECT_NAME, ":", VERSION, ":1\n", NULL);

	if ( strlen ( buf) != write ( fic, buf, strlen ( buf)))
	{
		gw_msg_box_create ( window, _( "Save catalog"), _( "Error : cannot write file version."));

		if ( buf != NULL )
		{
			g_free ( buf);
			buf = NULL;
		}

		return -1;
	}

	if ( buf != NULL )
	{
		g_free ( buf);
		buf = NULL;
	}

	/* Saves all disks */
	node = GTK_CTREE_ROW ( parent)->children;
	while ( node != NULL )
	{
		/* Gets disk informations */
		disk_data = gtk_ctree_node_get_row_data ( ctree, node);
		disk_info = disk_data_get_info ( disk_data);
		buf = catalog_disk_info_to_file ( disk_info, 1);

		if ( strlen ( buf) != write ( fic, buf, strlen ( buf)))
		{
			perror ( "write");

			if ( buf != NULL )
			{
				g_free ( buf);
				buf = NULL;
			}

			return -1;
		}

		if ( buf != NULL )
		{
			g_free ( buf);
			buf = NULL;
		}

		/* Recursive call to catalog_save_file_dir_1 */
		catalog_save_file_dir_1 ( fic, ctree, node);
		buf = g_strdup ( "\\\n");
		if ( strlen ( buf) != write ( fic, buf, strlen ( buf)))
		{
			perror ( "write");

			if ( buf != NULL )
			{
				g_free ( buf);
				buf = NULL;
			}

			return -1;
		}

		if ( buf != NULL )
		{
			g_free ( buf);
			buf = NULL;
		}

		/* Goes to the next disk */
		node = GTK_CTREE_ROW ( node)->sibling;
	}


	/* Closes file */
	if ( close ( fic) == -1 )
	{
		gw_msg_box_create ( window, _( "Add disk"), _( "Error : cannot close catalog file."));
		return -1;
	}


	/* Catalog modifications were saved */
	file_info = gtk_ctree_node_get_row_data ( ctree, parent);
	catalog_file_info_set_ismodified ( file_info, FALSE);

	return 0;
}


gint catalog_save_file_version_2 ( gchar *file, GtkCTree *ctree, GtkCTreeNode *parent)
{
	gzFile fic;
	GtkCTreeNode *node = NULL;
	struct catalog_file_info *catalog_info = NULL;
	struct disk_data *disk_data = NULL;
	struct disk_info *disk_info = NULL;
	GList *categories;
	gint i = 0;
	gint size = 0;
	gchar *compression_level = NULL;
	gchar *mode = NULL;
	GtkWindow *window = gw_gui_manager_main_interface_get_main_window ( );
	gchar *tmp[7];
	GWStringBuffer *strbuf[2];


#if GW_DEBUG_TIME
	time_t begin;
	time ( &begin);
#endif

#ifdef GW_DEBUG_MODE
	gw_application_manager_log_msg ( 0, __FILE__, __LINE__, __PRETTY_FUNCTION__, NULL);
#endif

	/* Gets the file compression level. */
	compression_level = gw_application_manager_get_settings ( GW_VALUE_APP_CATALOG_COMPRESSION_LEVEL);

	if ( compression_level == NULL )
	{
		compression_level = g_strdup_printf ( "%d", GW_VALUE_APP_CATALOG_COMPRESSION_LEVEL_DEFAULT);
	}
	else
	{
		compression_level = g_strdup ( compression_level);
	}

	/* Builds the open file mode. */
	mode = g_strdup_printf ( "wb%s", compression_level);
	g_free ( compression_level);


#if GW_DEBUG_MODE
	gw_application_manager_log_msg ( 0, __FILE__, __LINE__, __PRETTY_FUNCTION__, "open file with '%s' mode", mode);
#endif

	/* Creates file */
	if ( (fic = gzopen ( file, mode)) == NULL )
	{
		gw_msg_box_create ( window, _( "Save catalog"), _( "Error : cannot create file."));
		g_free ( mode);

		return -1;
	}

	g_free ( mode);

	/* Writes file version */
	if ( gzprintf ( fic, "%s:%s:2\n", PROJECT_NAME, VERSION) == 0 )
	{
		gw_msg_box_create ( window, _( "Save catalog"), _( "Error : cannot write file version."));

		if ( gzclose ( fic) != Z_OK)
		{
			gw_msg_box_create ( window, _( "Save catalog"), _( "Error : cannot close file."));
		}

		return -1;
	}

	/* Writes catalog informations */
	catalog_info = gtk_ctree_node_get_row_data ( ctree, parent);
	tmp[0] = gw_str_to_file ( catalog_file_info_get_title ( catalog_info));
	tmp[1] = gw_str_to_file ( catalog_file_info_get_description ( catalog_info));
	if ( gzprintf ( fic, "%s:%s\n",
				(tmp[0]==NULL)?catalog_file_info_get_title ( catalog_info):tmp[0],
				(tmp[1]==NULL)?catalog_file_info_get_description ( catalog_info):tmp[1]) == 0 )
	{
		if ( gzclose ( fic) != Z_OK )
		{
			gw_msg_box_create ( window, _( "Save catalog"), _( "Error : cannot close file."));
		}

		if ( tmp[0] != NULL ) {g_free ( tmp[0]);tmp[0] = NULL;}
		if ( tmp[1] != NULL ) {g_free ( tmp[1]);tmp[1] = NULL;}

		return -1;
	}

	if ( tmp[0] != NULL ) {g_free ( tmp[0]);tmp[0] = NULL;}
	if ( tmp[1] != NULL ) {g_free ( tmp[1]);tmp[1] = NULL;}


#if GW_DEBUG_MODE
	gw_application_manager_log_msg ( 0, __FILE__, __LINE__, __PRETTY_FUNCTION__, "write categories list", NULL);
#endif

	/* Writes catelories list */
	if ( (categories = catalog_file_info_get_categories ( catalog_info)) != NULL)
	{
		/* Had to save categories */
		/* Puts one catagory by line */
		categories = g_list_first ( categories);
		size = g_list_length ( categories);

		for ( i = 1; i < size; i++ )
		{
			categories = g_list_next ( categories);

			tmp[0] = gw_str_to_file ( category_get_name ( categories->data));
			tmp[1] = gw_str_to_file ( category_get_description ( categories->data));
			if ( gzprintf ( fic, "%s:%s\n",
						(tmp[0]==NULL)?category_get_name ( categories->data):tmp[0],
						(tmp[1]==NULL)?category_get_description ( categories->data):tmp[1]) == 0 )
			{
				if ( gzclose ( fic) != Z_OK )
				{
					gw_msg_box_create ( window, _( "Save catalog"), _( "Error : cannot close file."));
				}

				if ( tmp[0] != NULL ) {g_free ( tmp[0]);tmp[0] = NULL;}
				if ( tmp[1] != NULL ) {g_free ( tmp[1]);tmp[1] = NULL;}

				return -1;
			}

			if ( tmp[0] != NULL ) {g_free ( tmp[0]);tmp[0] = NULL;}
			if ( tmp[1] != NULL ) {g_free ( tmp[1]);tmp[1] = NULL;}
		}
	}
	else
	{
		if ( gzputs ( fic, "\n") == -1 )
		{
			if ( gzclose ( fic) != Z_OK )
			{
				gw_msg_box_create ( window, _( "Save catalog"), _( "Error : cannot close file."));
			}

			return -1;
		}
	}


#if GW_DEBUG_MODE
		gw_application_manager_log_msg ( 0, __FILE__, __LINE__, __PRETTY_FUNCTION__, "write all disk", NULL);
#endif

	strbuf[0] = gw_string_buffer_new ( );
	strbuf[1] = gw_string_buffer_new ( );

	/* Saves all disks */
	node = GTK_CTREE_ROW ( parent)->children;
	while ( node != NULL )
	{
		/* Gets disk informations */
		disk_data = gtk_ctree_node_get_row_data ( ctree, node);
		disk_info = disk_data_get_info ( disk_data);

		tmp[0] = gw_str_to_file ( disk_info_get_name ( disk_info));
		tmp[1] = gw_str_to_file ( disk_info_get_fsname ( disk_info));
		tmp[2] = gw_str_to_file ( disk_info_get_dir ( disk_info));
		tmp[3] = gw_str_to_file ( disk_info_get_type ( disk_info));
		tmp[4] = gw_str_to_file ( disk_info_get_volume ( disk_info));
		tmp[5] = gw_str_to_file ( disk_info_get_serial ( disk_info));
		tmp[6] = gw_str_to_file ( disk_info_get_description ( disk_info));

		if ( gzprintf ( fic, FORMAT2_DISK_LONG,
					(tmp[0]==NULL)?disk_info_get_name ( disk_info):tmp[0],
					disk_info_get_num ( disk_info),
					(tmp[1]==NULL)?disk_info_get_fsname ( disk_info):tmp[1],
					(tmp[2]==NULL)?disk_info_get_dir ( disk_info):tmp[2],
					(tmp[3]==NULL)?disk_info_get_type ( disk_info):tmp[3],
					(tmp[4]==NULL)?disk_info_get_volume ( disk_info):tmp[4],
					disk_info_get_full ( disk_info),
					disk_info_get_free ( disk_info),
					disk_info_get_date ( disk_info),
					(tmp[5]==NULL)?disk_info_get_serial ( disk_info):tmp[5],
					disk_info_get_category_index ( disk_info),
					(tmp[6]==NULL)?disk_info_get_description ( disk_info):tmp[6]) == 0 )
		{
			if ( gzclose ( fic) != Z_OK )
			{
				gw_msg_box_create ( window, _( "Save catalog"), _( "Error : cannot close file."));
			}

			if ( tmp[0] != NULL ) {g_free ( tmp[0]);tmp[0] = NULL;}
			if ( tmp[1] != NULL ) {g_free ( tmp[1]);tmp[1] = NULL;}
			if ( tmp[2] != NULL ) {g_free ( tmp[2]);tmp[2] = NULL;}
			if ( tmp[3] != NULL ) {g_free ( tmp[3]);tmp[3] = NULL;}
			if ( tmp[4] != NULL ) {g_free ( tmp[4]);tmp[4] = NULL;}
			if ( tmp[5] != NULL ) {g_free ( tmp[5]);tmp[5] = NULL;}
			if ( tmp[6] != NULL ) {g_free ( tmp[6]);tmp[6] = NULL;}

			gw_string_buffer_free ( strbuf[0]);
			gw_string_buffer_free ( strbuf[1]);

			return -1;
		}

		if ( tmp[0] != NULL ) {g_free ( tmp[0]);tmp[0] = NULL;}
		if ( tmp[1] != NULL ) {g_free ( tmp[1]);tmp[1] = NULL;}
		if ( tmp[2] != NULL ) {g_free ( tmp[2]);tmp[2] = NULL;}
		if ( tmp[3] != NULL ) {g_free ( tmp[3]);tmp[3] = NULL;}
		if ( tmp[4] != NULL ) {g_free ( tmp[4]);tmp[4] = NULL;}
		if ( tmp[5] != NULL ) {g_free ( tmp[5]);tmp[5] = NULL;}
		if ( tmp[6] != NULL ) {g_free ( tmp[6]);tmp[6] = NULL;}

		/* Recursive call to catalog_save_file_dir_2 */
		catalog_save_file_dir_2 ( fic, ctree, node, strbuf);

		if ( gzputs ( fic, "\\\n") == -1 )
		{
			if ( gzclose ( fic) != Z_OK )
			{
				gw_msg_box_create ( window, _( "Save catalog"), _( "Error : cannot close file."));
			}

			gw_string_buffer_free ( strbuf[0]);
			gw_string_buffer_free ( strbuf[1]);

			return -1;
		}

		/* Goes to next disk */
		node = GTK_CTREE_ROW ( node)->sibling;
	}

	gw_string_buffer_free ( strbuf[0]);
	gw_string_buffer_free ( strbuf[1]);

	/* Closes file */
	if ( gzclose ( fic) == -1 )
	{
		gw_msg_box_create ( window, _( "Add disk"), _( "Error : cannot close catalog file."));

		return -1;
	}


	/* Catalog informations were saved */
	catalog_info = gtk_ctree_node_get_row_data ( ctree, parent);
	catalog_file_info_set_ismodified ( catalog_info, FALSE);

#if GW_DEBUG_TIME
	gw_application_manager_log_msg ( 0, __FILE__, __LINE__, __PRETTY_FUNCTION__, _( "Elapsed time to save the catalog : %d secs"), time ( NULL) - begin);
#endif

	return 0;
}


gint catalog_save_file_disk_1 ( gint fic, GtkCTree *ctree, GtkCTreeNode *parent)
{
#ifdef GW_DEBUG_MODE
	gw_application_manager_log_msg ( 0, __FILE__, __LINE__, __PRETTY_FUNCTION__, NULL);
#endif

	return -1;
}


gint catalog_save_file_disk_2 ( gint fic, GtkCTree *ctree, GtkCTreeNode *parent)
{
#ifdef GW_DEBUG_MODE
	gw_application_manager_log_msg ( 0, __FILE__, __LINE__, __PRETTY_FUNCTION__, NULL);
#endif

	return -1;
}


gint catalog_save_file_dir_1 ( gint fic, GtkCTree *ctree, GtkCTreeNode *parent)
{
	gchar *buf;
	struct dir_data *dir_data = NULL;
	struct file_info *file_info = NULL;
	GList *dir = NULL;
	GtkCTreeNode *node = NULL;


#ifdef GW_DEBUG_MODE
	gw_application_manager_log_msg ( 0, __FILE__, __LINE__, __PRETTY_FUNCTION__, NULL);
#endif

	/* Gets current folder informations */
	dir_data = gtk_ctree_node_get_row_data ( ctree, parent);

	/* Gets folder files list */
	dir = dir_data_get_child ( dir_data);
	dir = g_list_first ( dir);
	while ( dir != g_list_last ( dir) )
	{
		/* Writes file informations */
		/* Checks if it's a folder different from "." and ".." */
		if ( file_info_get_isdirectory ( dir->data) == TRUE )
		{
			if ( (strcmp ( file_info_get_name ( dir->data), ".") != 0) && (strcmp ( file_info_get_name ( dir->data), "..") != 0) )
			{
				buf = g_strdup ("/\n");
				if ( strlen ( buf) != write ( fic, buf, strlen ( buf)))
				{
					perror ( "write");

					if ( buf != NULL )
					{
						g_free ( buf);
						buf = NULL;
					}

					return -1;
				}

				if ( buf != NULL )
				{
					g_free ( buf);
					buf = NULL;
				}
			}
		}

		/* Gets file informations */
		buf = catalog_file_info_to_file ( dir->data, 1);

		if ( strlen ( buf) != write ( fic, buf, strlen ( buf)))
		{
			perror ( "write");

			if ( buf != NULL )
			{
				g_free ( buf);
				buf = NULL;
			}

			return -1;
		}

		if ( buf != NULL )
		{
			g_free ( buf);
			buf = NULL;
		}


		/* Goes to the next file */
		dir = g_list_next ( dir);
	}

	if ( dir != NULL )
	{
		/* Writes file informations */
		if ( file_info_get_isdirectory ( dir->data) == TRUE )
		{
			if ( (strcmp ( file_info_get_name ( dir->data), ".") != 0) && (strcmp ( file_info_get_name ( dir->data), "..") != 0) )
			{
				buf = g_strdup ( "/\n");
				if ( strlen ( buf) != write ( fic, buf, strlen ( buf)))
				{
					perror ( "write");

					if ( buf != NULL )
					{
						g_free ( buf);
						buf = NULL;
					}

					return -1;
				}

				if ( buf != NULL )
				{
					g_free ( buf);
					buf = NULL;
				}
			}
		}

		/* Gets file informations */
		buf = catalog_file_info_to_file ( dir->data, 1);

		if ( strlen ( buf) != write ( fic, buf, strlen ( buf)))
		{
			perror ( "write");

			if ( buf != NULL )
			{
				g_free ( buf);
				buf = NULL;
			}

			return -1;
		}

		if ( buf != NULL )
		{
			g_free ( buf);
			buf = NULL;
		}
	}


	/* Saves all subfolders */
	node = GTK_CTREE_ROW ( parent)->children;
	while ( node != NULL )
	{
		/* Gets subfolder informations */
		dir_data = gtk_ctree_node_get_row_data ( ctree, node);
		file_info = dir_data_get_info ( dir_data);
		buf = g_strdup ( "/\n");
		if ( strlen ( buf) != write ( fic, buf, strlen ( buf)))
		{
			perror ( "write");

			if ( buf != NULL )
			{
				g_free ( buf);
				buf = NULL;
			}

			return -1;
		}

		if ( buf != NULL )
		{
			g_free ( buf);
			buf = NULL;
		}

		/* Gets subfolder informations */
		buf = catalog_file_info_to_file ( dir->data, 1);

		if ( strlen ( buf) != write ( fic, buf, strlen ( buf)))
		{
			perror ( "write");

			if ( buf != NULL )
			{
				g_free ( buf);
				buf = NULL;
			}

			return -1;
		}

		if ( buf != NULL )
		{
			g_free ( buf);
			buf = NULL;
		}

		/* Recursive call to catalog_save_file_dir_1 */
		catalog_save_file_dir_1 ( fic, ctree, node);
		buf = g_strdup ( "\\\n");
		if ( strlen ( buf) != write ( fic, buf, strlen ( buf)))
		{
			perror ( "write");

			if ( buf != NULL )
			{
				g_free ( buf);
				buf = NULL;
			}

			return -1;
		}

		if ( buf != NULL )
		{
			g_free ( buf);
			buf = NULL;
		}

		/* Goes to the next subfolder */
		node = GTK_CTREE_ROW ( node)->sibling;
	}

	return 0;
}


gint catalog_save_file_dir_2 ( gzFile fic, GtkCTree *ctree, GtkCTreeNode *parent, GWStringBuffer *buf[2])
{
	struct dir_data *dir_data = NULL;
	struct file_info *file_info = NULL;
	GList *dir = NULL;
	GtkCTreeNode *node = NULL;
	gchar *tmp[3];


#ifdef GW_DEBUG_MODE
	gw_application_manager_log_msg ( 0, __FILE__, __LINE__, __PRETTY_FUNCTION__, NULL);
#endif

	/* Gets current folder informations */
	dir_data = gtk_ctree_node_get_row_data ( ctree, parent);

	/* Gets folder files list */
	dir = dir_data_get_child ( dir_data);
	dir = g_list_first ( dir);
	while ( dir != g_list_last ( dir) )
	{
#ifdef GW_DEBUG_MODE
	gw_application_manager_log_msg ( 0, __FILE__, __LINE__, __PRETTY_FUNCTION__, "saving file %s ...", file_info_get_name ( dir->data));
#endif

		/* Writes file informations */
		/* Checks if it's a folder different from "." and ".." */
		if ( file_info_get_isdirectory ( dir->data) == TRUE )
		{
			if ( (strcmp ( file_info_get_name ( dir->data), ".") != 0) && (strcmp ( file_info_get_name ( dir->data), "..") != 0) )
			{
				if ( gzputs ( fic, "/\n") == -1 )
				{
					return -1;
				}
			}
		}

		/* Gets file informations */
		tmp[0] = gw_str_to_file_strb ( file_info_get_name ( dir->data), buf[0]);
		tmp[1] = gw_str_to_file_strb ( file_info_get_description ( dir->data), buf[1]);
		tmp[2] = file_info_get_rights_to_gchar ( dir->data);

		if ( gzprintf ( fic, FORMAT2_FILE_LONG,
					tmp[0],
					tmp[2],
					file_info_get_owner ( dir->data),
					file_info_get_group ( dir->data),
					file_info_get_inode ( dir->data),
					file_info_get_size ( dir->data),
					file_info_get_cdate ( dir->data),
					file_info_get_adate ( dir->data),
					file_info_get_mdate ( dir->data),
					file_info_get_category_index ( dir->data),
					tmp[1]) == 0 )
		{
			return -1;
		}

		/* Goes to the next file */
		dir = g_list_next ( dir);
	}

	if ( dir != NULL )
	{
#ifdef GW_DEBUG_MODE
	gw_application_manager_log_msg ( 0, __FILE__, __LINE__, __PRETTY_FUNCTION__, "saving last file %s ...", file_info_get_name ( dir->data));
#endif

		/* Writes file informations */
		if ( file_info_get_isdirectory ( dir->data) == TRUE )
		{
			if ( (strcmp ( file_info_get_name ( dir->data), ".") != 0) && (strcmp ( file_info_get_name ( dir->data), "..") != 0) )
			{
				if ( gzputs ( fic, "/\n") == -1 )
				{
					return -1;
				}
			}
		}

		/* Gets file informations */
		tmp[0] = gw_str_to_file_strb ( file_info_get_name ( dir->data), buf[0]);
		tmp[1] = gw_str_to_file_strb ( file_info_get_description ( dir->data), buf[1]);
		tmp[2] = file_info_get_rights_to_gchar ( dir->data);

		if ( gzprintf ( fic, FORMAT2_FILE_LONG,
					tmp[0],
					tmp[2],
					file_info_get_owner ( dir->data),
					file_info_get_group ( dir->data),
					file_info_get_inode ( dir->data),
					file_info_get_size ( dir->data),
					file_info_get_cdate ( dir->data),
					file_info_get_adate ( dir->data),
					file_info_get_mdate ( dir->data),
					file_info_get_category_index ( dir->data),
					tmp[1]) == 0 )
		{
			return -1;
		}
	}


	/* Saves all subfolders */
	node = GTK_CTREE_ROW ( parent)->children;
	while ( node != NULL )
	{
#ifdef GW_DEBUG_MODE
	gw_application_manager_log_msg ( 0, __FILE__, __LINE__, __PRETTY_FUNCTION__, "saving subfolder %s ...", file_info_get_name ( file_info));
#endif

		/* Gets subfolder informations */
		dir_data = gtk_ctree_node_get_row_data ( ctree, node);
		file_info = dir_data_get_info ( dir_data);
		if ( gzputs ( fic, "/\n") == -1 )
		{
			return -1;
		}

		/* Gets file informations */
		tmp[0] = gw_str_to_file_strb ( file_info_get_name ( file_info), buf[0]);
		tmp[1] = gw_str_to_file_strb ( file_info_get_description ( file_info), buf[1]);
		tmp[2] = file_info_get_rights_to_gchar ( file_info);

		if ( gzprintf ( fic, FORMAT2_FILE_LONG,
					tmp[0],
					tmp[2],
					file_info_get_owner ( file_info),
					file_info_get_group ( file_info),
					file_info_get_inode ( file_info),
					file_info_get_size ( file_info),
					file_info_get_cdate ( file_info),
					file_info_get_adate ( file_info),
					file_info_get_mdate ( file_info),
					file_info_get_category_index ( file_info),
					tmp[1]) == 0 )
		{
			return -1;
		}

		/* Recursive call to catalog_save_file_dir_2 */
		catalog_save_file_dir_2 ( fic, ctree, node, buf);
		if ( gzputs ( fic, "\\\n") == -1 )
		{
			return -1;
		}

		/* Goes to the next subfolder */
		node = GTK_CTREE_ROW ( node)->sibling;
	}

	return 0;
}


gint catalog_ctree_data_free ( GList *p)
{
#ifdef GW_DEBUG_MODE
	gw_application_manager_log_msg ( 0, __FILE__, __LINE__, __PRETTY_FUNCTION__, NULL);
#endif

	if ( p == NULL )
	{
		return -1;
	}

	g_list_foreach ( p, (GFunc)g_free, NULL);
	g_list_free ( p);

	return 0;
}


struct disk_info * catalog_disk_info_new_from_table ( gchar **attrib, gint file_version)
{
	struct disk_info *p = NULL;


#ifdef GW_DEBUG_MODE
	gw_application_manager_log_msg ( 0, __FILE__, __LINE__, __PRETTY_FUNCTION__, NULL);
#endif

	p = disk_info_new ( );

	switch ( file_version )
	{
		case 1:	disk_info_set_name ( p, gw_file_to_str ( attrib[0]));
			disk_info_set_fsname ( p, g_strdup ( NULL)); /* We can guess it. */

			if ( strncmp ( attrib[2], "CD-ROM", 6) == 0 )
			{
				disk_info_set_fsname ( p, g_strdup ( "/dev/cdrom"));
			}
			else { if ( strncmp ( attrib[2], "HD", 2) == 0 )
			{
				disk_info_set_fsname ( p, g_strdup ( "/dev/hd"));
			}
			else { if ( strncmp ( attrib[2], "Floppy", 6) == 0 )
			{
				disk_info_set_fsname ( p, g_strdup ( "/dev/fd"));
			}
			else { if ( strncmp ( attrib[2], "Zip", 3) == 0 )
			{
				disk_info_set_fsname ( p, g_strdup ( "/dev/zip"));
			}
			else
			{
				disk_info_set_fsname ( p, g_strdup ( ""));
			}}}}

			disk_info_set_dir ( p, g_strdup ( NULL));/* Hard to find!! */
			disk_info_set_num ( p, strtol ( attrib[1], NULL, 10));

			if ( strncmp ( attrib[2], "CD-ROM", 6) == 0 )
			{
				disk_info_set_type ( p, g_strdup ( "iso9660"));
			}
			else
			{
				disk_info_set_type ( p, g_strdup ( ""));
			}

			if ( strncmp ( attrib[3], "(null)", 6 ) != 0 )
			{
				if ( strncmp ( attrib[3], "(null)", 6 ) != 0 )
				{
					disk_info_set_volume ( p, gw_file_to_str ( attrib[3]));
				}
			}

			disk_info_set_full ( p, strtoui64 ( attrib[4], NULL, 10));
			disk_info_set_free ( p, strtoui64 ( attrib[5], NULL, 10));
			disk_info_set_date ( p, strtoul ( attrib[6], NULL, 10));

			if ( strncmp ( attrib[7], "(null)", 6 ) != 0 )
			{
				disk_info_set_serial ( p, gw_file_to_str ( attrib[7]));
			}

			disk_info_set_description ( p, gw_file_to_str ( attrib[8]));
			break;

		case 2:	disk_info_set_name ( p, gw_file_to_str ( attrib[0]));
				disk_info_set_num ( p, strtol ( attrib[1], NULL, 10));
				disk_info_set_fsname ( p, gw_file_to_str ( attrib[2]));
				disk_info_set_dir ( p, gw_file_to_str ( attrib[3]));
				disk_info_set_type ( p, gw_file_to_str ( attrib[4]));

				if ( strncmp ( attrib[5], "(null)", 6 ) != 0 )
				{
					if ( strncmp ( attrib[5], "", 6 ) != 0 )
					{
						disk_info_set_volume ( p, gw_file_to_str ( attrib[5]));
					}
				}

			disk_info_set_full ( p, strtoui64 ( attrib[6], NULL, 10));
			disk_info_set_free ( p, strtoui64 ( attrib[7], NULL, 10));
			disk_info_set_date ( p, strtoul ( attrib[8], NULL, 10));

			if ( strncmp ( attrib[9], "(null)", 6 ) != 0 )
			{
				disk_info_set_serial ( p, gw_file_to_str ( attrib[9]));
			}
			/* Bug to fix!! */
			/* disk_info_set_category ( p, g_strdup ( attrib[10])); */
			disk_info_set_description ( p, gw_file_to_str ( attrib[11]));
			break;

		default:disk_info_free ( p);
			return NULL;
	}

	return p;
}


struct disk_info * catalog_disk_info_new_from_str ( gchar *str, gint file_version, struct catalog_file_info *catalog_info)
{
	struct disk_info *p = NULL;
	gint i = 0, j = 0, begin = 0;
	gchar *tmp = NULL;


#ifdef GW_DEBUG_MODE
	gw_application_manager_log_msg ( 0, __FILE__, __LINE__, __PRETTY_FUNCTION__, NULL);
#endif

	if ( (p = disk_info_new ( )) != NULL ) {
		i = 0;
		begin = i;

		/* Read the disk name. */
		for ( j = i; ((str[j] != ':') && (str[j] != '\0')); j++ ) {
			if ( str[j] == '\\') {
				if ( str[j+1] == '#') {
					str[i++] = ':';
					j++;
				} else if ( str[j+1] == 'n') {
					str[i++] = '\n';
					j++;
				} else {
					str[i++] = str[j];
				}
			} else {
				str[i++] = str[j];
			}
		}

		if ( begin < i) {
			if ( (tmp = (gchar*)g_malloc0(sizeof(gchar)*(i-begin+1))) != NULL ) {
				memcpy(tmp,&str[begin],i-begin);
			} else {
				tmp = NULL;
			}
		} else {
			tmp = NULL;
		}

		disk_info_set_name ( p, tmp);
		tmp = NULL;

		i = j + 1;
		begin = i;

		/* Read the disk archive number. */
		for ( j = i; ((str[j] != ':') && (str[j] != '\0')); j++ ) {
			i++;
		}

		if ( str[j] == ':') {
			str[j] = '\0';
			disk_info_set_num ( p, strtol ( &str[begin], NULL, 10));
		} else {
			tmp = NULL;
		}

		if ( file_version == 1) {
		} else if ( file_version == 2 ) {
			i = j + 1;
			begin = i;

			/* Read the disk file system name. */
			for ( j = i; ((str[j] != ':') && (str[j] != '\0')); j++ ) {
				if ( str[j] == '\\') {
					if ( str[j+1] == '#') {
						str[i++] = ':';
						j++;
					} else if ( str[j+1] == 'n') {
						str[i++] = '\n';
						j++;
					} else {
						str[i++] = str[j];
					}
				} else {
					str[i++] = str[j];
				}
			}

			if ( begin < i) {
				if ( (tmp = (gchar*)g_malloc0(sizeof(gchar)*(i-begin+1))) != NULL ) {
					memcpy(tmp,&str[begin],i-begin);
				} else {
					tmp = NULL;
				}
			} else {
				tmp = NULL;
			}

			disk_info_set_fsname ( p, tmp);
			tmp = NULL;
		}

		if ( file_version == 1) {
		} else if ( file_version == 2 ) {
			i = j + 1;
			begin = i;

			/* Read the disk access directory. */
			for ( j = i; ((str[j] != ':') && (str[j] != '\0')); j++ ) {
				if ( str[j] == '\\') {
					if ( str[j+1] == '#') {
						str[i++] = ':';
						j++;
					} else if ( str[j+1] == 'n') {
						str[i++] = '\n';
						j++;
					} else {
						str[i++] = str[j];
					}
				} else {
					str[i++] = str[j];
				}
			}

			if ( begin < i) {
				if ( (tmp = (gchar*)g_malloc0(sizeof(gchar)*(i-begin+1))) != NULL ) {
					memcpy(tmp,&str[begin],i-begin);
				} else {
					tmp = NULL;
				}
			} else {
				tmp = NULL;
			}

			disk_info_set_dir ( p, tmp);
			tmp = NULL;
		}

		i = j + 1;
		begin = i;

		/* Read the disk type. */
		for ( j = i; ((str[j] != ':') && (str[j] != '\0')); j++ ) {
			if ( str[j] == '\\') {
				if ( str[j+1] == '#') {
					str[i++] = ':';
					j++;
				} else if ( str[j+1] == 'n') {
					str[i++] = '\n';
					j++;
				} else {
					str[i++] = str[j];
				}
			} else {
				str[i++] = str[j];
			}
		}

		if ( begin < i) {
			if ( (file_version==1) && (i-begin==6) && (memcmp(&str[begin],"CD-ROM",6)==0) ) {
				tmp = g_strdup("iso9660");
			} else if ( (tmp = (gchar*)g_malloc0(sizeof(gchar)*(i-begin+1))) != NULL ) {
				memcpy(tmp,&str[begin],i-begin);
			} else {
				tmp = NULL;
			}
		} else {
			tmp = NULL;
		}

		disk_info_set_type ( p, tmp);
		tmp = NULL;

		if ( file_version == 1 ) {
			/* We can guess disk_fsname but not disk_dir. */
			if ( (i-begin==6) && (memcmp ( &str[begin], "CD-ROM", 6) == 0) ) {
				disk_info_set_fsname ( p, g_strdup ( "/dev/cdrom"));
			} else if ( (i-begin>2) && (memcmp ( &str[begin], "HD", 2) == 0) ) {
				disk_info_set_fsname ( p, g_strdup ( "/dev/hd"));
			} else if ( (i-begin==6) && (memcmp ( &str[begin], "Floppy", 6) == 0) ) {
				disk_info_set_fsname ( p, g_strdup ( "/dev/fd"));
			} else if ( (i-begin==3) && (memcmp ( &str[begin], "Zip", 3) == 0) ) {
				disk_info_set_fsname ( p, g_strdup ( "/dev/zip"));
			} else {
			}
		}

		i = j + 1;
		begin = i;

		/* Read the disk volume name. */
		for ( j = i; ((str[j] != ':') && (str[j] != '\0')); j++ ) {
			if ( str[j] == '\\') {
				if ( str[j+1] == '#') {
					str[i++] = ':';
					j++;
				} else if ( str[j+1] == 'n') {
					str[i++] = '\n';
					j++;
				} else {
					str[i++] = str[j];
				}
			} else {
				str[i++] = str[j];
			}
		}

		if ( begin < i) {
			if ( memcmp ( &str[begin], "(null)", 6) != 0 ) {
				if ( (tmp = (gchar*)g_malloc0(sizeof(gchar)*(i-begin+1))) != NULL ) {
					memcpy(tmp,&str[begin],i-begin);
				} else {
					tmp = NULL;
				}
			} else {
				tmp = NULL;
			}
		} else {
			tmp = NULL;
		}

		disk_info_set_volume ( p, tmp);
		tmp = NULL;

		i = j + 1;
		begin = i;

		/* Read the disk used space. */
		for ( j = i; ((str[j] != ':') && (str[j] != '\0')); j++ ) {
			i++;
		}

		if ( str[j] == ':') {
			str[j] = '\0';
			disk_info_set_full ( p, strtoui64 ( &str[begin], NULL, 10));
		} else {
		}

		i = j + 1;
		begin = i;

		/* Read the disk free space. */
		for ( j = i; ((str[j] != ':') && (str[j] != '\0')); j++ ) {
			i++;
		}

		if ( str[j] == ':') {
			str[j] = '\0';
			disk_info_set_free ( p, strtoui64 ( &str[begin], NULL, 10));
		} else {
		}

		i = j + 1;
		begin = i;

		/* Read the disk adding date. */
		for ( j = i; ((str[j] != ':') && (str[j] != '\0')); j++ ) {
			i++;
		}

		if ( str[j] == ':') {
			str[j] = '\0';
			disk_info_set_date ( p, strtoul ( &str[begin], NULL, 10));
		} else {
		}

		i = j + 1;
		begin = i;

		/* Read the disk serial number. */
		for ( j = i; ((str[j] != ':') && (str[j] != '\0')); j++ ) {
			if ( str[j] == '\\') {
				if ( str[j+1] == '#') {
					str[i++] = ':';
					j++;
				} else if ( str[j+1] == 'n') {
					str[i++] = '\n';
					j++;
				} else {
					str[i++] = str[j];
				}
			} else {
				str[i++] = str[j];
			}
		}

		if ( begin < i) {
			if ( memcmp ( &str[begin], "(null)", 6) != 0 ) {
				if ( (tmp = (gchar*)g_malloc0(sizeof(gchar)*(i-begin+1))) != NULL ) {
					memcpy(tmp,&str[begin],i-begin);
				} else {
					tmp = NULL;
				}
			} else {
				tmp = NULL;
			}
		} else {
			tmp = NULL;
		}

		disk_info_set_serial ( p, tmp);
		tmp = NULL;

		if ( file_version == 2) {
			i = j + 1;
			begin = i;

			/* Read the disk category. */
			for ( j = i; ((str[j] != ':') && (str[j] != '\0')); j++ ) {
				i++;
			}

			if ( str[j] == ':') {
				str[j] = '\0';
				if ( catalog_info != NULL ) {
					disk_info_set_category ( p, catalog_file_info_get_category_by_index ( catalog_info, strtol ( &str[begin], NULL, 10)));
				}
			} else {
			}
		} else {
		}

		i = j + 1;
		begin = i;

		/* Read the disk description. */
		for ( j = i; ((str[j] != ':') && (str[j] != '\0')); j++ ) {
			if ( str[j] == '\\') {
				if ( str[j+1] == '#') {
					str[i++] = ':';
					j++;
				} else if ( str[j+1] == 'n') {
					str[i++] = '\n';
					j++;
				} else {
					str[i++] = str[j];
				}
			} else {
				str[i++] = str[j];
			}
		}

		if ( begin < i) {
			if ( (tmp = (gchar*)g_malloc0(sizeof(gchar)*(i-begin+1))) != NULL ) {
				memcpy(tmp,&str[begin],i-begin);
			} else {
				tmp = NULL;
			}
		} else {
			tmp = NULL;
		}

		disk_info_set_description ( p, tmp);
		tmp = NULL;
	}

	return p;
}


gchar * catalog_disk_info_to_file ( struct disk_info *p, gint file_version)
{
	/* Bug to fix : g_strdup_printf() fails with too long string...
	   In order to prevent this bug, I decomposed buffer in two parts : disk porperties and description. */
	gchar *buf = NULL;
	gchar *tmp[7];
	gchar *tmp_buf = NULL;


#ifdef GW_DEBUG_MODE
	gw_application_manager_log_msg ( 0, __FILE__, __LINE__, __PRETTY_FUNCTION__, NULL);
#endif

	if ( p == NULL )
	{
		return NULL;
	}

	switch ( file_version)
	{
		case 1:	tmp[0] = gw_str_to_file ( disk_info_get_name ( p));
			tmp[1] = gw_str_to_file ( disk_info_get_type ( p));
			tmp[2] = gw_str_to_file ( disk_info_get_volume ( p));
			tmp[3] = gw_str_to_file ( disk_info_get_serial ( p));
			tmp[4] = gw_str_to_file ( disk_info_get_description ( p));

			buf = g_strdup_printf ( FORMAT1_DISK_SHORT,
						tmp[0],
						disk_info_get_num ( p),
						tmp[1],
						tmp[2],
						disk_info_get_full ( p),
						disk_info_get_free ( p),
						disk_info_get_date ( p),
						tmp[3]/*,
						tmp[4]*/);

			tmp_buf = g_strconcat ( buf, tmp[4], "\n", NULL);
			g_free ( buf);
			buf = tmp_buf;

			if ( tmp[0] != NULL ) g_free ( tmp[0]);
			if ( tmp[1] != NULL ) g_free ( tmp[1]);
			if ( tmp[2] != NULL ) g_free ( tmp[2]);
			if ( tmp[3] != NULL ) g_free ( tmp[3]);
			if ( tmp[4] != NULL ) g_free ( tmp[4]);

			return buf;

		case 2: tmp[0] = gw_str_to_file ( disk_info_get_name ( p));
			tmp[1] = gw_str_to_file ( disk_info_get_fsname ( p));
			tmp[2] = gw_str_to_file ( disk_info_get_dir ( p));
			tmp[3] = gw_str_to_file ( disk_info_get_type ( p));
			tmp[4] = gw_str_to_file ( disk_info_get_volume ( p));
			tmp[5] = gw_str_to_file ( disk_info_get_serial ( p));
			tmp[6] = gw_str_to_file ( disk_info_get_description ( p));

			buf = g_strdup_printf ( FORMAT2_DISK_SHORT,
						tmp[0],
						disk_info_get_num ( p),
						tmp[1],
						tmp[2],
						tmp[3],
						tmp[4],
						disk_info_get_full ( p),
						disk_info_get_free ( p),
						disk_info_get_date ( p),
						tmp[5],
						disk_info_get_category_index ( p)/*,
						tmp[6]*/);

			tmp_buf = g_strconcat ( buf, tmp[6], "\n", NULL);
			g_free ( buf);
			buf = tmp_buf;

			if ( tmp[0] != NULL ) g_free ( tmp[0]);
			if ( tmp[1] != NULL ) g_free ( tmp[1]);
			if ( tmp[2] != NULL ) g_free ( tmp[2]);
			if ( tmp[3] != NULL ) g_free ( tmp[3]);
			if ( tmp[4] != NULL ) g_free ( tmp[4]);
			if ( tmp[5] != NULL ) g_free ( tmp[5]);
			if ( tmp[6] != NULL ) g_free ( tmp[6]);

			return buf;

		default:return NULL;
	}

	return NULL;
}


struct file_info * catalog_file_info_new_from_table ( gchar **attrib, gboolean folder, gint file_version)
{
	struct file_info *p = NULL;


#ifdef GW_DEBUG_MODE
	gw_application_manager_log_msg ( 0, __FILE__, __LINE__, __PRETTY_FUNCTION__, NULL);
#endif

	p = file_info_new ( );
	switch ( file_version )
	{
		case 1:	if ( folder )
			{
				file_info_set_name ( p, gw_file_to_str ( attrib[0]));
				file_info_set_rights_from_gchar ( p, attrib[1]);
				file_info_set_owner ( p, g_strdup ( attrib[2]));
				file_info_set_group ( p, g_strdup ( attrib[3]));
				file_info_set_inode ( p, strtol ( attrib[4], NULL, 10));
				file_info_set_size ( p, strtoui64 ( attrib[5], NULL, 10));
				file_info_set_cdate ( p, strtoul ( attrib[6], NULL, 10));
				file_info_set_adate ( p, strtoul ( attrib[7], NULL, 10));
				file_info_set_mdate ( p, strtoul ( attrib[8], NULL, 10));
				if ( (attrib[9]!= NULL) && (strlen(attrib[9])>0))file_info_set_description ( p, gw_file_to_str ( attrib[9]));
			}
			else
			{
				file_info_set_name ( p, gw_file_to_str ( attrib[0]));
				file_info_set_rights_from_gchar ( p, attrib[1]);
				file_info_set_owner ( p, g_strdup ( attrib[2]));
				file_info_set_group ( p, g_strdup ( attrib[3]));
				file_info_set_inode ( p, strtol( attrib[4], NULL, 10));
				file_info_set_size ( p, strtoui64 ( attrib[5], NULL, 10));
				file_info_set_cdate ( p, strtoul ( attrib[6], NULL, 10));
				file_info_set_adate ( p, strtoul ( attrib[7], NULL, 10));
				file_info_set_mdate ( p, strtoul ( attrib[8], NULL, 10));
				if ( strlen(attrib[9])>0)file_info_set_description ( p, gw_file_to_str ( attrib[9]));
			}

			break;

		case 2:	if ( folder )
			{
				file_info_set_name ( p, gw_file_to_str ( attrib[0]));
				file_info_set_rights_from_gchar ( p, attrib[1]);
				file_info_set_owner ( p, g_strdup ( attrib[2]));
				file_info_set_group ( p, g_strdup ( attrib[3]));
				file_info_set_inode ( p, strtol ( attrib[4], NULL, 10));
				file_info_set_size ( p, strtoui64 ( attrib[5], NULL, 10));
				file_info_set_cdate ( p, strtoul ( attrib[6], NULL, 10));
				file_info_set_adate ( p, strtoul ( attrib[7], NULL, 10));
				file_info_set_mdate ( p, strtoul ( attrib[8], NULL, 10));
				if ( (attrib[10]!= NULL) && (strlen(attrib[10])>0))file_info_set_description ( p, gw_file_to_str ( attrib[10]));
			}
			else
			{
				file_info_set_name ( p, gw_file_to_str ( attrib[0]));
				file_info_set_rights_from_gchar ( p, attrib[1]);
				file_info_set_owner ( p, g_strdup ( attrib[2]));
				file_info_set_group ( p, g_strdup ( attrib[3]));
				file_info_set_inode ( p, strtol ( attrib[4], NULL, 10));
				file_info_set_size ( p, strtoui64 ( attrib[5], NULL, 10));
				file_info_set_cdate ( p, strtoul ( attrib[6], NULL, 10));
				file_info_set_adate ( p, strtoul ( attrib[7], NULL, 10));
				file_info_set_mdate ( p, strtoul ( attrib[8], NULL, 10));
				if ( (attrib[10]!= NULL) && (strlen(attrib[10])>0))file_info_set_description ( p, gw_file_to_str ( attrib[10]));
			}
			break;

		default:file_info_free ( p);
			return NULL;
	}

	return p;
}


struct file_info * catalog_file_info_new_from_str ( gchar *str, gint file_version, struct catalog_file_info *catalog_info)
{
	struct file_info *p = NULL;
	gint i = 0, j = 0, begin = 0;
	gchar *tmp = NULL;


#ifdef GW_DEBUG_MODE
	gw_application_manager_log_msg ( 0, __FILE__, __LINE__, __PRETTY_FUNCTION__, NULL);
#endif

	if ( (p = file_info_new ( )) != NULL ) {
		i = 0;
		begin = i;

		/* Read the file name. */
		for ( j = i; ((str[j] != ':') && (str[j] != '\0')); j++ ) {
			if ( str[j] == '\\') {
				if ( str[j+1] == '#') {
					str[i++] = ':';
					j++;
				} else if ( str[j+1] == 'n') {
					str[i++] = '\n';
					j++;
				} else {
					str[i++] = str[j];
				}
			} else {
				str[i++] = str[j];
			}
		}

		if ( begin < i) {
			if ( (tmp = (gchar*)g_malloc0(sizeof(gchar)*(i-begin+1))) != NULL ) {
				memcpy(tmp,&str[begin],i-begin);
			} else {
				tmp = NULL;
			}
		} else {
			tmp = NULL;
		}

		file_info_set_name ( p, tmp);
		tmp = NULL;

		i = j + 1;
		begin = i;

		/* Read the file rights. */
		for ( j = i; ((str[j] != ':') && (str[j] != '\0')); j++ ) {
			i++;
		}

		if ( begin < i) {
			if ( (tmp = (gchar*)g_malloc0(sizeof(gchar)*(i-begin+1))) != NULL ) {
				memcpy(tmp,&str[begin],i-begin);
			} else {
				tmp = NULL;
			}
		} else {
			tmp = NULL;
		}

		if ( tmp != NULL )
		{
			file_info_set_rights_from_gchar ( p, tmp);
			g_free ( tmp);
			tmp = NULL;
		}

		i = j + 1;
		begin = i;

		/* Read the file owner. */
		for ( j = i; ((str[j] != ':') && (str[j] != '\0')); j++ ) {
			i++;
		}

		if ( begin < i) {
			if ( (tmp = (gchar*)g_malloc0(sizeof(gchar)*(i-begin+1))) != NULL ) {
				memcpy(tmp,&str[begin],i-begin);
			} else {
				tmp = NULL;
			}
		} else {
			tmp = NULL;
		}

		file_info_set_owner ( p, tmp);

		i = j + 1;
		begin = i;

		/* Read the file group. */
		for ( j = i; ((str[j] != ':') && (str[j] != '\0')); j++ ) {
			i++;
		}

		if ( begin < i) {
			if ( (tmp = (gchar*)g_malloc0(sizeof(gchar)*(i-begin+1))) != NULL ) {
				memcpy(tmp,&str[begin],i-begin);
			} else {
				tmp = NULL;
			}
		} else {
			tmp = NULL;
		}

		file_info_set_group ( p, tmp);

		i = j + 1;
		begin = i;

		/* Read the file inode. */
		for ( j = i; ((str[j] != ':') && (str[j] != '\0')); j++ ) {
			i++;
		}

		if ( str[j] == ':') {
			str[j] = '\0';
			file_info_set_inode ( p, strtol ( &str[begin], NULL, 10));
		} else {
			tmp = NULL;
		}

		i = j + 1;
		begin = i;

		/* Read the file size. */
		for ( j = i; ((str[j] != ':') && (str[j] != '\0')); j++ ) {
			i++;
		}

		if ( str[j] == ':') {
			str[j] = '\0';
			file_info_set_size ( p, strtoui64 ( &str[begin], NULL, 10));
		} else {
		}

		i = j + 1;
		begin = i;

		/* Read the file cdate. */
		for ( j = i; ((str[j] != ':') && (str[j] != '\0')); j++ ) {
			i++;
		}

		if ( str[j] == ':') {
			str[j] = '\0';
			file_info_set_cdate ( p, strtoul ( &str[begin], NULL, 10));
		} else {
		}

		i = j + 1;
		begin = i;

		/* Read the file adate. */
		for ( j = i; ((str[j] != ':') && (str[j] != '\0')); j++ ) {
			i++;
		}

		if ( str[j] == ':') {
			str[j] = '\0';
			file_info_set_adate ( p, strtoul ( &str[begin], NULL, 10));
		} else {
		}

		i = j + 1;
		begin = i;

		/* Read the file mdate. */
		for ( j = i; ((str[j] != ':') && (str[j] != '\0')); j++ ) {
			i++;
		}

		if ( str[j] == ':') {
			str[j] = '\0';
			file_info_set_mdate ( p, strtoul ( &str[begin], NULL, 10));
		} else {
			if ( file_version == 1) {
				if ( str[j] == '\0') {
					file_info_set_mdate ( p, strtoul ( &str[begin], NULL, 10));
				}
			}
		}

		if ( file_version == 2) {
			i = j + 1;
			begin = i;

			/* Read the file category. */
			for ( j = i; ((str[j] != ':') && (str[j] != '\0')); j++ ) {
				i++;
			}

			if ( str[j] == ':') {
				str[j] = '\0';
				if ( catalog_info != NULL ) {
					file_info_set_category ( p, catalog_file_info_get_category_by_index ( catalog_info, strtol ( &str[begin], NULL, 10)));
				}
			} else {
			}
		} else {
		}

		i = j + 1;
		begin = i;

		/* Read the file description. */
		for ( j = i; ((str[j] != ':') && (str[j] != '\0')); j++ ) {
			if ( str[j] == '\\') {
				if ( str[j+1] == '#') {
					str[i++] = ':';
					j++;
				} else if ( str[j+1] == 'n') {
					str[i++] = '\n';
					j++;
				} else {
					str[i++] = str[j];
				}
			} else {
				str[i++] = str[j];
			}
		}

		if ( begin < i) {
			if ( (tmp = (gchar*)g_malloc0(sizeof(gchar)*(i-begin+1))) != NULL ) {
				memcpy(tmp,&str[begin],i-begin);
			} else {
				tmp = NULL;
			}
		} else {
			tmp = NULL;
		}

		file_info_set_description ( p, tmp);
	}

	return p;
}


gchar * catalog_file_info_to_file ( struct file_info *p, gint file_version)
{
	/* Bug to fix : g_strdup_printf() fails with too long string...
	   In order to prevent this bug, I decomposed buffer in two parts : disk porperties and description. */
	gchar *buf = NULL, *tmp_buf = NULL;
	gchar *tmp[3];


#ifdef GW_DEBUG_MODE
	gw_application_manager_log_msg ( 0, __FILE__, __LINE__, __PRETTY_FUNCTION__, NULL);
#endif

	if ( p == NULL )
	{
		return NULL;
	}

	switch ( file_version)
	{
		case 1:	buf = g_strdup_printf ( FORMAT1_FILE_LONG,
						tmp[0] = gw_str_to_file ( file_info_get_name ( p)),
						tmp[2] = file_info_get_rights_to_gchar( p),
						file_info_get_owner ( p),
						file_info_get_group ( p),
						file_info_get_inode ( p),
						file_info_get_size ( p),
						file_info_get_cdate ( p),
						file_info_get_adate ( p),
						file_info_get_mdate ( p),
						tmp[1] = gw_str_to_file ( file_info_get_description ( p)));

			if ( tmp[0] != NULL ) g_free ( tmp[0]);
			if ( tmp[1] != NULL ) g_free ( tmp[1]);
			if ( tmp[2] != NULL ) g_free ( tmp[2]);

			return buf;

		case 2:	buf = g_strdup_printf ( /*FORMAT2_FILE_LONG*/FORMAT2_FILE_SHORT,
						tmp[0] = gw_str_to_file ( file_info_get_name ( p)),
						tmp[2] = file_info_get_rights_to_gchar ( p),
						file_info_get_owner ( p),
						file_info_get_group ( p),
						file_info_get_inode ( p),
						file_info_get_size ( p),
						file_info_get_cdate ( p),
						file_info_get_adate ( p),
						file_info_get_mdate ( p),
						file_info_get_category_index ( p)/*,
						tmp[1] = gw_str_to_file ( file_info_get_description ( p))*/);

			tmp_buf = g_strconcat ( buf, tmp[1] = gw_str_to_file ( file_info_get_description ( p)), "\n", NULL);
			g_free ( buf);
			buf = tmp_buf;

			if ( tmp[0] != NULL ) g_free ( tmp[0]);
			if ( tmp[1] != NULL ) g_free ( tmp[1]);
			if ( tmp[2] != NULL ) g_free ( tmp[2]);

			return buf;

		default:return NULL;
	}

	return NULL;
}
