/*  GWhere.
 *  Copyright (C) 2000  Sbastien LECACHEUR
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


/*!
 * @header	gwfileinfo.h
 *		GWhere<br>
 *		Copyright (C) 2000  Sbastien LECACHEUR<br><br>
 *		This Distributed under the terms of the <a href="http://www.gnu.org/copyleft/gpl.html">GNU General Public Licence</a><br><br>
 *		This component was developed for the GWhere project.<br>
 *		It's an object for file representation. His structure gets many informations as name, size, owner, ...<br><br>
 *		Category field is collaborative. All other fields are agregated.
 * @abstract	This is the file information data structure.
 * @author	Sbastien LECACHEUR <zero@gwhere.org>
 * @version	1.0 2001/08/21
 * @see		glib.h
 * @see		gwsupport.h
 * @see		gwcategory.h
 * @link		GWhere Web Site <http://www.gwhere.org>
 * @link		GTK+ Web Site <http://www.gtk.org>
 * @ref		gwfileinfo.c
 * @author	Author <zero@gwhere.org>
 */


#ifndef GW_DATA_FILE_INFO_H
#define GW_DATA_FILE_INFO_H


#include <glib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include "gwcategory.h"


/*!
 * @struct	file_info
 * @abstract	File information data structure
 * @discussion	File information is an opaque data structure containing information about file properties and informations.
 */
struct file_info;


/*!
 * @function	file_info_new
 * @abstract	Creates a new file information.
 * @discussion	This function creates a new file information. It's an allocation function.
 * @result	a pointer to the file information structure.<br>
 *		Returns <i>NULL</i> when an error occured.<br>
 */
struct file_info * file_info_new ( void);


/*!
 * @function	file_info_set_name
 * @abstract	Sets the file name.
 * @discussion	This function sets the name of the file
 * @param	p	the file information to set the name
 * @param	name	the name
 * @result	an error code.<br>
 *		Returns <i>0</i> when it's done.<br>
 *		Returns <i>-1</i> when an error occured.<br>
 */
gint file_info_set_name ( struct file_info *p, gchar *name);


/*!
 * @function	file_info_set_rights
 * @abstract	Sets the file rights.
 * @discussion	This function sets the file information rights.<br>
 *		Sample : 000766
 * @param	p	the file information to set rights
 * @param	rights	the rights under bit format
 * @result	an error code.<br>
 *		Returns <i>0</i> when it's done.<br>
 *		Returns <i>-1</i> when an error occured.<br>
 */
gint file_info_set_rights ( struct file_info *p, mode_t rights);


/*!
 * @function	file_info_set_rights_from_gchar
 * @abstract	Sets the file rights from the string format.
 * @discussion	This function sets the file information rights from string format.<br>
 *		Sample : -rwxrwxrwx
 * @param	p	the file information to set rights
 * @param	rights	the rights under string format
 * @result	an error code.<br>
 *		Returns <i>0</i> when it's done.<br>
 *		Returns <i>-1</i> when an error occured.<br>
 */
gint file_info_set_rights_from_gchar ( struct file_info *p, gchar *rights);


/*!
 * @function	file_info_set_owner
 * @abstract	Sets the file owner.
 * @discussion	This function sets the file information owner.<br>
 * @param	p	the file information to set owner
 * @param	owner	the owner
 * @result	an error code.<br>
 *		Returns <i>0</i> when it's done.<br>
 *		Returns <i>-1</i> when an error occured.<br>
 */
gint file_info_set_owner ( struct file_info *p, gchar *owner);


/*!
 * @function	file_info_set_group
 * @abstract	Sets the file owner group.
 * @discussion	This function sets the file information owner group.<br>
 * @param	p	the file information to set owner group
 * @param	group	the group
 * @result	an error code.<br>
 *		Returns <i>0</i> when it's done.<br>
 *		Returns <i>-1</i> when an error occured.<br>
 */
gint file_info_set_group ( struct file_info *p, gchar *group);


/*!
 * @function	file_info_set_size
 * @abstract	Sets the file size.
 * @discussion	This function sets the file information size.<br>
 * @param	p	the file information to set size
 * @param	size	the size
 * @result	an error code.<br>
 *		Returns <i>0</i> when it's done.<br>
 *		Returns <i>-1</i> when an error occured.<br>
 */
gint file_info_set_size ( struct file_info *p, guint64 size);


/*!
 * @function	file_info_set_inode
 * @abstract	Sets the file inode.
 * @discussion	This function sets the file information inode.<br>
 * @param	p	the file information to set inode
 * @param	inode	the inode
 * @result	an error code.<br>
 *		Returns <i>0</i> when it's done.<br>
 *		Returns <i>-1</i> when an error occured.<br>
 */
gint file_info_set_inode ( struct file_info *p, glong inode);


/*!
 * @function	file_info_set_cdate
 * @abstract	Sets the file creation date.
 * @discussion	This function sets the file information creation date.<br>
 * @param	p	the file information to set creation date
 * @param	cdate	the creation date
 * @result	an error code.<br>
 *		Returns <i>0</i> when it's done.<br>
 *		Returns <i>-1</i> when an error occured.<br>
 */
gint file_info_set_cdate ( struct file_info *p, gulong cdate);


/*!
 * @function	file_info_set_adate
 * @abstract	Sets the file last access date.
 * @discussion	This function sets the file information last access date.<br>
 * @param	p	the file information to set last access date
 * @param	adate	the last access date
 * @result	an error code.<br>
 *		Returns <i>0</i> when it's done.<br>
 *		Returns <i>-1</i> when an error occured.<br>
 */
gint file_info_set_adate ( struct file_info *p, gulong adate);


/*!
 * @function	file_info_set_mdate
 * @abstract	Sets the file last modification date.
 * @discussion	This function sets the file last information modification date.<br>
 * @param	p	the information to set last modification date
 * @param	mdate	the last modification date
 * @result	an error code.<br>
 *		Returns <i>0</i> when it's done.<br>
 *		Returns <i>-1</i> when an error occured.<br>
 */
gint file_info_set_mdate ( struct file_info *p, gulong mdate);


/*!
 * @function	file_info_set_category
 * @abstract	Sets the file category.
 * @discussion	This function sets the file information category. The category is <i>struct category</i><br>
 * @param	p	the information to set category
 * @param	category	the category
 * @result	an error code.<br>
 *		Returns <i>0</i> when it's done.<br>
 *		Returns <i>-1</i> when an error occured.<br>
 */
gint file_info_set_category ( struct file_info *p, struct category *category);


/*!
 * @function	file_info_set_description
 * @abstract	Sets the file description.
 * @discussion	This function sets the file information description.<br>
 * @param	p	the file information to set description
 * @param	description	the description
 * @result	an error code.<br>
 *		Returns <i>0</i> when it's done.<br>
 *		Returns <i>-1</i> when an error occured.<br>
 */
gint file_info_set_description ( struct file_info *p, gchar *description);


/*!
 * @function	file_info_get_name
 * @abstract	Gets the file name.
 * @discussion	This function gets the file information name.<br>
 * @param	p	the file information to get name
 * @result	The name.<br>
 *		Returns <i>NULL</i> when an error occured.<br>
 */
gchar * file_info_get_name ( struct file_info *p);


/*!
 * @function	file_info_get_isdirectory
 * @abstract	Gets if is a folder.
 * @discussion	This function gets if the file information is folder.<br>
 * @param	p	the file information to get if is a folder
 * @result	<i>TRUE</i> if is a folder or <i>FALSE</i> if is a file.<br>
 *		Returns <i>FALSE</i> when an error occured.<br>
 */
gboolean file_info_get_isdirectory ( struct file_info *p);


/*!
 * @function	file_info_get_is_real_directory
 * @abstract	Gets if is a real folder.
 * @discussion	This function gets if the file information is a real folder.<br>
 *		"." and ".." are not a real folder
 * @param	p	the file information to get if is a folder
 * @result	<i>TRUE</i> if is a folder or <i>FALSE</i> if is a file.<br>
 *		Returns <i>FALSE</i> when an error occured.<br>
 */
gboolean file_info_get_is_real_directory ( struct file_info *p);


/*!
 * @function	file_info_get_rights
 * @abstract	Gets the file rights under bits format.
 * @discussion	this function gets the file information rights under bits format.<br>
 * @param	p	the file information to get rights
 * @result	The bits format rights.<br>
 *		Returns <i>0</i> when an error occured.<br>
 */
mode_t file_info_get_rights ( struct file_info *p);


/*!
 * @function	file_info_get_rights_to_gchar
 * @abstract	Gets the file rights under string format.
 * @discussion	this function gets the file information rights under string format.<br>
 * @param	p	the file information to get rights
 * @result	The string format rights.<br>
 *		Returns <i>NULL</i> when an error occured.<br>
 */gchar* file_info_get_rights_to_gchar ( struct file_info *p);


/*!
 * @function	file_info_get_owner
 * @abstract	Gets the file owner.
 * @discussion	This function gets the file information owner.<br>
 * @param	p	the file information to get owner
 * @result	The owner.<br>
 *		Returns <i>NULL</i> when an error occured.<br>
 */
gchar * file_info_get_owner ( struct file_info *p);


/*!
 * @function	file_info_get_group
 * @abstract	Gets the file owner group.
 * @discussion	This function gets the file information owner group.<br>
 * @param	p	the file information to get owner group
 * @result	The owner group.<br>
 *		Returns <i>NULL</i> when an error occured.<br>
 */
gchar * file_info_get_group ( struct file_info *p);


/*!
 * @function	file_info_get_size
 * @abstract	Gets the file size.
 * @discussion	This functions gets the file information size.<br>
 * @param	p	the file information to get size
 * @result	The size.<br>
 *		Returns <i>-1</i> when an error occured.<br>
 */
guint64 file_info_get_size ( struct file_info *p);


/*!
 * @function	file_info_get_inode
 * @abstract	Gets the file inode.
 * @discussion	This function gets the file information inode.<br>
 * @param	p	the file information to get inode
 * @result	The inode.<br>
 *		Returns <i>-1</i> when an error occured.<br>
 */
glong file_info_get_inode ( struct file_info *p);


/*!
 * @function	file_info_get_cdate
 * @abstract	Gets the file creation date.
 * @discussion	This function gets the file information creation date.<br>
 * @param	p	the file information to get creation date
 * @result	The creation date.<br>
 *		Returns <i>0</i> when an error occured.<br>
 */
gulong file_info_get_cdate ( struct file_info *p);


/*!
 * @function	file_info_get_adate
 * @abstract	Gets the file last access date.
 * @discussion	This function gets the file information last access date.<br>
 * @param	p	the file information to get last access date
 * @result	The last access date.<br>
 *		Returns <i>0</i> when an error occured.<br>
 */
gulong file_info_get_adate ( struct file_info *p);


/*!
 * @function	file_info_get_mdate
 * @abstract	Gets the file last modification date.
 * @discussion	This function gets the file information last modification date.<br>
 * @param	p	the file information to get last modification date
 * @result	The last modification date.<br>
 *		Returns <i>0</i> when an error occured.<br>
 */
gulong file_info_get_mdate ( struct file_info *p);


/*!
 * @function	file_info_get_category
 * @abstract	Gets the file category.
 * @discussion	This function gets the file information category. The category is the category datas structure.<br>
 * @param	p	the file information to get category
 * @result	The category.<br>
 *		Returns <i>NULL</i> when an error occured.<br>
 */
struct category * file_info_get_category ( struct file_info *p);


/*!
 * @function	file_info_get_category_name
 * @abstract	Gets the file category name.
 * @discussion	This function gets the file category name.<br>
 * @param	p	the file information to get category name
 * @result	The category name.<br>
 *		Returns <i>""</i> when there are not category.<br>
 *		Returns <i>NULL</i> when an error occured.<br>
 */
gchar * file_info_get_category_name ( struct file_info *p);


/*!
 * @function	file_info_get_category_index
 * @abstract	Gets the file category index.
 * @discussion	This function gets the file information category index.<br>
 * @param	p	the file information to get category index
 * @result	The category index.<br>
 *		Returns <i>-1</i> when an error occured.<br>
 */
gint file_info_get_category_index ( struct file_info *p);


/*!
 * @function	file_info_get_description
 * @abstract	Gets the file description.
 * @discussion	This function gets the file information description.<br>
 * @param	p	the file information to get description
 * @result	The description.<br>
 *		Returns <i>NULL</i> when an error occured.<br>
 */
gchar * file_info_get_description ( struct file_info *p);


/*!
 * @function	file_info_get_cdate_to_gchar
 * @abstract	Gets the formated file creation date.
 * @discussion	This function gets the formated file information creation date.<br>
 * @param	p	the file information to get formated creation date
 * @result	The formated creation date. Must be freed with g_free().<br>
 *		Returns <i>NULL</i> when an error occured.<br>
 */
gchar * file_info_get_cdate_to_gchar ( struct file_info *p);


/*!
 * @function	file_info_get_adate_to_gchar
 * @abstract	Gets the formated file last access date.
 * @discussion	This function gets the formated file information last access date.<br>
 * @param	p	the file information to get formated last access date
 * @result	The formated last access date. Must be freed with g_free().<br>
 *		Returns <i>NULL</i> when an error occured.<br>
 */
gchar * file_info_get_adate_to_gchar ( struct file_info *p);


/*!
 * @function	file_info_get_mdate_to_gchar
 * @abstract	Gets the formated file last modification date.
 * @discussion	This function gets the formated file information last modification date.<br>
 * @param	p	the file information to get last modification date
 * @result	The formated last modification date. Must be freed with g_free().<br>
 *		Returns <i>NULL</i> when an error occured.<br>
 */
gchar * file_info_get_mdate_to_gchar ( struct file_info *p);


/*!
 * @function	file_info_free
 * @abstract	Frees datas structure.
 * @discussion	This function frees the file information datas structure.<br>
 *		Doesn't free Category because it's collaborative.<br>
 * @param	p	the file information to free
 * @result	an error code.<br>
 *		Returns <i>0</i> when it's done.<br>
 *		Returns <i>-1</i> when an error occured.<br>
 */
gint file_info_free ( struct file_info *p);


#endif
