# -*- coding: utf-8 -*-
# Gufw 13.10.0 - http://gufw.org
# Copyright (C) 2008-2013 Marcos Alvarez Costales https://launchpad.net/~costales
#
# Gufw is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
# 
# Gufw is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with Gufw; if not, see http://www.gnu.org/licenses for more
# information.


from gi.repository import Gtk, Gdk, GObject
from gi.repository import cairo, Pango, PangoCairo
import os

from constants import *

import gettext
from gettext import gettext as _
gettext.textdomain('gufw')


class UtilsView:
    def __init__(self, fw_frontend, rules_model, statusbar, log, log_txt):
        self.fw = fw_frontend
        self.rules_model = rules_model
        self.statusbar = statusbar
        self.log = log
        self.log_txt = log_txt
        
        self.txt_color_green  = self.log_txt.create_tag('color_green',    foreground=GREEN)
        self.txt_color_red    = self.log_txt.create_tag('color_red',      foreground=RED)
        self.txt_color_orange = self.log_txt.create_tag('colored_orange', foreground=ORANGE)
        self.txt_color_blue   = self.log_txt.create_tag('colored_blue',   foreground=BLUE)
        self.txt_color_gray   = self.log_txt.create_tag('colored_gray',   foreground=GRAY)
        self.txt_color_black  = self.log_txt.create_tag('colored_black',  foreground=BLACK)

    def _get_log_color(self, color):
        if color == GREEN:
            return self.txt_color_green
        if color == RED:
            return self.txt_color_red
        if color == ORANGE:
            return self.txt_color_orange
        if color == BLUE:
            return self.txt_color_blue
        if color == GRAY:
            return self.txt_color_gray
            
        return self.txt_color_black
    
    def show_dialog(self, header, msg):
        dialog = Gtk.MessageDialog(None, 0, Gtk.MessageType.INFO, Gtk.ButtonsType.OK, header)
        dialog.format_secondary_text(msg)
        dialog.run()
        dialog.destroy()
        
    def show_question(self, header, msg, question):
        reset_dialog = Gtk.MessageDialog(None,
                       Gtk.DialogFlags.MODAL | Gtk.DialogFlags.DESTROY_WITH_PARENT,
                       Gtk.MessageType.WARNING, Gtk.ButtonsType.NONE,
                       msg)
        reset_dialog.format_secondary_markup(question)
        reset_dialog.set_title(header)
        reset_dialog.add_buttons(Gtk.STOCK_CANCEL, Gtk.ResponseType.CANCEL, Gtk.STOCK_YES, Gtk.ResponseType.YES)
        reset_answer = reset_dialog.run()
        reset_dialog.destroy()
        if reset_answer == Gtk.ResponseType.YES:
            return True
        else:
            return False
    
    def add_to_log(self, msg, color=BLACK, force_save=True):
        if force_save:
            new_line = self.fw.add_to_log(msg)
            self.log_txt.insert_with_tags(self.log_txt.get_start_iter(), new_line, self._get_log_color(color)) 
        else:
            self.log_txt.insert_with_tags(self.log_txt.get_end_iter(), msg, self._get_log_color(color)) 
    
    def set_statusbar_msg(self, msg):
        cid = self.statusbar.get_context_id('default context')
        self.statusbar.push(cid, msg)
    
    def print_rules(self, rules):
        self.rules_model.clear()
        if not self.fw.get_status():
            return
        
        row = 1
        for rule in (rules):
            iter = self.rules_model.insert(row)
            translated_rule = rule['ufw_rule'].replace(" ALLOW ", _(" ALLOW "))
            translated_rule = translated_rule.replace(" DENY ",   _(" DENY "))
            translated_rule = translated_rule.replace(" REJECT ", _(" REJECT "))
            translated_rule = translated_rule.replace(" LIMIT ",  _(" LIMIT "))
            translated_rule = translated_rule.replace(" OUT ",    _(" OUT "))
            translated_rule = translated_rule.replace(" IN ",     _(" IN "))
            translated_rule = translated_rule.replace("Anywhere", _("Anywhere"))
            translated_rule = translated_rule.replace(" (out)",   _(" (out)"))
            translated_rule = translated_rule.replace(" on ",     _(" on "))
            
            self.rules_model.set_value(iter, 0,  translated_rule)    # ufw rule
            self.rules_model.set_value(iter, 1,  rule['description']) # description
            self.rules_model.set_value(iter, 2,  rule['command'])     # command
            self.rules_model.set_value(iter, 3,  rule['policy'])      # policy
            self.rules_model.set_value(iter, 4,  rule['direction'])   # direction
            self.rules_model.set_value(iter, 5,  rule['protocol'])    # proto
            self.rules_model.set_value(iter, 6,  rule['from_ip'])     # from_ip
            self.rules_model.set_value(iter, 7,  rule['from_port'])   # from_port
            self.rules_model.set_value(iter, 8,  rule['to_ip'])       # to_ip
            self.rules_model.set_value(iter, 9,  rule['to_port'])     # to_port
            self.rules_model.set_value(iter, 10, rule['iface'])       # iface
            self.rules_model.set_value(iter, 11, rule['logging'])     # logging
            if "ALLOW" in rule['ufw_rule']:
                self.rules_model.set_value(iter, 12, POLICY2COLOR['allow']) # color
            elif "DENY" in rule['ufw_rule']:
                self.rules_model.set_value(iter, 12, POLICY2COLOR['deny']) # color
            elif "REJECT" in rule['ufw_rule']:
                self.rules_model.set_value(iter, 12, POLICY2COLOR['reject']) # color
            elif "LIMIT" in rule['ufw_rule']:
                self.rules_model.set_value(iter, 12, POLICY2COLOR['limit']) # color
            else:
                self.rules_model.set_value(iter, 12, POLICY2COLOR['others']) # color
            
            row += 1
    
    def file_dialog(self, type_dialog, title):
        if type_dialog == 'open':
            type_win = Gtk.FileChooserAction.OPEN
            stock_win = Gtk.STOCK_OPEN
        else:
            type_win = Gtk.FileChooserAction.SAVE
            stock_win = Gtk.STOCK_SAVE
        
        dialog = Gtk.FileChooserDialog(title, None,
                                       type_win,
                                      (Gtk.STOCK_CANCEL, Gtk.ResponseType.CANCEL,
                                       stock_win, Gtk.ResponseType.OK))
        filter_profile = Gtk.FileFilter()
        filter_profile.set_name("Gufw profile")
        filter_profile.add_pattern("*.profile")
        dialog.add_filter(filter_profile)
        filter_all = Gtk.FileFilter()
        filter_all.set_name("All files")
        filter_all.add_pattern("*")
        dialog.add_filter(filter_all)
        
        response = dialog.run()
        select = ''
        if response == Gtk.ResponseType.OK:
            select = dialog.get_filename()
            if type_dialog == 'save' and os.path.exists(dialog.get_filename()):
                if not self.show_question(_("File exists"), _("The file already exists"), _(" Replacing it will overwrite its contents. Do you want to replace it?")):
                    select = ''
        dialog.destroy()    
        return select
        
    def restore_window_size(self, win):
        width = 0
        height = 0
        if self.fw.get_config_value('WindowWidth'):
            width = int(self.fw.get_config_value('WindowWidth'))
        if self.fw.get_config_value('WindowHeight'):
            height = int(self.fw.get_config_value('WindowHeight'))
        
        if width >= Gdk.Screen.width() and height >= Gdk.Screen.height():
            win.maximize()
        elif width != 0 and height != 0:
            win.resize(width, height)            
            
    def save_window_size(self, win):
        width, height = win.get_size()
        self.fw.set_config_value('WindowWidth',  str(width))
        self.fw.set_config_value('WindowHeight', str(height))
    
    def validate_rule(self, from_ip, from_port, to_ip, to_port):
        # At least 1 Port/IP
        if not from_ip and not from_port and not to_ip and not to_port:
            self.show_dialog(_("Insert IP/Ports"), _("You need to insert IP/ports in to/from fields"))
            return False
        
        # If 2 ports > 2 IPs
        if (from_port and to_port) and (not from_ip or not to_ip):
            self.show_dialog(_("Insert both IPs"), _("You need to insert both IPs if you're using both ports"))
            return False
            
        return True
