# -*- coding: utf-8 -*-
# Gufw 13.10.0 - http://gufw.org
# Copyright (C) 2008-2013 Marcos Alvarez Costales https://launchpad.net/~costales
#
# Gufw is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
# 
# Gufw is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with Gufw; if not, see http://www.gnu.org/licenses for more
# information.


from gi.repository import Gtk, GObject
import os, re

from constants import *
import gettext
from gettext import gettext as _
gettext.textdomain('gufw')


class WinPreferences:
    """All events"""
    def __init__(self, fw_frontend, utils, profile_cb, winMain):
        self.fw = fw_frontend
        self.utils = utils
        
        self.builder = Gtk.Builder()
        self.builder.add_from_file('/usr/share/gufw/ui/preferences.ui')
        self._set_objects_name(profile_cb)
        self._set_initial_values()
        self.win_preferences.set_transient_for(winMain)
        self.builder.connect_signals(self)
        self.win_preferences.show_all()

    def _set_objects_name(self, profile_cb):
        # Init from main Gufw window
        self.main_profile_cb = profile_cb
        # Preferences window
        self.win_preferences = self.builder.get_object('preferences')
        self.ufw_logging     = self.builder.get_object('ufw_logging')
        self.gufw_logging    = self.builder.get_object('gufw_logging')
        self.list_profiles   = self.builder.get_object('profiles_list')
        self.list_selection  = self.builder.get_object('profiles_selection')
    
    def _set_initial_values(self):
        self.ufw_logging.set_active(UFW_LEVEL2NUM[self.fw.get_ufw_logging()])
        
        if self.fw.get_logging():
            self.gufw_logging.set_active(True)
        
        # Profiles (model)
        self.profile_rows = Gtk.ListStore(str)
        for profile in self.fw.get_all_profiles():
            self.profile_rows.append([profile])
        self.list_profiles.set_model(self.profile_rows)
        # Profiles (Editable one column)
        renderer_editable_profile = Gtk.CellRendererText()
        renderer_editable_profile.set_property('editable', True)
        type_profile_column = Gtk.TreeViewColumn(_("Profile"), renderer_editable_profile, text=0)
        self.list_profiles.append_column(type_profile_column)
        # Profiles (Edit row)
        renderer_editable_profile.connect('edited', self._rename_profile)
    
    def _rename_profile(self, widget, path, new_name):
        """2 Click on profile"""
        # None is for internal use
        if new_name == 'None':
            self.utils.show_dialog(_("Profile not valid"), _("You can't use this profile name"))
            return
        # Not empty
        if not new_name:
            self.utils.show_dialog(_("Profile not valid"), _("Enter at least one character"))
            return
        # Check only ASCII characters and no spaces
        if not re.match('^[A-Za-z0-9_-]*$', new_name):
            self.utils.show_dialog(_("Profile not valid"), _("Use only letters and numbers"))
            return
        # Exist?
        for searched_profile in self.profile_rows:
            if searched_profile[0] == new_name:
                self.utils.show_dialog(_("Profile exist"), _("There is a profile with the same name"))
                return
        # Not the default
        previous_name = self.profile_rows[path][0]
        if previous_name == self.fw.get_profile():
            self.utils.show_dialog(_("Current profile"), _("You can't rename the current profile"))
            return
        
        self.fw.rename_profile(self.profile_rows[path][0], new_name)
        self.profile_rows[path][0] = new_name
        self.main_profile_cb.remove(int(path))
        self.main_profile_cb.insert_text(int(path), new_name)
        self.utils.add_to_log(_("Edited Profile: ") + new_name)
    
    def on_AddProfile_btn_clicked(self, widget, data=None):
        """Add a profile"""
        ind = 0
        exist_profile = True
        while exist_profile:
            ind += 1
            new_name = _("Profile") + str(ind)
            exist_profile = False
            for searched_profile in self.profile_rows:
                if searched_profile[0] == new_name:
                    exist_profile = True
                    break # Next ind
        
        self.fw.add_profile(new_name)
        ind = self.fw.get_all_profiles().index(new_name)
        self.profile_rows.append([new_name])
        self.main_profile_cb.append_text(new_name)
        self.utils.add_to_log(_("New Profile: ") + new_name)
        
    def on_DeleteProfile_btn_clicked(self, widget, data=None):
        model, treeiter = self.list_selection.get_selected()
        if treeiter == None:
            self.utils.show_dialog(_("Select a profile"), _("You need to select a profile for deleting"))
        else:
            deleted_profile = model.get_value(treeiter, 0)
            if deleted_profile == self.fw.get_profile():
                self.utils.show_dialog(_("Profile not erasable"), _("You can't remove the current profile"))
            else:
                self.main_profile_cb.remove(self.fw.get_all_profiles().index(deleted_profile))
                self.fw.delete_profile(deleted_profile)
                self.profile_rows.remove(treeiter)
                self.utils.add_to_log(_("Deleted Profile: ") + deleted_profile)
 
    def on_ufw_logging_changed(self, widget, data=None):
        self.fw.set_ufw_logging(NUM2UFW_LEVEL[self.ufw_logging.get_active()])
        self.utils.add_to_log(_("ufw Logging: ") + self.ufw_logging.get_active_text())
    
    def on_gufw_logging_toggled(self, widget, data=None):
        if self.gufw_logging.get_active():
            self.fw.set_logging(True)
            self.utils.add_to_log(_("Gufw Logging: Enabled"))
        else:
            self.fw.set_logging(False)
            self.utils.add_to_log(_("Gufw Logging: Disabled"))
    
    def on_close_btn_clicked(self, widget, data=None):
        self.win_preferences.destroy()
        
    def on_preferences_delete_event(self, widget, data=None):
        self.win_preferences.destroy()
    
