# -*- coding: utf-8 -*-
# Gufw 13.10.0 - http://gufw.org
# Copyright (C) 2008-2013 Marcos Alvarez Costales https://launchpad.net/~costales
#
# Gufw is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
# 
# Gufw is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with Gufw; if not, see http://www.gnu.org/licenses for more
# information.


from gi.repository import Gtk, Gdk, GObject, WebKit
from string import Template

import os, re, sys, subprocess

import gettext
from gettext import gettext as _
gettext.textdomain('gufw')
    
from about       import WinAbout
from preferences import WinPreferences
from add         import WinAdd
from update      import UpdateRule
from utilsview   import UtilsView
from listening   import ListeningReport
from constants   import *


class Gufw:
    def __init__(self, frontend):
        self.fw = frontend
        
        self.builder = Gtk.Builder()
        self.builder.add_from_file('/usr/share/gufw/ui/main.ui')
        
        self._set_objects_name()
        self.utils = UtilsView(self.fw, self.rules_model, self.statusbar, self.log, self.log_txt)
        self.listening = ListeningReport(self.fw, self.listening_model)
        self._set_initial_values()
        
        self.builder.connect_signals(self)
        
        Gtk.main()
    
    def _set_objects_name(self):
        self.winMain      = self.builder.get_object('main')
        
        self.profile      = self.builder.get_object('profile')
        self.switchStatus = self.builder.get_object('switchStatus')
        self.incoming     = self.builder.get_object('incoming')
        self.outgoing     = self.builder.get_object('outgoing')
        self.shield       = self.builder.get_object('shield')
        
        self.rules_expander  = self.builder.get_object('expanderRules')
        self.rules_box       = self.builder.get_object('boxRules')
        self.rules           = self.builder.get_object('Rules')
        self.show_add_win    = self.builder.get_object('btnAddRule')
        self.detele_rule_btn = self.builder.get_object('btnDeleteRule')
        
        self.report_expander  = self.builder.get_object('expanderReport')
        self.report_box       = self.builder.get_object('boxReport')
        self.report           = self.builder.get_object('Report')
        
        self.log_expander = self.builder.get_object('expanderLog')
        self.log_box      = self.builder.get_object('boxLog')
        self.log          = self.builder.get_object('log')
        self.log_txt      = self.log.get_buffer()
        self.log_copy     = self.builder.get_object('btnLogCopy')
        self.log_delete   = self.builder.get_object('btnLogCopy')
        
        self.statusbar    = self.builder.get_object('statusmsg')
        self.progress     = self.builder.get_object('progress')
        
        # Rules
        self.render_txt = Gtk.CellRendererText()
        self.render_txt.set_property('font', FONT)
        
        self.rules_model = Gtk.ListStore(GObject.TYPE_STRING,  # ufw rule
                                         GObject.TYPE_STRING,  # description
                                         GObject.TYPE_STRING,  # command
                                         GObject.TYPE_STRING,  # policy
                                         GObject.TYPE_STRING,  # direction
                                         GObject.TYPE_STRING,  # proto
                                         GObject.TYPE_STRING,  # from_ip
                                         GObject.TYPE_STRING,  # from_port
                                         GObject.TYPE_STRING,  # to_ip
                                         GObject.TYPE_STRING,  # to_port
                                         GObject.TYPE_STRING,  # iface
                                         GObject.TYPE_STRING,  # logging
                                         GObject.TYPE_STRING)  # color
        self.tv_rules = self.rules
        self.tv_rules.set_model(self.rules_model)
        self.tv_rules.get_selection().set_mode(Gtk.SelectionMode.MULTIPLE)
        
        # Listening report
        self.listening_model = Gtk.ListStore(GObject.TYPE_STRING, # Protocol
                                             GObject.TYPE_STRING, # Port
                                             GObject.TYPE_STRING, # Address
                                             GObject.TYPE_STRING, # App
                                             GObject.TYPE_STRING) # Color
        self.tv_report = self.report
        self.tv_report.set_model(self.listening_model)
        self.tv_report.get_selection().set_mode(Gtk.SelectionMode.NONE)
        
        self.donateBox = self.builder.get_object('donate_box')
    
    def _set_initial_values(self):
        for profile in self.fw.get_all_profiles():
            self.profile.append_text(profile)
        self.profile.set_active(self.fw.get_all_profiles().index(self.fw.get_profile()))
        
        self.switchStatus.set_active(self.fw.get_status())
        
        self.incoming.set_active(POLICY2NUM[self.fw.get_policy('incoming')])
        self.outgoing.set_active(POLICY2NUM[self.fw.get_policy('outgoing')])
        self.incoming.set_sensitive(self.fw.get_status())
        self.outgoing.set_sensitive(self.fw.get_status())
        self.detele_rule_btn.set_sensitive(self.fw.get_status())
        
        # Rules
        tree_header = Gtk.TreeViewColumn(_("Rule"), self.render_txt, text=0, foreground=12)
        self.tv_rules.append_column(tree_header)
        tree_header = Gtk.TreeViewColumn(_("Name"), self.render_txt, text=1, foreground=12)
        tree_header.set_expand(True)
        tree_header.set_resizable(True)
        self.tv_rules.append_column(tree_header)
        
        # Listening Report
        tree_header = Gtk.TreeViewColumn (_("Protocol"), self.render_txt, text=0, foreground=4)
        tree_header.set_resizable(True)
        self.tv_report.append_column (tree_header)
        tree_header = Gtk.TreeViewColumn (_("Port"), self.render_txt, text=1, foreground=4)
        tree_header.set_resizable(True)
        self.tv_report.append_column (tree_header)
        tree_header = Gtk.TreeViewColumn (_("Address"), self.render_txt, text=2, foreground=4)
        tree_header.set_resizable(True)
        self.tv_report.append_column (tree_header)
        tree_header = Gtk.TreeViewColumn (_("Application"), self.render_txt, text=3, foreground=4)
        self.tv_report.append_column (tree_header)
        
        self._load_tutorial()
        
        if self.fw.get_config_value('ExpandRules') == 'yes':
            self.tuto.hide()
            self.rules_expander.set_expanded(True)
            self.rules_box.show()
        
        if self.fw.get_config_value('ExpandListening') == 'yes':
            self.tuto.hide()
            self.report_expander.set_expanded(True)
            self.report_box.show()
            self.listening.starting()
        
        if self.fw.get_config_value('ExpandLog') == 'yes':
            self.tuto.hide()
            self.log_expander.set_expanded(True)
            self.log_box.show()
            self.utils.add_to_log(self.fw.get_log(), GRAY, False)
        
        self._set_shield()
        
        self.utils.restore_window_size(self.winMain)
        self.winMain.show()        
        
        if self.fw.get_status():
            self.utils.print_rules(self.fw.get_rules())
        
        if self.fw.get_config_value('ShowDonationBtn') == 'no':
            self.donateBox.hide()
    
    def _launch_browser(self, url):
        try:
            user = sys.argv[1]
        except:
            self.utils.show_dialog(_("Visit this web (copy & paste):"), url)
            return
        
        if not user or user == 'root':
            self.utils.show_dialog(_("Visit this web (copy & paste):"), url)
            return
            
        cmd = "su -l " + user + " -c 'python -mwebbrowser " + url + "'"
        
        subprocess.Popen(cmd, shell=True)
    
    def _load_tutorial(self):
        self.tuto = self.builder.get_object('boxTutorial')
        self.tuto_web = WebKit.WebView()
        self.tuto.add(self.tuto_web)
        
        self.tuto.show()
        self.tuto_web.show()
        #self.tuto_web.set_transparent(True)
        
        f = open(TUTORIAL_FILE, 'r')
        html_content = f.read()  
        f.close()
        
        replace_html = dict(heading1=_("Getting started"),
                            welcome=_("What is Gufw?"),
                            intro=_("An uncomplicated way to manage your firewall. Easy, simple and beautiful :)"),
                            heading2=_("Basic"),
                            heading3=_("FAQ"),
                            best_conf=_("If you are a normal user, you will be safe with this setting (Status=On, Incoming=Deny, Outgoing=Allow). Remember append allow rules for your P2P apps:"),
                            rename_profile=_("You can rename your profiles with just 2 clicks on them:"),
                            rule_name=_("The Rule Name will help you to identify your rules in the future:"),
                            faq1_q=_("How to autostart Gufw with the system?"),
                            faq1_a=_("You don't need it. After you do all of the changes in Gufw and close it, the settings are still in effect."),
                            faq2_q=_("Why is disabled by default?"),
                            faq2_a=_("By default, the firewall does not open ports to the outside world."),
                            faq3_q=_("Some rules are added by himself?"),
                            faq3_a=_("Well, when you change or import a profile, or when you edit a rule, Gufw is re-adding that rule. ufw is adding for IPv4 and IPv6, then this is the behavior. Sorry about that."),
                            faq4_q=_("What is Allow, Deny, Reject and Limit?"),
                            faq4_a1=_("Allow: Will allow traffic."),
                            faq4_a2=_("Deny: Will deny traffic."),
                            faq4_a3=_("Reject: Will deny traffic and will inform that it has been rejected."),
                            faq4_a4=_("Limit: Will deny traffic if an IP tried several connections."),
                            faq5_q=_("I see some rules in all profiles"),
                            faq5_a=_("All the ufw rules will be appear in all profiles."),
                            faq6_q=_("I want even more! :)"),
                            faq6_a=_("Visit the community documentation"))
        html = Template(html_content).safe_substitute(replace_html)
        self.tuto_web.load_html_string(html, "file:///")
    
    def on_expanderRules_activate(self, widget, data=None):
        if not self.rules_expander.get_expanded():
            self.rules_box.show()
            self.utils.print_rules(self.fw.get_rules())
            self.fw.set_config_value('ExpandRules', 'yes')
        else:
            self.rules_box.hide()
            self.fw.set_config_value('ExpandRules', 'no')
        self._show_tutorial(not self.rules_expander.get_expanded(), self.report_expander.get_expanded(), self.log_expander.get_expanded())
    
    def on_expanderReport_activate(self, widget, data=None):
        if not self.report_expander.get_expanded():
            self.report_box.show()
            self.fw.set_config_value('ExpandListening', 'yes')
            self.listening.starting()
        else:
            self.listening.stopping()
            self.fw.set_config_value('ExpandListening', 'no')
            self.report_box.hide()
            
        self._show_tutorial(self.rules_expander.get_expanded(), not self.report_expander.get_expanded(), self.log_expander.get_expanded())
    
    def _show_tutorial(self, rules, report, log):
        if not rules and not report and not log:
            self.tuto.show()
        else:
            self.tuto.hide()
    
    def on_expanderLog_activate(self, widget, data=None):
        if not self.log_expander.get_expanded():
            self.log_box.show()
            self.log_txt.set_text('')
            self.utils.add_to_log(self.fw.get_log(), GRAY, False)
            self.fw.set_config_value('ExpandLog', 'yes')
        else:
            self.log_box.hide()            
            self.fw.set_config_value('ExpandLog', 'no')
        self._show_tutorial(self.rules_expander.get_expanded(), self.report_expander.get_expanded(), not self.log_expander.get_expanded())
    
    def on_menu_import_activate(self, widget, data=None):
        import_profile = self.utils.file_dialog('open', _("Import Profile"))
        profile = os.path.basename(import_profile) #Filename
        profile = os.path.splitext(profile)[0] # Ext
        
        if not profile:
            self.utils.set_statusbar_msg(_("Import canceled"))
            return
        
        if not re.match('^[A-Za-z0-9_-]*$', profile):
            self.utils.show_dialog(_("Error"), _("Filename has not valid characters. Rename the file to only letters and numbers"))
            return
        
        if profile in self.fw.get_all_profiles():
            self.utils.set_statusbar_msg(_("Operation canceled"))
            self.utils.show_dialog(_("Error"), _("Profile already exists"))
        else:
            self.fw.import_profile(import_profile)       
            self.profile.append_text(profile)
            self.utils.add_to_log(_("Profile imported") + ": " + import_profile)
            self.utils.set_statusbar_msg(_("Profile imported, now you can choose in the profiles"))
    
    def on_menu_export_activate(self, widget, data=None):
        export_profile = self.utils.file_dialog('save', _("Export Profile"))
        
        if not export_profile:
            self.utils.set_statusbar_msg(_("Export canceled"))
            return
        
        if export_profile[-8:] != '.profile':
            export_profile = export_profile + '.profile'
        
        self.fw.export_profile(export_profile)
        self.utils.add_to_log(_("Profile exported with Gufw rules") + ": " + export_profile)
        self.utils.set_statusbar_msg(_("Profile exported"))
    
    def on_main_delete_event(self, widget, data=None):
        self._exit_gufw()
    
    def on_menu_quit_activate(self, widget, data=None):
        self._exit_gufw()
    
    def _exit_gufw(self):
        self.utils.save_window_size(self.winMain)
        self.listening.stopping()
        Gtk.main_quit()
    
    def _set_shield(self):
        """Set the shield color"""
        if self.fw.get_status():
            file_shield = os.path.join(SHIELDS_DIR, self.fw.get_policy('incoming').lower() + '_' + self.fw.get_policy('outgoing').lower() + '.png')
        else:
            file_shield = os.path.join(SHIELDS_DIR, 'disabled_disabled.png')
        self.shield.set_from_file(file_shield)
    
    def on_menu_reset_activate(self, widget, data=None):
        answer = self.utils.show_question(_("Reset Firewall"), _("This will remove all rules in the current profile and disable the firewall"), _("Do you want to continue?"))
        if answer:
            if self.fw.get_status():
                self.switchStatus.set_active(False)
            self.fw.reset()
            self.utils.add_to_log(_("Removed rules and reset firewall!"))
    
    def on_btnLogRemove_clicked(self, widget, data=None):
        self.fw.refresh_log()
        self.log_txt.set_text('')
        self.utils.add_to_log(_("Gufw Log: Refresh"))
        self.utils.set_statusbar_msg(_("Gufw Log removed"))
    
    def on_btnLogCopy_clicked(self, widget, data=None):
        self.clipboard = Gtk.Clipboard.get(Gdk.SELECTION_CLIPBOARD)
        self.clipboard.set_text(self.fw.get_log(), -1)
        self.utils.set_statusbar_msg(_("Text copied to clipboard"))
    
    def on_donate_btn_clicked(self, widget, data=None):
        self._launch_browser(URL_DONATE)
    
    def on_donate_hide_btn_clicked(self, widget, data=None):
        self.fw.set_config_value('ShowDonationBtn', 'no')
        self.donateBox.hide()
    
    def on_menu_about_activate(self, widget, data=None):
        about = WinAbout(self.winMain)
    
    def on_menu_doc_activate(self, widget, data=None):
        self._launch_browser(URL_DOC)
    
    def on_menu_answers_activate(self, widget, data=None):
        self._launch_browser(URL_ANSWERS)
    
    def on_menu_bugs_activate(self, widget, data=None):
        self._launch_browser(URL_BUGS)
    
    def on_menu_translations_activate(self, widget, data=None):
        self._launch_browser(URL_TRANSLATIONS)
    
    def on_menu_donate_activate(self, widget, data=None):
        self._launch_browser(URL_DONATE)
    
    def on_menu_gplus_activate(self, widget, data=None):
        self._launch_browser(URL_GPLUS)
    
    def on_menu_gplus_community_activate(self, widget, data=None):
        self._launch_browser(URL_GPLUSCOMUNITY)
    
    def on_menu_twitter_activate(self, widget, data=None):
        self._launch_browser(URL_TWITTER)
    
    def on_menu_identica_activate(self, widget, data=None):
        self._launch_browser(URL_IDENTICA)
    
    def on_menu_preferences_activate(self, widget, data=None):
        preferences = WinPreferences(self.fw, self.utils, self.profile, self.winMain)
    
    def on_incoming_changed(self, widget, data=None):
        self.fw.set_policy('incoming', NUM2POLICY[self.incoming.get_active()].lower())
        self._set_shield()
        self.utils.add_to_log(_("Incoming: ") + self.incoming.get_active_text(), POLICY2COLOR[self.fw.get_policy('incoming')])
        self.utils.set_statusbar_msg(_("Incoming policy changed"))
        if self.fw.get_policy('incoming') != NUM2POLICY[self.incoming.get_active()].lower():
            self.utils.show_dialog(_("There was an error changing the incoming policy"), _("Restart your firewall for refresh the real status and please report this bug"))
    
    def on_outgoing_changed(self, widget, data=None):
        self.fw.set_policy('outgoing', NUM2POLICY[self.outgoing.get_active()].lower())
        self._set_shield()
        self.utils.add_to_log(_("Outgoing: ") + self.outgoing.get_active_text(), POLICY2COLOR[self.fw.get_policy('outgoing')])
        self.utils.set_statusbar_msg(_("Outgoing policy changed"))
        if self.fw.get_policy('outgoing') != NUM2POLICY[self.outgoing.get_active()].lower():
            self.utils.show_dialog(_("There was an error changing the outgoing policy"), _("Restart your firewall for refresh the real status and please report this bug"))
    
    def on_btnAddRule_clicked(self, widget, data=None):
        self.show_add_win.set_sensitive(False)
        addWin = WinAdd(self.fw, self.utils, self.winMain, self.show_add_win)
    
    def on_switchStatus_active_notify(self, widget, data=None):
        self.fw.set_status(self.switchStatus.get_active())
        self.utils.print_rules(self.fw.get_rules())
        
        if self.fw.get_status():
            self.utils.add_to_log(_("Status: Enabled"), GREEN)
            self.utils.set_statusbar_msg(_("Firewall enabled"))
        else:
            self.utils.add_to_log(_("Status: Disabled"), RED)
            self.utils.set_statusbar_msg(_("Firewall disabled"))
        
        self.incoming.set_sensitive(self.fw.get_status())
        self.outgoing.set_sensitive(self.fw.get_status())
        self.detele_rule_btn.set_sensitive(self.fw.get_status())
        self._set_shield()
        if self.fw.get_status() != self.switchStatus.get_active():
            self.utils.show_dialog(_("There was an error changing the status firewall"), _("Restart your firewall for refresh the real status and please report this bug"))

    def on_btnDeleteRule_clicked(self, widget, data=None):
        rules_selected = False
        (model, rows) = self.tv_rules.get_selection().get_selected_rows()
        
        for row in reversed(rows):
            ufw_row = int(str(row)) + 1
            iter = self.rules_model.get_iter(row,)
            rules_selected = True
            
            rules_before = self.fw.get_rules()
            cmd = self.fw.delete_rule(ufw_row)
            rules_after = self.fw.get_rules()
            
            if len(rules_before) != len(rules_after):
                self.utils.add_to_log(cmd[0])
                self.utils.set_statusbar_msg(_("Rule/s deleted"))
            else:
                self.utils.add_to_log(_("Error running: ") + cmd[0] + ' > ' + cmd[1].replace('\n', ' | '))
                self.utils.set_statusbar_msg(_("Error. Review Gufw Log"))
        
        if rules_selected:
            self.utils.print_rules(self.fw.get_rules())
        else:
            self.utils.show_dialog(_("No rule selected"), _("You have to select a rule"))
    
    def _get_total_rows(self, model):
        i = 0
        while True:
            try:
                iter = model.get_iter(i,)
            except:
                return i
            i += 1
    
    def on_btnEditRule_clicked(self, widget, data=None):
        (model, rows) = self.tv_rules.get_selection().get_selected_rows()
        
        if len(rows) <> 1:
            self.utils.show_dialog(_("Select only one rule"), _("You can't edit several rules"))
            return
        
        # Just one rule selected
        ufw_row = int(str(rows[0])) + 1
        total_rows = self._get_total_rows(self.rules_model)
        
        iter = self.rules_model.get_iter(rows[0],)
        cmd   = self.rules_model.get_value(iter, 2)
        
        if not cmd: # ufw rule > inmutable
            self.utils.show_dialog(_("Inmutable Rule"), _("You can't edit a rule added from ufw"))
            return
        
        description = self.rules_model.get_value(iter, 1)
        policy      = self.rules_model.get_value(iter, 3)
        direction   = self.rules_model.get_value(iter, 4)
        proto       = self.rules_model.get_value(iter, 5)
        from_ip     = self.rules_model.get_value(iter, 6)
        from_port   = self.rules_model.get_value(iter, 7)
        to_ip       = self.rules_model.get_value(iter, 8)
        to_port     = self.rules_model.get_value(iter, 9)
        iface       = self.rules_model.get_value(iter, 10)
        logging     = self.rules_model.get_value(iter, 11)
        
        update_rule = UpdateRule(self.fw, self.utils, self.rules_model, ufw_row,
                                 description, cmd, policy, direction, proto, from_ip, from_port, to_ip, to_port, iface, logging,
                                 self.winMain)

    def on_profile_changed(self, widget, data=None):
        operation = self.fw.set_profile(self.profile.get_active_text())
        
        self.incoming.set_active(POLICY2NUM[self.fw.get_policy('incoming')])
        self.outgoing.set_active(POLICY2NUM[self.fw.get_policy('outgoing')])
        if self.switchStatus.get_active() != self.fw.get_status():
            self.switchStatus.set_active(self.fw.get_status())
        
        self.utils.print_rules(self.fw.get_rules())
        for msg in operation:
            self.utils.add_to_log(msg)
