# Gufw 11.04.1 - http://gufw.tuxfamily.org
# Copyright (C) 2008-2011 Marcos Alvarez Costales https://launchpad.net/~costales
#
# Gufw is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
# 
# Gufw is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with Gufw; if not, see http://www.gnu.org/licenses for more
# information.

import gtk
import threading
import gobject
import dbus
import webbrowser
from gettext import gettext as _
from util    import Path

__color__ = { "gray"   : "#BAB5AB",  # Basic 3D Medium
              "green"  : "#267726",  # Accent Green Dark
              "red"    : "#DF421E",  # Accent Red
              "orange" : "#D1940C",  # Accent Yellow Dark
              "blue"   : "#314E6C" } # Blue Shadow


class GuiGufw:
    """All events"""
    FONT = "ubuntu 10"
    TIME_REFRESH_REPORT = 1000
    TIME_STATUS_BAR = 8
    
    def __init__(self, firewall):
        self.fw = firewall
        self.path = Path()
        self.ui_builder = gtk.Builder()
        self._set_ui(self.ui_builder)
        gtk.main()
    
    def _set_ui(self, builder):
        """Set the interfaces"""
        builder.set_translation_domain("gufw")
        # Set windows
        self._set_ui_main(builder)
        self._set_ui_models(builder)
        self._set_ui_add(builder)
        self._set_ui_preferences(builder)
        self._set_ui_log(builder)
        self._set_ui_about(builder)
        self.win_main.show()
    
    def _set_ui_main(self, builder):
        """Set the window Main"""
        self.first_run_report = True
        self.previus_report = []
        builder.add_from_file(self.path.get_ui_path('main.ui'))
        self.win_main = builder.get_object('winMain')
        # Window size
        width, height = self.fw.get_window_size()
        if width == gtk.gdk.screen_width() and height== gtk.gdk.screen_height():
            self.win_main.maximize()
        else:
            self.win_main.resize(width,height)
        # Vpanel
        self.vpanelmain = builder.get_object("vpanelListeningRules")
        self.vpanelmain.set_position(self.fw.get_vpanel_pos())
        # Main Window Objects 
        self.cb_enabled_firewall = builder.get_object("cbEnabledFirewall")
        self.cb_enabled_fw_fake  = builder.get_object("cbEnabledFirewallFake")
        self.cb_policy_incoming  = builder.get_object("cbPolicyIncoming")
        self.cb_policy_in_fake   = builder.get_object("cbPolicyIncomingFake")
        self.cb_policy_outgoing  = builder.get_object("cbPolicyOutgoing")
        self.cb_policy_out_fake  = builder.get_object("cbPolicyOutgoingFake")
        self.image_shield        = builder.get_object("imgShield")
        self.block_report        = builder.get_object("blockReport")
        self.btn_unblock         = builder.get_object('btnUnblock')
        self.btn_add_window      = builder.get_object("btnAddWindow")
        self.btn_remove_rule     = builder.get_object("btnRemove")
        self.status_bar          = builder.get_object("statusBar")
        self.progress_bar        = builder.get_object("progressBar")
        self.progress_bar_block  = builder.get_object("progressBarBlock")
        # Objects for Global Menu in Unity
        self.menu_file      = builder.get_object("menu_file")
        self.menu_edit      = builder.get_object("menu_edit")
        self.menu_help      = builder.get_object("menu_help")
        # Menu
        self.menu_log       = builder.get_object("menuLogObject")
        self.menu_quit      = builder.get_object("menuQuit")
        self.menu_add       = builder.get_object("menuAddObject")
        self.menu_remove    = builder.get_object("menuRemoveObject")
        self.menu_reset     = builder.get_object("menuResetObject")
        self.menu_pref      = builder.get_object("menuPreferencesObject")
        self.menu_doc       = builder.get_object("menuDoc")
        self.menu_answers   = builder.get_object("menuHelp")
        self.menu_translate = builder.get_object("menuTranslate")
        self.menu_about     = builder.get_object("menuAbout")
        # Disable menus (not works from Glade)
        self.menu_log.set_sensitive(0)
        self.menu_add.set_sensitive(0)
        self.menu_remove.set_sensitive(0)
        self.menu_reset.set_sensitive(0)
        self.menu_pref.set_sensitive(0)
        # Set minimial signals for launch Gufw
        self.btn_unblock.connect('clicked', self.on_btnUnblock_clicked)
        self.win_main.connect('delete-event', self.on_winMain_delete_event)
        self.menu_quit.connect('activate', self.on_menuQuit_activate)
        self.menu_doc.connect('activate', self.on_menuDoc_activate)
        self.menu_answers.connect('activate', self.on_menuHelp_activate)
        self.menu_translate.connect('activate', self.on_menuTranslate_activate)
        self.menu_about.connect('activate', self.on_menuAbout_activate)
        # Show Listening report
        if self.fw.get_listening_status() == "enable":
            self.block_report.show()
            
        self.btn_unblock.grab_focus()
    
    def _set_ui_models(self, builder):
        """Set the models in main window"""
        self.render_txt = gtk.CellRendererText()
        self.render_txt.set_property("font", self.FONT)
        
        self.rules_model = gtk.ListStore(gobject.TYPE_STRING, gobject.TYPE_STRING, gobject.TYPE_STRING,
                                         gobject.TYPE_STRING, gobject.TYPE_STRING)
        
        self.tv_rules = builder.get_object("tvRules")
        self.tv_rules.set_model(self.rules_model)
        self.tv_rules.get_selection().set_mode(gtk.SELECTION_MULTIPLE)
        
        tree_header = gtk.TreeViewColumn (_("To"), self.render_txt, text=1, foreground=4)
        tree_header.set_expand(True)
        tree_header.set_resizable(True)
        self.tv_rules.append_column (tree_header)
        tree_header = gtk.TreeViewColumn (_("Action"), self.render_txt, text=2, foreground=4)
        tree_header.set_expand(True)
        tree_header.set_resizable(True)
        self.tv_rules.append_column (tree_header)
        tree_header = gtk.TreeViewColumn (_("From"), self.render_txt, text=3, foreground=4)
        tree_header.set_expand(True)
        self.tv_rules.append_column (tree_header)
        
        # Listening Report
        self.report_model = gtk.ListStore(gobject.TYPE_STRING, gobject.TYPE_STRING, gobject.TYPE_STRING,
                                          gobject.TYPE_STRING, gobject.TYPE_STRING, gobject.TYPE_STRING) 
        self.tv_report = builder.get_object("tvReport")
        self.tv_report.set_model(self.report_model)
        self.tv_report.get_selection().set_mode(gtk.SELECTION_NONE)
        
        tree_header = gtk.TreeViewColumn (_("Protocol"), self.render_txt, text=1, foreground=5)
        tree_header.set_resizable(True)
        self.tv_report.append_column (tree_header)
        tree_header = gtk.TreeViewColumn (_("Port"), self.render_txt, text=2, foreground=5)
        tree_header.set_resizable(True)
        self.tv_report.append_column (tree_header)
        tree_header = gtk.TreeViewColumn (_("Address"), self.render_txt, text=3, foreground=5)
        tree_header.set_resizable(True)
        self.tv_report.append_column (tree_header)
        tree_header = gtk.TreeViewColumn (_("Application"), self.render_txt, text=4, foreground=5)
        self.tv_report.append_column (tree_header)
    
    def _set_ui_add(self, builder):
        """Set the window Add"""
        builder.add_from_file(self.path.get_ui_path('add.ui'))
        self.win_add = builder.get_object('winAdd')
        self.win_add.set_transient_for(self.win_main)
        # Preconf
        self.insert_number_preconf = builder.get_object("sbInsertNumberPreconf")
        self.direction_preconf     = builder.get_object("cbDirectionPreconf")
        self.action_preconf        = builder.get_object("cbActionPreconf")
        self.log_preconf           = builder.get_object("cbLogPreconf")
        self.type_preconf          = builder.get_object("cbTypePreconf")
        self.service_preconf       = builder.get_object("cbServicePreconf")
        self.program_preconf       = builder.get_object("cbProgramPreconf")
        # Simple
        self.insert_number_simple = builder.get_object("sbInsertNumberSimple")
        self.direction_simple     = builder.get_object("cbDirectionSimple")
        self.action_simple        = builder.get_object("cbActionSimple")
        self.log_simple           = builder.get_object("cbLogSimple")
        self.port_simple          = builder.get_object("entryPortSimple")
        self.proto_simple         = builder.get_object("cbProtoSimple")
        # Advanced
        self.insert_number_advanced = builder.get_object("sbInsertNumberAdvanced")
        self.action_advanced        = builder.get_object("cbActionAdvanced")
        self.direction_advanced     = builder.get_object("cbDirectionAdvanced")
        self.log_advanced           = builder.get_object("cbLogAdvanced")
        self.proto_advanced         = builder.get_object("cbProtoAdvanced")
        self.fromip_advanced        = builder.get_object("entryFromIpAdvanced")
        self.portfrom_advanced      = builder.get_object("entryPortFromAdvanced")
        self.toip_advanced          = builder.get_object("entryToIpAdvanced")
        self.portto_advanced        = builder.get_object("entryPortToAdvanced")
        # Others
        self.rules_notebook   = builder.get_object("rulesNotebook")
        self.add_btn_add      = builder.get_object("btnAddRule")
        self.extended_actions = builder.get_object("cbExtendedActions")
        
    def _set_ui_preferences(self, builder):
        """Set the window Preferences"""
        builder.add_from_file(self.path.get_ui_path('preferences.ui'))
        self.win_preferences = builder.get_object('winPreferences')
        self.win_preferences.set_transient_for(self.win_main)
        # Preference Window
        self.cb_report       = builder.get_object("cbReport")
        self.cb_notify_popup = builder.get_object("cbNotifyPopup")
        self.lbl_ufw_level   = builder.get_object("lblLogLevel")
        self.cb_ufw_level    = builder.get_object("cbLogLevel")
        self.cb_gufw_log     = builder.get_object("cbGufwLog") 
        self.pref_btn_close  = builder.get_object("btnClosePref")
            
    def _set_ui_log(self, builder):
        """Set the window Log"""
        builder.add_from_file(self.path.get_ui_path('log.ui'))
        self.win_log = builder.get_object('winLog')
        self.win_log.set_transient_for(self.win_main)
        # Log Window
        self.log_txt        = builder.get_object("logTxt")
        self.log_txt_buffer = self.log_txt.get_buffer()
        self.log_btn_close  = builder.get_object("btnCloseLog")
        self.server_script  = builder.get_object("cbServerScript")
    
    def _set_ui_about(self,builder):
        """Set the window About"""
        builder.add_from_file(self.path.get_ui_path('about.ui'))
        self.win_about = builder.get_object('winAbout')
        self.win_about.set_transient_for(self.win_main)
    
    def _set_main_menus(self, status):
        """Block main menu with modal windows, this fix Global Menu"""
        self.menu_file.set_sensitive(status)
        self.menu_edit.set_sensitive(status)
        self.menu_help.set_sensitive(status)
    
    def _set_main_values(self, statusbar_msg):
        """Set initials status to GUI"""
        # Set sensitive values by status firewall
        if self.fw.get_status() == "enable":
            # Shield / Dropbox / Buttons / Menus
            incoming = self.fw.get_policy("incoming")
            outgoing = self.fw.get_policy("outgoing")
            self.image_shield.set_from_file(self.path.get_shield_path(incoming, outgoing))
            self.cb_policy_incoming.set_sensitive(1)
            self.cb_policy_outgoing.set_sensitive(1)
            self.add_btn_add.set_sensitive(1)
            if self.fw.get_number_rules() == 0:
                self.btn_remove_rule.set_sensitive(0)
                self.menu_remove.set_sensitive(0)
            else:
                self.btn_remove_rule.set_sensitive(1)
                self.menu_remove.set_sensitive(1)
        else:
            # Shield / Dropbox / Buttons / Menus
            self.image_shield.set_from_file(self.path.get_shield_path("disable", "disable"))
            self.cb_policy_incoming.set_sensitive(0)
            self.cb_policy_outgoing.set_sensitive(0)
            self.add_btn_add.set_sensitive(0)
            self.btn_remove_rule.set_sensitive(0)
            self.menu_remove.set_sensitive(0)
        # Gufw menu sensitive values by Gufw Log status
        if self.fw.get_gufw_logging() == "enable":
            self.menu_log.set_sensitive(1)
        else:
            self.menu_log.set_sensitive(0)
        # Rules
        self._set_rules_list()
        # StatusBar
        self._set_statusbar_msg(statusbar_msg)
    
    def _set_statusbar_msg(self, msg):
        """Set a message into StatusBar Main window"""
        cid = self.status_bar.get_context_id('default context')
        mid = self.status_bar.push(cid, msg)
        gobject.timeout_add_seconds(self.TIME_STATUS_BAR, self.status_bar.remove_message, cid, mid)
    
    def _set_rules_list(self):
        """Set rules in main window"""
        row = 0
        self.rules_model.clear()
        rules = self.fw.get_rule_list()
        for rule in rules:
            row += 1
            iterador = self.rules_model.insert(row)
            # Set value txt
            rule_formated = self._get_format_rules_txt(rule)
            # Get color
            color = self._get_rule_color(rule_formated[1])            
            # Set values
            self.rules_model.set_value(iterador, 0, row) # Use for remove rule
            self.rules_model.set_value(iterador, 1, rule_formated[0].strip())
            self.rules_model.set_value(iterador, 2, rule_formated[1].strip())
            self.rules_model.set_value(iterador, 3, rule_formated[2].strip())
            self.rules_model.set_value(iterador, 4, color) # Foreground color rule
    
    def _get_rule_color(self, rule):
        """Return color rule"""
        # Color Allow/Deny/Reject/Limit
        # IN mode (equal to normal mode, persist code for clear read)
        if rule == "ALLOW IN":
            if self.fw.get_policy("incoming") != "allow":
                return __color__["red"]
            else:
                return __color__["gray"]
        # Deny?
        elif rule == "DENY IN":
            if self.fw.get_policy("incoming") != "deny":
                return __color__["green"]
            else:
                return __color__["gray"]
        # Reject
        elif rule == "REJECT IN":
            if self.fw.get_policy("incoming") != "reject":
                return __color__["blue"]
            else:
                return __color__["gray"]
        # Limit?
        elif rule == "LIMIT IN":
            return __color__["orange"]
        
        # OUT mode
        elif rule == "ALLOW OUT": 
            if self.fw.get_policy("outgoing") != "allow":
                return __color__["red"]
            else:
                return __color__["gray"]
        # Deny?
        elif rule == "DENY OUT":
            if self.fw.get_policy("outgoing") != "deny":
                return __color__["green"]
            else:
                return __color__["gray"]
        # Reject
        elif rule == "REJECT OUT":
            if self.fw.get_policy("outgoing") != "reject":
                return __color__["blue"]
            else:
                return __color__["gray"]
        # Limit?
        elif rule == "LIMIT OUT":
            return __color__["orange"]
            
        # NORMAL mode
        # Allow?
        elif rule == "ALLOW": 
            if self.fw.get_policy("incoming") != "allow":
                return __color__["red"]
            else:
                return __color__["gray"]
        # Deny?
        elif rule == "DENY":
            if self.fw.get_policy("incoming") != "deny":
                return __color__["green"]
            else:
                return __color__["gray"]
        # Reject
        elif rule == "REJECT":
            if self.fw.get_policy("incoming") != "reject":
                return __color__["blue"]
            else:
                return __color__["gray"]
        # Limit?
        elif rule == "LIMIT":
            return __color__["orange"]
        
    def _get_format_rules_txt(self, rule):
        # IN mode (equal to normal mode, persist code for clear read)
        if rule.find("ALLOW IN") != -1:
            split_str = "ALLOW IN"
        # Deny?
        elif rule.find("DENY IN") != -1:
            split_str = "DENY IN"
        # Reject
        elif rule.find("REJECT IN") != -1:
            split_str = "REJECT IN"
        # Limit?
        elif rule.find("LIMIT IN") != -1:
            split_str = "LIMIT IN"
        
        # OUT mode
        elif rule.find("ALLOW OUT") != -1: 
            split_str = "ALLOW OUT"
        # Deny?
        elif rule.find("DENY OUT") != -1:
            split_str = "DENY OUT"
        # Reject
        elif rule.find("REJECT OUT") != -1:
            split_str = "REJECT OUT"
        # Limit?
        elif rule.find("LIMIT OUT") != -1:
            split_str = "LIMIT OUT"
            
        # NORMAL mode
        # Allow?
        elif rule.find("ALLOW") != -1: 
            split_str = "ALLOW"
        # Deny?
        elif rule.find("DENY") != -1:
            split_str = "DENY"
        # Reject
        elif rule.find("REJECT") != -1:
            split_str = "REJECT"
        # Limit?
        elif rule.find("LIMIT") != -1:
            split_str = "LIMIT"
        
        # Values
        rule_split = rule.split(split_str)
        return rule_split[0].strip(), split_str, rule_split[1].strip()
    
    def _do_refresh_report(self):
        """Refresh method in background (no freeze)"""
        lines = self.fw.get_listening_report()
        background_job = RefreshReport(self.fw.get_status(), self.report_model, lines, self.previus_report, self.first_run_report, self.fw.get_notify_popup())
        background_job.start()
        self.previus_report = lines
        if self.fw.get_listening_status() == "enable":
            self.first_run_report = False
            return True
        else:
            self.previus_report = []
            self.first_run_report = True
            return False
    
    def _refresh_report(self):
        """Refresh Listening Report"""
        gobject.timeout_add(self.TIME_REFRESH_REPORT , self._do_refresh_report)
        
    def _remove_rule(self):
        """Remove Rules Method"""
        number_rules = self.fw.get_number_rules()
        tree,iter = self.tv_rules.get_selection().get_selected_rows()
        removed = 0
        actual_row = 0
        total_rows = len(iter)
           
        if total_rows == 0:
            self._set_main_values(_("Select rule(s)"))
            yield None
            return

        # No sensitive buttons & msg
        self.progress_bar_block.show()
        self.cb_enabled_firewall.set_sensitive(0)
        self.cb_policy_incoming.set_sensitive(0)
        self.cb_policy_outgoing.set_sensitive(0)
        self.btn_remove_rule.set_sensitive(0)
        self.menu_remove.set_sensitive(0)
        self.add_btn_add.set_sensitive(0)
        self.menu_pref.set_sensitive(0)
        self.menu_reset.set_sensitive(0)
        self._set_statusbar_msg(_("Removing rules..."))

        # For one row selected
        iter.reverse() # Remove first the last rules for not overwrite rules
        for item in iter:
            
            # Get rule selected (row number)
            number_rule_row = tree.get_value(tree.get_iter(item),0)
            
            # Move Progress Bar
            actual_row += 1
            progress = float(actual_row) / float(total_rows)
            if progress > 1:
                progress = 1.0
            self.progress_bar.set_fraction(progress)
            yield True
            
            self.fw.remove_rule(number_rule_row)

        # Clean Progress Bar
        self.progress_bar.set_fraction(0)
        self.progress_bar_block.hide()
        self.cb_enabled_firewall.set_sensitive(1)
        self.menu_pref.set_sensitive(1)
        self.menu_reset.set_sensitive(1)
        
        if number_rules != self.fw.get_number_rules():
            self._set_main_values(_("Rule(s) removed"))
        else:
            self._set_main_values(_("Error performing operation"))
        
        yield None

    def _add_rule_preconf(self):
        """Add a preconfigured rule"""
        number_rules = self.fw.get_number_rules()
        # Insert Number
        if self.extended_actions.get_active() != 0: # Visible?
            insert_number = str(self.insert_number_preconf.get_value_as_int())
        else:
            insert_number = "0"
        # Allow|deny|Limit
        if self.action_preconf.get_active() == 0:
            action = "allow"
        elif self.action_preconf.get_active() == 1:
            action = "deny"
        elif self.action_preconf.get_active() == 2:
            action = "reject"
        else:
            action = "limit"
        # IN/OUT
        if self.direction_preconf.get_active() == 0:
            direction = "in"
        else:
            direction = "out"
        # Log
        log = "log-default"
        if self.extended_actions.get_active() != 0: # Visible?
            if self.log_preconf.get_active() == 1:
                log = "log"
            elif self.log_preconf.get_active() == 2:
                log = "log-all"
        
        # Service?
        if self.type_preconf.get_active() == 1:
            SERVICES = { "FTP"   : "ftp",
                         "HTTP"  : "http",
                         "IMAP"  : "imap",
                         "NFS"   : "nfs",
                         "POP3"  : "pop3",
                         "Samba" : "135,139,445tcp|137,138udp",
                         "SMTP"  : "smtp",
                         "SSH"   : "ssh",
                         "VNC"   : "5900tcp",
                         "CUPS"  : "631tcp" }        
            service_txt = SERVICES[self.service_preconf.get_active_text()]
            all_ports = service_txt.split("|")
            for port_proto in all_ports:
                if port_proto.find("tcp") != -1:
                    port     = port_proto.replace("tcp", "")
                    protocol = "tcp"
                    is_program = True
                elif port_proto.find("udp") != -1:
                    port     = port_proto.replace("udp", "")
                    protocol = "udp"
                    is_program = True
                elif port_proto.find("both") != -1:
                    port     = port_proto.replace("both", "")
                    protocol = "both"
                    is_program = True
                else:
                    port     = port_proto
                    protocol = ""
                    is_program = False
                    
                self.fw.add_rule(is_program, insert_number, action, direction, log, protocol, "", "", "", port)
            
            if number_rules != self.fw.get_number_rules():
                self._set_main_values(_("Rule added"))
            else:
                self._set_main_values(_("Error performing operation"))
                
        # Program?
        else:
            PROGRAMS = { "Amule"        : "4662tcp#4672udp",
                         "Deluge"       : "6881:6891tcp#6881:6891udp",
                         "KTorrent"     : "6881tcp#4444udp",
                         "Nicotine"     : "2234:2239tcp#2242tcp#2240tcp",
                         "qBittorrent"  : "6881tcp#6881udp",
                         "Transmission" : "51413tcp#51413udp",
                         "Skype"        : "443tcp" }
            port_proto = PROGRAMS[self.program_preconf.get_active_text()]
            ports_protos = port_proto.split("#")
            for prog in ports_protos:
                if prog.find("tcp") != -1:
                    port     = prog.replace("tcp", "")
                    protocol = "tcp"
                elif prog.find("udp") != -1:
                    port     = prog.replace("udp", "")
                    protocol = "udp"
                elif prog.find("both") != -1:
                    port     = prog.replace("both", "")
                    protocol = "both"
                    
                # TODO Add rule program
                self.fw.add_rule(True, insert_number, action, direction, log, protocol, "", "", "", port)
                    
            if number_rules != self.fw.get_number_rules(): 
                self._set_main_values(_("Rule added"))
            else:
                self._set_main_values(_("Error performing operation"))
    
    def _add_rule_simple(self):
        """Add a simple rule"""
        number_rules = self.fw.get_number_rules()
        # Insert Number
        if self.extended_actions.get_active() != 0: # Visible?
            insert_number = str(self.insert_number_simple.get_value_as_int())
        else:
            insert_number = "0"
        # Allow|deny|Limit
        if self.action_simple.get_active() == 0:
            action = "allow"
        elif self.action_simple.get_active() == 1:
            action = "deny"
        elif self.action_simple.get_active() == 2:
            action = "reject"
        else:
            action = "limit"
        # IN/OUT
        if self.direction_simple.get_active() == 0:
            direction = "in"
        else:
            direction = "out"
        # Log
        log = "log-default"
        if self.extended_actions.get_active() != 0: # Visible?
            if self.log_simple.get_active() == 1:
                log = "log"
            elif self.log_simple.get_active() == 2:
                log = "log-all"
        # Protocol
        if self.proto_simple.get_active() == 0:
            protocol = "tcp"
        elif self.proto_simple.get_active() == 1:
            protocol = "udp"
        else:
            protocol = "both"
        # Port
        port = self.port_simple.get_text()
        # ? -> ! Don't read the next!!
        if port == "stallman":
            dlg_egg = gtk.MessageDialog(parent=self.win_main, flags=0, type=gtk.MESSAGE_WARNING, buttons=gtk.BUTTONS_CLOSE, message_format="Value your freedom or you will lose it, teaches history. 'Don't bother us with politics', respond those who don't want to learn.")
            dlg_egg.format_secondary_markup("Richard Stallman")
            dlg_egg.set_title("It's time to think!")
            dlg_egg.run()
            dlg_egg.destroy()
            return
        # Validate port
        if port == "":
            self._set_statusbar_msg(_("Error: Insert a port number"))
            return
        # Validate both and not range ports
        if ( port.find(":") != -1 ) and protocol == "both":
            self._set_statusbar_msg(_("Error: Range ports only with tcp or udp protocol"))
            return
        # Add rule
        self.fw.add_rule(True, insert_number, action, direction, log, protocol, "", "", "", port)
        
        if number_rules != self.fw.get_number_rules():
            self._set_main_values(_("Rule added"))
        else:
            self._set_main_values(_("Error performing operation"))
        
    def _add_rule_advanced(self):
        """Add an advanced rule"""
        number_rules = self.fw.get_number_rules()
        # Insert Number
        if self.extended_actions.get_active() != 0: # Visible?
            insert_number = str(self.insert_number_advanced.get_value_as_int())
        else:
            insert_number = "0"
        # Deny|Reject|Allow|Limit
        if self.action_advanced.get_active() == 0:
            action = "allow"
        elif self.action_advanced.get_active() == 1:
            action = "deny"
        elif self.action_advanced.get_active() == 2:
            action = "reject"
        else:
            action = "limit"
        # IN/OUT
        if self.direction_advanced.get_active() == 0:
            direction = "in"
        else:
            direction = "out"
        # Log
        log = "log-default"
        if self.extended_actions.get_active() != 0: # Visible?
            if self.log_advanced.get_active() == 1:
                log = "log"
            elif self.log_advanced.get_active() == 2:
                log = "log-all"
        # Protocol
        if self.proto_advanced.get_active() == 0:
            protocol = "tcp"
        elif self.proto_advanced.get_active() == 1:
            protocol = "udp"
        else:
            protocol = "both"
        # From
        fromip   = self.fromip_advanced.get_text()
        fromport = self.portfrom_advanced.get_text()
        # To
        toip   = self.toip_advanced.get_text()
        toport = self.portto_advanced.get_text() 
        # Validate values
        if fromip == "" and fromport == "" and toip == "" and toport == "":
            self._set_statusbar_msg(_("Error: Fields filled out incorrectly"))
            return
        # Validate both and not range ports in FROM
        if ( fromport != "" and fromport.find(":") != -1 ) and protocol == "both":
            self._set_statusbar_msg(_("Error: Range ports only with tcp or udp protocol"))
            return
        # Validate both and not range ports in TO            
        if ( toport != "" and toport.find(":") != -1 ) and protocol == "both":
            self._set_statusbar_msg(_("Error: Range ports only with tcp or udp protocol"))
            return
        # Add rule program
        result = self.fw.add_rule(True, insert_number, action, direction, log, protocol, fromip, fromport, toip, toport)
        if number_rules != self.fw.get_number_rules():
            self._set_main_values(_("Rule added"))
        else:
            self._set_main_values(_("Error performing operation"))
    
    def on_btnAddWindow_clicked(self, widget):
        """Button Add"""
        self.btn_add_window.set_sensitive(0)
        self.menu_add.set_sensitive(0)
        self.add_btn_add.grab_focus()
        self.win_add.show()
    
    def on_btnCloseAdd_clicked(self, widget):
        """Button Close Add Rules"""
        self.btn_add_window.set_sensitive(1)
        self.menu_add.set_sensitive(1)
        self.win_add.hide()
        return True
    
    def on_btnClosePref_clicked(self, widget):
        """Close preferences Button"""
        self.win_preferences.hide()
        self._set_main_menus(1)
        return True
    
    def on_btnCloseLog_clicked(self, widget):
        """Close Gufw Log Window"""
        self.win_log.hide()
        self._set_main_menus(1)
        return True
    
    def on_btnClearLog_clicked(self, widget):
        """Clear Log"""
        self.fw.erase_gufw_log()
        self.log_txt_buffer.set_text("")
        self.log_btn_close.grab_focus()
    
    def on_btnRemove_clicked(self, widget):
        """Remove in background"""
        task = self._remove_rule()
        gobject.idle_add(task.next)
    
    def on_btnAddRule_clicked(self, widget):
        """Add rule Button"""
        # Simple rule
        if self.rules_notebook.get_current_page() == 0:
            self._add_rule_preconf()
        # Preconfigured rule
        elif self.rules_notebook.get_current_page() == 1:
            self._add_rule_simple()
        # Advanced rule
        elif self.rules_notebook.get_current_page() == 2:
            self._add_rule_advanced()
    
    def on_btnCleanAdvanced_clicked(self, widget):
        """Clear values in advanced tab"""
        self.fromip_advanced.set_text("")
        self.portfrom_advanced.set_text("")
        self.toip_advanced.set_text("")
        self.portto_advanced.set_text("")
    
    def on_btnUnblock_clicked(self, widget):
        """Show PolicyKit"""
        if self.fw.unblock() == "access":
            self._set_initial_objects_main()
            self._set_initial_objects_preferences()
            self._set_main_values("")
            self.ui_builder.connect_signals(self)
            if self.fw.get_listening_status() == "enable":
                self._refresh_report()
        else:
            self._set_statusbar_msg(_("Wrong identification"))
        
    def _set_initial_objects_main(self):
        """Set the initial status where unblock the FW"""
        # Sensitive buttons
        self.btn_unblock.set_sensitive(0)
        self.menu_add.set_sensitive(1)
        self.menu_reset.set_sensitive(1)
        self.menu_pref.set_sensitive(1)
        self.btn_add_window.set_sensitive(1)
        # Status
        self.cb_enabled_firewall.set_sensitive(1)
        if self.fw.get_status() == "enable":
            self.cb_enabled_firewall.set_active(True)
        else:
            self.cb_enabled_firewall.set_active(False)
        # Policy
        incoming = self.fw.get_policy("incoming")
        outgoing = self.fw.get_policy("outgoing")
        if incoming == "deny":
            self.cb_policy_incoming.set_active(0)
        elif incoming == "reject":
            self.cb_policy_incoming.set_active(1)
        elif incoming == "allow":
            self.cb_policy_incoming.set_active(2)
        if outgoing == "deny":
            self.cb_policy_outgoing.set_active(0)
        elif outgoing == "reject":
            self.cb_policy_outgoing.set_active(1)
        elif outgoing == "allow":
            self.cb_policy_outgoing.set_active(2)
    
    def _set_initial_objects_preferences(self):
        """Set the initial status where unblock the FW"""
        # Listening report
        if self.fw.get_listening_status() == "enable":
            self.cb_report.set_active(1)
        else:
            self.cb_report.set_active(0)
            self.cb_notify_popup.set_sensitive(0)            
        # Show Notify popups
        if self.fw.get_notify_popup() == "enable":
            self.cb_notify_popup.set_active(1)
        else:
            self.cb_notify_popup.set_active(0)
        # ufw Log
        if self.fw.get_ufw_logging() == "off":
            self.cb_ufw_level.set_active(0)
        elif self.fw.get_ufw_logging() == "low":
            self.cb_ufw_level.set_active(1)
        elif self.fw.get_ufw_logging() == "medium":
            self.cb_ufw_level.set_active(2)
        elif self.fw.get_ufw_logging() == "high":
            self.cb_ufw_level.set_active(3)
        else:
            self.cb_ufw_level.set_active(4)
        if self.fw.get_status() == "disable":
            self.cb_ufw_level.set_sensitive(0)
            self.lbl_ufw_level.set_sensitive(0)
        # Gufw Log
        if self.fw.get_gufw_logging() == "enable":
            self.cb_gufw_log.set_active(1)
        else:
            self.cb_gufw_log.set_active(0)
    
    def on_cbLogLevel_changed(self, widget):
        """Change Logging Level"""
        if ( self.cb_ufw_level.get_active() == 0 ):
            self.fw.set_ufw_logging("off")
        elif ( self.cb_ufw_level.get_active() == 1 ):
            self.fw.set_ufw_logging("low")
        elif ( self.cb_ufw_level.get_active() == 2 ):
            self.fw.set_ufw_logging("medium")
        elif ( self.cb_ufw_level.get_active() == 3 ):
            self.fw.set_ufw_logging("high")
        elif ( self.cb_ufw_level.get_active() == 4 ):
            self.fw.set_ufw_logging("full")
    
    def on_cbGufwLog_toggled(self, widget):
        """Gufw Log CheckButton"""
        if self.cb_gufw_log.get_active() == 1:
            self.fw.set_gufw_logging("enable")
            self.menu_log.set_sensitive(1)
        elif self.cb_gufw_log.get_active() == 0:
            self.fw.set_gufw_logging("disable")
            self.menu_log.set_sensitive(0)
    
    def on_cbReport_toggled(self, widget):
        """Listening report"""
        if self.cb_report.get_active() == 1:
            self.fw.set_listening_status("enable")
            self.cb_notify_popup.set_sensitive(1)
            self.block_report.show()
        else:
            self.fw.set_listening_status("disable")
            self.cb_notify_popup.set_sensitive(0)
            self.block_report.hide()
        self._refresh_report()
    
    def on_cbNotifyPopup_toggled(self, widget):
        """Show Notify Popups for Listening reports"""
        if self.cb_notify_popup.get_active() == 1:
            self.fw.set_notify_popup("enable")
        else:
            self.fw.set_notify_popup("disable")
        
    def on_cbServerScript_toggled(self, widget):
        """View Gufw Log as Server Script"""
        if self.server_script.get_active():
            self.log_txt_buffer.set_text(self.fw.get_gufw_log('server'))
        else:
            self.log_txt_buffer.set_text(self.fw.get_gufw_log('local'))
    
    def on_cbEnabledFirewall_toggled(self, widget):
        """Changed FW Status"""
        if self.fw.get_status() == "enable":
            self.fw.set_status("disable")
            self.cb_ufw_level.set_sensitive(0)
            self.lbl_ufw_level.set_sensitive(0)
            self._set_main_values(_("Disabled firewall"))
        else:
            self.fw.set_status("enable")
            self.cb_ufw_level.set_sensitive(1)
            self.lbl_ufw_level.set_sensitive(1)            
            self._set_main_values(_("Enabled firewall"))
    
    def on_cbPolicyIncoming_changed(self, widget):
        """Policy (Deny/Allow/Reject All) Incoming"""
        # Apply?
        if self.fw.get_policy("incoming") == "deny" and self.cb_policy_incoming.get_active() == 0:
            return
        if self.fw.get_policy("incoming") == "reject" and self.cb_policy_incoming.get_active() == 1:
            return
        if self.fw.get_policy("incoming") == "allow" and self.cb_policy_incoming.get_active() == 2:
            return
        
        if self.cb_policy_incoming.get_active() == 0:
            self.fw.set_policy("incoming","deny")
            self._set_main_values(_("Deny all INCOMING traffic"))
            return
        elif self.cb_policy_incoming.get_active() == 1:
            self.fw.set_policy("incoming","reject")
            self._set_main_values(_("Reject all INCOMING traffic"))
            return
        elif self.cb_policy_incoming.get_active() == 2:
            self.fw.set_policy("incoming","allow")
            self._set_main_values(_("Allow all INCOMING traffic"))
            
    def on_cbPolicyOutgoing_changed(self, widget):
        """Policy (Deny/Allow/Reject All) Outgoing"""
        # Apply?
        if self.fw.get_policy("outgoing") == "deny" and self.cb_policy_outgoing.get_active() == 0:
            return
        if self.fw.get_policy("outgoing") == "reject" and self.cb_policy_outgoing.get_active() == 1:
            return
        if self.fw.get_policy("outgoing") == "allow" and self.cb_policy_outgoing.get_active() == 2:
            return
        
        if self.cb_policy_outgoing.get_active() == 0:
            self.fw.set_policy("outgoing","deny")
            self._set_main_values(_("Deny all OUTGOING traffic"))
            return
        elif self.cb_policy_outgoing.get_active() == 1:
            self.fw.set_policy("outgoing","reject")
            self._set_main_values(_("Reject all OUTGOING traffic"))
            return
        elif self.cb_policy_outgoing.get_active() == 2:
            self.fw.set_policy("outgoing","allow")
            self._set_main_values(_("Allow all OUTGOING traffic"))
    
    def on_cbTypePreconf_changed(self, widget):
        """Change between Service/Program"""
        if self.type_preconf.get_active() == 0:
            self.service_preconf.hide()   
            self.program_preconf.show()   
        else:
            self.service_preconf.show()   
            self.program_preconf.hide()   
    
    def on_cbServicePreconf_changed(self,widget):
        """Change tooltip for actual Service"""
        if self.service_preconf.get_active() == 0:
            self.service_preconf.set_tooltip_text("File Transfer Protocol")
        elif self.service_preconf.get_active() == 1:
            self.service_preconf.set_tooltip_text("Hypertext Transfer Protocol")
        elif self.service_preconf.get_active() == 2:
            self.service_preconf.set_tooltip_text("Internet Message Access Protocol")
        elif self.service_preconf.get_active() == 3:
            self.service_preconf.set_tooltip_text("Network File System")
        elif self.service_preconf.get_active() == 4:
            self.service_preconf.set_tooltip_text("Post Office Protocol")
        elif self.service_preconf.get_active() == 5:
            self.service_preconf.set_tooltip_text("")
        elif self.service_preconf.get_active() == 6:
            self.service_preconf.set_tooltip_text("Simple Mail Transfer Protocol")
        elif self.service_preconf.get_active() == 7:
            self.service_preconf.set_tooltip_text("Secure Shell")
        elif self.service_preconf.get_active() == 8:
            self.service_preconf.set_tooltip_text("Virtual Network Computing")
        elif self.service_preconf.get_active() == 9:
            self.service_preconf.set_tooltip_text("Common Unix Printing System")
    
    def on_cbExtendedActions_toggled(self, widget):
        """Extended actions"""
        # Set hide extended actions
        if self.extended_actions.get_active() == 0:
            self.insert_number_preconf.hide()
            self.insert_number_simple.hide()
            self.insert_number_advanced.hide()
            self.log_preconf.hide()
            self.log_simple.hide()
            self.log_advanced.hide()
        else:
            self.insert_number_preconf.show()
            self.insert_number_simple.show()
            self.insert_number_advanced.show()
            self.log_preconf.show()
            self.log_simple.show()
            self.log_advanced.show()
    
    def on_menuQuit_activate(self, widget):
        """Menu Quit"""
        width, height = self.win_main.get_size()
        self.fw.update_config_file(width, height, self.vpanelmain.get_position())
        gtk.main_quit()
    
    def on_menuAdd_activate(self,widget):
        """Menu Add Rule"""
        self.add_btn_add.grab_focus()
        self.btn_add_window.set_sensitive(0)
        self.menu_add.set_sensitive(0)
        self.win_add.show()
    
    def on_menuPreferences_activate(self, widget):            
        """Show Window Preferences"""
        self.pref_btn_close.grab_focus()
        self._set_main_menus(0)
        self.win_preferences.show()
    
    def on_menuAbout_activate(self, widget):
        """View About Window"""
        self._set_main_menus(0)
        self.win_about.run()
        self.win_about.hide()
        self._set_main_menus(1)
    
    def on_menuLog_activate(self, widget):
        """View Gufw Log Window"""
        if self.server_script.get_active():
            self.log_txt_buffer.set_text(self.fw.get_gufw_log('server'))
        else:
            self.log_txt_buffer.set_text(self.fw.get_gufw_log('local'))
        
        self.log_btn_close.grab_focus()
        self._set_main_menus(0)
        self.win_log.show()
    
    def on_menuReset_activate(self, widget):
        """Reset ufw"""
        reset_dialog = gtk.MessageDialog(parent=self.win_main,
                                         flags=0,
                                         type=gtk.MESSAGE_WARNING,
                                         buttons=gtk.BUTTONS_OK_CANCEL,
                                         message_format=_("This will remove all rules and disable the firewall!"))
        reset_dialog.format_secondary_markup(_("Do you want to continue?"))
        reset_dialog.set_title(_("Reset Firewall"))
        reset_dialog.set_default_response(gtk.RESPONSE_CANCEL)
        self._set_main_menus(0)
        reset_answer = reset_dialog.run()
        reset_dialog.destroy()
        self._set_main_menus(1)
        if reset_answer == gtk.RESPONSE_OK:
            if self.fw.get_status() == "enable":
                self.cb_enabled_firewall.set_active(0)
            self.fw.reset_ufw()
            self._set_main_values(_("Removed rules and reset firewall!"))
    
    def on_menuRemove_activate(self, widget):
        """Remove rules in background"""
        task = self._remove_rule()
        gobject.idle_add(task.next)
    
    def on_menuDoc_activate(self, widget):
        """Launch browser with Documentation web"""
        webbrowser.open_new("https://help.ubuntu.com/community/Gufw")
        
    def on_menuHelp_activate(self, widget):
        """Launch browser with Documentation web"""
        webbrowser.open_new("https://answers.launchpad.net/gui-ufw")
        
    def on_menuTranslate_activate(self, widget):
        """Launch browser with Documentation web"""
        webbrowser.open_new("https://translations.launchpad.net/gui-ufw/trunk/+translations")
        
    def on_winMain_delete_event(self, widget, event):
        """Close Button Main Window"""
        width, height = self.win_main.get_size()
        self.fw.update_config_file(width, height, self.vpanelmain.get_position())
        gtk.main_quit()
    
    def on_winAdd_delete_event(self, widget, event):
        """Close Button Window Add Rules"""
        self.btn_add_window.set_sensitive(1)
        self.menu_add.set_sensitive(1)
        self.win_add.hide()
        return True
       
    def on_winPreferences_delete_event(self, widget, event):
        """Close Button preferences Window"""
        self.win_preferences.hide()
        self._set_main_menus(1)
        return True
    
    def on_winLog_delete_event(self, widget, event):
        """Close Button Log Window"""
        self.win_log.hide()
        self._set_main_menus(1)
        return True
    
    def on_winAbout_delete_event(self, widget, event):
        """Close Button About Window"""
        self.win_about.hide()
        return True
    
    def on_winAdd_key_press_event(self, widget, event):
        """Press Scape Key Add Window > Hide window"""
        if event.keyval == gtk.keysyms.Escape :
            self.win_add.hide()
            self.btn_add_window.set_sensitive(1)
            self.menu_add.set_sensitive(1)
            return True    
    
    def on_winPreferences_key_press_event(self, widget, event):
        """Press Scape Key Preferences Window > Hide window"""
        if event.keyval == gtk.keysyms.Escape :
            self.win_preferences.hide()
            self._set_main_menus(1)
            return True  
    
    def on_winLog_key_press_event(self, widget, event):
        """Press Scape Key Log Window > Hide window"""
        if event.keyval == gtk.keysyms.Escape :
            self.win_log.hide()
            self._set_main_menus(1)
            return True  
    
class RefreshReport(threading.Thread):
    """Refresh Listening report in background"""
    def __init__(self, fw_status, model, lines, previus_lines, first_run, show_popups):
        threading.Thread.__init__(self)
        bus = dbus.SessionBus()
        notify_object = bus.get_object('org.freedesktop.Notifications','/org/freedesktop/Notifications')
        self.notify_interface = dbus.Interface(notify_object,'org.freedesktop.Notifications')
        self.path = Path()
        self.firewall_status = fw_status
        self.listening_model = model
        self.lines = lines
        self.previus_lines = previus_lines
        self.first_run = first_run
        self.show_popup = show_popups
    
    def run(self):
        """Show listening report in GUI"""
        self.listening_model.clear() 
        row = 0
        notif_msg = ""
        for line in self.lines:
            # Component lines for next notify
            if ( self.show_popup == "enable" ) and ( not line in self.previus_lines ) and ( not self.first_run ):
                msg_split = line.split("%")
                if msg_split[3] == '-' and msg_split[2] != '*':
                    msg = msg_split[2] + _(" on ") + msg_split[1] + " " + msg_split[0] # IP
                else:
                    msg = msg_split[3] + _(" on ") + msg_split[1] + " " + msg_split[0] # App
                
                if notif_msg == "":
                    notif_msg = msg
                else:
                    notif_msg = "\n".join([notif_msg, msg])
            
            # Update the Listening Report
            row += 1
            iter = self.listening_model.insert(row)
            line_split = line.split("%")
                        
            self.listening_model.set_value(iter, 0, row)
            self.listening_model.set_value(iter, 1, line_split[0].strip()) # Protocol
            self.listening_model.set_value(iter, 2, line_split[1].strip()) # Port
            self.listening_model.set_value(iter, 3, line_split[2].strip()) # Address
            self.listening_model.set_value(iter, 4, line_split[3].strip()) # App
            
            if self.firewall_status == "enable":
                if line_split[4] == "allow":
                    self.listening_model.set_value(iter, 5, __color__["red"])
                elif line_split[4] == "deny":
                    self.listening_model.set_value(iter, 5, __color__["green"])
                elif line_split[4] == "reject":
                    self.listening_model.set_value(iter, 5, __color__["blue"])
                elif line_split[4] == "limit":
                    self.listening_model.set_value(iter, 5, __color__["orange"])
                    
        # Notifications system for new connections        
        if ( self.show_popup == "enable" ) and ( not notif_msg == "" ):
            self.notify_interface.Notify("Gufw", 0, self.path.get_icon_path(), _("Firewall"), notif_msg, '', {"x-canonical-append": dbus.String("allowed")}, -1) # Expired time notification blocked by bug #390508
