/* tracktype for files dropped onto the track manager plus additional
 * handling requirements for this tracktype (e.g. suffix2tracktype database) */

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <signal.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <regex.h>
#include <signal.h>

#include "int.h"
#include "dialog.h"
#include "helpings.h"
#include "preferences.h"
#include "dependencies.h"

#ifndef __svr4__
void swab(const void *from,void *to,size_t n);
#endif

#include "tracks.h"
#include "stdfiletrack.h"
#include "varman.h"
#include "main.h"
#include "preferences.h"
#include "filetypes.h"
#include "piping.h"
#include "helpings.h"
#include "calc.h"
#include "mp3func.h"

/* uncomment if you want to debug stdfiletrack.c */
//#define DEBUG
//
#ifdef DEBUG
# include "colors.h"
# include "rterm.h"
#endif

/* contains program output after detectsize calls */
char outbuf[16384];

#ifdef HAVE_PTHREADS
pthread_mutex_t outbuf_mutex=PTHREAD_MUTEX_INITIALIZER;
#endif

/* this function is run in background and implements the byteswapping used
 * to convert little endian streams to big endian streams and back */

/* cstdin and cstdout are the pipes used to communicate with the main process */
void stdfiletrack_byteswap(int cstdin,int cstdout,int cstderr,gpointer data)
{
   char buf[BUFSIZE];
   char conv[BUFSIZE];
   int readcount=0;
   int writecount=0;

#ifdef DEBUG
   dprintf(cstderr,"stdfiletrack: started byteswap\n");
#endif
   do
     {
	readcount=read(cstdin,&buf,BUFSIZE);
	if (readcount>0)
	  {
	     /* swap high and lowbyte */
	     swab(&buf,&conv,readcount);
	     writecount=write(cstdout,&conv,readcount);
	  }
	;
     }
   while ((readcount>0)&&(writecount==readcount));
   close(cstdin);close(cstdout);
#ifdef DEBUG
   dprintf(cstderr,"stdfiletrack: byteswap closed IO channels...\n");
#endif
   close(cstderr);

   _exit(0);
}
;

int stdfiletrack_openpipe(void*trackinfo)
{
   int outp=-1;
   stdfiletrack_info *i;
   char *call;
   int *errpp;

   i=(stdfiletrack_info*)((tracks_trackinfo*)trackinfo)->data;

   if (i->filedes==-1)
     {
	char *path=strdup(i->uri);
	vfs_filesystem *fs=vfs_parseuri(i->uri,path);
#ifdef DEBUG
	printf ("stdfiletrack.c: reading file \"%s\"\n",i->uri);
#endif
	if (fs)
	  i->filedes=vfs_openfile(fs,path);
	if (i->filedes==-1)
	  {
	     perror ("error opening source file");
	  }
	else
	  {
	     if (strlen(filetypes_getfiltername(i->suffix))>0)
	       {
		  char *realname=vfs_getrealname(fs,path);

		  /* make local copy of piping call */
		  call=varman_replacevars_copy(global_defs,
					       filetypes_getfiltername(i->suffix));

		  /* this is for "filters" that can't take their input from stdin */
		  if (realname)
		    {
		       char *esc_name=helpings_escape(realname,
						      "\"",'\\');
		       varman_replacestring(call,"$file",esc_name);
		       free(esc_name);
		       free(realname);
		    };

#ifdef DEBUG
		  printf ("calling converter: %s\n",call);
#endif
		  i->convpid=piping_createcommandchain(call,
						       &i->filedes,
						       &outp);
		  i->filedes=outp;
		  free(call);
	       }
	     else
	       i->convpid=-1;
			    /* do a little->big endian conversion if requested */
	     if (!strcmp(filetypes_getfilter(i->suffix),
			 "little Endian"))
	       {
#ifdef DEBUG
		  errpp=&i->errp;i->errp=-1;
#else
		  errpp=NULL;
#endif

		  outp=-1; /* create yet another outp descriptor */
		  i->filterpid=piping_create_function(stdfiletrack_byteswap,
						      NULL,
						      &i->filedes,
						      &outp,
						      errpp);
#ifdef DEBUG
		  i->debugfilter=rterm_connectpipe(&Red,i->errp);
#endif
		  i->filedes=outp;
	       }
	     else
	       i->filterpid=-1;
	  }
	;
	free(path);
     }
   ;
#ifdef DEBUG
   printf ("stdfiletrack_openpipe: returning track pipe %i\n",
	   i->filedes);
#endif
   return i->filedes;
}
;

void stdfiletrack_closepipe(void*trackinfo)
{
   stdfiletrack_info *i;
   int status;

   i=(stdfiletrack_info*)((tracks_trackinfo*)trackinfo)->data;
   if (i->filedes!=-1)
     {
#ifdef DEBUG
	printf ("stdfiletrack_closepipe: closing filedescriptor\n");
#endif
	close(i->filedes);
	i->filedes=-1;

	if (i->filterpid!=-1)
	  {
#ifdef DEBUG
	     rterm_print ("waiting for filter ...\n");
#endif
	     waitpid(i->filterpid,&status,0);
	  }
	;
	if (i->convpid!=-1)
	  {
	     // send SIGTERM to converter
	     kill(i->convpid,SIGTERM);
#ifdef DEBUG
	     rterm_print ("waiting for converter ...\n");
#endif
	     waitpid(i->convpid,&status,0);
	  }
	;
#ifdef DEBUG
	     /* disactivate debugging pipe if in debugging mode */
	if (i->debugfilter)
	  rterm_disconnectpipe(i->debugfilter);
	close(i->errp);
#endif
     }
   ;

}
;

int stdfiletrack_calculate(tracks_trackinfo *dt)
{
   int pos=0;
   unsigned int wavsize=0;
   unsigned int mp3size=0;
   int error=0; // becomes true if an error occurred

   int result=0;
   stdfiletrack_info *info=(stdfiletrack_info*)(dt->data);
   char *path=strdup(info->uri);
   vfs_filesystem *fs=vfs_parseuri(info->uri,path);
   if (fs)
     {
	char *realname=vfs_getrealname(fs,path);
	if (strstr(filetypes_getsizealg(info->suffix),
		   "$filesize")!=NULL)
	  pos=vfs_filesize(fs,path);

	/* the following functions need a "realname" to work properly */
	if (realname)
	  {
	     OUTBUF_LOCK;
	     /* is client output required ? */
	     if ((strstr(filetypes_getsizealg(info->suffix),
			 "getpos")!=NULL)||
		 (strstr(filetypes_getsizealg(info->suffix),
			 "regexp")!=NULL))
	       {
		  if (piping_isvalid_command(filetypes_getdetectsize(info->suffix)))
		    {
		       char *esc_name=helpings_escape(realname,
						      "\"",'\\');
		       piping_create_getoutput(varman_replacestring_copy(filetypes_getdetectsize(info->suffix),
									 "$file",
									 esc_name),
					       outbuf,16384,PIPING_WATCHALL
					       );
		       free(esc_name);
		    }
		  else
		    {
		       /* tracksize detection command not found.
			* Tell the user about it */
		       char *message=(char*)malloc(1024+strlen(info->suffix)+
						   strlen(filetypes_getdetectsize(info->suffix)));
		       sprintf(message,_("The tool necessary to detect the tracksize of\n"
					 "files with the suffix '%s' couldn't be found on\n"
					 "your system. In it's current configuration,\n"
					 "GnomeToaster is trying to execute the following command\n"
					 "to obtain the tracksize:\n%s\n"),
			       info->suffix,
			       filetypes_getdetectsize(info->suffix));
		       dependencies_showdep(filetypes_getdetectsize(info->suffix),
					    message);
		       free(message);
		       error=1;
		    }; // tracksize detect tool not found
	       }
	     ;
	     if (strstr(filetypes_getsizealg(info->suffix),
			"$mp3size")!=NULL)
	       mp3size=mp3func_playingtime_rawbytes(realname);

	     free(realname);
	     OUTBUF_UNLOCK;
	  };

	if (strstr(filetypes_getsizealg(info->suffix),
		   "$wavsize")!=NULL)
	  {
	     int desc;
	     unsigned char buffer[2048];
	     int valid=1;
	     unsigned int bytespersecond=0;
	/* bytes to skip to reach the next chunk */
	     unsigned int chunksize=0;

#ifdef DEBUG
	     printf ("stdfiletrack: wavfile size detected\n");
#endif
	     desc=vfs_openfile(fs,path);
	     if (desc!=-1)
	       {
		  read (desc,buffer,4);
	     /* if first 4 characters are neither 'RIFF' nor 'WAVE' */
		  if (strncmp(buffer,"RIFF",4)&&strncmp(buffer,"WAVE",4))
		    {
		       valid=0;
#ifdef DEBUG
		       printf ("stdfiletrack_calculate: couldn't detect neither RIFF nor WAVE signatur on .wav file\n");
#endif
		    }
		  else
		    {
		  /* if the first 4 characters were 'RIFF', read the next
		   * 8 characters and see if the last 4 of them are 'WAVE' */
		       if (strncmp(buffer,"WAVE",4))
			 {
#ifdef DEBUG
			    printf ("stdfiletrack_calculate: found RIFF signature...\n");
#endif
			    read(desc,buffer,8);
			    if (strncmp((char*)&buffer[4],"WAVE",4))
			      {
				 valid=0;
#ifdef DEBUG
				 printf("stdfiletrack_calculate: couldn't find WAVE signature...\n");
#endif
			      };
			 };
		    };
	     /* if wave file looks valid */
		  if (valid)
		    {
#ifdef DEBUG
		       printf ("stdfiletrack: RIFF WAV or WAV header detected\n");
#endif
		  /* get chunk descriptor + chunk size */
		       while (read (desc,buffer,8)==8)
			 {
		       /* get chunk size
			* Reading each singular byte will
			* make this work on CPUs that require 32bit alignment
			* for reading 32bit values while at the same time
			* ensuring endianness */
			    chunksize=helpings_readuint(buffer,4);
			    if (!strncmp(buffer,"fmt ",4))
			      {
			    /* get format chunk */
				 if (read(desc,buffer,16)==16)
				   {
				 /* get bytes per second */
				      bytespersecond=helpings_readuint(buffer,8);
#ifdef DEBUG
				      printf("stdfiletrack: Format header detected, Wav is %i bytes per second\n",bytespersecond);
#endif
				 /* we already read 16 bytes out of this one,
				  * so just skip the rest */
				      chunksize-=16;
				   };
			      }
			    else
			      {
				 if (!strncmp(buffer,"data",4))
				   {
				 /* add length of current data chunk to playing time */
				      if (bytespersecond>0)
					wavsize+=(unsigned int)(((double)chunksize/(double)bytespersecond)*44100*4);
#ifdef DEBUG
				      printf ("stdfiletrack: data chunk found,size: %i bytes,increasing wavtracksize to %i\n",
					      chunksize,
					      wavsize);
#endif
				   };
			      };
		       /* seek next chunk */
			    lseek(desc,chunksize,SEEK_CUR);
			 };
		       close(desc);
		    };
	       };
	  };

	if (!error)
	  result=calc_function(filetypes_getsizealg(info->suffix),
			       3,
			       "$filesize",
			       (double)pos,
			       "$wavsize",
			       (double)wavsize,
			       "$mp3size",
			       (double)mp3size
			       );

	/* adjust tracksizes to sector boundaries */
	result=
	  helpings_pad(result,
		       tracks_tracktypesectorsizes[tracks_tracktypeid(dt->tracktype)]);

     }; // do we have something valid at all ?
   free(path);
   return result;
};

int stdfiletrack_tracksize(void*trackinfo)
{
   stdfiletrack_info *i;

   i=(stdfiletrack_info*)((tracks_trackinfo*)trackinfo)->data;

   if (i->tracksize==-1)
     i->tracksize=stdfiletrack_calculate((tracks_trackinfo*)trackinfo);

   return i->tracksize;
}
;

int stdfiletrack_dataavail(void*trackinfo)
{
   stdfiletrack_info *i;

   i=(stdfiletrack_info*)((tracks_trackinfo*)trackinfo)->data;

   return (i->filedes!=-1);
}
;

int  stdfiletrack_valid(void*trackinfo)  /* check if our converter exists */
{
   char *message;
   int result=0;
   stdfiletrack_info *i=(stdfiletrack_info*)((tracks_trackinfo*)trackinfo)->data;

   if (i)
     {
	char *filter=filetypes_getfiltername(i->suffix);
	if (strlen(filter))
	  {
	     message=(char*)malloc(1024+strlen(i->suffix)+strlen(filter));
	     sprintf(message,_("The Filter program for the \n"
			       "suffix '%s' couldn't be found on your system.\n"
			       "Please make sure your filetype registry\n"
			       "is setup properly.\n"
			       "Your current configuration is calling the following\n"
			       "command to create the desired track:\n%s\n"
			       ),i->suffix,filter);
	     result=piping_isvalid_commandchain(filter,message);
	     free(message);
	  }
	else
	    /* either no filter at all or a valid program */
	  result=1;
     };
   return result;
}
;

void stdfiletrack_destroy(void*trackinfo)
{
   stdfiletrack_info *i;

   i=(stdfiletrack_info*)((tracks_trackinfo*)trackinfo)->data;

   if (stdfiletrack_dataavail(trackinfo))
     close (i->filedes);
   free(i->uri);
   free(i->suffix);
   free(i);
}
;

/* Create a filetrack */
tracks_trackinfo *stdfiletrack_create(char *uri)
{
   tracks_trackinfo *dt;
   stdfiletrack_info *info;
   char *tracktype;
   tracks_precache cachemode;

   char description[256];
   char *filename=strdup(uri);
   vfs_parseuri(uri,filename);

   info=(stdfiletrack_info*)malloc(sizeof(stdfiletrack_info));
   info->uri=strdup(uri);
   info->filedes=-1;
   info->suffix=strrchr(filename,'.');
   if (info->suffix==NULL) /* if suffix not present set it to "" */
     info->suffix="";
   info->suffix=strdup(info->suffix);
   tracktype=filetypes_gettracktype(info->suffix);
   if (strlen(tracktype)==0)
     tracktype="data"; /* default to a data track if no tracktype is given. */
   /* it is *intended* to overwrite the pointer to the string here as the
    * string returned by filetypes_gettracktype() is not a copy but the real one */

   strcpy(description,tracktype);
   strcat(description," file ");

   strcat(description,filename);

   /* this means that we didn't detect the tracksize yet
    * so this is gonna be done when the tracksize method of this track
    * is first invoked */
   info->tracksize=-1;

   /* precache track either "atadd" or not at all */
   if (filetypes_getprecache(info->suffix))
     cachemode=atadd;
   else
     cachemode=none;

   if (strrchr(filename,'/'))
     strcpy(filename,(char*)(strrchr(filename,'/')+1));
   if (strchr(filename,'.'))
     *strchr(filename,'.')=0;

   dt=tracks_create(description,

		    /* FIXME: pass mp3 tags here in case of an mp3 file
		     * one day */
		    filename,
		    NULL,

		    tracktype,cachemode,
		    stdfiletrack_openpipe,
		    stdfiletrack_closepipe,
		    stdfiletrack_tracksize,
		    stdfiletrack_dataavail,
		    stdfiletrack_valid,
		    stdfiletrack_destroy,
		    info);

   free(filename);

   return dt;
}
;

/* pos handler for getpos(y,x) functio */
void stdfiletrack_getpos_handler(char *arg1,char *arg2,char *output)
{
   char *xpos,*ypos;
   int xposv,yposv;

#ifdef DEBUG
   printf ("stdfiletrack_getpos_handler: arg2 \"%s\"\n",arg2);
#endif
   ypos=arg2;xpos=strchr(arg2,',');
   if (xpos==NULL)
     {
	printf ("stdfiletrack_getpos_handler: Illegal argument.\n");
	xpos="0";
     }
   else
     xpos++;
   sscanf(ypos,"%d",&yposv);
   sscanf(xpos,"%d",&xposv);
#ifdef DEBUG
   printf ("stdfiletrack_getpos_handler: get number (%d,%d)\n",
	   yposv,xposv);
#endif

   sprintf(output,"%i",
	   helpings_getvalueinstring(helpings_getlineno(outbuf,
							yposv),
				     xposv));
#ifdef DEBUG
   printf ("stdfiletrack_getpos_handler: result \"%s\"\n",
	   output);
#endif

}
;

void stdfiletrack_regexp_handler(char *arg1,char *arg2,char *output)
{
   int errorcode;

   strcpy(output,"");
   /* arg2 is not empty and a string in quotation marks*/
   if ((strlen(arg2)>0)&&
       (*arg2=='"')&&
       (arg2[strlen(arg2)-1]=='"'))
     {
	regex_t preg;
	regmatch_t pmatch[2];
	/* cut quotation marks and resolve escape sequences */
	char *expression=helpings_resolveescapedchars(arg2+1);
	expression[strlen(expression)-1]=0;
#ifdef DEBUG
	printf ("stdfiletrack_regexp_handler: our expression is '%s'\n",expression);
	printf ("stdfiletrack_regexp_handler: matching on:'%s'\n",outbuf);
#endif

	/* continue if expression could be compiled successfully */
	errorcode=regcomp(&preg,
			  expression,
			  REG_EXTENDED|REG_NEWLINE);
	if (!errorcode)
	  {
	     errorcode=regexec(&preg,
			       outbuf,
			       2,
			       &pmatch[0],
			       0);
	     if (!errorcode)
	       {
		  if ((pmatch[1].rm_so!=-1)&&(pmatch[1].rm_eo!=-1))
		    {
		       /* the regexp function returns all results as strings.
			* use strval() to convert into numbers */
		       *output='"';
		       strncpy((char*)(output+1),
			       (char*)(outbuf+pmatch[1].rm_so),
			       (int)(pmatch[1].rm_eo-pmatch[1].rm_so));
		       /* and terminate */
		       output[pmatch[1].rm_eo-pmatch[1].rm_so+1]='"';
		       output[pmatch[1].rm_eo-pmatch[1].rm_so+2]=0;
		    };
	       }
	     else
	       {
		  char buffer[2048];

		  regerror(errorcode,
			   &preg,
			   buffer,
			   2048);
#ifdef DEBUG
		  printf ("stdfiletrack_regexp_handler: execution error - %s\n",
			  buffer);
#endif
	       };
	     /* free compiled expression */
	     regfree(&preg);
	  }
	else
	  {
#ifdef DEBUG
	     printf ("stdfiletrack_regexp_handler: compilation error\n");
#endif
	  };
	if (expression)
	  free(expression);
     };

#ifdef DEBUG
   printf ("stdfiletrack_regexp_handler: result \"%s\"\n",
	   output);
#endif

}
;

void stdfiletrack_init()
{
   calc_expression_register_prepend(calc_expression_create("getpos",
							   stdfiletrack_getpos_handler));
   calc_expression_register_prepend(calc_expression_create("regexp",
							   stdfiletrack_regexp_handler));

}
;
