/* this file collects some functions managing those parts of a progress
 * dialog appearing whenever files are being copied,that are the same on
 * each such occasion like a throuput display,a display for the time
 * remaining,one for the time already passed 
 * 
 * This new version of the datacopydlg provides several options to
 * customize the appearance of the dialog,
 * including the possibility to display several related progress bars
 * in one window (e.g. Status of the whole cd + status of the current track) */

#include "int.h"

#include <gtk/gtk.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdio.h>
#include <string.h>
#include <time.h>
#include <stdlib.h>
#include <stdarg.h>

#include "datacopydlg.h"
#include "helpings.h"
#include "layoutconfig.h"

/* uncomment if debugging is desired */
/* #define DEBUG */

int datacopydlg_delete_event(GtkWidget *w,			     
			     GdkEvent *e,
			     gpointer data)
{
   return TRUE; /* keep window over delete event */
};

void datacopydlg_destroy(datacopydlg_dlginfo *dlg)
{
   int i;
   
   gtk_grab_remove(dlg->messagebox);
   gtk_widget_destroy(dlg->messagebox);
   
   /* free all threads of this progress dialog */
   for (i=0;i<dlg->numthreads;i++)
     free(dlg->threads[i]);
   
   free(dlg);
}
;

/* restart a specific progress thread */
void datacopydlg_restartthread(datacopydlg_dlginfo *dlg,int threadnum,
			       int totalsize)
{
   dlg->threads[threadnum]->bytesdone=0;
   dlg->threads[threadnum]->totalsize=totalsize;
   time((time_t*)&dlg->threads[threadnum]->starttime);
   datacopydlg_updatedisplay(dlg);
};

void datacopydlg_configurethread(datacopydlg_dlginfo *dlg,int threadnum,
				 unsigned int options,
				 char *framename,
				 char *label,
				 int  totalsize)
  
{
   if (options!=0)
     dlg->threads[threadnum]->options=options;
   if (framename!=NULL)
     strcpy((char*)dlg->threads[threadnum]->framename,framename);
   if (label!=NULL)
     strcpy((char*)dlg->threads[threadnum]->label,label);
   if (totalsize!=-1)
     dlg->threads[threadnum]->totalsize=totalsize;

   if ((dlg->threads[threadnum]->options&DATACOPYDLG_SHOWFRAME)&&
       framename!=NULL)     
     gtk_frame_set_label(GTK_FRAME(dlg->threads[threadnum]->frame),
			 framename);   
   
   if (dlg->threads[threadnum]->options&(DATACOPYDLG_SHOWLABEL|
					 DATACOPYDLG_SHOWTHROUGHPUT|
					 DATACOPYDLG_SHOWTIME_REMAINING|
					 DATACOPYDLG_SHOWTIME_ELAPSED))
     gtk_widget_show(dlg->threads[threadnum]->infofield);
   else
     gtk_widget_hide(dlg->threads[threadnum]->infofield);
   
   if (dlg->threads[threadnum]->options&DATACOPYDLG_SHOWPROGRESS)
     gtk_widget_show(dlg->threads[threadnum]->progress);
   else
     gtk_widget_hide(dlg->threads[threadnum]->progress);     
   
   datacopydlg_updatedisplay(dlg);
};

void datacopydlg_updatedisplay(datacopydlg_dlginfo *dlg)
{
   int newtime;
   char buf[256];
   int remaining;
   gfloat done;
   
   int i;
   
   time((time_t*)&newtime);
   /* update our "threads" */
   for (i=0;i<dlg->numthreads;i++)
     {	     
	buf[0]=0; /* start with an empty string buffer */	     
	if (dlg->threads[i]->options&DATACOPYDLG_SHOWLABEL)
	  {
	     strcat(buf,dlg->threads[i]->label);
	     strcat(buf,"\n");
	  };	     
	/* display throughput within this thread ? */
	if (dlg->threads[i]->options&DATACOPYDLG_SHOWTHROUGHPUT)
	  {		  
	     strcat(buf,_("Throughput:"));
	     sprintf((char*)&buf[strlen(buf)],
		     "%.2f",
		     dlg->currentthroughput/1024);
	     strcat(buf,_(" kb/sec.\n"));
	  };
	if (dlg->threads[i]->options&DATACOPYDLG_SHOWTIME_ELAPSED)
	  {		  
	     strcat(buf,
		    helpings_secs2hms(difftime(newtime,
					       dlg->threads[i]->starttime)
				      ,1));
	     strcat(buf,_(" elapsed\n"));
	  };
	if (dlg->threads[i]->options&DATACOPYDLG_SHOWTIME_REMAINING)
	  {
	     if (dlg->currentthroughput!=0)
	       {		  
		  remaining=((dlg->threads[i]->totalsize-
			      dlg->threads[i]->bytesdone)/
			     dlg->currentthroughput);
		  if (dlg->threads[i]->bytesdone>dlg->threads[i]->totalsize) 
		    remaining=0;
		  strcat(buf,helpings_secs2hms(remaining,1));
		  strcat(buf,_(" remaining\n"));
	       }
	     else
	       strcat(buf,_("please wait...\n"));
	  };
	/* and update display */
	gtk_label_set_text(GTK_LABEL(dlg->threads[i]->infofield),
			   buf);
#ifdef DEBUG
	printf ("Thread [%i]: %i bytes processed...\n",
		i,
		dlg->threads[i]->bytesdone);
#endif
	done=(float)dlg->threads[i]->bytesdone/
	  (float)dlg->threads[i]->totalsize;
	if (done>1) done=1;
	if (done<0) done=0;
	
	if (dlg->threads[i]->options&DATACOPYDLG_SHOWPROGRESS)
	  {
	     gtk_progress_bar_update(GTK_PROGRESS_BAR(dlg->threads[i]->progress),
				     done);
	  };
	if (dlg->threads[i]->options&DATACOPYDLG_SHOWPROGRESSINTITLE)
	  {
	     sprintf(buf,_("(%.0f%% done) %s"),100*done,dlg->title);
	     gtk_window_set_title(GTK_WINDOW(dlg->messagebox),buf);
	  };
     }; /* step through all threads */
};
   

/* update the informations in datacopydlg_dlginfo and display changes 
 * this function should be called whenever a new block of data has been
 * processed */
void datacopydlg_newdatablock(datacopydlg_dlginfo *dlg,int size)
{
   int newtime;
   
   int i;
   
   /* needed for throughput calculation */
   dlg->bytessincelastmeasure+=size;
   /* update our "threads" */
   for (i=0;i<dlg->numthreads;i++)
     {	     
	/* increase number of bytes done */
	dlg->threads[i]->bytesdone+=size;
     };

   /* keep an update interval of >1 sec. */
   time((time_t*)&newtime);
   if (difftime(newtime,dlg->lastmeasure)>=1)
     {
	/* calculate current throughput */	
	dlg->currentthroughput=
	  ((float)dlg->bytessincelastmeasure/
	   difftime(newtime,dlg->lastmeasure)
	   );
	
	/* update the display */
	datacopydlg_updatedisplay(dlg);
	   
	/* reset update timer */
	dlg->bytessincelastmeasure=0;
	dlg->lastmeasure=newtime;
     };
}
;
       
/* create a progress box with title for processing totalsize bytes of data,
 * stophandler gets called whenever the user terminates the process by
 * clicking the stop button. if NULL,stop button will not even be visible
 * data can be a pointer to some additional data passed to the signal handler
 * but can also be set to NULL if not needed */
datacopydlg_dlginfo *datacopydlg_create(char *title,
					GtkSignalFunc stophandler,
					gpointer data,
					int numthreads,
					...					
					)
{
   GtkWidget *stop;
   datacopydlg_dlginfo *dlg;
   
   int i;
   va_list threadargs;
	
   dlg=(datacopydlg_dlginfo*)malloc(sizeof(datacopydlg_dlginfo));
   
   dlg->messagebox=gtk_dialog_new();
#ifndef DEBUG
   gtk_grab_add(dlg->messagebox); /* do not make progress box modal if in debug mode */
#endif
   
   gtk_window_set_title(GTK_WINDOW(dlg->messagebox),
			title);
   gtk_window_set_position(GTK_WINDOW(dlg->messagebox),
			   GTK_WIN_POS_CENTER);
   
   strcpy(dlg->title,title);

   /* initialize throughput meter */
   time((time_t*)&dlg->lastmeasure);
   dlg->bytessincelastmeasure=0;
   dlg->currentthroughput=0;
   
   dlg->numthreads=numthreads;
   va_start(threadargs,numthreads);
   for (i=0;i<numthreads;i++)
     {
	datacopydlg_threadinfo *thread;
	GtkWidget *threadcontainer;
	
	thread=(datacopydlg_threadinfo*)malloc(sizeof(datacopydlg_threadinfo));
	/* get the arguments */
	thread->options=va_arg(threadargs,unsigned int);
	strcpy(thread->framename,va_arg(threadargs,char *));
	strcpy(thread->label,va_arg(threadargs,char *));
	thread->totalsize=va_arg(threadargs,int);
	
	threadcontainer=gtk_vbox_new(0,0);
	if (thread->options&DATACOPYDLG_SHOWFRAME)
	  {
#ifdef DEBUG
	     printf ("datacopydlg_create: creating framed thread\n");
#endif
	     thread->frame=gtk_frame_new(thread->framename);
	     gtk_widget_show(thread->frame);
	     gtk_container_add(GTK_CONTAINER(thread->frame),threadcontainer);
	     gtk_widget_show(threadcontainer);	
	  }
	else
	  {	     
	     thread->frame=threadcontainer;
#ifdef DEBUG
	     printf ("datacopydlg_create: creating unframed thread\n");
#endif
	  };
	
	thread->infofield=gtk_label_new(_("please wait..."));
	gtk_label_set_justify(GTK_LABEL(thread->infofield),
			      GTK_JUSTIFY_LEFT);
	gtk_box_pack_start(GTK_BOX(threadcontainer),
			   thread->infofield,
			   1,1,1);
	
	thread->progress=gtk_progress_bar_new();
	gtk_progress_set_format_string(GTK_PROGRESS(thread->progress),
				       _("%p%% done"));
	gtk_progress_set_show_text(GTK_PROGRESS(thread->progress),
				   TRUE);	     
	gtk_box_pack_start(GTK_BOX(threadcontainer),
			   thread->progress,
			   TRUE,TRUE,0);
	
	/* decide what parts of the dialog should be shown */
	if (thread->options&(DATACOPYDLG_SHOWLABEL|
			     DATACOPYDLG_SHOWTHROUGHPUT|
			     DATACOPYDLG_SHOWTIME_REMAINING|
			     DATACOPYDLG_SHOWTIME_ELAPSED))
	  gtk_widget_show(thread->infofield);
	
	if (thread->options&DATACOPYDLG_SHOWPROGRESS)
	  gtk_widget_show(thread->progress);	
	
	thread->bytesdone=0;
	thread->starttime=dlg->lastmeasure;	
	
	/* add thread to the list of threads */
	dlg->threads[i]=thread;
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dlg->messagebox)->vbox),
			   thread->frame,0,0,0);
	gtk_widget_show(thread->frame);				  				  
     }; /* threads */
   va_end(threadargs);   
	
   if (stophandler!=NULL)
     {		  
	stop=gtk_button_new_with_label(_("Stop"));
	gtk_signal_connect(GTK_OBJECT(stop),"clicked",
			   stophandler,data);
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dlg->messagebox)->action_area),
			   stop,TRUE,TRUE,0);
	gtk_widget_show(stop);
     };
   gtk_signal_connect(GTK_OBJECT(dlg->messagebox),"delete_event",	
		      GTK_SIGNAL_FUNC(datacopydlg_delete_event),data);
   
   /* show whole dialog */   
   gtk_widget_show(dlg->messagebox);
   
   return dlg;
}
;

	
